/* Copyright (C) 2000, 2001  SWsoft, Singapore                                  
 *                                                                              
 *  This program is free software; you can redistribute it and/or modify        
 *  it under the terms of the GNU General Public License as published by        
 *  the Free Software Foundation; either version 2 of the License, or           
 *  (at your option) any later version.                                         
 *                                                                              
 *  This program is distributed in the hope that it will be useful,             
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of              
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               
 *  GNU General Public License for more details.                                
 *                                                                              
 *  You should have received a copy of the GNU General Public License           
 *  along with this program; if not, write to the Free Software                 
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA   
 */

// This file contains implementation of data sources enumerator


#include "hfiles.h"
#include "headers.h"
#include "mysqlMeta.h"
#include "srcrwset.h"

// CImpISourcesRowset::GetRowset ------------------------------------------------
//
// @mfunc Returns a enumerated sources rowset
//
// @rdesc HRESULT
//      @flag S_OK                  | The method succeeded.
//      @flag E_INVALIDARG          | ppCommand was NULL
//      @flag E_NOINTERFACE         | Could not obtain requested interface on DBSession object
//      @flag E_OUTOFMEMORY         | Out of memory
//      @flag E_FAIL                | Could not initialize rowset
//      @flag DB_E_NOAGGREGATION    | pUnkOuter was not NULL (this object does not support
//                                    being aggregated)
//      @flag DB_E_OBJECTOPEN       | The provider would have to open a new connection to 
//									  support the operation and DBPROP_MULTIPLECONNECTIONS is set to VARIANT_FALSE
//		@flag DB_S_ERRORSOCCURED	| Rowset is opened but at least one property was not set
//		@flag DB_E_ERRORSOCCURED	| No rowset is opened because at least one property was not set
//		@flag DB_E_NOTSUPPORTED		| Restrictions are not supported
//
STDMETHODIMP CImpISourcesRowset::GetSourcesRowset
    (
    IUnknown*		pUnkOuter,			//@parm IN		| Controlling IUnknown if being aggregated 
	REFIID			riid,				//@parm IN		| IID of requested rowset interface
	ULONG			cProperties,		//@parm IN		| The number of properties
	DBPROPSET		rgProperties[],		//@parm IN/OUT	| Properties
    IUnknown**		ppRowset			//@parm OUT		| A pointer to memory in which to return the interface pointer
    )
{
	INTERFACE_METHOD_START("ISourcesRowset::GetSourcesRowset");

	HRESULT hr = S_OK;
	HRESULT hrProp = S_OK;

	// Set up OUT variable for the case if we fail
	if( ppRowset )
		*ppRowset = NULL;

    // check in-params and NULL out-params in case of error
    if (ppRowset == NULL || cProperties < 0)
        return E_INVALIDARG;

	// check aggregation
	if( pUnkOuter != NULL && riid != IID_IUnknown )
		return DB_E_NOAGGREGATION;

	// Allocate memory for properties. No data source
	if( m_pUtilProp == NULL )
	{
		m_pUtilProp = new CUtilProp();
		if( m_pUtilProp == NULL )
			return E_OUTOFMEMORY;
	}

	CRowset *pRowset;

	// set properties if necessary
	if (cProperties )
	{
		// Check Arguments for use by properties
		hr = m_pUtilProp->SetPropertiesArgChk(cProperties, rgProperties);
		if( FAILED( hr ) )
			return hr;

		// set the properties
		hr = m_pUtilProp->SetProperties(PROPSET_ROWSET, cProperties, rgProperties);
	}
	
	if( (hr == DB_E_ERRORSOCCURRED) || 
		(hr == DB_S_ERRORSOCCURRED) )
	{
		// If all the properties set were SETIFCHEAP then we can set 
		// our status to DB_S_ERRORSOCCURRED and continue.
		for(ULONG ul = 0; ul < cProperties; ul++)
		{
			for(ULONG ul2 = 0; ul2 < rgProperties[ul].cProperties; ul2++)
			{
				// Check for a required property that failed, if found, we must return
				// DB_E_ERRORSOCCURRED
				if( (rgProperties[ul].rgProperties[ul2].dwStatus != DBPROPSTATUS_OK) &&
					(rgProperties[ul].rgProperties[ul2].dwOptions != DBPROPOPTIONS_SETIFCHEAP) )
					return DB_E_ERRORSOCCURRED;
			}
		}
		hrProp = DB_S_ERRORSOCCURRED;
	}

	if (FAILED(hr) && hrProp == S_OK)
	{
		TRACE( "   ISourcesRowset::GetSourcesRowset, failed: properties" );
		return hr;
	}


	// open and initialize a file object
    CSourcesData *pData = new CSourcesData();
    if (!pData)
		return E_OUTOFMEMORY;

    TRACE( "ISourcesRowset::GetRowset, CSourcesData - before init" );

	hr = pData->Init();
    if (hr != S_OK)
    {
        delete pData;
		TRACE( "ISourcesRowset::GetRowset, failed pData->Init" );
		return hr;
    }

	TRACE( "ISourcesRowset::GetRowset, CSourcesData - inited" );

	// allocate memory for new rowset
    pRowset = new CRowset(pUnkOuter);
    if (!pRowset)
	{
        delete  pData;
		return E_OUTOFMEMORY;
	}

	// Initialize new rowset
    if (!pRowset->Init(0, pData, ROWSET_SCROLL))
	{
        delete  pRowset;
		TRACE( "   ISourcesRowset::GetSourcesRowset, failed IRowset->Init" );
		return  E_FAIL;
	}

 
	//At this point we have handed off the pData pointer to the
	//provider so null it out.
	pData = NULL;

    // get requested interface pointer on rowset\cursor
    hr = pRowset->QueryInterface(riid, (void**)ppRowset);
    if (FAILED(hr))
    {
        delete pRowset;
		TRACE2( "  ISourcesRowset::GetSourcesRowset, QI - hr=%d", hr );
		return hr;
    }

	return (hr == S_OK) ? hrProp : hr;

	INTERFACE_METHOD_END();
}


// CImpIParseDislpayName::ParseDisplayName ------------------------------------------------
//
// @mfunc Returns a moniker to data source
//
STDMETHODIMP CImpIParseDisplayName::ParseDisplayName
	(	
	IBindCtx *pbc,     //Pointer to bind context
	LPOLESTR pszDisplayName, //Pointer to string containing display name
	ULONG *pchEaten,   //Pointer to length, in characters, of display name
	IMoniker **ppmkOut //Address of output variable that receives the 
		               //resulting IMoniker interface pointer
	)
{
	INTERFACE_METHOD_START("IParseDislpayName::ParseDisplayName");

	// set up args for the case if an error occur
	if( ppmkOut )
		*ppmkOut = NULL;

	// check args
	if( ppmkOut == NULL ) 
		return E_UNEXPECTED;

	// check args
	if( pchEaten == NULL ) 
		return MK_E_NOOBJECT;

	*pchEaten = 0; // unappetizing string. Nothing is eaten
	return MK_E_NOOBJECT;  // Report that name is not in file namespace

	INTERFACE_METHOD_END();
}


///////////////////////////////////////////////
// CSourcesData
///////////////////////////////////////////////

//Skip relative
HRESULT CSourcesData::GetFetchedData(LONG lRow, DWORD* pdwBmk)
{
	// Check if array was used
	if( m_plRowPositions==NULL )
		return DB_S_ENDOFROWSET;
	
	// Sign does no matter
	lRow = abs(lRow);

	// Get appropriate row position value from buffer
	HRESULT hr = m_plRowPositions[ lRow ].m_hr;
	m_lCurrentRow = m_plRowPositions[ lRow ].m_lRowPos;
	if( pdwBmk != NULL )
		*pdwBmk = Pos2Bmk( m_lCurrentRow );

	return hr;
}


// Invalidate buffers
HRESULT CSourcesData::SetMoveInfo( HRESULT hr )
{
	delete m_plRowPositions;
	m_plRowPositions = NULL;
	return hr;
}

//@cmember Move relative
HRESULT CSourcesData::MovePending(LONG nRows, BOOL bSaveCurrentPosition)
{
	TRACE3( "CSourcesData::Move %d (total %d)", nRows, m_cRows );
	// Allocate new buffer for row information
	delete m_plRowPositions;
	m_plRowPositions = new CRowInfo[ abs(nRows) + 1 ];
	if( m_plRowPositions == NULL )
		return E_OUTOFMEMORY;

	HRESULT hr = S_OK;
	LONG lIdx = 0;

	// Save first row position if asked
	if( bSaveCurrentPosition )
	{
		m_plRowPositions[ lIdx ].m_lRowPos = m_lCurrentRow;
		m_plRowPositions[ lIdx ].m_hr = ( m_lCurrentRow >= 0 && m_lCurrentRow < m_cRows ) ? S_OK : DB_S_ENDOFROWSET;
		lIdx++;
	}

	// Save other row positions
	for( LONG i = 1; i <= abs(nRows); i++, lIdx++ )
	{
		LONG lCurrentRow = m_lCurrentRow + sgn(nRows) * i;
		
		if( m_lCurrentRow >= m_cRows || lCurrentRow >= m_cRows )
		{
			m_plRowPositions[ lIdx ].m_lRowPos = m_cRows;
			hr = m_plRowPositions[ lIdx ].m_hr = DB_S_ENDOFROWSET; //EOF
		}
		else if( m_lCurrentRow < 0 || lCurrentRow < 0)
		{
			m_plRowPositions[ lIdx ].m_lRowPos = -1; 
			hr = m_plRowPositions[ lIdx ].m_hr = DB_S_ENDOFROWSET; //BOF
		}
		else
		{
			m_plRowPositions[ lIdx ].m_lRowPos = lCurrentRow; 
			m_plRowPositions[ lIdx ].m_hr = S_OK; //BOF
		}
	}

	return hr;
}


//@cmember Move to first row
HRESULT CSourcesData::MoveFirst()
{
	TRACE( "CSourcesData::MoveFirst" );
	m_lCurrentRow = 0;
	return S_OK;
}

//@cmember Move to last row
HRESULT CSourcesData::MoveLast()
{
	TRACE( "CSourcesData::MoveLast" );
	m_lCurrentRow = m_cRows - 1;
	return S_OK;
}

//@cmember Move to specified bookmark
HRESULT CSourcesData::MoveBookmark(ULONG* pulBookmark)
{
	TRACE( "CSourcesData::MoveBookmark" );
	ULONG ulPos = Bmk2Pos(*pulBookmark);
	if (ulPos >= m_cRows || ulPos < 0)
		return DB_E_BADBOOKMARK;

	m_lCurrentRow = ulPos;
	return S_OK;
}


//@cmember Find row from current position
HRESULT CSourcesData::Find(ULONG icol, PCOLUMNDATA pDst/*void* pvValue*/,	DBCOMPAREOP	CompareOp, DBTYPE dbType, bool bForward)
{
	TRACE( "CSourcesData::Find" );
	return E_FAIL;
}


//@cmember Return the number of rows in the table	
HRESULT CSourcesData::GetRowCnt(DWORD* pdwRows)
{
	TRACE( "CSourcesData::GetRowCnt" );
	*pdwRows = m_cRows;
	return S_OK;
}

//@cmember Get relative position (by percentage) of the row in  table
HRESULT CSourcesData::GetPercentage(ULONG* pulBookmark, double *pdfPPos)
{
	TRACE( "CSourcesData::GetPercentage" );
	ULONG ulPos = Bmk2Pos(*pulBookmark);

	if (ulPos >= m_cRows || ulPos < 0)
		return E_INVALIDARG;

	*pdfPPos = double(ulPos + 1) / double(m_cRows);
	return S_OK;
}

//@cmember Get bookmark for current row
HRESULT CSourcesData::GetBookmark(ULONG *pulBookmark)
{
	TRACE( "CSourcesData::GetBookmark" );
	
	*pulBookmark = Pos2Bmk(m_lCurrentRow);
	return S_OK;
}

//@cmember Retrieve number of columns
HRESULT CSourcesData::GetColumnCnt(DWORD* pdwCount)
{
	TRACE( "CSourcesData::GetColumnCnt" );

	*pdwCount = m_cCols;
	return S_OK;
}

//@cmember Retrieve column information
HRESULT CSourcesData::GetColumnInfo(DWORD dwCol, DBCOLUMNINFO* pInfo)
{
	TRACE( "CSourcesData::GetColumnInfo" );
	
	if (dwCol < 1 || dwCol > m_cCols)
		return E_INVALIDARG;
	
	COLINFO *pCol = m_pColumns + dwCol - 1;
	
	//Copy column info	
	pInfo->iOrdinal		= dwCol; 
    pInfo->pTypeInfo	= NULL; 
    pInfo->dwFlags		= pCol->dwFlags; 
    pInfo->ulColumnSize = pCol->ulSize; 
    pInfo->wType		= pCol->wType; 
    pInfo->bPrecision	= 0; 
    pInfo->bScale		= 0; 
    pInfo->columnid.eKind          = DBKIND_PROPID;
    pInfo->columnid.uGuid.guid     = GUID_NULL;
    pInfo->columnid.uName.ulPropid = pInfo->iOrdinal;

	// Copy name in special way
	pInfo->pwszName = new WCHAR[MAX_COLUMN_NAME_CHARS];
	wcscpy0(pInfo->pwszName, pCol->wszName, MAX_COLUMN_NAME_CHARS - 1);

	return S_OK;
}

	
//@cmember Fetch row data
HRESULT CSourcesData::GetRow(ULONG* pulOffset, BYTE* pbProvRow)
{
	TRACE( "CSourcesData::GetRow" );

	SOURCEITEMDATA *pSource = m_pSourceItemData + m_lCurrentRow;
	for (int i = 1; i <= m_cCols; i++)
	{
		COLUMNDATA *pCol = (COLUMNDATA*)((BYTE*)pbProvRow + pulOffset[i]);
		pCol->dwLength = m_pColumns[i - 1].ulSize;
		pCol->dwStatus = DBSTATUS_S_OK;

		switch (i)
		{
		case 1:	// SOURCES_NAME
			wcscpy( (LPOLESTR)pCol->bData, pSource->m_wszSourceName );
			pCol->dwLength = wcslen( (LPOLESTR)pCol->bData ) * sizeof(WCHAR);
			break;

		case 2:	// SOURCES_PARSENAME
			wcscpy( (LPOLESTR)pCol->bData, pSource->m_wszParseName );
			//wcscat( (LPOLESTR)pCol->bData, L"\\file.ddf" );
			pCol->dwLength = wcslen( (LPOLESTR)pCol->bData ) * sizeof(WCHAR);
			break;

		case 3:	//SOURCES_DESCRIPTION
			wcscpy( (LPOLESTR)pCol->bData, pSource->m_wszDescription );
			pCol->dwLength = wcslen( pSource->m_wszDescription ) * sizeof(WCHAR);
			break;
		
		case 4:	//SOURCES_TYPE
			*(WORD*)pCol->bData = DBSOURCETYPE_DATASOURCE;
			break;

		case 5:	//SOURCES_ISPARENT
			*(VARIANT_BOOL*)pCol->bData = VARIANT_FALSE;
			break;

		default://All other columns are NULL
			pCol->dwStatus = DBSTATUS_S_ISNULL;
			pCol->dwLength = 0; 
		}
	}

	TRACE2("CSourcesData::GetRow: %S", pSource->m_wszDescription );
    return S_OK;	
}


//@cmember Update the current rows values
HRESULT CSourcesData::UpdateRow(
	ULONG*		pulBookmark,		//@parm IN | Bookmark of row to update	
    ULONG*      ulOffset,		//@parm IN | Array of offsets for the columns
    BYTE*       pbProvRow,		//@parm IN | Data to update row with.
	PACCESSOR	pAccessor,		//@parm IN | Accessor with bindings to update.
	BYTE*		pbProvRowOld
								)
{
	TRACE( "CSourcesData::UpdateRow" );
	return E_FAIL;
}

//@cmember Insert new row
HRESULT CSourcesData::InsertRow(ULONG* ulOffset, BYTE* pbProvRow, PACCESSOR pAccessor)
{
	TRACE( "CSourcesData::InsertRow" );
	return E_FAIL;
}

//@cmember Remove row with the bmk
HRESULT CSourcesData::DeleteRow(ULONG* pulBookmark)
{
	TRACE( "CSourcesData::DeleteRow" );
	return E_FAIL;
}

CSourcesData::CSourcesData()
{
	CLEAR_CONSTRUCT( CSourcesData );
	
	m_lCurrentRow	= -1;
}

CSourcesData::~CSourcesData()
{
	delete [] m_pSourceItemData;
	delete [] m_plRowPositions;
}


//Initialize object with schema data
HRESULT CSourcesData::Init()
{
	TRACE( "CSourcesData::Init" );

	HRESULT ret = E_FAIL;
	
	// Enumerate data sources only on local server
	DSNINFO dsnInfo;
	
	//Init columns
	static COLINFO Columns[] = 
	{							
		{L"SOURCES_NAME",		DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"SOURCES_PARSENAME",	DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"SOURCES_DESCRIPTION",DBTYPE_WSTR,	MAX_NAME_SIZE,	DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"SOURCES_TYPE",		DBTYPE_UI2,		sizeof(WORD),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL},
		{L"SOURCES_ISPARENT",	DBTYPE_BOOL,	sizeof(VARIANT_BOOL),	DBCOLUMNFLAGS_ISFIXEDLENGTH | DBCOLUMNFLAGS_ISNULLABLE | DBCOLUMNFLAGS_MAYBENULL}
	};
	m_pColumns = Columns;
	m_cCols = 5;

	/////////////////////////////////////////////////
	/// Enumerate all data sources
	/////////////////////////////////////////////////

	HRESULT hr = E_FAIL;
	
	// Enumerators for DSNs. (Should be initialized!) Count of their items
	EnumOfEnumsDataSources	enumDataSources;
	int nEnumsTotal;
	hr = enumDataSources.Init( &dsnInfo );
	if( FAILED( hr ) )
		return hr;
	hr = enumDataSources.Count( &nEnumsTotal );
	if( FAILED( hr ) )
	{
		enumDataSources.Done();
		return hr;
	}
	if( !nEnumsTotal )
	{
		enumDataSources.Done();
		return S_OK;  // Nothing to do
	}

	struct	EnumEnumTag {
		START_CLASS();
	public:
		EnumDatabase*	m_pEnum;
		int	m_count;
		BOOL			m_bOK;
		EnumEnumTag() { CLEAR_CONSTRUCT(EnumEnumTag); }
		FINISH_CLASS();
	} * dbEnums = new EnumEnumTag[ nEnumsTotal ];
	if( dbEnums == NULL )
	{
		enumDataSources.Done();
		return E_OUTOFMEMORY;
	}

	int nCountAll = 0;
	BOOL bHasSuccess = FALSE;
	for( int i = 0; i < nEnumsTotal; i++ )
	{
		dbEnums[ i ].m_bOK = SUCCEEDED( enumDataSources.Next( &dsnInfo ) );
		if( dbEnums[ i ].m_bOK ) 
			dbEnums[ i ].m_bOK = SUCCEEDED( enumDataSources.GetNewObject( dsnInfo.m_type, &dbEnums[ i ].m_pEnum ) );
		if( dbEnums[ i ].m_bOK ) 
			dbEnums[ i ].m_bOK = SUCCEEDED( dbEnums[ i ].m_pEnum->Init( &dsnInfo ) );
		if( dbEnums[ i ].m_bOK ) 
			dbEnums[ i ].m_bOK = SUCCEEDED( dbEnums[ i ].m_pEnum->Count( &dbEnums[ i ].m_count ) );
		if( dbEnums[ i ].m_bOK )
		{
			nCountAll += dbEnums[ i ].m_count;
			bHasSuccess = TRUE;
		}
	}

	
	if( !bHasSuccess ) // All failed!
		goto End;	
	
	ret = S_OK;	 // We will have  something
	
	if( nCountAll == 0 ) // Nothing to do!
		goto End;	
	
	//Init data
	m_pSourceItemData = new SOURCEITEMDATA[ nCountAll ];
	if( m_pSourceItemData == NULL ) 
	{
		ret = E_OUTOFMEMORY; 
		goto End;
	}
	
	//Fill out pointer array
	m_cRows = 0;
	HRESULT hrNext;
	for( i = 0; i < nEnumsTotal; i++ )
		if( dbEnums[ i ].m_bOK )
			while( SUCCEEDED( hrNext = dbEnums[ i ].m_pEnum->Next( &dsnInfo ) ) )	
				if( hrNext == S_OK )  // S_OK means that file "....\file.ddf" exists
				{
					A2Wsz( dsnInfo.m_szDataSource, m_pSourceItemData[ m_cRows ].m_wszSourceName );
					_snwprintf( m_pSourceItemData[ m_cRows ].m_wszParseName, MAXSTR(m_pSourceItemData[ m_cRows ].m_wszParseName), 
							L"DSN=Unknown;DB=%s;SYSDB=%s", m_pSourceItemData[ m_cRows ].m_wszSourceName, m_pSourceItemData[ m_cRows ].m_wszSourceName );
					A2Wsz( dsnInfo.m_szDescription, m_pSourceItemData[ m_cRows ].m_wszDescription );
					m_cRows++;
				}
		
End:
	// Completing: close all enums
	// delete subItems
	for( i = 0; i < nEnumsTotal; i++ )
		if( dbEnums[ i ].m_pEnum )
		{
			dbEnums[ i ].m_pEnum->Done();
			delete dbEnums[ i ].m_pEnum; // It is not COM object
		}

	delete dbEnums; // delete items holder

	enumDataSources.Done(); 

	// Move to first line if succeeded
	return FAILED(hr) ? hr : MoveFirst();
}


// CSources::AddRef -------------------------------------------------------
//
// @mfunc Increments a persistence count for the object
//
// @rdesc Current reference count
//
ULONG	CSources::AddRef(void)
{																	
    TRACE2( "CSources::AddRef: %d", int(m_cRef+1) );
	return ++m_cRef;
}
																	

// CSources::Release ------------------------------------------------------
//
// @mfunc Decrements a persistence count for the object and if
// persistence count is 0, the object destroys itself.
//
// @rdesc Current reference count
//
ULONG	CSources::Release(void)									
{																	
    TRACE2( "CSources::Release: %d", int(m_cRef-1) );
    if (!--m_cRef)
    {
        delete this;
        return 0;
    }

    return m_cRef;
}
																	

// CSources::QueryInterface -----------------------------------------------
//
// @mfunc Returns a pointer to a specified interface. Callers use
// QueryInterface to determine which interfaces the called object
// supports.
//
// @rdesc HRESULT indicating the status of the method
//      @flag S_OK          | Interface is supported and ppvObject is set.
//      @flag E_NOINTERFACE | Interface is not supported by the object
//      @flag E_INVALIDARG  | One or more arguments are invalid.
//
HRESULT CSources::QueryInterface(REFIID riid, LPVOID *ppv)
{																	
    // Is the pointer bad?
    if (ppv == NULL)
        return E_INVALIDARG;

    //  Place NULL in *ppv in case of failure
    *ppv = NULL;

    //  This is the non-delegating IUnknown implementation
    if (riid == IID_IUnknown)
        *ppv = (LPVOID) this;
    else if (riid == IID_ISourcesRowset)
        *ppv = (LPVOID) m_pISourcesRowset;
    else if (riid == IID_IParseDisplayName)
        *ppv = (LPVOID) m_pIParseDisplayName;

    if (*ppv == NULL)
		return E_NOINTERFACE;

    //  If we're going to return an interface, AddRef it first
    ((LPUNKNOWN) *ppv)->AddRef();
    return S_OK;
}


// CSources::Init ------------------------------------------------------------
//
// @mfunc Initialize the sources enumerator Object
//
// @rdesc Did the Initialization Succeed
//      @flag  TRUE | Initialization succeeded
//      @flag  FALSE | Initialization failed
//
BOOL CSources::Init ( BOOL bMySqlMode )
{
	TRACE( "CSources::Init" );

    LPUNKNOWN   pIUnknown = (LPUNKNOWN) this;

    if (m_pUnkOuter)
        pIUnknown = m_pUnkOuter;

	// Instantiate the data conversion service object
	if( !g_pIDataConvert )
	{
		HRESULT hr = CoCreateInstance(CLSID_OLEDB_CONVERSIONLIBRARY,
							  NULL,
							  CLSCTX_INPROC_SERVER,
							  IID_IDataConvert,
							  (void **)&g_pIDataConvert);
		if( FAILED(hr) )
			return FALSE;
	}
	else
		// Already instantiated, increment reference count
		g_pIDataConvert->AddRef();

    // Allocate contained interface objects
    // Note that our approach is simple - we always create *all* of the Enumerator interfaces
    // If our properties were read\write (i.e., could be set), we would need to
    // consult properties to known which interfaces to create.
    // Also, none of our interfaces conflict. If any did conflict, then we could
    // not just blindly create them all.
    m_pIParseDisplayName        = new CImpIParseDisplayName(pIUnknown);
    m_pISourcesRowset           = new CImpISourcesRowset(pIUnknown);

    // if all interfaces were created, return success
    return (BOOL) (m_pIParseDisplayName &&
                   m_pISourcesRowset );
}


// CSources::CSources  -- Constructs the sources enumerator Object
CSources::CSources(LPUNKNOWN pIUnknown )
{
	CLEAR_CONSTRUCT( CSources );
	m_pUnkOuter = pIUnknown;
}

// CSources::~CSources --- Destructs the sources enumerator Object
CSources::~CSources()
{
		// Decrement the ref count on the data conversion object
	if( g_pIDataConvert )
	{
		ULONG ulRefCount = g_pIDataConvert->Release();

		// Is it gone for good?
		if( !ulRefCount )
			g_pIDataConvert = NULL;
	}
	
	// Release allocated memory
	delete m_pIParseDisplayName;
	delete m_pISourcesRowset;
}
