/* Copyright (C) 2000, 2001  SWsoft, Singapore                                  
 *                                                                              
 *  This program is free software; you can redistribute it and/or modify        
 *  it under the terms of the GNU General Public License as published by        
 *  the Free Software Foundation; either version 2 of the License, or           
 *  (at your option) any later version.                                         
 *                                                                              
 *  This program is distributed in the hope that it will be useful,             
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of              
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the               
 *  GNU General Public License for more details.                                
 *                                                                              
 *  You should have received a copy of the GNU General Public License           
 *  along with this program; if not, write to the Free Software                 
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA   
 */

// Includes ------------------------------------------------------------------

#include "hfiles.h"
#include "headers.h"

// Code ----------------------------------------------------------------------




//  IRowsetChange specific methods

// CImpIRowsetChange::SetData ------------------------------------------------
//
// @mfunc Sets new data values into fields of a row.
//
// @rdesc HRESULT
//      @flag S_OK                   | The method succeeded
//      @flag E_OUTOFMEMORY          | Out of memory
//      @flag DB_E_BADACCESSORHANDLE | Bad accessor handle
//      @flag DB_E_READONLYACCESSOR  | Tried to write through a read-only accessor
//      @flag DB_E_BADROWHANDLE      | Invalid row handle
//      @flag E_INVALIDARG           | pData was NULL and accessor is not null one
//      @flag E_FAIL                 | Provider-specific error
//      @flag OTHER                  | Other HRESULTs returned by called functions
//
STDMETHODIMP CImpIRowsetChange::SetData
    (
    HROW        hRow,       //@parm IN | Handle of the row in which to set the data
    HACCESSOR   hAccessor,  //@parm IN | Handle to the accessor to use
    void*		pData       //@parm IN | Pointer to the data
    )
{
	INTERFACE_METHOD_START("IRowsetChange::SetData");

    BYTE*       pbProvRow;
    HRESULT     hr; 
    BYTE*       rgbRowDataSave = NULL;
	ULONG*		pColumns = NULL;
	bool		bOk = true;
    PACCESSOR	pAccessor;

	//Check reentrant
	if (m_pObj->m_bNotReentrant)
		return DB_E_NOTREENTRANT;

    if (m_pObj->m_pExtBufferAccessor == NULL  ||   
        FAILED( m_pObj->m_pExtBufferAccessor->GetItemOfExtBuffer((ULONG) hAccessor, &pAccessor)) ||
        pAccessor == NULL)
    {
        return DB_E_BADACCESSORHANDLE;
    }

	//Create internal arrays    
	rgbRowDataSave = (BYTE*) malloc(m_pObj->m_cbRowSize);
    if (NULL == rgbRowDataSave)
        return E_OUTOFMEMORY;

	pColumns = new ULONG[pAccessor->cBindings + 1];
    if (NULL == pColumns)
        return E_OUTOFMEMORY;

	//Fill out changed column array
    for (int i = 0; i < pAccessor->cBindings; i++)
        pColumns[i] = pAccessor->rgBindings[i].iOrdinal;


    // Is row handle right?
    //if ((m_pObj->m_prowbitsIBuffer)->IsSlotSet((ULONG) hRow ) != S_OK)
	if ((m_pObj->m_prowbitsIBuffer)->IsSlotSet( m_pObj->GetSlotNumber( hRow ) ) != S_OK)
        return DB_E_BADROWHANDLE;

    // Ensure a source of data
    if (pData == NULL && !pAccessor->bNullAccessor )
        return E_INVALIDARG;

    pbProvRow = (BYTE*)(m_pObj->GetRowBuff((ULONG)hRow));

    // Save the row.
    memcpy(rgbRowDataSave, pbProvRow, m_pObj->m_cbRowSize);


	//Send notifications
	hr = m_pObj->OnFieldChange
						(
						hRow,
						pAccessor->cBindings,
						pColumns, 
						DBREASON_COLUMN_SET,
						DBEVENTPHASE_OKTODO,
						FALSE
						);			

	if (hr != S_OK)
		return DB_E_CANCELED;

	hr = m_pObj->OnFieldChange
						(
						hRow,
						pAccessor->cBindings,
						pColumns, 
						DBREASON_COLUMN_SET,
						DBEVENTPHASE_ABOUTTODO,
						FALSE
						);			
	if (hr != S_OK)
		return DB_E_CANCELED;


	//Fill out row with new data
	hr = m_pObj->ApplyAccessor(hAccessor, pbProvRow, pData);
	if (FAILED(hr))
		goto SetData_Exit;
		
	bOk = (hr == S_OK);

	//Move to specified location  -- in UpdateRow now
	//hr = m_pObj->m_pData->MoveBookmark(PROWBUFF(pbProvRow)->dwBmks);
	//if (hr != S_OK)
	//	goto SetData_Exit;

	// Carry out the update.
	hr = m_pObj->m_pData->UpdateRow( PROWBUFF(pbProvRow)->dwBmks, m_pObj->m_rgdwDataOffsets, pbProvRow, pAccessor, rgbRowDataSave );
	if (hr != S_OK)
	{
		// Restore the row to its previous state 
		memcpy(pbProvRow, rgbRowDataSave, m_pObj->m_cbRowSize);
	}

SetData_Exit:
	// Notify consumer that row was updated. 
	m_pObj->OnFieldChange(hRow,
						  pAccessor->cBindings,
						  pColumns, 
						  DBREASON_COLUMN_SET,
						  (hr == S_OK) ? DBEVENTPHASE_DIDEVENT : DBEVENTPHASE_FAILEDTODO,
						  TRUE);			

    free(rgbRowDataSave);
	delete [] pColumns;

    // We report any lossy conversions with a special status.
    // Note that DB_S_ERRORSOCCURED is a success, rather than failure.
	if (hr == S_OK)
		return bOk ? S_OK : DB_S_ERRORSOCCURRED;
	else 
		return hr;

	INTERFACE_METHOD_END();
}



// CImpIRowsetChange::DeleteRows ---------------------------------------
//
// @mfunc Deletes rows from the provider.  If Errors on individual rows
// occur, the DBERRORINFO array is updated to reflect the error and S_FALSE
// is returned instead of S_OK.
//
// @rdesc HRESULT indicating the status of the method
//      @Flag S_OK                  | All row handles deleted
//      @Flag DB_S_ERRORSOCCURRED   | Some, but not all, row handles deleted
//      @Flag E_INVALIDARG          | Arguments did not match spec.
//      @Flag E_OUTOFMEMORY         | Could not allocated error array
//
STDMETHODIMP CImpIRowsetChange::DeleteRows
    (
		HCHAPTER		hReserved,		//@parm IN	| Reserved for future use
		ULONG           cRows,			//@parm IN	| Number of rows to delete
		const HROW      rghRows[],		//@parm IN	| Array of handles to delete
		DBROWSTATUS		rgRowStatus[]	//@parm OUT | Error information
    )
{
	INTERFACE_METHOD_START("IRowsetChange::DeleteRows");

	HRESULT		hr;

    ULONG       ihRow        = 0L;
    ULONG       cErrors      = 0L;
    ULONG       cRowReleased = 0L;
    BYTE*       pbProvRow;

	ULONG		ulPropIndex;
	bool		bSingleGranularity;

	//Check reentrant
	if (m_pObj->m_bNotReentrant)
		return DB_E_NOTREENTRANT;

    // If No Row handle, just return.
	if (0 == cRows)
        return S_OK;

    // Check for Invalid Arguments
    if ((cRows >= 1) && (NULL == rghRows))
        return E_INVALIDARG;

	if (m_pObj->m_pUtilProp->GetPropIndex(DBPROP_NOTIFICATIONGRANULARITY, &ulPropIndex) == TRUE)
		bSingleGranularity = m_pObj->m_pUtilProp->m_rgproperties[ulPropIndex].boolVal == VARIANT_TRUE;

	if (!bSingleGranularity)
	{
		//Send notifications
		hr = m_pObj->OnRowChange(cRows, rghRows, DBREASON_ROW_DELETE, DBEVENTPHASE_OKTODO, FALSE);			
		if (hr != S_OK)
			return DB_E_CANCELED;

		hr = m_pObj->OnRowChange(cRows, rghRows, DBREASON_ROW_DELETE, DBEVENTPHASE_ABOUTTODO, FALSE);			
		if (hr != S_OK)
			return DB_E_CANCELED;
	}

    // Process row handles
    while (ihRow < cRows)
    {
		if (bSingleGranularity)
		{
			//Send notifications
			hr = m_pObj->OnRowChange(1, &rghRows[ihRow], DBREASON_ROW_DELETE, DBEVENTPHASE_OKTODO, FALSE);			
			if (hr != S_OK)
				continue;

			hr = m_pObj->OnRowChange(1, &rghRows[ihRow], DBREASON_ROW_DELETE, DBEVENTPHASE_ABOUTTODO, FALSE);			
			if (hr != S_OK)
				continue;
		}

		if (rgRowStatus)
			rgRowStatus[ihRow] = DBROWSTATUS_S_OK;

        // Is row handle valid
        //if (S_OK != (m_pObj->m_prowbitsIBuffer)->IsSlotSet((ULONG)rghRows[ihRow]))
		if (S_OK != (m_pObj->m_prowbitsIBuffer)->IsSlotSet( m_pObj->GetSlotNumber( rghRows[ihRow])))
        {
            // Log Error
			if (rgRowStatus)
				rgRowStatus[ihRow]= DBROWSTATUS_E_INVALID;

			goto RowError;
		}

        // Get RowBuffer to look at which row this applies to
        pbProvRow = (BYTE*)(m_pObj->GetRowBuff((ULONG) rghRows[ihRow] ));

        // Has row already been deleted
        //  S_OK means deleted
        if (S_OK == m_pObj->IsDeleted((ULONG) ((PROWBUFF) pbProvRow)->dwBmkStatus ))
        {
			if (rgRowStatus)
				rgRowStatus[ihRow] = DBROWSTATUS_E_DELETED;

			goto RowError;
        }

		/* //Move to row that will be deleted **** Moved inside ...::DeleteRow
		hr = m_pObj->m_pData->MoveBookmark( PROWBUFF(pbProvRow)->dwBmks );
		if (hr != S_OK)
        {
			// Some better decision as to what rowstatus to set could be done here..
			if (rgRowStatus)
				rgRowStatus[ihRow] = DBROWSTATUS_E_PERMISSIONDENIED;
			
			goto RowError;
        }*/


		// Delete the Row
		hr = m_pObj->m_pData->DeleteRow( PROWBUFF(pbProvRow)->dwBmks );
        if (hr != S_OK)
        {
			// Some better decision as to what rowstatus to set could be done here..
			if (rgRowStatus)
				rgRowStatus[ihRow] = DBROWSTATUS_E_PERMISSIONDENIED;
			
			goto RowError;
        }

		m_pObj->MarkAsDeleted(&PROWBUFF(pbProvRow)->dwBmkStatus);

		if (bSingleGranularity)
			m_pObj->OnRowChange(1, &rghRows[ihRow], DBREASON_ROW_DELETE, DBEVENTPHASE_DIDEVENT, TRUE);			

		ihRow++;
		continue; //Proceed to the next row

RowError:
		if (bSingleGranularity)
			m_pObj->OnRowChange(1, &rghRows[ihRow], DBREASON_ROW_DELETE, DBEVENTPHASE_FAILEDTODO, TRUE);			

        cErrors++;
        ihRow++;
	} //while


	// If everything went OK except errors in rows use DB_S_ERRORSOCCURRED.
	hr = cErrors ? (cErrors < cRows) ? 
			DB_S_ERRORSOCCURRED : 
		 	DB_E_ERRORSOCCURRED : 
		 	S_OK;

	if (!bSingleGranularity)
	{
		m_pObj->OnRowChange(cRows,
							rghRows,
							DBREASON_ROW_DELETE,
							hr != S_OK ? DBEVENTPHASE_FAILEDTODO : DBEVENTPHASE_DIDEVENT,
							TRUE);			
	}

	return hr;

	INTERFACE_METHOD_END();
}


// CImpIRowsetChange::InsertRow  --------------------------------------------
//
// @mfunc Insert row into provider
//
// @rdesc HRESULT indicating the status of the method
//      @Flag S_OK                     | success
//      @Flag DB_S_ERRORSOCCURRED      | Some, but not all, rows are inserted successfully 
//      @Flag E_INVALIDARG             | Arguments did not match spec.
//      @Flag E_OUTOFMEMORY            | Could not allocated error array
//      @Flag E_UNEXPECTED	           | object has a zombi status
//      @Flag DB_E_ABORTLIMITREACHED   | server resources limites exceeded
//      @Flag DB_E_BADACCESSORHANDLE   | bad accessor
//      @Flag DB_E_BADACCESSORTYPE     | given accessor is a reference acessor or not an acessor at all
//      @Flag DB_E_ERRORSOCCURRED	   | 

//      @Flag DB_E_MAXPENDCHANGESEXCEEDED   | 
//      @Flag DB_E_NOTREENTRANT		   | 
//      @Flag DB_E_NOTSUPPORTED		   | 
//      @Flag DB_E_ROWLIMITEXCEEDED	   | 
//      @Flag DB_E_ROWSNOTRELEASED	   | 
//      @Flag DB_SEC_E_PERMISSIONDENIED		| 
//
STDMETHODIMP    CImpIRowsetChange::InsertRow
    (
	    HCHAPTER	hReserved,	//@parm IN	| The chapter handle. 
		HACCESSOR	hAccessor,	//@parm IN	| The handle of the accessor to use.
		void*		pData,		//@parm IN	| A pointer to memory containing the new data values.
		HROW*		phRow		//@parm OUT	| A pointer to memory in which to return the handle of the new row. 
	)
{
	INTERFACE_METHOD_START("IRowsetChange::InsertRow");

	HRESULT		hr = S_OK;

	PACCESSOR	pAccessor;
    ULONG		cBindings;
    DBBINDING*	pBinding;
	int			iSlot = -1;

	//Check reentrant
	if (m_pObj->m_bNotReentrant)
		return DB_E_NOTREENTRANT;

	// Check for valid hAccessor
	if (m_pObj->m_pExtBufferAccessor == NULL ||    
        FAILED(m_pObj->m_pExtBufferAccessor->GetItemOfExtBuffer((ULONG)hAccessor, &pAccessor)) ||
        pAccessor == NULL)
    {
        return DB_E_BADACCESSORHANDLE;
    }
	
    cBindings = pAccessor->cBindings;
    pBinding  = pAccessor->rgBindings;

    // Ensure a source of data. Note that null accessor is valid one
    if (pData == NULL && !pAccessor->bNullAccessor )
        return E_INVALIDARG;

	// we do not support default values
	//if (cBindings == 0)
    //    return DB_E_BADACCESSORTYPE;

	if (m_pObj->IsSlotLimitReached())
		return E_FAIL;

	// store current position
	ULONG ulBookmarks[MAXBMKS];
	hr = m_pObj->m_pData->GetBookmark(ulBookmarks);
 	bool bBookmarkLoaded = (hr == S_OK);

	// Find next free slot
	iSlot = m_pObj->GetFreeSlot();
	assert (iSlot != 0xFFFFFFFF);
	m_pObj->m_cRows++;
	m_pObj->m_ulRowRefCount++;

	PROWBUFF pRow = m_pObj->GetRowBuff(m_pObj->m_SlotRef[iSlot].hRow);

	//Fill out row with new data
	hr = m_pObj->ApplyAccessor(hAccessor, (BYTE*)pRow, pData);
	if (FAILED(hr))
		goto CImpIRowsetChange_InsertRow_Exit;

	pRow->ulRefCount = 1;
	pRow->cbBmk = BMKS_SIZE * m_pObj->m_pData->bookmarks();
	pRow->dwBmkStatus = BMKS_ROW_IN_BUFFER;

	//Send notifications
	hr = m_pObj->OnRowChange(1, &m_pObj->m_SlotRef[iSlot].hRow, DBREASON_ROW_INSERT, DBEVENTPHASE_OKTODO, FALSE);			
	if (hr != S_OK)
		return hr;

	hr = m_pObj->OnRowChange(1, &m_pObj->m_SlotRef[iSlot].hRow, DBREASON_ROW_INSERT, DBEVENTPHASE_ABOUTTODO, FALSE);			
	if (hr != S_OK)
		return hr;

	// --------------------------------------------
	// Insert row
	hr = m_pObj->m_pData->MoveLast();
	if( FAILED(hr) )//(hr != S_OK)
		goto CImpIRowsetChange_InsertRow_Exit;

	hr = m_pObj->m_pData->InsertRow(m_pObj->m_rgdwDataOffsets, (BYTE*)pRow, pAccessor );
	if( FAILED(hr) )// (hr != S_OK)
		goto CImpIRowsetChange_InsertRow_Exit;

	//Load bookmark for new row
	hr = m_pObj->m_pData->GetBookmark( pRow->dwBmks );
	if (FAILED(hr))
		goto CImpIRowsetChange_InsertRow_Exit;
    
	// if we have HROW*	phRow parameter != NULL, then return hRow
	if (phRow != NULL)
		*phRow = m_pObj->m_SlotRef[iSlot].hRow;


// Exit from function
CImpIRowsetChange_InsertRow_Exit:

	// restore current  position
	if (bBookmarkLoaded)
		hr = m_pObj->m_pData->MoveBookmark(ulBookmarks);

	m_pObj->OnRowChange(1, 
						&m_pObj->m_SlotRef[iSlot].hRow,
						DBREASON_ROW_INSERT,
						(hr == S_OK) ? DBEVENTPHASE_DIDEVENT :DBEVENTPHASE_FAILEDTODO, 
						TRUE);			

	//Release row if loaded ///     Commented against fixing vbAdoIns hRow bug
	/*if (iSlot >= 0)
	{
		m_pObj->ReleaseSlot(iSlot);
		--m_pObj->m_cRows;
	}*/

	//Release row if its row handle is not required and creation succeeded
	if (iSlot >= 0 && phRow == NULL)
	{
		m_pObj->ReleaseSlot(iSlot);
		--m_pObj->m_cRows;
	}

	return hr;

	INTERFACE_METHOD_END();	
}
