/*
** Copyright (c) 1998 by Timothy Gerard Endres
** <mailto:time@ice.com>  <http://www.ice.com>
** 
** This program is free software.
** 
** You may redistribute it and/or modify it under the terms of the GNU
** General Public License as published by the Free Software Foundation.
** Version 2 of the license should be included with this distribution in
** the file LICENSE, as well as License.html. If the license is not
** included	with this distribution, you may find a copy at the FSF web
** site at 'www.gnu.org' or 'www.fsf.org', or you may write to the
** Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139 USA.
**
** THIS SOFTWARE IS PROVIDED AS-IS WITHOUT WARRANTY OF ANY KIND,
** NOT EVEN THE IMPLIED WARRANTY OF MERCHANTABILITY. THE AUTHOR
** OF THIS SOFTWARE, ASSUMES _NO_ RESPONSIBILITY FOR ANY
** CONSEQUENCE RESULTING FROM THE USE, MODIFICATION, OR
** REDISTRIBUTION OF THIS SOFTWARE. 
** 
*/

/**
 * An adaptor, transforming the JDBC interface
 * to the TableModel interface.
 *
 */

package com.ice.sqlclient;

import java.sql.*;
import java.math.*;
import java.util.Vector;
import com.sun.java.swing.event.TableModelEvent; 
import com.sun.java.swing.table.AbstractTableModel; 


public class
JDBCAdapter extends AbstractTableModel
	{
    private Vector			rows = new Vector();
    private ColumnMeta[]	columnMetas = new ColumnMeta[0]; 


	public
	JDBCAdapter( ResultSet resultSet )
		{
		this.setQueryResult( resultSet );
		}

	public void
	setQueryResult( ResultSet resultSet )
		{
		ResultSetMetaData	metaData;

		if ( resultSet == null )
			return;

		try {
			metaData = resultSet.getMetaData();

			int numberOfColumns =
				metaData.getColumnCount();

			// Cache the column names... 
			this.columnMetas =
				new ColumnMeta[ numberOfColumns ];

			for ( int colIdx = 0
					; colIdx < numberOfColumns
						; colIdx++ )
				{
				ColumnMeta column =
					this.new ColumnMeta( colIdx + 1 );

				column.cacheMetaData( metaData );

				this.columnMetas[ colIdx ] = column;
				}
			}
		catch ( SQLException ex )
			{
			// UNDONE - make these work like the ErrMsgAdapter!!!!!
			System.err.println
				( "ERROR getting META DATA in MySQLAdapter!!!!" );
			System.err.println
				( "\t" + ex.getMessage() );
			return;
			}		

		int idx = 1;

		// Get all rows.
		for ( this.rows = new Vector() ; ; )
			{
			boolean haveNext = false;
			try { haveNext = resultSet.next(); }
			catch ( SQLException ex )
				{
				// UNDONE - make these work like the ErrMsgAdapter!!!!!
				System.err.println
					( "ERROR getting ROW DATA in JDBCAdapter!!!!" );
				System.err.println
					( "\tROW = " + (this.rows.size() + 1) );
				System.err.println
					( "\t" + ex.getMessage() );
				break;
				}

			if ( ! haveNext )
				break;

			Vector newRow = new Vector();

			for ( idx = 1
					; idx <= this.getColumnCount()
						; idx++ )
				{
				Object elem;

				try {
					elem = resultSet.getObject( idx );
					if ( elem instanceof byte[] )
						{
						byte[] belem = (byte[]) elem;
						elem = "byte[" + belem.length + "]";
						}
					}
				catch ( SQLException ex )
					{
					elem = new String( "" );

					System.err.println
						( "JDBCAdapter: ERROR getting column data: "
							+ " Row = " + (this.rows.size() + 1)
							+ " Col = " + idx
							+ " Type = "
							+ this.columnMetas[idx-1].dataType );

					System.err.println
						( "\t" + ex.getMessage() );
					}

				newRow.addElement( elem );
				}

			this.rows.addElement( newRow );
			}

		this.fireTableChanged( null );
		}

    ////////////////////////////////////////////////////////////////////
    //
    //         Implementation of the TableModel Interface
    //
    ////////////////////////////////////////////////////////////////////

	//
    // M E T A    D A T A
	//

    public String
	getColumnName( int colIdx )
		{
		if ( this.columnMetas == null
				|| colIdx < 0
				|| colIdx > this.columnMetas.length )
			return "";

		ColumnMeta column = this.columnMetas[ colIdx ];
		return column.getName();
		}

    public Class
	getColumnClass( int colIdx )
		{
        int		type;

		if ( this.columnMetas == null
				|| colIdx < 0
				|| colIdx > this.columnMetas.length )
			return Object.class;

		ColumnMeta column = this.columnMetas[ colIdx  ];

		return
			this.getJDBCTypeClass
				( column.getDataType() );
		}

    private Class
	getJDBCTypeClass( int type )
		{
		switch ( type )
			{
			case Types.CHAR:
			case Types.VARCHAR:
			case Types.LONGVARCHAR:
				return String.class;

			case Types.BINARY:
			case Types.VARBINARY:
			case Types.LONGVARBINARY:
				return String.class;

			case Types.BIT:
				return Boolean.class;

			case Types.TINYINT:
			case Types.SMALLINT:
			case Types.INTEGER:
				return Integer.class;

			case Types.BIGINT:
				return Long.class;

			case Types.DECIMAL:
			case Types.NUMERIC:
				return BigDecimal.class;

			case Types.REAL:
			case Types.FLOAT:
				return Float.class;

			case Types.DOUBLE:
				return Double.class;

			case Types.DATE:
				return java.sql.Date.class;

			case Types.TIME:
				return java.sql.Time.class;

			case Types.TIMESTAMP:
				return java.sql.Timestamp.class;

			default:
				return Object.class;
			}
		}

    public boolean
	isCellEditable( int row, int colIdx )
		{
		if ( this.columnMetas == null
				|| colIdx < 0
				|| colIdx > this.columnMetas.length )
			{
			return false;
			}

		ColumnMeta column = this.columnMetas[ colIdx ];

		return column.isEditable();
		}

    public int
	getColumnCount()
		{
        return this.columnMetas.length; 
		}

	//
    // D A T A    M E T H O D S
	//

    public int
	getRowCount()
		{
		return this.rows.size();
		}

    public Object
	getValueAt( int aRow, int aColumn )
		{
		if ( aRow > this.rows.size() || aRow < 0 )
			return null;

		if ( aColumn > this.columnMetas.length || aColumn < 0 )
			return null;

		Vector row = (Vector) rows.elementAt( aRow );

		return row.elementAt( aColumn );
		}

    private String
	dbRepresentation( ColumnMeta column, Object value )
		{
		int type = column.getDataType();

		System.err.println
			( "DBREP: Col '" +column.getName()+ "' Type=" + type );
		
		if ( value == null )
            return "(null)"; 

		switch ( type )
			{
			case Types.INTEGER:
			case Types.DOUBLE:
			case Types.FLOAT:
				return value.toString();

			case Types.BIT:
				return
					((Boolean)value).booleanValue()
						? "1" : "0";

			case Types.DATE:
				 // This will need some conversion.
				return value.toString();

			default:
				System.err.println
					( "DBREP: DEFAULT: Type=" + type );
				return "\""+value.toString()+"\"";
			}
		}

    public void
	setValueAt( Object value, int row, int colIdx )
		{
		if ( row < 0 || row > this.rows.size() )
			return;

 		if ( colIdx < 0 || colIdx > this.columnMetas.length )
			return;

		if ( false ) // UNDONE - do we want to do this?
			{
   //     try {
			ColumnMeta column =
				this.columnMetas[ colIdx ];

            String columnName = column.getName();
			String tableName = column.getTableName();

			StringBuffer qBuf = new StringBuffer();
            qBuf.append( "update " );
			qBuf.append( tableName );
            qBuf.append( " set " );
			qBuf.append( columnName );
			qBuf.append( " = " );
			qBuf.append( this.dbRepresentation( column, value ) );
			qBuf.append( " where " );

            // We don't have a model of the schema so we don't know the 
            // primary keys or which columns to lock on. To demonstrate 
            // that editing is possible, we'll just lock on everything.
			boolean addAnd = false;
			int count = this.getColumnCount();
            for ( int col = 0 ; col < count ; col++ )
				{ 
				ColumnMeta forcol =
					this.columnMetas[ col ];

                String colName = forcol.getName();

                if ( colName.equals("") )
                    continue; 

                if ( addAnd )
                    qBuf.append( " and " ); 

				addAnd = true;

                qBuf.append( colName );
				qBuf.append( " = " );
				qBuf.append(
					this.dbRepresentation
						( column, this.getValueAt(row, col) ) ); 
				}

            System.out.println( qBuf.toString() ); 
            System.out.println
				("Not sending update to database! NO update.");

            // statement.executeQuery(query);
	/* UNDONE - bring back when update is executed
			}
        catch ( SQLException ex )
			{ 
            //     e.printStackTrace(); 
            System.err.println
				( "Update failed: " + ex.getMessage() ); 
			}
	*/
			}

        Vector dataRow = (Vector) rows.elementAt( row );

        dataRow.setElementAt( value, colIdx );
		}

	/**
	 * The ColumnMeta inner class is used to represent
	 * each column and its meta data, such as name and
	 * data type.
	 */

	class ColumnMeta extends Object
		{
		int		index;
		String	name;
		String	label;
		String	tableName;
		int		dataType;
		boolean	isEditable;

		public
		ColumnMeta( int index )
			{
			this.index = index;
			}

		public void
		cacheMetaData( ResultSetMetaData metaData )
			throws SQLException
			{
			this.name = metaData.getColumnName( this.index );
			this.label = metaData.getColumnLabel( this.index );
			this.dataType = metaData.getColumnType( this.index );
            this.tableName = metaData.getTableName( this.index ); 
			this.isEditable = metaData.isWritable( this.index );
if ( false )
System.err.println
( "ColumnMeta.cacheMetaData[" + this.index + "]="
	+ this.name +","+ this.dataType +","+ this.tableName);

			// HACK to fixup for buggy drivers...
            if ( this.tableName == null )
                this.tableName = "(null)"; 
			}

		public int
		getDataType()
			{ return this.dataType; }

		public void
		setDataType( int type )
			{ this.dataType = type; }

		public String
		getName()
			{ return this.name; }

		public void
		setName( String name )
			{ this.name = name; }

		public String
		getTableName()
			{
			return this.tableName;
			}

		public boolean
		isEditable()
			{ return this.isEditable; }
		}

	}
