;
; UDMA2.ASM    Written 18-Dec-2004 by Jack R. Ellis
;
; UDMA2 is free software.   You can redistribute and/or modify it under
; the terms of the GNU General Public License (hereafter called GPL) as
; published by the Free Software Foundation, either version 2 of GPL or
; any later versions at your option.   UDMA2 is distributed in the hope
; that it will be useful, but WITHOUT ANY WARRANTY and without even the
; implied warranties of MERCHANTABILITY nor of FITNESS FOR A PARTICULAR
; PURPOSE!  See the GPL for details.  You ought to have received a copy
; of the GPL with your UDMA2 files.  If not, write to the Free Software
; Foundation Inc., 59 Temple Place Ste. 330, Boston, MA 02111-1307 USA.
; http://www.gnu.org/licenses/
;
; Special thanks to Luchezar I. Georgiev for his INESTIMABLE advice and
; help in research, revising, enhancing, and test of the original UDMA!
;
; This is a DOS driver for UltraDMA hard-disks on PC motherboards using
; a VIA VT8235 or similar chipset.   On loading, the driver finds which
; IDE drives are UltraDMA hard-disks and will run all such disks.   Any
; UltraDMA disks may be used.   LBA and CHS mode requests are accepted.
; "Read/Write Extended" DMA commands are issued for disk addresses over
; 28 bits and require ATA-6 or newer hard-disks.   Disk addresses of 28
; bits or less use normal DMA commands.   The driver handles BIOS Int13
; read or write requests only.   Other Int13 requests (including seeks)
; and I-O requests having bad parameters are "passed" back to the BIOS.
; If an input buffer is not 32-bit aligned, crosses a 64K DMA boundary,
; or fails VDS "lock", data goes through a 64K XMS memory buffer, using
; UltraDMA to or from the buffer.    UDMA2 offers output overlap, which
; buffers ALL output and does NOT wait for output end, so user work may
; "overlap" the DMA!   If a 400-msec timer expires, or if output errors
; occur, an error message is displayed.   Overlap is initially DISABLED
; and must be enabled with the /O switch (see below).    Output overlap
; requires that each IDE channel may NOT be set up with an UltraDMA and
; a non-UltraDMA drive!   Other drivers are unaware UDMA2 leaves output
; "running", and such drivers on the same IDE channel as UDMA2 may post
; "controller-busy" ERRORS!
;
; UDMA2 switch options, specified in the CONFIG.SYS line that loads the
; driver, are as follows:
;
;   /O   Enables output overlap.  REQUIRES an XMS manager (HIMEM.SYS or
;          equivalent) and 128K of XMS memory.  If XMS is not available
;          or if /B is also specified, output overlap will NOT be used,
;          and UDMA2 (V1.5 and up) will use "DMA only" mode, like UDMA.
;
;   /B   Causes XMS memory to be IGNORED, for "backward compatibility".
;          This switch overrides /O and so DISABLES output overlap too.
;
;   /Mn  Specifies the MAXIMUM UltraDMA "mode" to be used by all disks,
;          where  n  is a number between 0 and 7, as follows: 
;              0 = ATA-16, 16 MB/sec.     4 = ATA-66,   66 MB/sec.
;              1 = ATA-25, 25 MB/sec.     5 = ATA-100, 100 MB/sec.
;              2 = ATA-33, 33 MB/sec.     6 = ATA-133, 133 MB/sec.
;              3 = ATA-44, 44 MB/sec.     7 = ATA-166, 166 MB/sec.
;          Disks designed to a "mode" LESS than the given value will be
;          limited to their own highest "mode".  This switch IS NOT for
;          regular use and is meant for "problem" systems only!
;         
;   /X   Disables initialization "read tests".   This switch IS NOT for
;          regular use and is a "last chance" way of making UDMA2 load,
;          for "problem" systems only!  Any error detected by the "read
;          tests" should be CORRECTED, wherever possible!
;
; For each switch, a dash may replace the slash, and lower-case letters
; (o, b, m, x) may also be used.
;
; UDMA2 offers the same I-O return codes as UDMA, to help in diagnosing
; problem systems and chipsets.   On exit from successful I-O requests,
; the AH-register is zero, and the carry flag is reset.   When an error
; occurs, carry is SET, and the AH-reg. has one of the following codes:
;
;    Code 08h - DMA timed out
;	  0Fh - DMA error
;	  20h - Controller busy before I-O
;	  21h - Controller busy after I-O
;	  80h - First DRQ timed out
;	  AAh - Disk not ready before I-O
;	  ABh - Disk not ready after I-O
;	  CCh - Write FAULT before I-O
;	  CDh - Write FAULT after I-O
;	  E0h - Hard error at I-O end
;	  FEh - BIOS/driver read MISMATCH (init only)
;	  FFh - XMS memory error
;
; When output overlap is enabled, run-time output errors shall cause the
; following message to display:
;
;    UDMA2:  Output ERROR eeh Disk=d LBA=aaaaaaaaaaaah!
;
; where eeh is one of the return codes shown above, d is the disk number
; (0 if primary-master, 1 if primary-slave, 2 if secondary-master, and 3
; if secondary-slave), and aaaaaaaaaaaah shows the starting disk address
; expressed as a 12-digit "logical block address" (LBA).   Because UDMA2
; is a physical-level driver called by Int 13h (not a DOS "file driver")
; the DOS "file" or "directory" for an output error CANNOT be displayed!
;
;
; Revision History:
; ----------------
;  V1.9  18-Dec-04   JE   Fixed DRQ timeout code
;  V1.8  11-Dec-04   JE   Fixed FOOLISH disk-select bug (My apologies!)
;  V1.7  10-Dec-04   JE   No EDD BIOS error abort, more code reductions
;  V1.6   2-Dec-04   JE   Reduced code size, same as in V1.6 UDMA2S
;  V1.5  24-Nov-04   JE   Re-added "DMA only", added /B /M /X switches
;  V1.4  18-Nov-04   JE   Pooling OUT; best in a "cache", NOT a driver!
;  V1.3  16-Nov-04   JE   Output errors say "UDMA", other minor changes
;  V1.2  12-Nov-04   JE   Minor corrections to XMS init logic
;  V1.1   8-Nov-04   JE   Corrected initialization VDS "lock" error
;  V1.0	  6-Nov-04   JE   Original issue of "pooling" & "overlap" UDMA2
;
;
; General Program Equations.
;
%define VER 'V1.9, 18-Dec-2004.'
RDYTO	equ	008h		;384-msec minimum I-O timeout.
RBYTES	equ	(2*1024*65536*12/14318180+1)*512 ;Read-test byte count.
	;( = microseconds per tick, but adjusted for binary megabytes).
	;Number of reads this length per tick = transfer rate in MB/s.
RS_SC	equ	RBYTES/512	;"Read speed" input sector count.
RC_SC	equ	58		;"Read compare" total sector count.
RC_CYL	equ	3		;"Read compare" starting disk address.
RC_HD	equ	15
RC_SEC	equ	5
STACK	equ	332		;Stack size for the overlap driver.
BSTACK	equ	288		;Stack size for the "basic" driver.
BIOSTMR equ	0046Ch		;BIOS "tick" timer address.
HDISKS	equ	00475h		;BIOS hard-disk count address.
VDSFLAG equ	0047Bh		;BIOS "Virtual DMA" flag address.
CR	equ	00Dh		;ASCII carriage-return.
LF	equ	00Ah		;ASCII line-feed.
;
; Driver Return Codes.
;
DMATIMO equ	0E8h		;DMA timeout code, 008h at exit.
DMAERR	equ	0EFh		;DMA error   code, 00Fh at exit.
CTLRERR equ	000h		;Ctlr. busy  code, 020h/021h at exit.
DRQTIMO equ	080h		;DRQ timeout code.
DISKERR equ	08Ah		;Disk-busy   code, 0AAh/0ABh at exit.
WFLTERR equ	0ACh		;Write-fault code, 0CCh/0CDh at exit.
HARDERR equ	0BFh		;Hard-error  code, 0E0H at exit.
XMSERR	equ	0FFh		;XMS memory-error code.
;
; IDE Controller Register Definitions.
;
CDATA	equ	001F0h		;Data port.
CSUBCM	equ	CDATA+1		;Subcommand register.
CSECCT	equ	CDATA+2		;I-O sector count.
CDSEL	equ	CDATA+6		;Disk-select and upper LBA.
CCMD	equ	CDATA+7		;Command register.
CSTAT	equ	CDATA+7		;Primary status register.
CSTAT2	equ	CDATA+206h	;Alternate status register.
;
; Controller Status and Command Definitions.
;
BSY	equ	080h		;IDE controller is busy.
RDY	equ	040h		;IDE disk is "ready".
FLT	equ	020h		;IDE disk has a "fault".
DRQ	equ	008h		;IDE data request.
ERR	equ	001h		;IDE general error flag.
DMI	equ	004h		;DMA interrupt occured.
DME	equ	002h		;DMA error occurred.
DRCMD	equ	0C8h		;DMA read command (write is 0CAh,
				;    LBA48 commands are 025h/035h).
SETM	equ	003h		;Set Mode subcommand.
SETF	equ	0EFh		;Set Features command.
LBABITS equ	0E0h		;Fixed LBA command bits.
;
; LBA "Device Address Packet" Layout.
;
struc	DAP
DapPL	resb	1		;Packet length.
	resb	1		;(Reserved).
DapSC	resb	1		;I-O sector count.
	resb	1		;(Reserved).
DapBuf	resd	1		;I-O buffer address (segment:offset).
DapLBA	resw	1		;48-bit logical block address (LBA).
DapLBA1	resw	1
DapLBA2	resw	1
endstruc
;
; DOS "Request Packet" Layout.
;
struc	RP
	resb	2		;(Unused by us).
RPOp	resb	1		;Opcode.
RPStat	resw	1		;Status word.
	resb	9		;(Unused by us).
RPSize	resd	1		;Resident driver size.
RPCmdL	resd	1		;Command-line data pointer.
endstruc
RPERR	equ	08003h		;Packet "error" flags.
RPDON	equ	00100h		;Packet "done" flag.
;
; DOS Driver Device Header.
;
@	dd	0FFFFFFFFh	;Link to next header block.
	dw	08000h		;Driver "device attributes".
	dw	Strat		;"Strategy" routine offset.
	dw	DevInt		;"Device-Interrupt" routine offset.
CStack	equ	$-4		;(Caller's saved stack ptr. after Init).
	db	'UDMA2$',0,0	;Driver name (NO "arrowhead" for V1.5+).
;
; Resident Driver "Data Page" Variables.
;
VLF	db	0		;VDS lock flag (1 = user buffer locked).
Flags	db	0		;Driver control flags:
				;  080h = Driver or timer rtn. busy.
				;  001h = Overlapped output running.
Timer	db	0		;Overlapped output 400-msec timer.
IDEAd	db	0		;Current low-order IDE status address.
DMAAd	dw	0		;Current DMA command-register address.
Units	dd	0000000FFh	;IDE "active units" table, set by Init:
	dd	0000000FFh	;  Byte 0:    BIOS unit (0FFh inactive).
	dd	0000000FFh	;  Byte 1:    CHS sectors per head.
	dd	0000000FFh	;  Byte 2-3:  CHS sectors/cylinder.
PRDAd	dd	IOAdr-@		;PRD 32-bit command addr. (Init set).
	db	0		;IDE "upper" sector count (always 0).
LBA2	db	0FAh,0F0h,08Ah	;IDE "upper" LBA bits 24-47.
SecCt	db	080h		;IDE "lower" sector count.
LBA	db	0,0,0		;IDE "lower" LBA bits 0-23.
DSCmd	db	0		;IDE disk-select and LBA commands.
IOCmd	db	0		;IDE command byte.
XMHdl	dw	0		;XMS buffer handle number (Init set).
XMOfs	dd	0		;XMS 32-bit buffer offset (Init set).
VDSLn	dd	0		;VDS and XMS buffer length.
VDSOf	dd	0		;VDS 32-bit offset.
VDSSg	dd	0		;VDS 16-bit segment (hi-order zero).
IOAdr	dd	0		;VDS and DMA 32-bit address.
IOLen	dd	080000000h	;DMA byte count and "end" flag.
XMSSH	equ	VDSOf		;(XMS parameters share the VDS block).
XMSSA	equ	VDSOf+2
XMSDH	equ	VDSSg+2
;
; Driver Entry Routine.  For CHS requests, the registers contain:
;
;   AH      Request code.  We handle 002h read and 003h write.
;   AL      I-O sector count.
;   CH      Lower 8 bits of starting cylinder.
;   CL      Starting sector and upper 2 bits of cylinder.
;   DH      Starting head.
;   DL      BIOS unit number.  We handle 080h and up (hard-disks).
;   ES:BX   I-O buffer address.
;
; For LBA requests, the registers contain:
;
;   AH      Request code.  We handle 042h read and 043h write.
;   DL      BIOS unit number.  We handle 080h and up (hard-disks).
;   DS:SI   Pointer to Device Address Packet ("DAP"), described above.
;
Entry	push	bp		;Driver entry routine -- save BP-reg.
	mov	bp,16		;Reset active-units table index.
@LastU	equ	$-2		;(Last-unit index, set by Init.  MUST
				;  be 16 while Init TESTS all disks!). 
NextU	sub	bp,byte 4	;Any more active units to check?
	js	QuickX		;No, request NOT for us -- exit quick!
	cmp	dl,[cs:bp+Units-@] ;Does request unit match our table?
	jne	NextU		;No, see if more table entries remain.
	cli			;It's for us!  Disable CPU interrupts.
	cld			;Ensure FORWARD "string" commands!
	push	dx		;Save DX- & AX-regs. for tests below.
	push	ax
	mov	dl,0BEh		;Get I-O mask (used in "busy" test!).
	test	[cs:Flags],dl	;Handling another request or timer?
	js	InvalF		;Yes?  Set "invalid function" & exit!
	and	dl,ah		;Mask out LBA and write request bits.
	cmp	dl,002h		;Is this a CHS or LBA read or write?
	jne	Pass		;No, let BIOS handle this request.
	shl	ah,1		;Is this an LBA read or write request?
	jns	ValSC		;No, go validate CHS sector count.
	mov	al,[si+DapSC]	;Get "DAP" I-O sector count.
	cmp	dword [si+DapBuf],byte -1 ;64-bit "DAP" buffer address?
	jne	ValSC		;No, go validate "DAP" sector count.
Pass	pop	ax		;Pass request -- reload AX- & DX-regs.
	pop	dx
@NoOp1	pushf			;"Fake" a timer interrupt & let timer
	push	cs		;  routine dismiss any pending output!
	call	TmPend
QuickX	mov	bp,sp		;Reload CPU flags saved by Int 13h.
	push	word [bp+6]
	popf
	pop	bp		;Reload BP-register.
	jmp	0000:0000	;Go to next routine in Int 13h chain.
@PrvI13 equ	$-4		;(Previous INT13 vector, set by Init).
ValSC	dec	al		;Is sector count zero or over 128?
	js	Pass		;Yes?  Let BIOS handle this "No-No!".
	or	byte [cs:Flags],080h ;Valid request -- post "busy".
@NoOp2	mov	[cs:CStack],sp	;Save caller's stack pointer.
	mov	[cs:CStack+2],ss
	mov	sp,cs		;Switch to this driver's stack.
	mov	ss,sp
	mov	sp,ResEnd
@Stack	equ	$-2		;(Driver stack offset, set by Init).
	sti			;Re-enable CPU interrupts.
@NoOp3	call	OCheck		;Await end of any pending output.
	push	ds		;Save all remaining CPU registers.
	push	bx		;(DS- and BX-regs. first, for exit).
	push	cx
	push	si
	push	di
	push	es
	inc	ax		;Restore sector count -- LBA request?
	js	GetDAP		;Yes, get remaining "DAP" parameters.
	xchg	ax,cx		;CHS -- save request code and sectors.
	mov	si,0003Fh	;Set SI-reg. to starting sector.
	and	si,ax
	dec	si
	mov	di,dx		;Set DI-reg. to starting head.
	shr	al,6		;Set AX-reg. to cylinder number.
	xchg	al,ah
	mul	word [cs:bp+Units+2-@]  ;Convert cylinder to sectors.
	xchg	ax,di		     ;Swap low-order and head number.
	mov	al,[cs:bp+Units+1-@] ;Convert head to sectors.
	mul	ah
	add	si,ax		;Add to starting sector.
	add	si,di		;Add in cylinder sectors.
	adc	dl,dh
	xchg	ax,bx		;Get buffer offset in AX-register.
	xchg	ax,cx		;Swap offset with request/sectors.
	xor	di,di		;Reset upper LBA address bits.
	jmp	short SetDS	;Go set our DS-register.
InvalF	stc			;We're BUSY, you fool!  Post "invalid
	mov	al,001h		;  function" & error (most drivers do
	jmp	GetOut		;  NO "re-entry" test and can CRASH!).
	db	0		;(Unused alignment "filler").
GetDAP	les	cx,[si+DapBuf]	;Get "DAP" I-O buffer address.
	mov	di,[si+DapLBA2]	;Get "DAP" logical block address.
	mov	dx,[si+DapLBA1]
	mov	si,[si+DapLBA]
SetDS	push	cs		;Set our DS-register.
	pop	ds
	xor	bx,bx		;Zero BX-reg. for relative commands.
	mov	[bx+LBA-@],si	;Save 48-bit logical block address.
	mov	[bx+LBA+2-@],dl
	mov	[bx+LBA2-@],dh
	mov	[bx+LBA2+2-@],di
	shr	dx,12		;Shift out LBA bits 16-27.
	or	di,dx		;Anything in LBA bits 28-47?
	jz	LBA28		;No, use LBA28 read/write command.
	shl	ah,3		;LBA48 -- get command as 020h/030h.
	jmp	short GetAdr	;Go get IDE and LBA address bytes.
LBA28	xchg	dh,[bx+LBA2-@]	;LBA28 -- reload & reset bits 24-27.
	or	ah,(DRCMD+1)	;Get LBA28 read/write command + 5.
GetAdr	mov	di,CDSEL-00100h	;Get primary device-address bytes.
@DMALo1	equ	$-1		;(Lower DMA command address, Init set).
	shr	bp,3		;Is this a primary-channel request?
	jz	DevAdr		;Yes, set IDE & PCI address bytes.
	mov	di,CDSEL+00680h	;Get secondary device-address bytes.
@DMALo2	equ	$-1		;(Lower DMA command address, Init set).
DevAdr	mov	[bx+IDEAd-@],di ;Set current IDE & PCI address bytes.
	mov	dl,(LBABITS/32)	;Get disk-select & LBA command bits.
	rcl	dl,5
	or	dl,dh		;"Or" in LBA28 bits 24-27 (if any).
	mov	dh,005h		;Get final IDE command byte.
	xor	dh,ah		;(LBA28 = C8h/CAh, LBA48 = 25h/35h).
	mov	[bx+DSCmd-@],dx	;Set LBA and IDE command bytes.
	mov	[bx+SecCt-@],al	;Set I-O sector count.
	mov	ah,0		;Set VDS/XMS and DMA buffer lengths.
	shl	ax,1
	mov	[bx+VDSLn+1-@],ax
	mov	[bx+IOLen+1-@],ax
	mov	[bx+VDSOf-@],cx	;Set VDS offset and segment.
	mov	[bx+VDSOf+2-@],bx
	mov	[bx+VDSSg-@],es
@NoOp4	test	byte [bx+IOCmd-@],012h ;Is this a write request?
	jnz	GoToBuf		       ;Yes, do buffered output below.
	or	dword [bx+IOAdr-@],byte -1 ;Invalidate VDS address.
	mov	ax,08103h	       ;Do VDS "lock" of user buffer.
	mov	dx,0000Ch
	call	VDLock
	jc	GoToBuf		       ;VDS error -- do buffered I-O.
	cmp	dword [bx+IOAdr-@],byte -1 ;Got a valid VDS address?
	je	NoVDS		;No, set 20-bit user buffer address.
	inc	byte [bx+VLF-@]	;Set VDS "lock" flag.
	mov	ax,[bx+IOAdr-@]	;Get low-order VDS buffer address.
	jmp	short ChkOdd	;Go check for "odd" buffer alignment.
NoVDS	mov	ax,16		;No VDS -- get 20-bit buffer segment.
	mul	word [bx+VDSSg-@]
	add	ax,[bx+VDSOf-@]	;Add in buffer offset value.
	adc	dx,bx
	mov	[bx+IOAdr-@],ax	;Set 20-bit user buffer address.
	mov	[bx+IOAdr+2-@],dx
ChkOdd	test	al,003h		;Is user I-O buffer 32-bit aligned?
	jnz	NoLock		;No, use buffered I-O routines below.
	mov	cx,[bx+IOLen-@]	;Get lower ending DMA address.
	dec	cx		;(IOLen - 1 + IOAdr).
	add	ax,cx		;Would this I-O cross a 64K boundary?
	jc	NoLock		;Yes, use buffered I-O routines below.
	call	DoDMA		;Do direct DMA I-O with user's buffer.
Done	rcl	ah,1		;Done -- save error flag in AH-reg.
	push	ax		;Save error code (AL-reg.) and flag.
	call	VDUnlk		;If needed, "unlock" user I-O buffer.
	pop	ax		;Reload error code and flag.
@NoXMA	pop	es		;Reload registers unneeded for exit.
	pop	di
	pop	si
	pop	cx
	cli				;Disable CPU interrupts.
	and	byte [bx+Flags-@],07Fh	;Reset driver "busy" flag. 
	pop	bx			;Reload BX- and DS-registers.
	pop	ds
@NoXMB	lss	sp,[cs:CStack]	;Switch back to caller's stack.
	rcr	ah,1		;Put error flag back in carry bit.
GetOut	mov	bp,sp		;Point to caller's saved AX-reg.
	mov	[bp+1],al	;Set error code in exiting AH-reg.
	rcr	byte [bp+10],1	;Set error flag in exiting carry
	rol	byte [bp+10],1	;  bit, in flags saved by Int 13h.
	pop	ax		;Reload AX-, DX-, and BP-registers.
	pop	dx
	pop	bp
	iret			;Exit.
NoLock	call	VDUnlk		;Buffered I-O -- "unlock" user buffer.
GoToBuf	jmp	BufIO		;Go to buffered I-O routines below.
;
; Subroutine to execute read and write commands.
;
BufDMA	mov	dword [bx+IOAdr-@],0  ;Buffered -- set XMS buffer addr.
@XBufAd	equ	$-4		;(XMS 32-bit buffer address, Init set).
DoDMA	mov	dx,[bx+DMAAd-@]	;Ensure any previous DMA is stopped!
	in	al,dx		;(On some older chipsets, if DMA is
	and	al,0FEh		;  running, reading an IDE register
	out	dx,al		;  causes the chipset to "HANG"!!).
	mov	al,[bx+DSCmd-@]	;Select our desired disk.
	and	al,0F0h
	mov	dl,[bx+IDEAd-@]
	mov	dh,001h
	out	dx,al
	mov	di,dx		;Save IDE drive-select address.
	mov	es,bx		;Point to low-memory BIOS timer.
	mov	si,BIOSTMR
	mov	cx,(RDYTO*256)+FLT   ;Get timeout count & "fault" mask.
	add	ch,[es:si]	     ;Set CH-reg. with timeout limit.
	call	ChkRdy		     ;Await controller- and disk-ready.
	jc	DoDMAX		     ;If error, exit immediately!
	test	byte [bx+IOCmd-@],012h  ;Is this a write request?
	jnz	SetDMA		;Yes, reset DMA command register.
	mov	al,008h		;Get "DMA read" command bit.
SetDMA	mov	dx,[bx+DMAAd-@]	;Reset DMA commands and set DMA mode.
	out	dx,al
	inc	dx		;Point to DMA status register.
	inc	dx
	in	al,dx		;Reset DMA status register.
	or	al,006h		;(Done this way so we do NOT alter
	out	dx,al		;  the "DMA capable" status flags!).
	push	si		;Save BIOS timer pointer.
	inc	dx		;Set PRD pointer to our DMA address.
	inc	dx
	mov	si,(PRDAd-@)
	outsd
	mov	bx,001F7h	;Set IDE parameter-output flags.
NxtPar	lea	dx,[di+CSECCT-CDSEL-1] ;Point to IDE sector count -1.
IDEPar	inc	dx		;Output all ten LBA48 parameter bytes.
	outsb			;(1st 4 overlayed by 2nd 4 if LBA28!).
	shr	bx,1		;More parameters to go in this group?
	jc	IDEPar		;Yes, loop back and output next one.
	jnz	NxtPar		;If first 4 done, go output last 6.
	pop	si		;Reload BIOS timer pointer.
	mov	dh,003h		;Get IDE alternate-status address.
	dec	dx		;(Primary-status address | 300h - 1).
ChkDRQ	cmp	ch,[es:si]	;Too long without 1st data-request?
	je	ErrDRQ		;Yes?  Return carry and DRQ error!
	in	al,dx		;Read IDE alternate status.
	and	al,DRQ		;Has 1st data-request arrived?
	jz	ChkDRQ		;No, loop back and check again.
	mov	dx,[bx+DMAAd-@]	;Set DMA Start/Stop bit (starts DMA).
	in	al,dx
	inc	ax
	out	dx,al
	test	byte [bx+IOCmd-@],012h	;Read command, or NO overlap?
@OvlpF	equ	$-1			;(If 000h, overlap disabled).
	jz	ChkDMA			;If either, go await I-O end.
	inc	byte [bx+Flags-@]	;Post "overlapped output" flag.
	mov	byte [bx+Timer-@],RDYTO	;Post 400-msec output timeout.
	xor	ax,ax			;Return "success" & zero carry.
DoDMAX	ret				;Exit.
;
; Subroutine to await end of I-O.
;
ChkDMA	inc	dx		;Point to DMA status register.
	inc	dx
	in	al,dx		;Read DMA controller status.
	dec	dx		;Point back to DMA command register.
	dec	dx
	and	al,DMI+DME	;DMA interrupt or DMA error?
	jnz	HltDMA		;Yes, halt DMA and check results.
	cmp	ch,[es:si]	;Has our DMA transfer timed out?
	jne	ChkDMA		;No, loop back and check again.
HltDMA	push	ax		;Save ending DMA status.
	in	al,dx		;Reset DMA Start/Stop bit.
	and	al,0FEh
	out	dx,al
	pop	ax		;Reload ending DMA status.
	cmp	al,DMI		;Did DMA end with only an interrupt?
	jne	ErrDMA		;No?  Go see what went wrong.
	inc	dx		;Reread DMA controller status.
	inc	dx
	in	al,dx
	test	al,DME		;Any "late" DMA error after DMA end?
	jnz	DMAEnd		;Yes?  Return carry and DMA error!
	inc	cx		;Check "fault" and hard-error at end.
ChkRdy	lea	dx,[di+CSTAT-CDSEL] ;Read IDE primary status.
	in	al,dx
	test	al,BSY+RDY	;Controller or disk still busy?
	jg	ChkErr		;No, check for "fault" or hard-error.
	cmp	ch,[es:si]	;Too long without becoming ready?
	jne	ChkRdy		;No, loop back and check again.
	test	al,BSY		;BAAAD News!  Did controller go ready?
	mov	ax,(256*CTLRERR)+DISKERR ;(Get not-ready error codes).
	jmp	short WhichE	;Go see which error code to return.
ChkErr	and	al,cl		;Disk "fault" or hard-error?
	jz	ChkDMAX		;No, all is well -- go exit below.
	test	al,FLT		;BAAAD News!  Is the disk "faulted"?
	mov	ax,(256*WFLTERR)+HARDERR ;(Get hardware error codes).
WhichE	jz	EndErr		;If "zero", use AL-reg. return code.
	mov	al,ah		;Use AH-reg. return code of this pair.
EndErr	add	al,cl		;Add 1 if error occurred at I-O end.
	stc			;Set carry flag to denote "error"!
ChkDMAX	ret			;Exit.
ErrDRQ	mov	al,DRQTIMO	;BAAAD News!  Get DRQ error code.
	stc			;Set carry flag (error) and exit!
	ret
ErrDMA	test	al,DME		;BAAAD News!  Did DMA end with error?
DMAEnd	mov	ax,(256*DMAERR)+DMATIMO	 ;(Get DMA error codes).
	jmp	short WhichE	;Go see which error code to return.
;
; Subroutine to do a VDS "lock" or "unlock".
;
VDUnlk	shr	byte [bx+VLF-@],1 ;Was input buffer "locked" by VDS?
	jnc	VDExit		;No, just exit below.
	mov	ax,08104h	;Get VDS "unlock" parameters.
	xor	dx,dx
VDLock	mov	di,(VDSLn-@)	;Point to VDS parameter block.
VDInit	push	cs
	pop	es
	push	bx		;Save BX-reg. and execute VDS request.
	int	04Bh
VDDone	pop	bx		;Reload our BX-register.
	sti			;RESTORE all critical driver settings!
	cld			;(Never-NEVER "trust" external code!).
	push	cs
	pop	ds
VDExit	ret			;Exit.
;
; Buffered I-O routines, placed here so the latter part of this logic
;   which is NOT overlayed, and the "XMSMov" subroutine below, can be 
;   "dismissed" during driver-init when the non-XMS driver is used.
;
BufIO	shl	dword [bx+VDSOf-@],16  ;Convert to XMS handle & offset.
	test	byte [bx+IOCmd-@],012h ;Is this a write request?
	jnz	BufOut		;Yes, use output routine below.
	call	BufDMA		;Input all data to driver XMS buffer.
	jc	DoneJ		;If error, post return code & exit!
	call	XMSMov		;Move XMS data to user input buffer.
	jmp	short DoneJ	;Done -- post any return code & exit.
BufOut	call	XMSMov		;Output -- move data to XMS buffer.
	jc	DoneJ		;If error, post return code & exit!
	call	BufDMA		;Output all data from XMS buffer.
DoneJ	jmp	Done		;Done -- post any return code & exit.
NonXEnd	equ	BufIO+(@NoXMB+3-@NoXMA) ;End of resident non-XMS driver.
;
; Subroutine to move data to or from the driver's XMS buffer.
;
XMSMov	push	cs		;Point ES-reg. to our data.
	pop	es
	mov	di,(XMSDH-@)	;Point to XMS destination field.
	jnz	XMSet		;If output, just set XMS destination!
	mov	si,(XMSSH-@)	;Point to XMS user-buffer address.
	movsw			;Move user-buffer address from
	movsw			;  XMS source to XMS destination.
	movsw
	mov	di,(XMSSH-@)	;Point to XMS source field.
XMSet	mov	si,(XMHdl-@)	;Set XMS handle and buffer offset as
	movsw			;  input source or output destination.
	movsw
	movsw
XMGo	mov	ah,00Bh		;Get XMS "move data" request code.
XMCall	push	bx		;Save BX-reg. and execute XMS request.
	call	0000:0000	;(SI-reg. points to IOLen after move).
@XEntry equ	$-4		;(XMS "entry" address, set by init).
	dec	ax		;Return 0 if success, -1 if error.
	sar	ax,1		;If error, set carry flag on.
	jmp	short VDDone	;Go reload BX-reg. and exit above.
	db	0		;(Unused alignment "filler").
BaseEnd	equ	$+BSTACK	;End of resident "basic" driver.
;
; Timer-interrupt routine, called every 55-msec by IRQ8 timer "ticks"
;   and used to check pending output if our 400-msec timeout expires.
;
TmInt	pushf			;Entry -- save CPU flags and call
	db	09Ah		;  all other timer routines first!
@PrvTmr	dd	0		;(Prior timer vector, saved by Init).
	cli			;Disable CPU interrupts.
	dec	byte [cs:Timer]	;Decrement 400-msec output timer.
	jg	TmRet		;If it hasn't expired, exit fast!
	mov	byte [cs:Timer],0    ;If zero BEFORE, reset it!
TmPend	test	byte [cs:Flags],081h ;Driver busy, or NO output?
	jle	TmRet		     ;If either, exit fast!
	or	byte [cs:Flags],080h ;Post "driver busy" flag.
	mov	[cs:CStack],sp	;Save caller's stack pointer.
	mov	[cs:CStack+2],ss
	mov	sp,cs		;Switch to this driver's stack.
	mov	ss,sp
	mov	sp,ResEnd
	sti			;Re-enable CPU interrupts.
	call	OCheck		;Await end of all pending output.
	cli			;Disable CPU interrupts.
	lss	sp,[cs:CStack]	;Switch back to caller's stack.
	and	byte [cs:Flags],07Fh  ;Reset "driver busy" flag.
TmRet	iret			;Exit.
;
; Subroutine to check output and display any needed error message.
;
OCheck	pusha			;Save all CPU registers except ES-reg.
	push	ds		;(If needed, ES-reg. is saved below).
	push	cs		;Set our DS-register.
	pop	ds
	xor	bx,bx		;Zero BX-reg. for relative commands.
	btr	[bx+Flags-@],bx	;Is overlapped output running?
	jnc	OCExit		;No, just exit below.
	push	es		;Save ES-register now.
	mov	cx,256+FLT	;Set 1 extra count & "fault" mask.
	add	ch,[bx+Timer-@]	;Get ending output timer value +1.
	mov	es,bx		;Point to BIOS timer in low-memory.
	mov	si,BIOSTMR
	add	ch,[es:si]	;Set CH-reg. with timeout limit.
	mov	al,[bx+IDEAd-@]	;Set IDE status address in DI-reg.
	mov	ah,001h
	xchg	ax,di
	mov	dx,[bx+DMAAd-@]	;Get DMA command-register address.
	call	ChkDMA		;Await end of previous output.
	jnc	OCDone		;If no output errors, exit below.
	xchg	al,ah		;BAAAD News!  Set return code in msg.
	mov	cx,2
	mov	si,OECode
	call	HexCnv
	mov	al,[bx+DSCmd-@]	;Get disk-select from LBA command.
	shl	al,2
	xor	al,[bx+IDEAd-@]	;Get IDE channel (inverts select).
	shr	al,6		;Set error disk number, 0 thru 3.
	xor	al,031h		;(001h "re-inverts" disk select,
	mov	[OEDisk],al	;  030h converts number to ASCII).
	mov	si,OELBA	;Set LBA bits 32-47 in error message.
	mov	ax,[bx+LBA2+1-@]
	call	HexCnv4
	mov	ax,[bx+LBA+2-@]	;Get LBA bits 16-23 and LBA command.
	and	ah,00Fh		;Anything in "LBA28" bits 24-27?
	jnz	OCMsg1		;Yes, set bits 16-31 in message.
	mov	ah,[bx+LBA2-@]	;Get "LBA48" bits 24-31.
OCMsg1	call	HexCnv4		;Set LBA bits 16-31 in message.
	mov	ax,[bx+LBA-@]	;Set LBA bits  0-15 in message.
	call	HexCnv4
	mov	ah,00Fh		;Get BIOS video "page" in BH-reg.
	int	010h
	mov	si,OEMsg	;Point to output-error message.
	jmp	short OCMsg3	;Go start message output below.
OCMsg2	push	bx		;Have BIOS display next message byte.
	push	si		;(CANNOT call DOS in "BIOS" drivers!).
	mov	ah,00Eh
	int	010h
	pop	si
	pop	bx
OCMsg3	sti			;RESTORE all critical driver settings!
	cld			;(Never-NEVER "trust" external code!).
	push	cs
	pop	ds
	lodsb			;Get next error-message byte.
	or	al,al		;Are we at the terminating "null"?
	jnz	OCMsg2		;No, loop back & display next byte.
OCDone	pop	es		;Reload all CPU registers.
OCExit	pop	ds
	popa
	ret			;Exit.
;
; Subroutine to convert hex digits to ASCII for output messages.
;   At entry the AX-reg. has the data, the CX-reg. has the digit
;   count, and the SI-reg. has the message pointer.
;
HexCnv4	mov	cx,4		;Set normal 4-digit count.
HexCnv	rol	ax,4		;Get next hex digit in low-order.
	push	ax		;Save remaining digits.
	and	al,00Fh		;Mask off next hex digit.
	cmp	al,009h		;Is digit 0-9?
	jbe	HexCnvA		;Yes, convert to ASCII.
	add	al,007h		;Add A-F offset.
HexCnvA	add	al,030h		;Convert digit to ASCII.
	mov	[si],al		;Set next ASCII digit in message.
	inc	si		;Bump message address.
	pop	ax		;Reload remaining digits.
	loop	HexCnv		;If more digits to go, loop back.
	ret			;Exit.
;
; Output-Error Message (BAAAD News!).
;
OEMsg	db	CR,LF,'UDMA2:  Output ERROR '
OECode	db	'00h Disk='
OEDisk	db	'0 LBA='
OELBA	db	'000000000000h!',CR,LF,0
ResEnd	equ	$+STACK		;End of resident driver.
;
; Initialization Messages.   Those from "UDMsg" through "DspPF" display
;   once-only during initialization "start up".   Then, they become the
;   full size "overlap" driver's stack.   The rest are here in case the
;   BIOS gets our 57K diagnostic reads by MISTAKE and CLOBBERS all data
;   beyond "RBuff" (HAPPENED during debug)!   Placing the messages HERE
;   and "RBuff" and "start up" code LAST allows driver-init to SURVIVE!
;
UDMsg	db	CR,LF,'UDMA2 Disk Driver ',VER,'   '
	db	'Output Overlap is $'
DOMsg	db	'DISABLED!',CR,LF,'$'
EOMsg	db	'enabled.',CR,LF,'$'
NEMsg	db	'No controller found$'
MEMsg	db	'Bus-Master setup BAD$'
NXMsg	db	'No XMS manager$'
XEMsg	db	'XMS init error$'
XDMsg	db	'/B, XMS ignored$'
NBMsg	db	'; using only DMA I-O, NO Output Overlap!',CR,LF,'$'
VEMsg	db	'VDS lock failed$'
PCMsg	db	'UltraDMA controller found at I-O address '
DspAd	db	'0000h.',CR,LF,'    Vendor/Device I.D. '
DspVI	db	'0000h/'
DspDI	db	'0000h.   PCI bus '
DspPB	db	'00h, device '
DspPD	db	'00h, function '
DspPF	db	'0.',CR,LF,'$'
DNMsg	db	'is '
DName	equ	$		;All disk names overlay the next
DNEnd	equ	DName+40	;  two "start up" init messages.
PRMsg	db	CR,LF,'CPU is below an 80386$'
PEMsg	db	'PCI BIOS is not V2.0C+$'
NDMsg	db	'No disks to use$'
EBMsg	db	'EDD error!  BIOS unit ignored$'
HOMsg	db	'Hardware-only disk scan:',CR,LF,'$'
PMMsg	db	'Primary-master disk $'
PSMsg	db	'Primary-slave disk $'
SMMsg	db	'Secondary-master disk $'
SSMsg	db	'Secondary-slave disk $'
MSMsg	db	'.',CR,LF,'    Set to UltraDMA mode '
CurMode db	'0, ATA-'
DMode	db	'16. $'
RTMsg	db	'    Read test = $'
RRate	db	'    0 MB/sec'
CRMsg	db	'.',CR,LF,'$'
NRMsg	db	'    Read tests omitted.',CR,LF,'$'
AEMsg	db	'absent or non-ATA$'
DEMsg	db	'is not UltraDMA$'
NPMsg	db	'CHS-values ERROR$'
SEMsg	db	'Set-Mode error$'
BEMsg	db	'failed BIOS read!  $'
TEMsg	db	'FAILED read test!  $'
EMsg1	db	'DMA timed out$'
EMsg2	db	'DMA error$'
EMsg3	db	'Controller busy before I-O$'
EMsg4	db	'Controller busy after I-O$'
EMsg5	db	'First DRQ timed out$'
EMsg6	db	'Disk not ready before I-O$'
EMsg7	db	'Disk not ready after I-O$'
EMsg8	db	'Write FAULT before I-O$'
EMsg9	db	'Write FAULT after I-O$'
EMsg10	db	'Hard error at I-O end$'
EMsg11	db	'BIOS/driver read MISMATCH$'
EMsg12	db	'XMS memory error$'
RCMsg	db	'Return code '
RCode	db	'00h$'
Suffix	db	'; driver NOT loaded!',CR,LF,'$'
;
; Initialization Tables And Variables.
;
	align	4
HDNames dw	PMMsg		;Table of hard-disk "name" pointers.
	dw	PSMsg
	dw	SMMsg
	dw	SSMsg
Modes	db	'16. '		;Mode 0 = ATA-16  UltraDMA mode table.
	db	'25. '		;Mode 1 = ATA-25.
	db	'33. '		;Mode 2 = ATA-33.
	db	'44. '		;Mode 3 = ATA-44  (Rare but possible).
	db	'66. '		;Mode 4 = ATA-66.
	db	'100.'		;Mode 5 = ATA-100.
	db	'133.'		;Mode 6 = ATA-133.
	db	'166.'		;Mode 7 = ATA-166.
MsgTbl	db	008h		;Driver error-message codes/addresses.
	dw		EMsg1
	db	00Fh
	dw		EMsg2
	db	020h
	dw		EMsg3
	db	021h
	dw		EMsg4
	db	080h
	dw		EMsg5
	db	0AAh
	dw		EMsg6
	db	0ABh
	dw		EMsg7
	db	0CCh
	dw		EMsg8
	db	0CDh
	dw		EMsg9
	db	0E0h
	dw		EMsg10
	db	0FEh
	dw		EMsg11
MsgEnd	db	0FFh
	dw		EMsg12
IVDSLen	dd	ResEnd-@	;VDS length (driver size).
IVDSOfs	dd	0		;VDS 32-bit offset.
IVDSSeg	dd	0		;VDS 16-bit segment (hi-order zero).
IVDSAdr	dd	0		;VDS 32-bit driver address.
Packet	dd	0		;"Init" request packet address.
StackL	dw	STACK		;Stack length for exit logic.
OvlpSw	db	0		;Output overlap switch-option flag.
NoRdSw	db	0		;"No read tests" flag.
HDCount db	0		;Remaining hard-disk count.
EDDFlag db	0		;"EDD BIOS present" flag.
HDUnit	db	0		;Current BIOS unit number.
HDIndex db	0		;IDE "index" number.
;
; Main driver-initialization routine, entered from the DOS "device
;   interrupt" logic below, after it does one-time-only functions.
;
I_RScn	mov	dx,NDMsg	;Point to "No disk to use" message.
	xor	ax,ax		;Load & reset EDD BIOS flag.
	xchg	al,[EDDFlag]
	or	ax,ax		;Were we scanning v.s. DPTE data?
	jz	near I_Err	;No?  Display "No disk" and exit!
I_Scan	mov	ax,00080h	;Reset hard-disk unit number & index.
	mov	[HDUnit],ax
	mov	byte [HDCount],0  ;Reset remaining hard-disk count.
@BIOSHD equ	$-1		;(BIOS hard-disk count, set above).
	cmp	ah,[EDDFlag]	;Will disk scan use the EDD BIOS?
	jne	I_Next		;Yes, go start with BIOS unit 80h.
	mov	dx,HOMsg	;Display "hardware-only" message.
	call	I_Dspl
I_Next	movzx	bx,[HDIndex]	;Get disk unit-number index.
	cmp	bh,[EDDFlag]	;Are we using DPTE data from BIOS?
	je	I_ChMS		;No, check disk at "fixed" addresses.
	mov	ah,048h		;Get next BIOS disk's EDD parameters.
	mov	dl,[HDUnit]
	mov	si,EDDBuff
	call	I_Int13
	jc	I_ErEDD		;Error?  Ignore this BIOS unit!
	cmp	dword [si+26],byte -1  ;Valid DPTE pointer?
	je	near I_More	;No, ignore unit & check for more.
	les	si,[si+26]	;Get this disk's DPTE pointer.
	mov	bx,15		;Calculate DPTE checksum.
	xor	cx,cx
I_CkSm	add	cl,[es:bx+si]
	dec	bx
	jns	I_CkSm
	jcxz	I_EDOK		;If checksum O.K., use parameters.
I_ErEDD	mov	dx,EBMsg	;Display "EDD error" and ignore unit!
	jmp	short I_ErrD
I_EDOK	mov	bx,00010h	;Initialize IDE unit number index.
	and	bl,[es:si+4]
	shr	bl,4
	mov	ax,[es:si]	;Get disk's IDE base address.
	cmp	ax,CDATA	;Is this a primary-channel disk?
	je	I_Indx		;Yes, set disk unit-number index.
	cmp	ax,(CDATA-080h)	;Is this a secondary-channel disk?
	jne	I_More		;No, ignore unit & check for more.
	inc	bx		;Adjust for secondary channel.
	inc	bx
I_Indx	mov	[HDIndex],bl	;Set disk's unit number index.
I_ChMS	mov	ax,bx		;Separate channel and master/slave.
	shr	al,1
	mov	ah,(LBABITS/32)	;Get drive-select "nibble".
	rcl	ah,5
	ror	al,1		;Set channel offset (2nd = 080h).
	mov	[@HDOffs],al
	mov	dx,CDSEL	;Get IDE disk-select address.
	xor	dl,al
	mov	al,ah		;Select master or slave disk.
	out	dx,al
	shl	bx,2		;Save disk's "Units" table index.
	push	bx
	shr	bx,1		;Get "channel name" message index.
	mov	dx,[bx+HDNames] ;Display disk's IDE channel name.
	call	I_Dspl		;("Primary master", etc.).
	mov	ah,008h		;Get BIOS CHS values for this disk.
	mov	dl,[HDUnit]
	call	I_Int13
	xchg	ax,dx		;Set AX-reg. with head-number value.
	pop	bx		;Reload disk's unit number index.
	mov	dx,NPMsg	;Point to "parameter ERROR" message.
	jc	I_ErrD		;Error -- display msg. & ignore disk.
	mov	al,cl		;Save sectors per head value.
	and	al,03Fh		;(Clear 2 high-order cylinder bits).
	mov	[bx+Units+1-@],al
	inc	ah		;Get heads/cylinder (BIOS value + 1).
	mul	ah		;Save sectors per cylinder value.
	mov	[bx+Units+2-@],ax
	mov	al,[HDUnit]	;Activate this disk in main driver.
	mov	[bx+Units-@],al
	push	bx		;Validate this UltraDMA disk.
	call	I_ValD
	pop	bx
	jnc	I_More		;If no errors, check for more disks.
	mov	byte [bx+Units-@],0FFh  ;DELETE disk in main driver!
I_ErrD	call	I_Dspl		;Display error for this disk.
	mov	dx,CRMsg	;Display error-message suffix.
	call	I_Dspl
I_More	add	word [HDUnit],00101h  ;Bump BIOS unit & disk index.
	cmp	word [EDDFlag],08400h ;No EDD and all 4 units tested?
	je	I_AnyD		;Yes, see if we found any disks.
	dec	byte [HDCount]	;More BIOS disks to check?
	jnz	I_Next		;Yes, loop back and do next one.
I_AnyD	mov	bx,16		;Set up to scan for last unit.
I_ChkU	sub	bx,byte 4	;Any more active units to check?
	js	near I_RScn	;No, see if we should do a re-scan.
	cmp	byte [bx+Units-@],0FFh  ;Is this unit active?
	je	I_ChkU		;No, loop back and check next unit.
	add	bx,byte 4	;Post last-unit index in main driver.
	mov	[@LastU],bx
	mov	cx,[StackL]	;Using "DMA only" mode (no stack)?
	jcxz	I_SetV		;Yes, "hook" only Int 13h vector. 
	cld			;Set driver stack space to zeros.
	mov	di,[@Stack]	;(Helps debug if unused stack = 0!).
	sub	di,cx
	push	cs
	pop	es
	xor	al,al
	rep	stosb
	cmp	byte [@OvlpF],0	;Is output overlap disabled?
	je	I_SetV		;Yes, "hook" only Int 13h vector.
	mov	ax,03508h	;Get current IRQ8 timer vector.
	call	I_Int21
	push	es		;Save previous timer vector.
	push	bx
	pop	dword [@PrvTmr]
	mov	ax,02508h	;"Hook" IRQ8 timer to our driver.
	mov	dx,TmInt
	call	I_Int21
I_SetV	call	I_Hook		;"Hook" Int 13h calls to our driver.
	popad			;Reload all 32-bit registers.
	push	ax		;Save all 16-bit registers.
	push	bx
	push	cx
	push	dx
	push	si
	push	di
	mov	ax,[IVDSLen]	;Get driver size & "success" flag.
	mov	dx,RPDON
	jmp	short I_Exit	;Go post "init" packet results & exit.
I_VErr	mov	dx,VEMsg	;VDS "lock" error!  Point to message.
I_Err	push	dx		;Save error message pointer.
	shr	byte [IVDSOfs],1 ;Was driver "locked" by VDS?
	jnc	I_XUnl		;No, get rid of our XMS memory.
	mov	di,IVDSLen	;Point to initialization VDS block.
	mov	ax,08104h	;"Unlock" this driver from memory.
	xor	dx,dx
	call	VDInit
I_XUnl	mov	cx,[XMHdl]	;Did we reserve any XMS memory?
	jcxz	I_LdMP		;No, display desired error message.
	mov	ah,00Dh		;Unlock our XMS memory buffer.
	mov	dx,cx
	push	dx
	call	XMCall
	mov	ah,00Ah		;Free our XMS memory buffer.
	pop	dx
	call	XMCall
	mov	ah,006h		;Do local-disable of "A20 line".
	call	XMCall
I_LdMP	pop	dx		;Reload error message pointer.
I_EOut	call	I_Dspl		;Display desired error message.
	popad			;Reload all 32-bit registers.
	push	ax		;Save all 16-bit registers.
	push	bx
	push	cx
	push	dx
	push	si
	push	di
I_Quit	mov	dx,Suffix	;Display message suffix.
	call	I_Dspl
I_BadP	xor	ax,ax		;Get "null" length & error flags.
	mov	dx,RPDON+RPERR
I_Exit	les	bx,[Packet]	;Post results in "init" packet.
	mov	[es:bx+RPSize],ax
	mov	[es:bx+RPSize+2],cs
	mov	[es:bx+RPStat],dx
	pop	di		;Reload all CPU registers and exit.
	pop	si
	pop	dx
	pop	cx
	pop	bx
	pop	ax
	pop	es
	pop	ds
	popf
	retf
;
; Subroutine to "validate" an UltraDMA hard-disk.
;
I_ValD	mov	al,0ECh		;Issue "Identify Device" command.
	call	I_Cmd
	jnc	I_PIO		;If no error, get "identify" data.
I_AErr	mov	dx,AEMsg	;Absent or non-ATA!   Point to msg.
	stc			;Set carry flag (error!) and exit.
	ret
I_PIO	add	dx,byte (CDATA-CCMD)  ;Point to PIO data register.
	in	ax,dx		;Read I.D. bytes 0 and 1.
	xchg	ax,si		;Save "ATA/ATAPI" flag word.
	mov	cx,26		;Skip I.D. bytes 2-53.
I_Skp0	in	ax,dx
	loop	I_Skp0
	cld			;Ensure FORWARD "string" commands!
	push	ds		;Point to disk-name message.
	pop	es
	mov	di,DName
	mov	cl,20		;Read & swap disk name into message.
I_RdNm	in	ax,dx		;(I.D. bytes 54-93).
	xchg	ah,al
	stosw
	loop	I_RdNm
	mov	cl,6		;Skip I.D. bytes 94-105.
I_Skp1	in	ax,dx
	loop	I_Skp1
	in	ax,dx		;Read I.D. bytes 106 and 107.
	mov	bh,al		;Save "DMA valid" flag byte.
	mov	cl,34		;Skip I.D. bytes 108-175.
I_Skp2	in	ax,dx
	loop	I_Skp2
	in	ax,dx		;Read I.D. bytes 176 and 177.
	mov	bl,ah		;Save "UltraDMA selected" flag byte.
	mov	cl,167		;Skip remaining I.D. data.
I_Skp3	in	ax,dx
	loop	I_Skp3
	shl	si,1		;Is this an "ATA" hard-disk?
	jc	I_AErr		;No?  Exit & display message!
	test	bh,004h		;Are UltraDMA flag bits valid?
	jz	I_DErr		;No?  Exit & display message!
	mov	di,Modes	;Point to UltraDMA mode table.
	mov	al,'0'		;Initialize "current mode" value.
	mov	cl,002h		;Set rotating mode-check bit.
	cmp	bl,001h		;Will disk do UltraDMA mode 0?
	jae	I_NxtM		;Yes, find its best UltraDMA mode.
I_DErr	mov	dx,DEMsg	;Not an UltraDMA disk!   Point to msg.
I_SErr	stc			;Set carry flag (error!) and exit.
	ret
I_NxtM	mov	[CurMode],al	;Update "current mode" value in message.
	cmp	al,0FFh		;Are we at the desired MAXIMUM "mode"?
@MaxMod	equ	$-1		;(Max. UltraDMA "mode", set by startup).
	jae	I_GotM		;Yes, use current mode.
	cmp	bl,cl		;Will disk do next UltraDMA mode?
	jb	I_GotM		;No, use current mode.
	inc	ax		;Set up for next UltraDMA mode.
	add	di,byte 4
	shl	cl,1		;More UltraDMA modes to check?
	jnz	I_NxtM		;Yes, loop back.
I_GotM	push	ax		;Save "current mode" value.
	mov	eax,[di]	;Set UltraDMA mode in set-mode message.
	mov	[DMode],eax
	inc	dx		;Set mode-select subcode.
	mov	al,SETM
	out	dx,al
	pop	ax		;Set desired UltraDMA mode.
	add	al,010h
	inc	dx
	out	dx,al
	mov	al,SETF		;Issue set-features command to disk.
	call	I_Cmd
	mov	dx,SEMsg	;Point to "Set-mode" error message.
	jc	I_SErr		;If error, set carry flag and exit.
	mov	di,DNEnd	;Point to end of disk name.
I_NxtN	cmp	di,DName	;Are we at the disk-name start?
	je	I_Unnm		;Yes, disk name is all spaces!
	cmp	byte [di-1],' '	;Is preceding byte a space?
	jne	I_TrmN		;No, terminate disk name message.
	dec	di		;Decrement disk name pointer.
	jmp	short I_NxtN	;Go see if next byte is a space.
I_Unnm	mov	dword [di],"unna"   ;Set "unnamed" as disk name.
	mov	dword [di+4],"med "
	add	di,byte 7
I_TrmN	mov	byte [di],'$'	;Set message terminator after name.
	dec	byte [NoRdSw]	;Should we omit the "read tests"?
	jns	near I_DspNm	;Yes, just display disk name & mode.
	mov	byte [XMSMov],0C3h  ;Disable XMS moves with "ret" cmd.
	call	I_Hook		;"Hook" this driver into Int 13h.
	call	I_RdT		;Do initial read for synchronization.
	jnc	I_SetC		;If O.K., set up 4-pass read test.
I_RdEr	push	ax		;Read error!  Save driver return code.
	call	I_RVec		;Restore original INT 13h vector.
	pop	ax		;Reload driver return code.
I_Flunk	push	ax		;Display "FAILED read test" message.
	mov	dx,TEMsg
	call	I_Dspl
	pop	ax
	mov	di,(MsgTbl-3-@)	;Point to our error-message table.
I_EScn	add	di,byte 3	;Bump to next message code/address.
	mov	dx,[di+1]	;Set message address in DX-register.
	cmp	ah,[di]		;Driver return code = table code?
	je	I_RBye		;Yes, this is the message we want!
	cmp	di,(MsgEnd-@)	;Is driver return code unrecognized?
	jb	I_EScn		;No, loop back and check next entry.
I_RCMs	mov	cx,2		;Set return code in error message.
	mov	si,RCode
	call	HexCnv
	mov	dx,RCMsg	;Point to "Return code" message.
I_RBye	stc			;Set carry flag (error!) and exit.
	ret
I_SetC	xor	dx,dx		;Clear read counter.
	xor	si,si		;Point to low-memory BIOS timer.
	mov	es,si
	mov	si,BIOSTMR
	mov	cl,[es:si]	;Load current timer tick count LSB.
I_RDly	cmp	cl,[es:si]	;Has next timer "tick" occurred?
	je	I_RDly		;No, keep waiting.
	add	cl,1+4		;Yes, update & prepare for 4 passes.
I_RNxt	inc	dx		; Yes, count reads up
	push	es		;Save timer/counter registers.
	push	si
	push	dx
	push	cx
	call	I_RdT		;Read RBYTES into XMS buffer.
	pop	cx		;Reload timer/counter registers.
	pop	dx
	pop	si
	pop	es
	jc	I_RdEr		;Read error?  Display message & exit!
	cmp	cl,[es:si]	;Next timer interrupt arrived?
	jne	I_RNxt		;No, read once more.
	shr	dx,2		;Save average rate for 4 passes.
	mov	[Bucket],dx
	mov	al,RC_SC	;Set "read comparison" parameters.
	mov	[RCSects],al
	mov	cx,((RC_CYL*256)+RC_SEC)
	mov	[RCSecNo],cl
	mov	dh,RC_HD
	call	I_RdT1		;Input "read comparison" data to XMS.
	jc	I_RdEr		;Read error?  Display message & exit!
	call	I_RVec		;Restore original Int 13h vector.
	cld			;Set up to initialize XMS block.
	push	ds
	pop	es
	mov	di,(VDSLn-@)
	mov	eax,512		;Set 1-sector XMS block length.
	stosd
	mov	ax,[XMHdl]	;Reset XMS "source address".
	stosw
	mov	eax,[XMOfs]
	stosd
	xor	ax,ax		;Reset XMS "destination address".
	stosw
	mov	ax,(RBuff+3-@)
	stosw
	mov	ax,cs
	stosw
I_RCIn	mov	al,1		;Set BIOS input sector count of 1.
	mov	bx,(RBuff+3-@)	;Set BIOS input buffer address.
	mov	ch,RC_CYL	;Set BIOS input disk address.
	mov	cl,[RCSecNo]
	mov	dh,RC_HD
	call	I_RdT2		;Have BIOS read next data sector.
	jnc	I_RCkS		;Any error during BIOS read?
	push	ax		;Yes?  Save BIOS return code.
	mov	dx,BEMsg	;Display "failed BIOS read" message.
	call	I_Dspl
	pop	ax		;Reload BIOS return code.
	jmp	I_RCMs		;Go set up return-code msg. and exit.
I_RCkS	call	I_Roll		;"Roll" & save BIOS sector checksum.
	push	ax
	mov	si,(VDSLn-@)	;Get next data sector from XMS memory.
	call	XMGo
	pop	dx		;Reload BIOS sector checksum.
	or	ax,ax		;Any errors during XMS move?
I_RCEr	jnz	I_Flunk		;Yes?  Display DRIVER error and exit!
	call	I_Roll		;"Roll" checksum for XMS input sector.
	cmp	ax,dx		;BIOS and driver input "mismatch"?
	mov	ah,0FEh		;(Load "mismatch" return code if so).
	jne	I_RCEr		;Yes?  Display DRIVER error and exit!
	add	word [XMSSA+1],byte 2 ;Bump XMS source to next sector.
	inc	byte [RCSecNo]	;Update BIOS input sector number
	dec	byte [RCSects]	;More BIOS data to input?
	jnz	I_RCIn		;Yes, go read next sector.
I_DspNm	mov	dx,DNMsg	;Display disk "name" message.
	call	I_Dspl
	mov	dx,MSMsg	;Display "Set to mode" message.
	call	I_Dspl
	mov	dx,NRMsg	;Point to "Read tests DISABLED" msg.
	inc	byte [NoRdSw]	;Were the "read tests" omitted?
	jnz	I_Dspl		;Yes, display "DISABLED" msg. & exit.
	mov	ax,[Bucket]	;Reload average read rate.
	mov	di,(RRate+4-@)	;Point to "read test" message digits.
	mov	byte [di],'0'	;Initialize "read test" digits to 0.
	or	ax,ax		;Did the disk read NOTHING?
	jz	I_Rate		;Yes, display "read test = 0".
	mov	cx,10		;Set divisor for ASCII conversion.
I_ItoA	xor	dx,dx		;Divide remaining read rate by 10.
	div	cx
	xchg	dx,ax		;Save quotient & get remainder in AX.
	add	al,'0'		;Convert next digit to ASCII.
	mov	[di],al		;Store digit in "read test" message. 
	dec	di
	xchg	dx,ax		;Reload quotient.
	or	ax,ax		;Is quotient zero?
	jnz	I_ItoA		;No, loop back & get next digit.
	inc	di		;Point to high-order rate digit.
I_Rate	push	di		;Save pointer to read-rate digits.
	mov	dx,RTMsg	;Display "Read Test" message.
	call	I_Dspl
	pop	dx		;Display read rate for this disk.
I_Dspl	mov	ah,009h
	call	I_Int21
	clc			;Clear carry (no errors!) and exit.
	ret
;
; Subroutine to do all "read test" inputs.
;
I_RdT	mov	al,RS_SC	;"Read speed" -- set sector count.
	mov	cx,1		;Set cylinder 0, sector 1.
	xor	dh,dh		;Set head 0.
I_RdT1	mov	bx,(RBuff+3-@)	;Use "odd" offset to avoid VDS use.
I_RdT2	push	ds		;Point ES-register to our data.
	pop	es
	mov	ah,2		;Set "CHS read" request code.
	mov	dl,[HDUnit]	;Set desired unit number.
I_Int13	int	013h		;Issue desired Int 13h request.
	push	cs		;Reload our DS-register.
	pop	ds
	ret			;Exit.
;
; Subroutine to issue disk initialization commands.
;
I_Cmd	mov	dx,CCMD		;Issue desired init command.
	xor	dl,0
@HDOffs	equ	$-1
	out	dx,al
	xor	si,si		;Point to low-memory BIOS timer.
	mov	es,si
	mov	si,BIOSTMR
	mov	cl,RDYTO	;Set I-O timeout limit in CL-reg.
	add	cl,[es:si]
I_CmdW	cmp	cl,[es:si]	;Has our command timed out?
	je	I_CmdE		;Yes, set CPU carry flag & exit.
	in	al,dx		;Get IDE controller status.
	test	al,BSY+RDY	;Controller or disk still busy?
	jle	I_CmdW		;Yes, loop back and check again.
	test	al,ERR		;Did command cause any errors?
	jz	I_CmdX		;No, leave carry flag off & exit.
I_CmdE	stc			;Error!  Set CPU carry flag.
I_CmdX	ret			;Exit.
;
; Subroutine to "hook" this driver into the Int 13h chain.
;
I_Hook	mov	ax,03513h	;Get current Int 13h vector.
	call	I_Int21
	push	es		;Save previous Int 13h vector.
	push	bx
	pop	dword [@PrvI13]
	mov	ax,02513h	;"Hook" this driver into Int 13h.
	mov	dx,(Entry-@)
I_Int21	int	021h
	push	cs		;Reload our DS-register.
	pop	ds
	ret			;Exit.
;
; Subroutine to restore the Int 13h vector after initialization reads.
;
I_RVec	mov	ax,02513h	;Set back the old Int 13h vector.
	lds	dx,[@PrvI13]
	call	I_Int21
	mov	al,00Eh		;Re-enable XMS moves with "push cs".
	mov	[XMSMov],al
	ret			;Exit.
;
; Subroutine to "roll" the checksum for data input to "RBuff".
;
I_Roll	xor	ax,ax		;Reset checksum "bucket".
	mov	cx,256		;"Roll" 256 words of data.
	mov	si,(RBuff+3-@)	;Point to read-test buffer.
I_RolA	add	ax,[si]		;Add next word to checksum.
	adc	ax,0		;If carryout, add it back.
	inc	si		;Point to next data word.
	inc	si
	loop	I_RolA		;If more words to go, loop back.
	ret			;Exit -- checksum is in AX-reg.
;
; Initialization Buffers and "After Startup" Variables.   The 520
;   bytes starting at "RBuff" must be "disposable" and used once-
;   only.   After, this space will be used as a "read-test" input
;   buffer and for other temporary variables, as described below.
;
	align	4
EDDBuff dd	30		;Start of 30-byte EDD input buffer.
RBuff	equ	$		;Start of "read-test" input buffer.
Bucket	equ	RBuff+516	;Working 16-bit "bucket".
RCSecNo	equ	RBuff+518	;"Read compare" sector address.
RCSects	equ	RBuff+519	;"Read compare" remaining sectors.
;
; "Strategy" routine -- At entry, ES:BX points to the DOS init request
;   packet, whose address is saved for processing below.
;
Strat	mov	[cs:Packet],bx	;Save DOS request-packet address.
	mov	[cs:Packet+2],es
	retf			;Exit & await DOS "Device Interrupt".
;
; "Device-Interrupt" routine -- This routine initializes the driver.
;
DevInt	pushf			;Entry -- save CPU flags.
	push	ds		;Save all 16-bit CPU registers.
	push	es
	push	ax
	push	bx
	push	cx
	push	dx
	push	si
	push	di
	push	cs		;Set our DS-register.
	pop	ds
	les	bx,[Packet]	;Point to DOS request packet.
	xor	ax,ax		;Get a zero for below.
	cmp	[es:bx+RPOp],al	;Is this an "Init" packet?
	je	I_TTL		;Yes, display our title message.
	jmp	I_BadP		;Go post errors and exit quick!
I_TTL	mov	dx,UDMsg	;Display driver "title" message.
	call	I_Dspl
	pushf			;80386 test -- save CPU flags.
	push	sp		;See if CPU is an 80286 or newer.
	pop	ax
	cmp	ax,sp		;(80286+ push SP, then decrement it)
	jne	I_Junk		;CPU is 8086/80186 -- cannot USE it!
	push	07000h		;80286 and up -- try to set NT|IOPL.
	popf
	pushf
	pop	ax
	test	ah,070h		;Did any NT|IOPL bits get set?
	jnz	I_386		;Yes, CPU is at least an 80386.
I_Junk	popf			;Reload starting CPU flags.
	mov	dx,PRMsg	;Display "No 80386 CPU" message.
	call	I_Dspl
	jmp	I_Quit		;Go display suffix and exit quick!
I_386	popf			;Reload starting CPU flags.
	pop	di		;Reload all 16-bit registers.
	pop	si
	pop	dx
	pop	cx
	pop	bx
	pop	ax
	pushad			;Save all 32-bit registers.
	les	bx,[Packet]	;Point back to DOS request packet.
	les	bx,[es:bx+RPCmdL]  ;Get command-line data pointer.
I_NxtC	mov	al,[es:bx]	;Get next command-line byte.
I_NxtC1	inc	bx		;Bump pointer to following byte.
	cmp	al,0		;Is byte the command-line terminator?
	je	I_TermC		;Yes, check if overlap was enabled.
	cmp	al,CR		;Is byte an ASCII carriage-return?
	je	I_TermC		;Yes, check if overlap was enabled.
	cmp	al,LF		;Is byte an ASCII line-feed?
	je	I_TermC		;Yes, check if overlap was enabled.
	cmp	al,'/'		;Is byte a slash?
	je	I_ChkO		;Yes, see if "O" or "o" is next.
	cmp	al,'-'		;Is byte a dash?
	jne	I_NxtC		;No, check next command-line byte.
I_ChkO	mov	al,[es:bx]	;Get byte after the slash or dash.
	and	al,0DFh		;Mask out lower-case bit (020h).
	cmp	al,'O'		;Is byte an "O" or "o"?
	jne	I_ChkB		;No, go see if byte is "B" or "b".
	mov	[OvlpSw],al	;Set "overlap" switch-option flag.
	inc	bx		;Point to byte following the "O".
	jmp	short I_NxtC	;Continue scanning for a terminator.
I_ChkB	cmp	al,'B'		;Is byte a "B" or "b"?
	jne	I_ChkM		;No, go see if byte is "M" or "m".
	mov	[@IgnXMS],al	;Set "Ignore XMS" flag below.
	inc	bx		;Point to byte following the "B".
	jmp	short I_NxtC	;Continue scanning for a terminator.
I_ChkM	cmp	al,'M'		;Is byte an "M" or "m"?
	jne	I_ChkX		;No, go see if byte is "X" or "x".
	inc	bx		;Get byte following the "M" or "m".
	mov	al,[es:bx]
	cmp	al,'0'		;Is byte below a zero?
	jb	I_NxtC1		;Yes, see if byte is a terminator.
	cmp	al,'7'		;Is byte above a seven?
	ja	I_NxtC1		;Yes, see if byte is a terminator.
	mov	[@MaxMod],al	;Set maximum UltraDMA "mode" value.
	inc	bx		;Point to byte following the "mode".
	jmp	short I_NxtC	;Continue scanning for a terminator.
I_ChkX	cmp	al,'X'		;Is byte an "X" or "x"?
	jne	I_NxtC1		;No, see if byte is a terminator.
	mov	[NoRdSw],al	;Set "no read" switch-option flag.
	inc	bx		;Point to byte following the "X".
	jmp	short I_NxtC	;Continue scanning for a terminator.
I_TermC	mov	dx,EOMsg	;Point to overlap "enabled" message.
	cmp	byte [OvlpSw],0	;Was output overlap ever enabled?
	jne	I_Ovlp		;Yes, display output overlap status.
	call	I_DsbO		;Too bad -- DISABLE output overlap.
	mov	ax,BSTACK	;Set "basic" driver stack length.
	mov	[StackL],ax
	mov	ax,(BaseEnd-@)	;Set "basic" driver stack offset.
	mov	[@Stack],ax
	mov	[IVDSLen],ax	;Reduce resident driver size.
	mov	dx,DOMsg	;Point to overlap "disabled" message.
I_Ovlp	call	I_Dspl		;Display status of output overlap.
	xor	edi,edi		;Get PCI BIOS "I.D." code.
	mov	ax,0B101h
	int	01Ah
	push	cs		;Reload our DS-register.
	pop	ds
	cmp	edx,"PCI "	;Is PCI BIOS V2.0C or newer?
	mov	dx,PEMsg	;(Get error message pointer if not).
	jne	I_PCEr		;No?  Go display message and exit!
	mov	si,LBA2		;Point to interface byte table.
	cld			;Ensure FORWARD "string" commands!
I_GetD	mov	ecx,000010100h	;We want class 1 storage, subclass 1 IDE.
	lodsb			;Get next "valid" PCI interface byte.
	mov	cl,al
	push	si		;Search for our PCI class code.
	mov	ax,0B103h	;(Returns bus/device/function in BX).
	xor	si,si
	int	01Ah
	push	cs		;Reload our DS-register.
	pop	ds
	pop	si
	jnc	I_GotD		;Found our boy!  Go process it.
	cmp	si,LBA		;More interface bytes to try?
	jb	I_GetD		;Yes, go try next one.
	mov	dx,NEMsg	;BAAAD News!  Point to error message.
	jmp	short I_PCEr	;Go display error message and exit.
I_GotD	push	bx		;Save PCI bus/device/function.
	mov	ax,0B108h	;Get low-order PCI command byte.
	mov	di,4
	int	01Ah
	push	cs		;Reload our DS-register.
	pop	ds
	pop	bx		;Reload PCI bus/device/function.
	and	cl,005h		;Mask Bus-Master and I-O Space bits.
	cmp	cl,005h		;Is this how our controller is set up?
	je	I_Base		;Yes, get our PCI base address.
	mov	dx,MEMsg	;Cannot USE it -- point to error msg.
I_PCEr	jmp	I_EOut		;Go display "INVALID" message & exit!
I_Base	push	bx		;Save PCI bus/device/function.
	mov	ax,0B109h	;Get PCI base address (register 4).
	mov	di,32
	int	01Ah
	push	cs		;Reload our DS-register.
	pop	ds
	pop	bx		;Reload PCI bus/device/function.
	xchg	ax,cx		;Post our DMA controller address.
	and	al,0FCh
	mov	[DMAAd],ax
	mov	[@DMALo1],al	;Set low-order DMA address bytes.
	add	[@DMALo2],al
	mov	si,DspAd	;Set hex address in display message.
	call	HexCnv4
	push	bx		;Save PCI bus/device/function.
	mov	ax,0B10Ah	;Get Vendor and Device I.D.
	xor	di,di
	int	01Ah
	push	cs		;Reload our DS-register.
	pop	ds
	pop	bx		;Reload PCI bus/device/function.
	push	ecx		;Save Vendor and Device I.D.
	mov	eax,ecx		;Set vendor I.D. in display message.
	mov	si,DspVI
	call	HexCnv4
	pop	eax		;Reload Vendor and Device I.D.
	shr	eax,16		;Set Device I.D. in display message.
	mov	si,DspDI
	call	HexCnv4
	mov	ah,bh		;Set PCI bus number in message.
	mov	cl,2
	mov	si,DspPB
	call	HexCnv
	mov	ah,bl		;Set PCI device number in message.
	shr	ah,3
	mov	cl,2
	mov	si,DspPD
	call	HexCnv
	and	bl,007h		;Set PCI function number in message.
	or	bl,030h
	mov	[DspPF],bl
	mov	dx,PCMsg	;Display all controller information.
	call	I_Dspl
	mov	dx,XDMsg	;Get "XMS ignored" message pointer.
	mov	ax,04300h	;Get "XMS present" request code.
	cmp	al,0		;Is XMS memory to be ignored?
@IgnXMS	equ	$-1		;("Ignore XMS" flag, set by startyp).
	jne	I_XErr		;Yes, display message & disable XMS.
	int	02Fh		;Inquire about an XMS manager.
	push	cs		;Reload our DS-register.
	pop	ds
	mov	dx,NXMsg	;Point to "No XMS manager" message.
	cmp	al,080h		;Is an XMS manager installed?
	jne	I_XErr		;No, display message & disable XMS.
	mov	ax,04310h	;Get XMS manager "entry" address.
	int	02Fh
	push	cs		;Reload our DS-register.
	pop	ds
	push	es		;Save XMS manager "entry" address.
	push	bx
	pop	dword [@XEntry]
	mov	ah,009h		;Ask XMS manager for 128K of memory.
	mov	dx,128
	call	XMCall
	jc	I_XMErr		;If error, display msg. & disable XMS.
	mov	[XMHdl],dx	;Save XMS buffer handle number.
	mov	ah,00Ch		;"Lock" our XMS memory (direct call,
	call	far [@XEntry]	;  as our subroutine ZEROS BX-reg.!).
	push	cs		;Reload our DS-register.
	pop	ds
	dec	ax		;Did XMS memory get locked?
	jnz	I_RidX		;No?  Unuseable -- get RID of it!
	shl	edx,16		;Get unaligned XMS buffer address.
	mov	dx,bx
	mov	eax,edx		;Find 1st 64K boundary after start.
	add	eax,65536
	xor	ax,ax
	mov	[@XBufAd],eax	;Set final XMS 32-bit buffer address.
	sub	eax,edx		;Initialize "offset" into XMS memory.
	mov	[XMOfs],eax
	mov	ah,005h		;Local-enable "A20 line" FOREVER!
	call	XMCall		;("A20" CANNOT turn off during DMA!).
	jnc	I_Stop		;If no error, ensure DMA is stopped.
	mov	ah,00Dh		;Unuseable!  Unlock our XMS memory.
	mov	dx,[XMHdl]
	call	XMCall
I_RidX	xor	dx,dx		;Load & reset our XMS buffer handle.
	xchg	dx,[XMHdl]
	mov	ah,00Ah		;Free our XMS memory.
	call	XMCall
I_XMErr mov	dx,XEMsg	;Point to "XMS setup error" message.
I_XErr	call	I_Dspl		;Display desired XMS error message.
	mov	dx,NBMsg	;Display "no buffered I-O" message.
	call	I_Dspl
	inc	byte [NoRdSw]	;Disable all "read tests".
	call	I_DsbO		;Disable output overlap.
	mov	di,@NoOp2	;No-op all stack switching logic.
	stosb			;("DMA only" mode uses NO stack!).
	mov	ax,0DB87h
	mov	cx,8
	rep	stosw
	mov	di,@NoXMB
	stosw
	stosw
	stosw
	mov	si,@NoXMA	;Reject buffered I-O by overlaying
	mov	di,BufIO	;  "BufIO" with driver exit logic.
	mov	cx,(@NoXMB-@NoXMA)
	rep	movsb
	mov	al,0E9h		;End "BufIO" overlay with "jmp Pass".
	stosb
	db	0B8h
	dw	(Pass-(BufIO+(@NoXMB+3-@NoXMA)))
	stosw
	mov	[StackL],cx	;Set stack length to zero!
	mov	ax,(NonXEnd-@)	;Reduce resident driver size.
	mov	[IVDSLen],ax
I_Stop	mov	dx,[DMAAd]	;Ensure any previous DMA is stopped.
	in	al,dx		;(See "DoDMA" routine notes below).
	and	al,0FEh
	out	dx,al
	add	dx,byte 8	;Stop secondary-channel DMA, also!
	in	al,dx
	and	al,0FEh
	out	dx,al
	xor	eax,eax		;Zero EAX-reg. for 20-bit addressing.
	mov	es,ax		;Point ES-reg. to low memory.
	or	al,[es:HDISKS]	;Did BIOS find any hard-disks?
	jz	near I_RScn	;No?  Display "No disk" and exit!
	mov	[@BIOSHD],al	;Set BIOS hard-disk count below.
	mov	ax,cs		;Set our code segment in VDS block.
	mov	[IVDSSeg],ax
	shl	eax,4		;Get 20-bit driver virtual address.
	cli			;Avoid interrupts during VDS tests.
	test	byte [es:VDSFLAG],020h ;Are "VDS services" active?
	jz	I_SetA		;No, set 20-bit virtual addresses.
	mov	di,IVDSLen	;Point to initialization VDS block.
	mov	ax,08103h	;"Lock" this driver into memory.
	mov	dx,0000Ch
	call	VDInit
	jc	near I_VErr	;Error?  Display error msg. & exit!
	inc	byte [IVDSOfs]	;Set initialization VDS "lock" flag.
	mov	eax,[IVDSAdr]	;Get 32-bit starting driver address.
I_SetA	sti			;Re-enable CPU interrupts.
	add	[PRDAd],eax	;Set relocated 32-bit PRD address.
	mov	ah,041h		;See if we have an EDD BIOS.
	mov	bx,055AAh
	mov	dl,080h
	call	I_Int13
	jc	I_Srch		;No, search for disks without EDD.
	cmp	bx,0AA55h	;Did BIOS "reverse" our entry code?
	jne	I_Srch		;No, search for disks without EDD.
	xchg	ax,cx		;Set "EDD BIOS present" flag.
	and	al,004h
	mov	[EDDFlag],al
I_Srch	jmp	I_Scan		;Go search for UltraDMA disks to use.
;
; Subroutine to DISABLE output overlap in the resident driver.
;
I_DsbO	mov	byte [@OvlpF],0	;Zero resident driver "overlap" flag.
	cld			;Ensure FORWARD "string" commands!
	push	cs		;Point ES-register to our driver code.
	pop	es
	mov	al,090h		;No-op calls to overlap routines.
	mov	di,@NoOp3
	stosb
	mov	ax,0DB87h
	stosw
	mov	di,@NoOp4
	stosw
	stosw
	stosw
	mov	di,@NoOp1	;No-op "pass" call to timer routine.
	stosw
	stosw
	mov	al,090h
	stosb
	ret			;Exit.
