;*****************************************************************************
;** This is the 32bit ASM part of JEMM386.
;**
;** JEMM386 is a branch of FreeDOS Emm386, which in turn used the source of
;** an EMM made public in c't (a german IT magazine) in 08/90, page 214,  
;** written by Harald Albrecht.
;**
;** original author and copyright
;**   (c) 1990		 c't/Harald Albrecht
;**   (c) 2001-2004  tom ehlert
;**
;**	Licensed under the Artistic License version
;**	please see LICENSE.TXT for details
;**
;*****************************************************************************
;**
;** c't/Harald Albrecht: created the v86 monitor part and EMS functions
;** 1990 
;** 
;** Thomas Gloeckler, Rechenzentrum FH Ulm: DMA enhancements + fixes
;** 1990 
;**
;** junk@drivesnapshot.de: put into current shape as a potential EMM386 
;** 2001				   (EMM386-VCPI-DPMI-...)
;**
;** tom ehlert: english comments, ... 
;** 2001-2004	may have cut/copy/pasted ... things around
;**
;** Imre Leber: All German strings finally translated to English.
;** 2003		Merged code from Eric Auer.
;**
;** michael devore: implemented support for VCPI, VDS (partially), dynamic
;** 2004-2006		XMS memory allocation, EMS v4.0 (partially)
;** - Modified for >64M and VCPI support
;** - Updated for EMS 4.0, some extended API returns unsupported, this will
;**   change as requested or needed by other software requirements
;** - Documented EMS 4.0 only supports up to 32M (0x800 pages), but some EMS
;**   drivers allow more than 32M.	The EMS 4.0 spec could be extended to 1G-32K
;**   (map value of 0xFFFF means unmap) without breaking the spec API,
;**   but unfortunately breaking unsophisticated programs which get confused.
;**   I have arbitrarily decided to allow up to 32M of EMS allocations,
;**   leaving the high bit of page allocation alone since specific drivers
;**   may make use of high bit values other than 0xFFFF for their own purposes,
;**   or may only check high bit for unmap -- as FreeDOS EMM386 does.
;**	- Minor English corrections where useful for comprehension
;**
;**   Michael Devore's changes are not copyrighted and are released
;**   to the public domain.
;**   This does not affect copyright on the rest of the code.

;--   the following modifications are done for JEMM386:
;--   generally: 
;--    + comments were deleted if they were definitely wrong or outdated.
;--    + All modifications are public domain.
;--    + The source is now best viewed with Tab Size 4!
;--
;--   1. bugfix: changes done in proc pm_entry: switch to host stack
;-- 	 and back.
;--   2. bugfix: in proc VCPI_GetInterface: do not return a fixed value
;-- 	 in DI for the page table 0 offset, instead calculate this value
;-- 	 from the V86-monitor code descriptor. Return with an error
;-- 	 if page table 0 is "full". (obsolete).
;--   3. moved V86 monitor stack into shared extended memory.
;-- 	 temp stack removed. saves 1 kB of conv. memory.
;--   4. moved GDT into shared extended memory, thrown away LDT and 
;-- 	 temp. TSS.
;--   5. IO-Bitmap and IDT created on the fly. Reduces size of binary by
;-- 	 about 10 kB.
;--   6. rearranged segments to simplify binary structure.
;-- 	 RSEG		 resident code and data (=GROUP RESGRP)
;-- 	 V86SEG 	 protected-mode code (+data) moved to extended memory
;-- 	 INITSEG	 code for monitor initialization
;-- 	 _TEXT,_DATA real-mode code+data shared with the C part
;--   7. illegal opcode error handler rewritten which further reduces
;-- 	 conv. memory requirements.
;--   8. most of the VDS code moved to extended memory, runs in protected
;-- 	 mode now. The real-mode part (about 40h bytes) will not stay 
;-- 	 resident if option NOVDS is given (this has been made obsolete
;-- 	 by later development, the real-mode part is now just a few bytes).
;--   9. display more infos for exceptions occuring in ring 0.
;-- 	 page faults and GPFs will display the proper protected mode CS:EIP.
;-- 	 external exceptions (caused by intruders) may be detected and 
;-- 	 will prompt the user to reboot.
;--  10. code rearranged. now paging is enabled much earlier and
;-- 	 the monitor code will always be located at linear address 110000h
;-- 	 regardless where the physical address is. large parts of the
;-- 	 data moved to linear address 0FFExxxxx, to release PTEs in
;-- 	 page table 0. This also makes 2. obsolete.
;--  11. the full physical memory is no longer mapped in the monitor's 
;-- 	 address space. The Int 15h emulation has been adjusted and will
;-- 	 now always use scratch PTEs for the copy operation. This reduces 
;-- 	 extended memory usage a lot. On 80486+ cpus the INVLPG opcode
;-- 	 will be used to flush TLB entries.
;--  12. option NOALTBOOT deleted. It is obsolete now since IRQ 1 is never
;-- 	 hooked. Instead a HLT breakpoint has been added at FFFF:0000.
;--  13. option MEMCHECK deleted. It is obsolete now, the Int 15h emulation
;-- 	 will *always* build the PTEs on the fly. NOCHECK still checks if
;-- 	 the memory range is inside physical ram. Else a page fault is
;-- 	 generated.
;--  14. command line switch S=XXXX-YYYY added to allow Jemm386 to include
;-- 	 upper memory being activated by UMBPCI as UMBs. This allows UMBPCI
;-- 	 to be called before Himem and then removed entirely from DOS memory.
;--  15. if a DOS app hooks IVT vector 67h, the interrupt is now routed to
;-- 	 v86-mode (and catched at last by a breakpoint, which will transfer
;-- 	 control to the V86 monitor). Previously hooking IVT vector 67h in
;-- 	 real-mode was useless, because this interrupt was handled in 
;-- 	 protected-mode and the hook proc was never called. This is mainly 
;-- 	 to increase compatibility with MS Emm386.
;-------- v5.00pre
;--  16. implemented the EMMXXXX0 device, and allow some minimal
;-- 	 communication with it. Makes several apps happy, among them MS-DOS.
;--  17. DMA buffering code redesigned. Int 13h and 40h are no longer
;-- 	 hooked, instead it is checked in int 15h, ax=9100h and ax=9101h 
;-- 	 if the DMA buffer content should be transfered back (this was
;-- 	 reverted later, Int 13h and 40h are hooked again).
;--  17a.All variables handling DMA moved to V86SEG segment. Furthermore all
;-- 	 real-mode ints are hooked *after* the monitor has been installed.
;--  18. DMA bug fixed: the buffer is 64k max, but phys. addr. wasn't checked 
;-- 	 if it crosses a 64 kB boundary. Now the check is done and also size
;-- 	 is checked if it exceeds maximum size. (no error msg is displayed,
;-- 	 though). The DMA buffer size may be below 64 kB now (fixed with 26.).
;-- 	 Bugs still open: A truly safe buffer would beginn on a 128 kB border
;-- 	 and be 128 kB in size. It also must be located entirely in the first
;-- 	 16 MB (should be no problem usually).
;--  19. DMA: Single Mask and Master Mask ports also trapped now. It is
;-- 	 checked if the channel is active, and nothing is done unless it is.
;--  20. DMA bugfix: the DMA buffer is not always used when it has to be
;-- 	 used. The CONTINUOUS? proc was not strict enough.
;--  21. bugfix accessing XMS handle array: unused handles got the FREE flag
;-- 	 set (bit 0). Now unused handles got the "INPOOL" flag set (bit 2).
;-- 	 This makes JEMM386 work better with MS Himem (at least).
;--  22. bugfix: mapped page at F0000 (to catch RESET) wasn't write protected.
;-- 	 The fix requires to catch page faults caused by write accesses and
;-- 	 "emulate" them.
;--  23. display FS and GS in invalid opcode handler and wait for ESC key
;-- 	 before trying to abort the current program with int 21h, ah=4Ch.
;--  24. A20 disable/enable emulation activated
;--  25. VME extension supported. Software interrupt redirection bitmap 
;-- 	 added to the TSS. Detection of CPUID opcode support. If VME
;-- 	 extensions are present (Pentium+), the VME bit in CR4 is set.
;--  26. DMA buffer is now 64 kB in size again and begins on a physical
;-- 	 64 kB border. This is achieved by rearranging PTEs. 
;--  27. the EMMXXXX0 device interface may be used to query the current EMM 
;-- 	 status.
;-------- v5.00
;--  28. bugfix: the VDS disable/enable DMA translation cannot be dummies.
;-- 	 They must decrease/increase a counter and if zero the automatic
;-- 	 translation for the channel must be disabled.
;--  29. debug display of "unsupported" VDS functions removed. These functions
;-- 	 will only succeed if a valid buffer ID has been supplied, which 
;-- 	 currently is never the case. Now just the proper error code is
;-- 	 returned.
;-------- v5.01
;--  30. support for variable DMA buffer size added.
;--  31. VDS functions 07/08 (req/rel buffer) and 09/0A (copy in/out of 
;-- 	 DMA buffer) implemented. VDS functions 03/04 (lock/unlock region)
;-- 	 now really implemented (they will use functions 07/08 if needed). 
;--  32. option ALTBOOT implemented. Uses Int 15h, ah=4Fh as keyboard hook.
;--  33. bugfix: install an XMS handler even if no UMBs are supplied, since
;-- 	 the XMS handler is also used for A20 handling.
;--  34. bugfix: NoVCPI no longer requires NoEMS to work. Why was this ever
;-- 	 implemented?
;--  35. bugfix: INT 67h, AH=4Eh, Subfunc 2 didn't set state.
;--  36. INT 67h, AH=4Fh implemented
;--  37. bugfix: INT 67h, AH=43h left EMM in an inconsistent state if it 
;-- 	 ran out of free pages during the allocation.
;-------- v5.10
;--  38. bugfix: ALTBOOT not set and a write into FF00 segment caused a
;-- 	 page fault which was not handled correctly, causing crash.
;--  39. EMS/VCPI memory management improved, no space wasted.
;--  40. 32 MB limit for EMS may be overcome with MIN= option.
;--  41. if pool sharing cannot be activated and MIN= parameter is not
;-- 	 set in command line, a reasonable assumption is made about amount 
;-- 	 of EMS/VCPI memory to allocate.
;--  42. debug messages during protected-mode initialization implemented.
;--  43. JEMM386.ASM can be assembled with MASM 6.1
;--  44. protected-mode segments V86SEG and INITSEG made 32-bit.
;--  45. save/restore CR2 on page faults at page FF (ALTBOOT).
;-- 	 (reverted in v5.21 since it was useless).
;--  46. bugfix: if pool sharing is on, A20 disabled and Himem has placed
;-- 	 its handle table into the HMA ...
;--  47. A20 emulation now (also) done on the "port" level. The XMS hook
;-- 	 still is useful if XMS host has decided not to disable A20 ever
;-- 	 (because it has found A20 on on startup).
;--  48. all assembler code in JEMM386C.C moved to JEMM386.ASM to make
;-- 	 JEMM386C.C compileable with other compilers (Open Watcom, MS VC).
;-- 	 Compiler specific helper functions for DWORD div/mod/mul/shl/shr
;-- 	 added.
;--  49. bugfix: some EMS functions (int 67h, ah=57h) destroyed hiword(eax).
;--  50. int 67h, ax=5900 "implemented".
;--  51. bugfix: in MAP_PAGE, the logical page to map may be invalid
;-- 	 (because the page was mapped into the page frame, but its handle
;-- 	 has long been released). Then do not try to get pool information
;-- 	 for this page, this may map "random" pages into the page frame.
;-- 	 Instead, unmap the page, which will result in linear=phys.
;-------- v5.20
;--  52. check CX parameter is not > 8000h in int 15h, ah=87h emulation.
;--  53. LOAD command line parameter support added.
;-------- v5.21
;--  54. bugfix: EMS realloc may have changed logical page order of a handle.
;-------- v5.22
;--  55. close files before going resident as TSR.
;--  56. implemented VDS 8105 (scatter lock) with bit 6 of DL set 
;-- 	 (returning PTEs).	
;--  57. EMX option added.
;-------- v5.23
;--  58. bugfix: int 67h, ax=5001h, didn't work, but may have reported
;-- 	 success.
;--  59. PGE option added.
;-------- v5.24
;--  60. bugfix: if Ctrl-Alt-Del is detected the real-mode int 15h chain 
;-- 	 is called first before rebooting (allows SmartDrv and others to
;-- 	 cleanup).
;-------- v5.25
;--  61. bugfix: VDS Request/Release Buffer called VDS Copy In/Out of buffer,
;-- 	 but the latter has an additional parameter (offset in buffer in 
;-- 	 BX:CX), which has to be set to 0 first.
;--  62. bugfix: VDS Request Buffer was unable to alloc the last kb of the
;-- 	 DMA buffer.
;--  63. bugfix: VDS Release Buffer was unable to release a buffer if size
;-- 	 was < 400h and > 0.
;-------- v5.26
;--  64. bugfix: int 67h, ah=5Ah with BX=0 didn't return a valid handle in DX
;--  65. bugfix: int 67h, ah=51h destroyed content of AL register
;--  66. bugfix: int 67h, ah=51h may have called function TryFreeToXMS with
;-- 	 DF set, which might have caused data corruption.
;-------- v5.27
;--  67. bugfix: int 4Bh, ax=8105h (scatter lock): if bits 0-11 of offset in
;-- 	 EDDS were <> 000h, the first page was returned as a separate region.
;--  68. bugfix: int 4Bh, ax=8105h (scatter lock): if this function failed
;-- 	 with AL=9, it should have set field NumUsed in EDDS with a value 
;-- 	 required to describe the full region. This wasn't always the case.
;--  69. bugfix: Int 67h, ah=53h and 54h accessed (nonexistant) name table
;-- 	 entries. Since it is possible to alloc a handle even with NOEMS,
;-- 	 the name table must always exist. 
;--  70. bugfix: int 67h, ah=57h (move to/from expanded memory) didn't work
;-- 	 without a page frame.
;-------- v5.28
;--  71. int 67h, ah=58h now works with NOEMS.
;--  72. error code of int 67h, ah=41h, 47h, 48h if no page frame is defined
;-- 	 changed to 80h from 91h, because 91h seems to be defined for EMS
;-- 	 v4.0 only.
;--  73. NOINVLPG option added.
;-------- v5.29
;--  74. int 67h, ah=55h/56h implemented.
;--  75. bugfix: int 67h, ah=57h didn't always flush the correct TLB entries.
;--  76. bugfix: int 67h, ax=4F01h didn't work.
;-------- v5.30
;--  77. bugfix int 67h, ah=57h: test for overlapping EMS regions didn't
;-- 	 work.
;--  78. bugfix int 67h, ah=57h: conventional memory regions weren't tested 
;-- 	 if they exceed the 1 MB boundary, in which case error A2h should be
;-- 	 returned.
;--  79. bugfix: int 67h, ah=57h didn't return status 92h if an overlapp
;-- 	 occured (92h is a hint that part of the source has been overwritten).
;--  80. bugfix: if an exception occured in the v86-monitor, the register
;-- 	 contents were displayed wrongly in v5.30.
;--  81. exceptions occured in protected-mode are now displayed the same way
;-- 	 as the v86-exceptions, that is, through int 29h in v86-mode. The 
;-- 	 previous solution (writing directly into the video screen buffer  
;-- 	 from protected-mode) didn't work if screen was in graphics mode.
;--  82. handler for v86-int06 (invalid opcode exception) moved to protected
;-- 	 mode, reduces monitor's conventional memory usage to 192 bytes.
;-------- v5.31
;--  83. removed an optimisation in MAP_PAGE, which caused problems with	
;-- 	 PKZIP/PKUNZIP.
;-------- v5.32
;--  84. changes in REBOOT (some old machines still didn't reboot with
;-- 	 Ctrl-Alt-Del).
;--  85. monitor stack, GDT and IDT added to V86SEG. Thus the extra
;-- 	 monitor stack segment is no longer needed, and SS can be used
;-- 	 to access monitor data. Small catch: file size increases by about
;-- 	 2,5 kB because stack, GDT and IDT are now a static part of the
;-- 	 binary.
;--  86. V86SEG + INITSEG grouped into one 32bit segment (_TEXT32),
;-- 	 and RSEG + _TEXT grouped into one 16bit code segment (_TEXT16).
;-- 	 This finally reduces number of physical segments to 3:
;-- 	 _TEXT16: 16 bit code (real-mode)
;-- 	 _TEXT32: 32 bit code+data (protected-mode)
;-- 	  DGROUP: 16 bit data (real-mode)
;--  87. bugfix: EMS might have been disabled due to a query with a
;-- 	 wrong segment register assume.
;-------- v5.33
;--  88. bugfix: ALTBOOT was erroneously always enabled since int 15h
;-- 	 is always hooked in protected-mode.
;-------- v5.34
;--  89. bugfix: msg "unable to continue - please reboot" wasn't displayed
;-- 	 in versions 5.31-5.34
;--  90. bugfix: clearing "Dirty" and "Accessed" bits in PTEs of page 
;-- 	 table 0 on initialization didn't work in versions 5.24-5.34
;--  91. bugfix: int15h, ah=87h emulation cleared TF, not CF
;--  92. bugfix: exc06 display did always print the v86 exc type display
;--      (that is, with segment registers), although they weren't rendered.
;--  93. ASM source splitted into Jemm32.asm (32bit) and Jemm16.asm.
;-- 	 The 32bit part is now true FLAT and assembled+linked in a
;--      separate module in Win32 COFF format.
;--  94. NOHI option implemented.
;--  95. ExecIntV86 implemented to execute v86 interrupts directly from inside 
;--      the monitor, which makes some hacks unnecessary and further reduces
;--      conventional memory usage.
;-------- v5.4
;--  96. "int 67h, ah=87h" hack removed. The int 15h extended memory block 
;--      move emulation code is now called the same way as the other code,
;--      by a v86 breakpoint.
;--  97. Jemm386 made unloadable.
;-------- v5.45
;--  98. bugfix: 16-bit DMA channel I/O was wrong (wrong I/O port for
;--      base address calculated in Dma_Checkchannel.
;--  99. bugfix: 2 lines 'disappeared' in v5.45, making Jemm to always
;--      display "can't continue" message.
;-- 100. FASTBOOT option implemented.
;-------- v5.46
;*****************************************************************************
		TITLE	JEMM386 - Virtual 8086-Monitor for 80386 PCs
		NAME	JEMM386
;
; (c) 1990 c't/Harald Albrecht
; (c) 2001 tom.ehlert@ginko.de
;
; to be assembled with MASM 6.1 or TASM 4.1/5.0
;
; LIM-EMS-Driver for 386 PCs (EMS Version 4.0)
;
; (c) 1989 Harald Albrecht
;
		.486P

		include jemm386.inc		;common declarations
		include jemm32.inc		;declarations for Jemm32
		include debug.inc

ife ?MASM
		LOCALS
else
		option proc:private
endif

;--- equates

?FLAT		equ 0		; 1=use the .model FLAT directive
?WT			equ 0		; set WT bit in PTE on int 15h, ah=87 emulation
device_name equ 000Ah	; offset device name in RSEG
?BASE		equ 110000h	; base of binary
?FASTMON	equ 1		; 1=faster monitor jmp for int 00-0F
?FREEXMS	equ 1		; 1=free all XMS on exit

if ?FASTMON
?INTTABSIZ	equ 0E0h * 7
else
?INTTABSIZ	equ 100h * 7
endif

;--- publics/externals

; no entries anymore!

;--- segment definitions

if ?FLAT
		.model FLAT
		.code
else
  if ?MASM
  		option dotname
  endif      
.text   SEGMENT PARA USE32 public 'CODE'	;protected mode code + data
.text	ENDS
.text	SEGMENT
  ife ?MASM
		assume CS:.text
  endif      
		assume SS:.text
		assume DS:.text
		assume ES:.text
endif

;--- the binary's entry must be at offset 0!

ife ?MASM
		public _start
endif        
_start:
		jmp InitMonitor
		db "TAK"
		dd (V86_TOS-8)/4 dup ("KATS")	;monitor stack

; GDT - Global Descriptor Table

descriptor struc
		dw ?
		dw ?
		db ?
		db ?
		db ?
		db ?
descriptor ends 	   

;--- FLAT_CODE_SEL must be first descriptor in GDT
;--- Furthermore, Jemm16.asm expects GDT located just behind the stack.
;--- if this changes, Jemm16.asm has to be adjusted!

TSS_LEN equ 68h+2000h+8+20h				; size of TSS "segment"

GDT 	LABEL BYTE
	DQ 0								; NULL-Entry
	descriptor <-1,0,0,9AH,0CFh,0>		; flat 4 GB Code (32 Bit)
	descriptor <-1,0,0,92H,0CFh,0>		; flat 4 GB Data (32 Bit)
	descriptor <TSS_LEN-1,0,0,89H,0,0>	; TSS V86
	descriptor <-1,0,0,9AH,0,0>			; std 64k Code Descriptor
	descriptor <-1,0,0,92H,0,0>			; std 64k Data Descriptor
if ?CLEARHIGHESP
	descriptor <V86_TOS-1,0,?BASE shr 16,92H,040h,?BASE shr 24>; monitor stack segment
endif

FLAT_CODE_SEL	equ 1 * size descriptor
FLAT_DATA_SEL	equ 2 * size descriptor
V86_TSS_SEL 	equ 3 * size descriptor
REAL_CODE_SEL	equ 4 * size descriptor
REAL_DATA_SEL	equ 5 * size descriptor
if ?CLEARHIGHESP
V86_STACK_SEL	equ 6 * size descriptor
endif

GDT_SIZE	equ 200h

; IDT - Interrupt Descriptor Table

	ORG V86_TOS+GDT_SIZE
	
IDT 	LABEL BYTE

	DQ 100h dup (0)

;--- global monitor data

_CR3	DD 0				; CR3 for monitor address context

IDT_PTR LABEL FWORD			; size+base IDT
		DW 7FFH
		DD offset IDT
wMasterPICBase  dw 0008h	; put it here for alignment
		
GDT_PTR LABEL FWORD			; size+base GDT
		DW GDT_SIZE-1
		DD offset GDT
wSlavePICBase   dw 0070h	; put it here for alignment
		
dwFeatures	dd	0			; features from CPUID 

dwTotalMemory	dd 0		; XMS highest address (r/o)				
dwMaxMem16K		DD 0		; maximum amount of VCPI memory in 16K
OldInt06		dd 0
OldInt67		dd 0
if ?VDS
OldInt4B		dd 0
endif

;--- EMS variables

?PHYSPG			equ 4	; max physical pages (don't set below 4!)

EMSSTATUSTABLE	DD	0	; table of EMS handle status (size 2*4/handle)
EMSNAMETABLE	DD	0	; table of EMS handle names (8 bytes/handle)
EMSPAGETABLE	DD	0	; table of EMS pages (1 byte/EMS page)
FIRSTPAGE		DD	0	; start EMS pages (physical address!)
dwMaxEMSPages	DD	0	; total EMS pages
EMSPAGESAVAIL	DD	0	; free EMS pages

PAGE2SEGM		DB	?PHYSPG DUP (0)		; segments mapped to phys.pgs
PHYSPAGETABLE	DW	?PHYSPG DUP (-1)	; log. EMS pgs mapped to phys. pgs

				align 4

;--- table of EMS page descriptors (EMSPD), one item for each EMS page
;--- each item consists of 2 words, addressing a nibble in PoolAllocationTable
;--- initialized with -1

EMSPageAllocationStart DD	0
EMSPageAllocationEnd   DD	0

;-- address of EMS/VCPI allocation block start
;-- each item consists of 64 bytes

PoolAllocationTable 	DD	0
PoolAllocationEnd		DD	0

LastBlockAllocator		DD 0	; pointer to last block allocated from
LastBlockFreed			DD 0	; pointer to last block freed from
XMSBlockSize			DD 0	; current size of XMS block to allocate for sharing, in K
XMSPoolBlockCount		DD 0	; count of XMS handles allocated for pool blocks (not counting initial UMB block)

if ?DMA or ?VDS

			align 4

DMABuffStart	DD	0	; Beginning of the DMA-Buffer (linear address)
DMABuffSize 	DD	0	; max size of the DMA-Buffer in bytes
DMABuffStartPhys DD 0	; Beginning of the DMA-Buffer (physical address)
DMABuffFree 	DD ?DMABUFFMAX/32 dup (0)	; bits for DMA buffer handling
				db 0	; space for 2 additional bits is needed!
			align 4
endif

if ?DMA

DMAREQ struc		
ChanMode	db ?	;saved "mode" value (register 0Bh or D6h)
bFlags		db ?	;flags, see below
BlockLen	dw ?	;saved "block length" value
BaseAdr 	dw ?	;saved "block address" value
PageReg 	db ?	;saved "page register" value
cDisable	db ?	;is automatic translation disabled? (VDS)
DMAREQ ends

;bFlags values

DMAF_ENABLED	equ 1	;1=channel is enabled/unmasked


DmaTargetAdr	DD	0	; Original adress for DMA
DmaTargetLen	DD	0	; Utilized part of the Buffer

DmaChn			DMAREQ 8 dup (<0,0,0,0,0,0>)
DmaGblFlags		DW 0	; global flags
				align 4

endif

dwRes			dd	0	; linear address of resident segment
dwRFlags		dd	0	; linear address bRFlags
dwRSeg			label dword
wRSeg			dw	0	; resident segment
				dw  0
dwReqPtr		dd	0	; device strategy request ptr
if ?FASTBOOT
pSavedVecs		dd 0	;vectors saved during startup
dwInt19Org		dd 0	;original int 19 vector saved by DOS
endif


bEmsPhysPages	DB	0	; current physical pages

;-- byte vars (variable)

if ?A20PORTS
bLast64			DB	0
endif
if ?A20XMS
wA20			dw 1	; local + global XMS A20 count
endif
if ?A20PORTS or ?A20XMS
bNoA20			DB	0
bA20Stat		DB	1	; default is A20 enabled
endif

;-- byte vars (rather constant)

bNoEMS			DB	0	; flags no EMS services
bNoFrame		DB	0	; flags no page frame
bNoVCPI			DB	0	; flags no VCPI services
bNoPool			DB	0	; flags pooling with XMS memory
bNoInvlPg		DB	-1	; <> 0 -> dont use INVLPG
bV86Flags		DB	0	; other flags 
bFatal			db  0	; fatal exception, dont call DOS

V86F_SB			equ 1	; soundblaster driver compat
V86F_NOCHECK	equ 2	; flag NOCHECK option
if ?EMX
V86F_EMX		equ 4	; EMX compat
endif
if ?FASTBOOT
V86F_FASTBOOT	equ 8	; fastboot active
endif

; Flag to enable/disable special
; handling of INT67/44 (MAP_PAGE)
; during init phase, abused to map UMB everywhere
bInitDone		db	0	; reset after initialization
bIs486			db	0	; cpu is 486 or better (INVLPG opcode available)
if ?PGE
bPageMask		db	0	;mask for PTEs in 0-110000h
endif
bBpTab			db  0	;offset start bptable in RSEG
bBpBack 		db  BPBACK

		align	4

bpstart	dd 0	;linear address start ob bp table

RunEmuInstr:
EmuInstr	DB	90h,90h,90h	; run self-modifying code here
		ret
		
;--- place rarely modified data here
;--- so data and code is separated by at least 64 "constant" bytes

if ?ROMRO
dwSavedRomPTE dd 0	;saved PTE for FF000 page faults
dqSavedInt01  dq 0	;saved INT 01
endif

UMBsegments	UMBBLK UMB_MAX_BLOCKS dup (<>)
XMS_Handle_Table	XMS_HANDLE_TABLE_STRUC	<0,0,0,0>
XMSControlHandle dw 0

	align 16

;--- start code
	
if ?CODEIN2PAGE
	ORG 1000h	;locate code in 2. page (so data and code are separated)
endif

if ?VCPI

;--- the following 3 procs must be located in the first page.
;--- If this is no longer possible,
;--- the VCPI shared address space has to be increased.

; VCPI switch V86 mode to protected mode
; inp: AX=DE0C
;     ESI -> linear address of "v86 to protected-mode switch" data
;   set by EMM entry code:
;     ESP -> ecx, edi, esi
;     DS,ES=FLAT; FS,GS=NULL (done by cpu)
; out: GDTR, IDTR, LDTR, TR loaded for client
;     SS:ESP=V86_TOS (must provide at least 16 byte space for client)
;                    HIWORD(ESP) is cleared
; modifies EAX, ESI
;

?CLEARTB	equ 1

VCPI_V86toPM	PROC

	mov ecx,[ebp].EMMFRAME.eEcx		; restore ECX
    pop ebp							; restore EBP
if ?CLEARHIGHESP
	mov ax, V86_STACK_SEL 			; some poorly written VCPI clients expect
    mov ss, eax     				; hiword(esp) to be cleared.
endif
	mov	eax,[esi].V862PM.swCR3	; set client's context *first*
	mov	cr3,eax

	mov	eax,[esi].V862PM.swIDTOFFS	; set up client's IDT
	lidt fword ptr [eax]
	mov	eax,[esi].V862PM.swGDTOFFS	; set up client's GDT
	movzx esp,[esi].V862PM.swTR
	lgdt fword ptr [eax]
if ?CLEARTB    
	mov	eax,[eax+2] 				; EAX == linear address of client's GDT
	and	BYTE PTR ds:[esp+eax+5],NOT 2	; clear task busy bit in TSS descriptor
endif    
	ltr	sp							; set up client's TSS
	lldt [esi].V862PM.swLDTR		; set up client's LDT
if ?CLEARHIGHESP
	mov sp,V86_TOS
else
	mov esp,?BASE+V86_TOS
endif
	jmp	[esi].V862PM.swCSEIP		; jump to client's entry point

VCPI_V86toPM	ENDP

	align 4
	
; VCPI switch protected mode to v86 mode
;  inp: SS:ESP set for IRETD to V86, PM far call return address to discard;
;       stack must be in shared address space.
;       AX=DE0C
;		DS -> selector containing full shared address space 0-11xxxx
;  out: CR3, GDTR, IDTR, LDTR, TR loaded for server,
;	 IRETD will load SS:ESP and all segment registers, then enable v86 mode.
;  modifies: EAX
;
;-- for speed reasons the following lines are a bit mixed.
;-- but what is done is simple:
;-- 1. switch to host context (SS:ESO will stay valid, stack is in 1. MB)
;-- 2. load IDTR, GDTR, LDTR, TR for VCPI host
;-- 3. clear task busy bit in host's TSS descriptor
;-- 4. clear task switch flag in CR0
;-- 5. IRETD will switch to v86 mode, interrupts disabled

VCPI_PMtoV86	PROC

	cli					; client should have disabled interrupts, but not all do
	add	esp,5*4			; position ESP to EFL on stack
	mov	eax,[_CR3]
    push 23002h			; set flags VM=1, IOPL=3, IF=0
	mov	cr3,eax
    sub esp,2*4
	lgdt	fword ptr [GDT_PTR]
	xor	eax,eax
	lidt	fword ptr [IDT_PTR]
	lldt ax
	and	BYTE PTR [GDT+V86_TSS_SEL+5],NOT 2
	mov	al,V86_TSS_SEL
	clts
	ltr	ax
	iretd

VCPI_PMtoV86	ENDP


;-- put the VCPI PM entry at the very beginning. This entry
;-- must be in the shared address space, and just the first page
;-- is shared!
;-- entry for EMM routines called from protected mode

	align 4

VCPI_PM_Entry	PROC

	cmp	ax,0de0ch		; see if switch from protected mode to V86 mode
	je	VCPI_PMtoV86	; yes, give it special handling
	pushfd
	push	ds			; have to save segments for p-mode entry
	push	es

	push	esi
	push	edi
	push	ecx

	mov	ecx,cs
	add	ecx,8
	mov	ds,ecx			; load DS,ES with a copy of FLAT_DATA_SEL
	mov	es,ecx

;--- segment registers should *all* be set *before* switching contexts!
;--- why? because GDT of client might be located in unreachable space.
;--- get client SS:ESP + CR3, switch to host stack and context

	mov		edi, V86_TOS + ?BASE - 3*4
	mov		esi, [_CR3]
	cli					; don't allow interruptions
    mov		[edi+4],esp
    mov		[edi+8],ss
	mov		ss, ecx
	mov		esp, edi
	mov		ecx, cr3
	mov		cr3, esi
	mov 	[edi+0],ecx ;save client CR3

if ?SAFEMODE
	sub		esp,8
	sgdt	[esp]
	lgdt	fword ptr [GDT_PTR]
	mov 	cx, FLAT_DATA_SEL
    mov		ss, ecx
	mov		ds, ecx
	mov		es, ecx
endif

	cld

if ?VCPIDBG
	@DbgOutS <"VCPI call in PM, ax=">,1
	@DbgOutW ax,1
	@DbgOutS <10>,1
endif

	cmp	ah,0deh
	jne	@@INV_CALL	; only allow VCPI calls from protected mode interface

; other than de0ch, don't allow anything other than de03h,de04h,de05h from PM
	cmp	al,3
	jb	@@INV_CALL
	cmp	al,5
	ja	@@INV_CALL

	movzx	ecx,al
	call	[VCPI_CALL_TABLE+ECX*4]

@@PM_ret:
if ?VCPIDBG
	lgdt fword ptr [esp]
	add esp,8
endif
	pop	ecx
if 1	
	mov	cr3,ecx
	lss esp,[esp]
else	
	pop esi
	pop edi
	mov	cr3,ecx		; restore client context *first* and
	mov ss, edi		; then restore client SS:ESP, thus client's GDT
	mov esp, esi	; need not be located in shared address space
endif	 
	pop	ecx
	pop edi
	pop esi
	pop	es
	pop	ds
	popfd
	retf
@@INV_CALL:
	mov	ah,8fh			; use bad subfunction code, per VCPI spec
	jmp @@PM_ret
    
VCPI_PM_Entry	ENDP

endif	 

	align 4
	
EMS_CALL_TABLE label dword
	Dd EMS_GET_STATUS			; 40h
	Dd EMS_GET_PAGE_FRAME_ADDRESS		;41h
	Dd EMS_GET_UNALLOCATED_PAGE_COUNT	;42h
	Dd EMS_ALLOCATE_PAGES		; 43h
	Dd EMS_MAP_HANDLE_PAGE		; 44h
	Dd EMS_DEALLOCATE_PAGES		; 45h
	Dd EMS_GET_VERSION			; 46h
	Dd EMS_SAVE_PAGES			; 47h
	Dd EMS_RESTORE_PAGES		; 48h
	Dd EMS_NOT_IMPL				; 49h (get io port addresses)
	Dd EMS_NOT_IMPL				; 4ah (get translation array)
	Dd EMS_GET_OPEN_HANDLES_COUNT		; 4bh
	Dd EMS_GET_NR_OF_ALLOCATED_PAGES	; 4ch
	Dd EMS_GET_ALLOCATED_PAGES	; 4dh
	Dd EMS_GET_SET_PAGE_MAP		; 4eh	
	Dd ems4_get_set_partial_page_map	; 4fh (4: get/set partial page map)
	Dd ems4_map_multi			; 50h
	Dd ems4_realloc				; 51h
	Dd ems4_attribute			; 52h
	Dd ems4_handle_names		; 53h
	Dd ems4_get_handle_info		; 54h
	Dd ems4_alter_map_jump		; 55h (4: alter page map and jump)
	Dd ems4_alter_map_call		; 56h (4: alter page map and call)
	Dd ems4_memory_region		; 57h
	Dd ems4_get_mappable_info	; 58h
	Dd ems4_get_config			; 59h
	Dd ems4_allocate_pages		; 5ah
	Dd EMS_NOT_IMPL				; 5bh (4: alternate map register set)
	Dd EMS_NOT_IMPL				; 5ch (4: prepare EMS for warm boot)
	Dd EMS_NOT_IMPL				; 5dh (4: enable/disable OS functions)
EMS_MAX equ ($ - EMS_CALL_TABLE) / 4

if ?VCPI

VCPI_CALL_TABLE label dword
	Dd VCPI_Presence	; 0
	Dd VCPI_GetInterface
	Dd VCPI_GetMax		; 2
	Dd VCPI_GetFree
	Dd VCPI_Allocate4K	; 4
	Dd VCPI_Free4K
	Dd VCPI_GetAddress	; 6
	Dd VCPI_GetCR0
	Dd VCPI_ReadDR		; 8
	Dd VCPI_WriteDR
	Dd VCPI_GetMappings	; 0ah
	Dd VCPI_SetMappings
;	Dd VCPI_V86toPM		; 0ch
VCPI_MAX equ ($ - VCPI_CALL_TABLE) / 4
endif

if ?VDS

VDS_CALL_TABLE label dword
	Dd vds_reserved		; 00
	Dd vds_reserved		; 01
	Dd vds_version		; 02 get version
	Dd vds_lock			; 03 lock region, ES:DI -> DDS	
	Dd vds_unlock		; 04 unlock region, ES:DI -> DDS
	Dd vds_scatterlock	; 05 scatter lock, ES:DI -> EDDS
	Dd vds_scatterunlock; 06 scatter unlock, ES:DI -> EDDS
	Dd vds_reqbuff		; 07 request DMA buffer, ES:DI -> DDS
	Dd vds_relbuff		; 08 release DMA buffer, ES:DI -> DDS
	Dd vds_copyinbuff	; 09 copy into DMA buffer, ES:DI -> DDS
	Dd vds_copyoutbuff	; 0A copy out of DMA buffer, ES:DI -> DDS
	Dd vds_disabletrans	; 0B disable DMA translation
	Dd vds_enabletrans	; 0C enable DMA translation
VDS_MAX equ ($ - VDS_CALL_TABLE) / 4
endif
	

if ?FASTMON

?FASTENTRIES 	equ 32

@FASTMON macro x
	push x
    jmp short V86_Monitor
	endm

	align 4
int00 label byte
	INTNO = 0
	REPT ?FASTENTRIES
		@FASTMON INTNO
        INTNO = INTNO+1
    ENDM
endif

	align 4

@v86pushreg macro
	PUSH	EAX
	PUSH	EBX
	PUSH	ECX
    endm
@v86popreg macro
	POP		ECX
    POP		EBX
    POP		EAX
    endm

; all entries in the IDT (except 15h and 67h) jump to V86_Monitor.
; It has to check if an exception has occured. If no, just reflect the 
; interrupt to v86-mode. If yes, check what the reason for the exception
; was and do the appropriate actions.
;
; inp: the cpu has set DS,ES,FS,GS==NULL if switch from v86 mode!
;  [ESP+0] = INT#

V86_Monitor PROC

	@v86pushreg
    
;-- we don't know for sure what size the stack is.
;-- There might exist other ring0 code which intruded in the  
;-- monitors address context and modified GDT/IDT entries
;-- But at least they should have established a 32-bit stack or
;-- at the very least cleared HiWord(ESP).

	MOVZX	ECX,byte ptr [ESP].V86FRAME.fIntNo  ;load 1 byte ONLY!

; Three cases are considered:
; - INT executed in V86 mode     ESP = ?BASE+V86_TOS - size V86FRAME
; - exception in V86 mode        ESP = ?BASE+V86_TOS - size V86FRGP
; - EXC/IRQ (HLT) in v86 monitor ESP < ?BASE+V86_TOS - size V86FRGP

	CMP 	ESP, ?BASE + V86_TOS - size V86FRGP	   ; What did happen?
	JZ		V86_Exception 		; an exception in V86 mode
	JB		@@IRQOREXC			; Monitor is being interrupted (IRQ or EXC)

;--- this is almost the same code as for SimIntV86, but this one uses
;--- a V86FRAME stack frame instead of V86FRGP

	MOVZX	EBX, word ptr [ESP].V86FRAME.fSS
	MOV		EAX, [ESP].V86FRAME.fESP
	SHL 	EBX,4
	SUB		AX, 6					; Create space for IRET frame
	MOV 	[ESP].V86FRAME.fESP,EAX
	MOVZX	EAX,AX					; use LOWORD(ESP) only!
	ADD 	EBX,EAX

; copy Interrupt frame down. Use SS, since DS is unset

	MOV 	EAX,[ESP].V86FRAME.fCS
	SHL		EAX, 16
	MOV 	AX,word ptr [ESP].V86FRAME.fEIP
	MOV 	SS:[EBX+0],EAX
	MOV 	EAX,[ESP].V86FRAME.fEFL
	MOV 	SS:[EBX+4],AX
	AND		AH, NOT (1 or 2)		; Clear IF+TF as it is done in real-mode
	MOV 	EBX,SS:[ECX*4]			; route call to vector in real-mode IVT
	MOV 	word ptr [ESP].V86FRAME.fEIP,BX
	SHR 	EBX, 16
	MOV 	[ESP].V86FRAME.fCS, EBX
	MOV 	[ESP].V86FRAME.fEFL, EAX

	@v86popreg				; Clean everything properly
	add 	ESP,4			; skip int#
	IRETD					; Return to virtual 86-Modus

	align 4

;-- IRQ or exception in ring 0
;-- an exception may be with or without error code
;-- an IRQ will have offset "BEHINDHLT" as EIP and FLAT_CODE_SEL as CS

@@IRQOREXC:

if ?V86DBG
	@DbgOutS <"V86 IRQ/EXC, cx=">,1
	@DbgOutW cx,1
	@DbgOutS <" CS=">,1
	mov ax, word ptr [esp+3*4+4+4]
	@DbgOutW ax,1
	@DbgOutS <10>,1
endif
	cmp		[esp].V86FRAME.fEIP,offset BEHINDHLT
	JNZ		ring0_exc
	cmp		word ptr [esp].V86FRAME.fCS,FLAT_CODE_SEL
	JNZ		ring0_exc

;-- ESP before halt -> IRETD frame [eip, cs, efl, esp, ss, es, ds, fs, gs)
;-- now 7 dword are additionally pushed: ECX, EBX, EAX, INT#, EIP, CS, EFL
;-- that gives 16*4=64 bytes on stack

if ?HLTDBG
	@DbgOutS <"HLT triggered, int=">,1
	@DbgOutW cx,1
	@DbgOutS <10>,1
endif
	MOV 	[ESP+6*4],ECX		 ; set INT# in correct position
	pop		ecx					 ; restore ECX
	add		esp, 5*4			 ; skip EBX, EAX, INT#, EIP, CS
	jmp		V86_Monitor 		 ; and restart

V86_Monitor 	ENDP

;--- handle exc 06 in v86-mode

HandleExc06V86 proc
	push	ebp
    mov		ebp,esp
	call	SimIRET
	mov		[ebp+4].V86FRGP.gIntNo,6
    pop		ebp
    @v86popreg
	jmp		errorcode_pushed
HandleExc06V86 endp

EXCFR struc
	PUSHADS <>
	V86FRGP <>
EXCFR ends

;--- handle exceptions (ring 0 protected-mode and v86-mode)
;--- display current register set
;--- if it is caused by external protected-mode code, display REBOOT option
;--- else jump to v86-mode and try to abort current PSP

;--- handle exceptions in ring0 protected-mode

ring0_exc proc
ring0_exc endp

HandleException proc
	mov		eax,ss					; the stack size is not known
	lar		eax, eax
	test	eax,400000h
	jnz		@@is32
	movzx	esp,sp					; make sure ESP is valid
@@is32:
    @v86popreg

	cmp		dword ptr [esp],8		; exc #8?
	jnc		errorcode_pushed
	push	dword ptr [esp]			; emulate error code for exc 0-7 
;	 mov		dword ptr [esp+4],0
if ?MASM
errorcode_pushed::
else
errorcode_pushed:
endif
	@v86pushreg
	pushad
	mov		ebp,esp

	mov		ax, FLAT_DATA_SEL
	mov		ds, eax
	mov 	ES, eax
	cld

	test	byte ptr [ebp].EXCFR.gEFL+2,2	;V86 mode?
	jnz		@@isv86
    mov		byte ptr [exc_ds],' '
	lar		eax, [ebp].EXCFR.gCS
	and 	ah,60h
	jnz 	@@isring3
	mov		ebx, ss
	lea		ecx,[ebp + size PUSHADS +3*4 +4 +4*4]	;is this correct?
	jmp		@@isring0
@@isv86:	
	movzx	esi,word ptr [ebp].EXCFR.gCS
	shl		esi,4
	add		esi,[ebp].EXCFR.gEIP
	mov		cl,8
	mov		edi,offset exc_csip
@@nextitem:
	lods	byte ptr [esi]
	call	b2a
	inc		edi
	dec		cl
	jnz		@@nextitem
	push	offset exc_v86segregs	;render V86 segment registers
	call	renderitems
@@isring3:	  
	mov		ebx,dword ptr [ebp].EXCFR.gSS
	MOV 	ecx,dword ptr [ebp].EXCFR.gESP
@@isring0:	  
	push	ebx 					; ebp-4 == SS
	push	ecx						; ebp-8 == ESP
	mov		eax, cr0
	push	eax						; ebp-12
	mov		eax, cr2
	push	eax						; ebp-16
	
	push	offset exc_format
	call	renderitems

	mov		esp,ebp

	mov		eax,ss
	cmp 	ax, FLAT_DATA_SEL
	jnz 	@@external_exc
	cmp		[ebp].EXCFR.gIntNo, 0Ch
	jz		@@external_exc
	cmp		[bInitDone],0
	jnz 	@@printexcstr
@@external_exc:
	mov		[bFatal],1
@@printexcstr:
	popad
    @v86popreg
	push	ds
	pop		ss
	mov		esp, ?BASE + V86_TOS - size V86FRGP + 3*4
    @v86pushreg

	push	offset exc_str
	call	PrintString 		;this will call v86-mode

    cmp		byte ptr [exc_ds],' '	;segments rendered? (v86 mode exc?)
	jz		@@nov86
	push	offset exc_str2
	call	PrintString
@@nov86:
@@waitkey:
	mov		byte ptr [esp].V86FRGP.gEAX+1, 00
    push	16h
    call	ExecIntV86
    cmp		byte ptr [esp].V86FRGP.gEAX, 1Bh
    jnz		@@waitkey
	push	offset exc_str4		;print a CR/LF    	
    call	PrintString
	cmp		[bFatal],1
	jz		@@nodoscall
    mov		word ptr [esp].V86FRGP.gEAX, 4C7Fh
    push	21h
    call	ExecIntV86
;--- if v86 returned (or exception is fatal), there is nothing to do than wait
@@nodoscall:
	push	offset exc_str3	
	call	PrintString
    jmp		@@waitkey
	
HandleException endp

	align 4

;--- v86 breakpoint table
;--- order must match the one defined in Jemm16.asm

_NUMBP_	= 0

@DefineBP macro equate, target
equate	equ _NUMBP_
_NUMBP_ = _NUMBP_ + 1
	dd target
	endm
    
bptable label dword
	@DefineBP BP67, Int67_V86Entry		; int 67h called by IVT vector
	@DefineBP BP06, HandleExc06V86		; BP06, default v86 int 06 handler
	@DefineBP BPXMS, xms_handler		; XMS handler (UMB+A20)
	@DefineBP BPI1587, I15_Simulate87	; simulate Int 15h, ah=87h
	@DefineBP BPBACK, V86_Back			; return to pm after calling v86-code
	@DefineBP BPSTRAT, EMMXXXX0_Strategy; EMMXXXX0 device strategy routine
	@DefineBP BPDEV, EMMXXXX0_Interrupt	; EMMXXXX0 device interrupt routine
if ?VDS
	@DefineBP BPVDS, vds_handler		; VDS handler, v86 int 4Bh
endif    
if ?DMA
	@DefineBP BP1340, Dma_CopyBuffer	; INT13/40 DMA read op
endif        
if ?UNLOAD
	@DefineBP BPRESET, Reset			; exit Jemm386, return to real-mode
endif
    
NUMBP equ _NUMBP_    


;--- exception in V86 mode
;--- ESP->V86FRGP
	
V86_Exception proc

if 0;?V86DBG  ;this happens quite often, so mostly not good to activate
	@DbgOutS <"exception in v86-mode, #=">,1
	mov eax, [esp].V86FRGP.gIntNo
	@DbgOutW ax,1
	@DbgOutS <", CS:IP=">,1
	mov eax,  [esp].V86FRGP.gCS
	@DbgOutW ax,1
	@DbgOutS <":">,1
	mov eax, [esp].V86FRGP.gEIP
	@DbgOutW ax,1
	@DbgOutS <10>,1
endif
	MOV 	EAX,SS
	MOV 	DS,EAX
	CMP 	ECX,0DH				; general protection exception?
	JNZ 	@@V86_TEST_EXC		; no, check further
@@Is_BP:
	MOVZX	EBX,word ptr [ESP].V86FRGP.gCS
    mov		ecx,[ESP].V86FRGP.gEIP
	SHL 	EBX,4
	ADD 	EBX,ECX				; EBX = linear CS:EIP, ECX=EIP

; check what triggered the GPF. Possible reasons are:

; - trapped I/O command. I/O only causes GPF for masked ports.
;	Currently the DMA, KBC and P92 ports are trapped. These ports
;	have no 32 registers, so only 8/16bit IN/OUT is trapped. String 
;	IN/OUT is not trapped either, but makes no sense in these cases.
; - HLT opcode, which then might be:
;	+ a "Breakpoint" if CS is RSEG. Do special handling if the offset
;	  is "known" as well.
;	+ other HLTs. Is handled by doing HLT for the user in the monitor.
; - other privileged opcode. Some are emulated (mov CRx, reg ...),	
;	some are not and then are just translated to an int 6, illegal opcode,
;	which is then reflected to the V86 task!).
;
	MOV 	AL,[EBX]				; check opcode
	cmp		AL, ?BPOPC
	jnz		@@NoBPOPC				; breakpoint?
	mov		eax, [bpstart]
    sub		ebx, eax
    jb		@@No_BP
    cmp		ebx, NUMBP
    jae		@@No_BP
	jmp		[offset bptable + ebx*4]

if ?BPOPC ne 0F4h
	align 4
if ?MASM	
Int06_Entry::
else
Int06_Entry:
endif
	push	0		;exc 06 has no error code, set a dummy one
	push	6
    @v86pushreg
	mov		eax, ss
	mov		ds, eax
	jmp 	@@Is_BP
endif

@@No_BP:
	add		ebx, eax
	cmp		ebx, 0FFFF0h
	jz		Reboot
if ?BPOPC ne 0F4h
	jmp		@@V86_EXC06
endif
@@Is_Hlt:

if ?HLTDBG
	@DbgOutS <"True HLT occured at CS:IP=">,1
	@DbgOutD [esp].V86FRGP.gCS,1
	@DbgOutS <":">,1
	@DbgOutD [esp].V86FRGP.gEIP,1
	@DbgOutS <10>,1
endif	 

	INC 	[ESP].V86FRGP.gEIP	; Jump over the HLT instruction
    @v86popreg
	ADD 	ESP,4+4 			; throw away errorcode & INT#
	STI 						; give Interrupts free and then wait,
	HLT 						; wait, wait........
if ?MASM    
BEHINDHLT::
else
BEHINDHLT:
endif
	iretd						; will it ever hit this?

@@NoBPOPC:

if ?BPOPC ne 0F4h
	CMP 	AL,0F4H 				; HLT-
	JZ		@@Is_Hlt				; command ?
endif

if ?SB
; see if SoundBlaster INT 3 forced to 1ah error code GPF

	CMP	[ESP].V86FRGP.gErrC,1ah
	jne	@@notsb
	test [bV86Flags],V86F_SB
	je	@@notsb					; SB option not turned on
    @v86popreg
	add	esp,4+4 				; discard excess GPF error code
	inc	[ESP].IRETDV86.vEIP		; skip INT 3 opcode
	push 3						; simulate an INT 3
	jmp	V86_Monitor
@@notsb:
endif

if ?EMUDBG
	@DbgOutS <"Opcode ">,1
	@DbgOutB AL,1
	mov ah, [ebx+1]
	@DbgOutS <" ">,1
	@DbgOutB AH,1
	mov ah, [ebx+2]
	@DbgOutS <" ">,1
	@DbgOutB AH,1
	@DbgOutS <" caused GPF at ">,1
	@DbgOutD ebx,1
	@DbgOutS <10>,1
endif

	cmp		al,0Fh					; check if potentially mov <reg>,cr#
	je		@@ExtendedOp

	CMP 	AL,0E4H 				; IN/OUT ??
	JB		@@V86_EXC06
	CMP 	AL,0E7H
	JBE 	@@DoIO_Im
	CMP 	AL,0ECH
	JB		@@V86_EXC06
	CMP 	AL,0EFH
	JBE 	@@DoIO_DX

	; no handling for 32bit IN/OUT or other potential GPF causes

@@NIX:
	JMP 	@@V86_EXC06				; else complain!

@@ExtendedOp:
	mov ah,al
	mov al,[ebx+1]
	cmp	al,9
	je	@@wbinvd
	cmp	al,8
	je	@@invd
	cmp	al,30h
	je	@@wrmsr
	cmp	al,31h
	je	@@rdtsc
	cmp	al,32h
	je	@@rdmsr
	cmp	al,20h
	jb	@@NIX		; not an opcode we emulate
	cmp	al,23h
	ja	@@NIX

; opcodes 0F 20 xx to 0F 23 xx emulated via self-modifying code

	mov	cl,[ebx+2]	; get third byte of opcode
    xchg al,ah
	mov	WORD PTR [EmuInstr+0],ax
	mov	BYTE PTR [EmuInstr+2],cl
    @v86popreg
	call RunEmuInstr
	add	esp,4+4					; eat return address and error code
	add	[esp].IRETDV86.vEIP,3	; jump over emulated instruction
	iretd

@@invd:
	invd				; 0f 08 is invd opcode
	jmp	@@invdshare
@@wbinvd:
	wbinvd				; 0f 09 is wbinvd opcode
@@invdshare:
	@v86popreg
	jmp	@@twoeat

@@wrmsr:
	@v86popreg
	@wrmsr
	jmp	@@twoeat

; early pentiums and such will throw an exception on rdtsc instruction in V86
;  regardless of CR4 setting, later CPU versions won't

@@rdtsc:
	@v86popreg
	@rdtsc
	jmp	@@twoeat
    
@@rdmsr:
	@v86popreg
	@rdmsr				; rdmsr opcode

@@twoeat:
	add	esp,4+4			; eat return address and error code
	add	[esp].IRETDV86.vEIP,2	; jump over instruction
	iretd
	
;--- exception (not 0Dh) in V86 mode
;--- DS=FLAT, ECX=INT#

@@V86_TEST_EXC:
if ?ROMRO
	cmp		ECX, 0Eh
	jnz		@@V86_EXC_NOT0D0E
	mov		eax, CR2
	shr		eax, 12
	cmp		eax, 0FFh	   ;it is the FF000 page (which is r/o!)
	jnz 	@@V86_EXC06
	@DbgOutS <"Write access to FF000",10>, ?V86DBG
	@GETPTEPTR ebx, 1000h+0FFh*4, 1
	mov 	ecx, [ebx]	;get PTE for FF000
	mov		dword ptr [ebx], 0FF000h + 111B	;set the "original" PTE
	mov		[dwSavedRomPTE], ecx
	call	@@invlpgFF

	mov		ecx, offset @@resettf
	mov		ax, FLAT_CODE_SEL
	shl		eax, 16
	mov		ax, cx
	mov		cx, 0EE00h

	mov		ebx, offset IDT
	xchg	eax, [ebx+1*8+0]
	xchg	ecx, [ebx+1*8+4]
	mov		dword ptr [dqSavedInt01+0], eax
	mov		dword ptr [dqSavedInt01+4], ecx
	or		byte ptr [esp].V86FRGP.gEFL+1, 1	;set TF
    @v86popreg				; return to V86, execute 1 instruction
	ADD 	ESP,4+4
	iretd

;--- returned to monitor after 1 instruction run in v86 mode

@@resettf:
	pushad
	push	ss
	pop		ds
	@DbgOutS <"After write access to FF000",10>, ?V86DBG
	mov		ecx, [dwSavedRomPTE]
	@GETPTEPTR eax, 1000h+0FFh*4, 1
	mov		[eax],ecx
	xor		eax, eax
	mov		cr2, eax
	mov		eax, dword ptr [dqSavedInt01+0]
	mov		ecx, dword ptr [dqSavedInt01+4]
	mov		ebx, offset IDT
	mov		[ebx+1*8+0], eax
	mov		[ebx+1*8+4], ecx
	call	@@invlpgFF
	popad
	and		byte ptr [esp].IRETDV86.vEFL+1,not 1	;clear TF
	iretd
@@invlpgFF:
if ?INVLPG
	cmp		[bNoInvlPg],0
	jnz		@@noinvlpg
	invlpg	ds:[0FF000h]
	ret
@@noinvlpg:
endif
	mov		eax, cr3
	mov		cr3, eax
	ret
endif	 

;*************************************************************
; unhandled exception in v86-mode. simulate a v86 exception 06.
; this notifies any hookers about the problem. If noone has hooked
; v86-int 06, we finally end at the monitor again, display a register
; dump and then try to terminate the current PSP.
;*************************************************************

@@V86_EXC_NOT0D0E:

;--- exception occured in V86 mode (except 0D or "0E at page FF")

@@V86_EXC06:

;--- simulate invalid opcode interrupt in v86 mode
;--- DS=FLAT

	push	ebp
	mov		ebp, esp
	push	6
	call	SimIntV86
	pop		ebp
    @v86popreg
	add 	ESP,4+4 		; remove error code + old calling address
	IRETD					; return to virtual 86-Mode

;***************************************

; an IN/OUT-command now HAS to be emulated and the data has to be checked
; esp = ecx, ebx, eax, ret, errc, eip, cs, ...

IOOFS	equ 3*4	;pushed EDX, ESI, EDI

if 0

;--- this code is to take over handling of port trapping.
;--- It's table driven and therefore will allow to add 
;--- ports to trap dynamically. Currently offers no benefits.

portmap label byte
if ?DMA
	db 000h,00Fh
    db 080h,08Fh
    db 0C0h,0DFh
endif    
if ?A20PORTS
	db 060h,060h
    db 064h,064h
    db 092h,092h
endif
endportmap label byte

portproc label dword
	dd Dma_HandleDmaPorts8
    dd Dma_HandlePagePorts
    dd Dma_HandleDmaPorts16
	dd A20_Handle60
    dd A20_Handle64
    dd A20_Handle92
    
@@DoIO_DX:
	PUSH	EDX
	PUSH	ESI
	PUSH	EDI
	INC 	[ESP+IOOFS].V86FRGP.gEIP		; Jump over instruction
	JMP 	@@WithDX
@@DoIO_Im:
	PUSH	EDX
	PUSH	ESI
	PUSH	EDI
	MOVZX	EDX,BYTE PTR [EBX+1]	; get I/O port in DX
	ADD 	[ESP+IOOFS].V86FRGP.gEIP,2 	; jump over instruction
@@WithDX:
	mov bl,al
    mov eax, [esp+IOOFS].V86FRGP.gEAX
    mov esi, offset portmap
@@nextport:    
    cmp dl, [esi+0]
    jb @@skipport
    cmp dl, [esi+1]
    jbe @@portfound
@@skipport:
	add esi,2
    cmp esi, offset endportmap
    jnz @@nextport
    jmp defaultporthandler
@@portfound:
	sub esi, offset portmap
	mov	ecx, ss
	mov	es, ecx
    call dword ptr [esi*2+offset portproc]
    jnc @@ishandled
    call defaultporthandler
@@ishandled:
	test	bl,2
    jnz		@@isout
    mov [esp+IOOFS].V86FRGP.gEAX, eax
@@isout:    
	POP		EDI
	POP		ESI
	POP		EDX
    @v86popreg
	ADD 	ESP,4+4
	IRETD
    
defaultporthandler:    
    test bl,2
    jnz @@isout
    test bl,1
    jnz @@iswordin
    in al,dx
    ret
@@iswordin:
	in ax,dx
    ret
@@isout:
	test bl,1
    jnz @@iswordout
    out dx,al
    ret
@@iswordout:
	out dx,ax
    ret
    
endif

;--- DS=FLAT
;--- AL=opcode
;--- EBX=linear CS:EIP

if 1

;--- this is the "old" port trapping code. It's still active
;--- since the port trapping API isn't implemented yet.

@@DoIO_DX:
	PUSH	EDX
	PUSH	ESI
	PUSH	EDI
	INC 	[ESP+IOOFS].V86FRGP.gEIP		; Jump over instruction
	JMP 	@@WithDX
@@DoIO_Im:
	PUSH	EDX
	PUSH	ESI
	PUSH	EDI
	MOVZX	EDX,BYTE PTR [EBX+1]	; get I/O port in DX
	ADD 	[ESP+IOOFS].V86FRGP.gEIP,2 	; jump over instruction
@@WithDX:
	mov		bl,al
    mov		eax,[ESP+IOOFS].V86FRGP.gEAX
	TEST	BL,2		  	; is it IN or OUT ?
	JNZ 	@@Is_Out			; 
	TEST	BL,1		  	; IN AL,xx or IN AX,xx ?
	JNZ 	@@In_Word
	push	offset @@In_Done
if ?A20PORTS
	cmp		DL,60h
	jz		A20_Handle60
	cmp		DL,92h
	jz		A20_Handle92
endif
	IN		AL,DX
    ret
@@In_Word:
	IN		AX,DX
@@In_Done:
	MOV 	[ESP+IOOFS].V86FRGP.gEAX,EAX
@@Bye:
	POP		EDI
	POP		ESI
	POP		EDX
    @v86popreg
	ADD 	ESP,4+4
	IRETD

;--- immediate and DX OUT

@@Is_Out:
	mov		ecx, ss
	mov		es, ecx
	push	offset @@Bye
	TEST	BL,1			; is it a WORD op?
	JZ		@@Do_ByteOut
	push	edx
	push	eax
	call	@@Do_ByteOut
	pop		eax
	pop		edx
	INC 	EDX
	MOV 	AL,AH

;--- output AL to DX
;--- DS=ES=FLAT

@@Do_ByteOut:
if ?A20PORTS
	CMP 	DL,60H
	JZ		A20_Handle60
	CMP 	DL,64H
	JZ		A20_Handle64
	CMP 	DL,92H
	JZ		A20_Handle92
endif
if ?DMA
	CMP 	DL,10H				; Has a Page-register
	JB		Dma_HandleDmaPorts8	; been adressed, or one
	CMP 	DL,80H				; Has a Page-register
	JB		@@nodma
	CMP 	DL,90H				; of both DMA-
	JB		Dma_HandlePagePorts	; Controllers ?
	CMP 	DL,0C0H
	JB		@@nodma
	CMP 	DL,0E0H
	JB		Dma_HandleDmaPorts16
@@nodma:	
endif
	OUT		DX, AL
	RET

endif

V86_Exception endp

	align 4

if ?DMA

PageLookUp		DB		0,2,3,1,0,0,0,0,0,6,7,5,0,0,0,0
PageXLat		DB		87H,83H,81H,82H,0,8BH,89H,8AH

;
; Access to a pageregister
;
; In: DX : Port-address (80h-8Fh)
; AL : Value
; BL : type of op (bit 1: in/out, bit 0: byte/word)
;
Dma_HandlePagePorts PROC

if 0
	test	bl,2
    jnz		@@isout
    stc
    ret
@@isout:    
endif
if ?DMADBG
	@DbgOutS <"DMA page reg OUT, dx=">
	@DbgOutW dx
	@DbgOutS <" AL=">
	@DbgOutB al
	@DbgOutS <10>
endif	 

	OUT 	DX,AL					; release data
	MOVZX	EBX,DL					; Look up the number that corresponds to the port
	MOVZX	EDI,BYTE PTR PageLookUp[EBX-80H]
	MOV 	[DmaChn+EDI*8].PageReg,AL		  ; Buffer Bits 24-16
	JMP 	Dma_IsReady

Dma_HandlePagePorts ENDP


Dma_HandleDmaPorts proc
;
; Supervise certain registers of the 8bit DMA controller
; DX = port
; AL = value
;

if ?MASM
Dma_HandleDmaPorts8::
else
Dma_HandleDmaPorts8:
endif

if 0
	test	bl,2
    jnz		@@isout
    stc
    ret
@@isout:    
endif
if ?DMADBG
	@DbgOutS <"8 bit DMA OUT, dx=">,1
	@DbgOutW dx,1
	@DbgOutS <" AL=">,1
	@DbgOutB al,1
	@DbgOutS <" 08=">,1
	mov ah,al
	in al,8
	@DbgOutB al,1
	mov al,ah	 
	@DbgOutS <10>,1
	@WaitKey 1, 0;?DMADBG
endif	 

	CMP 	DL,8					; Program addresses and/or length of the blocks
	JB		@@BlockSet8
	CMP 	DL,0AH					; single mask?
	JZ		@@SMask8
	CMP 	DL,0BH					; "Mode Register" responded ?
	JZ		@@Mode8 				; DMA-Controller #1 ?
	CMP 	DL,0CH
	JZ		@@Clear8				; clear flip-flop?
if ?MMASK	 
	CMP 	DL,0DH
	JZ		@@MMask8On
	CMP 	DL,0EH
	JZ		@@MMask8Off
	CMP 	DL,0FH
	JZ		@@MMask8				; master mask?
endif	 
	@DbgOutS <"port was unhandled!?",10>, ?DMADBG
	out		dx, al
    ret
@@Clear8:
	out		dx, al
	BTR 	[DmaGblFlags],HiLoFlag1
	ret
    

;
; Supervise certain registers of the 16bit DMA controller
; DX = port
; AL = value
;

if ?MASM
Dma_HandleDmaPorts16::
else
Dma_HandleDmaPorts16:
endif

if 0
	test	bl,2
    jnz		@@isout2
    stc
    ret
@@isout2:    
endif
	CMP 	DL,0C0h+10h 			; dto. 2nd DMA-Controller ?
	JB		@@BlockSet16
	CMP 	DL,0C0h+0AH*2			; single mask?
	JZ		@@SMask16
	CMP 	DL,0C0h+0BH*2			; mode set?
	JZ		@@Mode16
	CMP 	DL,0C0h+0Ch*2
	JZ		@@Clear16
if ?MMASK	 
	CMP 	DL,0C0h+0Dh*2
	JZ		@@MMask16On
	CMP 	DL,0C0h+0Eh*2
	JZ		@@MMask16Off
	CMP 	DL,0C0h+0Fh*2			; master mask?
	JZ		@@MMask16
endif	 
	@DbgOutS <"port was unhandled!?",10>, ?DMADBG
	out		dx, al
    ret
@@Clear16:
	out		dx, al
	BTR 	[DmaGblFlags],HiLoFlag2
	ret

    
@@BlockSet8:					;--- I/O addresses 0-7 (DMA 8bit)
	MOV 	BX,HiLoFlag1
	MOVZX 	EDI,DL
	SHR 	EDI,1				; get channel# into EDI
	jc		@@Length
	JMP		@@Adr
@@BlockSet16:					;--- I/O addresses C0-CF (DMA 16bit)
	MOV 	BX,HiLoFlag2
	MOV 	EDI,EDX
	SHR 	EDI,2				; get channel# into EDI
	AND 	EDI,3
	ADD 	EDI,4
	TEST	DL,02H				; block length or address?
	JNZ 	@@Length
@@Adr:
	mov		ecx,offset DmaChn.BaseAdr
    jmp		@@LenOrAdr
@@Length:
	mov		ecx,offset DmaChn.BlockLen
@@LenOrAdr:
	OUT 	DX,AL
	BTC 	[DmaGblFlags],BX		; toggle the Hi/Lo-Flag
	adc		ecx,0
	MOV 	BYTE PTR [ECX+EDI*8],AL
	test	cl,1
    jnz		Dma_IsReady
	ret

;
; Monitor "Mode Register" for the Transferdirection DMA <--> I/O
;
@@Mode16:
	MOV 	EDI,EAX
	AND 	EDI,011B			; mask the DMA channel #
	ADD 	EDI,4				; 16bit channels are 4-7
	JMP 	@@Mode
@@Mode8:
	MOV 	EDI,EAX
	AND 	EDI,011B
@@Mode:
	out		dx, al
	mov		[DmaChn+EDI*8].ChanMode,al

if ?MASM
Dma_IsReady::
else
Dma_IsReady:
endif
if 0
	mov		ecx,[dwRFlags]
	test	byte ptr [ecx],2	;new Int 13h/40h DMA op?
	jz		@@nonewreq
	and		byte ptr [ecx],not 2
	and 	[DmaChn+EDI*8].bFlags,not DMAF_ENABLED	;wait until channel is enabled
@@nonewreq:    
endif
	test	[DmaChn+EDI*8].bFlags,DMAF_ENABLED
	jz		@@Bye
	CALL	Dma_CheckChannel	; Then check value
@@Bye:
	RET

;--- Single mask port (000A, 00D4)
;--- bits 0-1 select channel
;--- bit 2=1 -> enable channel mask (=channel inactive)

@@SMask16:
	mov		edi, eax
	AND 	EDI,0011B
	add		edi,4
	jmp		@@SMask
@@SMask8:
	mov		edi, eax
	AND 	EDI,0011B
@@SMask:
	and		[DmaChn+EDI*8].bFlags,not DMAF_ENABLED
	test	al,4
	jnz		@@isdisable
	or		[DmaChn+EDI*8].bFlags,DMAF_ENABLED
if 0    
	mov		ecx,[dwRFlags]
	and		byte ptr [ecx],not 2
endif    
	push	eax
	call	Dma_CheckChannel
	pop		eax
@@isdisable:
	out		dx, al
	ret

if ?MMASK
;--- Master mask port (000F, 00DE)
;--- bit 0 -> 1=channel 0 inactive, 0=channel 0 active
;--- bit 1 -> ... chn 1
;--- bit 2 -> ... chn 2
;--- bit 3 -> ... chn 3

;--- this port cannot be read reliably !!!!!

@@MMask16On:			;mask all 4 channels
	mov		al,0Fh
	jmp		@@MMask16
@@MMask16Off:			;unmask all 4 channels
	mov		al,0
@@MMask16:
	mov		ah, al
	shl		ah, 4
	mov		edi, 4
	jmp		@@MMask
@@MMask8On:				;mask all 4 channels
	mov		al,0Fh
	jmp		@@MMask8
@@MMask8Off:			;unmask all 4 channels
	mov		al,0
@@MMask8:
	mov		ah, al
	xor		edi, edi
@@MMask:
	mov		cl,4
	xchg	al,ah	;value to write to port now in AH, in AL transformed mask
@@nextcnl:
;	test	[DmaChn+EDI*8].bFlags,DMAF_ENABLED
;	jnz 	@@skipchannel		;channel is already active
	and		[DmaChn+EDI*8].bFlags,not DMAF_ENABLED
	bt		eax, edi
	jc		@@skipchannel		;channel will become/stay inactive
	or		[DmaChn+EDI*8].bFlags,DMAF_ENABLED
	push	eax
	push	ecx
	call	Dma_CheckChannel
	pop		ecx
	pop		eax
@@skipchannel:
	inc		edi
	dec		cl
	jnz 	@@nextcnl
	mov		al, ah
	out		dx, al
	ret
endif

Dma_HandleDmaPorts ENDP

;
; A DMA-Channel is completely with data about beginning and length
; supplied, so that a check can (and has to) take place.
;
; In: EDI: Channelnumber 0..7
; modifies ECX, ESI, EBX
;
Dma_CheckChannel PROC

	@DbgOutS <"Dma_CheckChannel enter",10>, ?DMADBG
	@WaitKey 1, 0;?DMADBG
;	 int		3

	cmp		[DmaChn+EDI*8].cDisable,0	;translation disabled by VDS?
	jnz		@@Bye
	test	[DmaChn+EDI*8].ChanMode,1100B  ; If a verify is wanted
	JZ		@@Bye						   ; nothing is to be done

if 1
	and 	[DmaChn+EDI*8].bFlags,not DMAF_ENABLED
endif

	MOVZX	ECX,[DmaChn+EDI*8].BlockLen	; get block length into ECX
	MOVZX	ESI,[DmaChn+EDI*8].PageReg	; and base address into ESI
	INC 	ECX
	CMP 	EDI,4						; for 16bit DMA channels,
	JB		@@Only8
	ADD 	ECX,ECX 					; words are transferred
	SHL 	ESI,15						; and Bit 0 of the page register
	MOV 	SI,[DmaChn+EDI*8].BaseAdr	; will be ignored
	SHL 	ESI,1
	JMP 	@@Chk
@@Only8:
	SHL 	ESI,16						; for 8bit DMA, address calculation
	MOV 	SI,[DmaChn+EDI*8].BaseAdr	; is simple
@@Chk:

;--- now ESI holds the start address (linear!)

	BTR 	[DmaGblFlags],NeedBuffer	; Initialise.

;--- Is the block occupying contiguous physical pages?
;--- Or does it cross a 64kB/128 kB boundary?
;--- after the call ESI will hold the physical address (if NC)

	push	esi
	CALL	Dma_IsContiguous
	pop		ebx
	JNC 	@@Set	;NC if buffer *is* contiguous				

;	test	[DmaChn+EDI*8].ChanMode,10h	;auto-init? Then a buffer is useless
;	jnz		@@Set

	@DbgOutS <"block not contiguous or crosses 64k border, DMA buffer needed!",10>, ?DMADBG
	@WaitKey 1, 0;?DMADBG
	
	MOV 	ESI,[DMABuffStartPhys]		; get DMA Buffer physical address
	BTS		[DmaGblFlags], NeedBuffer
	cmp		ecx, [DMABuffSize]
	jbe		@@blocksizeok
	mov		ecx, [DMABuffSize]			; error: DMA buffer overflow	
@@blocksizeok:
    call	@@SetX
	MOV 	AL,[DmaChn+EDI*8].ChanMode	; copy data into buffer required?
	and		al,1100b
	cmp		al,1000b
	JNZ 	@@IsRead

	@DbgOutS <"Dma_CheckChannel: copy into DMA buffer",10>, ?DMADBG

	push	edi	
	MOV 	ESI,EBX
	MOV 	EDI,[DMABuffStart]
	CLD
	MOV 	EAX,ECX
	AND 	ECX,3
	REP 	MOVS BYTE PTR [ESI],BYTE PTR [EDI]
	@BIG_NOP
	MOV 	ECX,EAX
	SHR 	ECX,2
	REP 	MOVS DWORD PTR [ESI], DWORD PTR [EDI]
	@BIG_NOP
    pop		edi
    RET
@@IsRead:
	mov		eax,[dwRFlags]
if 1
    test	byte ptr [eax],2			; is a int 13h/40h read op pending?
    jz		@@Bye
	mov		byte ptr [eax],1
else    
	or 		byte ptr [eax],1
endif
	MOV 	[DmaTargetLen],ECX
	MOV 	[DmaTargetAdr],EBX 			; save linear address of block
@@Bye:
	RET

;--- set DMA base address and page register
;--- modifies ESI, EAX

@@Set:

if ?DMADBG
	@DbgOutS <"target lin. start address=">,1
	@DbgOutD ebx,1
	@DbgOutS <" length=">,1
	@DbgOutD ecx,1
	@DbgOutS <" phys=">,1
	@DbgOutD esi,1
	@DbgOutS <" chn=">,1
	@DbgOutD edi,1
	cmp esi, ebx
	jz @@nothingtodo
	@DbgOutS <"*">,1
@@nothingtodo:	  
	@DbgOutS <10>,1
	@WaitKey 1, 0
endif	 

	cmp		esi, ebx				; has address changed?
	jz		@@Bye
@@SetX:	
	push	edx
	CMP 	EDI,4					; 8-bit or 16-bit channel ?
	JB		@@Set8

;-- calc I/O-address from DMA channel: 4->C0, 5->C4, 6->C8, 7->CC

	lea		edx, [edi-4]
    shl		edx, 2                  ; edx=0,4,8,C
	or	 	dl, 0C0h
	BTR 	[DmaGblFlags],HiLoFlag2	; DMA #2, HiLo-FlipFlop-Flag
	OUT 	[0D8H],AL				; clear Hi/Lo-FlipFlop
	JMP 	$+2
	SHR 	ESI,1
	MOV 	EAX,ESI 				; reprogram base address
	SHR 	ESI,15
	JMP 	@@Cont

;-- calc I/O-address from DMA channel: 0->0, 1->2, 2->4, 3->6

@@Set8:
	LEA 	EDX,[EDI+EDI]
	BTR 	[DmaGblFlags],HiLoFlag1	; DMA #1, HiLo-FlipFlop-Flag
	OUT 	[0CH],AL				; clear Hi/Lo-FlipFlop
	JMP 	$+2
	MOV 	EAX,ESI 				; reprogram base address
	SHR 	ESI,16					; get bits 16-23 into 0-7
@@Cont:
	OUT 	DX,AL					;
	MOV 	AL,AH
	JMP 	$+2
	OUT 	DX,AL
	MOV		DL,PageXLat[EDI]		; get I/O-Adress of page-register
	MOV 	EAX,ESI 				; set the page register
	OUT 	DX,AL
    pop		edx
	ret
    align	4
    
Dma_CheckChannel ENDP

;
; Check a memory-area to be in physical contiguous memory.
; Furthermore, the physical region must not cross a 64 kB
; (or 128 kB for 16bit) boundary
;
; In:  ESI: linear start address (bits 0-23 only)
;	   ECX: Length of the range (?DMABUFFMAX is max in kB)
;	   EDI: channel
; Out: NC: contiguous, ESI = physical adress
;		C: not contiguous
; modifies: EAX, EBX, ESI 

Dma_IsContiguous PROC
	cmp 	ESI, 400000h-20000h 	; don't touch PTEs > 3E0000h
	jnc		@@Ok2
	PUSH	ECX

if 0	;ecx cannot be 0 (is 1-20000h)
	cmp		ecx,1					; make sure ecx is at least 1
	adc 	ecx,0
endif

	PUSH	ESI
	lea 	ECX,[ecx+esi-1] 		; ecx -> last byte 

	SHR 	ESI,12					; linear start address -> PTE
	SHR 	ECX,12					; linear end   address -> PTE
	sub		ecx, esi
	
	@GETPTEPTR ESI, ESI*4+1000h 	; ESI -> PTE
	MOV 	EAX,[ESI]				; get PTE

if ?DMADBG
	@DbgOutS <"Dma_IsContiguous: PTE=">
	@DbgOutD eax
	@DbgOutS <" ecx=">
	@DbgOutD ecx
	@DbgOutS <10>
endif

	shr		eax, 12					; ignore bits 0-11 of PTE

;--- ESI -> PTE, EAX=1. PTE

	test	eax, 0FF000h 			; if physical address is > 16 MB		
	jnz		@@Fail2 				; a buffer is needed in any case

	JECXZ	@@Ok					; exit if region fits in one page
	
	PUSH	EAX 					; save Bits 31-12
@@Loop:
	INC 	EAX						; Go one page and
	ADD 	ESI,4					; one entry further
	MOV 	EBX,[ESI]
	shr 	EBX, 12
	CMP 	EAX,EBX 				; contiguous?
	JNZ 	@@Fail
	loop	@@Loop
	mov		eax, [esp]
	shr		eax, 4
	shr		ebx, 4
	cmp		edi,4
	jb		@@is8bit
	and		al,not 1
	and		bl,not 1
@@is8bit:	 
	cmp		ebx,eax
	JNZ 	@@Fail					; a 64/128 kB Border has been crossed!!!
	POP 	EAX
@@Ok:
	pop 	ESI
    shl		eax,12
	AND 	ESI,0FFFh
	OR		ESI,EAX
	pop 	ECX
@@Ok2:
	CLC
	RET
@@Fail: 
	POP 	EAX
@@Fail2: 
	POP 	ESI
	POP 	ECX
	STC
	RET
    align 4
    
Dma_IsContiguous ENDP


; Copy the DMA-buffercontents after termination of the DISK-I/O to the
; wanted target/location
; * This is triggered by an HLT at a magical location in our own int 13
; * and int 40 handlers. First the original int 13 / 40 is done, and
; * then things are copied to / from where the mapped memory -really- is.
; * Should only trigger if a matching DMA is pending. Tricky!
; DS=FLAT

Dma_CopyBuffer PROC

	push	ebp
    mov		ebp,esp
	call	SimIRET
    and		[ebp+4].V86FRGP.gEFL, not 1	;clear carry
    pop		ebp

	PUSH	ESI
	PUSH	EDI
	
	MOV 	ESI,[DMABuffStart]	; Get the basic data of the block which
	MOV 	EDI,[DmaTargetAdr] 	; can be shifted. Read currently
								; simulated DMA source / dest. / length
	mov 	ECX,[DmaTargetLen]

if ?DMADBG								  
	@DbgOutS <"Dma_CopyBuffer: dst=">,1
	@DbgOutD edi,1
	@DbgOutS <" src=">,1
	@DbgOutD esi,1
	@DbgOutS <" siz=">,1
	@DbgOutD ecx,1
	@DbgOutS <10>,1
endif

	MOV 	EAX,DS
	MOV 	ES,EAX
	CLD
	MOV 	EAX,ECX
	AND 	ECX,3
	REP 	MOVS BYTE PTR [ESI],BYTE PTR [EDI]
	@BIG_NOP
	MOV 	ECX,EAX
	SHR 	ECX,2
	REP 	MOVS DWORD PTR [ESI], DWORD PTR [EDI]
	@BIG_NOP
	mov		eax,[dwRFlags]
    mov		byte ptr [eax],0
	POP		EDI
	pop		ESI
    @v86popreg
	ADD 	ESP,4+4 			; throw away error code + return-address.
	IRETD						; back to virtual 8086-Mode.

Dma_CopyBuffer ENDP

endif	;?DMA

	align 4
    
;--- this breakpoint allows v86 code to be called by the monitor
;--- the caller has to push a v86_call item on the v86-stack:
;--- out: ebx=v86_call item

V86_Back proc
	MOVZX	ebx, word ptr [ESP].V86FRGP.gESP
	MOVZX	ecx, word ptr [ESP].V86FRGP.gSS
	add 	[ESP].V86FRGP.gESP,size v86_call - 4
	SHL		ecx, 4
	add		ebx, ecx
	sub		ebx, 4
	mov		eax, [ebx].v86_call.dwOldCSIP
	mov		word ptr [ESP].V86FRGP.gEIP,ax
	shr		eax,16
	mov		[ESP].V86FRGP.gCS, eax
	jmp 	[ebx].v86_call.dwRetAddr
V86_Back endp

	align 4

EMMXXXX0_Strategy proc
    call	SimRETF				; do a RETF in V86
    movzx	eax,word ptr [esp].V86FRGP.gES
    movzx	ecx,word ptr [esp].V86FRGP.gEBX
    shl		eax, 4
    add		eax, ecx
    mov		[dwReqPtr], eax
    @v86popreg
	ADD 	ESP,4+4 			; throw away errorcode & returnaddress.
	iretd
EMMXXXX0_Strategy endp

;--- EMMXXXX0 device driver interrupt routine 
;--- DS=FLAT

request_hdr struc
rhSize		db	?		; number of bytes stored
rhUnit_id	db	?		; unit ID code
rhCmd		db	?		; command code
rhStatus	dw	?		; status word
rhReserved	db	8 dup (?)	; reserved
request_hdr ends

	
EMMXXXX0_Interrupt proc
    call	SimRETF				; do a RETF in V86
	push	ds
	pop		es
	mov		ecx, [dwReqPtr]
if ?EMMXXXX0
	mov		al, [ecx].request_hdr.rhCmd
if ?EMXDBG
	@DbgOutS <"EMMXXXX0 request ">,1
	@DbgOutB al,1
	@DbgOutS <10>,1
endif
	cmp 	al, 3			;IOCTL input?
	jz		@@ioctl_read
	cmp 	al, 8			;write?
	jz		@@resp_writeerr
	cmp 	al, 9			;write+verify?
	jz		@@resp_writeerr
	cmp 	al, 10			;write status
	jz		@@resp_ok
	cmp 	al, 11			;write+flush?
	jz		@@resp_writeerr
	cmp 	al, 12			;IOCTL output?
	jz		@@ioctl_write
	cmp 	al, 13			;open device?
	jz		@@resp_ok
	cmp 	al, 14			;close device?
	jz		@@resp_ok
	mov		ax, 8103h
	jmp		@@device_done
@@ioctl_read:
	pushad
	call	IoctlRead
	popad
	jc		@@resp_readerr
	jmp		@@resp_ok
@@ioctl_write:
	pushad
	call	IoctlWrite
	popad
	jc		@@resp_writeerr
	jmp		@@resp_ok
@@resp_readerr:    
	mov		ax, 810Bh
	jmp		@@device_done
@@resp_writeerr:	
	mov		ax, 810Ah
	jmp		@@device_done
@@resp_ok:
	mov		ax, 100h
@@device_done:
else
	mov		ax, 8103h
endif
	mov		[ecx].request_hdr.rhStatus, ax
    @v86popreg
	ADD 	ESP,4+4 			; throw away errorcode & returnaddress.
	iretd
	align 4
EMMXXXX0_Interrupt endp

;--- simulate an RETF in v86-mode
;--- INP: SS:ESP+4 -> V86FRGP
;--- modifies EAX,ECX

SimRETF proc
	MOVZX	eax, word ptr [ESP+4].V86FRGP.gESP
	MOVZX	ecx, word ptr [ESP+4].V86FRGP.gSS
	SHL		ecx, 4
	add		ecx, eax
	MOV		eax, [ecx+0]
	mov		word ptr [ESP+4].V86FRGP.gEIP, ax
	shr		eax,16
	mov		[ESP+4].V86FRGP.gCS, eax
	ADD 	[ESP+4].V86FRGP.gESP,2*2
	ret
	align 4
SimRETF endp

;--- simulate an IRET in v86-mode
;--- INP: SS:EBP+4 -> V86FRGP
;--- modifies EAX,EBX,ECX

SimIRET proc
	MOVZX	eax, word ptr [EBP+4].V86FRGP.gESP
	MOVZX	ecx, word ptr [EBP+4].V86FRGP.gSS
	SHL		ecx, 4
	add		eax, ecx
	MOV		ebx, [eax+0]
	MOV		cx, [eax+4]
	mov		word ptr [EBP+4].V86FRGP.gEIP,bx
	shr		ebx,16
	mov		[EBP+4].V86FRGP.gCS, ebx
	mov		word ptr [EBP+4].V86FRGP.gEFL,cx
	ADD 	[EBP+4].V86FRGP.gESP,3*2
	ret
	align 4
SimIRET endp

;--- simulate an V86 Int
;--- INP: SS:EBP+4 -> V86FRGP
;--- INP: [ESP+4] == INT #
;--- DS=FLAT
;--- modifies EAX, EBX, ECX
;--- this is a stdcall proc

SimIntV86 proc	  

	movzx	eax,word ptr [EBP+4].V86FRGP.gESP
	MOVZX	EBX,word ptr [EBP+4].V86FRGP.gSS	; get address of v86 SS:SP
    SUB		AX, 3*2 				; make room for IRET frame
	SHL 	EBX,4
	ADD 	EBX, EAX
	MOV 	[EBP+4].V86FRGP.gESP,eax

	MOV 	EAX,[EBP+4].V86FRGP.gCS	; get v86 CS:IP into EAX
	shl		EAX, 16
	MOV 	AX,word ptr [EBP+4].V86FRGP.gEIP
	MOV 	ECX,[EBP+4].V86FRGP.gEFL; + v86 Flags
	MOV 	[EBX+0],EAX
	MOV 	[EBX+4],CX				; and store them onto v86 stack
    and		ch,not (1+2)			; clear TF+IF
	mov 	[EBP+4].V86FRGP.gEFL,ecx

	pop		ecx
    pop		ebx						; get int#
	MOV 	EAX,[EBX*4]				; now set new v86 CS:IP to value
	MOV 	word ptr [EBP+4].V86FRGP.gEIP,AX	;found in IVT
	SHR 	EAX,16
	MOV 	[EBP+4].V86FRGP.gCS,EAX
	jmp 	ecx
SimIntV86 endp

;--- exec a v86 int immediately
;--- ESP+8 -> V86FRGP
;--- [ESP+4] = INT# to execute

ExecIntV86 proc

	movzx	ebx, word ptr [esp+8].V86FRGP.gESP
	movzx	ecx, word ptr [esp+8].V86FRGP.gSS

;--- since SimIntV86 is called later, we dont need a full v86_call
;--- item onto the stack. size (v86_call - 4) is enough.

	sub		ebx, size v86_call - 4
	shl		ecx, 4
	mov		word ptr [esp+8].V86FRGP.gESP, bx
	add		ebx, ecx
	pop		[ebx-4].v86_call.dwRetAddr
    
	movzx	ecx, [bBpBack]
	mov 	eax, [dwRSeg]
	xchg	ecx, [esp+4].V86FRGP.gEIP
	xchg	eax, [esp+4].V86FRGP.gCS
	mov		word ptr [ebx-4].v86_call.dwOldCSIP+0, cx
	mov		word ptr [ebx-4].v86_call.dwOldCSIP+2, ax
    
    pop		eax
    
	push	ebp
    mov		ebp,esp
    push	eax
    call	SimIntV86
    pop		ebp

	@v86popreg
	add 	ESP,4+4 		; skip call return + error code
	IRETD
ExecIntV86 endp

if ?VME
;--- set/reset the VME flag in CR4 if supported
;--- INP: AL[0] new state of flag

SetVME proc
	test	byte ptr [dwFeatures], 2		;VME supported?
	jz		@@novme
	@mov_ecx_cr4
	and		al,1
	and		cl,not 1
	or		cl,al
	@mov_cr4_ecx
@@novme:
	ret
SetVME endp

endif

;--- print a string. Since ring3 code cannot be "called", the
;--- ring3 stack has to be used to save the state.
;--- a v86disp item is saved on v86-SS:SP:
;--- DS=FLAT

v86disp struc
dwESI	dd ?
wAX		dw ?
dwRet	dd ?
v86disp ends

PrintString proc	 
	movzx	ebx, word ptr [esp+8].V86FRGP.gESP
	movzx	ecx, word ptr [esp+8].V86FRGP.gSS
	sub		ebx, size v86disp
	shl		ecx, 4
	mov		word ptr [esp+8].V86FRGP.gESP, bx
	add		ebx, ecx

	mov		[ebx].v86disp.dwESI, esi	;save ESI
	mov		[ebx].v86disp.wAX, ax
	pop		[ebx].v86disp.dwRet			;return address

	pop		esi							;get the string to render
@@nextchar:
	lodsb
	and		al,al
	jz		@@done
	mov		byte ptr [ESP].V86FRGP.gEAX,al
	push	29h
    call	ExecIntV86
    add		ebx, size v86_call	;V86_Back has set EBX to v86 stack
    jmp		@@nextchar
@@done:
	mov		esi,[ebx].v86disp.dwESI
	mov		ax, [ebx].v86disp.wAX
	mov		word ptr [esp].V86FRGP.gEAX, ax
	add 	[ESP].V86FRGP.gESP, size v86disp
	jmp		[ebx].v86disp.dwRet

PrintString endp


if ?A20PORTS

A20_Handle60 proc
	test bl,2			;is it in or out?
    jz A20_Inp60
	cmp bLast64,0D1h	;last value written to 64 was "write output port"?
	jnz @@notoutp
	mov bLast64,0
	@DbgOutS <"write to port 60h kbc output port, al=">,?A20DBG
	@DbgOutB al,?A20DBG
	@DbgOutS <10>,?A20DBG
	push eax
	shr al,1
	and al,1
	mov bA20Stat,al
	call A20_Set
	pop eax
	or	al,2
@@notoutp:	  
	out dx, al
	ret
A20_Inp60:
	cmp [bLast64],0D0h	;last value written to 64 was "read output port"?
	jnz @@default
A20_Inp92::
	in al,dx
	and al, not 2
	mov cl, [bA20Stat]
	shl cl, 1
	or	al, cl
    ret
@@default:
	in al,dx
    clc
    ret
A20_Handle60 endp

A20_Handle64 proc
	test bl,2
    jz @@default
	mov [bLast64],al	;save last value written to port 64h
if ?A20DBG	  
	cmp al,0D1h
	jnz @@nokbcout
	@DbgOutS <"write to port 64h, al=">,?A20DBG
	@DbgOutB al,?A20DBG
	@DbgOutS <10>,?A20DBG
@@nokbcout:    
endif	 
	out dx, al
	ret
@@default:
	stc
    ret
A20_Handle64 endp

A20_Handle92 proc
	test bl,2			;is it in or out?
    jz A20_Inp92
	@DbgOutS <"write to port 92h, al=">,?A20DBG
	@DbgOutB al,?A20DBG
	@DbgOutS <10>,?A20DBG
	push eax
	shr al,1
	and al,1
	mov bA20Stat,al
	call A20_Set
	pop eax
	or	al, 2		;dont allow disable
	out dx, al
	ret
A20_Handle92 endp

	
endif

if ?EMMXXXX0

;--- read ioctl EMMXXXX0 device
;--- all registers may be modified!
;--- DS,ES=FLAT
;--- ECX=request header

IoctlRead proc 
	mov 	eax, [ecx+14]	;get buffer real-mode address
	movzx	ebx, ax
	shr		eax, 12
	and		al, 0F0h
	add		ebx, eax        ;ebx=buffer linear address

	movzx	edx, word ptr [ecx+18]	;get buffer size
	mov		al, [EBX+0]
	cmp 	al, 0			;get "API"
	jz		@@func00
if 0	
	cmp		al, 1			;GEMMIS not supported
	jz		@@func01
endif	 
	cmp 	al, 2			;get version
	jz		@@func02
	cmp 	al, 4			;get Emm386 resident segment/size?
	jz		@@func04
	cmp 	al, 6			;get system vars
	jz		@@func06
	cmp 	al, 7			;get UMBs
	jz		@@func07
	jmp		@@error
@@func00:	 
	cmp		edx,6	;bytes to read
	jb		@@error
	mov 	word ptr [ebx+0], 0028h
	mov 	dword ptr [ebx+2], 0	;API entry
	jmp		@@ok
@@func02:	 
	cmp		edx,2	;bytes to read
	jb		@@error
	mov 	word ptr [ebx+0], ?VERSIONHIGH + ?VERSIONLOW * 256
	jmp		@@ok
@@func04:	 
	cmp		edx,4	;bytes to read
	jb		@@error
    mov		eax,[dwRSeg]
	mov 	[ebx+0], eax
	jmp		@@ok
@@func06:	 

	cmp		edx,16	;bytes to read
	jb		@@error
	mov		al, [bNoEMS]
	mov 	[ebx].EMX06.e06_NoEMS, al
	xor		eax, eax
	cmp		[bNoFrame],0
	jnz		@@nopf
	mov		ah, [PAGE2SEGM]			;get mapping of phys page 0
@@nopf:    
	mov 	[ebx].EMX06.e06_Frame, ax
	mov		al, [bNoVCPI]
	mov 	[ebx].EMX06.e06_NoVCPI, al
	cmp		edx,24					;to get VCPI memory info, we need 24 byte
	jb		@@novcpimem
	mov		eax, [dwMaxMem16K]		;VCPI in 16 kB 
	shl		eax, 2
	mov 	[ebx].EMX06.e06_VCPITotal, eax
	call	Pool_Get4KPageCount	;return total in edx, free in eax
	sub		edx, eax
	mov 	[ebx].EMX06.e06_VCPIUsed, edx
@@novcpimem:	
if ?DMA    
	mov		eax, [DMABuffStartPhys]
else
	xor		eax, eax
endif	 
	mov 	[ebx].EMX06.e06_DMABuff, eax
if ?DMA    
	mov		eax, [DMABuffSize]
	shr		eax, 10
else
	xor		eax, eax
endif	 
	mov 	[ebx].EMX06.e06_DMASize, ax
if ?VME    
	mov		al,1
	test	byte ptr [dwFeatures],2
	jz		@@novme
	@mov_eax_cr4
	and 	al,1
	xor		al,1
@@novme:
	mov		[ebx].EMX06.e06_NoVME,al
endif	 
if ?PGE    
	mov		al,1
	test	byte ptr [dwFeatures+1],20h
	jz		@@nopge
	@mov_eax_cr4
	shr		al,7
	xor		al,1
@@nopge:
	mov		[ebx].EMX06.e06_NoPGE,al
endif	 
if ?A20PORTS or ?A20XMS
	mov		al,[bNoA20]
	mov		[ebx].EMX06.e06_NoA20,al
endif
@@ok:
	clc
	ret
@@error:
	stc
	ret
@@func07:	 
	cmp		dx, UMB_MAX_BLOCKS * size UMBBLK	;buffer large enough to get the UMB entries?
	jb		@@error
	mov		edi, ebx
	mov		esi, offset UMBsegments
	mov		ecx, UMB_MAX_BLOCKS
	rep		movs dword ptr [edi], [esi]
	clc
	ret
	
IoctlRead endp    

;--- all registers may be modified!
;--- DS,ES=FLAT
;--- ECX=request header
;--- the "update" command is 15

IoctlWrite proc 

	mov 	eax, [ecx+14];buffer
	movzx	esi, ax
	shr		eax, 12
	and		al, 0F0h
	add		esi, eax

	lods	byte ptr [esi]	;get function to run
	cmp 	al, 15
	jz		@@func15
@@error:
	stc
	ret
@@func15:

;--- esi -> EMX15W variable

	mov		al, [ecx+18]	;buffer size
    cmp		al, 5
    jb		@@error
	
	lods	byte ptr [esi]	;e15_bVME
if ?VME
	cmp		al,-1
	jz		@@novme
    xor		al,1
    call	SetVME
@@novme:
endif
	lods	byte ptr [esi]	;e15_bA20
if ?A20PORTS or ?A20XMS
	cmp		al,-1
	jz		@@noa20
	and		al, al
	jz		@@a20emuon
	push	eax
	mov		al,1
	call	A20_Set		;enable A20 gate
	pop		eax
@@a20emuon:
	mov		[bNoA20], al
@@noa20:
endif	 
	lods	byte ptr [esi]	;e15_bVCPI
	cmp		al,-1
	jz		@@novcpi
	mov		[bNoVCPI],al
@@novcpi:
	lods	byte ptr [esi]	;e15_bPGE
if ?PGE
	cmp		al,-1
	jz		@@nopge
	test	byte ptr [dwFeatures+1], 20h   ;PGE supported?
	jz		@@nopge
	pushad
	and		al,1
	xor		al,1
	mov		[bPageMask],al
	@GETPTEPTR edi, 1000h, 1	;start of pagetab 0
	mov		ecx,110h+1			;00000000-00110FFF
@@FILL_PAGETAB0:
	mov		edx, [edi]
	and		dh,not 1		;mask out G
	or		dh,al
	MOV 	[EDI],EDX
	ADD 	EDI,4
	loop	@@FILL_PAGETAB0
	@mov_ecx_cr4
	shl		al,7
	and		cl,not 80h
	or		cl,al
	@mov_cr4_ecx
	popad
@@nopge:
endif
	clc
	ret
IoctlWrite endp	

endif

    align 4

;--- XMS UMB handler
;--- EAX holds value from v86 mode
;--- EBX, ECX are saved on stack

umb_handler proc

	push esi
	mov	ecx, UMB_MAX_BLOCKS		;set CL to max umbs, clear CH!
	
	cmp	ah,11h			;free UMB, DX=segment address to release
	je	UMBfree
	cmp	ah,12h			;realloc UMB, DX=segment to resize, BX=new size
	je	UMBrealloc

;--- UMBalloc

UMBalloc:		;DX=size of block in paragraphs

if ?UMBDBG
	@DbgOutS <"UMBalloc enter, DX=">,1
	@DbgOutW dx,1
	@DbgOutS <10>,1
endif

					; find first available memory block
	mov	esi, offset UMBsegments
	xor	ebx,ebx		; holds largest too-small block size

@@UMBloop:
	cmp	[esi].UMBBLK.wSegm,0	; see if valid UMB
	je	@@UMBnext			; no
	test BYTE PTR [esi].UMBBLK.wSize+1,80h	; see if UMB already allocated (high bit size set)
	jne	@@UMBnext			;  yes
	cmp	dx,[esi].UMBBLK.wSize; dx = requested block size (high bit of UMB size known reset)
	jbe	@@UMBfound			; enough memory available in UMB
	mov	ch,1				; flag UMB was found, although too small
	cmp	bx,[esi].UMBBLK.wSize
	ja	@@UMBnext
	mov	bx,[esi].UMBBLK.wSize; update largest too-small block size
@@UMBnext:
	add	esi,size UMBBLK
	dec cl
	jnz @@UMBloop
	or	ch,ch
	jne	@@umb_too_small
	mov	bl,0B1h		; error "no UMB's are available"
	xor	dx,dx
	xor	eax,eax		; flag failure
	jmp	@@umb_exit
@@umb_too_small:
	mov	dx,bx		; return largest UMB in DX
	mov	bl,0B0h		; error "only smaller UMB available"
	xor	eax,eax		; flag failure
	jmp @@umb_exit

@@UMBfound:

; see if actual UMB size exceeds request size by >=2K
	mov	ax,80h				; 128 paras == 2K
	add	ax,dx
	cmp	ax,[esi].UMBBLK.wSize
	ja	@@good_umb			; can't split it, just use it

;  2K or over would be unused, see if we can split the block

	mov	cl, UMB_MAX_BLOCKS
	mov	ebx, offset UMBsegments

@@splitloop:
	cmp	WORD PTR [ebx].UMBBLK.wSegm,0
	jne	@@splitnext

; split the block
	mov	ax,dx
	add	ax,7fh
	and	ax,0ff80h			; round up allocation to next 2K in paras
	mov	cx,[esi].UMBBLK.wSegm
	add	cx,ax
	mov	[ebx].UMBBLK.wSegm,cx	; new block has segment offset of old block+allocation
	mov	cx,[esi].UMBBLK.wSize	; get original UMB block size, in paras
	sub	cx,ax				; subtract allocation
	mov	[ebx].UMBBLK.wSize,cx	; update new block with old block size minus allocation
	mov	[esi].UMBBLK.wSize,ax	; update original UMB block size to allocation

	jmp	@@good_umb

@@splitnext:
	add	ebx,size UMBBLK
	dec	cl
	jne	@@splitloop

@@good_umb:
	mov	dx,[esi].UMBBLK.wSize	; actual block size to dx
	or	BYTE PTR [esi].UMBBLK.wSize+1,80h	; flag UMB allocated
	mov	bx,[esi].UMBBLK.wSegm	; get UMB address in bx
	mov word ptr [ESP+4].V86FRGP.gEBX,bx
	mov ax,1
	
@@umb_exit:	 
if ?UMBDBG
	@DbgOutS <"UMB exit, ax=">,1
	@DbgOutW ax,1
	@DbgOutS <", bx=">,1
	@DbgOutW bx,1
	@DbgOutS <", dx=">,1
	@DbgOutW dx,1
	@DbgOutS <10>,1
endif	 
	pop	esi
	mov word ptr [esp].V86FRGP.gEAX, ax
	and ax,ax
	jnz @@umbexit_noerror
	mov byte ptr [esp].V86FRGP.gEBX, bl
@@umbexit_noerror:
	@v86popreg
	add esp,4+4
	iretd

;--- UMBFree

UMBfree:
if ?UMBDBG
	@DbgOutS <"UMBfree enter, DX=">, 1
	@DbgOutW DX, 1
	@DbgOutS <10>, 1
endif    
    
	mov	esi,offset UMBsegments
	xor	eax,eax			; flag failure
@@freeloop:
	cmp	[esi].UMBBLK.wSegm,dx	; see if matches existing UMB allocation
	je	@@free_found
	add	esi,size UMBBLK
	loop @@freeloop
@@free_error:	 
	mov	bl,0b2h			; invalid UMB segment number error code
	jmp @@umb_exit

@@free_found:
	test byte ptr [esi].UMBBLK.wSize+1,80h	; is block allocated?
	jz @@free_error
	and	BYTE PTR [esi].UMBBLK.wSize+1,7fh	; flag UMB not allocated
	inc	eax			; flag success
	jmp @@umb_exit

;--- UMBrealloc

UMBrealloc:		;DX=segment, BX=new size of block in paragraphs

	mov ebx,[esp+4].V86FRGP.gEBX	; restore EBX
    
if ?UMBDBG
	@DbgOutS <"UMBrealloc enter, DX=">, 1
    @DbgOutW DX,1
	@DbgOutS <", BX=">, 1
    @DbgOutW BX,1
	@DbgOutS <10>, 1
endif

	mov	esi,offset UMBsegments
	xor	eax,eax			; flag failure
@@realloop:
	cmp	[esi].UMBBLK.wSegm,dx	; see if matches existing UMB allocation
	je	@@real_found
	add	esi, size UMBBLK
	loop @@realloop
	mov	bl,0b2h			; invalid UMB segment number error code
	jmp @@umb_exit

@@real_found:
	test byte ptr [esi].UMBBLK.wSize+1,80h	;is it an allocated block?
	jnz @@umbreal_error1
	mov cx, [esi].UMBBLK.wSize
	and ch, 7Fh
	cmp bx, cx
	ja @@umbreal_error2
	inc	eax			; flag success
	jmp @@umb_exit
@@umbreal_error1:	;block is not allocated
	mov bl,0B2h
	jmp @@umb_exit
@@umbreal_error2:	;block is too small
	mov dx,cx
	mov bl,0B0h
	jmp @@umb_exit
	
umb_handler endp


if ?VDS

;--- flags in DX used by various VDS functions

VDSF_COPY		equ 02h
VDSF_NOBUFFER	equ 04h
VDSF_64KALIGN	equ 10h
VDSF_128KALIGN	equ 20h
	

; entry ax = 4k page number 
; exit edx = physical address

vds_GetPhysAddr	PROC
	movzx edx, ax
	@GETPTE edx, edx*4+1000h
	and dx, 0F000h
	ret
vds_GetPhysAddr	ENDP

;--- test if a region is contiguous and crosses 64/128 kB borders
;--- ax = start page, bx = end page, dx=flags
;--- for first 4 MB only
;--- returns initial physical address in EAX
;--- in case of error: size which would be ok in edx

VDS_retcode equ <byte ptr [ebp+4].V86FRGP.gIntNo>	;save VDS returncode on stack

vds_contiguous proc

	push ecx
	push esi
	push edi
	movzx eax, ax

	@GETPTEPTR esi, 1000h, 1	;get start of page tab 0

	mov edi, [esi+eax*4]
	and di, 0F000h
	push edi
@@nextitem:    
	cmp ax, bx
	jz @@iscontiguous
	inc eax
	add edi, 1000h
	mov ecx, [esi+eax*4]
	and cx,0F000h
	cmp edi, ecx
	je @@nextitem
	pop eax
	mov	VDS_retcode,1
	mov edx, edi
	sub edx, eax
@@failed:
	pop edi
	pop esi
	pop ecx
	stc
	ret
@@failed2:			;failed cause 64 kb border crossing
	mov edx, esi
	xor dx, dx
	sub edx, eax
	mov	VDS_retcode,2
	jmp @@failed
@@failed3:			;failed cause 128 kb border crossing
	mov edx, esi
	and edx, 0FFFE0000h
	sub edx, eax
	mov	VDS_retcode,2
	jmp @@failed
@@iscontiguous:
	pop edi			;get start physical address
	mov eax, edi
	mov esi, ecx	;save ecx in case of border cross errors
	shr edi, 16
	shr ecx, 16
	test dl, VDSF_64KALIGN	;check for 64 kb border cross?
	jz @@no64check
	cmp di,cx
	jnz @@failed2
@@no64check:
	test dl,VDSF_128KALIGN	;check for 128 kb?
	jz @@no128check
	shr di,1
	shr cx,1
	cmp di,cx
	jnz @@failed3
@@no128check:
	pop edi
	pop esi
	pop ecx
	ret
	align 4
	
vds_contiguous endp

;-- simplified version of vds_contiguous
;-- assumes linear == physical (for regions above 400000h)
;-- eax = linear start address
;-- ebx = size of region
;-- ret C -> boundary error (eax==size ok)
;-- ret NC -> ok, eax == physical address [== linear address]

vds_contiguous2 proc

;-- see if boundary alignment check

	test	dl,VDSF_64KALIGN or VDSF_128KALIGN	; boundary check?
	jz	@@ok
	mov	ecx,eax
	lea	ebx,[eax+ebx-1]
	shr	ebx,16		; convert start/end to alignment 64K frames
	shr	ecx,16
	test dl,VDSF_64KALIGN
	jne	@@aligncheck	; yes, if 64K works, then 128K will too
	shr	ecx,1			;-- 128K alignment check
	shr	ebx,1
@@aligncheck:
	cmp	bx,cx
	je	@@ok
	inc	ecx			; ecx == starting alignment frame+1
	shl	ecx,16		; convert to next alignment frame address
	test dl,VDSF_64KALIGN
	jne	@@check2
	shl	ecx,1
@@check2:
	sub	ecx,eax		; get bytes to next alignment frame address from start
	mov	VDS_retcode,2	; region crossed alignment boundary error code
	mov	eax,ecx		; eax == size ok
	stc
@@ok:
	ret
	align 4
vds_contiguous2 endp

	align 4

;--- bit vector which vds func needs ES:DI translation
;------ CBA9876543210
bDDS dd 0011111111000b

;--- DDS, used by 03-04, 07-08, 09-0A

DDS struc
dwSize	dd ?	;+0  size of region
dwOfs	dd ?	;+4  offset virtual start address
wSeg	dw ?	;+8  segment/selector virtual start address (or 0000)
wID		dw ?	;+10 buffer ID
dwPhys	dd ?	;+12 physical address
DDS ends

;--- EDDS, used by 05-06

EDDS struc
dwSize	dd ?	;+0
dwOfs	dd ?	;+4
wSeg	dw ?	;+8
wRes	dw ?	;+10
wNumAvail	dw ?	;+12
wNumUsed	dw ?	;+14
EDDS ends

;--- EDDS suffix for regions

EDDSRG struc
dwPhysAddr	dd ?	;+16
dwSizeRg	dd ?	;+20
EDDSRG ends

;--- EDDS suffix for PTEs

EDDSPT struc
dwPTE		dd ?	;+16
EDDSPT ends

;--- EDDS, used by 05-06, for PTEs

;--- handle VDS in protected mode
;--- DS=flat, others=NULL
;--- ESP->V86FRGP (includes EAX, EBX, ECX)
;--- other std registers not modified
;--- segment registers in V86FRGP

vds_handler proc

	push ebp
	mov ebp, esp
	push edi
	mov eax, [EBP+4].V86FRGP.gEAX
	cmp ah,81h
	jnz @@isnotvds
	movzx eax,al

if 0;?VDSDBG
	@DbgOutS <"VDS entry, al=">,1
	@DbgOutB al,1
	@DbgOutS <10>,1
endif
	mov	VDS_retcode,0

	cmp al,VDS_MAX
	jb @@vds_funcok
@@isnotvds:
	xor eax, eax
@@vds_funcok:
	bt bDDS, eax
	jnc @@nodds
;--- make DDS/EDDS accessible
	movzx	edi,di
	movzx	ecx,word ptr [ebp+4].V86FRGP.gES 
	shl		ecx, 4
	add		edi, ecx
@@nodds:
	mov		ecx,ds
	mov		es,ecx
	call	[VDS_CALL_TABLE+eax*4]
	pushfd
    call	SimIRET
    pop	eax
	and eax,1
    and [ebp+4].V86FRGP.gEFL,not 1
    or  [ebp+4].V86FRGP.gEFL,eax
if ?MASM
vds_exit2::
else
vds_exit2:	  
endif
	pop edi
	pop ebp
    @v86popreg
	add esp,4+4		;skip error code + int#
	IRETD
vds_handler endp

vds_reserved proc
	mov eax, [OldInt4B]
    and eax, eax
    jnz @@callvdsold
    stc
    ret
@@callvdsold:    
if ?VDSDBG
	@DbgOutS <"VDS_reserved, jmp to cs:ip=">,1
	@DbgOutD eax,1
	@DbgOutS <10>,1
	@WaitKey 1, 0
endif	 
	mov	word ptr [ebp+4].V86FRGP.gEIP, AX
	shr eax, 16
	mov	[ebp+4].V86FRGP.gCS, EAX
	pop eax				;skip return address
	jmp vds_exit2
vds_reserved endp

vds_scatterunlock proc	;dummy, just fall through vds_ok
vds_scatterunlock endp

vds_ok proc
	clc
	ret
vds_ok endp


vds_unsup proc
	mov al, 0Fh		; error "function not supported"
vds_unsup endp		; fall through

vds_fail proc
	mov byte ptr [ebp+4].V86FRGP.gEAX,al	; set code in AL
	stc
	ret
vds_fail endp
	
;--- int 4b, ax=8102h

vds_version proc
	mov al,10h	;"reserved bit set in dx"
	and dx,dx
	jnz vds_fail
	mov word ptr [ebp+4].V86FRGP.gEAX, 100h	; major/minor spec version
	mov word ptr [ebp+4].V86FRGP.gEBX, 1	; product number
	mov	word ptr [ebp+4].V86FRGP.gECX, 1	; product revision
	mov di, word ptr [DMABuffSize+0]
	mov word ptr [ebp-4],di					; EDI is saved on stack!
	mov si, word ptr [DMABuffSize+2]
	xor	dx,dx								; flags
	jmp	vds_ok
vds_version endp

;--- int 4b, ax=8107h, request DMA buffer
;--- DX = flags, bit 1: copy data into buffer
;--- ES:DI -> DDS

vds_reqbuff proc

	mov [edi].DDS.wID, 0
	mov ecx, [edi].DDS.dwSize
if ?VDSDBG
	@DbgOutS <"VDS request buff, flgs=">,1
	@DbgOutW dx,1
	@DbgOutS <" siz=">,1
	@DbgOutD ecx,1
	@DbgOutS <10>,1
endif
	and ecx, ecx
	jz vds_ok
	mov	al,5		; error "region too large for buffer"
	add ecx, 400h-1	; align to kB
	and cx, 0FC00h
	cmp ecx, [DMABuffSize]
	ja vds_fail
	
;--- scan for a free region in buffer

	mov eax, [DMABuffSize]
	shr eax, 10			; transform in KB
	inc eax				; the bit vector is 1-based!
	shr ecx, 10
	inc ecx				; test for 1 bit more!
@@rescan:	 
	mov ebx, ecx
@@nextbit:
	dec eax
	js @@fail
	bt [DMABuffFree], eax
	jnc @@rescan
	dec ebx
	jnz @@nextbit
	
	dec ecx		; now use the real size
	inc eax		; skip the last bit found

;--- a free region large enough found

if ?VDSDBG
	@DbgOutS <"VDS free region found at ">,1
	@DbgOutW ax,1
	@DbgOutS <10>,1
endif

	mov ebx, eax
	mov [edi].DDS.wID, ax
@@marknextbit:
	btr [DMABuffFree], ebx
	inc ebx
	loop @@marknextbit
	dec eax
	shl eax, 10		;convert to byte
	add eax, [DMABuffStartPhys]
	mov [edi].DDS.dwPhys, eax
if ?VDSDBG
	@DbgOutS <"VDS req buff, phys=">,1
	@DbgOutD eax,1
	@DbgOutS <10>,1
endif
	test dl,VDSF_COPY		; copy into buffer?
	jz vds_ok
	xor ecx, ecx
	jmp vds_copyinbuffEx
@@fail:    
if ?VDSDBG
	@DbgOutS <"VDS req buff aborted, eax=">,1
	@DbgOutD eax,1
	@DbgOutS <", ebx=">,1
	@DbgOutD ebx,1
	@DbgOutS <10>,1
endif
	mov al, 6				; error "buffer currently in use"
	jmp vds_fail
	
vds_reqbuff endp

;--- int 4b, ax=8104h, unlock region
;--- DX=Flags, bit 1: copy data out of buffer
;--- ES:DI -> DDS
;--- "unlock region" is the same as "release buffer"

vds_unlock proc

if 0;?VDSDBG
	@DbgOutS <"VDS unlock flgs=">,1
	@DbgOutW dx,1
	@DbgOutS <" addr=">,1
	@DbgOutW [edi].DDS.wSeg,1
	@DbgOutS <":">,1
	@DbgOutD [edi].DDS.dwOfs,1
	@DbgOutS <" siz=">,1
	@DbgOutD [edi].DDS.dwSize,1
	@DbgOutS <" id=">,1
	@DbgOutW [edi].DDS.wID,1
	@DbgOutS <" phys=">,1
	@DbgOutD [edi].DDS.dwPhys,1
	@DbgOutS <10>,1
endif
	
vds_unlock endp	;fall through


;--- int 4b, ax=8108h, release DMA buffer
;--- DX = flags, bit 1: copy data out of buffer
;--- ES:DI -> DDS

vds_relbuff proc

	movzx ebx, [edi].DDS.wID
if ?VDSDBG
	@DbgOutS <"VDS release buff, flgs=">,1
	@DbgOutW dx,1
	@DbgOutS <" DDS=[siz=">,1
	@DbgOutD [edi].DDS.dwSize,1
	@DbgOutS <" ID=">,1
	@DbgOutW bx,1
	@DbgOutS <" addr=">,1
	@DbgOutD [edi].DDS.dwPhys,1
	@DbgOutS <"]",10>,1
endif
if 0;?VDSDBG
	@DbgOutS <"DMA buffer bits:">,1
	@DbgOutD DMABuffFree+0,1
	@DbgOutS <"-">,1
	@DbgOutD DMABuffFree+4,1
	@DbgOutS <"-">,1
	@DbgOutD DMABuffFree+8,1
	@DbgOutS <"-">,1
	@DbgOutD DMABuffFree+12,1
	@DbgOutS <"-">,1
	@DbgOutB <byte ptr DMABuffFree+16>,1
	@DbgOutS <10>,1
endif
	and ebx, ebx
	jz vds_ok

	mov al,0Ah	;"invalid buffer id"
	cmp bx, ?DMABUFFMAX
	ja vds_fail

;--- the bit at position -1 must be "1"
;--- and bit 0 must be "0"
;--- the region ends with a "1" bit

	movzx ebx, bx
	dec ebx
	bt [DMABuffFree], ebx
	jnc vds_fail
	inc ebx
	bt [DMABuffFree], ebx
	jc vds_fail

	test dl, VDSF_COPY
	jz @@nextbit
	push ebx
	xor  ecx, ecx
	call vds_copyoutbuffEx
	pop ebx    
	jc @@error

@@nextbit:
	bts [DMABuffFree], ebx
	inc ebx
	jnc @@nextbit
	jmp vds_ok
@@error:
	ret

vds_relbuff endp

;-- test if a buffer is valid and as large as required
;-- used by copy into/out of DMA buffer
;-- ECX == offset in buffer
;-- returns with EBX == linear address of src/dst in DMA-buffer

vds_testbuff proc	 

	movzx ebx, [edi].DDS.wID
if ?VDSDBG
	@DbgOutS <"VDS testbuff id=">,1
	@DbgOutW bx,1
	@DbgOutS <10>,1
endif	 
	mov	al,0Ah					;"invalid buffer ID"
	and ebx, ebx
	jz	@@fail
	cmp bx, ?DMABUFFMAX
	ja	@@fail
	dec ebx
	bt [DMABuffFree], ebx	;bit at -1 *must* be set
	jnc @@fail
	inc ebx
	bt [DMABuffFree], ebx	;bit at 0 bit *must* be clear
	jc @@fail
	mov eax, ecx			;eax == offset in DMA buffer
	
	mov ecx, [edi].DDS.dwSize
	and ecx, ecx
	jz @@ok

	lea ecx, [ecx+eax+3FFh]
	shr ecx, 10

	push eax
	lea eax, [ebx+ecx]
	cmp eax, ?DMABUFFMAX
	pop eax
	ja @@fail3
	
	push ebx
@@nextbit:
	bt [DMABuffFree], ebx
	jc @@fail2
	inc ebx
	dec ecx
	jnz @@nextbit
	pop ebx    
	
	dec ebx
	shl ebx, 10
	add ebx, [DMABuffStart]
	add ebx, eax
@@ok:	 
if ?VDSDBG
	@DbgOutS <"VDS testbuff ok, buff start (linear)=">,1
	@DbgOutD ebx,1
	@DbgOutS <10>,1
endif	 
	clc
	ret
@@fail2:
if ?VDSDBG
	@DbgOutS <"VDS testbuff @@fail2, ebx=">, 1
	@DbgOutD ebx, 1
	@DbgOutS <" ecx=">, 1
	@DbgOutD ecx, 1
	@DbgOutS <10>, 1
endif	 
	pop ebx
@@fail3:
	mov al,0Bh	;"copy out of buffer range"
@@fail:
	@DbgOutS <"VDS testbuff failed, al=">,?VDSDBG
	@DbgOutB al,?VDSDBG
	@DbgOutS <10>,?VDSDBG
	stc
	ret
	
vds_testbuff endp

;--- int 4b, ax=8109h, copy into DMA buffer
;--- DX = 0000
;--- BX:CX = offset in buffer
;--- ES:DI -> DDS

vds_copyinbuff proc

	mov cx, word ptr [ebp+4].V86FRGP.gEBX
	shl ecx, 16
	mov cx, word ptr [ebp+4].V86FRGP.gECX	;ecx == offset in DMA buffer
if ?MASM	
vds_copyinbuffEx::
else
vds_copyinbuffEx:
endif
	call vds_testbuff
	jc vds_fail
	mov ecx, [edi].DDS.dwSize
	movzx eax, [edi].DDS.wSeg
	shl eax, 4
	add eax, [edi].DDS.dwOfs
if ?VDSDBG
	@DbgOutS <"VDS copyinbuff src=">,1
	@DbgOutD eax,1
	@DbgOutS <" dst=">,1
	@DbgOutD ebx,1
	@DbgOutS <" siz=">,1
	@DbgOutD ecx,1
	@DbgOutS <10>,1
endif	 
	pushad
	mov esi, eax
	mov edi, ebx
	mov dl,cl
	shr ecx, 2
	cld
	rep movs dword ptr [edi], [edi]
	mov cl,dl
	and cl, 3
	rep movs byte ptr [edi], [edi]
	@BIG_NOP
	popad
	jmp	vds_ok
	
vds_copyinbuff endp

;--- int 4b, ax=8109h, copy out of DMA buffer
;--- DX = 0000
;--- BX:CX = offset in buffer
;--- ES:DI -> DDS

vds_copyoutbuff proc

	mov cx, word ptr [ebp+4].V86FRGP.gEBX
	shl ecx, 16
	mov cx, word ptr [ebp+4].V86FRGP.gECX	;ecx == offset in DMA buffer
if ?MASM	
vds_copyoutbuffEx::    
else
vds_copyoutbuffEx:
endif
	call vds_testbuff
	jc vds_fail
	mov ecx, [edi].DDS.dwSize
	movzx eax, [edi].DDS.wSeg
	shl eax, 4
	add eax, [edi].DDS.dwOfs
	pushad
	mov esi, ebx
	mov edi, eax
	mov dl,cl
	shr ecx, 2
	rep movs dword ptr [edi], [edi]
	mov cl,dl
	and cl, 3
	rep movs byte ptr [edi], [edi]
	@BIG_NOP
	popad
	jmp	vds_ok

vds_copyoutbuff endp

;--- int 4b, ax=8103h, lock region
;--- ES:DI -> DDS
;--- DX=Flags
;--- 0:reserved
;--- 1:data should be copied into buffer (if necessary) [requires 2 cleared]
;--- 2:buffer disable (buffer should not be allocated if noncontiguous or crosses 64/128 kB)
;--- 3:dont attempt automatic remap
;--- 4:region must not cross 64 kb border
;--- 5:region must not cross 128 kb border

;--- there are several cases:
;--- 1. region is contiguous (and does not cross borders)
;---	return with carry clear, buffer ID == 0
;--- 2. region is not contiguous and/or does cross borders
;---	2.1 buffer disable flag set
;---		return with carry set and code 1,2,3 
;---	2.2 buffer disable flag cleared
;---	   2.2.1 buffer available and large enough
;---			 alloc buffer
;---			 if copy required copy data into buffer
;---			 return with carry clear and buffer ID <> 0
;---	   2.2.2 buffer too small
;---			 return with carry set and code 5
;---	   2.2.3 buffer not available
;---			 return with carry set and code 6
;---
;---	 field "Physical Address" may be filled by "Lock Region"

vds_lock proc

	push	edx

if ?VDSDBG
	@DbgOutS <"VDS lock flgs=">,?VDSDBG
	@DbgOutW dx,?VDSDBG
	@DbgOutS <" addr=">,?VDSDBG
	@DbgOutW [edi].DDS.wSeg,?VDSDBG
	@DbgOutS <":">,?VDSDBG
	@DbgOutD [edi].DDS.dwOfs,?VDSDBG
	@DbgOutS <" siz=">,?VDSDBG
	@DbgOutD [edi].DDS.dwSize,?VDSDBG
	@DbgOutS <10>,?VDSDBG
endif

	xor eax, eax
	mov	ebx, [edi].DDS.dwSize	; region size
	cmp	ebx, eax
if 0
	je	@@locksuccess		; zero byte-sized region always works
else
	setz al 		;size 0 is always ok, but the physical address must be set
	add ebx, eax	;as well. So handle size 0 as if it is size 1
endif
	mov ax, [edi].DDS.wSeg
	shl	eax,4
	add	eax, [edi].DDS.dwOfs
;	 jc --> overflow error
	mov	ecx,eax		; ecx == start linear address
	lea eax, [eax+ebx-1]
	cmp eax, 400000h; region in first 4 MB ?
	jb	@@below4MB
	mov	eax,ecx		; restore linear address to eax

;-- assume linear == physical
;-- call a simplified version of vds_contiguous

	call vds_contiguous2
	jnc @@locksuccess
	jmp @@lenfail

@@below4MB:
	mov ebx, eax	; ebx == final linear address
	shr	ebx,12		; convert to 4K frame
	mov eax, ecx
	shr	eax,12		; convert to 4K frame
	call vds_contiguous
	jc @@notcontiguous
	and ch, 0Fh
	or ax, cx
	jmp @@locksuccess

; physical memory is noncontiguous, error code is 1 or 2
; return maximum length which would be ok

@@notcontiguous:

	mov eax, edx
	and	ecx, 0fffh
	sub eax, ecx
	mov dx, [esp]		;restore flags
	test dl, VDSF_NOBUFFER ;buffering disabled?
	jnz @@nobuffering
	push eax
	call vds_reqbuff
	pop ecx
	jnc @@lockok
	mov VDS_retcode,al
	mov eax, ecx
@@nobuffering:	  

if ?VDSDBG
	@DbgOutS <"VDS lock failed, ret size=">,1
	@DbgOutD eax,1
	@DbgOutS <" addr=">,1
	@DbgOutD edx,1
	@DbgOutS <" rc=">,1
	@DbgOutB VDS_retcode,1
	@DbgOutS <10>,1
endif

@@lenfail:
	mov	[edi].DDS.dwSize,eax	; update maximum contiguous length
	mov	[edi].DDS.wID,0 	  ; zero buffer id?
	pop edx
	mov	al,VDS_retcode
	jmp	vds_fail

@@locksuccess:
if ?VDSDBG
	@DbgOutS <"VDS lock ok, ret size=">,1
	@DbgOutD [edi].DDS.dwSize,1
	@DbgOutS <" phys=">,1
	@DbgOutD eax,1
	@DbgOutS <10>,1
endif
	mov	[edi].DDS.dwPhys,eax ; physical address
	mov	[edi].DDS.wID,0 	 ; zero buffer id

@@lockok:
	pop	edx
	jmp vds_ok

vds_lock endp

;--- int 4b, ax=8105h
;--- inp: ES:DI -> EDDS
;---	  DX=flags
;--- out: NC
;---	  wNumUsed: no of entries used to describe region
;---	  regions/PTEs behind EDDS are set
;--- error: C set
;---		AL=error code
;---		dwSize: length that can be locked with current entries
;---		wNumUsed: number of entries required to fully describe region!

vds_scatterlock proc

	push	edx

if ?VDSDBG
	@DbgOutS <"VDS scatlock flgs=">,1
	@DbgOutW dx,1
	@DbgOutS <" addr=">,1
	@DbgOutW [edi].EDDS.wSeg,1
	@DbgOutS <":">,1
	@DbgOutD [edi].EDDS.dwOfs,1
	@DbgOutS <" siz=">,1
	@DbgOutD [edi].EDDS.dwSize,1
	@DbgOutS <" avl=">,1
	@DbgOutW [edi].EDDS.wNumAvail,1
	@DbgOutS <10>,1
endif

	mov	ebx,[edi].EDDS.dwSize
	xor	ecx,ecx		; cx holds entries used
	mov	eax,ecx		; zero eax for later calcs
	mov	[edi].EDDS.wNumUsed,cx	; EDDS number entries used
	cmp	ebx,ecx
	je	@@lockok	; zero sized region always successful

	mov	ax,[edi].EDDS.wSeg
	shl	eax,4
	add	eax,[edi].EDDS.dwOfs

	test dl,40h		; PTEs flagged?
	jne	@@getptes

	mov	edx,eax			; edx == start linear address
	shr	eax,12			; convert start to 4K frame
	cmp	eax,256
	jb	@@checklock		; inside of UMB remapping range (<1M)

; outside of UMB remapping range, assuming linear == physical
	mov	[edi].EDDS.wNumUsed,1	; one region used/needed
	cmp	cx,[edi].EDDS.wNumAvail
	jnc	@@notenoughregions
	mov	[edi+size EDDS].EDDSRG.dwPhysAddr,edx	; region physical address
	mov	[edi+size EDDS].EDDSRG.dwSizeRg,ebx		; region size
@@lockok:
	pop edx
	jmp	vds_ok
@@notenoughregions:
	pop edx
	mov	al,9		;error "NumAvail too small"
	jmp vds_fail

;--- return regions, 1. MB

@@checklock:
	push	esi

	push	edx			; save start linear address
	
	lea	ebx,[ebx+edx-1]	; ebx == final linear address
	shr	ebx,12			; convert to 4K frame
	call vds_GetPhysAddr	; expects page in AX!
	mov	esi,edx			; current physical address of 4K page
	cmp	cx,[edi].EDDS.wNumAvail
	jnc	@@bumpused
	mov	[edi+size EDDS].EDDSRG.dwPhysAddr,edx
	mov	[edi+size EDDS].EDDSRG.dwSizeRg,0
	jmp	@@bumpused

@@entryloop:
	cmp	cx,[edi].EDDS.wNumAvail	; entries available count
	jnc	@@bumpused
	mov	[edi+ecx*8+size EDDS].EDDSRG.dwSizeRg,1000h	; init region size
	mov [edi+ecx*8+size EDDS].EDDSRG.dwPhysAddr, esi

@@bumpused:
	inc	[edi].EDDS.wNumUsed	; bump count of used/needed entries

@@nextframe:
	inc	eax				; next linear page frame
	cmp	eax,ebx
	ja	@@scatdone		; no more regions to map
	cmp	ah,0			; page below 100h?
	jz	@@next2			; not at end of first 1M
	cmp	ax,256
	ja	@@scatdone		; finishing off final region entry
	cmp	esi,0ff000h		; end of 1M, see if final 4K block was identity mapped
	je	@@scatdone		; yes

; start new region for final
	inc	ecx
	mov	esi, 100000h
	jmp	@@entryloop

@@next2:
	add	esi, 1000h
	call vds_GetPhysAddr	;expects page in AX!
	cmp	edx, esi
	je	@@samereg
	inc	ecx				; have to move to next region/entry
	mov	esi,edx			; update current physical address
	jmp	@@entryloop

@@samereg:
	cmp	cx,[edi].EDDS.wNumAvail	; entries available count
	jnc	@@nextframe
	add	[edi+ecx*8+size EDDS].EDDSRG.dwSizeRg,1000h
	jmp	@@nextframe

; calculate final region byte size or maximum allowed
@@scatdone:
	pop edx
	cmp	[edi].EDDS.wNumAvail,0	; entries available count
	jz	@@noregions
	and	edx,0fffh
	add	[edi+size EDDS].EDDSRG.dwPhysAddr, edx
	mov	ebx,1000h
	sub	ebx,edx	
	add	[edi+size EDDS].EDDSRG.dwSizeRg,ebx
@@noregions:
	xor	ebx,ebx
	mov	edx,ebx
	movzx ecx,[edi].EDDS.wNumUsed	; number regions used (cannot be 0)
	mov al,0
	cmp cx, [edi].EDDS.wNumAvail
	jbe	@@finalloop
	mov	cx,[edi].EDDS.wNumAvail	; only count up to minimum of available/used
	mov al,9
if ?VDSDBG
	and ecx, ecx
	jz @@scatfail
else
	jecxz @@scatfail 
endif	 
@@finalloop:
	mov	esi,edx			; keep previous, last known valid value
	add	edx,[edi+ebx*8+size EDDS].EDDSRG.dwSizeRg
	inc	ebx
	loop @@finalloop
	cmp	al,0
	jne	@@scatfail		; not all regions represented, update EDDS.dwSize

	mov	edx,[edi].EDDS.dwSize	; update final region byte count
	sub	edx,esi
	dec	ebx
	mov [edi+ebx*8+size EDDS].EDDSRG.dwSizeRg, edx
if ?VDSDBG
	@DbgOutS <"VDS scatlock exit, rc=0, used=">,1
	@DbgOutW [edi].EDDS.wNumUsed, 1
	@DbgOutS <", addr=">,1
	@DbgOutD [edi+size EDDS].EDDSRG.dwPhysAddr, 1
	@DbgOutS <", siz=">,1
	@DbgOutD [edi+size EDDS].EDDSRG.dwSizeRg, 1
	@DbgOutS <10>,1
endif
	pop	esi
	pop edx
	jmp vds_ok
@@scatfail:
	mov	[edi].EDDS.dwSize,edx
if ?VDSDBG
	@DbgOutS <"VDS scatlock exit, rc=">,1
	@DbgOutB al,1
	@DbgOutS <", siz=">,1
	@DbgOutD [edi].EDDS.dwSize, 1
	@DbgOutS <" used=">,1
	@DbgOutW [edi].EDDS.wNumUsed, 1
	@DbgOutS <10>,1
endif
	pop	esi
	pop edx
	jmp	vds_fail
	
;--- return PTEs

@@getptes:
	push eax
	
	shr	eax,12			; convert linear start to PTE index
	push esi
	@GETPTEPTR esi, 1000h, 1
@@loop:
	xor edx, edx
	cmp eax, 400h		; don't cross page table 0 border!
	jnc @@noPTE
	mov edx, [esi+eax*4]
	and dx, 0F001h
@@noPTE:	
	cmp cx, [edi].EDDS.wNumAvail
	jnc @@noPTEupdate
	mov [edi+ecx*4+size EDDS].EDDSPT.dwPTE, edx
@@noPTEupdate:
	inc ecx
	inc eax
	sub ebx, 1000h
	ja @@loop
	mov [edi].EDDS.wNumUsed,cx
	pop esi
	pop eax
	
	pop edx
	and ah,0Fh
	cmp cx, [edi].EDDS.wNumAvail
	ja @@notenoughregions2
	mov word ptr [ebp+4].V86FRGP.gEBX, ax
	jmp vds_ok
@@notenoughregions2:
	movzx ecx, [edi].EDDS.wNumAvail
	shl ecx, 12
	movzx eax,ax
	sub ecx, eax
	mov	[edi].EDDS.dwSize, ecx
	mov	al,9		;error "NumAvail too small"
	jmp vds_fail

vds_scatterlock endp

;--- disable automatic translation for a DMA channel
;--- BX=DMA channel number
;--- DX=flags (all reserved and must be 0)

vds_disabletrans proc

	mov ebx, [EBP+4].V86FRGP.gEBX
if ?VDSDBG
	@DbgOutS <"VDS enable translation for channel ">,1
	@DbgOutW bx,1
	@DbgOutS <10>,1
endif
	cmp bx, 8
	mov al, 0Ch		;error "invalid channel"
	jnc vds_fail
	mov al, 10h		;error "reserved flags set in DX"
	and dx, dx
	jnz vds_fail
if ?DMA    
	mov al, 0Dh		;error "disable count overflow"
	movzx ebx, bx
	cmp [DmaChn+ebx*8].cDisable,255
	jz vds_fail
	inc [DmaChn+ebx*8].cDisable
endif
	jmp vds_ok
	
vds_disabletrans endp

;--- enable automatic translation for a DMA channel
;--- BX=DMA channel number
;--- DX=flags (all reserved and must be 0)
	
vds_enabletrans proc

	mov ebx, [EBP+4].V86FRGP.gEBX
if ?VDSDBG
	@DbgOutS <"VDS disable translation for channel ">,1
	@DbgOutW bx,1
	@DbgOutS <10>,1
endif
	cmp bx, 8
	mov al, 0Ch		;error "invalid channel"
	jnc vds_fail
	mov al, 10h		;error "reserved flags set in DX"
	and dx, dx
	jnz vds_fail
if ?DMA    
	mov al, 0Eh		;error "disable count underflow"
	movzx ebx, bx
	cmp [DmaChn+ebx*8].cDisable,0
	jz vds_fail
	dec [DmaChn+ebx*8].cDisable
endif	 
	jmp vds_ok

vds_enabletrans endp

VDS_retcode equ <>	;undefine stack variable

endif ;?VDS

if 1

;--- helper routines

dw2a	proc			; display DWORD in eax into EDI
		push	eax
		shr		eax,16
		call	w2a
		pop 	eax
dw2a	endp		 	; fall through
w2a		proc			; display WORD in ax into EDI
		push	eax
		mov 	al,ah
		call	b2a
		pop 	eax
w2a		endp			; fall through
b2a		proc			; display BYTE in al into EDI
		push	eax
		shr 	al,4
		call	@@nibout
		pop		eax		; fall through
@@nibout: 				; display NIBBLE in al[0..3] into EDI
		and 	al,0Fh
		cmp 	al,10
		sbb 	al,69H
		das
		stosb
		ret
b2a		endp

excitem struc
_bSize		db ?
bOfs		db ?
dwTarget	dd ?
excitem ends

;--- render register contents
;--- [ESP+4]: item descriptor
;--- may modify all general purpose registers except EBP
;--- DS,ES=FLAT

renderitems proc
	pop		eax
	pop		esi		;get parameter
	push	eax
@@nextitem:    
	mov		al, [esi].excitem._bSize
	cmp		al, -1
	jz		@@done_exc
	mov		bl, al
	movsx	eax, [esi].excitem.bOfs
	mov		edi, [esi].excitem.dwTarget
	add		esi, size excitem
	MOV 	eax, ss:[ebp+eax]	;use SS prefix here!
	push	offset @@nextitem
	cmp		bl,2
	jz		b2a
	cmp		bl,4
	jz		w2a
	jmp 	dw2a
@@done_exc:
	retn
renderitems endp


exc_str label byte
	DB 13,10,"JEMM386: exception "
exc_no db 2 dup (' ')
	db " occured at CS:EIP="
exc_cs db 4 dup (' ')
	db ':'
exc_eip db 8 dup (' ')
	db 13,10
	db "SS:ESP="
exc_ss db 4 dup (' ')
	db ':'
exc_esp db 8 dup (' ')
	db " EBP="
exc_ebp db 8 dup (' ')
	db " EFL="
exc_efl db 8 dup (' ')
	DB " CR0="
exc_cr0 db 8 dup (' ')
	DB " CR2="
exc_cr2 db 8 dup (' ')
	db 13,10
	db "EAX="
exc_eax db 8 dup (' ')
	db " EBX="
exc_ebx db 8 dup (' ')
	db " ECX="
exc_ecx db 8 dup (' ')
	db " EDX="
exc_edx db 8 dup (' ')
	db " ESI="
exc_esi db 8 dup (' ')
	db " EDI="
exc_edi db 8 dup (' ')
	db 13,10
	db 0

exc_str2 label byte
	db "DS="
exc_ds db 4 dup (' ')
	db " ES="
exc_es db 4 dup (' ')
	db " FS="
exc_fs db 4 dup (' ')
	db " GS="
exc_gs db 4 dup (' ')
	db ' [CS:IP]='
exc_csip db 8*3 dup (' ')
	db CR,LF,'Press ESC to abort program ', 0
	db 0

exc_str3 db 13,'JEMM386: unable to continue. Please reboot '
	db 0
exc_str4 db CR,LF
	db 0

exc_format label excitem
	excitem <2, EXCFR.gIntNo,offset exc_no>
	excitem <4, EXCFR.gCS,	 offset exc_cs>
	excitem <8, EXCFR.gEIP,	 offset exc_eip>
	excitem <4, -4, 		 offset exc_ss>
	excitem <8, -8, 		 offset exc_esp>
	excitem <8, PUSHADS.rEBP,offset exc_ebp>
	excitem <8, EXCFR.gEFL,	 offset exc_efl>
	excitem <8, -12,		 offset exc_cr0>
	excitem <8, -16,		 offset exc_cr2>
	excitem <8, PUSHADS.rEAX,offset exc_eax>
	excitem <8, PUSHADS.rEBX,offset exc_ebx>
	excitem <8, PUSHADS.rECX,offset exc_ecx>
	excitem <8, PUSHADS.rEDX,offset exc_edx>
	excitem <8, PUSHADS.rESI,offset exc_esi>
	excitem <8, PUSHADS.rEDI,offset exc_edi>
	db -1

exc_v86segregs label excitem
	excitem <4, EXCFR.gDS, offset exc_ds>
	excitem <4, EXCFR.gES, offset exc_es>
	excitem <4, EXCFR.gFS, offset exc_fs>
	excitem <4, EXCFR.gGS, offset exc_gs>
	db -1

endif

if ?DBGOUT

;--- display DWORD in eax

VDWORDOUT proc
		push	eax
		shr		eax,16
		call	VWORDOUT
		pop 	eax
VDWORDOUT endp		  
VWORDOUT proc
		push	eax
		mov 	al,ah
		call	VBYTEOUT
		pop 	eax
VWORDOUT endp
VBYTEOUT proc
		pushfd
		push	eax
		mov		ah,al
		shr 	al,4
		call	VNIBOUT
		mov		al,ah
		call	VNIBOUT
		pop 	eax
		popfd
		ret
VBYTEOUT endp
VNIBOUT proc		
		and 	al,0Fh
		cmp 	al,10
		sbb 	al,69H
		das
		jmp 	VPUTCHR
VNIBOUT endp		

VPUTCHR PROC
	push ds
	PUSHAD
	push	FLAT_DATA_SEL
	pop		ds
if ?USEMONO    
	mov		edi,0B0000h
	mov		ebx,7
else
	MOV		EDI,0B8000h
	CMP		BYTE ptr DS:[463h],0B4h
	JNZ 	@@IS_COLOR
	XOR 	DI,DI
@@IS_COLOR:
	movzx	EBX, WORD PTR DS:[44Eh]
	ADD 	EDI, EBX
	MOVZX	EBX, BYTE PTR DS:[462h]
endif	 
	mov		esi, edi
	MOVZX	ECX, BYTE PTR DS:[EBX*2+450h+1]	;ROW
if ?USEMONO
	MOV 	EAX, 80
else
	MOVZX	EAX, WORD PTR DS:[44Ah]
endif	 
	MUL 	ECX
	MOVZX	EDX, BYTE PTR DS:[EBX*2+450h]	;COL
	ADD 	EAX, EDX
	MOV 	DH,CL
	LEA 	EDI, [EDI+EAX*2]
	MOV 	AL, [ESP+1Ch]
	CMP 	AL, 10
	JZ		@@NEWLINE
	MOV 	[EDI], AL
	MOV 	byte ptr [EDI+1], 07
	INC 	DL
if ?USEMONO
	cmp		dl,80
else
	CMP 	DL, BYTE PTR DS:[44Ah]
endif	 
	JB		@@OLDLINE
@@NEWLINE:	  
	MOV 	DL, 00
	INC 	DH
if ?USEMONO
	CMP 	DH, 24
else
	CMP 	DH, BYTE PTR DS:[484h]
endif	 
	JBE 	@@OLDLINE
	DEC 	DH
	CALL	@@SCROLL_SCREEN
@@OLDLINE:	  
	MOV 	DS:[EBX*2+450h],DX
	POPAD
	pop ds
	RET

;--- scroll screen up 1 line
;--- esi -> start screen

@@SCROLL_SCREEN:
	push es
	push ds
	pop	es
	CLD
	mov   edi,esi
if ?USEMONO
	mov   eax,80
else
	movzx eax,word ptr ds:[44Ah]
endif
	push  eax
	lea   esi, [esi+2*eax]
if ?USEMONO
	mov   CL, 24
else
	MOV   CL, DS:[484h]
endif
	mul   cl
	mov   ecx,eax
	rep   MOVS WORD PTR [EDI], WORD PTR [ESI]
	pop   ecx
	mov   ax,0720h
	rep   stos WORD PTR [EDI]
	pop es
	retn

VPUTCHR ENDP


VPRINTSTR PROC
	XCHG EBX,[ESP]
	PUSH EAX
@@NEXTCHAR:
	MOV AL,CS:[EBX]		; using CS prefix should always work here
	INC EBX
	CMP AL,0
	JZ	@@DONE
	call VPUTCHR
	JMP @@NEXTCHAR
@@DONE:
	POP EAX
	XCHG EBX,[ESP]
	RET
VPRINTSTR endp

endif

if ?VDSDBG

;--- support for 386SWAT: check if int 3 vector still points to
;--- the monitor code segment. If no, assume 386SWAT has intruded

DebugBreak proc
		push  eax
		sub   esp,8
		sidt  [esp]
		mov   eax,[esp+2]
		cmp   word ptr [eax+3*8+4],FLAT_CODE_SEL
		lea   esp,[esp+8]
		pop   eax
		jnz   swatfound
		ret
swatfound:
		pushfd
		or	byte ptr [esp+1],1	;set TF
		popfd
		ret 	   
DebugBreak endp

endif

if ?ALTBOOT or ?A20PORTS

	align 4

;; Special Ctrl-Alt-Del handler (should be off by default and only
;; be enabled by the ALTBOOT option!). catches Ctrl-Alt-Del 
;; and makes sure that mapping / protected mode
;; is not blocking the proper reboot process.
;
Int15_Entry PROC
if ?ALTBOOT
	cmp		ax,4F53h
	jz		@@isdel
endif
if ?MASM
Int15_Entry_Ex::		;<- entry if NOALTBOOT (default)
else
Int15_Entry_Ex:
endif
if ?A20PORTS
	cmp		ah,24h
	jz		@@isa20
endif	 
@@v86mon:
	push	15h
	JMP 	V86_Monitor

if ?A20PORTS

;--- catch int 15h, ax=2400h and ax=2401h

@@isa20:
	cmp		al,2
	jnb		@@v86mon
	@DbgOutS <"Int 15h, ax=">,?A20DBG	
	@DbgOutW ax,?A20DBG
	@DbgOutS <" called",10>,?A20DBG	
 if 0
	call	A20_Set
	mov		ah,0
	and		[esp].IRETDV86.vEFL, not 1
 else
	mov		ah,86h
	or		[esp].IRETDV86.vEFL, 1
 endif
	iretd
endif

if ?ALTBOOT    
@@isdel:
	PUSH	EAX
	MOV 	AL,CS:[@KB_FLAG]	; Have the keys CTRL & ALT
	AND 	AL,1100B			; been pressed ?
	CMP 	AL,1100B			; If not,  continue working
	POP 	EAX
	JNZ 	@@v86mon
    push	FLAT_DATA_SEL
    pop		ds					; required for ExecIntV86
	push	0
    push	0
    @v86pushreg
    push	15h
    call	ExecIntV86
  if ?FASTBOOT
  	test	[bV86Flags],V86F_FASTBOOT
    jnz 	fastboot
  endif  
    jmp		Reboot
endif

Int15_Entry ENDP

endif

if ?UNLOAD

;--- unload monitor, return to real-mode

Reset proc

	call SimRETF

if ?VME
    mov  al,0
    call SetVME
endif

if ?FREEXMS
	push ds
    pop es
	call FreeXMSHandles
endif

if ?RESDBG    
	@DbgOutS <"Reset, v86CS:EIP=">,1
    mov eax, [esp].V86FRGP.gCS
	@DbgOutD eax,1
	@DbgOutS <":">,1
	mov eax, [esp].V86FRGP.gEIP
	@DbgOutD eax,1
	@DbgOutS <", v86SS:ESP=">,1
	mov eax, [esp].V86FRGP.gSS
	@DbgOutD eax,1
	@DbgOutS <":">,1
	mov eax, [esp].V86FRGP.gESP
	@DbgOutD eax,1
	@DbgOutS <10>,1
endif    
	movzx eax, word ptr [esp].V86FRGP.gCS
    shl eax, 4
    mov word ptr [GDT + REAL_CODE_SEL+2], ax
    shr eax, 16
    mov [GDT + REAL_CODE_SEL+4], al
    movzx eax, word ptr [esp].V86FRGP.gSS
    shl eax, 4
    mov word ptr [GDT + REAL_DATA_SEL+2], ax
    shr eax, 16
    mov [GDT + REAL_DATA_SEL+4], al
    
    mov eax, [OldInt06]
    mov ds:[06h*4],eax
    mov eax, [OldInt67]
    mov ds:[67h*4],eax
    mov ecx, [OldInt4B]
    jecxz @@novds
    mov ds:[4Bh*4],ecx
    and byte ptr ds:[47Bh],not 20h
@@novds:
    mov edx, [esp].V86FRGP.gEIP
    mov ecx, [esp].V86FRGP.gESP
    mov bx, [XMSControlHandle]
    
	push	0
	push	word ptr 3FFh
	LIDT	FWORD ptr [esp] 	; reset IDT to real-mode
    
	MOV 	AX,REAL_DATA_SEL	; before returning to real-mode set the
	MOV 	DS,EAX				; segment register caches
	MOV 	ES,EAX
    MOV		FS,EAX
    MOV		GS,EAX
	MOV 	SS,EAX
    mov		ESP,ECX
    push REAL_CODE_SEL
    push edx
    retf
    
Reset endp
endif

if ?FASTBOOT

;--- how is FASTBOOT implemented?
;--- the important thing is to restore the IVT to the values
;--- *before* DOS has been installed. This requires:
;--- a). DOS must hook int 19h and restore the vectors it has modified
;---     (vectors which count are 00-1F, 40-5F and 68-77). It must also
;---     save vector 15h (which is modified by himem.sys).
;--- b). the vectors must be save at 0070:100h
;---     msdos saves at least 10,13,15,19,1B.
;--- c). jemm must be loaded as a device driver, so it is loaded very 
;---     early before other drivers/tsrs.
;--- if these requirements are met, Jemm will do with FASTBOOT: 
;--- 1. save int vectors 0-1F, 40-5F, 68-77 (20h+20h+10h = 50h*4=320 bytes)
;---    on init.
;--- 2. on ctrl-alt-del, restore these vectors, save the vector for int 19h
;---    which DOS has saved internally at 0070:0100+x and modify it to point
;---    to a breakpoint.
;--- 3. call v86-int 19h. DOS will restore the vectors it has saved.
;--- 4. Jemm regains control, with DOS already deactivated. Now clear
;---    vectors 20-3F and 60-67, restore int 19h to the value saved 
;---    previously.
;--- 5. jump to real-mode and do an int 19h again.

restorevecs proc
    pushad
	xor		eax, eax
	mov		edi, eax
    mov		esi, [pSavedVecs]
	push	ds
	pop		es
if ?RBTDBG    
	@DbgOutS <"restoring vectors 00-1F, 40-5F, 68-77",10>,1
endif
	mov		ecx, 20h
    rep		movsd   		;set 00-1F
    add		edi, 20h*4
    mov		cl, 20h
    rep		movsd			;set 40-5F
    add		edi, 8*4
    mov		cl, 10h
    rep		movsd			;set 68-77

;--- search the int 19h vector stored at 0070:100h

    mov		esi,700h+100h
    mov		cl,5
@@nextitem:    
    lodsb
    mov		bl,al
    lodsd
    cmp		bl,19h
    loopnz	@@nextitem
    stc
    jnz		@@norestore		;not found. no FASTBOOT possible
    
    mov		edx, [dwRSeg]
    shl		edx, 16
    mov		dl, [bBpTab]	;use the first BP for returning to the monitor
if ?RBTDBG    
	@DbgOutS <"vector 19h saved by DOS=">,1
    @DbgOutD eax,1
	@DbgOutS <", temp vector=">,1
    @DbgOutD edx,1
    @DbgOutS <10>,1
endif
    mov		[dwInt19Org],eax
    mov		[esi-4],edx

if 0    ;restoring the XBDA should be done by DOS (MS-DOS does)
	mov		dx,ds:[40Eh]
    and		dx,dx
    jz		@@noxbda
    mov		ax,ds:[413h]	;move it to top of RAM
    dec		eax
    mov		ds:[413h],ax	;should be 27Fh (=639 kb) again
    shl		eax,6			;27Fh -> 9FC0
    movzx	esi,dx
    shl		esi,4
    movzx	edi,ax
    shl		edi,4
    mov		ecx, 400h/4
    rep		movsd
@@noxbda:
endif

if 0
    in		al,0A1h
    or		al,03Fh
    out		0A1h,al
    in		al,021h
    or		al,0F8h
    out		21h,al
endif

	clc
@@norestore:
	popad
	ret
restorevecs endp

fastboot proc

if ?RBTDBG    
	@DbgOutS <"fastboot reached",10>,1
endif

    mov		al,0AEh				; reenable keyboard
    out		64h,al
    mov		al,20h				; send EOI to master PIC (for keyboard)
    out		20h,al
    call	restorevecs
    jc		Reboot
	mov     ds:[bptable],offset fastboot_1
    push	19h
    call	ExecIntV86
fastboot_1:    

;--- now DOS has restored its vectors
;--- restore the previously modified int 19h vector

	mov ecx,[dwInt19Org]
    jecxz @@noint19org
    mov ds:[19h*4],ecx
@@noint19org: 


if 0
	mov	dword ptr ds:[1Eh*4],0F000EFC7h
endif

;--- set vectors 20-3F and 60-67

	push ds
    pop es
    mov edi,20h*4
    mov ecx,20h
;    mov eax,0F000EEF3h
    xor eax,eax
    rep stosd
;    xor eax,eax
    mov edi,60h*4
    mov cl,8
    rep stosd

if 1
    movzx ecx, word ptr ds:[40Eh]
    jecxz @@noxbda
    shl ecx, 4
    mov word ptr [ecx+90h],0	;clear boot flags (might be BIOS specific)
@@noxbda:
endif

if ?RBTDBG
	mov ecx,20h
    xor esi,esi
    xor ebx,ebx
@@: 
    lodsd
    @DbgOutB bl,1
    @DbgOutS <": ">,1
    @DbgOutD eax, 1
    @DbgOutS <"        ">,1
    inc ebx
    loop @B
    @DbgOutS <10>,1
;    @WaitKey 1,1
endif

	mov dl,1		;set flag for fastboot
    jmp reboot_1
    
fastboot endp
endif

;--- Reboot - reboot the machine
;--- inp: DS=FLAT

Reboot proc

if ?RBTDBG    
	@DbgOutS <"Reboot",10>,1
endif

if ?FASTBOOT
	mov		dl,0	;flag: do a "full" reboot
endif

reboot_1::
	MOV 	WORD PTR DS:[@RESET_FLAG],1234H	; 1234h=warm boot

if 1
	MOV		AL,0Fh	;disable NMI, set shutdown byte
	out		70h,al
	MOV		AL,0	;software-reset
	out		71h,al
endif

ife ?USETRIPLEFAULT
	mov		edi, 7E00h
    mov		esi, offset rmcode
    mov		ecx, offset endofrmcode - offset rmcode
	push ds
    pop es
    rep	movsb
  if ?FASTBOOT
	cmp		dl,1
	jnz		@@nofast2
    sub		edi, 5
    mov		esi, offset rmcode2
    mov		ecx, offset endofrmcode2 - offset rmcode2
    rep		movsb
    and		byte ptr ds:[47Bh],not 20h	;reset VDS bit
@@nofast2:
  endif
	xor		edx, edx
	push	edx
	push	word ptr 3FFh
	LIDT	FWORD ptr [esp] 	; reset the IDT to real-mode

  if 1
	cmp		edx,[dwFeatures]
	jz		@@nocr4
	@mov_cr4_edx				; reset CR4 to 0
@@nocr4:
  endif

	MOV 	AX,REAL_DATA_SEL	; before returning to real-mode set the
	MOV 	DS,EAX				; segment register caches
	MOV 	ES,EAX
	MOV 	FS,EAX
	MOV 	GS,EAX
	MOV 	SS,EAX				; set SS:ESP
	MOV 	ESP,7C00h
	MOV 	ECX,CR0 			; prepare to reset CR0 PE and PG bits
	AND 	ECX,7FFFFFFEH
	XOR		EAX, EAX
    db		66h
	db 		0eah
	Dw		7E00h
	dw		REAL_CODE_SEL
else
	xor		edx,edx				; cause a triple fault to reboot
	push	edx
	push	edx
	lidt	fword ptr [esp]
	int		3
endif

rmcode:
	db 0Fh, 22h, 0C1h		;mov cr0, ecx
    db 0Fh, 22h, 0D8h		;mov cr3, eax
    db 8Eh,0D0h				;mov ss, ax
	db 0EAh,0, 0, -1, -1	;jmp ffff:0000
endofrmcode:
if ?FASTBOOT
rmcode2:
	db 0EAh,0Dh,7Eh,0, 0	;jmp 0000:7E0D
if 0
	db 0Eh					;push cs
    db 1Fh					;pop ds
    db 0Eh					;push cs
    db 07h					;pop es
    db 0FBh					;sti
    db 0B8h, 00h, 00h		;mov ax,0000h
    db 0BAh, 80h, 00h		;mov dx,0080h
    db 0CDh, 13h			;int 13h
    db 0B8h, 01h, 02h		;mov ax,0201h
    db 0B9h, 01h, 00h		;mov cx,0001h
    db 0BAh, 80h, 00h		;mov dx,0080h
    db 0BBh, 00h, 7Ch		;mov bx,7C00h
    db 0CDh, 13h			;int 13h
    db 06h					;push es
    db 53h					;push bx
    db 0CBh					;retf
else
    db 0CDh, 19h			;int 19h
endif
endofrmcode2:
endif

Reboot endp

	align	4

xms_handler proc
	call SimRETF				; emulate a RETF in v86
	mov eax,[esp].V86FRGP.gEAX	; restore EAX
if ?A20XMS
	cmp ah,10h
    jae umb_handler
else
	jmp umb_handler
endif
    
xms_handler endp	;fall through for A20


if ?A20XMS

; handles XMS A20 functions
; 3 = global enable
; 4 = global disable
; 5 = local enable
; 6 = local disable

XMS_HandleA20 proc

	mov al,ah
	@DbgOutS <"XMS A20 emulation, ah=">,?A20DBG
	@DbgOutB al,?A20DBG
	@DbgOutS <", curr cnt=">,?A20DBG
	@DbgOutW [wA20],?A20DBG
	@DbgOutS <", curr state=">,?A20DBG
	@DbgOutB [bA20Stat],?A20DBG
	@DbgOutS <10>,?A20DBG
	
	mov cx, word ptr [wA20]
	cmp al,4
	jb @@glen
	jz @@gldi
	cmp al,6
	jb @@loen
	jmp @@lodi
	
@@glen:
	or ch,1
	jmp @@testa20
@@gldi:
	and ch,not 1
	jcxz @@testa20
	jmp @@stillenabled
@@loen:
	inc cl
	jz	@@localerr
	jmp @@testa20
@@lodi:    
	sub cl,1
	jc	@@localerr2
	and cx, cx
	jnz @@stillenabled
@@testa20:
	and cx, cx
	setnz al
	cmp al, [bA20Stat]
	jz @@notchanged
	mov [bA20Stat],al
	call A20_Set
@@notchanged:
	mov ax,1
	mov bl,0
	mov [wA20],cx
	jmp	@@a20_exit
@@localerr2:
if 1			;if this is to be changed, check EDR-DOS first!
	inc cl
	dec ch
	jz	@@testa20
endif	 
@@localerr:
if 1
	xor eax,eax
	mov bl,82h
else
	mov ax,1
endif
	jmp @@a20_exit
@@stillenabled:    
	mov [wA20],cx
	xor eax,eax
	mov bl,94h

@@a20_exit:	 
	mov word ptr [esp].V86FRGP.gEAX, ax
	and ax,ax
	jnz @@a20exit_noerror
	mov byte ptr [esp].V86FRGP.gEBX, bl
@@a20exit_noerror:
	@v86popreg	
	add esp,4+4
	iretd

XMS_HandleA20 endp

endif

if ?A20PORTS or ?A20XMS

;--- set PTEs for HMA to emulate enable/disable A20
;--- in: AL bit 0 == 1 : enable, else disable

A20_Set proc    
	cmp [bNoA20], 0
	jnz @@exit
	pushad
	and  al,1
	shl  al,4		;00h or 10h
	xor	ecx,ecx
	@GETPTEPTR edi, 1000h+256*4+2, 1
@@spec_loop:
	mov [edi+ecx*4],al
	inc	ecx
	cmp	cl,16
	jb	@@spec_loop
if ?INVLPG
	cmp [bNoInvlPg],0
	jnz @@noinvlpg
	mov edx, 100000h
@@nextpte:
	invlpg ds:[edx]
	add dh, 10h
	jnz @@nextpte
	popad
	ret
@@noinvlpg:
endif
; flush TLB to update page maps
	mov	eax,CR3
	mov	CR3,eax
	popad
@@exit:    
	ret

A20_Set endp

endif

	align 4

;--- inp: eax = linear address
;--- cl = size in PTEs
;--- ebx = pagedir
;--- edx = page table

if ?WT
?PA	equ 1+2+8	;PRESENT + R/W + WRITE THROUGH
else
?PA	equ 1+2		;PRESENT + R/W
endif

I15_SetPTEs proc
	and ax, 0F000h
	or al,?PA
if ?INVLPG	  
	cmp [bNoInvlPg],0
	jz @@setPTEs486
endif	 
@@nextPTE1:   
	mov [edx], eax
	add eax, 1000h
	add edx,4
	dec cl
	jnz  @@nextPTE1
	ret
if ?INVLPG	  
@@setPTEs486:
@@nextPTE2:   
	mov [edx], eax
	invlpg ds:[ebx]
	add edx,4
	add eax, 1000h
	add ebx, 1000h
	dec cl
	jnz  @@nextPTE2
	ret
endif	 
I15_SetPTEs endp

	align 4
    
;************************************************************
; simulate INT15/87
;
;INT 15 - SYSTEM - COPY EXTENDED MEMORY (by RBIL)
;
;		 AH = 87h
;		 CX = number of words to copy (max 8000h)
;		 ES:SI -> global descriptor table
;Return: CF set on error
;		 CF clear if successful
;		 AH = status 
;
;Values for extended-memory copy status:
; 00h	 source copied into destination
; 01h	 parity error
; 02h	 interrupt error
; 03h	 address line 20 gating failed
; 80h	 invalid command (PC,PCjr)
; 86h	 unsupported function (XT,PS30)
;
;Format of global descriptor table:
;Offset  Size	 Description
; 00h 16 BYTEs	 zeros (used by BIOS)
; 10h	 WORD	 source segment length in bytes (2*CX-1 or greater)
; 12h  3 BYTEs	 24-bit linear source address, low byte first
; 15h	 BYTE	 source segment access rights (93h)
; 16h	 BYTE	 more rights
; 17h	 BYTE	 8 bit	linear source adress, high
; 18h	 WORD	 destination segment length in bytes (2*CX-1 or greater)
; 1Ah  3 BYTEs	 24-bit linear destination address, low byte first
; 1Dh	 BYTE	 destination segment access rights (93h)
; 1eh	 byte	 more rights
; 1fh	 BYTE	 8 bit	linear source adress, high
;************************************************************

I15MOVE struc
	dq ?
	dq ?
wSrcLim 	dw ?
wSrcA0015	dw ?
bSrcA1623	db ?
wSrcAcc		dw ?
bSrcA2431	db ?
wDstLim 	dw ?
wDstA0015	dw ?
bDstA1623	db ?
wDstAcc		dw ?
bDstA2431	db ?
I15MOVE ends

;--- registers EAX, EBX, ECX are saved on host stack!
;--- DS=FLAT

I15_Simulate87 proc

	push	ebp
    mov		ebp,esp
	call	SimIRET
	movzx	ecx,word ptr [ebp+4].V86FRGP.gECX	;restore destroyed CX register

	push edi
    push esi
	push edx
    cld

;-- MS Emm386 returns with error AH=2 if CX > 8000h!

	cmp cx, 8000h	
	ja @@error02
	or	ecx,ecx
	je	@@ok			; nothing to do
    
	MOVZX	edi,WORD PTR [ebp+4].V86FRGP.gES	; make edi = linear address of command
	MOVZX	esi,si
	SHL 	edi,4
	add 	esi,edi

	lea eax, [ecx*2-1]  ; verify that src and dst descriptors are ok.
						; we don't care about segment access rights
	cmp	ax, [esi].I15MOVE.wSrcLim	; 16-bit overflow not an issue (0->ffff)
	ja	@@error80
	cmp	ax, [esi].I15MOVE.wDstLim
	ja	@@error80

	mov al,[esi].I15MOVE.bSrcA1623
	mov ah,[esi].I15MOVE.bSrcA2431	; get linear source address
	mov dl,[esi].I15MOVE.bDstA1623
	mov dh,[esi].I15MOVE.bDstA2431	; get linear destination address
	shl eax,16
	shl edx,16
	mov ax,[esi].I15MOVE.wSrcA0015
	mov dx,[esi].I15MOVE.wDstA0015
	mov esi,eax
	mov edi,edx

	mov eax,ds
    mov es,eax

if ?I15DBG
	@DbgOutS <"Int 15h, ah=87h, src=">,1
	@DbgOutD esi,1
	@DbgOutS <", dst=">,1
	@DbgOutD edi,1
	@DbgOutS <", siz=">,1
	@DbgOutW cx,1
	@DbgOutS <10>,1
endif

;-- NOCHECK -> moves for addresses not backuped with RAM/ROM will fail
;-- (cause an exception)

	test [bV86Flags], V86F_NOCHECK
	je	@@memcheck

	lea eax, [esi+ecx*2]
	lea edx, [edi+ecx*2]
	cmp	eax, [dwTotalMemory]
	jae	@@fail
	cmp	edx, [dwTotalMemory]
	jae	@@fail

;-- Now 1 scratch page table is used and there is no more
;-- linear == physical mapping

@@memcheck:
	push ecx
					;get no of PTEs involved. This depends on the
					;bases of the descriptors. test cases:

	add cx,800h-1	;round up size to page boundary
	shr ecx,11		;words -> PTEs (8000h -> 10h)
	inc ecx			;low12(base) == FFFh, cx=8000h -> 10h+1 PTEs
	mov ch, cl

	cmp esi, 110000h+0000h	;the 111xxx page contains GDT/IDT
	jc	@@src_is_shared
	@GETPTEPTR edx, 2000h+?SCRATCHPTE, 1
	mov eax, esi
	and esi, 0FFFh
	or	esi, ?SCRATCHLINEAR
	mov ebx, esi
	call I15_SetPTEs
	mov cl, ch
@@src_is_shared:
	cmp edi, 110000h+0000h
	jc	@@dst_is_shared
	@GETPTEPTR edx, 2000h+?SCRATCHPTE+11h*4, 1
	mov eax, edi
	and edi, 0FFFh
	or	edi, ?SCRATCHLINEAR + 11000h
	mov ebx, edi
	call I15_SetPTEs
@@dst_is_shared:
if ?INVLPG
	cmp [bNoInvlPg],0
	jz @@flushdone
endif	 
	mov eax, cr3
	mov cr3, eax
@@flushdone:
	pop ecx

@@accessok:

	shr ecx,1
	REP MOVS DWORD PTR [ESI],DWORD PTR [EDI]
	@BIG_NOP
	adc ecx,ecx
	REP MOVS WORD PTR [ESI],WORD PTR [EDI]
	@BIG_NOP

@@ok:
	mov AH,0	; everything OK and finished
	and	[ebp+4].V86FRGP.gEFL, not 1
@@i1587_exit:
	mov byte ptr [ebp+4].V86FRGP.gEAX+1, ah
	pop edx
    pop esi
    pop edi
    pop	ebp
    @v86popreg
    add	esp,4+4
    iretd
@@error02:
	mov ah,02h
	or	[ebp+4].V86FRGP.gEFL, 1
	jmp @@i1587_exit
@@fail:
	mov ah,ds:[0FFFFFFFCh]
@@error80:
	mov ah,80h
	or	[ebp+4].V86FRGP.gEFL, 1
    jmp @@i1587_exit

I15_Simulate87 endp

		align 4

@emmpushreg macro
		PUSH	ECX
		PUSH	ESI
		PUSH	EDI
        PUSH	EBP
        MOV		EBP,ESP
		endm		
@emmpopreg macro        
		POP		EBP
		POP 	EDI
		POP 	ESI
		POP 	ECX
        endm
        
;--- if int 67h vector is hooked in real-mode, the hooker code is
;--- called and will finally met a breakpoint which will get us here.

Int67_V86Entry proc
		push	ebp
		mov		ebp,esp
		call	SimIRET
		pop		ebp
        
;--- clear V86FRGP stack frame

		@v86popreg
		ADD 	ESP,4+4 		; throw away errorcode & returnaddress.
        
;--- build EMMFRAME stack frame

		@emmpushreg
		MOV		ECX,SS
		MOV		DS,ECX
		MOV 	ES,ECX
		JMP		EMM_ENTRY_2
Int67_V86Entry endp

Int67_Indirect:
		@emmpopreg
		push	67h
		jmp		V86_Monitor

		align 4
;
; Here starts the Expanded Memory Manager (EMM) Version 4.0
;

Int67_Entry PROC

		@emmpushreg

		MOV 	ECX,SS		; address everything
		MOV 	DS,ECX
		MOV 	ES,ECX

		mov		ecx,[dwRSeg]
		cmp		cx,ds:[67h*4+2]		;IVT vector 67h modified?
		jnz		Int67_Indirect
if ?MASM		
EMM_ENTRY_2::
else
EMM_ENTRY_2:		
endif
		CLD
		
if ?VCPI
		cmp	ah,0deh			; see if VCPI function
		jne	@@not_vcpi_api
		cmp al,0Ch
		jz VCPI_V86toPM
		movzx ecx,al

  if ?VCPIDBG
		@DbgOutS <"VCPI rm, ax=">,1
		@DbgOutW ax,1
		@DbgOutS <" edx=">,1
		@DbgOutD edx,1
		@DbgOutS <" ... ">,1
  endif

		cmp	[bNoVCPI],0		; check if VCPI turned off
		jne	@@emm_INV_CALL	; yes, return invalid code, flags VCPI not present for 0de00h
		cmp	al,VCPI_MAX
		jae	@@emm_INV_CALL	; invalid VCPI call
		call [VCPI_CALL_TABLE+ECX*4]
  if ?VCPIDBG
		@DbgOutS <"VCPI exit, ax=">,1
		@DbgOutW ax,1
		@DbgOutS <", edx=">,1
		@DbgOutD edx,1
		@DbgOutS <10>,1
  endif
		jmp	@@BYEEMSX
		align 4
@@not_vcpi_api:
endif

if ?EMSDBG
@dircall macro func
local isnotfunc
		cmp ah,func
		jnz isnotfunc
		movzx ecx,ah
		call [EMS_CALL_TABLE+ecx*4-40h*4]
		jmp @@BYEEMSX
isnotfunc:
		endm
;		@dircall 4Fh	;to exclude AH=4Fh from debug displays
;		@dircall 50h
		@dircall 57h
		@DbgOutS <"EMM entry, ax=">,1
		@DbgOutW ax,1
		@DbgOutS <" dx=">,1
		@DbgOutW dx,1
		@DbgOutS <" bx=">,1
		@DbgOutW bx,1
		@DbgOutS <" ... ">,1
endif

		MOVZX	ECX,AH				; check permitted range
		SUB 	CL,40H
		JB		@@emm_INV_CALL
		CMP 	CL,EMS_MAX
		JAE		@@emm_INV_CALL
		CALL	[EMS_CALL_TABLE+ECX*4]
@@BYEEMS:
if ?EMSDBG
		@DbgOutS <"EMM exit, ax=">,1
		@DbgOutW ax,1
		@DbgOutS <", dx=">,1
		@DbgOutW dx,1
		@DbgOutS <10>,1
endif
@@BYEEMSX:
		@emmpopreg
		IRETD
@@emm_INV_CALL:
		MOV 	AH,84H				 ; "Invalid Functioncode in AH"
		JMP 	@@BYEEMS
		
Int67_Entry ENDP

;
; 6740: AH = 40h: return the actual state of the EMM-driver.
;
EMS_GET_STATUS PROC
	MOV 	AH,00					 ; Allways everything OK.
	RET
EMS_GET_STATUS ENDP
;
; 6741: AH = 41h: request the segment address of the EMS-window
;
EMS_GET_PAGE_FRAME_ADDRESS PROC
	mov		ah,80h
	cmp		[bNoFrame],0
	jnz		@@BYE
	MOV 	AH,00				  ; No error occurred
	MOV 	BH,[PAGE2SEGM+0] 	  ; Segment address of EMS-Window/Frame
    MOV		BL,0
@@BYE:	  
	RET
EMS_GET_PAGE_FRAME_ADDRESS ENDP

;
; 6742: AH = 42h: Request number of ( also maximum) available EMS-pages
;

EMS_GET_UNALLOCATED_PAGE_COUNT PROC

	call	GetEMSPageCount	;update PAGESAVAIL

	MOV 	BX,word ptr [EMSPAGESAVAIL] ; free EMS pages
	MOV 	DX,word ptr [dwMaxEMSPages] ; total EMS pages

; follow MS-DOS EMM386 7.x lead and don't throttle pages on NOEMS
;	cmp	cs:[bNoEMS],0
;	je	@@unalloc_ret
;	or	bx,bx
;	je	@@unalloc_ret
;	mov	bx,1			; only show maximum of 1 EMS page if NOEMS set

@@unalloc_ret:
	MOV 	AH,00
	RET
EMS_GET_UNALLOCATED_PAGE_COUNT ENDP
;
; 6743: AH = 43h: Reserve Memoryspace of the EMS-add-on-card (ahemm..extinct...)
; in  BX = EMS pages to reserve
; out AH=00, DX = handle

EMS_ALLOCATE_PAGES PROC
	MOV 	AH,89H			; "Request, to reserve null pages"
	AND 	BX,BX
	JZ		SHORT @@BYE

if ?MASM
allocate_pages_plus_zero::	;this entry allows to alloc zero pages!
else
allocate_pages_plus_zero:
endif
	MOV 	AH,87H					; "Not enough pages available"
	CMP 	BX,word ptr [dwMaxEMSPages]
	JA		SHORT @@BYE

	call	GetEMSPageCount			; update PAGESAVAIL
	MOV 	AH,88H					; "Not enough pages available anymore"
	CMP 	BX,word ptr [EMSPAGESAVAIL]
	JA		SHORT @@BYE

	MOV 	ESI,[EMSSTATUSTABLE]	; Now search for a free Handle in the table
	MOV 	CX,MAX_HANDLES		;
@@SEARCH:
	CMP 	WORD PTR [ESI],EMSH_FREE; Is there one free ... ?
	JZ		SHORT @@FOUND
	ADD 	ESI,8
	dec		cx
	jnz 	@@SEARCH
	MOV 	AH,85H					; "No more free Handles"
@@BYE:
	RET

@@FOUND:
	MOV 	WORD PTR [ESI],EMSH_USED ; mark Handle as occupied

	MOV 	DX,MAX_HANDLES		; Set in DX now the actual Handle-
	SUB 	DX,CX				; number

; zero page allocations allowed, so test and bypass code if found

	or		bx,bx
	je		@@allocate_exit
	
	push	ebx
	MOV 	EDI,[EMSPAGETABLE]	; mark the pages in the
	MOV 	ECX,[dwMaxEMSPages]	; page-usage-table as used
	PUSH	EAX
	MOV 	AL,FREEPAGE_ID
@@SEARCH_PAGES:
	REPNZ	SCAS BYTE PTR [EDI]	; After searching for a free page
	jnz	@@nofind			; free page could not be located
	call AllocateEMSPage	; allocate the page
	jc	@@nofind
	DEC		[EMSPAGESAVAIL]
	MOV 	[EDI-1],DL		; Assign the handle
	DEC 	BX			; Continue until all desired pages are occupied
	JNZ 	@@SEARCH_PAGES
	POP		EAX
	POP 	EBX
@@allocate_exit:
	MOV 	AH,00
	RET
@@nofind:
if ?POOLDBG
	@DbgOutS <"EMS_ALLOCATE_PAGES: @@nofind reached, BX=">,1
	@DbgOutW bx,1
	@DbgOutS <10>,1
endif
	call EMS_DEALLOCATE_PAGES	;free the handle in DX	
	pop eax
	pop	ebx
	MOV AH,88H					; "Not enough pages available anymore"
	ret
	
EMS_ALLOCATE_PAGES ENDP

;--- get EMS absolute page
;--- inp BX = logical page
;--- inp DX = handle
;--- out BX = absolute page if NC
;--- modifes ECX, EDI

EMS_get_abs_page proc	 
	INC 	EBX 				; fade out
	XCHG	EAX,EDX				; get the handle to search in AL
	MOV 	ECX,[dwMaxEMSPages]	; the range to search
	MOV 	EDI,[EMSPAGETABLE]	; Search the allocation table by the pages
    jecxz	@@NIX
@@LOOP:
	REPNZ	SCAS BYTE PTR [EDI]	; occupied by the handle
	JNZ 	@@NIX
	DEC 	BX
	JNZ 	@@LOOP
	MOV 	BX,word ptr [dwMaxEMSPages]
	SUB 	BX,CX				; abs pagenumber -> BX
	DEC 	BX
	XCHG	EAX, EDX
	ret
@@NIX:
	XCHG	EAX, EDX
	MOV 	AH,8AH				; "logical page out of reserved area"
	stc
	ret
EMS_get_abs_page endp	 

; 6744
; AH = 44h: mirror logical page in the EMS-window
; AL = physical page (0-3)
; DX = handle
; BX = logical page #
;
EMS_MAP_HANDLE_PAGE PROC
	CMP 	AL,[bEmsPhysPages]		; not" Only pages 0..3
	JAE 	SHORT @@PAGE_TOO_LARGE	; are allowed!

	CALL	EMS_TEST_HANDLE
	PUSH	EBX 			; save BX  (since it is changed)
	AND 	BH,BH			; bx < 0 means unmap this phys. page
	JS		SHORT @@MAP
	call	EMS_get_abs_page
	jc		@@NIX
@@MAP:
	CALL	EMS_MAP_ABS_PAGE
	MOV 	AH,00
@@NIX:
	POP 	EBX
	RET
@@ERR8B:
	MOV 	AH,8BH				; "Indicated physical page does not exist"
	RET

;--- int 67h, AX=449Fh
;--- DS:SI -> UMB segments (8 items)
;--- DX = handle 0
;--- CL bit 0: NOHI option
    
@@endinit:
	or 		[bInitDone],1	; finish initialization
	movzx	ecx, word ptr [ebp].EMMFRAME.eDS
	shl		ecx, 4
	movzx	esi, word ptr [ebp].EMMFRAME.eEsi
	add		esi, ecx
	mov		edi, offset UMBsegments
	mov		ecx, UMB_MAX_BLOCKS
	rep		movs dword ptr [edi], [esi]
if ?MOVEHIGH    
    test	byte ptr [ebp].EMMFRAME.eEcx,1
    jnz		@@NoHigh
	mov		cx, ds:[06h*4+2]
    mov		ds:[67h*4+2],cx
    mov		[dwRSeg],ecx
    mov		esi, [dwRes]
    shl		ecx, 4
    mov		[dwRes],ecx
    sub		ecx, esi
	add		[dwRFlags],ecx
	add		[bpstart],ecx
@@NoHigh:    
endif
	mov		ah,00
	RET

;*******************************************************************************
; here comes the funny part (by Tom)
;		during initialization phase, calling EMM_MAP_PAGE (0x44)
;		with pysical page (AL) > 3 is possible.
;		meaning:
;		AL = highest 8 bits of logical adress, AL=E4 --> address E4000
;		initialization phase is terminated with AL=9F
;*******************************************************************************

@@PAGE_TOO_LARGE:
	cmp 	[bInitDone],0		; still in init phase ?
	jne 	@@ERR8B
	cmp		al,09fh				; AL=9f finishes init phase
	je		@@endinit

; the fun part - map in page BX at address AL
; BX is interpreted as a 4k page here.
; That is, bits 2-15 are the true logical page
; and bits 0-1 are the page offset in this page

	AND 	BH,BH		  ; In case the log. page number
	JS		@@ERR8B 	  ; negative is, no search

; code stolen above
; find the memory for handle/page

	CALL	EMS_TEST_HANDLE
	PUSH	EBX

; convert 4K-based logical page to 16K-based
	shr		bx,2

	call	EMS_get_abs_page
	jc		@@NIX
	
;****
;**** stolen from MAP_PAGE (hiword(eax) is destroyed here!)
;****

	MOVZX	EAX,AL					; use the page # to get ptr to PTE
	@GETPTEPTR esi, [eax*4+1000h]	; in ESI   

	MOVZX	EDI,BX					; Now convert EMS absolute page in BX
	SHL 	EDI,12+2				; to a physical address in EDI
	ADD 	EDI,[FIRSTPAGE]
	ADD 	DI,111B 				; Statusbits: R/W=1,U/S=1,P=1

;	MOV 	CX,4					; 1 EMS page 4 (virtual) pages

	POP EBX
	movzx ecx,bl
	and	cl,3		; get 4K multiplier offset into 16K page
	shl	ecx,12		; convert multiplier offset to true 4K offset
	add	edi,ecx

; if segment frame value is 0ffh, then we know we're shadowing the ROM into RAM
; to catch jumps to FFFF:0, so copy ROM image of block to RAM

	cmp	al,0ffh
	jne	@@noshadow

;-- map the new page in linear address space at linear scratch pos
;-- and copy the content of page AL into in

	push	esi
	push	edi
	@GETPTEPTR ECX, 2000h+?SCRATCHPTE, 1
	mov		[ecx], edi
	mov		edi, cr3			;flush TLB
	mov		cr3, edi
	mov		edi, ?SCRATCHLINEAR
	movzx	esi,al
	shl	esi,12		; convert segment frame to absolute address
	mov	ecx,1000h/4	; map 4K block in dwords
	rep movs dword ptr [edi], [esi] 
	@BIG_NOP
	
	mov	BYTE PTR [edi-10h],?BPOPC	; set breakpoint at FFFF:0000
	pop	edi
	pop	esi
	and	di, not 2			;since it shadows ROM, make PTE r/o

@@noshadow:
	MOV 	[ESI],EDI		; set the PTE
	
if ?EMSDBG
	@DbgOutS <"page ">,1
	@DbgOutB al,1
	@DbgOutS <" got PTE ">,1
	@DbgOutD edi,1
	@DbgOutS <", mapped at ">,1
	@DbgOutD esi,1
	@DbgOutS <", PAGESAVAIL=">,1
	@DbgOutD [EMSPAGESAVAIL],1
	@DbgOutS <10>,1
endif

	MOV 	AH,00
if ?INVLPG
	cmp		[bNoInvlPg],0
	jnz		@@noinvlpg
	movzx	esi,al
	shl	esi,12
	invlpg	ds:[esi]
	RET
@@noinvlpg:
endif
	MOV		ESI, CR3
	MOV 	CR3, ESI		; and flush TLB
	RET

;*******************************************************************************
; end of fun :-)
;*******************************************************************************

EMS_MAP_HANDLE_PAGE ENDP

;
; 6745: AH = 45h: Release reserved memoryspace again
;		DX = handle to release
; any pages of this handle mapped in page frame remain mapped!
; this is same behaviour as MS Emm386

EMS_DEALLOCATE_PAGES PROC
	CALL	EMS_TEST_HANDLE
	MOV 	AH,86H					; "A saved state still
	CMP 	WORD PTR [ESI],EMSH_USED; exists" ?
	JNZ 	SHORT @@BYE
	MOV 	ECX,[dwMaxEMSPages]		; All pages in the allocation table
	XCHG	EAX,EDX 				; have to be marked again as free
	MOV 	EDI,[EMSPAGETABLE]
    jecxz	@@OK
@@LOOP:
	REPNZ	SCAS BYTE PTR [EDI] 	; after searching pages with Handlenumber,
	JNZ 	SHORT @@OK				; done?

	call	FreeEMSPage			; preserves registers

	MOV 	BYTE PTR [EDI-1],FREEPAGE_ID; Mark page as free
	INC 	[EMSPAGESAVAIL]
	JMP 	SHORT @@LOOP
@@OK:
	MOV 	WORD PTR [ESI],EMSH_FREE ; release Handle

; zero handle name on free
	mov	edi, [EMSNAMETABLE]
	xor esi, esi
	movzx	ecx, al	; handle (index)
	mov	DWORD PTR [edi+ecx*8+0],esi
	mov	DWORD PTR [edi+ecx*8+4],esi

	XCHG	EAX,EDX
	MOV 	AH,00					; Function executed duly
@@BYE:
	RET
EMS_DEALLOCATE_PAGES ENDP
;
; 6746: AH = 46h: determine Version-number of EMM
;
EMS_GET_VERSION PROC
;	MOV 	AX,0032H				; Currently version 3.2
	MOV 	AX,0040H				; Currently version 4.0
	RET
EMS_GET_VERSION ENDP

;--- save the frame window mapping state

EMS_SaveFrameToEsi proc
	mov		cl,4
EMS_SaveFrameToEsi endp	;fall through

;--- save CL (0-4) pages to ESI
;--- modifies CL

EMS_SavePagesToEsi PROC
	PUSH	EAX
	PUSH	EDX
	XOR		EDX, EDX	;start with physical page 0
	INC		CL
@@NEXTPAGE:
	DEC		CL
	jz		@@done
	MOV 	AX, [PHYSPAGETABLE+EDX*2]
	MOV 	[ESI+EDX*2],AX
	INC		DL
	JMP		@@NEXTPAGE
@@done:    
	POP		EDX
	POP 	EAX
	RET
EMS_SavePagesToEsi ENDP

;--- restore the frame window mapping state

EMS_RestoreFrameFromEsi PROC
	mov		cl, 4
EMS_RestoreFrameFromEsi ENDP		;fall through

;-- restore CL (0 - 4) pages for frame
;-- esi -> ems handle (8 bytes, where 4 EMS pages stored are stored)

EMS_RestorePagesFromEsi PROC

	PUSH	EAX				; save everything! Data- and System-
	PUSH	EBX				; registers first!
	xor		eax, eax		; start with physical page 0
	INC		CL
@@NEXTPAGE:    
	DEC		CL
	jz		@@done
	MOV 	BX,[ESI+EAX*2]
	push	Ecx
	push	Eax
	PUSH	ESI
	CALL	EMS_MAP_ABS_PAGE;expects phys. page in AL, log. page in BX
	POP 	ESI
	pop		Eax
	pop		Ecx
	INC 	EAX
	jmp 	@@NEXTPAGE
@@done:
	POP 	EBX
	POP 	EAX
	MOV 	AH,00			; report OK (because of functions $4E01/$4E02)
	RET
	
EMS_RestorePagesFromEsi ENDP


;
; 6747: AH = 47h: Save number of the fit-in page in EMS-window
; DX = handle
; it might be that there are pages mapped into the EMS page frame
; which don't belong to any handles!
;
EMS_SAVE_PAGES PROC
	mov		ah,80h
	cmp		[bNoFrame],0
	jnz		@@BYE
	CALL	EMS_TEST_HANDLE
	MOV 	AH,8DH				   ; "State for Handle already saved"
	CMP 	WORD PTR [ESI],EMSH_USED
	JNZ 	@@BYE
	call	EMS_SaveFrameToEsi
	MOV 	AH,00					 ; report 'everything OK'
@@BYE:
	RET
EMS_SAVE_PAGES ENDP
;
; 6748: AH = 48h: Restore saved state of the EMS-window
; DX = handle
;
EMS_RESTORE_PAGES PROC
	mov		ah,80h
	cmp		[bNoFrame],0
	jnz		@@BYE
	CALL	EMS_TEST_HANDLE
	MOV 	AH,8EH					  ; "A saved stated does not exist"
	CMP 	WORD PTR [ESI],EMSH_USED
	JZ		SHORT @@BYE
	CALL	EMS_RestoreFrameFromEsi
	MOV 	WORD PTR [ESI],EMSH_USED  ; Nothing stored for Handle
	MOV 	AH,00					  ; report 'everything OK'
@@BYE:
	RET

EMS_RESTORE_PAGES ENDP

;
; report the failure so that we can maybe support it in the future
;
EMS_NOT_IMPL PROC

IF	?UNIMPL_EMS_DBG
	mov		edi,offset unimpl_ax
	call	w2a
	mov		esi,offset unimpl_func
	call	PrintString
ENDIF
	MOV 	AH,84H					  ; "Invalid function code"
	RET

if ?UNIMPL_EMS_DBG	  
unimpl_func db "Unimplemented EMS function called, ax="
unimpl_ax	db "	"
			db 10
			db 0
endif

EMS_NOT_IMPL ENDP
;
; 674B: AH = 4Bh: return Number of open Handles in BX
;
EMS_GET_OPEN_HANDLES_COUNT PROC
	MOV 	ESI,[EMSSTATUSTABLE] ; Search Handle-status-table for
	MOV 	ECX,MAX_HANDLES 	 ; assigned/given handles
	XOR 	BX,BX
@@LOOP:
	CMP 	WORD PTR [ESI],EMSH_FREE ; Free ?
	JZ		SHORT @@CLOSED
	INC 	EBX
@@CLOSED:
	ADD 	ESI,8					 ; Next entry.
	loop	@@LOOP
	MOV 	AH,00
	RET
EMS_GET_OPEN_HANDLES_COUNT ENDP
;
; 674C: AH = 4Ch: Determine number of reserved pages for a Handle
;  inp: handle in DX.
;  out: pages in BX.
;
EMS_GET_NR_OF_ALLOCATED_PAGES PROC
	CALL	EMS_TEST_HANDLE
	xchg	EAX,EDX 				 ; get handle in AL, save EAX in EDX
	MOV 	ECX,[dwMaxEMSPages]
	MOV 	EDI,[EMSPAGETABLE]
	XOR 	BX,BX
	mov		esi,edi 			;dont remove, esi should return last pos!
    jecxz	@@OK
@@LOOP:
	REPNZ	SCAS BYTE PTR [EDI]
	JNZ 	SHORT @@OK		; No more found, so done
	mov		esi,edi			; update last find pos
	INC 	EBX				; one page more
	JMP 	@@LOOP
@@OK:
	xchg	EAX,EDX 		; restore EAX and EDX
	MOV 	AH,00			; Ok.
	RET
EMS_GET_NR_OF_ALLOCATED_PAGES ENDP
;
; 674D: AH = 4Dh: determine Number of reserved pages for all Handles
;	ES:DI -> array of 2 WORD entries (handle, pages)
; out: AH=00 success, BX=number of handles stored in array
;
EMS_GET_ALLOCATED_PAGES PROC
	MOVZX	ESI,WORD PTR [ebp].EMMFRAME.eES
	SHL 	ESI,4						; ES:DI ^ from the storage area
	MOVZX	EDI,DI
	ADD 	ESI,EDI
	PUSH	EAX
	PUSH	EDX
	MOV 	EDI,[EMSSTATUSTABLE]
	XOR 	EAX,EAX 					 ; actual/current Handle-Number
	XOR 	BX,BX						 ; sofar no Handle open
@@NEXT_HANDLE:
	CMP 	WORD PTR [EDI],EMSH_FREE	 ; Assign handle at all ? If
	JZ		SHORT @@NEXT				 ; not, then jump over
	INC 	EBX 						 ; One more Handle is open...
	MOV 	[ESI+0],AX					 ; Place handle
	PUSH	EDI
	MOV 	EDI,[EMSPAGETABLE]			 ; count reserved pages
	XOR 	EDX,EDX 					 ; EDX is counter
	MOV 	ECX,[dwMaxEMSPages]
    jecxz	@@NOPE
@@LOOP:
	REPNZ	SCAS BYTE PTR [EDI] 		 ; Search through Page-Allocation-Table
	JNZ 	SHORT @@NOPE				 ; No more found
	INC 	EDX
	JMP 	@@LOOP
@@NOPE:
	MOV 	[ESI+2],DX					 ; Set the number of determined pages
	add		ESI, 4
	POP 	EDI
@@NEXT:
	ADD 	EDI,8						 ; next Handle, please !
	INC 	EAX
	CMP 	AL,MAX_HANDLES				 ; All Handles processed ?
	JB		SHORT @@NEXT_HANDLE
	POP 	EDX
	POP 	EAX
	MOV 	AH,00						 ; Everything ok.
	RET
EMS_GET_ALLOCATED_PAGES ENDP

;
; 674E: AH = 4Eh: Get & Set Map
; AL = 0,1,2,3
; AL = 0: ES:DI -> array to get info
; AL = 1: DS:SI -> array to set info
; AL = 2: DS:SI -> array to set info, ES:DI -> array to get info
; AL = 3: AL returns size of array (bytes)

@CHKSUM	MACRO	REG
	MOV 	AX,[REG+2]			 ; Calculate checksum
	ADD 	AX,[REG+4]
	ADD 	AX,[REG+6]
	ADD 	AX,[REG+8]
	ENDM

EMS_GET_SET_PAGE_MAP PROC
	CMP 	AL,3					; Subfunction 0 to 3 ?
	JA		bad_subfunc
	JZ		SHORT @@SUBF_3			; Size of field
	CMP 	AL,1
	JZ		SHORT @@SUBF_1			; Set Page Map
; AL = 2: Get & Set Page Map
; AL = 0: Get Page Map - save Hardwarestatus
@@SUBF_0:
	MOVZX	ECX, WORD PTR [ebp].EMMFRAME.eES; ES:DI ^ convert from statusfield in
	SHL 	ECX, 4					; usual REAL-Mode-Format
	MOVZX	EDI, DI
	ADD 	EDI, ECX
	PUSH	ESI
	PUSH	EAX						; save Subfunctioncode
	LEA		esi, [edi+2]			; Currently skip checksum
	mov		cl, [bEmsPhysPages]
	CALL	EMS_SavePagesToEsi
	DEC 	ESI
	DEC		ESI
	@CHKSUM	ESI 					; Calculate checksum and ...
	MOV 	[ESI],AX				; ... store
	POP 	EAX 					; restore and examen subfunctioncode
	POP 	ESI
	CMP 	AL,2					; if subfuction 2
	JZ		SHORT @@SUBF_1			; is wanted , since then also
	MOV 	AH,00					; Subf. 1 still needs be done.
	RET 							; Everything OK.

; Subf. 1: Set Page Map - restore Hardwarestatus
@@SUBF_1:
	MOVZX	ECX,WORD PTR [ebp].EMMFRAME.eDS	; DS:SI ^ convert from statusfield in
	SHL 	ECX,4					 ; usual REAL-Mode-Format
	MOVZX	ESI,SI
	ADD 	ESI,ECX
	@CHKSUM	ESI 					 ; Calculate checksum and check it
	CMP 	[ESI],AX
	JNZ 	SHORT @@CHKERR
	ADD 	ESI,2					 ; Jump over checksum
	mov		cl, [bEmsPhysPages]
	JMP 	EMS_RestorePagesFromEsi

; Checksum is incorrect !
@@CHKERR:
	MOV 	AH,0A3H 				 ; data is destroyed !
	RET

; Subf. 3: Size of the field
@@SUBF_3:
	MOV 	AL, [bEmsPhysPages]
	add		al,al					; need 2 bytes for a page
	add		al,2					; + 2 Bytes for checksum
	MOV 	AH,00					; ChkSum
	RET								; That was it then.

EMS_GET_SET_PAGE_MAP ENDP

bad_subfunc:
	MOV 	AH,8FH					 ; Invalid subfunctioncode !
	RET

;--- check if segment in AX is a mapped page
;--- if yes, return NC and convert segment to physical page in AL

EMS_IsMappedSegm proc
	and al,al			;must begin on a page boundary
    jnz @@notvalid
    mov al,ah
	push ecx
    push edi
    movzx ecx, [bEmsPhysPages]
    mov ah,cl
    mov edi, offset PAGE2SEGM
    repnz scasb
    jnz @@notvalid2
    mov al, ah
    dec al
    sub al, cl
    pop edi
    pop ecx
	clc
	ret
@@notvalid2:
	pop edi
    pop ecx
@@notvalid:
	stc
	ret
EMS_IsMappedSegm endp

; Paste a (absolute) logical page anywhere in address space
; ESI == ptr to PTE
; EAX == linear address
; BX = absolute logical page
; modifies eax, esi, edi, ecx

EMS_MAP_ABS_PAGE_EX proc	
	AND 	BH,BH				; log. page < 0, then fade out
	JS		@@UNMAP
	MOVZX	EDI,BX				; Calculate now the (absolute) pagenumber

	push esi
	shl	edi,2					; convert absolute page to dword entry offset
	add	edi, [EMSPageAllocationStart]	; edi -> EMS page descriptor entry
	movzx esi,[edi].EMSPD.wPD		; pool allocation block count
	movzx ecx,[edi].EMSPD.wNibOfs	; 16K offset from pool block base
if 1
	cmp si,-1	;bad pointer?
	jz @@unmap2	;then unmap this page
endif
	shl	esi,6						; convert to 64-byte block offset
	add	esi, [PoolAllocationTable]	; esi -> pool allocation block for page

	@assumem esi,<LPPOOL_SYSTEM_INFO>

	shl	ecx,14				; 16K to bytes
	mov	edi,[esi].psi_addressK
	shl	edi,10				; K address to bytes
	add	edi,ecx				; edi -> page memory
	
	pop	esi					;restore PTE ptr
	
	@assumem esi,nothing

@@SET:
	ADD 	EDI,111B		; Statusbits: R/W=1,U/S=1,P=1
	MOV 	CL,4			; 1 EMS page are 4 (virtual) pages
@@LOOP:
	MOV 	[ESI],EDI		; Register the new physical Address of
	ADD 	ESI,4			; window
	ADD 	EDI,4096		; Process next 4K-page
	DEC 	CL
	JNZ 	@@LOOP
if ?INVLPG	  
	cmp		[bNoInvlPg],0
	jnz		@@noinvlpg
	mov		cl, 4			; EMS page == 4 physical pages
@@nextpage:    
	INVLPG	ds:[eax]
	add		eax, 1000h
	dec		cl
	jnz 	@@nextpage
	RET
@@noinvlpg:    
endif
	MOV 	EAX,CR3 		; flush TLB
	MOV 	CR3,EAX
	RET
@@unmap2:
	pop		esi
@@UNMAP:
	mov		edi, eax
	
	@DbgOutS <"Unmap EMS page at ">, ?EMSDBG
	@DbgOutD edi, ?EMSDBG
	@DbgOutS <", page table ptr=">, ?EMSDBG
	@DbgOutD esi, ?EMSDBG
	@DbgOutS <10>, ?EMSDBG
	
	JMP 	@@SET
	
EMS_MAP_ABS_PAGE_EX endp
	
; assign a logical page to a physical page
; AL = physical page, BX = logical page (absolute) or -1 (to unmap)
; modifies ESI, EDI, ECX
; the logical page in BX might belong to no handle!
;
EMS_MAP_ABS_PAGE PROC
	PUSH	EAX
	movzx	eax,al
	MOV 	ESI,EAX				;physical page -> esi
    mov		al, [PAGE2SEGM+ESI]
if 0
;--- this optimisation should be safe, but it turned out it is not.
;--- some versions of PKZIP/PKUNZIP don't like it.
	cmp		bx,[PHYSPAGETABLE+ESI*2]
	jz		@@nochange
endif
	MOV 	[PHYSPAGETABLE+ESI*2],BX
	@GETPTEPTR ESI, EAX*4+1000h	; ESI -> PTE
    SHL		eax, 12				; C0 -> C0000, C4-> C4000
	call	EMS_MAP_ABS_PAGE_EX
if ?EMSDBG
	jmp 	@@change
@@nochange:
	@DbgOutS <"MAP_PAGE: mapping of ">,1
	@DbgOutW bx,1
	@DbgOutS <" to page ">,1
	@DbgOutB al,1
	@DbgOutS <" optimised",10>,1
@@change:	 
else	
@@nochange:    
endif
	pop		eax
	ret
EMS_MAP_ABS_PAGE ENDP

	align 4
;
; Check a given Handle for validness.
; In case the Handle is invalid the returnaddress is thrown away
; and afterwards through RET returned to dispatcher.
; Else ESI will point to the handle in EMSSTATUSTABLE array

EMS_TEST_HANDLE PROC
	CMP 	DX,MAX_HANDLES			  ; Out of area ?
	JAE 	SHORT @@BYE
	MOVZX	ESI,DL					  ; Form the pointer from the Handle-Status-
	SHL 	ESI,3					  ; Table and ...
	ADD 	ESI, [EMSSTATUSTABLE]
	CMP 	WORD PTR [ESI], EMSH_FREE ; ... examine, if the Handle has
	JZ		SHORT @@BYE 			  ; been marked as free
	RET
@@BYE:
	ADD 	ESP,4					  ; Throw away call(ing)-address
	MOV 	AH,83H					  ; "Handed over Handle is unknown"
	RET
EMS_TEST_HANDLE ENDP

;--- begin EMS 4.0 functions

;
; 674F: AH = 4Fh: Get & Set partial Map
; AL = 0,1,2 
; AL = 0 (get map), DS:SI -> map to get, ES:DI -> status to receive
; AL = 1 (set map), DS:SI -> map to restore
; AL = 2 (get size), items in BX, returns size of map in AL
; the list for sub function 0 DS:SI points to has following structure:
; WORD	 : items in list
; WORD[] ; segment! addresses for which to get the map info 

?DBG4F equ 0

ems4_get_set_partial_page_map PROC

if ?EMM4F
	cmp al,2
	ja bad_subfunc
	jz @@getsize
	movzx ecx,WORD PTR [ebp].EMMFRAME.eDS
	shl	ecx,4
	movzx esi, si
	add	esi,ecx
	
	cmp al,1
	jz @@setmap

	movzx ecx,WORD PTR [ebp].EMMFRAME.eES
	shl	ecx,4
	movzx edi, di
	add	edi,ecx

	push eax
	lods word ptr [esi]
	movzx eax,ax
	movzx ecx, [bEmsPhysPages]
	cmp eax,ecx
	ja @@failA3
	mov ecx, eax
	stos word ptr [edi]
	jecxz @@done00
@@nextsegm:
	lods word ptr [esi]
	call EMS_IsMappedSegm
	jc @@fail8B
	stos byte ptr [edi]
	movzx eax, al
	mov ax, [PHYSPAGETABLE+eax*2]
	stos word ptr [edi]
	loop @@nextsegm
@@done00:
	pop eax
	mov ah,00
	ret
	
@@fail8B:
	pop eax
	mov ah,8Bh
	ret
@@failA3:
	pop eax
@@failA3_1:
	mov ah,0A3h	;segment count exceeds mappable pages
	ret

@@setmap:
	push eax
	lods word ptr [esi]
	movzx eax,ax
	movzx ecx, [bEmsPhysPages]
	cmp eax,ecx
	ja @@failA3
	mov ecx, eax
	jecxz @@done01
@@nextsegm1:
	mov al, [esi+0]
	cmp al, [bEmsPhysPages]
	jae @@fail8B
	pushad
	mov bx, [esi+1]
	call EMS_MAP_ABS_PAGE
	popad
	add esi, 3
	loop @@nextsegm1
@@done01:
	pop eax
	mov ah,0
	ret

@@getsize:
	cmp bx, 4
	ja @@failA3_1
	mov al, bl
if 0	
	add al, al
	add al, bl		;3 bytes for each entry required
	add al, 2		;+2 bytes for size
else
	inc al			;4 extra bytes for size + checksum?
	shl al,2		;4 bytes (makes Kyrandia 3 work [better]!?)
endif
	mov ah, 00
	ret
else
	jmp EMS_NOT_IMPL
endif

ems4_get_set_partial_page_map ENDP

; 6750:
; AH = 50h: EMS 4.0 map multiple pages
; DX = handle
;  DS:SI -> mapping array
;  CX = items in array (ecx is destroyed, reload it from stack)
;  structure of mapping array:
;  WORD logical page (or -1 to unmap page)
;  WORD physical page (AL=0) or segment address (AL=1)

ems4_map_multi PROC

	cmp	al,1
	ja	bad_subfunc
	
	movzx ecx, word ptr [ebp].EMMFRAME.eEcx	; load EMM entry CX value
	movzx edi, word ptr [ebp].EMMFRAME.eDS
	shl	edi,4
	movzx esi,si
	add	edi,esi		; edi -> map address buffer

if ?MASM
ems4_map_multi_edi::
else
ems4_map_multi_edi:
endif

; perform handle check here so that stack return address isn't blown

	call EMS_TEST_HANDLE
	mov esi, edi	

	push ebx
	push eax
	jecxz @@success
@@multi_loop:
	mov	bx,[esi+0]
	mov	ax,[esi+2]
	add	esi,4
	cmp byte ptr [esp+0],1	;subfunction 1?
	jne	@@mappage
	call EMS_IsMappedSegm	;convert segment in ax to a page no
	mov ah,8Bh
	jc	@@multi_out
@@mappage:
	push ecx
	push esi
	call EMS_MAP_HANDLE_PAGE
	pop	esi
	pop	ecx
	test ah,ah
	jne	@@multi_out	; error occurred
	loop @@multi_loop
@@success:	  
	MOV	ah,00	; no error return
@@multi_out:
	mov [esp+1],ah
	pop eax
	pop	ebx
	ret
	
ems4_map_multi ENDP

; 6751:
; AH = 51h: EMS 4.0 reallocate pages for handle
; DX = handle
; BX = new page count for handle
; out: BX=pages assigned to handle
;
ems4_realloc PROC
	call	EMS_TEST_HANDLE
	call	GetEMSPageCount	;update PAGESAVAIL

	push ebx		; save new pages

	call	EMS_GET_NR_OF_ALLOCATED_PAGES
	
	mov	edi,esi
	mov esi,ebx		; current pages for this handle
	mov	ecx,ebx
	pop	ebx			; restore new page request amount
	add	ecx,[EMSPAGESAVAIL]	; get current pages for handle + available
	cmp	bx,cx
	ja	@@toomany
	
	push ebx
	push eax
	cmp	bx, si		; check new page count against original
	jb	@@shrinkalloc
	je	@@realloc_success	; no change needed

	MOV ECX,[dwMaxEMSPages]
	add ecx,[EMSPAGETABLE]
	sub ecx,edi

; growing the allocation of pages

	sub	bx,si		; get count of new pages needed
@@growpages:
	mov	al,FREEPAGE_ID
	repnz scas BYTE PTR [edi]

	jnz	@@nofind			; couldn't find new page
	call AllocateEMSPage	; allocate the page (preserves all registers)
	jc	@@nofind			; allocation successful?
	mov	[edi-1],dl
	dec	[EMSPAGESAVAIL]
	dec	bx
	jnz	@@growpages
	jmp	@@realloc_success
@@nofind:
	pop eax
	pop	ebx			; todo: release the pages already allocated
@@toomany:
	mov	bx,si		; return original pages owned by handle
	mov	ah,88h
	ret

; trim off the top pages, this avoids modifying the page order!

@@shrinkalloc:
	xchg esi,ebx
	sub	bx,si		; get count of pages to reduce
	mov ecx, edi
	sub ecx, [EMSPAGETABLE]
	dec	edi			
	mov	al,dl
@@shrinkpages:
	std				; scan backwards
	repnz	scas BYTE PTR [edi]
	cld

	add	edi,2		; FreeEMSPage expects edi-1 ptr, not edi+1
	call FreeEMSPage	;preserves registers (but expects DF to be clear!)
	sub	edi,2

	mov	BYTE PTR [edi+1],FREEPAGE_ID
	inc	[EMSPAGESAVAIL]
	dec	bx
	jnz	@@shrinkpages

@@realloc_success:
	pop eax
	pop	ebx
	mov	ah,00		; no error occurred
	ret

ems4_realloc ENDP

; 6752
; AH = 52h: EMS 4.0 attribute related
; AL = 0/1/2, DX=handle, BL=0/1
;
ems4_attribute PROC
	cmp	al,2
	jb	@@get_set_attribute
	ja	bad_subfunc	; this is an invalid subfunction
@@is_volatile:
	xor	ax,ax	; al == 0, volatile attribute only, ah == successful call
	ret
@@get_set_attribute:
	call EMS_TEST_HANDLE; only valid handles please
	or al,al			; 0 is get, 1 is set
	jz @@is_volatile	; only volatile here (none survive warm reboot)
	or bl,bl			; 0 is "make volatile" (true anyway)
	jnz @@cannot_make_nonvolatile
	mov ah,00			; be happy
	ret
@@cannot_make_nonvolatile:
	mov ah,91h		; feature not supported
	ret
ems4_attribute ENDP

; 6753:
; AH = 53h: EMS 4.0 get/set handle name
; AL = 0: get handle name in ES:DI
; AL = 1: set handle name in DS:SI
; DX = handle
;
ems4_handle_names PROC
	cmp	al,1
	ja	bad_subfunc	; this is an invalid subfunction
	jz	@@ems4_setname

	call EMS_TEST_HANDLE
	movzx esi,WORD PTR [ebp].EMMFRAME.eES
	shl	esi,4
	movzx	edi,di
	add	edi,esi		; edi -> handle name buffer address (dest)
	mov	esi, [EMSNAMETABLE]
	movzx	ecx,dx	; handle (index)
	lea esi, [esi+ecx*8]
	jmp @@ems4_getsetname

@@ems4_setname:
	mov edi, esi	;save SI
	call EMS_TEST_HANDLE
	movzx esi,WORD PTR [ebp].EMMFRAME.eDS
	shl	esi,4
	movzx edi, di	;this is original SI
	add	esi,edi		; esi -> handle name (source)
	mov	edi, [EMSNAMETABLE]
	movzx	ecx,dx	; handle (index)
	lea edi, [edi+ecx*8]
@@ems4_getsetname:	  
	mov	ecx,[esi+0]	; transfer handle name to es:di
	mov	[edi+0],ecx
	mov	ecx,[esi+4]
	mov	[edi+4],ecx
	mov	ah,00	; no error return
	ret

ems4_handle_names ENDP

; 6754:
; AH = 54h: EMS 4.0 get various handle info
;
; AL = 0: get handle directory into ES:DI
; AL = 1: search handle by name in DS:SI, return handle in DX
; AL = 2: get total handles in BX

ems4_get_handle_info PROC
	cmp	al,1
	jb	getallhand
	je	@@find_handle_by_name
	cmp	al,2
	ja	bad_subfunc	; this is an invalid subfunction

	mov	bx,MAX_HANDLES
	mov	ah,00	; zero ah, no error return
	ret

; write handle directory to caller buffer
; return in AL number of open handles
getallhand:
	movzx	esi,WORD PTR [ebp].EMMFRAME.eES
	shl	esi,4
	movzx edi,di
	add	esi,edi
	mov	edi, [EMSSTATUSTABLE]
	mov	al,0		; AL will be count of open handles
	xor ecx, ecx
@@scan_handles:
	cmp	WORD PTR [edi], EMSH_FREE
	jz	@@free_handle
	inc	al		; count that open handle

	push eax
	mov	[esi+0],cx
	mov	eax,edi
	sub	eax, [EMSSTATUSTABLE]	; convert to table offset
	add	eax, [EMSNAMETABLE]		; offsets are identical
	push DWORD PTR [eax+0]		; copy handle name
	pop	DWORD PTR [esi+2]
	push DWORD PTR [eax+4]
	pop	DWORD PTR [esi+6]
	add	esi,10
	pop	eax
	
@@free_handle:
	add	edi,8
	inc ecx
	cmp cl, MAX_HANDLES
	jb	@@scan_handles
	mov	ah,00
	ret

@@find_handle_by_name:
	movzx	edi,WORD PTR [ebp].EMMFRAME.eDS
	shl	edi,4
	movzx	esi,si
	add	edi,esi
	
	push ebx
	push eax
	mov	eax,[edi+0]		; fetch to-be-searched name
	mov	ebx,[edi+4]		; (8 byte binary string)
if 0 ;it is valid to search a handle with no name!
	or	eax,eax
	jnz	@@valid_search_term
	or	ebx,ebx
	jz	@@invalid_search_term
@@valid_search_term:
endif
	xor	ecx,ecx
	mov	edi, [EMSNAMETABLE]
	mov	esi, [EMSSTATUSTABLE]
@@scan_for_name:
	cmp	WORD PTR [esi], EMSH_FREE	; closed, auto-fail
	jz	@@another_handle
	cmp	[edi+0],eax
	jnz	@@another_handle
	cmp	[edi+4],ebx			; Note that open handles do not have
	jz	@@found_handle		; to have a name.
@@another_handle:
	add	esi,8
	add edi,8
	inc	ecx
	cmp	cl,MAX_HANDLES
	jb	@@scan_for_name
	pop eax
	pop ebx
	mov	ah,0a0h			; "no handle could be found"
	ret
@@found_handle:
	pop eax
	pop ebx
	mov dx,cx
	mov	ah,00
	ret
if 0
@@invalid_search_term:	; The error code descr. is misleading:
	pop eax
	pop ebx
	mov	ah,0a1h			; "handle found had no name"
	ret
endif

ems4_get_handle_info ENDP

log_phys_map struc
wLogPage	   DW ?
wPhysPage	   DW ?	;physical pages if AL=0, segment if AL=1
log_phys_map ends

map_and_jump struc
dwJmpAddr	 DD ? ; far16 jump address
bSize		 DB ? ; items in log_phys_map member
pLogPhysMap  DD ? ; pointer to log_phys_map structure
map_and_jump ends

if ?EMM5556

;--- EDI = v86-DS

ems4_mapmultiplepages proc
	shl	edi,4
	movzx esi,si
	add	edi,esi		; edi -> map address buffer
	push edi
	movzx ecx, word ptr [edi].map_and_jump.pLogPhysMap+0
	movzx esi, word ptr [edi].map_and_jump.pLogPhysMap+2
	shl esi, 4
	add esi, ecx
	movzx ecx, [edi].map_and_jump.bSize
	mov edi, esi
	call ems4_map_multi_edi	;map ECX pages from EDI, DX=handle
	pop edi
	ret
ems4_mapmultiplepages endp
endif

; 6755:
; AH = 55h: EMS 4.0 alter page map and jump
; AL = 0/1
; DX = handle
; DS:SI -> map_and_jump structure

ems4_alter_map_jump proc
if ?EMM5556
	movzx	edi, word ptr [ebp].EMMFRAME.eDS
	call ems4_mapmultiplepages
	and ah,ah
	jnz @@bye
	mov ecx,[edi].map_and_jump.dwJmpAddr
	mov word ptr [ebp].EMMFRAME.eEip, cx
	shr ecx, 16
	mov [ebp].EMMFRAME.eCS, ecx
@@bye:
	ret
else
	jmp EMS_NOT_IMPL
endif
ems4_alter_map_jump endp

; 6756:
; AH = 56h: EMS 4.0 alter page map and call
; AL = 0/1/2
; DX = handle
; if AL=0/1: [in] DS:SI -> map_and_jump structure ()
; if AL=2: [out] BX = additional stack space required

ems4_alter_map_call proc
if ?EMM5556
	cmp al,2
	jz	@@getstackspace
	movzx	edi, word ptr [ebp].EMMFRAME.eDS

	push dword ptr [PHYSPAGETABLE+4]	;save the Page Frame status
	push dword ptr [PHYSPAGETABLE+0]
	
	call ems4_mapmultiplepages
	and ah,ah
	jnz @@bye

;--- now store a v86_call structure onto the v86 stack
;--- + 8 extra bytes for the old frame state

	movzx	esi, word ptr [ebp].EMMFRAME.eEsp
	movzx	ecx, word ptr [ebp].EMMFRAME.eSS
	sub		si, size v86_call + 2*4
	shl		ecx, 4
	mov		word ptr [ebp].EMMFRAME.eEsp, si
	add		esi, ecx

	pop		dword ptr [esi+size v86_call+0]	;save the old frame state
	pop		dword ptr [esi+size v86_call+4]
	
	mov ecx,[edi].map_and_jump.dwJmpAddr	;set new v86 CS:IP and
	xchg cx, word ptr [ebp].EMMFRAME.eEip
	mov word ptr [esi].v86_call.dwOldCSIP+0, cx	;save old CS:IP in v86_call
	shr ecx, 16
	xchg cx, word ptr [ebp].EMMFRAME.eCS
	mov word ptr [esi].v86_call.dwOldCSIP+2, cx

	mov		ecx, [dwRSeg]
    shl		ecx, 16
	mov		cl, [bBpBack]
	mov		[esi].v86_call.dwBP+0, ecx		;store breakpoint on top of stack
	mov		[esi].v86_call.dwRetAddr, offset @@restore_page_frame
	
	ret
@@bye:
	add		esp,2*4
	ret
@@getstackspace:
	mov bx,size v86_call + 2*4
	ret

;--- this is called when the v86 code executes a RETF
;--- ebx -> v86 stack (the first DWORD of v86_call is skipped already!)
	
@@restore_page_frame:	
	add 	[ESP].V86FRGP.gESP,8	;adjust v86 stack for the 8 extra bytes
	add		ebx, size v86_call		;skip v86_call structure
	pushad
	mov		esi,ebx
	mov		cl, [bEmsPhysPages]
	call	EMS_RestorePagesFromEsi
	mov		byte ptr [ESP+size PUSHADS].V86FRGP.gEAX+1,ah
	popad
	@v86popreg
	add 	ESP,4+4 		; remove error code + int #
	IRETD
else
	jmp EMS_NOT_IMPL
endif
ems4_alter_map_call endp

; 6757:
; AH = 57h: EMS 4.0 move/exchange memory region
; AL = 0: move memory region
; AL = 1: exchange memory region
; DS:SI -> EMM57

;-- this function should work even if no EMS page frame is defined!
;-- EMS regions may overlapp!

EMM57 struc
e57_dwSize	DD ?	; +0  size of region
e57_bSrcTyp DB ?	; +4  src memory type
e57_wSrcHdl DW ?	; +5  src handle
e57_wSrcOfs DW ?	; +7  src ofs
e57_wSrcSeg DW ?	; +9  src segm./log. page
e57_bDstTyp DB ?	; +11 dst memory type
e57_wDstHdl	DW ?	; +12 dst handle
e57_wDstOfs	DW ?	; +14 dst ofs
e57_wDstSeg	DW ?	; +16 dst segm./log. page
EMM57 ends

;--- memory type:
;--- 00: conv. memory
;--- 01: expanded memory
;--- handle:
;--- == NULL: conv. memory
;--- <> NULL: EMS handle

Saved_Eax	equ <ebp+0>
RegionDest	equ <ebp-4>

ems4_memory_region PROC

	cmp	al,1
	ja	bad_subfunc	; this is an invalid subfunction
	movzx edi,WORD PTR [ebp].EMMFRAME.eDS
	movzx esi,si
	shl	edi,4
	add	edi,esi		; edi -> EMS region buffer address

	push	ebx
	push	edx
	push	eax
;	push	ebp			;ebp is already saved
	mov		ebp, esp
	sub		esp, 1*4
	push	offset @@exit	;push error return address for test_handle

if ?EMSDBG
	@DbgOutS <"EMM 57h: siz=">,1
	@DbgOutD [edi].EMM57.e57_dwSize,1
	@DbgOutS <", src=">,1
	@DbgOutB [edi].EMM57.e57_bSrcTyp,1
	@DbgOutS <"/">,1
	@DbgOutW [edi].EMM57.e57_wSrcHdl,1
	@DbgOutS <"/">,1
	@DbgOutW [edi].EMM57.e57_wSrcSeg,1
	@DbgOutS <"/">,1
	@DbgOutW [edi].EMM57.e57_wSrcOfs,1
	@DbgOutS <", dst=">,1
	@DbgOutB [edi].EMM57.e57_bDstTyp,1
	@DbgOutS <"/">,1
	@DbgOutW [edi].EMM57.e57_wDstHdl,1
	@DbgOutS <"/">,1
	@DbgOutW [edi].EMM57.e57_wDstSeg,1
	@DbgOutS <"/">,1
	@DbgOutW [edi].EMM57.e57_wDstOfs,1
endif

	mov	ecx,[edi].EMM57.e57_dwSize
	test ecx,ecx
	je	@@ok		; always succeeds if no bytes are moved
	mov	ah,96h		; preload error code, region length exceeds 1M
	cmp	ecx,65536*16
	ja	@@exit

; process region destination information

	movzx	ecx,[edi].EMM57.e57_wDstOfs
	movzx	ebx,[edi].EMM57.e57_wDstSeg
	cmp		[edi].EMM57.e57_bDstTyp,0	;0 = conv, 1 = expanded
	je	@@calc_dest_conv

; destination is EMS

	mov dx, [edi].EMM57.e57_wDstHdl
	call EMS_TEST_HANDLE
	
	mov	ah,95h		; preload error code, specified offset is outside logical page
	test ch,0C0h	; ecx must be < 4000h
	jnz	@@exit

	mov		eax, ?SCRATCHLINEAR+104000h
	lea		esi, [ecx+eax]
	mov		[RegionDest],esi
	add		ecx,[edi].EMM57.e57_dwSize
	
; try to map in all needed pages (max space is 1 MB + 16 kB)

	@GETPTEPTR esi, 2000h+?SCRATCHPTE+104h*4, 1
	call	mappages
	jmp 	@@calc_dest_done

@@calc_dest_conv:
	shl	ebx,4		; convert seg to memory offset
	add	ebx,ecx
	mov	[RegionDest],ebx
	mov ah,0A2h		; conv memory region must not exceed 1 MB boundary
	add	ebx, [edi].EMM57.e57_dwSize
	cmp ebx, 100000h
	ja	@@exit
@@calc_dest_done:	 

; process region source information

	movzx	ecx,[edi].EMM57.e57_wSrcOfs
	movzx	ebx,[edi].EMM57.e57_wSrcSeg
	cmp		[edi].EMM57.e57_bSrcTyp,0	; 0 = conv, 1 = expanded
	je	@@calc_src_conv

; source is EMS

	mov dx, [edi].EMM57.e57_wSrcHdl
	call EMS_TEST_HANDLE
	
	mov	ah,95h		; preload error code to specified offset is outside logical page
	test ch,0C0h	; ecx must be < 4000h
	jnz	@@exit
	
	mov		eax,?SCRATCHLINEAR
	add		ecx,[edi].EMM57.e57_dwSize
	
; try to map in all needed pages (max size is 1 MB)

	@GETPTEPTR esi, 2000h+?SCRATCHPTE, 1
	call	mappages
	mov		ecx, [edi].EMM57.e57_dwSize
	movzx	esi, [edi].EMM57.e57_wSrcOfs
	add		esi, ?SCRATCHLINEAR

; if src and dest are EMS test if they overlapp
	
	cmp		[edi].EMM57.e57_bDstTyp,0	; is dst also expanded memory?
	jz		@@calc_src_done
	cmp		dx, [edi].EMM57.e57_wDstHdl	; are handles equal?
	jnz 	@@calc_src_done

	movzx eax, [edi].EMM57.e57_wDstSeg
	movzx edx, [edi].EMM57.e57_wSrcSeg
	shl eax, 14
	shl edx, 14
	or	ax, [edi].EMM57.e57_wDstOfs
	or	dx, [edi].EMM57.e57_wSrcOfs

;--- the problem case is:
;--- source < destination AND source + size > destination

	cmp edx, eax		; source < destination?
	jnc @@calc_src_done	; no, use std copy
	add edx, ecx		; edx == source + size
	cmp eax, edx		; destination >= source + size?
	jnc @@calc_src_done	; yes, use std copy
	jmp @@overlapped

@@calc_src_conv:
	shl	ebx,4			; convert seg to memory offset
	add	ebx,ecx
	mov	ecx, [edi].EMM57.e57_dwSize
	mov	esi,ebx
	mov ah,0A2h			; conv memory region must not exceed 1 MB boundary
	add ebx, ecx
	cmp ebx, 100000h
	ja	@@exit
	
@@calc_src_done:
	mov	edi, [RegionDest]
	cmp	byte ptr [Saved_Eax],0
	jne	@@xchg
	MOV 	EAX,ECX
	SHR 	ECX,2
	REP 	MOVS DWORD PTR [ESI], DWORD PTR [EDI]
	@BIG_NOP
	MOV 	ECX,EAX
	AND 	ECX,3
	REP 	MOVS BYTE PTR [ESI],BYTE PTR [EDI]
	@BIG_NOP
@@ok:
	mov	ah,00	; zero ah, no error return

; exit with code in AH

@@exit:
if ?EMSDBG
	@DbgOutS <"=">,1
	@DbgOutB ah,1
	@DbgOutS <10>,1
endif
	mov [Saved_Eax+1], ah	;set AH in saved EAX on stack
	mov esp, ebp
;   pop ebp
	pop eax
	pop	edx
	pop	ebx
	ret

@@overlapped:
	mov	edi, [RegionDest]
	cmp	byte ptr [Saved_Eax],0
	jne	@@xchg
	std
	lea		esi,[esi+ecx-1]
	lea		edi,[edi+ecx-1]
	mov		eax,ecx
	and		ecx,3
	REP 	MOVS BYTE PTR [ESI],BYTE PTR [EDI]
	MOV 	ECX,EAX
	sub		esi,3
	sub		edi,3
	SHR 	ECX,2
	REP 	MOVS DWORD PTR [ESI], DWORD PTR [EDI]
	cld
	mov		ah,92h	;status "overlapp occured"
	jmp		@@exit
@@xchg:
	mov	ah,cl
	and	cl,3
	je	@@fullx
@@xb:
	mov	al,[edi]
	movsb
	mov	[esi-1],al
	dec cl
	jnz	@@xb
@@fullx:
	mov	cl,ah
	shr	ecx,2
	jz @@ok
@@xdw:
	mov	eax,[edi]
	movsd
	mov	[esi-4],eax
	dec ecx
	jnz @@xdw
	jmp	@@ok
	
mappages:
	add		ecx, eax		;let ecx point to end of region
@@nextpagetomap:
	pushad
	call	EMS_get_abs_page;get abs page of DX:BX in BX
	jc		@@exit
	call	EMS_MAP_ABS_PAGE_EX		;requires EAX,BX,ESI to be set
	popad
	inc		ebx				;next EMS page
	add		eax, 4000h		;proceed with linear address
	add		esi, 4*4		;proceed with PTE pointer
	cmp		eax,ecx
	jb		@@nextpagetomap
	retn
	
Saved_Eax	equ <>
RegionDest	equ <>

ems4_memory_region ENDP

; 6758:
; AH = 58h: EMS 4.0 get addresses of mappable pages, number of mappable pages
; AL = 0: ES:DI -> buffer, returns segments in CX
; structure of buffer item: WORD segment, WORD physical page
; AL = 1: returns segments in CX

ems4_get_mappable_info PROC
	cmp	al,1
	ja	bad_subfunc	; this is an invalid subfunction

	movzx ecx, [bEmsPhysPages]
	mov	WORD PTR [ebp].EMMFRAME.eEcx,cx	; return mappable pages in CX

	cmp al,1
	je	nummap
	jecxz nummap
	
	movzx	esi,WORD PTR [ebp].EMMFRAME.eES
	shl	esi,4
	movzx	edi,di
	add	edi,esi	; edi -> handle name buffer address
	mov esi, offset PAGE2SEGM
    xor ecx, ecx
    push eax
@@mapinfo_loop:
	lodsb
    mov ah, al
    mov al, 00
    stosw			; base address
    mov ax,cx
    stosw			; physical page
	inc ecx			; next physical page
	cmp cl,[bEmsPhysPages]
	jnz	@@mapinfo_loop
    pop eax
nummap:
	mov	ah,00	; no error return
	ret

ems4_get_mappable_info ENDP

; 6759:
; AH = 59h: EMS 4.0 get hardware config/get number of raw pages
; AL = 1: return raw pages in DX and BX
; AL = 0: get hardware config in ES:DI

EMM59 struc
e59_pgsize	dw ?	;raw page size in paragraphs
e59_altsets dw ?	;number of alternate register sets
e59_sizcont dw ?	;size of mapping context save area in bytes
e59_dmasets dw ?	;dma register sets
e59_dmaflgs dw ?	;dma flags
EMM59 ends

ems4_get_config PROC
	cmp	al,1	; only subfunctions 0+1 supported
	ja	bad_subfunc
	je	EMS_GET_UNALLOCATED_PAGE_COUNT
	
if 0
	mov	al,ds:[7FFF0000h]	;generate an exception 0Eh
endif

	movzx	esi, WORD PTR [ebp].EMMFRAME.eES
	shl		esi, 4
	movzx	edi, di
	add		edi, esi
	mov		[edi].EMM59.e59_pgsize, 1024
	mov		[edi].EMM59.e59_altsets, 0
	mov		[edi].EMM59.e59_sizcont, 4*2
	mov		[edi].EMM59.e59_dmasets, 0
	mov		[edi].EMM59.e59_dmaflgs, 0
	mov		ah,0
	ret

ems4_get_config ENDP

; 675A:
; AH = 5ah: EMS 4.0 allocate handle and standard/raw pages
; in  AL = 0/1
; in  BX = pages to allocate (may be 0)
; out DX = handle
;
ems4_allocate_pages PROC

	cmp	al,1	; subfunction must be 0 or 1, we don't care if either
	ja	bad_subfunc
	jmp	allocate_pages_plus_zero

ems4_allocate_pages ENDP

if ?VCPI
;
; AX=DE00: VCPI presence detection
;  return BH = 1 (major version), BL = 0 (minor version)
;
VCPI_Presence	PROC
	mov	bx,100h
	mov	ah,00
	ret
VCPI_Presence	ENDP

;
; AX=DE01: VCPI get protected mode interface
;  inp: es:di -> client zero page table (to fill)
;		ds:si -> three descriptor table entries in client's GDT (to fill)
;  out: [es:di] page table filled
;		di: first uninitialized page table entry (advanced by 4K)
;	   ebx: offset to server's protect mode code segment
;
;--- dont forget: ECX, ESI, EDI are saved on the stack

VCPI_GetInterface	PROC
	movzx edi,WORD PTR [ebp].EMMFRAME.eDS
	shl	edi,4
	movzx esi,si
	add	edi,esi			; esi -> client GDT entries
	mov	esi, offset GDT + FLAT_CODE_SEL
	movsd
    movsd
    movsd
    movsd

	movzx esi,WORD PTR [ebp].EMMFRAME.eES
    movzx edi,WORD PTR [ebp].EMMFRAME.eEdi
	shl	esi,4
	add	edi,esi				; edi -> client zero page table
	@GETPTEPTR	esi,1000h,1	; esi -> page table for first 1M

;--- Jemm386 must ensure that 
;--- the VCPI_PM_ENTRY label will be in shared memory. Since this label is
;--- now at the very beginning of V86 segment, 1 page should suffice

if ?CODEIN2PAGE
	mov 	ecx, 440h+8 ;this is offset in page table for 112000h
else
	mov 	ecx, 440h+4 ;this is offset in page table for 111000h
endif	 
	add		word ptr [ebp].EMMFRAME.eEdi,cx	; modify edi
	shr		ecx, 2
	push	eax
vgiloop:
	lods dword ptr [esi]
	and	ah,0F1h		; clear bits 9-11
	stos dword ptr [edi]
	loop vgiloop
;;	and byte ptr es:[edi-4], not 2	;reset r/w for 110xxx page.
	pop	eax

	mov	ebx,OFFSET VCPI_PM_Entry
	mov	ah,00
	ret

VCPI_GetInterface	ENDP

; AX=DE02: VCPI get maximum physical memory address
;  return edx == physical address of highest 4K page available
;
VCPI_GetMax	PROC
	mov	edx,[dwTotalMemory]
if ?EMX
	test bV86Flags, V86F_EMX
	jz @@noemx
	mov	edx,[dwMaxMem16K]	;mem in 16K
	shl edx, 12+2			;convert to bytes
	add edx, [FIRSTPAGE]
@@noemx:	
endif	 
	dec	edx
	and	dx,NOT 0fffh

	mov	ah,00			; flag success
	ret
VCPI_GetMax	ENDP

;
; AX=DE03: VCPI get number of free pages
;  return edx == number of free pages
;
; this functions may also be called from protected-mode.
; then SS+DS+ES are flat, but they are NOT FLAT_DATA_SEL!!!

VCPI_GetFree	PROC
	push eax
	call GetFreeXMS4KPageCount	;edx==free 4k pages in XMS
	mov	ecx,edx

	call Pool_Get4KPageCount	; current free in eax, current total in edx
	add	ecx,eax		; ecx == all free pages
	sub	edx,eax		; edx == current allocated (total - free)
	jc	@@nofree	; this shouldn't happen, but return no free it if does
	xchg edx,ecx	; free pages count in edx, allocated count in ecx

; total free must be <= MAXMEM16K * 4 - current allocated
;  otherwise set total free = MAXMEM16K * 4 - current allocated
	mov	eax,[dwMaxMem16K]
	shl	eax,2			; convert to maximum 4K blocks
	sub	eax,ecx
	jc	@@nofree
	cmp	eax,edx
	jae	@@done
	mov	edx,eax

@@done:
	pop	eax
	mov	ah,00			; flag success
	ret

@@nofree:
	xor	edx,edx
	jmp	@@done

VCPI_GetFree	ENDP

;
; AX=DE04: VCPI allocate a 4K page
;  return edx == physical address of 4K page allocated
;
; this functions may also be called from protected-mode.
; then SS+DS+ES are flat, but they are NOT FLAT_DATA_SEL!!!

VCPI_Allocate4K	PROC
	push	eax		; save high word of eax
	push	edx
	call	Pool_Get4KPageCount	; current free in eax, current total in edx

; fail if current total - current free (allocated) >= MAXMEM16K * 4

	sub	edx,eax			; edx == current allocated (total - free)
	jc	@@fail			; shouldn't happen, but fail if it does
	mov	eax,[dwMaxMem16K]
	shl	eax,2			; convert to maximum 4K blocks
	cmp	eax,edx
	jbe	@@fail

	call Pool_Allocate4KPage
	jnc	@@success

	call Pool_ExpandAnyBlock
	or	edx,edx			; see if any pool block has 4K free
	je	@@tryxms

; pool block was expanded, so an allocation should succeed
	call Pool_Allocate4KPage
	jnc	@@success

; shouldn't reach this point since pool expansion complete,
;  fall through to trying XMS

@@tryxms:
	call AllocateXMSForPool
	jc	@@fail
	call Pool_Allocate4KPage	; this should always succeed
	jc	@@bad2			; failed due to internal fault, so fail allocation

@@success:
	pop eax				; throw away saved EDX on stack
	pop	eax
	mov	ah,00
	ret

@@bad2:
@@fail:
	pop edx
	pop eax
	mov	ah,88h
	ret

VCPI_Allocate4K	ENDP

;
; AX=DE05: VCPI free a 4K page
;  entry edx == physical address of 4K page to free
;
VCPI_Free4K	PROC
	call Pool_Free4KPage
	jc	@@bad
	mov	ah,00			; flag success
	ret

@@bad:
	mov	ah,8ah
	ret

VCPI_Free4K	ENDP

;
; AX=DE06: VCPI get physical address of 4K page in first megabyte
;  entry cx = page number (cx destroyed, use stack copy)
;  return edx == physical address of 4K page
;
VCPI_GetAddress	PROC
	movzx	ecx, word ptr [ebp].EMMFRAME.eEcx
	cmp	cx,256
	jae	@@vga_bad		; page outside of first megabyte

	@GETPTE	edx, ecx*4+1000h
	and	dx,0f000h		; mask to page frame address
	mov	ah,00			; flag success
	ret

@@vga_bad:
;;	xor	edx,edx			; do not modify EDX in case of failure!
	mov	ah,8bh
	ret

VCPI_GetAddress	ENDP

;
; AX=DE07: VCPI read CR0
;  return EBX == CR0
;
VCPI_GetCR0	PROC
	mov	ebx,cr0
	mov	ah,00			; flag success
	ret
VCPI_GetCR0	ENDP
 
;
; AX=DE08: VCPI read debug registers
;  call with ES:DI buffer pointer. Returns with buffer filled.
;  (8 dwords, dr0 first, dr4/5 undefined)
;
VCPI_ReadDR	PROC
	movzx esi,WORD PTR [ebp].EMMFRAME.eES
	shl	esi,4
	movzx edi,di
	add	edi,esi
    push eax
	mov	eax,dr0
    stosd
	mov	eax,dr1
    stosd
	mov	eax,dr2
    stosd
	mov	eax,dr3
    stosd
	mov	eax,dr6
    mov [edi+8], eax
    stosd
    mov eax,dr7
    mov [edi+8], eax
    stosd
    pop eax
	mov	ah,00			; flag success
	ret
VCPI_ReadDR	ENDP

;
; AX=DE09: VCPI write debug registers
;  call with ES:DI buffer pointer. Updates debug registers.
;  (8 dwords, dr0 first, dr4/5 ignored)
;
VCPI_WriteDR	PROC
	movzx	esi,WORD PTR [ebp].EMMFRAME.eES
	shl	esi,4
	movzx	edi,di
	add	esi,edi
    push eax
    lodsd
	mov	dr0,eax
    lodsd
	mov	dr1,eax
    lodsd
	mov	dr2,eax
    lodsd
	mov	dr3,eax
    add esi,8
    lodsd
	mov	dr6,eax
    lodsd
	mov	dr7,eax
    pop eax
	mov	ah,00			; flag success
	ret
VCPI_WriteDR	ENDP

;
; AX=DE0A: VCPI get 8259A interrupt vector mappings
;  return bx == 1st vector mapping for master 8259A (IRQ0-IRQ7)
;	 cx == 1st vector mapping for slave 8259A (IRQ8-IRQ15)
;
VCPI_GetMappings	PROC
	mov	bx, [wMasterPICBase]
	mov	cx, [wSlavePICBase]
	mov	word ptr [ebp].EMMFRAME.eEcx,cx	; client CX is on stack
	mov	ah,00			; flag success
	ret
VCPI_GetMappings	ENDP

; AX=DE0B: VCPI set 8259A interrupt vector mappings
;  entry bx == 1st vector mapping for master 8259A (IRQ0-IRQ7)
;	 cx == 1st vector mapping for slave 8259A (IRQ8-IRQ15)
;-- this is meant just as info, the client has to program the PIC itself

VCPI_SetMappings	PROC

	mov ecx, [ebp].EMMFRAME.eEcx
	mov [wMasterPICBase],bx
	mov [wSlavePICBase],cx
	mov	ah,00			; flag success
	ret
VCPI_SetMappings	ENDP


endif	;?VCPI


; Pool memory management routines

; dynamically compute EMS pages, store to EMSPAGESAVAIL
;  if fixed allocation, check only current EMS store
; destroy no registers

GetEMSPageCount	PROC

	pushad

	call GetFreeXMS4KPageCount
	shr	edx,2			; convert 4K count to 16K pages
	mov	ecx,edx

; get used EMS pages in edx, available in eax (VCPI/EMS used in edi)
	call Pool_Get16KPageCount
	add	eax,ecx			; eax == potential pages available
    
if 1    

	mov ecx, [dwMaxEMSPages]
    sub ecx, edx		; ecx == free EMS pages numbers
    jbe @@nofree
    cmp ecx, eax		; eax == free 16k pages usable for EMS
    jnc @@eaxok
    mov eax, ecx		; use the lower value
@@eaxok:    

else
	cmp	[bNoPool],0		; check if pool sharing
	jne	@@ret

; total free must be <= MAXMEM16K - current VCPI/EMS allocated/used
;  otherwise set total free = MAXMEM16K - allocated

	mov	ecx, [dwMaxMem16K]
if ?POOLDBG
	@DbgOutS <"GetEMSPageCount: dwMaxMem16k=">,1
	@DbgOutD ecx,1
	@DbgOutS <"VCPI/EMS used=">,1
	@DbgOutD edi,1
	@DbgOutS <"avail EMS=">,1
	@DbgOutD eax,1
	@DbgOutS <10>,1
endif
	sub	ecx,edi			; amount that can still be allocated (total - used)
	jc	@@nofree
	cmp	eax,ecx
	jbe	@@pagemax		; available less than amount allowed for allocation
	mov	eax,ecx
@@pagemax:
	cmp	eax, [dwMaxEMSPages]; limit possible EMS pages to maximum allowed
	jb	@@noadj1
	mov	eax, [dwMaxEMSPages]
@@noadj1:
	cmp	eax,edx			; ensure no overflow from improper current value
	ja	@@noadj2
	mov	eax,edx
@@noadj2:
	sub	eax,edx			; max - used == max available

endif

@@ret:
if ?POOLDBG
	@DbgOutS <"GetEMSPageCount: PAGESAVAIL=">,1
	@DbgOutD eax,1
	@DbgOutS <10>,1
endif
	mov	[EMSPAGESAVAIL],eax
	popad
	ret

@@nofree:
	xor	eax,eax
	jmp	@@ret

GetEMSPageCount	ENDP


; allocate an EMS page
; upon entry [edi-1] -> EMS page table entry for page
; return carry if fail, due to internal failure
; destroy no registers

AllocateEMSPage	PROC
	push	eax
	push	ebx
	push	edx

	lea	ebx,[edi-1]
	sub	ebx, [EMSPAGETABLE]		; absolute offset in table
	shl	ebx,2					; dword/entry
	add	ebx, [EMSPageAllocationStart]	; ebx -> ptr EMSPD
	cmp	ebx, [EMSPageAllocationEnd]
	jae	@@bad1				 	; out of range

	call Pool_Allocate16KPage
	jnc	@@success

	call Pool_ExpandAnyBlock
	or	edx,edx			; see if any pool block has 16K free
	je	@@tryxms

; pool block was expanded, so an allocation should succeed
	call Pool_Allocate16KPage
	jnc	@@success

; shouldn't reach this point since pool expansion complete,
;  fall through to trying XMS

	@DbgOutS <"AllocateEMSPage, internal error",10>,?POOLDBG

@@tryxms:
	call AllocateXMSForPool
	jc	@@fail
	call Pool_Allocate16KPage	; this should always succeed
	jc	@@bad2			; failed due to internal fault, so fail allocation
@@success:

;  set EMSPD.wPD == 64-byte allocation block count from start
;  set EMSPD.wNibOfs == nibble offset in allocation block (2 * 48)

	shr	eax,6		; convert to 64-byte offset (block count)
	mov	[ebx].EMSPD.wPD,ax
	mov	[ebx].EMSPD.wNibOfs,dx
    clc
@@exit:
	pop	edx
	pop	ebx
	pop	eax
	ret

@@bad1:
	@DbgOutS <"AllocateEMSPage, bad1 reached",10>,?POOLDBG
@@bad2:
	@DbgOutS <"AllocateEMSPage, bad2 reached",10>,?POOLDBG
@@fail:
	@DbgOutS <"AllocateEMSPage, fail reached",10>,?POOLDBG
	stc
	jmp	@@exit

AllocateEMSPage	ENDP


; release EMS page, [edi-1] -> EMS page table entry for page
; destroy no registers

FreeEMSPage	PROC
	push	edx
	mov	edx,edi
	dec	edx
	sub	edx, [EMSPAGETABLE]				; absolute offset in table
	shl	edx,2							; dword/entry
	add	edx, [EMSPageAllocationStart]	; edx -> EMS page descriptor entry
	cmp	edx, [EMSPageAllocationEnd]
	jae	@@ret							; out of range

	call Pool_Free16KPage

@@ret:
	pop	edx
	ret
FreeEMSPage	ENDP

; get total and free 4K page count in pool
; out: EDX=total pages
; out: EAX=free pages
; destroy no other registers
; do not push/pop segment registers here!
; the function is called from VCPI protected mode API

Pool_Get4KPageCount	PROC
	push	esi
	push	ecx
	mov	esi, [PoolAllocationTable]

	xor	eax,eax
	mov	edx,eax

@@findblkloop:
	cmp	[esi].POOL_SYSTEM_INFO.psi_addressK,0
	je	@@nextblock		; unused/deallocated block
	movzx ecx,[esi].POOL_SYSTEM_INFO.psi_16kmax
	lea	edx,[edx+ecx*4]	; convert to 4K count, known 16-bit quantity
	mov cx,[esi].POOL_SYSTEM_INFO.psi_4kfree
	add	eax,ecx

@@nextblock:
	add	esi,POOLBLOCK_TOTAL_SPACE
	cmp	esi, [PoolAllocationEnd]
	jb	@@findblkloop

	pop	ecx
	pop	esi
	ret

Pool_Get4KPageCount	ENDP


; get used 16K (EMS) page count in pool
; out: EDX = used EMS page count
; out: EAX = free EMS pages
; [out: EDI = VCPI/EMS used pages] obsolete
; destroy no other registers

Pool_Get16KPageCount	PROC
	push	esi
	push	ecx

	mov ecx, [dwMaxEMSPages]
	and ecx, ecx
	jz @@noems
	mov	edi, [EMSPageAllocationStart]
	or eax,-1
	xor edx, edx
@@countloop:
	repz scas dword ptr [edi]
	jz @@done1	  
	inc	edx
	and ecx, ecx
	jnz @@countloop
@@done1:	
;	xor edi, edi
	xor eax, eax
	mov	esi, [PoolAllocationTable]

	@assumem esi,LPPOOL_SYSTEM_INFO

@@findblkloop:

	cmp	[esi].psi_addressK,0
	je	@@nextfind

;	movzx	ecx,[esi].psi_16kmax
;	sub	cl,[esi].psi_16kfree
;	add	edi,ecx					; update total pages used
	mov	cl,[esi].psi_16kfree	; high words known zero
	add	eax,ecx

@@nextfind:
	add	esi,POOLBLOCK_TOTAL_SPACE
	cmp	esi, [PoolAllocationEnd]
	jb	@@findblkloop
@@ret:

if ?POOLDBG
	@DbgOutS <"GetCurrEMSPageCount: free=">,1
	@DbgOutD eax,1
	@DbgOutS <", used=">,1
	@DbgOutD edi,1
	@DbgOutS <", used EMS=">,1
	@DbgOutD edx,1
	@DbgOutS <" pool =">,1
	@DbgOutD [PoolAllocationTable],1
	@DbgOutS <"-">,1
	@DbgOutD [PoolAllocationEnd],1
	@DbgOutS <")",10>,1
endif
	pop	ecx
	pop	esi
	ret
@@noems:
	xor	eax,eax
;	mov	edi,eax
	mov	edx,eax
	jmp @@ret

	@assumem esi, nothing

Pool_Get16KPageCount	ENDP


; locate any adjacent free XMS block to current pool allocation block
; if found, try consuming 32K of it for pool allocation block
; the adjacent XMS block must be at the end of current pool block,
;  since the pool block base cannot be changed once set
; pool block base+block size == owner XMS handle base+handle size (end match end)
; ends must match since you can't span noncontiguous sub-blocks of an owner XMS
;  block with a single EMS/VCPI pool allocation block
; INP: EDX -> Pool block to expand
; OUT: NC if success, C if fail
; other registers preserved
; do not push/pop segment registers here!

Pool_ExpandBlock	PROC

	pushad

	@assumem edx,LPPOOL_SYSTEM_INFO

if ?POOLDBG
	@DbgOutS <"Pool_ExpandBlock: edx=">,1
	@DbgOutD edx,1
	@DbgOutS <", addrK=">
	@DbgOutD [edx].psi_addressK
	@DbgOutS <", pArray=">
	@DbgOutD [edx].psi_descptr
	@DbgOutS <10>,1
endif

	cmp	[bNoPool],0	; no XMS handle table info available
	jne	@@locfail
	cmp	[edx].psi_16kmax,2*POOLBLOCK_ALLOCATION_SPACE
	jae	@@locfail				; current block is full
	test [edx].psi_flags,POOLBLOCK_FLAG_DONTEXPAND
	jne	@@locfail				; can't expand this block

	mov	edi,[edx].psi_descptr
	
	@assumem edi, LPXMS_ARRAY_STRUC
	
	mov	ebx,[edi].xas_addressK
	add	ebx,[edi].xas_sizeK	; ebx -> end of current pool block owner XMS

; see if owner XMS for EMS/VCPI allocation block end matches
;  end of pool allocation block

	movzx ecx,[edx].psi_startadj
	mov	eax,[edx].psi_addressK
	sub	eax,ecx					; true XMS start when pool allocation block created
	movzx ecx,[edx].psi_16kmax
	shl	ecx,4					; convert to K
	add	eax,ecx
	movzx ecx,[edx].psi_endadj
	add	eax,ecx					; true XMS end when block created
	cmp	eax,ebx
	jne	@@locfail				; owner XMS end no longer matches initial pool block owner XMS end

	movzx ecx, [XMS_Handle_Table.xht_numhandle]
	mov	esi, [XMS_Handle_Table.xht_array]
	or	ecx,ecx
	je	@@locfail
	test esi,esi
	je	@@locfail

	@assumem esi, LPXMS_ARRAY_STRUC

; esi -> test XMS block

	movzx	eax, [XMS_Handle_Table.xht_sizeof]
@@hanloop:
	cmp	ebx,[esi].xas_addressK		; see if test block immediately follows current block	
	je	@@found
	add	esi,eax		; move to next handle descriptor
	dec	ecx
	jne	@@hanloop
@@locfail:
	popad
	stc				; flag failure
	ret

@@found:
	test [esi].xas_flag,XMS_FREE	; if block is not free, abort scan
	je	@@locfail
	movzx eax,[edx].psi_endadj
	add	eax,[esi].xas_sizeK
	cmp	eax,32		; free block plus unused end overlap must be >=32K
	jb	@@locfail

; transfer 32K of following block to current block - unused end K in current
	mov	eax,32
	movzx	ecx,[edx].psi_endadj
	sub	eax,ecx					; adjust amount to change preceding block
	add	[esi].xas_addressK,eax	; move changed block address ahead
	sub	[esi].xas_sizeK,eax		; and adjust size
	mov	edi,[edx].psi_descptr
	add	[edi].xas_sizeK,eax		; increase EMS/VCPI associated XMS block size
	mov	[edx].psi_endadj,0		; no end overlap

	add	[edx].psi_16kmax,2		; adjust allocation tracking bytes
	add	[edx].psi_16kfree,2
	add	[edx].psi_4kfree,2*4
	movzx	eax,[edx].psi_16kmax
	shr	eax,1					; byte offset in allocation space (32K/byte)
	dec	eax						; relative 0
	mov	BYTE PTR [edx+eax+POOLBLOCK_SYSTEM_SPACE],0	; zero tracking allocation byte

; see if changed contiguous XMS block size went to <32K,
;  if so, transfer any remainder to pool block and zero XMS block

	mov	eax,[esi].xas_sizeK
	cmp	eax,31
	ja	@@loc2
	mov	[edx].psi_endadj,al

	xor	eax,eax
	mov	[esi].xas_addressK,eax
	mov	[esi].xas_sizeK,eax
	mov	[esi].xas_lockcount,al
	mov	[esi].xas_flag,XMS_INPOOL	; flag: free handle!

@@loc2:
	mov	[LastBlockFreed],edx	; expanding block size is same as freeing space
	popad
	clc					; flag success
	ret

	@assumem edi, nothing
	@assumem esi, nothing
	@assumem edx, nothing

Pool_ExpandBlock	ENDP


; expand any available allocation pool block by 32K, if possible
;  return edx -> expanded allocation pool block, zero if none
; destroy no other registers
; do not push/pop segment registers here!

Pool_ExpandAnyBlock	PROC
	push	esi
	cmp	[bNoPool],0	; no XMS handle table info available
	jne	@@fail

	mov	esi, [PoolAllocationTable]
	
	@assumem esi, LPPOOL_SYSTEM_INFO

@@findblkloop:
	cmp	[esi].psi_addressK,0	; unused/deallocated block
	je	@@nextblock
	cmp	[esi].psi_16kmax,2*POOLBLOCK_ALLOCATION_SPACE
	jae	@@nextblock				; current block is full
	mov	edx,esi
	call Pool_ExpandBlock
	jnc @@done
@@nextblock:
	add	esi,POOLBLOCK_TOTAL_SPACE
	cmp	esi, [PoolAllocationEnd]
	jb	@@findblkloop

@@fail:
	xor	edx,edx			; failure

@@done:
	pop	esi
	ret

	@assumem esi, nothing

Pool_ExpandAnyBlock	ENDP


; find and allocate free 4K (VCPI) block in pool blocks
; return edx == physical address, zero if none found
; destroy ecx,esi,edi
; do not push/pop segment registers here! this function is
; called by VCPI protected-mode API

Pool_Allocate4KPage	PROC
	push	eax

	@assumem esi, LPPOOL_SYSTEM_INFO

; first try last block allocated, to avoid searching full blocks if possible

	xor	edx,edx
	mov	esi, [LastBlockAllocator]
	or	esi,esi
	je	@@nolastalloc
	cmp	[esi].psi_addressK,edx
	je	@@nolastalloc
	cmp	[esi].psi_4kfree,dx
	jne	@@searchbytes

; try last freed chunk

@@nolastalloc:
	mov	esi, [LastBlockFreed]
	or	esi,esi
	je	@@nolastfreed
	cmp	[esi].psi_addressK,edx
	je	@@nolastfreed
	cmp	[esi].psi_4kfree,dx
	jne	@@searchbytes

@@nolastfreed:
	mov	esi, [PoolAllocationTable]

@@findblkloop:
	cmp	[esi].psi_addressK,edx	; unused/deallocated block
	je	@@nextblock
	cmp	[esi].psi_4kfree,dx
	jne	@@searchbytes
@@nextblock:
	add	esi,POOLBLOCK_TOTAL_SPACE
	cmp	esi, [PoolAllocationEnd]
	jb	@@findblkloop
    pop eax						;no free page found
    stc
    ret

@@searchbytes:
	movzx ecx,[esi].psi_16kmax
	shr	ecx,1			; count of allocation bytes in block
	xor	edi,edi
@@findbyteloop:
	mov	al,[esi+edi+POOLBLOCK_SYSTEM_SPACE]
	xor	al,-1			; unallocated 4K areas show as bit set
	jne	@@freebyte		; at least one unallocated area
	inc	edi
	loop @@findbyteloop
	@CheckBlockIntegrity; nothing free, although block indicated there was
	jmp	@@nextblock		; continue search

@@freebyte:
	mov	ecx,edi
	shl	ecx,15		; each byte covers 32K
	mov	edx,[esi].psi_addressK
	shl	edx,10		; convert from K to bytes
	add	edx,ecx		; compute base address of block addressed by byte

	mov	ah,1

; al holds bitcodes of allocations, set if available due to xor
@@bitloop:
	shr	al,1
	jc	@@found
	add	edx,4096
	shl	ah,1
	jmp	@@bitloop

@@found:
	or	[esi+edi+POOLBLOCK_SYSTEM_SPACE],ah	; flag that 4K area has been allocated
	dec	[esi].psi_4kfree
	mov	al,[esi+edi+POOLBLOCK_SYSTEM_SPACE]
	cmp	ah,0fh		; see if allocated from low nybble or high
	ja	@@highnyb

; low nybble
	and	al,0fh
	jmp	@@nybshared

@@highnyb:
	and	al,0f0h

; see if first allocation in that nybble
;  if so, then reduce free 16K region count since it was partially consumed

@@nybshared:
	mov	[LastBlockAllocator],esi	; update last block allocated from

	xor	al,ah			; turn off bit we turned on
	jne	@@finddone		; not the first bit allocated in 16K region (nybble)
	dec	[esi].psi_16kfree	;this counter's valid values are 0-96
	jns	@@finddone

	@CheckBlockIntegrity

@@finddone:
	pop	eax
    clc
	ret
	@assumem esi, nothing

Pool_Allocate4KPage	ENDP

; find and allocate free 16K (EMS) block in pool blocks
; inp: ebx -> EMS page descriptor entry
; out: NC if ok,
;     eax == offset PoolAllocationTable,
;     edx == nibble offset in descriptor
;      C on errors
; destroy no other registers

Pool_Allocate16KPage	PROC
	push	ecx
	push	esi
	push	edi

	@assumem esi, LPPOOL_SYSTEM_INFO

	xor	edx,edx

; first try last block allocated, to avoid searching full blocks if possible
	mov	esi, [LastBlockAllocator]
	or	esi,esi
	je	@@nolastalloc
	cmp	[esi].psi_addressK,edx
	je	@@nolastalloc
	cmp	[esi].psi_16kfree,dl
	jne	@@searchbytes

; try last freed chunk
@@nolastalloc:
	mov	esi, [LastBlockFreed]
	or	esi,esi
	je	@@nolastfreed
	cmp	[esi].psi_addressK,edx
	je	@@nolastfreed
	cmp	[esi].psi_16kfree,dl
	jne	@@searchbytes

@@nolastfreed:
	mov	esi, [PoolAllocationTable]

@@findblkloop:
	cmp	[esi].psi_addressK,edx	; unused/deallocated block
	je	@@nextblock
	cmp	[esi].psi_16kfree,dl
	jne	@@searchbytes
@@nextblock:
	add	esi,POOLBLOCK_TOTAL_SPACE
	cmp	esi, [PoolAllocationEnd]
	jb	@@findblkloop
    stc
	jmp	@@exit			;no 16k page free anymore

@@searchbytes:
	movzx ecx,[esi].psi_16kmax
	shr	ecx,1			; count of allocation bytes in block
	xor	edi,edi

@@findbyteloop:
	mov	al,[esi+edi+POOLBLOCK_SYSTEM_SPACE]
	xor	al,-1			; unallocated 4K areas show as bit set
	mov	ah,al
	and	al,0fh
	cmp	al,0fh
	je	@@lowfree		; low nybble unallocated, free 16K area

	and	ah,0f0h
	cmp	ah,0f0h
	je	@@highfree		; high nybble unallocated, free 16K area

; no free 16K area
	inc	edi
	loop @@findbyteloop
	@CheckBlockIntegrity
	jmp	@@nextblock

@@lowfree:
	or	BYTE PTR [esi+edi+POOLBLOCK_SYSTEM_SPACE],0fh
	mov	cl,0
	jmp	@@freeshared	; edx == 0

@@highfree:
	or	BYTE PTR [esi+edi+POOLBLOCK_SYSTEM_SPACE],0f0h
	mov	cl,1
	mov	edx,16		; nybble offset is four 4K pages, 16K

@@freeshared:
	mov	eax,edi
	shl	eax,15		; each byte covers 32K
	add	edx,[esi].psi_addressK	; add in base value
	shl	edx,10		; convert K to bytes
	add	edx,eax		; edx == 16k page memory address
	dec	[esi].psi_16kfree
	sub	[esi].psi_4kfree,4
	jnc	@@valid2
    
	@CheckBlockIntegrity	; force valid value

; update ebx pointer

@@valid2:
	mov	[LastBlockAllocator],esi	; update last block allocated from
	mov	eax,esi
	sub	eax, [PoolAllocationTable]
    mov edx,edi
    shl edx,1
    or dl,cl
    clc
@@exit:
	pop	edi
	pop	esi
	pop	ecx
	ret

	@assumem esi, nothing

Pool_Allocate16KPage	ENDP

; upon entry edx -> 4K page absolute address to free
; return carry clear on success, set on fail
; destroy ecx,esi,edi

Pool_Free4KPage	PROC
	push	eax
	push	ebx
	push	edx

	mov	ebx,edx
	shr	ebx,10			; convert bytes to K
	and	bl,NOT 3		; ensure 4K alignment

	@assumem esi, LPPOOL_SYSTEM_INFO

	mov	esi, [LastBlockFreed]
	or	esi,esi
	je	@@notlastfreed
	mov	eax,[esi].psi_addressK
	or	eax,eax
	je	@@notlastfreed	; unused/deallocated block

; ebx == start of 4K page in K after alignment adjustment
	cmp	ebx,eax
	jb	@@notlastfreed	; pool block starts after page
	movzx	ecx,[esi].psi_16kmax
	shl	ecx,4			; convert 16K to 1K
	add	ecx,eax			; ecx == end of range holding 4K pages
	cmp	ebx,ecx
	jb	@@rightblock	; after start, before end

@@notlastfreed:
	mov	esi, [LastBlockAllocator]
	or	esi,esi
	je	@@notlastalloc
	mov	eax,[esi].psi_addressK
	or	eax,eax
	je	@@notlastalloc	; unused/deallocated block

	cmp	ebx,eax
	jb	@@notlastalloc	; pool block starts after page
	movzx	ecx,[esi].psi_16kmax
	shl	ecx,4			; convert 16K to 1K
	add	ecx,eax			; ecx == end of range holding 4K pages
	cmp	ebx,ecx
	jb	@@rightblock	; after start, before end

@@notlastalloc:
	mov	esi, [PoolAllocationTable]

@@findblkloop:
	mov	eax,[esi].psi_addressK
	or	eax,eax
	je	@@nextblock		; unused/deallocated block

	cmp	ebx,eax
	jb	@@nextblock	; pool block starts after page
	movzx	ecx,[esi].psi_16kmax
	shl	ecx,4			; convert 16K to 1K
	add	ecx,eax			; ecx == end of range holding 4K pages
	cmp	ebx,ecx
	jb	@@rightblock	; page to free within pool block

@@nextblock:
	add	esi,POOLBLOCK_TOTAL_SPACE
	cmp	esi, [PoolAllocationEnd]
	jb	@@findblkloop

@@fail:
	@CheckBlockIntegrity
	stc
	jmp	@@ret

; this is the proper pool allocation block
@@rightblock:
	sub	ebx,eax		; 4K offset from block base in K
	mov	eax,ebx
	shr	eax,2			; K to 4K page
	mov	cl,al			; keep bit offset
	shr	eax,3			; 4K page to 32K byte offset
	and	ecx,7

	btr	dword ptr [esi+eax+POOLBLOCK_SYSTEM_SPACE],ecx	; see if bit set (was allocated)
	jnc @@fail			; no

	inc	[esi].psi_4kfree
	mov	[LastBlockFreed],esi

; check if this frees up a 16K chunk

	mov bl,0Fh
	mov	al,[esi+eax+POOLBLOCK_SYSTEM_SPACE]
    cmp cl,4
	jb	@@nothigh
    mov bl,0F0h
@@nothigh:
	test al,bl			; see if all bits of nybble cleared
	jne	@@success		; no
	inc	[esi].psi_16kfree
	call TryFreeToXMS	; free empty pool allocation block to XMS if appropriate
@@success:
	clc

@@ret:
	pop	edx
	pop	ebx
	pop	eax
	ret
	@assumem esi,nothing

Pool_Free4KPage	ENDP


; upon entry edx -> EMS page descriptor pointer
;  upon return set page descriptor pointer to -1 (unused)
; destroy no registers

Pool_Free16KPage	PROC
	push	esi
	push	eax
	push	ecx


	movzx esi,[edx].EMSPD.wPD	; [e]si == 64-byte block count
	cmp	si,-1
	je	@@fail			; bad pointer
	shl	esi,6			; convert 64-byte count to byte offset
	add	esi, [PoolAllocationTable]	; esi -> pool allocation block

	movzx ecx,byte ptr [edx].EMSPD.wNibOfs	; half-byte offset
    mov ah,0Fh
	shr	ecx,1			; byte offset
    jnc @@islow
    mov ah,0F0h
@@islow:
	mov	al,[esi+ecx+POOLBLOCK_SYSTEM_SPACE]
	and	al,ah 			; mask out bits which dont interest
    cmp al,ah
    jne @@fail
    rol al,4
	and	[esi+ecx+POOLBLOCK_SYSTEM_SPACE],al	; reset all expected bits

@@success:
	inc	[esi].POOL_SYSTEM_INFO.psi_16kfree
	add	[esi].POOL_SYSTEM_INFO.psi_4kfree,4
	mov	[LastBlockFreed],esi
	call TryFreeToXMS	; free empty pool allocation block to XMS if appropriate
	clc
@@ret:
	mov	DWORD PTR [edx].EMSPD.wNibOfs,-1
	pop	ecx
	pop	eax
	pop	esi
	ret

@@fail:
if ?POOLDBG
	@DbgOutS <"Pool_Free16kPage failed page=">,1
    mov ecx, edx
    sub ecx, [EMSPageAllocationStart]
    shr ecx, 2
    @DbgOutD ecx,1
	@DbgOutS <", masks=">,1
    @DbgOutD eax,1
    @DbgOutS <10>,1
endif    
	@CheckBlockIntegrity
	stc
	jmp	@@ret

Pool_Free16KPage	ENDP


; find an unused Pool block
; return NC and edx -> Pool block
; or C if none and no space available
; no other registers modified

GetUnusedPoolBlock	PROC
	mov	edx, [PoolAllocationTable]
@@findblkloop:
	cmp	[edx].POOL_SYSTEM_INFO.psi_addressK,0	; unused/deallocated block?
	je	@@found
	add	edx,POOLBLOCK_TOTAL_SPACE
	cmp	edx, [PoolAllocationEnd]
	jb	@@findblkloop
    stc
@@found:
	ret

GetUnusedPoolBlock	ENDP

; prepare pool block for use
; upon entry:
;  edx -> pool allocation block
;  ecx == raw size in K before alignment (max 1536+3)
;  edi == raw address in K before alignment
;  esi == owner XMS handle psi_descptr value, do NOT use XMS handle values for
;	size and address since this call may be part of a multi-pool block span
; destroys no registers

PreparePoolBlock	PROC

	pushad
	
	@assumem edx, LPPOOL_SYSTEM_INFO
	
	mov	[edx].psi_descptr,esi

	mov	ebx,edi			; raw address - must be aligned to page boundary
    add ebx,3
    and ebx,not 3
	mov	[edx].psi_addressK,ebx
    sub ebx, edi		; 00->00, 01->03, 02->02, 03->01
	mov	[edx].psi_startadj,bl

; block size = (raw size - start adjustment) rounded down to 32K boundary
;  since each allocation byte covers 32K

	mov	eax,ecx			; raw size
	sub	eax,ebx
	and	al,NOT 31
	shr	eax,4			; 16K count, known 16-bit value going to 8-bit

	cmp	ax,POOLBLOCK_ALLOCATION_SPACE*2
	jbe	@@setmax
	mov	ax,POOLBLOCK_ALLOCATION_SPACE*2

@@setmax:
	mov	[edx].psi_16kmax,al
	mov	[edx].psi_16kfree,al
	shl	eax,2			; 8-bit value potentially going to 16-bit
	mov	[edx].psi_4kfree,ax

; compute end adjustment, raw size - (block size + start adjustment)
	movzx ebx,[edx].psi_16kmax
	shl	ebx,4			; convert to K
	movzx eax,[edx].psi_startadj
	add	eax,ebx
	sub	ecx,eax			; ecx == raw size
	mov	[edx].psi_endadj,cl

; zero allocation entries
	xor	eax,eax
	lea	edi,[edx+POOLBLOCK_SYSTEM_SPACE]
	mov	ecx,POOLBLOCK_ALLOCATION_SPACE/4
	rep stos dword ptr [edi]
	@BIG_NOP

	popad
	ret
	@assumem edx, nothing

PreparePoolBlock	ENDP

; populate empty pool blocks with XMS owner info
; inp:
;  esi -> XMS (pseudo-)handle
;  ecx == size in kB XMS block 
;  edi == physical address (shifted 10 to right!)
; NOTE: ecx and edi may not match owner XMS size/address
; out: NC success
;       C if insufficient number of empty blocks to cover XMS handle range
; destroys eax,ebx,ecx,edx,edi

GetBlocksForXMSHandle	PROC

	mov	ebx,ecx
@@allocloop:
	call GetUnusedPoolBlock
	jc	@@exit		; no more blocks, remainder of XMS is effectively discarded

	mov	eax,edi		; compute size of candidate block/offset to new
    add eax,3
    and al,not 3
    sub eax,edi
	add	eax,1536	; 1.5M (in K) plus alignment adjustment size
	cmp	eax,ebx
	jbe	@@sizeok
	mov	eax,ebx
@@sizeok:
	mov	ecx,eax
	call PreparePoolBlock	; uses esi entry condition
	add	edi,eax			; update pool allocation block address
	sub	ebx,eax			; update size left to allocate
	cmp	ebx,32			; see if should remainder what's left
	jnb	@@allocloop
	mov	[edx].POOL_SYSTEM_INFO.psi_endadj,bl
	clc
@@exit:
	ret

GetBlocksForXMSHandle	ENDP

; walk XMS blocks, find largest XMS block which is sized 1.5M or smaller >=32K
;  after 4K alignment and allocate it for new EMS/VCPI pool allocation block
; if all XMS blocks >1.5M, then pick smallest and try to put remainder
;  into a free handle.
; If no free handle, allocate sufficient new EMS/VCPI pool blocks to cover full range.
; If not enough free EMS/VCPI blocks available, then remaining allocation is lost
;  until handle is freed.  This could only happen under very bad XMS fragmentation,
;  if at all.
; return carry clear if success, set if fail
; destroy no other registers
; do not push/pop segment registers here!

AllocateXMSForPool	PROC

	pushad

	cmp	[bNoPool],0	; check if pool sharing
	jne	@@allocfail

	mov	esi, [XMS_Handle_Table.xht_array]
	movzx ecx, [XMS_Handle_Table.xht_numhandle]

if ?POOLDBG
	@DbgOutS <"AllocateXMSForPool: XMS array=">,1
	@DbgOutD esi,1
	@DbgOutS <", handles=">,1
	@DbgOutW cx,1
	@DbgOutS <10>,1
endif

	or	esi,esi
	je	@@allocfail
	or	ecx,ecx
	je	@@allocfail

	@assumem esi, LPXMS_ARRAY_STRUC
	
	call GetUnusedPoolBlock	; test only, don't keep pointer
	jc	@@allocfail			; unable to make initial pool block allocation
	xor	edx,edx				; edx -> largest block <= 1.5M or smallest if none <=1.5M

@@hanloop:
	test [esi].xas_flag,XMS_FREE
	je	@@next				; yes, don't check
	mov	ebx,[esi].xas_addressK
	mov	eax,[esi].xas_sizeK
	and	ebx,ebx
	je	@@next				; FD Himem bug, can't check blank or zero-sized handle
	or	eax,eax
	je	@@next
	or	edx,edx
	je	@@newcandidate		; auto-match if first xms block available

; adjust for alignment loss (1-3K) in test

	and	bl,3
	mov	bh,4
	sub	bh,bl
	and	bh,3
	movzx ebx,bh
	sub	eax,ebx

	@assumem edx, LPXMS_ARRAY_STRUC

; adjust for alignment lost in candidate

	cmp	[edx].xas_sizeK,32	; ensure candidate size isn't so small that adjustment will underflow
	jb	@@next			; doesn't even meet minimum requirements
	mov	bl,byte ptr [edx].xas_addressK
	and	bl,3
	mov	bh,4
	sub	bh,bl
	and	bh,3
	movzx ebx,bh
	neg	ebx
	add	ebx,[edx].xas_sizeK

; eax holds test value size, ebx holds current candidate, both alignment adjusted
	cmp	eax,ebx
	je	@@next
	ja	@@larger

; test XMS block smaller than candidate block
	cmp	ebx,1536		; in K
	jbe	@@next			; current candidate closer to match size
	cmp	eax,32
	jb	@@next			; test too small
	jmp	@@newcandidate

; test XMS block larger than candidate block
@@larger:
	cmp	ebx,1536
	jae	@@next			; current candidate closer to match size
	cmp	eax,1536
	ja	@@next			; test too large

@@newcandidate:
	cmp	[esi].xas_sizeK,32
	jb	@@next			; candidate doesn't even meet unadjusted requirements
	mov	edx,esi			; new best candidate

@@next:
	movzx eax, [XMS_Handle_Table.xht_sizeof]
	add	esi,eax			; move to next handle descriptor
	dec	ecx
	jne	@@hanloop
	or	edx,edx
	je	@@allocfail

; candidate must be at least 32K, after 4K alignment adjustment
	mov	bl,BYTE PTR [edx].xas_addressK
	and	bl,3
	mov	bh,4
	sub	bh,bl
	and	bh,3
	movzx ebx,bh
	neg	ebx
	add	ebx,[edx].xas_sizeK
	cmp	ebx,32
	jb	@@allocfail				; candidate too small

if ?POOLDBG
	@DbgOutS <"AllocateXMSForPool: used XMS block handle=">,1
	@DbgOutD edx,1
	@DbgOutS <" addr=">,1
	@DbgOutD [edx].xas_addressK,1
	@DbgOutS <" siz=">,1
	@DbgOutD [edx].xas_sizeK,1
	@DbgOutS <10>,1
endif

	mov	[edx].xas_flag,XMS_USED	; flag candidate as used
	mov	[edx].xas_lockcount,1	; and locked

	mov	[XMSBlockSize],1536	; default allocation maximum size
	mov	eax, [XMSPoolBlockCount]
	cmp	eax,1
	jbe	@@trailadj			; use standard 1.5M size for first two blocks
	dec	eax					; should never overflow before we hit 4G total allocated

@@noadj:
	and	al,0fh				; but ensure that overflow doesn't happen anyway
	mov	cl,al				; shift the block size higher by factor of two
	shl	[XMSBlockSize],cl

; default is same as a MAX setting
;	cmp	[dwMaxMem16K],MAXMEM16K_DEFAULT
;	je	@@trailadj			; no MAX setting
; MAX setting, don't overallocate XMS we can't use

	push	edx
	call	Pool_Get4KPageCount	; current free in eax, current total in edx
	sub	edx,eax			; edx == current allocated (total - free)
	jc	@@adjusted		; shouldn't happen, continue without adjustment if it does

; if XMSBlockSize >= MAXMEM16K * 4 - allocated, then reduce XMSBlockSize

	mov	eax,[dwMaxMem16K]
	shl	eax,2			; convert to maximum 4K blocks
	sub	eax,edx
	jc	@@adjusted		; shouldn't happen, continue without adjustment
	shl	eax,2			; convert to 1K blocks

@@checksize:
	cmp	eax, [XMSBlockSize]
	jae	@@adjusted
	cmp	[XMSBlockSize],1536	; see if XMSBlockSize is at minimum default
	jbe	@@adjusted		; yes, can't reduce it any further
	shr	[XMSBlockSize],1	; reduce block size by one shift and try again
	jmp	@@checksize

@@adjusted:
	pop	edx

; allow up to 31K trailing bytes
@@trailadj:
	mov	eax, [XMSBlockSize]
	add	eax,31				; adjust for possible trail
	cmp	ebx,eax
	jbe	@@setblock			; no need to split XMS handle allocation

; search for a free XMS handle

	mov	  edi, [XMS_Handle_Table.xht_array]
	movzx ecx, [XMS_Handle_Table.xht_numhandle]
	movzx eax, [XMS_Handle_Table.xht_sizeof]
	@assumem edi, LPXMS_ARRAY_STRUC
@@freeloop:
	test [edi].xas_flag,XMS_INPOOL
	jnz @@gotfree
	cmp	[edi].xas_flag,XMS_USED	; some Himems dont set XMS_INPOOL, so
	je	@@nextfree				; check FREE items if address/size is NULL
	cmp	[edi].xas_addressK,0
	je	@@gotfree
	cmp	[edi].xas_sizeK,0
	je	@@gotfree
@@nextfree:
	add	edi,eax			; move to next handle descriptor
	loop @@freeloop

; no free handle found, try to allocate multiple blocks, discarding excess

	jmp @@setblock

@@gotfree:
	mov	cl,BYTE PTR [edx].xas_addressK	; compute size of candidate block/offset to new
	and	cl,3
	mov	ch,4
	sub	ch,cl
	and	ch,3
	movzx	ecx,ch

;	add	ecx,1536				; 1.5M (in K) plus alignment adjustment size
	add	ecx, [XMSBlockSize]	; maximum size (exceeded) plus alignment adjustment size

; edx -> candidate block being allocated, edi -> new block receiving remainder
; update candidate XMS block size
	mov	eax,[edx].xas_sizeK		; keep original size for updating new block
	mov	[edx].xas_sizeK,ecx

; update new XMS block info
	sub	eax,ecx					; new block size == old block original size - old block new size
	mov	[edi].xas_sizeK,eax
	mov	[edi].xas_flag,XMS_FREE	; explicitly flag free
	mov	[edi].xas_lockcount,0
	mov	eax,[edx].xas_addressK
	add	eax,ecx
	mov	[edi].xas_addressK,eax	; new block start == old block start + old block new size

if ?POOLDBG
	@DbgOutS <"AllocateXMSForPool: free XMS block handle=">,1
	@DbgOutD edi,1
	@DbgOutS <" addr=">,1
	@DbgOutD [edi].xas_addressK,1
	@DbgOutS <" siz=">,1
	@DbgOutD [edi].xas_sizeK,1
	@DbgOutS <10>,1
endif

; edx -> owner XMS handle for new pool allocation block(s)
; may be multiple blocks due to XMSBlockCount shifter

@@setblock:
	mov	esi,edx
	mov	ecx,[esi].xas_sizeK
	mov	edi,[esi].xas_addressK
	call GetBlocksForXMSHandle

	inc	[XMSPoolBlockCount]
	popad
	clc
	ret

@@allocfail:
	popad
	stc
	ret
	@assumem edx,nothing
	@assumem esi,nothing
	@assumem edi,nothing

AllocateXMSForPool	ENDP


; find count of available XMS 4K-aligned 4K pages in 32K chunks
;  return count in edx
;  destroys no other registers
;  do NOT push/pop segment registers here! this function
;  is called by VCPI protected mode API.

GetFreeXMS4KPageCount	PROC
	push esi
	push ecx
	xor	edx,edx
	cmp	[bNoPool], 0	; XMS memory pool?
	jne	@@countdone
	movzx ecx, [XMS_Handle_Table.xht_numhandle]
	jecxz @@countdone
	mov	esi, [XMS_Handle_Table.xht_array]
	or	esi,esi
	je	@@countdone
    
	push eax
	push ebx

	@assumem esi, <LPXMS_ARRAY_STRUC>

@@hanloop:
	test [esi].xas_flag,XMS_FREE
	jz	@@next
	xor	eax,eax
	cmp	eax,[esi].xas_addressK	; account for FD Himem bug
	je	@@next
	cmp	eax,[esi].xas_sizeK
	je	@@next

	mov	eax,[esi].xas_addressK
	mov	ebx,eax
	add	ebx,3		; round up
	add	eax,[esi].xas_sizeK
	and	al,0fch		; align to 4K boundary
	and	bl,0fch
	sub	eax,ebx		; compute size of block after alignments
	jbe	@@next
	and	al,NOT 1fh	; mask to 32K
	shr	eax,2		; convert 1K to 4K
	add	edx,eax		; update total count

@@next:
	movzx eax, [XMS_Handle_Table.xht_sizeof]
	add	esi,eax	; move to next handle descriptor
	dec	ecx
	jne	@@hanloop

	pop	ebx
	pop	eax
@@countdone:
	pop	ecx
	pop	esi
	ret

	@assumem esi, nothing

GetFreeXMS4KPageCount	ENDP


if ?FREEXMS

FreeXMSHandles proc

	pushad
	mov	esi, [PoolAllocationTable]	; esi -> pool allocation block
@@nextitem:
	cmp [esi].POOL_SYSTEM_INFO.psi_addressK,0
    jz @@skipitem
	movzx eax, [esi].POOL_SYSTEM_INFO.psi_16kmax
	mov [esi].POOL_SYSTEM_INFO.psi_16kfree,al
    shl eax, 2
	mov [esi].POOL_SYSTEM_INFO.psi_4kfree,ax
    push esi
	call TryFreeToXMS
    pop esi
@@skipitem:    
	add	esi,POOLBLOCK_TOTAL_SPACE
	cmp	esi, [PoolAllocationEnd]
    jb @@nextitem
    popad
    ret
    
FreeXMSHandles endp

endif

; upon entry esi -> pool allocation block to check if freeable to XMS
; perform the free if possible
; destroys eax,ecx,esi

TryFreeToXMS	PROC
	push	edx
	push	edi

	cmp	[bNoPool],0	; check if pool sharing
	jne	@@checkdone		; no

	@assumem esi, LPPOOL_SYSTEM_INFO
	
	test	[esi].psi_flags,POOLBLOCK_FLAG_DONTEXPAND
	jne	@@checkdone		; can't free this block

@@proccheck:
	mov	al,[esi].psi_16kfree
	cmp	al,[esi].psi_16kmax
	ja	@@bad			; free more than max, try to fix
	jne	@@checkdone		; free is less than maximum, used

	movzx eax,[esi].psi_4kfree
	shr	eax,2
	or	ah,ah
	jne	@@bad
	cmp	al,[esi].psi_16kmax
	ja	@@bad
	jne	@@checkdone		; free less than max

; walk all pool blocks, see if all blocks for XMS handle are empty or nonexistent
;  if so, then mark XMS handle as free
	mov	esi,[esi].psi_descptr
	mov	edx,[PoolAllocationTable]
	
	@assumem edx, LPPOOL_SYSTEM_INFO
	@assumem esi, LPXMS_ARRAY_STRUC

@@checkblkloop:
	cmp	[edx].psi_addressK,0
	je	@@checknext			; unused block

	cmp	esi,[edx].psi_descptr
	jne	@@checknext

	test	[edx].psi_flags,POOLBLOCK_FLAG_DONTEXPAND
	jne	@@checkdone		; can't free this block

; see if block empty
	movzx ecx,[edx].psi_16kmax
	cmp	cl,[edx].psi_16kfree
	jne	@@checkdone

	shl	ecx,2				; convert to 4K max
	cmp	cx,[edx].psi_4kfree
	jne	@@checkdone

@@checknext:
	add	edx, POOLBLOCK_TOTAL_SPACE
	cmp	edx, [PoolAllocationEnd]
	jb	@@checkblkloop

; checked all blocks as empty, go through them again and mark unused

	mov	edx, [PoolAllocationTable]

@@freeblkloop:
	cmp	[edx].psi_addressK,0
	je	@@freenext			; unused block

	cmp	esi,[edx].psi_descptr
	jne	@@freenext

; zero the block
	mov	ecx,POOLBLOCK_TOTAL_SPACE/4
	xor	eax,eax
	mov	edi,edx
	rep stos dword ptr [edi]
	@BIG_NOP

@@freenext:
	add	edx,POOLBLOCK_TOTAL_SPACE
	cmp	edx, [PoolAllocationEnd]
	jb	@@freeblkloop

	mov	[esi].xas_lockcount,0
	mov	[esi].xas_flag,XMS_FREE		; XMS block is free
	dec	[XMSPoolBlockCount]

	call	DefragXMS

@@checkdone:
	pop	edi
	pop	edx
	ret

@@bad:
	@CheckBlockIntegrity
;;	jmp	@@proccheck		;this would loop "forever"
	jmp	@@checkdone

	@assumem esi, nothing
	@assumem edx, nothing

TryFreeToXMS	ENDP

; try to defrag XMS blocks
; scan through the XMS handle array
; and try to merge FREE blocks
; destroys eax,ecx,edx,esi,edi

DefragXMS	PROC
	push	ebx
	mov	esi, [XMS_Handle_Table.xht_array]
	movzx ecx, [XMS_Handle_Table.xht_numhandle]

	@assumem esi, LPXMS_ARRAY_STRUC

	or	esi,esi
	je	@@nodefrag
	or	ecx,ecx
	je	@@nodefrag
	
	movzx	eax, [XMS_Handle_Table.xht_sizeof]
@@defragloop:
	cmp	[esi].xas_flag,XMS_FREE	; see if free
	jne	@@defragnext			; anything else is to ignore
	cmp	[esi].xas_addressK,0
	je	@@defragnext			; can't check blank handle

@@rescan:
	mov	ebx,[esi].xas_addressK
	add	ebx,[esi].xas_sizeK		; ebx -> end of test block

;--- now check array again if a successor exist which is also FREE

	mov	edi, [XMS_Handle_Table.xht_array]
	movzx edx, [XMS_Handle_Table.xht_numhandle]

	@assumem edi, LPXMS_ARRAY_STRUC

@@checkloop:
	cmp	[edi].xas_flag,XMS_FREE ; check if a FREE block exists 
	jne @@checknext
	cmp	ebx,[edi].xas_addressK	; which starts at EBX (end of test block)
	jne	@@checknext
	mov	edx,[edi].xas_sizeK
	add	[esi].xas_sizeK,edx	; update merger block size
	mov	[esi].xas_lockcount,0	; ensure not showing as locked, shouldn't be

	mov	[edi].xas_flag,XMS_INPOOL	; flag handle as free
	mov	[edi].xas_lockcount,0
	mov	[edi].xas_addressK,0
	mov	[edi].xas_sizeK,0
	jmp	@@rescan		;there might come just another free block to merge
@@checknext:
	add	edi,eax			; move to next handle descriptor
	dec	edx
	jne	@@checkloop

@@defragnext:
	add	esi,eax			; move to next handle descriptor
	loop @@defragloop
@@nodefrag:
	pop	ebx
	ret

	@assumem esi, nothing
	@assumem edi, nothing

DefragXMS	ENDP

; hook left for debugging, no current actions taken
;; upon entry esi -> pool allocation block to perform integrity check upon
;;	update with valid information if allocation counts mismatch
;; destroy no registers

if ?POOLDBG
CheckBlockIntegrity	PROC
	ret
CheckBlockIntegrity	ENDP
endif

	align 16

V86_RES equ $ - _start

V86_ENDRES proc	;declare a public label so the size is seen in .MAP
V86_ENDRES endp

;--- here start protected mode initialization code
;--- which is *not* copied to extended memory

;--- IO permission bitmap init values

if ?MASM
IOBM label byte
else
IOBM:
endif

; I/O-control of the DMA-port
; * trap ports 0..7, A, B, C, (D, E, F)
; * trap ports 81,82,83,87, 89,8A,8B
; * trap ports c0..cf
; * trap port d4 d6 d8 (da, dc, de)

if ?DMA or ?A20PORTS
 if ?DMA
  if ?MMASK
;----- FEDCBA9876543210
	DW 1111110011111111b	; DMA-Controller #1 (00-0F)
  else
;----- FEDCBA9876543210
	DW 0001110011111111b	; DMA-Controller #1 (00-0F)
  endif
 else
	DW 0					; ports 00-0F
 endif
	DW 0,0,0,0,0			; ports 10-5F
 if ?A20PORTS	 
;----- FEDCBA9876543210
	DW 0000000000010001b	; ports 60-6F
 else
	DW 0					; ports 60-6F
 endif	 
	DW 0					; ports 70-7F
 if ?DMA
;----- FEDCBA9876543210
	DW 0000111010001110b	; page register (80-8F)
 else
	DW 0					; ports 80-8F
 endif
 if ?A20PORTS
;----- FEDCBA9876543210
	DW 0000000000000100b	; ports 90-9F
 else
	DW 0					; ports 90-9F
 endif	 
	DW 0,0					; ports A0-BF
 if ?DMA
;----- FEDCBA9876543210
	DW 1111111111111111b	; DMA-Controller #2 (C0-CF)
  if ?MMASK
;----- FEDCBA9876543210
	DW 0101010101010000b	; DMA-Controller #2 (D8-DF)
  else
;----- FEDCBA9876543210
	DW 0000000101010000b	; DMA-Controller #2 (D8-DF)
  endif
 else
	DW 0,0					; ports C0-DF
 endif
endif

IOBM_COPY_LEN equ $ - IOBM


;--- check cpu type, return features if CPUID is supported

Is486 proc
	pushfd						; save EFlags
	xor	edx,edx
	push 240000h				; set AC+ID bits in eflags
	popfd
	pushfd
	pop eax
	popfd						; restore EFlags
	shr eax, 16
	test al,04					; AC bit set? then it is a 486+
	mov ah,0
	je @@no_486
	inc ah
	test al,20h 				; CPUID supported?
	jz @@nocpuid
	xor	eax,eax
	inc	eax						; get register 1
	@cpuid
	mov ah,1
@@nocpuid:
@@no_486:
	mov al,ah
	ret
Is486 endp

setstatustable proc    
	cmp		[EMSSTATUSTABLE],0
	jnz 	@@done
	MOV 	[EMSSTATUSTABLE],EDI; Here starts the status table of the
	MOV 	AX,EMSH_SYSTEM		; handles. Handle 0 is reserved
	STOS	WORD PTR [EDI]		; for the System, all others are free
	@BIG_NOP
	MOV 	AX,EMSH_FREE
	MOV 	ECX, MAX_HANDLES*4	; Entries still to fill for 255
	REP 	STOS WORD PTR [EDI]	; Handles (REP STOSW, addr. size =
	@BIG_NOP
@@done:    
	ret
setstatustable endp

;--- set int EAX to value of ECX

SetIDTEntry proc
	mov		word ptr [offset IDT+eax*8+0], cx
	shr		ecx,16
	mov		word ptr [offset IDT+eax*8+6], cx
    ret
SetIDTEntry endp

;--- set int 15h to be handled by monitor
;--- in: edx -> software int redirection bitmap in TSS
;--- in: ecx = int15 protected-mode entry

ActivateI15 proc	
	mov		al, 15h
	bts		[edx], eax
	jc		@@vectorset
    call	SetIDTEntry
@@vectorset:	
	ret
ActivateI15 endp	

;--- Monitor initialization

;--- input expected:
;--- CPU in protected-mode, interrupts disabled, paging disabled
;--- GDTR = GDT
;--- CS = FLAT selector (FLAT_CODE_SEL)
;--- SS = DGROUP
;--- DS = RSEG
;--- ES = _TEXT32 (=conv. memory segment where this code has been loaded)
;--- BX = offset BRPTAB
;--- DI = offset JEMMINIT, contains:
;---  MonitorStart		: start of EMB for the monitor
;---  MonitorEnd		: end of EMB for monitor
;---  MaxMem16k 		: MAX=xxxx cmdline option given, or 1E00 (120 MB)
;---  TotalMemory		: total size of physical memory
;---  XMS_Handle_Table	: returned by int 2f, ax=4309h
;---  MaxEMSPages		: maximum EMS 16k pages, default 32 MB
;---  XMS_Control_Handle: XMS handle for monitor extended memory block
;---  DMA_Buffer_Size	: size of DMA buffer in kB
;---  Frame 			: FRAME=xxxx parameter
;---  NoEMS, NoFrame, NoPool, AltBoot, NoVME, NoVDS, NoPGE, NoA20, NoVCPI

;--- memory layout installed by the initialization code:
;--- shared memory (110000 - 11xxxx):
;---  0-3.0 kB	  wasted space because of rounding to page border
;---	0.5 kB	  monitor stack (size V86_TOS)
;---	0.5 kB	  GDT  [GDT_PTR] (size GDT_SIZE)
;---	2.0 kB	  IDT  [IDT_PTR]
;---  ~13.5 kB	  resident monitor data+code (size V86_RES)
;---   ~1.7 kB	  interrupt table (size E0h*7)
;---  0-3.0 kB	  wasted space because of rounding to page border
;---			  (if > 2 kB it is filled with STATUSTABLE)

;--- linear addresses in sys space (000400000 - 0007FFFFF):

;---    4   kB    reserved
;---   12   kB    page directory, page table 0, sys page table
;---              DMABuffStart (0-?DMABUFFMAX kB)
;---   ~2.2 kB    TSS (size TSS_LEN)
;---              para align
;---    2   kB    EMSSTATUSTABLE (size 8*256)
;---    2   kB    EMSNAMETABLE (size 8*256)
;---              EMSPageAllocationStart (EMS pages * 4 bytes) 
;---              EMSPAGETABLE (EMS pages * 1 bytes)
;---              64 byte align
;---              PoolAllocationTable (128*64 + x)
;---  0-3.0 kB    page align
;---              start of "free" memory in this very first XMS block
;---              (used for UMBs, if NODYN used for EMS/VCPI as well)

InitMonitor PROC

;--- local variables:

jemmini 	equ <[ebp-4]>
rsofsinit	equ <[ebp-8]>	 
tmpPageMask equ <[ebp-11]>	  
tmpIs486	equ <[ebp-12]>	  
tmpFeatures	equ <[ebp-16]>	  
dwSizeV86	equ <[ebp-20]>	  

	CLD
	pop		esi					; get return EIP
	pop		edx					; get return CS (real-mode)
	mov		eax,esp
	mov		ecx,ss				; SS contains DGROUP
	movzx	ecx,cx
	mov		SP,FLAT_DATA_SEL
	mov		SS,ESP
	mov		SP,ES
    movzx	ESP,SP
    shl		ESP, 4
    ADD		ESP, V86_TOS

;--- now SS:ESP is monitor stack in conv. memory image
	
	push	gs			; +32 real-mode GS
	push	fs			; +28 real-mode FS
	push	ds			; +24 real-mode DS
	push	es			; +20 real-mode ES
	PUSH	ecx			; +16 SS (=DGROUP)
	PUSH	eax			; +12 ESP
	PUSH	00023002H	;  +8 EFL (VM=1, IOPL=3, IF=0)
	PUSH	edx			;  +4 CS
	PUSH	esi			;  +0 EIP
	mov		ebp,esp		; ebp -> IRETDV86

	XOR 	EAX,EAX
	LLDT	AX					; initialise LDT (it is not used)
	mov		FS,EAX
	mov		GS,EAX
	MOV 	AL,FLAT_DATA_SEL	; Addressing everything
	MOV 	DS,EAX
	MOV 	ES,EAX

;--- segment registers now:
;--- CS,SS,DS,ES: FLAT  
;--- FS,GS: NULL  
;--- no access to global variables possible until paging is enabled
;--- and memory block has been moved to extended memory!

if ?INITDBG
	@DbgOutS <"Welcome in PM",10>,1
	@WaitKey 1,0
endif	 

	shl		ecx, 4
	movzx	edi, di
	add		edi, ecx			;get linear address of jemmini
	push	edi					;== jemmini [ebp-4]


	movzx	ebx, bx
	add		ebx, ecx
	push	ebx					;== rsofsinit [ebp-8]

	pushfd							; clear the NT flag to avoid a "reboot"
	and		byte ptr [esp+1],not 40h; at next IRET in protected-mode
	popfd
	
	mov		esi, jemmini

	call	Is486
	push	eax					;== tmpPageMask [ebp-12]
	push	edx					;== tmpFeatures [ebp-16]
	and		al,al
	jz		@@is486
	cmp		[esi].JEMMINIT.jiNoInvlPg,-1
	jnz		@@is486
	mov		[esi].JEMMINIT.jiNoInvlPg,0
@@is486:

	MOV 	EDI,[esi].JEMMINIT.jiMonitorStart ; Start of Monitor-Code

	ADD 	EDI,4095		; Round to the next page border
	AND 	DI,NOT 4095		; may loose up to 3 kB	

;-- calc size of the items which must be shared:
;-- GDT, IDT, stack and code.

;-- ?INTTABSIZ is size of INT_TABLE

	mov		eax, V86_RES + ?INTTABSIZ	;size of resident code+data
	add		eax, 1000h-1
	and		ax, not 1000h-1	;round to next page (may waste another 3 kB!)

	mov 	ebx, edi		;EBX = physical address of shared items
	add		edi, eax
   
	push	eax				;== dwSizeV86 [ebp-20], size of code+GDT+IDT+stk

if 0
	mov		eax, cr3
    and		eax, 0FFFh		;don't touch low 12 bits of CR3
    or		eax, edi
	MOV 	CR3, eax		;set CR3 (paging is still disabled)
else
	mov		cr3, edi
endif

;-- clear pagedir, page table 0 + system page table

	push	edi
	mov		ecx, 3000h/4
	xor		eax, eax
	rep		stos dword ptr [edi]
	@BIG_NOP
	pop		edi

;-- init page dir (2 PDEs)

	lea		edx, [edi+1000h]
	MOV 	[EDI+000h],EDX
	OR		DWORD PTR [EDI+000h],111B
	add		edx, 1000h
	mov		[EDI+?SYSPDE], EDX
	OR		DWORD PTR [EDI+?SYSPDE],111B
	
	add		edi, 1000h	;edi -> mapped page table 0
	
	push	edx 		;save mapped system page table

;-- init page table 0 address space 0-110000h

	mov		edx, 0+7	;set page flags u/ser, w/riteable, p/resent
if ?PGE
	cmp		[esi].JEMMINIT.jiNoPGE,0
	jnz		@@nopge
	test	byte ptr tmpFeatures+1,20h	;PGE supported?
	jz		@@nopge
	@mov_eax_cr4
	or al,80h	
	@mov_cr4_eax
	or		dh,1		;set G bit (page global)
@@nopge:	
	mov		tmpPageMask,dh
endif	 
	mov		cx,110h		;hiword ecx is cleared
@@FILL_PAGETAB0:	
	MOV 	[EDI],EDX
	ADD 	EDX,1000h
	ADD 	EDI,4
	loop	@@FILL_PAGETAB0
	
if 0
;-- give the video region A000-BFFF some special attributes
	push edi
	sub  edi, (110h*4 - 0A0h*4)
	mov		cl,20h
@@FILL_VPAGE:
	or	dword ptr [edi],8h	;set "WT"
	add edi,4
	loop @@FILL_VPAGE
	pop edi	
endif

;-- init page table 0 address space 110000h-? (shared space)
	
	mov		ecx, dwSizeV86	;size of shared space above 110000h (page aligned)
	shr		ecx, 12		;is just a handful of pages
	mov		edx, ebx	;get physical address of this space
	or		dl, 111b
if ?PGE
;--- set first page (which is shared) global
	or		dh, tmpPageMask		
endif
	
;	or		dl, 101b	;set R/O
@@FILL_PAGETAB0X:	 
	MOV 	[EDI],EDX
	ADD 	EDX,1000h
if 1	
	and		dl, not 2	;all pages except the first r/o
endif	 
if ?PGE
	and		dh, 0F0h
endif
	ADD 	EDI,4
	loop	@@FILL_PAGETAB0X

;-- for VCPI intruders, set remaining PTEs in page tab 0
;-- if they still aren't satisfied, they may be just stupid.

	push	edx
if 1	;SBEINIT.COM *is* stupid, needs linear=physical mapping
	movzx	edx, di
	and		dh, 0Fh
	shl		edx, 10		;transform offset in page table -> linear address
	or		dl,7
endif
	mov		cx, 400h	;hiword ecx is clear
@@FILL_PAGETAB0XX:	 
	MOV 	[EDI],EDX
	ADD 	EDX,1000h
	ADD 	EDI,4
	test	di,0FFFh
	loopnz	@@FILL_PAGETAB0XX
	pop		edx

;-- init system page table with the rest we need

	pop		edi			;get saved mapped system page table
	add		edi, ?SYSPTE

;-- set PTEs in system page table

;-- what is the maximum which is needed?
;-- 3			  for page tables
;-- dmasize/4	  for dma buffer 
;-- 15			  max wasted because of dma buffer alignment
;-- 3			  for tss
;-- 1			  for ems handles + names
;-- maxems*5/4096 for ems management
;-- maxmem16k * 16 (kb needed) ->
;-- 		  / 1536 (blocks needed) ->
;			  * 64 (bytes needed)
;-- -> maxmem16k * 1024 / 1536 -> bytes
;-- -> maxmem16k * 2 / 3 -> bytes
;-- -> maxmem16k / 1536 -> kb
;-- -> maxmem16k / 6144 -> pages

if 0
	mov 	ecx, [esi].JEMMINIT.jiMonitorEnd]
	sub 	ecx, [esi].JEMMINIT.jiMonitorStart]
	sub		ecx, dwSizeV86		;subtract the low space
	shr		ecx, 12				;size -> PTEs
	inc		ecx
	movzx	ecx, cx
else
	push	edx
	mov 	eax, [esi].JEMMINIT.jiMaxMem16k
	cdq
	mov		ecx, 6144
	div		ecx
	pop		edx
	add		eax, 1+3+15+3 +2	   ;add 2 for rounding issues
	movzx	ecx, [esi].JEMMINIT.jiMaxEMSPages
	lea		ecx, [ecx+ecx*4]
	shr		ecx, 12
	add		eax, ecx
	movzx	ecx, [esi].JEMMINIT.jiDMABufferSize
	shr		ecx, 2
	add		ecx, eax
endif

@@FILL_PAGETABF:
	MOV 	[EDI],EDX
	ADD 	EDX,1000h
	ADD 	EDI,4
;	 test	 DI, 0FFFh			;make sure not to write beyond the page table
	loopnz	@@FILL_PAGETABF

;	@DbgOutS <"page tables initialized",10>,?INITDBG

;--- page dir, page tab 000 and sys page tab are now initialized

;--- switch the context now!

	MOV 	EAX,CR0
	OR		EAX,80000000H		; enable Paging
	MOV 	CR0,EAX
if ?INITDBG
	@DbgOutS <"Paging is enabled",10>,1
	@WaitKey 1,0
endif	 

;--- ok, paging enabled, now move monitor code+data in extended memory

	mov		edi, ?BASE
	movzx	esi, word ptr [ebp].IRETDV86.vES	;real-mode ES contained _TEXT32
	shl		esi, 4
	
if 0
;--- resolve base fixups
;--- this is usually not needed, since the binary has been linked
;--- for base address 110000h

	push	esi
	add		esi, V86_TOTAL	;the relocs are just behind the 32bit block
	xor		edx, edx
	xchg	dx,[esi]		;get size of header (hack!)
nextpage:
	mov		ebx, [esi+0]
	mov		ecx, [esi+4]
	and		ecx, ecx
	jz		reloc_done
if ?INITDBG
	@DbgOutS <"rlcs at ">
	@DbgOutD esi,1
	@DbgOutS <" for page ">,1
	@DbgOutD ebx,1
	@DbgOutS <" size=">,1
	@DbgOutD ecx,1
	@DbgOutS <" edx=">,1
	@DbgOutD edx,1
	@DbgOutS <10>,1
	@WaitKey 1,0
endif	 
	add		ecx, esi
	add		esi, 8
	sub		ebx, edx		;subtract size of header from RVA
    add		ebx, [esp]		;add conv. base to address
	xor		eax, eax
nextreloc:
	lodsw	
	test	ah,0F0h
	jz		ignreloc
	and		ah,0Fh
	add		[eax+ebx], edi
ignreloc: 
	cmp 	esi, ecx
	jb		nextreloc
	jmp		nextpage
reloc_done:
	pop		esi					; restore source base
if ?INITDBG
	@DbgOutS <"base relocs done",10>,1
	@WaitKey 1,0
endif	 
endif

;-- copy all to extended memory (linear address 110000h)

	MOV 	ECX, V86_RES shr 2	; Length of resident part (para aligned)
	rep		movs dword ptr [edi], [esi]
	@BIG_NOP

if ?INITDBG
	@DbgOutS <"_TEXT32 moved to ext mem",10>,1
	@WaitKey 1,0
endif	 

;--- after code + data has been moved to extended memory
;--- access to global variables is possible

;--- load final values for GDTR + IDTR

	LGDT	FWORD PTR [GDT_PTR]
	LIDT	FWORD PTR [IDT_PTR]
if ?INITDBG
	@DbgOutS <"GDTR + IDTR set",10>,1
	@WaitKey 1,0
endif	 

;--- switch to the stack in extended memory

	mov		ebp, ?BASE + V86_TOS - size IRETDV86
	lea		esp, [ebp-20]	;take care for the local variables

if ?INITDBG
	@DbgOutS <"ESP reset",10>,1
	@WaitKey 1,0
endif	 

;-- create INT_TABLE + set IDT

	mov		ebx, 0EE00h shl 16 + FLAT_CODE_SEL
	mov		ecx, 100h
    mov		esi, offset IDT
if ?FASTMON
	mov		eax, offset int00
@@nextfastidt:
	mov		edx, eax
    shr		edx, 16
	mov		[esi+0],ax
	mov		[esi+2],ebx
	mov		[esi+6],dx
	add		eax, 4
	add		esi, 8
	dec		ecx
    cmp		cl,100h - ?FASTENTRIES
    jnz		@@nextfastidt
    mov		edx, 0E9006Ah
    mov		dh,?FASTENTRIES
else
    mov		edx, 0E9006Ah		;push byte 00h, jmp xxxx
endif
@@nextidtentry:
	mov		eax, edi
	mov		[esi+0],ax
	mov		[esi+2],ebx
    shr		eax, 16
	mov		[esi+6],ax
    mov		[edi+0],edx
    mov		eax, offset V86_Monitor
	add		edi,7		;7 bytes for each entry in INT_TABLE!
    sub		eax, edi
    mov		[edi-4],eax
    inc		dh			;next INT #
	add		esi,8
	loop	@@nextidtentry

if ?INITDBG
	@DbgOutS <"IDT + int table initialized",10>,1
	@WaitKey 1,0
endif	 


;--- edi para aligned again (increased by 700h/690h)
;--- EDI -> free linear memory

	MOV		EAX,CR3
	MOV		[_CR3],eax

	mov		eax, tmpFeatures
	mov		[dwFeatures], eax
	mov		eax, tmpIs486
	mov 	[bIs486], al
if ?PGE    
	mov		[bPageMask],ah
endif

	mov		esi, jemmini
	mov		eax, [esi].JEMMINIT.jiTotalMemory
	mov		[dwTotalMemory],eax
	mov		eax, [esi].JEMMINIT.jiMaxMem16k
	mov		[dwMaxMem16K],eax
	movzx	eax, [esi].JEMMINIT.jiMaxEMSPages
	mov		[dwMaxEMSPages],eax
	mov		al, [esi].JEMMINIT.jiNoEMS
	mov		[bNoEMS],al
	mov		al, [esi].JEMMINIT.jiNoFrame
	mov		[bNoFrame],al
    and		al,al
    jnz		@@noframe
    mov 	[bEmsPhysPages], 4
	mov		ax, [esi].JEMMINIT.jiFrame
    push	edi
    mov		edi, offset PAGE2SEGM	;set the first 4 contiguous pages as frame
    mov		al,ah
    add		ah,4
    stosw
    add		ax,0808h
    stosw
    pop		edi
@@noframe:
	mov		al, [esi].JEMMINIT.jiNoPool
	mov		[bNoPool],al
if ?A20PORTS or ?A20XMS
	mov		al, [esi].JEMMINIT.jiNoA20
	mov		[bNoA20],al
endif
if ?VCPI
	mov		al, [esi].JEMMINIT.jiNoVCPI
	mov		[bNoVCPI],al
endif
	mov		al, [esi].JEMMINIT.jiNoInvlPg
	mov		[bNoInvlPg],al
	mov		al, [esi].JEMMINIT.jiV86Flags
	mov		[bV86Flags],al
    mov		ax, [esi].JEMMINIT.jiXMSControlHandle
    mov		[XMSControlHandle],ax

if ?FASTBOOT
	test	[bV86Flags],V86F_FASTBOOT
    jz		@@nofastboot
	mov	 [pSavedVecs],edi 
    xor  esi,esi
    mov  ecx,20h
    rep  movsd		;save vecs 00-1F
    add  esi,20h*4	;skip 20-3F
    mov  cl,20h
    rep  movsd		;save vecs 40-5F
    add  esi,8*4	;skip 60-67
    mov  cl,10h
    rep  movsd		;save vecs 68-77h
if ?RBTDBG
	mov  esi,[pSavedVecs]
    mov  ecx,20h
    xor ebx,ebx
@@: 
    lodsd
    @DbgOutB bl,1
    @DbgOutS <": ">,1
    @DbgOutD eax, 1
    @DbgOutS <"        ">,1
    inc ebx
    loop @B
    @DbgOutS <10>,1
endif
@@nofastboot:    
endif

	MOVZX	ECX,word ptr [ebp].IRETDV86.vDS
	MOV 	[dwRSeg], ecx
	SHL 	ECX,4
	mov		[dwRes],ecx

	mov		esi, rsofsinit

	movzx	eax,[esi].RSOFS.wBpTab
    mov		[bBpTab],al
    add		[bBpBack],al
	add		eax,ecx
	mov		[bpstart],eax

if ?DMA    
	movzx	eax,[esi].RSOFS.wRFlags
	add		eax,ecx
	mov		[dwRFlags],eax
endif
if ?INITDBG
	@DbgOutS <"variables copied",10>,1
	@WaitKey 1,0
endif	 

;--- test if 2 kB of the wasted space could be used for EMS

	mov  cx, di
	and  cx, 0FFFh
	jz	 @@nothingtoregain
	cmp  cx, 800h
	ja	 @@nothingtoregain
	call setstatustable
@@nothingtoregain:


;--- the memory in page tab 0 is initialized

;--- until now are consumed:
;--- 3 pages for pagedir+pagetab
;--- 4-5 pages for monitor stack, GDT, IDT, data, code

if ?INITDBG
	@DbgOutS <"GDT, IDT, stack and code in ext mem initialised",10>,1
	@WaitKey 1,0
endif	 

	mov 	eax, ?SYSLINEAR
;	MOV 	[PAGEDIR],eax
	lea		edi, [eax + 3000h]	; skip page dir + 2 page tables

if ?DMA

	MOV 	[DMABuffStart], EDI
	mov		eax, jemmini
	movzx	eax, [eax].JEMMINIT.jiDMABufferSize	;buffer size in kB
	shl		eax, 10
	mov		ecx, eax
	shr		ecx, 10
	inc		ecx		;add start and end bit
	inc		ecx
	xor		edx, edx
@@nextdmabuffbit:	 
	bts		[DMABuffFree], edx
	inc		edx
	loop	@@nextdmabuffbit

;-- get the physical address of current EDI in ECX

	mov 	ecx, edi
	sub		ecx, ?SYSLINEAR
	add		ecx, dwSizeV86
	mov		edx, jemmini
	add		ecx, [edx].JEMMINIT.jiMonitorStart
	cmp		ecx, 1000000h-20000h
	jc		@@dmaaddrok
	xor		eax, eax
@@dmaaddrok:
	mov		[DMABuffSize], eax
	and		eax, eax
	jz		@@buffergood
	
	lea		edx, [ecx+eax-1];edx -> last byte of buffer
	mov		eax, edx
	shr		eax, 16
	mov		esi, ecx
	shr		esi, 16
	cmp		ax, si	;does buffer cross a 64 kB boundary?
	jz		@@buffergood

if ?INITDBG
	@DbgOutS <"linear address DMA buffer=">,1
	@DbgOutD ?DMALINEAR,1
	@DbgOutS <10>,1
	@WaitKey 1,0
endif	 
	
	mov		[DMABuffStart],?DMALINEAR

;-- align it to the next 64 kB boundary
	
	inc		edx
	xor		dx, dx
	sub		edx, ecx	; some pages left till next physical 64 kb border
	add		ecx, edx

	pushad				; rearrange PTEs
	shr		edx, 12		; bytes -> pages
	mov		eax, edi
	sub		eax, ?SYSLINEAR
	shr		eax, 12
	@GETPTEPTR edi, eax*4+2000h+?SYSPTE	;edi -> curr. PTEs for DMA buffer
	lea		esi, [edi+edx*4]	;esi -> PTEs to be used for DMA buffer
	mov 	ebx, esi
	mov		ecx, [DMABuffSize]	;will be page aligned
	jecxz	@@NoDMABuff
	shr		ecx, 12
;--- now move the PTEs to the place for the
;--- DMA buffer and clear them here!
	@GETPTEPTR edi,	2000h+?DMAPTE, 1
@@nextpage1:
	lods	dword ptr [esi]
	and 	eax, eax			;shouldnt happen
	jz		@@ptedone
	stos	dword ptr [edi]
	mov 	dword ptr [esi-4],0 
	loop	@@nextpage1

;-- PTEs for DMA buff are moved out, there might be a rest of PTEs 
;-- which has to be moved now "upwards", num PTEs in EDX.
;-- as a result, there is a hole in address space of size "DMABuffSize",
;-- but this is required for linear to phys translation!

	mov		ecx, edx
	jecxz	@@ptedone
	mov		edx, 4
	mov		edi, esi			;this is the destination
	mov		esi, ebx
@@nextpage2:	
	sub		esi, edx
	sub		edi, edx
	xor		eax, eax
	xchg	eax, [esi]
	mov		[edi], eax
	loop	@@nextpage2
@@ptedone:
	mov		eax, cr3			;flush TLB
	mov		cr3, eax
if ?INITDBG
	mov		edi, ?DMALINEAR
	mov		ecx, [DMABuffSize]
	shr		ecx, 2
	mov		eax, "BAMD"
	rep		stos dword ptr [edi]
endif
@@NoDMABuff:
	popad
@@buffergood:
	add		edi, [DMABuffSize]
	MOV		[DMABuffStartPhys], ecx

endif

if ?INITDBG
	@DbgOutS <"DMA done",10>,1
	@WaitKey 1,0
endif	 
	

;--- now create the TSS (will begin on a page boundary, since DMA buffer
;--- size is rounded to 4 kB).

;	mov		[dwTss], edi
	mov		ebx, offset GDT
	mov 	eax, edi
	MOV 	WORD PTR [EBX+V86_TSS_SEL+2],AX
	SHR 	EAX,16
	MOV 	BYTE PTR [EBX+V86_TSS_SEL+4],AL
	MOV 	BYTE PTR [EBX+V86_TSS_SEL+7],AH

;--- init TSS and the software interrupt redirection bitmap (256 bits)
;--- it is known by Pentium+ cpus only, but doesn't hurt for previous cpus

	mov		edx, edi	
	mov		ecx, size TSSSEG/4 + 32/4
	xor		eax, eax
	rep		stos dword ptr [edi]	;clear TSS
	@BIG_NOP
	mov		dword ptr [edx].TSSSEG.tsEsp0, ?BASE + V86_TOS
	mov		dword ptr [edx].TSSSEG.tsSS0, FLAT_DATA_SEL
if 1	
	mov		eax, [_CR3] 		; save value for CR3 in TSS (not needed)
	mov 	[edx].TSSSEG.tsCR3, eax
endif
	mov		[edx].TSSSEG.tsOfs,size TSSSEG+32 ;let 32 bytes space below IO Bitmap

	lea		edx, [edx+size TSSSEG]

;--- int 67h must be handled by the monitor in any case

	xor		eax, eax
	mov		al, 67h
    mov		ecx, offset Int67_Entry
	bts		[edx], eax
    call	SetIDTEntry

if ?BPOPC ne 0F4h
	mov		al,6
	mov		ecx, offset Int06_Entry
;   bts		[edx], eax			;not required since it is an exception
	call	SetIDTEntry
endif
    
	mov		esi,jemmini
if ?ALTBOOT
	cmp		[esi].JEMMINIT.jiAltBoot,0
	jz		@@noaltboot
	mov		ecx, offset Int15_Entry
	call	ActivateI15
@@noaltboot:	
endif
if ?A20PORTS
	mov		ecx, offset Int15_Entry_Ex
	call	ActivateI15
endif
if ?VME
	mov		al,[esi].JEMMINIT.jiNoVME
    xor		al,1
    call	SetVME
endif

;-- init the io permission bitmap

	movzx esi, word ptr [ebp].IRETDV86.vES
	shl	esi, 4
	add	esi, offset IOBM - offset _start
	mov edx, edi
	mov	ecx,IOBM_COPY_LEN
	rep movs byte ptr [edi], [esi]
	@BIG_NOP
	mov ecx,2000h -	IOBM_COPY_LEN	;2000h == 65536/8
	mov al,0
	REP STOS BYTE PTR [EDI]
	@BIG_NOP
	
	mov al, 0FFh
	stos BYTE PTR [EDI]		;the IO bitmap must be terminated by a FF byte

if ?INITDBG
	@DbgOutS <"TSS done",10>,1
	@WaitKey 1,0
endif	 

;-- finally load TR

	mov		ax, V86_TSS_SEL
	ltr 	ax
	
if ?A20PORTS
 if 0
	xor		eax, eax
	test	byte ptr wBiosA20,1	;keyboard controller can switch A20?
	jnz		@@kbdA20
	mov		al, 60h
	btr		[edx], eax
	mov		al, 64h
	btr		[edx], eax
@@kbdA20:
	test	byte ptr wBiosA20,2	;port 92h can switch A20?
	jnz		@@ps2A20
	mov		al, 92h
	btr		[edx], eax
@@ps2A20:
 endif
endif

;--- here CR3, GDTR, IDTR, TR and LDTR all have their final values

	ADD 	EDI,15			; round to para
	AND 	EDI,NOT 15

if ?INITDBG
	@DbgOutS <"Jemm386 initialised, edi=">,1
	@DbgOutD edi,1
	@DbgOutS <10>,1
	@WaitKey 1,0
endif	 

;---  is XMS pooling on? then direct access to XMS handle table required?

	cmp		[bNoPool],0
	jne		@@noxmsarray
	mov		ecx, jemmini
	mov		ecx, [ecx].JEMMINIT.jiXMSHandleTable
	movzx	eax, cx
	shr		ecx, 12
	and		cl, 0F0h
	add 	ecx, eax
; transfer XMS table info to fixed memory location, assume two dwords
	mov		eax,[ecx+0]
	mov		dword ptr [XMS_Handle_Table],eax
	movzx edx,word ptr [ecx+4]
	movzx eax,word ptr [ecx+6]
	shl	eax,4
	add	eax,edx
	cmp eax, 100000h	;is handle array in HMA?
	jb @@nohmaadjust
	@GETPTEPTR edx, 2000h+?HMAPTE, 1	;make a copy of the HMA PTEs

if ?INITDBG    
	@DbgOutS <"copy of HMA PTEs at ">, 1
	@DbgOutD edx, 1
	@DbgOutS <10>, 1
endif

	mov  ecx,10h
	push eax
	mov eax, 100000h + 7
@@loophmacopy:
	mov  [edx], eax
	add  edx, 4
	add  eax, 1000h
	loop @@loophmacopy
	pop eax
	mov edx, ?HMALINEAR
	lea eax, [eax + edx - 100000h]
@@nohmaadjust:
	mov	[XMS_Handle_Table.xht_array],eax
@@noxmsarray:

;--- EMS/VCPI init

	cmp		[bNoEMS],0
	jz		@@emsactive
	mov		[bNoFrame], 1
	mov		eax,[dwRes]
if ?INITDBG    
	@DbgOutS <"EMS disable, eax=">,1
	@DbgOutD eax,1
	@DbgOutS <10>,1
	@WaitKey 1,0
endif	 
	mov		byte ptr [eax + device_name + 3],'Q'
@@emsactive:	

; alloc 2K status table (8 bytes * 256 handles)
;  each entry is 4 words, first word is either
;  -1 == system handle
;  -2 == free handle
;  -3 == occupied handle
; != the above values, first page number of saved mapping
; 2-4 words are 2nd-4th page number of saved mapping

	call	setstatustable
	
;--- the EMS name table requires at least 1 "SYSTEM" entry

	mov	[EMSNAMETABLE],EDI
	mov	eax,'TSYS'		; store "SYSTEM" as first handle name
	mov	[edi+0],eax
	mov	eax,'ME'
	mov	[edi+4],eax
	add	edi,8

if 0
	cmp	[bNoEMS],0
	jnz @@nonamesetup
endif

; allocate room for handle names (8*256 = 2K) and zero them

	mov	ecx,8*MAX_HANDLES/4
	xor	eax,eax
	rep	stos DWORD PTR [edi]
	@BIG_NOP
@@nonamesetup:

if ?INITDBG
	@DbgOutS <"EMS status table=">,1
	@DbgOutD EMSSTATUSTABLE,1
	@DbgOutS <", name table=">,1
	@DbgOutD EMSNAMETABLE,1
	@DbgOutS <10>,1
	@WaitKey 1,0
endif	 

; allocate and -1 store space for EMS page allocation pointer/descriptors
;  dword per page, points into EMS/VCPI block which controls page allocation
; allocate 4 bytes per 16K XMS total, 128K-4 max (no more than 32K-1 pages controlled)
;  page descriptor high word == 64-byte EMS/VCPI allocation block count from start
;  page descriptor low word ==	half-byte offset of allocation page (2 * 48)
;	within allocation block, not including system bytes

	mov ecx,[dwMaxEMSPages]
	mov	[EMSPageAllocationStart],edi
    push ecx
	or	eax,-1
	rep	stos DWORD PTR [edi]
	@BIG_NOP
    pop ecx
	mov	[EMSPageAllocationEnd],edi
	MOV	[EMSPAGETABLE],EDI

if ?INITDBG
	@DbgOutS <"EMS page alloc start/end=">,1
	@DbgOutD [EMSPageAllocationStart],1
	@DbgOutS <"/">,1
	@DbgOutD edi,1
	@DbgOutS <10>,1
endif

	MOV 	AL,FREEPAGE_ID			; all currently free
	REP 	STOS BYTE PTR [EDI]		; REP STOSB (addr. size = 32)
	@BIG_NOP

; initialize memory pool block descriptors
; each descriptor describes a memory block <= 1.5M (48*8 * 4K)
; and is 16 + 48 = 64 bytes in size.
; required are: ((dwMaxMem16K / 1.5M) + 128) * 64 bytes bytes for these items
; 128 is max number XMS handles

	add	edi,64-1
	and	edi,NOT (64-1)	; 64-byte align tables for proper location by count
	mov	[PoolAllocationTable],edi

	mov	eax,[dwMaxMem16K]	;=7680 (7680x16=120 MB)
	cdq
	mov	ecx,1536/16		; since dwMaxMem16k is in 16k units
	div	ecx				; default: 7680/96 = 80
	inc eax				; round up, not down
	cmp [bNoPool],0
	jnz @@isnopool
	movzx ecx,[XMS_Handle_Table.xht_numhandle]
	add eax, ecx
@@isnopool:    
	shl	eax,6-2			; 64 bytes/16 dwords each
	mov	ecx,eax

	xor	eax,eax
	rep	stos DWORD PTR [edi]
	@BIG_NOP

	mov	[PoolAllocationEnd],edi

if ?INITDBG
	@DbgOutS <"pool start/end=">,1
	@DbgOutD PoolAllocationTable,1
	@DbgOutS <"/">,1
	@DbgOutD PoolAllocationEnd,1
	@DbgOutS <10>,1
endif	 

;--- convert EDI back into a physical address
;--- use the page directory for the conversion

	mov		eax, edi
	sub		eax, ?SYSLINEAR
	shr		eax, 12
	@GETPTE eax, 2000h+eax*4+?SYSPTE
	and 	ah, 0F0h
	mov		al, 0
	
	and		edi, 0FFFh
	
;--- now check if current phys address is < DMA buffer	  
;--- if yes, all pages below must be skipped and are wasted
;--- since the EMS/VCPI memory managment needs a contiguous block
;--- of physical memory as input.

	mov		ecx, [DMABuffStartPhys]
	cmp		eax, ecx
	jnc		@@abovedma
if ?INITDBG
	@DbgOutS <"must waste space, phys end of monitor=">,1
	@DbgOutD eax,1
	@DbgOutS <" is below DMA buff=">,1
	@DbgOutD ecx,1
	@DbgOutS <10>,1
endif
	add		ecx, [DMABuffSize]		;get physical end of the DMA buff
	mov		eax, ecx
	xor		edi, edi
@@abovedma:    
	add		edi, eax
	mov		eax, jemmini
	mov eax, [eax].JEMMINIT.jiMonitorEnd

	cmp eax, EDI
	jc	@@killvcpi		;we run out of memory!!!

if ?INITDBG
	@DbgOutS <"end of monitor data, physical=">,1
	@DbgOutD edi,1
	@DbgOutS <", end of XMS memory block=">,1
	@DbgOutD eax,1
	@DbgOutS <10>,1
endif	 

; force 4K alignment for EMS/VCPI fixed pages and UMB's

	ADD 	EDI,4095				; Round to the next page border
	AND 	DI,NOT 4095
	MOV 	[FIRSTPAGE],EDI			; Mark the beginning of first EMS-page
;	 mov		ebp, edi				; save it in ebp
	
	sub		eax, edi
	jnc		@@isvalid
	xor		eax, eax
@@isvalid:	  
	shr	eax,14			; compute 16K pages of memory available
	
	cmp	[bNoPool], 0
	je	@@setblocks

if ?INITDBG
	@DbgOutS <"pool sharing is off, mem=">,1
	@DbgOutD eax,1
	@DbgOutS <10>,1
endif	 

;-- pool sharing is off, rest of block is used for EMS/VCPI
;-- make sure current values of MAXMEM16 and MAXEMSPAGES
;-- are not too high

	cmp eax,[dwMaxMem16K]
	jnc @@isnodecrease
	mov	[dwMaxMem16K], eax
@@isnodecrease:
	mov ecx,[dwMaxEMSPages]	;value is 8000 max
	cmp eax, ecx
	jnc @@isnodecrease2
	mov	[dwMaxEMSPages],eax
@@isnodecrease2:   
	mov eax, [dwMaxMem16K]

; if pool sharing, first pool allocation block points to remainder of initial
;  XMS allocated memory so that UMB code still has linear==physical memory

; setup control blocks (pool allocation blocks which are fixed) for the
;  fixed EMS/VCPI allocations
; start and end adjustments leave as initialized zero value since the memory
;  is originally 4K aligned, via edi alignment above

@@setblocks:
	mov	ecx,eax			; count of available 16K blocks
	mov	esi, [PoolAllocationTable]
	mov edx, jemmini
	movzx edx, [edx].JEMMINIT.jiXMSControlHandle

	@assumem esi, LPPOOL_SYSTEM_INFO

@@fixblkloop:
	mov	[esi].psi_descptr,edx
	mov	eax,ecx
	cmp	eax,POOLBLOCK_ALLOCATION_SPACE * 2	; 92 == 1536 / 16
	jb	@@fix2
	mov	eax,POOLBLOCK_ALLOCATION_SPACE * 2
@@fix2:
	
	sub	ecx,eax

if 0;?INITDBG
	@DbgOutS <"pool desc init, block=">,1
	@DbgOutD esi,1
	@DbgOutS <" 16k pg=">,1
	@DbgOutD eax,1
	@DbgOutS <" remaining=">,1
	@DbgOutD ecx,1
	@DbgOutS <10>,1
endif	 
	mov bl,al
	mov	[esi].psi_16kmax,al
	mov	[esi].psi_16kfree,al
	shl	eax,2
	mov	[esi].psi_4kfree,ax
	test bl,1		;was it uneven?
	jz	@@iseven
	inc bl						;add one page since pool must have an even
	mov	[esi].psi_16kmax,bl		;number for max
	movzx ebx, bl
	shr ebx, 1
	mov byte ptr [esi+ebx+POOLBLOCK_SYSTEM_SPACE],0F0h	;last page is "used"
@@iseven:
	mov	ebx,edi
	shr	ebx,10			; convert byte address to K
	mov	[esi].psi_addressK,ebx
	shl	eax,12			; convert 4K count to bytes
	add	edi,eax			; okay to increment edi free memory ptr, it's not used further

; never expand these blocks, the descriptor holds a real XMS handle rather than
;  an XMS pseudo-handle/pointer taken from the XMS handle array
	or	[esi].psi_flags,POOLBLOCK_FLAG_DONTEXPAND

	and ecx, ecx
	jz	@@fixblkdone	; no more memory to distribute to allocation blocks
	
	add	esi,POOLBLOCK_TOTAL_SPACE
	cmp	esi, [PoolAllocationEnd]
	jb	@@fixblkloop
	jmp @@fixblkdone

@@killvcpi:

; ran out of memory, shouldn't happen, avoid disaster by shutting off vcpi,
;  pool-sharing, and leaving max/available memory at 0

if ?INITDBG
	@DbgOutS <"out of memory condition on init!, MonitorEnd=">,1
	mov eax, jemmini
	@DbgOutD [eax].JEMMINIT.jiMonitorEnd,1
	@DbgOutS <" EDI=">,1
	@DbgOutD edi,1
	@DbgOutS <10>,1
endif	 

	mov	[bNoVCPI], 1	
	mov	[bNoEMS], 1	
	mov	[bNoPool], 1

@@fixblkdone:

	@assumem esi, nothing

if ?INITDBG
	@DbgOutS <"EMS/VCPI memory handling initialised, MAXMEM16K=">,1
	@DbgOutD [dwMaxMem16K],1
	@DbgOutS <10,"jump to v86-mode",10>,1
endif	 

;-- clear all dirty + accessed flags in page table 0

	@GETPTEPTR esi, 1000h, 1; let esi -> page table 0	
	mov		ecx, 1000h/4
	mov		al, not (20h or 40h)
@@nextpte:
	and		[esi],al
	add		esi, 4
	loop	@@nextpte
	
	mov		eax, ds:[06h*4]
	mov		[OldInt06],eax
	mov		eax, ds:[67h*4]
	mov		[OldInt67],eax
    
if ?VDS
	mov		eax, jemmini
	cmp		[eax].JEMMINIT.jiNoVDS, 0
	jnz		@@novds
	mov		eax, ds:[4Bh*4]	;vector may be 0000:0000
	mov		[OldInt4B],eax
	mov		eax, [dwRSeg]
	shl		eax, 16
	mov		al, [bBpTab]
if BPVDS
	add		al, BPVDS
endif
	mov		ds:[4Bh*4], eax
	or		byte ptr ds:[47Bh],20h
@@novds:
endif

	mov		eax, [FIRSTPAGE]
	mov		esp, ebp

if ?INITDBG
	@DbgOutS <"activate V86 mode",10>,1
	@WaitKey 1,0
endif

; Now finally all is prepared, so jump to v86 mode with an IRETD
	
	CLTS	   ; TS-Flag Clear (Task Switch) absolutely
			   ; thus the next x87-command without INT 7 is being executed

	IRETD	   ; this command switches to v86 mode
	
InitMonitor ENDP

	align 16
	
V86_TOTAL equ $ - _start

ife ?FLAT	 
.text	ENDS
endif

		END _start
