/*
// Program:  Free FDISK
// Written By:  Brian E. Reifsnyder
// Module:  FDISK.CPP
// Module Description:  Main Free FDISK Code Module
// Version:  0.98 Pre 1
// Copyright:  1998-2000 under the terms of the GNU GPL
*/

/*
/////////////////////////////////////////////////////////////////////////////
//  DEFINES
/////////////////////////////////////////////////////////////////////////////
*/

#define MAIN

/*
/////////////////////////////////////////////////////////////////////////////
//  INCLUDES
/////////////////////////////////////////////////////////////////////////////
*/

#include <bios.h>
#include <conio.h>
#include <dir.h>
#include <dos.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "fdisk.h"

/*
/////////////////////////////////////////////////////////////////////////////
//  PROTOTYPES
/////////////////////////////////////////////////////////////////////////////
*/

/* External Prototype Declarations */
/* ******************************* */
extern int Read_Partition_Tables();

extern void Automatically_Partition_Hard_Drive();
extern void Check_For_INT13_Extensions();
extern void Clear_Partition_Table();
extern void Clear_Flag(int flag_number);
extern void Load_Brief_Partition_Table();
extern void Load_External_Lookup_Table();
extern void Process_Fdiskini_File();
extern void Remove_MBR();
extern void Save_MBR();
extern void Set_Flag(int flag_number);
extern void Test_Flag(int flag_number);

/* Internal Prototype Declarations */
/* ******************************* */
int LBA_Partition_Type_To_Create(int standard_partition_type);
int Create_DOS_Partition_Interface(int numeric_type, long size_in_MB);
int Create_Logical_Drive(int numeric_type, long size_in_MB);
int Create_Logical_Drive_Interface();
int Create_Primary_Partition(int numeric_type,long size_in_MB);
int Delete_Logical_Drive_Interface();
int Determine_Drive_Letters();
int More_Than_One_Hard_Disk();
int Partition_Type_To_Create(unsigned long size_in_mb);
int Set_Active_Partition_Interface();
int Standard_Menu(int menu);
int Write_Partition_Tables();

unsigned long Combine_Cylinder_and_Sector(unsigned long cylinder, unsigned long sector);
unsigned long Decimal_Number(unsigned long hex1, unsigned long hex2
 ,unsigned long hex3, unsigned long hex4);
unsigned long Extract_Cylinder(unsigned long hex1, unsigned long hex2);
unsigned long Extract_Cylinder_From_LBA_Value(long double lba_value
 ,long double head,long double sector,long double total_heads
 ,long double total_sectors);
unsigned long Extract_Sector(unsigned long hex1, unsigned long hex2);
unsigned long Input(int size_of_field,int x_position,int y_position,int type, int min_range, long max_range,int return_message,int default_value,long maxiumum_possible_percentage);

void Ask_User_About_FAT32_Support();
void Calculate_Partition_Ending_Cylinder(long start_cylinder,long size);
void Clear_Active_Partition();
void Clear_Boot_Sector(int drive,long cylinder,long head,long sector);
void Clear_Extended_Partition_Table(int drive);
void Clear_Screen(int type);
void Convert_Long_To_Integer(long number);
void Create_Alternate_MBR();
void Create_MBR();
void Create_MBR_If_Not_Present();
void Delete_Extended_DOS_Partition_Interface();
void Delete_Logical_Drive(int type);
void Delete_N_DOS_Partition_Interface();
void Delete_Primary_Partition(int partition_number);
void Delete_Primary_DOS_Partition_Interface();
void Determine_Color_Video_Support();
void Determine_Free_Space();
void Display_All_Drives();
void Display_CL_Partition_Table();
void Display_Help_Screen();
void Display_Information();
void Display_Label();
void Display_Extended_Partition_Information_SS();
void Display_Or_Modify_Partition_Information();
void Display_Partition_Information();
void Display_Primary_Partition_Information_SS();
void Dump_Partition_Information();
void Initialization();
void Interactive_User_Interface();
void Menu_Routine();
void Modify_Partition_Type(int partition_number,int type_number);
void Pause();
void Position_Cursor(int row,int column);
void Print_Centered(int y,char *text,int style);
void Reboot_PC();
void Set_Active_Partition(int partition_number);
/*
/////////////////////////////////////////////////////////////////////////////
//  FUNCTIONS
/////////////////////////////////////////////////////////////////////////////
*/

/* Ask user if they want to use large disk support (FAT 32) */
void Ask_User_About_FAT32_Support()
{
  Clear_Screen(NULL);

  Print_Centered(5,"Free FDISK is capable of using large disk support to allow you to    ",0);
  Print_Centered(6,"create partitions that are greater than 2,048 MB by using FAT32      ",0);
  Print_Centered(7,"partitions.  If you enable large disk support, any partitions or     ",0);
  Print_Centered(8,"logical drives greater than 512 MB will be created using FAT32.      ",0);
  Print_Centered(10,"IMPORTANT:  If you enable large disk support many operating systems,",0);
  Print_Centered(11,"including FreeDOS, will be unable to access the partitions and      ",0);
  Print_Centered(12,"logical drives created that are over 512 MB in size.                ",0);

  Print_Centered(17,"Do you want to use large disk support (Y/N)....?    ",0);

  flags.fat32=Input(1,62,17,YN,0,0,NONE,1,0);
}

/* Calculate the end of a Partition */
void Calculate_Partition_Ending_Cylinder(long start_cylinder,unsigned long size)
{
  /* unsigned long size has to be in sectors @ 512 bytes/sector */
  /* Returns computed_ending_cylinder and computed_partition_size. */
  unsigned long delta_cylinders;

  unsigned long cylinder_size=(part_table[(flags.drive_number-0x80)]
   .total_head+1)*(part_table[(flags.drive_number-0x80)].total_sect);

  unsigned long number_of_cylinders_required=(size/cylinder_size);

  computed_partition_size=(cylinder_size*number_of_cylinders_required);
  computed_ending_cylinder=number_of_cylinders_required+start_cylinder-1;

  /* Keep the ending cylinder from going past the end of the hard disk. */
  if(computed_ending_cylinder>=part_table[(flags.drive_number-0x80)].
   total_cyl)
    {
    delta_cylinders=(computed_ending_cylinder
     -part_table[(flags.drive_number-0x80)].total_cyl)+1;
    computed_ending_cylinder=part_table[(flags.drive_number-0x80)]
     .total_cyl-1;
    computed_partition_size=computed_partition_size
     -(cylinder_size*delta_cylinders);
    }
}

/* Change Current Fixed Disk Drive */
void Change_Current_Fixed_Disk_Drive()
{
  int new_drive_number;
  int old_drive_number=flags.drive_number;

  Clear_Screen(NULL);
  Print_Centered(0,"Change Current Fixed Disk Drive",BOLD);

  Display_All_Drives();

  Position_Cursor(4,21);
  printf("Enter Fixed Disk Drive Number (1-%d)......................."
   ,(flags.maximum_drive_number-127));

  new_drive_number=Input(1,62,21,NUM,1,(flags.maximum_drive_number-127)
   ,ESCR,(flags.drive_number-127),0);

  if( (new_drive_number<=0)
   || (new_drive_number>(flags.maximum_drive_number-127)) )
    {
    flags.drive_number=old_drive_number;
    }
  else
    {
    flags.drive_number=new_drive_number+127;
    }
}

/* Clear the Active Partition */
void Clear_Active_Partition()
{
  int index=0;

  do
    {
    part_table[(flags.drive_number-128)].active_status[index]=0x00;
    index++;
    }while(index<4);

  part_table[(flags.drive_number-128)].part_values_changed=TRUE;
  flags.partitions_have_changed=TRUE;
}

/* Clear the Extended Partition Table Buffers */
void Clear_Extended_Partition_Table(int drive)
{
  int index;

  part_table[drive].ext_part_exists=FALSE;
  part_table[drive].ext_part_size_in_MB=0;
  part_table[drive].ext_part_num_sect=0;
  part_table[drive].ext_part_largest_free_space=0;

  part_table[drive].log_drive_free_space_start_cyl=0;
  part_table[drive].log_drive_free_space_end_cyl=0;

  part_table[drive].log_drive_largest_free_space_location=0;
  part_table[drive].num_of_ext_part=UNUSED;
  part_table[drive].num_of_log_drives=0;

  index=0;
  do
    {
    part_table[drive].log_drive_num_type[index]=0;

    part_table[drive].log_drive_start_cyl[index]=0;
    part_table[drive].log_drive_start_head[index]=0;
    part_table[drive].log_drive_start_sect[index]=0;

    part_table[drive].log_drive_end_cyl[index]=0;
    part_table[drive].log_drive_end_head[index]=0;
    part_table[drive].log_drive_end_sect[index]=0;

    part_table[drive].log_drive_rel_sect[index]=0;
    part_table[drive].log_drive_num_sect[index]=0;

    part_table[drive].log_drive_size_in_MB[index]=0;
    part_table[drive].log_drive_created[index]=FALSE;

    part_table[drive].next_ext_exists[index]=FALSE;

    part_table[drive].next_ext_num_type[index]=0;

    part_table[drive].next_ext_start_cyl[index]=0;
    part_table[drive].next_ext_start_head[index]=0;
    part_table[drive].next_ext_start_sect[index]=0;

    part_table[drive].next_ext_end_cyl[index]=0;
    part_table[drive].next_ext_end_head[index]=0;
    part_table[drive].next_ext_end_sect[index]=0;

    part_table[drive].next_ext_rel_sect[index]=0;
    part_table[drive].next_ext_num_sect[index]=0;

    index++;
    }while(index<24);
}

/* Clear Screen */
void Clear_Screen(int type)
{
  asm{
    mov ah,0
    mov al,3
    int 0x10
    }

  if(type!=NOEXTRAS)
    {
    Display_Information();

    Display_Label();
    }
}

/* Convert Long number to 2 integers */
void Convert_Long_To_Integer(long number)
{
  integer1=0;
  integer2=0;

  asm{
    mov ax,WORD PTR number

    mov BYTE PTR integer1, al
    mov BYTE PTR integer2, ah
    }
}

/* Create DOS Partition Interface */
int Create_DOS_Partition_Interface(int type)
{
  int numeric_type;
  int partition_created=FALSE;
  int partition_slot_just_used;

  long maximum_possible_percentage;

  unsigned long input=0;
  unsigned long maximum_partition_size_in_MB;

  Determine_Free_Space();

  maximum_partition_size_in_MB
   =(((part_table[(flags.drive_number-128)]
   .pri_part_largest_free_space+1)
   *(part_table[(flags.drive_number-128)].total_head+1)
   *(part_table[(flags.drive_number-128)].total_sect))/2048);

  /* Adjust maximum_partition_size_in_MB depending upon version */
  if( (type!=EXTENDED) && (flags.version==FOUR)
   && (maximum_partition_size_in_MB>2048) )
   maximum_partition_size_in_MB=2048;

  if( (type!=EXTENDED) && (flags.version==FIVE)
   && (maximum_partition_size_in_MB>2048) )
   maximum_partition_size_in_MB=2048;

  if( (type!=EXTENDED) && (flags.version==SIX)
   && (maximum_partition_size_in_MB>2048) )
   maximum_partition_size_in_MB=2048;

  if( (type!=EXTENDED) && (flags.version==W95)
   && (maximum_partition_size_in_MB>2048) )
   maximum_partition_size_in_MB=2048;

  if( (type!=EXTENDED) && ( (flags.version==W95B) || (flags.version==W98) )
   && (flags.fat32==FALSE) && (maximum_partition_size_in_MB>2048) )
   maximum_partition_size_in_MB=2048;

  if(type==PRIMARY)
    {
    Clear_Screen(NULL);

    Print_Centered(4,"Create Primary DOS Partition",BOLD);

    Position_Cursor(4,6);
    printf("Current fixed disk drive: ");
    cprintf("%d",(flags.drive_number-127));

    Position_Cursor(4,8);
    printf("Do you wish to use the maximum available size for a Primary DOS Partition");

    if((flags.drive_number-128)==0)
      {
      Position_Cursor(4,9);
      printf("and make the partition active (Y/N).....................? ");
      }
    else
      {
      Position_Cursor(4,9);
      printf("(Y/N)...................................................? ");
      }

    flags.esc=FALSE;
    input=Input(1,62,9,YN,0,0,ESCR,1,0);
    if(flags.esc==TRUE) return(1);

    if(input==1)
      {
      input=maximum_partition_size_in_MB;

      /* Use the maximum available free space to create a DOS Partition */

      /* Adjust numeric type depending upon partition size and the FDISK */
      /* version emulated.                                               */
      numeric_type=Partition_Type_To_Create(input);

      partition_slot_just_used=Create_Primary_Partition(numeric_type,input);
      if((flags.drive_number-128)==0) Set_Active_Partition(partition_slot_just_used);
      partition_created=TRUE;
      }
    }

  if(partition_created==FALSE)
    {
    Clear_Screen(NULL);

    if(type==PRIMARY) Print_Centered(4,"Create Primary DOS Partition",BOLD);
    else Print_Centered(4,"Create Extended DOS Partition",BOLD);

    Position_Cursor(4,6);
    printf("Current fixed disk drive: ");
    cprintf("%d",(flags.drive_number-127));

    Display_Primary_Partition_Information_SS();

    Position_Cursor(4,15);
    printf("Maximum space available for partition is ");
    cprintf("%4d",maximum_partition_size_in_MB);
    printf(" Mbytes ");

    maximum_possible_percentage=(100*maximum_partition_size_in_MB)/part_table[(flags.drive_number-128)].total_hard_disk_size_in_MB;
    cprintf("(%3d%%)",maximum_possible_percentage);

    Position_Cursor(4,18);
    printf("Enter partition size in Mbytes or percent of disk space (%) to");
    Position_Cursor(4,19);

    if(type==PRIMARY) printf("create a Primary DOS Partition.................................: ");
    else printf("create an Extended DOS Partition...............................: ");

    flags.esc=FALSE;

    if( (flags.version==4) || (flags.version==5) || (flags.version==6) )
     input=Input(4,69,19,NUMP,1,maximum_partition_size_in_MB,ESCR
     ,maximum_partition_size_in_MB,maximum_possible_percentage);
    else input=Input(5,69,19,NUMP,1,maximum_partition_size_in_MB,ESCR
     ,maximum_partition_size_in_MB,maximum_possible_percentage);

    if(flags.esc==TRUE) return(1);

    if(type==PRIMARY) numeric_type=Partition_Type_To_Create(input);
    else numeric_type=5;

    Create_Primary_Partition(numeric_type,input);
    }

  Clear_Screen(NULL);

  if(type==PRIMARY) Print_Centered(4,"Create Primary DOS Partition",BOLD);
  else Print_Centered(4,"Create Extended DOS Partition",BOLD);

  Position_Cursor(4,6);
  printf("Current fixed disk drive: ");
  cprintf("%d",(flags.drive_number-127));

  Display_Primary_Partition_Information_SS();

  Position_Cursor(4,21);
  if(type==PRIMARY) cprintf("Primary DOS Partition created");
  else cprintf("Extended DOS Partition created");

  Input(0,0,0,ESC,0,0,ESCC,0,0);

  if(type==EXTENDED) Create_Logical_Drive_Interface();

  return(0);
}

/* Create Logical Drive */
/* Returns a 0 if successful and a 1 if unsuccessful */
int Create_Logical_Drive(int numeric_type, long size_in_MB)
{
  int index;
  int offset;

  unsigned long maximum_size_in_logical_sectors
   =((part_table[(flags.drive_number-128)]
   .ext_part_largest_free_space+1)
   *(part_table[(flags.drive_number-128)].total_head+1)
   *(part_table[(flags.drive_number-128)].total_sect));
  unsigned long maximum_size_in_MB=maximum_size_in_logical_sectors/2048;
  unsigned long size_in_logical_sectors;

  /* Adjust the size of the partition to fit boundaries, if necessary. */
  if(size_in_MB>maximum_size_in_MB)
    {
    size_in_MB=maximum_size_in_MB;
    size_in_logical_sectors=maximum_size_in_MB*2048;

    numeric_type=Partition_Type_To_Create(size_in_MB);
    }
  else
    {
    size_in_logical_sectors=size_in_MB*2048;
    }

  /* Make space in the part_table structure, if necessary. */
  if( (part_table[(flags.drive_number-128)].log_drive_largest_free_space_location<=part_table[(flags.drive_number-128)].num_of_log_drives) && (part_table[(flags.drive_number-128)].log_drive_largest_free_space_location>0) )
    {
    index=part_table[(flags.drive_number-128)].num_of_log_drives+1;
    do
      {
      part_table[(flags.drive_number-128)].log_drive_num_type[index]=part_table[(flags.drive_number-128)].log_drive_num_type[(index-1)];
      strcpy(part_table[(flags.drive_number-128)].log_drive_vol_label[index],part_table[(flags.drive_number-128)].log_drive_vol_label[(index-1)]);

      part_table[(flags.drive_number-128)].log_drive_start_cyl[index]=part_table[(flags.drive_number-128)].log_drive_start_cyl[(index-1)];
      part_table[(flags.drive_number-128)].log_drive_start_head[index]=part_table[(flags.drive_number-128)].log_drive_start_head[(index-1)];
      part_table[(flags.drive_number-128)].log_drive_start_sect[index]=part_table[(flags.drive_number-128)].log_drive_start_sect[(index-1)];

      part_table[(flags.drive_number-128)].log_drive_end_cyl[index]=part_table[(flags.drive_number-128)].log_drive_end_cyl[(index-1)];
      part_table[(flags.drive_number-128)].log_drive_end_head[index]=part_table[(flags.drive_number-128)].log_drive_end_head[(index-1)];
      part_table[(flags.drive_number-128)].log_drive_end_sect[index]=part_table[(flags.drive_number-128)].log_drive_end_sect[(index-1)];

      part_table[(flags.drive_number-128)].log_drive_rel_sect[index]=part_table[(flags.drive_number-128)].log_drive_rel_sect[(index-1)];
      part_table[(flags.drive_number-128)].log_drive_num_sect[index]=part_table[(flags.drive_number-128)].log_drive_num_sect[(index-1)];

      part_table[(flags.drive_number-128)].log_drive_size_in_MB[index]=part_table[(flags.drive_number-128)].log_drive_size_in_MB[(index-1)];

      part_table[(flags.drive_number-128)].next_ext_exists[index]=part_table[(flags.drive_number-128)].next_ext_exists[(index-1)];

      part_table[(flags.drive_number-128)].next_ext_num_type[index]=part_table[(flags.drive_number-128)].next_ext_num_type[(index-1)];

      part_table[(flags.drive_number-128)].next_ext_start_cyl[index]=part_table[(flags.drive_number-128)].next_ext_start_cyl[(index-1)];
      part_table[(flags.drive_number-128)].next_ext_start_head[index]=part_table[(flags.drive_number-128)].next_ext_start_head[(index-1)];
      part_table[(flags.drive_number-128)].next_ext_start_sect[index]=part_table[(flags.drive_number-128)].next_ext_start_sect[(index-1)];

      part_table[(flags.drive_number-128)].next_ext_end_cyl[index]=part_table[(flags.drive_number-128)].next_ext_end_cyl[(index-1)];
      part_table[(flags.drive_number-128)].next_ext_end_head[index]=part_table[(flags.drive_number-128)].next_ext_end_head[(index-1)];
      part_table[(flags.drive_number-128)].next_ext_end_sect[index]=part_table[(flags.drive_number-128)].next_ext_end_sect[(index-1)];

      part_table[(flags.drive_number-128)].next_ext_rel_sect[index]=part_table[(flags.drive_number-128)].next_ext_rel_sect[(index-1)];
      part_table[(flags.drive_number-128)].next_ext_num_sect[index]=part_table[(flags.drive_number-128)].next_ext_num_sect[(index-1)];

      index--;
      }while(index>=part_table[(flags.drive_number-128)].log_drive_largest_free_space_location);
    }

  /* Add the logical drive entry. */
  part_table[(flags.drive_number-128)].log_drive_num_type[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=numeric_type;
  strcpy(part_table[(flags.drive_number-128)].log_drive_vol_label[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location],"");

  part_table[(flags.drive_number-128)].log_drive_start_cyl[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=part_table[(flags.drive_number-128)].log_drive_free_space_start_cyl;
  part_table[(flags.drive_number-128)].log_drive_start_head[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=1;
  part_table[(flags.drive_number-128)].log_drive_start_sect[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=1;

  Calculate_Partition_Ending_Cylinder(part_table[(flags.drive_number-128)].log_drive_free_space_start_cyl,size_in_logical_sectors);

  part_table[(flags.drive_number-128)].log_drive_end_cyl[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=computed_ending_cylinder;
  part_table[(flags.drive_number-128)].log_drive_end_head[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=part_table[(flags.drive_number-128)].total_head;
  part_table[(flags.drive_number-128)].log_drive_end_sect[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=part_table[(flags.drive_number-128)].total_sect;

  if( (part_table[(flags.drive_number-128)]
   .log_drive_end_cyl[part_table[(flags.drive_number-128)]
   .log_drive_largest_free_space_location]>1023)
   && (part_table[(flags.drive_number-128)].ext_int_13==TRUE) )
   part_table[(flags.drive_number-128)]
   .log_drive_num_type[part_table[(flags.drive_number-128)]
   .log_drive_largest_free_space_location]
   =LBA_Partition_Type_To_Create(numeric_type);

  part_table[(flags.drive_number-128)].log_drive_rel_sect[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=part_table[(flags.drive_number-128)].total_sect;
  part_table[(flags.drive_number-128)].log_drive_num_sect[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=computed_partition_size-part_table[(flags.drive_number-128)].total_sect;
  part_table[(flags.drive_number-128)].log_drive_size_in_MB[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=computed_partition_size/2048;

  part_table[(flags.drive_number-128)].num_of_log_drives++;
  part_table[(flags.drive_number-128)].log_drive_created[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=TRUE;

  /* Add the linkage entry. */

  /* Add the linkage entry if there is a logical drive after this one. */
  if(part_table[(flags.drive_number-128)].log_drive_num_type[(part_table[(flags.drive_number-128)].log_drive_largest_free_space_location+1)]>0)
    {
    part_table[(flags.drive_number-128)].next_ext_exists[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=TRUE;

    part_table[(flags.drive_number-128)].next_ext_num_type[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=5;

    part_table[(flags.drive_number-128)].next_ext_start_cyl[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=part_table[(flags.drive_number-128)].log_drive_start_cyl[(part_table[(flags.drive_number-128)].log_drive_largest_free_space_location+1)];
    part_table[(flags.drive_number-128)].next_ext_start_head[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=0;
    part_table[(flags.drive_number-128)].next_ext_start_sect[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=1;

    part_table[(flags.drive_number-128)].next_ext_end_cyl[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=part_table[(flags.drive_number-128)].log_drive_end_cyl[(part_table[(flags.drive_number-128)].log_drive_largest_free_space_location+1)];
    part_table[(flags.drive_number-128)].next_ext_end_head[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=part_table[(flags.drive_number-128)].total_head;
    part_table[(flags.drive_number-128)].next_ext_end_sect[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=part_table[(flags.drive_number-128)].total_sect;

    part_table[(flags.drive_number-128)].next_ext_rel_sect[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=(part_table[(flags.drive_number-128)].log_drive_start_cyl[(part_table[(flags.drive_number-128)].log_drive_largest_free_space_location+1)]-part_table[(flags.drive_number-128)].pri_part_start_cyl[part_table[(flags.drive_number-128)].num_of_ext_part])*(part_table[(flags.drive_number-128)].total_head+1)*part_table[(flags.drive_number-128)].total_sect;
    part_table[(flags.drive_number-128)].next_ext_num_sect[part_table[(flags.drive_number-128)].log_drive_largest_free_space_location]=part_table[(flags.drive_number-128)].log_drive_num_sect[(part_table[(flags.drive_number-128)].log_drive_largest_free_space_location+1)]+part_table[(flags.drive_number-128)].total_sect;
    }

  /* Add the linkage entry if there is a logical drive before this one. */
  if(part_table[(flags.drive_number-128)].log_drive_largest_free_space_location>0)
    {
    part_table[(flags.drive_number-128)].next_ext_exists[(part_table[(flags.drive_number-128)].log_drive_largest_free_space_location-1)]=TRUE;

    part_table[(flags.drive_number-128)].next_ext_num_type[(part_table[(flags.drive_number-128)].log_drive_largest_free_space_location-1)]=5;

    part_table[(flags.drive_number-128)].next_ext_start_cyl[(part_table[(flags.drive_number-128)].log_drive_largest_free_space_location-1)]=part_table[(flags.drive_number-128)].log_drive_free_space_start_cyl;
    part_table[(flags.drive_number-128)].next_ext_start_head[(part_table[(flags.drive_number-128)].log_drive_largest_free_space_location-1)]=0;
    part_table[(flags.drive_number-128)].next_ext_start_sect[(part_table[(flags.drive_number-128)].log_drive_largest_free_space_location-1)]=1;

    part_table[(flags.drive_number-128)].next_ext_end_cyl[(part_table[(flags.drive_number-128)].log_drive_largest_free_space_location-1)]=computed_ending_cylinder;
    part_table[(flags.drive_number-128)].next_ext_end_head[(part_table[(flags.drive_number-128)].log_drive_largest_free_space_location-1)]=part_table[(flags.drive_number-128)].total_head;
    part_table[(flags.drive_number-128)].next_ext_end_sect[(part_table[(flags.drive_number-128)].log_drive_largest_free_space_location-1)]=part_table[(flags.drive_number-128)].total_sect;

    part_table[(flags.drive_number-128)].next_ext_rel_sect[(part_table[(flags.drive_number-128)].log_drive_largest_free_space_location-1)]=( (part_table[(flags.drive_number-128)].next_ext_start_cyl[(part_table[(flags.drive_number-128)].log_drive_largest_free_space_location-1)]) - (part_table[(flags.drive_number-128)].pri_part_start_cyl[part_table[(flags.drive_number-128)].num_of_ext_part]) )*(part_table[(flags.drive_number-128)].total_head+1)*part_table[(flags.drive_number-128)].total_sect;

    part_table[(flags.drive_number-128)].next_ext_num_sect
     [(part_table[(flags.drive_number-128)]
     .log_drive_largest_free_space_location-1)]=computed_partition_size;
    }

  part_table[(flags.drive_number-128)].part_values_changed=TRUE;
  flags.partitions_have_changed=TRUE;

  if(debug.create_partition==TRUE)
    {
    Clear_Screen(NULL);
    Print_Centered(1,"int Create_Logical_Drive(int numeric_type,long size_in_MB)",BOLD);
    Position_Cursor(4,3);
    printf("int numeric_type=%d",numeric_type);
    Position_Cursor(4,4);
    printf("long size_in_MB=%d",size_in_MB);
    Position_Cursor(4,5);
    printf("Number of partition that was created:  %d",part_table[(flags.drive_number-128)].log_drive_largest_free_space_location);
    Position_Cursor(4,7);
    printf("Brief logical drive table:");

    index=part_table[(flags.drive_number-128)].log_drive_largest_free_space_location-1;
    offset=9;

    Position_Cursor(4,8);
    printf(" #  NT     SC    SH    SS      EC   EH   ES      Rel. Sect.    Size in MB ");

    do
      {
      if( (index>=0) && (index<24) )
	{
	Position_Cursor(4,offset);
	printf("%2d",index);
	Position_Cursor(7,offset);
	printf("%3d",part_table[(flags.drive_number-128)].log_drive_num_type[index]);
	Position_Cursor(13,offset);
	printf("%4d",part_table[(flags.drive_number-128)].log_drive_start_cyl[index]);
	Position_Cursor(19,offset);
	printf("%4d",part_table[(flags.drive_number-128)].log_drive_start_head[index]);
	Position_Cursor(25,offset);
	printf("%4d",part_table[(flags.drive_number-128)].log_drive_start_sect[index]);

	Position_Cursor(33,offset);
	printf("%4d",part_table[(flags.drive_number-128)].log_drive_end_cyl[index]);
	Position_Cursor(38,offset);
	printf("%4d",part_table[(flags.drive_number-128)].log_drive_end_head[index]);
	Position_Cursor(43,offset);
	printf("%4d",part_table[(flags.drive_number-128)].log_drive_end_sect[index]);

	Position_Cursor(58,offset);
	printf("%d",part_table[(flags.drive_number-128)].log_drive_rel_sect[index]);

	Position_Cursor(72,offset);
	printf("%5d",part_table[(flags.drive_number-128)].log_drive_size_in_MB[index]);
	}
      else
	{
	Position_Cursor(4,offset);
	printf("N/A");
	}

      offset++;
      index++;

      }while(offset<=11);

    Position_Cursor(4,15);
    printf("Next extended location table:");

    Position_Cursor(4,16);
    printf(" #         SC    SH    SS      EC   EH   ES      Rel. Sect.    Size in MB ");

    index=part_table[(flags.drive_number-128)].log_drive_largest_free_space_location-1;
    offset=17;

    do
      {
      if( (index>=0) && (index<24) && (part_table[(flags.drive_number-128)].next_ext_exists[index]==TRUE) )
	{
	Position_Cursor(4,offset);
	printf("%2d",index);

	Position_Cursor(13,offset);
	printf("%4d",part_table[(flags.drive_number-128)].next_ext_start_cyl[index]);
	Position_Cursor(19,offset);
	printf("%4d",part_table[(flags.drive_number-128)].next_ext_start_head[index]);
	Position_Cursor(25,offset);
	printf("%4d",part_table[(flags.drive_number-128)].next_ext_start_sect[index]);

	Position_Cursor(33,offset);
	printf("%4d",part_table[(flags.drive_number-128)].next_ext_end_cyl[index]);
	Position_Cursor(38,offset);
	printf("%4d",part_table[(flags.drive_number-128)].next_ext_end_head[index]);
	Position_Cursor(43,offset);
	printf("%4d",part_table[(flags.drive_number-128)].next_ext_end_sect[index]);

	/*
	Temporarily removed because the size of the relative sector field
	exceeds that handled by the printf statement.  As a result,
	incorrect information is returned.

	Position_Cursor(58,offset);
	printf("%4d",part_table[(flags.drive_number-128)].next_ext_rel_sect[index]);
	*/

	Position_Cursor(72,offset);
	printf("%4d",((part_table[(flags.drive_number-128)].next_ext_num_sect[index])/2048));
	}
      else
	{
	Position_Cursor(4,offset);
	printf("N/A");
	}

      offset++;
      index++;

      }while(offset<=19);

    Pause();
    }

  return(0);
}

/* Create Logical Drive Interface */
/* Returns a 0 if successful and a 1 if unsuccessful */
int Create_Logical_Drive_Interface()
{
  long input=0;

  int drive_created=FALSE;
  int maximum_possible_percentage;
  int numeric_type;

  unsigned long maximum_partition_size_in_MB;

  Determine_Free_Space();

  maximum_partition_size_in_MB
   =(((part_table[(flags.drive_number-128)]
   .ext_part_largest_free_space+1)
   *(part_table[(flags.drive_number-128)].total_head+1)
   *(part_table[(flags.drive_number-128)].total_sect))/2048);

  if(part_table[(flags.drive_number-128)]
   .ext_part_largest_free_space>=2)
    {
    do
      {
      /* Adjust maximum_partition_size_in_MB depending upon version */
      if( (flags.version==FOUR) && (maximum_partition_size_in_MB>2048) )
       maximum_partition_size_in_MB=2048;
      if( (flags.version==FIVE) && (maximum_partition_size_in_MB>2048) )
       maximum_partition_size_in_MB=2048;
      if( (flags.version==SIX) && (maximum_partition_size_in_MB>2048) )
       maximum_partition_size_in_MB=2048;
      if( (flags.version==W95) && (maximum_partition_size_in_MB>2048) )
       maximum_partition_size_in_MB=2048;
      if( ( (flags.version==W95B) || (flags.version==W98) )
       && (flags.fat32==FALSE) && (maximum_partition_size_in_MB>2048) )
       maximum_partition_size_in_MB=2048;

      Clear_Screen(NULL);

      if(drive_created==TRUE)
	{
	Position_Cursor(4,22);
	cprintf("Logical DOS Drive created, drive letters changed or added");
	}

      Print_Centered(1,"Create Logical DOS Drive in the Extended DOS Partition",BOLD);

      Display_Extended_Partition_Information_SS();

      if(1==Determine_Drive_Letters())
	{
	Position_Cursor(4,22);
	printf("                                                           ");
	Position_Cursor(4,22);
	cprintf("Maximum number of Logical DOS Drives installed.");
	Input(0,0,0,ESC,0,0,ESCC,0,0);
	return(1);
	}

      Position_Cursor(4,17);
      printf("Total Extended DOS Partition size is ");
      cprintf("%4d",part_table[(flags.drive_number-128)].ext_part_size_in_MB);
      printf(" Mbytes (1 Mbyte = 1048576 bytes)");

      Position_Cursor(4,18);
      printf("Maximum space available for partition is ");

      if( (flags.version==4) || (flags.version==5) || (flags.version==6) )
	cprintf("%4d",maximum_partition_size_in_MB);
      else cprintf("%5d",maximum_partition_size_in_MB);

      printf(" Mbytes ");
      maximum_possible_percentage=(100*maximum_partition_size_in_MB)/part_table[(flags.drive_number-128)].ext_part_size_in_MB;
      cprintf("(%3d%%)",maximum_possible_percentage);

      Position_Cursor(4,20);
      printf("Enter logical drive size in Mbytes or percent of disk space (%)...");

      flags.esc=FALSE;

      if( (flags.version==4) || (flags.version==5) || (flags.version==6) )
       input=Input(4,70,20,NUMP,1,maximum_partition_size_in_MB,ESCR
       ,maximum_partition_size_in_MB,maximum_possible_percentage);
      else input=Input(5,70,20,NUMP,1,maximum_partition_size_in_MB,ESCR
       ,maximum_partition_size_in_MB,maximum_possible_percentage);

      if(flags.esc==TRUE) return(1);

      numeric_type=Partition_Type_To_Create(input);

      Create_Logical_Drive(numeric_type,input);
      drive_created=TRUE;

      Determine_Free_Space();
      maximum_partition_size_in_MB
       =(((part_table[(flags.drive_number-128)]
       .ext_part_largest_free_space+1)
       *(part_table[(flags.drive_number-128)].total_head+1)
       *(part_table[(flags.drive_number-128)].total_sect))/2048);
      }while(part_table[(flags.drive_number-128)].ext_part_largest_free_space>=2);
    }

  Clear_Screen(NULL);
  Print_Centered(1,"Create Logical DOS Drive in the Extended DOS Partition",BOLD);
  Display_Extended_Partition_Information_SS();
  Position_Cursor(4,22);
  cprintf("All available space in the Extended DOS Partition");
  Position_Cursor(4,23);
  cprintf("is assigned to logical drives.");
  Input(0,0,0,ESC,0,0,ESCC,0,0);

  return(0);
}

/* Create a Primary Partition */
/* Returns partition number if successful and a 99 if unsuccessful */
int Create_Primary_Partition(int numeric_type,long size_in_MB)
{
  int index=0;
  int empty_partition_number=99;

  unsigned long maximum_size_in_logical_sectors
   =((part_table[(flags.drive_number-128)].pri_part_largest_free_space)
   *(part_table[(flags.drive_number-128)].total_head+1)
   *(part_table[(flags.drive_number-128)].total_sect))
   -part_table[(flags.drive_number-128)].total_sect;

  unsigned long maximum_size_in_MB=maximum_size_in_logical_sectors/2048;
  unsigned long size_in_logical_sectors;

  /* Ensure that an empty primary partition exists. */
  do
    {
    if( (empty_partition_number==99) && (part_table[(flags.drive_number-128)].pri_part_num_type[index]==0) )
      {
      empty_partition_number=index;
      }
    index++;
    }while(index<4);

  /* If all primary partitions are full, report failure. */
  if(empty_partition_number==99) return(99);

  /* Adjust the size of the partition to fit boundaries, if necessary. */
  if(size_in_MB>maximum_size_in_MB)
    {
    size_in_MB=maximum_size_in_MB;
    size_in_logical_sectors=maximum_size_in_MB*2048;

    if( (numeric_type!=5) && (numeric_type!=0x0f) )
     numeric_type=Partition_Type_To_Create(size_in_MB);
    }
  else
    {
    size_in_logical_sectors=size_in_MB*2048;
    }

  /* Make sure the starting cylinder of an extended partition is at least  */
  /* 1.  If the cylinder number is 0, increment it to 1.                   */
  if( (numeric_type==5) || (numeric_type==0x0f) )
    {
    if(part_table[(flags.drive_number-128)].pp_largest_free_space_start_cyl==0)
      part_table[(flags.drive_number-128)].pp_largest_free_space_start_cyl=1;
      size_in_logical_sectors=size_in_logical_sectors
       -( (part_table[(flags.drive_number-128)].total_head+1)
       *(part_table[(flags.drive_number-128)].total_sect)
       -part_table[(flags.drive_number-128)].total_sect );
    }

  Calculate_Partition_Ending_Cylinder(part_table[(flags.drive_number-128)].pp_largest_free_space_start_cyl,size_in_logical_sectors);

  part_table[(flags.drive_number-128)].active_status[empty_partition_number]=0;
  part_table[(flags.drive_number-128)].pri_part_num_type[empty_partition_number]=numeric_type;

  part_table[(flags.drive_number-128)].pri_part_start_cyl[empty_partition_number]=part_table[(flags.drive_number-128)].pp_largest_free_space_start_cyl;
  /* If the starting cylinder is 0, then the starting head is 1...otherwise */
  /* the starting head is 1.                                                */
  if(part_table[(flags.drive_number-128)].pp_largest_free_space_start_cyl==0) part_table[(flags.drive_number-128)].pri_part_start_head[empty_partition_number]=1;
  else part_table[(flags.drive_number-128)].pri_part_start_head[empty_partition_number]=0;
  part_table[(flags.drive_number-128)].pri_part_start_sect[empty_partition_number]=1;

  part_table[(flags.drive_number-128)].pri_part_end_cyl[empty_partition_number]=computed_ending_cylinder;
  part_table[(flags.drive_number-128)].pri_part_end_head[empty_partition_number]=part_table[(flags.drive_number-128)].total_head;
  part_table[(flags.drive_number-128)].pri_part_end_sect[empty_partition_number]=part_table[(flags.drive_number-128)].total_sect;

  if( (part_table[(flags.drive_number-128)]
   .pri_part_end_cyl[empty_partition_number]>1023)
   && (part_table[(flags.drive_number-128)].ext_int_13==TRUE) )
    {
    numeric_type=LBA_Partition_Type_To_Create(numeric_type);
    part_table[(flags.drive_number-128)]
     .pri_part_num_type[empty_partition_number]=numeric_type;
    }

  if(part_table[(flags.drive_number-128)].pri_part_start_cyl[empty_partition_number]>0)
    {
    part_table[(flags.drive_number-128)].pri_part_rel_sect[empty_partition_number]=(part_table[(flags.drive_number-128)].pri_part_start_cyl[empty_partition_number])*(part_table[(flags.drive_number-128)].total_head+1)*part_table[(flags.drive_number-128)].total_sect;
    }
  else part_table[(flags.drive_number-128)].pri_part_rel_sect[empty_partition_number]=part_table[(flags.drive_number-128)].total_sect;

  if(part_table[(flags.drive_number-128)]
   .pp_largest_free_space_start_cyl==0)
   computed_partition_size=computed_partition_size-part_table
   [(flags.drive_number-128)].total_sect;

  part_table[(flags.drive_number-128)]
   .pri_part_num_sect[empty_partition_number]
   =computed_partition_size;

  part_table[(flags.drive_number-128)].pri_part_size_in_MB[empty_partition_number]=computed_partition_size/2048;

  part_table[(flags.drive_number-128)].part_values_changed=TRUE;
  flags.partitions_have_changed=TRUE;

  part_table[(flags.drive_number-128)].pri_part_created[empty_partition_number]=TRUE;

  if( (numeric_type==5) || (numeric_type==0x0f) )
    {
    part_table[(flags.drive_number-128)].ext_part_exists=TRUE;
    part_table[(flags.drive_number-128)].num_of_ext_part=empty_partition_number;
    part_table[(flags.drive_number-128)].ext_part_num_sect=computed_partition_size;
    part_table[(flags.drive_number-128)].ext_part_size_in_MB=size_in_MB;
    }

  if(debug.create_partition==TRUE)
    {
    Clear_Screen(NULL);
    Print_Centered(1,"int Create_Primary_Partition(int numeric_type,long size_in_MB)",BOLD);
    Position_Cursor(4,3);
    printf("int numeric_type=%d",numeric_type);
    Position_Cursor(4,4);
    printf("long size_in_MB=%d",size_in_MB);
    Position_Cursor(4,5);
    printf("empty_partition_number=%d",empty_partition_number);

    Position_Cursor(4,8);
    printf("New Partition Information:");
    Position_Cursor(4,10);
    printf("Starting Cylinder:  %d",part_table[(flags.drive_number-128)].pri_part_start_cyl[empty_partition_number]);
    Position_Cursor(4,11);
    printf("Starting Head:      %d",part_table[(flags.drive_number-128)].pri_part_start_head[empty_partition_number]);
    Position_Cursor(4,12);
    printf("Starting Sector:    %d",part_table[(flags.drive_number-128)].pri_part_start_sect[empty_partition_number]);

    Position_Cursor(40,10);
    printf("Ending Cylinder:    %d",part_table[(flags.drive_number-128)].pri_part_end_cyl[empty_partition_number]);
    Position_Cursor(40,11);
    printf("Ending Head:        %d",part_table[(flags.drive_number-128)].pri_part_end_head[empty_partition_number]);
    Position_Cursor(40,12);
    printf("Ending Sector:      %d",part_table[(flags.drive_number-128)].pri_part_end_sect[empty_partition_number]);

    Position_Cursor(4,14);
    printf("Relative Sectors:   %d",part_table[(flags.drive_number-128)].pri_part_rel_sect[empty_partition_number]);

    Position_Cursor(40,14);
    printf("Size of partition in MB:    %d",part_table[(flags.drive_number-128)].pri_part_size_in_MB[empty_partition_number]);

    Pause();
    }

  return(empty_partition_number);
}

/* Delete Extended DOS Partition Interface */
void Delete_Extended_DOS_Partition_Interface()
{
  int input=0;

  Clear_Screen(NULL);

  Print_Centered(4,"Delete Extended DOS Partition",BOLD);

  Display_Primary_Partition_Information_SS();

  Position_Cursor(4,18);
  if(flags.monochrome!=TRUE) textcolor(WHITE | BLINK);
  cprintf("WARNING!");
  if(flags.monochrome!=TRUE) textcolor(15);

  printf(" Data in the deleted Extended DOS Partition will be lost.");
  Position_Cursor(4,19);
  printf("Do you wish to continue (Y/N).................? ");

  flags.esc=FALSE;
  input=Input(1,52,19,YN,0,0,ESCR,0,0);

  if( (flags.esc==FALSE) && (input==TRUE) )
    {
    Delete_Primary_Partition(part_table[(flags.drive_number-128)].num_of_ext_part);
    Clear_Extended_Partition_Table(flags.drive_number-128);

    Position_Cursor(4,21);
    cprintf("Extended DOS Partition deleted");

    Position_Cursor(4,24);
    printf("                                    ");

    Input(0,0,0,ESC,0,0,ESCC,0,0);
    }
}

/* Delete Logical DOS Drive */
void Delete_Logical_Drive(int logical_drive_number)
{
  int index;

  /* Zero out the partition table entry for the logical drive. */
  part_table[(flags.drive_number-128)]
   .log_drive_num_type[logical_drive_number]=0;

  strcpy(part_table[(flags.drive_number-128)]
   .log_drive_vol_label[logical_drive_number],"           ");

  part_table[(flags.drive_number-128)]
   .log_drive_start_cyl[logical_drive_number]=0;
  part_table[(flags.drive_number-128)]
   .log_drive_start_head[logical_drive_number]=0;
  part_table[(flags.drive_number-128)]
   .log_drive_start_sect[logical_drive_number]=0;

  part_table[(flags.drive_number-128)]
   .log_drive_end_cyl[logical_drive_number]=0;
  part_table[(flags.drive_number-128)]
   .log_drive_end_head[logical_drive_number]=0;
  part_table[(flags.drive_number-128)]
   .log_drive_end_sect[logical_drive_number]=0;

  part_table[(flags.drive_number-128)]
   .log_drive_rel_sect[logical_drive_number]=0;
  part_table[(flags.drive_number-128)]
   .log_drive_num_sect[logical_drive_number]=0;

  part_table[(flags.drive_number-128)]
   .log_drive_size_in_MB[logical_drive_number]=0;

  /* If the logical drive to be deleted is not the first logical drive.     */
  /* Assume that there are extended partition tables after this one.  If    */
  /* there are not any more extended partition tables, nothing will be      */
  /* harmed by the shift.                                                   */
  if(logical_drive_number>0)
    {
    /* Move the extended partition information from this table to the       */
    /* previous table.                                                      */
    part_table[(flags.drive_number-128)]
     .next_ext_start_cyl[(logical_drive_number-1)]
     =part_table[(flags.drive_number-128)]
     .next_ext_start_cyl[logical_drive_number];

    part_table[(flags.drive_number-128)]
     .next_ext_start_head[(logical_drive_number-1)]
     =part_table[(flags.drive_number-128)]
     .next_ext_start_head[logical_drive_number];

    part_table[(flags.drive_number-128)]
     .next_ext_start_sect[(logical_drive_number-1)]
     =part_table[(flags.drive_number-128)]
     .next_ext_start_sect[logical_drive_number];

    part_table[(flags.drive_number-128)]
     .next_ext_end_cyl[(logical_drive_number-1)]
     =part_table[(flags.drive_number-128)]
     .next_ext_end_cyl[logical_drive_number];

    part_table[(flags.drive_number-128)]
     .next_ext_end_head[(logical_drive_number-1)]
     =part_table[(flags.drive_number-128)]
     .next_ext_end_head[logical_drive_number];

    part_table[(flags.drive_number-128)]
     .next_ext_end_sect[(logical_drive_number-1)]
     =part_table[(flags.drive_number-128)]
     .next_ext_end_sect[logical_drive_number];

    part_table[(flags.drive_number-128)]
     .next_ext_rel_sect[(logical_drive_number-1)]
     =part_table[(flags.drive_number-128)]
     .next_ext_rel_sect[logical_drive_number];

    part_table[(flags.drive_number-128)]
     .next_ext_num_sect[(logical_drive_number-1)]
     =part_table[(flags.drive_number-128)]
     .next_ext_num_sect[logical_drive_number];

    /* Shift all the following extended partition tables left by 1.         */
    index=logical_drive_number;

    do
      {
      part_table[(flags.drive_number-128)].log_drive_num_type[index]=part_table[(flags.drive_number-128)].log_drive_num_type[(index+1)];

      strcpy(part_table[(flags.drive_number-128)].log_drive_vol_label[index],part_table[(flags.drive_number-128)].log_drive_vol_label[(index+1)]);

      part_table[(flags.drive_number-128)].log_drive_start_cyl[index]=part_table[(flags.drive_number-128)].log_drive_start_cyl[(index+1)];
      part_table[(flags.drive_number-128)].log_drive_start_head[index]=part_table[(flags.drive_number-128)].log_drive_start_head[(index+1)];
      part_table[(flags.drive_number-128)].log_drive_start_sect[index]=part_table[(flags.drive_number-128)].log_drive_start_sect[(index+1)];

      part_table[(flags.drive_number-128)].log_drive_end_cyl[index]=part_table[(flags.drive_number-128)].log_drive_end_cyl[(index+1)];
      part_table[(flags.drive_number-128)].log_drive_end_head[index]=part_table[(flags.drive_number-128)].log_drive_end_head[(index+1)];
      part_table[(flags.drive_number-128)].log_drive_end_sect[index]=part_table[(flags.drive_number-128)].log_drive_end_sect[(index+1)];

      part_table[(flags.drive_number-128)].log_drive_rel_sect[index]=part_table[(flags.drive_number-128)].log_drive_rel_sect[(index+1)];
      part_table[(flags.drive_number-128)].log_drive_num_sect[index]=part_table[(flags.drive_number-128)].log_drive_num_sect[(index+1)];

      part_table[(flags.drive_number-128)].log_drive_size_in_MB[index]=part_table[(flags.drive_number-128)].log_drive_size_in_MB[(index+1)];

      part_table[(flags.drive_number-128)].next_ext_num_type[index]=part_table[(flags.drive_number-128)].next_ext_num_type[(index+1)];

      part_table[(flags.drive_number-128)].next_ext_start_cyl[index]=part_table[(flags.drive_number-128)].next_ext_start_cyl[(index+1)];
      part_table[(flags.drive_number-128)].next_ext_start_head[index]=part_table[(flags.drive_number-128)].next_ext_start_head[(index+1)];
      part_table[(flags.drive_number-128)].next_ext_start_sect[index]=part_table[(flags.drive_number-128)].next_ext_start_sect[(index+1)];

      part_table[(flags.drive_number-128)].next_ext_end_cyl[index]=part_table[(flags.drive_number-128)].next_ext_end_cyl[(index+1)];
      part_table[(flags.drive_number-128)].next_ext_end_head[index]=part_table[(flags.drive_number-128)].next_ext_end_head[(index+1)];
      part_table[(flags.drive_number-128)].next_ext_end_sect[index]=part_table[(flags.drive_number-128)].next_ext_end_sect[(index+1)];

      part_table[(flags.drive_number-128)].next_ext_rel_sect[index]=part_table[(flags.drive_number-128)].next_ext_rel_sect[(index+1)];
      part_table[(flags.drive_number-128)].next_ext_num_sect[index]=part_table[(flags.drive_number-128)].next_ext_num_sect[(index+1)];

      if(part_table[(flags.drive_number-128)].log_drive_num_type[index]>0)
	{
	part_table[(flags.drive_number-128)].next_ext_exists[(index-1)]=TRUE;
	}
      else
	{
	part_table[(flags.drive_number-128)].next_ext_exists[(index-1)]=FALSE;
	}

      index++;
      }while(index<22);
    }
  part_table[(flags.drive_number-128)].num_of_log_drives--;

  /* If there aren't any more logical drives, clear the extended        */
  /* partition table to prevent lockups by any other partition utils.   */
  if(part_table[(flags.drive_number-128)].num_of_log_drives==0)
    {
    index=0;
    do
      {
      part_table[(flags.drive_number-128)].log_drive_num_type[index]=0;

      part_table[(flags.drive_number-128)].log_drive_start_cyl[index]=0;
      part_table[(flags.drive_number-128)].log_drive_start_head[index]=0;
      part_table[(flags.drive_number-128)].log_drive_start_sect[index]=0;

      part_table[(flags.drive_number-128)].log_drive_end_cyl[index]=0;
      part_table[(flags.drive_number-128)].log_drive_end_head[index]=0;
      part_table[(flags.drive_number-128)].log_drive_end_sect[index]=0;

      part_table[(flags.drive_number-128)].log_drive_rel_sect[index]=0;
      part_table[(flags.drive_number-128)].log_drive_num_sect[index]=0;

      part_table[(flags.drive_number-128)].log_drive_size_in_MB[index]=0;
      part_table[(flags.drive_number-128)].log_drive_created[index]=FALSE;

      part_table[(flags.drive_number-128)].next_ext_exists[index]=FALSE;

      part_table[(flags.drive_number-128)].next_ext_num_type[index]=0;

      part_table[(flags.drive_number-128)].next_ext_start_cyl[index]=0;
      part_table[(flags.drive_number-128)].next_ext_start_head[index]=0;
      part_table[(flags.drive_number-128)].next_ext_start_sect[index]=0;

      part_table[(flags.drive_number-128)].next_ext_end_cyl[index]=0;
      part_table[(flags.drive_number-128)].next_ext_end_head[index]=0;
      part_table[(flags.drive_number-128)].next_ext_end_sect[index]=0;

      part_table[(flags.drive_number-128)].next_ext_rel_sect[index]=0;
      part_table[(flags.drive_number-128)].next_ext_num_sect[index]=0;

      index++;
      }while(index<24);
    }

  part_table[(flags.drive_number-128)].part_values_changed=TRUE;
  flags.partitions_have_changed=TRUE;
}

/* Delete Logical Drive Interface */
int Delete_Logical_Drive_Interface()
{
  int drive_to_delete=0;
  int index=0;
  int input=0;
  int input_ok;

  Clear_Screen(NULL);

  Print_Centered(1,"Delete Logical DOS Drive(s) in the Extended DOS Partition",BOLD);

  Display_Extended_Partition_Information_SS();

  Position_Cursor(4,19);
  if(flags.monochrome!=TRUE) textcolor(WHITE | BLINK);
  cprintf("WARNING!");
  if(flags.monochrome!=TRUE) textcolor(15);

  printf(" Data in a deleted Logical DOS Drive will be lost.");
  Position_Cursor(4,20);
  printf ("What drive do you want to delete...............................? ");

  Determine_Drive_Letters();

  //char drive_lettering_buffer[8] [27];   this line is for reference
  /* Place code to find the min and max drive letter here. */


  input_ok=FALSE;

  do
    {
    flags.esc=FALSE;
    input=Input(1,69,20,CHAR,68,90,ESCR,0,0);
    /* Note:  min_range and max_range will need adjusted!!!!! */
    /* Changes will have to be made because the first logical drive letter */
    /* on the selected drive may not be D:, the drive letters on the       */
    /* drive may not be sequential.                                        */

    if(flags.esc==TRUE) return(1);

    /* Ensure that the entered character is legitimate. */
    index=4;
    do
      {
      if( (drive_lettering_buffer[(flags.drive_number-128)] [index]>0) && (drive_lettering_buffer[(flags.drive_number-128)] [index]==input) )
	{
	input=index-4;
	input_ok=TRUE;
	index=30; /* break out of the loop */
	}

      index++;
      }while(index<=26);

    }while(input_ok==FALSE);

  drive_to_delete=input;

  Position_Cursor(4,22);
  printf("Are you sure (Y/N)..............................? ");
  flags.esc=FALSE;
  input=Input(1,54,22,YN,0,0,ESCR,0,0);

  if( (input==TRUE) && (flags.esc==FALSE) )
    {
    Delete_Logical_Drive(drive_to_delete);

    Clear_Screen(NULL);
    Print_Centered(1,"Delete Logical DOS Drive(s) in the Extended DOS Partition",BOLD);
    Display_Extended_Partition_Information_SS();
    input=Input(0,0,0,ESC,0,0,ESCC,0,0);
    }

  return(0);
}

/* Delete Non-DOS Partition User Interface */
void Delete_N_DOS_Partition_Interface()
{
  int input=0;

  Clear_Screen(NULL);
  Print_Centered(4,"Delete Non-DOS Partition",BOLD);

  Display_Primary_Partition_Information_SS();

  Position_Cursor(4,18);
  if(flags.monochrome!=TRUE) textcolor(WHITE | BLINK);
  cprintf("WARNING!");
  if(flags.monochrome!=TRUE) textcolor(15);

  printf(" Data in the deleted Non-DOS Partition will be lost.");
  Position_Cursor(4,19);
  printf("What Non-DOS Partition do you want to delete..? ");

  flags.esc=FALSE;
  input=Input(1,52,19,NUM,1,4,ESCR,-1,0); /* 4 needs changed to the max num of partitions */

  if(flags.esc==FALSE)
    {
    Delete_Primary_Partition(input-1);

    Position_Cursor(4,21);
    cprintf("Non-DOS Partition deleted");
    Position_Cursor(4,24);
    printf("                                    ");

    Input(0,0,0,ESC,0,0,ESCC,0,0);
    }
}

/* Delete Primary Partition */
void Delete_Primary_Partition(int partition_number)
{
  int index;

  /* If partition_number is an extended partition, first delete the */
  /* extended partition.                                            */
  if( (part_table[(flags.drive_number-128)].pri_part_num_type[partition_number]==5)
    || (part_table[(flags.drive_number-128)].pri_part_num_type[partition_number]==0x0f) )
    {
    part_table[(flags.drive_number-128)].ext_part_exists=FALSE;
    part_table[(flags.drive_number-128)].num_of_ext_part=0;

    part_table[(flags.drive_number-128)].ext_part_size_in_MB=0;
    part_table[(flags.drive_number-128)].ext_part_num_sect=0;
    part_table[(flags.drive_number-128)].ext_part_largest_free_space=0;

    part_table[(flags.drive_number-128)].log_drive_largest_free_space_location=0;
    part_table[(flags.drive_number-128)].log_drive_free_space_start_cyl=0;
    part_table[(flags.drive_number-128)].log_drive_free_space_end_cyl=0;

    part_table[(flags.drive_number-128)].num_of_log_drives=0;

    index=0;
    do
      {
      part_table[(flags.drive_number-128)].log_drive_num_type[index]=0;

      strcpy(part_table[(flags.drive_number-128)].log_drive_vol_label[index],"            ");;

      part_table[(flags.drive_number-128)].log_drive_start_cyl[index]=0;
      part_table[(flags.drive_number-128)].log_drive_start_head[index]=0;
      part_table[(flags.drive_number-128)].log_drive_start_sect[index]=0;

      part_table[(flags.drive_number-128)].log_drive_end_cyl[index]=0;
      part_table[(flags.drive_number-128)].log_drive_end_head[index]=0;
      part_table[(flags.drive_number-128)].log_drive_end_sect[index]=0;

      part_table[(flags.drive_number-128)].log_drive_rel_sect[index]=0;
      part_table[(flags.drive_number-128)].log_drive_num_sect[index]=0;

      part_table[(flags.drive_number-128)].log_drive_size_in_MB[index]=0;

      part_table[(flags.drive_number-128)].next_ext_exists[index]=0;

      part_table[(flags.drive_number-128)].next_ext_num_type[index]=0;

      part_table[(flags.drive_number-128)].next_ext_start_cyl[index]=0;
      part_table[(flags.drive_number-128)].next_ext_start_head[index]=0;
      part_table[(flags.drive_number-128)].next_ext_start_sect[index]=0;

      part_table[(flags.drive_number-128)].next_ext_end_cyl[index]=0;
      part_table[(flags.drive_number-128)].next_ext_end_head[index]=0;
      part_table[(flags.drive_number-128)].next_ext_end_sect[index]=0;

      part_table[(flags.drive_number-128)].next_ext_rel_sect[index]=0;
      part_table[(flags.drive_number-128)].next_ext_num_sect[index]=0;

      index++;
      }while(index<23);
    }

  part_table[(flags.drive_number-128)].active_status[partition_number]=0;
  part_table[(flags.drive_number-128)].pri_part_num_type[partition_number]=0;

  part_table[(flags.drive_number-128)].pri_part_start_cyl[partition_number]=0;
  part_table[(flags.drive_number-128)].pri_part_start_head[partition_number]=0;
  part_table[(flags.drive_number-128)].pri_part_start_sect[partition_number]=0;

  part_table[(flags.drive_number-128)].pri_part_end_cyl[partition_number]=0;
  part_table[(flags.drive_number-128)].pri_part_end_head[partition_number]=0;
  part_table[(flags.drive_number-128)].pri_part_end_sect[partition_number]=0;

  part_table[(flags.drive_number-128)].pri_part_rel_sect[partition_number]=0;
  part_table[(flags.drive_number-128)].pri_part_num_sect[partition_number]=0;

  part_table[(flags.drive_number-128)].pri_part_size_in_MB[partition_number]=0;

  part_table[(flags.drive_number-128)].part_values_changed=TRUE;
  flags.partitions_have_changed=TRUE;

  /* Delete volume label from buffer */
  strcpy(part_table[(flags.drive_number-128)].pri_part_vol_label[partition_number],"           ");
}

/* Delete Primary DOS Partition Interface */
void Delete_Primary_DOS_Partition_Interface()
{
  int input=0;
  int partition_to_delete;

  Clear_Screen(NULL);

  Print_Centered(4,"Delete Primary DOS Partition",BOLD);
  Display_Primary_Partition_Information_SS();

  Position_Cursor(4,19);
  if(flags.monochrome!=TRUE) textcolor(WHITE | BLINK);
  cprintf("WARNING!");
  if(flags.monochrome!=TRUE) textcolor(15);

  printf(" Data in the deleted Primary DOS Partition will be lost.");
  Position_Cursor(4,20);
  printf("What primary partition do you want to delete..? ");

  flags.esc=FALSE;
  input=Input(1,52,20,NUM,1,4,ESCR,-1,0); /* 4 needs changed to the max num of partitions */

  if(flags.esc==FALSE)
    {
    partition_to_delete=input-1;

    Position_Cursor(4,22);
    printf("Are you sure (Y/N)..............................? ");
    flags.esc=FALSE;
    input=Input(1,54,22,YN,0,0,ESCR,0,0);

    if( (input==TRUE) && (flags.esc==FALSE) )
      {
      Delete_Primary_Partition(partition_to_delete);

      Clear_Screen(NULL);

      Print_Centered(4,"Delete Primary DOS Partition",BOLD);
      Display_Primary_Partition_Information_SS();
      Position_Cursor(4,21);
      cprintf("Primary DOS Partition deleted");

      Input(0,0,0,ESC,0,0,ESCC,0,0);
      }
    }
}

/* Determine drive letters */
int Determine_Drive_Letters()
/* Normally returns a 0, returns a 1 if current_letter>='Z' */
{
  int current_letter='C';
  int drive_found=FALSE;
  int index=0;
  int sub_index=0;

  int first=FALSE;

  Load_Brief_Partition_Table();

  /* Clear drive_lettering_buffer[8] [27] */
  do
    {
    sub_index=0;
    do
      {
      drive_lettering_buffer[index] [sub_index]=0;

      sub_index++;
      }while(sub_index<27);

    index++;
    }while(index<8);

  /* Begin placement of drive letters */

  /* Start with the primary partitions... */
  index=0;
  do
    {
    sub_index=0;
    drive_found=FALSE;
    do
      {
      if( (brief_partition_table[index] [sub_index]==1) || (brief_partition_table[index] [sub_index]==4) || (brief_partition_table[index] [sub_index]==6) )
	{
	drive_lettering_buffer[index] [sub_index]=current_letter;
	current_letter++;
	drive_found=TRUE;
	}

      if( (brief_partition_table[index] [sub_index]==0x0e) && ( (flags.version==W95) || (flags.version==W95B) || (flags.version==W98) ) )
	{
	drive_lettering_buffer[index] [sub_index]=current_letter;
	current_letter++;
	drive_found=TRUE;
	}

      if( ( (brief_partition_table[index] [sub_index]==0x0b) || (brief_partition_table[index] [sub_index]==0x0c) ) && ( (flags.version==W95B) || (flags.version==W98) ) )
	{
	drive_lettering_buffer[index] [sub_index]=current_letter;
	current_letter++;
	drive_found=TRUE;
	}

      sub_index++;
      }while( (sub_index<4) && (drive_found==FALSE) );

    index++;
    }while(index<8);

  /* Next examine the extended partitions... */
  index=0;
  do
    {
    sub_index=4;
    drive_found=FALSE;
    do
      {
      if( (brief_partition_table[index] [sub_index]==1) || (brief_partition_table[index] [sub_index]==4) || (brief_partition_table[index] [sub_index]==6) )
	{
	drive_lettering_buffer[index] [sub_index]=current_letter;
	current_letter++;
	}

      if( (brief_partition_table[index] [sub_index]==0x0e) && ( (flags.version==W95) || (flags.version==W95B) || (flags.version==W98) ) )
	{
	drive_lettering_buffer[index] [sub_index]=current_letter;
	current_letter++;
	}

      if( ( (brief_partition_table[index] [sub_index]==0x0b) || (brief_partition_table[index] [sub_index]==0x0c) ) && ( (flags.version==W95B) || (flags.version==W98) ) )
	{
	drive_lettering_buffer[index] [sub_index]=current_letter;
	current_letter++;
	}

      sub_index++;
      }while(sub_index<27);

    index++;
    }while(index<8);

  /* Return to the primary partitions... */
  index=0;
  do
    {
    sub_index=0;
    first=FALSE;
    do
      {
      if( (first==TRUE) && ( (brief_partition_table[index] [sub_index]==1) || (brief_partition_table[index] [sub_index]==4) || (brief_partition_table[index] [sub_index]==6) ) )
	{
	drive_lettering_buffer[index] [sub_index]=current_letter;
	current_letter++;
	}

      if( (first==TRUE) && (brief_partition_table[index] [sub_index]==0x0e) && ( (flags.version==W95) || (flags.version==W95B) || (flags.version==W98) ) )
	{
	drive_lettering_buffer[index] [sub_index]=current_letter;
	current_letter++;
	}

      if( (first==TRUE) && ( (brief_partition_table[index] [sub_index]==0x0b) || (brief_partition_table[index] [sub_index]==0x0c) ) && ( (flags.version==W95B) || (flags.version==W98) ) )
	{
	drive_lettering_buffer[index] [sub_index]=current_letter;
	current_letter++;
	}

      if( (first==FALSE) && ( (brief_partition_table[index] [sub_index]==1) || (brief_partition_table[index] [sub_index]==4) || (brief_partition_table[index] [sub_index]==6) ) )
	{
	first=TRUE;
	}

      if( (first==FALSE) && (brief_partition_table[index] [sub_index]==0x0e) && ( (flags.version==W95) || (flags.version==W95B) || (flags.version==W98) ) )
	{
	first=TRUE;
	}

      if( (first==FALSE) && ( (brief_partition_table[index] [sub_index]==0x0b) || (brief_partition_table[index] [sub_index]==0x0c) ) && ( (flags.version==W95B) || (flags.version==W98) ) )
	{
	first=TRUE;
	}


      sub_index++;
      }while(sub_index<4);

    index++;
    }while(index<8);

  if(current_letter>'Z') return(1);
  else return(0);
}

/* Determine if the video display will support boldfacing text */
void Determine_Color_Video_Support()
{
  asm{
    mov bx,es:0x10
    and bx,0x30
    cmp bx,0x30
    jne color
    }

  flags.monochrome=TRUE;
  textcolor(7);
  goto type_determined;

  color:
  flags.monochrome=FALSE;
  textcolor(15);

  type_determined:
}

/* Determine the locations of free space in the partition table */
void Determine_Free_Space()
{
  int first_used_partition=UNUSED;
  int last_used_partition=UNUSED;
  int index;
  int sub_index;
  int swap;

  int drive=flags.drive_number-0x80;

  unsigned long free_space_after_last_used_partition=0;
  unsigned long free_space_before_first_used_partition=0;
  unsigned long free_space_between_partitions_0_and_1=0;
  unsigned long free_space_between_partitions_1_and_2=0;
  unsigned long free_space_between_partitions_2_and_3=0;

  unsigned long cylinder_size=(part_table[drive].total_head+1)*(part_table[drive].total_sect);

  /* Reset the physical order to default */
  index=0;
  do
    {
    part_table[drive].pri_part_physical_order[index]=index;

    index++;
    }while(index<4);

  /* Determine the location and size of the largest free space in the */
  /* primary partition.                                               */

  /* 1.  Sort the primary partitions based upon starting cylinder and their*/
  /*     contents...or lack of use.                                        */

  /* Place all empty partitions at the end of the table. */
  index=0;
  do
    {

    sub_index=0;
    do
      {
      if(part_table[drive].pri_part_num_type[part_table[drive].pri_part_physical_order[sub_index]]==0)
	{
	swap=part_table[drive].pri_part_physical_order[sub_index];
	part_table[drive].pri_part_physical_order[sub_index]=part_table[drive].pri_part_physical_order[(sub_index+1)];
	part_table[drive].pri_part_physical_order[(sub_index+1)]=swap;
	}
      sub_index++;
      }while(sub_index<3);

    index++;
    }while(index<4);

  if(debug.determine_free_space==TRUE)
    {
    Clear_Screen(NULL);
    Print_Centered(0,"Determine_Free_Space(int drive) debugging screen 1",BOLD);

    printf("\n\nCylinder Size (total heads * total sectors)=%d\n",cylinder_size);

    printf("\nContents after initial sorting of unused partitions to end:\n\n");

    index=0;
    do
      {
      printf("Partition %1d:  %1d    ",index,part_table[drive].pri_part_physical_order[index]);
      printf("SC:  %4d    ",part_table[drive].pri_part_start_cyl[part_table[drive].pri_part_physical_order[index]]);
      printf("EC:  %4d    ",part_table[drive].pri_part_end_cyl[part_table[drive].pri_part_physical_order[index]]);
      printf("Size in MB:  %4d\n",part_table[drive].pri_part_size_in_MB[part_table[drive].pri_part_physical_order[index]]);

      index++;
      }while(index<4);
    Position_Cursor(4,20);
    Pause();
    }

  /* Order the partitions based upon starting cylinder */
  index=0;
  do
    {
    sub_index=0;
    do
      {
      if( (part_table[drive].pri_part_num_type
       [part_table[drive].pri_part_physical_order
       [sub_index]]!=0) && (part_table[drive]
       .pri_part_num_type[part_table[drive]
       .pri_part_physical_order[(sub_index+1)]]!=0)
       && (part_table[drive].pri_part_start_cyl
       [part_table[drive].pri_part_physical_order[sub_index]]
       >part_table[drive].pri_part_start_cyl
       [part_table[drive].pri_part_physical_order[(sub_index+1)]]) )
	{
	swap=part_table[drive].pri_part_physical_order[sub_index];
	part_table[drive].pri_part_physical_order[sub_index]=part_table[drive].pri_part_physical_order[(sub_index+1)];
	part_table[drive].pri_part_physical_order[(sub_index+1)]=swap;
	}
      sub_index++;
      }while(sub_index<3);
    index++;
    }while(index<4);

  if(debug.determine_free_space==TRUE)
    {
    Clear_Screen(NULL);
    Print_Centered(0,"Determine_Free_Space(int drive) debugging screen 2",BOLD);

    printf("\n\nCylinder Size (total heads * total sectors)=%d\n",cylinder_size);
    printf("\nContents after sorting partitions by starting cylinder:\n\n");

    index=0;
    do
      {
      printf("Partition %d:  %1d    ",index,part_table[drive].pri_part_physical_order[index]);
      printf("SC:  %4d    ",part_table[drive].pri_part_start_cyl[part_table[drive].pri_part_physical_order[index]]);
      printf("EC:  %4d    ",part_table[drive].pri_part_end_cyl[part_table[drive].pri_part_physical_order[index]]);
      printf("Size in MB:  %4d\n",part_table[drive].pri_part_size_in_MB[part_table[drive].pri_part_physical_order[index]]);

      index++;
      }while(index<4);
    }

  /* 2.  Is there any free space before the first partition? */

  /* Find the first used partition and the last used partition. */
  index=0;
  do
    {
    if( (first_used_partition==UNUSED)
     && (part_table[drive].pri_part_num_type[part_table[drive]
     .pri_part_physical_order[index]]>0) )
      {
      first_used_partition=index;
      }

    if(part_table[drive].pri_part_num_type
     [part_table[drive].pri_part_physical_order[index]]>0)
      {
      last_used_partition=index;
      }

    index++;
    }while(index<4);

  if(first_used_partition!=UNUSED)
    {
    if(part_table[drive].pri_part_start_cyl
     [part_table[drive].pri_part_physical_order[first_used_partition]]>0)
      {
      free_space_before_first_used_partition
       =(part_table[drive].pri_part_start_cyl
       [part_table[drive].pri_part_physical_order[first_used_partition]]);
      }
    else free_space_before_first_used_partition=0;
    }

  /* 3.  Is there any free space after the last used partition? */
  if(first_used_partition!=UNUSED)
    {
    if(part_table[drive].pri_part_end_cyl
     [part_table[drive].pri_part_physical_order[last_used_partition]]<=part_table[drive].total_cyl)
      {
      free_space_after_last_used_partition
       =(part_table[drive].total_cyl-part_table[drive].pri_part_end_cyl
       [part_table[drive].pri_part_physical_order[last_used_partition]]);
      }
    }

  /* 4.  Is there any free space between partitions?                    */
  /*                                                                    */
  if( (first_used_partition!=UNUSED) && (last_used_partition>=1) )
    {
    if( (part_table[drive].pri_part_end_cyl
     [part_table[drive].pri_part_physical_order[0]]+1)
     <(part_table[drive].pri_part_start_cyl[part_table[drive]
     .pri_part_physical_order[1]]) )
      {
      free_space_between_partitions_0_and_1
       =(part_table[drive].pri_part_start_cyl
       [part_table[drive].pri_part_physical_order[1]]
       -part_table[drive].pri_part_end_cyl
       [part_table[drive].pri_part_physical_order[0]]);
      }
    }

  if( (first_used_partition!=UNUSED) && (last_used_partition>=2) )
    {
    if( (part_table[drive].pri_part_end_cyl[part_table[drive].pri_part_physical_order[1]]+1)<(part_table[drive].pri_part_start_cyl[part_table[drive].pri_part_physical_order[2]]) )
      {
      free_space_between_partitions_1_and_2=(part_table[drive].pri_part_start_cyl[part_table[drive].pri_part_physical_order[2]]-part_table[drive].pri_part_end_cyl[part_table[drive].pri_part_physical_order[1]]);
      }
    }

  if( (first_used_partition!=UNUSED) && (last_used_partition==3) )
    {
    /* */
    if( (part_table[drive].pri_part_end_cyl[part_table[drive].pri_part_physical_order[2]]+1)<(part_table[drive].pri_part_start_cyl[part_table[drive].pri_part_physical_order[3]]) )
      {
      free_space_between_partitions_2_and_3=(part_table[drive].pri_part_start_cyl[part_table[drive].pri_part_physical_order[3]]-part_table[drive].pri_part_end_cyl[part_table[drive].pri_part_physical_order[2]]);
      }
    }

  /* Locate the largest free space */
  if(first_used_partition!=UNUSED)
    {
    /* */
    part_table[drive].pp_largest_free_space_start_head=0;
    part_table[drive].pp_largest_free_space_start_sect=1;

    /* Default the largest free space to before the first used partition */
    part_table[drive].pri_part_largest_free_space=free_space_before_first_used_partition;
    part_table[drive].pp_largest_free_space_start_cyl=0;
    part_table[drive].pp_largest_free_space_end_cyl
     =part_table[drive].pri_part_start_cyl
     [part_table[drive].pri_part_physical_order[first_used_partition]]-1;

    /* If the largest free space is not before the first used partition  */
    /* make the correct adjustments.                                     */
    if(free_space_after_last_used_partition>part_table[drive].pri_part_largest_free_space)
      {
      part_table[drive].pri_part_largest_free_space=free_space_after_last_used_partition;
      part_table[drive].pp_largest_free_space_start_cyl
       =part_table[drive].pri_part_end_cyl
       [part_table[drive].pri_part_physical_order[last_used_partition]]+1;
      part_table[drive].pp_largest_free_space_end_cyl=part_table[drive].total_cyl;
      }

    if(free_space_between_partitions_0_and_1>part_table[drive].pri_part_largest_free_space)
      {
      part_table[drive].pri_part_largest_free_space=free_space_between_partitions_0_and_1;
      part_table[drive].pp_largest_free_space_start_cyl=part_table[drive].pri_part_end_cyl[part_table[drive].pri_part_physical_order[0]]+1;
      part_table[drive].pp_largest_free_space_end_cyl=part_table[drive].pri_part_start_cyl[part_table[drive].pri_part_physical_order[1]]-1;
      }

    if(free_space_between_partitions_1_and_2>part_table[drive].pri_part_largest_free_space)
      {
      part_table[drive].pri_part_largest_free_space=free_space_between_partitions_1_and_2;
      part_table[drive].pp_largest_free_space_start_cyl=part_table[drive].pri_part_end_cyl[part_table[drive].pri_part_physical_order[1]]+1;
      part_table[drive].pp_largest_free_space_end_cyl=part_table[drive].pri_part_start_cyl[part_table[drive].pri_part_physical_order[2]]-1;
      }

    if(free_space_between_partitions_2_and_3>part_table[drive].pri_part_largest_free_space)
      {
      part_table[drive].pri_part_largest_free_space=free_space_between_partitions_2_and_3;
      part_table[drive].pp_largest_free_space_start_cyl=part_table[drive].pri_part_end_cyl[part_table[drive].pri_part_physical_order[2]]+1;
      part_table[drive].pp_largest_free_space_end_cyl=part_table[drive].pri_part_start_cyl[part_table[drive].pri_part_physical_order[3]]-1;
      }
    }
  else
    {
    part_table[drive].pri_part_largest_free_space=part_table[drive].total_cyl;
    part_table[drive].pp_largest_free_space_start_cyl=0;
    part_table[drive].pp_largest_free_space_end_cyl=part_table[drive].total_cyl-1;
    }

  if(part_table[drive].pri_part_largest_free_space<=0) part_table[drive].pri_part_largest_free_space=0;

  if(debug.determine_free_space==TRUE)
    {
    printf("\n\nFree space (in cylinders) in primary partition table:\n");
    printf("\nFree space before first used partition:  %d",free_space_before_first_used_partition);
    printf("\nFree space after last used partition:  %d",free_space_after_last_used_partition);
    printf("\nFree space between partitions 0 and 1:  %d",free_space_between_partitions_0_and_1);
    printf("\nFree space between partitions 1 and 2:  %d",free_space_between_partitions_1_and_2);
    printf("\nFree space between partitions 2 and 3:  %d",free_space_between_partitions_2_and_3);
    printf("\n\nLargest free space in primary partition table:  %d",part_table[drive].pri_part_largest_free_space);
    printf("\nStarting cylinder of largest free space:  %d",part_table[drive].pp_largest_free_space_start_cyl);
    printf("\nEnding cylinder of largest free space:  %d",part_table[drive].pp_largest_free_space_end_cyl);
    Pause();
    }

  /* Determine the location and size of the largest free space in the */
  /* extended partition, if it exists.                                */
  if(part_table[drive].ext_part_exists==TRUE)
    {
    part_table[drive].ext_part_largest_free_space=0;
    part_table[drive].log_drive_free_space_start_cyl=0;
    part_table[drive].log_drive_free_space_end_cyl=0;
    part_table[drive].log_drive_largest_free_space_location=0;

    if(debug.determine_free_space==TRUE)
      {
      Clear_Screen(NULL);
      Print_Centered(0,"Determine_Free_Space(int drive) debugging screen 3",BOLD);
      printf("\n\n");
      }

    if(part_table[drive].num_of_log_drives>0)
      {
      /* If there are logical drives in the extended partition first find  */
      /* the largest free space between the logical drives...if it exists. */
      last_used_partition=UNUSED;
      index=0;

      /* Check to see if the first possible entry in the extended partition */
      /* is unused.  If it is unused and there is a logical drive after it  */
      /* then skip checking for free space between entry 0 and 1.           */
      if(part_table[drive].log_drive_num_type[0]==0) index=1;

      do
	{
	if(part_table[drive].log_drive_num_type[index]>0)
	  {
	  last_used_partition=index;
	  }

	if( (part_table[drive].log_drive_start_cyl[(index+1)]-part_table[drive].log_drive_end_cyl[index]) >1 )
	  {
	  part_table[drive].ext_part_largest_free_space=(part_table[drive].log_drive_start_cyl[(index+1)]-1)-(part_table[drive].log_drive_end_cyl[index]+1);
	  part_table[drive].log_drive_free_space_start_cyl=part_table[drive].log_drive_end_cyl[index]+1;
	  part_table[drive].log_drive_free_space_end_cyl=part_table[drive].log_drive_start_cyl[(index+1)]-1;
	  part_table[drive].log_drive_largest_free_space_location=index+1;
	  }

	if(debug.determine_free_space==TRUE)
	  {
	  if(index==12)
	    {
	    printf("\n");
	    Pause();
	    }

	  printf("\nLogical Drive #: %2d    ",index);
	  printf("SC: %4d    ",part_table[drive].log_drive_start_cyl[index]);
	  printf("EC: %4d    ",part_table[drive].log_drive_end_cyl[index]);
	  }

	index++;
	}while(index<22);

      if(debug.determine_free_space==TRUE)
	{
	printf("\nLogical Drive #: %2d    ",index);
	printf("SC: %4d    ",part_table[drive].log_drive_start_cyl[22]);
	printf("EC: %4d    \n",part_table[drive].log_drive_end_cyl[22]);
	Pause();

	Clear_Screen(NULL);
	Print_Centered(0,"Determine_Free_Space(int drive) debugging screen 4",BOLD);
	printf("\n\nNote:  All values are in cylinders.\n\n");
	printf("Results of free space calculations after computing spaces between\n  logical drives:\n\n");
	printf("Location of largest free space:  %d\n",part_table[drive].log_drive_largest_free_space_location);
	printf("Size of largest free space:  %4d\n",part_table[drive].ext_part_largest_free_space);
	printf("Starting cylinder of largest free space:  %d\n",part_table[drive].log_drive_free_space_start_cyl);
	printf("Ending cylinder of largest free space:  %d\n",part_table[drive].log_drive_free_space_end_cyl);
	Pause();
	}

      /* Determine if there is any free space before the first logical      */
      /* drive in the extended partition.                                   */
      if(part_table[drive].log_drive_num_type[0]!=0)
	{
	if( (part_table[drive].log_drive_start_cyl[0]-part_table[drive].pri_part_start_cyl[part_table[drive].num_of_ext_part])>part_table[drive].ext_part_largest_free_space)
	  {
	  part_table[drive].ext_part_largest_free_space=(part_table[drive].log_drive_start_cyl[0]-part_table[drive].pri_part_start_cyl[part_table[drive].num_of_ext_part]);
	  part_table[drive].log_drive_free_space_start_cyl=part_table[drive].pri_part_start_cyl[part_table[drive].num_of_ext_part];
	  part_table[drive].log_drive_free_space_end_cyl=part_table[drive].log_drive_start_cyl[0]-1;
	  part_table[drive].log_drive_largest_free_space_location=0;
	  }
	}
      else
	{
	if( (part_table[drive].log_drive_start_cyl[1]-part_table[drive].pri_part_start_cyl[part_table[drive].num_of_ext_part])>part_table[drive].ext_part_largest_free_space)
	  {
	  part_table[drive].ext_part_largest_free_space=(part_table[drive].log_drive_start_cyl[1]-part_table[drive].pri_part_start_cyl[part_table[drive].num_of_ext_part]);
	  part_table[drive].log_drive_free_space_start_cyl=part_table[drive].pri_part_start_cyl[part_table[drive].num_of_ext_part];
	  part_table[drive].log_drive_free_space_end_cyl=part_table[drive].log_drive_start_cyl[1]-1;
	  part_table[drive].log_drive_largest_free_space_location=0;
	  }
	}

      if(debug.determine_free_space==TRUE)
	{
	Clear_Screen(NULL);
	Print_Centered(0,"Determine_Free_Space(int drive) debugging screen 5",BOLD);
	printf("\n\nResults of free space calculations after computing space before\n  the first logical drive:\n\n");
	printf("Location of largest free space:  %d\n",part_table[drive].log_drive_largest_free_space_location);
	printf("Size of largest free space:  %4d\n",part_table[drive].ext_part_largest_free_space);
	printf("Starting cylinder of largest free space:  %d\n",part_table[drive].log_drive_free_space_start_cyl);
	printf("Ending cylinder of largest free space:  %d\n",part_table[drive].log_drive_free_space_end_cyl);
	}

      /* Determine if there is any free space after the last logical drive  */
      /* in the extended partition.                                         */
      if( (last_used_partition<23) && (part_table[drive].log_drive_end_cyl[last_used_partition]<part_table[drive].pri_part_end_cyl[part_table[drive].num_of_ext_part]) )
	{
	if( (part_table[drive].pri_part_end_cyl[part_table[drive].num_of_ext_part]-part_table[drive].log_drive_end_cyl[last_used_partition])>(part_table[drive].ext_part_largest_free_space) )
	  {
	  part_table[drive].ext_part_largest_free_space=(part_table[drive].pri_part_end_cyl[part_table[drive].num_of_ext_part]-part_table[drive].log_drive_end_cyl[last_used_partition]);
	  part_table[drive].log_drive_free_space_start_cyl=part_table[drive].log_drive_end_cyl[last_used_partition]+1;
	  part_table[drive].log_drive_free_space_end_cyl=part_table[drive].pri_part_end_cyl[part_table[drive].num_of_ext_part]-1;
	  part_table[drive].log_drive_largest_free_space_location=last_used_partition+1;
	  }
	}

      if(debug.determine_free_space==TRUE)
	{
	printf("\n\nResults of free space calculations after computing space after\n  the last logical drive:\n\n");
	printf("Location of largest free space:  %d\n",part_table[drive].log_drive_largest_free_space_location);
	printf("Size of largest free space:  %4d\n",part_table[drive].ext_part_largest_free_space);
	printf("Starting cylinder of largest free space:  %d\n",part_table[drive].log_drive_free_space_start_cyl);
	printf("Ending cylinder of largest free space:  %d\n",part_table[drive].log_drive_free_space_end_cyl);
	Pause();
	}
      }
    else
      {
      /* If the extended partition is empty. */
      part_table[drive].ext_part_largest_free_space=part_table[drive].pri_part_end_cyl[part_table[drive].num_of_ext_part]-part_table[drive].pri_part_start_cyl[part_table[drive].num_of_ext_part]-1;

      part_table[drive].log_drive_free_space_start_cyl=part_table[drive].pri_part_start_cyl[part_table[drive].num_of_ext_part];
      part_table[drive].log_drive_free_space_end_cyl=part_table[drive].pri_part_end_cyl[part_table[drive].num_of_ext_part];

      if(debug.determine_free_space==TRUE)
	{
	printf("\n\nThere are not any Logical DOS Drives in the Extended DOS Partition\n\n");
	printf("Location of largest free space:     0\n");
	printf("Size of largest free space:  %d\n",part_table[drive].ext_part_largest_free_space);
	printf("Starting cylinder of largest free space:  %d\n",part_table[drive].log_drive_free_space_start_cyl);
	printf("Ending cylinder of largest free space:  %d\n",part_table[drive].log_drive_free_space_end_cyl);
	Pause();
	}
      }
    }
}

/* Display information for all hard drives */
void Display_All_Drives()
{
  int current_column_offset_of_general_drive_information;
  int current_column_offset=4;
  int current_line=3;
  int current_line_of_general_drive_information;
  int drive=1;
  int drive_letter_index=0;
  int index;

  long space_used_on_drive_in_MB;

  unsigned long usage;

  Determine_Drive_Letters();

  Position_Cursor(2,2);
  printf("Disk   Drv   Mbytes   Free   Usage");

  do
    {
    if(current_line>18)
      {
      current_line=3;
      current_column_offset=45;

      Position_Cursor(43,2);
      printf("Disk   Drv   Mbytes   Free   Usage");
      }

    /* Print physical drive information */
    current_column_offset_of_general_drive_information=current_column_offset;
    current_line_of_general_drive_information=current_line;
    space_used_on_drive_in_MB=0;

    /* Print drive number */
    Position_Cursor(current_column_offset_of_general_drive_information,current_line);
    cprintf("%d",drive);

    /* Print size of drive */
    Position_Cursor((current_column_offset_of_general_drive_information+12),current_line);
    printf("%4d",part_table[(drive-1)].total_hard_disk_size_in_MB);

    /* Get space_used_on_drive_in_MB */
    index=0;
    do
      {
      if( (part_table[(drive-1)].pri_part_num_type[index]!=5)
       && (part_table[(drive-1)].pri_part_num_type[index]!=15)
       && (part_table[(drive-1)].pri_part_num_type[index]!=0) )
       space_used_on_drive_in_MB
       =space_used_on_drive_in_MB
       +part_table[(drive-1)].pri_part_size_in_MB[index];

      index++;
      }while(index<=3);

    index=0;
    do
      {
      if(part_table[(drive-1)].log_drive_num_type[index]>0)
       space_used_on_drive_in_MB
       =space_used_on_drive_in_MB
       +part_table[(drive-1)].log_drive_size_in_MB[index];

      index++;
      }while(index<=22);

    /* Print logical drives on disk, if applicable */

    drive_letter_index=0;
    do
      {
      if(drive_lettering_buffer[(drive-1)] [drive_letter_index]>0)
	{
	current_line++;

	if(current_line>18)
	  {
	  current_line=3;
	  current_column_offset=45;

	  Position_Cursor(43,2);
	  printf("Disk   Drv   Mbytes   Free   Usage");
	  }

	/* Print drive letter of logical drive */
	Position_Cursor((current_column_offset+6),current_line);
	printf("%c",drive_lettering_buffer[(drive-1)] [drive_letter_index]);
	Position_Cursor((current_column_offset+7),current_line);
	printf(":");

	/* Print size of logical drive */
	Position_Cursor((current_column_offset+12),current_line);

	if(drive_letter_index<4)
	  {
	  printf("%4d",part_table[(drive-1)].pri_part_size_in_MB[drive_letter_index]);
	  }
	else
	  {
	  printf("%4d",part_table[(drive-1)].log_drive_size_in_MB[(drive_letter_index-4)]);
	  }
	}

      drive_letter_index++;
      }while(drive_letter_index<27);

    /* Print amount of free space on drive */
    if( (part_table[(drive-1)].total_hard_disk_size_in_MB-space_used_on_drive_in_MB)>0)
      {
      Position_Cursor((current_column_offset_of_general_drive_information+20),current_line_of_general_drive_information);
      printf("%4d",(part_table[(drive-1)].total_hard_disk_size_in_MB-space_used_on_drive_in_MB));
      }

    /* Print drive usage percentage */
    if(space_used_on_drive_in_MB==0) usage=0;
    else usage=1+((100*space_used_on_drive_in_MB)/(part_table[(drive-1)].total_hard_disk_size_in_MB));
    if(usage>100) usage=100;

    Position_Cursor((current_column_offset_of_general_drive_information+28),current_line_of_general_drive_information);
    printf("%3d%%",usage);

    current_line++;
    drive++;
    }while(drive<=(flags.maximum_drive_number-127));

  Position_Cursor(4,20);
  printf("(1 Mbyte = 1048576 bytes)");
}

void Display_CL_Partition_Table()
{
  int index=0;

  unsigned long usage=0;

  Determine_Drive_Letters();

  printf("\n\nCurrent fixed disk drive: %1d",(flags.drive_number-127));
  if(flags.extended_options_flag==TRUE)
    {
    printf("                  (TC: %4d",part_table[(flags.drive_number-128)].total_cyl);
    printf(" TH: %3d",part_table[(flags.drive_number-128)].total_head);
    printf(" TS: %3d)",part_table[(flags.drive_number-128)].total_sect);
    }

  printf("\n\nPartition   Status   Mbytes   Description     Usage  ");
  if(flags.extended_options_flag==TRUE) printf("Start Cyl  End Cyl");
  printf("\n");

  index=0;
  do
    {
    if(part_table[(flags.drive_number-128)].pri_part_num_type[index]>0)
      {
      /* Drive Letter of Partition */
      if( (part_table[(flags.drive_number-128)].pri_part_num_type[index]==1) || (part_table[(flags.drive_number-128)].pri_part_num_type[index]==4) || (part_table[(flags.drive_number-128)].pri_part_num_type[index]==6) || ( (part_table[(flags.drive_number-128)].pri_part_num_type[index]==0x0b) && ( (flags.version==W95B) || (flags.version==W98) ) ) )
	{
	printf(" %1c:",drive_lettering_buffer[(flags.drive_number-128)] [index]);
	}
      else printf("   ");

      /* Partition Number */
      printf(" %1d",(index+1));

      if(flags.extended_options_flag==TRUE)
	{
	/* Partition Type */
	printf(" %3d",(part_table[(flags.drive_number-128)].pri_part_num_type[index]));
	}
      else printf("    ");

      /* Status */
      if(part_table[(flags.drive_number-128)].active_status[index]>0)
	{
	printf("      A");
	}
      else printf("       ");

      /* Mbytes */
      printf("    %6d",part_table[(flags.drive_number-128)].pri_part_size_in_MB[index]);

      /* Description */
      printf("   %15s",partition_lookup_table_buffer_long[part_table[(flags.drive_number-128)].pri_part_num_type[(index)]]);

      /* Usage */
      usage=((part_table[(flags.drive_number-128)].pri_part_size_in_MB[index]*100)/part_table[(flags.drive_number-128)].total_hard_disk_size_in_MB);
      if(usage>100) usage=100;

      printf("   %3d%%",usage);

      if(flags.extended_options_flag==TRUE)
	{
	/* Starting Cylinder */
	printf("    %4d",part_table[(flags.drive_number-128)].pri_part_start_cyl[index]);

	/* Ending Cylinder */
	printf("      %4d",part_table[(flags.drive_number-128)].pri_part_end_cyl[index]);
	}
      printf("\n");
      }

    index++;
    }while(index<4);

  /* Check to see if there are any drives to display */
  if( (brief_partition_table[(flags.drive_number-128)] [4]>0) || (brief_partition_table[(flags.drive_number-128)] [5]>0) )
    {
    printf("\nContents of Extended DOS Partition:\n");
    printf("Drv Volume Label  Mbytes  System  Usage\n");

    /* Display information for each Logical DOS Drive */
    index=4;
    do
      {
      if( (brief_partition_table[(flags.drive_number-128)] [index]==1) || (brief_partition_table[flags.drive_number-128] [index]==4) || (brief_partition_table[flags.drive_number-128] [index]==6) )
	{
	/* Display drive letter */
	printf(" %1c:",drive_lettering_buffer[(flags.drive_number-128)] [index]);

	/* Display volume label */
	printf(" %11s",part_table[(flags.drive_number-128)].log_drive_vol_label[index-4]);

	/* Display size in MB */
	printf("    %4d",part_table[(flags.drive_number-128)].log_drive_size_in_MB[(index-4)]);

	/* Display file system type */
	printf("  %-8s",partition_lookup_table_buffer_short[part_table[(flags.drive_number-128)].log_drive_num_type[(index-4)]]);

	/* Display usage in % */
	usage=((part_table[(flags.drive_number-128)].log_drive_num_sect[index-4]*100)/part_table[(flags.drive_number-128)].ext_part_num_sect);
	if(usage>100) usage=100;

	printf("  %3d%%",usage);

	printf("\n");
	}

      index++;
      }while(index<27);
    }

}

/* Display Help Screens */
void Display_Help_Screen()
{
  char name[20];

  if(flags.use_freedos_label==TRUE)
    {
    strcpy(name,ALTNAME);
    strcat(name," FDISK");
    }
  else strcpy(name,PRINAME);

  char version[40];
  strcpy(version,"Version ");
  strcat(version,VERSION);

  if(flags.do_not_pause_help_information==FALSE) Clear_Screen(NOEXTRAS);

  printf("\n\n%-20s                   %40s\n", name, version);
  printf("Written By:  Brian E. Reifsnyder\n\n");
  printf("Syntax:\n\n");
  printf("  %s                  Runs %s in interative mode.\n",filename,name);
  printf("  %s /? [/NOPAUSE]    Displays this help information.\n\n",filename);

  printf("%s compatibility switches:\n\n",name);
  printf("  %s /ACTOK                          (functions with an \"*\" are incomplete)\n",filename);
  printf("  %s /CMBR <drive#>\n",filename);
  printf("* %s /EXT:<size> <drive#> [{/LOG:<size>} | {/LOGO:<size>}]\n",filename);
  printf("* %s /FPRMT\n",filename);
  printf("  %s /MBR\n",filename);
  printf("  %s /PRI:<size> <drive#>\n",filename);
  printf("* %s /PRIO:<size> <drive#>\n",filename);
  printf("  %s /STATUS\n",filename);
  printf("  %s /X\n",filename);
  printf("\n\n");

  if(flags.do_not_pause_help_information==FALSE)
    {
    Pause();
    Clear_Screen(NOEXTRAS);
    }

  printf("%s native switches:\n\n",name);
  printf("For partition table modification:\n");

  printf("  %s /A [drive#]\n",filename);
  printf("  %s /ACTIVATE <drive#> <partition#>\n",filename);
  printf("  %s /C [<drive#>]\n",filename);
  printf("  %s /D drive# {<partition#> | /P | /E | /L <drive_letter>} ]\n",filename);
  printf("  %s /DEACTIVATE <drive#>\n",filename);
  printf("  %s /M drive# <primary_partition#> <newtype#>\n",filename);
  printf("  %s /N <drive#> <partition_type> <size> [/P] [/S <type#>] ]\n",filename);
  printf("* %s /RESIZE <drive#> <partition#> <new_size>\n",filename);

  printf("\nFor MBR modification:\n");
  printf("  %s /AMBR [<drive#>]\n",filename);
  printf("  %s /MBR [<drive#>]\n",filename);
  printf("  %s /RMBR [<drive#>]\n",filename);
  printf("  %s /SMBR [<drive#>]\n",filename);

  if(flags.do_not_pause_help_information==FALSE)
    {
    printf("\n\n");
    Pause();
    Clear_Screen(NOEXTRAS);
    printf("%s native switches(cont.):\n",name);
    }

  printf("\nFor handling flags on a hard disk:\n");
  printf("  %s /CLEARFLAG [<drive#> <flag#>]\n",filename);
  printf("  %s /SETFLAG [<drive#> <flag#>]\n",filename);
  printf("  %s /TESTFLAG [<drive#> <flag#>]\n",filename);

  printf("\nFor obtaining information about the hard disk(s):\n");
  printf("  %s /DUMP\n",filename);
  printf("  %s /I [<drive#> [/TECH] ]\n",filename);
  printf("  %s /L\n",filename);

  printf("\nFor rebooting the computer:\n");
  printf("  %s /REBOOT\n",filename);
  printf("\n\n");

  if(flags.do_not_pause_help_information==FALSE)
    {
    Pause();
    Clear_Screen(NOEXTRAS);
    }
  printf("Description of command line switches:\n\n");
  printf("  /?          Displays these help screens.  If /NOPAUSE is used then the text\n");
  printf("                scrolls without pausing.  The /NOPAUSE option makes it easy\n");
  printf("                to re-direct the output to a file.\n");
  printf("\n%s compatability switches:\n\n",name);
  printf("  /ACTOK      Retained for compatibility.  On versions of FDISK where this\n");
  printf("                is a valid option, FDISK does not check the drive integrity.\n");
  printf("  /CMBR       Creates the Master Boot Record on the specified <drive#>.\n");
  printf("  /EXT:       Creates an extended partition on <drive#> of <size> in MB.\n");
  printf("                If \"/LOG:<size>\" is used then a Logical DOS Drive of\n");
  printf("                <size> is created.  \"LOGO:<size>\" is the same as \"LOG:<size>,\"\n");
  printf("                except the logical drive is forced to FAT16.\n");
  printf("  /FPRMT      When Free FDISK emulates version W95B, or greater, then the\n");
  printf("                support for large disks no longer appears.  Instead you will\n");
  printf("                get prompted everytime you create a parition.\n");
  printf("  /MBR        Writes the MAster Boot Record to disk 1.\n");
  printf("  /PRI:       Creates a primary partition on <drive#> with a size of <size>.\n");
  printf("                the partition is set active.\n");
  printf("  /PRIO:      Same as \"/PRI:\" except it forces the partition type to FAT16.\n");

  if(flags.do_not_pause_help_information==FALSE)
    {
    Pause();
    Clear_Screen(NOEXTRAS);
    printf("%s compatability switches(cont.):\n\n",name);
    }

  printf("  /STATUS     Shows information about all partitions on all hard disks.\n");
  printf("  /X          Forces %s to not use LBA partitions.  If this switch\n",name);
  printf("                is used %s will be limited to 8.4GB hard disks.\n",name);
  printf("                Note:  This switch is ignored if the emulated version is less\n");
  printf("                than W95.\n");
  printf("\n%s native switches:\n\n",name);
  printf("For partition table modification:\n");
  printf("  /A          Automatically partitions drive#.  If <drive#> is not specified,\n");
  printf("                the program will automatically partition the first drive.\n");
  printf("                If a primary and/or extended partition exists this function\n");
  printf("                will abort.\n");
  printf("  /ACTIVATE   Sets the partition active as specified by <drive#> and\n");
  printf("                <partition#>.\n");
  printf("  /C          Clears the partition sector.  (Use with extreme caution!)\n");
  printf("  /DEACTIVATE Sets all partitions on <drive#> as not active.\n");
  printf("  /M          Modifies the partition type of <primary_partition#> to\n");
  printf("                <newtype#>.\n");

  if(flags.do_not_pause_help_information==FALSE)
    {
    Pause();
    Clear_Screen(NOEXTRAS);
    printf("%s native switches(cont.):\n\n",name);
    printf("For partition table modification(cont.):\n");
    }

  printf("  /N          Creates a new partition based upon <drive#> (1-8), type (P, E,\n");
  printf("                or L), and <size> (in megabytes).  If the /P switch is added,\n");
  printf("                the size is a percentage of the total free drive space.  If\n");
  printf("                the /S switch is used then a <type#> partition is created.\n");
  printf("  /D          Deletes a partition based upon the drive# (1-4) and the partition\n");
  printf("                number, logical partition drive letter (/L), the primary\n");
  printf("                partition (/P) or the extended partition (/E).\n");
  printf("  /RESIZE     Destructively resizes a partition. \n");
  printf("\nFor MBR modification:\n\n");
  printf("  /AMBR       Writes the Master Boot program in \"BOOT.MBR\" to <drive#>.\n");
  printf("  /CMBR       Same as \"/MBR.\"\n");
  printf("  /MBR        Writes the Master Boot Record to <drive#>.\n");
  printf("  /RMBR       Removes the Master Boot Record from <drive#>.\n");
  printf("  /SMBR       Saves the current Master Boot program to the \"BOOT.MBR\" file.\n");

  if(flags.do_not_pause_help_information==FALSE)
    {
    Pause();
    Clear_Screen(NOEXTRAS);
    printf("%s native switches(cont.):\n",name);
    }

  printf("\nFor handling flags on a hard disk:\n");
  printf("  /CLEARFLAG  Clears <flag#> on <drive#>.\n");
  printf("  /SETFLAG    Sets a <flag#> on <drive#>.\n");
  printf("  /TESTFLAG   Test for a <flag#> on <drive#>.\n");
  printf("\nFor obtaining information about the hard disk(s):\n\n");
  printf("  /DUMP       Dumps technical partition information on all installed hard\n");
  printf("                disks to the screen.  (The output can optionally be redirected\n");
  printf("                with the standard redirection commands.\n");
  printf("                (i.e. \"%s /DUMP > filename.ext \")\n",filename);
  printf("  /I          Displays the partition table for drive# (1-4).  If the \"/tech\"\n");
  printf("                switch is used, it also shows the starting and ending\n");
  printf("                information for each partition.\n");
  printf("  /L          Same as \"/STATUS.\"\n");
  printf("\nFor rebooting the computer:\n\n");
  printf("  /REBOOT     Forces the computer to reboot.\n");

  if(flags.do_not_pause_help_information==FALSE)
    {
    Pause();
    Clear_Screen(NOEXTRAS);
    }
  else printf("\n\n\n\n");

  printf("\This program is Copyright %s, by Brian E. Reifsnyder, under\n",COPYLEFT);
  printf("the terms of the GNU General Public License.\n");
  printf("\nThis program comes as-is and without warranty of any kind.  The author of\n");
  printf("this software assumes no responsibility pertaining to the use or mis-use of\n");
  printf("this software.  By using this software, the operator is understood to be\n");
  printf("agreeing to the terms of the above.\n");
}

/* Display Information */
void Display_Information()
{
  if(flags.extended_options_flag==TRUE)
    {
    Position_Cursor(11,0);
    if(flags.fat32==TRUE) cprintf("FAT32");

    Position_Cursor(0,0);
    if(flags.version==FOUR) cprintf("4");
    if(flags.version==FIVE) cprintf("5");
    if(flags.version==SIX) cprintf("6");
    if(flags.version==W95) cprintf("W95");
    if(flags.version==W95B) cprintf("W95B");
    if(flags.version==W98) cprintf("W98");

    Position_Cursor(5,0);
    if(flags.use_extended_int_13==TRUE) cprintf("INT13");

    Position_Cursor(79,0);
    if(flags.extended_options_flag==TRUE) cprintf("X");

    Position_Cursor(72,0);
    if(flags.use_ambr==TRUE) cprintf("AMBR");

    Position_Cursor(77,0);
    if(flags.partitions_have_changed==TRUE) cprintf("C");
    }

  if(debug.emulate_disk>0)
    {
    Position_Cursor(66,0);
    cprintf("E%1d",debug.emulate_disk);
    }

  if(debug.write==FALSE)
    {
    Position_Cursor(69,0);
    cprintf("RO");
    }
}

/* Display Label */
void Display_Label()
{
  if(flags.label==TRUE)
    {
    int index=0;

    char label[20];

    strcpy(label,PRINAME);

    do
      {
      Position_Cursor(79,((index*2)+3));
      printf("%c",label[index]);
      index++;
      }while(index<10);
    }
}

/* Display Extended Partition Information Sub Screen */
void Display_Extended_Partition_Information_SS()
{
  int column_index=0;
  int index;
  int print_index=4;

  unsigned long usage;

  Determine_Drive_Letters();

  /* Check to see if there are any drives to display */
  if( (brief_partition_table[(flags.drive_number-128)] [4]>0) || (brief_partition_table[(flags.drive_number-128)] [5]>0) )
    {
    Position_Cursor(0,3);
    printf("Drv Volume Label  Mbytes  System  Usage");

    /* Display information for each Logical DOS Drive */
    index=4;
    print_index=4;
    do
      {
      if( (brief_partition_table[(flags.drive_number-128)] [index]==1)
       || (brief_partition_table[(flags.drive_number-128)] [index]==4)
       || (brief_partition_table[(flags.drive_number-128)] [index]==6)

       || ( ( (brief_partition_table[(flags.drive_number-128)] [index]==0x0b)
       || (brief_partition_table[(flags.drive_number-128)] [index]==0x0c) )
       && ( (flags.version==W95B) || (flags.version==W98) ) )

       || ( (brief_partition_table[(flags.drive_number-128)] [index]==0x0e)
       && ( (flags.version==W95) || (flags.version==W95B)
       || (flags.version==W98) ) ) )
	{
	if(print_index>15)
	  {
	  column_index=41;
	  print_index=4;

	  Position_Cursor(41,3);
	  printf("Drv Volume Label  Mbytes  System  Usage");
	  }

	/* Display drive letter */
	Position_Cursor((column_index+0),print_index);
	cprintf("%c",drive_lettering_buffer[(flags.drive_number-128)] [index]);

	Position_Cursor((column_index+1),print_index);
	cprintf(":");

	/* Display volume label */
	Position_Cursor((column_index+4),print_index);
	printf("%11s",part_table[(flags.drive_number-128)].log_drive_vol_label[index-4]);

	/* Display size in MB */
	Position_Cursor((column_index+19),print_index);
	printf("%4d",part_table[(flags.drive_number-128)].log_drive_size_in_MB[(index-4)]);

	/* Display file system type */
	Position_Cursor((column_index+25),print_index);
	printf("%s",partition_lookup_table_buffer_short[part_table[(flags.drive_number-128)].log_drive_num_type[(index-4)]]);

	/* Display usage in % */
	usage=((part_table[(flags.drive_number-128)].log_drive_num_sect[index-4]*100)/part_table[(flags.drive_number-128)].ext_part_num_sect);
	if(usage>100) usage=100;

	Position_Cursor((column_index+35),print_index);
	printf("%3d%%",usage);

	print_index++;
	}

      else if(brief_partition_table[(flags.drive_number-128)] [index]>0)
	     {
	     if(print_index>15)
	       {
	       column_index=41;
	       print_index=4;

	       Position_Cursor(41,3);
	       printf("Drv Volume Label  Mbytes  System  Usage");
	       }

	     /* Display size in MB */
	     Position_Cursor((column_index+19),print_index);
	     printf("%4d",part_table[(flags.drive_number-128)].log_drive_size_in_MB[(index-4)]);

	     /* Display file system type */
	     Position_Cursor((column_index+25),print_index);
	     printf("%s",partition_lookup_table_buffer_short[part_table[(flags.drive_number-128)].log_drive_num_type[(index-4)]]);

	     /* Display usage in % */
	     usage=((part_table[(flags.drive_number-128)].log_drive_num_sect[index-4]*100)/part_table[(flags.drive_number-128)].ext_part_num_sect);
	     if(usage>100) usage=100;

	     Position_Cursor((column_index+35),print_index);
	     printf("%3d%%",usage);

	     print_index++;
	     }
      index++;
      }while(index<27);
    }
  else
    {
    Position_Cursor(4,10);
    cprintf("No logical drives defined");
    }

  Position_Cursor(4,17);
  printf("Total Extended DOS Partition size is ");
  cprintf("%4d",(part_table[flags.drive_number-128].ext_part_size_in_MB) );
  printf(" Mbytes (1 Mbyte = 1048576 bytes)");
}

/* Display/Modify Partition Information */
void Display_Or_Modify_Partition_Information()
{
  int finished=FALSE;
  int input;
  int partition_number;

  unsigned long new_partition_size_in_sectors;
  unsigned long usage;

  do
    {
    Clear_Screen(NULL);
    Print_Centered(4,"Display/Modify Partition Information",BOLD);

    Display_Primary_Partition_Information_SS();

    Position_Cursor(4,18);
    printf("Which partition do you wish to modify...........................?");

    flags.esc=FALSE;
    partition_number=Input(1,69,18,NUM,1,4,ESCC,-1,0)-1;

    if(flags.esc==TRUE) finished=TRUE;
    else
      {
      Clear_Screen(NULL);
      Print_Centered(4,"Display/Modify Partition Information",BOLD);

      Determine_Drive_Letters();

      Position_Cursor(4,6);
      printf("Current fixed disk drive: ");
      cprintf("%d",(flags.drive_number-127));

      Position_Cursor(4,8);
      printf("Partition   Status   Mbytes    Description    Usage  Start Cyl  End Cyl");

      /* Drive Letter of Partition */
      if( (part_table[(flags.drive_number-128)].pri_part_num_type[partition_number]==1) || (part_table[(flags.drive_number-128)].pri_part_num_type[partition_number]==4) || (part_table[(flags.drive_number-128)].pri_part_num_type[partition_number]==6) )
	{
	Position_Cursor(5,9);
	printf("%c:",drive_lettering_buffer[(flags.drive_number-128)] [partition_number]);
	}

      if( ( (part_table[(flags.drive_number-128)].pri_part_num_type[partition_number]==0x0b) || (part_table[(flags.drive_number-128)].pri_part_num_type[partition_number]==0x0c) ) && ( (flags.version==W95B) || (flags.version==W98) ) )
	{
	Position_Cursor(5,9);
	printf("%c:",drive_lettering_buffer[(flags.drive_number-128)] [partition_number]);
	}

      if( (part_table[(flags.drive_number-128)].pri_part_num_type[partition_number]==0x0e) && ( (flags.version==W95) || (flags.version==W95B) || (flags.version==W98) ) )
	{
	Position_Cursor(5,9);
	printf("%c:",drive_lettering_buffer[(flags.drive_number-128)] [partition_number]);
	}

      /* Partition Number */
      Position_Cursor(8,9);
      cprintf("%d",(partition_number+1));

      /* Partition Type */
      Position_Cursor(10,9);
      printf("%3d",(part_table[(flags.drive_number-128)].pri_part_num_type[partition_number]));

      /* Status */
      if(part_table[(flags.drive_number-128)].active_status[partition_number]>0)
	{
	Position_Cursor(19,9);
	printf("A");
	}

      /* Mbytes */
      Position_Cursor(24,9);
      printf("%6d",part_table[(flags.drive_number-128)].pri_part_size_in_MB[partition_number]);

      /* Description */
      Position_Cursor(33,9);
      printf("%15s",partition_lookup_table_buffer_long[part_table[(flags.drive_number-128)].pri_part_num_type[partition_number]]);

      /* Usage */
      usage=((part_table[(flags.drive_number-128)].pri_part_size_in_MB[partition_number]*100)/part_table[(flags.drive_number-128)].total_hard_disk_size_in_MB);
      if(usage>100) usage=100;

      Position_Cursor(51,9);
      printf("%3d%%",usage);

      /* Starting Cylinder */
      Position_Cursor(59,9);
      printf("%4d",part_table[(flags.drive_number-128)].pri_part_start_cyl[partition_number]);

      /* Ending Cylinder */
      Position_Cursor(69,9);
      printf("%4d",part_table[(flags.drive_number-128)].pri_part_end_cyl[partition_number]);

      Position_Cursor(4,12);
      printf("Choose one of the following:");

      Position_Cursor(4,14);
      cprintf("1.");
      printf("  Change partition type");
      Position_Cursor(4,15);
      cprintf("2.");
      printf("  Remove active status");
      Position_Cursor(44,14);
      cprintf("3.");
      printf("  Hide/Unhide partition");
      Position_Cursor(44,15);
      cprintf("4.");
      printf("  Resize partition (destructive)");

      Position_Cursor(4,17);
      printf("Enter choice: ");

      flags.esc=FALSE;
      input=Input(1,19,17,NUM,1,4,ESCC,-1,0);
      if(flags.esc==TRUE) input=99;

       if(input==1)
	{
	Position_Cursor(4,19);
	printf("Enter new partition type (1-255)...................................");

	flags.esc=FALSE;
	input=Input(3,71,19,NUM,1,255,ESCC,-1,0);
	if(flags.esc==FALSE)
	  {
	  Modify_Partition_Type(partition_number,input);
	  input=99;
	  }
	else input=99;
	}

      if(input==2) Clear_Active_Partition();

      if(input==3)
	{
	if(part_table[(flags.drive_number-128)].pri_part_num_type[partition_number]>140)
	  {
	  part_table[(flags.drive_number-128)].pri_part_num_type[partition_number]=part_table[(flags.drive_number-128)].pri_part_num_type[partition_number]-140;

	  part_table[(flags.drive_number-128)].part_values_changed=TRUE;
	  flags.partitions_have_changed=TRUE;
	  input=99;
	  }
	if( (input==3) && (part_table[(flags.drive_number-128)].pri_part_num_type[partition_number]<16) )
	  {
	  part_table[(flags.drive_number-128)].pri_part_num_type[partition_number]=part_table[(flags.drive_number-128)].pri_part_num_type[partition_number]+140;

	  part_table[(flags.drive_number-128)].part_values_changed=TRUE;
	  flags.partitions_have_changed=TRUE;
	  }
	}

      if(input==4)
	{
	Position_Cursor(4,19);
	printf("Enter new partition ending cylinder...............................");
	flags.esc=FALSE;
	/* */
	input=Input(3,71,19,NUM,(part_table[(flags.drive_number-128)].pri_part_start_cyl[partition_number]+1),part_table[(flags.drive_number-128)].total_cyl,ESCC,-1,0);
	if(flags.esc==FALSE)
	  {
	  new_partition_size_in_sectors=(input-part_table[(flags.drive_number-128)].pri_part_start_cyl[partition_number])*(part_table[(flags.drive_number-128)].total_head+1)*part_table[(flags.drive_number-128)].total_sect;
	  Calculate_Partition_Ending_Cylinder(part_table[(flags.drive_number-128)].pri_part_start_cyl[partition_number],new_partition_size_in_sectors);

	  part_table[(flags.drive_number-128)].pri_part_end_cyl[partition_number]=computed_ending_cylinder;
	  part_table[(flags.drive_number-128)].pri_part_num_sect[partition_number]=computed_partition_size;
	  part_table[(flags.drive_number-128)].pri_part_size_in_MB[partition_number]=computed_partition_size/2048;

	  part_table[(flags.drive_number-128)].part_values_changed=TRUE;
	  flags.partitions_have_changed=TRUE;

	  input=99;
	  }
	else input=99;
	}
      }
    }while(finished==FALSE);

  if(part_table[(flags.drive_number-128)].num_of_log_drives>0)
    {
    Position_Cursor(4,17);
    printf("The Extended DOS Partition contains Logical DOS Drives.");

    Position_Cursor(4,18);
    printf("                                                                       ");

    Position_Cursor(4,18);
    printf("Do you want to display the logical drive information (Y/N)......?");

    input=Input(1,69,18,YN,0,0,ESCR,1,0);

    if(input==TRUE)
      {
      /* Display logical drive information */

      Clear_Screen(NOEXTRAS);

      Print_Centered(1,"Display Logical DOS Drive Information",BOLD);

      Display_Extended_Partition_Information_SS();

      Input(0,0,0,ESC,0,0,ESCC,0,0);
      }
    }
}

/* Display Partition Information */
void Display_Partition_Information()
{
  int input;

  Clear_Screen(NULL);
  Print_Centered(4,"Display Partition Information",BOLD);

  Display_Primary_Partition_Information_SS();

  if(part_table[(flags.drive_number-128)].num_of_log_drives>0)
    {
    Position_Cursor(4,17);
    printf("The Extended DOS Partition contains Logical DOS Drives.");

    Position_Cursor(4,18);
    printf("Do you want to display the logical drive information (Y/N)......?");

    input=Input(1,69,18,YN,0,0,ESCR,1,0);

    if(input==TRUE)
      {
      /* Display logical drive information */

      Clear_Screen(NOEXTRAS);

      Print_Centered(1,"Display Logical DOS Drive Information",BOLD);

      Display_Extended_Partition_Information_SS();

      Input(0,0,0,ESC,0,0,ESCC,0,0);
      }
    }
  else
    {
    Input(0,0,0,ESC,0,0,ESCC,0,0);
    }
}

/* Display Primary Partition information Sub-screen */
void Display_Primary_Partition_Information_SS()
{
  int cursor_offset=0;
  int index=0;

  unsigned long usage=0;

  Determine_Drive_Letters();

  Position_Cursor(4,6);
  printf("Current fixed disk drive: ");
  cprintf("%d",(flags.drive_number-127));

  if( (part_table[(flags.drive_number-128)].pri_part_num_type[0]>0) || (part_table[(flags.drive_number-128)].pri_part_num_type[1]>0) || (part_table[(flags.drive_number-128)].pri_part_num_type[2]>0) || (part_table[(flags.drive_number-128)].pri_part_num_type[3]>0) )
    {
    if(flags.extended_options_flag==FALSE)
      {
      Position_Cursor(4,8);
      printf("Partition  Status   Type    Volume Label  Mbytes   System   Usage");

      index=0;
      do
	{
	if(part_table[(flags.drive_number-128)].pri_part_num_type[index]>0)
	  {
	  /* Drive Letter of Partition */
	  if( (part_table[(flags.drive_number-128)].pri_part_num_type[index]==1) || (part_table[(flags.drive_number-128)].pri_part_num_type[index]==4) || (part_table[(flags.drive_number-128)].pri_part_num_type[index]==6) )
	    {
	    Position_Cursor(5,(cursor_offset+9));
	    printf("%c:",drive_lettering_buffer[(flags.drive_number-128)] [index]);
	    }

	  if( ( (part_table[(flags.drive_number-128)].pri_part_num_type[index]==0x0b) || (part_table[(flags.drive_number-128)].pri_part_num_type[index]==0x0c) ) && ( (flags.version==W95B) || (flags.version==W98) ) )
	    {
	    Position_Cursor(5,(cursor_offset+9));
	    printf("%c:",drive_lettering_buffer[(flags.drive_number-128)] [index]);
	    }

	  if( (part_table[(flags.drive_number-128)].pri_part_num_type[index]==0x0e) && ( (flags.version==W95) || (flags.version==W95B) || (flags.version==W98) ) )
	    {
	    Position_Cursor(5,(cursor_offset+9));
	    printf("%c:",drive_lettering_buffer[(flags.drive_number-128)] [index]);
	    }

	  /* Partition Number */
	  Position_Cursor(8,(cursor_offset+9));
	  cprintf("%d",(index+1));

	  /* Status */
	  if(part_table[(flags.drive_number-128)].active_status[index]>0)
	    {
	    Position_Cursor(18,(cursor_offset+9));
	    printf("A");
	    }

	  /* Type */
	  if(  (part_table[(flags.drive_number-128)].pri_part_num_type[index]==1) || (part_table[(flags.drive_number-128)].pri_part_num_type[index]==4) || (part_table[(flags.drive_number-128)].pri_part_num_type[index]==6)  )
	    {
	    Position_Cursor(23,(cursor_offset+9));
	    printf("PRI DOS");
	    }

	  if(part_table[(flags.drive_number-128)].pri_part_num_type[index]==5)
	    {
	    Position_Cursor(23,(cursor_offset+9));
	    printf("EXT DOS");
	    }

	  if( (part_table[(flags.drive_number-128)].pri_part_num_type[index]!=1) && (part_table[(flags.drive_number-128)].pri_part_num_type[index]!=4) && (part_table[(flags.drive_number-128)].pri_part_num_type[index]!=5) && (part_table[(flags.drive_number-128)].pri_part_num_type[index]!=6)  )
	    {
	    Position_Cursor(23,(cursor_offset+9));
	    printf("Non-DOS");
	    }

	  if( (part_table[(flags.drive_number-128)].pri_part_num_type[index]==0x0e) && ( (flags.version==W95) || (flags.version==W95B) || (flags.version==W98) ) )
	    {
	    Position_Cursor(23,(cursor_offset+9));
	    printf("PRI DOS");
	    }

	  if( (part_table[(flags.drive_number-128)].pri_part_num_type[index]==0x0f) && ( (flags.version==W95) || (flags.version==W95B) || (flags.version==W98) ) )
	    {
	    Position_Cursor(23,(cursor_offset+9));
	    printf("EXT DOS");
	    }

	  if( (part_table[(flags.drive_number-128)].pri_part_num_type[index]==0x0b) && ( (flags.version==W95B) || (flags.version==W98) ) )
	    {
	    Position_Cursor(23,(cursor_offset+9));
	    printf("PRI DOS");
	    }

	  if( (part_table[(flags.drive_number-128)].pri_part_num_type[index]==0x0c) && ( (flags.version==W95B) || (flags.version==W98) ) )
	    {
	    Position_Cursor(23,(cursor_offset+9));
	    printf("PRI DOS");
	    }

	  /* Volume Label */
	  Position_Cursor(33,(cursor_offset+9));
	  printf("%11s",part_table[(flags.drive_number-128)].pri_part_vol_label[index]);

	  /* Mbytes */
	  Position_Cursor(45,(cursor_offset+9));
	  printf("%6d",part_table[(flags.drive_number-128)].pri_part_size_in_MB[index]);

	  /* System */
	  Position_Cursor(54,(cursor_offset+9));
	  printf("%s",partition_lookup_table_buffer_short[part_table[(flags.drive_number-128)].pri_part_num_type[(index)]]);

	  /* Usage */
	  usage=((part_table[(flags.drive_number-128)].pri_part_size_in_MB[index]*100)/part_table[(flags.drive_number-128)].total_hard_disk_size_in_MB);
	  if(usage>100) usage=100;

	  Position_Cursor(65,(cursor_offset+9));
	  printf("%3d%%",usage);

	  cursor_offset++;
	  }

	index++;
	}while(index<4);
      }
    else
      {
      Position_Cursor(4,8);
      printf("Partition   Status   Mbytes    Description    Usage  Start Cyl  End Cyl");

      index=0;
      do
	{
	if(part_table[(flags.drive_number-128)].pri_part_num_type[index]>0)
	  {
	  /* Drive Letter of Partition */
	  if( (part_table[(flags.drive_number-128)].pri_part_num_type[index]==1) || (part_table[(flags.drive_number-128)].pri_part_num_type[index]==4) || (part_table[(flags.drive_number-128)].pri_part_num_type[index]==6) )
	    {
	    Position_Cursor(5,(cursor_offset+9));
	    printf("%c:",drive_lettering_buffer[(flags.drive_number-128)] [index]);
	    }

	  if( ( (part_table[(flags.drive_number-128)].pri_part_num_type[index]==0x0b) || (part_table[(flags.drive_number-128)].pri_part_num_type[index]==0x0c) ) && ( (flags.version==W95B) || (flags.version==W98) ) )
	    {
	    Position_Cursor(5,(cursor_offset+9));
	    printf("%c:",drive_lettering_buffer[(flags.drive_number-128)] [index]);
	    }

	  if( (part_table[(flags.drive_number-128)].pri_part_num_type[index]==0x0e) && ( (flags.version==W95) || (flags.version==W95B) || (flags.version==W98) ) )
	    {
	    Position_Cursor(5,(cursor_offset+9));
	    printf("%c:",drive_lettering_buffer[(flags.drive_number-128)] [index]);
	    }

	  /* Partition Number */
	  Position_Cursor(8,(cursor_offset+9));
	  cprintf("%d",(index+1));

	  /* Partition Type */
	  Position_Cursor(10,(cursor_offset+9));
	  printf("%3d",(part_table[(flags.drive_number-128)].pri_part_num_type[index]));

	  /* Status */
	  if(part_table[(flags.drive_number-128)].active_status[index]>0)
	    {
	    Position_Cursor(19,(cursor_offset+9));
	    printf("A");
	    }

	  /* Mbytes */
	  Position_Cursor(24,(cursor_offset+9));
	  printf("%6d",part_table[(flags.drive_number-128)].pri_part_size_in_MB[index]);

	  /* Description */
	  Position_Cursor(33,(cursor_offset+9));
	  printf("%15s",partition_lookup_table_buffer_long[part_table[(flags.drive_number-128)].pri_part_num_type[index]]);

	  /* Usage */
	  usage=((part_table[(flags.drive_number-128)].pri_part_size_in_MB[index]*100)/part_table[(flags.drive_number-128)].total_hard_disk_size_in_MB);
	  if(usage>100) usage=100;

	  Position_Cursor(51,(cursor_offset+9));
	  printf("%3d%%",usage);

	  /* Starting Cylinder */
	  Position_Cursor(59,(cursor_offset+9));
	  /* */
	  printf("%4d",part_table[(flags.drive_number-128)].pri_part_start_cyl[index]);

	  /* Ending Cylinder */
	  Position_Cursor(69,(cursor_offset+9));
	  printf("%4d",part_table[(flags.drive_number-128)].pri_part_end_cyl[index]);

	  cursor_offset++;
	  }

	index++;
	}while(index<4);
      }
    }
  else
    {
    Position_Cursor(4,21);
    cprintf("No partitions defined");
    }

  Position_Cursor(4,14);
  printf("Total disk space is ");
  cprintf("%4d",part_table[(flags.drive_number-128)].total_hard_disk_size_in_MB);
  printf(" Mbytes (1 Mbyte = 1048576 bytes)");
}

/* Dump the partition tables from all drives to screen */
void Dump_Partition_Information()
{
  int index=0;
  flags.extended_options_flag=TRUE;

  do
    {
    flags.drive_number=index+128;
    Display_CL_Partition_Table();
    index++;
    }while(index<=7);
}

/* Initialize flags, variables, load fdisk.ini, load part.dat, etc. */
void Initialization()
{
  int value1;
  int value2;

  long index;

  unsigned long sector_buffer_segment;
  unsigned long sector_buffer_offset;

  flags.display_name_description_copyright=TRUE;
  flags.monochrome=FALSE;
  flags.partitions_have_changed=FALSE;

  flags.drive_number=128;

  Load_External_Lookup_Table();

  /* If the part.ini file is not found, load the internal lookup table. */
  if(flags.partition_type_lookup_table==INTERNAL)
    {
    index=1;
    do
      {
      if( (index!=5) && (index!=15) )
	{
	/* */
	strcpy(partition_lookup_table_buffer_short[index],"Unknown ");
	strcpy(partition_lookup_table_buffer_long[index],"Unknown        ");
	}
      index++;
      }while(index<=255);

    strcpy(partition_lookup_table_buffer_short[1],"FAT12   ");
    strcpy(partition_lookup_table_buffer_short[4],"FAT16   ");
    strcpy(partition_lookup_table_buffer_short[5],"Extended");
    strcpy(partition_lookup_table_buffer_short[6],"FAT16   ");
    strcpy(partition_lookup_table_buffer_short[11],"FAT32   ");
    strcpy(partition_lookup_table_buffer_short[12],"FAT32   ");
    strcpy(partition_lookup_table_buffer_short[14],"FAT16   ");
    strcpy(partition_lookup_table_buffer_short[15],"Extended");

    strcpy(partition_lookup_table_buffer_long[1],"FAT12          ");
    strcpy(partition_lookup_table_buffer_long[4],"FAT16          ");
    strcpy(partition_lookup_table_buffer_long[5],"Extended       ");
    strcpy(partition_lookup_table_buffer_long[6],"FAT16          ");
    strcpy(partition_lookup_table_buffer_long[11],"FAT32          ");
    strcpy(partition_lookup_table_buffer_long[12],"FAT32          ");
    /* */
    strcpy(partition_lookup_table_buffer_long[14],"FAT16          ");
    strcpy(partition_lookup_table_buffer_long[15],"Extended       ");
    }

  Process_Fdiskini_File();

  Determine_Color_Video_Support();

  /* Check for interrupt 0x13 extensions (If the proper version is set.) */
  if( (flags.version==W95) || (flags.version==W95B) || (flags.version==W98) )
   Check_For_INT13_Extensions();

  /* If the version is W95B or W98 then default to FAT32 support.  */
  if( (flags.version==W95B) || (flags.version==W98) ) flags.fat32=TRUE;

  /* Initialize LBA structures, if necessary. */
  if(flags.use_extended_int_13==TRUE)
    {
    /* Initialize the Disk Address Packet */
    /* ---------------------------------- */

    /* Packet size = 16 */
    disk_address_packet[0]=16;

    /* Reserved, must be 0 */
    disk_address_packet[1]=0;

    /* Number of blocks to transfer = 1 */
    disk_address_packet[2]=1;

    /* Reserved, must be 0 */
    disk_address_packet[3]=0;

    /* Determine the location of sector_buffer[512]             */
    /* and place the address of sector_buffer[512] into the DAP */

    Convert_Long_To_Integer(FP_OFF(sector_buffer));
    disk_address_packet[4]=integer1;
    disk_address_packet[5]=integer2;

    Convert_Long_To_Integer(FP_SEG(sector_buffer));
    disk_address_packet[6]=integer1;
    disk_address_packet[7]=integer2;

    /* Zero out the offsets 8-11 to clear the LBA address field. */
    index=8;
    do
      {
      disk_address_packet[index]=0;
      index++;
      }while(index<12);

    /* Zero out offsets 12-15 because they are currently unused. They are  */
    /* part of the 64 bit LBA address...at the present time only the lower */
    /* 28 bits are used.                                                   */
    index=12;
    do
      {
      disk_address_packet[index]=0;
      index++;
      }while(index<16);

    /* Initialize the Result Buffer */
    /* ---------------------------- */

    /* Buffer Size = 26 */
    result_buffer[0]=26;
    }

  Read_Partition_Tables();

  if( (part_table[1].total_cyl+part_table[2].total_cyl+part_table[3].total_cyl+part_table[4].total_cyl+part_table[5].total_cyl+part_table[6].total_cyl+part_table[7].total_cyl)>0)
    {
    flags.more_than_one_drive=TRUE;
    }
  else flags.more_than_one_drive=FALSE;

  if(flags.flag_sector>part_table[(flags.drive_number-128)].total_sect)
    {
    printf("The \"FLAG_SECTOR\" value in the \"fdisk.ini\" file is out of range...\n");
    printf("Operation Terminated.\n");
    exit(3);
    }
}

/* Get input from keyboard */
unsigned long Input(int size_of_field,int x_position,int y_position,int type, int min_range, long max_range,int return_message,int default_value,long maximum_possible_percentage)
{
  /*
  size_of_field:                 number of characters for the user to enter,
				 if size of field is 0 then no input box
				 is drawn on the screen.
  x_position, y_position:        screen coordinates to place the input box
  type                           type of input--CHAR  A single character as
						      specified by min_range
						      and max_range.  min_range
						      and max_range are the
						      min. and max. ASCII
						      values possible for this
						      input field. The ASCII
						      values in these 2 fields
						      should be of capital
						      characters only.  If the
						      user enters a lower case
						      character it will be
						      converted to uppercase.
						YN    Either a yes or no.
						NUM   A number as specified
						      by min_range and
						      max_range.
						NUMP  A number or percentage
						      within the limits of
						      the size_of_field.
						ESC   Waits for the ESC key
						      only
  return_message				ESCR  Displays "Press
						      Esc to return to FDISK
						      Options"
						ESCE  Displays "Press
						      Esc to exit FDISK"
						ESCC  Displays "Press Esc to
						      continue"
						NONE  Does not display a
						      return message.
  default_value                  The default value that is displayed for
				 input.  This option only works with the NUM
				 type or the YN type.
				 Set this to -1 if it is not used.
  maximum_possible_percentage                   If type is NUMP, this is the
						maximum percentage possible.

  */

  char input;
  char line_buffer[10];

  unsigned long multiplier;

  int index;
  int invalid_input=FALSE;
  int line_buffer_index=0;
  int proper_input_given=FALSE;
  int percent_entered=FALSE;
  int percent_just_entered=FALSE;

  unsigned long data_max_range=max_range;
  unsigned long data;

  /* Clear line buffer */
  index=0;
  do
    {
    line_buffer[index]=0;
    index++;
    }while(index<10);

  /* Place appropriate text on the screen prior to obtaining input */
  if(type!=ESC)
    {
    Position_Cursor(x_position,y_position);

    cprintf("[");

    index=0;
    do
      {
      cprintf(" ");
      index++;
      }while(index<size_of_field);

    cprintf("]");
    }

  /* Display the return message */
  if( (return_message==ESCR) || (return_message==ESCE) || (return_message==ESCC) )
    {
    Position_Cursor(4,24);
    printf("Press ");
    cprintf("Esc");
    printf(" to ");
    }

  if(return_message==ESCR) printf("return to FDISK options");

  if(return_message==ESCE) printf("exit FDISK");

  if(return_message==ESCC) printf("continue");

  /* Set the default value for NUM type, if applicable */
  if( (default_value>=0) && (type==NUM) && (size_of_field==1) )
    {
    Position_Cursor(x_position+1,y_position);
    printf("%d",default_value);
    line_buffer_index=0;
    line_buffer[0]=default_value+48;
    }
  if( (default_value>=0) && (type==NUM) && (size_of_field>1) )
    {
    /* Needs added! */


    }

  /* Set the default value for YN type, if applicable */
  if( (default_value>=0) && (type==YN) && (size_of_field==1) )
    {
    Position_Cursor(x_position+1,y_position);

    if(default_value==1)
      {
      printf("Y");
      line_buffer_index=0;
      line_buffer[0]='Y';
      data=TRUE;
      }

    if(default_value==0)
      {
      printf("N");
      line_buffer_index=0;
      line_buffer[0]='N';
      data=FALSE;
      }

    }

  do
    {
    if(type!=ESC) Position_Cursor((size_of_field+x_position),y_position);

    /* Obtain keypress from keyboard */
    asm{
      mov ah,7
      int 0x21
      mov BYTE PTR input,al
      }

    /* Clear error messages from screen */
    if(type!=YN)
      {
      Position_Cursor(4,22);
      printf("                                                            ");
      }

    Position_Cursor(4,23);
    printf("                                                    ");

    /* Esc key has been hit */
    if(input==27)
      {
      flags.esc=TRUE;
      proper_input_given=TRUE;
      data=0;
      type=99;
      }

    /* Enter key has been hit */
    if(input==13)
      {
      if( ( (type==CHAR) || (type==YN) ) && (line_buffer[0]!=0) && ( (data==TRUE) || (data==FALSE) || (data!=99) ) )
	{
	proper_input_given=TRUE;

	type=99;
	}

      if( (type==NUM) && (line_buffer[0]!=0) )
	{
	proper_input_given=TRUE;

	/* Convert line_buffer to an unsigned integer in data */
	data=0;
	index=strlen(line_buffer)-1;
	multiplier=1;
	do
	  {
	  data=data+((line_buffer[index]-48)*multiplier);
	  index--;
	  multiplier=multiplier*10;
	  }while(index>=0);

	/* Make sure that data is <= max_range */
	if(data>data_max_range)
	  {
	  data=0;
	  proper_input_given=FALSE;

	  Position_Cursor(4,22);
	  cprintf("Requested partition size exceeds the maximum available space");

	  /* Set input=0xff to avoid processing this time around */
	  input='\xff';
	  }
	else type=99;
	}

      if( (type==NUMP) && (line_buffer[0]!=0) )
	{
	proper_input_given=TRUE;

	/* Convert line_buffer to an unsigned integer in data */
	data=0;
	index=strlen(line_buffer)-1;

	if(percent_entered==TRUE) index--;

	multiplier=1;
	do
	  {
	  data=data+((line_buffer[index]-48)*multiplier);
	  index--;
	  multiplier=multiplier*10;
	  }while(index>=0);


	if(percent_entered==TRUE) data=(data*data_max_range)/maximum_possible_percentage;

	/* Make sure that data is <= max_range */
	if(data>data_max_range)
	  {
	  data=0;
	  proper_input_given=FALSE;

	  Position_Cursor(4,22);
	  cprintf("Requested partition size exceeds the maximum available space");

	  /* Set input=0xff to avoid processing this time around */
	  input='\xff';
	  }
	else type=99;
	}

      if( (debug.input_routine==TRUE) && (type==99) )
	{
	Clear_Screen(NULL);

	printf("Input entered by user:  %d",data);
	Pause();
	}
      }

    if(debug.input_routine==TRUE)
      {
      Position_Cursor(50,22);
      printf("                  ");

      Position_Cursor(50,22);
      printf("Input:  %d",input);
      }

    if(type==CHAR)
      {
      /* Convert to upper case, if necessary. */
      if(input>=97) input=input-32;

      if( (input>=min_range) && (input<=max_range) )
	{
	line_buffer[0]=input;
	data=input;
	}
      else
	{
	proper_input_given=FALSE;
	line_buffer[0]=' ';
	data=99;

	Position_Cursor(4,23);
	cprintf("Invalid entry, please enter %c-",min_range);
	cprintf("%c.",max_range);
	}

      Position_Cursor((x_position+1),y_position);
      cprintf("%c",line_buffer[0]);
      }

    if(type==YN)
      {
      switch (input) {
	case 'Y':
	  line_buffer[0]='Y';
	  data=TRUE;
	  break;
	case 'y':
	  line_buffer[0]='Y';
	  data=TRUE;
	  break;
	case 'N':
	  line_buffer[0]='N';
	  data=FALSE;
	  break;
	case 'n':
	  line_buffer[0]='N';
	  data=FALSE;
	  break;
	default:
	  proper_input_given=FALSE;
	  line_buffer[0]=' ';
	  data=99;

	  Position_Cursor(4,23);
	  cprintf("Invalid entry, please enter Y-N.");

	}

      Position_Cursor((x_position+1),y_position);
      cprintf("%c",line_buffer[0]);
      }

    if( (type==NUM) && (input!='\xff') )
      {
      /* If the backspace key has not been hit. */
      if(input!=8)
	{
	invalid_input=FALSE;

	if(size_of_field>1)
	  {
	  min_range=0;
	  max_range=9;
	  }

	if( (input>='0') && (input<='9') )input=input-48;
	else
	  {
	  if(input<10) input=11;
	  }

	if( ( (size_of_field>1) && (input>max_range) ) || (input>9) )
	  {
	  proper_input_given=FALSE;

	  Position_Cursor(4,23);
	  cprintf("Invalid entry, please enter %d-%d.",min_range,max_range);
	  invalid_input=TRUE;
	  }

	if( (size_of_field==1) && ( (input<min_range) || ( (input>max_range) && (input<10) ) ) )
	  {
	  proper_input_given=FALSE;

	  Position_Cursor(4,23);
	  cprintf("%d is not a choice, please enter ",input);
	  cprintf("%d-%d.",min_range,max_range);
	  invalid_input=TRUE;
	  }

	if( (invalid_input==FALSE) && (line_buffer_index==size_of_field) && (size_of_field>1) )
	  {
	  proper_input_given=FALSE;

	  Position_Cursor(4,23);
	  cprintf("Invalid entry.");
	  invalid_input=TRUE;
	  }

	if( (invalid_input==FALSE) && (line_buffer_index==size_of_field) && (size_of_field==1) )
	  {
	  line_buffer_index=0;
	  }

	if(invalid_input==FALSE)
	  {
	  if( (line_buffer_index==1) && (line_buffer[0]=='0') )
	    {
	    line_buffer[0]=0;
	    line_buffer_index=0;
	    }

	  line_buffer[line_buffer_index]=(input+48);
	  line_buffer_index++;
	  }
	}
      else
	{
	/* If the backspace key has been hit */
	line_buffer_index--;
	if(line_buffer_index<0) line_buffer_index=0;
	line_buffer[line_buffer_index]=0;

	if(line_buffer_index==0)
	  {
	  line_buffer[0]='0';
	  line_buffer_index=1;
	  }
	}

      /* Clear text box before displaying line_buffer */
      index=0;
      do
	{
	Position_Cursor((x_position+1+index),y_position);
	printf(" ");

	index++;
	}while(index<size_of_field);

      /* Display line_buffer */
      index=line_buffer_index;
      do
	{
	Position_Cursor((x_position+size_of_field-line_buffer_index+index),y_position);
	index--;
	cprintf("%c",line_buffer[index]);
	}while(index>0);
      }

    if( (type==NUMP) && (input!='\xff') )
      {
      /* If the backspace key has not been hit. */
      if(input!=8)
	{
	invalid_input=FALSE;

	if(size_of_field>1)
	  {
	  min_range=0;
	  max_range=9;
	  }

	if( (input=='%') && (percent_entered==FALSE) )
	  {
	  percent_entered=TRUE;
	  percent_just_entered=TRUE;
	  }

	if( (input>='0') && (input<='9') )input=input-48;
	else
	  {
	  if(input<10) input=11;
	  }

	if( (percent_entered==FALSE) && (percent_just_entered==FALSE) && ( ( (size_of_field>1) && (input>max_range) ) || (input>9) ) )
	  {
	  proper_input_given=FALSE;

	  Position_Cursor(4,23);
	  cprintf("Invalid entry, please enter %d-%d.",min_range,max_range);
	  invalid_input=TRUE;
	  }

	if( (percent_entered==FALSE) && (size_of_field==1) && ( (input<min_range) || ( (input>max_range) && (input<10) ) ) )
	  {
	  proper_input_given=FALSE;

	  Position_Cursor(4,23);
	  cprintf("%d is not a choice, please enter ",input);
	  cprintf("%d-%d.",min_range,max_range);
	  invalid_input=TRUE;
	  }

	if( ( (percent_entered==TRUE) && (percent_just_entered==FALSE) ) || ( (invalid_input==FALSE) && (line_buffer_index==size_of_field) && (size_of_field>1) ) )
	  {
	  proper_input_given=FALSE;

	  Position_Cursor(4,23);
	  cprintf("Invalid entry.");
	  invalid_input=TRUE;
	  }

	if( (invalid_input==FALSE) && (line_buffer_index==size_of_field) && (size_of_field==1) )
	  {
	  line_buffer_index=0;
	  }

	if(invalid_input==FALSE)
	  {
	  if( (line_buffer_index==1) && (line_buffer[0]=='0') )
	    {
	    line_buffer[0]=0;
	    line_buffer_index=0;
	    }

	  if(percent_just_entered==TRUE)
	    {
	    percent_just_entered=FALSE;
	    line_buffer[line_buffer_index]='%';
	    line_buffer_index++;
	    }
	  else
	    {
	    line_buffer[line_buffer_index]=(input+48);
	    line_buffer_index++;
	    }
	  }
	}
      else
	{
	/* If the backspace key has been hit */
	line_buffer_index--;
	if(line_buffer_index<0) line_buffer_index=0;
	line_buffer[line_buffer_index]=0;

	if(line_buffer_index==0)
	  {
	  line_buffer[0]='0';
	  line_buffer_index=1;
	  }

	if(percent_entered==TRUE) percent_entered=FALSE;
	}

      /* Clear text box before displaying line_buffer */
      index=0;
      do
	{
	Position_Cursor((x_position+1+index),y_position);
	printf(" ");

	index++;
	}while(index<size_of_field);

      /* Display line_buffer */
      index=line_buffer_index;
      do
	{
	Position_Cursor((x_position+size_of_field-line_buffer_index+index),y_position);
	index--;
	cprintf("%c",line_buffer[index]);
	}while(index>0);
      }

    if(debug.input_routine==TRUE)
      {
      Position_Cursor(60,23);
      printf("                ");

      Position_Cursor(60,24);
      printf("                ");

      Position_Cursor(50,23);
      printf("Line Buffer:  %10s",line_buffer);

      Position_Cursor(50,24);
      printf("Line Buffer Index:  %d",line_buffer_index);

      Position_Cursor(75,24);
      if(percent_entered==TRUE)
	{
	printf("P");
	}
      else
	{
	printf("  ");
	}
      }

    /* Place brackets back on screen as a precautionary measure. */
    if(type!=ESC)
      {
      Position_Cursor(x_position,y_position);
      cprintf("[");

      Position_Cursor((x_position+size_of_field+1),y_position);
      cprintf("]");
      }

    }while(proper_input_given==FALSE);


  percent_entered++; /* dummy */

  return(data);
}

/* Interactive User Interface Control Routine */
void Interactive_User_Interface()
{
  int counter=0;
  int index=0;
  int menu=MM;

  Create_MBR_If_Not_Present();

  do
    {
    menu=Standard_Menu(menu);

/* Definitions for the menus */
/* MM   0x00                  Main Menu                     */

/*   CP   0x10                Create PDP or LDD             */

/*     CPDP 0x11              Create Primary DOS Partition  */
/*     CEDP 0x12              Create Extended DOS Partition */
/*     CLDD 0x13              Create Logical DOS Drive      */

/*   SAP  0x20                Set Active Partition          */

/*   DP   0x30                Delete partition or LDD       */

/*     DPDP 0x31              Delete Primary DOS Partition  */
/*     DEDP 0x32              Delete Extended DOS Partition */
/*     DLDD 0x33              Delete Logical DOS Drive      */
/*     DNDP 0x34              Delete Non-DOS Partition      */

/*   DPI  0x40                Display Partition Information */

/*   CD   0x50                Change Drive                  */

/* EXIT 0x0f                  Code to Exit from Program     */

    if( (menu==CPDP) || (menu==CEDP) )
      {
      /* Ensure that space is available in the primary partition table */
      /* to create a partition.                                        */

      /* First make sure that an empty slot is available.  */
      index=0;
      counter=0;
      do
	{
	if(part_table[(flags.drive_number-128)]
	 .pri_part_num_type[index]>0) counter++;
	index++;
	}while(index<4);

      /* Next, make sure that there is a space available of at least   */
      /* two cylinders.                                                */
      Determine_Free_Space();
      if(part_table[(flags.drive_number-128)]
       .pri_part_largest_free_space<2) counter=4;

      if(counter>3)
	{
	Clear_Screen(NULL);

	if(menu==CPDP)
	 Print_Centered(4,"Create Primary DOS Partition",BOLD);
	else Print_Centered(4,"Create Extended DOS Partition",BOLD);
	Position_Cursor(4,6);
	printf("Current fixed disk drive: ");
	cprintf("%d",(flags.drive_number-127));

	Display_Primary_Partition_Information_SS();

	Position_Cursor(4,22);
	cprintf("No space to create a DOS partition.");

	Input(0,0,0,ESC,0,0,ESCC,0,0);
	menu=MM;
	}
      }

    if(menu==CPDP) Create_DOS_Partition_Interface(PRIMARY);
    if(menu==CEDP)
      {
      if(part_table[(flags.drive_number-128)]
       .ext_part_exists==TRUE)
	{
	Clear_Screen(NULL);

	Print_Centered(4,"Create Extended DOS Partition",BOLD);
	Position_Cursor(4,6);
	printf("Current fixed disk drive: ");
	cprintf("%d",(flags.drive_number-127));

	Display_Primary_Partition_Information_SS();

	Position_Cursor(4,22);
	cprintf("Extended DOS Partition already exists.");

	Input(0,0,0,ESC,0,0,ESCC,0,0);
	}
      else Create_DOS_Partition_Interface(EXTENDED);
      }

    if(menu==CLDD)
      {
      if(part_table[(flags.drive_number-128)].ext_part_exists==FALSE)
	{
	Position_Cursor(4,22);
	cprintf("Cannot create Logical DOS Drive without");
	Position_Cursor(4,23);
	cprintf("an Extended DOS Partition on the current drive.");
	Position_Cursor(4,24);
	printf("                                        ");
	Input(0,0,0,ESC,0,0,ESCC,0,0);
	menu=MM;
	}
      else Create_Logical_Drive_Interface();
      }

    if(menu==SAP) Set_Active_Partition_Interface();

    if(menu==DPDP)
      {
      /* Ensure that primary partitions are available to delete. */
      counter=0;
      index=0;

      do
	{
	if(part_table[(flags.drive_number-128)]
	 .pri_part_num_type[index]>0)
	  {
	  if(part_table[(flags.drive_number-128)]
	   .pri_part_num_type[index]==1) counter++;
	  if(part_table[(flags.drive_number-128)]
	   .pri_part_num_type[index]==4) counter++;
	  if(part_table[(flags.drive_number-128)]
	   .pri_part_num_type[index]==6) counter++;

	  if( (flags.version==W95) || (flags.version==W95B)
	   || (flags.version==W98) )
	    {
	    if(part_table[(flags.drive_number-128)]
	     .pri_part_num_type[index]==0x0e) counter++;
	    }

	  if( (flags.version==W95B) || (flags.version==W98) )
	    {
	    if(part_table[(flags.drive_number-128)]
	     .pri_part_num_type[index]==0x0b) counter++;
	    if(part_table[(flags.drive_number-128)]
	     .pri_part_num_type[index]==0x0c) counter++;
	    }
	  }

	index++;
	}while(index<4);

      if(counter==0)
	{
	Position_Cursor(4,22);
	cprintf("No Primary DOS Partition to delete.");
	Position_Cursor(4,24);
	printf("                                        ");
	Input(0,0,0,ESC,0,0,ESCC,0,0);
	menu=MM;
	}
      else Delete_Primary_DOS_Partition_Interface();
      }

    if(menu==DEDP)
      {
      if(part_table[(flags.drive_number-128)].ext_part_exists==FALSE)
	{
	Position_Cursor(4,22);
	cprintf("No Extended DOS Partition to delete.");
	Position_Cursor(4,24);
	printf("                                        ");
	Input(0,0,0,ESC,0,0,ESCC,0,0);
	menu=MM;
	}
      else Delete_Extended_DOS_Partition_Interface();
      }

    if(menu==DLDD)
      {
      if( (part_table[(flags.drive_number-128)].num_of_log_drives==0) || (part_table[(flags.drive_number-128)].ext_part_exists==FALSE) )
	{
	Position_Cursor(4,22);
	cprintf("No Logical DOS Drive(s) to delete.");
	Position_Cursor(4,24);
	printf("                                        ");
	Input(0,0,0,ESC,0,0,ESCC,0,0);
	menu=MM;
	}
      else Delete_Logical_Drive_Interface();
      }

    if(menu==DNDP)
      {
      /* First Ensure that Non-DOS partitions are available to delete. */
      index=0;
      counter=0;

      do
	{
	if(part_table[(flags.drive_number-128)]
	 .pri_part_num_type[index]>0)
	  {
	  counter++;
	  if(part_table[(flags.drive_number-128)]
	   .pri_part_num_type[index]==1) counter--;
	  if(part_table[(flags.drive_number-128)]
	   .pri_part_num_type[index]==4) counter--;
	  if(part_table[(flags.drive_number-128)]
	   .pri_part_num_type[index]==5) counter--;
	  if(part_table[(flags.drive_number-128)]
	   .pri_part_num_type[index]==6) counter--;

	  if( (flags.version==W95) || (flags.version==W95B)
	   || (flags.version==W98) )
	    {
	    if(part_table[(flags.drive_number-128)]
	     .pri_part_num_type[index]==0x0e) counter--;
	    if(part_table[(flags.drive_number-128)]
	     .pri_part_num_type[index]==0x0f) counter--;
	    }

	  if( (flags.version==W95B) || (flags.version==W98) )
	    {
	    if(part_table[(flags.drive_number-128)]
	     .pri_part_num_type[index]==0x0b) counter--;
	    if(part_table[(flags.drive_number-128)]
	     .pri_part_num_type[index]==0x0c) counter--;
	    }
	  }
	index++;
	}while(index<4);

      if(counter==0)
	{
	Position_Cursor(4,22);
	cprintf("No Non-DOS Partition to delete.");
	Position_Cursor(4,24);
	printf("                                        ");
	Input(0,0,0,ESC,0,0,ESCC,0,0);
	menu=MM;
	}
      else Delete_N_DOS_Partition_Interface();
      }
    if( (menu==DPI) && (flags.extended_options_flag==FALSE) ) Display_Partition_Information();
    if( (menu==DPI) && (flags.extended_options_flag==TRUE) ) Display_Or_Modify_Partition_Information();

    if(menu==CD)  Change_Current_Fixed_Disk_Drive();

    if(menu!=EXIT)
      {
      if( (menu>0x0f) || (menu==MM) )menu=MM;
      else menu=menu&0xf0;
      }

    }while(menu!=EXIT);

  if(flags.partitions_have_changed==TRUE)
    {
    Write_Partition_Tables();

    Clear_Screen(NOEXTRAS);

    if(flags.reboot==FALSE)
      {
      Position_Cursor(4,11);
      printf("You ");
      cprintf("MUST");
      printf(" restart your system for your changes to take effect.");
      Position_Cursor(4,12);
      printf("Any drives you have created or changed must be formatted");
      Position_Cursor(4,13);
      cprintf("AFTER");
      printf(" you restart.");

      Input(0,0,0,ESC,0,0,ESCE,0,0);
      Clear_Screen(NOEXTRAS);
      }
    else
      {
      Position_Cursor(4,13);
      cprintf("System will now restart");
      Position_Cursor(4,15);
      printf("Press any key when ready . . .");

      /* Wait for a keypress. */
      asm{
	mov ah,7
	int 0x21
	}

      Reboot_PC();
      }
    }
  else Clear_Screen(NOEXTRAS);
}

/* Convert the standard_partition_type to an LBA partition type */
int LBA_Partition_Type_To_Create(int standard_partition_type)
{
  int lba_partition_type;

  /* Extended int 0x13 FAT 32 */
  if(standard_partition_type==0x0b) lba_partition_type=0x0c;

  /* Extended int 0x13 FAT 16 */
  if( (standard_partition_type==1) || (standard_partition_type==4)
   || (standard_partition_type==6) ) lba_partition_type=0x0e;

  /* Extended int 0x13 Extended Partition */
  if(standard_partition_type==0x05) lba_partition_type=0x0f;

  return(lba_partition_type);
}

/* Load the brief_partition_table[8] [27] */
void Load_Brief_Partition_Table()
{
  int index=0;
  int sub_index=0;

  /* Clear brief_partition_table[8] [27] */
  index=0;
  do
    {
    sub_index=0;
    do
      {
      brief_partition_table[index] [sub_index]=0;

      sub_index++;
      }while(sub_index<27);
    index++;
    }while(index<8);

  index=0;
  do
    {
    /* Load the primary partitions into brief_partition_table[8] [27] */
    sub_index=0;
    do
      {
      brief_partition_table[index] [sub_index]=part_table[index].pri_part_num_type[sub_index];

      sub_index++;
      }while(sub_index<4);

    /* Load the extended partitions into brief_partition_table[8] [27] */
    sub_index=0;
    do
      {
      brief_partition_table[index] [(sub_index+4)]=part_table[index].log_drive_num_type[sub_index];

      sub_index++;
      }while(sub_index<23);

    index++;
    }while(index<8);
}

/* Modify Partition Type */
/* This will eventually be modified to include logical drives */
void Modify_Partition_Type(int partition_number,int type_number)
{
  part_table[(flags.drive_number-128)].pri_part_num_type[partition_number]=type_number;

  part_table[(flags.drive_number-128)].part_values_changed=TRUE;
  flags.partitions_have_changed=TRUE;
}

/* Compute the partition type to create. */
int Partition_Type_To_Create(unsigned long size_in_mb)
{
  int numeric_type;

  /* FAT 12 */
  if(size_in_mb<=16) numeric_type=1;

  /* Small FAT 16 */
  if( (size_in_mb>16) && (size_in_mb<=32) ) numeric_type=4;

  /* Large FAT 16 */
  if(size_in_mb>32) numeric_type=6;

  /* FAT 32 */
  if( (size_in_mb>512) && ( (flags.version==W95B) || (flags.version==W98) )
   && (flags.fat32==TRUE) ) numeric_type=0x0b;

  return(numeric_type);
}

/* Pause Routine */
void Pause()
{
  printf("\nPress any key to continue.\n");

  asm{
    mov ah,7
    int 0x21
    }
}

/* Position cursor on the screen */
void Position_Cursor(int column,int row)
{
  asm{
    /* Get video page number */
    mov ah,0x0f
    int 0x10

    /* Position Cursor */
    mov ah,0x02
    mov dh,BYTE PTR row
    mov dl,BYTE PTR column
    int 0x10
    }
}

/* Print Centered Text */
void Print_Centered(int y,char *text,int style)
{
  int x=40-strlen(text)/2;

  Position_Cursor(x,y);

  if(style==BOLD) cprintf(text);
  else printf(text);
}

/* Reboot the PC */
void Reboot_PC()
{
  /* Note:  Reboot is a cold start. */
  void ((far * fp) ())=(void(far*) () ) ((0xffffL<<16) | 0x0000L);
  *(int far *) ((0x0040L << 16) | 0x0072)=0;
  fp();
}

/* Set Active Partition */
void Set_Active_Partition(int partition_number)
{
  int index=0;

  do
    {
    if(index==partition_number) part_table[(flags.drive_number-128)].active_status[index]=0x80;
    else part_table[(flags.drive_number-128)].active_status[index]=0x00;

    index++;
    }while(index<4);

  part_table[(flags.drive_number-128)].part_values_changed=TRUE;
  flags.partitions_have_changed=TRUE;
}

/* Set Active Partition Interface */
int Set_Active_Partition_Interface()
{
  int correct_input=FALSE;
  int index=0;
  int input;

  int available_partition_counter=0;
  int first_available_partition_active=FALSE;
  int only_active_partition_active=FALSE;

  int partition_settable[4];

  /* Check to see if other partitions that can be set active exist.*/
  /* Also check to see what partitions are available to set active.*/
  do
    {
    partition_settable[index]=FALSE;

    if( (part_table[(flags.drive_number-128)].pri_part_num_type[index]==1) || (part_table[(flags.drive_number-128)].pri_part_num_type[index]==4) || (part_table[(flags.drive_number-128)].pri_part_num_type[index]==6) || ( (part_table[(flags.drive_number-128)].pri_part_num_type[index]==0x0b) && ( (flags.version==W95B) || (flags.version==W98) ) ) )
      {
      available_partition_counter++;
      if( (available_partition_counter==1) && (part_table[(flags.drive_number-128)].active_status[index]==0x80) )first_available_partition_active=TRUE;
      partition_settable[index]=TRUE;
      }

    index++;
    }while(index<=3);

  if( (available_partition_counter==1) && (first_available_partition_active==TRUE) ) only_active_partition_active=TRUE;

  Clear_Screen(NULL);
  Print_Centered(4,"Set Active Partition",BOLD);

  Display_Primary_Partition_Information_SS();

  if(available_partition_counter==0)
    {
    Position_Cursor(4,22);
    cprintf("No partitions to make active.");

    Input(0,0,0,ESC,0,0,ESCC,0,0);
    }

  if( (only_active_partition_active==FALSE) && (available_partition_counter>0) )
    {
    Position_Cursor(4,16);
    printf("Enter the number of the partition you want to make active...........: ");

    do
      {
      flags.esc=FALSE;
      input=Input(1,70,16,NUM,1,4,ESCR,-1,0);
      if(flags.esc==TRUE) return(1);

      /* Ensure that input is valid. */
      if(partition_settable[(input-1)]==TRUE) correct_input=TRUE;
      else
	{
	Position_Cursor(4,23);
	cprintf("%d is not a choice. Please enter a valid choice.",input);
	}

      }while(correct_input==FALSE);

    Set_Active_Partition(input-1);

    Clear_Screen(NULL);
    Print_Centered(4,"Set Active Partition",BOLD);

    /* */
    Display_Primary_Partition_Information_SS();

    Input(0,0,0,ESC,0,0,ESCC,0,0);
    }

  if(only_active_partition_active==TRUE)
    {
    Position_Cursor(4,22);
    cprintf("The only startable partition on Drive %d is already set active.",(flags.drive_number-127));

    Input(0,0,0,ESC,0,0,ESCC,0,0);
    }

  return(0);
}

/* Standard Menu Routine */
/* Displays the menus laid out in a standard format and returns the */
/* selection chosen by the user.                                    */
int Standard_Menu(int menu)
{
  int counter;
  int index;

  int display_menu=TRUE;
  int result;
  int maximum_number_of_options=0;

  int input;

  char program_name[60]="";
  char program_description[60]="";
  char copyleft[60]="";

  char title[60]="";

  char option_1[60]="";
  char option_2[60]="";
  char option_3[60]="";
  char option_4[60]="";
  char option_5[60]="Change current fixed disk drive";

  do
    {
    /* Load Menu Text */

    if(flags.use_freedos_label==FALSE)
      {
      strcpy(program_name,PRINAME);
      strcat(program_name,"    Version ");
      strcat(program_name,VERSION);
      }
    else strcpy(program_name,ALTNAME);

    strcpy(program_description,"Fixed Disk Setup Program");
    strcpy(copyleft,"GNU GPL Copyright Brian E. Reifsnyder ");
    strcat(copyleft,COPYLEFT);

    if(menu==MM)
      {
      maximum_number_of_options=4;
      strcpy(title,"FDISK Options");
      strcpy(option_1,"Create DOS partition or Logical DOS Drive");
      strcpy(option_2,"Set Active partition");
      strcpy(option_3,"Delete partition or Logical DOS Drive");

      if(flags.extended_options_flag==FALSE) strcpy(option_4,"Display partition information");
      else strcpy(option_4,"Display/Modify partition information");
      }

    if(menu==CP)
      {
      maximum_number_of_options=3;
      strcpy(title,"Create DOS Partition or Logical DOS Drive");
      strcpy(option_1,"Create Primary DOS Partition");
      strcpy(option_2,"Create Extended DOS Partition");
      strcpy(option_3,"Create Logical DOS Drive(s) in the Extended DOS Partition");
      strcpy(option_4,"");
      }

    if(menu==DP)
      {
      maximum_number_of_options=4;
      strcpy(title,"Delete DOS Partition or Logical DOS Drive");
      strcpy(option_1,"Delete Primary DOS Partition");
      strcpy(option_2,"Delete Extended DOS Partition");
      strcpy(option_3,"Delete Logical DOS Drive(s) in the Extended DOS Partition");
      strcpy(option_4,"Delete Non-DOS Partition");
      if(flags.version==FOUR) maximum_number_of_options=3;
      }

    /* Display Program Name and Copyright Information */
    Clear_Screen(NULL);

    if(menu!=MM) flags.display_name_description_copyright=FALSE;

    if(flags.display_name_description_copyright==TRUE)
      {
      Print_Centered(0,program_name,STANDARD);
      Print_Centered(1,program_description,STANDARD);
      Print_Centered(2,copyleft,STANDARD);
      }

    /* Display Menu Title(s) */
    Print_Centered(4,title,BOLD);

    /* Display Current Drive Number */
    Position_Cursor(4,6);
    printf("Current fixed disk drive: ");
    cprintf("%d",(flags.drive_number-127));

    if(menu==DP)
      {
      /* Ensure that primary partitions are available to delete. */
      counter=0;
      index=0;

      do
	{
	if(part_table[(flags.drive_number-128)]
	 .pri_part_num_type[index]>0)
	 counter++;
	index++;
	}while(index<4);

      if(counter==0)
	{
	Position_Cursor(4,22);
	cprintf("No partitions to delete.");
	Position_Cursor(4,24);
	printf("                                        ");
	Input(0,0,0,ESC,0,0,ESCC,0,0);
	menu=MM;
	return(menu);
	}
      }

    /* Display Menu */
    Position_Cursor(4,8);
    printf("Choose one of the following:");

    Position_Cursor(4,10);
    cprintf("1.  ");
    printf("%s",option_1);

    if(maximum_number_of_options>1)
      {
      Position_Cursor(4,11);
      cprintf("2.  ");
      printf("%s",option_2);
      }

    if(maximum_number_of_options>2)
      {
      Position_Cursor(4,12);
      cprintf("3.  ");
      printf("%s",option_3);
      }

    if(maximum_number_of_options>3)
      {
      Position_Cursor(4,13);
      cprintf("4.  ");
      printf("%s",option_4);
      }

    if( (menu==MM) && (flags.more_than_one_drive==TRUE) )
      {
      maximum_number_of_options=5;
      Position_Cursor(4,14);
      cprintf("5.  ");
      printf("%s",option_5);
      }

    /* Display Special Messages */

    /* If there is not an active partition */
    if( ( (part_table[(flags.drive_number-128)].pri_part_num_type[0]>0) || (part_table[(flags.drive_number-128)].pri_part_num_type[1]>0) || (part_table[(flags.drive_number-128)].pri_part_num_type[2]>0) || (part_table[(flags.drive_number-128)].pri_part_num_type[3]>0) ) && (flags.drive_number==0x80) && (menu==MM) && (part_table[flags.drive_number-128].active_status[0]==0) && (part_table[flags.drive_number-128].active_status[1]==0) && (part_table[flags.drive_number-128].active_status[2]==0) && (part_table[flags.drive_number-128].active_status[3]==0) )
      {
      Position_Cursor(4,21);
      cprintf("WARNING! ");
      printf("No partitions are set active - disk 1 is not startable unless");
      Position_Cursor(4,22);
      printf("a partition is set active");
      }

    /* Get input from user */
    Position_Cursor(4,17);
    printf("Enter choice: ");

    if(menu==MM) input=Input(1,19,17,NUM,1,maximum_number_of_options,ESCE,1,0);
    else input=Input(1,19,17,NUM,1,maximum_number_of_options,ESCR,-1,0);

    /* Process the input */
    if(input!=0)
      {
      if(menu==MM) menu=input<<4;
      else menu=menu|input;
      }
    else
      {
      if(menu==MM)
	{
	menu=EXIT;
	}
      else
	{
	if(menu>0x0f) menu=MM;
	else menu=menu&0xf0;
	}
      }

    if( (menu==MM) || (menu==CP) || (menu==DP) ) display_menu=TRUE;
    else display_menu=FALSE;

    }while(display_menu==TRUE);

  /* */
  flags.display_name_description_copyright=FALSE;

  return(menu);
}

/*
/////////////////////////////////////////////////////////////////////////////
//  MAIN ROUTINE
/////////////////////////////////////////////////////////////////////////////
*/
void main(int argc, char *argv[])
{
  int index;
  int location;

  /* Place the filename of this program into filename */
  index=strlen(argv[0]);
  location=0;
  do
    {
    if(argv[0] [index]=='\\')
      {
      location=index+1;
      index=-1;
      }
    index--;
    }while(index>=0);

  index=location;
  do
    {
    filename[index-location]=argv[0] [index];
    index++;
    }while(index<=(strlen(argv[0])) );

  index=0;
  do
    {
    if(filename[index]=='.') filename[index]=0;
    index++;
    }while(index<12);

  /* Place the path of this program into path. */
  if(location>0)
    {
    index=0;
    do
      {
      path[index]=argv[0] [index];

      index++;
      }while(index<location);
    path[index]=0;
    }
  else path[0]=0;

  Initialization();

  if(debug.path==TRUE)
    {
    printf("\nThe PATH to \"%s\" is:  ",filename);
    printf("\"%s\"\n\n",path);
    Pause();
    }

  /* If "FDISK" is typed without any options */
  if(argc==1)
    {
    /* Ask the user if FAT32 is desired. */
    if( (flags.version==W95B) || (flags.version==W98) ) Ask_User_About_FAT32_Support();

    Interactive_User_Interface();
    exit(0);
    }

  /* If "FDISK" is typed with MS-FDISK switches */
  /* ****************************************** */

  if(argc>=2)
    {

    /* if "FDISK /ACTOK" is entered */
    if(0==stricmp("/ACTOK",argv[1]))
      {
      /* Ask the user if FAT32 is desired. */
      if( (flags.version==W95B) || (flags.version==W98) ) Ask_User_About_FAT32_Support();

      Interactive_User_Interface();
      exit(0);
      }

    /* if "FDISK /EXT:<size> <disk>" is entered */
    if( (0==strncmp("/EXT",argv[1],3) ) || (0==strncmp("/ext",argv[1],3) ) )
      {
      char char_size[4];

      long size=0;

      char_size[0]=argv[1] [5];
      char_size[1]=argv[1] [6];
      char_size[2]=argv[1] [7];
      char_size[3]=argv[1] [8];

      flags.drive_number=((argv[2] [0])-48)+127;
      size=atol(char_size);

      /* check to make sure the drive is a legitimate number */
      if( (flags.drive_number<128) || (flags.drive_number>flags.maximum_drive_number) )
	{
	printf("\nInvalid drive designation...Operation Terminated.\n");
	exit(5);
	}

      Determine_Free_Space();

      if(part_table[(flags.drive_number-128)].ext_int_13==TRUE)
       Create_Primary_Partition(0x0f,size);
      else Create_Primary_Partition(5,size);

      Write_Partition_Tables();
      exit(0);
      }

    /* if "FDISK /FPRMT" is entered */
    if(0==stricmp("/FPRMT",argv[1]))
      {
      /* Functionallity to be added later */
      printf("\nThe function selected has not yet been implemented...Operation Terminated.\n");
      exit(10);

      //exit(0);
      }

    /* if "FDISK /PRI:<size> <disk>" is entered */
    if( (0==strncmp("/PRI",argv[1],3) ) || (0==strncmp("/pri",argv[1],3) ) )
      {
      int numeric_type;
      char char_size[4];
      long size=0;

      char_size[0]=argv[1] [5];
      char_size[1]=argv[1] [6];
      char_size[2]=argv[1] [7];
      char_size[3]=argv[1] [8];

      flags.drive_number=((argv[2] [0])-48)+127;
      size=atol(char_size);

      /* check to make sure the drive is a legitimate number */
      if( (flags.drive_number<128) || (flags.drive_number>flags.maximum_drive_number) )
	{
	printf("\nInvalid drive designation...Operation Terminated.\n");
	exit(5);
	}

      numeric_type=Partition_Type_To_Create(size);

      Determine_Free_Space();
      Set_Active_Partition(Create_Primary_Partition(numeric_type,size));

      Write_Partition_Tables();
      exit(0);
      }

    /* if "FDISK /PRIO:<size> <disk>" is entered */
    if( (0==strncmp("/PRIO",argv[1],4) ) || (0==strncmp("/prio",argv[1],4) ) )
      {
      /* Functionallity to be added later */
      printf("\nThe function selected has not yet been implemented...Operation Terminated.\n");
      exit(10);

      //exit(0);
      }

    /* if "FDISK /STATUS" is entered */
    if(0==stricmp("/STATUS",argv[1]))
      {
      flags.monochrome=TRUE;
      textcolor(7);
      Clear_Screen(NULL);
      Print_Centered(1,"Fixed Disk Drive Status",0);
      Display_All_Drives();
      exit(0);
      }

    /* if "FDISK /X" is entered */
    if(0==stricmp("/X",argv[1]))
      {
      /* Ask the user if FAT32 is desired. */
      if( (flags.version==W95B) || (flags.version==W98) ) Ask_User_About_FAT32_Support();

      flags.use_extended_int_13=FALSE;
      index=0;
      do
	{
	part_table[index].ext_int_13=FALSE;
	index++;
	}while(index<8);

      Read_Partition_Tables();
      Interactive_User_Interface();
      exit(0);
      }
    }

  if(0==stricmp("/?",argv[1]))
    {
    if(0==stricmp("/NOPAUSE",argv[2])) flags.do_not_pause_help_information=TRUE;
    else flags.do_not_pause_help_information=FALSE;

    Display_Help_Screen();
    exit(0);
    }

  /* If FDISK is typed with Free FDISK switches */
  /* ****************************************** */

  /* First, check to see if a drive number was entered */
  if(argc>=3) flags.drive_number=((argv[2] [0])-48)+127;
  else flags.drive_number=0x80;

  /* Ensure that the drive number is valid */
  if( (flags.drive_number<0x80) || (flags.drive_number>flags.maximum_drive_number) )
    {
    printf("\nIncorrect drive designation...Operation Terminated.\n");
    exit(5);
    }

  if(argc>=2)
    {
    /* if "FDISK /A" is entered */
    if(0==stricmp("/A",argv[1]))
      {
      Automatically_Partition_Hard_Drive();
      Write_Partition_Tables();

      exit(0);
      }

    /* if "FDISK /ACTIVATE" is entered */
    /* Syntax:  FDISK [/ACTIVATE drive# partition#] */
    if(0==stricmp("/ACTIVATE",argv[1]))
      {
      if(argc<4)
	{
	printf("\nSyntax Error...Operation Terminated.\n");
	exit(1);
	}
      int partition_number=atoi(argv[3]);

      if((partition_number<1) || (partition_number>4))
	{
	printf("\nPartition number is out of range (1-4)...Operation Terminated.\n");
	exit(9);
	}

      Set_Active_Partition(partition_number-1);
      Write_Partition_Tables();

      exit(0);
      }

    /* if "FDISK /AMBR" is typed */
    /* Syntax: FDISK [/AMBR drive#] */
    if(0==stricmp("/AMBR",argv[1]))
      {
      Create_Alternate_MBR();
      exit(0);
      }

    /*if "FDISK /C" is typed */
    if(0==stricmp("/C",argv[1]))
      {
      Clear_Partition_Table();
      exit(0);
      }

    /* if "FDISK /CLEARFLAG" is typed */
    if(0==stricmp("/CLEARFLAG",argv[1]))
      {
      int flag_number;

      if( (argc!=2) && (argc!=4) )
	{
	printf("\nSyntax Error...Operation Terminated.\n");
	exit(1);
	}

      if(argc==4) flag_number=atoi(argv[3]);
      else flag_number=1;

      if( (flag_number<1) || (flag_number>64) )
	{
	printf("\nFlag number is out of range...Operation Terminated.\n");
	exit(9);
	}

      Clear_Flag(flag_number);
      }

    /* if "FDISK /D" is typed */
    /* Syntax:  FDISK [/D drive# {partition# || /E || [/L driveletter]} ] */
    if(0==stricmp("/D",argv[1]))
      {
      int partition_number;

      if(argc<4)
	{
	printf("\nSyntax Error...Operation Terminated.\n");
	exit(1);
	}

      if(0==stricmp("/L",argv[3]))
	{
	if(part_table[(flags.drive_number-128)].num_of_log_drives==0)
	  {
	  printf("\nNo Logical DOS Drives exist...Operation Terminated.\n");
	  exit(9);
	  }

	printf("\nThe function selected has not yet been implemented...Operation Terminated.\n");
	exit(10);

	/* Write_Partition_Tables();
	exit(0);  */
	}

      if(0==stricmp("/E",argv[3]))
	{
	if(part_table[(flags.drive_number-128)].ext_part_exists!=TRUE)
	  {
	  printf("\nExtended partitition does not exist...Operation Terminated.\n");
	  exit(9);
	  }

	Delete_Primary_Partition(part_table[(flags.drive_number-128)].num_of_ext_part);
	Write_Partition_Tables();
	exit(0);
	}

      /* If the program has made it to this point, assume that the user */
      /* entered the primary partition number to delete.                */
      partition_number=atoi(argv[3]);

      if((partition_number<1) || (partition_number>4))
	{
	printf("\nPartition number is out of range (1-4)...Operation Terminated.\n");
	exit(9);
	}

      Delete_Primary_Partition(partition_number);
      Write_Partition_Tables();
      exit(0);
      }

    /*if "FDISK /DEACTIVATE" is typed */
    if(0==stricmp("/DEACTIVATE",argv[1]))
      {
      Clear_Active_Partition();
      Write_Partition_Tables();
      exit(0);
      }

    /* if "FDISK /DUMP" is typed */
    if(0==stricmp("/DUMP",argv[1]))
      {
      Dump_Partition_Information();
      exit(0);
      }

    /*if "FDISK /I [/drive# [/tech]]" is typed */
    if(0==stricmp("/I",argv[1]))
      {
      if( (argc==4) && (0==stricmp("/TECH",argv[3])) ) flags.extended_options_flag=TRUE;
      if( (argc==4) && (0!=stricmp("/TECH",argv[3])) )
	{
	printf("\nSyntax Error...Operation Terminated.\n");
	exit(1);
	}

      Display_CL_Partition_Table();
      exit(0);
      }

    /* if "FDISK /L" is typed */
    if(0==stricmp("/L",argv[1]))
      {
      flags.monochrome=TRUE;
      textcolor(7);
      Clear_Screen(NULL);
      Display_All_Drives();
      exit(0);
      }

    /* if "FDISK [/M drive# primarypartition# newtype#]" is entered */
    if(0==stricmp("/M",argv[1]))
      {
      int partition_number;
      int type_number;

      partition_number=atoi(argv[3]);

      if((partition_number<1) || (partition_number>4))
	{
	printf("\nPrimary partition number is out of range...Operation Terminated.\n");
	exit(9);
	}

      type_number=atoi(argv[4]);

      if((type_number<=0) || (type_number>255))
	{
	printf("\nNew partition type is out of range...Operation Terminated.\n");
	exit(9);
	}

      Modify_Partition_Type((partition_number-1),type_number);

      Write_Partition_Tables();
      exit(0);
      }

    /* if "FDISK /MBR" is typed */
    /* Syntax:  FDISK [/MBR drive#] */
    /* "/CMBR" is added for MS-FDISK compatibility */
    if( (0==stricmp("/MBR",argv[1])) || (0==stricmp("/CMBR",argv[1])) )
      {
      Create_MBR();
      exit(0);
      }

    /* if "FDISK /N" is typed*/
    /*   Syntax: FDISK /N drive# partition_type size [/P] [/S type]*/
    if(0==stricmp("/N",argv[1]))
      {
      int numeric_type;
      int partition_type=NULL;
      int special_flag=FALSE;

      unsigned long maximum_partition_size_in_MB;
      unsigned long size;

      if(argc<5)
	{
	printf("\nSyntax Error...Operation Terminated.\n");
	exit(1);
	}

      Determine_Free_Space();

      /* Get partition size */
      size=atol(argv[4]);

      /* Determine partition_type */
      if(0==stricmp("P",argv[3])) partition_type=PRIMARY;

      if(0==stricmp("E",argv[3])) partition_type=EXTENDED;

      if(0==stricmp("L",argv[3]))
	{
	if(part_table[(flags.drive_number-128)].ext_part_exists==FALSE)
	  {
	  printf("\nNo Extended Partition Exists...Operation Terminated.\n");
	  exit(9);
	  }

	partition_type=LOGICAL;

	if(1==Determine_Drive_Letters())
	  {
	  printf("\nMaximum number of Logical DOS Drives installed...Operation Terminated.\n");
	  exit(9);
	  }
	}

      if(partition_type==NULL)
	{
	printf("\nSyntax Error...Operation Terminated.\n");
	exit(1);
	}

      /* Compute size if it is a percentage, if necessary. */
      if( ( (argc==6) || (argc==8) ) && ( (0==stricmp("/P",argv[5])) || (0==stricmp("/P",argv[7])) ) )
	{
	Determine_Free_Space();
	if( (partition_type==PRIMARY) || (partition_type==EXTENDED) )size=(size*(((part_table[(flags.drive_number-128)].pri_part_largest_free_space+1)*(part_table[(flags.drive_number-128)].total_head+1)*(part_table[(flags.drive_number-128)].total_sect))/2048))/100;
	if(partition_type==LOGICAL) size=(size*(((part_table[(flags.drive_number-128)].ext_part_largest_free_space+1)*(part_table[(flags.drive_number-128)].total_head+1)*(part_table[(flags.drive_number-128)].total_sect))/2048))/100;
	}

      /* Set special partition type, if necessary. */
      if( (argc>=6) && (argc<=8) && (0==stricmp("/S",argv[(argc-2)]) ) )
	{
	if(0==stricmp("/S",argv[5])) numeric_type=atoi(argv[6]);
	if( (argc==8) && (0==stricmp("/S",argv[6])) ) numeric_type=atoi(argv[7]);

	if( (numeric_type<1) || (numeric_type>255) )
	  {
	  printf("\nSpecial partition type is out of range...Operation Terminated.\n");
	  exit(9);
	  }

	special_flag=TRUE;
	}

      if( (partition_type==PRIMARY) || (partition_type==EXTENDED) )
	{
	/* */
	maximum_partition_size_in_MB=(((part_table[(flags.drive_number-128)].pri_part_largest_free_space+1)*(part_table[(flags.drive_number-128)].total_head+1)*(part_table[(flags.drive_number-128)].total_sect))/2048);

	/* Adjust maximum_partition_size_in_MB depending upon version */
	if( (flags.version==FOUR) && (partition_type!=EXTENDED) && (maximum_partition_size_in_MB>2048) ) maximum_partition_size_in_MB=2048;
	if( (flags.version==FIVE) && (partition_type!=EXTENDED) && (maximum_partition_size_in_MB>2048) ) maximum_partition_size_in_MB=2048;
	if( (flags.version==SIX) && (partition_type!=EXTENDED) && (maximum_partition_size_in_MB>2048) ) maximum_partition_size_in_MB=2048;
	if( (flags.version==W95) && (partition_type!=EXTENDED) && (maximum_partition_size_in_MB>2048) ) maximum_partition_size_in_MB=2048;
	if( ( (flags.version==W95B) || (flags.version==W98) ) && (flags.fat32==FALSE) && (maximum_partition_size_in_MB>2048) ) maximum_partition_size_in_MB=2048;

	if(size>maximum_partition_size_in_MB) size=maximum_partition_size_in_MB;

	if(special_flag==FALSE)
	  {
	  if(partition_type!=EXTENDED) numeric_type=Partition_Type_To_Create(size);
	  else
	    {
	    if(part_table[(flags.drive_number-128)].ext_int_13==TRUE)
	     numeric_type=0x0f;
	    else numeric_type=5;
	    }
	  }

	Create_Primary_Partition(numeric_type,size);
	}
      else
	{
	/* */
	maximum_partition_size_in_MB=(((part_table[(flags.drive_number-128)].ext_part_largest_free_space+1)*(part_table[(flags.drive_number-128)].total_head+1)*(part_table[(flags.drive_number-128)].total_sect))/2048);

	/* Adjust maximum_partition_size_in_MB depending upon version */
	if( (flags.version==FOUR) && (maximum_partition_size_in_MB>2048) ) maximum_partition_size_in_MB=2048;
	if( (flags.version==FIVE) && (maximum_partition_size_in_MB>2048) ) maximum_partition_size_in_MB=2048;
	if( (flags.version==SIX) && (maximum_partition_size_in_MB>2048) ) maximum_partition_size_in_MB=2048;
	if( (flags.version==W95) && (maximum_partition_size_in_MB>2048) ) maximum_partition_size_in_MB=2048;
	if( ( (flags.version==W95B) || (flags.version==W98) ) && (flags.fat32==FALSE) && (maximum_partition_size_in_MB>2048) ) maximum_partition_size_in_MB=2048;

	if(size>maximum_partition_size_in_MB) size=maximum_partition_size_in_MB;

	if(special_flag==FALSE) numeric_type=Partition_Type_To_Create(size);

	Create_Logical_Drive(numeric_type,size);
	}

      Write_Partition_Tables();
      exit(0);
      }

    /* if "FDISK /REBOOT" is typed */
    if(0==stricmp("/REBOOT",argv[1])) Reboot_PC();


    /* if "FDISK /RESIZE" is typed */
    if(0==stricmp("/RESIZE",argv[1]))
      {
      printf("\nThe function selected has not yet been implemented...Operation Terminated.\n");
      exit(10);
      }

    /* if "FDISK /RMBR" is typed */
    /* Syntax:  FDISK [/RMBR drive#] */
    if(0==stricmp("/RMBR",argv[1]))
      {
      Remove_MBR();
      exit(0);
      }

    /* if "FDISK /SETFLAG" is typed */
    /* Syntax: FDISK /SETFLAG [drive# flag#] */
    if(0==stricmp("/SETFLAG",argv[1]))
      {
      int flag_number;

      if( (argc!=2) && (argc!=4) )
	{
	printf("\nSyntax Error...Operation Terminated.\n");
	exit(1);
	}

      if(argc==4) flag_number=atoi(argv[3]);
      else flag_number=1;

      if( (flag_number<1) || (flag_number>64) )
	{
	printf("\nFlag number is out of range...Operation Terminated.\n");
	exit(9);
	}

      Set_Flag(flag_number);
      }

    /* if "FDISK /SMBR" is typed */
    /* Syntax: FDISK [/SMBR drive#] */
    if(0==stricmp("/SMBR",argv[1]))
      {
      Save_MBR();
      exit(0);
      }

    /* if "FDISK /TESTFLAG [drive# flag#]" is typed */
    if(0==stricmp("/TESTFLAG",argv[1]))
      {
      int flag_number;

      if( (argc!=2) && (argc!=4) )
	{
	printf("\nSyntax Error...Operation Terminated.\n");
	exit(1);
	}

      if(argc==4) flag_number=atoi(argv[3]);
      else flag_number=1;

      if( (flag_number<1) || (flag_number>64) )
	{
	printf("\nFlag number is out of range...Operation Terminated.\n");
	exit(9);
	}

      Test_Flag(flag_number);
      }
    }

  printf("\nSyntax Error...Operation Terminated.\n");
  exit(1);
}
