	page    59,132
	title   UDVD -- DOS CD/DVD Driver.
;
; This is a DOS driver for up to 3 CD/DVD drives on PC systems.   On
; loading, it tests the IDE channels and runs the first three CD/DVD 
; drives found.   Drives capable of UltraDMA will use it, unless /UX
; disables UltraDMA or no controller is found.   An XMS manager with
; 128K of XMS memory is required.   When UDMA is loaded, UDVD shares
; UDMA's 64K "main" XMS buffer.   If CD/DVD "raw" input is unneeded,
; all UDVD file data will be cached through the UDMA driver!
;
; General Program Equations.
;
	.386p			;Allow use of 80386 instructions.
s	equ	<short>		;Default conditional jumps to "short".
MAXBIOS	equ	17		;Maximum BIOS units supported by UDMA.
MSEL	equ	0A0h		;"Master" device-select bits.
SSEL	equ	0B0h		;"Slave"  device-select bits.
RMAXLBA	equ	00006DD39h	;Redbook (audio) maximum LBA value.
COOKSL	equ	2048		;CD/DVD "cooked" sector length.
RAWSL	equ	2352		;CD/DVD "raw" sector length.
CMDTO	equ	00Ah		;500-msec minimum command timeout.
SEEKTO	equ	037h		;3-second minimum "seek"  timeout.
STARTTO	equ	07Fh		;7-second minimum startup timeout.
INT13S	equ	((4*013h)+2)	;Int 13h low-memory segment address.
BIOSTMR equ	0046Ch		;BIOS "tick" timer address.
VDSFLAG equ	0047Bh		;BIOS "Virtual DMA" flag address.
HDI_OFS	equ	0048Eh-BIOSTMR	;BIOS hard-disk int. flag "offset".
IXM	equ	2048		;IOCTL transfer-length multiplier.
CR	equ	00Dh		;ASCII carriage-return.
LF	equ	00Ah		;ASCII line-feed.
TAB	equ	009h		;ASCII "tab".
;
; IDE Controller Register Definitions.
;
CDATA	equ	001F0h		;Data port.
CSECCT	equ	CDATA+2		;I-O sector count.
CDSEL	equ	CDATA+6		;Drive-select and upper LBA.
CCMD	equ	CDATA+7		;Command register.
CSTAT	equ	CDATA+7		;Primary status register.
;
; Controller Status and Command Definitions.
;
BSY	equ	080h		;IDE controller is busy.
DRQ	equ	008h		;IDE data request.
ERR	equ	001h		;IDE general error flag.
DMI	equ	004h		;DMA interrupt occured.
DME	equ	002h		;DMA error occurred.
LBABITS equ	0E0h		;Fixed LBA command bits.
;
; Byte, Word, and Double-Word Definitions.
;
BDF	struc
lb	db	?
hb	db	?
BDF	ends

WDF	struc
lw	dw	?
hw	dw	?
WDF	ends

DDF	struc
dwd	dd	?
DDF	ends
;
; DOS "Request Packet" Layout.
;
RP	struc
RPHLen	db	?		;Header byte count.
RPSubU	db	?		;Subunit number.
RPOp	db	?		;Opcode.
RPStat	dw	?		;Status word.
	db	8 dup (?)	;(Unused by us).
RPUnit	db	?		;Number of units found.
RPLen	dw	?		;Resident driver 32-bit length.
RPSeg	dw	?		;(Segment must ALWAYS be set!).
RPCL	dd	?		;Command-line data pointer.
RP	ends
RPERR	equ	08003h		;Packet "error" flags.
RPDON	equ	00100h		;Packet "done" flag.
RPBUSY	equ	00200h		;Packet "busy" flag.
;
; IOCTL "Request Packet" Layout.
;
IOC	struc
	db	13 dup (?)	;Request "header" (unused by us).
	db	?		;Media descriptor byte (Unused by us).
IOCAdr	dd	? 		;Data-transfer address.
IOCLen	dw	?		;Data-transfer length.
	dw	?		;Starting sector (unused by us).
	dd	?		;Volume I.D. pointer (unused by us).
IOC	ends
;
; Read Long "Request Packet" Layout.
;
RL	struc
	db	13 dup (?)	;Request "header" (unused by us).
RLAM	db	?		;Addressing mode.
RLAddr	dd	?		;Data-transfer address.
RLSC	dw	?		;Data-transfer sector count.
RLSec	dd	?		;Starting sector number.
RLDM	db	?		;Data-transfer mode.
RLIntlv	db	?		;Interleave size.
RLISkip	db	?		;Interleave skip factor.
RL	ends
;
; Segment Declarations.
;
CODE	segment	public use16 'CODE'
	assume	cs:CODE,ds:CODE
;
; DOS Driver Device Header.
;
@	dd	0FFFFFFFFh	;Link to next header block.
	dw	0C800h		;Driver "device attributes".
	dw	Strat		;"Strategy" routine pointer.
	dw	DevInt		;"Device Interrupt" routine pointer.
DvrNam	db	'UDVD1   '	;DOS "device name" (UDVD1 default).
	dw	0		;(Reserved).
	db	0		;First assigned drive letter.
Units	db	0		;Number of CD/DVD drives (1 or 2).
;
; Resident Driver Variables.
;
EntryP	dw	(I_Init-@)	;"Device Interrupt" entry pointer.
Flush	dd	(RLRdX-@)	;UDMA "flush" subroutine address.
RqPkt	dd	0		;DOS request-packet address.
UserB	dd	0		;Working user I-O buffer address.
UserL	dw	0		;Working user I-O buffer length.
AudAP	dw	ScanP		;Current audio-start address pointer.
DMAAd	dw	0FFFFh		;Current DMA command address.
IDEAd	dw	00100h		;Current IDE data-register address.
VLF	db	0		;VDS "lock" flag (001h = buffer lock).
Try	db	0		;I-O retry counter.
DMF	db	0		;DMA input flag  (001h if so).
XMF	db	0		;XMS "move" flag (001h = move needed).
;
; ATAPI "Packet" Area (always 12 bytes for a CD/DVD).
;
Packet	db	0		;Opcode.
	db	0		;Unused (LUN and reserved).
PktLBA	dd	0		;CD/DVD logical block address.
PktLH	db	0		;"Transfer length" (sector count).
PktLn	dw	0		;Middle- and low-order sector count.
PktRM	db	0		;Read mode ("Raw" Read Long only).
	dw	0		;Unused ATAPI "pad" bytes (required).
;
; VDS Parameter Block.
;
PRDAd	dd	(IOAdr-@)	;PRD 32-bit command addr. (Init set).
VDSLn	dd	(ResEnd-@)	;VDS buffer length.
VDSOf	dd	0		;VDS 32-bit buffer offset.
VDSSg	dd	0		;VDS 16-bit segment (hi-order zero).
VDSAd	dd	0		;VDS 32-bit "virtual" buffer address.
;
; UltraDMA Command-List.
;
IOAdr	dd	0		;VDS and DMA 32-bit address.
IOLen	dd	080000000h	;DMA byte count and "end" flag.
;
; Audio Function Buffer (16 bytes) for most CD/DVD "audio" requests.
;   The variables below are used only during driver initialization.
;
InBuf	equ	$
UTblP	dw	UTable		;Initialization unit table pointer.
XBHdl	dw	0		;XMS buffer "handle".
XBOfs	dw	0		;XMS buffer "offset".
NoDMA	db	0		;"No UltraDMA" flag.
UMode	dw	0		;UltraDMA "mode select" bits.
UFlag	db	0		;UltraDMA "mode valid" flags.
CCTbl	db	0FAh,0F0h,08Ah	;PCI Class-Code "Interface" bytes.
	db	080h,0BAh,0B0h
CCEnd	equ	$
;
; Unit Parameter Tables.
;
UTable	db	0FFh		;Unit 0 low DMA addr. (set by Init).
	db	0FFh		;	low IDE addr. (set by Init).
	db	0FFh		;	Device-select (set by Init).
	db	0FFh		;	Media-change flag.
	dd	0FFFFFFFFh	;	Current audio-start address.
	dd	0FFFFFFFFh	;	Current audio-end   address.
	dd	0FFFFFFFFh	;	Last-session starting LBA.
	db	0FFh		;Unit 1 low DMA addr. (set by Init).
	db	0FFh		;	low IDE addr. (set by Init).
	db	0FFh		;	Device-select (set by Init).
	db	0FFh		;	Media-change flag.
	dd	0FFFFFFFFh	;	Current audio-start address.
	dd	0FFFFFFFFh	;	Current audio-end   address.
	dd	0FFFFFFFFh	;	Last-session starting LBA.
	db	0FFh		;Unit 2 low DMA addr. (set by Init).
	db	0FFh		;	low IDE addr. (set by Init).
	db	0FFh		;	Device-select (set by Init).
	db	0FFh		;	Media-change flag.
	dd	0FFFFFFFFh	;	Current audio-start address.
	dd	0FFFFFFFFh	;	Current audio-end   address.
	dd	0FFFFFFFFh	;	Last-session starting LBA.
UTblEnd	equ	$		;(End of all unit tables).
;
; Dispatch Table for DOS CD/DVD request codes 0 through 14.
;
DspTbl1	dw	DspLmt1		;Number of valid request codes.
	dw	Try2D		;Invalid-request handler address.
DspTblA	dw	UnSupp		;00 -- Initialization  (special).
	dw	UnSupp		;01 -- Media Check	(unused).
	dw	UnSupp		;02 -- Build BPB	(unused).
	dw	Try3D		;03 -- IOCTL Input.
	dw	UnSupp		;04 -- Input		(unused).
	dw	UnSupp		;05 -- Input no-wait	(unused).
	dw	UnSupp		;06 -- Input Status	(unused).
	dw	UnSupp		;07 -- Input flush	(unused).
	dw	UnSupp		;08 -- Output		(unused).
	dw	UnSupp		;09 -- Output & verify	(unused).
	dw	UnSupp		;10 -- Output status	(unused).
	dw	UnSupp		;11 -- Output flush	(unused).
	dw	Try4D		;12 -- IOCTL Output.
	dw	Ignore		;13 -- Device Open     (ignored).
	dw	Ignore		;14 -- Device Close    (ignored).
DspLmt1	equ	($-DspTblA)/2	;Request-code limit for this table.
;
; Dispatch Table for DOS CD/DVD request codes 128 through 136.
;
DspTbl2	dw	DspLmt2		;Number of valid request codes.
	dw	UnSupp		;Invalid-request handler address.
DspTblB	dw	RqRead		;128 -- Read Long.
	dw	UnSupp		;129 -- Reserved	(unused).
	dw	RqSeek		;130 -- Read Long Prefetch.
	dw	RqSeek		;131 -- Seek.
	dw	RqPlay		;132 -- Play Audio.
	dw	RqStop		;133 -- Stop Audio.
	dw	UnSupp		;134 -- Write Long	(unused).
	dw	UnSupp		;135 -- Wr. Long Verify	(unused).
	dw	RqRsum		;136 -- Resume Audio.
DspLmt2	equ	($-DspTblB)/2	;Request-code limit for this table.
;
; Dispatch table for IOCTL Input requests.
;
DspTbl3	dw	DspLmt3		;Number of valid request codes.
	dw	UnSupp		;Invalid-request handler address.
DspTblC	dw	ReqDHA +5*IXM	;00 -- Device-header address.
	dw	RqHLoc +6*IXM	;01 -- Current head location.
	dw	UnSupp		;02 -- Reserved		(unused).
	dw	UnSupp		;03 -- Error Statistics	(unused).
	dw	UnSupp		;04 -- Audio chan. info (unused).
	dw	UnSupp		;05 -- Read drive bytes	(unused).
	dw	ReqDS  +5*IXM	;06 -- Device status.
	dw	ReqSS  +4*IXM	;07 -- Sector size.
	dw	ReqVS  +5*IXM	;08 -- Volume size.
	dw	ReqMCS +2*IXM	;09 -- Media-change status.
	dw	ReqADI +7*IXM	;10 -- Audio disk info.
	dw	ReqATI +7*IXM	;11 -- Audio track info.
	dw	ReqAQI +11*IXM	;12 -- Audio Q-channel info.
	dw	UnSupp		;13 -- Subchannel info	(unused).
	dw	UnSupp		;14 -- Read UPC code	(unused).
	dw	ReqASI +11*IXM	;15 -- Audio status info.
DspLmt3	equ	($-DspTblC)/2	;Request-code limit for this table.
;
; Dispatch table for IOCTL Output requests.
;
DspTbl4	dw	DspLmt4		;Number of valid request codes.
	dw	UnSupp		;Invalid-request handler address.
DspTblD	dw	ReqEJ +1*IXM	;00 -- Eject Disk.
	dw	ReqLU +2*IXM	;01 -- Lock/Unlock Door.
	dw	ReqRS +1*IXM	;02 -- Reset drive.
	dw	UnSupp		;03 -- Audio control	(unused).
	dw	UnSupp		;04 -- Write ctl. bytes	(unused).
	dw	ReqCl +1*IXM	;05 -- Close tray.
DspLmt4	equ	($-DspTblD)/2	;Request-code limit for this table.
;
; Driver Request Processor, for all but the initialization request.
;
ReqPr:	sti			;Ensure CPU interrupts are enabled.
	cld			;Ensure FORWARD "string" commands!
	pushad			;Save all CPU registers we will use.
	push	es
	xor	bx,bx		;Zero BX-reg. for relative commands.
	call	ClrPkt		;Clear our ATAPI packet area.
	les	si,[bx+RqPkt-@]	      ;Point to DOS request packet.
	mov	es:[si.RPStat],RPDON  ;Init status to "done".
	movzx	di,es:[si+RPSubU].lb  ;Get unit-table offset.
	shl	di,4
	add	di,(UTable+4-@)	;Set unit's audio-start address ptr.
	mov	[bx+AudAP-@],di
	mov	ax,[di-4]	;Set drive DMA and IDE addresses.
	mov	[bx+DMAAd-@],al
	mov	[bx+IDEAd-@],ah
	mov	al,es:[si+RPOp]	;Get packet request code.
	mov	di,(DspTbl1-@)	;Point to 1st DOS dispatch table.
	call	Dispat		;Dispatch to desired request handler.
	cli			;Disable CPU interrupts.
	pop	es		;Reload all CPU registers we used.
	popad
	pop	ds		;Reload DS-register and CPU flags.
	popf
	retf			;Exit.
;
; Function-Code "Dispatch" Routines.
;
Try2D:	sub	al,080h		;Not request code 0-15:  subtract 128.
	mov	di,(DspTbl2-@)	;Point to 2nd DOS dispatch table.
	jmp s	Dispat		;Go try request-dispatch again.
Try3D:	mov	di,(DspTbl3-@)	;Point to IOCTL Input dispatch table.
	jmp s	TryIOC
Try4D:	mov	di,(DspTbl4-@)	;Point to IOCTL Output dispatch table.
TryIOC:	les	si,es:[si+IOCAdr]  ;Get actual IOCTL request code.
	mov	al,es:[si]
	les	si,[bx+RqPkt-@]	;Reload DOS request-packet address.
Dispat:	cmp	al,[di]		;Is request code out-of-bounds?
	inc	di		;(Skip past table-limit value).
	inc	di
	jae s	Dspat1		;Yes?  Dispatch to error handler!
	inc	di		;Skip past error-handler address.
	inc	di
	xor	ah,ah		;Point to request-handler address.
	shl	ax,1
	add	di,ax
Dspat1:	mov	dx,[di]		;Get handler address from table.
	mov	di,007FFh
	and	di,dx
	xor	dx,di		;IOCTL request (xfr length > 0)?
	jz s	DspGo		;No, dispatch to desired handler.
	shr	dx,11		   ;Ensure correct IOCTL transfer
	mov	es:[si+IOCLen],dx  ;  length is set in DOS packet.
	les	si,es:[si+IOCAdr]  ;Get IOCTL data-transfer address.
DspGo:	push	di		   ;Dispatch to desired handler.
	ret
GenErr:	mov	al,12		;General error!  Get error code.
	jmp s	ReqErr		;Go post packet error code & exit.
UnSupp:	mov	al,3		;Unsupported request!  Get error code.
	jmp s	ReqErr		;Go post packet error code & exit.
SectNF:	mov	al,8		;Sector not found!  Get error code.
ReqErr:	les	si,[bx+RqPkt-@]	;Reload DOS request-packet address.
	mov	ah,081h		;Post error flags & code in packet.
	mov	es:[si+RPStat],ax
Ignore:	ret			;Exit ("ignored" request handler).
;
; DOS "Read Long" handler.
;
RqRead:	call	ValSN		;Validate starting sector number.
	call	MultiS		;Handle Multi-Session disk if needed.
	jc s	ReqErr		;If error, post return code & exit.
	mov	cx,es:[si+RLSC]	;Get request sector count.
	jcxz	Ignore		;If zero, simply exit.
	xchg	cl,ch		;Save swapped sector count.
	mov	[bx+PktLn-@],cx
	cmp	es:[si.RLDM],1	;"Cooked" or "raw" read mode?
@RM1	equ	[$-1].lb	;(Set to 0 if no "Raw mode" by Init).
	ja s	SectNF		;No?  Return "sector not found"!
	mov	dl,028h		;Get "cooked" input values.
	mov	ax,COOKSL
@RM2:	jb s	RqRL1		;If "cooked" input, set values.
	mov	dl,0BEh		;Get "raw" input values.
	mov	ax,RAWSL
	mov	[bx+PktRM-@].lb,0F8h  ;Set "raw" input flags.
RqRL1:	mov	[bx+Packet-@].lb,dl   ;Set "packet" opcode.
	mul	es:[si+RLSC]	      ;Get desired input byte count.
	test	dx,dx		      ;More than 64K bytes desired?
	jnz s	SectNF		      ;Yes?  Return sector not found!
	mov	[bx+VDSLn-@],ax	      ;Set VDS and DMA byte counts.
	mov	[bx+IOLen-@],ax
	les	ax,es:[si+RLAddr]     ;Set user input-buffer address.
	mov	[bx+VDSOf-@],ax
	mov	[bx+VDSSg-@],es
	mov	[bx+UserB+2-@],es
	or	[bx+VDSAd-@].dwd,-1   ;Invalidate VDS address.
	mov	ax,08103h	      ;VDS "lock" user buffer.
	mov	dl,00Ch
	call	VDSReq
	jnc s	RqRL2		      ;If no error, get buffer addr.
	call	Flush		      ;Ask UDMA to "flush" its cache.
	jmp	RqRL8		      ;Must use our "PIO mode" input.
RqRL2:	les	si,[bx+RqPkt-@]	      ;Reload request packet address.
	mov	eax,[bx+VDSAd-@]      ;Get 32-bit VDS buffer address.
	cmp	eax,-1		      ;Is VDS address not "all-ones"?
	jb s	RqRL3		      ;Yes, set VDS "lock" flag.
	movzx	eax,[bx+VDSSg-@].lw   ;VDS logic is NOT present --
	shl	eax,4		      ;  set 20-bit buffer address.
	add	eax,[bx+VDSOf-@]
	mov	[bx+VDSAd-@],eax
	nop			      ;(Unused alignment "filler").
RqRL3:	adc	[bx+VLF-@],bl	      ;Set VDS lock flag from carry.
	test	[bx+DMAAd-@].lb,001h  ;Is this drive using UltraDMA?
	jnz s	RqRL6		      ;No, must do "PIO mode" input.
	test	al,003h		      ;Is user buffer 32-bit aligned?
	jnz s	RqRL4		      ;No, use our XMS memory buffer.
	cmp	[bx+VDSAd-@].hw,009h  ;Is DMA beyond our 640K limit?
	ja s	RqRL4		      ;Yes, use our XMS memory buffer.
	mov	cx,ax		      ;Get lower ending DMA address.
	add	cx,[bx+IOLen-@]
	cmc			      ;Would I-O cross a 64K boundary?
	jbe s	RqRL5		      ;No, set final buffer address.
RqRL4:	inc	[bx+XMF-@].lb	      ;Post XMS "move" flag for below.
	mov	eax,0		      ;Get our 32-bit XMS buffer addr.
@XBufAd	equ	[$-4].dwd	      ;(XMS buffer address, Init set).
RqRL5:	mov	[bx+IOAdr-@],eax      ;Set final 32-bit buffer addr.
	inc	[bx+DMF-@].lb	      ;Set "DMA input" flag.
RqRL6:	jmp s	RqRL8		      ;If no caching, use logic below.
;
; ***** NOTE *****
;
;   USB or "Firewire" disk drivers can use the following logic to call
;   UDMA and have it cache their I-O requests!   They must assign UDMA
;   unit 20-255 to their devices, NOT 17-19 as below that are reserved
;   for CD/DVD units!    UDMA will either "call back" the USB/Firewire
;   driver if the desired LBA address was not in cache and I-O must be
;   done, or it returns below WITHOUT making a "call back" if data was
;   already cached and the user buffer has been loaded with it!    The
;   carry flag signals an ERROR, and the "call back" logic can post an
;   error code in the AL-reg.   UDMA returns this code in the AH-reg.,
;   to stay compatible with I-O requests handled by the BIOS.
;
	mov	eax,[bx+VDSAd-@]      ;Get 32-bit I-O buffer address.
	movzx	cx,es:[si+RLSC].lb    ;Get sector count and I-O code.
				      ;  CH must be 0 (read) for CDs!
	movzx	bp,es:[si+RPSubU].lb  ;Get our CD/DVD drive number.
	add	bp,MAXBIOS	      ;Convert to UDMA "cache unit".
	xor	di,di		   ;Zero hi-order LBA, unused by CDs.
	push	[bx+PktLBA-@].dwd  ;Swap "packet LBA" back to regular
	pop	si		   ;  "little endian" format for UDMA
	pop	dx		   ;  as the ATAPI fools specify "big
	xchg	dl,dh		   ;  endian" format for I-O packets!
	xchg	dx,si
	xchg	dl,dh
	shl	cl,2		;Multiply sector count and LBA by 4.
	shl	si,1		;  This is only for a CD/DVD driver, 
	rcl	dx,1		;  that has 2048-byte ATAPI sectors.
	rcl	di,1		;  This should be OMITTED by drivers
	shl	si,1		;  using normal 512-byte sectors, as
	rcl	dx,1		;  UDMA "expects" to handle!
	rcl	di,1
	mov	bx,offset (RLRead-@)  ;Load our I-O "call back" addr.
	pushf			;Stack CPU flags, to make this appear
	db	09Ah		;  like an "interrupt", and call UDMA
@ExEntr	dd	0		;  "External Entry" caching routine.
				;  (UDMA "ExEntr" addr. set by Init).
	call	VDRset		;We're BACK!  Restore driver settings.
	jnc s	RqRL10		;If no errors, go reset our I-O flags.
	mov	al,12		;Get CD/DVD "general error" code.
	cmp	ah,001h		;Did UDMA return "invalid function"?
	je s	RqRL7		;Yes, post "general error" instead.
	cmp	ah,0FFh		;Did UDMA return "XMS error"?
	je s	RqRL7		;Yes, post "general error" instead.
	mov	al,ah		;Must be our code -- put it in AL-reg.
RqRL7:	call	ReqErr		;Set error code in I-O request packet.
	jmp s	RqRL10		;Go reset flags & see about "unlock".
	db	0		;(Unused alignment "filler").
;
; End of special "UDMA caching" logic.
;
RqRL8:	mov	ah,005h		;"Non-cached" data input routine --
	call	A20Req		;  issue "A20 local-enable" request.
	mov	al,12		;(Get CD/DVD "general error" code).
	jnz s	RqRL7		;"A20" KAPUT?  Return "general error"!
	push	cs		;Read all desired data.
	call	RLRead
	jnc s	RqRL9		;Any errors during input?
	call	ReqErr		;Yes, post returned error return code.
RqRL9:	mov	ah,006h		;Issue "A20 local-disable" request.
	call	A20Req
	nop			;(Unused alignment "filler").
RqRL10:	mov	[bx+DMF-@],bx	;Reset "DMA input" & "XMS move" flags.
	shr	[bx+VLF-@].lb,1	;Is user buffer "locked" by VDS?
	jnc s	VDExit		;No, just exit below.
	mov	ax,08104h	;"VDS unlock" user buffer, then exit.
	xor	dx,dx
;
; Subroutine to do "VDS lock" and "VDS unlock" requests.
;
VDSReq:	mov	di,(VDSLn-@)	;Point to VDS parameter block.
	push	cs
	pop	es
	int	04Bh		;Execute VDS "lock" or "unlock".
VDRset:	sti			;RESTORE all critical driver settings!
	cld			;(Never-NEVER "trust" external code!).
	push	cs		;Reload this driver's DS-reg.
	pop	ds
	mov	bx,0		;Rezero BX-reg. but save carry bit.
VDExit:	ret			;Exit.
;
; Subroutine to do "A20 local-enable" & "A20 local-disable" requests.
;
A20Req:	db	09Ah		;Call XMS manager to do this request.
@XEntry	dd	0		;(XMS "entry" address, set by Init).
	dec	ax		;Zero AX-reg. if success, -1 if error.
	jmp s	VDRset		;Go restore driver settings and exit.
;
; Subroutine to read all data for a "Read Long" request.   This logic
;   can be "called back" by the UDMA driver in caching CD/DVD drives. 
;
RLRead:	call	VDRset		  ;Make all critical driver settings.
	mov	ecx,8		  ;Move DMA command-list up to XMS.
	mov	esi,(IOAdr-@)
@CmdLst	equ	[$-4].dwd	  ;(Command-list addr., set by Init).
	mov	edi,[bx+PRDAd-@]
	push	cs
	call	RLRdMv
	jc s	RLRdX		  ;Failed?  Return "general error"!
	call	DoIO		  ;Do desired UltraDMA input request.
	jc s	RLRdX		  ;If any errors, exit below.
	shr	[bx+XMF-@].lb,1	  ;Do we need an XMS buffer "move"?
	jnc s	RLRdX		  ;No, just exit below.
	mov	ecx,[bx+VDSLn-@]  ;Get parameters for an XMS move.
	mov	esi,[bx+IOAdr-@]
	mov	edi,[bx+VDSAd-@]
	nop			;(Unused alignment "filler").
RLRdMv:	db	09Ah		  ;Move data from XMS to user buffer.
MoveRtn	dd	(MvData-@)	  ;"XMS Move" routine ptr., Init set).
	mov	al,12		  ;Use "general error" code if error.
RLRdX:	retf			  ;Exit back to UDMA for data caching.
;
; DOS "Seek" handler.
;
RqSeek:	call	RdAST1		;Read current "audio" status.
	call	ClrPkt		;Reset our ATAPI packet area.
	jc s	RqSK2		;If status error, do DOS seek.
	mov	al,[di+1]	;Get "audio" status flag.
	cmp	al,011h		;Is drive in "play audio" mode?
	je s	RqSK1		;Yes, validate seek address.
	cmp	al,012h		;Is drive in "pause" mode?
	jne s	RqSK2		;No, do DOS seek below.
RqSK1:	call	ValSN		;Validate desired seek address.
	mov	di,[bx+AudAP-@]	;Point to audio-start address.
	cmp	eax,[di+4]	;Is address past "play" area?
	ja s	RqSK2		;Yes, do DOS seek below.
	mov	[di],eax	;Update audio-start address.
	call	PlayA		;Issue "Play Audio" command.
	jc s	RqPLE		;If error, post code & exit.
	cmp	[di+1].lb,011h	;Were we playing audio before?
	je s	RqPLX		;Yes, post "busy" status and exit.
	call	ClrPkt		;Reset our ATAPI packet area.
	jmp s	RqStop		;Go put drive back in "pause" mode.
RqSK2:	call	ValSN		;Validate desired seek address.
	call	MultiS		;Handle Multi-Session disk if needed.
	jc s	RqPLE		;If error, post return code & exit.
	mov	[bx+Packet-@].lb,02Bh  ;Set "seek" command code.
RqSK3:	call	DoCmd		;Issue desired command to drive.
	jc s	RqPLE		;If error, post return code & exit.
RqSKX:	ret			;Exit.
;
; DOS "Play Audio" handler.
;
RqPlay:	cmp	es:[si.RLSC].dwd,0  ;Is sector count zero?
	je s	RqSKX		    ;Yes, just exit above.
	mov	eax,es:[si+RLAddr]  ;Validate audio-start address.
	call	ValSN1
	mov	di,[bx+AudAP-@]	;Save drive's audio-start address.
	mov	[di],eax
	add	eax,es:[si+18]	;Calculate audio-end address.
	mov	edx,@MAXLBA.dwd	;Get maximum audio address.
	jc s	RqPL1		;If "end" WAY too big, use max.
	cmp	eax,edx		;Is "end" address past maximum?
	jbe s	RqPL2		;No, use "end" address as-is.
RqPL1:	mov	eax,edx		;Set "end" address to maximum.
RqPL2:	mov	[di+4],eax	;Save drive's audio-end address.
	call	PlayA		;Issue "Play Audio" command.
RqPLE:	jc	ReqErr		;Error!  Post return code & exit.
RqPLX:	jmp	RdAST4		;Go post "busy" status and exit.
;
; DOS "Stop Audio" handler.
;
RqStop:	mov	[bx+Packet-@].lb,04Bh ;Set "Pause/Resume" command.
	jmp	DoCmd		      ;Go pause "audio" and exit.
;
; DOS "Resume Audio" handler.
;
RqRsum:	inc	[bx+PktLn+1-@].lb  ;Set "Resume" flag for above.
	call	RqStop		;Issue "Pause/Resume" command.
	jmp s	RqPLE		;Go exit through "RqPlay" above.
;
; IOCTL Input "Device Header Address" handler.
;
ReqDHA:	push	cs		;Return our base driver address.
	push	bx
	pop	es:[si+1].dwd
	ret			;Exit.
;
; IOCTL Input "Current Head Location" handler.
;
RqHLoc:	mov	[bx+Packet-@].dwd,001400042h   ;Set command bytes.
	mov	al,16		;Set input byte count of 16.
	call	RdAST3		;Issue "Read Subchannel" request.
	jc s	RqPLE		;If error, post return code & exit.
	mov	es:[si+1],bl	;Return "HSG" addressing mode.
	call	SwpLBA		;Return "swapped" head location.
	mov	es:[si+2],eax
	jmp s	RqDSX		;Go post "busy" status and exit.
;
; IOCTL Input "Device Status" handler.
;
ReqDS:	mov	[bx+Packet-@].dwd,0002A005Ah  ;Set up mode-sense.
	mov	al,16		;Use input byte count of 16.
	call	DoBuf1		;Issue mode-sense for hardware data.
RqDS0:	jc s	RqPLE		;If error, post return code & exit.
	mov	eax,00214h	;Get our basic driver status flags.
@RMF	equ	[$-4].lb	;(00210h if no "Raw" mode, Init set).
	mov	cl,070h		;Get media status byte.
	xor	cl,[di+2]
	shr	cl,1		;Door open, or closed with no disk?
	jnz s	RqDS1		;No, check "drive locked" status.
	adc	ax,00800h	;Post "door open" & "no disk" flags.
RqDS1:	test	[di+14].lb,002h	;Drive pushbutton "locked out"?
	jnz s	RqDS2		;No, set flags in IOCTL.
	or	al,002h		;Set "door locked" status flag.
RqDS2:	mov	es:[si+1],eax	;Set status flags in IOCTL buffer.
RqDSX:	jmp	RdAST		;Go post "busy" status and exit.
;
; IOCTL Input "Sector Size" handler.
;
ReqSS:	cmp	es:[si+1].lb,1	;Is read mode "cooked" or "raw"
	ja	GenErr		;No?  Post "general error" & exit.
	mov	ax,RAWSL	;Get "raw" sector length.
	je s	RqSS1		;If "raw" mode, set sector length.
	mov	ax,COOKSL	;Get "cooked" sector length.
RqSS1:	mov	es:[si+2],ax	;Post sector length in IOCTL packet.
	ret			;Exit.
;
; IOCTL Input "Volume Size" handler.
;
ReqVS:	mov	[bx+Packet-@].lb,025h  ;Set "Read Capacity" code.
	mov	al,008h		;Get 8 byte data-transfer length.
	call	DoBuf2		;Issue "Read Capacity" command.
	jc s	RqDS0		;If error, post return code & exit.
	mov	eax,[di]	;Set "swapped" size in IOCTL packet.
	call	Swp32
	mov	es:[si+1],eax
	jmp s	RqATIX		;Go post "busy" status and exit.
;
; IOCTL Input "Media-Change Status" handler.
;
ReqMCS:	call	DoCmd		;Issue "Test Unit Ready" command.
	mov	di,[bx+AudAP-@]	;Get media-change flag from table.
	mov	al,[di-1]
	mov	es:[si+1],al	;Return media-change flag to user.
	ret			;Exit.
;
; IOCTL Input "Audio Disk Info" handler.
;
ReqADI:	mov	al,0AAh		;Specify "lead-out" session number.
	call	RdTOC		;Read disk table-of-contents (TOC).
	jc s	RqASIE		;If error, post return code & exit.
	mov	es:[si+3],eax	;Set "lead out" LBA addr. in IOCTL.
	mov	ax,[di+2]	;Set first & last tracks in IOCTL.
	mov	es:[si+1],ax
	jmp s	RqATIX		;Go post "busy" status and exit.
;
; IOCTL Input "Audio Track Info" handler.
;
ReqATI:	mov	al,es:[si+1]	;Specify desired session (track) no.
	call	RdTOC		;Read disk table-of-contents (TOC).
	jc s	RqASIE		;If error, post return code & exit.
	mov	es:[si+2],eax	;Set track LBA address in IOCTL.
	mov	al,[di+5]
	shl	al,4
	mov	es:[si+6],al
RqATIX:	jmp	RdAST		;Go post "busy" status and exit.
;
; IOCTL Input "Audio Q-Channel Info" handler.
;
ReqAQI:	mov	ax,04010h	;Set "data in", use 16-byte count.
	call	RdAST2		;Read current "audio" status.
	jc s	RqASIE		;If error, post return code & exit.
	mov	eax,[di+5]	;Set ctrl/track/index in IOCTL.
	mov	es:[si+1],eax
	mov	eax,[di+13]	;Set time-on-track in IOCTL.
	mov	es:[si+4],eax
	mov	edx,[di+9]	;Get time-on-disk & clear high
	shl	edx,8		;  order time-on-track in IOCTL.
	jmp s	RqASI4		;Go set value in IOCTL and exit.
;
; IOCTL Input "Audio Status Info" handler.
;
ReqASI:	mov	ax,04010h	;Set "data in", use 16-byte count.
	call	RdAST2		;Read current "audio" status.
RqASIE:	jc	ReqErr		;If error, post return code & exit.
	mov	es:[si+1],bx	;Reset audio "paused" flag.
	xor	eax,eax		;Reset starting audio address.
	xor	edx,edx		;Reset ending audio address.
	cmp	[di+1].lb,011h  ;Is drive now "playing" audio?
	jne s	RqASI1		;No, check for audio "pause".
	mov	di,[bx+AudAP-@]	;Point to drive's audio data.
	mov	eax,[di]	;Get current audio "start" addr.
	jmp s	RqASI2		;Go get current audio "end" addr.
RqASI1:	cmp	[di+1].lb,012h	;Is drive now in audio "pause"?
	jne s	RqASI3		;No, return "null" addresses.
	inc	es:[si+1].lb	;Set audio "paused" flag.
	call	SwpLBA		;Convert time-on-disk to LBA addr.
	call	CvtLBA
	mov	di,[bx+AudAP-@]	;Point to drive's audio data.
RqASI2:	mov	edx,[di+4]	;Get current audio "end" address.
RqASI3:	mov	es:[si+3],eax	;Set audio "start" addr. in IOCTL.
RqASI4:	mov	es:[si+7],edx	;Set audio "end" address in IOCTL.
	ret			;Exit.
;
; IOCTL Output "Eject Disk" handler.
;
ReqEJ:	mov	[bx+Packet-@].lw,0011Bh  ;Set "eject" commands.
	mov	[bx+PktLBA+2-@].lb,002h  ;Set "eject" function.
	jmp	RqSK3			 ;Go do "eject" & exit.
;
; IOCTL Output "Lock/Unlock Door" handler.
;
ReqLU:	mov	al,es:[si+1]	;Get "lock" or "unlock" function.
	cmp	al,001h		;Is function byte too big?
	ja s	RqASIE		;Yes, post "General Error" & exit.
	mov	cx,0001Eh	;Get "lock" & "unlock" commands.
RqLU1:	mov	[bx+Packet-@],cx    ;Set "packet" command bytes.
	mov	[bx+PktLBA+2-@],al  ;Set "packet" function byte.
	call	DoCmd		;Issue desired command to drive.
	jc s	RqASIE		;If error, post return code & exit.
	jmp s	RdAST		;Go post "busy" status and exit.
;
; IOCTL Output "Reset Drive" handler.
;
ReqRS:	call	HltDMA		;Stop previous DMA & select drive.
	inc	dx		;Point to IDE command register.
	mov	al,008h		;Do an ATAPI "soft reset" command.
	out	dx,al
	call	ChkTO		;Await controller-ready.
	jc s	RqASIE		;Timeout!  Return "General Error".
	ret			;Exit.
;
; IOCTL Output "Close Tray" handler.
;
ReqCL:	mov	al,003h		;Get "close tray" function byte.
	mov	cx,0011Bh	;Get "eject" & "close" commands.
	jmp s	RqLU1		;Go do "close tray" command above.
;
; Subroutine to read the current "audio" status and disk address.
;
RdAST:	call	ClrPkt		  ;Status only -- reset ATAPI packet.
RdAST1:	mov	ax,00004h	  ;Clear "data in", use 4-byte count.
RdAST2:	mov	[bx+Packet-@].dwd,001000242h  ;Set command bytes.
	mov	[bx+PktLBA-@],ah  ;Set "data in" flag (RdAST2 only).
RdAST3:	call	DoBuf1		  ;Issue "Read Subchannel" command.
	jc s	RdASTX		  ;If error, exit immediately.
	cmp	[di+1].lb,011h	  ;Is a "play audio" in progress?
	jne s	RdTOC1		  ;No, clear carry flag and exit.
RdAST4:	push	si		  ;Save SI- and ES-regs.
	push	es
	les	si,[bx+RqPkt-@]	  ;Reload DOS request-packet addr.
	or	es:[si.RPStat],RPBUSY  ;Set "busy" status bit.
	pop	es		  ;Reload ES- and SI-regs.
	pop	si
RdASTX:	ret			  ;Exit.
;
; Subroutine to read disk "Table of Contents" (TOC) values.
;
RdTOC:	mov	[bx+Packet-@].lw,00243h  ;Set TOC and MSF bytes.
	call	DoTOC1		;Issue "Read Table of Contents" cmd.
	jc s	RdTOCX		;If error, exit immediately.
	call	SwpLBA		;Return "swapped" starting address.
RdTOC1:	clc			;Clear carry flag (no error).
RdTOCX:	ret			;Exit.
;
; Subroutine to issue a "Play Audio" command.   At entry, the
;   DI-reg. points to the audio-start address for this drive.
;
PlayA:	mov	eax,[di]	;Set "packet" audio-start address.
	call	CvtMSF
	mov	[bx+PktLBA+1-@],eax
	mov	eax,[di+4]	;Set "packet" audio-end address.
	call	CvtMSF
	mov	[bx+PktLH-@],eax
	mov	[bx+Packet-@].lb,047h  ;Set "Play Audio" command.
	jmp s	DoCmd		;Start drive playing audio & exit.
;
; Subroutine to handle a Multi-Session disk for DOS reads and seeks.
;   Multi-Session disks require (A) saving the last-session starting
;   LBA for a new disk after any media-change and (B) "offsetting" a
;   read of the VTOC or initial directory block, sector 16 or 17, to
;   access the VTOC/directory of the disk's last session.
;
MultiS:	mov	di,[bx+AudAP-@]	;Point to drive variables.
	cmp	[di+11].lb,0FFh	;Is last-session LBA valid?
	jne s	MultS1			;Yes, proceed with request.
	mov	[bx+Packet-@].lb,043h	;Set "Read TOC" command.
	inc	[bx+PktLBA-@].lb	;Set "format 1" request.
	call	DoTOC2			;Read first & last session.
	jc s	MultSX			;If any error, exit below.
	mov	[bx+PktLBA-@],bl	;Reset "format 1" request.
	mov	al,[di+3]		;Get last-session number.
	call	DoTOC1		;Read disk info for last session.
	jc s	MultSX		;If error, exit with carry set.
	call	SwpLBA		;"Swap" & save last-session LBA addr.
	mov	di,[bx+AudAP-@]
	mov	[di+8],eax
	call	ClrPkt		   ;Reset our ATAPI packet area.
MultS1:	mov	eax,es:[si+RLSec]  ;Get starting sector number.
	mov	edx,eax		   ;"Mask" sector to an even number.
	and	dl,0FEh
	cmp	edx,16		;Sector 16 (VTOC) or 17 (directory)?
	jne s	MultS2		;No, set sector in packet.
	add	eax,[di+8]	;Offset sector to last-session start.
MultS2:	call	Swp32		;"Swap" sector into packet as LBA.
	mov	[bx+PktLBA-@],eax
	clc			;Clear carry flag (no errors).
MultSX:	ret			;Exit.
;
; Subroutine to ensure UltraDMA is stopped and then select our CD/DVD
;   drive.   For some older chipsets, if UltraDMA is running, reading
;   an IDE register causes the chipset to "HANG"!!
;
HltDMA:	mov	dx,[bx+DMAAd-@]	;Get drive UltraDMA command address.
	test	dl,001h		;Is this drive using UltraDMA?
	jnz s	HltDM1		;No, just select "master" or "slave".
	in	al,dx		;Ensure any previous DMA is stopped!
	and	al,0FEh
	out	dx,al
HltDM1:	mov	dx,[bx+IDEAd-@]	;Point to IDE device-select register.
	add	dx,6
	mov	di,[bx+AudAP-@]	;Get drive's IDE device-select byte.
	mov	al,[di-2]
	out	dx,al		;Select IDE "master" or "slave" unit.
	ret			;Exit.
;
; Ye Olde I-O Subroutine.   ALL of our CD/DVD I-O is executed here!
;
DoTOC1:	mov	[bx+PktLH-@],al	;"TOC" -- set session no. in packet.
DoTOC2:	mov	al,12		;Use 12-byte "TOC" allocation count.
DoBuf1:	mov	[bx+PktLn+1-@],al  ;Buffered -- set packet count.
DoBuf2:	xor	ah,ah		   ;Save data-transfer length.
	mov	[bx+VDSLn-@],ax
	mov	[bx+VDSOf-@].lw,(InBuf-@)  ;Use our buffer for I-O.
	mov	[bx+UserB+2-@],cs
	jmp s	DoIO		;Go start I-O below.
DoCmd:	mov	[bx+VDSLn-@],bx	;Command only -- reset xfr length.
DoIO:	push	si		;Save SI- and ES-registers.
	push	es
	mov	[bx+Try-@].lb,4	;Set request retry count of 4.
DoIO1:	call	HltDMA		;Stop previous DMA & select drive.
	call	ChkTO		;Await controller-ready.
	jc s	DoIO3		;Timeout!  Handle as a "hard error".
	mov	ax,[bx+VDSOf-@]	;Reset data-transfer buffer address.
	mov	[bx+UserB-@],ax
	mov	ax,[bx+VDSLn-@]	;Reset data-transfer byte count.
	mov	[bx+UserL-@],ax
	cmp	[bx+DMF-@],bl	;UltraDMA input request?
	je s	DoIO2		;No, output our ATAPI "packet".
	mov	dx,[bx+DMAAd-@]	;Point to DMA command register.
	mov	al,008h		;Reset DMA commands & set read mode.
	out	dx,al
	inc	dx		;Point to DMA status register.
	inc	dx
	in	al,dx		;Reset DMA status register.
	or	al,006h		;(Done this way so we do NOT alter
	out	dx,al		;  the "DMA capable" status flags!).
	inc	dx		;Set PRD pointer to our DMA address.
	inc	dx
	mov	si,(PRDAd-@)
	outsd
DoIO2:	mov	dx,[bx+IDEAd-@]	;Point to IDE "features" register.
	inc	dx
	mov	al,[bx+DMF-@]	;If UltraDMA input, set "DMA" flag.
	out	dx,al
	add	dx,3		;Point to byte count registers.
	mov	ax,[bx+UserL-@]	;Output data-transfer length.
	out	dx,al
	inc	dx
	mov	al,ah
	out	dx,al
	inc	dx		;Point to command register.
	inc	dx
	mov	al,0A0h		;Issue "Packet" command.
	out	dx,al
	mov	cl,DRQ		;Await controller- and data-ready.
	call	ChkTO1
DoIO3:	jc s	DoIO6		;Timeout!  Handle as a "hard error".
	xchg	ax,si		;Save BIOS timer address.
	mov	dx,[bx+IDEAd-@]	;Point to IDE data register.
	mov	cx,6		;Output all 12 "Packet" bytes.
	mov	si,(Packet-@)
	rep	outsw
	xchg	ax,si		;Reload BIOS timer address.
	mov	ah,STARTTO	;Allow 7 seconds for drive startup.
	cmp	[bx+DMF-@],bl	;UltraDMA input request?
	je s	DoIO8		;No, do "PIO mode" transfer below.
	mov	[bx+UserL-@],bx	;Reset transfer length (DMA does it).
	add	ah,es:[si]	    ;Set 7-second timeout in AH-reg.
	mov	es:[si+HDI_OFS],bl  ;Reset BIOS disk-interrupt flag.
	mov	dx,[bx+DMAAd-@]	    ;Point to DMA command register.
	in	al,dx		;Set DMA Start/Stop bit (starts DMA).
	or	al,001h
	out	dx,al
DoIO4:	inc	dx		;Point to DMA status register.
	inc	dx
	in	al,dx		;Read DMA controller status.
	dec	dx		;Point back to DMA command register.
	dec	dx
	and	al,DMI+DME	;DMA interrupt or DMA error?
	jnz s	DoIO5		;Yes, halt DMA and check results.
	cmp	ah,es:[si]	;Has our DMA transfer timed out?
	je s	DoIO5		    ;Yes?  Halt DMA & post timeout!
	cmp	es:[si+HDI_OFS],bl  ;Did BIOS get a disk interrupt?
	je s	DoIO4		    ;No, loop back & retest status.
	mov	al,DMI		;Set "simulated" interrupt flag.
DoIO5:	xchg	ax,si		;Save ending DMA status.
	in	al,dx		;Reset DMA Start/Stop bit.
	and	al,0FEh
	out	dx,al
	xchg	ax,si		;Reload ending DMA status.
	cmp	al,DMI		;Did DMA end with only an interrupt?
	jne s	DoIO13		;No?  Handle as a "hard error"!
	inc	dx		;Reread DMA controller status.
	inc	dx
	in	al,dx
	test	al,DME		;Any "late" DMA error after DMA end?
	jnz s	DoIO13		;Yes?  Handle as a "hard error"!
	call	ChkTO		;Await final controller-ready.
DoIO6:	jc s	DoIO13		;Timeout!  Handle as a "hard error"!
	jmp s	DoIO12		;Go check for other input errors.
DoIO7:	mov	ah,SEEKTO	;"PIO mode" -- get "seek" timeout.
DoIO8:	xor	cl,cl		;Await controller-ready.
	call	ChkTO2
	jc s	DoIO13		;Timeout!  Handle as a "hard error".
	test	al,DRQ		;Did we also get a data-request?
	jz s	DoIO12		;No, go check for any input errors.
	dec	dx		;Get controller-buffer byte count.
	dec	dx
	in	al,dx
	mov	ah,al
	dec	dx
	in	al,dx
	mov	dx,[bx+IDEAd-@]	;Point to IDE data register.
	mov	si,[bx+UserL-@]	;Get our data-transfer length.
	or	si,si		;Any remaining bytes to input?
	jz s	DoIO10		;No, "eat" all residual data.
	cmp	si,ax		;Remaining bytes > buffer count?
	jbe s	DoIO9		;No, input all remaining bytes.
	mov	si,ax		;Use buffer count as input count.
DoIO9:	les	di,[bx+UserB-@]	;Get input data-transfer address.
	mov	cx,si		;Input all 16-bit data words.
	shr	cx,1
	rep	insw
	add	[bx+UserB-@],si	;Increment data-transfer address.
	sub	[bx+UserL-@],si	;Decrement data-transfer length.
	sub	ax,si		;Any data left in controller buffer?
	jz s	DoIO7		;No, await next controller-ready.
	nop			;(Unused alignment "filler").
DoIO10:	xchg	ax,cx		;"Eat" all residual input data.
	shr	cx,1		;(Should be NO residual data as we
DoIO11:	in	ax,dx		;  always set an exact byte count.
	loop	DoIO11		;  This logic is only to be SAFE!).
	jmp s	DoIO7		;Go await next controller-ready.
DoIO12:	mov	si,[bx+AudAP-@]	;Get drive media-change flag pointer.
	dec	si
	and	ax,00001h	;Did controller detect any errors?
	jz s	DoIO15		;No, see if all data was transferred.
	sub	dx,6		;Get controller's sense key value.
	in	al,dx
	shr	al,4
	cmp	al,006h		;Is sense key "Unit Attention"?
	je s	DoIO16		;Yes, check for prior media-change.
	mov	ah,0FFh		;Get 0FFh M.C. flag for "Not Ready".
	cmp	al,002h		;Is sense key "Drive Not Ready"?
	je s	DoIO17		;Yes, go set our media-change flag.
DoIO13:	mov	dx,[bx+IDEAd-@]	;Hard error!  Point to command reg.
	add	dx,7
	mov	al,008h		;Issue ATAPI "soft reset" to drive.
	out	dx,al
	mov	al,11		;Get "hard error" return code.
DoIO14:	dec	[bx+Try-@].lb	;Do we have more I-O retries left?
	jz s	DoIO18		;No, set carry & return error code.
	jmp	DoIO1		;Try re-executing this I-O request.
DoIO15:	cmp	[bx+UserL-@],bx	;Was all desired data input?
	jne s	DoIO13		;No?  Handle as a hard error.
	mov	[si].lb,001h	;Set "no media change" flag.
	clc			;Reset carry flag (no error).
	jmp s	DoIO19		;Go reload regs. and exit below.
DoIO16:	mov	al,002h		;"Attention":  Get "Not Ready" code.
	cmp	[si],bl		;Is media-change flag already set?
	jle s	DoIO14		;Yes, retry & see if it goes away!
DoIO17:	xchg	ah,[si]		;Load & set our media-change flag.
	call	Flush		;Ask UDMA to "flush" its cache.
	mov	[si+12].lb,0FFh	;Make last-session LBA invalid.
	dec	ah		;Is media-change flag already set?
	jnz s	DoIO18		;Yes, set carry flag and exit.
	mov	al,15		;Return "Invalid Media Change".
DoIO18:	stc			;Set carry flag (error!).
	nop			;(Unused alignment "filler").
DoIO19:	pop	es		;Reload ES- and SI-registers.
	pop	si
	mov	di,(InBuf-@)	;For audio, point to our buffer.
	ret			;Exit.
;
; Subroutine to convert an LBA sector number to "RedBook" MSF format.
;
CvtMSF:	add	eax,150		;Add in offset.
	push	eax		;Get address in DX:AX-regs.
	pop	ax
	pop	dx
	mov	cx,75		;Divide by 75 "frames"/second.
	div	cx
	shl	eax,16		;Set "frames" remainder in upper EAX.
	mov	al,dl
	ror	eax,16
	mov	cl,60		;Divide quotient by 60 seconds/min.
	div	cl
	ret			;Exit -- EAX-reg. contains MSF value.
;
; Subroutine to clear our ATAPI "packet" area.
;
ClrPkt:	mov	[bx+Packet-@],bx    ;Zero 1st 10 ATAPI packet bytes.
	mov	[bx+Packet+2-@],bx  ;(Last 2 are unused "pad" bytes).
	mov	[bx+Packet+4-@],bx
	mov	[bx+Packet+6-@],bx
	mov	[bx+Packet+8-@],bx
	ret			    ;Exit.
;
; Subroutine to validate the starting RedBook disk sector number.
;
ValSN:	mov	eax,es:[si+RLSec]  ;Get starting sector number.
ValSN1:	mov	dl,es:[si+RLAM]	;Get desired addressing mode.
	cmp	dl,001h		;HSG or RedBook addressing?
	ja s	ValSNE		;No?  Return "sector not found".
	je s	ValSN3		;RedBook -- get starting sector.
ValSN2:	ret			;HSG -- exit (accept any DVD value).
ValSN3:	call	CvtLBA		;RedBook -- get starting sector.
	cmp	eax,RMAXLBA	;Is starting sector too big?
@MAXLBA	equ	[$-4].dwd	;(Maximum "audio" LBA, used above).
	jbe s	ValSN2		;No, all is well -- go exit above.
ValSNE:	pop	ax		;Error!  Discard our exit address.
	jmp	SectNF		;Post "sector not found" and exit.
;
; Subroutine to test for I-O timeouts.   At entry, the CL-reg. is
;   008h to test for a data-request, also.   At exit, the DX-reg.
;   points to the IDE primary-status register.   The AH-, SI- and
;   ES-regs. will be lost.
;
ChkTO:	xor	cl,cl		;Check for only controller-ready.
ChkTO1:	mov	ah,CMDTO	;Use 500-msec command timeout.
ChkTO2:	mov	es,bx		;Point to low-memory BIOS timer.
	mov	si,BIOSTMR
	add	ah,es:[si]	;Set timeout limit in AH-reg.
ChkTO3:	cmp	ah,es:[si]	;Has our I-O timed out?
	stc			;(If so, set carry flag).
	je s	ChkTOX		;Yes?  Exit with carry flag on.
	mov	dx,[bx+IDEAd-@]	;Read IDE primary status.
	add	dx,7
	in	al,dx
	test	al,BSY		;Is our controller still busy?
	jnz s	ChkTO3		;Yes, loop back and test again.
	or	cl,cl		;Are we also awaiting I-O data?
	jz s	ChkTOX		;No, just exit.
	test	al,cl		;Is data-request (DRQ) also set?
	jz s	ChkTO3		;No, loop back and test again.
ChkTOX:	ret			;Exit -- carry indicates timeout.
;
; Subroutine to convert "RedBook" MSF values to an LBA sector number.
;
CvtLBA:	mov	cx,ax		;Save "seconds" & "frames" in CX-reg.
	shr	eax,16		;Get "minute" value.
	cmp	ax,99		;Is "minute" value too large?
	ja s	CvtLBE		;Yes, return -1 error value.
	cmp	ch,60		;Is "second" value too large?
	ja s	CvtLBE		;Yes, return -1 error value.
	cmp	cl,75		;Is "frame" value too large?
	ja s	CvtLBE		;Yes, return -1 error value.
	xor	edx,edx		;Zero EDX-reg. for 32-bit math below.
	mov	dl,60		;Convert "minute" value to "seconds".
	mul	dl		;(Multiply by 60, obviously!).
	mov	dl,ch		;Add in "second" value.
	add	ax,dx
	mov	dl,75		;Convert "second" value to "frames".
	mul	edx		;(Multiply by 75 "frames"/second).
	mov	dl,150		;Subtract offset - "frame".
	sub	dl,cl		;("Adds" frame, "subtracts" offset).
	sub	eax,edx
	ret			;Exit.
CvtLBE:	or	eax,-1		;Too large!  Set -1 error value.
	ret			;Exit.
;
; Subroutine to "swap" the 4 bytes of a a 32-bit value.
;
SwpLBA:	mov	eax,[di+8]	;Get audio-end or buffer LBA value.
Swp32:	xchg	al,ah		;"Swap" original low-order bytes.
	rol	eax,16		;"Exchange" low- and high-order.
	xchg	al,ah		;"Swap" ending low-order bytes.
	ret			;Exit.
;
; DOS "Strategy" Routine.
;
Strat:	mov	cs:RqPkt.lw,bx	;Save DOS request-packet address.
	mov	cs:RqPkt.hw,es
	retf			;Exit & await DOS "Device Interrupt".
;
; DOS "Device Interrupt" Routine (must be AFTER all modified code!).
;
DevInt:	pushf			;Entry -- save CPU flags.
	push	ds		;Save and set our DS-register.
	push	cs
	pop	ds
	jmp	EntryP		;Dispatch to actual request handler.
;
; Subroutine to do physical XMS moves.   At entry, the ECX-reg. has
;   the byte length, and the ESI/EDI-regs. have the move addresses.
;   At exit, the carry flag is reset for no error or is SET for any
;   XMS move error.   If so, the AL-reg. has a 0FFh XMS error code.
;   The DS- and ES-regs. are saved and restored, and the BX-reg. is
;   zeroed at exit.   The other registers are NOT saved, for speed.
;   Note that the "real mode" move logic has no XMS errors to post!
;   Only "protected mode" moves (BIOS or EMM386) may return errors.
;   All logic after this point is DISMISSED by driver init, if UDMA
;   is "linked" to this driver and its "MvData" logic will be used.
;
MvData:	push	ds		;Save caller's segment registers.
	push	es
	push	cs		;Set this driver's data segment.
	pop	ds
	smsw	ax		;Get CPU "machine status word".
	shr	ax,1		;Are we running in "protected" mode?
	jc s	MvBIOS		;Yes, use BIOS move routine below.
	shr	ecx,2		;Convert byte count to dword count.
	mov	bp,cx		;Set move byte count in BP-reg.
	mov	dx,512		;Set 512-dword section count.
MvRNxt:	mov	cx,dx		;Get move section count.
	cmp	cx,bp		;At least 512 dwords left?
	jbe s	MvRGo		;Yes, use full section count.
	mov	cx,bp		;Use remaining dword count.
MvRGo:	cli			;Disable CPU interrupts.
	db	02Eh,00Fh	;"lgdt cs:GDTP", coded the hard-way
	db	001h,016h	;   to avoid any annoying V5.1 MASM
	dw	(GDTP-@)	;   warning messages about it!
	mov	eax,cr0		;Set "protected-mode" control bit.
	or	al,001h
	mov	cr0,eax
	mov	bx,offset (GDT_DS-OurGDT)  ;Set segment "selectors".
	mov	ds,bx
	mov	es,bx
	cld			;Ensure FORWARD "string" commands.
	db	0F3h,067h	;Move all 32-bit words using "rep"
	movsd			;  and "address-override" prefixes.
	db	067h,090h	;("Override" & "nop" for 386 BUG!).
	dec	ax		;Clear "protected-mode" control bit.
	mov	cr0,eax
	sti			;Allow interrupts after next command.
	sub	bp,dx		;Any more data sections to move?
	ja s	MvRNxt		;Yes, go move next data section.
MvDone:	xor	ax,ax		;Success!  Reset carry and error code.
MvExit:	sti			;RESTORE all critical driver settings!
	cld			;(Never-NEVER "trust" external code!).
	pop	es		;Reload caller's segment registers.
	pop	ds
	mov	bx,0		;Rezero BX-reg. but save carry bit.
	retf			;Exit.
;
; GDT Table Pointer, used by our "lgdt" command (here for alignment).
;
GDTP	dw	GDTLen		;GDT length (always 24 bytes).
GDTPAdr	dd	(OurGDT-@)	;GDT 32-bit address (Set By Init).
;
; BIOS Move Routine.
;
MvBIOS:	mov	eax,16384	;BIOS move:  Get 16K section count.
@BiosSC	equ	[$-4].dwd	;(BIOS section count, set by Init).
MvBNxt:	push	ecx		;Save remaining move byte count.
	push	esi		;Save move source address.
	cmp	ecx,eax		;At least 1 full move section left?
	jb s	MvBGo		;No, use remaining byte count.
	mov	ecx,eax		;Use full move section count.
MvBGo:	push	edi		;Save move destination address.
	shr	cx,1		;Convert byte count to word count.
	cld			;Ensure FORWARD "string" commands!
	push	cs		;Ensure ES-reg. points to our driver.
	pop	es
	push	edi		;Set up destination descriptor.
	mov	di,offset (DstDsc+2-@)
	pop	ax
	cli			;(Must disable CPU interrupts here
	stosw			;  to obey XMS "re-entrancy" rule!).
	pop	ax
	stosb
	mov	[di+2],ah
	push	esi		;Set up source descriptor.
	mov	di,offset (SrcDsc+2-@)
	pop	ax
	stosw
	pop	ax
	stosb
	mov	[di+2],ah
	mov	si,offset (MoveDT-@)  ;Point to move descriptor table.
	clc			      ;Reset carry -- BIOS might NOT!
	mov	ah,087h		;Have BIOS move next data section.
	int	015h
	pop	edi		;Reload destination/source addresses.
	pop	esi
	pop	ecx		;Reload remaining move byte count.
	jc s	MvExit		;If any BIOS error, exit immediately.
	push	cs		;Ensure DS-reg. points to our driver.
	pop	ds
	mov	eax,@BiosSC	;Reload move section count.
	sti			;Re-enable CPU interrupts.
	add	esi,eax		;Update source and dest. addresses.
	add	edi,eax
	sub	ecx,eax		;Any more data sections to move?
	ja s	MvBNxt		;Yes, go move next section of data.
	jmp s	MvDone		;Success!  Reset carry flag and exit.
;
; Global Descriptor Table, used during our own "real-mode" moves.
;
OurGDT	dd	0,0		;"Null" segment to start.
	dw	0FFFFh		;Conforming code segment.
GDT_CSL	dw	0
GDT_CSM	db	0
	dw	0009Fh
GDT_CSH	db	0
GDT_DS	dd	00000FFFFh,000CF9300h  ;4-GB data segment in pages.
GDTLen	equ	($-OurGDT)
;
; Int 15h Descriptor Table, used for BIOS "protected-mode" moves.
;
MoveDT	dd	0,0		       ;"Null" descriptor.
	dd	0,0		       ;GDT descriptor.
SrcDsc	dd	00000FFFFh,000009300h  ;Source segment descriptor.
DstDsc	dd	00000FFFFh,000009300h  ;Dest.  segment descriptor.
	dd	0,0		       ;(Used by BIOS).
	dd	0,0		       ;(Used by BIOS).
ResEnd	equ	$		;End of resident logic.
;
; Initialization ATA "Mode" Table.
;
Modes	dw	01602h		;Mode 0, ATA-16.
	dw	02502h		;Mode 1, ATA-25.
	dw	03302h		;Mode 2, ATA-33.
	dw	04402h		;Mode 3, ATA-44.
	dw	06602h		;Mode 4, ATA-66.
	dw	01003h		;Mode 5, ATA-100.
	dw	01333h		;Mode 6, ATA-133.
	dw	01663h		;Mode 7, ATA-166.
;
; Initialization IDE Parameter Table.
;
ScanP	db	(CDATA-0100h),0A0h  ;Primary-master   parameters.
	db	(CDATA-0100h),0B0h  ;Primary-slave    parameters.
ScanS	db	(CDATA-0180h),0A0h  ;Secondary-master parameters.
	db	(CDATA-0180h),0B0h  ;Secondary-slave  parameters.
ScanE	equ	$		    ;End of IDE parameters table.
;
; Initialization Messages.
;
TTLMsg	db	CR,LF,'UDVD, 8-13-2007.   Driver name is '
TTLName	db	'        .',CR,LF,'$'
DNMsg	db	' is '
DName	equ	$
VEMsg	db	'VDS init error$'
IBMsg	db	'No V2.0C+ PCI BIOS$'
OmitMsg	db	', UltraDMA omitted.',CR,LF,'$'
NXMsg	db	'No XMS manager$'
BCMsg	db	'Bad '
PCMsg	db	'UltraDMA controller'
PCMsg1	db	'$at I-O address '
PCMsg2	db	'xxxxh, Chip I.D. '
PCMsg3	db	'xxxxxxxxh.',CR,LF,'$'
UNMsg	db	'Unit '
UMsgNo	db	'0:  $'
PriMs	db	'Primary-$'
SecMs	db	'Secondary-$'
MstMs	db	'master$'
SlvMs	db	'slave$'
NDMsg	db	'No drive to use$'
PrMsg	db	'No 80386+ CPU'
Suffix	db	'; UDVD not loaded!',CR,LF,'$'
;
; Driver Initialization Routine.
;
I_Init:	push	es		;Save needed 16-bit CPU registers.
	push	ax
	push	bx
	push	dx
	push	cs		;Set our DS-register.
	pop	ds
	les	bx,RqPkt	;Point to DOS "Init" packet.
	cmp	es:[bx].RPOp,0	;Is this really an "Init" packet?
	jne s	I_Quit			    ;No?  Reload regs. & exit!
	mov	es:[bx].RPStat,RPDON+RPERR  ;Set init packet defaults.
	mov	es:[bx].RPSeg,cs
	and	es:[bx].RPLen,0
	push	sp		;See if CPU is an 80286 or newer.
	pop	ax		;(80286+ push SP, then decrement it).
	cmp	ax,sp		;Did SP-reg. get saved "decremented"?
	jne s	I_Junk		;Yes, CPU is an 8086/80186, TOO OLD!
	pushf			;80386 test -- save CPU flags.
	push	07000h		;Try to set NT|IOPL status flags.
	popf
	pushf			;Get resulting CPU status flags.
	pop	ax
	popf			;Reload starting CPU flags.
	test	ah,070h		;Did any NT|IOPL bits get set?
	jnz s	I_386		;Yes, CPU is at least an 80386.
I_Junk:	mov	dx,(PRMsg-@)	;Display "No 80386+ CPU" message.
	call	I_Msg
I_Quit:	jmp	I_Exit		;Go reload registers and exit quick!
I_386:	pushad			;80386+ CPU:  Save 32-bit registers.
	les	si,es:[bx].RPCL	;Point to command line that loaded us.
	xor	bx,bx		;Zero our BX-reg. and avoid re-entry.
	mov	[bx+EntryP-@].lw,(ReqPr-@)
I_NxtC:	mov	al,es:[si]	;Get next command-line byte.
	inc	si		;Bump pointer past this byte.
	cmp	al,0		;Is byte the command-line terminator?
	je s	I_TrmJ		;Yes, go validate driver name.
	cmp	al,LF		;Is byte an ASCII line-feed?
	je s	I_TrmJ		;Yes, go validate driver name.
	cmp	al,CR		;Is byte an ASCII carriage-return?
I_TrmJ:	je	I_Term		;Yes, go validate driver name.
	cmp	al,'-'		;Is byte a dash?
	je s	I_NxtS		;Yes, see what next "switch" byte is.
	cmp	al,'/'		;Is byte a slash?
	jne s	I_NxtC		;No, check next command-line byte.
I_NxtS:	mov	ax,es:[si]	;Get next 2 command-line bytes.
	and	al,0DFh		;Mask out 1st lower-case bit (020h).
	cmp	al,'A'		;Is this byte an "A" or "a"?
	jne s	I_ChkC		;No, see if byte is "C" or "C".
	inc	si		;Bump pointer past "alternate" switch.
	mov	al,(CDATA-108h)	;Set "alternate" primary addresses.
	mov	ScanP.lb,al
	mov	[ScanP+2].lb,al
	mov	al,(CDATA-188h)	;Set "alternate" secondary addresses.
	mov	ScanS.lb,al
	mov	[ScanS+2].lb,al
	jmp s	I_NxtC		;Go check next command-line byte.
I_ChkC:	cmp	al,'C'		;Is this byte a "C" or "c"?
	jne s	I_ChkU		;No, see if byte is "U" or "u".
	inc	si		;Bump pointer past "C" or "c".
	mov	@RMF,010h	;Reset "Raw mode" user status flag.
	mov	@RM1,0		;Disable "Raw mode" input requests.
	mov	@RM2.lb,0EBh
	jmp s	I_NxtC		;Go check next command-line byte.
I_ChkU:	cmp	al,'U'		;Is this byte a "U" or "u"?
	jne s	I_ChkD		;No, see if 2 bytes are "D:" or "d:".
	inc	si		;Bump pointer past "U" or "u".
	mov	al,ah		;Get 2nd command-line byte.
	and	al,0DFh		;Mask out 1st lower-case bit (020h).
	cmp	al,'X'		;Is 2nd byte an "X" or "x"?
	jne s	I_NxtC		;No, continue scan for a terminator.
	inc	si		;Bump pointer past "X" or "x".
	mov	[bx+NoDMA-@],al	;Set "No UltraDMA" flag.
I_NxtJ:	jmp s	I_NxtC		;Go check next command-line byte.
I_ChkD:	cmp	ax,":D"		;Are next 2 bytes "D:" or "d:"?
	jne s	I_NxtC		;No, continue scan for a terminator.
	inc	si		;Bump pointer past "device" switch.
	inc	si
	mov	di,(DvrNam-@)	;Blank out device name.
	mov	eax,"    "
	mov	[di],eax
	mov	[di+4],eax
I_Nam1:	mov	al,es:[si]	;Get next device-name byte.
	cmp	al,TAB		;Is byte a "tab"?
	je s	I_NxtJ		;Yes, handle above, "name" has ended!
	cmp	al,' '		;Is byte a space?
	je s	I_NxtJ		;Yes, handle above, "name" has ended!
	cmp	al,'/'		;Is byte a slash?
	je s	I_NxtJ		;Yes, handle above, "name" has ended!
	cmp	al,0		;Is byte the command-line terminator?
	je s	I_Term		;Yes, go test for UltraDMA controller.
	cmp	al,LF		;Is byte an ASCII line-feed?
	je s	I_Term		;Yes, go test for UltraDMA controller.
	cmp	al,CR		;Is byte an ASCII carriage-return?
	je s	I_Term		;Yes, go test for UltraDMA controller.
	cmp	al,'a'		;Ensure letters are upper-case.
	jb s	I_Nam2
	cmp	al,'z'
	ja s	I_Nam2
	and	al,0DFh
I_Nam2:	cmp	al,'!'		;Is this byte an exclamation point?
	je s	I_Nam3		;Yes, store it in device name.
	cmp	al,'#'		;Is byte below a pound-sign?
	jb s	I_Nam4		;Yes, Invalid!  Blank first byte.
	cmp	al,')'		;Is byte a right-parenthesis or less?
	jbe s	I_Nam3		;Yes, store it in device name.
	cmp	al,'-'		;Is byte a dash?
	je s	I_Nam3		;Yes, store it in device name.
	cmp	al,'0'		;Is byte below a zero?
	jb s	I_Nam4		;Yes, invalid!  Blank first byte.
	cmp	al,'9'		;Is byte a nine or less?
	jbe s	I_Nam3		;Yes, store it in device name.
	cmp	al,'@'		;Is byte below an "at sign"?
	jb s	I_Nam4		;Yes, invalid!  Blank first byte.
	cmp	al,'Z'		;Is byte a "Z" or less?
	jbe s	I_Nam3		;Yes, store it in device name.
	cmp	al,'^'		;Is byte below a carat?
	jb s	I_Nam4		;Yes, invalid!  Blank first byte.
	cmp	al,'~'		;Is byte above a tilde?
	ja s	I_Nam4		;Yes, invalid!  Blank first byte.
	cmp	al,'|'		;Is byte an "or" symbol?
	je s	I_Nam4		;Yes, invalid!  Blank first byte.
I_Nam3:	mov	[di],al		;Store next byte in device name.
	inc	si		;Bump command-line pointer.
	inc	di		;Bump device-name pointer.
	cmp	di,(DvrNam+8-@)	;Have we stored 8 device-name bytes?
	jb s	I_Nam1		;No, go get next byte.
	jmp s	I_Nam5		;Go get next byte & check terminator.
I_Nam4:	mov	DvrNam.lb,' '	;Invalid!  Blank first byte.
I_Nam5:	jmp	I_NxtC		;Go get next command byte.
I_Term:	xor	bx,bx			;Zero BX-reg. for below.
	cmp	[bx+DvrNam-@].lb,' '	;Is driver "name" valid?
	jne s	I_SetN			;Yes, eliminate any spaces.
	mov	[bx+DvrNam-@],"DVDU"	;Set our default "name".
	mov	[bx+DvrNam+4-@],"   1"
I_SetN:	mov	si,(TTLName+8-@)	;Set driver "name" in title.
	mov	eax,[bx+DvrNam-@]
	mov	[si-8],eax
	mov	eax,[bx+DvrNam+4-@]
	mov	[si-4],eax
I_ScnN:	dec	si			;Decrement "name" pointer.
	cmp	[si].lb,' '		;Is this "name" byte a space?
	je s	I_ScnN			;Yes, scan for a non-space.
	mov	[si+1].dwd,0240A0D2Eh	;Set terminators after name.
	mov	dx,(TTLMsg-@)		;Display driver "title" msg.
	call	I_Msg
;
; ***** NOTE *****
;
;   USB or Firewire disk drivers can do the following to "link" with
;   the UDMA driver and initialize themselves to allow UDMA caching.
;
	mov	ax,cs		     ;Set default segment values.
	mov	MoveRtn.hw,ax
	mov	Flush.hw,ax
	xor	ax,ax		     ;Point ES-reg. to low memory.
	mov	es,ax
	mov	es,es:[INT13S]	     ;Get Int 13h segment address.
	mov	di,(DvrNam-@)	     ;Get driver-name offset.
	cmp	es:[di].dwd,"AMDU"   ;Is the UDMA driver present?
	jne s	I_VDCh		     ;No, forget about UDMA "link".
	cmp	es:[di+4].dwd,"$X5"  ;Is it our nice "Type 5" UDMA?
	jne s	I_VDCh		     ;No, forget about UDMA "link".
	movzx	di,es:[di-4].lb	;Point to UDMA "External Variables".
	shl	di,2
	mov	ax,es:[di]	;Save high-order XMS buffer address.
	mov	@XBufAd.hw,ax	;(It is 64K aligned, low-order = 0).
	mov	ax,es:[di+2]	;Save buffer "offset" in XMS memory.
	mov	XBOfs,ax	;(Only necessary for UDVD UltraDMA).
	mov	eax,es:[di+6]	;Save UDMA's "XMS Move" subroutine
	mov	MoveRtn,eax	;  address, if you use that routine!
	mov	ax,(MvData-@)	;Dismiss our own "XMS Move" logic by
	mov	VDSLn.lw,ax	;  reducing the size of this driver.
	cmp	@RMF,014h	;Will this driver allow "raw" input?
	je s	I_VDCh		;Yes, we CANNOT permit UDMA caching!
				;(This is only for a CD/DVD driver).
	mov	dx,es:[di+4]	;Get UDMA "External Entry" addr.
	or	dx,dx		;"Stand alone" UDMA disk driver?
	jz s	I_VDCh		;Yes, forget about UDMA caching.
	push	es		;Set UDMA "External Entry" address.
	push	dx
	pop	@ExEntr
	sub	dx,8		;Set UDMA "flush" routine address, 8
	push	es		;  bytes behind its "External Entry"
	push	dx		;  code.   A cache "flush" is needed
	pop	Flush		;  for "passed" I-O, a new removable
				;  disk, and maybe for a hard-error!
;
; End of initialization logic for UDMA caching.
;
	mov	RqRL6.lw,0DB87h	;Enable CD/DVD caching through UDMA.
I_VDCh:	xor	bx,bx		;Zero BX-reg. for below.
	xor	eax,eax		;Zero EAX-reg. for 20-bit addressing.
	mov	es,ax		;Point ES-reg. to low memory.
	mov	ax,cs		;Set default 20-bit driver address.
	mov	[bx+VDSSg-@],ax
	shl	eax,4
	mov	[bx+VDSAd-@],eax
	cli			  ;Avoid interrupts during VDS tests.
	mov	ax,08103h	  ;Set up VDS "lock" call.
	mov	dx,0000Ch
	test	es:[VDSFLAG].lb,020h  ;Are "VDS services" active?
	jz s	I_REnI		      ;No, re-enable interrupts.
	call	I_VDS		  ;"Lock" this driver into memory.
	mov	dx,(VEMsg-@)	  ;Point to "VDS init error" msg.
	jc	I_Err		  ;If error, display msg. & exit!
	inc	[bx+VDSOf-@].lb	  ;Set init VDS "lock" flag.
I_REnI:	sti			  ;Re-enable CPU interrupts.
	mov	eax,[bx+VDSAd-@]  ;Relocate command-block source addr.
	add	@CmdLst,eax
	add	GDTPAdr,eax	  ;Relocate "real mode" GDT base addr.
	mov	GDT_CSL,ax	  ;Set lower CS-descriptor address.
	shr	eax,16		  ;Set upper CS-descriptor address.
	mov	GDT_CSM,al
	mov	GDT_CSH,ah
	cmp	NoDMA,0		;Does our boy want no UltraDMA?
	jne s	I_NoUC		;Yes, see if we have an XMS manager.
	xor	edi,edi		;UltraDMA controller check:  Request
	mov	al,001h		;  PCI BIOS I.D. (should be "PCI ").
	call	I_In1A
	cmp	edx," ICP"	;Do we have a V2.0C or newer PCI BIOS?
	mov	dx,(IBMsg-@)	;(Get "No V2.0C+ PCI" message ptr.).
	jne s	I_CtlM		;No, display message and check XMS.
	mov	si,(CCTbl-@)	;Point to interface byte table.
I_GetC:	cmp	si,(CCEnd-@)	;More interface bytes to check?
	jae s	I_NoUC		;No, set "No UltraDMA" flag & proceed.
	mov	ecx,000010100h	;Find class 1 storage, subclass 1 IDE.
	lodsb			;Use next class-code "interface" byte.
	mov	cl,al
	push	si		;Save class-code table pointer.
	mov	al,003h		;Inquire about an UltraDMA controller.
	xor	si,si		;(Returns bus/device/function in BX).
	call	I_In1A
	pop	si		;Reload class-code table pointer.
	jc s	I_GetC		;Not found -- test for more bytes.
	push	bx		;Save PCI bus/device/function.
	mov	di,4		;Get low-order PCI command byte.
	call	I_PCID
	pop	bx		;Reload PCI bus/device/function.
	not	cl		;Mask Bus-Master and I-O Space bits.
	and	cl,005h		;Is this how our controller is set up?
	jz s	I_GotC		;Yes, save UltraDMA controller data.
	mov	dx,(BCMsg-@)	;Point to "Bad controller" message.
I_CtlM:	call	I_Msg		;Display controller-related error.
	mov	dx,(OmitMsg-@)	;Display "UltraDMA omitted" message.
	call	I_Msg
I_NoUC:	mov	NoDMA,1		;Ensure "No UltraDMA" flag is set.
	jmp s	I_XMgr		;Go see if we have an XMS manager.
I_GotC:	push	bx		;Save PCI bus/device/function.
	xor	di,di		;Get Vendor and Device I.D.
	call	I_PCID
	pop	bx		;Reload PCI bus/device/function.
	push	ecx		;Save Vendor and Device I.D.
	mov	di,32		;Get PCI base address (register 4).
	call	I_PCID
	xchg	ax,cx		;Save our DMA controller address.
	and	al,0FCh
	mov	DMAAd,ax
	mov	si,(PCMsg2-@)	;Set controller address in message.
	call	I_Hex
	mov	si,(PCMsg3-@)	;Set Vendor & Device I.D. in message.
	pop	ax
	call	I_Hex
	pop	ax
	call	I_Hex
	mov	dx,(PCMsg-@)	;Display all UltraDMA controller data.
	mov	PCMsg1.lb,' '
	call	I_Msg
I_XMgr:	mov	ax,04300h	;Inquire about an XMS manager.
	call	I_In2F
	mov	dx,(NXMsg-@)	;Point to "No XMS manager" message.
	cmp	al,080h		;Is an XMS manager installed?
	jne	I_Err		;No?  Display error message and exit!
	mov	ax,04310h	;Get and save XMS "entry" address.
	call	I_In2F
	mov	si,bx
	push	es
	push	bx
	pop	@XEntry
	mov	ax,@XBufAd.hw	;Get XMS buffer address & offset.
	mov	cx,XBOfs
	or	ax,ax		;Did we get an XMS buffer from UDMA?
	jnz s	I_GtX3		;Yes, analyze it & set PRD address. 
	mov	ah,009h		;Ask XMS manager for 128K of memory.
	mov	dx,128
	call	I_XMS
	jnz s	I_GtXE		;If error, display message and exit!
	mov	XBHdl,dx	;Save XMS buffer handle number.
	mov	ah,00Ch		;"Lock" our XMS memory.
	call	I_XMS
	jz s	I_GtX1		;If no error, set XMS buffer address.
I_GtXE:	mov	dx,(VEMsg-@)	;Set up "XMS init error" message.
	mov	VEMsg.lw,"MX"
	jmp	I_Err		;Go display error message and exit.
I_GtX1:	shl	edx,16		;Get unaligned XMS buffer address.
	or	dx,bx		;Is buffer ALREADY on a 64K boundary?
	mov	eax,edx		;(Copy 32-bit address to EAX-reg.).
	jz s	I_GtX2		;Yes, use XMS buffer address as-is.
	add	eax,65536	;Find 1st 64K boundary after start.
	xor	ax,ax
I_GtX2:	push	eax		;Save high-order buffer address.
	pop	cx
	pop	cx
	sub	ax,dx		;Get buffer "offset" in XMS memory.
	xchg	ax,cx		;Swap buffer offset & address.
	mov	@XBufAd.hw,ax	;Save our XMS buffer address.
I_GtX3:	xor	bx,bx		;Rezero BX-reg. for our logic.
	shl	eax,16		;Get 32-bit XMS buffer address.
	mov	edx,-16		;Initialize buffer adjustment. 
	cmp	cx,32		;At least 32 bytes of "offset"?
	jae s	I_GtX4		;Yes, set command-list offset. 
	mov	edx,000010010h	   ;Adjust to beyond XMS buffer.
I_GtX4:	add	eax,edx		   ;Set our 32-bit PRD address.
	mov	[bx+PRDAd-@],eax
I_Scan:	mov	si,[bx+AudAP-@]	   ;Load & update parameters ptr.
	add	[bx+AudAP-@].lw,2
	cmp	si,(ScanE-@)	   ;Any more IDE units to check?
	je s	I_AnyD		   ;No, check for any drives to use.
	mov	ax,[si]		   ;Set this unit's IDE address.
	mov	[bx+IDEAd-@],al
	mov	[bx+CCTbl-@],ah	      ;Save unit device-select byte.
	or	[bx+DMAAd-@].lb,00Fh  ;Invalidate UltraDMA address.
	cmp	[bx+NoDMA-@],bl	      ;Is all UltraDMA disabled?
	jne s	I_Scn1		      ;Yes, validate unit as ATAPI.
	and	[bx+DMAAd-@].lb,0F0h  ;Reset secondary-channel bit.
	test	al,080h		      ;Primary channel CD/DVD unit?
	jnz s	I_Scn1		      ;Yes, validate unit as ATAPI.
	or	[bx+DMAAd-@].lb,008h  ;Use secondary DMA channel.
I_Scn1:	call	I_ValD		      ;Validate unit as ATAPI CD/DVD.
	jc s	I_Scan		      ;If invalid, merely ignore it.
	mov	si,[bx+UTblP-@]	      ;Update unit-table parameters.
	mov	al,[bx+DMAAd-@]
	mov	ah,[bx+IDEAd-@]
	mov	[si],ax
	mov	al,[bx+CCTbl-@]
	mov	[si+2],al
	add	si,16		;Update unit-table pointer.
	mov	[bx+UTblP-@],si
	inc	[bx+Units-@].lb	;Bump number of active units.
	inc	UMsgNo		;Bump display unit number.
	cmp	si,(UTblEnd-@)	;Can we install another drive?
	jb	I_Scan		;Yes, loop back & check for more.
I_AnyD:	cmp	[bx+Units-@],bl	;Do we have any CD/DVD drives to use?
	ja s	I_Done		;Yes, success -- post "init" packet.
	mov	dx,(NDMsg-@)	;Point to "No drive to use" message.
I_Err:	push	dx		;Init ERROR!  Save message pointer.
	mov	dx,XBHdl	;Get XMS "handle" number.
	or	dx,dx		;Did we reserve any XMS memory?
	jz s	I_ErrV		;No, see about a VDS "unlock".
	mov	ah,00Dh		;Unlock and "free" our XMS memory.
	push	dx
	call	I_XMS
	mov	ah,00Ah
	pop	dx
	call	I_XMS
I_ErrV:	shr	[bx+VDSOf-@].lb,1  ;Was driver "locked" by VDS?
	jnc s	I_ErrD		   ;No, go display error message.
	mov	ax,08104h	;"Unlock" this driver from memory.
	xor	dx,dx
	call	I_VDS
I_ErrD:	pop	dx		;Reload error-message pointer.
	call	I_Msg		;Display error message and suffix.
	mov	dx,(Suffix-@)
	call	I_Msg
	jmp s	I_Fail		;Go reload all registers and exit.
I_Done:	xor	ax,ax		;Success!  Load & reset driver length.
	xchg	ax,VDSLn.lw
	lds	bx,RqPkt	;Set result values in "init" packet.
	mov	[bx].RPLen,ax
	mov	[bx].RPStat,RPDON
I_Fail:	popad			;Reload all 32-bit CPU registers.
I_Exit:	pop	dx		;Reload 16-bit CPU regs. we used.
	pop	bx
	pop	ax
	pop	es		;Reload CPU segment registers.
	pop	ds
	popf			;Reload CPU flags and exit.
	retf
;
; Subroutine to "validate" an IDE unit as an ATAPI CD/DVD drive.
;
I_ValD:	mov	dx,[bx+DMAAd-@]	;Get unit UltraDMA command address.
	test	dl,001h		;Will this unit be using UltraDMA?
	jnz s	I_Va0		;No, just select "master" or "slave".
	in	al,dx		;Ensure any previous DMA is stopped!
	and	al,0FEh
	out	dx,al
I_Va0:	mov	dx,[bx+IDEAd-@]	;Point to IDE device-select register.
	add	dx,6
	mov	al,[bx+CCTbl-@]	;Select IDE "master" or "slave" unit.
	out	dx,al
	call	ChkTO		;Await controller-ready.
	jc s	I_Va7		;If timeout, go exit below.
	mov	al,0A1h		;Issue "Identify Packet Device" cmd.
	out	dx,al
	call	ChkTO		;Await controller-ready.
	jc s	I_Va7		;If timeout, go exit below.
	test	al,DRQ		;Did we also get a data-request?
	jz s	I_Va6		;No, go set carry & exit below.
	sub	dx,7		;Point back to IDE data register.
	in	ax,dx		;Read I.D. word 0, main device flags.
	and	ax,0DF03h	;Mask off flags for an ATAPI CD/DVD.
	xchg	ax,si		;Save main device flags in SI-reg.
	mov	cx,26		;Skip I.D. words 1-26 (unimportant).
I_Va1:	in	ax,dx
	loop	I_Va1
	mov	di,(DName-@)	;Point to drive "name" input buffer.
	push	cs
	pop	es
	mov	cl,20		;Read & swap words 27-46 into buffer.
I_Va2:	in	ax,dx		;(Manufacturer "name" of this drive).
	xchg	ah,al
	stosw
	loop	I_Va2
	mov	cl,7		;Skip I.D. words 47-52 (unimportant)
I_Va3:	in	ax,dx		;  and read I.D. word 53 into AX-reg.
	loop	I_Va3
	mov	[bx+UFlag-@],al	;Save UltraDMA "valid" flags.
	mov	cl,35		;Skip I.D. words 54-87 (unimportant)
I_Va4:	in	ax,dx		;  and read I.D. word 88 into AX-reg.
	loop	I_Va4
	mov	[bx+UMode-@],ah	;Save posted UltraDMA "mode" value.
	mov	cl,167		;Skip all remaining I.D. data.
I_Va5:	in	ax,dx
	loop	I_Va5
	cmp	si,08500h	;Do device flags say "ATAPI CD/DVD"?
	je s	I_Va8		;Yes, see about UltraDMA use.
I_Va6:	stc			;Set carry flag on (error!).
I_Va7:	ret			      ;Exit.
I_Va8:	test	[bx+UFlag-@].lb,004h  ;Valid UltraDMA "mode" bits?
	jz s	I_Va9		      ;No, disable drive UltraDMA.
	cmp	[bx+UMode-@],bx	      ;Can drive do mode 0 minimum?
	jne s	I_Va10		      ;Yes, display "Unit n:" msg.
I_Va9:	or	[bx+DMAAd-@].lb,1     ;Disable this drive's UltraDMA.
I_Va10:	mov	dx,(UnMsg-@)	      ;Display "Unit n:" message.
	call	I_Msg
	mov	dx,(PriMs-@)	      ;Point to "Primary" message.
	test	[bx+IDEAd-@].lb,080h  ;Primary-channel drive?
	jnz s	I_Va11		      ;Yes, display "Primary" msg.
	mov	dx,(SecMs-@)	      ;Point to "Secondary" message.
I_Va11:	call	I_Msg		      ;Display CD/DVD's IDE channel.
	mov	dx,(MstMs-@)	      ;Point to "Master" message.
	cmp	[bx+CCTbl-@].lb,SSEL  ;Is this drive a "slave"?
	jnz s	I_Va12		      ;No, display "Master".
	mov	dx,(SlvMs-@)	;Point to "Slave" message.
I_Va12:	call	I_Msg		;Display "Master" or "Slave".
	mov	si,(DName+40-@)	;Point to end of CD/DVD vendor name.
I_Va13:	cmp	si,(DName-@)	;Are we at the vendor-name start?
	je s	I_Va14		;Yes, vendor name is all spaces!
	dec	si		;Decrement vendor-name pointer.
	cmp	[si].lb,' '	;Is this name byte a space?
	je s	I_Va13		;No, continue scan for non-space.
	inc	si		;Skip non-space character.
	mov	[si].lw," ,"	;End disk name with comma/space.
	inc	si		;Skip comma and space.
	inc	si
I_Va14:	test	[bx+DMAAD-@].lb,1  ;Will this drive use "PIO mode"?
	jz s	I_Va15		   ;No, set "ATA" mode after name.
	mov	[si].dwd," OIP"	   ;Set "PIO mode" after drive name.
	mov	[si+4].dwd,"edom"
	add	si,8
	jmp s	I_Va17		;Go set message terminators.
I_Va15:	mov	[si].dwd,"-ATA"	;Set "ATA-" after drive name.
	add	si,4
	mov	cx,[bx+UMode-@]	;Initialize UltraDMA "mode" scan.
	mov	di,(Modes-2-@)
I_Va16:	inc	di		;Advance to next UltraDMA "mode".
	inc	di
	shr	cx,1		;Will drive do next "mode"?
	jnz s	I_Va16		;Yes, keep scanning for maximum.
	mov	ax,[di]		;Set UltraDMA "mode" in message.
	mov	cl,00Fh
	and	cl,al
	call	I_HexA
I_Va17:	mov	[si].dwd,0240A0D2Eh  ;Set message terminators.
	mov	dx,(DNMsg-@)	     ;Display vendor name & "mode".
	call	I_Msg
	clc			;Reset carry (no errors) and exit.
	ret
;
; Subroutines to do all initialization "external" calls.
;
I_In2F:	int	02Fh		;Issue desired Int 2Fh request.
	jmp s	I_IntX		;Go restore driver settings and exit.
I_XMS:	call	@XEntry		;Issue desired XMS request.
	dec	ax		;Zero AX-reg. if success, -1 if error.
	jmp s	I_IntX		;Go restore driver settings and exit.
I_PCID:	mov	al,00Ah		;Set "PCI doubleword" request code.
I_In1A:	mov	ah,0B1h		;PCI BIOS -- execute desired request.
	int	01Ah
	jmp s	I_IntX		;Go restore driver settings and exit.
I_VDS:	push	bx		;VDS -- save our BX-register.
	mov	di,(VDSLn-@)	;Point to VDS parameter block.
	push	cs
	pop	es
	int	04Bh		;Execute VDS "lock" or "unlock".
	pop	bx		;Reload our BX-register.
	jmp s	I_IntX		;Go restore driver settings and exit.
I_Msg:	mov	ah,009h		;Get DOS "display string" command.
I_In21:	Int	021h		;Execute desired DOS request.
I_IntX:	sti			;RESTORE all critical driver settings!
	cld			;(Never-NEVER "trust" external code!).
	push	cs
	pop	ds
	ret			;Exit.
;
; Subroutine to convert a 4-digit hex number to ASCII for messages.
;   At entry, the number is in the AX-reg., and the message pointer
;   is in the SI-reg.   At exit, the SI-reg. is updated and the CX-
;   reg. is zero.
;
I_Hex:	mov	cx,4		;Set 4-digit count.
I_HexA:	rol	ax,4		;Get next hex digit in low-order.
	push	ax		;Save remaining digits.
	and	al,00Fh		;Mask off next hex digit.
	cmp	al,009h		;Is digit 0-9?
	jbe s	I_HexB		;Yes, convert to ASCII.
	add	al,007h		;Add A-F offset.
I_HexB:	add	al,030h		;Convert digit to ASCII.
	mov	[si],al		;Store next digit in message.
	inc	si		;Bump message pointer.
	pop	ax		;Reload remaining digits.
	loop	I_HexA		;If more digits to go, loop back.
	ret			;Exit.
CODE	ends
	end
