
/************************************************************************/
/*                                                                      */
/* FILE: doslib.c                                                       */
/* Preliminary DOS SDK Functions                                        */
/*                                                                      */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* This file is part of G-COM.                                          */
/* G-COM is derived from DOS-C source (GPL).                            */
/*                                                                      */
/* (C) Copyright 1999-2000  Roberto Gordo Saez   (GCOM)                 */
/* (C) Copyright 1995-1998  Pasquale J. Villani  (DOSC)                 */
/*                                                                      */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2, or (at your option)  */
/* any later version.                                                   */
/*                                                                      */
/* This program is distributed in the hope that it will be useful, but  */
/* WITHOUT ANY WARRANTY; without even the implied warranty of           */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU     */
/* General Public License for more details.                             */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to the Free Software          */
/* Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.                  */
/*                                                                      */
/************************************************************************/


#include "globals.h"


/* -------------------------------------------------------------------- */
/* DosAllocMem                                                          */
/* -------------------------------------------------------------------- */

UWORD DosAllocMem(UWORD MemSize, DWORD FAR *error)
{
    SEGMENT MemSeg;
    UWORD error_code = 0;

    ASM
    {
        mov     bx,MemSize
        mov     ah,0x48
        int     0x21
        mov     MemSeg,ax
        jnc     exit
        mov     MemSeg,bx
        mov     error_code,ax
    }

    exit:
    *error = (error_code == 0) ? 0 : -error_code;
    return MemSeg;
}


/* -------------------------------------------------------------------- */
/* DosCd                                                                */
/* -------------------------------------------------------------------- */

DWORD DosCd(char FAR *Dir)
{
    SEGMENT Dir_seg = FP_SEG(Dir);
    OFFSET Dir_off = FP_OFF(Dir);
    UWORD status;

    ASM
    {
        push    ds
        mov     ds,Dir_seg
        mov     dx,Dir_off
        mov     ah,0x3B
        int     0x21
        pop     ds
        mov     status,ax
        jnc     exit
    }

    return -((DWORD)status);

    exit:
    return 0;
}


/* -------------------------------------------------------------------- */
/* DosClose                                                             */
/* -------------------------------------------------------------------- */

DWORD DosClose(UWORD Handle)
{
    UWORD status;

    ASM
    {
        mov     bx,Handle
        mov     ah,0x3E
        int     0x21
        mov     status,ax
        jnc     exit
    }

    return -((DWORD)status);

    exit:
    return 0;
}


/* -------------------------------------------------------------------- */
/* DosCreate                                                            */
/* -------------------------------------------------------------------- */

DWORD DosCreate(char FAR *FileName, UWORD Attributes)
{
    SEGMENT fname_seg = FP_SEG(FileName);
    OFFSET fname_off = FP_OFF(FileName);
    UWORD Handle;

    ASM
    {
        push    ds
        mov     ds,fname_seg
        mov     dx,fname_off
        mov     cx,Attributes
        mov     ah,0x3C
        int     0x21
        pop     ds
        mov     Handle,ax
        jnc     exit
    }

    return -((DWORD)Handle);

    exit:
    return (DWORD)Handle;
}


/* -------------------------------------------------------------------- */
/* DosDelete                                                            */
/* -------------------------------------------------------------------- */

DWORD DosDelete(char FAR *FileName)
{
    SEGMENT FileName_seg = FP_SEG(FileName);
    OFFSET FileName_off = FP_OFF(FileName);
    UWORD status;

    ASM
    {
        push    ds
        mov     ds,FileName_seg
        mov     dx,FileName_off
        mov     ah,0x41
        int     0x21
        pop     ds
        mov     status,ax
        jnc     exit
    }

    return -((DWORD)status);

    exit:
    return 0;
}


/* -------------------------------------------------------------------- */
/* DosDupHandle                                                         */
/* -------------------------------------------------------------------- */

DWORD DosDupHandle(DWORD OldHandle, DWORD FAR *pNewHandle)
{
    UWORD Handle = (UWORD)OldHandle;

    ASM
    {
        mov     bx,Handle
        mov     ah,0x45
        int     0x21
        mov     Handle,ax
        jnc     exit
    }

    return -((DWORD)Handle);

    exit:
    *pNewHandle = (DWORD)Handle;
    return 0;
}


/* -------------------------------------------------------------------- */
/* DosExec                                                              */
/* -------------------------------------------------------------------- */

DWORD DosExec(char FAR *ProgName, exec_block FAR *ProgArgs)
{
    SEGMENT ProgArgs_off = FP_OFF(ProgArgs);
    OFFSET ProgArgs_seg = FP_SEG(ProgArgs);
    SEGMENT ProgName_off = FP_OFF(ProgName);
    OFFSET ProgName_seg = FP_SEG(ProgName);
    UWORD status;

    ASM
    {
        push    dx
        push    ds
        push    es
        push    bx
        push    di
        push    si
        mov     ds,ProgName_seg
        mov     dx,ProgName_off
        mov     es,ProgArgs_seg
        mov     bx,ProgArgs_off
        mov     ax,0x4B00
        int     0x21
        pop     si
        pop     di
        pop     bx
        pop     es
        pop     ds
        pop     dx
        mov     status,ax
        jnc     exit
    }

    return -((DWORD)status);

    exit:
    return 0;
}


/* -------------------------------------------------------------------- */
/* DosExit                                                              */
/* -------------------------------------------------------------------- */

void DosExit(UBYTE ReturnValue)
{
    ASM
    {
        mov     al,ReturnValue
        mov     ah,0x4C
        int     0x21
    }
}


/* -------------------------------------------------------------------- */
/* DosForceDupHandle                                                    */
/* -------------------------------------------------------------------- */

DWORD DosForceDupHandle(DWORD OpenHandle, DWORD DuplicateHandle)
{
    UWORD status;
    UWORD handle1 = (UWORD)OpenHandle;
    UWORD handle2 = (UWORD)DuplicateHandle;

    ASM
    {
        mov     bx,handle1
        mov     cx,handle2
        mov     ah,0x46
        int     0x21
        mov     status,ax
        jnc     exit
    }

    return -((DWORD)status);

    exit:
    return 0;
}


/* -------------------------------------------------------------------- */
/* DosFreeMem                                                           */
/* -------------------------------------------------------------------- */

DWORD DosFreeMem(UWORD SegmentMem)
{
    UWORD status;

    ASM
    {
        push    es
        mov     es,SegmentMem
        mov     ah,0x49
        int     0x21
        pop     es
        mov     status,ax
        jnc     exit
    }

    return -((DWORD)status);

    exit:
    return 0;
}


/* -------------------------------------------------------------------- */
/* DosGetBreak                                                          */
/* -------------------------------------------------------------------- */

BOOL DosGetBreak(void)
{
    UBYTE status;

    ASM
    {
        mov     ax,0x3300
        int     0x21
        mov     status,dl
    }

    return (status == 0) ? FALSE : TRUE;
}


/* -------------------------------------------------------------------- */
/* DosGetChar                                                           */
/* -------------------------------------------------------------------- */

char DosGetChar(void)
{
    char character;

    ASM
    {
        mov     ah,0x08
        int     0x21
        mov     character,al
    }

    return character;
}


/* -------------------------------------------------------------------- */
/* DosGetDate                                                           */
/* -------------------------------------------------------------------- */

void DosGetDate(UBYTE *wdp, UBYTE *mp, UBYTE *mdp, UWORD *yp)
{
    UBYTE WeekDay = 0,
    Month = 0,
    MonthDay = 0;
    UWORD Year = 0;

    ASM
    {
        mov     ah,0x2A
        int     0x21
        mov     WeekDay,al
        mov     Year,cx
        mov     Month,dh
        mov     MonthDay,dl
    }

    *wdp = WeekDay;
    *mp = Month;
    *mdp = MonthDay;
    *yp = Year;
}


/* -------------------------------------------------------------------- */
/* DosGetDrive                                                          */
/* -------------------------------------------------------------------- */

UBYTE DosGetDrive(void)
{
    UBYTE DriveNumber;

    ASM
    {
        mov     ah,0x19
        int     0x21
        mov     DriveNumber,al
    }

    return DriveNumber;
}


/* -------------------------------------------------------------------- */
/* DosGetInfo                                                           */
/* -------------------------------------------------------------------- */

DWORD DosGetInfo(info FAR *p_info)
{
    SEGMENT info_seg = FP_SEG(p_info);
    OFFSET info_off = FP_OFF(p_info);
    UWORD status;

    ASM
    {
        push    ds
        mov     ds,info_seg
        mov     dx,info_off
        xor     bx,bx
        mov     ax,0x3800
        int     0x21
        pop     ds
        mov     status,ax
        jnc     exit
    }

    return -((DWORD)status);

    exit:
    return (DWORD)status;
}


/* -------------------------------------------------------------------- */
/* DosGetPSP                                                            */
/* -------------------------------------------------------------------- */

psp FAR *DosGetPSP(void)
{
    SEGMENT psp_seg;

    ASM
    {
        mov     ah,0x62
        int     0x21
        mov     psp_seg,bx
    }

    return MK_FP(psp_seg,0);
}


/* -------------------------------------------------------------------- */
/* DosGetTime                                                           */
/* -------------------------------------------------------------------- */

void DosGetTime(UBYTE *hp, UBYTE *mp, UBYTE *sp, UBYTE *hdp)
{
    UBYTE Hour = 0,
    Minutes = 0,
    Seconds = 0,
    Hundredths = 0;

    ASM
    {
        mov     ah,0x2C
        int     0x21
        mov     Hour,ch
        mov     Minutes,cl
        mov     Seconds,dh
        mov     Hundredths,dl
    }

    *hp = Hour;
    *mp = Minutes;
    *sp = Seconds;
    *hdp = Hundredths;
}


/* -------------------------------------------------------------------- */
/* DosGetVerify                                                         */
/* -------------------------------------------------------------------- */

BOOL DosGetVerify(void)
{
    UBYTE status;

    ASM
    {
        mov     ah,0x54
        int     0x21
        mov     status,al
    }

    return (status == 0) ? FALSE : TRUE;
}


/* -------------------------------------------------------------------- */
/* DosOpen                                                              */
/* -------------------------------------------------------------------- */

DWORD DosOpen(char FAR *FileName, UBYTE FileAccess)
{
    SEGMENT fname_seg = FP_SEG(FileName);
    OFFSET fname_off = FP_OFF(FileName);
    UWORD Handle;

    ASM
    {
        push    ds
        mov     ds,fname_seg
        mov     dx,fname_off
        mov     al,FileAccess
        mov     ah,0x3D
        int     0x21
        pop     ds
        mov     Handle,ax
        jnc     exit
    }

    return -((DWORD)Handle);

    exit:
    return (DWORD)Handle;
}


/* -------------------------------------------------------------------- */
/* DosParseFilename                                                     */
/* -------------------------------------------------------------------- */

BOOL DosParseFilename(char FAR *pParseInput, fcb FAR *pFileFCB, UBYTE ParseControl)
{
    SEGMENT SegParseInput = FP_SEG(pParseInput);
    OFFSET OffsetParseInput = FP_OFF(pParseInput);
    SEGMENT SegFileFCB = FP_SEG(pFileFCB);
    OFFSET OffsetFileFCB = FP_OFF(pFileFCB);

    ASM
    {
        push    ds
        mov     ds,SegParseInput
        mov     si,OffsetParseInput
        push    es
        mov     es,SegFileFCB
        mov     di,OffsetFileFCB
        mov     al,ParseControl
        mov     ah,0x29
        int     0x21
        pop     es
        pop     ds
        cmp     al,0xFF
        jne     exit
    }

    return FALSE;

    exit:
    return TRUE;
}


/* -------------------------------------------------------------------- */
/* DosPwd                                                               */
/* -------------------------------------------------------------------- */

DWORD DosPwd(UBYTE Drive, char FAR *CurDir)
{
    SEGMENT CurDir_seg = FP_SEG(CurDir);
    OFFSET CurDir_off = FP_OFF(CurDir);
    UWORD status;

    ASM
    {
        push    ds
        mov     ds,CurDir_seg
        mov     si,CurDir_off
        mov     dl,Drive
        mov     ah,0x47
        int     0x21
        pop     ds
        mov     status,ax
        jnc     exit
    }

    return -((DWORD)status);

    exit:
    return 0;
}


/* -------------------------------------------------------------------- */
/* DosRead                                                              */
/* -------------------------------------------------------------------- */

DWORD DosRead(UWORD fd, char FAR *buffer, UWORD count)
{
    SEGMENT buff_seg = FP_SEG(buffer);
    OFFSET buff_off = FP_OFF(buffer);
    UWORD status;

    ASM
    {
        mov     bx,fd
        mov     cx,count
        push    ds
        mov     ds,buff_seg
        mov     dx,buff_off
        mov     ah,0x3F
        int     0x21
        pop     ds
        mov     status,ax
        jnc     exit
    }

    return -((DWORD)status);

    exit:
    return (DWORD)status;
}


/* -------------------------------------------------------------------- */
/* DosRtnValue                                                          */
/* -------------------------------------------------------------------- */

UWORD DosRtnValue(void)
{
    UWORD status;

    ASM
    {
        mov     ah,0x4d
        int     0x21
        mov     status,ax
    }

    return status;
}


/* -------------------------------------------------------------------- */
/* DosSeek                                                              */
/* -------------------------------------------------------------------- */

DWORD DosSeek(UWORD Handle, UBYTE MoveMethod, DWORD FOffset)
{
    UWORD hi_pos = (UWORD)((FOffset & 0xFFFF0000L) >> 16);
    UWORD lo_pos = (UWORD)(FOffset & 0x0000FFFFL);

    ASM
    {
        mov     bx,Handle
        mov     dx,lo_pos
        mov     cx,hi_pos
        mov     al,MoveMethod
        mov     ah,0x42
        int     0x21
        mov     hi_pos,dx
        mov     lo_pos,ax
    }

    return (DWORD)(((DWORD)hi_pos << 16) | (DWORD)lo_pos);
}


/* -------------------------------------------------------------------- */
/* DosSetBreak                                                          */
/* -------------------------------------------------------------------- */

void DosSetBreak(BOOL BreakFlag)
{
    UBYTE bflag = BreakFlag ? 1 : 0;

    ASM
    {
        mov     dl,bflag
        mov     ax,0x3301
        int     0x21
    }
}


/* -------------------------------------------------------------------- */
/* DosSetDrive                                                          */
/* -------------------------------------------------------------------- */

void DosSetDrive(UBYTE DriveNumber)
{
    ASM
    {
        mov     dl,DriveNumber
        mov     ah,0x0E
        int     0x21
    }
}


/* -------------------------------------------------------------------- */
/* DosSetInt                                                            */
/* -------------------------------------------------------------------- */

void DosSetInt(UBYTE IntNum, void FAR *IntHandler)
{
    SEGMENT handler_seg = FP_SEG(IntHandler);
    OFFSET handler_off = FP_OFF(IntHandler);

    ASM
    {
        mov     dx,handler_off
        push    ds
        mov     ds,handler_seg
        mov     al,IntNum
        mov     ah,0x25
        int     0x21
        pop     ds
    }
}


/* -------------------------------------------------------------------- */
/* DosSetVerify                                                         */
/* -------------------------------------------------------------------- */

void DosSetVerify(BOOL mode)
{
    UBYTE nmode = mode ? 1 : 0;

    ASM
    {
        mov     al,nmode
        mov     ah,0x2E
        int     0x21
    }
}


/* -------------------------------------------------------------------- */
/* DosWrite                                                             */
/* -------------------------------------------------------------------- */

DWORD DosWrite(UWORD fd, char FAR *buffer, UWORD count)
{
    SEGMENT buff_seg = FP_SEG(buffer);
    OFFSET buff_off = FP_OFF(buffer);
    UWORD status;

    ASM
    {
        mov     bx,fd
        mov     cx,count
        push    ds
        mov     ds,buff_seg
        mov     dx,buff_off
        mov     ah,0x40
        int     0x21
        pop     ds
        mov     status,ax
        jnc     exit
    }

    return -((DWORD)status);

    exit:
    return (DWORD)status;
}
