/*
   dblfiles.c - double files checking.
   Copyright (C) 2002 Imre Leber

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   If you have any questions, comments, suggestions, or fixes please
   email me at:  imre.leber@worldonline.be
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "fte.h"
#include "..\chkdrvr.h"
#include "..\struct\FstTrMap.h"
#include "..\errmsgs\errmsgs.h"
#include "..\errmsgs\PrintBuf.h"

/*
** We use a hash table (26*26 +1 in size) and an array large enough to contain
** all the needed info from a directory.
**
** For every directory entry we hash the file name to a certain slot. If the 
** slot already has entries assigned, then it is looked wether any of the
** elements has the same file name and extension.
*/

#define HASHTABLESIZE ((26 * 26) + 1)

struct ArrayElement
{
    char filename[8];
    char extension[3];
    unsigned next;
};

struct HashSlot
{
    unsigned ptr;
};

struct Pipe
{
    struct HashSlot*     hashtable;
    struct ArrayElement* filenamearray;

    CLUSTER surroundingdir;
    BOOL fixit;
    BOOL doublefound;
};

static int ArrayFreePointer;

static BOOL SearchDirForDoubles(RDWRHandle handle, CLUSTER firstcluster,
                                BOOL fixit);
static BOOL DoubleFilenameFinder(RDWRHandle handle,
				 struct DirectoryPosition* pos,
				 struct DirectoryEntry* entry,
				 void** structure);
static BOOL RenameDoubleFile(RDWRHandle handle,
                             struct DirectoryPosition* pos,
                             struct DirectoryEntry* entry,
                             CLUSTER firstcluster);
static BOOL SlowSearchDirForDoubles(RDWRHandle handle, CLUSTER firstcluster,
                                    BOOL fixit);
static BOOL FilenameHasher(RDWRHandle handle, struct DirectoryPosition* pos,
                           void** structure);
static BOOL DoubleFilenameRenamer(RDWRHandle handle,
                                  struct DirectoryPosition* pos,
				  struct DirectoryEntry* entry,
                                  void** structure);

/*========================== Checking ====================================/

/*************************************************************************
**                           FindDoubleFiles
**************************************************************************
** Searches trough all directories for files that occur twice in a directory.
**************************************************************************/

RETVAL FindDoubleFiles(RDWRHandle handle)
{
    BOOL invalid = FALSE, *pinvalid = &invalid, inroot;

    /* Search the root directory */
    inroot = SearchDirForDoubles(handle, 0, FALSE);
    if (inroot == FAIL) return ERROR;

    if (!FastWalkDirectoryTree(handle, DoubleFilenameFinder,
                               (void**) &pinvalid))
       return ERROR;

    return (inroot || invalid) ? FAILED : SUCCESS;
}

/*************************************************************************
**                           SearchDirForDoubles
**************************************************************************
** Searches through a directory for doubles and returns:
**    
**  TRUE  if there were doubles.
**  FALSE if there were no doubles.
**  FAIL  if there was a medium error.
**************************************************************************/

static BOOL SearchDirForDoubles(RDWRHandle handle, CLUSTER firstcluster,
                                BOOL fixit)
{
    struct HashSlot* hashtable;
    struct ArrayElement* filenamearray;
    unsigned long dircount;
    struct Pipe pipe, *ppipe = &pipe;

    ArrayFreePointer = 0;

    /* Try to allocate memory for the hash table. */
    hashtable = (struct HashSlot*)
                malloc(sizeof(struct HashSlot) *  HASHTABLESIZE);
    if (!hashtable)
       return SlowSearchDirForDoubles(handle, firstcluster, fixit);

    /* Make sure all next pointers are invalidated */
    memset(hashtable, 0xff, sizeof(struct HashSlot) *  HASHTABLESIZE);
    
    /* 0xFF means all the files in the directory */
    dircount = low_dircount(handle, firstcluster, 0xFF); 
    if (dircount == FAIL)                               /* FAIL == -1 */
    {
       free(hashtable);
       return SlowSearchDirForDoubles(handle, firstcluster, fixit);
    }

    /* Try to allocate memory for the array. */
    filenamearray = (struct ArrayElement*)
		    malloc(((unsigned)dircount) * sizeof(struct ArrayElement));
    if (!filenamearray)
    {
       free(hashtable);
       return SlowSearchDirForDoubles(handle, firstcluster, fixit);
    }

    /* All memory allocated, start up the fast method. */
    pipe.hashtable      = hashtable;
    pipe.filenamearray  = filenamearray;
    pipe.doublefound    = FALSE;
    pipe.fixit          = fixit;
    pipe.surroundingdir = firstcluster;

    if (!TraverseSubdir(handle, firstcluster, FilenameHasher,
                        (void**) &ppipe, TRUE))
    {
       free(hashtable);
       free(filenamearray);
       return FAIL;
    }

    free(hashtable);
    free(filenamearray);

    return pipe.doublefound;
}

/*************************************************************************
**                           HashFilename
**************************************************************************
** The hash function
**************************************************************************/

static unsigned HashFilename(char* filename)
{
#if 0
   char buf[2];
   memcpy(buf, filename, 2);

   if (((buf[0] < 'A') || (buf[0] > 'Z')) ||
       ((buf[1] < 'A') || (buf[1] > 'Z')))
   {
      return 26*26;
   }

   buf[0] -= 'A';
   buf[1] -= 'A';

   return ((unsigned)buf[0]) * 26 + ((unsigned)buf[1]);
#endif
#if 1                           /* Alternative method */
   return (unsigned)
             (
              (*((unsigned long*)  filename) +
               *(((unsigned long*) filename) + 1)) %
          HASHTABLESIZE);

#endif
}

/*************************************************************************
**                           FilenameHasher
**************************************************************************
** Fast method of searching for doubles.
**
** Use a hash table.
**************************************************************************/

static BOOL FilenameHasher(RDWRHandle handle, struct DirectoryPosition* pos,
                           void** structure)
{
    unsigned slot, now, previous = 0xffff;
    struct DirectoryEntry entry;
    struct Pipe* pipe = *((struct Pipe**) structure);

    if (!GetDirectory(handle, pos, &entry))
       return FAIL;

    if ((entry.attribute & FA_LABEL) ||
        IsLFNEntry(&entry)           ||
        IsDeletedLabel(entry))
    {
       return TRUE;
    }

    slot = HashFilename(entry.filename);

    /* Now walk the list starting at the head in the hash table. */
    now = pipe->hashtable[slot].ptr;
    while (now != 0xFFFF)
    {
       if ((memcmp(entry.filename,
                   pipe->filenamearray[now].filename, 8) == 0) &&
           (memcmp(entry.extension,
                   pipe->filenamearray[now].extension, 3) == 0))
       {
           /* Print out the message and the file name */
           ShowDirectoryViolation(handle, pos, &entry,  
                                  "Found double file %s");            
           
           if (pipe->fixit)
           {
              if (!RenameDoubleFile(handle, pos, &entry,
                                    pipe->surroundingdir))
	         return FAIL;
           }

           pipe->doublefound = TRUE;
           return TRUE;
       }

       previous = now;
       now = pipe->filenamearray[now].next;
    }

    /* Not found, add it to the end of the list */
    if (previous == 0xFFFF) /* First one in the slot. */
    {
       pipe->hashtable[slot].ptr = ArrayFreePointer;
    }
    else
    {  /* We already have the end of the list */
       pipe->filenamearray[previous].next = ArrayFreePointer;
    }

    memcpy(pipe->filenamearray[ArrayFreePointer].filename, entry.filename, 8);
    memcpy(pipe->filenamearray[ArrayFreePointer].extension, entry.extension, 3);
    pipe->filenamearray[ArrayFreePointer].next = 0xFFFF;

    ArrayFreePointer++;

    return TRUE;
}

/*************************************************************************
**                           SlowSearchDirForDoubles
**************************************************************************
** Slow method of searching for doubles.
**
** Compare every file name with every other file name.
**************************************************************************/

static BOOL SlowSearchDirForDoubles(RDWRHandle handle, CLUSTER firstcluster,
                                    BOOL fixit)
{
    unsigned long i, j;
    struct DirectoryPosition pos = {1,1};
    struct DirectoryEntry entry;
    
    char filename[8];
    char extension[3];
    BOOL doublesfound = FALSE;    

    for (i = 1;; i++)
    {
        pos.sector = 0;
        pos.offset = 0;

	if (!GetNthDirectoryPosition(handle, firstcluster, i, &pos))
           return FAIL;

        if ((pos.sector == 0) && (pos.offset == 0)) break;

        if (!GetDirectory(handle, &pos, &entry))
           return FAIL;

	if ((entry.attribute & FA_LABEL) ||
	    IsLFNEntry(&entry)           ||
	    IsDeletedLabel(entry))
	{
	   continue;
	}

        memcpy(filename, entry.filename, 8);
        memcpy(extension, entry.extension, 3);

        for (j = 0; j < i; j++)
        {
	    if (!GetNthDirectoryPosition(handle, firstcluster, j, &pos))
               return FAIL;

           if (!GetDirectory(handle, &pos, &entry))
	      return FAIL;

	   if ((entry.attribute & FA_LABEL) ||
	       IsLFNEntry(&entry)           ||
	       IsDeletedLabel(entry))
	   {
	      continue;
	   }

	   if (IsLFNEntry(&entry) || IsDeletedLabel(entry))
	      continue;

	   if ((memcmp(filename, entry.filename, 8) == 0) &&
	       (memcmp(extension, entry.extension, 3) == 0))
	   {
              doublesfound = TRUE;
           
              /* Print out the message and the file name */
              ShowDirectoryViolation(handle, &pos, &entry,  
                                     "Found double file %s");  

              if (fixit)
              {
                 if (!RenameDoubleFile(handle, &pos, &entry, firstcluster))
	    	    return FAIL;
              }
      	   }
	}
    }
    
    return doublesfound;
}

/*************************************************************************
**                           DoubleFilenameFinder
**************************************************************************
** Intermediate function to get to every directory in the volume.
**************************************************************************/

static BOOL DoubleFilenameFinder(RDWRHandle handle,
                                 struct DirectoryPosition* pos,
				 struct DirectoryEntry* entry,
                                 void** structure)
{
    CLUSTER firstcluster;
    BOOL* invalid = *((BOOL**) structure);

    pos = pos;

    if (IsLFNEntry(entry))
       return TRUE;

    if (IsDeletedLabel(*entry))
       return TRUE;

    if (entry->attribute & FA_DIREC)
    {
       firstcluster = GetFirstCluster(entry);
       if (firstcluster)
       {
          switch (SearchDirForDoubles(handle, firstcluster, FALSE))
          {
             case TRUE:
                  *invalid = TRUE;
                  return TRUE;
               
             case FAIL:
                  return FAIL;
          }
       }
    }

    return TRUE;
}


/*=========================== Fixing ====================================*/

/*************************************************************************
**                           RenameDoubleFiles
**************************************************************************
** Looks through all the directories to find double file names. If there
** are files with the same name the second one found is renamed.
**************************************************************************/

RETVAL RenameDoubleFiles(RDWRHandle handle)
{
    if (SearchDirForDoubles(handle, 0, TRUE) == FAIL)
       return ERROR;

    return (FastWalkDirectoryTree(handle, DoubleFilenameRenamer, NULL)) ?
           SUCCESS : ERROR;
}

/*************************************************************************
**                           DoubleFilenameRenamer
**************************************************************************
** Takes a directory entry and if that is a directory searches it for double
** file names. If it contains doubles then they are renamed.
**************************************************************************/

static BOOL DoubleFilenameRenamer(RDWRHandle handle,
                                  struct DirectoryPosition* pos,
				  struct DirectoryEntry* entry,
                                  void** structure)
{
    CLUSTER firstcluster;
    
    structure = structure, pos = pos;

    if (IsLFNEntry(entry))
       return TRUE;

    if (IsDeletedLabel(*entry))
       return TRUE;

    if (entry->attribute & FA_DIREC)
    {
       firstcluster = GetFirstCluster(entry);
       if (firstcluster &&
           (SearchDirForDoubles(handle, firstcluster, TRUE) == FAIL))
       {
	  return FAIL;
       }
    }

    return TRUE;
}

/*************************************************************************
**                           RenameDoubleFile
**************************************************************************
** Renames a double file. It does this by taking a file name and adding a 
** a numeric extension to it.
**
** eg. defrag.lst => defrag.000
**************************************************************************/

static BOOL RenameDoubleFile(RDWRHandle handle,
                             struct DirectoryPosition* pos,
                             struct DirectoryEntry* entry,
                             CLUSTER firstcluster)
{
    BOOL retval;
    int counter=0;
    char newext[3];

    for (;;)
    {
        sprintf(newext, "%3d", counter);
        
        retval = LoFileNameExists(handle, firstcluster,
                                  entry->filename, newext);

        if (retval == FALSE) break;
        if (retval == FAIL)  return FALSE;

        counter++;
    }

    memcpy(entry->extension, newext, 3);

    return WriteDirectory(handle, pos, entry);
}