/*
   DtDtPnt.c - '..' checking.
   Copyright (C) 2002 Imre Leber

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   If you have any questions, comments, suggestions, or fixes please
   email me at:  imre.leber@worldonline.be
*/

#include <stdio.h>
#include <string.h>

#include "fte.h"
#include "..\chkdrvr.h"
#include "..\struct\FstTrMap.h"
#include "..\errmsgs\printbuf.h"

static BOOL RootDirChecker(RDWRHandle handle,
                           struct DirectoryPosition* pos,
                           void** structure);

static BOOL FirstClusterFinder(RDWRHandle handle,
                               struct DirectoryPosition* pos,
                               struct DirectoryEntry* entry,
                               void** structure);

static BOOL SubDirChecker(RDWRHandle handle, struct DirectoryPosition* pos,
                          void** structure);
                          
static BOOL SubDirChecker(RDWRHandle handle, struct DirectoryPosition* pos,
                          void** structure);

static BOOL CheckDblDots(RDWRHandle handle, CLUSTER firstcluster,
                         CLUSTER topointto, BOOL* invalid,
                         struct DirectoryPosition* pos,
                         struct DirectoryEntry* entry,
                         BOOL fixit);
                      
struct Pipe
{
   CLUSTER topointto;
   BOOL*   invalid;
   BOOL    fixit;
};

/*=========================== Checking ==================================*/

/*************************************************************************
**                           CheckDotDotPointer
**************************************************************************
** Checks wether the '..' directory entries point to the parent of the parent
** directory.
**************************************************************************/

RETVAL CheckDotDotPointer(RDWRHandle handle)
{
     struct Pipe pipe, *ppipe = &pipe;
     
     pipe.invalid = FALSE;
     pipe.fixit   = FALSE;

     /* Root directory */
     if (!TraverseRootDir(handle, RootDirChecker, (void**) &ppipe, TRUE))
        return ERROR;

     /* Sub directories */   
     if (!FastWalkDirectoryTree(handle, FirstClusterFinder,
                                (void**) &ppipe))
        return ERROR;

     return (pipe.invalid) ? FAILED : SUCCESS;
}

/*============================ Fixing ==================================*/

/*************************************************************************
**                           AdjustDotDotPointer
**************************************************************************
** Checks wether the '..' directory entries point to the parent of the parent
** directory, and if they do not they are adjusted.
**************************************************************************/

RETVAL AdjustDotDotPointer(RDWRHandle handle)
{
     struct Pipe pipe, *ppipe = &pipe;
     
     pipe.invalid = FALSE;
     pipe.fixit   = TRUE;

     /* Root directory */
     if (!TraverseRootDir(handle, RootDirChecker, (void**) &ppipe, TRUE))
        return ERROR;

     /* Sub directories */   
     if (!FastWalkDirectoryTree(handle, FirstClusterFinder,
                                (void**) &ppipe))
        return ERROR;

     return SUCCESS;
}

/*============================ Common ==================================*/

/*************************************************************************
**                           RootDirChecker
**************************************************************************
** Checks wether the '..' directory entries point to the parent of the parent
** directory.
**************************************************************************/

static BOOL RootDirChecker(RDWRHandle handle,
                           struct DirectoryPosition* pos,
                           void** structure)
{
     CLUSTER firstcluster;
     struct Pipe* pipe = *((struct Pipe**) structure); 
     struct DirectoryEntry entry;

     if (!GetDirectory(handle, pos, &entry))
        return FAIL;

     if (IsLFNEntry(&entry)    ||
         IsDeletedLabel(entry) ||
         (entry.attribute & FA_LABEL))
     {
        return TRUE;
     }

     if (entry.attribute & FA_DIREC)
     {
        firstcluster = GetFirstCluster(&entry);
        if (firstcluster &&
            (!CheckDblDots(handle, firstcluster, 0, 
                           pipe->invalid, 
                           pos, &entry,
                           pipe->fixit)))
        {
           return FAIL;
        }
     }

     return TRUE;
}

/*************************************************************************
**                         FirstClusterFinder
**************************************************************************
** Takes a certain directory and looks wether in that directory all sub
** directories have '..' entries that point to the given directory.
**************************************************************************/

static BOOL FirstClusterFinder(RDWRHandle handle,
                               struct DirectoryPosition* pos,
			       struct DirectoryEntry* entry,
                               void** structure)
{
     struct Pipe* pipe = *((struct Pipe**) structure);
    
     pos = pos;
     
     if ((entry->attribute & FA_LABEL) ||
         IsLFNEntry(entry)             ||
         IsDeletedLabel(*entry)        ||         
         IsPreviousDir(*entry)         || 
         IsCurrentDir(*entry))
     {
        return TRUE;     
     }
     
     if (entry->attribute & FA_DIREC)
     {
        pipe->topointto = GetFirstCluster(entry);

        if (pipe->topointto)
        {
           if (!TraverseSubdir(handle, pipe->topointto,
			       SubDirChecker, (void**) &pipe, TRUE))
           {
              return FAIL;
           }
        }
     }

     return TRUE;
}

/*************************************************************************
**                             SubDirChecker
**************************************************************************
** Goes through all the sub directories of a certain directory and checks 
** wether the sub directory has a '..' entry that points to the directory.
**************************************************************************/

static BOOL SubDirChecker(RDWRHandle handle, struct DirectoryPosition* pos,
                          void** structure)
{
     CLUSTER firstcluster;
     struct Pipe *pipe = *((struct Pipe**) structure);
     struct DirectoryEntry entry;

     if (!GetDirectory(handle, pos, &entry))
        return FAIL;

     if ((entry.attribute & FA_LABEL) ||
         IsLFNEntry(&entry)           ||
         IsDeletedLabel(entry)        ||         
         IsPreviousDir(entry)         || 
         IsCurrentDir(entry))
     {
        return TRUE;     
     }

     if (entry.attribute & FA_DIREC)    /* sub directory */
     {
        firstcluster =  GetFirstCluster(&entry);
        if (firstcluster &&
            (!CheckDblDots(handle, firstcluster,
                           pipe->topointto, pipe->invalid,
                           pos, &entry,
                           pipe->fixit)))
        {
           return FAIL;
        }
     }

     return TRUE;
}

/*************************************************************************
**                             CheckDblDots
**************************************************************************
** Checks the directory pointed to by firstcluster to see wether the '..' 
** entry points to the directory pointed to by topointto.
**************************************************************************/

static BOOL CheckDblDots(RDWRHandle handle, CLUSTER firstcluster,
                         CLUSTER topointto, BOOL* invalid,
                         struct DirectoryPosition* dirpos,
                         struct DirectoryEntry* direntry,
                         BOOL fixit)
{
   struct DirectoryEntry entry;
   struct DirectoryPosition pos = {0,0};

   /* Get the position of the second directory entry */  
   if (!GetNthDirectoryPosition(handle, firstcluster, 1, &pos))
   {
      return FALSE;
   }
   
   /* See wether there was a second directory entry */
   if ((pos.sector == 0) && (pos.offset == 0))
   {
      ShowDirectoryViolation(handle, dirpos, direntry,  
                             "%s is a directory without '..'");            
      return TRUE;
   }

   /* Read the second directory entry from disk. */
   if (!GetDirectory(handle, &pos, &entry))
      return FALSE;

   /* See wether it is a '..' entry */   
   if (!IsPreviousDir(entry))
   {
      ShowDirectoryViolation(handle, dirpos, direntry,  
                             "%s is a directory without '..'");   
      return TRUE;
   }

   /* If it is a see wether it is pointing to the right directory */
   if (GetFirstCluster(&entry) != topointto)
   {
      ShowDirectoryViolation(handle, dirpos, direntry,  
                             "The '..' entry in %s is not pointing to the parent directory");              

      if (fixit)
      {
         SetFirstCluster(topointto, &entry);
         if (!WriteDirectory(handle, &pos, &entry))
            return FALSE;
      }
      else
      {
         *invalid = TRUE;
      }
   }

   return TRUE;
}