/*
   Chain.c - file chain checking.
   Copyright (C) 2002 Imre Leber

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   If you have any questions, comments, suggestions, or fixes please
   email me at:  imre.leber@worldonline.be
*/

#include <stdio.h>

#include "fte.h"

#include "..\chkdrvr.h"
#include "..\struct\FstTrMap.h"

static BOOL FileChainChecker(RDWRHandle handle, struct DirectoryPosition* pos,
                             struct DirectoryEntry* entry, void** structure);

static BOOL CheckTheFileChain(RDWRHandle handle, CLUSTER startcluster,
                              BOOL fixit);

static BOOL FileChainFixer(RDWRHandle handle, struct DirectoryPosition* pos,
                           struct DirectoryEntry* entry,  void** structure);

/* Checking */

RETVAL CheckFileChain(RDWRHandle handle)
{
    BOOL invalid = FALSE, *pinvalid = &invalid;

    if (!FastWalkDirectoryTree(handle, FileChainChecker, (void**) &pinvalid))
       return ERROR;

    return (invalid) ? FAILED : SUCCESS;
}

static BOOL FileChainChecker(RDWRHandle handle, struct DirectoryPosition* pos,
                             struct DirectoryEntry* entry, void** structure)
{
    CLUSTER firstcluster;
    BOOL* invalid = *((BOOL**) structure), retval;

    pos = pos;

    if ((entry->attribute & FA_LABEL) ||
	IsLFNEntry(entry)             ||
	IsCurrentDir(*entry)          ||
	IsPreviousDir(*entry)         ||
	IsDeletedLabel(*entry))
       return TRUE;

    firstcluster = GetFirstCluster(entry);
    if (firstcluster)
    {
       retval = CheckTheFileChain(handle, firstcluster, FALSE);

       if (retval == FALSE) *invalid = TRUE;
       if (retval == FAIL)  return ERROR;
    }

    return TRUE;
}

/* Fixing */
RETVAL BreakInvalidChain(RDWRHandle handle)
{
    return (FastWalkDirectoryTree(handle, FileChainFixer, NULL)) ?
           SUCCESS : ERROR;
}

static BOOL FileChainFixer(RDWRHandle handle, struct DirectoryPosition* pos,
                           struct DirectoryEntry* entry, void** structure)
{
    CLUSTER firstcluster;
    BOOL* invalid = *((BOOL**) structure), retval;

    pos = pos;

    if ((entry->attribute & FA_LABEL) ||
	IsLFNEntry(entry)             ||
	IsCurrentDir(*entry)          ||
	IsPreviousDir(*entry)         ||
	IsDeletedLabel(*entry))
       return TRUE;

    firstcluster = GetFirstCluster(entry);
    if (firstcluster)
    {
       retval = CheckTheFileChain(handle, firstcluster, TRUE);

       if (retval == FALSE) *invalid = TRUE;
       if (retval == FAIL)  return ERROR;
    }

    return TRUE;
}

/* Common */

/*
   Returns:
      FALSE if there was an invalid cluster
      TRUE  if there was no invalid cluster

      FAIL  if there was an error
*/

static BOOL CheckTheFileChain(RDWRHandle handle, CLUSTER startcluster, BOOL fixit)
{
    BOOL invalidfound = FALSE;
    CLUSTER current = startcluster, previous;

    while (!FAT_LAST(current))
    {
        previous = current;
        
        if (!GetNthCluster(handle, current, &current))
           return FAIL;

        if (FAT_BAD(current) || FAT_FREE(current))
        {
	   printf("Found a free or bad cluster in a file chain\n");
           invalidfound = TRUE;   
             
           if (fixit)
           {
              if (FAT_BAD(current))
              {
                 if (!WriteFatLabel(handle, previous, FAT_LAST_LABEL))
                    return FAIL;
              }
              else
              {
                 if (!WriteFatLabel(handle, current, FAT_LAST_LABEL))
                    return FAIL;
              }
           }
           break;
        }
    }
    
    return !invalidfound;
}
