\ ----------------------------------------------------------------------
\ LANGUAGE    : ANS Forth 
\ PROJECT     : CHForth, a Dutch public domain ANS Forth for PC's 
\ DESCRIPTION : Searching words in files
\ CATEGORY    : Utilities
\ AUTHOR      : Coos Haak 
\ LAST CHANGE : September 09, 1994, Coos Haak 
\ ----------------------------------------------------------------------



        ?DEF -searcher [IF] -searcher [THEN]


        MARKER -searcher


DOC

The word display contains inside information, it is the number of items on
the stack what SAVE-INPUT has put there. I move those temporarily to the
return stack so the number does not clutter up the stack depth info on the
status line. This number could be changed in the future without warning!

ENDDOC

privates

0 value line-counter                    private \ Count printed lines
0 value terminate?                      private \ User break

: reset-counter     ( -- )
        l/scr 1- to line-counter                \ Fresh display
        clear terminate?                        \ Don't stop at the beginning
    ;  private

: cr?               ( -- flag )
        cr -1 +to line-counter
        line-counter
        if      false                           \ Not yet at end of screen
        else    reset-counter ?at               \ New display 
                ." Press a key to continue"     \ Message
                key ^[ =                        \ Escape pressed ?
                dup to terminate?               \ Stop whole program
                -rot at-xy eol                  \ Clear whole line
        then
    ;  private

create findbuffer                       private \ Buffer with string to find
        here /line cell+ dup allot erase

create linebuffer                       private \ Save line here
        here /line cell+ dup allot erase

0 value total                           private \ Signals any help found

: ?upper        ( c-addr -- )                   \ convert to uppercase
        casesensitive @ invert
        if      count upper
        else    drop
        then
        ;  private

: display       ( -- )                          \ some echo on status line
        [ also internal ]
        showstatus?
        [ previous ]
        if      >r 2>r 2>r 2>r                  \ Secret information
                .status                         \ other status info
                2r> 2r> 2r> r>                  \ Secret information
                ?at attr @                      \ save location and attribute
                statusattr @ attr !             \ set screen attribute
                #53 0 at-xy                     \ free on status line
                found-file tuck type            \ type file name, keep count
                #14 swap - spaces               \ fill with spaces
                attr ! at-xy                    \ restore loc and attribute
        then
        ;  private

0 value new-file                private         \ flag

: get-string    ( x -- )                        \ delimiter
        parse-word nip 0=                       \ null string
        abort" Needs a string parameter"
        ;  private

: open-the-file ( c-addr u -- )
        2dup r/o open-file throw to source-id   \ open the file
        'name place                             \ keep name
        true to new-file                        \ a new opened file
        #lines off                              \ no lines read yet
        ;  private

: search-and-display
        linebuffer count                        \ first string
        findbuffer count                        \ second string
        search nip nip                          \ keep only flag
        if      1 +to total                     \ increment counter
                cr?
                if      r>drop exit             \ User break
                then
                total 3 .r space
                #14 'name count #13 min tuck type - spaces
                #lines @ 4 .r space
                source linebuffer expand        \ discard tabs, no uppercase
                c/l 1- ?at drop - min type      \ type the line
        then
        ;  private

: (sf)
        begin   refill                  \ get a line
        while   source dup              \ when not length zero
                if      linebuffer 1+
                        expand          \ no tabs
                        swap 1- tuck    \ save count
                        c! ?upper       \ to capitals
                        search-and-display
                else    2drop           \ throw string away
                then
        repeat
    ;  private

: stop2?        ( -- flag )                     \ Other version of STOP?
        key?
        if      key ^[ =
        else    false
        then
    ;  private

\G Skip leading SEPARATOR delimiters. Parse ccc delimited by
\G SEPARATOR . Search the files with extension given by FEXT$ in the
\G directory given by LIBPATH . Find ccc in the files. Display the
\G number of lines found, the name of the file, the line number and
\G the line depending on the width of the screen. If a full screen
\G is displayed, wait for the user to press a key. Stop if the key
\G is the escape key.
: SL            ( "ccc" -- )                    \ SEARCHER "search-libraries"
        reset-counter
        separator get-string                    \ get a string
        libpath count set-directory throw       \ set searchpath
        fext$ count                             \ default extension
        s" *" here pack append                  \ make "*.frt" name
        here count find-first-file throw        \ look for first occurence
        parsed-word here pack                   \ place string on here
                                                \ when you have an error
        count findbuffer pack ?upper            \ put in find buffer
        save-input                              \ you can open files now
        clear total                             \ no files yet
        begin   found-file open-the-file        \ open the file
                display                         \ echo file name
                (sf)
                source-id close-file throw      \ close the file
                terminate? invert               \ Early break
        while   find-next-file stop2? or        \ next file, user break
        until   then
        restore-input throw
        current-directory count set-directory throw     \ reset path
        total 0=                                \ give message
        if      cr ." No file found with " parsed-word type
        then
        ;

\G Skip leading SEPARATOR delimiters. Parse ccc delimited by
\G SEPARATOR . Search the files with extension given by FEXT$ in the
\G current directory. Find ccc in the files. Display the number of
\G lines found, the name of the file, the line number and the line
\G depending on the width of the screen. If a full screen is
\G displayed, wait for the user to press a key. Stop if the key is
\G the escape key.
: SF            ( "ccc" -- )                    \ SEARCHER "search-forth"
        reset-counter
        separator get-string                    \ get a string
        fext$ count                             \ default extension
        s" *" here pack append                  \ make "*.frt" name
        here count find-first-file throw        \ look for first occurence
        parsed-word here pack                   \ place string on here
                                                \ when you have an error
        count findbuffer pack ?upper            \ put in find buffer
        save-input                              \ you can open files now
        clear total                             \ no files yet
        begin   found-file open-the-file        \ open the file
                display                         \ echo file name
                (sf)
                source-id close-file throw      \ close the file
                terminate? invert               \ Early break
        while   find-next-file stop2? or        \ next file, user break
        until   then
        restore-input throw
        total 0=                                \ give message
        if      cr ." No file found with " parsed-word type
        then
        ;

: .lines        ( -- )
        new-file clear new-file                 \ is it a new file
        if      cr?
                if      r>drop exit             \ User break
                then
                ." File: " found-file type      \ type the name
        then
        begin   cr?                             \ User break
                if      r>drop exit
                then
                source type                     \ type the line
                refill                          \ another line
        while   source 0= swap c@ bl <> or      \ not empty and space at start
        until   then             
        1 +to total                             \ there is at least one found
        cr?                                     \ Empty line
        if      r>drop exit                     \ User break
        then
        ;  private

: (any)
        begin   refill                  \ get a line
        while   source dup              \ when not length zero
                if      linebuffer 1+
                        expand          \ no tabs
                        swap 1- tuck    \ save count
                        c! ?upper       \ to capitals
                        linebuffer 1+ c@ bl <>  \ no space
                        if      linebuffer count    \ first string
                                #16 min             \ start of line
                                findbuffer count    \ second string
                                search nip nip      \ keep only flag
                                if      .lines
                                then
                        then
                else    2drop           \ throw string away
                then
        repeat
    ;  private

\G Skip leading space delimiters. Parse ccc delimited by a space.
\G Search the files with extension given by HEXT$ in the directory
\G given by HELPPATH . Display the description of the names that
\G contain ccc. If a full screen is displayed, wait for the user to
\G press a key. Stop if the key is the escape key.
: ANY           ( "ccc" -- )                    \ SEARCHER
        reset-counter                           \ Fresh display
        bl get-string                           \ get a string
        helppath count set-directory throw      \ set searchpath
        hext$ count                             \ default extension
        s" *" here pack append                  \ make "*.hlp" name
        here count find-first-file throw        \ look for first occurence
        parsed-word here pack                   \ place string on here
                                                \ when you have an error
        count findbuffer pack ?upper            \ put in find buffer
        save-input                              \ you can open files now
        clear total                             \ no files yet
        begin   found-file open-the-file        \ open the file
                display                         \ echo file name
                (any)                           \ Do the work
                source-id close-file throw      \ close the file
                terminate? invert               \ Early break
        while   find-next-file stop2? or        \ next file, user break
        until   then
        restore-input throw
        current-directory count set-directory throw     \ reset path
        total 0=                                \ give message
        if      cr ." No help for " parsed-word type
        then
        ;

: (look)
        begin   refill                          \ get a line
        while   source dup                      \ when not length zero
                if      linebuffer 1+
                        expand                  \ no tabs
                        swap 1- tuck            \ save count
                        c! ?upper               \ to capitals
                        linebuffer count        \ first string
                        findbuffer count        \ second string
                        search nip nip          \ keep only flag
                        if      1 +to total     \ increment counter
                                cr?
                                if      exit    \ User break
                                then
                                total 3 .r
                                #lines @ 5 .r space
                                source c/l 1- ?at
                                drop - min type \ type the line
                        then
                else    2drop                   \ throw string away
                then
                stop2?                          \ user break
        until
        then
    ;  private

\G Skip leading space delimiters. Parse name delimited by a space.
\G Skip leading SEPARATOR delimiters. Parse ccc delimited by
\G SEPARATOR . Search file name with optional extension given by
\G FEXT$ . Find ccc in the file. Display the number of the lines 
\G found, the line number and the line containing ccc depending on
\G the width of the screen. If a full screen is displayed, wait for
\G the user to press a key. Stop if the key is the escape key.
: LOOK          ( "name" "ccc" --- )            \ SEARCHER
        reset-counter                           \ Fresh display
        from >in                                \ save >in
        separator get-string                    \ get a string
        to >in                                  \ restore >in
        getname 2>r                             \ get file name
        separator parse-word here pack          \ get string
        count findbuffer pack ?upper            \ and place in buffer
        save-input                              \ time to open the file
        2r> open-the-file                       \ open file
        clear total                             \ nothing found yet
        (look)                                  \ Do the work
        source-id close-file throw              \ close the file
        restore-input throw
        total 0=                                \ give message
        if      cr ." File does not contain " parsed-word type
        then
        ;

deprive

                            \ (* End of Source *) /
