## ----setup, include = FALSE---------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>",
  fig.width = 7,
  fig.height = 4
)

## ----message=FALSE, warning=FALSE---------------------------------------------
library(RegimeChange)

# Generate data with multiple changepoints
set.seed(42)
data <- c(
  rnorm(100, 0, 1),   # Regime 1
  rnorm(100, 3, 1),   # Regime 2
  rnorm(100, 1, 2),   # Regime 3
  rnorm(100, 4, 0.5)  # Regime 4
)
true_cps <- c(100, 200, 300)

# Detect with PELT
result_pelt <- detect_regimes(data, method = "pelt", penalty = "BIC")
print(result_pelt)

## -----------------------------------------------------------------------------
# Different penalties
result_bic <- detect_regimes(data, method = "pelt", penalty = "BIC")
result_aic <- detect_regimes(data, method = "pelt", penalty = "AIC")
result_mbic <- detect_regimes(data, method = "pelt", penalty = "MBIC")

cat("BIC:", result_bic$n_changepoints, "changepoints\n")
cat("AIC:", result_aic$n_changepoints, "changepoints\n")
cat("MBIC:", result_mbic$n_changepoints, "changepoints\n")

## -----------------------------------------------------------------------------
result <- detect_regimes(data, method = "pelt", min_segment = 30)

## -----------------------------------------------------------------------------
result_binseg <- detect_regimes(data, method = "binseg", n_changepoints = 5)
print(result_binseg)

## -----------------------------------------------------------------------------
result_wbs <- detect_regimes(data, method = "wbs", M = 100)
print(result_wbs)

## -----------------------------------------------------------------------------
result_mean <- detect_regimes(data, type = "mean")

## -----------------------------------------------------------------------------
# Data with variance change
set.seed(123)
var_data <- c(rnorm(100, 0, 1), rnorm(100, 0, 3))

result_var <- detect_regimes(var_data, type = "variance")
print(result_var)

## -----------------------------------------------------------------------------
result_both <- detect_regimes(data, type = "both")

## -----------------------------------------------------------------------------
# Basic plot with changepoints
plot(result_pelt, type = "data")

## -----------------------------------------------------------------------------
# Segment-colored plot
plot(result_pelt, type = "segments")

## -----------------------------------------------------------------------------
# Get segment details
for (i in seq_along(result_pelt$segments)) {
  seg <- result_pelt$segments[[i]]
  cat(sprintf("Segment %d: [%d, %d] - Mean: %.2f, SD: %.2f\n", 
              i, seg$start, seg$end, seg$params$mean, seg$params$sd))
}

## -----------------------------------------------------------------------------
result_ci <- detect_regimes(data, method = "pelt", 
                            uncertainty = TRUE, bootstrap_reps = 100)

if (length(result_ci$confidence_intervals) > 0) {
  print(result_ci$confidence_intervals[[1]])
}

## -----------------------------------------------------------------------------
eval_result <- evaluate(result_pelt, true_changepoints = true_cps)
print(eval_result)

## -----------------------------------------------------------------------------
comparison <- compare_methods(
  data = data,
  methods = c("pelt", "binseg", "wbs"),
  true_changepoints = true_cps
)
print(comparison)

## ----eval=FALSE---------------------------------------------------------------
# detect_regimes(data, method = "wbs", M = 200)

## ----eval=FALSE---------------------------------------------------------------
# detect_regimes(data, method = "pelt", penalty = "AIC")

## ----eval=FALSE---------------------------------------------------------------
# detect_regimes(data, method = "ensemble",
#                methods = c("pelt", "wbs", "binseg"))

