/*
 * Decompiled with CFR 0.152.
 */
package htsjdk.samtools.reference;

import htsjdk.samtools.Defaults;
import htsjdk.samtools.SAMException;
import htsjdk.samtools.SAMSequenceDictionary;
import htsjdk.samtools.SAMSequenceRecord;
import htsjdk.samtools.reference.AbstractFastaSequenceFile;
import htsjdk.samtools.reference.FastaSequenceIndex;
import htsjdk.samtools.reference.FastaSequenceIndexEntry;
import htsjdk.samtools.reference.ReferenceSequence;
import htsjdk.samtools.util.IOUtil;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.Iterator;

public class IndexedFastaSequenceFile
extends AbstractFastaSequenceFile
implements Closeable {
    private final FileChannel channel;
    private final FastaSequenceIndex index;
    private Iterator<FastaSequenceIndexEntry> indexIterator;

    public IndexedFastaSequenceFile(File file, FastaSequenceIndex index) {
        super(file);
        FileInputStream in;
        if (index == null) {
            throw new IllegalArgumentException("Null index for fasta " + file);
        }
        this.index = index;
        IOUtil.assertFileIsReadable(file);
        try {
            in = new FileInputStream(file);
        }
        catch (FileNotFoundException e) {
            throw new SAMException("Fasta file should be readable but is not: " + file, e);
        }
        this.channel = in.getChannel();
        this.reset();
        if (this.getSequenceDictionary() != null) {
            IndexedFastaSequenceFile.sanityCheckDictionaryAgainstIndex(file.getAbsolutePath(), this.sequenceDictionary, index);
        }
    }

    public IndexedFastaSequenceFile(File file) throws FileNotFoundException {
        this(file, new FastaSequenceIndex(IndexedFastaSequenceFile.findRequiredFastaIndexFile(file)));
    }

    @Override
    public boolean isIndexed() {
        return true;
    }

    private static File findFastaIndex(File fastaFile) {
        File indexFile = IndexedFastaSequenceFile.getFastaIndexFileName(fastaFile);
        if (!indexFile.exists()) {
            return null;
        }
        return indexFile;
    }

    private static File getFastaIndexFileName(File fastaFile) {
        return new File(fastaFile.getAbsolutePath() + ".fai");
    }

    private static File findRequiredFastaIndexFile(File fastaFile) throws FileNotFoundException {
        File ret = IndexedFastaSequenceFile.findFastaIndex(fastaFile);
        if (ret == null) {
            throw new FileNotFoundException(IndexedFastaSequenceFile.getFastaIndexFileName(fastaFile) + " not found.");
        }
        return ret;
    }

    public static boolean canCreateIndexedFastaReader(File fastaFile) {
        return fastaFile.exists() && IndexedFastaSequenceFile.findFastaIndex(fastaFile) != null;
    }

    protected static void sanityCheckDictionaryAgainstIndex(String fastaFile, SAMSequenceDictionary sequenceDictionary, FastaSequenceIndex index) {
        if (sequenceDictionary.getSequences().size() != index.size()) {
            throw new SAMException("Sequence dictionary and index contain different numbers of contigs");
        }
        Iterator<SAMSequenceRecord> sequenceIterator = sequenceDictionary.getSequences().iterator();
        Iterator<FastaSequenceIndexEntry> indexIterator = index.iterator();
        while (sequenceIterator.hasNext() && indexIterator.hasNext()) {
            SAMSequenceRecord sequenceEntry = sequenceIterator.next();
            FastaSequenceIndexEntry indexEntry = indexIterator.next();
            if (!sequenceEntry.getSequenceName().equals(indexEntry.getContig())) {
                throw new SAMException(String.format("Mismatch between sequence dictionary fasta index for %s, sequence '%s' != '%s'.", fastaFile, sequenceEntry.getSequenceName(), indexEntry.getContig()));
            }
            if ((long)sequenceEntry.getSequenceLength() == indexEntry.getSize()) continue;
            throw new SAMException("Index length does not match dictionary length for contig: " + sequenceEntry.getSequenceName());
        }
    }

    @Override
    public SAMSequenceDictionary getSequenceDictionary() {
        return this.sequenceDictionary;
    }

    @Override
    public ReferenceSequence getSequence(String contig) {
        return this.getSubsequenceAt(contig, 1L, (int)this.index.getIndexEntry(contig).getSize());
    }

    @Override
    public ReferenceSequence getSubsequenceAt(String contig, long start, long stop) {
        if (start > stop + 1L) {
            throw new SAMException(String.format("Malformed query; start point %d lies after end point %d", start, stop));
        }
        FastaSequenceIndexEntry indexEntry = this.index.getIndexEntry(contig);
        if (stop > indexEntry.getSize()) {
            throw new SAMException("Query asks for data past end of contig");
        }
        int length = (int)(stop - start + 1L);
        byte[] target = new byte[length];
        ByteBuffer targetBuffer = ByteBuffer.wrap(target);
        int basesPerLine = indexEntry.getBasesPerLine();
        int bytesPerLine = indexEntry.getBytesPerLine();
        int terminatorLength = bytesPerLine - basesPerLine;
        long startOffset = (start - 1L) / (long)basesPerLine * (long)bytesPerLine + (start - 1L) % (long)basesPerLine;
        ByteBuffer channelBuffer = ByteBuffer.allocate(Math.min(Defaults.NON_ZERO_BUFFER_SIZE, (length % basesPerLine + 2) * bytesPerLine));
        while (targetBuffer.position() < length) {
            startOffset += (long)Math.max((int)(startOffset % (long)bytesPerLine - (long)basesPerLine + 1L), 0);
            try {
                startOffset += (long)this.channel.read(channelBuffer, indexEntry.getLocation() + startOffset);
            }
            catch (IOException ex) {
                throw new SAMException("Unable to load " + contig + "(" + start + ", " + stop + ") from " + this.file);
            }
            channelBuffer.flip();
            int positionInContig = (int)start - 1 + targetBuffer.position();
            int nextBaseSpan = Math.min(basesPerLine - positionInContig % basesPerLine, length - targetBuffer.position());
            int bytesToTransfer = Math.min(nextBaseSpan, channelBuffer.capacity());
            channelBuffer.limit(channelBuffer.position() + bytesToTransfer);
            while (channelBuffer.hasRemaining()) {
                targetBuffer.put(channelBuffer);
                bytesToTransfer = Math.min(basesPerLine, length - targetBuffer.position());
                channelBuffer.limit(Math.min(channelBuffer.position() + bytesToTransfer + terminatorLength, channelBuffer.capacity()));
                channelBuffer.position(Math.min(channelBuffer.position() + terminatorLength, channelBuffer.capacity()));
            }
            channelBuffer.flip();
        }
        return new ReferenceSequence(contig, indexEntry.getSequenceIndex(), target);
    }

    @Override
    public ReferenceSequence nextSequence() {
        if (!this.indexIterator.hasNext()) {
            return null;
        }
        return this.getSequence(this.indexIterator.next().getContig());
    }

    @Override
    public void reset() {
        this.indexIterator = this.index.iterator();
    }

    @Override
    public String toString() {
        return this.file.getAbsolutePath();
    }

    @Override
    public void close() throws IOException {
        this.channel.close();
    }
}

