## ----GenSetup, include = FALSE----------------------------------------------------------------------------------------
knitr::opts_chunk$set(
    collapse = TRUE,
    comment = "#>",
    crop = NULL
    ## Related to
    ## https://stat.ethz.ch/pipermail/bioc-devel/2020-April/016656.html
)

## ----vignetteSetup, echo=FALSE, message=FALSE, warning = FALSE--------------------------------------------------------
## Bib setup
library("RefManageR")

## Write bibliography information
bib <- c(
    R = citation(),
    BiocStyle = citation("BiocStyle")[1],
    knitr = citation("knitr")[1],
    RefManageR = citation("RefManageR")[1],
    rmarkdown = citation("rmarkdown")[1],
    sessioninfo = citation("sessioninfo")[1],
    testthat = citation("testthat")[1],
    ISAnalytics = citation("ISAnalytics")[1],
    VISPA2 = BibEntry(bibtype = "Article", 
     title = paste("VISPA2: a scalable pipeline for", 
                   "high-throughput identification and",
                   "annotation of vector integration sites"),
     author = "Giulio Spinozzi, Andrea Calabria, Stefano Brasca", 
     journaltitle = "BMC Bioinformatics", 
     date = "2017-11-25",
     doi = "10.1186/s12859-017-1937-9")
)

library(ISAnalytics)
library(BiocStyle)
library(DT)
main_fig <- fs::path("../man", "figures", "dyn_vars_general.png")

## ----echo=FALSE-------------------------------------------------------------------------------------------------------
inst_chunk_path <- system.file("rmd", "install_and_options.Rmd", package = "ISAnalytics")

## ----include=FALSE----------------------------------------------------------------------------------------------------
knitr::opts_chunk$set(echo = TRUE)

## ----eval=FALSE-------------------------------------------------------------------------------------------------------
#  if (!requireNamespace("BiocManager", quietly = TRUE))
#      install.packages("BiocManager")
#  BiocManager::install("ISAnalytics")

## ----eval=FALSE-------------------------------------------------------------------------------------------------------
#  if (!requireNamespace("BiocManager", quietly = TRUE))
#      install.packages("BiocManager")
#  
#  # The following initializes usage of Bioc devel
#  BiocManager::install(version='devel')
#  
#  BiocManager::install("ISAnalytics")

## ----eval=FALSE-------------------------------------------------------------------------------------------------------
#  if (!require(devtools)) {
#    install.packages("devtools")
#  }
#  devtools::install_github("calabrialab/ISAnalytics",
#                           ref = "RELEASE_3_15",
#                           dependencies = TRUE,
#                           build_vignettes = TRUE)

## ----eval=FALSE-------------------------------------------------------------------------------------------------------
#  if (!require(devtools)) {
#    install.packages("devtools")
#  }
#  devtools::install_github("calabrialab/ISAnalytics",
#                           ref = "master",
#                           dependencies = TRUE,
#                           build_vignettes = TRUE)

## ----OptVerbose, eval=FALSE-------------------------------------------------------------------------------------------
#  # DISABLE
#  options("ISAnalytics.verbose" = FALSE)
#  
#  # ENABLE
#  options("ISAnalytics.verbose" = TRUE)
#  

## ----OptWidg, eval=FALSE----------------------------------------------------------------------------------------------
#  # DISABLE HTML REPORTS
#  options("ISAnalytics.reports" = FALSE)
#  
#  # ENABLE HTML REPORTS
#  options("ISAnalytics.reports" = TRUE)

## ---------------------------------------------------------------------------------------------------------------------
inspect_tags("chromosome")

## ----echo=FALSE-------------------------------------------------------------------------------------------------------
all_tags <- available_tags()
all_tags[, needed_in := purrr::map_chr(needed_in,
                                       ~ paste0(.x, collapse = ", "))]
mand_tags <- all_tags[eval(rlang::sym("dyn_vars_tbl")) == "mand_vars", 
                      mget(c("tag", "needed_in", "description"))]
annot_tags <- all_tags[eval(rlang::sym("dyn_vars_tbl")) == "annot_vars", 
                      mget(c("tag", "needed_in", "description"))]
af_tags <- all_tags[eval(rlang::sym("dyn_vars_tbl")) == "af_vars", 
                      mget(c("tag", "needed_in", "description"))]
iss_tags <- all_tags[eval(rlang::sym("dyn_vars_tbl")) == "iss_vars", 
                      mget(c("tag", "needed_in", "description"))]

## ----echo=FALSE-------------------------------------------------------------------------------------------------------
datatable(
  mand_tags,
  class = "stripe",
  options = list(dom = 't'), 
  rownames = FALSE
)

## ----echo=FALSE-------------------------------------------------------------------------------------------------------
datatable(
  annot_tags,
  class = "stripe",
  options = list(dom = 't'), 
  rownames = FALSE
)

## ----echo=FALSE-------------------------------------------------------------------------------------------------------
datatable(
  af_tags,
  class = "stripe",
  options = list(dom = 't'), 
  rownames = FALSE
)

## ----echo=FALSE-------------------------------------------------------------------------------------------------------
datatable(
  iss_tags,
  class = "stripe",
  options = list(dom = 't'), 
  rownames = FALSE
)

## ---------------------------------------------------------------------------------------------------------------------
mandatory_IS_vars(TRUE)

## ---------------------------------------------------------------------------------------------------------------------
new_mand_vars <- tibble::tribble(
  ~names, ~types, ~transform, ~flag, ~tag,
  "chrom", "char", ~stringr::str_replace_all(.x, "chr", ""), "required",
  "chromosome",
  "position", "int", NULL, "required", "locus",
  "strand", "char", NULL, "required", "is_strand",
  "gap", "int", NULL, "required", NA_character_,
  "junction", "int", NULL, "required", NA_character_
)

## ---------------------------------------------------------------------------------------------------------------------
set_mandatory_IS_vars(new_mand_vars)
mandatory_IS_vars(TRUE)

## ---------------------------------------------------------------------------------------------------------------------
new_mand_vars[1, ]$tag <- NA_character_
set_mandatory_IS_vars(new_mand_vars)
mandatory_IS_vars(TRUE)

## ---------------------------------------------------------------------------------------------------------------------
reset_mandatory_IS_vars()
mandatory_IS_vars(TRUE)

## ---------------------------------------------------------------------------------------------------------------------
matrix_file_suffixes()

## ----echo=FALSE-------------------------------------------------------------------------------------------------------
sample_sparse_matrix <- tibble::tribble(
  ~ chr, ~ integration_locus, ~ strand, ~ GeneName, ~GeneStrand, 
  ~ exp1, ~ exp2, ~ exp3,
  "1", 12324, "+", "NFATC3", "+", 4553,5345,NA_integer_,
  "6", 657532, "+", "LOC100507487", "+", 76,545,5,
  "7", 657532, "+", "EDIL3", "-", NA_integer_,56,NA_integer_,
)
print(sample_sparse_matrix, width = Inf)

## ---------------------------------------------------------------------------------------------------------------------
fs_path <- generate_default_folder_structure()
withr::with_options(list(ISAnalytics.reports = FALSE), code = {
  af <- import_association_file(fs_path$af, root = fs_path$root)
})

## ----echo=FALSE-------------------------------------------------------------------------------------------------------
print(head(af))

## ----results='hide'---------------------------------------------------------------------------------------------------
withr::with_options(list(ISAnalytics.reports = FALSE), {
    vispa_stats <- import_Vispa2_stats(
        association_file = af,
        join_with_af = FALSE
    )
})

## ----echo=FALSE-------------------------------------------------------------------------------------------------------
print(head(vispa_stats))

## ----message=FALSE, results='hide'------------------------------------------------------------------------------------
matrix_path <- fs::path(fs_path$root,
                        "PJ01",
                        "quantification",
                        "POOL01-1",
                        "PJ01_POOL01-1_seqCount_matrix.no0.annotated.tsv.gz")
matrix <- import_single_Vispa2Matrix(matrix_path)

## ----echo=FALSE-------------------------------------------------------------------------------------------------------
matrix

## ---------------------------------------------------------------------------------------------------------------------
withr::with_options(list(ISAnalytics.reports = FALSE), {
    matrices <- import_parallel_Vispa2Matrices(af,
        c("seqCount", "fragmentEstimate"),
        mode = "AUTO"
    )
})

## ----echo=FALSE-------------------------------------------------------------------------------------------------------
print(head(matrices))

## ----echo=FALSE-------------------------------------------------------------------------------------------------------
library(ISAnalytics)
ex_coll <- tibble::tribble(
  ~ chr, ~ integration_locus, ~ strand, ~ seqCount, ~ CompleteAmplificationID,
  ~ SubjectID, ~ ProjectID, 
  "1", 123454,  "+", 653, "SAMPLE1", "SUBJ01", "PJ01",
  "1", 123454, "+", 456, "SAMPLE2", "SUBJ02", "PJ01"
)
knitr::kable(ex_coll, caption = paste("Example of collisions: the same",
                                      "integration (1, 123454, +) is found",
                                      "in 2 different independent samples",
                                      "((SUBJ01, PJ01) & (SUBJ02, PJ01))"))

## ---------------------------------------------------------------------------------------------------------------------
data("integration_matrices", package = "ISAnalytics")
data("association_file", package = "ISAnalytics")
## Multi quantification matrix
no_coll <- remove_collisions(x = integration_matrices,
                             association_file = association_file,
                             report_path = NULL)
## Matrix list
separated <- separate_quant_matrices(integration_matrices)
no_coll_list <- remove_collisions(x = separated,
                             association_file = association_file,
                             report_path = NULL)
## Only sequence count
no_coll_single <- remove_collisions(x = separated$seqCount,
                             association_file = association_file,
                             quant_cols = c(seqCount = "Value"),
                             report_path = NULL)

## ----realign----------------------------------------------------------------------------------------------------------
other_realigned <- realign_after_collisions(
  sc_matrix = no_coll_single,
  other_matrices = list(fragmentEstimate = separated$fragmentEstimate)
)

## ----echo=FALSE-------------------------------------------------------------------------------------------------------
library(ISAnalytics)
print(default_meta_agg(), width = Inf)

## ---------------------------------------------------------------------------------------------------------------------
data("association_file", package = "ISAnalytics")
aggregated_meta <- aggregate_metadata(association_file = association_file)

## ----echo=FALSE-------------------------------------------------------------------------------------------------------
print(aggregated_meta)

## ---------------------------------------------------------------------------------------------------------------------
data("integration_matrices", package = "ISAnalytics")
data("association_file", package = "ISAnalytics")
aggreg <- aggregate_values_by_key(
  x = integration_matrices, 
  association_file = association_file,
  value_cols = c("seqCount", "fragmentEstimate")
)

## ----echo=FALSE-------------------------------------------------------------------------------------------------------
print(aggreg, width = Inf)

## ---------------------------------------------------------------------------------------------------------------------
agg1 <- aggregate_values_by_key(
    x = integration_matrices,
    association_file = association_file,
    key = c("SubjectID", "ProjectID"),
    value_cols = c("seqCount", "fragmentEstimate")
)

## ----echo=FALSE-------------------------------------------------------------------------------------------------------
print(agg1, width = Inf)

## ---------------------------------------------------------------------------------------------------------------------
agg2 <- aggregate_values_by_key(
    x = integration_matrices,
    association_file = association_file,
    key = "SubjectID",
    lambda = list(mean = ~ mean(.x, na.rm = TRUE)),
    value_cols = c("seqCount", "fragmentEstimate")
)

## ----echo=FALSE-------------------------------------------------------------------------------------------------------
print(agg2, width = Inf)

## ---------------------------------------------------------------------------------------------------------------------
agg3 <- aggregate_values_by_key(
    x = integration_matrices,
    association_file = association_file,
    key = "SubjectID",
    lambda = list(describe = ~ list(psych::describe(.x))),
    value_cols = c("seqCount", "fragmentEstimate")
)

## ----echo=FALSE-------------------------------------------------------------------------------------------------------
print(agg3, width = Inf)

## ---------------------------------------------------------------------------------------------------------------------
agg4 <- aggregate_values_by_key(
    x = integration_matrices,
    association_file = association_file,
    key = "SubjectID",
    lambda = list(sum = sum, mean = mean),
    value_cols = c("seqCount", "fragmentEstimate")
)

## ----echo=FALSE-------------------------------------------------------------------------------------------------------
print(agg4, width = Inf)

## ---------------------------------------------------------------------------------------------------------------------
agg5 <- aggregate_values_by_key(
    x = integration_matrices,
    association_file = association_file,
    key = "SubjectID",
    lambda = list(sum = sum, mean = mean),
    group = c(mandatory_IS_vars()),
    value_cols = c("seqCount", "fragmentEstimate")
)

## ----echo=FALSE-------------------------------------------------------------------------------------------------------
print(agg5, width = Inf)

## ----reproduce3, echo=FALSE-------------------------------------------------------------------------------------------
## Session info
library("sessioninfo")
options(width = 120)
session_info()

## ----vignetteBiblio, results = "asis", echo = FALSE, warning = FALSE, message = FALSE---------------------------------
## Print bibliography
PrintBibliography(bib, .opts = list(hyperlink = "to.doc", style = "html"))

