## ---- echo=FALSE, results="hide", message=FALSE-------------------------------
knitr::opts_chunk$set(error=FALSE, message=FALSE, warning=FALSE)
library(BiocStyle)

## -----------------------------------------------------------------------------
suppressPackageStartupMessages(library(escape))
suppressPackageStartupMessages(library(dittoSeq))
suppressPackageStartupMessages(library(SingleCellExperiment))
suppressPackageStartupMessages(library(Seurat))
suppressPackageStartupMessages(library(SeuratObject))
pbmc_small <- get("pbmc_small")
data("pbmc_small")
pbmc_small <- suppressMessages(UpdateSeuratObject(pbmc_small))
sce <- as.SingleCellExperiment(pbmc_small)

## -----------------------------------------------------------------------------
## We'll use the HALLMARK gene set collection ("H")
GS.hallmark <- getGeneSets(library = "H")

## -----------------------------------------------------------------------------
ES.seurat <- enrichIt(obj = pbmc_small, gene.sets = GS.hallmark, groups = 1000, cores = 2)
ES.sce <- enrichIt(obj = sce, gene.sets = GS.hallmark, groups = 1000, cores = 2)

## -----------------------------------------------------------------------------
## if working with a Seurat object
pbmc_small <- Seurat::AddMetaData(pbmc_small, ES.seurat)

## if working with an SCE object
met.data <- merge(colData(sce), ES.sce, by = "row.names", all=TRUE)
row.names(met.data) <- met.data$Row.names
met.data$Row.names <- NULL
colData(sce) <- met.data

## -----------------------------------------------------------------------------
colors <- colorRampPalette(c("#0348A6", "#7AC5FF", "#C6FDEC", "#FFB433", "#FF4B20"))

## -----------------------------------------------------------------------------
dittoHeatmap(pbmc_small, genes = NULL, metas = names(ES.seurat), 
             annot.by = "groups", 
             fontsize = 7, 
             cluster_cols = TRUE,
             heatmap.colors = colors(50))

## -----------------------------------------------------------------------------
dittoHeatmap(sce, genes = NULL, 
             metas = c("HALLMARK_APOPTOSIS", "HALLMARK_DNA_REPAIR", "HALLMARK_P53_PATHWAY"), 
             annot.by = "groups", 
             fontsize = 7,
             heatmap.colors = colors(50))

## -----------------------------------------------------------------------------
multi_dittoPlot(sce, vars = c("HALLMARK_APOPTOSIS", "HALLMARK_DNA_REPAIR", "HALLMARK_P53_PATHWAY"), 
                group.by = "groups", plots = c("jitter", "vlnplot", "boxplot"), 
                ylab = "Enrichment Scores", 
                theme = theme_classic() + theme(plot.title = element_text(size = 10)))

## -----------------------------------------------------------------------------
dittoScatterHex(sce, x.var = "HALLMARK_DNA_REPAIR", 
                    y.var = "HALLMARK_MTORC1_SIGNALING", 
                    do.contour = TRUE) + 
        scale_fill_gradientn(colors = colors(11)) 

## -----------------------------------------------------------------------------
dittoScatterHex(sce, x.var = "HALLMARK_DNA_REPAIR", 
                y.var = "HALLMARK_MTORC1_SIGNALING", 
                do.contour = TRUE,
                split.by = "groups")  + 
        scale_fill_gradientn(colors = colors(11)) 

## -----------------------------------------------------------------------------
## Seurat object example
ES2 <- data.frame(pbmc_small[[]], Idents(pbmc_small))
colnames(ES2)[ncol(ES2)] <- "cluster"

## plot
ridgeEnrichment(ES2, gene.set = "HALLMARK_DNA_REPAIR", group = "cluster", add.rug = TRUE)

## -----------------------------------------------------------------------------
ridgeEnrichment(ES2, gene.set = "HALLMARK_DNA_REPAIR", group = "cluster", 
                facet = "letter.idents", add.rug = TRUE)

## -----------------------------------------------------------------------------
splitEnrichment(ES2, split = "groups", gene.set = "HALLMARK_DNA_REPAIR")
splitEnrichment(ES2, x.axis = "cluster", split = "groups", gene.set = "HALLMARK_DNA_REPAIR")

## -----------------------------------------------------------------------------
PCA <- performPCA(enriched = ES2, groups = c("groups", "cluster"))

## -----------------------------------------------------------------------------
pcaEnrichment(PCA, PCx = "PC1", PCy = "PC2", contours = TRUE)

## -----------------------------------------------------------------------------
pcaEnrichment(PCA, PCx = "PC1", PCy = "PC2", contours = FALSE, facet = "cluster") 

## -----------------------------------------------------------------------------
masterPCAPlot(ES2, PCx = "PC1", PCy = "PC2", top.contribution = 10)

## -----------------------------------------------------------------------------
output <- getSignificance(ES2, group = "cluster", fit = "linear.model")

## -----------------------------------------------------------------------------
sessionInfo()

