## ----setup, include = FALSE---------------------------------------------------
library(RiboDiPA)
knitr::opts_chunk$set(
    collapse = TRUE,
    comment = "#>"
)

## ---- warning=FALSE, message=FALSE, eval=FALSE--------------------------------
#  if(!requireNamespace("BiocManager", quietly = TRUE))
#      install.packages("BiocManager")
#  BiocManager::install("RiboDiPA")

## ---- warning=FALSE, message=FALSE--------------------------------------------
## Download sample files from GitHub
library(BiocFileCache)
file_names <- c("WT1.bam", "WT2.bam", "MUT1.bam", "MUT2.bam", "eg.gtf")
url <- "https://github.com/jipingw/RiboDiPA-data/raw/master/"
bfc <- BiocFileCache()
bam_path <- bfcrpath(bfc,paste0(url,file_names))

## ---- warning=FALSE, message=FALSE--------------------------------------------
## Make the class label for the experiment
classlabel <- data.frame(
    condition = c("mutant", "mutant", "wildtype", "wildtype"),
    comparison = c(2, 2, 1, 1)
)
rownames(classlabel) <- c("mutant1","mutant2","wildtype1","wildtype2")

## ---- warning=FALSE, message=FALSE--------------------------------------------
## Run the RiboDiPA pipeline with default parameters
result.pst <- RiboDiPA(bam_path[1:4], bam_path[5], classlabel, cores = 2)

## -----------------------------------------------------------------------------
## View the table of output from RiboDiPA
head(result.pst$gene[order(result.pst$gene$qvalue), ])

## ---- warning=FALSE, message=FALSE--------------------------------------------
## Perform individual P-site mapping procedure
data.psite <- psiteMapping(bam_file_list = bam_path[1:4], 
    bam_path[5], psite.mapping = "auto", cores = 2)

## -----------------------------------------------------------------------------
## P-site mapping offset rule generated
data.psite$psite.mapping

## ---- warning=FALSE, message=FALSE, eval=FALSE--------------------------------
#  ## Use user specified psite mapping offset rule
#  offsets <- cbind(qwidth = c(28, 29, 30, 31, 32),
#      psite = c(18, 18, 18, 19, 19))
#  data.psite2 <- psiteMapping(bam_path[1:4], bam_path[5],
#      psite.mapping = offsets, cores = 2)

## ---- warning=FALSE, message=FALSE--------------------------------------------
## Merge the P-site data into bins with a fixed or an adaptive width
data.binned <- dataBinning(data = data.psite$coverage, bin.width = 0, 
    zero.omit = FALSE, bin.from.5UTR = TRUE, cores = 2)

## ---- warning=FALSE, message=FALSE--------------------------------------------
## Merge the P-site data on each codon
data.codon <- dataBinning(data = data.psite$coverage, bin.width = 1, 
    zero.omit = FALSE, bin.from.5UTR = TRUE, cores = 2)

## ---- warning=FALSE, message=FALSE--------------------------------------------
## Merge the P-site data on each exon and perform differential pattern analysis
result.exon <- diffPatternTestExon(psitemap = data.psite, 
    classlabel = classlabel, method = c('gtxr', 'qvalue'))

## ---- warning=FALSE, message=FALSE--------------------------------------------
## Perform differential pattern analysis
result.pst <- diffPatternTest(data = data.binned, 
    classlabel = classlabel, method=c('gtxr', 'qvalue'))

## ----fig2, fig.height=6, fig.width=4.5, fig.align="center", results='hide'----
## Plot ribosome per nucleotide tracks of specified genes.
plotTrack(data = data.psite, genes.list = c("YDR050C", "YDR064W"),
    replicates = NULL, exons = FALSE)

## ----fig3, fig.height = 9, fig.width = 6, fig.align = "center",results='hide'----
## Plot binned ribosome tracks of siginificant genes: YDR086C and YDR210W.
## you can specify the thrshold to redefine the significant level
plotTest(result = result.pst, genes.list = NULL, threshold = 0.05) 

## ----sessionInfo--------------------------------------------------------------
sessionInfo()

