## ----style, echo = FALSE, results = 'asis'----------------------------------------------------------------------------
BiocStyle::markdown()

## ----setup, include = FALSE-------------------------------------------------------------------------------------------
knitr::opts_chunk$set(
    collapse = TRUE,
    comment = "#>",
    crop = NULL
    ## Related to
    ## https://stat.ethz.ch/pipermail/bioc-devel/2020-April/016656.html
)

## ----vignetteSetup, echo=FALSE, message=FALSE, warning = FALSE--------------------------------------------------------
## Track time spent on making the vignette
startTime <- Sys.time()

## Bib setup
library("knitcitations")

## Load knitcitations with a clean bibliography
cleanbib()
cite_options(hyperlink = "to.doc", citation_format = "text", style = "html")

## Write bibliography information
bib <- c(
    R = citation(),
    BiocStyle = citation("BiocStyle")[1],
    knitcitations = citation("knitcitations")[1],
    knitr = citation("knitr")[1],
    rmarkdown = citation("rmarkdown")[1],
    sessioninfo = citation("sessioninfo")[1],
    testthat = citation("testthat")[1],
    ISAnalytics = citation("ISAnalytics")[1]
)

write.bibtex(bib, file = "how_to_import_functions.bib")

## ----installBioc, eval=FALSE------------------------------------------------------------------------------------------
#  ## For release version
#  if (!requireNamespace("BiocManager", quietly = TRUE)) {
#        install.packages("BiocManager")
#    }
#  BiocManager::install("ISAnalytics")
#  
#  ## For devel version
#  if (!requireNamespace("BiocManager", quietly = TRUE)) {
#        install.packages("BiocManager")
#    }
#  # The following initializes usage of Bioc devel
#  BiocManager::install(version = "devel")
#  BiocManager::install("ISAnalytics")

## ----installGitHub, eval=FALSE----------------------------------------------------------------------------------------
#  # For release version
#  if (!require(devtools)) {
#      install.packages("devtools")
#  }
#  devtools::install_github("calabrialab/ISAnalytics",
#      ref = "RELEASE_3_12",
#      dependencies = TRUE,
#      build_vignettes = TRUE
#  )
#  
#  ## Safer option for vignette building issue
#  devtools::install_github("calabrialab/ISAnalytics",
#      ref = "RELEASE_3_12"
#  )
#  
#  # For devel version
#  if (!require(devtools)) {
#      install.packages("devtools")
#  }
#  devtools::install_github("calabrialab/ISAnalytics",
#      ref = "master",
#      dependencies = TRUE,
#      build_vignettes = TRUE
#  )
#  
#  ## Safer option for vignette building issue
#  devtools::install_github("calabrialab/ISAnalytics",
#      ref = "master"
#  )

## ---------------------------------------------------------------------------------------------------------------------
library(ISAnalytics)

## ----OptVerbose, eval=FALSE-------------------------------------------------------------------------------------------
#  # DISABLE
#  options("ISAnalytics.verbose" = FALSE)
#  
#  # ENABLE
#  options("ISAnalytics.verbose" = TRUE)

## ----OptWidg, eval=FALSE----------------------------------------------------------------------------------------------
#  # DISABLE HTML REPORTS
#  options("ISAnalytics.widgets" = FALSE)
#  
#  # ENABLE HTML REPORTS
#  options("ISAnalytics.widgets" = TRUE)

## ---------------------------------------------------------------------------------------------------------------------
root_correct <- system.file("extdata", "fs.zip", package = "ISAnalytics")
root_correct <- unzip_file_system(root_correct, "fs")
fs::dir_tree(root_correct)

## ---- echo = FALSE----------------------------------------------------------------------------------------------------
example_matrix_path <- system.file("extdata", "ex_annotated_ISMatrix.tsv.xz",
    package = "ISAnalytics"
)
example_matrix <- read.csv(example_matrix_path,
    sep = "\t", header = TRUE,
    stringsAsFactors = FALSE, check.names = FALSE
)
knitr::kable(head(example_matrix),
    caption = "A simple example of messy matrix.", align = "l"
)

## ---------------------------------------------------------------------------------------------------------------------
example_matrix_path <- system.file("extdata", "ex_annotated_ISMatrix.tsv.xz",
    package = "ISAnalytics"
)
imported_im <- import_single_Vispa2Matrix(
    path = example_matrix_path,
    to_exclude = NULL,
    separator = "\t"
)

## ---- echo = FALSE----------------------------------------------------------------------------------------------------
knitr::kable(head(imported_im), caption = "Example of tidy integration matrix")

## ---------------------------------------------------------------------------------------------------------------------
# Displays the mandatory vars, can be called also for manipulation purposes
# on tibble instead of calling individual variables
mandatory_IS_vars()

# Displays the annotation variables
annotation_IS_vars()

## ---------------------------------------------------------------------------------------------------------------------
path_as_file <- system.file("extdata", "ex_association_file.tsv",
    package = "ISAnalytics"
)
withr::with_options(list(ISAnalytics.widgets = FALSE), {
    association_file <- import_association_file(
        path = path_as_file,
        root = root_correct,
        tp_padding = 4,
        dates_format = "dmy",
        separator = "\t"
    )
})
association_file

## ---------------------------------------------------------------------------------------------------------------------
withr::with_options(list(ISAnalytics.widgets = FALSE), {
    association_file_filtered <- import_association_file(
        path = path_as_file,
        root = root_correct,
        tp_padding = 4,
        dates_format = "dmy",
        separator = "\t",
        filter_for = list(ProjectID = "CLOEXP")
    )
})
association_file_filtered

## ---------------------------------------------------------------------------------------------------------------------
withr::with_options(list(ISAnalytics.widgets = FALSE), {
  vispa_stats <- import_Vispa2_stats(
  association_file = association_file_filtered, 
  join_with_af = FALSE
)
})

## ---- eval = FALSE----------------------------------------------------------------------------------------------------
#  withr::with_options(list(ISAnalytics.widgets = FALSE), {
#      matrices <- import_parallel_Vispa2Matrices_interactive(
#          association_file = path_as_file,
#          root = root_correct,
#          quantification_type = c("fragmentEstimate", "seqCount"),
#          matrix_type = "annotated",
#          workers = 2
#      )
#  })

## ---- eval = FALSE----------------------------------------------------------------------------------------------------
#  matrices <- import_parallel_Vispa2Matrices_interactive(
#      association_file = association_file,
#      root = NULL,
#      quantification_type = c("fragmentEstimate", "seqCount"),
#      matrix_type = "annotated",
#      workers = 2
#  )

## ---- eval=FALSE------------------------------------------------------------------------------------------------------
#  matrices$fragmentEstimate
#  matrices$seqCount

## ---------------------------------------------------------------------------------------------------------------------
library(magrittr)
refined_af <- association_file %>% dplyr::filter(.data$ProjectID == "CLOEXP")

## ---------------------------------------------------------------------------------------------------------------------
withr::with_options(list(ISAnalytics.widgets = FALSE), {
    matrices_auto <- import_parallel_Vispa2Matrices_auto(
        association_file = refined_af,
        root = NULL,
        quantification_type = c("fragmentEstimate", "seqCount"),
        matrix_type = "annotated",
        workers = 2,
        patterns = NULL,
        matching_opt = "ANY" # Same if you choose "ALL" or "OPTIONAL"
    )
})
matrices_auto

## ---------------------------------------------------------------------------------------------------------------------
root_err <- system.file("extdata", "fserr.zip", package = "ISAnalytics")
root_err <- unzip_file_system(root_err, "fserr")
fs::dir_tree(root_err)
withr::with_options(list(ISAnalytics.widgets = FALSE), {
    association_file_fserr <- import_association_file(path_as_file, root_err)
    refined_af_err <- association_file_fserr %>%
        dplyr::filter(.data$ProjectID == "CLOEXP")
    matrices_auto2 <- import_parallel_Vispa2Matrices_auto(
        association_file = refined_af_err,
        root = NULL,
        quantification_type = c("fragmentEstimate", "seqCount"),
        matrix_type = "annotated",
        workers = 2,
        patterns = "NoMate",
        matching_opt = "ANY" # Same if you choose "ALL" or "OPTIONAL"
    )
})
matrices_auto

## ----reproduce3, echo=FALSE-------------------------------------------------------------------------------------------
## Session info
library("sessioninfo")
options(width = 120)
session_info()

## ----results = "asis", echo = FALSE, warning = FALSE, message = FALSE-------------------------------------------------
## Print bibliography
bibliography()

