## ----knitr-options, include = FALSE-------------------------------------------
knitr::opts_chunk$set(collapse = TRUE, comment = "#>")

## ----setup--------------------------------------------------------------------
suppressPackageStartupMessages({
  library("splatter")
  library("scater")
  library("VariantAnnotation")
  library("ggplot2")
})

## ----quick-start--------------------------------------------------------------
vcf <- mockVCF(n.samples = 6)

sim <- splatPopSimulate(vcf = vcf, "nGenes" = 100)

sim <- logNormCounts(sim)
sim <- runPCA(sim, ncomponents = 10)
plotPCA(sim, colour_by = "Sample")

## ----default-SplatPopParams---------------------------------------------------
params <- newSplatPopParams()
params

## ----eqtlEstimate-------------------------------------------------------------
bulk.means <- mockBulkMatrix(n.genes=100, n.samples=100)
bulk.eqtl <- mockBulkeQTL(n.genes=100)
counts <- mockSCE()

params.est <- splatPopEstimate(means = bulk.means,
                               eqtl = bulk.eqtl,
                               counts = counts)
params.est

## ----splatPopSimulateMeans----------------------------------------------------
vcf <- mockVCF(n.samples = 6)
gff <- mockGFF(n.genes = 100)

sim.means <- splatPopSimulateMeans(vcf = vcf, gff = gff,
                                   params = newSplatPopParams())

round(sim.means$means[1:5, 1:6], digits = 2)

print(sim.means$key[1:5, ], digits = 2)

## ----splatPopSimulateMeans-from-key-------------------------------------------
sim.means.rep2 <- splatPopSimulateMeans(vcf = vcf, key=sim.means$key,
                                        params = newSplatPopParams())

round(sim.means.rep2$means[1:5, 1:6], digits = 2)

## ----quant-normalize-population-data------------------------------------------
bulk.qnorm <- splatPopQuantNorm(newSplatPopParams(), bulk.means)
round(bulk.qnorm[1:5, 1:5], 3)

## ----eqtl-splatPopSimulateSC-simple-object------------------------------------
sim.sc <- splatPopSimulateSC(params=params, 
                             key = sim.means$key,
                             sim.means=sim.means$means, 
                             batchCells=50)
sim.sc

## ----eqtl-splatPopSimulateSC-simple-plots-------------------------------------
sim.sc <- logNormCounts(sim.sc)
sim.sc <- runPCA(sim.sc, ncomponents = 10)
plotPCA(sim.sc, colour_by = "Sample")

## ----group-specific-eQTL-simulations------------------------------------------
params.group <- newSplatPopParams(nGenes = 50,
                                  batchCells = 40,
                                  group.prob = c(0.5, 0.5))

sim.sc.gr2 <- splatPopSimulate(vcf = vcf, params = params.group)

sim.sc.gr2 <- logNormCounts(sim.sc.gr2)
sim.sc.gr2 <- runPCA(sim.sc.gr2, ncomponents = 10)
plotPCA(sim.sc.gr2, colour_by = "Group", shape_by = "Sample")

## ----group-specific-eQTL-simulations-bigger-----------------------------------
params.group <- newSplatPopParams(batchCells = 40,
                                  nGenes = 50,
                                  similarity.scale = 6,
                                  eqtl.group.specific = 0.6,
                                  de.prob = 0.5,
                                  de.facLoc = 0.5, 
                                  de.facScale = 0.4,
                                  group.prob = c(0.5, 0.5))

sim.sc.gr2 <- splatPopSimulate(vcf = vcf, params = params.group)

sim.sc.gr2 <- logNormCounts(sim.sc.gr2)
sim.sc.gr2 <- runPCA(sim.sc.gr2, ncomponents = 10)
plotPCA(sim.sc.gr2, colour_by = "Group", shape_by = "Sample")

## ----simulate-population-with-batch-effects-----------------------------------
params.batches <- newSplatPopParams(batchCells = c(20, 20),
                                    nGenes = 50,
                                    similarity.scale = 5,
                                    batch.facLoc = 0.3,
                                    batch.facScale = 0.3)

sim.pop.batches <- splatPopSimulate(vcf = vcf, params = params.batches)
sim.pop.batches <- logNormCounts(sim.pop.batches)
sim.pop.batches <- runPCA(sim.pop.batches, ncomponents = 10)
plotPCA(sim.pop.batches, colour_by = "Batch", shape_by = "Sample",
        ncomponents = 5:6)


## ----simulate-population-with-path-effects------------------------------------
params.paths <- newSplatPopParams(batchCells = 40,
                                  nGenes = 50,
                                  similarity.scale = 6,
                                  de.facLoc = 0.5,
                                  de.facScale = 0.5,
                                  de.prob = 0.5)

sim.pop.paths <- splatPopSimulate(vcf = vcf, params = params.paths,
                                  method = "paths")
sim.pop.paths <- logNormCounts(sim.pop.paths)
sim.pop.paths <- runPCA(sim.pop.paths, ncomponents = 10)
plotPCA(sim.pop.paths, colour_by = "Step", shape_by = "Sample", 
        ncomponents = 5:6)

