## ---- include = FALSE---------------------------------------------------------
knitr::opts_chunk$set(
    collapse = TRUE,
    comment = "#>"
)
library(ggplot2)
theme_set(theme_classic())

## ----load-libraries, message=FALSE, warning=FALSE-----------------------------
library(schex)
library(dplyr)
library(scater)
library(Seurat)
library(TENxPBMCData)

## ----load, eval=TRUE----------------------------------------------------------
tenx_pbmc3k <- TENxPBMCData(dataset = "pbmc3k")

rownames(tenx_pbmc3k) <- uniquifyFeatureNames(rowData(tenx_pbmc3k)$ENSEMBL_ID, 
    rowData(tenx_pbmc3k)$Symbol_TENx)

pbmc <- as.Seurat(tenx_pbmc3k, data = NULL)

## ----norm, message=FALSE, warning=FALSE---------------------------------------
pbmc <- NormalizeData(pbmc, normalization.method = "LogNormalize", 
    scale.factor = 10000, verbose=FALSE)

## ----highly-variable, message=FALSE, warning=FALSE----------------------------
pbmc <- FindVariableFeatures(pbmc, selection.method = "vst", nfeatures = 2000,
    verbose = FALSE)

## ----scaling, message=FALSE, warning=FALSE------------------------------------
all.genes <- rownames(pbmc)
pbmc <- ScaleData(pbmc, features = all.genes, verbose = FALSE)

## ----pca, message=FALSE, warning=FALSE----------------------------------------
pbmc <- RunPCA(pbmc, features = VariableFeatures(object = pbmc), 
      verbose = FALSE)

## ----umap, message=FALSE, warning=FALSE---------------------------------------
set.seed(10)
pbmc <- RunUMAP(pbmc, dims = 1:10, verbose=FALSE)

## ----calc-hexbin--------------------------------------------------------------
pbmc <- make_hexbin(pbmc, nbins = 40, 
    dimension_reduction = "UMAP")

## ----plot-density, fig.height=7, fig.width=7----------------------------------
plot_hexbin_density(pbmc)

## ----plot-meta-1, fig.height=7, fig.width=7-----------------------------------
pbmc$nCount_RNA <- colSums(GetAssayData(pbmc, assay="RNA", "data"))
plot_hexbin_meta(pbmc, col="nCount_RNA", action="median")

## ----plot-gene, fig.height=7, fig.width=7-------------------------------------
gene_id <-"CD19"
schex::plot_hexbin_feature(pbmc, type="scale.data", feature=gene_id, 
    action="mean", xlab="UMAP1", ylab="UMAP2", 
    title=paste0("Mean of ", gene_id))

## -----------------------------------------------------------------------------
gene_id <-"CD19"
gg <- schex::plot_hexbin_feature(pbmc, type="scale.data", feature=gene_id, 
    action="mean", xlab="UMAP1", ylab="UMAP2", 
    title=paste0("Mean of ", gene_id))
gg + theme_void()

## ---- eval=FALSE--------------------------------------------------------------
#  ggsave(gg, file="schex_plot.pdf")

