/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.net;

import com.google.common.annotations.Beta;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.io.ByteStreams;
import com.google.common.primitives.Ints;
import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.ByteBuffer;
import java.util.Arrays;
import javax.annotation.Nullable;

@Beta
public final class InetAddresses {
    private static final int IPV4_PART_COUNT = 4;
    private static final int IPV6_PART_COUNT = 8;
    private static final Inet4Address LOOPBACK4 = (Inet4Address)InetAddresses.forString("127.0.0.1");
    private static final Inet4Address ANY4 = (Inet4Address)InetAddresses.forString("0.0.0.0");

    private InetAddresses() {
    }

    private static Inet4Address getInet4Address(byte[] bytes) {
        Preconditions.checkArgument(bytes.length == 4, "Byte array has invalid length for an IPv4 address: %s != 4.", bytes.length);
        try {
            InetAddress ipv4 = InetAddress.getByAddress(bytes);
            if (!(ipv4 instanceof Inet4Address)) {
                throw new UnknownHostException(String.format("'%s' is not an IPv4 address.", ipv4.getHostAddress()));
            }
            return (Inet4Address)ipv4;
        }
        catch (UnknownHostException e2) {
            throw new IllegalArgumentException(String.format("Host address '%s' is not a valid IPv4 address.", Arrays.toString(bytes)), e2);
        }
    }

    public static InetAddress forString(String ipString) {
        byte[] addr = InetAddresses.textToNumericFormatV4(ipString);
        if (addr == null) {
            addr = InetAddresses.textToNumericFormatV6(ipString);
        }
        if (addr == null) {
            throw new IllegalArgumentException(String.format("'%s' is not an IP string literal.", ipString));
        }
        try {
            return InetAddress.getByAddress(addr);
        }
        catch (UnknownHostException e2) {
            throw new IllegalArgumentException(String.format("'%s' is extremely broken.", ipString), e2);
        }
    }

    public static boolean isInetAddress(String ipString) {
        try {
            InetAddresses.forString(ipString);
            return true;
        }
        catch (IllegalArgumentException e2) {
            return false;
        }
    }

    private static byte[] textToNumericFormatV4(String ipString) {
        if (ipString.contains(":")) {
            return null;
        }
        String[] address = ipString.split("\\.");
        if (address.length != 4) {
            return null;
        }
        byte[] bytes = new byte[4];
        try {
            for (int i = 0; i < bytes.length; ++i) {
                int piece = Integer.parseInt(address[i]);
                if (piece < 0 || piece > 255) {
                    return null;
                }
                if (address[i].startsWith("0") && address[i].length() != 1) {
                    return null;
                }
                bytes[i] = (byte)piece;
            }
        }
        catch (NumberFormatException ex) {
            return null;
        }
        return bytes;
    }

    private static byte[] textToNumericFormatV6(String ipString) {
        int totalParts;
        int i;
        String[] parts;
        if (!ipString.contains(":")) {
            return null;
        }
        if (ipString.contains(":::")) {
            return null;
        }
        if (ipString.contains(".") && (ipString = InetAddresses.convertDottedQuadToHex(ipString)) == null) {
            return null;
        }
        ByteBuffer rawBytes = ByteBuffer.allocate(16);
        int partsHi = 0;
        int partsLo = 0;
        String[] addressHalves = ipString.split("::", 2);
        if (!addressHalves[0].equals("")) {
            parts = addressHalves[0].split(":", 8);
            try {
                for (i = 0; i < parts.length; ++i) {
                    if (parts[i].equals("")) {
                        return null;
                    }
                    int piece = Integer.parseInt(parts[i], 16);
                    rawBytes.putShort(2 * i, (short)piece);
                }
                partsHi = parts.length;
            }
            catch (NumberFormatException ex) {
                return null;
            }
        } else {
            partsHi = 1;
        }
        if (addressHalves.length > 1) {
            if (!addressHalves[1].equals("")) {
                parts = addressHalves[1].split(":", 8);
                try {
                    for (i = 0; i < parts.length; ++i) {
                        int partsIndex = parts.length - i - 1;
                        if (parts[partsIndex].equals("")) {
                            return null;
                        }
                        int piece = Integer.parseInt(parts[partsIndex], 16);
                        int bytesIndex = 2 * (8 - i - 1);
                        rawBytes.putShort(bytesIndex, (short)piece);
                    }
                    partsLo = parts.length;
                }
                catch (NumberFormatException ex) {
                    return null;
                }
            } else {
                partsLo = 1;
            }
        }
        if ((totalParts = partsHi + partsLo) > 8) {
            return null;
        }
        if (addressHalves.length == 1 && totalParts != 8) {
            return null;
        }
        return rawBytes.array();
    }

    private static String convertDottedQuadToHex(String ipString) {
        int lastColon = ipString.lastIndexOf(58);
        String initialPart = ipString.substring(0, lastColon + 1);
        String dottedQuad = ipString.substring(lastColon + 1);
        byte[] quad = InetAddresses.textToNumericFormatV4(dottedQuad);
        if (quad == null) {
            return null;
        }
        String penultimate = Integer.toHexString((quad[0] & 0xFF) << 8 | quad[1] & 0xFF);
        String ultimate = Integer.toHexString((quad[2] & 0xFF) << 8 | quad[3] & 0xFF);
        return initialPart + penultimate + ":" + ultimate;
    }

    public static String toUriString(InetAddress ip) {
        if (ip instanceof Inet6Address) {
            return "[" + ip.getHostAddress() + "]";
        }
        return ip.getHostAddress();
    }

    public static InetAddress forUriString(String hostAddr) {
        Preconditions.checkNotNull(hostAddr);
        Preconditions.checkArgument(hostAddr.length() > 0, "host string is empty");
        InetAddress retval = null;
        try {
            retval = InetAddresses.forString(hostAddr);
            if (retval instanceof Inet4Address) {
                return retval;
            }
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
        if (!hostAddr.startsWith("[") || !hostAddr.endsWith("]")) {
            throw new IllegalArgumentException("Not a valid address: \"" + hostAddr + '\"');
        }
        retval = InetAddresses.forString(hostAddr.substring(1, hostAddr.length() - 1));
        if (retval instanceof Inet6Address) {
            return retval;
        }
        throw new IllegalArgumentException("Not a valid address: \"" + hostAddr + '\"');
    }

    public static boolean isUriInetAddress(String ipString) {
        try {
            InetAddresses.forUriString(ipString);
            return true;
        }
        catch (IllegalArgumentException e2) {
            return false;
        }
    }

    public static boolean isCompatIPv4Address(Inet6Address ip) {
        if (!ip.isIPv4CompatibleAddress()) {
            return false;
        }
        byte[] bytes = ip.getAddress();
        return bytes[12] != 0 || bytes[13] != 0 || bytes[14] != 0 || bytes[15] != 0 && bytes[15] != 1;
    }

    public static Inet4Address getCompatIPv4Address(Inet6Address ip) {
        Preconditions.checkArgument(InetAddresses.isCompatIPv4Address(ip), "Address '%s' is not IPv4-compatible.", ip.getHostAddress());
        return InetAddresses.getInet4Address(InetAddresses.copyOfRange(ip.getAddress(), 12, 16));
    }

    public static boolean is6to4Address(Inet6Address ip) {
        byte[] bytes = ip.getAddress();
        return bytes[0] == 32 && bytes[1] == 2;
    }

    public static Inet4Address get6to4IPv4Address(Inet6Address ip) {
        Preconditions.checkArgument(InetAddresses.is6to4Address(ip), "Address '%s' is not a 6to4 address.", ip.getHostAddress());
        return InetAddresses.getInet4Address(InetAddresses.copyOfRange(ip.getAddress(), 2, 6));
    }

    public static boolean isTeredoAddress(Inet6Address ip) {
        byte[] bytes = ip.getAddress();
        return bytes[0] == 32 && bytes[1] == 1 && bytes[2] == 0 && bytes[3] == 0;
    }

    public static TeredoInfo getTeredoInfo(Inet6Address ip) {
        Preconditions.checkArgument(InetAddresses.isTeredoAddress(ip), "Address '%s' is not a Teredo address.", ip.getHostAddress());
        byte[] bytes = ip.getAddress();
        Inet4Address server = InetAddresses.getInet4Address(InetAddresses.copyOfRange(bytes, 4, 8));
        int flags = ByteStreams.newDataInput(bytes, 8).readShort() & 0xFFFF;
        int port = ~ByteStreams.newDataInput(bytes, 10).readShort() & 0xFFFF;
        byte[] clientBytes = InetAddresses.copyOfRange(bytes, 12, 16);
        for (int i = 0; i < clientBytes.length; ++i) {
            clientBytes[i] = ~clientBytes[i];
        }
        Inet4Address client = InetAddresses.getInet4Address(clientBytes);
        return new TeredoInfo(server, client, port, flags);
    }

    public static boolean isIsatapAddress(Inet6Address ip) {
        if (InetAddresses.isTeredoAddress(ip)) {
            return false;
        }
        byte[] bytes = ip.getAddress();
        if ((bytes[8] | 3) != 3) {
            return false;
        }
        return bytes[9] == 0 && bytes[10] == 94 && bytes[11] == -2;
    }

    public static Inet4Address getIsatapIPv4Address(Inet6Address ip) {
        Preconditions.checkArgument(InetAddresses.isIsatapAddress(ip), "Address '%s' is not an ISATAP address.", ip.getHostAddress());
        return InetAddresses.getInet4Address(InetAddresses.copyOfRange(ip.getAddress(), 12, 16));
    }

    public static boolean hasEmbeddedIPv4ClientAddress(Inet6Address ip) {
        return InetAddresses.isCompatIPv4Address(ip) || InetAddresses.is6to4Address(ip) || InetAddresses.isTeredoAddress(ip);
    }

    public static Inet4Address getEmbeddedIPv4ClientAddress(Inet6Address ip) {
        if (InetAddresses.isCompatIPv4Address(ip)) {
            return InetAddresses.getCompatIPv4Address(ip);
        }
        if (InetAddresses.is6to4Address(ip)) {
            return InetAddresses.get6to4IPv4Address(ip);
        }
        if (InetAddresses.isTeredoAddress(ip)) {
            return InetAddresses.getTeredoInfo(ip).getClient();
        }
        throw new IllegalArgumentException(String.format("'%s' has no embedded IPv4 address.", ip.getHostAddress()));
    }

    public static Inet4Address getCoercedIPv4Address(InetAddress ip) {
        if (ip instanceof Inet4Address) {
            return (Inet4Address)ip;
        }
        byte[] bytes = ip.getAddress();
        boolean leadingBytesOfZero = true;
        for (int i = 0; i < 15; ++i) {
            if (bytes[i] == 0) continue;
            leadingBytesOfZero = false;
            break;
        }
        if (leadingBytesOfZero && bytes[15] == 1) {
            return LOOPBACK4;
        }
        if (leadingBytesOfZero && bytes[15] == 0) {
            return ANY4;
        }
        Inet6Address ip6 = (Inet6Address)ip;
        long addressAsLong = 0L;
        addressAsLong = InetAddresses.hasEmbeddedIPv4ClientAddress(ip6) ? (long)InetAddresses.getEmbeddedIPv4ClientAddress(ip6).hashCode() : ByteBuffer.wrap(ip6.getAddress(), 0, 8).getLong();
        int coercedHash = InetAddresses.hash64To32(addressAsLong);
        if ((coercedHash |= 0xE0000000) == -1) {
            coercedHash = -2;
        }
        return InetAddresses.getInet4Address(Ints.toByteArray(coercedHash));
    }

    @VisibleForTesting
    static int hash64To32(long key) {
        key = (key ^ 0xFFFFFFFFFFFFFFFFL) + (key << 18);
        key ^= key >>> 31;
        key *= 21L;
        key ^= key >>> 11;
        key += key << 6;
        key ^= key >>> 22;
        return (int)key;
    }

    public static int coerceToInteger(InetAddress ip) {
        return ByteStreams.newDataInput(InetAddresses.getCoercedIPv4Address(ip).getAddress()).readInt();
    }

    public static Inet4Address fromInteger(int address) {
        return InetAddresses.getInet4Address(Ints.toByteArray(address));
    }

    public static InetAddress fromLittleEndianByteArray(byte[] addr) throws UnknownHostException {
        byte[] reversed = new byte[addr.length];
        for (int i = 0; i < addr.length; ++i) {
            reversed[i] = addr[addr.length - i - 1];
        }
        return InetAddress.getByAddress(reversed);
    }

    private static byte[] copyOfRange(byte[] original, int from, int to) {
        Preconditions.checkNotNull(original);
        int end = Math.min(to, original.length);
        byte[] result = new byte[to - from];
        System.arraycopy(original, from, result, 0, end - from);
        return result;
    }

    @Beta
    public static final class TeredoInfo {
        private final Inet4Address server;
        private final Inet4Address client;
        private final int port;
        private final int flags;

        public TeredoInfo(@Nullable Inet4Address server, @Nullable Inet4Address client, int port, int flags) {
            Preconditions.checkArgument(port >= 0 && port <= 65535, "port '%d' is out of range (0 <= port <= 0xffff)", port);
            Preconditions.checkArgument(flags >= 0 && flags <= 65535, "flags '%d' is out of range (0 <= flags <= 0xffff)", flags);
            this.server = server != null ? server : ANY4;
            this.client = client != null ? client : ANY4;
            this.port = port;
            this.flags = flags;
        }

        public Inet4Address getServer() {
            return this.server;
        }

        public Inet4Address getClient() {
            return this.client;
        }

        public int getPort() {
            return this.port;
        }

        public int getFlags() {
            return this.flags;
        }
    }
}

