## ----setup, include = FALSE---------------------------------------------------
# knitr::knit("vignettes/quick_start_guide.Rmd", output = "README.md")
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)
library(SharedObject)


## -----------------------------------------------------------------------------
library(parallel)
## Create the cluster
cl <- makeCluster(2)
## Create data
n <- 3
A <- matrix(1:(n^2), n, n)
## Create a shared object
shared_A <- share(A)
## Export the shared object
clusterExport(cl,"shared_A")
## Check the exported object
clusterEvalQ(cl, shared_A)

stopCluster(cl)

## -----------------------------------------------------------------------------
## Check the data
A
shared_A
## Check the class
class(A)
class(shared_A)
## Check if they are identical
identical(A, shared_A)

## -----------------------------------------------------------------------------
## Check if an object is shared
is.shared(A)
is.shared(shared_A)

## -----------------------------------------------------------------------------
## This function has been renamed to `sharedObjectProperties` 
## in Bioc 3.13
## get a summary report
getSharedObjectProperty(shared_A)

## get the individual properties
getCopyOnWrite(shared_A)
getSharedSubset(shared_A)
getSharedCopy(shared_A)

## -----------------------------------------------------------------------------
## the element `A` is sharable and `B` is not
x <- list(A = 1:3, B = as.symbol("x"))

## No error will be given, 
## but the element `B` is not shared
shared_x <- share(x)

## Use the `mustWork` argument
## An error will be given for the non-sharable object `B`
tryCatch({
  shared_x <- share(x, mustWork = TRUE)
},
error=function(msg)message(msg$message)
)

## -----------------------------------------------------------------------------
## A single logical is returned
is.shared(shared_x)
## Check each element in x
is.shared(shared_x, depth = 1)

## -----------------------------------------------------------------------------
## This function has been renamed to `sharedObjectPkgOptions` 
## in Bioc 3.13
getSharedObjectOptions()

## -----------------------------------------------------------------------------
## These functions have been renamed to `sharedObjectPkgOptions` 
## in Bioc 3.13
## change the default setting
setSharedObjectOptions(mustWork = TRUE)
## Check if the change is made
getSharedObjectOptions("mustWork")
## Resume to default
setSharedObjectOptions(mustWork = FALSE)

## -----------------------------------------------------------------------------
x1 <- share(1:4)
x2 <- x1

## x2 became a regular R object after the change
is.shared(x2)
x2[1] <- 10L
is.shared(x2)

## x1 is not changed
x1
x2

## -----------------------------------------------------------------------------
x1 <- share(1:4, copyOnWrite=FALSE)
x2 <- x1

## x2 will not be duplicated when a change is made
is.shared(x2)
x2[1] <- 0L
is.shared(x2)

## x1 has been changed
x1
x2

## -----------------------------------------------------------------------------
x <- share(1:4, copyOnWrite = FALSE)
x
-x
x

## -----------------------------------------------------------------------------
## Create x1 with copy-on-write off
x1 <- share(1:4, copyOnWrite=FALSE)
x2 <- x1
## change the value of x2
x2[1] <- 0L
## Both x1 and x2 are affected
x1
x2

## Enable copy-on-write
## x2 is now independent with x1
setCopyOnWrite(x2, TRUE)
x2[2] <- 0L
## only x2 is affected
x1
x2

## -----------------------------------------------------------------------------
x1 <- share(1:4)
x2 <- x1
## x2 is not shared after the duplication
is.shared(x2)
x2[1] <- 0L
is.shared(x2)


x1 <- share(1:4, sharedCopy = TRUE)
x2 <- x1
## x2 is still shared(but different from x1) 
## after the duplication
is.shared(x2)
x2[1] <- 0L
is.shared(x2)

## -----------------------------------------------------------------------------
sessionInfo()

