/*
 * Decompiled with CFR 0.152.
 */
package net.sf.picard.sam;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import net.sf.picard.PicardException;
import net.sf.picard.metrics.MetricBase;
import net.sf.picard.metrics.MetricsFile;
import net.sf.picard.reference.ReferenceSequence;
import net.sf.picard.reference.ReferenceSequenceFile;
import net.sf.picard.reference.ReferenceSequenceFileWalker;
import net.sf.picard.sam.ReservedTagConstants;
import net.sf.picard.util.Histogram;
import net.sf.picard.util.Log;
import net.sf.samtools.FileTruncatedException;
import net.sf.samtools.SAMFileHeader;
import net.sf.samtools.SAMFileReader;
import net.sf.samtools.SAMFormatException;
import net.sf.samtools.SAMRecord;
import net.sf.samtools.SAMSortOrderChecker;
import net.sf.samtools.SAMTag;
import net.sf.samtools.SAMValidationError;
import net.sf.samtools.util.BlockCompressedInputStream;
import net.sf.samtools.util.CloserUtil;
import net.sf.samtools.util.IOUtil;
import net.sf.samtools.util.SequenceUtil;
import net.sf.samtools.util.StringUtil;

public class SamFileValidator {
    private Histogram<SAMValidationError.Type> errorsByType = new Histogram();
    private final PrintWriter out;
    private Map<String, PairEndInfo> pairEndInfoByName;
    private ReferenceSequenceFileWalker refFileWalker = null;
    private boolean verbose = false;
    private int maxVerboseOutput = 100;
    private SAMSortOrderChecker orderChecker;
    private Set<SAMValidationError.Type> errorsToIgnore = EnumSet.noneOf(SAMValidationError.Type.class);
    private boolean ignoreWarnings = false;
    private boolean bisulfiteSequenced = false;
    private boolean sequenceDictionaryEmptyAndNoWarningEmitted = false;
    private static final Log log = Log.getInstance(SamFileValidator.class);

    public SamFileValidator(PrintWriter out) {
        this.out = out;
    }

    public void setErrorsToIgnore(Collection<SAMValidationError.Type> types) {
        if (!types.isEmpty()) {
            this.errorsToIgnore = EnumSet.copyOf(types);
        }
    }

    public void setIgnoreWarnings(boolean ignoreWarnings) {
        this.ignoreWarnings = ignoreWarnings;
    }

    public boolean validateSamFileSummary(SAMFileReader samReader, ReferenceSequenceFile reference) {
        this.init(reference);
        this.validateSamFile(samReader, this.out);
        boolean result = this.errorsByType.isEmpty();
        if (this.errorsByType.getCount() > 0.0) {
            Histogram<String> errorsAndWarningsByType = new Histogram<String>("Error Type", "Count");
            for (Histogram.Bin bin : this.errorsByType.values()) {
                errorsAndWarningsByType.increment(((SAMValidationError.Type)((Object)bin.getId())).getHistogramString(), bin.getValue());
            }
            MetricsFile metricsFile = new MetricsFile();
            this.errorsByType.setBinLabel("Error Type");
            this.errorsByType.setValueLabel("Count");
            metricsFile.setHistogram(errorsAndWarningsByType);
            metricsFile.write(this.out);
        }
        this.cleanup();
        return result;
    }

    public boolean validateSamFileVerbose(SAMFileReader samReader, ReferenceSequenceFile reference) {
        this.init(reference);
        try {
            this.validateSamFile(samReader, this.out);
        }
        catch (MaxOutputExceededException e) {
            this.out.println("Maximum output of [" + this.maxVerboseOutput + "] errors reached.");
        }
        boolean result = this.errorsByType.isEmpty();
        this.cleanup();
        return result;
    }

    public void validateBamFileTermination(File inputFile) {
        BufferedInputStream inputStream = null;
        try {
            inputStream = IOUtil.toBufferedStream(new FileInputStream(inputFile));
            if (!BlockCompressedInputStream.isValidFile(inputStream)) {
                return;
            }
            BlockCompressedInputStream.FileTermination terminationState = BlockCompressedInputStream.checkTermination(inputFile);
            if (terminationState.equals((Object)BlockCompressedInputStream.FileTermination.DEFECTIVE)) {
                this.addError(new SAMValidationError(SAMValidationError.Type.TRUNCATED_FILE, "BAM file has defective last gzip block", inputFile.getPath()));
            } else if (terminationState.equals((Object)BlockCompressedInputStream.FileTermination.HAS_HEALTHY_LAST_BLOCK)) {
                this.addError(new SAMValidationError(SAMValidationError.Type.BAM_FILE_MISSING_TERMINATOR_BLOCK, "Older BAM file -- does not have terminator block", inputFile.getPath()));
            }
        }
        catch (IOException e) {
            throw new PicardException("IOException", e);
        }
        finally {
            if (inputStream != null) {
                CloserUtil.close(inputStream);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void validateSamFile(SAMFileReader samReader, PrintWriter out) {
        try {
            samReader.setValidationStringency(SAMFileReader.ValidationStringency.SILENT);
            this.validateHeader(samReader.getFileHeader());
            this.orderChecker = new SAMSortOrderChecker(samReader.getFileHeader().getSortOrder());
            this.validateSamRecords(samReader);
            if (this.errorsByType.isEmpty()) {
                out.println("No errors found");
            }
        }
        finally {
            out.flush();
        }
    }

    private void validateSamRecords(Iterable<SAMRecord> samRecords) {
        long recordNumber = 1L;
        try {
            for (SAMRecord record : samRecords) {
                List<SAMValidationError> errors = record.isValid();
                if (errors != null) {
                    for (SAMValidationError error : errors) {
                        error.setRecordNumber(recordNumber);
                        this.addError(error);
                    }
                }
                this.validateMateFields(record, recordNumber);
                this.validateSortOrder(record, recordNumber);
                boolean cigarIsValid = this.validateCigar(record, recordNumber);
                if (cigarIsValid) {
                    this.validateNmTag(record, recordNumber);
                }
                this.validateSecondaryBaseCalls(record, recordNumber);
                this.validateTags(record, recordNumber);
                if (this.sequenceDictionaryEmptyAndNoWarningEmitted && !record.getReadUnmappedFlag()) {
                    this.addError(new SAMValidationError(SAMValidationError.Type.MISSING_SEQUENCE_DICTIONARY, "Sequence dictionary is empty", null));
                    this.sequenceDictionaryEmptyAndNoWarningEmitted = false;
                }
                if (++recordNumber % 10000000L != 0L) continue;
                log.info(recordNumber + " reads validated.");
            }
        }
        catch (SAMFormatException e) {
            this.out.println("SAMFormatException on record " + ++recordNumber);
            throw new PicardException("SAMFormatException on record " + recordNumber, e);
        }
        catch (FileTruncatedException e) {
            this.addError(new SAMValidationError(SAMValidationError.Type.TRUNCATED_FILE, "File is truncated", null));
        }
    }

    private void validateTags(SAMRecord record, long recordNumber) {
        for (SAMRecord.SAMTagAndValue tagAndValue : record.getAttributes()) {
            if (!(tagAndValue.value instanceof Long)) continue;
            this.addError(new SAMValidationError(SAMValidationError.Type.TAG_VALUE_TOO_LARGE, "Numeric value too large for tag " + tagAndValue.tag, record.getReadName(), recordNumber));
        }
    }

    private void validateSecondaryBaseCalls(SAMRecord record, long recordNumber) {
        String u2;
        String e2 = (String)record.getAttribute(SAMTag.E2.name());
        if (e2 != null) {
            if (e2.length() != record.getReadLength()) {
                this.addError(new SAMValidationError(SAMValidationError.Type.MISMATCH_READ_LENGTH_AND_E2_LENGTH, String.format("E2 tag length (%d) != read length (%d)", e2.length(), record.getReadLength()), record.getReadName(), recordNumber));
            }
            byte[] bases = record.getReadBases();
            byte[] secondaryBases = StringUtil.stringToBytes(e2);
            for (int i = 0; i < Math.min(bases.length, secondaryBases.length); ++i) {
                if (SequenceUtil.isNoCall(bases[i]) || SequenceUtil.isNoCall(secondaryBases[i]) || !SequenceUtil.basesEqual(bases[i], secondaryBases[i])) continue;
                this.addError(new SAMValidationError(SAMValidationError.Type.E2_BASE_EQUALS_PRIMARY_BASE, String.format("Secondary base call  (%c) == primary base call (%c)", Character.valueOf((char)secondaryBases[i]), Character.valueOf((char)bases[i])), record.getReadName(), recordNumber));
                break;
            }
        }
        if ((u2 = (String)record.getAttribute(SAMTag.U2.name())) != null && u2.length() != record.getReadLength()) {
            this.addError(new SAMValidationError(SAMValidationError.Type.MISMATCH_READ_LENGTH_AND_U2_LENGTH, String.format("U2 tag length (%d) != read length (%d)", u2.length(), record.getReadLength()), record.getReadName(), recordNumber));
        }
    }

    private boolean validateCigar(SAMRecord record, long recordNumber) {
        if (record.getReadUnmappedFlag()) {
            return true;
        }
        SAMFileReader.ValidationStringency savedStringency = record.getValidationStringency();
        record.setValidationStringency(SAMFileReader.ValidationStringency.LENIENT);
        List<SAMValidationError> errors = record.validateCigar(recordNumber);
        record.setValidationStringency(savedStringency);
        if (errors == null) {
            return true;
        }
        boolean valid = true;
        for (SAMValidationError error : errors) {
            this.addError(error);
            valid = false;
        }
        return valid;
    }

    private void validateSortOrder(SAMRecord record, long recordNumber) {
        SAMRecord prev = this.orderChecker.getPreviousRecord();
        if (!this.orderChecker.isSorted(record)) {
            this.addError(new SAMValidationError(SAMValidationError.Type.RECORD_OUT_OF_ORDER, String.format("The record is out of [%s] order, prior read name [%s], prior coodinates [%d:%d]", record.getHeader().getSortOrder().name(), prev.getReadName(), prev.getReferenceIndex(), prev.getAlignmentStart()), record.getReadName(), recordNumber));
        }
    }

    private void init(ReferenceSequenceFile reference) {
        this.pairEndInfoByName = new HashMap<String, PairEndInfo>();
        if (reference != null) {
            this.refFileWalker = new ReferenceSequenceFileWalker(reference);
        }
    }

    private void cleanup() {
        this.errorsByType = null;
        this.pairEndInfoByName = null;
        this.refFileWalker = null;
    }

    private void validateNmTag(SAMRecord record, long recordNumber) {
        if (!record.getReadUnmappedFlag()) {
            ReferenceSequence refSequence;
            int actualNucleotideDiffs;
            Integer tagNucleotideDiffs = record.getIntegerAttribute(ReservedTagConstants.NM);
            if (tagNucleotideDiffs == null) {
                this.addError(new SAMValidationError(SAMValidationError.Type.MISSING_TAG_NM, "NM tag (nucleotide differences) is missing", record.getReadName(), recordNumber));
            } else if (this.refFileWalker != null && !tagNucleotideDiffs.equals(actualNucleotideDiffs = SequenceUtil.calculateSamNmTag(record, (refSequence = this.refFileWalker.get(record.getReferenceIndex())).getBases(), 0, this.isBisulfiteSequenced()))) {
                this.addError(new SAMValidationError(SAMValidationError.Type.INVALID_TAG_NM, "NM tag (nucleotide differences) in file [" + tagNucleotideDiffs + "] does not match reality [" + actualNucleotideDiffs + "]", record.getReadName(), recordNumber));
            }
        }
    }

    private void validateMateFields(SAMRecord record, long recordNumber) {
        if (!record.getReadPairedFlag() || record.getNotPrimaryAlignmentFlag()) {
            return;
        }
        PairEndInfo pairEndInfo = this.pairEndInfoByName.remove(record.getReadName());
        if (pairEndInfo == null) {
            this.pairEndInfoByName.put(record.getReadName(), new PairEndInfo(record, recordNumber));
        } else {
            List<SAMValidationError> errors = pairEndInfo.validateMates(new PairEndInfo(record, recordNumber), record.getReadName());
            for (SAMValidationError error : errors) {
                this.addError(error);
            }
        }
    }

    private void validateHeader(SAMFileHeader fileHeader) {
        for (SAMValidationError error : fileHeader.getValidationErrors()) {
            this.addError(error);
        }
        if (fileHeader.getVersion() == null) {
            this.addError(new SAMValidationError(SAMValidationError.Type.MISSING_VERSION_NUMBER, "Header has no version number", null));
        } else if (!fileHeader.getVersion().equals("1.0")) {
            this.addError(new SAMValidationError(SAMValidationError.Type.INVALID_VERSION_NUMBER, "Header version: " + fileHeader.getVersion() + " does not match expected version: " + "1.0", null));
        }
        if (fileHeader.getSequenceDictionary().isEmpty()) {
            this.sequenceDictionaryEmptyAndNoWarningEmitted = true;
        }
        if (fileHeader.getReadGroups().isEmpty()) {
            this.addError(new SAMValidationError(SAMValidationError.Type.MISSING_READ_GROUP, "Read groups is empty", null));
        }
    }

    private void addError(SAMValidationError error) {
        if (this.errorsToIgnore.contains((Object)error.getType())) {
            return;
        }
        if (this.ignoreWarnings && error.getType().severity == SAMValidationError.Severity.WARNING) {
            return;
        }
        this.errorsByType.increment(error.getType());
        if (this.verbose) {
            this.out.println(error);
            this.out.flush();
            if (this.errorsByType.getCount() >= (double)this.maxVerboseOutput) {
                throw new MaxOutputExceededException();
            }
        }
    }

    public void setVerbose(boolean verbose, int maxVerboseOutput) {
        this.verbose = verbose;
        this.maxVerboseOutput = maxVerboseOutput;
    }

    public boolean isBisulfiteSequenced() {
        return this.bisulfiteSequenced;
    }

    public void setBisulfiteSequenced(boolean bisulfiteSequenced) {
        this.bisulfiteSequenced = bisulfiteSequenced;
    }

    private static class MaxOutputExceededException
    extends PicardException {
        MaxOutputExceededException() {
            super("maxVerboseOutput exceeded.");
        }
    }

    private static class PairEndInfo {
        private final int readAlignmentStart;
        private final int readReferenceIndex;
        private final boolean readNegStrandFlag;
        private final boolean readUnmappedFlag;
        private final int mateAlignmentStart;
        private final int mateReferenceIndex;
        private final boolean mateNegStrandFlag;
        private final boolean mateUnmappedFlag;
        private final long recordNumber;

        public PairEndInfo(SAMRecord record, long recordNumber) {
            this.recordNumber = recordNumber;
            this.readAlignmentStart = record.getAlignmentStart();
            this.readNegStrandFlag = record.getReadNegativeStrandFlag();
            this.readReferenceIndex = record.getReferenceIndex();
            this.readUnmappedFlag = record.getReadUnmappedFlag();
            this.mateAlignmentStart = record.getMateAlignmentStart();
            this.mateNegStrandFlag = record.getMateNegativeStrandFlag();
            this.mateReferenceIndex = record.getMateReferenceIndex();
            this.mateUnmappedFlag = record.getMateUnmappedFlag();
        }

        public List<SAMValidationError> validateMates(PairEndInfo mate, String readName) {
            ArrayList<SAMValidationError> errors = new ArrayList<SAMValidationError>();
            this.validateMateFields(this, mate, readName, errors);
            this.validateMateFields(mate, this, readName, errors);
            return errors;
        }

        private void validateMateFields(PairEndInfo end1, PairEndInfo end2, String readName, List<SAMValidationError> errors) {
            if (end1.mateAlignmentStart != end2.readAlignmentStart) {
                errors.add(new SAMValidationError(SAMValidationError.Type.MISMATCH_MATE_ALIGNMENT_START, "Mate alignment does not match alignment start of mate", readName, end1.recordNumber));
            }
            if (end1.mateNegStrandFlag != end2.readNegStrandFlag) {
                errors.add(new SAMValidationError(SAMValidationError.Type.MISMATCH_FLAG_MATE_NEG_STRAND, "Mate negative strand flag does not match read negative strand flag of mate", readName, end1.recordNumber));
            }
            if (end1.mateReferenceIndex != end2.readReferenceIndex) {
                errors.add(new SAMValidationError(SAMValidationError.Type.MISMATCH_MATE_REF_INDEX, "Mate reference index (MRNM) does not match reference index of mate", readName, end1.recordNumber));
            }
            if (end1.mateUnmappedFlag != end2.readUnmappedFlag) {
                errors.add(new SAMValidationError(SAMValidationError.Type.MISMATCH_FLAG_MATE_UNMAPPED, "Mate unmapped flag does not match read unmapped flag of mate", readName, end1.recordNumber));
            }
        }
    }

    public static class ValidationMetrics
    extends MetricBase {
    }
}

