### Name: combineAffyBatch
### Title: A function to combine data from different Affymetrix genechip
###   types into one AffyBatch.
### Aliases: combineAffyBatch
### Keywords: manip

### ** Examples

  ## In this example, we show how to combine the data from two different
  ## array types whose reporter sequences are partially the same.

  library("affy")
  library("hu6800cdf")
  library("hu6800probe")

  ## Load the data:
  f1 <- system.file("extdata", "118T1.cel", package="matchprobes")
  f2 <- system.file("extdata", "CL2001032020AA.cel", package="matchprobes")
  pd1 <- new("AnnotatedDataFrame", data=data.frame(fromFile=I(f1), row.names="f1"))
  pd2 <- new("AnnotatedDataFrame", data=data.frame(fromFile=I(f2), row.names="f2"))
  x1 <- read.affybatch(filenames=f1, compress=TRUE, phenoData=pd1)
  x2 <- read.affybatch(filenames=f2, compress=TRUE, phenoData=pd2)

  ## 'f1' and 'f2' contain the filenames of the CEL files. In practice,
  ## they will be vectors with many filenames per array type, here,
  ## for demonstration, we only use one single CEL file per array type,
  ## which are provided in the 'extdata' subdirectory of the package.

  ## Combine the data. For this to work it is required that the packages
  ## hu6800probe and hgu95av2probe are installed:
  res <- combineAffyBatch(list(x1, x2), c("hu6800probe","hgu95av2probe"), newcdf="comb")

  ## The function returns a list 'res' with two elements: an AffyBatch
  ## object (dat) and a CDF enviroment (cdf):
  comb <- res$cdf
  z    <- rma(res$dat)

  ## First, let us look at the distribution of the number of common
  ## probes per probeset:
  prs <- mget(ls(comb), comb, ifnotfound=NA)
  nrprobes <- sapply(prs, function(x) nrow(x))
  barplot(table(nrprobes),
     xlab="number of probes per probeset",
     ylab="frequency")

  ## Let us also make scatterplots of the two combined arrays, first on
  ## the probe and then at the probe set level. This plot shows the
  ## comparison of the intensity data from common probes (left panel)
  ## and probesets (right panel) of the two array platforms:
  png("matchprobes-scp.png", width=900, height=480)
  par(mfrow=c(1,2))
  plot(exprs(res$dat), main="after combine",
    pch=".", asp=1, xlab="f1", ylab="f2", log="xy")
  plot(exprs(z), main="after RMA",
    pch=".", asp=1, xlab="f1", ylab="f2")
  dev.off()

  ## Here, we explicitely used the png() device to write this plot to.
  ## PDF, for example, can become cumbersomely large in size for
  ## feature-rich plots such as this one.



