package net.eisele.nio2;

import java.io.IOException;
import java.nio.file.ClosedWatchServiceException;
import java.nio.file.FileSystems;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.util.List;

/**
 * Change Watch for /temp
 * @author eiselem
 */
public class ChangeWatch {

    public static void main(String[] args) throws InterruptedException, IOException {

        WatchService watchService = FileSystems.getDefault().newWatchService();
        Path watchedPath = Paths.get("/temp");

        WatchKey key = null;
        try {
            key = watchedPath.register(watchService, StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_DELETE);
        } catch (UnsupportedOperationException uox) {
            System.err.println("No native Notification support!");
            // handle this error here
        } catch (IOException iox) {
            System.err.println("I/O errors");
            iox.printStackTrace(System.err);
        }

        for (;;) {
            // take() will block until a file has been created/deleted
            WatchKey signalledKey;
            try {
                signalledKey = watchService.take();
            } catch (InterruptedException ix) {
                // we'll ignore being interrupted
                continue;
            } catch (ClosedWatchServiceException cwse) {
                // other thread closed watch service
                System.out.println("Watch service closed, terminating.");
                break;
            }

            // get list of events from key
            List<WatchEvent<?>> list = signalledKey.pollEvents();

            // Call reset() AFTER pollEvents() to allow the
            // key to be reported again by the watch service
            signalledKey.reset();

            // Print what has happened
            for (WatchEvent e : list) {
                String message = "";
                if (e.kind() == StandardWatchEventKinds.ENTRY_CREATE) {
                    Path context = (Path) e.context();
                    message = context.toString() + " created";
                } else if (e.kind() == StandardWatchEventKinds.ENTRY_DELETE) {
                    Path context = (Path) e.context();
                    message = context.toString() + " deleted";
                } else if (e.kind() == StandardWatchEventKinds.OVERFLOW) {
                    message = "OVERFLOW: more changes happened than we could retreive";
                }
                System.out.println(message + "\n");
            }
        }


    }
}
