/*****************************************************************************
*  ADBotPlayerV1                                                             *
*                                                                            *
*  Copyright (C) 2008 Andreas Dittrich                                       *
*                                                                            *
*  Author:  Andreas Dittrich                                                 *
*           Am Schimmelberg 29                                               *
*           67729 Sippersfeld (Germany)                                      *
*****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "inttypes.h"
#include "vector.h"
#include "tables.h"

#include "../interface/ADBotPlayerV1.h"
#include "ADBotPlayerV1_private.h"

#include "gamestate.h"



#pragma warning( disable : 4127 )


#define MIN(a,b) ( (a)<(b) ? (a) : (b) )
#define MAX(a,b) ( (a)>(b) ? (a) : (b) )


#ifndef RELEASE_CONTEST
	#define ADD_DBGTEXT(x) \
	{ \
		strcpy_s( pGS->dbgtext[pGS->dbgtext_linenumber & 0x07], 40, x ); \
		pGS->dbgtext_linenumber += 1; \
	}
#else
	#define ADD_DBGTEXT(x)
#endif



typedef struct {

	int is_ship_present;

	struct ship_tag {
		vector_t r;
		vector_t d;
	} ship;


	struct saucer_tag {
		vector_t r;
		int state;
	} saucer;


	int num_asteroids;
	int num_explosions;

	struct asteroid_tag {
		vector_t r;
		int shape;			// 0 .. 3
		int state;
	} asteroids[MAX_NUM_ASTEROIDS];


	int num_shots;

	struct shot_tag {
		vector_t r;
	} shots[MAX_NUM_SHOTS];


	uint32_t phi2;

	int score;

	int is_E_visible;

} ScreenObjects_t;





static void UpdateShipState( 
	GameState_t *pGS, 
	ScreenObjects_t *pSO,
	int time );

static void InitRandomGenerator( 
	GameState_t *pGS, 
	int time );

static void AddNewAsteroids( 
	GameState_t *pGS,
	asteroid_t *pC,
	int time );

static void UpdateAsteroidsState( 
	GameState_t *pGS, 
	ScreenObjects_t *pSO,
	int time );

static void UpdateShotsState( 
	GameState_t *pGS, 
	ScreenObjects_t *pSO,
	int time );

static void UpdateSaucerState( 
	GameState_t *pGS, 
	ScreenObjects_t *pSO,
	int time );

static void DecodeVectorRam( 
	ScreenObjects_t *pScreenObj, 
	uint8_t *pVectorRam );

static int getNearestAngleCode( vector_t v );

static int getAngleCode( vector_t v );



int UpdateGameState( 
	GameState_t *pGS, 
	uint8_t *pVectorRam,
	int time,
	KeysInfo_t *pKeysInfo )
{
	//first time we are called?
	if (pGS->t0_run==0)
		pGS->t0_run = time;

	//save keys
	pGS->keysinfo_last  = pGS->keysinfo_curr;
	pGS->keysinfo_curr  = pGS->keysinfo_next;
	pGS->keysinfo_next  = *pKeysInfo;

	pGS->keysLog[time&((1<<KEYSLOG_SIZE_LOG2)-1)] = *pKeysInfo;

	//decode Screen
	ScreenObjects_t SO;
	ScreenObjects_t * pSO = &SO;

	DecodeVectorRam( pSO, pVectorRam );

	//if ( (pGS->phi4&0x01) != pSO->phi2 )
	//	pGS->phi4++;

	//start of new game?
	if ( (pSO->is_ship_present) && 
		 (pGS->ship.state<=0) && 
		 (pGS->t0_game <= 0) )
	{
		pGS->t0_game = time;

		pGS->time = time - 1;
		pGS->level = 0;
		pGS->num_asteroids = 0;
		pGS->saucer.state = 0;
		pGS->num_shots = 0;
		pGS->next_asteroid_id = 0;

		//clear ship state
		memset( &pGS->ship, 0x00, sizeof(pGS->ship) );
	}

	//start of new level?
	if ( (pGS->num_asteroids<=0) && 
		 (pSO->num_asteroids>0) && 
		 (pGS->t0_game>0) )
	{
		pGS->level += 1;
		pGS->t0_level = time;

		//clear asteroids state
		pGS->num_asteroids = 0;
		for (int j=0; j<MAX_NUM_ASTEROIDS; j++)
			pGS->asteroids[j].state = 0;

		//clear shots state
		pGS->num_shots = 0;
		memset( pGS->shots, 0x00, sizeof(pGS->shots) );

		//clear saucer state
		memset( &pGS->saucer, 0x00, sizeof(pGS->saucer) );

		//random generator
		pGS->n_rgen_resync = 0;
	}

	if (pGS->t0_game <= 0)
		goto Set_Time_And_Return;

	//frame lost?
	if ( (time - pGS->time) != 1 )
	{
		ASSERT_MSG( !"frame discontinuity" );

		if ( (time - pGS->time) < 1 ) //shall never happen
		{
			pGS->keysinfo_next  = pGS->keysinfo_curr;
			pGS->keysinfo_curr  = pGS->keysinfo_last;

			return 0;
		}

		if ( (time - pGS->time) > 3 ) //to bad, we lost too many frames
			return -1;

		//try to recover somehow
		pGS->keysinfo_next  = pGS->keysinfo_curr;
		pGS->keysinfo_curr  = pGS->keysinfo_last;

		if ( (time - pGS->time) > 1 )
			PredictGameState( pGS, pGS->time+1, pKeysInfo );

		if ( (time - pGS->time) > 1 )
			PredictGameState( pGS, pGS->time+1, pKeysInfo );

		pGS->keysinfo_last  = pGS->keysinfo_curr;
		pGS->keysinfo_curr  = pGS->keysinfo_next;
		pGS->keysinfo_next  = *pKeysInfo;
	}


	//misc info from vectorram
	//========================

	//score
	if ( time > (pGS->t0_game + 10) )
	{
		//score
		int score_delta = pSO->score - pGS->score;
		if ( score_delta<0 )
			score_delta += 100000;

		pGS->score += score_delta;
	}


	//saucer
	//======

	UpdateSaucerState( 
		pGS, 
		pSO,
		time );


	//asteroids
	//=========

	UpdateAsteroidsState( 
		pGS, 
		pSO, 
		time );


	//shots
	//=====

	UpdateShotsState( 
		pGS, 
		pSO, 
		time );


	//ship
	//====

	UpdateShipState( 
		pGS, 
		pSO,
		time );


	//random generator
	//================

	InitRandomGenerator( 
		pGS, 
		time );

	CycleRandomGenerator(
		pGS,
		time,
		1 );

Set_Time_And_Return:

	//set new timestamp
	pGS->time = time;

	return 0;
}


void PredictGameState( 
	GameState_t *pGS, 
	int time,
	KeysInfo_t *pKeysInfo )
{
	KeysInfo_t keysinfo = {};
	if (pKeysInfo==NULL)
		pKeysInfo = &keysinfo;

	//save keys
	pGS->keysinfo_last  = pGS->keysinfo_curr;
	pGS->keysinfo_curr  = pGS->keysinfo_next;
	pGS->keysinfo_next  = *pKeysInfo;


	//check collision of saucer<->shots
	if (pGS->saucer.state>0)
	{
		saucer_t * const pA = &(pGS->saucer);

		int width_saucer = asteroid_width_table[pA->state&0x07];

		//check collision with shots
		for (int n=0; n<pGS->num_shots; n++)
		{
			vector_t u = pA->r;
			vsub( &u, pGS->shots[n].r );

			if ( is_hit_octagon( u, width_saucer ) )
			{
				//remove shot
				pGS->shots[n] = pGS->shots[pGS->num_shots-1];
				pGS->num_shots -= 1;

				//remove saucer
				pA->state = 0;

				break;
			}
		}
	}

	//check collision asteroids
	for (int j=0; j<MAX_NUM_ASTEROIDS; j++)
	{
		asteroid_t * const pA = &(pGS->asteroids[j]);
		
		if (pA->state<=0)
			continue;

		int width_asteroid = asteroid_width_table[pA->state&0x07];

		//check collision with shots
		for (int n=0; n<pGS->num_shots; n++)
		{
			vector_t u = pA->r;
			vsub( &u, pGS->shots[n].r );

			if ( is_hit_octagon( u, width_asteroid ) )
			{
				//remove shot
				pGS->shots[n] = pGS->shots[pGS->num_shots-1];
				pGS->num_shots -= 1;

				if ( (pGS->rgen.s != 0) && 
					 (pA->flags & (IS_TRACKED|IS_ESTIMATED)) && 
					 (pGS->shots[n].flags & IS_TRACKED) )
				{
					//add new asteroid(s)
					AddNewAsteroids(
						pGS,
						pA,
						time );
				}

				//remove current asteroid
				pA->state = -38;
			}
		}

		//check collision with saucer
		if (pGS->saucer.state>0)
		{
			int width = width_asteroid + 0x1C;
			if (pGS->saucer.state>1)
				width += 0x1C;

			vector_t u = pA->r;
			vsub( &u, pGS->saucer.r );

			if ( is_hit_octagon( u, width ) )
			{
				//remove saucer
				pGS->saucer.state = 0;

				//add new asteroid(s)
				if ( (pGS->rgen.s != 0) && 
					 (pA->flags & (IS_TRACKED|IS_ESTIMATED) ) && 
					 (pGS->saucer.flags & IS_TRACKED) )
				{
					//add new asteroid(s)
					AddNewAsteroids(
						pGS,
						pA,
						time );
				}

				//remove current asteroid
				pA->state = -38;
			}
		}
	}

	//new position of shots
	for (int j=0; j<pGS->num_shots; j++)
	{
		shot_t * const pA = &(pGS->shots[j]);

		//remove shot?
		if ( ( time - pA->t0 ) > 67 )
		{
			//remove shot
			pGS->shots[j] = pGS->shots[pGS->num_shots-1];
			pGS->num_shots -= 1;

			continue;
		}

		vadd( &pA->r, pA->v );
		normalize_r( &(pA->r) );
	}

	//new position of ship
	if (pGS->ship.state>0)
	{
		ship_t * const pA = &(pGS->ship);

		vadd( &pA->r, pA->v );
		normalize_r( &(pA->r) );
	}

	//new position of saucer
	if (pGS->saucer.state>0)
	{
		saucer_t * const pA = &(pGS->saucer);

		vadd( &pA->r, pA->v );
		normalize_r( &(pA->r) );
	}

	//new position of asteroids
	for (int j=0; j<MAX_NUM_ASTEROIDS; j++)
	{
		asteroid_t * const pA = &(pGS->asteroids[j]);
		
		if (pA->state<0)
			pA->state += 1;

		if (pA->state<=0)
			continue;

		vadd( &pA->r, pA->v );
		normalize_r( &(pA->r) );
	}

	//state of gun
	int num_shots_available = 4;
	for (int j=0; j<pGS->num_shots; j++)
	{
		if (pGS->shots[j].flags & IS_SHIP_SHOT)
			num_shots_available--;
	}

	int is_gun_ready = 
		!(pGS->keysinfo_last.keys & KEY_FIRE) &&
		(num_shots_available>0);

	//fire?
	if ( (pGS->keysinfo_curr.keys & KEY_FIRE) && 
		 is_gun_ready )
	{
		is_gun_ready = 0;
		num_shots_available--;

		shot_t * const pA = &(pGS->shots[pGS->num_shots]);
		pGS->num_shots++;

		vector_t r = pGS->ship.r;
		vector_t v = pGS->ship.v0;
		vadd( &v, pGS->ship.v );

		r.x += ((5*pGS->ship.v0.x)>>1);
		r.y += ((5*pGS->ship.v0.y)>>1);
		normalize_r( &r );

		memset( pA, 0x00, sizeof(shot_t) );
		pA->flags = (pGS->ship.flags&IS_TRACKED) | IS_SHIP_SHOT;
		pA->r =  r;
		pA->rr[time&0x07] = r;
		pA->t0 = time;
		pA->v = v;
		pA->id = pGS->keysinfo_curr.id_shot;
	}

	pGS->ship.is_gun_ready        = is_gun_ready;
	pGS->ship.num_shots_available = num_shots_available;

	//rotate ship?
	{
		ship_t * const pA = &(pGS->ship);

		if (pGS->keysinfo_curr.keys & KEY_LEFT)
			pA->angle_code += 3;
		
		if (pGS->keysinfo_curr.keys & KEY_RIGHT)
			pA->angle_code -= 3;

		pA->angle_code &= 0xFF;
		pA->v0 = v0_shots[pA->angle_code];
	}

	CycleRandomGenerator(
		pGS,
		time,
		1 );

	//set new timestamp
	pGS->time = time;
}


int IsSaveHyperspaceJump( GameState_t *pGS, vector_t *prdest )
{
	vector_t r_dest;

	int is_save = CheckHyperspaceJump( pGS, 0, &r_dest );
	if ( !is_save )
		return 0;

	*prdest = r_dest;

	static GameState_t GS_0;
	GameState_t * const pGS_0 = &GS_0;

	memcpy( pGS_0, pGS, sizeof(GameState_t) );

	int time = pGS->time;

	KeysInfo_t keysinfo_none = {};

	for (int i=1; i<48; i++)
	{
		PredictGameState(
			pGS_0,
			time + i,
			&keysinfo_none );
	}

	pGS_0->ship.r = r_dest;

	//check collision shot with ship
	int dt_collision = 1000;
	for (int j=0; j<pGS_0->num_shots; j++)
	{
		const int width_correction = -15;
		int width = MAX( 0, asteroid_width_table[1] - width_correction );

		shot_t * const pC = &(pGS_0->shots[j]);

		vector_t r = pGS_0->ship.r;
		vsub( &r, pC->r );
		normalize_d( &r );
		
		for (int dt=0; dt<10; dt++)
		{
			vsub( &r, pC->v );

			if ( is_hit_octagon(r, width) )
			{
				if (dt<dt_collision)
					dt_collision = dt;

				break;
			}
		}
	}

	//check collision of asteroid or saucer with ship
	for( int i=0; i<(MAX_NUM_ASTEROIDS+1); i++)
	{
		moving_object_t * pA = (moving_object_t*)&pGS_0->asteroids[i];
		if ( i>=MAX_NUM_ASTEROIDS )
			pA = (moving_object_t*)&pGS_0->saucer;

		if (pA->state <= 0)
			continue;

		//check collision of asteroid or saucer with ship
		{
			const int width_correction = -15;

			int width = MAX( 0, ship_width_table[pA->state&0x07] - width_correction );

			vector_t r = pA->r;
			vsub( &r, pGS_0->ship.r );
			normalize_d( &r );		

			for (int dt=0; dt<10; dt++)
			{
				vadd( &r, pA->v );

				if ( is_hit_octagon(r, width) )
				{
					if (dt<dt_collision)
						dt_collision = dt;

					break;
				}
			}
		}
	}

	if (dt_collision > 10)
		return 1;

	return 0;
}


int CheckHyperspaceJump(
	GameState_t *pGS,
	int dt,
	vector_t *pr )
{
	int res = 1;

	//save random generator
	randgen_t rgen_temp = pGS->rgen;

	//advance generator
	CycleRandomGenerator( pGS, 0, 6+dt );
	uint16_t s = pGS->rgen.s;

	if (s==0)
		return 0;

	uint32_t rxh = (s>>5)&0x1F;
	if (rxh >= 0x1D)
		rxh = 0x1C;
	if (rxh < 0x03)
		rxh = 0x03;

	uint32_t ryh = s&0x1F;

	if (ryh >= 0x18)
	{
		ryh &= 0x07;
		ryh <<= 1;
		ryh += 4;

		if ( (int)ryh >= pGS->num_asteroids )
			res = 0;
	}

	if (ryh >= 0x15)
		ryh = 0x14;
	if (ryh < 0x03)
		ryh = 0x03;

	pr->x = (int)( (rxh<<8) | (pGS->ship.r.x & 0xFF) );
	pr->y = (int)( (ryh<<8) | ((pGS->ship.r.y-128*8) & 0xFF) ) + 128*8;

	//restor rgen
	pGS->rgen = rgen_temp;

	return res;
}


void CycleRandomGenerator( 
	GameState_t *pGS, 
	int time,
	int nCycles)
{
	uint16_t * const ps = &(pGS->rgen.s);

	for (int n=0; n<nCycles; n++)
		*ps = (*ps<<1) | ( ((*ps>>14) ^ *ps)&1 );

	pGS->rgen.ss[time&0x07] = *ps;
}


static void InitRandomGenerator( 
	GameState_t *pGS, 
	int time )
{
	if ( (pGS->t0_level<=0) || 
		 ( (time - pGS->t0_level) != 8 ) )
		 return;

	if ( (pGS->num_asteroids<4) || (pGS->num_asteroids>15) )
		return;

	randgen_t * const prgen = &pGS->rgen;

	if (prgen->s != 0)
		return;

	int8_t sr[128];
	memset(sr, -1, sizeof(sr) );

	int k = 7*(pGS->num_asteroids);
	
	for (int i=0; i<(pGS->num_asteroids); i++)
	{
		asteroid_t * const pA = &(pGS->asteroids[i]);
		int vx, vy, pos;

		if (!( pA->flags & IS_TRACKED ))
		{
			ASSERT_MSG(!"asteroids not locked");
			return;
		}

		//get bits from shape
		k-=1;
		sr[k+4] = ((pA->shape)>>1)&1;
		sr[k+3] = ((pA->shape)>>0)&1;

		// get bits from vx
		k-=1;

		vx = pA->v.x;

		sr[k+7] = (vx>>7)&1;
		sr[k+3] = (vx>>3)&1;

		if ( (vx<-6) || (vx>6) )
		{
			sr[k+2] = (vx>>2)&1;
			sr[k+1] = (vx>>1)&1;
		}

		//get bits from vy
		k-=4;

		vy = pA->v.y;

		sr[k+3] = (vy>>3)&1;

		//get bits from x or y position
		k-=1;

		vector_t r = pA->r;
		vmulsub( &r, pA->v, 9 );
		normalize_r( &r );

		//save some info for debugging
		prgen->asteroids_init[i].shape = pA->shape;
		prgen->asteroids_init[i].r = r;
		prgen->asteroids_init[i].v = pA->v;

		r.y -= 128*8;

		// try to adjust the last three bits, because they are not 100% accurate
		if ( pA->v.x & 1 )
			; //ok
		else if ( pA->v.x & 2 )
			r.x &= ~0x01;
		else if ( pA->v.x & 4 )
			r.x &= ~0x03;
		else
			r.x &= ~0x07;

		if ( pA->v.y & 1 )
			; //ok
		else if ( pA->v.y & 2 )
			r.y &= ~0x01;
		else if ( pA->v.y & 4 )
			r.y &= ~0x03;
		else
			r.y &= ~0x07;

		if (r.x != 0)
		{
			pos = r.x>>8;
			sr[k+0] = 0;
		}
		else if (r.y != 0)
		{
			pos = r.y>>8;
			sr[k+0] = 1;
		}
		else
		{
			pos = 0;
		}

		//adjust the position of the asteroids
		r.y += 128*8;

		if ((r.x==0) && (r.y!=0))
		{
			if ( pA->v.y & 1 )
				; //ok
			else if ( pA->v.y & 2 )
				r.y |= (pA->r_last.y & 1);
			else if ( pA->v.y & 4 )
				r.y |= (pA->r_last.y & 3);
			else
				r.y |= (pA->r_last.y & 7);

			pA->r.x = 0;
			pA->r.y = r.y;

			vmuladd( &pA->r, pA->v, 9 );
			normalize_r( &r );
		}
		else if ((r.y==0) && (r.x!=0))
		{
			if ( pA->v.x & 1 )
				; //ok
			else if ( pA->v.x & 2 )
				r.x |= (pA->r_last.x & 1);
			else if ( pA->v.x & 4 )
				r.x |= (pA->r_last.x & 3);
			else
				r.x |= (pA->r_last.x & 7);

			pA->r.x = r.x;
			pA->r.y = 0;

			vmuladd( &pA->r, pA->v, 9 );
			normalize_r( &pA->r );
		}

		//verify the adjustment

		//..todo

		sr[k+5] = (pos>>4)&1;

		sr[k+3] = (pos>>2)&1;
		sr[k+2] = (pos>>1)&1;
		sr[k+1] = (pos>>0)&1;

		if (sr[k+15] >= 0)
			sr[k+0] = ( sr[k+1] + sr[k+15] )&1;

		if ( (sr[k+6+1] >= 0) && (sr[k+6+15] >= 0) )
			sr[k+6] = ( sr[k+6+1] + sr[k+6+15] )&1;
		else if ( (sr[k+6-1] >= 0) && (sr[k+6+14] >= 0) )
			sr[k+6] = ( sr[k+6-1] + sr[k+6+14] )&1;
	}


	uint16_t s = 0;

	for (k=15; k>=0; k--)
	{
		s <<= 1;

		if (sr[k]<0)
		{
			s = 0;
			break;
		}

		s |= sr[k];
	}

	for (k=0; k<8; k++)
		s = (s<<1) | ( ((s>>14) ^ s)&1 );

	pGS->rgen.s = s;
}


static void	UpdateVelocityAndPosition( 
	shot_t *pA, 
	shot_t *pB, 
	int time,
	uint32_t flags )
{
	if ( (pA->flags&IS_TRACKED) || (pA->flags&IS_ESTIMATED) )
	{
		if (  ( (pA->r.x&(~0x07)) != pB->r.x ) ||
			  ( (pA->r.y&(~0x07)) != pB->r.y )    )
		{
			ASSERT_MSG( !"prediction failed!" || (flags&IS_SAUCER) || (!(pA->flags&IS_TRACKED)) || (pA->flags&IS_INITIALIZED) );
			pA->flags &= ~(IS_TRACKED | IS_ESTIMATED);
			pA->t0 = time - 1;
		}
	}

	if ( (!(pA->flags&IS_TRACKED)) && 
		 ( (time - pA->t0) >= 8) )
	{
		int is_tracked = 1;

		//velocity
		pA->v.x = normalize_dx( pB->r.x - pA->rr[time&0x07].x )>>3;
		pA->v.y = normalize_dy( pB->r.y - pA->rr[time&0x07].y )>>3;

		//x-position
		int n = 0;
		int pos;
		
		pos = pA->r.x & 0x07;

		for (int p=0; p<8; p++)
		{
			n = 0;
			while (1)
			{
				if ( 0 != normalize_dx( pA->rr[(time+n)&0x07].x - (( pB->r.x + pos - (8-n) * pA->v.x )&(~0x07)) ) )
					break;

				n++;
				if (n==8)
					break;
			}

			if (n==8)
			{
				pA->r.x = pB->r.x + pos;
				break;
			}

			pos = (pos+1)&0x07;
		}

		if (n!=8)
			is_tracked = 0;

		//y-position

		pos = pA->r.y & 0x07;

		for (int p=0; p<8; p++)
		{
			n = 0;
			while (1)
			{
				if ( 0 != normalize_dy( pA->rr[(time+n)&0x07].y - (( pB->r.y + pos - (8-n) * pA->v.y )&(~0x07)) ) )
					break;

				n++;
				if (n==8)
					break;
			}

			if (n==8)
			{
				pA->r.y = pB->r.y + pos;
				break;
			}

			pos = (pos+1)&0x07;
		}

		if (n!=8)
			is_tracked = 0;

		if (is_tracked)
		{
			pA->flags |= IS_TRACKED;
		}
		else
		{
			//ASSERT_MSG( !"locking failed"  || (flags&IS_SAUCER) );
			pA->t0 = time - 1;
			pA->flags &= ~IS_TRACKED;
		}
	}

	if (!(pA->flags&IS_TRACKED) && !(pA->flags&IS_ESTIMATED) )
	{
		int time_delta = time - pA->t0;

		ASSERT_MSG( time_delta<8 );

		//update velocity
		if ( time_delta > 0 )
		{
			pA->v.x = normalize_dx(pB->r.x - pA->rr[(pA->t0)&0x07].x) / time_delta;
			pA->v.y = normalize_dy(pB->r.y - pA->rr[(pA->t0)&0x07].y) / time_delta;
		
			if (pA->flags&IS_SAUCER)
				pA->flags |= IS_TRACKED;
		}

		//set current position
		pA->r = pB->r;
	}

	//save current position
	pA->rr[time&0x07] = pB->r;
}


static void limit_velocity( int *pxy )
{
	if (*pxy >= 0)
	{
		if (*pxy > 31)
			*pxy = 31;
		else if (*pxy < 6)
			*pxy = 6;
	}
	else
	{
		if (*pxy < -31)
			*pxy = -31;
		else if (*pxy > -6)
			*pxy = -6;
	}
}


static void AddNewAsteroids( 
	GameState_t *pGS,
	asteroid_t *pC,
	int time )
{
	int j=0;

	if (pC->state<=1)
		return;

	//add two new asteroids
	for (int n=0; n<2; n++)
	{
		for (; j<MAX_NUM_ASTEROIDS; j++)
		{
			if (pGS->asteroids[j].state==0)
				break;
		}

		if (j>=MAX_NUM_ASTEROIDS)
			break;

		asteroid_t * const pA = &(pGS->asteroids[j]);
		memset( pA, 0x00, sizeof(asteroid_t) );

		CycleRandomGenerator( pGS, time, 6 );
		uint16_t s = pGS->rgen.s >> 1;

		//state, size
		if (pC->state == 4)
			pA->state = 2;
		else
			pA->state = 1;

		//shape
		pA->shape = ( s >> 8 )&3;

		//velocity
		pA->v = pC->v;

		int8_t ax = ( s >> 4 )&0x8F;
		if ( ax<0 )
			ax |= 0xF0;

		pA->v.x += (int)ax;
		limit_velocity( &(pA->v.x) );

		int8_t ay = s & 0x8F;
		if ( ay<0 )
			ay |= 0xF0;

		pA->v.y += (int)ay;
		limit_velocity( &(pA->v.y) );

		//position	(may not 100% exact because last 3 bit of old position is not exactly known)
		pA->r = pC->r;

		if (n==0)
			pA->r.x ^= (( pA->v.x & 0x1F )<<1);
		else
			pA->r.y ^= (( pA->v.y & 0x1F )<<1);

		pA->t0 = time;
		pA->flags = IS_ASTEROID | IS_ESTIMATED;

		vector_t rr = pA->r;
		rr.x &= ~0x07;
		rr.y &= ~0x07;

		pA->rr[time&0x07] = rr;

		pGS->num_asteroids += 1;
		j++;
	}
}


static void UpdateAsteroidsState( 
	GameState_t *pGS, 
	ScreenObjects_t *pSO,
	int time )
{
	//calculate new (predicted) position 
	//and update status
	for (int j=0; j<MAX_NUM_ASTEROIDS; j++)
	{
		asteroid_t * const pA = &(pGS->asteroids[j]);
		
		if (pA->state<=0)
			continue;

		vadd( &pA->r, pA->v );
		normalize_r( &(pA->r) );
	}

	if (pSO==NULL)
		return;

	ScreenObjects_t::asteroid_tag * const pBB = pSO->asteroids;
	const int NumAsteroidsAndExplosions = pSO->num_asteroids + pSO->num_explosions;

	if (pGS->t0_level==0)
		return;

	//at level start, init list
	if (pGS->t0_level == time)
	{
		//save the current position
		for (int j=0; j<MAX_NUM_ASTEROIDS; j++)
		{
			asteroid_t * const pA = &(pGS->asteroids[j]);

			pA->r_last = pA->r;
			pA->v_last = pA->v;
		}

		pGS->num_asteroids = 0;
		pGS->list_sync_lost = 0;

		for (int i=0; i<NumAsteroidsAndExplosions; i++)
		{
			ScreenObjects_t::asteroid_tag * const pB = &(pBB[i]);

			if (pB->shape>=0)
			{
				asteroid_t * const pA = &(pGS->asteroids[pGS->num_asteroids]);
				vector_t r_last = pA->r_last;
				vector_t v_last = pA->v_last;

				memset( pA, 0x00, sizeof(asteroid_t) );
				pA->r_last = r_last;
				pA->v_last = v_last;

				pA->r     = pB->r;
				pA->shape = pB->shape;
				pA->state = pB->state;

				pA->t0 = time;

				pA->flags = IS_INITIALIZED;

				pGS->num_asteroids += 1;
			}
		}

		//check against the random generator
		int is_prediction_valid = 1;
		int is_rgen_valid = 1;
		for (int j=0; j<MAX_NUM_ASTEROIDS; j++)
		{
			asteroid_t * const pA = &(pGS->asteroids[j]);
			if (pA->state<=0)
				continue;

			CycleRandomGenerator( pGS, time, 7 );
			uint16_t s = pGS->rgen.s;

			//check shape
			if ( pA->shape != ((s>>9)&3) )
				is_rgen_valid = 0;

			//velocity
			int8_t ax = (s>>5)&0x8F;
			if ( ax<0 )
				ax |= 0xF0;

			int vx = (int)ax;
			limit_velocity( &vx );

			int8_t ay = (s>>1) & 0x8F;
			if ( ay<0 )
				ay |= 0xF0;

			int vy = (int)ay;
			limit_velocity( &vy );

			pA->v.x = vx;
			pA->v.y = vy;

			//position
			int rx=0; 
			int ry=0;

			uint32_t pos = (s>>1) & 0x1F;
			
			if ( s&1 )
			{
				if (pos>0x17)
					pos &= 0x17;

				ry = (int)( (pos<<8) | ( (pA->r_last.y-128*8) & 0xFF) );
			}
			else
			{
				rx = (int)( (pos<<8) | (pA->r_last.x & 0xFF) );
			}

			rx = normalize_rx( rx + vx );
			ry = normalize_ry( ry + 128*8 + vy );

			int dx = normalize_dx( rx - pA->r.x ) >> NUM_BITS_FRACTION;
			int dy = normalize_dy( ry - pA->r.y ) >> NUM_BITS_FRACTION;
			int sqr_d = dx*dx + dy*dy;

			if (sqr_d>2)
				is_prediction_valid = 0;

			if (sqr_d>100)
				is_rgen_valid = 0;

			if (!is_rgen_valid)
				is_prediction_valid = 0;

			pA->r.x = rx;
			pA->r.y = ry;

			//set flags
			pA->flags |= IS_TRACKED;
		}

		#ifndef RELEASE_CONTEST
		//memcpy( pGS->asteroids_predicted, pGS->asteroids, sizeof(pGS->asteroids) );
		#endif

		//failed?
		if (!is_prediction_valid)
		{
			for (int j=0; j<MAX_NUM_ASTEROIDS; j++)
			{
				asteroid_t * const pA = &(pGS->asteroids[j]);
				if (pA->state<=0)
					continue;

				//clear flags
				pA->flags = 0;
			}
		}

		if (!is_rgen_valid)
		{
			//clear random generator
			memset( &pGS->rgen, 0x00, sizeof(pGS->rgen) );
		}

	}

	//update list: check if asteroids have exploded
	{
		int i_min = (pGS->saucer.state<0) ? 1 : 0;

		for (int i=i_min; i<NumAsteroidsAndExplosions; i++)
		{
			ScreenObjects_t::asteroid_tag * const pB = &(pBB[i]);
			asteroid_t * pC = NULL;

			if ( (pB->shape==-2) && (pB->state==-11) )
			{
				//if a big or mid size asteroid has exploded, add two new asteroids
				int k;

				for (k=0; k<MAX_NUM_ASTEROIDS; k++)
				{
					pC = &(pGS->asteroids[k]);

					if ( (pC->state<=0) || (pC->state==1) )
						continue;

					vector_t u = pC->rr[(time-1)&0x07];
					vsub( &u, pB->r );
					normalize_d( &u );

					if ( ((u.x>>NUM_BITS_FRACTION)==0) &&
						 ((u.y>>NUM_BITS_FRACTION)==0) )
						break;
				}

				if (k>=MAX_NUM_ASTEROIDS)
					continue;

				asteroid_t ast = *pC;
				vsub( &ast.r, ast.v );	//revert update of position

				AddNewAsteroids( pGS, &ast, time );
			}
		}
	}


	//state of explosions
	for (int j=0; j<MAX_NUM_ASTEROIDS; j++)
	{
		asteroid_t * const pA = &(pGS->asteroids[j]);
		
		if (pA->state<0)
			pA->state++;
	}


	//update state of each asteroid
	{
		int i_bad = -1;
		int j_bad = 0;

		int rgen_resynced = 0;

		int j=0;
		int i=0;

		if (pGS->saucer.state<0)
			i=1;

		for (; i<NumAsteroidsAndExplosions; i++)
		{
			ScreenObjects_t::asteroid_tag * const pB = &(pBB[i]);

			for(; j<MAX_NUM_ASTEROIDS; j++)
				if (pGS->asteroids[j].state!=0)	break;

			if (j>=MAX_NUM_ASTEROIDS)
			{
				if (pB->shape < 0)
					continue;

				i_bad = 0;
				j_bad = 0;
				break;
			}

			asteroid_t * const pA = &(pGS->asteroids[j]);

			if (pA->state<0)
			{
				if (pB->shape>=0)
				{
					ASSERT_MSG( !"pB->shape >= 0" );
					ADD_DBGTEXT("pB->shape >= 0");

					i_bad = i;
					j_bad = j;
					break;
				}
			}
			else if (pB->shape < 0)
			{
				pA->state = -37;

				vsub( &pA->r, pA->v );	//revert update of position
				normalize_r( &(pA->r) );

				pGS->num_asteroids -= 1;
			}
			else if ( pA->t0 == time )
			{
				int prediction_failed = 1;
				int do_rgen_resync = 0;

				if (!(pA->flags&IS_INITIALIZED))
					pA->flags = 0;

				if (pA->flags&IS_INITIALIZED)
					prediction_failed = 0;

				if ( (pGS->rgen.s != 0) && (!(pA->flags&IS_INITIALIZED)) )
				{
					vadd( &pA->r, pA->v );
					normalize_r( &pA->r );

					//verify position shape and size
					int dx = normalize_dx( pA->r.x - pB->r.x ) >> NUM_BITS_FRACTION;
					int dy = normalize_dy( pA->r.y - pB->r.y ) >> NUM_BITS_FRACTION;
					int sqr_d = dx*dx + dy*dy;

					if ( (pA->shape == pB->shape) &&
						 (pA->state == pB->state) )
					{
						prediction_failed = 0;
					}
					else if (pA->state == pB->state)
					{
						do_rgen_resync = 1;
					}

					if ( sqr_d == 0 )
					{
						pA->flags |= IS_ESTIMATED;
					}
					else
					{
						if (dx!=0)
							pA->r.x = pB->r.x;

						if (dy!=0)
							pA->r.y = pB->r.y;
					}
				}

				if (prediction_failed)
				{
					pA->r     = pB->r;
					pA->shape = pB->shape;
					pA->state = pB->state;

					if ( do_rgen_resync && (!rgen_resynced) )
					{
						CycleRandomGenerator( pGS, time, 1 );
						rgen_resynced = 1;

						pGS->n_rgen_resync += 1;
						if (pGS->n_rgen_resync > 5)
						{
							pGS->rgen.s = 0;
							pGS->n_rgen_resync = 0;
						}
					}
				}

				pA->rr[time&0x07] = pB->r;

				pA->id = pGS->next_asteroid_id;
				pGS->next_asteroid_id += 1;

				pA->flags |= IS_ASTEROID;
			}
			else
			{
				int list_missmatch = 0;

				//check size and shape
				if ( pA->shape != pB->shape )
				{
					list_missmatch = 1;
					ASSERT_MSG(!"list missmatch 1");
					ADD_DBGTEXT("list missmatch 1");
				}

				//check distance to predicted position
				int dx = normalize_dx( pA->r.x - pB->r.x ) >> NUM_BITS_FRACTION;
				int dy = normalize_dy( pA->r.y - pB->r.y ) >> NUM_BITS_FRACTION;
				int sqr_d = dx*dx + dy*dy;

				if ( (pA->flags&IS_TRACKED) && (sqr_d>0) && (!(pA->flags&IS_INITIALIZED)) )
				{
					list_missmatch = 2;
					ASSERT_MSG(!"list missmatch 2");
					ADD_DBGTEXT("list missmatch 2");
				}

				if (sqr_d >= 100)
				{
					list_missmatch = 3;
					ASSERT_MSG(!"list missmatch 3");
					ADD_DBGTEXT("list missmatch 3");
				}

				if (pA->state != pB->state)
				{
					list_missmatch = 4;
					ASSERT_MSG(!"list missmatch 4");
					ADD_DBGTEXT("list missmatch 4");
				}

				if (list_missmatch >= 3)
				{
					i_bad = i;
					j_bad = j;
					break;
				}

				UpdateVelocityAndPosition( 
					(shot_t*)pA, 
					(shot_t*)pB, 
					time, 
					IS_ASTEROID );
			}

			j++;
		}

		if (i_bad>=0)
		{
			if ( (NumAsteroidsAndExplosions - i_bad ) >= (MAX_NUM_ASTEROIDS - j_bad) )
			{
				i_bad = 0;
				j_bad = 0;
			}

			i = i_bad;
			j = j_bad;

			for (; i<NumAsteroidsAndExplosions; i++)
			{
				ScreenObjects_t::asteroid_tag * const pB = &(pBB[i]);
				asteroid_t * const pA = &(pGS->asteroids[j]);

				pA->t0    = time;
				pA->r     = pB->r;
				pA->flags = IS_ASTEROID;

				if (pB->shape>=0)
				{
					pA->shape = pB->shape;
					pA->state = pB->state;

					pA->id = pGS->next_asteroid_id;
					pGS->next_asteroid_id += 1;

					pA->rr[time&0x07] = pB->r;
				}
				else
				{
					pA->state = -19;
				}

				j++;
			}
		}

		//if there are asteroids in our list which do not exist, remove them
		for(; j<MAX_NUM_ASTEROIDS; j++)
		{
			asteroid_t * const pA = &(pGS->asteroids[j]);
			if (pA->state<=0)
				continue;

			ASSERT_MSG(!"list missmatch 5");
			ADD_DBGTEXT("list missmatch 5");
			pA->state = 0;
			pGS->num_asteroids--;

			if (pGS->num_asteroids<0)
				pGS->num_asteroids = 0;
		}
	}


	//if no asteroids are on the screen remove all asteroids from the list
	if (pSO->num_asteroids==0)
	{
		for (int j=0; j<MAX_NUM_ASTEROIDS; j++)
			pGS->asteroids[j].state = 0;

		pGS->num_asteroids = 0;
		pGS->list_sync_lost = 0;
	}
}


static void UpdateShotsState( 
	GameState_t *pGS, 
	ScreenObjects_t *pSO,
	int time )
{
	//calculate new (predicted) position
	for (int j=0; j<pGS->num_shots; j++)
	{
		shot_t * const pA = &(pGS->shots[j]);

		pA->r.x += pA->v.x;
		pA->r.y += pA->v.y;
		normalize_r( &(pA->r) );
	}

	if (pSO==NULL)
		return;

	ScreenObjects_t::shot_tag * const pBB = pSO->shots;
	const int NumShots = pSO->num_shots;

	int angle_code_ship = -1;

	//find corresponding object on the screen
	for (int i=0; i<NumShots; i++)
	{
		int j_min = 0;
		int sqr_dmin = (1<<30);

		shot_t * const pA = &(pGS->shots[i]);
		ScreenObjects_t::shot_tag * const pB = &(pBB[i]);

		for (int j=i; j<pGS->num_shots; j++)
		{
			shot_t * const pC = &(pGS->shots[j]);

			//check distance to predicted position
			int dx = normalize_dx( pC->r.x - pB->r.x ) >> NUM_BITS_FRACTION;
			int dy = normalize_dy( pC->r.y - pB->r.y ) >> NUM_BITS_FRACTION;

			int sqr_d = dx*dx + dy*dy;

			if ( (pC->flags&IS_TRACKED) && (sqr_d>0) )
				continue;

			if (sqr_d < sqr_dmin)
			{
				sqr_dmin = sqr_d;
				j_min = j;
			}
		}

		int time_delta = time - pGS->shots[j_min].t0;

		if (  ( (sqr_dmin<=2) && ( time_delta >1 ) ) ||
			  ( (sqr_dmin<=200) && ( time_delta<=1 ) )    )
		{
			//reorder the list
			if (i != j_min)
			{
				shot_t A = *pA;
				*pA = pGS->shots[j_min];
				pGS->shots[j_min] = A;
			}

			UpdateVelocityAndPosition( 
				(shot_t*)pA, 
				(shot_t*)pB, 
				time, 
				IS_SHOT );

			if ( ( pA->flags&IS_TRACKED ) && 
				  (pA->flags&IS_SHIP_SHOT)  &&
				  !(pGS->ship.flags&IS_TRACKED) )
			{
				vector_t v0;
				v0.x = pA->v.x - pGS->ship.v.x;
				v0.y = pA->v.y - pGS->ship.v.y;

				int angle_code = getAngleCode( v0 );
				if (angle_code<0)
				{
					ASSERT_MSG( !"angle detection failed!" );
				}
				else
				{
					angle_code_ship = ( pGS->ship.angle_code - pA->angle_code + angle_code) & 0xFF;
				}
			}
		}
		else if (pGS->num_shots < MAX_NUM_SHOTS)
		{
			//add new object
			if (i != pGS->num_shots)
			{
				//copy position "i" to "end+1"
				pGS->shots[pGS->num_shots] = *pA;
			}

			pGS->num_shots++;

			memset( pA, 0x00, sizeof(shot_t) );

			pA->r =  pB->r;
			pA->rr[time&0x07] = pB->r;
			pA->t0 = time;

			pA->flags |= IS_SHOT;

			//if fire was pressed check if it is from us
			if ( (pGS->keysinfo_curr.keys & KEY_FIRE) && 
				 (pGS->ship.state>0) )
			{
				vector_t r;

				r.x = pGS->ship.r.x + ((5*pGS->ship.v0.x)>>1);
				r.y = pGS->ship.r.y + ((5*pGS->ship.v0.y)>>1);
				normalize_r( &r );

				int dx = normalize_dx( r.x - pA->r.x ) >> NUM_BITS_FRACTION;
				int dy = normalize_dy( r.y - pA->r.y ) >> NUM_BITS_FRACTION;
				int sqr_d = dx*dx + dy*dy;

				if ( (sqr_d <= (3*3)) || 
					 ( !(pGS->ship.flags&IS_TRACKED) ) )
				{
					pA->flags |= IS_SHIP_SHOT;
				}

				if ( pA->flags & IS_SHIP_SHOT)
				{
					pA->angle_code = pGS->ship.angle_code;
					pA->id = pGS->keysinfo_curr.id_shot;
				}

				if ( (pA->flags & IS_SHIP_SHOT) && (pGS->ship.flags & IS_TRACKED) )
				{
					pA->r.x = r.x;
					pA->r.y = r.y;
					pA->v.x = pGS->ship.v.x + pGS->ship.v0.x;
					pA->v.y = pGS->ship.v.y + pGS->ship.v0.y;

					pA->flags |= IS_TRACKED;
				}
			}

			//a saucer shot increases the random generator
			if (!(pA->flags&IS_SHIP_SHOT))
			{
				CycleRandomGenerator( pGS, time, 1 );
			}
		}
		else
		{
			ASSERT_MSG( !"too many shots" );
		}
	}

	if (angle_code_ship >= 0)
	{
		pGS->ship.angle_code = angle_code_ship;
		pGS->ship.flags |= IS_TRACKED;
	}

	//remove destroyed objects
	pGS->num_shots = NumShots;
}


static void UpdateSaucerState( 
	GameState_t *pGS, 
	ScreenObjects_t *pSO,
	int time )
{
	saucer_t * const pA = &(pGS->saucer);

	if (pA->state>0)
	{
		//calculate new (predicted) position
		pA->r.x += pA->v.x;
		pA->r.y += pA->v.y;
		normalize_r( &(pA->r) );
	}

	if (pA->state<0)
		pA->state++;

	if (pSO==NULL)
		return;

	ScreenObjects_t::saucer_tag * const pB = &(pSO->saucer);

	if (pB->state<=0)
	{
		//if saucer disappears, check if it was destroyed
		if (pA->state>0)
		{
			pA->state = 0;

			if  ( (pSO->num_explosions>0) && 
				  (pSO->asteroids[0].shape==-2) && 
				  (pSO->asteroids[0].state==-11) )
			{
				saucer_t * const pA = &(pGS->saucer);

				vector_t u = pA->rr[(time-1)&0x07];
				vsub( &u, pSO->asteroids[0].r );
				normalize_d( &u );

				if ( ((u.x>>NUM_BITS_FRACTION)==0) &&
					 ((u.y>>NUM_BITS_FRACTION)==0) )
					pA->state = -37;
			}
		}

		return;
	}

	if (pA->state>0)
	{
		if ( (time - pA->t0) >= 2 )
		{
			//check if velocity has changed
			if (  ( (pA->r.x&(~0x07)) != pB->r.x ) ||
				  ( (pA->r.y&(~0x07)) != pB->r.y )    )
			{
				pGS->phi128_offset = time;
			}
		}

		if ( (pGS->phi128_offset>0) && 
			 ( ((time - pGS->phi128_offset)&0x7F) == 0 ) )
			CycleRandomGenerator( pGS, time, 1 );
	}

	if (pA->state==0)
	{
		memset( pA, 0x00, sizeof(saucer_t) );

		pA->state = pB->state;
		pA->t0 = time;

		pA->flags |= IS_SAUCER;

		CycleRandomGenerator( pGS, time, 1 );
	}

	UpdateVelocityAndPosition( 
		(shot_t*)pA, 
		(shot_t*)pB, 
		time, 
		IS_SAUCER );
}


static void UpdateShipState( 
	GameState_t *pGS, 
	ScreenObjects_t *pSO,
	int time )
{
	ship_t * const pA = &pGS->ship;
	int is_present = pSO->is_ship_present;

	if (!is_present)
	{
		pGS->ship.state = 0;
		return;
	}

	if (pGS->keysinfo_next.keys & KEY_HYPERSPACE)
		CycleRandomGenerator( pGS, time, 6 );

	if (pGS->ship.state<=0)
	{
		pGS->ship.state = 1;
		pGS->t0_ship = time;

		pA->flags &= IS_TRACKED;	//clear all but not tracked flag

		pA->r = pSO->ship.r;		//(524*8,524*8)

		pA->v.x = 0;
		pA->v.y = 0;

		pA->flags |= IS_SHIP;
	}

	//calculate new (predicted) position
	vadd( &pA->r, pA->v );
	normalize_r( &(pA->r) );

	if (pGS->keysinfo_curr.keys & KEY_LEFT)
		pA->angle_code += 3;
	
	if (pGS->keysinfo_curr.keys & KEY_RIGHT)
		pA->angle_code -= 3;

	pA->angle_code &= 0xFF;
	pA->v0 = v0_shots[pA->angle_code];

	ScreenObjects_t::ship_tag * const pB = &(pSO->ship);

	if ( !pA->is_init_angle )
	{
		pA->angle_code = getNearestAngleCode( pB->d );
		pA->v0 = v0_shots[pA->angle_code];

		pA->is_init_angle = 1;
	}
}



#define PUT_NEW_ASTEROID(vecx, vecy, vecs, shape_) \
	if (pScreenObj->num_asteroids<MAX_NUM_ASTEROIDS) \
	{ \
		int NumAsteroidsAndExplosions = pScreenObj->num_asteroids + pScreenObj->num_explosions; \
		pScreenObj->asteroids[NumAsteroidsAndExplosions].r.x = vecx<<NUM_BITS_FRACTION; \
		pScreenObj->asteroids[NumAsteroidsAndExplosions].r.y = vecy<<NUM_BITS_FRACTION; \
		if (vecs==0) \
			pScreenObj->asteroids[NumAsteroidsAndExplosions].state = 4; \
		else if (vecs==15) \
			pScreenObj->asteroids[NumAsteroidsAndExplosions].state = 2; \
		else if (vecs==14) \
			pScreenObj->asteroids[NumAsteroidsAndExplosions].state = 1; \
		else \
			pScreenObj->asteroids[NumAsteroidsAndExplosions].state = -vecs; \
		pScreenObj->asteroids[NumAsteroidsAndExplosions].shape = shape_; \
		if (shape_>=0) \
			pScreenObj->num_asteroids++; \
		else \
			pScreenObj->num_explosions++; \
	} \
	else \
	{ \
		ASSERT_MSG( !"too many asteroids" ); \
	}

#define ADD_DIGIT_SCORE(d) \
	if ( parse_score_state < 2 ) {\
		parse_score_state = 1; \
		pScreenObj->score = pScreenObj->score*10 + d; } 

void DecodeVectorRam( ScreenObjects_t *pScreenObj, uint8_t *pVectorRam )
{
	uint16_t * const vector_ram = (uint16_t*)pVectorRam;

	int dx, dy, sf, vecz;
	int vecx=0; int vecy=0; int vecs=0;
	int vec1x = 0;
	int vec1y = 0;
	int shipdetect = 0;
	int parse_score_state = 0;

	//clear screen objects
	memset( pScreenObj, 0x00, sizeof(ScreenObjects_t) );

	//check vector ram
	if (vector_ram[0] == 0xe001)
		pScreenObj->phi2 = 0;
	else if (vector_ram[0] == 0xe201)
		pScreenObj->phi2 = 1;
	else 
		return;

	//decode vector ram
	int pc = 1;
	while (pc < 512)
	{
		int op = vector_ram[pc] >> 12;

		switch (op)
		{

		case 0xa: // LABS
			vecy = vector_ram[pc  ] & 0x3ff;
			vecx = vector_ram[pc+1] & 0x3ff;
			vecs = vector_ram[pc+1] >> 12;

			if (parse_score_state == 1)
				parse_score_state = 2;

			break;

		case 0xb: // HALT
			return;

		case 0xc: // JSRL
			switch (vector_ram[pc] & 0xfff)
			{
			case 0x8f3:
				PUT_NEW_ASTEROID( vecx, vecy, vecs, 0 );
				break;

			case 0x8ff:
				PUT_NEW_ASTEROID( vecx, vecy, vecs, 1 );
				break;

			case 0x90d:
				PUT_NEW_ASTEROID( vecx, vecy, vecs, 2 );
				break;

			case 0x91a:
				PUT_NEW_ASTEROID( vecx, vecy, vecs, 3 );
				break;

			case 0x880:
				PUT_NEW_ASTEROID( vecx, vecy, vecs, -4 );
				break;

			case 0x896:
				PUT_NEW_ASTEROID( vecx, vecy, vecs, -3 );
				break;

			case 0x8B5:
				PUT_NEW_ASTEROID( vecx, vecy, vecs, -2 );
				break;

			case 0x8D0:
				PUT_NEW_ASTEROID( vecx, vecy, vecs, -1 );
				break;

			case 0x929:
				pScreenObj->saucer.r.x = vecx<<NUM_BITS_FRACTION;
				pScreenObj->saucer.r.y = vecy<<NUM_BITS_FRACTION;
				pScreenObj->saucer.state = ( vecs == 15 ? 2 : 1 );
				break;

			case 0xadd:    // O
				ADD_DIGIT_SCORE(0);
				break;

			case 0xb2e:    // 1
				ADD_DIGIT_SCORE(1);
				break;

			case 0xb32:    // 2
				ADD_DIGIT_SCORE(2);
				break;

			case 0xb3a:    // 3
				ADD_DIGIT_SCORE(3);
				break;

			case 0xb41:    // 4
				ADD_DIGIT_SCORE(4);
				break;

			case 0xb48:    // 5
				ADD_DIGIT_SCORE(5);
				break;

			case 0xb4f:    // 6
				ADD_DIGIT_SCORE(6);
				break;

			case 0xb56:    // 7
				ADD_DIGIT_SCORE(7);
				break;

			case 0xb5b:    // 8
				ADD_DIGIT_SCORE(8);
				break;

			case 0xb63:    // 9
				ADD_DIGIT_SCORE(9);
				break;

			case 0xA9B: //'E'
				pScreenObj->is_E_visible = 1;
				break;

			}  
			break;

		case 0xd: // RTSL
			return;

		case 0xe: // JMPL
			return;

		case 0xf: // SVEC
			break;

		default:
			dy = vector_ram[pc] & 0x3ff;
			if ((vector_ram[pc] & 0x400) != 0)
				dy = -dy;

			dx = vector_ram[pc+1] & 0x3ff;
			if ((vector_ram[pc+1] & 0x400) != 0)
				dx = -dx;

			sf = op;
			vecz = vector_ram[pc+1] >> 12;
			if ( (dx == 0) && 
				 (dy == 0) && 
				 (vecz == 15) )
			{
				if (pScreenObj->num_shots < MAX_NUM_SHOTS)
				{
					pScreenObj->shots[pScreenObj->num_shots].r.x = vecx<<NUM_BITS_FRACTION;
					pScreenObj->shots[pScreenObj->num_shots].r.y = vecy<<NUM_BITS_FRACTION;
					pScreenObj->num_shots++;
				}
				else
				{
					ASSERT_MSG( !"too many shots" );
				}
			}

			if ( (op   ==  6) && 
				 (vecz == 12) && 
				 (dx != 0)    && 
				 (dy != 0)       )
			{
				switch (shipdetect)
				{
				case 0:
					vec1x = dx;
					vec1y = dy;
					shipdetect++;
					break;

				case 1:
					pScreenObj->is_ship_present = 1;
					pScreenObj->ship.r.x = vecx<<NUM_BITS_FRACTION;
					pScreenObj->ship.r.y = vecy<<NUM_BITS_FRACTION;
					pScreenObj->ship.d.x = vec1x - dx;
					pScreenObj->ship.d.y = vec1y - dy;
					shipdetect++;
					break;
				}
			}
			else if (shipdetect == 1)
			{
				shipdetect = 0;
			}

			break;
		}

		if (op <= 0xa)
			++pc;

		if (op != 0xe) // JMPL
			++pc;
	}   
}


static int getNearestAngleCode( vector_t v )
{
	int i_arg_max = 0;
	int z_max = 0;

	for (int i=0; i<256; i++)
	{
		vector_t u = v0_shots[i];

		int z = vscalarprod( u, v );
		if (z>z_max)
		{
			z_max = z;
			i_arg_max = i;
		}
	}

	return i_arg_max;
}


static int getAngleCode( vector_t v )
{
	for (int i=0; i<256; i++)
	{
		vector_t u = v0_shots[i];
		vsub( &u, v );

		if ( (u.x==0) && (u.y==0) )
			return i;
	}

	return -1;
}


