/*****************************************************************************
*  ADBotPlayerV1                                                             *
*                                                                            *
*  Copyright (C) 2008 Andreas Dittrich                                       *
*                                                                            *
*  Author:  Andreas Dittrich                                                 *
*           Am Schimmelberg 29                                               *
*           67729 Sippersfeld (Germany)                                      *
*****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <winsock2.h>

#include "inttypes.h"

#include "../interface/ADBotPlayerV1.h"
#include "ADBotPlayerV1_private.h"

#include "gameinterface.h"



#define NUM_BYTES_CPU_RAM 1024

#define ADDRESS DWORD

#pragma pack(1)

typedef struct {
	uint8_t vectorram[NUM_BYTES_VECTOR_RAM];
	uint8_t frame_number; 
	uint8_t key_id;
#ifdef RECEIVE_CPU_RAM
	uint8_t cpuram[NUM_BYTES_CPU_RAM];
#endif
} FramePacket_t;


typedef struct {
	uint8_t signature[6];
	uint8_t keys;
	uint8_t key_id; 
} KeysPacket_t;

#pragma pack()



typedef struct {

	FramePacket_t FramePacketBuffer;

	int frame_counter;
	int frame_counter_offset;

	SOCKET sd;
	ADDRESS server_ip;

	int is_WS2_init;

	struct keys_info_queue_t {
		uint32_t keys;
		void* keys_tag;
	} keys_info_queue[256];

	int keys_counter;

} GameInterface_context_t;



static int ReceivePacket(
	SOCKET sd, 
	FramePacket_t *pFramePacket,
	int timeout_ms );

static int SendPacket(
	SOCKET sd, 
	ADDRESS server_ip, 
	KeysPacket_t *pKeysPacket );





void * GameInterface_open( char InetAddressStr[] )
{
	GameInterface_context_t * h = 
		(GameInterface_context_t*)malloc( sizeof(GameInterface_context_t) );

	if (h==NULL)
		return NULL;

	memset( h, 0x00, sizeof(GameInterface_context_t) );

	unsigned long server_ip = inet_addr( InetAddressStr );
	if (server_ip == INADDR_NONE)
	{
		fprintf(stderr, "Ungueltige IP-Adresse: '%s'\n", InetAddressStr);
		goto Failed;
	}

	WSADATA wsadata;
	if (WSAStartup(MAKEWORD(2,2), &wsadata))
	{
		fprintf(stderr, "Fehler beim Initialisieren von Winsock.\n");
		goto Failed;
	}

	h->is_WS2_init = 1;

	SOCKET sd;
	sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
	if (sd == INVALID_SOCKET)
	{
		fprintf(stderr, "Fehler %d bei socket().\n", WSAGetLastError());
		goto Failed;
	}

	unsigned long enable_nonblocking = 1;
	if ( ioctlsocket(sd, FIONBIO, &enable_nonblocking) )
	{
		fprintf(stderr, "Kann Socket nicht auf nonblocking setzen (%d)", WSAGetLastError());
		goto Failed;
	}

	sockaddr_in sa;
	memset(&sa, 0, sizeof sa);
	sa.sin_family = AF_INET;
	sa.sin_addr.s_addr = 0;
	sa.sin_port = 0;

	if ( bind(sd, (struct sockaddr*) &sa, sizeof(sa)) )
	{
		fprintf(stderr, "Fehler %d bei bind().\n", WSAGetLastError());
		goto Failed;
	}

	h->sd = sd;
	h->server_ip = server_ip;

//Success
	return h;

Failed:
	GameInterface_close( h );
	
	return NULL;
}


void GameInterface_close( void *hGameInterface )
{
	GameInterface_context_t * h = (GameInterface_context_t*)hGameInterface;
	if (h==NULL)
		return;

	if ( h->is_WS2_init )
	{
		WSACleanup();
		h->is_WS2_init = 0;
	}

	if ( h->sd )
	{
		closesocket( h->sd );
		h->sd = NULL;
	}

	free(h);
}


int GameInterface_reset( void *hnd )
{
	GameInterface_context_t * h = (GameInterface_context_t*)hnd;
	if (h==NULL)
		return -1;

	memset( &h->FramePacketBuffer, 0x00, sizeof(FramePacket_t) );

	h->frame_counter = 0;
	h->frame_counter_offset = 0;

	memset( h->keys_info_queue, 0x00, sizeof(h->keys_info_queue) );

	h->keys_counter = 0;

	return 0;
}


#define RECEIVE_PACKET_TIMEOUT_MS 250

int GameInterface_receiveNextFrame( 
	void *hnd, 
	Frame_t *pFrame,
	int *pTimestamp )
{
	GameInterface_context_t * h = (GameInterface_context_t*)hnd;
	if (h==NULL)
		return -1;

	if (pFrame==NULL)
		return -1;

	FramePacket_t * const pPacket = &(h->FramePacketBuffer);

	int res = ReceivePacket(
		h->sd, 
		pPacket,
		RECEIVE_PACKET_TIMEOUT_MS );

	if (res!=0)
		return res;

	pFrame->pVectorRam = pPacket->vectorram;

	GameInterface_context_t::keys_info_queue_t * pQueue = h->keys_info_queue;
	const int keys_counter = pPacket->key_id;

	pFrame->keys     = pQueue[keys_counter].keys;
	pFrame->keys_tag = pQueue[keys_counter].keys_tag;

	if (h->frame_counter<=0)
	{
		h->frame_counter = pPacket->frame_number;
		h->frame_counter_offset = h->frame_counter;
	}

	int delta_frame_counter = (int)(uint8_t)( pPacket->frame_number  - (uint8_t)h->frame_counter );

	if ( delta_frame_counter != 0 )
	{
		printf("frames dropped: %d\n", 
			delta_frame_counter );

		h->frame_counter += delta_frame_counter;
	}

	if (pTimestamp)
	{
		*pTimestamp = h->frame_counter - h->frame_counter_offset;
	}

	h->frame_counter++;

	return 0;
}


int GameInterface_sync( 
	void *hnd )
{
	GameInterface_context_t * h = (GameInterface_context_t*)hnd;
	if (h==NULL)
		return -1;

	FramePacket_t * const pPacket = &(h->FramePacketBuffer);
	int res = 0;

	while (1)
	{
		res = ReceivePacket(
			h->sd, 
			pPacket,
			RECEIVE_PACKET_TIMEOUT_MS );

		if (res<0)
			break;

		fd_set readfds = {};
		timeval timeout = { 0, 0 };

		FD_SET(h->sd, &readfds);

		select(
			0, 
			&readfds, 
			NULL,
			NULL,
			&timeout );

		if ( !(FD_ISSET(h->sd, &readfds)) )
			break;
	}

	return res;
}


int GameInterface_sendNewKeys( 
	void *hnd, 
	uint32_t keys,
	void* keys_tag )
{
	GameInterface_context_t * h = (GameInterface_context_t*)hnd;
	if (h==NULL)
		return -1;

	KeysPacket_t KeysPacket;

	GameInterface_context_t::keys_info_queue_t * pQueue = h->keys_info_queue;

	pQueue[h->keys_counter&0xFF].keys     = keys;
	pQueue[h->keys_counter&0xFF].keys_tag = keys_tag;

	if (keys==0)
		keys = '@';

	KeysPacket.signature[0] = 'c';
	KeysPacket.signature[1] = 't';
	KeysPacket.signature[2] = 'm';
	KeysPacket.signature[3] = 'a';
	KeysPacket.signature[4] = 'm';
	KeysPacket.signature[5] = 'e';

	KeysPacket.keys   = (uint8_t)keys;
	KeysPacket.key_id = (uint8_t)h->keys_counter;

	int res = SendPacket( 
		h->sd, 
		h->server_ip,
		&KeysPacket );

	h->keys_counter++;

	return res;
}


static int ReceivePacket(
	SOCKET sd, 
	FramePacket_t *pPacket,
	int timeout_ms )
{
	fd_set readfds = {};
	timeval timeout = { 0, timeout_ms*1000 };

	FD_SET(sd, &readfds);

	int res = select(
		0, 
		&readfds, 
		NULL,
		NULL,
		&timeout );

	if (res==0)	//timeout
		return 1;

	if (res==SOCKET_ERROR)
	{
		fprintf(stderr, "SOCKET_ERROR\n");
		return -1;
	}

	if ( !(FD_ISSET(sd, &readfds)) )
		return 2;

	int bytes_received = recv(sd, (char*)pPacket, sizeof(FramePacket_t), 0);
	if (bytes_received != sizeof(FramePacket_t))
	{
		if (bytes_received>0)
		{
			char msg[128] = {0};
			memcpy( msg, pPacket->vectorram, min(sizeof(msg)-1, bytes_received) );

			if ( 0 == strncmp(msg, "busy ", 5) )
			{
				char *pmsg = msg+5;
				char *pend;

				int value = strtol( pmsg, &pend, 10 );
				if ( (pend!=pmsg) && (value>(11*60)) )
					return 4;

				return 3;
			}

			if ( 0 == strncmp(msg, "game over", 9) )
				return 3;

			fprintf(stderr, "msg: %s\n", msg);
		}

		int err = WSAGetLastError();
		fprintf(stderr, "error: %d, recvfrom()\n", err);

		return -2;
	}

	return 0;
}


static int SendPacket(
	SOCKET sd, 
	ADDRESS server_ip, 
	KeysPacket_t *pPacket )
{
	sockaddr_in server;

	memset(&server, 0, sizeof(server));
	server.sin_family       = AF_INET;
	server.sin_port         = htons(1979);
	server.sin_addr.s_addr  = server_ip;

	int NumBytesToTransmit = sizeof(KeysPacket_t);
	int NumBytesTransmitted = sendto(
		sd, 
		(char*)pPacket, 
		NumBytesToTransmit, 
		0, 
		(sockaddr*)&server, 
		sizeof(server) );

	if ( NumBytesToTransmit != NumBytesTransmitted )
	{
		int err = WSAGetLastError();
		if (err != WSAEWOULDBLOCK)
		{
			fprintf(stderr, "error: %d, sendto()\n", err);
			return -2;
		}
	}

	return 0;
}


int SendName(
	void *hnd, 
	char *name )
{
	GameInterface_context_t * h = (GameInterface_context_t*)hnd;
	if (h==NULL)
		return -1;

	sockaddr_in server;

	memset(&server, 0, sizeof(server));
	server.sin_family       = AF_INET;
	server.sin_port         = htons(1979);
	server.sin_addr.s_addr  = h->server_ip;

	char msg[38] = "ctname";
	strcat_s( msg, sizeof(msg), name );

	int NumBytesToTransmit = sizeof(msg);
	int NumBytesTransmitted = sendto(
		h->sd, 
		msg, 
		NumBytesToTransmit, 
		0, 
		(sockaddr*)&server, 
		sizeof(server) );

	if ( NumBytesToTransmit != NumBytesTransmitted )
	{
		int err = WSAGetLastError();
		if (err != WSAEWOULDBLOCK)
		{
			fprintf(stderr, "error: %d, sendto()\n", err);
			return -2;
		}
	}

	return 0;
}