/*
 * Created on Apr 19, 2008
 *
 * (c) 2006-2007 dka - edv, media, webdesign
 *
 */
package com.dkaedv.asteroids.ki;

import java.util.Collections;
import java.util.List;
import java.util.Vector;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.dkaedv.asteroids.data.Asteroid;
import com.dkaedv.asteroids.data.GameStatus;
import com.dkaedv.asteroids.data.IPositionable;
import com.dkaedv.asteroids.data.Shot;
import com.dkaedv.asteroids.data.Ufo;
import com.dkaedv.asteroids.net.KeysDatagram;
import com.dkaedv.asteroids.util.DistanceComparator;
import com.dkaedv.asteroids.util.VectorCalculations;

public class CollisionProtector extends Thread {
    private Log log = LogFactory.getLog(CollisionProtector.class);

    protected GameStatus gameStatus;
    protected KeysDatagram keysDatagram;
    protected TargetList targetList;

    // below this threshold warp is executed
    private static final double DISTANCE_THRESHOLD = 25;

    // below this threshold a move is executed if useful
    private static final double DISTANCE_THRUST_THRESHOLD = 300;

    // if the predicted position is below this a thrust is started
    private static final double DISTANCE_PREDICTED_THRUST_THRESHOLD = 2 * DISTANCE_THRESHOLD;

    // prediction for thrusting interval (4 predictions are made)
    private static final int THRUST_PREDICTION_INTERVAL = 70;

    private static final double DISTANCE_SHOT_THRESHOLD = 40;

    private static final boolean ENABLE_THRUST = false;

    @Override
    public void run() {
        while(true) {
            doProcessing();

            try {
                gameStatus.waitForUpdate();
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        } // while
    }


    public void doProcessing() {
        boolean statusUpdated = false;

        List<IPositionable> collCandidates = new Vector<IPositionable>();
        collCandidates.addAll(gameStatus.getAsteroids());

        if (gameStatus.getUfo() != null)
            collCandidates.add(gameStatus.getUfo());

        if (collCandidates.size() > 0
                && gameStatus.getShip() != null) {
            Collections.sort(collCandidates, new DistanceComparator(gameStatus.getShip().getPosition().getCurVector()));

            IPositionable collObj = collCandidates.get(0);

            double minDistance = VectorCalculations.getDistance(gameStatus.getShip().getPosition().getCurVector(),
                    collObj.getPosition().getCurVector());

            // Correct distance by size of the asteroid
            if (collObj instanceof Asteroid) {
                minDistance -= ((Asteroid)collObj).getRadius();
            }

            // Correct distance by size of the ufo
            if (collObj instanceof Ufo) {
                minDistance -= ((Ufo)collObj).getRadius();
            }

            if (minDistance < DISTANCE_THRESHOLD) {
                    log.warn("Warping for collision protection");
                    keysDatagram.warp(1);
                    gameStatus.setCollisionStatus(3);
                    statusUpdated = true;
            }

            // Handle shots
            List<IPositionable> shots = new Vector<IPositionable>();
            shots.addAll(gameStatus.getShots());

            if (shots.size() > 0
                    && gameStatus.getShip() != null
                    && gameStatus.getUfo() != null) {
                Collections.sort(shots, new DistanceComparator(gameStatus.getShip().getPosition().getCurVector()));

                IPositionable shot = shots.get(0);

                minDistance = VectorCalculations.getDistance(gameStatus.getShip().getPosition().getCurVector(),
                        shot.getPosition().getCurVector());


                if (minDistance < DISTANCE_SHOT_THRESHOLD
                        && shot.getPosition().getLastVector() != null
                        && (System.currentTimeMillis() - shot.getPosition().getLastVector().timestamp) > 50
                        && ((Shot)shot).getType() == Shot.TYPE_UFO) {
                    double lastDistance = VectorCalculations.getDistance(gameStatus.getShip().getPosition().getCurVector(),
                            shot.getPosition().getLastVector());

                    if (minDistance < lastDistance) {
                        log.warn("Warping for collision protection - shot moving towards me");
                        keysDatagram.warp(1);
                    }
                }
            }


            // No warp needed for Asteroids and Ufo
            // Check if thrust would be useful

            if (ENABLE_THRUST) {
                int counter = 0;

                boolean doThrust = false; // Gets true if a thrust is useful to get away
                boolean blockThrust = false; // Gets true if a thrust would hit another asteroid

                for (IPositionable collCandidate : collCandidates) {
                    if (counter < 10) {
                        double curDistance = VectorCalculations.getDistance(gameStatus.getShip().getPosition().getCurVector(),
                                collCandidate.getPosition().getCurVector());

                        // Don't thrust to target and only if target is within threshold
                        if (curDistance < DISTANCE_THRUST_THRESHOLD) {

                            double diffAngle = VectorCalculations.getAngleBetween(gameStatus.getShip().getDirectionVector(), VectorCalculations.getDifference(gameStatus.getShip().getDirectionVector(), collCandidate.getPosition().getCurVector()));

                            // If candidate is not in front (within +/- 45 degrees)
                            if (diffAngle > Math.PI / 4) {
                                com.dkaedv.asteroids.data.Vector predVector1 = collCandidate.getPosition().getPredictedVector(THRUST_PREDICTION_INTERVAL);
                                com.dkaedv.asteroids.data.Vector predVector2 = collCandidate.getPosition().getPredictedVector(THRUST_PREDICTION_INTERVAL * 2);
                                com.dkaedv.asteroids.data.Vector predVector3 = collCandidate.getPosition().getPredictedVector(THRUST_PREDICTION_INTERVAL * 3);
                                com.dkaedv.asteroids.data.Vector predVector4 = collCandidate.getPosition().getPredictedVector(THRUST_PREDICTION_INTERVAL * 4);

                                double predDist1 = VectorCalculations.getDistance(gameStatus.getShip().getPosition().getPredictedVector(THRUST_PREDICTION_INTERVAL), predVector1);
                                double predDist2 = VectorCalculations.getDistance(gameStatus.getShip().getPosition().getPredictedVector(THRUST_PREDICTION_INTERVAL * 2), predVector2);
                                double predDist3 = VectorCalculations.getDistance(gameStatus.getShip().getPosition().getPredictedVector(THRUST_PREDICTION_INTERVAL * 3), predVector3);
                                double predDist4 = VectorCalculations.getDistance(gameStatus.getShip().getPosition().getPredictedVector(THRUST_PREDICTION_INTERVAL * 4), predVector4);

                                if (predDist1 < DISTANCE_PREDICTED_THRUST_THRESHOLD
                                        || predDist2 < DISTANCE_PREDICTED_THRUST_THRESHOLD
                                        || predDist3 < DISTANCE_PREDICTED_THRUST_THRESHOLD
                                        || predDist4 < DISTANCE_PREDICTED_THRUST_THRESHOLD) {

                                    doThrust = true;
                                }
                            } else {
                                // Candidate is in front, block thrust
                                if (curDistance < 200) {
                                    blockThrust = true;
                                }
                            }
                        }
                    counter++;
                    } // counter if

                    if (doThrust
                            && !blockThrust
                            && gameStatus.getShip().getPosition().getSpeed() < 150) {
                        //log.info("Thrusting to avoid collision");
                        keysDatagram.thrust(40);

                        if (!statusUpdated) {
                            gameStatus.setCollisionStatus(2);
                            statusUpdated = true;
                        }
                    }
                } // for loop
            }
        }





        if (!statusUpdated) {
            gameStatus.setCollisionStatus(1);
            statusUpdated = true;
        }

    }

    /**
     * @param gameStatus the gameStatus to set
     */
    public void setGameStatus(GameStatus gameStatus) {
        this.gameStatus = gameStatus;
    }


    /**
     * @param keysDatagram the keysDatagram to set
     */
    public void setKeysDatagram(KeysDatagram keysDatagram) {
        this.keysDatagram = keysDatagram;
    }

    /**
     * @param targetList the targetList to set
     */
    public void setTargetList(TargetList targetList) {
        this.targetList = targetList;
    }
}
