package se.jupp.asteroids;

import java.awt.Graphics2D;
import java.io.IOException;
import java.util.*;

import se.jupp.asteroids.Frame.AsteroidPos;

public class Game implements FrameListener {
    private Ufo ufo;
    private List<Asteroid> asteroids = new LinkedList<Asteroid>();
    private List<Asteroid> virtual = new LinkedList<Asteroid>();
    private Ship ship;
    private List<Bullet> bullets = new LinkedList<Bullet>();
    private Communication com;
    int fired = 0;
    boolean goingLeft;
    int time = 0;
    int score = 0, minuteStartScore = 0, minuteStartTime = 0;
    int lastFrameScore = 0;
    int levelStartTime = 0, levelEndTime = 0;
    List<GameObject> important = Collections.emptyList();

    public Game(Communication com) {
        this.com = com;
    }

    static class Move implements Comparable<Move> {
        int start, wait, turns, hitTime, time;
        boolean left;
        GameObject enemy;

        public Move(int start, int wait, boolean left, int turns, int hitTime,
                GameObject enemy) {
            this.start = start;
            this.wait = wait;
            this.left = left;
            this.turns = turns;
            this.hitTime = hitTime;
            this.enemy = enemy;
            this.time = wait + turns + hitTime;
        }

        @Override
        public int hashCode() {
            final int prime = 31;
            int result = 1;
            result = prime * result + hitTime;
            result = prime * result + (left ? 1231 : 1237);
            result = prime * result + start;
            result = prime * result + time;
            result = prime * result + turns;
            result = prime * result + wait;
            return result;
        }

        @Override
        public boolean equals(Object obj) {
            if (this == obj)
                return true;
            if (obj == null)
                return false;
            if (getClass() != obj.getClass())
                return false;
            final Move other = (Move) obj;
            if (hitTime != other.hitTime)
                return false;
            if (left != other.left)
                return false;
            if (start != other.start)
                return false;
            if (time != other.time)
                return false;
            if (turns != other.turns)
                return false;
            if (wait != other.wait)
                return false;
            return true;
        }

        public String toString() {
            StringBuilder buf = new StringBuilder();
            if (wait > 0)
                buf.append(wait).append("w");
            buf.append(turns);
            buf.append(left ? 'L' : 'R');
            buf.append(hitTime).append("h");
            buf.append('/').append(enemy.id);
            return buf.toString();
        }

        public int compareTo(Move o) {
            if (time < o.time)
                return -1;
            if (o.time < time)
                return 1;
            return 0;
        }
    }

    List<Move> moves(Ship shipclone, int start,
            List<? extends GameObject> enemies, List<Move> done,
            List<GameObject> important) {
        List<Move> result = new ArrayList<Move>();
        List<Move> underway = new ArrayList<Move>(4);
        int time = 0;
        for (Move m : done) {
            time += m.wait;
            time += m.turns;
            Iterator<Move> i = underway.iterator();
            while (i.hasNext()) {
                Move u = i.next();
                if (u.start + u.time < time)
                    i.remove();
            }
            underway.add(m);
        }
        int earliestShotTime;
        if (underway.size() < 4)
            earliestShotTime = 0;
        else {
            earliestShotTime = Integer.MAX_VALUE;
            for (Move m : underway) {
                if (m.hitTime < earliestShotTime)
                    earliestShotTime = m.hitTime;
            }
        }
        boolean d = important.size() > 0;
        for (int wait = 0; wait < 10; wait++) {
            for (int i = 0; i < 60; i++) {
                if (i + wait < earliestShotTime)
                    continue;
                Bullet b = shipclone.predictedBullet(shipclone.turns + i);
                Hit x = willHitFirst(enemies, b, i + wait);
                if (x != null && (!d || important.contains(x.enemy)))
                    result.add(new Move(start, wait, true, i, x.time, x.enemy));
                b = shipclone.predictedBullet(shipclone.turns - i);
                x = willHitFirst(enemies, b, i);
                if (x != null && (!d || important.contains(x.enemy)))
                    result
                        .add(new Move(start, wait, false, i, x.time, x.enemy));
            }
        }
        Collections.sort(result);
        for (int i = 0; i < result.size(); i++) {
            GameObject enemy = result.get(i).enemy;
            for (int j = result.size() - 1; j > i; j--)
                if (result.get(j).enemy == enemy)
                    result.remove(j);
        }
        return result;
    }

    /*
     * 10 + 20 = 30
     *  5 + 30 = 45
     * 15 + 40 = 70
     *  5 + 30 = 65
     * 
     */
    int eval(List<Move> moves) {
        int time = 0;
        List<Move> underway = new ArrayList<Move>(4);
        for (Move m : moves) {
            time += m.wait;
            time += m.turns;
            Iterator<Move> i = underway.iterator();
            while (i.hasNext()) {
                Move u = i.next();
                if (u.start + u.time < time)
                    i.remove();
            }
            underway.add(m);
        }
        for (Move m : underway)
            time += m.hitTime / 4;
        return time;
    }

    boolean findMove(List<GameObject> important) {
        Move best = null;
        boolean d = important.size() > 0;
        int wait = 0;
        for (int i = 0; i < 60; i++) {
            Bullet b = ship.predictedBullet(ship.turns + i);
            Hit x = willHitFirst(b, i + wait);
            if (x != null && (best == null || x.time + i + wait < best.time)
                    && (!d || important.contains(x.enemy))) {
                best = new Move(time, wait, true, i, x.time, x.enemy);
            }
            b = ship.predictedBullet(ship.turns - i);
            x = willHitFirst(b, i);
            if (x != null && (best == null || x.time + i + wait < best.time)
                    && (!d || important.contains(x.enemy))) {
                best = new Move(time, wait, false, i, x.time, x.enemy);
            }
        }
        return (best != null) ? best.left : true;
    }

    boolean bulletUnderway(GameObject e) {
        if (e == null)
            return false;
        //        if (e instanceof Asteroid && e.r > 8)
        //            return false;
        for (Bullet bullet : bullets)
            if (bullet.target(e) && bullet.willHit)
                return true;
        return false;
    }

    int evaluateEnemy(GameObject b, GameObject e, int future) {
        if (e == null)
            return -1;
        if (bulletUnderway(e))
            return -1;
        int t = b.timeToCollision(e, future);
        if (t >= 0 && t < 70)
            return t;
        return -1;
    }

    List<Hit> collisions() {
        List<Hit> result = new LinkedList<Hit>();
        for (Asteroid a : asteroids) {
            int d = ship.timeToCollision(a, 0);
            if (d >= 0)
                if (!bulletUnderway(a))
                    result.add(new Hit(d, a, ship));
        }
        if (ufo != null) {
            int d = ship.timeToCollision(ufo, 0);
            if (d >= 0)
                if (!bulletUnderway(ufo))
                    result.add(new Hit(d, ufo, ship));
        }
        Collections.sort(result);
        return result;
    }

    List<GameObject> importantEnemies() {
        List<GameObject> result = new LinkedList<GameObject>();
        for (Hit h : collisions())
            result.add(h.enemy);
        if (ufo != null && !bulletUnderway(ufo))
            result.add(ufo);
        return result;
    }

    Position diffOfClosestEnemy() {
        Position minDifference = new Position();
        int dist2 = Integer.MAX_VALUE;
        for (Asteroid a : asteroids) {
            Position difference = a.pos.subtract(ship.pos).normalize();
            int d = difference.square();
            if (d < dist2) {
                if (bulletUnderway(a))
                    continue;
                dist2 = d;
                minDifference = difference;
            }
        }
        if (ufo != null) {
            Position difference = ufo.pos.subtract(ship.pos).normalize();
            int d = difference.square();
            if (d < dist2) {
                dist2 = d;
                minDifference = difference;
            }
        }
        return minDifference;
    }

    public static class Hit implements Comparable<Hit> {
        GameObject friend;
        int time;
        GameObject enemy;

        Hit(int time, GameObject target, GameObject friend) {
            this.time = time;
            this.enemy = target;
            this.friend = friend;
        }

        public int compareTo(Hit o) {
            if (time < o.time)
                return -1;
            if (o.time < time)
                return 1;
            return 0;
        }
    }

    Hit willHitFirst(List<? extends GameObject> enemies, GameObject friend,
            int future) {
        GameObject target = null;
        int time = Integer.MAX_VALUE;
        for (GameObject a : enemies) {
            int t = evaluateEnemy(friend, a, future);
            if (t >= 0 && t < time) {
                target = a;
                time = t;
            }
        }
        if (target != null)
            return new Hit(time, target, friend);
        else
            return null;
    }

    Hit willHitFirst(GameObject friend, int future) {
        GameObject target = null;
        int time = Integer.MAX_VALUE;
        for (Asteroid a : asteroids) {
            int t = evaluateEnemy(friend, a, future);
            if (t >= 0 && t < time) {
                target = a;
                time = t;
            }
        }
        for (Asteroid a : virtual) {
            int t = evaluateEnemy(friend, a, future);
            if (t >= 0 && t < time) {
                target = a;
                time = t;
            }
        }
        for (Bullet a : bullets) {
            if (a.ship != null)
                continue; // my own bullet
            int t = evaluateEnemy(friend, a, future);
            if (t >= 0 && t < time) {
                target = a;
                time = t;
            }
        }
        int t = evaluateEnemy(friend, ufo, future);
        if (t >= 0 && t < time) {
            target = ufo;
            time = t;
        }
        if (target != null)
            return new Hit(time, target, friend);
        else
            return null;
    }

    boolean fireIfEnemyInSight(List<GameObject> important) {
        Bullet b = ship.predictedBullet();
        Hit hit = willHitFirst(b, 0);
        int importantToShoot = 0;
        for (GameObject imp : important)
            if (!bulletUnderway(imp))
                importantToShoot++;
        int n = 4 - ship.bulletsUnderway() - importantToShoot;
        if (hit != null) {
            if (n > 0 || important.contains(hit.enemy)) {
                // if (hit.enemy == ufo
                //    && (ufo.lastTurn == 0 || ufo.age < ufo.lastTurn + 60))
                //    return false;
                // if (hit.enemy != ufo && asteroids.size() <= 1)
                //    return false;
                ship.fire(hit);
                return true;
            } else {
                // System.out.println("Not shooting at unimportant target");
            }
        }
        return false;
    }

    public synchronized void frameReceived(Frame frame) {
        if (frame == null) {
            System.out.println("Game over after " + time
                    + " frames. Total score " + score);
            return;
        }
        int step = ((frame.getId() & 255) - (time & 255)) & 255;
//        int pingStep = ((frame.getPing() & 255) - (time & 255)) & 255;
        time += step;
//        if (step > 1 || pingStep > 1)
//            System.out.println("step = " + step + " time=" + time + " ping="
//                    + pingStep);
        boolean oldHasEnemies = asteroids.size() > 0 || ufo != null;
        updateObjects(frame);
        if (lastFrameScore > 98000 && frame.score < 2000)
            score += 100000;
        score = (score / 100000) * 100000 + frame.score;
        if (time - minuteStartTime >= 60 * 60) {
            System.out.println("Minute score = " + (score - minuteStartScore));
            minuteStartTime = time;
            minuteStartScore = score;
        }
        lastFrameScore = frame.score;
        if (asteroids.size() == 0 && ufo == null && oldHasEnemies) {
            //            System.out.println("end of level at frame " + time + " (took "
            //                    + (time - levelStartTime) + " frames)");
            levelEndTime = time;
        } else if ((asteroids.size() > 0 || ufo != null) && !oldHasEnemies) {
            //            System.out.println("start of level at frame " + time
            //                    + " (pause was " + (time - levelEndTime) + ")");
            levelStartTime = time;
        }
        if (ship != null) {
            important = importantEnemies();
            for (GameObject o : asteroids)
                o.dangerous = false;
            if (ufo != null)
                ufo.dangerous = false;
            for (GameObject o : important)
                o.dangerous = true;
            if (asteroids.size() > 0 || ufo != null) {
                if (ship.canFire())
                    if (fireIfEnemyInSight(important))
                        fired++;
                if (findMove(important))
                    ship.goLeft();
                else
                    ship.goRight();
            }
            Hit danger = willHitFirst(ship, 0);
            if (danger != null && danger.time <= 3)
                ship.goHyper();
        }
        try {
            com.sendKeys(ship != null ? ship.getKeys() : 0, time);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    private void updateObjects(Frame frame) {
        if (frame.spaceShip != null && ship == null)
            ship = new Ship(frame.spaceShip, time);
        else if (ship != null) {
            ship.update(frame.spaceShip);
            if (frame.spaceShip == null)
                ship = null;
        }
        if (frame.ufo != null && ufo == null)
            ufo = new Ufo(frame.ufo, time);
        else if (ufo != null) {
            ufo.update(frame.ufo);
            if (frame.ufo == null)
                ufo = null;
        }
        int ai = 0, pi = 0;
        while (true) {
            Asteroid a = asteroidAt(ai);
            AsteroidPos ap = frame.asteroidAt(pi);
            if (a == null && ap == null)
                break;
            if (ap == null) {
                // destroyed
                a.update(null);
                asteroids.remove(ai);
            } else if (a == null) {
                asteroids.add(new Asteroid(ap, time));
                pi++;
                ai++;
            } else {
                if (a.match(ap)) {
                    a.update(ap);
                    pi++;
                    ai++;
                } else {
                    // destroyed
                    a.update(null);
                    asteroids.remove(ai);
                    Asteroid n = asteroidAt(ai);
                    while (n != null && ap != null && !n.match(ap)) {
                        asteroids.add(ai++, new Asteroid(ap, time));
                        n = asteroidAt(ai);
                        ap = frame.asteroidAt(++pi);
                    }
                }
            }
        }
        for (Iterator<Asteroid> i = virtual.iterator(); i.hasNext();) {
            Asteroid a = i.next();
            if (time - a.timeBorn > 10)
                i.remove();
        }
        updateBullets(frame);
    }

    Bullet closestBullet(Position p) {
        int mindist = Integer.MAX_VALUE;
        Bullet closest = null;
        for (Bullet b : bullets) {
            int d = b.pos.add(b.velocity).distanceSquared(p);
            if (d < mindist) {
                mindist = d;
                closest = b;
            }
        }
        return closest;
    }

    private void updateBullets(Frame f) {
        List<Bullet> done = new LinkedList<Bullet>();
        for (Position b : f.bullets) {
            Bullet closest = closestBullet(b);
            if (closest != null && closest.match(b)) {
                closest.update(b);
                done.add(closest);
                bullets.remove(closest);
            } else {
                Bullet x;
                done.add(x = new Bullet(b, ship, time));
                // Seems shooting on asteroid fragments before they've
                // been created might be bad
                if (false && x.hit != null && x.hit.enemy instanceof Asteroid) {
                    Asteroid a = (Asteroid) x.hit.enemy;
                    //                    if (virtual.contains(a))
                    //                        System.out.println("shooting at virtual " + a);
                    int size = a.smallerSize();
                    if (size == 15) {
                        Position p = a.pos.add(a.velocity.multiply(x.hit.time));
                        AsteroidPos ap1, ap2;
                        Position p2 = x.velocity.divide(x.velocity.length());
                        Position p3 = new Position(p2.y, -p2.x, true);
                        Position p4 = new Position(-p3.x, -p3.y, true);
                        Position p5 = p.add(p3.multiply(a.getSize() / 4));
                        Position p6 = p.add(p4.multiply(a.getSize() / 4));
                        ap1 = new AsteroidPos(p5.x(), p5.y(), 14, a.type);
                        ap2 = new AsteroidPos(p6.x(), p6.y(), 14, a.type);
                        x.virtual.add(new Asteroid(ap1, time));
                        x.virtual.add(new Asteroid(ap2, time));
                        virtual.addAll(x.virtual);
                    } else if (size == 14) {
                        Position p = a.pos.add(a.velocity.multiply(x.hit.time));
                        AsteroidPos ap1, ap2;
                        Position p2 = x.velocity.divide(x.velocity.length());
                        Position p3 = new Position(p2.y, -p2.x, true);
                        Position p4 = new Position(-p3.x, -p3.y, true);
                        Position p5 = p.add(p3.multiply(a.getSize() / 4));
                        Position p6 = p.add(p4.multiply(a.getSize() / 4));
                        ap1 = new AsteroidPos(p5.x(), p5.y(), 14, a.type);
                        ap2 = new AsteroidPos(p6.x(), p6.y(), 14, a.type);
                        x.virtual.add(new Asteroid(ap1, time));
                        x.virtual.add(new Asteroid(ap2, time));
                        virtual.addAll(x.virtual);
                    }
                }
            }
        }
        for (Bullet b : bullets) {
            virtual.removeAll(b.virtual);
            b.update(null);
        }
        bullets = done;
    }

    private Asteroid asteroidAt(int i) {
        return i < asteroids.size() ? asteroids.get(i) : null;
    }

    public synchronized void draw(Graphics2D g) {
        for (Asteroid asteroid : asteroids)
            asteroid.draw(g);
        if (ufo != null)
            ufo.draw(g);
        if (ship != null)
            ship.draw(g);
        for (Bullet bullet : bullets)
            bullet.draw(g);
    }
}
