package se.jupp.asteroids;

import java.awt.Color;
import java.awt.Graphics2D;
import java.util.LinkedList;
import java.util.List;

import se.jupp.asteroids.Game.Hit;

/**
 * Bullet object in automated player of Asteroids game.
 *
 * Bullets are either shot by the ship or by an ufo.
 *
 * This class is part of a solution for a
 * <a href="http://www.heise.de/ct/creativ/08/02/details/">competition by the German computer magazine c't</a>.
 */
public class Bullet extends GameObject {
    private static final int BIG_MAX_SQUARED_DIST = 350;
    private static final int SMALL_MAX_SQUARED_DIST = 10;
    Ship ship;
    Hit hit;
    boolean willHit = false;
    boolean future = false;
    List<Asteroid> virtual = new LinkedList<Asteroid>();
    Position origVelocity = null;

    /**
     *  Constructor.
     *  @param x  x coordinate
     *  @param y  y coordinate
     */
    public Bullet(Position pos, Ship ship, int timeBorn) {
        super(pos, 0, timeBorn);
        if (ship != null) {
            if (this.pos.distanceSquared(ship.pos) < 450) {
                this.ship = ship;
                this.velocity = ship.bulletSpeed();
                this.origVelocity = velocity;
                //                if (ship.lastFire == ship.age - 2) {
                hit = ship.hits.poll();
                if (hit != null)
                    willHit = true;
                //                }
                ship.registerBullet(this);
                //                System.out.println("new " + this);
            }
        }
        if (hit != null) {
            //            if (!hit.friend.pos.equals(pos))
            //                System.out.println("bad " + this + " from " + ship);
            //            else
            //                System.out.println("good " + this + " from " + ship);
        }
    }

    public Bullet(Position pos, Position velocity, int timeBorn) {
        super(pos, 0, timeBorn);
        this.velocity = velocity;
        future = true;
    }

    //    @Override
    //    public int timeToCollision(GameObject b) {
    //        if (future) {
    //            // two frames into the future
    //            // move enemy two steps ahead, and add two to result
    //            Position futurePos = b.pos.add(b.velocity.multiply(2));
    //            int t =
    //                    Position.timeToCollision(pos, velocity, futurePos,
    //                        b.velocity, r + b.r);
    //            if (t >= 0)
    //                return t + 2;
    //            else
    //                return t;
    //        } else
    //            return super.timeToCollision(b);
    //    }

    @Override
    public int timeToCollision(GameObject b, int future) {
        if (this.future) {
            // two frames into the future
            int t = super.timeToCollision(b, future + 2);
            if (t >= 0 && t < 70) // bullets only live for 70 frames
                return t + 2;
            else
                return -1;
        } else
            return super.timeToCollision(b, future);
    }

    boolean target(GameObject t) {
        return hit != null && hit.enemy == t;
    }

    boolean match(Position pos) {
        if (pos == null) {
            return false;
        } else {
            int max;
            if (age == 0)
                max = BIG_MAX_SQUARED_DIST;
            else
                max = SMALL_MAX_SQUARED_DIST;
            Position np = this.pos.add(velocity);
            int dist2 = np.distanceSquared(pos);
            if (dist2 > max) {
                return false;
            }
            return true;
        }
    }

    void update(Position pos) {
        super.update(pos);
        if (pos != null) {
            if (age == 8) {
                Position v = pos.subtract(bornAt).divide(age).normalize();
                if (!v.equals(velocity)) {
                    velocity = v;
                }
                this.pos = this.pos.add(velocity).unnormalize();
            } else if (age < 8) {
                this.pos = pos;
                Position v = pos.subtract(bornAt).divide(age).normalize();
                if (!velocity.equals(origVelocity)
                        || v.distanceSquared(velocity) > 10) {
                    velocity = v;
                    //                    System.out.println("updated velocity of " + this);
                }
            } else
                this.pos = this.pos.add(velocity).unnormalize();
        }
        if (age == 8) {
            if (ship != null)
                ship.calibrate(this);
            //            if (hit != null && !velocity.equals(hit.friend.velocity))
            //                System.out.println(" bad velocity " + this);
        }
        if (hit != null && age > 8) {
            int t = timeToCollision(hit.enemy, 0);
            if (t < 0 || t >= 70) {
                //                System.out
                //                    .println("" + this + " will miss " + hit.enemy + " t=" + t
                //                            + " age = " + age + " hit time=" + hit.time);
                willHit = false;
            } else {
                willHit = true;
            }

        }
        if (pos == null && ship != null) {
            ship.deregisterBullet(this);
            //            if (age < 68)
            //                System.out.println("" + this + " died young");
            //            else
            //                System.out.println("" + this + " destroyed");
        }
        //        System.out.println("updated " + this);
    }

    /**
     *  Get the size of the object.
     *
     *  The size returned by this method is half the length of a square which contains the object,
     *  so the object's bounding box is between (x - size, y - size) and (x + size, y + size).
     *  @return object size
     */
    public int getSize() {
        return 0;
    }

    /**
     *  Draw the object.
     *  @param g graphics context
     */
    public void draw(Graphics2D g) {
        //        super.draw(g);
        if (false && hit != null) {
            if (willHit)
                g.setColor(Color.YELLOW);
            else
                g.setColor(Color.BLUE);
            Position bhit =
                    hit.friend.pos.add(hit.friend.velocity.multiply(hit.time));
            hit.friend.pos.drawLine(bhit, g);
            Position thit =
                    hit.enemy.pos.add(hit.enemy.velocity.multiply(hit.time
                            - age));
            bhit.drawLine(thit, g);
            thit.drawLine(hit.enemy.pos, g);
        }
        g.setColor(Color.white);
        g.fillOval(pos.x() - 1, pos.y() - 1, 3, 3);
        drawVelocityVector(g);
    }

    @Override
    public String toString() {
        return "Bullet@" + id + pos + " velocity " + velocity + " age " + age;
    }
}
