package se.jupp.asteroids;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Shape;
import java.awt.geom.AffineTransform;
import java.awt.geom.GeneralPath;

/**
 * Asteroid object in automated player of Asteroids game.
 *
 * This class is part of a solution for a
 * <a href="http://www.heise.de/ct/creativ/08/02/details/">competition by the German computer magazine c't</a>.
 */
public class Asteroid extends GameObject implements Cloneable {
    /** Scaling for shapes when drawing big asteroid. */
    final private static int NORMAL_SCALE = 8;
    /** Scaling of this asteroid. */
    int size, scale;
    /** Type of this asteroid. There are 4 different types numbered from 0 to 3. */
    int type;
    private static final GeneralPath PATH0 = new GeneralPath();
    private static final GeneralPath PATH1 = new GeneralPath();
    private static final GeneralPath PATH2 = new GeneralPath();
    private static final GeneralPath PATH3 = new GeneralPath();
    static {
        PATH0.moveTo(1, 0);
        PATH0.lineTo(4, 1);
        PATH0.lineTo(4, 2);
        PATH0.lineTo(1, 4);
        PATH0.lineTo(-2, 4);
        PATH0.lineTo(-1, 3);
        PATH0.lineTo(-4, 3);
        PATH0.lineTo(-4, -1);
        PATH0.lineTo(-2, -4);
        PATH0.lineTo(1, -3);
        PATH0.lineTo(2, -4);
        PATH0.lineTo(4, -2);
        PATH0.closePath();

        PATH1.moveTo(3, 0);
        PATH1.lineTo(4, 2);
        PATH1.lineTo(2, 4);
        PATH1.lineTo(0, 2);
        PATH1.lineTo(-2, 4);
        PATH1.lineTo(-4, 2);
        PATH1.lineTo(-4, -2);
        PATH1.lineTo(-2, -4);
        PATH1.lineTo(1, -4);
        PATH1.lineTo(4, -2);
        PATH1.closePath();

        PATH2.moveTo(4, 1);
        PATH2.lineTo(2, 4);
        PATH2.lineTo(-1, 4);
        PATH2.lineTo(-4, 1);
        PATH2.lineTo(-2, 0);
        PATH2.lineTo(-4, -1);
        PATH2.lineTo(-2, -4);
        PATH2.lineTo(0, -1);
        PATH2.lineTo(0, -4);
        PATH2.lineTo(2, -4);
        PATH2.lineTo(4, -1);
        PATH2.closePath();

        PATH3.moveTo(2, 1);
        PATH3.lineTo(4, 2);
        PATH3.lineTo(2, 4);
        PATH3.lineTo(0, 3);
        PATH3.lineTo(-2, 4);
        PATH3.lineTo(-4, 2);
        PATH3.lineTo(-3, 0);
        PATH3.lineTo(-4, -2);
        PATH3.lineTo(-2, -4);
        PATH3.lineTo(-1, -3);
        PATH3.lineTo(2, -4);
        PATH3.lineTo(4, -1);
        PATH3.closePath();
    }
    /** The asteroid shapes sorted by type. */
    private static final Shape[] SHAPES =
            new Shape[] { PATH1, PATH3, PATH2, PATH0, };

    /**
     * Constructor.
     * @param x position (x coordinate)
     * @param y position (y coordinate)
     * @param size size (as given by game, so 0 is big and 14 is small)
     * @param type type of asteroid (0, 1, 2, or 3)
     */
    public Asteroid(Frame.AsteroidPos pos, int timeBorn) {
        super(pos, 4 * scale(pos.size), timeBorn);
        this.size = pos.size;
        scale = scale(size);
        assert (pos.type >= 0 && pos.type < SHAPES.length);
        this.type = pos.type;
        //        System.out.println("New asteroid " + this + " r = " + r);
    }

    @Override
    protected Object clone() throws CloneNotSupportedException {
        Asteroid c = (Asteroid) super.clone();
        return c;
    }

    private static final int BIG_MAX_SQUARED_DIST = 8 * 8;
    private static final int SMALL_MAX_SQUARED_DIST = 10;

    boolean match(Frame.AsteroidPos pos) {
        if (pos == null) {
            return false;
        } else {
            if (pos.size != this.size)
                return false;
            if (pos.type != this.type)
                return false;
            int max;
            if (age == 0)
                max = BIG_MAX_SQUARED_DIST;
            else
                max = SMALL_MAX_SQUARED_DIST;
            if (this.pos.add(velocity).distanceSquared(pos) > max) {
                //                System.out.println("match failed " + this.pos + " against "
                //                        + pos + " velocity " + velocity + " distance "
                //                        + this.pos.add(velocity).subtract(pos) + " distance2 "
                //                        + this.pos.add(velocity).distanceSquared(pos));
                return false;
            }
            return true;
        }
    }

    void update(Frame.AsteroidPos pos) {
        super.update(pos);
        if (pos != null) {
            if (age <= 8) {
                this.pos = pos;
                velocity = pos.subtract(bornAt).divide(age).normalize();
            } else
                this.pos = this.pos.add(velocity).unnormalize();
        }

        if (pos == null) {
            //            System.out.println("Asteroid " + this + " destroyed");
        } else {
            //            System.out.println("Asteriod " + id + " position delta "
            //                    + pos.subtract(old));
        }
    }

    int smallerSize() {
        switch (size) {
        case 0:
            return 15;
        case 15:
            return 14;
        case 14:
            return -1;
        default:
            return -1;
        }
    }

    private static int scale(int size) {
        switch (size) {
        case 0:
            return NORMAL_SCALE;
        case 15:
            return NORMAL_SCALE / 2;
        case 14:
            return NORMAL_SCALE / 4;
        default:
            return 0;
        }
    }

    /**
     *  Get the size of the object.
     *
     *  The size returned by this method is half the length of a square which contains the object,
     *  so the object's bounding box is between (x - size, y - size) and (x + size, y + size).
     *  @return object size
     */
    public int getSize() {
        return 4 * scale;
    }

    public int getType() {
        return type;
    }

    /**
     *  Draw the object.
     *  @param g graphics context
     */
    @Override
    public void draw(Graphics2D g) {
        //        super.draw(g);
        if (dangerous)
            g.setColor(Color.RED);
        else
            g.setColor(Color.gray);
        Position p = pos.unnormalize();
        AffineTransform trafo =
                new AffineTransform(scale, 0, 0, scale, p.x(), p.y());
        g.draw(trafo.createTransformedShape(SHAPES[type]));

        drawVelocityVector(g);
        //super.draw(g);
    }

    /**
     * Returns a string representation of the object.
     *
     * @return a string representation of the object.
     */
    @Override
    public String toString() {
        return "Asteroid@" + id + pos;
    }

}
