/*
 * Copyright (C) 2008 Henning Faber
 * 
 * This file is part of Sitting Duck Asteroids Bot project.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. 
 */

package de.hfaber.asteroids.bot.sittingduck;

import de.hfaber.asteroids.game.objects.GameObject;

/**
 * A tracked shot is a move that has been executed and thus resulted in
 * a bullet being fired. As soon as the bullet appears in the game 
 * status, it's object id may be assigned to the tracked shot.
 * 
 * @author Henning Faber
 */
public class TrackedShot extends Move {

    /**
     * The next available fake target id. 
     */
    private static int nextFakeTargetId = -1;
    
    /**
     * The object id of the bullet that belongs to this tracked shot,
     * or {@link GameObject#UNTRACKED}, if the bullet has not yet been
     * assigned.
     */
    private int m_bulletId;

    /**
     * Creates a tracked shot for the given move that does not have a
     * bullet id assigned.
     * 
     * @param m the move that caused this tracked shot to be fired
     */
    public TrackedShot(Move m, int impactFrameCorrection) {
        this(m, GameObject.UNTRACKED, impactFrameCorrection);
    }
    
    /**
     * Creates a tracked shot for the given move and bullet id.
     * 
     * @param m the move that caused this tracked shot to be fired
     * @param bulletId the object id of the bullet that belongs to this
     *  tracked shot
     * @param impactFrameCorrection 
     */
    public TrackedShot(Move m, int bulletId, int impactFrameCorrection) {
        super(m.getShotFrameNo(), m.getImpactFrameNo() + impactFrameCorrection, 
                m.getAngle(), m.getTargetId(), m.getTargetSize(), 
                m.getShotDistance(), m.getCollisionFrameNo());
        m_bulletId = bulletId;
    }
    
    /**
     * Creates a dummy tracked shot that only has bullet id but no 
     * move that caused the bullet to appear. Thus, it also does not have
     * a target that the tracked shot was fired at. Since the equals()
     * implementation considers two tracked shots identical, if they
     * were fired at the same target, a fake (but unique) target id is
     * used. 
     * 
     * @param bulletId the object id of the bullet that belongs to this
     *  tracked shot
     * @param eolFrameNo the frame number when the bullet that belongs to
     *  this tracked shot is expected to reach its end of lifetime (eol)
     */
    public TrackedShot(int bulletId, int eolFrameNo) {
        super(0, eolFrameNo, 0, getNextFakeTargetId(), 0, 0, NO_COLLISION);
        m_bulletId = bulletId;
    }
    
    /**
     * Returns the next available fake target id.
     * 
     * @return the next available fake target id
     */
    private static synchronized int getNextFakeTargetId() {
        return nextFakeTargetId--;
    }

    /**
     * Determines, if this tracked shot has a 'fake' target assigned.
     * 
     * @return <code>true</code>, if the the target assigned for this
     *  tracked shot is a 'fake' target, <code>false</code>, if not
     */
    public final boolean hasFakeTarget() {
        return getTargetId() <= GameObject.UNTRACKED;
    }
    
    /**
     * @return the bulletId
     */
    public final int getBulletId() {
        return m_bulletId;
    }

    /* (non-Javadoc)
     * @see de.hfaber.asteroids.bot.sittingduck.Move#toString()
     */
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(getClass().getSimpleName());
        sb.append("[shotFrameNo=");
        sb.append(getShotFrameNo());
        sb.append(" impactFrameNo=");
        sb.append(getImpactFrameNo());
        sb.append(" targetId=");
        sb.append(getTargetId());
        sb.append(" bulletId=");
        sb.append(m_bulletId);
        sb.append("]");
        return sb.toString();
    }
}
