/*
 * Copyright (C) 2008 Henning Faber
 * 
 * This file is part of Sitting Duck Asteroids Bot project.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. 
 */
package de.hfaber.asteroids.game.state;

import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import de.hfaber.asteroids.client.Commands;
import de.hfaber.asteroids.game.field.Point;
import de.hfaber.asteroids.game.objects.AngleByteRange;
import de.hfaber.asteroids.game.objects.Asteroid;
import de.hfaber.asteroids.game.objects.Bullet;
import de.hfaber.asteroids.game.objects.Explosion;
import de.hfaber.asteroids.game.objects.GameObject;
import de.hfaber.asteroids.game.objects.IScaleableGameObject;
import de.hfaber.asteroids.game.objects.ITypeableGameObject;
import de.hfaber.asteroids.game.objects.Saucer;
import de.hfaber.asteroids.game.objects.ScaleableGameObject;
import de.hfaber.asteroids.game.objects.Ship;

/**
 * @author Henning Faber
 */
public class GameStatus {

    /** 
     * Position of frame counter in receive buffer. 
     */
    private static final int FRAME_COUNT_POSITION = 1024;
    
    /**
     * Position of ping byte in receive buffer.
     */
    private static final int PING_POSITION = FRAME_COUNT_POSITION + 1;

    // op codes
    private static final int OP_LABS = 0xa;
    private static final int OP_HALT = 0xb;
    private static final int OP_JSRL = 0xc;
    private static final int OP_RTSL = 0xd;
    private static final int OP_JMPL = 0xe;
    private static final int OP_SVEC = 0xf;
    
    // addresses of sub routines
    private static final int SUB_ASTEROID_TYPE1 = 0x8f3;
    private static final int SUB_ASTEROID_TYPE2 = 0x8ff;
    private static final int SUB_ASTEROID_TYPE3 = 0x90d;
    private static final int SUB_ASTEROID_TYPE4 = 0x91a;
    private static final int SUB_SAUCER = 0x929;
    private static final int SUB_SHIP = 0xa6d;
    private static final int SUB_EXPLOSION_XXL = 0x880;
    private static final int SUB_EXPLOSION_XL = 0x896;
    private static final int SUB_EXPLOSION_L = 0x8b5;
    private static final int SUB_EXPLOSION_S = 0x8d0;

    // addresses of charaters 
    private static final Map<Integer, Character> CHAR_MAP = 
        new HashMap<Integer, Character>();
    static {
        CHAR_MAP.put(0xA78, 'A');
        CHAR_MAP.put(0xA80, 'B');
        CHAR_MAP.put(0xA8D, 'C');
        CHAR_MAP.put(0xA93, 'D');
        CHAR_MAP.put(0xA9B, 'E');
        CHAR_MAP.put(0xAA3, 'F');
        CHAR_MAP.put(0xAAA, 'G');
        CHAR_MAP.put(0xAB3, 'H');
        CHAR_MAP.put(0xABA, 'I');
        CHAR_MAP.put(0xAC1, 'J');
        CHAR_MAP.put(0xAC7, 'K');
        CHAR_MAP.put(0xACD, 'L');
        CHAR_MAP.put(0xAD2, 'M');
        CHAR_MAP.put(0xAD8, 'N');
        CHAR_MAP.put(0xADD, '0'); // zero instead if oh!
        CHAR_MAP.put(0xAE3, 'P');
        CHAR_MAP.put(0xAEA, 'Q');
        CHAR_MAP.put(0xAF3, 'R');
        CHAR_MAP.put(0xAFB, 'S');
        CHAR_MAP.put(0xB02, 'T');
        CHAR_MAP.put(0xB08, 'U');
        CHAR_MAP.put(0xB0E, 'V');
        CHAR_MAP.put(0xB13, 'W');
        CHAR_MAP.put(0xB1A, 'X');
        CHAR_MAP.put(0xB1F, 'Y');
        CHAR_MAP.put(0xB26, 'Z');
        CHAR_MAP.put(0xB2C, ' ');
        CHAR_MAP.put(0xB2E, '1');
        CHAR_MAP.put(0xB32, '2');
        CHAR_MAP.put(0xB3A, '3');
        CHAR_MAP.put(0xB41, '4');
        CHAR_MAP.put(0xB48, '5');
        CHAR_MAP.put(0xB4F, '6');
        CHAR_MAP.put(0xB56, '7');
        CHAR_MAP.put(0xB5B, '8');
        CHAR_MAP.put(0xB63, '9');
    }
    
    /**
     * The maximum number of asteroids that may occur within one game
     * status.
     */
    public static final int MAX_ASTEROIDS = 26;
    
    /**
     * The value when the score overflow occurs. 
     */
    private static final int SCORE_OVERFLOW = 100000;
    
    /**
     * Value for numbers that could not be extracted from the vector ram.
     */
    private static final int NO_NUMBER = -1;

    /** 
     * Game start string, may differ with other (non-German) ROMs.
     * If changing use only uppercase letters and 0 (zero) instead 
     * of O (oh).
     */
    private static final String GAME_START_STRING = "STARTKN0EPFE DRUECKEN";
    
    /** 
     * Game starting string, may differ with other (non-German) ROMs.
     * If changing use only uppercase letters and 0 (zero) instead 
     * of O (oh).
     */
    private static final String PLAYER1_STRING = "SPIELER 1";

    /** 
     * Game end string, may differ with other (non-German) ROMs.
     * If changing use only uppercase letters and 0 (zero) instead 
     * of O (oh).
     */
    private static final String GAME_END_STRING = "SPIELENDE";

    /** 
     * Location of the score of player one during play. 
     */
    private static final Point PLAYER_ONE_SCORE_LOCATION_GAME =
        new Point(160, 852);

    /** 
     * Location of score text in highscore and teaser display. 
     */
    private static final Point PLAYER_ONE_SCORE_LOCATION_OTHER = 
        new Point(100, 876);
    
    /** 
     * Location of the score of player two in highscore and teaser display. 
     */
    private static final Point PLAYER_TWO_SCORE_LOCATION_OTHER =
        new Point(768, 876);
    
    /** 
     * Highscore location in game. 
     */
    private static final Point HIGHSCORE_LOCATION   = new Point(480, 876);
    
    /**
     * Re-usable exact match strategy.
     */
    private static final IMatchStrategy EXACT_MATCH_STRATEGY = 
        new ExactMatchStrategy();

    /**
     * Re-usable approximate match strategy.
     */
    private static final IMatchStrategy APPROXIMATE_BULLET_MATCH_STRATEGY = 
        new ApproximateMatchStrategy(false);

    /**
     * Re-usable approximate match strategy.
     */
    private static final IMatchStrategy APPROXIMATE_ASTEROID_MATCH_STRATEGY = 
        new ApproximateMatchStrategy(true);

    /**
     * The ship or <code>null</code>, if no ship present. 
     */
    private Ship m_ship;
    
    /**
     * The last known shot angle. If the ship disappears, the last
     * known shot angle is still tracked and when the ship re-appears,
     * the initial angle is set to the tracked value.
     */
    private AngleByteRange m_lastShotAngle;
    
    /**
     * The list of asteroids. 
     */
    private final List<Asteroid> m_asteroids;

    /**
     * The list of bullets. 
     */
    private final List<Bullet> m_bullets;
    
    /**
     * The lisst of explosions.
     */
    private final List<Explosion> m_explosions;
    
    /**
     * The saucer or <code>null</code>, if no saucer present. 
     */
    private Saucer m_saucer;
    
    /**
     * The number of lifes left. 
     */
    private int m_lifeCount;
    
    /**
     * <code>true</code>, if this game status originates from the lower
     * part of the vector ram, <code>false</code>, if not. 
     */
    private boolean m_evenFrame;

    /**
     * The frame number as received from the mame server. 
     */
    private final byte m_serverFrame;
    
    /**
     * The ping byte as received from the mame server. 
     */
    private final byte m_ping;
    
    /**
     * The number of frames that this game status was projected ahead from
     * another game status.
     */
    private final int m_latencyCorrection;
    
    /**
     * The current score for player one.
     */
    private int m_playerOneScore;
    
    /**
     * The current score for player two.
     */
    private int m_playerTwoScore;

    /** 
     * The high score. 
     */
    private int m_highscore;
    
    /**
     * The current round.
     */
    private int m_round;
    
    /**
     * The number of frames that have elapsed in the current round.
     */
    private int m_roundDuration;

    /**
     * The number of frames, this game has seen, since the last time that 
     * {@link #m_gameStarting} flag was <code>true</code>.
     */
    private int m_frameNo;
    
    /** 
     * <code>true</code>, if teaser screen or highscore list is displayed,
     * <code>false</code> otherwise  
     */
    private boolean m_waitingForStart;

    /**
     * <code>true</code>, if game is about to start, <code>false</code>
     *  otherwise 
     */
    private boolean m_gameStarting;
    
    /** 
     * <code>true</code>, if game is over, but teaser screen is not yet
     * displayed, <code>false</code> otherwise 
     */
    private boolean m_gameEnding;
    
    /**
     * Creates a game status and fills it with the data from the given
     * receive buffer.
     * 
     * @param receiveBuffer The data used to setup the game status
     * @param prevGameStatus A previous game status that should be used
     *  to track objects. May be <code>null</code>, which results in all
     *  objects having zero velocity and a new id
     * @param commandMap A map that maps ping bytes to the command objects 
     *  that were sent by the mame client with the corresponding ping byte.
     *  This is used for tracking ship movements and shot angle. May be 
     *  <code>null</code>, which results in the ship not being tracked.  
     * @throws InvalidVectorRamException if the data in the receive buffer
     *  is invalid
     */
    public GameStatus(ByteBuffer receiveBuffer, GameStatus prevGameStatus,
            Map<Byte, Commands> commandMap) throws InvalidVectorRamException {
        super();

        // check, if buffer contains a valid vector ram frame
        int frameStart = receiveBuffer.getShort() & 0xffff;
        if ((frameStart != 0xe001) && (frameStart != 0xe201)) {
            throw new InvalidVectorRamException(
                    "Vector ram signature not found.");
        }
        
        // retrieve server frame and ping
        m_serverFrame = receiveBuffer.get(FRAME_COUNT_POSITION);
        m_ping = receiveBuffer.get(PING_POSITION);
        
        // do not consider old frames
        if (prevGameStatus != null) {
            byte frameGap = (byte)(m_serverFrame - prevGameStatus.getServerFrame());
            if (frameGap < 1) {
                throw new InvalidVectorRamException(
                        "Vector ram data belongs to an old frame.");
            }
        }
    
        // initialize asteroid list, bullet list and explosion list
        m_asteroids = new ArrayList<Asteroid>();
        m_bullets = new ArrayList<Bullet>();
        m_explosions = new ArrayList<Explosion>();
        
        // this game status is not projected from another game status 
        m_latencyCorrection = 0;
        
        // initialize scores
        m_playerOneScore = NO_NUMBER;
        m_playerTwoScore = NO_NUMBER;
        m_highscore = NO_NUMBER;
        
        // detect objects in vector ram
        detectObjects(receiveBuffer);
        
        // perform operations that necessarily require a previous game status
        if (prevGameStatus != null) {
            
            // track objects
            trackObjects(prevGameStatus, commandMap);
            
            // update game duration
            updateFrameCounter(prevGameStatus);
            
            // correct score overflow
            correctScoreOverflow(prevGameStatus);
        }
        
        // set location relative to ship
        setShipCoordinateSystem();
        
        // update round number
        updateRoundNumber(prevGameStatus);
    }

    /**
     * Creates a game status from the given game status and projects
     * all objects ahead for the given number of frames.
     * 
     * @param toProject the game status to project
     * @param shipRoationSteps the number of steps the ship will rotate 
     *  until the new frame is reached; a negative number denotes a 
     *  clockwise rotation, a positive number denotes an anti-clockwise
     *  rotation    
     * @param frameCount the number of frames the game status should be
     *  projected ahead
     */
    public GameStatus(GameStatus toProject, int shipRoationSteps,
            int frameCount) {
        super();
        
        // set latency correction
        m_latencyCorrection = frameCount;
        
        // project asteroids
        m_asteroids = new ArrayList<Asteroid>();
        for (Asteroid a : toProject.m_asteroids) {
            Asteroid projectedAsteroid = new Asteroid(a, frameCount);
            m_asteroids.add(projectedAsteroid);
        }
        
        // project bullets
        m_bullets = new ArrayList<Bullet>();
        for (Bullet bullet : toProject.m_bullets) {
            Bullet projectedBullet = new Bullet(bullet, frameCount);
            m_bullets.add(projectedBullet);
        }
        
        // project explosions
        m_explosions = new ArrayList<Explosion>();
        for (Explosion explosion : toProject.m_explosions) {
            Explosion projectedExplosion = new Explosion(explosion, 
                    frameCount);
            m_explosions.add(projectedExplosion);
        }
        
        // project ship, if present
        if (toProject.m_ship != null) {
            Ship projectedShip = new Ship(toProject.m_ship, frameCount, 
                    shipRoationSteps);
            m_ship = projectedShip;
        }
        
        // project saucer, if present
        if (toProject.m_saucer != null) {
            Saucer projectedSaucer = new Saucer(toProject.m_saucer, frameCount);
            m_saucer = projectedSaucer;
        }
        
        // project ping byte and server frame and frame number 
        m_ping = (byte)(toProject.m_ping + frameCount);
        m_serverFrame = (byte)(toProject.m_serverFrame + frameCount);
        m_frameNo = toProject.m_frameNo + frameCount;
        
        // determine even frame flag of projected game status
        m_evenFrame = toProject.m_evenFrame;
        if (frameCount % 2 != 0) {
            m_evenFrame = !m_evenFrame;
        }

        // adopt status information from given game status
        m_lastShotAngle = toProject.m_lastShotAngle;
        m_lifeCount = toProject.m_lifeCount;
        m_playerOneScore = toProject.m_playerOneScore;
        m_playerTwoScore = toProject.m_playerTwoScore;
        m_highscore = toProject.m_highscore;
        m_round = toProject.m_round;
        m_roundDuration = toProject.m_roundDuration + frameCount;
        m_waitingForStart = toProject.m_waitingForStart;
        m_gameStarting = toProject.m_gameStarting;
        m_gameEnding = toProject.m_gameEnding;
    }
    
    /**
     * Creates a game status from the given game status and projects
     * all objects ahead for the given number of frames.
     * 
     * @param toProject the game status to project
     * @param commandMap A map that maps ping bytes to the command objects 
     *  that were sent by the mame client with the corresponding ping byte.
     *  This is used for projecting the ship's shot angle. May be 
     *  <code>null</code>, which results in the shot angle not being
     *  projected.     
     * @param lastPing the last ping byte that was used to send commands
     * @param frameCount the number of frames the game status should be
     *  projected ahead
     */
    public GameStatus(GameStatus toProject, Map<Byte, Commands> commandMap,
            byte lastPing, int frameCount) {
        this(toProject, getShipRoationSteps(toProject, commandMap, lastPing, 
                frameCount), frameCount);
    }
    
    /**
     * Determines the rotation the ship has done after the given number
     * of frames, starting from the situation described by the given game 
     * status. The rotation is determined by the left/right commands that
     * have been issued and that are recorded in the given command map.
     * A negative return value denotes a clocwise rotation, a positive 
     * return value denotes an anti-clockwise rotation. 
     * 
     * @param toProject the game status object
     * @param commandMap A map that maps ping bytes to the command objects 
     *  that were sent by the mame client with the corresponding ping byte.
     *  May be <code>null</code>, which will yield a return value of zero.
     * @param frameCount the number of frames the game status should be
     *  projected ahead
     * @param lastPing the last ping byte that was used to send commands 
     * @return the number of steps the ship has been rotated
     */
    private static int getShipRoationSteps(GameStatus toProject,
            Map<Byte, Commands> commandMap, byte lastPing, int frameCount) {
        int steps = 0;
        for (int i = 0; i < frameCount; i++) {
            byte pingByte = (byte)(toProject.m_ping + i + 1);
            if ((byte)(lastPing - pingByte) < 0) {
                pingByte = lastPing;
            }
            Commands c = commandMap.get(pingByte);
            if (c != null) {
                if (c.isLeftPressed()) {
                    steps++;
                }
                if (c.isRightPressed()) {
                    steps--;
                }
            }
        }
        
        return steps;
    }
    
    /**
     * Detects game objects in the given vector ram and fills the
     * instance variables with the results.
     * 
     * @param vectorRam the vector ram
     */
    private void detectObjects(ByteBuffer vectorRam) {
        
        // initialize position pointer
        int position = 0;
        
        // initialize vector generator parameters
        int vx = 0;
        int vy = 0;
        int vs = 0;
        
        // initialize ship detection parameters
        ShipDetectionParameters shipDetection = new ShipDetectionParameters();
        
        // initialize variables for text detection
        TextDetectionParameters textDetection = new TextDetectionParameters();
        
        // loop, while valid position available 
        while (position < vectorRam.limit() - 2) {

            textDetection.resetAddFlag();
            
            // retrieve the next two words from the vector ram
            int curWord = vectorRam.getShort(position) & 0xffff;
            int nextWord = vectorRam.getShort(position + 2) & 0xffff;

            // extract op code
            int op = curWord >> 12;
        
            // distinguish op code
            switch (op) {
                case OP_LABS: 
                    vy = curWord & 0x3ff;
                    vx = nextWord & 0x3ff;
                    vs = nextWord >> 12;
                    break;
                    
                case OP_HALT:
                    return;
                    
                case OP_JSRL:
                    detectStaticallyDrawnObjects(textDetection, curWord, vx, 
                            vy, vs);  
                    break;
                    
                case OP_RTSL: 
                    return;
                    
                case OP_JMPL: 
                    m_evenFrame = (curWord & 0x3ff) == 1;
                    position = (curWord & 0x1ff) * 2;
                    break;
                    
                case OP_SVEC:
                    // not used for object detection!
                    break;
                    
                default:
                    detectDynamicallyDrawnObjects(shipDetection, curWord, 
                            nextWord, op, vx, vy);
                    break;
            }
            
            // extract information form text, if text string is complete
            if (!textDetection.isAddedLetter()) {
                StringBuilder text = textDetection.getText();
                if (text.length() > 0) {
                    getDataFromText(text.toString(), textDetection.getTx(), 
                            textDetection.getTy());
                    text.setLength(0);
                }
            }
            
            // move to next position
            if (op <= 0xa) {
                position += 2;
            }
            if (op != OP_JMPL) {  
                position += 2;
            }
        }
    }

    /**
     * Detects objects that the vector generator draws by jumping to a
     * fixed sub routine. This applies to asteroids, explosions, the
     * saucer, charcter letters and the ship
     * 
     * @param textDetection text detection parmeters 
     * @param curWord the 
     * @param curWord the word at the currently processed position in the
     *  vector ram
     * @param vx current x co-ordinate of the drawing ray
     * @param vy current y co-ordinate of the drawing ray
     * @param vs the current scaling factor
     */
    private void detectStaticallyDrawnObjects(
            TextDetectionParameters textDetection, int curWord, int vx, int vy,
            int vs) {
        Asteroid asteroid;
        int subAddress = curWord & 0xfff;
        switch (subAddress) {
            case SUB_ASTEROID_TYPE1:
                asteroid = new Asteroid(vx, vy, vs, Asteroid.TYPE_1);
                m_asteroids.add(asteroid);
                break;
                
            case SUB_ASTEROID_TYPE2:
                asteroid = new Asteroid(vx, vy, vs, Asteroid.TYPE_2);
                m_asteroids.add(asteroid);
                break;
                
            case SUB_ASTEROID_TYPE3:
                asteroid = new Asteroid(vx, vy, vs, Asteroid.TYPE_3);
                m_asteroids.add(asteroid);
                break;
                
            case SUB_ASTEROID_TYPE4:
                asteroid = new Asteroid(vx, vy, vs, Asteroid.TYPE_4);
                m_asteroids.add(asteroid);
                break;
                
            case SUB_SAUCER:
                m_saucer = new Saucer(vx, vy, vs);
                break;
                
            case SUB_SHIP:
                m_lifeCount++;
                break;
                
            case SUB_EXPLOSION_XXL:
            case SUB_EXPLOSION_XL:
            case SUB_EXPLOSION_L:
            case SUB_EXPLOSION_S:
                Explosion explosion = new Explosion(vx, vy);
                m_explosions.add(explosion);
                break;
                
            default:
                Character ch = CHAR_MAP.get(subAddress);
                if (ch != null) {
                    textDetection.addLetter(ch, vx, vy);
                }
                break;
        }
    }

    /**
     * Detects objects that the vector generator draws by with several
     * successive VCTR operations. This applies to bullets and the ship.
     * 
     * @param shipDetection the parameters needed for ship detection
     * @param curWord the word at the currently processed position in the
     *  vector ram
     * @param nextWord the word after the currently processed position in the
     *  vector ram
     * @param op the current op code
     * @param vx current x co-ordinate of the drawing ray
     * @param vy current y co-ordinate of the drawing ray
     */
    private void detectDynamicallyDrawnObjects(
            ShipDetectionParameters shipDetection, int curWord, int nextWord,
            int op, int vx, int vy) {
        
        shipDetection.m_dy = curWord & 0x3ff;
        if ((curWord & 0x400) != 0) {
            shipDetection.m_dy = -shipDetection.m_dy;
        }
        
        shipDetection.m_dx = nextWord & 0x3ff;
        if ((nextWord & 0x400) != 0) {
            shipDetection.m_dx = -shipDetection.m_dx;
        }
        
        shipDetection.m_vz = nextWord >> 12;
            
        if ((shipDetection.m_dx == 0) && (shipDetection.m_dy == 0) 
                && (shipDetection.m_vz == 15)) {
            Bullet bullet = new Bullet(vx, vy);
            m_bullets.add(bullet);
        }

        Ship ship = shipDetection.detectShip(op, vx, vy);
        if (ship != null) {
            m_ship = ship;
        }
    }

    /**
     * Updates game status properties depending on the given text or the 
     * given position.
     * 
     * @param text the text from which information should be extracted
     * @param tx the x coordinate of the text on the screen
     * @param ty the y coordinate of the text on the screen
     */
    private void getDataFromText(String text, int tx, int ty) {
        // check, if text denotes the score for player one
        if (((tx == PLAYER_ONE_SCORE_LOCATION_GAME.getX()) 
                && (ty == PLAYER_ONE_SCORE_LOCATION_GAME.getY()))
            || ((tx == PLAYER_ONE_SCORE_LOCATION_OTHER.getX()) 
                    && (ty == PLAYER_ONE_SCORE_LOCATION_OTHER.getY()))) {
            m_playerOneScore = extractNumber(text);
            
            // check, if text denotes the score for player two
        } else if ((tx == PLAYER_TWO_SCORE_LOCATION_OTHER.getX())
                && (ty == PLAYER_ONE_SCORE_LOCATION_OTHER.getY())) {
            m_playerTwoScore = extractNumber(text);
                    
        // check, if text denotes the high score
        } else if ((tx == HIGHSCORE_LOCATION.getX()) 
                && (ty == HIGHSCORE_LOCATION.getY())) {
            m_highscore = extractNumber(text);
            
        // set flag for game start display, if text is game start text
        } else if (GAME_START_STRING.equals(text)) {
            m_waitingForStart = true;
            
        // set game starting flag, if player 1 string is displayed
        } else if (PLAYER1_STRING.equals(text)) {
            m_gameStarting = true;
            
        // set flag for game end display, if text is game end text
        } else if (GAME_END_STRING.equals(text)) {
            m_gameEnding = true;
            
        }
    }

    /**
     * Parse string into a number.
     * 
     * @param s string to parse
     * @return number or 0 if string cannot be converted
     */
    private int extractNumber(String s) {
        try {
            int number = Integer.parseInt(s.trim());
            return number;
        } catch (NumberFormatException e) {
            return NO_NUMBER;
        }
    }

    /**
     * Sets the coordinate system of the ship for all objects of the
     * game status. In other words, the coordinates of all objects will
     * be moved, so that the origin of the system is the location of
     * the ship.
     */
    private void setShipCoordinateSystem() {
        if (m_ship != null) {
            // set coordinate system for ship
            m_ship.setLocationRelativeTo(m_ship.getLocation());
            
            // set coordinate system for asteroids
            for (Asteroid a : m_asteroids) {
                a.setLocationRelativeTo(m_ship.getLocation());
            }
            
            // set coordinate system for bullets
            for (Bullet s : m_bullets) {
                s.setLocationRelativeTo(m_ship.getLocation());
            }
            
            // set coordinate system for saucer, if present
            if (m_saucer != null) {
                m_saucer.setLocationRelativeTo(m_ship.getLocation());
            }
        }
    }

    /**
     * Identifies the objects from this game status in the given previous game
     * status. Identified objects receive the same id number as their
     * counterpart in the previous game status. and their motion vector is
     * calculated and set.
     * 
     * @param prev a previous game status
     * @param commandMap maps ping bytes to the command objects that were 
     *  sent by the mame client with the corresponding ping byte. May be 
     *  <code>null</code>, which results in the ship not being tracked.  
     */
    private void trackObjects(GameStatus prev, Map<Byte, Commands> commandMap) {
        // determine frame gap
        byte frameGap = (byte)(m_serverFrame - prev.getServerFrame());
        
        // track asteroids
        List<Asteroid> prevAsteroids = prev.getAsteroids();
        findMatches(m_asteroids, prevAsteroids, frameGap, 
                EXACT_MATCH_STRATEGY);
        findMatches(m_asteroids, prevAsteroids, frameGap, 
                APPROXIMATE_ASTEROID_MATCH_STRATEGY);
        
        // track bullets
        List<Bullet> prevBullets = prev.getBullets();
        findMatches(m_bullets, prevBullets, frameGap, EXACT_MATCH_STRATEGY);
        findMatches(m_bullets, prevBullets, frameGap, 
                APPROXIMATE_BULLET_MATCH_STRATEGY);
        
        // determind source for bullets
        for (Bullet bullet : m_bullets) {
            bullet.setSource(m_ship, m_saucer);
        }
        
        // track explosions
        List<Explosion> prevExplosions = prev.getExplosions();
        findMatches(m_explosions, prevExplosions, frameGap, 
                EXACT_MATCH_STRATEGY);
        
        // track saucer
        Saucer prevSaucer = prev.getSaucer();
        if ((m_saucer != null) && (prevSaucer != null)) {
            m_saucer.track(prevSaucer, frameGap);
        }
        
        // track ship
        if (m_ship != null) {
            if (prev.m_ship != null) {
            
                // movement
                m_ship.track(prev.m_ship, frameGap);
                
                // internal shot direction
                if (commandMap != null) {
                    Commands commands = commandMap.get(prev.getPing());
                    if (commands != null) {
                        if (commands.isRightPressed()) {
                            m_ship.rotateRight(frameGap);
                        } else if (commands.isLeftPressed()) {
                            m_ship.rotateLeft(frameGap);
                        }
                    }
    
                    commands = commandMap.get(m_ping);
                    if (commands != null) {
                        if (commands.isRightPressed()) {
                            m_ship.correctRight();
                        } else if (commands.isLeftPressed()) {
                            m_ship.correctLeft();
                        }
                    }
                }
            } else if (prev.m_lastShotAngle != null) {
                m_ship.setLastKnownShotAngle(prev.m_lastShotAngle);
            }
            
            m_lastShotAngle = m_ship.getAngleByte();
        } else {
            m_lastShotAngle = prev.m_lastShotAngle;
        }
    }

    /**
     * Finds a matching game object from a previous frame for each given 
     * object from the current frame. Uses the given match strategy. 
     * 
     * @param current the list of game objects from the current frame
     * @param prev the list of game object from a previous frame
     * @param frameGap the difference between the frame numbers of the
     *  current frame and the the frame from where the given list of
     *  previous objects orginates 
     * @param strategy the match strategy
     */
    private void findMatches(List<? extends GameObject> currentList, 
            List<? extends GameObject> prevList, int frameGap, 
            IMatchStrategy strategy) {
        
        // initialize match list
        List<GameObject> matchList = new ArrayList<GameObject>();
        
        // track each object
        for (GameObject o : currentList) {
            
            // do not track objects twice
            if (o.getId() != GameObject.UNTRACKED) {
                continue;
            }
            
            // determine possible matches
            matchList.clear();
            for (GameObject p : prevList) {

                // objects must have same size in order to possibly match
                if (o instanceof IScaleableGameObject) {
                    IScaleableGameObject so = (IScaleableGameObject)o;
                    if (!(p instanceof IScaleableGameObject)
                            || ((IScaleableGameObject)p).getScaleFactor() != so.getScaleFactor()) {
                        continue;
                    }
                }
                
                // objects must have same type in order to possibly match
                if (o instanceof ITypeableGameObject) {
                    ITypeableGameObject to = (ITypeableGameObject)o;
                    if (!(p instanceof ITypeableGameObject)
                            || ((ITypeableGameObject)p).getType() != to.getType()) {
                        continue;
                    }
                }
                
                // check precondition from match strategy
                if (strategy.satisfiesPrecondition(o, p, frameGap)) {
                    matchList.add(p);
                }
            }
            
            // select best match
            GameObject bestMatch = null;
            if (matchList.size() == 1) {
                
                // only one possible match found -> use it
                bestMatch = matchList.get(0);
            } else if (matchList.size() > 1) {
                
                // more than one possible match -> select best match
                bestMatch = strategy.selectBestMatch(o, matchList, frameGap);
            }

            // perform the actual tracking, if match has been found
            if (bestMatch != null) {
                
                // set id and calculate direction vector
                o.track(bestMatch, frameGap);
                
                // make sure this match is not found again for any other
                // game object
                if (!prevList.remove(bestMatch)) {
                    throw new IllegalStateException("oops!");
                }
            }
        }
    }
    
    /**
     * Inherits the round number from the previous frame; increases
     * it, if a new round has just started. 
     * 
     * @param prev A previous game status or <code>null</code>, if
     *  no previous game status is available. If no previous game
     *  status is available, the round number will be set to <code>one</code>
     *  if a game is in progress, or to <code>zero</code> if not. 
     */
    private void updateRoundNumber(GameStatus prev) {
        if (m_waitingForStart) {
            
            // no game running
            m_round = 0;
            m_roundDuration = 0;
        } else if ((prev != null) && (!m_gameStarting)) {
            
            // inherit round duration value from previous game status
            m_roundDuration += prev.m_roundDuration;
            
            // increase round duration, if round is still running
            if (m_asteroids.size() > 0) {
                int frameGap = m_frameNo - prev.m_frameNo;
                m_roundDuration += frameGap;
            }
            
            // inherit round value from previous game status
            m_round = prev.getRound();
            
            // increase value, if a new round has just started
            boolean prevClear = prev.getAsteroidCount() == 0;
            boolean nowWithTargets = m_asteroids.size() > 0;
            if  (prevClear && nowWithTargets) {
                m_round++;
                m_roundDuration = 0;
            }
        } else {
            
            // game just starting or no previous game status available  
            // -> start with round zero
            m_round = 0;
            m_roundDuration = 0;
        }
    }

    /**
     * Upates the frame counter. Takes the current number of elapsed
     * frames from the given previous game status and adds the number of
     * frames that have elapsed between this game status and the given
     * one. Sets the game duration to <code>zero</code>, if the
     * {@link #m_gameStarting} flag is set. 
     * 
     * @param prev a previous game status
     */
    private void updateFrameCounter(GameStatus prev) {
        if (m_gameStarting) {
            m_frameNo = 0;
        } else {
            byte frameGap = (byte)(m_serverFrame - prev.getServerFrame());
            m_frameNo = prev.getFrameNo() + frameGap;
        }
    }

    /**
     * Checks the score of this game status versus the score of the
     * given previous game status. If an overflow is detected
     * 
     * @param prev a previous game status
     */
    private void correctScoreOverflow(GameStatus prev) {
        // do not modify the score, if the game is not running
        if (isGameRunning()) {
            // get score from previous frame
            int prevScore = prev.getPlayerOneScore();
            int overflowCount = prevScore / SCORE_OVERFLOW; 
            int prevGameScore = prevScore % SCORE_OVERFLOW;
            
            // detect overflow
            if (prevGameScore > m_playerOneScore) {
                overflowCount++;
            }
            
            if (!m_gameStarting) {
                // carry over previous overflows
                m_playerOneScore += overflowCount * SCORE_OVERFLOW;
            }
        }
    }

    /**
     * Returns a list of all shootable targets. This includes all asteroids
     * and the saucer, if present.
     * 
     * @return the list of targets
     */
    public final List<ScaleableGameObject> getTargetList() {
        ArrayList<ScaleableGameObject> targetList 
            = new ArrayList<ScaleableGameObject>();
        targetList.addAll(m_asteroids);
        if (m_saucer != null) {
            targetList.add(m_saucer);
        }
        return targetList;
    }
    
    /**
     * Returns a list of all obstacles that could technically collide with
     * the ship and thus destroy it. This includes all asteroids, all 
     * hostile bullets and the saucer, if present.
     * 
     * @return the list of obstacles
     */
    public final List<GameObject> getObstacleList() {
        
        // create obstacle list
        ArrayList<GameObject> obstacleList = new ArrayList<GameObject>();
       
        // add asteroids
        obstacleList.addAll(m_asteroids);
       
        // add hostile bullets
        for (Bullet bullet : m_bullets) {
            if (bullet.getSource() == Bullet.HOSTILE) {
                obstacleList.add(bullet);
            }
        }
       
        // add saucer, if present
        if (m_saucer != null) {
            obstacleList.add(m_saucer);
        }
       
        // return the obstacle list
        return obstacleList;
    }
    
    /**
     * @return the ship
     */
    public final Ship getShip() {
        return m_ship;
    }

    /**
     * Returns the asteroid with index i.
     * 
     * @param i the index of the asteroid to return
     */
    public final Asteroid getAsteroid(int i) {
        return m_asteroids.get(i);
    }
    
    /**
     * Searches the list of asteroids for an asteroid with the given
     * id and returns it.
     * 
     * @param id the id to search for
     * @return the determined asteroid or <code>null</code>, if
     *  this game status does not have an astroid with the given id
     */
    public final Asteroid getAsteroidById(int id) {
        for (Asteroid a : m_asteroids) {
            if (a.getId() == id) {
                return a;
            }
        }
        return null;
    }

    /**
     * @return the number of available asteroids
     */
    public final int getAsteroidCount() {
        return m_asteroids.size();
    }

    /**
     * Returns a list of all asteroids.
     * 
     * @return list of asteroids
     */
    public final List<Asteroid> getAsteroids() {
        ArrayList<Asteroid> asteroidList = new ArrayList<Asteroid>();
        asteroidList.addAll(m_asteroids);
        return asteroidList;
    }
    
    /**
     * Returns the bullet with index i.
     * 
     * @param i the index of the bullet to return
     */
    public final Bullet getBullet(int i) {
        return m_bullets.get(i);
    }
    
    /**
     * Searches the list of bullets for a bullet with the given
     * id and returns it.
     * 
     * @param id the id to search for
     * @return the determined bullet or <code>null</code>, if
     *  this game status does not have a bullet with the given id
     */
    public final Bullet getBulletById(int id) {
        for (Bullet b : m_bullets) {
            if (b.getId() == id) {
                return b;
            }
        }
        return null;
    }

    /**
     * @return the number of available bullets
     */
    public final int getBulletCount() {
        return m_bullets.size();
    }

    /**
     * Returns a list of all bullets.
     * 
     * @return list of bullets
     */
    public final List<Bullet> getBullets() {
        List<Bullet> bulletList = new ArrayList<Bullet>();
        bulletList.addAll(m_bullets);
        return bulletList;
    }
    
    /**
     * Returns the explosion with index i.
     * 
     * @param i the index of the explosion to return
     */
    public final Bullet getExplosion(int i) {
        return m_bullets.get(i);
    }
    
    /**
     * Searches the list of explosions for an explosion with the given
     * id and returns it.
     * 
     * @param id the id to search for
     * @return the determined explosion or <code>null</code>, if
     *  this game status does not have an explosion with the given id
     */
    public final Explosion getExplosionById(int id) {
        for (Explosion e : m_explosions) {
            if (e.getId() == id) {
                return e;
            }
        }
        return null;
    }

    /**
     * @return the number of available explosions
     */
    public final int getExplosionCount() {
        return m_explosions.size();
    }

    /**
     * Returns a list of all explosions.
     * 
     * @return list of explosions
     */
    public final List<Explosion> getExplosions() {
        List<Explosion> explosionList = new ArrayList<Explosion>();
        explosionList.addAll(m_explosions);
        return explosionList;
    }
    
    /**
     * @return the saucer
     */
    public final Saucer getSaucer() {
        return m_saucer;
    }

    /**
     * @return the lifeCount
     */
    public final int getLifeCount() {
        return m_lifeCount;
    }

    /**
     * @return the evenFrame
     */
    public final boolean isEvenFrame() {
        return m_evenFrame;
    }

    /**
     * @return the frameNo
     */
    public final byte getServerFrame() {
        return m_serverFrame;
    }

    /**
     * @return the ping
     */
    public final byte getPing() {
        return m_ping;
    }
    
    /**
     * @return the latencyCorrection
     */
    public final int getLatencyCorrection() {
        return m_latencyCorrection;
    }

    /**
     * @return the score
     */
    public final int getPlayerOneScore() {
        return m_playerOneScore;
    }

    /**
     * @return the score_player_two
     */
    public final int getPlayerTwoScore() {
        return m_playerTwoScore;
    }

    /**
     * @return the highscore
     */
    public final int getHighscore() {
        return m_highscore;
    }

    /**
     * @return the round
     */
    public final int getRound() {
        return m_round;
    }

    /**
     * @return the roundDuration
     */
    public final int getRoundDuration() {
        return m_roundDuration;
    }

    /**
     * @return the frame number
     */
    public final int getFrameNo() {
        return m_frameNo;
    }

    /**
     * @return the gameInProgress
     */
    public final boolean isWaitingForStart() {
        return m_waitingForStart;
    }

    /**
     * @return the gameStarting
     */
    public final boolean isGameStarting() {
        return m_gameStarting;
    }

    /**
     * @return <code>true</code>, if the game is currently running,
     *  <code>false</code>, if not
     */
    public final boolean isGameRunning() {
        return m_playerTwoScore == NO_NUMBER;
    }
    
    /**
     * @return the gameEnding
     */
    public final boolean isGameEnding() {
        return m_gameEnding;
    }
    
    /**
     * @author Henning Faber
     */
    private static class ShipDetectionParameters {
        
        private static final int PHASE_NO_DATA_YET = 0;
        private static final int PHASE_POSSIBLE_SHIP = 1;
        private static final int PHASE_DETECTION_COMPLETE = 2;
        
        private int m_vz;
        private int m_dx;
        private int m_dy;
        private int m_v1x;
        private int m_v1y;
        private int m_detectionPhase;

        /**
         * @param op the current op code
         * @param vx current x co-ordinate of the drawing ray
         * @param vy current y co-ordinate of the drawing ray
         * @return a ship object that represents the detected ship, or
         *  <code>null</code>, if the ship was not represented by the
         *  given vector ram data
         */
        public Ship detectShip(int op, int vx, int vy) {
            Ship ship = null;
            if ((op == 6) && (m_vz == 12) && (m_dx != 0) && (m_dy != 0)) {
                if (m_detectionPhase == PHASE_NO_DATA_YET) {
                    m_v1x = m_dx;
                    m_v1y = m_dy;
                    m_detectionPhase = PHASE_POSSIBLE_SHIP;
                } else if (m_detectionPhase == PHASE_POSSIBLE_SHIP) {
                    Point viewDirection = new Point(m_v1x - m_dx, 
                            m_v1y - m_dy);
                    ship = new Ship(vx, vy, viewDirection);
                    m_detectionPhase = PHASE_DETECTION_COMPLETE;
                }
            } else if (m_detectionPhase == PHASE_POSSIBLE_SHIP) {
                m_detectionPhase = PHASE_NO_DATA_YET;
            }
            
            return ship;
        }

        /**
         * @return the m_dx
         */
        public final int getDx() {
            return m_dx;
        }

        /**
         * @return the m_dy
         */
        public final int getDy() {
            return m_dy;
        }

        /**
         * @return the m_vz
         */
        public final int getVz() {
            return m_vz;
        }
    }
    
    /**
     * @author Henning Faber
     */
    private static class TextDetectionParameters {
        
        private final StringBuilder m_text;
        private boolean m_addedLetter; 
        private int m_tx;
        private int m_ty;
        
        public TextDetectionParameters() {
            super();
            m_text = new StringBuilder();
            m_addedLetter = false;
            m_tx = 0;
            m_ty = 0;
        }
        
        /**
         * Resets the m_addedLetter flag. 
         */
        public void resetAddFlag() {
            m_addedLetter = false;
        }

        /**
         * Adds the given letter to the text string. Remembers the given
         * location, if this is the first letter that is added.
         * 
         * @param ch the letter to add
         * @param vx current x co-ordinate of the drawing ray
         * @param vy current y co-ordinate of the drawing ray
         */
        public final void addLetter(Character ch, int vx, int vy) {
            m_text.append(ch);
            if (!m_addedLetter) {
                m_addedLetter = true;
                m_tx = vx;
                m_ty = vy;
            }
        }
        
        /**
         * @return the text
         */
        public final StringBuilder getText() {
            return m_text;
        }

        /**
         * @return the addedLetter
         */
        public final boolean isAddedLetter() {
            return m_addedLetter;
        }

        /**
         * @return the tx
         */
        public final int getTx() {
            return m_tx;
        }

        /**
         * @return the ty
         */
        public final int getTy() {
            return m_ty;
        }
    }
}
