/*
 * Copyright (C) 2008 Henning Faber
 * 
 * This file is part of Sitting Duck Asteroids Bot project.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. 
 */
package de.hfaber.asteroids.game.objects;

import java.util.HashMap;
import java.util.Map;

import de.hfaber.asteroids.game.field.Point;
import de.hfaber.asteroids.game.field.Screen;


/**
 * <p>The ship. In addition to the standard attributes of a game object, 
 * the ship also has...</p>
 * 
 * <ul>
 *  <li>a view direction. This is the direction, which is visually 
 *   observable from the data in the vector ram. There are 64 possible
 *   view directions. However, they cannot all be adopted within a single 
 *   360  turnaround.
 *  <li>a shot direction. This is the direction, where a shot will go, 
 *   if fired. There are 256 posible shot directions. However, they 
 *   cannot all be adopted within a single 360  turnaround. The shot
 *   direction is internally represented by an 'angle byte'. 
 * </ul> 
 * 
 * <p>Since the shot direction cannot be determined directly from the
 * data in the vector ram, it is interpolated over multiple successive
 * frames.</p>
 * 
 * @author Henning Faber
 */
public class Ship extends GameObject {
    
    /**
     * Maximum number of simultaneous bullets that a ship may fire.  
     */
    public static final int MAX_BULLETS = 4;
    
    /**
     * The square radius of the ship measured in sub pixels. 
     */
    private static final int SHIP_SQUARE_RADIUS = 27 * 27
            * Screen.INTERNAL_PRECISION * Screen.INTERNAL_PRECISION;

    // this code block has been created with the RecordAngleByte class
    private static final Map<Double, AngleByteRange> ANGLE_BYTE_MAP =
        new HashMap<Double, AngleByteRange>();

    static {
        ANGLE_BYTE_MAP.put(0.0, new AngleByteRange(253, 3));
        ANGLE_BYTE_MAP.put(0.0991502492198197, new AngleByteRange(4, 7));
        ANGLE_BYTE_MAP.put(0.19432494514023965, new AngleByteRange(8, 11));
        ANGLE_BYTE_MAP.put(0.2904592885742987, new AngleByteRange(12, 15));
        ANGLE_BYTE_MAP.put(0.39117522545472644, new AngleByteRange(16, 19));
        ANGLE_BYTE_MAP.put(0.48689923181126904, new AngleByteRange(20, 23));
        ANGLE_BYTE_MAP.put(0.589443524740529, new AngleByteRange(24, 27));
        ANGLE_BYTE_MAP.put(0.6860948744267478, new AngleByteRange(28, 31));
        ANGLE_BYTE_MAP.put(0.7853981633974483, new AngleByteRange(32, 35));
        ANGLE_BYTE_MAP.put(0.8847014523681488, new AngleByteRange(36, 39));
        ANGLE_BYTE_MAP.put(0.9813528020543676, new AngleByteRange(40, 43));
        ANGLE_BYTE_MAP.put(1.0838970949836275, new AngleByteRange(44, 47));
        ANGLE_BYTE_MAP.put(1.17962110134017, new AngleByteRange(48, 51));
        ANGLE_BYTE_MAP.put(1.2803370382205979, new AngleByteRange(52, 55));
        ANGLE_BYTE_MAP.put(1.376471381654657, new AngleByteRange(56, 59));
        ANGLE_BYTE_MAP.put(1.4716460775750768, new AngleByteRange(60, 63));
        ANGLE_BYTE_MAP.put(1.5707963267948966, new AngleByteRange(64, 64));
        ANGLE_BYTE_MAP.put(1.6699465760147165, new AngleByteRange(65, 68));
        ANGLE_BYTE_MAP.put(1.7651212719351363, new AngleByteRange(69, 72));
        ANGLE_BYTE_MAP.put(1.8612556153691955, new AngleByteRange(73, 76));
        ANGLE_BYTE_MAP.put(1.9619715522496233, new AngleByteRange(77, 80));
        ANGLE_BYTE_MAP.put(2.0576955586061656, new AngleByteRange(81, 84));
        ANGLE_BYTE_MAP.put(2.1602398515354255, new AngleByteRange(85, 88));
        ANGLE_BYTE_MAP.put(2.256891201221644, new AngleByteRange(89, 92));
        ANGLE_BYTE_MAP.put(2.356194490192345, new AngleByteRange(93, 96));
        ANGLE_BYTE_MAP.put(2.4554977791630455, new AngleByteRange(97, 100));
        ANGLE_BYTE_MAP.put(2.552149128849264, new AngleByteRange(101, 104));
        ANGLE_BYTE_MAP.put(2.654693421778524, new AngleByteRange(105, 108));
        ANGLE_BYTE_MAP.put(2.7504174281350666, new AngleByteRange(109, 112));
        ANGLE_BYTE_MAP.put(2.8511333650154946, new AngleByteRange(113, 116));
        ANGLE_BYTE_MAP.put(2.9472677084495538, new AngleByteRange(117, 120));
        ANGLE_BYTE_MAP.put(3.0424424043699734, new AngleByteRange(121, 124));
        ANGLE_BYTE_MAP.put(3.141592653589793, new AngleByteRange(125, 131));
        ANGLE_BYTE_MAP.put(-3.0424424043699734, new AngleByteRange(132, 135));
        ANGLE_BYTE_MAP.put(-2.9472677084495538, new AngleByteRange(136, 139));
        ANGLE_BYTE_MAP.put(-2.8511333650154946, new AngleByteRange(140, 143));
        ANGLE_BYTE_MAP.put(-2.7504174281350666, new AngleByteRange(144, 147));
        ANGLE_BYTE_MAP.put(-2.654693421778524, new AngleByteRange(148, 151));
        ANGLE_BYTE_MAP.put(-2.552149128849264, new AngleByteRange(152, 155));
        ANGLE_BYTE_MAP.put(-2.4554977791630455, new AngleByteRange(156, 159));
        ANGLE_BYTE_MAP.put(-2.356194490192345, new AngleByteRange(160, 163));
        ANGLE_BYTE_MAP.put(-2.256891201221644, new AngleByteRange(164, 167));
        ANGLE_BYTE_MAP.put(-2.1602398515354255, new AngleByteRange(168, 171));
        ANGLE_BYTE_MAP.put(-2.0576955586061656, new AngleByteRange(172, 175));
        ANGLE_BYTE_MAP.put(-1.9619715522496233, new AngleByteRange(176, 179));
        ANGLE_BYTE_MAP.put(-1.8612556153691955, new AngleByteRange(180, 183));
        ANGLE_BYTE_MAP.put(-1.7651212719351363, new AngleByteRange(184, 187));
        ANGLE_BYTE_MAP.put(-1.6699465760147165, new AngleByteRange(188, 191));
        ANGLE_BYTE_MAP.put(-1.5707963267948966, new AngleByteRange(192, 192));
        ANGLE_BYTE_MAP.put(-1.4716460775750768, new AngleByteRange(193, 196));
        ANGLE_BYTE_MAP.put(-1.376471381654657, new AngleByteRange(197, 200));
        ANGLE_BYTE_MAP.put(-1.2803370382205979, new AngleByteRange(201, 204));
        ANGLE_BYTE_MAP.put(-1.17962110134017, new AngleByteRange(205, 208));
        ANGLE_BYTE_MAP.put(-1.0838970949836275, new AngleByteRange(209, 212));
        ANGLE_BYTE_MAP.put(-0.9813528020543676, new AngleByteRange(213, 216));
        ANGLE_BYTE_MAP.put(-0.8847014523681488, new AngleByteRange(217, 220));
        ANGLE_BYTE_MAP.put(-0.7853981633974483, new AngleByteRange(221, 224));
        ANGLE_BYTE_MAP.put(-0.6860948744267478, new AngleByteRange(225, 228));
        ANGLE_BYTE_MAP.put(-0.589443524740529, new AngleByteRange(229, 232));
        ANGLE_BYTE_MAP.put(-0.48689923181126904, new AngleByteRange(233, 236));
        ANGLE_BYTE_MAP.put(-0.39117522545472644, new AngleByteRange(237, 240));
        ANGLE_BYTE_MAP.put(-0.2904592885742987, new AngleByteRange(241, 244));
        ANGLE_BYTE_MAP.put(-0.19432494514023965, new AngleByteRange(245, 248));
        ANGLE_BYTE_MAP.put(-0.0991502492198197, new AngleByteRange(249, 252));
    }
    
    /**
     * The view direction. 
     */
    private Point m_viewDirection;

    /**
     * The angle byte, represented as a range of values that are possible
     * for the ship's view direction. When tracking the ship over multiple 
     * successive frames, the range is reduced until its distance is one and
     * thus the exact angle byte has been found.  
     */
    private AngleByteRange m_angleByte;
    
    /**
     * <p>Holds the corrected angle byte value for the actual shot angle in 
     * relation to the shot angle tracked from the observations of the 
     * current vector ram.</p>
     * <p>Before receiving a game status, commands are sent to the mame
     * server. The server returns the current game status before executing
     * these commands. Now, if the commands contained actions for
     * rotating left or right, this rotation is not yet available in the
     * received vector ram. However, it needs to be considered in order
     * to have the correct shot angle.</p>  
     */
    private AngleByteRange m_correctedAngleByte;
    
    private int m_correctionAngle;
    
    /**
     * Creates a ship at the given location with the given view direction.
     * The angle byte will be set to the pre-calculated possible range
     * that is assigned to the given view direction. 
     * 
     * @param x the x coordinate of the ship
     * @param y the y coordinate of the ship
     * @param viewDirection the view direction
     */
    public Ship(int x, int y, Point viewDirection) {
        super(x, y);
        m_viewDirection = viewDirection;
        m_angleByte = ANGLE_BYTE_MAP.get(getViewAngle());
        m_correctionAngle = 0;
        preCalculateCorrectedAngleByte();
    }
    
    /**
     * Creates a ship that represents what the given ship is expected to 
     * be like in the given number of frames.
     * 
     * @param toProject the ship to project
     * @param frameCount the number of frames to project
     * @param steps the number of steps the ship will rotate until the
     *  new frame number is reached; a negative number denotes a clocwise
     *  rotation, a positive number denotes an anti-clockwise rotation      
     */
    public Ship(Ship toProject, int frameCount, int steps) {
        super(toProject, frameCount);
        m_viewDirection = toProject.m_viewDirection;
        m_angleByte = toProject.m_angleByte.addAngle(
                steps * AngleByteRange.ANGLE_BYTE_STEP_SIZE);
        m_correctionAngle = toProject.m_correctionAngle;
        preCalculateCorrectedAngleByte();
    }

    /**
     * Calculates the value of {@link #m_correctedAngleByte}, from 
     * the current value of the {@link #m_angleByte} and 
     * {@link #m_correctionAngle}. The value is hold redundantly to
     * avoid re-calculating it whenever the shot angle is quried.
     */
    private void preCalculateCorrectedAngleByte() {
        m_correctedAngleByte = m_angleByte.addAngle(m_correctionAngle);
    }

    /* (non-Javadoc)
     * @see de.hfaber.asteroids.game.objects.GameObject#trackCustomProperties(de.hfaber.asteroids.game.objects.GameObject, int)
     */
    @Override
    protected final void trackCustomProperties(GameObject prev, int frameGap) {
        // inherit the ship's shot angle
        Ship prevShip = (Ship)prev;
        m_angleByte = prevShip.m_angleByte;
        preCalculateCorrectedAngleByte();
    }

    /* (non-Javadoc)
     * @see de.hfaber.asteroids.asteroids.game.objects.GameObject#getPrecision()
     */
    @Override
    protected int getPrecision() {
        // TODO: implement decent ship tracking!
        return 1;
    }

    /* (non-Javadoc)
     * @see de.hfaber.asteroids.asteroids.game.objects.GameObject#getMaxFrameDist()
     */
    @Override
    public final int getMaxFrameDist() {
        // TODO: what is the maximum ship speed?
        return 0;
    }

    /* (non-Javadoc)
     * @see de.hfaber.asteroids.asteroids.game.objects.GameObject#getRadius()
     */
    @Override
    public int getSquareRadius() {
        return SHIP_SQUARE_RADIUS;
    }

    /**
     * Adjusts the internal angle of the ship to reflect a rotation
     * to the left by the number of given steps.
     *  
     * @param steps the number of steps the ship should be rotated
     */
    public final void rotateLeft(int steps) {
        m_angleByte = m_angleByte.addAngle(steps * AngleByteRange.ANGLE_BYTE_STEP_SIZE);
        m_angleByte = m_angleByte.intersect(ANGLE_BYTE_MAP.get(getViewAngle()));
        preCalculateCorrectedAngleByte();
    }
    
    /**
     * Adjusts the internal angle of the ship to reflect a rotation
     * to the right by one frame.
     *  
     * @param steps the number of steps the ship should be rotated
     */
    public final void rotateRight(int steps) {
        m_angleByte = m_angleByte.addAngle(steps * -AngleByteRange.ANGLE_BYTE_STEP_SIZE);
        m_angleByte = m_angleByte.intersect(ANGLE_BYTE_MAP.get(getViewAngle()));
        preCalculateCorrectedAngleByte();
    }
    
    public final void correctLeft() {
        m_correctionAngle = AngleByteRange.ANGLE_BYTE_STEP_SIZE;
        preCalculateCorrectedAngleByte();
    }

    public final void correctRight() {
        m_correctionAngle = -AngleByteRange.ANGLE_BYTE_STEP_SIZE;
        preCalculateCorrectedAngleByte();
    }
    
    /**
     * Allows to specify the value of the angle byte. This may be used
     * on the first frame the ship re-appears in the game after a hyperspace
     * jump or a lost life. In this situation, there is no previous 
     * ship available, so the angle byte synchronisation would re-start
     * from the beginning. However, the last known angle byte is still
     * valid. 
     */
    public final void setLastKnownShotAngle(AngleByteRange lastKnownShotAngle) {
        m_angleByte = m_angleByte.intersect(lastKnownShotAngle);
        preCalculateCorrectedAngleByte();
    }
    
    /**
     * @return the angleByte
     */
    public final AngleByteRange getAngleByte() {
        return m_angleByte;
    }

    /**
     * Returns the number of rotation steps that are required to reach
     * the given internal angle byte from the current internal angle byte. 
     * 
     * @param targetAngle the angle byte position to reach
     * @return the number of requuired steps; a negative number denotes 
     *  a clocwise rotation, a positive number denotes an anti-clockwise 
     *  rotation 
     */
    public final int getRequiredSteps(int targetAngle) {
        int rotation = targetAngle - getShotAngle();
        rotation = Point.normalize(rotation, AngleByteRange.ANGLE_BYTE_VALUE_COUNT);
        return rotation;
    }
    
    /**
     * Calculates the view angle from the view direction and
     * returns it. 
     * 
     * @return the view angle
     */
    public final double getViewAngle() {
        double angle = Math.atan2(m_viewDirection.getY(), 
                m_viewDirection.getX());
        return angle;
    }
    
    /**
     * Returns the current angle byte value.
     * 
     * @return the current angle
     */
    public final int getShotAngle() {
        return m_correctedAngleByte.getAngle();
    }
    
    /**
     * Returns the internal shot direction according to the value
     * of the angle byte.
     * 
     * @return the shot direction
     */
    public final Point getShotDirection() {
        // get angle
        int angleByte = getShotAngle();
        double angle = Math.PI * angleByte / 128;
        
        Point direction = getDirection(); 
        int astsin = (int)Math.round(127 * Math.sin(angle));
        int astcos = (int)Math.round(127 * Math.cos(angle));
        int fx = (int)(direction.getX() + Math.floor(astcos >> 1));
        int fy = (int)(direction.getY() + Math.floor(astsin >> 1));
        
        // correct values, if over limit
        if (fx > 111) {
            fx = 111;
        }
        if (fx < -111) {
            fx = -111;
        }
        if (fy > 111) {
            fy = 111;
        }
        if (fy < -111) {
            fy = -111;
        }

        // return the shot direction vector
        Point shotDirection = new Point(fx, fy);
        return shotDirection;
    }
    
    /**
     * Returns the speed a bullet would have if the ship fires with
     * the current angle and speed.
     * 
     * @return the bullet speed
     */
    public final double getBulletSpeed() {
        Point shotDirection = getShotDirection();
        double shotSpeed = shotDirection.length();
        return shotSpeed;
    }

    /**
     * @return the viewDirection
     */
    public final Point getViewDirection() {
        return m_viewDirection;
    }

    /* (non-Javadoc)
     * @see de.hfaber.asteroids.asteroids.gameobjects.GameObject#toString()
     */
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(getClass().getSimpleName());
        sb.append("[id=");
        sb.append(getId());
        sb.append(" x=");
        sb.append(getRelativeLocation().getX());
        sb.append(" y=");
        sb.append(getRelativeLocation().getY());
        sb.append(" vx=");
        sb.append(m_viewDirection.getX());
        sb.append(" vy=");
        sb.append(m_viewDirection.getY());
        sb.append("]");
        return sb.toString();
    }
}
