/*
 * Copyright (C) 2008 Henning Faber
 * 
 * This file is part of Sitting Duck Asteroids Bot project.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. 
 */
package de.hfaber.asteroids.game.objects;

import de.hfaber.asteroids.game.field.Point;
import de.hfaber.asteroids.game.field.Screen;

/**
 * <p>An abstract game object. A game object provides the following basic
 * attributes:</p>
 * 
 * <ul>
 *  <li>A unique object id. As long as an object is successfully tracked 
 *   over multiple successive frames, its id remains identical.
 *  <li>A location, given as a point with x and y coordinates.
 *  <li>A direction, given as vector.
 *  <li>A speed, which is determined by the length of the direction vector.
 * </ul>
 * 
 * @author Henning Faber
 */
public abstract class GameObject implements ITrackableGameObject, 
        Comparable<GameObject> {

    /**
     * Id of a game object that has not yet been tracked.
     */
    public static final int UNTRACKED = 0;

    /**
     * <p>Motion signature array used for calculating the sub pixel
     * location, which cannot be determined from the observation of
     * the data in the vector ram, but can at best be interpolated
     * from observing the movement of the object over several successive
     * frames.</p>  
     * <p>A delta value is the number of pixels the object has moved between 
     * two successive frames. An array of delta values holds the observed
     * motion over several frames. The values in this array are cyclic,
     * i.e. they repeat at the latest after {@link Screen#INTERNAL_PRECISION} 
     * frames. However, when recording the delta values, it is undefined
     * where within this cycle the object is when the recording starts.
     * Thus, the observed delta values have to be counted back to the
     * beginning of the cycle. The motion signature array holds the
     * signatures that define the start of the cycle for the different 
     * available motion speeds.</p>
     * <p>A signature stores a zero bit for each position where the
     * array of delta values has the lower of two possible values.
     * It stores a one bit for each position where the array holds
     * the higher delta value of two possible values.</p>
     */
    private static final int[] MOTION_SIGNATURES = {
        255,    // 1111 1111
        128,    // 0000 0001
        136,    // 0001 0001
        164,    // 0010 0101
        170,    // 0101 0101
        218,    // 0101 1011
        238,    // 0111 0111
        254     // 0111 1111
    };
    
    /**
     * The next available unique game object id. 
     */
    private static int nextId = 1; 
    
    /**
     * A unique id that identifies an object between successive frames.   
     */
    private int m_id;
    
    /**
     * The location of the object (in sub space coordinates). 
     */
    private Point m_location;
    
    /**
     * The location relative to a certain origin.
     */
    private Point m_relativeLocation;

    /**
     * The location of the object as observed from the screen (but still
     * in sub space coordinates). This value is only needed internally
     * when calculating the corrected object location from the observed
     * motion of the object.  
     */
    private Point m_screenLocation;

    /**
     * The moving direction of this game object. 
     */
    private Point m_direction;
    
    /**
     * The number of delta values that have been recorded from previous
     * frames. 
     */
    private int m_deltaCount;
    
    /**
     * Pointer within the delta value arrays.
     */
    private int m_nextDeltaPtr;
    
    /**
     * Array of recorded movements of this object along the y axis.
     */
    private int[] m_deltaValuesX;
    
    /**
     * Array of recorded movements of this object along the y axis.
     */
    private int[] m_deltaValuesY;
    
    /**
     * Creates a game object at the given location. Speed and direction
     * are set to zero.
     * 
     * @param x the x coordinate of the game object
     * @param y the y coordinate of the game object
     */
    public GameObject(int x, int y) {
        super();
        m_id = UNTRACKED;
        m_screenLocation = new Point(x * Screen.INTERNAL_PRECISION, 
                y * Screen.INTERNAL_PRECISION, true);
        m_location = m_screenLocation;
        m_relativeLocation = m_location;
        m_direction = new Point(0, 0);
        m_deltaValuesX = new int[getPrecision()];
        m_deltaValuesY = new int[getPrecision()];
        m_deltaCount = 0;
        m_nextDeltaPtr = 0;
    }
    
    /**
     * Creates a game object that represents what the given game
     * object is expected to be like in the given number of frames.
     * 
     * @param toProject the game object to project
     * @param frameCount the number of frames to project
     */
    public GameObject(GameObject toProject, int frameCount) {
        super();
        m_id = toProject.m_id;
        m_screenLocation = toProject.m_screenLocation;
        m_location = toProject.project(frameCount);
        m_relativeLocation = toProject.relativeProject(frameCount);
        m_direction = toProject.m_direction;
        m_deltaValuesX = new int[getPrecision()];
        System.arraycopy(toProject.m_deltaValuesX, 0, m_deltaValuesX, 0,
                getPrecision());
        m_deltaValuesY = new int[getPrecision()];
        System.arraycopy(toProject.m_deltaValuesY, 0, m_deltaValuesY, 0,
                getPrecision());
        m_deltaCount = toProject.m_deltaCount;
        m_nextDeltaPtr = toProject.m_nextDeltaPtr;
    }


    /**
     * Return the precision used for tracking directions of objects of
     * this class. 
     */
    protected abstract int getPrecision();
    
    /**
     * Return the square of the maximum distance that an object of this 
     * class may move on the screen between two successive frames.
     * 
     * @return the maximum frame distance
     */
    public abstract int getMaxFrameDist();
    
    /**
     * Return the square of the radius of this game object.
     * 
     * @return the square radius
     */
    public abstract int getSquareRadius();
    
    /* (non-Javadoc)
     * @see de.hfaber.asteroids.game.objects.ITrackableGameObject#track(de.hfaber.asteroids.game.objects.GameObject, int)
     */
    public final void track(GameObject prev, int frameGap) {
        // track object id
        trackId(prev);
        
        // update direction vector
        trackMovement(prev, frameGap);
        
        // track custom properties
        trackCustomProperties(prev, frameGap);
    }

    /**
     * Inherits the id from the given tracked game object or assigns
     * a new id, if the given object was untracked. 
     * 
     * @param prev the game object that represented this game object
     *  in a previous frame 
     */
    private void trackId(GameObject prev) {
        // inherit id from tracked object
        m_id = prev.getId();
        
        // if tracked obect was new, assign a new id
        if (m_id == UNTRACKED) {
            m_id = getNextId();
        }
    }
    
    /**
     * Returns the next available object id.
     * 
     * @return the next available object id
     */
    private static synchronized int getNextId() {
        return nextId++;
    }

    /**
     * Updates the array of recorded movements with the number of pixels 
     * that this game object has moved in relation to the given previous
     * game object. Sub classes may overwrite this method to implement
     * their own behaviour.
     * 
     * @param prev the game object that represented this game object
     *  in a previous frame 
     * @param frameGap the difference between the frame numbers of the
     *  frame where this game object belongs to and the frame from where 
     *  the given previous game object originates 
     */
    private void trackMovement(GameObject prev, int frameGap) {
        // inherit direction parameters from previous object
        m_nextDeltaPtr = prev.m_nextDeltaPtr;
        m_deltaCount = prev.m_deltaCount;
        System.arraycopy(prev.m_deltaValuesX, 0, m_deltaValuesX, 0, 
                getPrecision());
        System.arraycopy(prev.m_deltaValuesY, 0, m_deltaValuesY, 0, 
                getPrecision());

        // calculate next delta value
        Point delta = prev.m_screenLocation.delta(m_screenLocation);

        // frames lost?
        if (frameGap > 1) {
            
            // maximum precision already available?
            if (m_deltaCount == getPrecision()) {
                
                // maximum number of movements required for maximum 
                // precision have already been recorded before
                // -> simply adjust the delta pointer
                m_nextDeltaPtr = (m_nextDeltaPtr + frameGap) % getPrecision();
            } else {
                
                // maximum precision is not yet available
                // -> interpolate movement over the lost frames
                double dx = delta.getX() / (double)frameGap;
                double dy = delta.getX() / (double)frameGap;
                double x = 0;
                double y = 0;
                int correctedFrameGap = (frameGap - 1) % getPrecision() + 1;
                for (int i = 0; i < correctedFrameGap; i++) {
                    
                    // set interpolated delta
                    m_deltaValuesX[m_nextDeltaPtr] = (int)(Math.floor(x + dx)
                            - Math.floor(x)); 
                    m_deltaValuesY[m_nextDeltaPtr] = (int)(Math.floor(y + dy) 
                            - Math.floor(y));
                    m_nextDeltaPtr = (m_nextDeltaPtr + 1) % getPrecision();
                    
                    // remember values for next loop
                    x += dx;
                    y += dy;
                    
                    // increase counter, if maximum precision is not yet reached
                    if (m_deltaCount < getPrecision()) {
                        m_deltaCount++;
                    }
                }
            }
        } else {
            // no frames lost
            m_deltaValuesX[m_nextDeltaPtr] = delta.getX(); 
            m_deltaValuesY[m_nextDeltaPtr] = delta.getY();
            m_nextDeltaPtr = (m_nextDeltaPtr + 1) % getPrecision();
        }
        
        // increase counter, if maximum precision is not yet reached
        if (m_deltaCount < getPrecision()) {
            m_deltaCount++;
        }

        // correct location by sub pixel remainder that cannot be
        // observed from the vector ram, but can be interpolated from
        // the observed movement
        int remainderX = getSubPixelRemainder(m_deltaValuesX, m_deltaCount,
                m_nextDeltaPtr);
        int remainderY = getSubPixelRemainder(m_deltaValuesY, m_deltaCount,
                m_nextDeltaPtr);
        m_location = new Point(m_screenLocation.getX() + remainderX, 
                m_screenLocation.getY() + remainderY);
        
        // calculate direction vector
        m_direction = calculateDirection();
    }
    
    /**
     * Sub classes may use this method to track their own custom
     * properties.
     * 
     * @param prev the game object that represented this game object
     *  in a previous frame 
     * @param frameGap the difference between the frame numbers of the
     *  frame where this game object belongs to and the frame from where 
     *  the given previous game object originates 
     */
    protected abstract void trackCustomProperties(GameObject prev, 
            int frameGap);
    
    /**
     * Tries to determine the sub pixel remainder for an observed
     * sequence of delta values and a given position inside this
     * sequence.
     *   
     * @param deltaValues the observed sequence of delta values
     * @param length the lenght of the <code>deltaValues</code> array
     * @param deltaPtr the position within the <code>deltaValues</code> 
     *  array, for which the sub pixel remainder should be calculated
     * @return the sub pixel remainder
     */
    private int getSubPixelRemainder(int[] deltaValues, int length, 
            int deltaPtr) {
        // initialize the return value
        int remainder = 0;

        // get lowe and high values
        int lowValue = Integer.MAX_VALUE;
        int highValue = Integer.MIN_VALUE;
        for (int i = 0; i < length; i++) {
            int value = Math.abs(deltaValues[i]);
            if (value < lowValue) {
                lowValue = value; 
            }
            if (value > highValue) {
                highValue = value; 
            }
        }        

        // build signature for delta values
        int signature = 0;
        for (int i = 0; i < length; i++) {
            int value = Math.abs(deltaValues[i]);
            if (value == highValue) {
                signature = signature | (1 << i);
            }
        }

        // determine taret signature
        int averageDelta = getAverageDeltaValue(deltaValues, length);
        int targetSignature = MOTION_SIGNATURES[Math.abs(averageDelta)
                % Screen.INTERNAL_PRECISION];

        // determine the phase shift of the delta value array
        for (int phase = 0; phase < length; phase++) {
            int shiftedSignature = ((signature << phase) 
                    & ((1 << Screen.INTERNAL_PRECISION) - 1))
                    | (signature >> (Screen.INTERNAL_PRECISION - phase));
            if (shiftedSignature == targetSignature) {
                
                // calculate the offset of the given delta pointer in
                // relation to the determined phase shift
                int offset = (deltaPtr + phase) % length;
                
                // multiply the offset with the average delta value
                int overallSubPixels = offset * averageDelta;

                // determine the sub pixel remainder by dividing
                // to obtain screen coordinates
                remainder = overallSubPixels % Screen.INTERNAL_PRECISION;

                // remainder has been determined
                break;
            }
        }
        
        // return the result
        return remainder;
    }

    /**
     * Calculates the current direction vector for this game object.
     * 
     * @return the direction vector for this game object
     */
    private Point calculateDirection() {
        // moving direction can only be determined, if at least
        // one delta value between the current and a previous position
        // has been observed
        Point direction;
        if (m_deltaCount > 0) {
            int sumX = getAverageDeltaValue(m_deltaValuesX, m_deltaCount);        
            int sumY = getAverageDeltaValue(m_deltaValuesY, m_deltaCount);        
            direction = new Point(sumX, sumY);
        } else {
            direction = new Point(0, 0);
        }
        
        // return the objects moving direction
        return direction;
    }

    /**
     * Calculates the average deltta value for the given list of delta 
     * values.
     * 
     * @param deltaValues the list of delta values, for which the average
     *  should be calculated
     * @param length the length of the list of delta values
     * @return the average delta value
     */
    private int getAverageDeltaValue(int[] deltaValues, int length) {
        int sum = 0;
        for (int i = 0; i < length; i++) {
            sum += deltaValues[i];
        }
        int average = sum / length;
        return average;
    }
    
    /**
     * Projects where this game object will be located after the given
     * number of frames have elapsed.
     * 
     * @param frames the numer of frames to project
     * @return the projected new location of the game object
     */
    public final Point project(int frames) {
        Point newLocation = new Point(
                m_location.getX() + frames * m_direction.getX(), 
                m_location.getY() + frames * m_direction.getY(), true);
        return newLocation;
    }

    /**
     * Projects where this game object will be located (based on the relative
     * location given by {@linkplain #m_relativeLocation}) after the given 
     * number of frames have elapsed.
     * 
     * @param frames the numer of frames to project
     * @return the projected new location of the game object based on the 
     *  relative location given by {@linkplain #m_relativeLocation}
     */
    public final Point relativeProject(int frames) {
        Point newLocation = new Point(
                m_relativeLocation.getX() + frames * m_direction.getX(), 
                m_relativeLocation.getY() + frames * m_direction.getY(), true);
        return newLocation;
    }
    
    /**
     * Returns the reliability of the direction vector. If the vector has
     * only been tracked over a one or few frames, it may not yet be the
     * correct vector and could thus change in successive frames.  
     * 
     * @return <code>true</code>, if the direction vector is reliable, i.e.
     *  will not change anymore in successive frames, <code>false</code>,
     *  if not
     */
    public final boolean isDirectionReliable() {
        return m_deltaCount == getPrecision();
    }
    
    /**
     * @return the id
     */
    public final int getId() {
        return m_id;
    }

    /**
     * @return the location
     */
    public final Point getLocation() {
        return m_location;
    }

    /**
     * @return the relativeLocation
     */
    public final Point getRelativeLocation() {
        return m_relativeLocation;
    }

    /**
     * Sets the relative location of the object in relation to the given
     * origin.
     * 
     * @param origin the point to relate to
     */
    public final void setLocationRelativeTo(Point origin) {
        m_relativeLocation = origin.delta(m_location);
    }
    
    /**
     * @return the direction
     */
    public final Point getDirection() {
        return m_direction;
    }


    /* (non-Javadoc)
     * @see java.lang.Comparable#compareTo(java.lang.Object)
     */
    @Override
    public int compareTo(GameObject o) {
        return m_id - o.getId();
    }

    /* (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(getClass().getSimpleName());
        sb.append("[id=");
        sb.append(getId());
        sb.append(" x=");
        sb.append(m_relativeLocation.getX());
        sb.append(" y=");
        sb.append(m_relativeLocation.getY());
        sb.append("]");
        return sb.toString();
    }
}
