/*
 * Copyright (C) 2008 Henning Faber
 * 
 * This file is part of Sitting Duck Asteroids Bot project.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. 
 */
package de.hfaber.asteroids.game.objects;

import org.apache.log4j.Logger;

import de.hfaber.asteroids.game.field.Point;

/**
 * <p>Represents a range of possible angle bytes.</p> 
 * 
 * <p>Each view angle of the ship, which is observable from the data 
 * in the vector ram, has a range of possible internal shot angles 
 * assigned. These shot angles can be represented with a single 
 * 'angle byte'.</p>
 * 
 * <p>This method for tracking the angle byte was inspired by the
 * following post: 
 * http://www.heise.de/ct/foren/forum-135513/msg-14793968/read/</p>
 * 
 * @author Henning Faber
 */
public class AngleByteRange {

    /**
     * The LOGGER instance. 
     */
    private static final Logger LOGGER = Logger.getLogger(
            AngleByteRange.class);

    /**
     * The number of possible angle byte values.
     */
    public static final int ANGLE_BYTE_VALUE_COUNT = 256;

    /**
     * The absolute value that the angel byte is increased or decreased  when rotating a single step to the left or right.
     */
    public static final int ANGLE_BYTE_STEP_SIZE = 3;

    /**
     * The maximum 'sane' distance between the low byte and the high 
     * byte. If this distance is exceeded, something went wrong with
     * the interpolation of the angle byte and the whole process is
     * restarted.   
     */
    private static final int MAX_RANGE = 7;
    
    /**
     * The lower boundary.
     */
    private final int m_lowByte;
    
    /**
     * The upper boundary.
     */
    private final int m_highByte;

    /**
     * The angle that is located in the middle between the low and 
     * the high byte.
     */
    private final int m_middleByte;

    /**
     * Creates an angle byte range with the given parameters.
     * 
     * @param lowByte the lower boundary
     * @param highByte the upper boundary
     */
    public AngleByteRange(int lowByte, int highByte) {
        super();
        
        // assign low and high boundary
        m_lowByte = lowByte;
        m_highByte = highByte;
        
        // calculate best assumable angle for the this byte range
        int diff = (ANGLE_BYTE_VALUE_COUNT + m_highByte - m_lowByte)
                % ANGLE_BYTE_VALUE_COUNT;
        m_middleByte = (m_lowByte + (diff >> 1)) & 0xff;
    }
    
    /**
     * Returns the angle that is located in the middle between the
     * low and the high byte.
     * 
     * @return the best assumable angle for the angle byte range
     */
    public final int getAngle() {
        return m_middleByte;
    }
    
    /**
     * Adds the given value to the low and high bytes.
     * 
     * @param value the value to add
     */
    public final AngleByteRange addAngle(int value) {
        int newLowByte = (m_lowByte + value) & 0xff;
        int newHighByte = (m_highByte + value) & 0xff;
        AngleByteRange newRange = new AngleByteRange(newLowByte, 
                newHighByte);
        return newRange;
    }
    
    /**
     * Calculates the intersection between the given angle byte
     * range and this angle byte range. If the intersection is
     * empty (which should usually not happen), the given angle 
     * byte range is returned instead.
     * 
     * @param other the other ange byte range
     * @return a new angle byte range object that represents the 
     *  intersection between this angle byte range and the given
     *  range
     */
    public final AngleByteRange intersect(AngleByteRange other) {
        int lowByteDiff = other.m_lowByte - m_lowByte;
        lowByteDiff = Point.normalize(lowByteDiff, ANGLE_BYTE_VALUE_COUNT);
        int lowByte;
        if (lowByteDiff > 0) {
            lowByte = other.m_lowByte;
        } else {
            lowByte = m_lowByte;
        }
        
        int highByteDiff = other.m_highByte - m_highByte;
        highByteDiff = Point.normalize(highByteDiff, ANGLE_BYTE_VALUE_COUNT);
        int highByte;
        if (highByteDiff < 0) {
            highByte = other.m_highByte;
        } else {
            highByte = m_highByte;
        }
        
        int diff = (ANGLE_BYTE_VALUE_COUNT + highByte - lowByte)
                % ANGLE_BYTE_VALUE_COUNT;
        if (diff > MAX_RANGE) {
            LOGGER.warn("Angle byte synchronisation lost! Resetting.");
            lowByte = other.m_lowByte;
            highByte = other.m_highByte;
        } else if (LOGGER.isDebugEnabled() && (lowByte != highByte)) {
            StringBuilder sb = new StringBuilder();
            sb.append("Intesect resulted in a range [lowByte=");
            sb.append(lowByte);
            sb.append(" highByte=");
            sb.append(highByte);
            sb.append("]");
            LOGGER.debug(sb.toString());
        }
        
        AngleByteRange newRange = new AngleByteRange(lowByte, highByte);
        return newRange;
    }
    
    /* (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(getClass().getSimpleName());
        sb.append("[");
        sb.append(m_lowByte);
        sb.append(":");
        sb.append(m_highByte);
        sb.append("]");
        return sb.toString();
    }
}
