package de.wens02;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Point;
import java.util.Vector;

public class Ship extends MovingObject
{
	// JNI-Methoden benutzen:
	private static boolean useJNIMethods = true;
	
	/** Aus Performanzgrnden die zeitfressende Routine findBestTarget in C++ */
	public static native Target findBestTargetJNI(
			final int x, final int y,
			Vector<Target> targets,
			final int wbStart, final int minKeyPress, final int maxKeyPress,
			final int currentShotFrames,
			final int frameHorizon );

	static {
		System.loadLibrary( "ship" );
	}
	/** Richtungsvektoren des Schiffs, ungenau */
	protected int					x2									= 0;
	protected int					y2									= 0;
	protected boolean			visible							= false;

	/** Winkelbyte, liefert Kenntnis ber exakte Ausrichtung des Schiffs */
	public static boolean winkelbyte_resync		= false;
	public static int			winkelbyte					= 0;

	Vector<Integer>				possiblePosition		= new Vector<Integer>();

	protected static Vector<Shot> shots				= new Vector<Shot>();

	private Point					lPoint							= new Point(0,0);
	private Point					rPoint							= new Point(0,0);

	/** Fr die Mittlung der letzten 8 dx,dy-Werte */
	private int[]					ship_dx							= new int[8];
	private int[]					ship_dy							= new int[8];
	private int						ship_di							= 0;

	/** Indiziert ber das winkelbyte die Schussweite/Frame */
	public static double	shot_dist[]					= new double[256];

	/** Indiziert ber das winkelbyte die Winkel */
	public static double	shot_alpha[]				= new double[256];

	/** Indiziert ber Winkelbyte die Schussoffsets bei stehendem Schiff */
	public static int			shot_offset_x[][]		= new int[256][74];
	public static int			shot_offset_y[][]		= new int[256][74];
	
	/** Dito, normalisiert und auf aktuelle Schiffsposition bezogen */
	public static short		shot_offset_norm_x[][]	= new short[256][74];
	public static short		shot_offset_norm_y[][]	= new short[256][74];
	
	/** Aus der Tabelle shot_delta_tmp extrahierte Werte */
	public static int			shot_origin[][]			= new int[256][2];
	public static int			ship_delta[][]			= new int[256][2];
	public static double	shot_delta[][]			= new double[256][2];

	/** Indiziert ber das winkelbyte shot_dx[][0], shot_dy[][1], ship_dx[][2], ship_dy[][3] */
	public static double	shot_delta_tmp[][]	= {
				//  shot_dx,dy           ship_dx,dy    x0,y0
		 {   7.875000,   0.000000,  1536,     0,  19,   0 }, // alpha =  90.000000 [  0]
		 {   7.875000,   0.500000,  1536,     0,  19,   1 }, // alpha =  86.367049 [  1]
		 {   7.875000,   1.125000,  1528,   152,  19,   2 }, // alpha =  81.869898 [  2]
		 {   7.750000,   1.750000,  1504,   296,  19,   4 }, // alpha =  77.275644 [  3]
		 {   7.625000,   2.250000,  1472,   440,  19,   5 }, // alpha =  73.559620 [  4]
		 {   7.375000,   2.875000,  1472,   440,  18,   7 }, // alpha =  68.702646 [  5]
		 {   7.125000,   3.375000,  1416,   584,  17,   8 }, // alpha =  64.653824 [  6]
		 {   6.875000,   3.875000,  1360,   720,  17,   9 }, // alpha =  60.592811 [  7]
		 {   6.625000,   4.375000,  1280,   856,  16,  10 }, // alpha =  56.560131 [  8]
		 {   6.250000,   4.875000,  1280,   856,  15,  12 }, // alpha =  52.045769 [  9]
		 {   5.875000,   5.250000,  1192,   976,  14,  13 }, // alpha =  48.215484 [ 10]
		 {   5.500000,   5.750000,  1088,  1088,  13,  14 }, // alpha =  43.726970 [ 11]
		 {   5.000000,   6.125000,   976,  1192,  12,  15 }, // alpha =  39.225674 [ 12]
		 {   4.500000,   6.500000,   976,  1192,  11,  16 }, // alpha =  34.695154 [ 13]
		 {   4.000000,   6.750000,   856,  1280,  10,  16 }, // alpha =  30.650668 [ 14]
		 {   3.500000,   7.000000,   720,  1360,   8,  17 }, // alpha =  26.565051 [ 15]
		 {   3.000000,   7.250000,   584,  1416,   7,  18 }, // alpha =  22.479434 [ 16]
		 {   2.500000,   7.500000,   584,  1416,   6,  18 }, // alpha =  18.434949 [ 17]
		 {   1.875000,   7.625000,   440,  1472,   4,  19 }, // alpha =  13.815025 [ 18]
		 {   1.375000,   7.750000,   296,  1504,   3,  19 }, // alpha =  10.060690 [ 19]
		 {   0.750000,   7.875000,   152,  1528,   1,  19 }, // alpha =   5.440332 [ 20]
		 {   0.125000,   7.875000,   152,  1528,   0,  19 }, // alpha =   0.909380 [ 21]
		 {  -0.375000,   7.875000,  -152,  1528,  -1,  19 }, // alpha =  -2.726311 [ 22]
		 {  -1.000000,   7.875000,  -296,  1504,  -3,  19 }, // alpha =  -7.236922 [ 23]
		 {  -1.625000,   7.750000,  -296,  1504,  -5,  19 }, // alpha = -11.842076 [ 24]
		 {  -2.125000,   7.625000,  -440,  1472,  -6,  19 }, // alpha = -15.572544 [ 25]
		 {  -2.750000,   7.500000,  -584,  1416,  -7,  18 }, // alpha = -20.136303 [ 26]
		 {  -3.250000,   7.250000,  -720,  1360,  -9,  18 }, // alpha = -24.145542 [ 27]
		 {  -3.750000,   7.000000,  -720,  1360, -10,  17 }, // alpha = -28.178590 [ 28]
		 {  -4.250000,   6.625000,  -856,  1280, -11,  16 }, // alpha = -32.680555 [ 29]
		 {  -4.750000,   6.375000,  -976,  1192, -12,  15 }, // alpha = -36.689723 [ 30]
		 {  -5.250000,   6.000000, -1088,  1088, -14,  15 }, // alpha = -41.185925 [ 31]
		 {  -5.625000,   5.625000, -1088,  1088, -15,  14 }, // alpha = -45.000000 [ 32]
		 {  -6.000000,   5.125000, -1192,   976, -15,  12 }, // alpha = -49.497152 [ 33]
		 {  -6.375000,   4.750000, -1280,   856, -16,  11 }, // alpha = -53.310277 [ 34]
		 {  -6.750000,   4.250000, -1360,   720, -17,  10 }, // alpha = -57.804266 [ 35]
		 {  -7.000000,   3.750000, -1360,   720, -18,   9 }, // alpha = -61.821410 [ 36]
		 {  -7.250000,   3.125000, -1416,   584, -19,   7 }, // alpha = -66.682292 [ 37]
		 {  -7.500000,   2.625000, -1472,   440, -19,   6 }, // alpha = -70.709954 [ 38]
		 {  -7.625000,   2.125000, -1504,   296, -20,   5 }, // alpha = -74.427456 [ 39]
		 {  -7.875000,   1.500000, -1504,   296, -20,   3 }, // alpha = -79.215702 [ 40]
		 {  -7.875000,   1.000000, -1528,   152, -20,   2 }, // alpha = -82.763078 [ 41]
		 {  -8.000000,   0.375000, -1536,     0, -20,   0 }, // alpha = -87.316225 [ 42]
		 {  -8.000000,  -0.250000, -1536,     0, -20,  -1 }, // alpha = -91.789911 [ 43]
		 {  -7.875000,  -0.750000, -1528,  -152, -20,  -2 }, // alpha = -95.440332 [ 44]
		 {  -7.875000,  -1.375000, -1528,  -152, -20,  -4 }, // alpha = -99.904183 [ 45]
		 {  -7.750000,  -2.000000, -1504,  -296, -20,  -5 }, // alpha = -104.470294 [ 46]
		 {  -7.625000,  -2.500000, -1472,  -440, -20,  -7 }, // alpha = -108.152706 [ 47]
		 {  -7.375000,  -3.125000, -1416,  -584, -19,  -8 }, // alpha = -112.963773 [ 48]
		 {  -7.125000,  -3.625000, -1416,  -584, -18, -10 }, // alpha = -116.965714 [ 49]
		 {  -6.875000,  -4.125000, -1360,  -720, -18, -11 }, // alpha = -120.963757 [ 50]
		 {  -6.500000,  -4.625000, -1280,  -856, -17, -12 }, // alpha = -125.433314 [ 51]
		 {  -6.125000,  -5.125000, -1192,  -976, -16, -13 }, // alpha = -129.920392 [ 52]
		 {  -5.750000,  -5.500000, -1192,  -976, -15, -14 }, // alpha = -133.726970 [ 53]
		 {  -5.375000,  -5.875000, -1088, -1088, -14, -15 }, // alpha = -137.544804 [ 54]
		 {  -4.875000,  -6.250000,  -976, -1192, -13, -16 }, // alpha = -142.045769 [ 55]
		 {  -4.500000,  -6.625000,  -856, -1280, -12, -17 }, // alpha = -145.813878 [ 56]
		 {  -4.000000,  -7.000000,  -856, -1280, -10, -18 }, // alpha = -150.255119 [ 57]
		 {  -3.375000,  -7.250000,  -720, -1360,  -9, -19 }, // alpha = -155.037210 [ 58]
		 {  -2.875000,  -7.375000,  -584, -1416,  -8, -19 }, // alpha = -158.702646 [ 59]
		 {  -2.375000,  -7.625000,  -440, -1472,  -6, -20 }, // alpha = -162.699473 [ 60]
		 {  -1.750000,  -7.750000,  -440, -1472,  -5, -20 }, // alpha = -167.275644 [ 61]
		 {  -1.250000,  -7.875000,  -296, -1504,  -4, -20 }, // alpha = -170.980678 [ 62]
		 {  -0.625000,  -8.000000,  -152, -1528,  -2, -20 }, // alpha = -175.532841 [ 63]
		 {   0.000000,  -8.000000,     0, -1536,   0, -20 }, // alpha = 180.000000 [ 64]
		 {   0.500000,  -8.000000,   152, -1528,   1, -20 }, // alpha = 176.423666 [ 65]
		 {   1.125000,  -7.875000,   296, -1504,   2, -20 }, // alpha = 171.869898 [ 66]
		 {   1.750000,  -7.750000,   440, -1472,   4, -20 }, // alpha = 167.275644 [ 67]
		 {   2.250000,  -7.625000,   440, -1472,   5, -20 }, // alpha = 163.559620 [ 68]
		 {   2.875000,  -7.375000,   584, -1416,   7, -19 }, // alpha = 158.702646 [ 69]
		 {   3.375000,  -7.250000,   720, -1360,   8, -19 }, // alpha = 155.037210 [ 70]
		 {   3.875000,  -7.000000,   856, -1280,   9, -18 }, // alpha = 151.032339 [ 71]
		 {   4.375000,  -6.625000,   856, -1280,  10, -17 }, // alpha = 146.560131 [ 72]
		 {   4.875000,  -6.250000,   976, -1192,  12, -16 }, // alpha = 142.045769 [ 73]
		 {   5.250000,  -5.875000,  1088, -1088,  13, -15 }, // alpha = 138.215484 [ 74]
		 {   5.750000,  -5.500000,  1192,  -976,  14, -14 }, // alpha = 133.726970 [ 75]
		 {   6.125000,  -5.125000,  1192,  -976,  15, -13 }, // alpha = 129.920392 [ 76]
		 {   6.500000,  -4.625000,  1280,  -856,  16, -12 }, // alpha = 125.433314 [ 77]
		 {   6.750000,  -4.125000,  1360,  -720,  16, -11 }, // alpha = 121.429566 [ 78]
		 {   7.000000,  -3.625000,  1416,  -584,  17, -10 }, // alpha = 117.377703 [ 79]
		 {   7.250000,  -3.125000,  1416,  -584,  18,  -8 }, // alpha = 113.317708 [ 80]
		 {   7.500000,  -2.500000,  1472,  -440,  18,  -7 }, // alpha = 108.434949 [ 81]
		 {   7.625000,  -2.000000,  1504,  -296,  19,  -5 }, // alpha = 104.697319 [ 82]
		 {   7.750000,  -1.375000,  1528,  -152,  19,  -4 }, // alpha = 100.060690 [ 83]
		 {   7.875000,  -0.750000,  1528,  -152,  19,  -2 }, // alpha =  95.440332 [ 84]
		 {   7.875000,  -0.250000,  1536,     0,  19,  -1 }, // alpha =  91.818303 [ 85]
		 {   7.875000,   0.375000,  1536,     0,  19,   0 }, // alpha =  87.273689 [ 86]
		 {   7.875000,   1.000000,  1528,   152,  19,   2 }, // alpha =  82.763078 [ 87]
		 {   7.750000,   1.500000,  1504,   296,  19,   3 }, // alpha =  79.045937 [ 88]
		 {   7.625000,   2.125000,  1504,   296,  19,   5 }, // alpha =  74.427456 [ 89]
		 {   7.500000,   2.625000,  1472,   440,  18,   6 }, // alpha =  70.709954 [ 90]
		 {   7.250000,   3.125000,  1416,   584,  18,   7 }, // alpha =  66.682292 [ 91]
		 {   7.000000,   3.750000,  1360,   720,  17,   9 }, // alpha =  61.821410 [ 92]
		 {   6.625000,   4.250000,  1360,   720,  16,  10 }, // alpha =  57.319445 [ 93]
		 {   6.375000,   4.750000,  1280,   856,  15,  11 }, // alpha =  53.310277 [ 94]
		 {   6.000000,   5.125000,  1192,   976,  15,  12 }, // alpha =  49.497152 [ 95]
		 {   5.625000,   5.625000,  1088,  1088,  14,  14 }, // alpha =  45.000000 [ 96]
		 {   5.125000,   6.000000,  1088,  1088,  12,  15 }, // alpha =  40.502848 [ 97]
		 {   4.750000,   6.375000,   976,  1192,  11,  15 }, // alpha =  36.689723 [ 98]
		 {   4.250000,   6.625000,   856,  1280,  10,  16 }, // alpha =  32.680555 [ 99]
		 {   3.750000,   7.000000,   720,  1360,   9,  17 }, // alpha =  28.178590 [100]
		 {   3.125000,   7.250000,   720,  1360,   7,  18 }, // alpha =  23.317708 [101]
		 {   2.625000,   7.500000,   584,  1416,   6,  18 }, // alpha =  19.290046 [102]
		 {   2.125000,   7.625000,   440,  1472,   5,  19 }, // alpha =  15.572544 [103]
		 {   1.500000,   7.750000,   296,  1504,   3,  19 }, // alpha =  10.954063 [104]
		 {   1.000000,   7.875000,   296,  1504,   2,  19 }, // alpha =   7.236922 [105]
		 {   0.375000,   7.875000,   152,  1528,   0,  19 }, // alpha =   2.726311 [106]
		 {  -0.250000,   7.875000,  -152,  1528,  -1,  19 }, // alpha =  -1.818303 [107]
		 {  -0.750000,   7.875000,  -152,  1528,  -2,  19 }, // alpha =  -5.440332 [108]
		 {  -1.375000,   7.750000,  -296,  1504,  -4,  19 }, // alpha = -10.060690 [109]
		 {  -2.000000,   7.625000,  -440,  1472,  -5,  19 }, // alpha = -14.697319 [110]
		 {  -2.500000,   7.500000,  -584,  1416,  -7,  18 }, // alpha = -18.434949 [111]
		 {  -3.125000,   7.250000,  -584,  1416,  -8,  18 }, // alpha = -23.317708 [112]
		 {  -3.625000,   7.000000,  -720,  1360, -10,  17 }, // alpha = -27.377703 [113]
		 {  -4.125000,   6.750000,  -856,  1280, -11,  16 }, // alpha = -31.429566 [114]
		 {  -4.625000,   6.500000,  -976,  1192, -12,  16 }, // alpha = -35.433314 [115]
		 {  -5.125000,   6.125000,  -976,  1192, -13,  15 }, // alpha = -39.920392 [116]
		 {  -5.500000,   5.750000, -1088,  1088, -14,  14 }, // alpha = -43.726970 [117]
		 {  -5.875000,   5.250000, -1192,   976, -15,  13 }, // alpha = -48.215484 [118]
		 {  -6.250000,   4.875000, -1280,   856, -16,  12 }, // alpha = -52.045769 [119]
		 {  -6.625000,   4.375000, -1280,   856, -17,  10 }, // alpha = -56.560131 [120]
		 {  -7.000000,   3.875000, -1360,   720, -18,   9 }, // alpha = -61.032339 [121]
		 {  -7.250000,   3.375000, -1416,   584, -19,   8 }, // alpha = -65.037210 [122]
		 {  -7.375000,   2.875000, -1472,   440, -19,   7 }, // alpha = -68.702646 [123]
		 {  -7.625000,   2.250000, -1472,   440, -20,   5 }, // alpha = -73.559620 [124]
		 {  -7.750000,   1.750000, -1504,   296, -20,   4 }, // alpha = -77.275644 [125]
		 {  -7.875000,   1.125000, -1528,   152, -20,   2 }, // alpha = -81.869898 [126]
		 {  -8.000000,   0.500000, -1536,     0, -20,   1 }, // alpha = -86.423666 [127]
		 {  -8.000000,   0.000000, -1536,     0, -20,   0 }, // alpha = -90.000000 [128]
		 {  -8.000000,  -0.625000, -1536,     0, -20,  -2 }, // alpha = -94.467159 [129]
		 {  -7.875000,  -1.250000, -1528,  -152, -20,  -4 }, // alpha = -99.019322 [130]
		 {  -7.750000,  -1.750000, -1504,  -296, -20,  -5 }, // alpha = -102.724356 [131]
		 {  -7.625000,  -2.375000, -1472,  -440, -20,  -6 }, // alpha = -107.300527 [132]
		 {  -7.375000,  -2.875000, -1472,  -440, -19,  -8 }, // alpha = -111.297354 [133]
		 {  -7.250000,  -3.375000, -1416,  -584, -19,  -9 }, // alpha = -114.962790 [134]
		 {  -7.000000,  -4.000000, -1360,  -720, -18, -10 }, // alpha = -119.744881 [135]
		 {  -6.625000,  -4.500000, -1280,  -856, -17, -12 }, // alpha = -124.186122 [136]
		 {  -6.250000,  -4.875000, -1280,  -856, -16, -13 }, // alpha = -127.954231 [137]
		 {  -5.875000,  -5.375000, -1192,  -976, -15, -14 }, // alpha = -132.455196 [138]
		 {  -5.500000,  -5.750000, -1088, -1088, -14, -15 }, // alpha = -136.273030 [139]
		 {  -5.125000,  -6.125000,  -976, -1192, -13, -16 }, // alpha = -140.079608 [140]
		 {  -4.625000,  -6.500000,  -976, -1192, -12, -17 }, // alpha = -144.566686 [141]
		 {  -4.125000,  -6.875000,  -856, -1280, -11, -18 }, // alpha = -149.036243 [142]
		 {  -3.625000,  -7.125000,  -720, -1360, -10, -18 }, // alpha = -153.034286 [143]
		 {  -3.125000,  -7.375000,  -584, -1416,  -8, -19 }, // alpha = -157.036227 [144]
		 {  -2.500000,  -7.625000,  -584, -1416,  -7, -20 }, // alpha = -161.847294 [145]
		 {  -2.000000,  -7.750000,  -440, -1472,  -5, -20 }, // alpha = -165.529706 [146]
		 {  -1.375000,  -7.875000,  -296, -1504,  -4, -20 }, // alpha = -170.095817 [147]
		 {  -0.750000,  -7.875000,  -152, -1528,  -2, -20 }, // alpha = -174.559668 [148]
		 {  -0.250000,  -8.000000,  -152, -1528,  -1, -20 }, // alpha = -178.210089 [149]
		 {   0.375000,  -8.000000,   152, -1528,   0, -20 }, // alpha = 177.316225 [150]
		 {   1.000000,  -7.875000,   296, -1504,   2, -20 }, // alpha = 172.763078 [151]
		 {   1.500000,  -7.875000,   296, -1504,   3, -20 }, // alpha = 169.215702 [152]
		 {   2.125000,  -7.625000,   440, -1472,   5, -20 }, // alpha = 164.427456 [153]
		 {   2.625000,  -7.500000,   584, -1416,   6, -19 }, // alpha = 160.709954 [154]
		 {   3.125000,  -7.250000,   720, -1360,   7, -19 }, // alpha = 156.682292 [155]
		 {   3.750000,  -7.000000,   720, -1360,   9, -18 }, // alpha = 151.821410 [156]
		 {   4.250000,  -6.750000,   856, -1280,  10, -17 }, // alpha = 147.804266 [157]
		 {   4.750000,  -6.375000,   976, -1192,  11, -16 }, // alpha = 143.310277 [158]
		 {   5.125000,  -6.000000,  1088, -1088,  12, -15 }, // alpha = 139.497152 [159]
		 {   5.625000,  -5.625000,  1088, -1088,  14, -15 }, // alpha = 135.000000 [160]
		 {   6.000000,  -5.250000,  1192,  -976,  15, -14 }, // alpha = 131.185925 [161]
		 {   6.375000,  -4.750000,  1280,  -856,  15, -12 }, // alpha = 126.689723 [162]
		 {   6.625000,  -4.250000,  1360,  -720,  16, -11 }, // alpha = 122.680555 [163]
		 {   7.000000,  -3.750000,  1360,  -720,  17, -10 }, // alpha = 118.178590 [164]
		 {   7.250000,  -3.250000,  1416,  -584,  18,  -9 }, // alpha = 114.145542 [165]
		 {   7.500000,  -2.750000,  1472,  -440,  18,  -7 }, // alpha = 110.136303 [166]
		 {   7.625000,  -2.125000,  1504,  -296,  19,  -6 }, // alpha = 105.572544 [167]
		 {   7.750000,  -1.625000,  1504,  -296,  19,  -5 }, // alpha = 101.842076 [168]
		 {   7.875000,  -1.000000,  1528,  -152,  19,  -3 }, // alpha =  97.236922 [169]
		 {   7.875000,  -0.375000,  1536,     0,  19,  -1 }, // alpha =  92.726311 [170]
		 {   7.875000,   0.125000,  1536,     0,  19,   0 }, // alpha =  89.090620 [171]
		 {   7.875000,   0.750000,  1528,   152,  19,   1 }, // alpha =  84.559668 [172]
		 {   7.750000,   1.375000,  1528,   152,  19,   3 }, // alpha =  79.939310 [173]
		 {   7.625000,   1.875000,  1504,   296,  19,   4 }, // alpha =  76.184975 [174]
		 {   7.500000,   2.500000,  1472,   440,  18,   6 }, // alpha =  71.565051 [175]
		 {   7.250000,   3.000000,  1416,   584,  18,   7 }, // alpha =  67.520566 [176]
		 {   7.000000,   3.500000,  1416,   584,  17,   8 }, // alpha =  63.434949 [177]
		 {   6.750000,   4.000000,  1360,   720,  16,  10 }, // alpha =  59.349332 [178]
		 {   6.500000,   4.500000,  1280,   856,  16,  11 }, // alpha =  55.304846 [179]
		 {   6.125000,   5.000000,  1192,   976,  15,  12 }, // alpha =  50.774326 [180]
		 {   5.750000,   5.500000,  1192,   976,  14,  13 }, // alpha =  46.273030 [181]
		 {   5.250000,   5.875000,  1088,  1088,  13,  14 }, // alpha =  41.784516 [182]
		 {   4.875000,   6.250000,   976,  1192,  12,  15 }, // alpha =  37.954231 [183]
		 {   4.375000,   6.625000,   856,  1280,  10,  16 }, // alpha =  33.439869 [184]
		 {   3.875000,   6.875000,   856,  1280,   9,  17 }, // alpha =  29.407189 [185]
		 {   3.375000,   7.125000,   720,  1360,   8,  17 }, // alpha =  25.346176 [186]
		 {   2.875000,   7.375000,   584,  1416,   7,  18 }, // alpha =  21.297354 [187]
		 {   2.250000,   7.625000,   440,  1472,   5,  19 }, // alpha =  16.440380 [188]
		 {   1.750000,   7.750000,   440,  1472,   4,  19 }, // alpha =  12.724356 [189]
		 {   1.125000,   7.875000,   296,  1504,   2,  19 }, // alpha =   8.130102 [190]
		 {   0.500000,   7.875000,   152,  1528,   1,  19 }, // alpha =   3.632951 [191]
		 {   0.000000,   7.875000,     0,  1536,   0,  19 }, // alpha =   0.000000 [192]
		 {  -0.625000,   7.875000,  -152,  1528,  -2,  19 }, // alpha =  -4.537773 [193]
		 {  -1.250000,   7.875000,  -296,  1504,  -4,  19 }, // alpha =  -9.019322 [194]
		 {  -1.750000,   7.750000,  -440,  1472,  -5,  19 }, // alpha = -12.724356 [195]
		 {  -2.375000,   7.625000,  -440,  1472,  -6,  19 }, // alpha = -17.300527 [196]
		 {  -2.875000,   7.375000,  -584,  1416,  -8,  18 }, // alpha = -21.297354 [197]
		 {  -3.375000,   7.125000,  -720,  1360,  -9,  17 }, // alpha = -25.346176 [198]
		 {  -4.000000,   6.875000,  -856,  1280, -10,  17 }, // alpha = -30.191623 [199]
		 {  -4.500000,   6.625000,  -856,  1280, -12,  16 }, // alpha = -34.186122 [200]
		 {  -4.875000,   6.250000,  -976,  1192, -13,  15 }, // alpha = -37.954231 [201]
		 {  -5.375000,   5.875000, -1088,  1088, -14,  14 }, // alpha = -42.455196 [202]
		 {  -5.750000,   5.500000, -1192,   976, -15,  13 }, // alpha = -46.273030 [203]
		 {  -6.125000,   5.000000, -1192,   976, -16,  12 }, // alpha = -50.774326 [204]
		 {  -6.500000,   4.500000, -1280,   856, -17,  11 }, // alpha = -55.304846 [205]
		 {  -6.875000,   4.000000, -1360,   720, -18,  10 }, // alpha = -59.808377 [206]
		 {  -7.125000,   3.500000, -1416,   584, -18,   8 }, // alpha = -63.838434 [207]
		 {  -7.375000,   3.000000, -1416,   584, -19,   7 }, // alpha = -67.864508 [208]
		 {  -7.625000,   2.500000, -1472,   440, -20,   6 }, // alpha = -71.847294 [209]
		 {  -7.750000,   1.875000, -1504,   296, -20,   4 }, // alpha = -76.399457 [210]
		 {  -7.875000,   1.375000, -1528,   152, -20,   3 }, // alpha = -80.095817 [211]
		 {  -7.875000,   0.750000, -1528,   152, -20,   1 }, // alpha = -84.559668 [212]
		 {  -8.000000,   0.125000, -1536,     0, -20,   0 }, // alpha = -89.104826 [213]
		 {  -8.000000,  -0.375000, -1536,     0, -20,  -1 }, // alpha = -92.683775 [214]
		 {  -7.875000,  -1.000000, -1528,  -152, -20,  -3 }, // alpha = -97.236922 [215]
		 {  -7.875000,  -1.625000, -1504,  -296, -20,  -5 }, // alpha = -101.659293 [216]
		 {  -7.625000,  -2.125000, -1504,  -296, -20,  -6 }, // alpha = -105.572544 [217]
		 {  -7.500000,  -2.750000, -1472,  -440, -19,  -7 }, // alpha = -110.136303 [218]
		 {  -7.250000,  -3.250000, -1416,  -584, -19,  -9 }, // alpha = -114.145542 [219]
		 {  -7.000000,  -3.750000, -1360,  -720, -18, -10 }, // alpha = -118.178590 [220]
		 {  -6.750000,  -4.250000, -1360,  -720, -17, -11 }, // alpha = -122.195734 [221]
		 {  -6.375000,  -4.750000, -1280,  -856, -16, -12 }, // alpha = -126.689723 [222]
		 {  -6.000000,  -5.250000, -1192,  -976, -15, -14 }, // alpha = -131.185925 [223]
		 {  -5.625000,  -5.625000, -1088, -1088, -15, -15 }, // alpha = -135.000000 [224]
		 {  -5.250000,  -6.000000, -1088, -1088, -14, -15 }, // alpha = -138.814075 [225]
		 {  -4.750000,  -6.375000,  -976, -1192, -12, -16 }, // alpha = -143.310277 [226]
		 {  -4.250000,  -6.750000,  -856, -1280, -11, -17 }, // alpha = -147.804266 [227]
		 {  -3.750000,  -7.000000,  -720, -1360, -10, -18 }, // alpha = -151.821410 [228]
		 {  -3.250000,  -7.250000,  -720, -1360,  -9, -19 }, // alpha = -155.854458 [229]
		 {  -2.750000,  -7.500000,  -584, -1416,  -7, -19 }, // alpha = -159.863697 [230]
		 {  -2.125000,  -7.625000,  -440, -1472,  -6, -20 }, // alpha = -164.427456 [231]
		 {  -1.625000,  -7.875000,  -296, -1504,  -5, -20 }, // alpha = -168.340707 [232]
		 {  -1.000000,  -7.875000,  -296, -1504,  -3, -20 }, // alpha = -172.763078 [233]
		 {  -0.375000,  -8.000000,  -152, -1528,  -1, -20 }, // alpha = -177.316225 [234]
		 {   0.125000,  -8.000000,   152, -1528,   0, -20 }, // alpha = 179.104826 [235]
		 {   0.750000,  -7.875000,   152, -1528,   1, -20 }, // alpha = 174.559668 [236]
		 {   1.375000,  -7.875000,   296, -1504,   3, -20 }, // alpha = 170.095817 [237]
		 {   1.875000,  -7.750000,   440, -1472,   4, -20 }, // alpha = 166.399457 [238]
		 {   2.500000,  -7.625000,   584, -1416,   6, -20 }, // alpha = 161.847294 [239]
		 {   3.000000,  -7.375000,   584, -1416,   7, -19 }, // alpha = 157.864508 [240]
		 {   3.500000,  -7.125000,   720, -1360,   8, -18 }, // alpha = 153.838434 [241]
		 {   4.000000,  -6.875000,   856, -1280,  10, -18 }, // alpha = 149.808377 [242]
		 {   4.500000,  -6.500000,   976, -1192,  11, -17 }, // alpha = 145.304846 [243]
		 {   5.000000,  -6.125000,   976, -1192,  12, -16 }, // alpha = 140.774326 [244]
		 {   5.500000,  -5.750000,  1088, -1088,  13, -15 }, // alpha = 136.273030 [245]
		 {   5.875000,  -5.375000,  1192,  -976,  14, -14 }, // alpha = 132.455196 [246]
		 {   6.250000,  -4.875000,  1280,  -856,  15, -13 }, // alpha = 127.954231 [247]
		 {   6.625000,  -4.500000,  1280,  -856,  16, -12 }, // alpha = 124.186122 [248]
		 {   6.875000,  -4.000000,  1360,  -720,  17, -10 }, // alpha = 120.191623 [249]
		 {   7.125000,  -3.375000,  1416,  -584,  17,  -9 }, // alpha = 115.346176 [250]
		 {   7.375000,  -2.875000,  1472,  -440,  18,  -8 }, // alpha = 111.297354 [251]
		 {   7.625000,  -2.375000,  1472,  -440,  19,  -6 }, // alpha = 107.300527 [252]
		 {   7.750000,  -1.750000,  1504,  -296,  19,  -5 }, // alpha = 102.724356 [253]
		 {   7.875000,  -1.250000,  1528,  -152,  19,  -4 }, // alpha =  99.019322 [254]
		 {   7.875000,  -0.625000,  1536,     0,  19,  -2 }  // alpha =  94.537773 [255]
};
	
	public Ship( int x, int y, int x2, int y2 )
	{
		super( x, y, 10, 0 );

		this.x2 = x2;
		this.y2 = y2;

		pixelSize					= 25;
		squaredPixelSize	= pixelSize * pixelSize;

		double shot_dist_min = 1000.0;
		double shot_dist_max = 0.0;
		double shot_dist_sum = 0.0;
		
		for ( int wb=0; wb < 256; wb++ ) {
			shot_delta[wb][0]	= shot_delta_tmp[wb][0];
			shot_delta[wb][1]	= shot_delta_tmp[wb][1];

			ship_delta[wb][0]	= (int)shot_delta_tmp[wb][2];
			ship_delta[wb][1]	= (int)shot_delta_tmp[wb][3];

			shot_origin[wb][0]	= (int)shot_delta_tmp[wb][4];
			shot_origin[wb][1]	= (int)shot_delta_tmp[wb][5];

			// Schussoffsets bei stehendem Schiff, pro Winkelbyte, pro Frame
			for ( int f = 2; f < 74; f++ ) {
				getShotPrediction( lPoint, f, wb );

				shot_offset_x[wb][f] = (short)( lPoint.x - x );
				shot_offset_y[wb][f] = (short)( lPoint.y - y );
			}
			double dx = shot_delta[wb][0];
			double dy = shot_delta[wb][1];

			shot_dist[wb]	= Math.sqrt( dx*dx + dy*dy );
			shot_alpha[wb]= Math.toDegrees( Math.atan2( dx, dy ) );

			if ( Debug.shot_dist ) {
				if ( shot_dist[wb] < shot_dist_min ) shot_dist_min = shot_dist[wb];
				if ( shot_dist[wb] > shot_dist_max ) shot_dist_max = shot_dist[wb];
				shot_dist_sum += shot_dist[wb];
			}
		}
		if ( Debug.shot_dist ) {
			System.out.println( String.format(
					"Ship.shot: min,avg,max = %5f/%5f/%5f",
					shot_dist_min, shot_dist_sum / 256.0, shot_dist_max ) );
		}
	}
	
	public void setVisible( int vx, int vy )
	{
		visible	= true;
		x				= vx;
		y				= vy;
		dx			= 0;
		dy			= 0;
		
		updateShotOffsets();
	}

	public void draw( Graphics g )
	{
		g.setColor( Color.blue );
		g.fillOval( x - 5, 1024 - y - 5, 10, 10 );
		g.drawLine( x, 1024 - y, x + ( x2 - x ) / 50, 1024 - ( y + ( y2 - y ) / 50 ) );

		int svx = (int)( x + 100 * shot_delta[winkelbyte][0] );
		int svy = (int)( y + 100 * shot_delta[winkelbyte][1] );

		g.drawLine( x, 1024 - y, svx, 1024 - svy );
	}

	public void update( int x_new, int y_new )
	{
		ship_dx[ship_di] = x_new - x;
		ship_dy[ship_di] = y_new - y;

		ship_di = ( ship_di + 1 ) % ship_dx.length;
		
		dx = 0.0;
		dy = 0.0;
		
		for ( int i=0; i < ship_dy.length; i++ ) {
			dx += ship_dx[i];
			dy += ship_dy[i];
		}
		dx /= ship_dx.length;
		dy /= ship_dx.length;

		boolean changedPos = ( x != x_new ) || ( y != y_new );
		
		x = x_new;
		y = y_new;
		
		if ( changedPos ) {
			updateShotOffsets();
		}
	}
	
	private void updateShotOffsets()
	{
		Log.debug( String.format(
				"Frame#%5d: ship.updateShotOffsets!",
				AsteroidApp.frame_cnt ) );

		for ( int wb = 0; wb < 256; wb++ ) {
			for ( int f = 2; f < 74; f++ ) {
				int sx = x + shot_offset_x[wb][f];
				int sy = y + shot_offset_y[wb][f];
				
				if ( sx < 0 ) {
					sx += AsteroidApp.EXTENT_X;
				}
				else if ( sx > AsteroidApp.MAX_X ) {
					sx -= AsteroidApp.EXTENT_X;
				}
				if ( sy < 0 ) {
					sy += AsteroidApp.EXTENT_Y;
				}
				else if ( sy > AsteroidApp.MAX_Y ) {
					sy -= AsteroidApp.EXTENT_Y;
				}
				shot_offset_norm_x[wb][f] = (short)sx;
				shot_offset_norm_y[wb][f] = (short)sy;
			}
		}
	}

	/** Positionen eines Schusses in #n Frames bei Winkelbyte wb */
	public void getShotPrediction( Point p, int frameHorizon, int wb )
	{
		// In Abstand von ca. 20 Abstand vom Schiffsmittelpunkt treten die Schsse aus:
		final int x0 = x + Ship.shot_origin[wb][0];
		final int y0 = y + Ship.shot_origin[wb][1];

		// Korrigiert auf -2: Im nchsten Frame wird erst der Bildschirminhalt zurckgegeben und dann
		// der Key "FIRE" ausgewertet, erst ab dem dritten Frame entfernt sich der Schuss
		p.x = x0 + (int)( ( frameHorizon - 2 ) * Ship.shot_delta[wb][0] + frameHorizon * dx );
		p.y = y0 + (int)( ( frameHorizon - 2 ) * Ship.shot_delta[wb][1] + frameHorizon * dy );
	}

	public void getShotPrediction( Point p, int frameHorizon )
	{
		getShotPrediction( p, frameHorizon, AsteroidApp.getWinkelbyte() );
	}
	
	/** Schuss so in den Vektor einsortieren, dass eine Sortierung frames absteigend erreicht wird */
	public static void addShot( Shot newShot )
	{
		for ( int i = 0; i < shots.size(); i++ ) {
			Shot shot = shots.elementAt( i );
			
			if ( newShot.frames > shot.frames ) {
				shots.insertElementAt( newShot, i );
				return;
			}
		}
		shots.add( newShot );
	}
	
	/** In wieviel Frames wird der nchste Schuss verfgbar sein? */
	public int nextShotAvailable()
	{
		if ( shots.size() < 4 ) {
			return 0;
		}
		int minFrame = 100;

		for ( Shot shot : shots ) {
			if ( shot.calc_remaining_frames < minFrame ) {
				minFrame = shot.calc_remaining_frames;
			}
		}
		return minFrame;
	}
	
	/** Ist innerhalb #n Frames ein Schuss am Ende seiner Lebenszeit und ermglicht einen neuen Schuss? */
	public boolean shotAvailable( int frameHorizon )
	{
		for ( Shot shot : shots ) {
			if ( shot.calc_remaining_frames < frameHorizon ) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Schiff hat sich nach links gedreht. Schussrichtung korrigieren und kontrollieren
	 */
	public void rotate( boolean left )
	{
		if ( winkelbyte_resync ) {
			rotate_resync( left );
		}
		else {
			int sdx = ship_delta[winkelbyte][0];
			int sdy = ship_delta[winkelbyte][1];

			if ( sdx != x2 - x || sdy != y2 - y ) {
				possiblePosition.clear();
	
				if ( Debug.winkelbyte ) {
					Log.warn( String.format(
							"Frame#%5d: WB1 resync [%5d,%5d]", AsteroidApp.frame_cnt, x2 - x, y2 - y ) );
				}
				for ( int i = 0; i < 255; i++ ) {
					if (    x2 - x == ship_delta[i][0]
					     && y2 - y == ship_delta[i][1] ) {
						possiblePosition.add( i );
					}
				}
				winkelbyte_resync = true;
				rotate_resync( left );
			}
		}
		winkelbyte = left ? ( winkelbyte + 1 ) & 0xFF : ( winkelbyte + 255 ) & 0xFF;
	}
	
	public void rotate_resync( boolean left )
	{
		Vector<Integer> tmpVector = new Vector<Integer>();

		for ( Integer i : possiblePosition ) {
			int j = left ? ( i + 1 ) & 0xFF : ( i + 255 ) & 0xFF;

			if (  x2 - x == ship_delta[j][0]
			   && y2 - y == ship_delta[j][1] ) {
				tmpVector.add( i );
			}
		}
		possiblePosition = tmpVector;

		if ( possiblePosition.size() == 1 ) {
			winkelbyte_resync	= false;
			winkelbyte				= possiblePosition.elementAt(0);
			return;
		}
		if ( possiblePosition.size() == 0 ) {
			for ( int i = 0; i < 255; i++ ) {
				int j = left ? ( i + 1 ) & 0xFF : ( i + 255 ) & 0xFF;

				if ( Debug.winkelbyte ) {
					System.err.println( String.format(
							"Frame#%5d: WB2 resync [%5d,%5d]", AsteroidApp.frame_cnt, x2 - x, y2 - y ) );
				}
				if (    x2 - x == ship_delta[j][0]
				     && y2 - y == ship_delta[j][1] ) {
					possiblePosition.add( j );
					winkelbyte = i;
				}
			}
		}
	}

	public Asteroid findNearestAsteroid( Vector<Asteroid> asteroids )
	{
		Asteroid nearest = null;

		for ( Asteroid a : asteroids ) {
			if ( null == nearest || a.distance < nearest.distance ) {
				nearest = a;
			}
		}
		return nearest;
	}
	
	/** Asteroiden einsammeln, die in den nchsten #frameHorizon Frames nicht nher
	 *  als #distance an das Schiff herankommen
	 */
	public int findHardTargets( Vector<Target> result, Vector<Target> targets, int frameHorizon, int distance )
	{
		result.clear();

		int distanceSq = distance * distance;
		
		for ( Target target : targets ) {
			boolean near = false;

			for ( int f = 1; !near && f < frameHorizon; f += 10 ) {
				if ( getSquaredTorusDistance( target, f ) < distanceSq ) {
					near = true;
				}
			}
			if ( !near || target instanceof UFO
					|| ( target instanceof Asteroid && ((Asteroid)target).sizeType != Asteroid.SIZE_SMALL ) ) {
				result.add( target );
			}
		}
		return result.size();
	}
	
	/** Alle Asteroiden mit Abstandsinformation versehen und solche auf Kollisionskurs
	 *  innerhalb der nchsten #minFrame Frames markieren. Nchsten Asteroiden auf Kollisionskurs zurckgeben.
	 */
	public Target findCollisionTarget( Vector<Target> targets, int minFrame )
	{
		Target	nearest_frame	= null;
		int			frame_min			= minFrame;

		for ( Target target : targets ) {
			target.collision			= false;
			target.collisionFrame	= 1000;

			final double collisionDist				= ( pixelSize + target.pixelSize ) / 2;
			final double squaredCollisionDist	= collisionDist * collisionDist;

			for ( int f = 1; f < minFrame; f++ ) {
				double squaredDist = getSquaredTorusDistance( target, f );

				if ( 1 == f ) {
					// Im Asteroiden die Distanz zum Schiff speichern
					target.distance = Math.sqrt( squaredDist );
				}
				if ( squaredDist <= squaredCollisionDist ) {
					// Asteroid ist auf Kollisionskurs.
					target.collision			= true;
					target.collisionFrame	= Math.min( f, target.collisionFrame );

					if ( 0 == target.framewait && f < frame_min ) {
						nearest_frame = target;
						frame_min			= f;
					}
				}
			}
		}
		return nearest_frame;
	}

	/** Max. Anzahl von Schssen von der Entfernung zum Schiff abhngig machen */
	public int getMaxShots( Target target, int frameHorizon )
	{
		if ( target instanceof UFO ) {
			return 1;
		}
		Asteroid a = (Asteroid)target;

		if ( Asteroid.SIZE_SMALL == a.sizeType ) {
			return 1;
		}
		if ( Asteroid.SIZE_MEDIUM == a.sizeType ) {
			return getTorusDistance( a, frameHorizon ) <= Config.MediumMinDist[ AsteroidApp.level_part ] ? 3 : 2;
		}
		// SIZE_LARGE:
		return getTorusDistance( a, frameHorizon ) <= Config.LargeMinDist[ AsteroidApp.level_part ] ? 3 : 2;
	}

	public void checkShotHits( Vector<Target> targets, final int level_frame )
	{
		int nChecks = 0;

		for ( Shot shot : Ship.shots ) {
			// Schsse sind absteigend nach frames sortiert, d.h. Entfernung vom Schiff.
			// 1: Berechnung auf Basis des Winkelbytes
			// 9: Erneute Berechnung zur Kontrolle, 8 Frames Delta haben neue dx,dy ermittelt
			boolean check = ( 1 == shot.frames ) || ( 9 == shot.frames ) || ( shot.frames > 60 && level_frame < 10 );

			if ( null != shot.target && shot.target.exploding ) {
				// Schuss ist noch auf dem Bildschirm, aber das Target ist schon explodiert
				// --> evtll. fliegt der Schuss in ein neues Target, Prfung erzwingen
				if ( Debug.shot_target ) {
					Log.debug( String.format(
						"Frame#%5d: Shot Target explodiert, shot.lifetime=%2d",
						AsteroidApp.frame_cnt, shot.calc_remaining_frames ) );
				}
				check = true;
			}
			if ( check ) {
				nChecks++;

				boolean target_found = false;

				// Edit 6.6.08: -1, weil Schuss im letzten Frame nicht trifft
				for ( int f = 1; !target_found && f < shot.max_remaining_frames - 1; f++ ) {
					for ( Target target : targets ) {
						if ( target.shotWillHit( shot, f ) ) {
							if ( null == target.shot ) {
								// Der Schuss, der zuerst trifft, wird in target vermerkt:
								target.shot = shot;
								target.hit_in_frames = f;
							}
							if ( target.maxShots() == 1 && target.shot != shot ) {
								// Anderer Schuss trifft dieses Target bereits
								if ( Debug.shot_target ) {
									Log.info( String.format( "Frame#%5d: staggered Asteroid shot", AsteroidApp.frame_cnt ) );
								}
								continue;
							}
							target.shots_fired_at++;
	
							shot.target									= target;
							shot.calc_remaining_frames	= f;
	
							if ( target.shots_fired_at >= getMaxShots( target, 0 ) || shot.frames >= 9 ) {
								target.framewait = f + 2;	// Falls es nicht stimmt, frhestens dann wieder versuchen
							}
							target_found = true;
						}
					}
				}
				if ( !target_found ) {
					if ( null != shot.target ) {
						shot.target.framewait	= 0;
						shot.target.shot			= null;
						shot.target						= null;
					}
					shot.calc_remaining_frames = shot.max_remaining_frames;
				}
			}
		}
		if ( Debug.shot_target && nChecks > 0 ) {
			Log.debug( String.format(
					"Frame#%5d: %2d Shot Prfungen",
					AsteroidApp.frame_cnt, nChecks ) );
		}
	}

	/** Erreichbarkeiten der Targets (Asteroiden/UFO) markieren, am schnellsten
	 *  abschiebares Target zurckgeben.
	 *  @param maxKeyPress = max. Tastendrcke rechts/links
	 */
	public Target markTargetReachability( Target[] aBest, Vector<Target> targets, int maxKeyPress )
	{
		aBest[1] = null;
		aBest[0] = useJNIMethods
							? findBestTargetJNI( x, y, targets, AsteroidApp.getWinkelbyte(),
																	 0 /* nextShotAvailable() */, maxKeyPress, Shot.getLifeSpan(), 0 )
							: findBestTarget( targets, AsteroidApp.getWinkelbyte(),
																0, maxKeyPress, 0 );

		if ( Debug.findTargetsJNI ) {
			for ( Target target : targets ) {
				System.out.println( String.format( "L%4d|%4d R%4d|%4d",
						target.leftMinHitInFrames, target.leftMinHitInFramesKeys,
						target.rightMinHitInFrames, target.rightMinHitInFramesKeys ) );
			}
			if ( null != aBest[0] ) {
				System.out.println( String.format( "Frame#%5d: C++ tBest L%2d|%2d R%2d|%2d",
						AsteroidApp.frame_cnt,
						aBest[0].leftMinHitInFrames, aBest[0].leftMinHitInFramesKeys,
						aBest[0].rightMinHitInFrames, aBest[0].rightMinHitInFramesKeys ) );
			}
		}
		if ( null != aBest[0] && 0 == aBest[0].leftMinHitInFramesKeys ) {
			// Zweitbestes Ziel genau dann suchen, wenn das beste Ziel ohne Drehung abgeschossen werden kann
			targets.remove( aBest[0] );

			aBest[1] = useJNIMethods
								? findBestTargetJNI(	x, y, targets, AsteroidApp.getWinkelbyte(),
																			0 /* nextShotAvailable() */, maxKeyPress, Shot.getLifeSpan(), 2 )
								: findBestTarget( targets, AsteroidApp.getWinkelbyte(),
																	0, maxKeyPress, 2 );

			targets.add( aBest[0] );
		}
		return aBest[0];
	}

	private Target findBestTarget(	Vector<Target> targets,
																	final int wbStart,
																	final int minKeyPress, final int maxKeyPress,
																	final int frameHorizon )
	{
		Target	tBest		= null;
		int			tFrames	= 1000;

		for ( Target target : targets ) {
			target.leftMinHitInFrames				= 1000;
			target.leftMinHitInFramesKeys		= 1000;

			target.rightMinHitInFrames			= 1000;
			target.rightMinHitInFramesKeys	= 1000;
		}
		// Mit welchem Winkelbyte msste man schiessen, um zu treffen?
		// Nur max. eine halbe Drehung bercksichtigen
		int wb_left		= wbStart;
		int wb_right	= wbStart;

		for ( int keypress = minKeyPress; keypress < maxKeyPress; keypress++ ) {
			// Edit 6.6.08: -1 wegen der Tatsache, dass der Schuss im letzten Frame nichts mehr trifft
			final int maxShotFrame = keypress + 2 + Shot.getLifeSpan( frameHorizon + keypress ) - 1;

			for ( int f = keypress + 2; f < maxShotFrame; f++ ) {
				if ( f > tFrames ) {
					// Ein schneller erreichbares Ziel ist schon gefunden.
					break;
				}
				final int fk = f - keypress;

				lPoint.x = shot_offset_norm_x[wb_left][fk];
				lPoint.y = shot_offset_norm_y[wb_left][fk];

				rPoint.x = shot_offset_norm_x[wb_right][fk];
				rPoint.y = shot_offset_norm_y[wb_right][fk];

				for ( Target target : targets ) {
					if ( target.shotWillHit( lPoint, frameHorizon + f ) ) {
						// Minimale Frameanzahl, bis target abgeschossen ist, incl. Drehung
						if ( f < target.leftMinHitInFrames ) {
							target.leftMinHitInFrames			= f;
							target.leftMinHitInFramesKeys	= keypress;
						}
						if ( null == tBest || target.leftMinHitInFrames < tFrames ) {
							tBest		= target;
							tFrames	= target.leftMinHitInFrames;
						}
					}
					if ( target.shotWillHit( rPoint, frameHorizon + f ) ) {
						if ( f < target.rightMinHitInFrames ) {
							target.rightMinHitInFrames			= f;
							target.rightMinHitInFramesKeys	= keypress;
						}
						if ( null == tBest || target.rightMinHitInFrames < tFrames ) {
							tBest		= target;
							tFrames	= target.rightMinHitInFrames;
						}
					}
				}
			}
			wb_left		= ( wb_left  +   1 ) & 0xFF;
			wb_right	= ( wb_right + 255 ) & 0xFF;
		}
		if ( null != tBest ) {
			System.out.println( String.format( "Frame#%5d: Java Best L%2d|%2d R%2d|%2d",
					AsteroidApp.frame_cnt,
					tBest.leftMinHitInFrames, tBest.leftMinHitInFramesKeys,
					tBest.rightMinHitInFrames, tBest.rightMinHitInFramesKeys ) );
		}
		return tBest;
	}

	/** Wrde ein Schuss mit der aktuellen Schiffsdrehung etwas treffen? */
	public Target findTarget( Vector<Target> targets, final int waitFrames )
	{
//		if ( useJNIMethods ) {
//			// BUG: targets.left/right... wird gelscht, auch fr das selektierte Target!
//			Target tBest = findBestTargetJNI( x, y, targets, AsteroidApp.getWinkelbyte(),
//																				0, 0, Shot.getLifeSpan(), waitFrames );
//			
//			return tBest;
//		}
		// Die max. Reichweite eines Schusses ist 69...72, via Shot.getLifeSpan ermittelt
		// Der Schuss tritt erst in +2 Frames aus, deswegen die max. Reichweite 2+getLifeSpan Frames
		final int wb						= AsteroidApp.getWinkelbyte();
		final int maxShotFrames = 2 + Shot.getLifeSpan( waitFrames ) - 1;

		for ( int f = 2; f < maxShotFrames; f++ ) {
			lPoint.x = x + shot_offset_x[wb][f];
			lPoint.y = y + shot_offset_y[wb][f];

			MovingObject.normalize( lPoint );

			for ( Target target : targets ) {
				if ( target.shotWillHit( lPoint, f + waitFrames ) ) {
					target.leftMinHitInFrames				= f;
					target.rightMinHitInFrames			= f;
					target.leftMinHitInFramesKeys		=	0;
					target.rightMinHitInFramesKeys	= 0;

					return target;
				}
			}
		}
		return null;
	}
	
	/** Drehung in Richtung von best, falls dieses erreicht ist, Drehung in Richtung second  */
	protected byte moveToTarget( Target best, Target second )
	{
		if ( 0 == best.leftMinHitInFramesKeys ) {
			return ( null != second ) ? moveToTarget( second, null ) : (byte)0;
		}
		if ( best.leftMinHitInFramesKeys < best.rightMinHitInFramesKeys ) {
			return AsteroidApp.KEY_LEFT;
		}
		return AsteroidApp.KEY_RIGHT;
	}
}
