// ============================================================================
// File:               $File$
//
// Project:            
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   (c) 2005  Rammi (rammi@caff.de)
//                     This code is in the public domain.
//                     Use at own risk.
//                     No guarantees given.
//
// Latest change:      $Date: 2006-02-07 20:25:27 +0100 (Di, 07 Feb 2006) $
//
// History:	       $Log$
//=============================================================================
package de.caff.util.settings.swing;

import de.caff.util.settings.DoublePreferenceProperty;

import java.util.Locale;
import java.util.prefs.Preferences;

/**
 *  Editable integer preference property.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @version $Revision: 112 $
 */
public class SwingDoublePreferenceProperty
        extends AbstractBasicSimpleEditablePreferenceProperty
        implements DoublePreferenceProperty
{
  /** Minimal value of property. */
  private final double minValue;
  /** Maximal value of property. */
  private final double maxValue;
  /** Step of property. */
  private final double step;
  /** Current value of property. */
  private double value;

  /**
   *  Create a double preference property with maximum range, step 1 and the
   *  given default value.
   *  @param basicName  property name
   *  @param baseTag    basic i18n tag
   *  @param value      default value
   */
  public SwingDoublePreferenceProperty(String basicName, String baseTag, double value)
  {
    this(basicName, baseTag, value, 1);
  }

  /**
   *  Create a double preference property with maximum range, step and the
   *  given default value.
   *  @param basicName  property name
   *  @param baseTag    basic i18n tag
   *  @param value      default value
   *  @param step       step to be used for value when stepping
   */
  public SwingDoublePreferenceProperty(String basicName, String baseTag, double value, double step)
  {
    this(basicName, baseTag, value, -Double.MAX_VALUE, Double.MAX_VALUE, step);
  }

  /**
   *  Create a double preference property with the given range, step 1 and the
   *  given default value.
   *  @param basicName  property name
   *  @param baseTag    basic i18n tag
   *  @param value      default value
   *  @param minValue   minimal allowed value (included)
   *  @param maxValue   maximal allowed value (included)
   */
  public SwingDoublePreferenceProperty(String basicName, String baseTag, double value, double minValue, double maxValue)
  {
    this(basicName, baseTag, value, minValue, maxValue, 1);
  }

  /**
   *  Create a double preference property with the given range, step and the
   *  given default value.
   *  @param basicName  property name
   *  @param baseTag    basic i18n tag
   *  @param value      default value
   *  @param minValue   minimal allowed value (included)
   *  @param maxValue   maximal allowed value (included)
   *  @param step       step to be used for value when stepping
   */
  public SwingDoublePreferenceProperty(String basicName, String baseTag,
                                       double value, double minValue, double maxValue, double step)
  {
    super(basicName, baseTag);
    this.minValue = minValue;
    this.maxValue = maxValue;
    this.step = step;
    this.value = value;
  }

  /**
   * Read the property value from the preferences.
   *
   * @param preferences preferences from where to read the property value
   */
  public void readFrom(Preferences preferences)
  {
    setValue(preferences.getDouble(getBasicName(), value));
  }

  /**
   * Store the current property value in the preferences.
   *
   * @param preferences preferences where to store the property value
   */
  public void storeTo(Preferences preferences)
  {
    preferences.putDouble(getBasicName(), value);
  }

  /**
   * Get the editor components for editing this preference property.
   *
   * @param l locale used for i18n
   * @return editor components
   */
  public EditorProvider getEditorProvider(Locale l)
  {
    return new DoubleEditor(this, l);
  }

  /**
   *  Get the minimal allowed value.
   *  @return minimal value
   */
  public double getMinValue()
  {
    return minValue;
  }

  /**
   *  Get the maximal allowed value.
   *  @return maximal value
   */
  public double getMaxValue()
  {
    return maxValue;
  }

  /**
   *  Get the step used in steppers.
   *  @return step
   */
  public double getStep()
  {
    return step;
  }

  /**
   *  Get the properties value.
   *  @return property value
   */
  public double getValue()
  {
    return value;
  }

  /**
   *  Set the properties value.
   *  @param value value to set
   */
  public void setValue(double value)
  {
    double oldValue = this.value;
    if (value > maxValue) {
      this.value = maxValue;
    }
    else if (value < minValue) {
      this.value = minValue;
    }
    else {
      this.value = value;
    }
    if (oldValue != this.value) {
      fireValueChange(getBasicName(), new Double(oldValue), new Double(this.value));
    }
  }
}
