// ============================================================================
// File:               ResourceException.java
//
// Project:            I18n
//
// Purpose:            Exception with resource. 
//
// Author:             Rammi
//-----------------------------------------------------------------------------
// Copyright Notice:   (c) 2002  Rammi (rammi@caff.de)
//                     This code is in the public domain.
//                     Use at own risk.
//                     No guarantees given.
//
// Latest change:      $Date$
//
// History:	       $Log$
//=============================================================================

package de.caff.i18n;

import java.util.Locale;

/** 
 *  An internationalized exception.
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 *  @version $Revision$
 */
public class ResourcedException
        extends Exception
{
  /**
   * The resource tag.
   * @see de.caff.i18n.I18n#getString
   * @see de.caff.i18n.I18n#format
   */
  protected  String   resTag;
  /**
   * The arguments for the resource.
   * @see de.caff.i18n.I18n#format
   */
  protected  Object[] args  = null;
  /**
   *  The original exception (if any).
   */
  protected Throwable originalException;

  /**
   *  Constructor for a simple I18n message.
   *  @param  tag  I18n resource tag of message
   *  @see I18n#getString(String)
   */
  public ResourcedException(String tag)
  {
    resTag = tag;
  }

  /**
   *  Constructor for an I18n message with arguments.
   *  @param  tag   I18n resource tag for a message with arguments
   *  @param  args  arguments
   *  @see I18n#format(String, Object[])
   */
  public ResourcedException(String tag, Object[] args)
  {
    resTag    = tag;
    this.args = args;
  }

  /**
   *  Constructor for an I18n message with one argument.
   *  @param  tag   resource tag
   *  @param  arg   argument
   *  @see I18n#format(String, Object[])
   */
  public ResourcedException(String tag, Object arg)
  {
    this(tag, new Object[] { arg });
  }

  /**
   *  Constructor for an I18n message with two argument.
   *  @param  tag   resource tag
   *  @param  arg1  first argument
   *  @param  arg2  second argument
   *  @see I18n#format(String, Object[])
   */
  public ResourcedException(String tag, Object arg1, Object arg2)
  {
    this(tag, new Object[] { arg1, arg2 });
  }

  /**
   *  Constructor for an I18n message with three argument.
   *  @param  tag   resource tag
   *  @param  arg1  first argument
   *  @param  arg2  second argument
   *  @param  arg3  second argument
   *  @see I18n#format(String, Object[])
   */
  public ResourcedException(String tag, Object arg1, Object arg2, Object arg3)
  {
    this(tag, new Object[] { arg1, arg2, arg3 });
  }

  /**
   *  Constructor for a simple I18n message after receiving another exeption.
   *
   *  @param originalException wrapped exception
   *  @param tag   I18n resource tag for a message with arguments
   *  @see I18n#getString(String)
   */
  public ResourcedException(Throwable originalException, String tag)
  {
    this(tag);
    setOriginalException(originalException);
  }

  /**
   *  Constructor for a I18n message with arguments after receiving another exeption.
   *
   *  @param originalException wrapped exception
   *  @param tag               resource tag
   *  @param  args  arguments
   *  @see I18n#format(String, Object[])
   */
  public ResourcedException(Throwable originalException, String tag, Object[] args)
  {
    this(tag, args);
    setOriginalException(originalException);
  }

  /**
   *  Constructor for a I18n message with one argument after receiving another exeption.
   *
   *  @param originalException wrapped exception
   *  @param tag               resource tag
   *  @param arg  argument
   *  @see I18n#format(String, Object[])
   */
  public ResourcedException(Throwable originalException, String tag, Object arg)
  {
    this(originalException, tag, new Object[] { arg });
  }

  /**
   *  Constructor for a I18n message with two arguments after receiving another exeption.
   *
   *  @param originalException wrapped exception
   *  @param tag               resource tag
   *  @param arg1  first argument
   *  @param arg2  second argument
   *  @see I18n#format(String, Object[])
   */
  public ResourcedException(Throwable originalException, String tag, Object arg1, Object arg2)
  {
    this(originalException, tag, new Object[] { arg1, arg2 });
  }

  /**
   *  Constructor for a I18n message with three arguments after receiving another exeption.
   *
   *  @param originalException wrapped exception
   *  @param tag               resource tag
   *  @param arg1  first argument
   *  @param arg2  second argument
   *  @param arg3  third argument
   *  @see I18n#format(String, Object[])
   */
  public ResourcedException(Throwable originalException, String tag, Object arg1, Object arg2, Object arg3)
  {
    this(originalException, tag, new Object[] { arg1, arg2, arg3 });
  }

  /**
   *  Get the message with default locale.
   *  @return  message 
   */
  public String getMessage()
  {
    return getMessage(null);
  }

  /**
   *  Get the message for a locale.
   *  @param l requested locale
   *  @return message for the given locale
   */
  public String getMessage(Locale l)
  {
    if (args != null) {
      return I18n.format(resTag, args, l);
    }
    else {
      return I18n.getString(resTag, l);
    }
  }

  /**
   *  Get the exception from where this resourced exception originated.
   *  @return original exception or <code>null</code> if there is no original
   *          exception
   */
  public Throwable getOriginalException()
  {
    return originalException;
  }

  /**
   *  Set the exception from where this resourced exception originated.
   *  @param originalException original exception
   */
  public final void setOriginalException(Throwable originalException)
  {
    this.originalException = originalException;
  }
}
