// Ported to C++ by Eduard Heidt

/*****************************************************************************************
 
	A S T E R O I D S

******************************************************************************************
	
	This is an exact Java port of the arcade game Asteroids, which was originally
	written by Ed Logg of Atari in 1979.

	The Java version was created by translating the Asteroids program code,
	originally written for the 6502 CPU of the Asteroids machine, to Java.
	This translation was made by a static binary recompilation program, which generates
	corresponding Java program code for each 6502 instruction and applies some
	optimization techniques to the generated program (e.g., removal of redundant
	flag calculations).

	Norbert Kehrer        http://web.utanet.at/nkehrer/

	Austria, September 2005.


******************************************************************************************

	Credits:
	The coin dropping sound is from Peter Hirschberg's Vector Dream emulator.

	Restrictions:
	This software may not be used for any commercial purposes and may not
	be placed on any commercial web site.
	If you improve, port, or modify this program, please refer to me in your version
	and let me know of your efforts. You will find my email address on my 
	website http://web.utanet.at/nkehrer/. 
	If you place the applet on your (non-commercial) web site, please mention
	me as the author and provide a link to my site http://web.utanet.at/nkehrer/.

******************************************************************************************/


#pragma once

#include "d:\development\zlib\include\zlib.h"
#include <windows.h>
#include <stdio.h>
#include "asteroid_roms.h"

#include "game.h"


class Emulator
{
	friend class Game;
private:

	// *** Flags
	int 	flagN;
	int 	flagV;
	int 	flagB;
	int 	flagD;
	int 	flagI;
	int 	flagC;
	int 	flagZ;

	// *** Registers
	int 	regA;
	int 	regX;
	int 	regY;
	int 	regS;
	int 	regP;
	int	regPC;

	// *** Memory
	int mem[0x10000];

	// *** Stack for fast stack access (not yet implemented)
	int	stack[0x100];
	int	sp; 	// stack pointer

	// *** temporary variables
	int	temp;
	int 	z, zl, zh;	// for ADC and SBC

	// *** Interrupt modes
	static const int INT_NONE = 0;
	static const int INT_IRQ  = 1;
	static const int INT_NMI  = 2;

	// *** Vectors
	int nmiVec;
	int irqVec;
	int resetVec;

	// *** Cycle count
	long cycles;
	long cyclesPeriodicTask;
	
	
	char keys;
	

	// *** The frontend
	int readMem (int addr) 
	{	
		bool keyHyperSpace = keys&1;
		bool keyFire       = keys&2;
		bool keyCoin1      = true;  
		bool keyCoin2      = true; 
		bool keyCoin3      = true; 
		bool keyStart1     = keys&0x20; 
		bool keyStart2     = false;      
		bool keyThrust     = keys&0x4;       
		bool keyRight      = keys&0x8;       
		bool keyLeft       = keys&0x10;       	
	
		switch ( addr ) {
    
			// *** Input ports
			case 0x2003:	return (keyHyperSpace  ? 0xff : 0x7f);
			case 0x2004:	return (keyFire        ? 0xff : 0x7f);
			case 0x2400:	return (keyCoin1       ? 0xff : 0x7f);
			case 0x2401:	return (keyCoin2       ? 0xff : 0x7f);
			case 0x2402:	return (keyCoin3       ? 0xff : 0x7f);
			case 0x2403:	return (keyStart1      ? 0xff : 0x7f);
			case 0x2404:	return (keyStart2      ? 0xff : 0x7f);
			case 0x2405:	return (keyThrust      ? 0xff : 0x7f);
			case 0x2406:	return (keyRight       ? 0xff : 0x7f);
			case 0x2407:	return (keyLeft        ? 0xff : 0x7f);

			// *** DIP switches
			case 0x2800:	return 0xfc | 0x00;	// 2 = 1 Coin 1 Play,  0 = free play
			case 0x2801:	return 0xfc | 0x00;	// Unknown
			case 0x2802:	return 0xfc | 0x01;	// 4-ship game
			case 0x2803:	return 0xfc | 0x00;	// English
		};


		return mem[addr];
	};
	
	void writeMem (int addr, int value) 
	{
		static int asteroid_bank = 0;
		static int asteroid_newbank = 0;
	
		int temp, i;

		switch ( addr ) {

		// *** Start the DVG engine and draw the screen
		case 0x3000:
			callback();
			
/*			frameCounter++;
			if ((frameCounter % frameSkip) == 0) {
				dvg.dvgRun();
				if (showCredits)
					displayCredits();
				canvasGraphics.drawImage(bufferImage,0,0,null);
			};
*/
			break;

		// *** Memory bank switch
		case 0x3200:
			asteroid_newbank = (value >> 2) & 0x01;
			if (asteroid_bank != asteroid_newbank) {
				// Perform bankswitching on page 2 and page 3 
				asteroid_bank = asteroid_newbank;
				for (i = 0; i < 0x100; i++) {
					temp = mem[0x200 + i];
					mem[0x200 + i] = mem[0x300 + i];
					mem[0x300 + i] = temp;
				};
			};
			break;


		// *** Sound circuit control
/*
		case 0x3600: // Explosions
			if (newByte == 0x3d) largeExplSFX.play(); 
			if (newByte == 0xfd) mediumExplSFX.play(); 
			if (newByte == 0xbd) smallExplSFX.play(); 
			break;

		case 0x3a00: // Heartbeat
			if ((newByte&0x10) != 0) {
				if ((newByte&0x04) != 0) beat2SFX.play(); 
				else			 beat1SFX.play(); 
			}
			break;	

		case 0x3c00: // Saucer
			if ((newByte&0x80) != 0) {
				if ((r.mem[0x3c02]&0x80) != 0)	 largeSaucerSFX.loop();	
				else				 smallSaucerSFX.loop();	
			}	
			else {
				largeSaucerSFX.stop();	
				smallSaucerSFX.stop();	
			}
 			break;	
		
		case 0x3c01: // Ufo fire
			thisUfoFire = (newByte&0x80);
			if (thisUfoFire != 0) 
				if (thisUfoFire != lastUfoFire) ufoFireSFX.play();	
			lastUfoFire = thisUfoFire;	
 			break;
		
		case 0x3c03: // Thrust
			if ((newByte&0x80) != 0) thrustSFX.loop();	
			else			 thrustSFX.stop();	
 			break;	

		case 0x3c04: // Player fire
			thisPlayerFire = (newByte&0x80);
			if (thisPlayerFire != 0) 
				if (thisPlayerFire != lastPlayerFire) yourFireSFX.play();	
			lastPlayerFire = thisPlayerFire;	
 			break;	

		case 0x3c05: // Extra ship
			if (newByte != 0) 	extraShipSFX.loop();	
			else			extraShipSFX.stop();	
 			break;
*/
		} // end case


		// *** Normal RAM
		
		if ( addr < 0x5000 )         // 0x6800 - 0xffff ROM chips
			mem[addr] = value;	
		
		return;
	};

	int periodicProcessing ()
	{
		//return appl.periodicProcessing() ;
		trace("Period:");
		Sleep(0);
		return INT_NMI;
	};



	bool dotrace;

	void trace (char* disAss) 
	{
		if (dotrace) 
		{
			printf("%s PC=%x\n", disAss, regPC);
			/*
			+"\t"+disAss+
				"\tA="+hex(regA)+"\tX="+hex(regX)+"\tY="+hex(regY)+"\tS="+hex(regS)      
				+ "\tt="+System.currentTimeMillis());
				
			checkFlag (flagC,"flagC");
			checkFlag (flagN,"flagN");
			checkFlag (flagZ,"flagZ");
			checkFlag (flagV,"flagV");
			checkFlag (flagD,"flagD");
			checkFlag (flagI,"flagI");
			
			checkReg(regA,"A");
			checkReg(regX,"X");
			checkReg(regY,"Y");
			*/
		};
	};


	void  checkFlag(int f, char* fs) 
	{
		//if ((f!=0)&&(f!=1))
		//	System.out.println("**** ERROR: "+fs+"="+f);
	};

	void checkReg(int r, char* rs) 
	{
		//if ((r<0)||(r>255))
		//	System.out.println("**** ERROR: "+rs+"="+r);
	};

public:

	char* state;

	void SaveState()
	{
		if(!state)
			state = new char[sizeof(Emulator)];
		memcpy(state, this, sizeof(Emulator));
	}
	void RestoreState()
	{
		if(state)
			memcpy(this, state, sizeof(Emulator));
	}

	Game* game;
	void callback()
	{
		static FramePacket frame;
		static KeysPacket k;
		static int t = 0;
		
		
		unsigned short start = (unsigned char)mem[0x4000+0] | (unsigned char)mem[0x4000+1] << 8;
		//unsigned short start2 = (unsigned char)mem[0x4000+1024] | (unsigned char)mem[0x4000+1024+1] << 8;
		//printf("--> %x\t%x", start, start2);
		
		int offset = 0;
		
		if(start == 0xe201)
			offset = 1024;
			
		for(int i = 0; i < 2;i++)
			frame.vectorram[i] = (char)mem[0x4000+i];

		for(int i = 2; i < 1024;i++)
			frame.vectorram[i] = (char)mem[0x4000+i+offset];
			
		t++;

		keys = k.keys;
		game->InterpretScreen(frame, game->state);

		if(t == 1 && !game->state.ship.IsPresent())
			t = 0;
		
		k.clear();   // alle Tasten loslassen
		game->state.t = t;
		game->player.MakeTurn (game->state, k);

		if(t>0)
		{	
			gzwrite(gzf, frame.vectorram, 1024);
			gzwrite(gzf, &k.keys, 1);

			static int count = 0;
			count++;
			if(count >= 18000)
			{
				count = 0;
				throw connection_error("t>18000");
			}
		}
		
		//Sleep(10);


	}
	
	
	gzFile gzf;
	// *** Constructor
	Emulator(Game* g )
	{
		ZeroMemory(this, sizeof(Emulator));
		game = g;
		
		gzf = gzopen("game_dump.gz", "wb");
		char header[] = {0x63,0x74,0x6D,0x61,0x6D,0x65,0x32,0x0D,0x0A,0x02,0x00,0x0B,0x01,0x7F,0x00,0x00,
								01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
								00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
								00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00};
		gzwrite(gzf, header, 0x39);
			
		// load Asteroids ROMs
		
		//035145.02  	 	2048  	cpu1  	6800  	good  	no  	0  
		//035144.02  	 	2048  	cpu1  	7000  	good  	no  	0  
		//035143.02  	 	2048  	cpu1  	7800  	good  	no  	0  
		//035127.02  	 	2048  	cpu1  	5000  	good  	no  	0 
		 
/*
		 int j = 0;
		 if(FILE* f = fopen("roms\\035145.02", "rb"))
		 {
			for (int i=0; i<2048; i++)
				 mem6800[j++] = (unsigned char) fgetc(f);
			fclose(f);
		 }
		 
		 if(FILE* f = fopen("roms\\035144.02", "rb"))
		 {
			for (int i=0; i<2048; i++)
				mem6800[j++] = (unsigned char) fgetc(f);
			fclose(f);
		 }
		 
		 if(FILE* f = fopen("roms\\035143.02", "rb"))
		 {
			for (int i=0; i<2048; i++)
				mem6800[j++] = (unsigned char) fgetc(f);
			fclose(f);
		 }
		 
		  if(FILE* f = fopen("roms\\035127.02", "rb"))
		 {
			for (int i=0; i<2048; i++)
				mem[0x5000+i] = (unsigned char) fgetc(f);
			fclose(f);
		 }
		 
*/
		int i;
		for (i=0x0000; i<0x1800; i++)
			mem[0x6800+i] = mem6800[i];
			
		for (i=0x0000; i<0x0800; i++)
			mem[0xf800+i] = mem6800[0x1000+i];
			
		for (i=0x0000; i<0x0800; i++)
			mem[0x5000+i] = mem5000[i];

		// set up periodic task
		cyclesPeriodicTask = 1500000 / 250;

		// set interrupt vectors
		nmiVec = ((mem[0xfffb]<<8) | mem[0xfffa]);
		irqVec = ((mem[0xffff]<<8) | mem[0xfffe]);
		resetVec = ((mem[0xfffd]<<8) | mem[0xfffc]);

		// initialize program counter
		regPC = resetVec;
	};
	
	~Emulator()
	{
		if(state)
			delete state;
		gzclose(gzf);
	}


	// *** Program
	void prog () 
	{

		int interrupt;


		for (;;) 
		{
			// perform periodic task
			if (cycles >= cyclesPeriodicTask) {
				cycles -= cyclesPeriodicTask;
				interrupt = periodicProcessing();

				// take an interrupt if necessary
				if((interrupt == INT_NMI) || ((interrupt == INT_IRQ) && (flagI == 0))) { 
					// ICount -= 7; 
 					mem[0x0100 | regS] = (regPC >> 8);
					regS = (regS - 1) & 0xff;
					mem[0x0100 | regS] = (regPC & 0xff);
					regS = (regS - 1) & 0xff;
					flagB = 0;
					mem[0x0100 | regS] =((flagN<<7)|(flagV<<6)|(0x20)|(flagB<<4)|(flagD<<3)|(flagI<<2)|(flagZ<<1)|flagC);
					regS = (regS - 1) & 0xff;
					flagD = 0;
					if (interrupt == INT_NMI) {
						regPC = nmiVec; 	// NMI
					}
					else {
						flagI = 1;
						regPC = irqVec;		// IRQ
					}
				};
			};

			// run program
			switch ((regPC>>10)<<2) {

				case 0x68:	exec6800();	break;
				case 0x6c:	exec6c00();	break;
				case 0x70:	exec7000();	break;
				case 0x74:	exec7400();	break;
				case 0x78:	exec7800();	break;
				case 0x7c:	exec7c00();	break;
			};

		}

	};


// ********************** generated code begins here






// Reading ROM ... 035145.02 ... 2048 bytes.
// Reading ROM ... 035144.02 ... 2048 bytes.
// Reading ROM ... 035143.02 ... 2048 bytes.
// Reading ROM ... 035143.02 ... 2048 bytes.
// Reading ROM ... 035127.02 ... 2048 bytes.

	void exec6800 () 
	{

		switch (regPC) {

case 0x6800:
	// $6800	jmp $7cf3
	regPC = 0x7cf3;
	cycles += 3;
	break;

case 0x6803:
	// $6803	jsr $6efa
	mem[0x0100 | regS] = 0x68;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x6;
	regS = (regS - 1) & 0xff;
	regPC = 0x6efa;
	cycles += 6;
	break;

case 0x6806:
	// $6806	jsr $6ed8
	mem[0x0100 | regS] = 0x68;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x9;
	regS = (regS - 1) & 0xff;
	regPC = 0x6ed8;
	cycles += 6;
	break;

case 0x6809:
	// $6809	jsr $7168
	mem[0x0100 | regS] = 0x68;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xc;
	regS = (regS - 1) & 0xff;
	regPC = 0x7168;
	cycles += 6;
	break;

case 0x680c:
	// $680c	lda $2007
	regA = readMem(0x2007);
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 4;
	regPC = 0x680f;

case 0x680f:
	// $680f	bmi $680f
	if (flagN == 1) {
		cycles += 2;
		regPC = 0x680f;
		break;
	};
	// $6811	lsr $5b
	temp = mem[0x5b];
	flagC = temp & 1;
	temp >>= 1;
	mem[0x5b] = (temp & 0xff);
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = ((temp & 0xff) >> 7) & 1;
	// $6813	nop
	// No operation
	// $6814	nop
	// No operation
	cycles += 11;
	regPC = 0x6815;

case 0x6815:
	// $6815	lda $2002
	regA = readMem(0x2002);
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6818	bmi $6815
	if (flagN == 1) {
		cycles += 6;
		regPC = 0x6815;
		break;
	};
	// $681a	lda $4001
	regA = mem[0x4001];
	// $681d	eor #$02
	regA = (regA ^ 0x2) & 0xff;
	// $681f	sta $4001
	mem[0x4001] = regA;
	// $6822	sta $3000
	writeMem(0x3000, regA);
	// $6825	sta $3400
	writeMem(0x3400, regA);
	// $6828	inc $5c
	temp = (mem[0x5c] + 1) & 0xff;
	mem[0x5c] = temp;
	flagZ = (temp == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $682a	bne $682e
	if (flagZ == 0) {
		cycles += 31;
		regPC = 0x682e;
		break;
	};
	// $682c	inc $5d
	temp = (mem[0x5d] + 1) & 0xff;
	mem[0x5d] = temp;
	flagZ = (temp == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	cycles += 36;
	regPC = 0x682e;

case 0x682e:
	// $682e	ldx #$40
	regX = 0x40;
	// $6830	and #$02
	regA = (regA & 0x2) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6832	bne $6836
	if (flagZ == 0) {
		cycles += 6;
		regPC = 0x6836;
		break;
	};
	// $6834	ldx #$44
	regX = 0x44;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	cycles += 8;
	regPC = 0x6836;

case 0x6836:
	// $6836	lda #$02
	regA = 0x2;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6838	sta $02
	mem[0x2] = regA;
	// $683a	stx $03
	mem[0x3] = regX;
	// $683c	jsr $6885
	mem[0x0100 | regS] = 0x68;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x3f;
	regS = (regS - 1) & 0xff;
	regPC = 0x6885;
	cycles += 14;
	break;

case 0x683f:
	// $683f	bcs $6803
	if (flagC == 1) {
		cycles += 2;
		regPC = 0x6803;
		break;
	};
	// $6841	jsr $765c
	mem[0x0100 | regS] = 0x68;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x44;
	regS = (regS - 1) & 0xff;
	regPC = 0x765c;
	cycles += 8;
	break;

case 0x6844:
	// $6844	jsr $6d90
	mem[0x0100 | regS] = 0x68;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x47;
	regS = (regS - 1) & 0xff;
	regPC = 0x6d90;
	cycles += 6;
	break;

case 0x6847:
	// $6847	bpl $6864
	if (flagN == 0) {
		cycles += 2;
		regPC = 0x6864;
		break;
	};
	// $6849	jsr $73c4
	mem[0x0100 | regS] = 0x68;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x4c;
	regS = (regS - 1) & 0xff;
	regPC = 0x73c4;
	cycles += 8;
	break;

case 0x684c:
	// $684c	bcs $6864
	if (flagC == 1) {
		cycles += 2;
		regPC = 0x6864;
		break;
	};
	// $684e	lda $5a
	regA = mem[0x5a];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6850	bne $685e
	if (flagZ == 0) {
		cycles += 7;
		regPC = 0x685e;
		break;
	};
	// $6852	jsr $6cd7
	mem[0x0100 | regS] = 0x68;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x55;
	regS = (regS - 1) & 0xff;
	regPC = 0x6cd7;
	cycles += 13;
	break;

case 0x6855:
	// $6855	jsr $6e74
	mem[0x0100 | regS] = 0x68;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x58;
	regS = (regS - 1) & 0xff;
	regPC = 0x6e74;
	cycles += 6;
	break;

case 0x6858:
	// $6858	jsr $703f
	mem[0x0100 | regS] = 0x68;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x5b;
	regS = (regS - 1) & 0xff;
	regPC = 0x703f;
	cycles += 6;
	break;

case 0x685b:
	// $685b	jsr $6b93
	mem[0x0100 | regS] = 0x68;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x5e;
	regS = (regS - 1) & 0xff;
	regPC = 0x6b93;
	cycles += 6;
	break;

case 0x685e:
	// $685e	jsr $6f57
	mem[0x0100 | regS] = 0x68;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x61;
	regS = (regS - 1) & 0xff;
	regPC = 0x6f57;
	cycles += 6;
	break;

case 0x6861:
	// $6861	jsr $69f0
	mem[0x0100 | regS] = 0x68;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x64;
	regS = (regS - 1) & 0xff;
	regPC = 0x69f0;
	cycles += 6;
	break;

case 0x6864:
	// $6864	jsr $724f
	mem[0x0100 | regS] = 0x68;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x67;
	regS = (regS - 1) & 0xff;
	regPC = 0x724f;
	cycles += 6;
	break;

case 0x6867:
	// $6867	jsr $7555
	mem[0x0100 | regS] = 0x68;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x6a;
	regS = (regS - 1) & 0xff;
	regPC = 0x7555;
	cycles += 6;
	break;

case 0x686a:
	// $686a	lda #$7f
	regA = 0x7f;
	// $686c	tax
	regX = regA;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $686d	jsr $7c03
	mem[0x0100 | regS] = 0x68;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x70;
	regS = (regS - 1) & 0xff;
	regPC = 0x7c03;
	cycles += 10;
	break;

case 0x6870:
	// $6870	jsr $77b5
	mem[0x0100 | regS] = 0x68;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x73;
	regS = (regS - 1) & 0xff;
	regPC = 0x77b5;
	cycles += 6;
	break;

case 0x6873:
	// $6873	jsr $7bc0
	mem[0x0100 | regS] = 0x68;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x76;
	regS = (regS - 1) & 0xff;
	regPC = 0x7bc0;
	cycles += 6;
	break;

case 0x6876:
	// $6876	lda $02fb
	regA = mem[0x2fb];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6879	beq $687e
	if (flagZ == 1) {
		cycles += 6;
		regPC = 0x687e;
		break;
	};
	// $687b	dec $02fb
	temp = (mem[0x2fb] - 1) & 0xff;
	mem[0x2fb] = temp;
	flagZ = (temp == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	cycles += 12;
	regPC = 0x687e;

case 0x687e:
	// $687e	ora $02f6
	regA = (regA | mem[0x2f6]) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6881	bne $680c
	if (flagZ == 0) {
		cycles += 6;
		regPC = 0x680c;
		break;
	};
	// $6883	beq $6809
	if (flagZ == 1) {
		cycles += 8;
		regPC = 0x6809;
		break;
	};
	cycles += 8;
	regPC = 0x6885;

case 0x6885:
	// $6885	lda $1c
	regA = mem[0x1c];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6887	beq $689d
	if (flagZ == 1) {
		cycles += 5;
		regPC = 0x689d;
		break;
	};
	// $6889	lda $5a
	regA = mem[0x5a];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $688b	bne $6890
	if (flagZ == 0) {
		cycles += 10;
		regPC = 0x6890;
		break;
	};
	// $688d	jmp $6960
	regPC = 0x6960;
	cycles += 13;
	break;

case 0x6890:
	// $6890	dec $5a
	temp = (mem[0x5a] - 1) & 0xff;
	mem[0x5a] = temp;
	flagZ = (temp == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6892	jsr $69e2
	mem[0x0100 | regS] = 0x68;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x95;
	regS = (regS - 1) & 0xff;
	regPC = 0x69e2;
	cycles += 11;
	break;

case 0x6895:
	// $6895	clc
	flagC = 0;
	// $6896	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 8;
	break;

case 0x6897:
	// $6897	lda #$02
	regA = 0x2;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6899	sta $70
	mem[0x70] = regA;
	// $689b	bne $68b0
	if (flagZ == 0) {
		cycles += 7;
		regPC = 0x68b0;
		break;
	};
	cycles += 7;
	regPC = 0x689d;

case 0x689d:
	// $689d	lda $71
	regA = mem[0x71];
	// $689f	and #$03
	regA = (regA & 0x3) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $68a1	beq $6897
	if (flagZ == 1) {
		cycles += 7;
		regPC = 0x6897;
		break;
	};
	// $68a3	clc
	flagC = 0;
	// $68a4	adc #$07
	temp = 0x7;
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $68a6	tay
	regY = regA;
	// $68a7	lda $32
	regA = mem[0x32];
	// $68a9	and $33
	regA = (regA & mem[0x33]) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $68ab	bpl $68b0
	if (flagN == 0) {
		cycles += 21;
		regPC = 0x68b0;
		break;
	};
	// $68ad	jsr $77f6
	mem[0x0100 | regS] = 0x68;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xb0;
	regS = (regS - 1) & 0xff;
	regPC = 0x77f6;
	cycles += 27;
	break;

case 0x68b0:
	// $68b0	ldy $70
	regY = mem[0x70];
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $68b2	beq $6895
	if (flagZ == 1) {
		cycles += 5;
		regPC = 0x6895;
		break;
	};
	// $68b4	ldx #$01
	regX = 0x1;
	// $68b6	lda $2403
	regA = readMem(0x2403);
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $68b9	bmi $68de
	if (flagN == 1) {
		cycles += 13;
		regPC = 0x68de;
		break;
	};
	// $68bb	cpy #$02
	temp = regY - 0x2;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $68bd	bcc $693b
	if (flagC == 0) {
		cycles += 17;
		regPC = 0x693b;
		break;
	};
	// $68bf	lda $2404
	regA = readMem(0x2404);
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $68c2	bpl $693b
	if (flagN == 0) {
		cycles += 23;
		regPC = 0x693b;
		break;
	};
	// $68c4	lda $6f
	regA = mem[0x6f];
	// $68c6	ora #$04
	regA = (regA | 0x4) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $68c8	sta $6f
	mem[0x6f] = regA;
	// $68ca	sta $3200
	writeMem(0x3200, regA);
	// $68cd	jsr $6ed8
	mem[0x0100 | regS] = 0x68;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xd0;
	regS = (regS - 1) & 0xff;
	regPC = 0x6ed8;
	cycles += 41;
	break;

case 0x68d0:
	// $68d0	jsr $7168
	mem[0x0100 | regS] = 0x68;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xd3;
	regS = (regS - 1) & 0xff;
	regPC = 0x7168;
	cycles += 6;
	break;

case 0x68d3:
	// $68d3	jsr $71e8
	mem[0x0100 | regS] = 0x68;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xd6;
	regS = (regS - 1) & 0xff;
	regPC = 0x71e8;
	cycles += 6;
	break;

case 0x68d6:
	// $68d6	lda $56
	regA = mem[0x56];
	// $68d8	sta $58
	mem[0x58] = regA;
	// $68da	ldx #$02
	regX = 0x2;
	// $68dc	dec $70
	temp = (mem[0x70] - 1) & 0xff;
	mem[0x70] = temp;
	flagZ = (temp == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	cycles += 13;
	regPC = 0x68de;

case 0x68de:
	// $68de	stx $1c
	mem[0x1c] = regX;
	// $68e0	dec $70
	temp = (mem[0x70] - 1) & 0xff;
	mem[0x70] = temp;
	// $68e2	lda $6f
	regA = mem[0x6f];
	// $68e4	and #$f8
	regA = (regA & 0xf8) & 0xff;
	// $68e6	eor $1c
	regA = (regA ^ mem[0x1c]) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $68e8	sta $6f
	mem[0x6f] = regA;
	// $68ea	sta $3200
	writeMem(0x3200, regA);
	// $68ed	jsr $71e8
	mem[0x0100 | regS] = 0x68;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xf0;
	regS = (regS - 1) & 0xff;
	regPC = 0x71e8;
	cycles += 29;
	break;

case 0x68f0:
	// $68f0	lda #$01
	regA = 0x1;
	// $68f2	sta $02fa
	mem[0x2fa] = regA;
	// $68f5	sta $03fa
	mem[0x3fa] = regA;
	// $68f8	lda #$92
	regA = 0x92;
	// $68fa	sta $02f8
	mem[0x2f8] = regA;
	// $68fd	sta $03f8
	mem[0x3f8] = regA;
	// $6900	sta $03f7
	mem[0x3f7] = regA;
	// $6903	sta $02f7
	mem[0x2f7] = regA;
	// $6906	lda #$7f
	regA = 0x7f;
	// $6908	sta $02fb
	mem[0x2fb] = regA;
	// $690b	sta $03fb
	mem[0x3fb] = regA;
	// $690e	lda #$05
	regA = 0x5;
	// $6910	sta $02fd
	mem[0x2fd] = regA;
	// $6913	sta $03fd
	mem[0x3fd] = regA;
	// $6916	lda #$ff
	regA = 0xff;
	// $6918	sta $32
	mem[0x32] = regA;
	// $691a	sta $33
	mem[0x33] = regA;
	// $691c	lda #$80
	regA = 0x80;
	// $691e	sta $5a
	mem[0x5a] = regA;
	// $6920	asl a
	temp = (regA << 1);
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $6921	sta $18
	mem[0x18] = regA;
	// $6923	sta $19
	mem[0x19] = regA;
	// $6925	lda $56
	regA = mem[0x56];
	// $6927	sta $57
	mem[0x57] = regA;
	// $6929	lda #$04
	regA = 0x4;
	// $692b	sta $6c
	mem[0x6c] = regA;
	// $692d	sta $6e
	mem[0x6e] = regA;
	// $692f	lda #$30
	regA = 0x30;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6931	sta $02fc
	mem[0x2fc] = regA;
	// $6934	sta $03fc
	mem[0x3fc] = regA;
	// $6937	sta $3e00
	writeMem(0x3e00, regA);
	// $693a	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 103;
	break;

case 0x693b:
	// $693b	lda $32
	regA = mem[0x32];
	// $693d	and $32
	regA = (regA & mem[0x32]) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $693f	bpl $694c
	if (flagN == 0) {
		cycles += 8;
		regPC = 0x694c;
		break;
	};
	// $6941	lda $5c
	regA = mem[0x5c];
	// $6943	and #$20
	regA = (regA & 0x20) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6945	bne $694c
	if (flagZ == 0) {
		cycles += 15;
		regPC = 0x694c;
		break;
	};
	// $6947	ldy #$06
	regY = 0x6;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $6949	jsr $77f6
	mem[0x0100 | regS] = 0x69;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x4c;
	regS = (regS - 1) & 0xff;
	regPC = 0x77f6;
	cycles += 23;
	break;

case 0x694c:
	// $694c	lda $5c
	regA = mem[0x5c];
	// $694e	and #$0f
	regA = (regA & 0xf) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6950	bne $695e
	if (flagZ == 0) {
		cycles += 7;
		regPC = 0x695e;
		break;
	};
	// $6952	lda #$01
	regA = 0x1;
	// $6954	cmp $70
	temp = regA - mem[0x70];
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	// $6956	adc #$01
	temp = 0x1;
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $6958	eor #$01
	regA = (regA ^ 0x1) & 0xff;
	// $695a	eor $6f
	regA = (regA ^ mem[0x6f]) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $695c	sta $6f
	mem[0x6f] = regA;
	cycles += 22;
	regPC = 0x695e;

case 0x695e:
	// $695e	clc
	flagC = 0;
	// $695f	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 8;
	break;

case 0x6960:
	// $6960	lda $5c
	regA = mem[0x5c];
	// $6962	and #$3f
	regA = (regA & 0x3f) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6964	bne $6970
	if (flagZ == 0) {
		cycles += 7;
		regPC = 0x6970;
		break;
	};
	// $6966	lda $02fc
	regA = mem[0x2fc];
	// $6969	cmp #$08
	temp = regA - 0x8;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $696b	beq $6970
	if (flagZ == 1) {
		cycles += 15;
		regPC = 0x6970;
		break;
	};
	// $696d	dec $02fc
	temp = (mem[0x2fc] - 1) & 0xff;
	mem[0x2fc] = temp;
	flagZ = (temp == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	cycles += 21;
	regPC = 0x6970;

case 0x6970:
	// $6970	ldx $18
	regX = mem[0x18];
	// $6972	lda $57,x
	regA = mem[(0x57+regX)&0xff];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6974	bne $6992
	if (flagZ == 0) {
		cycles += 9;
		regPC = 0x6992;
		break;
	};
	// $6976	lda $021f
	regA = mem[0x21f];
	// $6979	ora $0220
	regA = (regA | mem[0x220]) & 0xff;
	// $697c	ora $0221
	regA = (regA | mem[0x221]) & 0xff;
	// $697f	ora $0222
	regA = (regA | mem[0x222]) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6982	bne $6992
	if (flagZ == 0) {
		cycles += 27;
		regPC = 0x6992;
		break;
	};
	// $6984	ldy #$07
	regY = 0x7;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $6986	jsr $77f6
	mem[0x0100 | regS] = 0x69;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x89;
	regS = (regS - 1) & 0xff;
	regPC = 0x77f6;
	cycles += 35;
	break;

case 0x6989:
	// $6989	lda $1c
	regA = mem[0x1c];
	// $698b	cmp #$02
	temp = regA - 0x2;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $698d	bcc $6992
	if (flagC == 0) {
		cycles += 7;
		regPC = 0x6992;
		break;
	};
	// $698f	jsr $69e2
	mem[0x0100 | regS] = 0x69;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x92;
	regS = (regS - 1) & 0xff;
	regPC = 0x69e2;
	cycles += 13;
	break;

case 0x6992:
	// $6992	lda $021b
	regA = mem[0x21b];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6995	bne $69cd
	if (flagZ == 0) {
		cycles += 6;
		regPC = 0x69cd;
		break;
	};
	// $6997	lda $02fa
	regA = mem[0x2fa];
	// $699a	cmp #$80
	temp = regA - 0x80;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $699c	bne $69cd
	if (flagZ == 0) {
		cycles += 14;
		regPC = 0x69cd;
		break;
	};
	// $699e	lda #$10
	regA = 0x10;
	// $69a0	sta $02fa
	mem[0x2fa] = regA;
	// $69a3	ldx $1c
	regX = mem[0x1c];
	// $69a5	lda $57
	regA = mem[0x57];
	// $69a7	ora $58
	regA = (regA | mem[0x58]) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $69a9	beq $69cf
	if (flagZ == 1) {
		cycles += 31;
		regPC = 0x69cf;
		break;
	};
	// $69ab	jsr $702d
	mem[0x0100 | regS] = 0x69;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xae;
	regS = (regS - 1) & 0xff;
	regPC = 0x702d;
	cycles += 37;
	break;

case 0x69ae:
	// $69ae	dex
	regX--;
	regX&= 0xff;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $69af	beq $69cd
	if (flagZ == 1) {
		cycles += 4;
		regPC = 0x69cd;
		break;
	};
	// $69b1	lda #$80
	regA = 0x80;
	// $69b3	sta $5a
	mem[0x5a] = regA;
	// $69b5	lda $18
	regA = mem[0x18];
	// $69b7	eor #$01
	regA = (regA ^ 0x1) & 0xff;
	// $69b9	tax
	regX = regA;
	// $69ba	lda $57,x
	regA = mem[(0x57+regX)&0xff];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $69bc	beq $69cd
	if (flagZ == 1) {
		cycles += 22;
		regPC = 0x69cd;
		break;
	};
	// $69be	stx $18
	mem[0x18] = regX;
	// $69c0	lda #$04
	regA = 0x4;
	// $69c2	eor $6f
	regA = (regA ^ mem[0x6f]) & 0xff;
	// $69c4	sta $6f
	mem[0x6f] = regA;
	// $69c6	sta $3200
	writeMem(0x3200, regA);
	// $69c9	txa
	regA = regX;
	// $69ca	asl a
	temp = (regA << 1);
	flagC = temp >> 8;
	regA = (temp & 0xff);
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $69cb	sta $19
	mem[0x19] = regA;
	cycles += 44;
	regPC = 0x69cd;

case 0x69cd:
	// $69cd	clc
	flagC = 0;
	// $69ce	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 8;
	break;

case 0x69cf:
	// $69cf	stx $1a
	mem[0x1a] = regX;
	// $69d1	lda #$ff
	regA = 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $69d3	sta $1c
	mem[0x1c] = regA;
	// $69d5	jsr $6efa
	mem[0x0100 | regS] = 0x69;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xd8;
	regS = (regS - 1) & 0xff;
	regPC = 0x6efa;
	cycles += 14;
	break;

case 0x69d8:
	// $69d8	lda $6f
	regA = mem[0x6f];
	// $69da	and #$f8
	regA = (regA & 0xf8) & 0xff;
	// $69dc	ora #$03
	regA = (regA | 0x3) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $69de	sta $6f
	mem[0x6f] = regA;
	// $69e0	clc
	flagC = 0;
	// $69e1	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 18;
	break;

case 0x69e2:
	// $69e2	ldy #$01
	regY = 0x1;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $69e4	jsr $77f6
	mem[0x0100 | regS] = 0x69;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xe7;
	regS = (regS - 1) & 0xff;
	regPC = 0x77f6;
	cycles += 8;
	break;

case 0x69e7:
	// $69e7	ldy $18
	regY = mem[0x18];
	// $69e9	iny
	regY++;
	regY&= 0xff;
	// $69ea	tya
	regA = regY;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $69eb	jsr $7bd1
	mem[0x0100 | regS] = 0x69;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xee;
	regS = (regS - 1) & 0xff;
	regPC = 0x7bd1;
	cycles += 13;
	break;

case 0x69ee:
	// $69ee	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 6;
	break;

case 0x69ef:
	// dc.b $71		; $69ef

case 0x69f0:
	// $69f0	ldx #$07
	regX = 0x7;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	cycles += 2;
	regPC = 0x69f2;

case 0x69f2:
	// $69f2	lda $021b,x
	regA = mem[0x21b+regX];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $69f5	beq $69f9
	if (flagZ == 1) {
		cycles += 6;
		regPC = 0x69f9;
		break;
	};
	// $69f7	bpl $69fd
	if (flagN == 0) {
		cycles += 8;
		regPC = 0x69fd;
		break;
	};
	cycles += 8;
	regPC = 0x69f9;

case 0x69f9:
	// $69f9	dex
	regX--;
	regX&= 0xff;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $69fa	bpl $69f2
	if (flagN == 0) {
		cycles += 4;
		regPC = 0x69f2;
		break;
	};
	// $69fc	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 10;
	break;

case 0x69fd:
	// $69fd	ldy #$1c
	regY = 0x1c;
	// $69ff	cpx #$04
	temp = regX - 0x4;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6a01	bcs $6a0a
	if (flagC == 1) {
		cycles += 6;
		regPC = 0x6a0a;
		break;
	};
	// $6a03	dey
	regY--;
	regY&= 0xff;
	// $6a04	txa
	regA = regX;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6a05	bne $6a0a
	if (flagZ == 0) {
		cycles += 12;
		regPC = 0x6a0a;
		break;
	};
	cycles += 12;
	regPC = 0x6a07;

case 0x6a07:
	// $6a07	dey
	regY--;
	regY&= 0xff;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $6a08	bmi $69f9
	if (flagN == 1) {
		cycles += 4;
		regPC = 0x69f9;
		break;
	};
	cycles += 4;
	regPC = 0x6a0a;

case 0x6a0a:
	// $6a0a	lda $0200,y
	regA = mem[0x200+regY];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6a0d	beq $6a07
	if (flagZ == 1) {
		cycles += 6;
		regPC = 0x6a07;
		break;
	};
	// $6a0f	bmi $6a07
	if (flagN == 1) {
		cycles += 8;
		regPC = 0x6a07;
		break;
	};
	// $6a11	sta $0b
	mem[0xb] = regA;
	// $6a13	lda $02af,y
	regA = mem[0x2af+regY];
	// $6a16	sec
	flagC = 1;
	// $6a17	sbc $02ca,x
	temp = mem[0x2ca+regX];
	if (flagD == 0) {   // no decimal mode
		z = regA - temp - ((flagC == 0) ? 1 : 0);
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh == 0) ? 1 : 0;
		regA = zl;
	}
	else {   // decimal mode
		zl = ((regA & 0x0f) - (temp & 0x0f) - ((flagC == 0) ? 1 : 0)) & 0xff;
		if (zl > 15)
			zl -= 6;
		zh = (regA >> 4) - (temp >> 4) - (zl&0x10);
		if (zh > 15)
			zh -= 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $6a1a	sta $08
	mem[0x8] = regA;
	// $6a1c	lda $0269,y
	regA = mem[0x269+regY];
	// $6a1f	sbc $0284,x
	temp = mem[0x284+regX];
	if (flagD == 0) {   // no decimal mode
		z = regA - temp - ((flagC == 0) ? 1 : 0);
		zl = z & 0xff;
		zh = z >> 8;
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		regA = zl;
	}
	else {   // decimal mode
		zl = ((regA & 0x0f) - (temp & 0x0f) - ((flagC == 0) ? 1 : 0)) & 0xff;
		if (zl > 15)
			zl -= 6;
		zh = (regA >> 4) - (temp >> 4) - (zl&0x10);
		if (zh > 15)
			zh -= 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
	};
	// $6a22	lsr a
	temp = regA;
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	// $6a23	ror $08
	temp = mem[0x8] | (flagC << 8);
	temp >>= 1;
	mem[0x8] = (temp & 0xff);
	// $6a25	asl a
	temp = (regA << 1);
	flagC = temp >> 8;
	regA = (temp & 0xff);
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6a26	beq $6a34
	if (flagZ == 1) {
		cycles += 45;
		regPC = 0x6a34;
		break;
	};
	// $6a28	bpl $6a97
	if (flagN == 0) {
		cycles += 47;
		regPC = 0x6a97;
		break;
	};
	// $6a2a	eor #$fe
	regA = (regA ^ 0xfe) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6a2c	bne $6a97
	if (flagZ == 0) {
		cycles += 51;
		regPC = 0x6a97;
		break;
	};
	// $6a2e	lda $08
	regA = mem[0x8];
	// $6a30	eor #$ff
	regA = (regA ^ 0xff) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6a32	sta $08
	mem[0x8] = regA;
	cycles += 59;
	regPC = 0x6a34;

case 0x6a34:
	// $6a34	lda $02d2,y
	regA = mem[0x2d2+regY];
	// $6a37	sec
	flagC = 1;
	// $6a38	sbc $02ed,x
	temp = mem[0x2ed+regX];
	if (flagD == 0) {   // no decimal mode
		z = regA - temp - ((flagC == 0) ? 1 : 0);
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh == 0) ? 1 : 0;
		regA = zl;
	}
	else {   // decimal mode
		zl = ((regA & 0x0f) - (temp & 0x0f) - ((flagC == 0) ? 1 : 0)) & 0xff;
		if (zl > 15)
			zl -= 6;
		zh = (regA >> 4) - (temp >> 4) - (zl&0x10);
		if (zh > 15)
			zh -= 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $6a3b	sta $09
	mem[0x9] = regA;
	// $6a3d	lda $028c,y
	regA = mem[0x28c+regY];
	// $6a40	sbc $02a7,x
	temp = mem[0x2a7+regX];
	if (flagD == 0) {   // no decimal mode
		z = regA - temp - ((flagC == 0) ? 1 : 0);
		zl = z & 0xff;
		zh = z >> 8;
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		regA = zl;
	}
	else {   // decimal mode
		zl = ((regA & 0x0f) - (temp & 0x0f) - ((flagC == 0) ? 1 : 0)) & 0xff;
		if (zl > 15)
			zl -= 6;
		zh = (regA >> 4) - (temp >> 4) - (zl&0x10);
		if (zh > 15)
			zh -= 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
	};
	// $6a43	lsr a
	temp = regA;
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	// $6a44	ror $09
	temp = mem[0x9] | (flagC << 8);
	temp >>= 1;
	mem[0x9] = (temp & 0xff);
	// $6a46	asl a
	temp = (regA << 1);
	flagC = temp >> 8;
	regA = (temp & 0xff);
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6a47	beq $6a55
	if (flagZ == 1) {
		cycles += 34;
		regPC = 0x6a55;
		break;
	};
	// $6a49	bpl $6a97
	if (flagN == 0) {
		cycles += 36;
		regPC = 0x6a97;
		break;
	};
	// $6a4b	eor #$fe
	regA = (regA ^ 0xfe) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6a4d	bne $6a97
	if (flagZ == 0) {
		cycles += 40;
		regPC = 0x6a97;
		break;
	};
	// $6a4f	lda $09
	regA = mem[0x9];
	// $6a51	eor #$ff
	regA = (regA ^ 0xff) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6a53	sta $09
	mem[0x9] = regA;
	cycles += 48;
	regPC = 0x6a55;

case 0x6a55:
	// $6a55	lda #$2a
	regA = 0x2a;
	// $6a57	lsr $0b
	temp = mem[0xb];
	flagC = temp & 1;
	temp >>= 1;
	mem[0xb] = (temp & 0xff);
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = ((temp & 0xff) >> 7) & 1;
	// $6a59	bcs $6a63
	if (flagC == 1) {
		cycles += 9;
		regPC = 0x6a63;
		break;
	};
	// $6a5b	lda #$48
	regA = 0x48;
	// $6a5d	lsr $0b
	temp = mem[0xb];
	flagC = temp & 1;
	temp >>= 1;
	mem[0xb] = (temp & 0xff);
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = ((temp & 0xff) >> 7) & 1;
	// $6a5f	bcs $6a63
	if (flagC == 1) {
		cycles += 18;
		regPC = 0x6a63;
		break;
	};
	// $6a61	lda #$84
	regA = 0x84;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 20;
	regPC = 0x6a63;

case 0x6a63:
	// $6a63	cpx #$01
	temp = regX - 0x1;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6a65	bcs $6a69
	if (flagC == 1) {
		cycles += 4;
		regPC = 0x6a69;
		break;
	};
	// $6a67	adc #$1c
	temp = 0x1c;
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		flagZ = (zl == 0 ? 1 : 0);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		flagN = (zl >> 7) & 1;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	cycles += 6;
	regPC = 0x6a69;

case 0x6a69:
	// $6a69	bne $6a77
	if (flagZ == 0) {
		cycles += 2;
		regPC = 0x6a77;
		break;
	};
	// $6a6b	adc #$12
	temp = 0x12;
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $6a6d	ldx $021c
	regX = mem[0x21c];
	// $6a70	dex
	regX--;
	regX&= 0xff;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $6a71	beq $6a75
	if (flagZ == 1) {
		cycles += 12;
		regPC = 0x6a75;
		break;
	};
	// $6a73	adc #$12
	temp = 0x12;
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		flagZ = (zl == 0 ? 1 : 0);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		flagN = (zl >> 7) & 1;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	cycles += 14;
	regPC = 0x6a75;

case 0x6a75:
	// $6a75	ldx #$01
	regX = 0x1;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	cycles += 2;
	regPC = 0x6a77;

case 0x6a77:
	// $6a77	cmp $08
	temp = regA - mem[0x8];
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6a79	bcc $6a97
	if (flagC == 0) {
		cycles += 5;
		regPC = 0x6a97;
		break;
	};
	// $6a7b	cmp $09
	temp = regA - mem[0x9];
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6a7d	bcc $6a97
	if (flagC == 0) {
		cycles += 10;
		regPC = 0x6a97;
		break;
	};
	// $6a7f	sta $0b
	mem[0xb] = regA;
	// $6a81	lsr a
	temp = regA;
	temp >>= 1;
	regA = (temp & 0xff);
	// $6a82	clc
	flagC = 0;
	// $6a83	adc $0b
	temp = mem[0xb];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $6a85	sta $0b
	mem[0xb] = regA;
	// $6a87	lda $09
	regA = mem[0x9];
	// $6a89	adc $08
	temp = mem[0x8];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		flagZ = (zl == 0 ? 1 : 0);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		flagN = (zl >> 7) & 1;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $6a8b	bcs $6a97
	if (flagC == 1) {
		cycles += 31;
		regPC = 0x6a97;
		break;
	};
	// $6a8d	cmp $0b
	temp = regA - mem[0xb];
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6a8f	bcs $6a97
	if (flagC == 1) {
		cycles += 36;
		regPC = 0x6a97;
		break;
	};
	// $6a91	jsr $6b0f
	mem[0x0100 | regS] = 0x6a;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x94;
	regS = (regS - 1) & 0xff;
	regPC = 0x6b0f;
	cycles += 42;
	break;

case 0x6a94:
	// $6a94	jmp $69f9
	regPC = 0x69f9;
	cycles += 3;
	break;

case 0x6a97:
	// $6a97	dey
	regY--;
	regY&= 0xff;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $6a98	bmi $6a94
	if (flagN == 1) {
		cycles += 4;
		regPC = 0x6a94;
		break;
	};
	// $6a9a	jmp $6a0a
	regPC = 0x6a0a;
	cycles += 7;
	break;

case 0x6a9d:
	// $6a9d	lda $0200,y
	regA = mem[0x200+regY];
	// $6aa0	and #$07
	regA = (regA & 0x7) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6aa2	sta $08
	mem[0x8] = regA;
	// $6aa4	jsr $77b5
	mem[0x0100 | regS] = 0x6a;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xa7;
	regS = (regS - 1) & 0xff;
	regPC = 0x77b5;
	cycles += 15;
	break;

case 0x6aa7:
	// $6aa7	and #$18
	regA = (regA & 0x18) & 0xff;
	// $6aa9	ora $08
	regA = (regA | mem[0x8]) & 0xff;
	// $6aab	sta $0200,x
	mem[0x200+regX] = regA;
	// $6aae	lda $02af,y
	regA = mem[0x2af+regY];
	// $6ab1	sta $02af,x
	mem[0x2af+regX] = regA;
	// $6ab4	lda $0269,y
	regA = mem[0x269+regY];
	// $6ab7	sta $0269,x
	mem[0x269+regX] = regA;
	// $6aba	lda $02d2,y
	regA = mem[0x2d2+regY];
	// $6abd	sta $02d2,x
	mem[0x2d2+regX] = regA;
	// $6ac0	lda $028c,y
	regA = mem[0x28c+regY];
	// $6ac3	sta $028c,x
	mem[0x28c+regX] = regA;
	// $6ac6	lda $0223,y
	regA = mem[0x223+regY];
	// $6ac9	sta $0223,x
	mem[0x223+regX] = regA;
	// $6acc	lda $0246,y
	regA = mem[0x246+regY];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6acf	sta $0246,x
	mem[0x246+regX] = regA;
	// $6ad2	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 70;
	break;

case 0x6ad3:
	// $6ad3	sta $0b
	mem[0xb] = regA;
	// $6ad5	stx $0c
	mem[0xc] = regX;
	cycles += 6;
	regPC = 0x6ad7;

case 0x6ad7:
	// $6ad7	ldy #$00
	regY = 0x0;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	cycles += 2;
	regPC = 0x6ad9;

case 0x6ad9:
	// $6ad9	iny
	regY++;
	regY&= 0xff;
	// $6ada	lda ($0b),y
	regA = mem[(mem[0xb] | (mem[0xc]<<8))+regY];
	// $6adc	eor $09
	regA = (regA ^ mem[0x9]) & 0xff;
	// $6ade	sta ($02),y
	mem[(mem[0x2] | (mem[0x3]<<8)) + regY] = regA;
	// $6ae0	dey
	regY--;
	regY&= 0xff;
	// $6ae1	cmp #$f0
	temp = regA - 0xf0;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6ae3	bcs $6b03
	if (flagC == 1) {
		cycles += 22;
		regPC = 0x6b03;
		break;
	};
	// $6ae5	cmp #$a0
	temp = regA - 0xa0;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6ae7	bcs $6aff
	if (flagC == 1) {
		cycles += 26;
		regPC = 0x6aff;
		break;
	};
	// $6ae9	lda ($0b),y
	regA = mem[(mem[0xb] | (mem[0xc]<<8))+regY];
	// $6aeb	sta ($02),y
	mem[(mem[0x2] | (mem[0x3]<<8)) + regY] = regA;
	// $6aed	iny
	regY++;
	regY&= 0xff;
	// $6aee	iny
	regY++;
	regY&= 0xff;
	// $6aef	lda ($0b),y
	regA = mem[(mem[0xb] | (mem[0xc]<<8))+regY];
	// $6af1	sta ($02),y
	mem[(mem[0x2] | (mem[0x3]<<8)) + regY] = regA;
	// $6af3	iny
	regY++;
	regY&= 0xff;
	// $6af4	lda ($0b),y
	regA = mem[(mem[0xb] | (mem[0xc]<<8))+regY];
	// $6af6	eor $08
	regA = (regA ^ mem[0x8]) & 0xff;
	// $6af8	adc $17
	temp = mem[0x17];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		flagZ = (zl == 0 ? 1 : 0);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		flagN = (zl >> 7) & 1;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $6afa	sta ($02),y
	mem[(mem[0x2] | (mem[0x3]<<8)) + regY] = regA;
	cycles += 71;
	regPC = 0x6afc;

case 0x6afc:
	// $6afc	iny
	regY++;
	regY&= 0xff;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $6afd	bne $6ad9
	if (flagZ == 0) {
		cycles += 4;
		regPC = 0x6ad9;
		break;
	};
	cycles += 4;
	regPC = 0x6aff;

case 0x6aff:
	// $6aff	dey
	regY--;
	regY&= 0xff;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $6b00	jmp $7c39
	regPC = 0x7c39;
	cycles += 5;
	break;

case 0x6b03:
	// $6b03	lda ($0b),y
	regA = mem[(mem[0xb] | (mem[0xc]<<8))+regY];
	// $6b05	eor $08
	regA = (regA ^ mem[0x8]) & 0xff;
	// $6b07	clc
	flagC = 0;
	// $6b08	adc $17
	temp = mem[0x17];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $6b0a	sta ($02),y
	mem[(mem[0x2] | (mem[0x3]<<8)) + regY] = regA;
	// $6b0c	iny
	regY++;
	regY&= 0xff;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $6b0d	bne $6afc
	if (flagZ == 0) {
		cycles += 23;
		regPC = 0x6afc;
		break;
	};
	cycles += 23;
	regPC = 0x6b0f;

case 0x6b0f:
	// $6b0f	cpx #$01
	temp = regX - 0x1;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6b11	bne $6b1b
	if (flagZ == 0) {
		cycles += 4;
		regPC = 0x6b1b;
		break;
	};
	// $6b13	cpy #$1b
	temp = regY - 0x1b;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6b15	bne $6b29
	if (flagZ == 0) {
		cycles += 8;
		regPC = 0x6b29;
		break;
	};
	// $6b17	ldx #$00
	regX = 0x0;
	// $6b19	ldy #$1c
	regY = 0x1c;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	cycles += 12;
	regPC = 0x6b1b;

case 0x6b1b:
	// $6b1b	txa
	regA = regX;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6b1c	bne $6b3c
	if (flagZ == 0) {
		cycles += 4;
		regPC = 0x6b3c;
		break;
	};
	// $6b1e	lda #$81
	regA = 0x81;
	// $6b20	sta $02fa
	mem[0x2fa] = regA;
	// $6b23	ldx $18
	regX = mem[0x18];
	// $6b25	dec $57,x
	temp = (mem[(0x57+regX)&0xff] - 1) & 0xff;
	mem[(0x57+regX)&0xff] = temp;
	// $6b27	ldx #$00
	regX = 0x0;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	cycles += 21;
	regPC = 0x6b29;

case 0x6b29:
	// $6b29	lda #$a0
	regA = 0xa0;
	// $6b2b	sta $021b,x
	mem[0x21b+regX] = regA;
	// $6b2e	lda #$00
	regA = 0x0;
	// $6b30	sta $023e,x
	mem[0x23e+regX] = regA;
	// $6b33	sta $0261,x
	mem[0x261+regX] = regA;
	// $6b36	cpy #$1b
	temp = regY - 0x1b;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6b38	bcc $6b47
	if (flagC == 0) {
		cycles += 23;
		regPC = 0x6b47;
		break;
	};
	// $6b3a	bcs $6b73
	if (flagC == 1) {
		cycles += 25;
		regPC = 0x6b73;
		break;
	};
	cycles += 25;
	regPC = 0x6b3c;

case 0x6b3c:
	// $6b3c	lda #$00
	regA = 0x0;
	// $6b3e	sta $021b,x
	mem[0x21b+regX] = regA;
	// $6b41	cpy #$1b
	temp = regY - 0x1b;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6b43	beq $6b66
	if (flagZ == 1) {
		cycles += 11;
		regPC = 0x6b66;
		break;
	};
	// $6b45	bcs $6b73
	if (flagC == 1) {
		cycles += 13;
		regPC = 0x6b73;
		break;
	};
	cycles += 13;
	regPC = 0x6b47;

case 0x6b47:
	// $6b47	jsr $75ec
	mem[0x0100 | regS] = 0x6b;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x4a;
	regS = (regS - 1) & 0xff;
	regPC = 0x75ec;
	cycles += 6;
	break;

case 0x6b4a:
	// $6b4a	lda $0200,y
	regA = mem[0x200+regY];
	// $6b4d	and #$03
	regA = (regA & 0x3) & 0xff;
	// $6b4f	eor #$02
	regA = (regA ^ 0x2) & 0xff;
	// $6b51	lsr a
	temp = regA;
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	// $6b52	ror a
	temp = regA | (flagC << 8);
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	// $6b53	ror a
	temp = regA | (flagC << 8);
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	// $6b54	ora #$3f
	regA = (regA | 0x3f) & 0xff;
	// $6b56	sta $69
	mem[0x69] = regA;
	// $6b58	lda #$a0
	regA = 0xa0;
	// $6b5a	sta $0200,y
	mem[0x200+regY] = regA;
	// $6b5d	lda #$00
	regA = 0x0;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6b5f	sta $0223,y
	mem[0x223+regY] = regA;
	// $6b62	sta $0246,y
	mem[0x246+regY] = regA;
	// $6b65	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 44;
	break;

case 0x6b66:
	// $6b66	txa
	regA = regX;
	// $6b67	ldx $18
	regX = mem[0x18];
	// $6b69	dec $57,x
	temp = (mem[(0x57+regX)&0xff] - 1) & 0xff;
	mem[(0x57+regX)&0xff] = temp;
	// $6b6b	tax
	regX = regA;
	// $6b6c	lda #$81
	regA = 0x81;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6b6e	sta $02fa
	mem[0x2fa] = regA;
	// $6b71	bne $6b4a
	if (flagZ == 0) {
		cycles += 21;
		regPC = 0x6b4a;
		break;
	};
	cycles += 21;
	regPC = 0x6b73;

case 0x6b73:
	// $6b73	lda $02f8
	regA = mem[0x2f8];
	// $6b76	sta $02f7
	mem[0x2f7] = regA;
	// $6b79	lda $1c
	regA = mem[0x1c];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6b7b	beq $6b4a
	if (flagZ == 1) {
		cycles += 13;
		regPC = 0x6b4a;
		break;
	};
	// $6b7d	stx $0d
	mem[0xd] = regX;
	// $6b7f	ldx $19
	regX = mem[0x19];
	// $6b81	lda $021c
	regA = mem[0x21c];
	// $6b84	lsr a
	temp = regA;
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	// $6b85	lda #$99
	regA = 0x99;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6b87	bcs $6b8b
	if (flagC == 1) {
		cycles += 29;
		regPC = 0x6b8b;
		break;
	};
	// $6b89	lda #$20
	regA = 0x20;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 31;
	regPC = 0x6b8b;

case 0x6b8b:
	// $6b8b	jsr $7397
	mem[0x0100 | regS] = 0x6b;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x8e;
	regS = (regS - 1) & 0xff;
	regPC = 0x7397;
	cycles += 6;
	break;

case 0x6b8e:
	// $6b8e	ldx $0d
	regX = mem[0xd];
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $6b90	jmp $6b4a
	regPC = 0x6b4a;
	cycles += 6;
	break;

case 0x6b93:
	// $6b93	lda $5c
	regA = mem[0x5c];
	// $6b95	and #$03
	regA = (regA & 0x3) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6b97	beq $6b9a
	if (flagZ == 1) {
		cycles += 7;
		regPC = 0x6b9a;
		break;
	};
	cycles += 7;
	regPC = 0x6b99;

case 0x6b99:
	// $6b99	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 6;
	break;

case 0x6b9a:
	// $6b9a	lda $021c
	regA = mem[0x21c];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6b9d	bmi $6b99
	if (flagN == 1) {
		cycles += 6;
		regPC = 0x6b99;
		break;
	};
	// $6b9f	beq $6ba4
	if (flagZ == 1) {
		cycles += 8;
		regPC = 0x6ba4;
		break;
	};
	// $6ba1	jmp $6c34
	regPC = 0x6c34;
	cycles += 11;
	break;

case 0x6ba4:
	// $6ba4	lda $1c
	regA = mem[0x1c];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6ba6	beq $6baf
	if (flagZ == 1) {
		cycles += 5;
		regPC = 0x6baf;
		break;
	};
	// $6ba8	lda $021b
	regA = mem[0x21b];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6bab	beq $6b99
	if (flagZ == 1) {
		cycles += 11;
		regPC = 0x6b99;
		break;
	};
	// $6bad	bmi $6b99
	if (flagN == 1) {
		cycles += 13;
		regPC = 0x6b99;
		break;
	};
	cycles += 13;
	regPC = 0x6baf;

case 0x6baf:
	// $6baf	lda $02f9
	regA = mem[0x2f9];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6bb2	beq $6bb7
	if (flagZ == 1) {
		cycles += 6;
		regPC = 0x6bb7;
		break;
	};
	// $6bb4	dec $02f9
	temp = (mem[0x2f9] - 1) & 0xff;
	mem[0x2f9] = temp;
	flagZ = (temp == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	cycles += 12;
	regPC = 0x6bb7;

case 0x6bb7:
	// $6bb7	dec $02f7
	temp = (mem[0x2f7] - 1) & 0xff;
	mem[0x2f7] = temp;
	flagZ = (temp == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6bba	bne $6b99
	if (flagZ == 0) {
		cycles += 8;
		regPC = 0x6b99;
		break;
	};
	// $6bbc	lda #$12
	regA = 0x12;
	// $6bbe	sta $02f7
	mem[0x2f7] = regA;
	// $6bc1	lda $02f9
	regA = mem[0x2f9];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6bc4	beq $6bd0
	if (flagZ == 1) {
		cycles += 20;
		regPC = 0x6bd0;
		break;
	};
	// $6bc6	lda $02f6
	regA = mem[0x2f6];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6bc9	beq $6b99
	if (flagZ == 1) {
		cycles += 26;
		regPC = 0x6b99;
		break;
	};
	// $6bcb	cmp $02fd
	temp = regA - mem[0x2fd];
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6bce	bcs $6b99
	if (flagC == 1) {
		cycles += 32;
		regPC = 0x6b99;
		break;
	};
	cycles += 32;
	regPC = 0x6bd0;

case 0x6bd0:
	// $6bd0	lda $02f8
	regA = mem[0x2f8];
	// $6bd3	sec
	flagC = 1;
	// $6bd4	sbc #$06
	temp = 0x6;
	if (flagD == 0) {   // no decimal mode
		z = regA - temp - ((flagC == 0) ? 1 : 0);
		zl = z & 0xff;
		zh = z >> 8;
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		regA = zl;
	}
	else {   // decimal mode
		zl = ((regA & 0x0f) - (temp & 0x0f) - ((flagC == 0) ? 1 : 0)) & 0xff;
		if (zl > 15)
			zl -= 6;
		zh = (regA >> 4) - (temp >> 4) - (zl&0x10);
		if (zh > 15)
			zh -= 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
	};
	// $6bd6	cmp #$20
	temp = regA - 0x20;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6bd8	bcc $6bdd
	if (flagC == 0) {
		cycles += 12;
		regPC = 0x6bdd;
		break;
	};
	// $6bda	sta $02f8
	mem[0x2f8] = regA;
	cycles += 16;
	regPC = 0x6bdd;

case 0x6bdd:
	// $6bdd	lda #$00
	regA = 0x0;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6bdf	sta $02cb
	mem[0x2cb] = regA;
	// $6be2	sta $0285
	mem[0x285] = regA;
	// $6be5	jsr $77b5
	mem[0x0100 | regS] = 0x6b;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xe8;
	regS = (regS - 1) & 0xff;
	regPC = 0x77b5;
	cycles += 16;
	break;

case 0x6be8:
	// $6be8	lsr a
	temp = regA;
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	// $6be9	ror $02ee
	temp = mem[0x2ee] | (flagC << 8);
	temp >>= 1;
	mem[0x2ee] = (temp & 0xff);
	// $6bec	lsr a
	temp = regA;
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	// $6bed	ror $02ee
	temp = mem[0x2ee] | (flagC << 8);
	temp >>= 1;
	mem[0x2ee] = (temp & 0xff);
	// $6bf0	lsr a
	temp = regA;
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	// $6bf1	ror $02ee
	temp = mem[0x2ee] | (flagC << 8);
	temp >>= 1;
	mem[0x2ee] = (temp & 0xff);
	// $6bf4	cmp #$18
	temp = regA - 0x18;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6bf6	bcc $6bfa
	if (flagC == 0) {
		cycles += 28;
		regPC = 0x6bfa;
		break;
	};
	// $6bf8	and #$17
	regA = (regA & 0x17) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 30;
	regPC = 0x6bfa;

case 0x6bfa:
	// $6bfa	sta $02a8
	mem[0x2a8] = regA;
	// $6bfd	ldx #$10
	regX = 0x10;
	// $6bff	bit $60
	temp =mem[0x60];
	flagN = (temp >> 7);
	flagV = ((temp >> 6) & 1);
	flagZ = ((temp & regA) != 0) ? 0 : 1;
	// $6c01	bvs $6c0f
	if (flagV == 1) {
		cycles += 11;
		regPC = 0x6c0f;
		break;
	};
	// $6c03	lda #$1f
	regA = 0x1f;
	// $6c05	sta $0285
	mem[0x285] = regA;
	// $6c08	lda #$ff
	regA = 0xff;
	// $6c0a	sta $02cb
	mem[0x2cb] = regA;
	// $6c0d	ldx #$f0
	regX = 0xf0;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	cycles += 25;
	regPC = 0x6c0f;

		};

	};

	void exec6c00 () {

		switch (regPC) {

case 0x6c0f:
	// $6c0f	stx $023f
	mem[0x23f] = regX;
	// $6c12	ldx #$02
	regX = 0x2;
	// $6c14	lda $02f8
	regA = mem[0x2f8];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6c17	bmi $6c30
	if (flagN == 1) {
		cycles += 12;
		regPC = 0x6c30;
		break;
	};
	// $6c19	ldy $19
	regY = mem[0x19];
	// $6c1b	lda $0053,y
	regA = mem[0x53+regY];
	// $6c1e	cmp #$30
	temp = regA - 0x30;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6c20	bcs $6c2f
	if (flagC == 1) {
		cycles += 23;
		regPC = 0x6c2f;
		break;
	};
	// $6c22	jsr $77b5
	mem[0x0100 | regS] = 0x6c;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x25;
	regS = (regS - 1) & 0xff;
	regPC = 0x77b5;
	cycles += 29;
	break;

case 0x6c25:
	// $6c25	sta $08
	mem[0x8] = regA;
	// $6c27	lda $02f8
	regA = mem[0x2f8];
	// $6c2a	lsr a
	temp = regA;
	temp >>= 1;
	regA = (temp & 0xff);
	// $6c2b	cmp $08
	temp = regA - mem[0x8];
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6c2d	bcs $6c30
	if (flagC == 1) {
		cycles += 14;
		regPC = 0x6c30;
		break;
	};
	cycles += 14;
	regPC = 0x6c2f;

case 0x6c2f:
	// $6c2f	dex
	regX--;
	regX&= 0xff;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	cycles += 2;
	regPC = 0x6c30;

case 0x6c30:
	// $6c30	stx $021c
	mem[0x21c] = regX;
	// $6c33	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 10;
	break;

case 0x6c34:
	// $6c34	lda $5c
	regA = mem[0x5c];
	// $6c36	asl a
	temp = (regA << 1);
	flagC = temp >> 8;
	regA = (temp & 0xff);
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6c37	bne $6c45
	if (flagZ == 0) {
		cycles += 7;
		regPC = 0x6c45;
		break;
	};
	// $6c39	jsr $77b5
	mem[0x0100 | regS] = 0x6c;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x3c;
	regS = (regS - 1) & 0xff;
	regPC = 0x77b5;
	cycles += 13;
	break;

case 0x6c3c:
	// $6c3c	and #$03
	regA = (regA & 0x3) & 0xff;
	// $6c3e	tax
	regX = regA;
	// $6c3f	lda $6cd3,x
	regA = mem[0x6cd3+regX];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6c42	sta $0262
	mem[0x262] = regA;
	cycles += 12;
	regPC = 0x6c45;

case 0x6c45:
	// $6c45	lda $1c
	regA = mem[0x1c];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6c47	beq $6c4e
	if (flagZ == 1) {
		cycles += 5;
		regPC = 0x6c4e;
		break;
	};
	// $6c49	lda $02fa
	regA = mem[0x2fa];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6c4c	bne $6c53
	if (flagZ == 0) {
		cycles += 11;
		regPC = 0x6c53;
		break;
	};
	cycles += 11;
	regPC = 0x6c4e;

case 0x6c4e:
	// $6c4e	dec $02f7
	temp = (mem[0x2f7] - 1) & 0xff;
	mem[0x2f7] = temp;
	flagZ = (temp == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6c51	beq $6c54
	if (flagZ == 1) {
		cycles += 8;
		regPC = 0x6c54;
		break;
	};
	cycles += 8;
	regPC = 0x6c53;

case 0x6c53:
	// $6c53	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 6;
	break;

case 0x6c54:
	// $6c54	lda #$0a
	regA = 0xa;
	// $6c56	sta $02f7
	mem[0x2f7] = regA;
	// $6c59	lda $021c
	regA = mem[0x21c];
	// $6c5c	lsr a
	temp = regA;
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6c5d	beq $6c65
	if (flagZ == 1) {
		cycles += 14;
		regPC = 0x6c65;
		break;
	};
	// $6c5f	jsr $77b5
	mem[0x0100 | regS] = 0x6c;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x62;
	regS = (regS - 1) & 0xff;
	regPC = 0x77b5;
	cycles += 20;
	break;

case 0x6c62:
	// $6c62	jmp $6cc4
	regPC = 0x6cc4;
	cycles += 3;
	break;

case 0x6c65:
	// $6c65	lda $023f
	regA = mem[0x23f];
	// $6c68	cmp #$80
	temp = regA - 0x80;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	// $6c6a	ror a
	temp = regA | (flagC << 8);
	temp >>= 1;
	regA = (temp & 0xff);
	// $6c6b	sta $0c
	mem[0xc] = regA;
	// $6c6d	lda $02ca
	regA = mem[0x2ca];
	// $6c70	sec
	flagC = 1;
	// $6c71	sbc $02cb
	temp = mem[0x2cb];
	if (flagD == 0) {   // no decimal mode
		z = regA - temp - ((flagC == 0) ? 1 : 0);
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh == 0) ? 1 : 0;
		regA = zl;
	}
	else {   // decimal mode
		zl = ((regA & 0x0f) - (temp & 0x0f) - ((flagC == 0) ? 1 : 0)) & 0xff;
		if (zl > 15)
			zl -= 6;
		zh = (regA >> 4) - (temp >> 4) - (zl&0x10);
		if (zh > 15)
			zh -= 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $6c74	sta $0b
	mem[0xb] = regA;
	// $6c76	lda $0284
	regA = mem[0x284];
	// $6c79	sbc $0285
	temp = mem[0x285];
	if (flagD == 0) {   // no decimal mode
		z = regA - temp - ((flagC == 0) ? 1 : 0);
		zl = z & 0xff;
		zh = z >> 8;
		regA = zl;
	}
	else {   // decimal mode
		zl = ((regA & 0x0f) - (temp & 0x0f) - ((flagC == 0) ? 1 : 0)) & 0xff;
		if (zl > 15)
			zl -= 6;
		zh = (regA >> 4) - (temp >> 4) - (zl&0x10);
		if (zh > 15)
			zh -= 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
	};
	// $6c7c	asl $0b
	temp = (mem[0xb] << 1);
	flagC = temp >> 8;
	mem[0xb] = (temp & 0xff);
	// $6c7e	rol a
	temp = (regA << 1)  | flagC;
	regA = (temp & 0xff);
	// $6c7f	asl $0b
	temp = (mem[0xb] << 1);
	flagC = temp >> 8;
	mem[0xb] = (temp & 0xff);
	// $6c81	rol a
	temp = (regA << 1)  | flagC;
	regA = (temp & 0xff);
	// $6c82	sec
	flagC = 1;
	// $6c83	sbc $0c
	temp = mem[0xc];
	if (flagD == 0) {   // no decimal mode
		z = regA - temp - ((flagC == 0) ? 1 : 0);
		zl = z & 0xff;
		zh = z >> 8;
		regA = zl;
	}
	else {   // decimal mode
		zl = ((regA & 0x0f) - (temp & 0x0f) - ((flagC == 0) ? 1 : 0)) & 0xff;
		if (zl > 15)
			zl -= 6;
		zh = (regA >> 4) - (temp >> 4) - (zl&0x10);
		if (zh > 15)
			zh -= 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
	};
	// $6c85	tax
	regX = regA;
	// $6c86	lda $0262
	regA = mem[0x262];
	// $6c89	cmp #$80
	temp = regA - 0x80;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	// $6c8b	ror a
	temp = regA | (flagC << 8);
	temp >>= 1;
	regA = (temp & 0xff);
	// $6c8c	sta $0c
	mem[0xc] = regA;
	// $6c8e	lda $02ed
	regA = mem[0x2ed];
	// $6c91	sec
	flagC = 1;
	// $6c92	sbc $02ee
	temp = mem[0x2ee];
	if (flagD == 0) {   // no decimal mode
		z = regA - temp - ((flagC == 0) ? 1 : 0);
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh == 0) ? 1 : 0;
		regA = zl;
	}
	else {   // decimal mode
		zl = ((regA & 0x0f) - (temp & 0x0f) - ((flagC == 0) ? 1 : 0)) & 0xff;
		if (zl > 15)
			zl -= 6;
		zh = (regA >> 4) - (temp >> 4) - (zl&0x10);
		if (zh > 15)
			zh -= 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $6c95	sta $0b
	mem[0xb] = regA;
	// $6c97	lda $02a7
	regA = mem[0x2a7];
	// $6c9a	sbc $02a8
	temp = mem[0x2a8];
	if (flagD == 0) {   // no decimal mode
		z = regA - temp - ((flagC == 0) ? 1 : 0);
		zl = z & 0xff;
		zh = z >> 8;
		regA = zl;
	}
	else {   // decimal mode
		zl = ((regA & 0x0f) - (temp & 0x0f) - ((flagC == 0) ? 1 : 0)) & 0xff;
		if (zl > 15)
			zl -= 6;
		zh = (regA >> 4) - (temp >> 4) - (zl&0x10);
		if (zh > 15)
			zh -= 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
	};
	// $6c9d	asl $0b
	temp = (mem[0xb] << 1);
	flagC = temp >> 8;
	mem[0xb] = (temp & 0xff);
	// $6c9f	rol a
	temp = (regA << 1)  | flagC;
	regA = (temp & 0xff);
	// $6ca0	asl $0b
	temp = (mem[0xb] << 1);
	flagC = temp >> 8;
	mem[0xb] = (temp & 0xff);
	// $6ca2	rol a
	temp = (regA << 1)  | flagC;
	regA = (temp & 0xff);
	// $6ca3	sec
	flagC = 1;
	// $6ca4	sbc $0c
	temp = mem[0xc];
	if (flagD == 0) {   // no decimal mode
		z = regA - temp - ((flagC == 0) ? 1 : 0);
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh == 0) ? 1 : 0;
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		regA = zl;
	}
	else {   // decimal mode
		zl = ((regA & 0x0f) - (temp & 0x0f) - ((flagC == 0) ? 1 : 0)) & 0xff;
		if (zl > 15)
			zl -= 6;
		zh = (regA >> 4) - (temp >> 4) - (zl&0x10);
		if (zh > 15)
			zh -= 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $6ca6	tay
	regY = regA;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $6ca7	jsr $76f0
	mem[0x0100 | regS] = 0x6c;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xaa;
	regS = (regS - 1) & 0xff;
	regPC = 0x76f0;
	cycles += 112;
	break;

case 0x6caa:
	// $6caa	sta $62
	mem[0x62] = regA;
	// $6cac	jsr $77b5
	mem[0x0100 | regS] = 0x6c;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xaf;
	regS = (regS - 1) & 0xff;
	regPC = 0x77b5;
	cycles += 9;
	break;

case 0x6caf:
	// $6caf	ldx $19
	regX = mem[0x19];
	// $6cb1	ldy $53,x
	regY = mem[(0x53+regX)&0xff];
	// $6cb3	cpy #$35
	temp = regY - 0x35;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	// $6cb5	ldx #$00
	regX = 0x0;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $6cb7	bcc $6cba
	if (flagC == 0) {
		cycles += 13;
		regPC = 0x6cba;
		break;
	};
	// $6cb9	inx
	regX++;
	regX&= 0xff;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	cycles += 15;
	regPC = 0x6cba;

case 0x6cba:
	// $6cba	and $6ccf,x
	regA = (regA & mem[0x6ccf+regX]) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6cbd	bpl $6cc2
	if (flagN == 0) {
		cycles += 7;
		regPC = 0x6cc2;
		break;
	};
	// $6cbf	ora $6cd1,x
	regA = (regA | mem[0x6cd1+regX]) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 12;
	regPC = 0x6cc2;

case 0x6cc2:
	// $6cc2	adc $62
	temp = mem[0x62];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		flagZ = (zl == 0 ? 1 : 0);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		flagN = (zl >> 7) & 1;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	cycles += 3;
	regPC = 0x6cc4;

case 0x6cc4:
	// $6cc4	sta $62
	mem[0x62] = regA;
	// $6cc6	ldy #$03
	regY = 0x3;
	// $6cc8	ldx #$01
	regX = 0x1;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $6cca	stx $0e
	mem[0xe] = regX;
	// $6ccc	jmp $6cf2
	regPC = 0x6cf2;
	cycles += 13;
	break;

case 0x6ccf:
	// dc.b $8f		; $6ccf
	// dc.b $87		; $6cd0
	// dc.b $70		; $6cd1
	// dc.b $78		; $6cd2
	// dc.b $f0		; $6cd3
	// dc.b $00		; $6cd4
	// dc.b $00		; $6cd5
	// dc.b $10		; $6cd6

case 0x6cd7:
	// $6cd7	lda $1c
	regA = mem[0x1c];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6cd9	beq $6cfc
	if (flagZ == 1) {
		cycles += 5;
		regPC = 0x6cfc;
		break;
	};
	// $6cdb	asl $2004
	temp = (readMem(0x2004) << 1);
	flagC = temp >> 8;
	writeMem(0x2004, (temp & 0xff));
	// $6cde	ror $63
	temp = mem[0x63] | (flagC << 8);
	flagC = temp & 1;
	temp >>= 1;
	mem[0x63] = (temp & 0xff);
	// $6ce0	bit $63
	temp =mem[0x63];
	flagN = (temp >> 7);
	flagV = ((temp >> 6) & 1);
	flagZ = ((temp & regA) != 0) ? 0 : 1;
	// $6ce2	bpl $6cfc
	if (flagN == 0) {
		cycles += 21;
		regPC = 0x6cfc;
		break;
	};
	// $6ce4	bvs $6cfc
	if (flagV == 1) {
		cycles += 23;
		regPC = 0x6cfc;
		break;
	};
	// $6ce6	lda $02fa
	regA = mem[0x2fa];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6ce9	bne $6cfc
	if (flagZ == 0) {
		cycles += 29;
		regPC = 0x6cfc;
		break;
	};
	// $6ceb	tax
	regX = regA;
	// $6cec	lda #$03
	regA = 0x3;
	// $6cee	sta $0e
	mem[0xe] = regA;
	// $6cf0	ldy #$07
	regY = 0x7;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	cycles += 38;
	regPC = 0x6cf2;

case 0x6cf2:
	// $6cf2	lda $021b,y
	regA = mem[0x21b+regY];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6cf5	beq $6cfd
	if (flagZ == 1) {
		cycles += 6;
		regPC = 0x6cfd;
		break;
	};
	// $6cf7	dey
	regY--;
	regY&= 0xff;
	// $6cf8	cpy $0e
	temp = regY - mem[0xe];
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6cfa	bne $6cf2
	if (flagZ == 0) {
		cycles += 13;
		regPC = 0x6cf2;
		break;
	};
	cycles += 13;
	regPC = 0x6cfc;

case 0x6cfc:
	// $6cfc	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 6;
	break;

case 0x6cfd:
	// $6cfd	stx $0d
	mem[0xd] = regX;
	// $6cff	lda #$12
	regA = 0x12;
	// $6d01	sta $021b,y
	mem[0x21b+regY] = regA;
	// $6d04	lda $61,x
	regA = mem[(0x61+regX)&0xff];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6d06	jsr $77d2
	mem[0x0100 | regS] = 0x6d;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x9;
	regS = (regS - 1) & 0xff;
	regPC = 0x77d2;
	cycles += 20;
	break;

case 0x6d09:
	// $6d09	ldx $0d
	regX = mem[0xd];
	// $6d0b	cmp #$80
	temp = regA - 0x80;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	// $6d0d	ror a
	temp = regA | (flagC << 8);
	temp >>= 1;
	regA = (temp & 0xff);
	// $6d0e	sta $09
	mem[0x9] = regA;
	// $6d10	clc
	flagC = 0;
	// $6d11	adc $023e,x
	temp = mem[0x23e+regX];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		flagZ = (zl == 0 ? 1 : 0);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		flagN = (zl >> 7) & 1;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $6d14	bmi $6d1e
	if (flagN == 1) {
		cycles += 19;
		regPC = 0x6d1e;
		break;
	};
	// $6d16	cmp #$70
	temp = regA - 0x70;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6d18	bcc $6d24
	if (flagC == 0) {
		cycles += 23;
		regPC = 0x6d24;
		break;
	};
	// $6d1a	lda #$6f
	regA = 0x6f;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6d1c	bne $6d24
	if (flagZ == 0) {
		cycles += 27;
		regPC = 0x6d24;
		break;
	};
	cycles += 27;
	regPC = 0x6d1e;

case 0x6d1e:
	// $6d1e	cmp #$91
	temp = regA - 0x91;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6d20	bcs $6d24
	if (flagC == 1) {
		cycles += 4;
		regPC = 0x6d24;
		break;
	};
	// $6d22	lda #$91
	regA = 0x91;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 6;
	regPC = 0x6d24;

case 0x6d24:
	// $6d24	sta $023e,y
	mem[0x23e+regY] = regA;
	// $6d27	lda $61,x
	regA = mem[(0x61+regX)&0xff];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6d29	jsr $77d5
	mem[0x0100 | regS] = 0x6d;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x2c;
	regS = (regS - 1) & 0xff;
	regPC = 0x77d5;
	cycles += 15;
	break;

case 0x6d2c:
	// $6d2c	ldx $0d
	regX = mem[0xd];
	// $6d2e	cmp #$80
	temp = regA - 0x80;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	// $6d30	ror a
	temp = regA | (flagC << 8);
	temp >>= 1;
	regA = (temp & 0xff);
	// $6d31	sta $0c
	mem[0xc] = regA;
	// $6d33	clc
	flagC = 0;
	// $6d34	adc $0261,x
	temp = mem[0x261+regX];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		flagZ = (zl == 0 ? 1 : 0);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		flagN = (zl >> 7) & 1;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $6d37	bmi $6d41
	if (flagN == 1) {
		cycles += 19;
		regPC = 0x6d41;
		break;
	};
	// $6d39	cmp #$70
	temp = regA - 0x70;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6d3b	bcc $6d47
	if (flagC == 0) {
		cycles += 23;
		regPC = 0x6d47;
		break;
	};
	// $6d3d	lda #$6f
	regA = 0x6f;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6d3f	bne $6d47
	if (flagZ == 0) {
		cycles += 27;
		regPC = 0x6d47;
		break;
	};
	cycles += 27;
	regPC = 0x6d41;

case 0x6d41:
	// $6d41	cmp #$91
	temp = regA - 0x91;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6d43	bcs $6d47
	if (flagC == 1) {
		cycles += 4;
		regPC = 0x6d47;
		break;
	};
	// $6d45	lda #$91
	regA = 0x91;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 6;
	regPC = 0x6d47;

case 0x6d47:
	// $6d47	sta $0261,y
	mem[0x261+regY] = regA;
	// $6d4a	ldx #$00
	regX = 0x0;
	// $6d4c	lda $09
	regA = mem[0x9];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6d4e	bpl $6d51
	if (flagN == 0) {
		cycles += 12;
		regPC = 0x6d51;
		break;
	};
	// $6d50	dex
	regX--;
	regX&= 0xff;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	cycles += 14;
	regPC = 0x6d51;

case 0x6d51:
	// $6d51	stx $08
	mem[0x8] = regX;
	// $6d53	ldx $0d
	regX = mem[0xd];
	// $6d55	cmp #$80
	temp = regA - 0x80;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	// $6d57	ror a
	temp = regA | (flagC << 8);
	temp >>= 1;
	regA = (temp & 0xff);
	// $6d58	clc
	flagC = 0;
	// $6d59	adc $09
	temp = mem[0x9];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
	};
	// $6d5b	clc
	flagC = 0;
	// $6d5c	adc $02ca,x
	temp = mem[0x2ca+regX];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $6d5f	sta $02ca,y
	mem[0x2ca+regY] = regA;
	// $6d62	lda $08
	regA = mem[0x8];
	// $6d64	adc $0284,x
	temp = mem[0x284+regX];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $6d67	sta $0284,y
	mem[0x284+regY] = regA;
	// $6d6a	ldx #$00
	regX = 0x0;
	// $6d6c	lda $0c
	regA = mem[0xc];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6d6e	bpl $6d71
	if (flagN == 0) {
		cycles += 47;
		regPC = 0x6d71;
		break;
	};
	// $6d70	dex
	regX--;
	regX&= 0xff;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	cycles += 49;
	regPC = 0x6d71;

case 0x6d71:
	// $6d71	stx $0b
	mem[0xb] = regX;
	// $6d73	ldx $0d
	regX = mem[0xd];
	// $6d75	cmp #$80
	temp = regA - 0x80;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	// $6d77	ror a
	temp = regA | (flagC << 8);
	temp >>= 1;
	regA = (temp & 0xff);
	// $6d78	clc
	flagC = 0;
	// $6d79	adc $0c
	temp = mem[0xc];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
	};
	// $6d7b	clc
	flagC = 0;
	// $6d7c	adc $02ed,x
	temp = mem[0x2ed+regX];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $6d7f	sta $02ed,y
	mem[0x2ed+regY] = regA;
	// $6d82	lda $0b
	regA = mem[0xb];
	// $6d84	adc $02a7,x
	temp = mem[0x2a7+regX];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $6d87	sta $02a7,y
	mem[0x2a7+regY] = regA;
	// $6d8a	lda #$80
	regA = 0x80;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6d8c	sta $66,x
	mem[(0x66+regX)&0xff] = regA;
	// $6d8e	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 52;
	break;

case 0x6d8f:
	// dc.b $d8		; $6d8f

case 0x6d90:
	// $6d90	lda $32
	regA = mem[0x32];
	// $6d92	and $33
	regA = (regA & mem[0x33]) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6d94	bpl $6d97
	if (flagN == 0) {
		cycles += 8;
		regPC = 0x6d97;
		break;
	};
	// $6d96	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 14;
	break;

case 0x6d97:
	// $6d97	lda $1a
	regA = mem[0x1a];
	// $6d99	lsr a
	temp = regA;
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6d9a	beq $6db4
	if (flagZ == 1) {
		cycles += 7;
		regPC = 0x6db4;
		break;
	};
	// $6d9c	ldy #$01
	regY = 0x1;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $6d9e	jsr $77f6
	mem[0x0100 | regS] = 0x6d;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xa1;
	regS = (regS - 1) & 0xff;
	regPC = 0x77f6;
	cycles += 15;
	break;

case 0x6da1:
	// $6da1	ldy #$02
	regY = 0x2;
	// $6da3	ldx $33
	regX = mem[0x33];
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $6da5	bpl $6da8
	if (flagN == 0) {
		cycles += 7;
		regPC = 0x6da8;
		break;
	};
	// $6da7	dey
	regY--;
	regY&= 0xff;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	cycles += 9;
	regPC = 0x6da8;

case 0x6da8:
	// $6da8	sty $18
	mem[0x18] = regY;
	// $6daa	lda $5c
	regA = mem[0x5c];
	// $6dac	and #$10
	regA = (regA & 0x10) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6dae	bne $6db4
	if (flagZ == 0) {
		cycles += 10;
		regPC = 0x6db4;
		break;
	};
	// $6db0	tya
	regA = regY;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6db1	jsr $7bd1
	mem[0x0100 | regS] = 0x6d;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xb4;
	regS = (regS - 1) & 0xff;
	regPC = 0x7bd1;
	cycles += 18;
	break;

case 0x6db4:
	// $6db4	lsr $18
	temp = mem[0x18];
	flagC = temp & 1;
	temp >>= 1;
	mem[0x18] = (temp & 0xff);
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = ((temp & 0xff) >> 7) & 1;
	// $6db6	jsr $73b2
	mem[0x0100 | regS] = 0x6d;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xb9;
	regS = (regS - 1) & 0xff;
	regPC = 0x73b2;
	cycles += 11;
	break;

case 0x6db9:
	// $6db9	ldy #$02
	regY = 0x2;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $6dbb	jsr $77f6
	mem[0x0100 | regS] = 0x6d;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xbe;
	regS = (regS - 1) & 0xff;
	regPC = 0x77f6;
	cycles += 8;
	break;

case 0x6dbe:
	// $6dbe	ldy #$03
	regY = 0x3;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $6dc0	jsr $77f6
	mem[0x0100 | regS] = 0x6d;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xc3;
	regS = (regS - 1) & 0xff;
	regPC = 0x77f6;
	cycles += 8;
	break;

case 0x6dc3:
	// $6dc3	ldy #$04
	regY = 0x4;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $6dc5	jsr $77f6
	mem[0x0100 | regS] = 0x6d;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xc8;
	regS = (regS - 1) & 0xff;
	regPC = 0x77f6;
	cycles += 8;
	break;

case 0x6dc8:
	// $6dc8	ldy #$05
	regY = 0x5;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $6dca	jsr $77f6
	mem[0x0100 | regS] = 0x6d;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xcd;
	regS = (regS - 1) & 0xff;
	regPC = 0x77f6;
	cycles += 8;
	break;

case 0x6dcd:
	// $6dcd	lda #$20
	regA = 0x20;
	// $6dcf	sta $00
	mem[0x0] = regA;
	// $6dd1	lda #$64
	regA = 0x64;
	// $6dd3	ldx #$39
	regX = 0x39;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $6dd5	jsr $7c03
	mem[0x0100 | regS] = 0x6d;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xd8;
	regS = (regS - 1) & 0xff;
	regPC = 0x7c03;
	cycles += 15;
	break;

case 0x6dd8:
	// $6dd8	lda #$70
	regA = 0x70;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6dda	jsr $7cde
	mem[0x0100 | regS] = 0x6d;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xdd;
	regS = (regS - 1) & 0xff;
	regPC = 0x7cde;
	cycles += 8;
	break;

case 0x6ddd:
	// $6ddd	ldx $18
	regX = mem[0x18];
	// $6ddf	ldy $32,x
	regY = mem[(0x32+regX)&0xff];
	// $6de1	sty $0b
	mem[0xb] = regY;
	// $6de3	tya
	regA = regY;
	// $6de4	clc
	flagC = 0;
	// $6de5	adc $31
	temp = mem[0x31];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		flagZ = (zl == 0 ? 1 : 0);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		flagN = (zl >> 7) & 1;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $6de7	sta $0c
	mem[0xc] = regA;
	// $6de9	jsr $6f1a
	mem[0x0100 | regS] = 0x6d;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xec;
	regS = (regS - 1) & 0xff;
	regPC = 0x6f1a;
	cycles += 26;
	break;

case 0x6dec:
	// $6dec	ldy $0b
	regY = mem[0xb];
	// $6dee	iny
	regY++;
	regY&= 0xff;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $6def	jsr $6f1a
	mem[0x0100 | regS] = 0x6d;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xf2;
	regS = (regS - 1) & 0xff;
	regPC = 0x6f1a;
	cycles += 11;
	break;

case 0x6df2:
	// $6df2	ldy $0b
	regY = mem[0xb];
	// $6df4	iny
	regY++;
	regY&= 0xff;
	// $6df5	iny
	regY++;
	regY&= 0xff;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $6df6	jsr $6f1a
	mem[0x0100 | regS] = 0x6d;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xf9;
	regS = (regS - 1) & 0xff;
	regPC = 0x6f1a;
	cycles += 13;
	break;

case 0x6df9:
	// $6df9	lda $2003
	regA = readMem(0x2003);
	// $6dfc	rol a
	temp = (regA << 1)  | flagC;
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $6dfd	rol $63
	temp = (mem[0x63] << 1)  | flagC;
	mem[0x63] = (temp & 0xff);
	// $6dff	lda $63
	regA = mem[0x63];
	// $6e01	and #$1f
	regA = (regA & 0x1f) & 0xff;
	// $6e03	cmp #$07
	temp = regA - 0x7;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6e05	bne $6e2e
	if (flagZ == 0) {
		cycles += 20;
		regPC = 0x6e2e;
		break;
	};
	// $6e07	inc $31
	temp = (mem[0x31] + 1) & 0xff;
	mem[0x31] = temp;
	// $6e09	lda $31
	regA = mem[0x31];
	// $6e0b	cmp #$03
	temp = regA - 0x3;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6e0d	bcc $6e22
	if (flagC == 0) {
		cycles += 32;
		regPC = 0x6e22;
		break;
	};
	// $6e0f	ldx $18
	regX = mem[0x18];
	// $6e11	lda #$ff
	regA = 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6e13	sta $32,x
	mem[(0x32+regX)&0xff] = regA;
	cycles += 41;
	regPC = 0x6e15;

case 0x6e15:
	// $6e15	ldx #$00
	regX = 0x0;
	// $6e17	stx $18
	mem[0x18] = regX;
	// $6e19	stx $31
	mem[0x31] = regX;
	// $6e1b	ldx #$f0
	regX = 0xf0;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $6e1d	stx $5d
	mem[0x5d] = regX;
	// $6e1f	jmp $73b2
	regPC = 0x73b2;
	cycles += 16;
	break;

case 0x6e22:
	// $6e22	inc $0c
	temp = (mem[0xc] + 1) & 0xff;
	mem[0xc] = temp;
	// $6e24	ldx $0c
	regX = mem[0xc];
	// $6e26	lda #$f4
	regA = 0xf4;
	// $6e28	sta $5d
	mem[0x5d] = regA;
	// $6e2a	lda #$0b
	regA = 0xb;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6e2c	sta $34,x
	mem[(0x34+regX)&0xff] = regA;
	cycles += 19;
	regPC = 0x6e2e;

case 0x6e2e:
	// $6e2e	lda $5d
	regA = mem[0x5d];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6e30	bne $6e3a
	if (flagZ == 0) {
		cycles += 5;
		regPC = 0x6e3a;
		break;
	};
	// $6e32	lda #$ff
	regA = 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6e34	sta $32
	mem[0x32] = regA;
	// $6e36	sta $33
	mem[0x33] = regA;
	// $6e38	bmi $6e15
	if (flagN == 1) {
		cycles += 15;
		regPC = 0x6e15;
		break;
	};
	cycles += 15;
	regPC = 0x6e3a;

case 0x6e3a:
	// $6e3a	lda $5c
	regA = mem[0x5c];
	// $6e3c	and #$07
	regA = (regA & 0x7) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6e3e	bne $6e71
	if (flagZ == 0) {
		cycles += 7;
		regPC = 0x6e71;
		break;
	};
	// $6e40	lda $2407
	regA = readMem(0x2407);
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6e43	bpl $6e49
	if (flagN == 0) {
		cycles += 13;
		regPC = 0x6e49;
		break;
	};
	// $6e45	lda #$01
	regA = 0x1;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6e47	bne $6e50
	if (flagZ == 0) {
		cycles += 17;
		regPC = 0x6e50;
		break;
	};
	cycles += 17;
	regPC = 0x6e49;

case 0x6e49:
	// $6e49	lda $2406
	regA = readMem(0x2406);
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6e4c	bpl $6e71
	if (flagN == 0) {
		cycles += 6;
		regPC = 0x6e71;
		break;
	};
	// $6e4e	lda #$ff
	regA = 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 8;
	regPC = 0x6e50;

case 0x6e50:
	// $6e50	ldx $0c
	regX = mem[0xc];
	// $6e52	clc
	flagC = 0;
	// $6e53	adc $34,x
	temp = mem[(0x34+regX)&0xff];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		flagZ = (zl == 0 ? 1 : 0);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		flagN = (zl >> 7) & 1;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $6e55	bmi $6e67
	if (flagN == 1) {
		cycles += 11;
		regPC = 0x6e67;
		break;
	};
	// $6e57	cmp #$0b
	temp = regA - 0xb;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6e59	bcs $6e69
	if (flagC == 1) {
		cycles += 15;
		regPC = 0x6e69;
		break;
	};
	// $6e5b	cmp #$01
	temp = regA - 0x1;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6e5d	beq $6e63
	if (flagZ == 1) {
		cycles += 19;
		regPC = 0x6e63;
		break;
	};
	// $6e5f	lda #$00
	regA = 0x0;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6e61	beq $6e6f
	if (flagZ == 1) {
		cycles += 23;
		regPC = 0x6e6f;
		break;
	};
	cycles += 23;
	regPC = 0x6e63;

case 0x6e63:
	// $6e63	lda #$0b
	regA = 0xb;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6e65	bne $6e6f
	if (flagZ == 0) {
		cycles += 4;
		regPC = 0x6e6f;
		break;
	};
	cycles += 4;
	regPC = 0x6e67;

case 0x6e67:
	// $6e67	lda #$24
	regA = 0x24;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 2;
	regPC = 0x6e69;

case 0x6e69:
	// $6e69	cmp #$25
	temp = regA - 0x25;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6e6b	bcc $6e6f
	if (flagC == 0) {
		cycles += 4;
		regPC = 0x6e6f;
		break;
	};
	// $6e6d	lda #$00
	regA = 0x0;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 6;
	regPC = 0x6e6f;

case 0x6e6f:
	// $6e6f	sta $34,x
	mem[(0x34+regX)&0xff] = regA;
	cycles += 4;
	regPC = 0x6e71;

case 0x6e71:
	// $6e71	lda #$00
	regA = 0x0;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6e73	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 8;
	break;

case 0x6e74:
	// $6e74	lda $1c
	regA = mem[0x1c];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6e76	beq $6ed7
	if (flagZ == 1) {
		cycles += 5;
		regPC = 0x6ed7;
		break;
	};
	// $6e78	lda $021b
	regA = mem[0x21b];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6e7b	bmi $6ed7
	if (flagN == 1) {
		cycles += 11;
		regPC = 0x6ed7;
		break;
	};
	// $6e7d	lda $02fa
	regA = mem[0x2fa];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6e80	bne $6ed7
	if (flagZ == 0) {
		cycles += 17;
		regPC = 0x6ed7;
		break;
	};
	// $6e82	lda $2003
	regA = readMem(0x2003);
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6e85	bpl $6ed7
	if (flagN == 0) {
		cycles += 23;
		regPC = 0x6ed7;
		break;
	};
	// $6e87	lda #$00
	regA = 0x0;
	// $6e89	sta $021b
	mem[0x21b] = regA;
	// $6e8c	sta $023e
	mem[0x23e] = regA;
	// $6e8f	sta $0261
	mem[0x261] = regA;
	// $6e92	lda #$30
	regA = 0x30;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6e94	sta $02fa
	mem[0x2fa] = regA;
	// $6e97	jsr $77b5
	mem[0x0100 | regS] = 0x6e;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x9a;
	regS = (regS - 1) & 0xff;
	regPC = 0x77b5;
	cycles += 49;
	break;

case 0x6e9a:
	// $6e9a	and #$1f
	regA = (regA & 0x1f) & 0xff;
	// $6e9c	cmp #$1d
	temp = regA - 0x1d;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6e9e	bcc $6ea2
	if (flagC == 0) {
		cycles += 6;
		regPC = 0x6ea2;
		break;
	};
	// $6ea0	lda #$1c
	regA = 0x1c;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 8;
	regPC = 0x6ea2;

case 0x6ea2:
	// $6ea2	cmp #$03
	temp = regA - 0x3;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6ea4	bcs $6ea8
	if (flagC == 1) {
		cycles += 4;
		regPC = 0x6ea8;
		break;
	};
	// $6ea6	lda #$03
	regA = 0x3;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 6;
	regPC = 0x6ea8;

case 0x6ea8:
	// $6ea8	sta $0284
	mem[0x284] = regA;
	// $6eab	ldx #$05
	regX = 0x5;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	cycles += 6;
	regPC = 0x6ead;

case 0x6ead:
	// $6ead	jsr $77b5
	mem[0x0100 | regS] = 0x6e;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xb0;
	regS = (regS - 1) & 0xff;
	regPC = 0x77b5;
	cycles += 6;
	break;

case 0x6eb0:
	// $6eb0	dex
	regX--;
	regX&= 0xff;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $6eb1	bne $6ead
	if (flagZ == 0) {
		cycles += 4;
		regPC = 0x6ead;
		break;
	};
	// $6eb3	and #$1f
	regA = (regA & 0x1f) & 0xff;
	// $6eb5	inx
	regX++;
	regX&= 0xff;
	// $6eb6	cmp #$18
	temp = regA - 0x18;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6eb8	bcc $6ec6
	if (flagC == 0) {
		cycles += 12;
		regPC = 0x6ec6;
		break;
	};
	// $6eba	and #$07
	regA = (regA & 0x7) & 0xff;
	// $6ebc	asl a
	temp = (regA << 1);
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $6ebd	adc #$04
	temp = 0x4;
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
	};
	// $6ebf	cmp $02f6
	temp = regA - mem[0x2f6];
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6ec2	bcc $6ec6
	if (flagC == 0) {
		cycles += 24;
		regPC = 0x6ec6;
		break;
	};
	// $6ec4	ldx #$80
	regX = 0x80;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	cycles += 26;
	regPC = 0x6ec6;

case 0x6ec6:
	// $6ec6	cmp #$15
	temp = regA - 0x15;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6ec8	bcc $6ecc
	if (flagC == 0) {
		cycles += 4;
		regPC = 0x6ecc;
		break;
	};
	// $6eca	lda #$14
	regA = 0x14;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 6;
	regPC = 0x6ecc;

case 0x6ecc:
	// $6ecc	cmp #$03
	temp = regA - 0x3;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6ece	bcs $6ed2
	if (flagC == 1) {
		cycles += 4;
		regPC = 0x6ed2;
		break;
	};
	// $6ed0	lda #$03
	regA = 0x3;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 6;
	regPC = 0x6ed2;

case 0x6ed2:
	// $6ed2	sta $02a7
	mem[0x2a7] = regA;
	// $6ed5	stx $59
	mem[0x59] = regX;
	cycles += 7;
	regPC = 0x6ed7;

case 0x6ed7:
	// $6ed7	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 6;
	break;

case 0x6ed8:
	// $6ed8	lda #$02
	regA = 0x2;
	// $6eda	sta $02f5
	mem[0x2f5] = regA;
	// $6edd	ldx #$03
	regX = 0x3;
	// $6edf	lsr $2802
	temp = readMem(0x2802);
	flagC = temp & 1;
	temp >>= 1;
	writeMem(0x2802, (temp & 0xff));
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = ((temp & 0xff) >> 7) & 1;
	// $6ee2	bcs $6ee5
	if (flagC == 1) {
		cycles += 16;
		regPC = 0x6ee5;
		break;
	};
	// $6ee4	inx
	regX++;
	regX&= 0xff;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	cycles += 18;
	regPC = 0x6ee5;

case 0x6ee5:
	// $6ee5	stx $56
	mem[0x56] = regX;
	// $6ee7	lda #$00
	regA = 0x0;
	// $6ee9	ldx #$04
	regX = 0x4;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	cycles += 7;
	regPC = 0x6eeb;

case 0x6eeb:
	// $6eeb	sta $021b,x
	mem[0x21b+regX] = regA;
	// $6eee	sta $021f,x
	mem[0x21f+regX] = regA;
	// $6ef1	sta $51,x
	mem[(0x51+regX)&0xff] = regA;
	// $6ef3	dex
	regX--;
	regX&= 0xff;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $6ef4	bpl $6eeb
	if (flagN == 0) {
		cycles += 18;
		regPC = 0x6eeb;
		break;
	};
	// $6ef6	sta $02f6
	mem[0x2f6] = regA;
	// $6ef9	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 28;
	break;

case 0x6efa:
	// $6efa	lda #$00
	regA = 0x0;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6efc	sta $3600
	writeMem(0x3600, regA);
	// $6eff	sta $3a00
	writeMem(0x3a00, regA);
	// $6f02	sta $3c00
	writeMem(0x3c00, regA);
	// $6f05	sta $3c01
	writeMem(0x3c01, regA);
	// $6f08	sta $3c03
	writeMem(0x3c03, regA);
	// $6f0b	sta $3c04
	writeMem(0x3c04, regA);
	// $6f0e	sta $3c05
	writeMem(0x3c05, regA);
	// $6f11	sta $69
	mem[0x69] = regA;
	// $6f13	sta $66
	mem[0x66] = regA;
	// $6f15	sta $67
	mem[0x67] = regA;
	// $6f17	sta $68
	mem[0x68] = regA;
	// $6f19	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 48;
	break;

case 0x6f1a:
	// $6f1a	lda $0034,y
	regA = mem[0x34+regY];
	// $6f1d	asl a
	temp = (regA << 1);
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $6f1e	tay
	regY = regA;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $6f1f	bne $6f35
	if (flagZ == 0) {
		cycles += 10;
		regPC = 0x6f35;
		break;
	};
	// $6f21	lda $32
	regA = mem[0x32];
	// $6f23	and $33
	regA = (regA & mem[0x33]) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6f25	bmi $6f35
	if (flagN == 1) {
		cycles += 18;
		regPC = 0x6f35;
		break;
	};
	// $6f27	lda #$72
	regA = 0x72;
	// $6f29	ldx #$f8
	regX = 0xf8;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $6f2b	jsr $7d45
	mem[0x0100 | regS] = 0x6f;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x2e;
	regS = (regS - 1) & 0xff;
	regPC = 0x7d45;
	cycles += 28;
	break;

case 0x6f2e:
	// $6f2e	lda #$01
	regA = 0x1;
	// $6f30	ldx #$f8
	regX = 0xf8;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $6f32	jmp $7d45
	regPC = 0x7d45;
	cycles += 7;
	break;

case 0x6f35:
	// $6f35	ldx $56d5,y
	regX = mem[0x56d5+regY];
	// $6f38	lda $56d4,y
	regA = mem[0x56d4+regY];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6f3b	jmp $7d45
	regPC = 0x7d45;
	cycles += 11;
	break;

case 0x6f3e:
	// $6f3e	beq $6f56
	if (flagZ == 1) {
		cycles += 2;
		regPC = 0x6f56;
		break;
	};
	// $6f40	sty $08
	mem[0x8] = regY;
	// $6f42	ldx #$d5
	regX = 0xd5;
	// $6f44	ldy #$e0
	regY = 0xe0;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $6f46	sty $00
	mem[0x0] = regY;
	// $6f48	jsr $7c03
	mem[0x0100 | regS] = 0x6f;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x4b;
	regS = (regS - 1) & 0xff;
	regPC = 0x7c03;
	cycles += 18;
	break;

case 0x6f4b:
	// $6f4b	ldx #$da
	regX = 0xda;
	// $6f4d	lda #$54
	regA = 0x54;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6f4f	jsr $7bfc
	mem[0x0100 | regS] = 0x6f;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x52;
	regS = (regS - 1) & 0xff;
	regPC = 0x7bfc;
	cycles += 10;
	break;

case 0x6f52:
	// $6f52	dec $08
	temp = (mem[0x8] - 1) & 0xff;
	mem[0x8] = temp;
	flagZ = (temp == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6f54	bne $6f4b
	if (flagZ == 0) {
		cycles += 7;
		regPC = 0x6f4b;
		break;
	};
	cycles += 7;
	regPC = 0x6f56;

case 0x6f56:
	// $6f56	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 6;
	break;

case 0x6f57:
	// $6f57	ldx #$22
	regX = 0x22;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	cycles += 2;
	regPC = 0x6f59;

case 0x6f59:
	// $6f59	lda $0200,x
	regA = mem[0x200+regX];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6f5c	bne $6f62
	if (flagZ == 0) {
		cycles += 6;
		regPC = 0x6f62;
		break;
	};
	cycles += 6;
	regPC = 0x6f5e;

case 0x6f5e:
	// $6f5e	dex
	regX--;
	regX&= 0xff;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $6f5f	bpl $6f59
	if (flagN == 0) {
		cycles += 4;
		regPC = 0x6f59;
		break;
	};
	// $6f61	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 10;
	break;

case 0x6f62:
	// $6f62	bpl $6fc7
	if (flagN == 0) {
		cycles += 2;
		regPC = 0x6fc7;
		break;
	};
	// $6f64	jsr $7708
	mem[0x0100 | regS] = 0x6f;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x67;
	regS = (regS - 1) & 0xff;
	regPC = 0x7708;
	cycles += 8;
	break;

case 0x6f67:
	// $6f67	lsr a
	temp = regA;
	temp >>= 1;
	regA = (temp & 0xff);
	// $6f68	lsr a
	temp = regA;
	temp >>= 1;
	regA = (temp & 0xff);
	// $6f69	lsr a
	temp = regA;
	temp >>= 1;
	regA = (temp & 0xff);
	// $6f6a	lsr a
	temp = regA;
	temp >>= 1;
	regA = (temp & 0xff);
	// $6f6b	cpx #$1b
	temp = regX - 0x1b;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6f6d	bne $6f76
	if (flagZ == 0) {
		cycles += 12;
		regPC = 0x6f76;
		break;
	};
	// $6f6f	lda $5c
	regA = mem[0x5c];
	// $6f71	and #$01
	regA = (regA & 0x1) & 0xff;
	// $6f73	lsr a
	temp = regA;
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6f74	beq $6f77
	if (flagZ == 1) {
		cycles += 21;
		regPC = 0x6f77;
		break;
	};
	cycles += 21;
	regPC = 0x6f76;

case 0x6f76:
	// $6f76	sec
	flagC = 1;
	cycles += 2;
	regPC = 0x6f77;

case 0x6f77:
	// $6f77	adc $0200,x
	temp = mem[0x200+regX];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		flagZ = (zl == 0 ? 1 : 0);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		flagN = (zl >> 7) & 1;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $6f7a	bmi $6fa1
	if (flagN == 1) {
		cycles += 7;
		regPC = 0x6fa1;
		break;
	};
	// $6f7c	cpx #$1b
	temp = regX - 0x1b;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6f7e	beq $6f93
	if (flagZ == 1) {
		cycles += 11;
		regPC = 0x6f93;
		break;
	};
	// $6f80	bcs $6f99
	if (flagC == 1) {
		cycles += 13;
		regPC = 0x6f99;
		break;
	};
	// $6f82	dec $02f6
	temp = (mem[0x2f6] - 1) & 0xff;
	mem[0x2f6] = temp;
	flagZ = (temp == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6f85	bne $6f8c
	if (flagZ == 0) {
		cycles += 21;
		regPC = 0x6f8c;
		break;
	};
	// $6f87	ldy #$7f
	regY = 0x7f;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $6f89	sty $02fb
	mem[0x2fb] = regY;
	cycles += 27;
	regPC = 0x6f8c;

case 0x6f8c:
	// $6f8c	lda #$00
	regA = 0x0;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6f8e	sta $0200,x
	mem[0x200+regX] = regA;
	// $6f91	beq $6f5e
	if (flagZ == 1) {
		cycles += 9;
		regPC = 0x6f5e;
		break;
	};
	cycles += 9;
	regPC = 0x6f93;

case 0x6f93:
	// $6f93	jsr $71e8
	mem[0x0100 | regS] = 0x6f;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x96;
	regS = (regS - 1) & 0xff;
	regPC = 0x71e8;
	cycles += 6;
	break;

case 0x6f96:
	// $6f96	jmp $6f8c
	regPC = 0x6f8c;
	cycles += 3;
	break;

case 0x6f99:
	// $6f99	lda $02f8
	regA = mem[0x2f8];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6f9c	sta $02f7
	mem[0x2f7] = regA;
	// $6f9f	bne $6f8c
	if (flagZ == 0) {
		cycles += 10;
		regPC = 0x6f8c;
		break;
	};
	cycles += 10;
	regPC = 0x6fa1;

case 0x6fa1:
	// $6fa1	sta $0200,x
	mem[0x200+regX] = regA;
	// $6fa4	and #$f0
	regA = (regA & 0xf0) & 0xff;
	// $6fa6	clc
	flagC = 0;
	// $6fa7	adc #$10
	temp = 0x10;
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
	};
	// $6fa9	cpx #$1b
	temp = regX - 0x1b;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6fab	bne $6faf
	if (flagZ == 0) {
		cycles += 15;
		regPC = 0x6faf;
		break;
	};
	// $6fad	lda #$00
	regA = 0x0;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 17;
	regPC = 0x6faf;

case 0x6faf:
	// $6faf	tay
	regY = regA;
	// $6fb0	lda $02af,x
	regA = mem[0x2af+regX];
	// $6fb3	sta $04
	mem[0x4] = regA;
	// $6fb5	lda $0269,x
	regA = mem[0x269+regX];
	// $6fb8	sta $05
	mem[0x5] = regA;
	// $6fba	lda $02d2,x
	regA = mem[0x2d2+regX];
	// $6fbd	sta $06
	mem[0x6] = regA;
	// $6fbf	lda $028c,x
	regA = mem[0x28c+regX];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6fc2	sta $07
	mem[0x7] = regA;
	// $6fc4	jmp $7027
	regPC = 0x7027;
	cycles += 33;
	break;

case 0x6fc7:
	// $6fc7	clc
	flagC = 0;
	// $6fc8	ldy #$00
	regY = 0x0;
	// $6fca	lda $0223,x
	regA = mem[0x223+regX];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6fcd	bpl $6fd0
	if (flagN == 0) {
		cycles += 10;
		regPC = 0x6fd0;
		break;
	};
	// $6fcf	dey
	regY--;
	regY&= 0xff;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	cycles += 12;
	regPC = 0x6fd0;

case 0x6fd0:
	// $6fd0	adc $02af,x
	temp = mem[0x2af+regX];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $6fd3	sta $02af,x
	mem[0x2af+regX] = regA;
	// $6fd6	sta $04
	mem[0x4] = regA;
	// $6fd8	tya
	regA = regY;
	// $6fd9	adc $0269,x
	temp = mem[0x269+regX];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
	};
	// $6fdc	cmp #$20
	temp = regA - 0x20;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6fde	bcc $6fec
	if (flagC == 0) {
		cycles += 24;
		regPC = 0x6fec;
		break;
	};
	// $6fe0	and #$1f
	regA = (regA & 0x1f) & 0xff;
	// $6fe2	cpx #$1c
	temp = regX - 0x1c;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $6fe4	bne $6fec
	if (flagZ == 0) {
		cycles += 30;
		regPC = 0x6fec;
		break;
	};
	// $6fe6	jsr $702d
	mem[0x0100 | regS] = 0x6f;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xe9;
	regS = (regS - 1) & 0xff;
	regPC = 0x702d;
	cycles += 36;
	break;

case 0x6fe9:
	// $6fe9	jmp $6f5e
	regPC = 0x6f5e;
	cycles += 3;
	break;

case 0x6fec:
	// $6fec	sta $0269,x
	mem[0x269+regX] = regA;
	// $6fef	sta $05
	mem[0x5] = regA;
	// $6ff1	clc
	flagC = 0;
	// $6ff2	ldy #$00
	regY = 0x0;
	// $6ff4	lda $0246,x
	regA = mem[0x246+regX];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $6ff7	bpl $6ffb
	if (flagN == 0) {
		cycles += 18;
		regPC = 0x6ffb;
		break;
	};
	// $6ff9	ldy #$ff
	regY = 0xff;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	cycles += 20;
	regPC = 0x6ffb;

case 0x6ffb:
	// $6ffb	adc $02d2,x
	temp = mem[0x2d2+regX];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $6ffe	sta $02d2,x
	mem[0x2d2+regX] = regA;
	// $7001	sta $06
	mem[0x6] = regA;
	// $7003	tya
	regA = regY;
	// $7004	adc $028c,x
	temp = mem[0x28c+regX];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
	};
	// $7007	cmp #$18
	temp = regA - 0x18;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $7009	bcc $7013
	if (flagC == 0) {
		cycles += 24;
		regPC = 0x7013;
		break;
	};
	// $700b	beq $7011
	if (flagZ == 1) {
		cycles += 26;
		regPC = 0x7011;
		break;
	};
	// $700d	lda #$17
	regA = 0x17;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $700f	bne $7013
	if (flagZ == 0) {
		cycles += 30;
		regPC = 0x7013;
		break;
	};
	cycles += 30;
	regPC = 0x7011;

		};

	};

	void exec7000 () {

		switch (regPC) {

case 0x7011:
	// $7011	lda #$00
	regA = 0x0;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 2;
	regPC = 0x7013;

case 0x7013:
	// $7013	sta $028c,x
	mem[0x28c+regX] = regA;
	// $7016	sta $07
	mem[0x7] = regA;
	// $7018	lda $0200,x
	regA = mem[0x200+regX];
	// $701b	ldy #$e0
	regY = 0xe0;
	// $701d	lsr a
	temp = regA;
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $701e	bcs $7027
	if (flagC == 1) {
		cycles += 18;
		regPC = 0x7027;
		break;
	};
	// $7020	ldy #$f0
	regY = 0xf0;
	// $7022	lsr a
	temp = regA;
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7023	bcs $7027
	if (flagC == 1) {
		cycles += 24;
		regPC = 0x7027;
		break;
	};
	// $7025	ldy #$00
	regY = 0x0;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	cycles += 26;
	regPC = 0x7027;

case 0x7027:
	// $7027	jsr $72fe
	mem[0x0100 | regS] = 0x70;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x2a;
	regS = (regS - 1) & 0xff;
	regPC = 0x72fe;
	cycles += 6;
	break;

case 0x702a:
	// $702a	jmp $6f5e
	regPC = 0x6f5e;
	cycles += 3;
	break;

case 0x702d:
	// $702d	lda $02f8
	regA = mem[0x2f8];
	// $7030	sta $02f7
	mem[0x2f7] = regA;
	// $7033	lda #$00
	regA = 0x0;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7035	sta $021c
	mem[0x21c] = regA;
	// $7038	sta $023f
	mem[0x23f] = regA;
	// $703b	sta $0262
	mem[0x262] = regA;
	// $703e	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 28;
	break;

case 0x703f:
	// $703f	lda $1c
	regA = mem[0x1c];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7041	beq $7085
	if (flagZ == 1) {
		cycles += 5;
		regPC = 0x7085;
		break;
	};
	// $7043	lda $021b
	regA = mem[0x21b];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7046	bmi $7085
	if (flagN == 1) {
		cycles += 11;
		regPC = 0x7085;
		break;
	};
	// $7048	lda $02fa
	regA = mem[0x2fa];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $704b	beq $7086
	if (flagZ == 1) {
		cycles += 17;
		regPC = 0x7086;
		break;
	};
	// $704d	dec $02fa
	temp = (mem[0x2fa] - 1) & 0xff;
	mem[0x2fa] = temp;
	flagZ = (temp == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $7050	bne $7085
	if (flagZ == 0) {
		cycles += 25;
		regPC = 0x7085;
		break;
	};
	// $7052	ldy $59
	regY = mem[0x59];
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $7054	bmi $706f
	if (flagN == 1) {
		cycles += 30;
		regPC = 0x706f;
		break;
	};
	// $7056	bne $7068
	if (flagZ == 0) {
		cycles += 32;
		regPC = 0x7068;
		break;
	};
	// $7058	jsr $7139
	mem[0x0100 | regS] = 0x70;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x5b;
	regS = (regS - 1) & 0xff;
	regPC = 0x7139;
	cycles += 38;
	break;

case 0x705b:
	// $705b	bne $7081
	if (flagZ == 0) {
		cycles += 2;
		regPC = 0x7081;
		break;
	};
	// $705d	ldy $021c
	regY = mem[0x21c];
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $7060	beq $7068
	if (flagZ == 1) {
		cycles += 8;
		regPC = 0x7068;
		break;
	};
	// $7062	ldy #$02
	regY = 0x2;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $7064	sty $02fa
	mem[0x2fa] = regY;
	// $7067	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 20;
	break;

case 0x7068:
	// $7068	lda #$01
	regA = 0x1;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $706a	sta $021b
	mem[0x21b] = regA;
	// $706d	bne $7081
	if (flagZ == 0) {
		cycles += 8;
		regPC = 0x7081;
		break;
	};
	cycles += 8;
	regPC = 0x706f;

case 0x706f:
	// $706f	lda #$a0
	regA = 0xa0;
	// $7071	sta $021b
	mem[0x21b] = regA;
	// $7074	ldx #$3e
	regX = 0x3e;
	// $7076	stx $69
	mem[0x69] = regX;
	// $7078	ldx $18
	regX = mem[0x18];
	// $707a	dec $57,x
	temp = (mem[(0x57+regX)&0xff] - 1) & 0xff;
	mem[(0x57+regX)&0xff] = temp;
	// $707c	lda #$81
	regA = 0x81;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $707e	sta $02fa
	mem[0x2fa] = regA;
	cycles += 26;
	regPC = 0x7081;

case 0x7081:
	// $7081	lda #$00
	regA = 0x0;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7083	sta $59
	mem[0x59] = regA;
	cycles += 5;
	regPC = 0x7085;

case 0x7085:
	// $7085	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 6;
	break;

case 0x7086:
	// $7086	lda $2407
	regA = readMem(0x2407);
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7089	bpl $708f
	if (flagN == 0) {
		cycles += 6;
		regPC = 0x708f;
		break;
	};
	// $708b	lda #$03
	regA = 0x3;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $708d	bne $7096
	if (flagZ == 0) {
		cycles += 10;
		regPC = 0x7096;
		break;
	};
	cycles += 10;
	regPC = 0x708f;

case 0x708f:
	// $708f	lda $2406
	regA = readMem(0x2406);
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7092	bpl $709b
	if (flagN == 0) {
		cycles += 6;
		regPC = 0x709b;
		break;
	};
	// $7094	lda #$fd
	regA = 0xfd;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 8;
	regPC = 0x7096;

case 0x7096:
	// $7096	clc
	flagC = 0;
	// $7097	adc $61
	temp = mem[0x61];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		flagZ = (zl == 0 ? 1 : 0);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		flagN = (zl >> 7) & 1;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $7099	sta $61
	mem[0x61] = regA;
	cycles += 8;
	regPC = 0x709b;

case 0x709b:
	// $709b	lda $5c
	regA = mem[0x5c];
	// $709d	lsr a
	temp = regA;
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $709e	bcs $7085
	if (flagC == 1) {
		cycles += 7;
		regPC = 0x7085;
		break;
	};
	// $70a0	lda $2405
	regA = readMem(0x2405);
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $70a3	bpl $70e1
	if (flagN == 0) {
		cycles += 13;
		regPC = 0x70e1;
		break;
	};
	// $70a5	lda #$80
	regA = 0x80;
	// $70a7	sta $3c03
	writeMem(0x3c03, regA);
	// $70aa	ldy #$00
	regY = 0x0;
	// $70ac	lda $61
	regA = mem[0x61];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $70ae	jsr $77d2
	mem[0x0100 | regS] = 0x70;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xb1;
	regS = (regS - 1) & 0xff;
	regPC = 0x77d2;
	cycles += 30;
	break;

case 0x70b1:
	// $70b1	bpl $70b4
	if (flagN == 0) {
		cycles += 2;
		regPC = 0x70b4;
		break;
	};
	// $70b3	dey
	regY--;
	regY&= 0xff;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	cycles += 4;
	regPC = 0x70b4;

case 0x70b4:
	// $70b4	asl a
	temp = (regA << 1);
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $70b5	clc
	flagC = 0;
	// $70b6	adc $64
	temp = mem[0x64];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $70b8	tax
	regX = regA;
	// $70b9	tya
	regA = regY;
	// $70ba	adc $023e
	temp = mem[0x23e];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		flagZ = (zl == 0 ? 1 : 0);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		flagN = (zl >> 7) & 1;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $70bd	jsr $7125
	mem[0x0100 | regS] = 0x70;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xc0;
	regS = (regS - 1) & 0xff;
	regPC = 0x7125;
	cycles += 21;
	break;

case 0x70c0:
	// $70c0	sta $023e
	mem[0x23e] = regA;
	// $70c3	stx $64
	mem[0x64] = regX;
	// $70c5	ldy #$00
	regY = 0x0;
	// $70c7	lda $61
	regA = mem[0x61];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $70c9	jsr $77d5
	mem[0x0100 | regS] = 0x70;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xcc;
	regS = (regS - 1) & 0xff;
	regPC = 0x77d5;
	cycles += 18;
	break;

case 0x70cc:
	// $70cc	bpl $70cf
	if (flagN == 0) {
		cycles += 2;
		regPC = 0x70cf;
		break;
	};
	// $70ce	dey
	regY--;
	regY&= 0xff;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	cycles += 4;
	regPC = 0x70cf;

case 0x70cf:
	// $70cf	asl a
	temp = (regA << 1);
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $70d0	clc
	flagC = 0;
	// $70d1	adc $65
	temp = mem[0x65];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $70d3	tax
	regX = regA;
	// $70d4	tya
	regA = regY;
	// $70d5	adc $0261
	temp = mem[0x261];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		flagZ = (zl == 0 ? 1 : 0);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		flagN = (zl >> 7) & 1;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $70d8	jsr $7125
	mem[0x0100 | regS] = 0x70;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xdb;
	regS = (regS - 1) & 0xff;
	regPC = 0x7125;
	cycles += 21;
	break;

case 0x70db:
	// $70db	sta $0261
	mem[0x261] = regA;
	// $70de	stx $65
	mem[0x65] = regX;
	// $70e0	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 13;
	break;

case 0x70e1:
	// $70e1	lda #$00
	regA = 0x0;
	// $70e3	sta $3c03
	writeMem(0x3c03, regA);
	// $70e6	lda $023e
	regA = mem[0x23e];
	// $70e9	ora $64
	regA = (regA | mem[0x64]) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $70eb	beq $7105
	if (flagZ == 1) {
		cycles += 15;
		regPC = 0x7105;
		break;
	};
	// $70ed	lda $023e
	regA = mem[0x23e];
	// $70f0	asl a
	temp = (regA << 1);
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $70f1	ldx #$ff
	regX = 0xff;
	// $70f3	clc
	flagC = 0;
	// $70f4	eor #$ff
	regA = (regA ^ 0xff) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $70f6	bmi $70fa
	if (flagN == 1) {
		cycles += 29;
		regPC = 0x70fa;
		break;
	};
	// $70f8	inx
	regX++;
	regX&= 0xff;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $70f9	sec
	flagC = 1;
	cycles += 33;
	regPC = 0x70fa;

case 0x70fa:
	// $70fa	adc $64
	temp = mem[0x64];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $70fc	sta $64
	mem[0x64] = regA;
	// $70fe	txa
	regA = regX;
	// $70ff	adc $023e
	temp = mem[0x23e];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		flagZ = (zl == 0 ? 1 : 0);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		flagN = (zl >> 7) & 1;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $7102	sta $023e
	mem[0x23e] = regA;
	cycles += 16;
	regPC = 0x7105;

case 0x7105:
	// $7105	lda $65
	regA = mem[0x65];
	// $7107	ora $0261
	regA = (regA | mem[0x261]) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $710a	beq $7124
	if (flagZ == 1) {
		cycles += 9;
		regPC = 0x7124;
		break;
	};
	// $710c	lda $0261
	regA = mem[0x261];
	// $710f	asl a
	temp = (regA << 1);
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $7110	ldx #$ff
	regX = 0xff;
	// $7112	clc
	flagC = 0;
	// $7113	eor #$ff
	regA = (regA ^ 0xff) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7115	bmi $7119
	if (flagN == 1) {
		cycles += 23;
		regPC = 0x7119;
		break;
	};
	// $7117	sec
	flagC = 1;
	// $7118	inx
	regX++;
	regX&= 0xff;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	cycles += 27;
	regPC = 0x7119;

case 0x7119:
	// $7119	adc $65
	temp = mem[0x65];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $711b	sta $65
	mem[0x65] = regA;
	// $711d	txa
	regA = regX;
	// $711e	adc $0261
	temp = mem[0x261];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		flagZ = (zl == 0 ? 1 : 0);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		flagN = (zl >> 7) & 1;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $7121	sta $0261
	mem[0x261] = regA;
	cycles += 16;
	regPC = 0x7124;

case 0x7124:
	// $7124	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 6;
	break;

case 0x7125:
	// $7125	bmi $7130
	if (flagN == 1) {
		cycles += 2;
		regPC = 0x7130;
		break;
	};
	// $7127	cmp #$40
	temp = regA - 0x40;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $7129	bcc $7138
	if (flagC == 0) {
		cycles += 6;
		regPC = 0x7138;
		break;
	};
	// $712b	ldx #$ff
	regX = 0xff;
	// $712d	lda #$3f
	regA = 0x3f;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $712f	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 16;
	break;

case 0x7130:
	// $7130	cmp #$c0
	temp = regA - 0xc0;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $7132	bcs $7138
	if (flagC == 1) {
		cycles += 4;
		regPC = 0x7138;
		break;
	};
	// $7134	ldx #$01
	regX = 0x1;
	// $7136	lda #$c0
	regA = 0xc0;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 8;
	regPC = 0x7138;

case 0x7138:
	// $7138	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 6;
	break;

case 0x7139:
	// $7139	ldx #$1c
	regX = 0x1c;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	cycles += 2;
	regPC = 0x713b;

case 0x713b:
	// $713b	lda $0200,x
	regA = mem[0x200+regX];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $713e	beq $715e
	if (flagZ == 1) {
		cycles += 6;
		regPC = 0x715e;
		break;
	};
	// $7140	lda $0269,x
	regA = mem[0x269+regX];
	// $7143	sec
	flagC = 1;
	// $7144	sbc $0284
	temp = mem[0x284];
	if (flagD == 0) {   // no decimal mode
		z = regA - temp - ((flagC == 0) ? 1 : 0);
		zl = z & 0xff;
		zh = z >> 8;
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		regA = zl;
	}
	else {   // decimal mode
		zl = ((regA & 0x0f) - (temp & 0x0f) - ((flagC == 0) ? 1 : 0)) & 0xff;
		if (zl > 15)
			zl -= 6;
		zh = (regA >> 4) - (temp >> 4) - (zl&0x10);
		if (zh > 15)
			zh -= 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
	};
	// $7147	cmp #$04
	temp = regA - 0x4;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $7149	bcc $714f
	if (flagC == 0) {
		cycles += 20;
		regPC = 0x714f;
		break;
	};
	// $714b	cmp #$fc
	temp = regA - 0xfc;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $714d	bcc $715e
	if (flagC == 0) {
		cycles += 24;
		regPC = 0x715e;
		break;
	};
	cycles += 24;
	regPC = 0x714f;

case 0x714f:
	// $714f	lda $028c,x
	regA = mem[0x28c+regX];
	// $7152	sec
	flagC = 1;
	// $7153	sbc $02a7
	temp = mem[0x2a7];
	if (flagD == 0) {   // no decimal mode
		z = regA - temp - ((flagC == 0) ? 1 : 0);
		zl = z & 0xff;
		zh = z >> 8;
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		regA = zl;
	}
	else {   // decimal mode
		zl = ((regA & 0x0f) - (temp & 0x0f) - ((flagC == 0) ? 1 : 0)) & 0xff;
		if (zl > 15)
			zl -= 6;
		zh = (regA >> 4) - (temp >> 4) - (zl&0x10);
		if (zh > 15)
			zh -= 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
	};
	// $7156	cmp #$04
	temp = regA - 0x4;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $7158	bcc $7163
	if (flagC == 0) {
		cycles += 14;
		regPC = 0x7163;
		break;
	};
	// $715a	cmp #$fc
	temp = regA - 0xfc;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $715c	bcs $7163
	if (flagC == 1) {
		cycles += 18;
		regPC = 0x7163;
		break;
	};
	cycles += 18;
	regPC = 0x715e;

case 0x715e:
	// $715e	dex
	regX--;
	regX&= 0xff;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $715f	bpl $713b
	if (flagN == 0) {
		cycles += 4;
		regPC = 0x713b;
		break;
	};
	// $7161	inx
	regX++;
	regX&= 0xff;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7162	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 12;
	break;

case 0x7163:
	// $7163	inc $02fa
	temp = (mem[0x2fa] + 1) & 0xff;
	mem[0x2fa] = temp;
	flagZ = (temp == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $7166	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 12;
	break;

case 0x7167:
	// dc.b $90		; $7167

case 0x7168:
	// $7168	ldx #$1a
	regX = 0x1a;
	// $716a	lda $02fb
	regA = mem[0x2fb];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $716d	bne $71df
	if (flagZ == 0) {
		cycles += 8;
		regPC = 0x71df;
		break;
	};
	// $716f	lda $021c
	regA = mem[0x21c];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7172	bne $71e7
	if (flagZ == 0) {
		cycles += 14;
		regPC = 0x71e7;
		break;
	};
	// $7174	sta $023f
	mem[0x23f] = regA;
	// $7177	sta $0262
	mem[0x262] = regA;
	// $717a	inc $02fd
	temp = (mem[0x2fd] + 1) & 0xff;
	mem[0x2fd] = temp;
	// $717d	lda $02fd
	regA = mem[0x2fd];
	// $7180	cmp #$0b
	temp = regA - 0xb;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $7182	bcc $7187
	if (flagC == 0) {
		cycles += 36;
		regPC = 0x7187;
		break;
	};
	// $7184	dec $02fd
	temp = (mem[0x2fd] - 1) & 0xff;
	mem[0x2fd] = temp;
	flagZ = (temp == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	cycles += 42;
	regPC = 0x7187;

case 0x7187:
	// $7187	lda $02f5
	regA = mem[0x2f5];
	// $718a	clc
	flagC = 0;
	// $718b	adc #$02
	temp = 0x2;
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
	};
	// $718d	cmp #$0b
	temp = regA - 0xb;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $718f	bcc $7193
	if (flagC == 0) {
		cycles += 12;
		regPC = 0x7193;
		break;
	};
	// $7191	lda #$0b
	regA = 0xb;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 14;
	regPC = 0x7193;

case 0x7193:
	// $7193	sta $02f6
	mem[0x2f6] = regA;
	// $7196	sta $02f5
	mem[0x2f5] = regA;
	// $7199	sta $08
	mem[0x8] = regA;
	// $719b	ldy #$1c
	regY = 0x1c;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	cycles += 13;
	regPC = 0x719d;

case 0x719d:
	// $719d	jsr $77b5
	mem[0x0100 | regS] = 0x71;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xa0;
	regS = (regS - 1) & 0xff;
	regPC = 0x77b5;
	cycles += 6;
	break;

case 0x71a0:
	// $71a0	and #$18
	regA = (regA & 0x18) & 0xff;
	// $71a2	ora #$04
	regA = (regA | 0x4) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $71a4	sta $0200,x
	mem[0x200+regX] = regA;
	// $71a7	jsr $7203
	mem[0x0100 | regS] = 0x71;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xaa;
	regS = (regS - 1) & 0xff;
	regPC = 0x7203;
	cycles += 15;
	break;

case 0x71aa:
	// $71aa	jsr $77b5
	mem[0x0100 | regS] = 0x71;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xad;
	regS = (regS - 1) & 0xff;
	regPC = 0x77b5;
	cycles += 6;
	break;

case 0x71ad:
	// $71ad	lsr a
	temp = regA;
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	// $71ae	and #$1f
	regA = (regA & 0x1f) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $71b0	bcc $71c5
	if (flagC == 0) {
		cycles += 6;
		regPC = 0x71c5;
		break;
	};
	// $71b2	cmp #$18
	temp = regA - 0x18;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $71b4	bcc $71b8
	if (flagC == 0) {
		cycles += 10;
		regPC = 0x71b8;
		break;
	};
	// $71b6	and #$17
	regA = (regA & 0x17) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 12;
	regPC = 0x71b8;

case 0x71b8:
	// $71b8	sta $028c,x
	mem[0x28c+regX] = regA;
	// $71bb	lda #$00
	regA = 0x0;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $71bd	sta $0269,x
	mem[0x269+regX] = regA;
	// $71c0	sta $02af,x
	mem[0x2af+regX] = regA;
	// $71c3	beq $71d0
	if (flagZ == 1) {
		cycles += 19;
		regPC = 0x71d0;
		break;
	};
	cycles += 19;
	regPC = 0x71c5;

case 0x71c5:
	// $71c5	sta $0269,x
	mem[0x269+regX] = regA;
	// $71c8	lda #$00
	regA = 0x0;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $71ca	sta $028c,x
	mem[0x28c+regX] = regA;
	// $71cd	sta $02d2,x
	mem[0x2d2+regX] = regA;
	cycles += 17;
	regPC = 0x71d0;

case 0x71d0:
	// $71d0	dex
	regX--;
	regX&= 0xff;
	// $71d1	dec $08
	temp = (mem[0x8] - 1) & 0xff;
	mem[0x8] = temp;
	flagZ = (temp == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $71d3	bne $719d
	if (flagZ == 0) {
		cycles += 9;
		regPC = 0x719d;
		break;
	};
	// $71d5	lda #$7f
	regA = 0x7f;
	// $71d7	sta $02f7
	mem[0x2f7] = regA;
	// $71da	lda #$30
	regA = 0x30;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $71dc	sta $02fc
	mem[0x2fc] = regA;
	cycles += 21;
	regPC = 0x71df;

case 0x71df:
	// $71df	lda #$00
	regA = 0x0;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 2;
	regPC = 0x71e1;

case 0x71e1:
	// $71e1	sta $0200,x
	mem[0x200+regX] = regA;
	// $71e4	dex
	regX--;
	regX&= 0xff;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $71e5	bpl $71e1
	if (flagN == 0) {
		cycles += 9;
		regPC = 0x71e1;
		break;
	};
	cycles += 9;
	regPC = 0x71e7;

case 0x71e7:
	// $71e7	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 6;
	break;

case 0x71e8:
	// $71e8	lda #$60
	regA = 0x60;
	// $71ea	sta $02ca
	mem[0x2ca] = regA;
	// $71ed	sta $02ed
	mem[0x2ed] = regA;
	// $71f0	lda #$00
	regA = 0x0;
	// $71f2	sta $023e
	mem[0x23e] = regA;
	// $71f5	sta $0261
	mem[0x261] = regA;
	// $71f8	lda #$10
	regA = 0x10;
	// $71fa	sta $0284
	mem[0x284] = regA;
	// $71fd	lda #$0c
	regA = 0xc;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $71ff	sta $02a7
	mem[0x2a7] = regA;
	// $7202	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 38;
	break;

case 0x7203:
	// $7203	jsr $77b5
	mem[0x0100 | regS] = 0x72;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x6;
	regS = (regS - 1) & 0xff;
	regPC = 0x77b5;
	cycles += 6;
	break;

case 0x7206:
	// $7206	and #$8f
	regA = (regA & 0x8f) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7208	bpl $720c
	if (flagN == 0) {
		cycles += 4;
		regPC = 0x720c;
		break;
	};
	// $720a	ora #$f0
	regA = (regA | 0xf0) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 6;
	regPC = 0x720c;

case 0x720c:
	// $720c	clc
	flagC = 0;
	// $720d	adc $0223,y
	temp = mem[0x223+regY];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		flagZ = (zl == 0 ? 1 : 0);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		flagN = (zl >> 7) & 1;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $7210	jsr $7233
	mem[0x0100 | regS] = 0x72;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x13;
	regS = (regS - 1) & 0xff;
	regPC = 0x7233;
	cycles += 12;
	break;

case 0x7213:
	// $7213	sta $0223,x
	mem[0x223+regX] = regA;
	// $7216	jsr $77b5
	mem[0x0100 | regS] = 0x72;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x19;
	regS = (regS - 1) & 0xff;
	regPC = 0x77b5;
	cycles += 11;
	break;

case 0x7219:
	// $7219	jsr $77b5
	mem[0x0100 | regS] = 0x72;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x1c;
	regS = (regS - 1) & 0xff;
	regPC = 0x77b5;
	cycles += 6;
	break;

case 0x721c:
	// $721c	jsr $77b5
	mem[0x0100 | regS] = 0x72;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x1f;
	regS = (regS - 1) & 0xff;
	regPC = 0x77b5;
	cycles += 6;
	break;

case 0x721f:
	// $721f	jsr $77b5
	mem[0x0100 | regS] = 0x72;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x22;
	regS = (regS - 1) & 0xff;
	regPC = 0x77b5;
	cycles += 6;
	break;

case 0x7222:
	// $7222	and #$8f
	regA = (regA & 0x8f) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7224	bpl $7228
	if (flagN == 0) {
		cycles += 4;
		regPC = 0x7228;
		break;
	};
	// $7226	ora #$f0
	regA = (regA | 0xf0) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 6;
	regPC = 0x7228;

case 0x7228:
	// $7228	clc
	flagC = 0;
	// $7229	adc $0246,y
	temp = mem[0x246+regY];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		flagZ = (zl == 0 ? 1 : 0);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		flagN = (zl >> 7) & 1;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $722c	jsr $7233
	mem[0x0100 | regS] = 0x72;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x2f;
	regS = (regS - 1) & 0xff;
	regPC = 0x7233;
	cycles += 12;
	break;

case 0x722f:
	// $722f	sta $0246,x
	mem[0x246+regX] = regA;
	// $7232	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 11;
	break;

case 0x7233:
	// $7233	bpl $7242
	if (flagN == 0) {
		cycles += 2;
		regPC = 0x7242;
		break;
	};
	// $7235	cmp #$e1
	temp = regA - 0xe1;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $7237	bcs $723b
	if (flagC == 1) {
		cycles += 6;
		regPC = 0x723b;
		break;
	};
	// $7239	lda #$e1
	regA = 0xe1;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 8;
	regPC = 0x723b;

case 0x723b:
	// $723b	cmp #$fb
	temp = regA - 0xfb;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $723d	bcc $724e
	if (flagC == 0) {
		cycles += 4;
		regPC = 0x724e;
		break;
	};
	// $723f	lda #$fa
	regA = 0xfa;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7241	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 12;
	break;

case 0x7242:
	// $7242	cmp #$06
	temp = regA - 0x6;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $7244	bcs $7248
	if (flagC == 1) {
		cycles += 4;
		regPC = 0x7248;
		break;
	};
	// $7246	lda #$06
	regA = 0x6;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 6;
	regPC = 0x7248;

case 0x7248:
	// $7248	cmp #$20
	temp = regA - 0x20;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $724a	bcc $724e
	if (flagC == 0) {
		cycles += 4;
		regPC = 0x724e;
		break;
	};
	// $724c	lda #$1f
	regA = 0x1f;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 6;
	regPC = 0x724e;

case 0x724e:
	// $724e	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 6;
	break;

case 0x724f:
	// $724f	lda #$10
	regA = 0x10;
	// $7251	sta $00
	mem[0x0] = regA;
	// $7253	lda #$50
	regA = 0x50;
	// $7255	ldx #$a4
	regX = 0xa4;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7257	jsr $7bfc
	mem[0x0100 | regS] = 0x72;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x5a;
	regS = (regS - 1) & 0xff;
	regPC = 0x7bfc;
	cycles += 15;
	break;

case 0x725a:
	// $725a	lda #$19
	regA = 0x19;
	// $725c	ldx #$db
	regX = 0xdb;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $725e	jsr $7c03
	mem[0x0100 | regS] = 0x72;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x61;
	regS = (regS - 1) & 0xff;
	regPC = 0x7c03;
	cycles += 10;
	break;

case 0x7261:
	// $7261	lda #$70
	regA = 0x70;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7263	jsr $7cde
	mem[0x0100 | regS] = 0x72;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x66;
	regS = (regS - 1) & 0xff;
	regPC = 0x7cde;
	cycles += 8;
	break;

case 0x7266:
	// $7266	ldx #$00
	regX = 0x0;
	// $7268	lda $1c
	regA = mem[0x1c];
	// $726a	cmp #$02
	temp = regA - 0x2;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $726c	bne $7286
	if (flagZ == 0) {
		cycles += 9;
		regPC = 0x7286;
		break;
	};
	// $726e	lda $18
	regA = mem[0x18];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7270	bne $7286
	if (flagZ == 0) {
		cycles += 14;
		regPC = 0x7286;
		break;
	};
	// $7272	ldx #$20
	regX = 0x20;
	// $7274	lda $021b
	regA = mem[0x21b];
	// $7277	ora $59
	regA = (regA | mem[0x59]) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7279	bne $7286
	if (flagZ == 0) {
		cycles += 25;
		regPC = 0x7286;
		break;
	};
	// $727b	lda $02fa
	regA = mem[0x2fa];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $727e	bmi $7286
	if (flagN == 1) {
		cycles += 31;
		regPC = 0x7286;
		break;
	};
	// $7280	lda $5c
	regA = mem[0x5c];
	// $7282	and #$10
	regA = (regA & 0x10) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7284	beq $7293
	if (flagZ == 1) {
		cycles += 38;
		regPC = 0x7293;
		break;
	};
	cycles += 38;
	regPC = 0x7286;

case 0x7286:
	// $7286	lda #$52
	regA = 0x52;
	// $7288	ldy #$02
	regY = 0x2;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $728a	sec
	flagC = 1;
	// $728b	jsr $773f
	mem[0x0100 | regS] = 0x72;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x8e;
	regS = (regS - 1) & 0xff;
	regPC = 0x773f;
	cycles += 12;
	break;

case 0x728e:
	// $728e	lda #$00
	regA = 0x0;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7290	jsr $778b
	mem[0x0100 | regS] = 0x72;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x93;
	regS = (regS - 1) & 0xff;
	regPC = 0x778b;
	cycles += 8;
	break;

case 0x7293:
	// $7293	lda #$28
	regA = 0x28;
	// $7295	ldy $57
	regY = mem[0x57];
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $7297	jsr $6f3e
	mem[0x0100 | regS] = 0x72;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x9a;
	regS = (regS - 1) & 0xff;
	regPC = 0x6f3e;
	cycles += 11;
	break;

case 0x729a:
	// $729a	lda #$00
	regA = 0x0;
	// $729c	sta $00
	mem[0x0] = regA;
	// $729e	lda #$78
	regA = 0x78;
	// $72a0	ldx #$db
	regX = 0xdb;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $72a2	jsr $7c03
	mem[0x0100 | regS] = 0x72;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xa5;
	regS = (regS - 1) & 0xff;
	regPC = 0x7c03;
	cycles += 15;
	break;

case 0x72a5:
	// $72a5	lda #$50
	regA = 0x50;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $72a7	jsr $7cde
	mem[0x0100 | regS] = 0x72;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xaa;
	regS = (regS - 1) & 0xff;
	regPC = 0x7cde;
	cycles += 8;
	break;

case 0x72aa:
	// $72aa	lda #$1d
	regA = 0x1d;
	// $72ac	ldy #$02
	regY = 0x2;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $72ae	sec
	flagC = 1;
	// $72af	jsr $773f
	mem[0x0100 | regS] = 0x72;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xb2;
	regS = (regS - 1) & 0xff;
	regPC = 0x773f;
	cycles += 12;
	break;

case 0x72b2:
	// $72b2	lda #$00
	regA = 0x0;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $72b4	jsr $7bd1
	mem[0x0100 | regS] = 0x72;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xb7;
	regS = (regS - 1) & 0xff;
	regPC = 0x7bd1;
	cycles += 8;
	break;

case 0x72b7:
	// $72b7	lda #$10
	regA = 0x10;
	// $72b9	sta $00
	mem[0x0] = regA;
	// $72bb	lda #$c0
	regA = 0xc0;
	// $72bd	ldx #$db
	regX = 0xdb;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $72bf	jsr $7c03
	mem[0x0100 | regS] = 0x72;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xc2;
	regS = (regS - 1) & 0xff;
	regPC = 0x7c03;
	cycles += 15;
	break;

case 0x72c2:
	// $72c2	lda #$50
	regA = 0x50;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $72c4	jsr $7cde
	mem[0x0100 | regS] = 0x72;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xc7;
	regS = (regS - 1) & 0xff;
	regPC = 0x7cde;
	cycles += 8;
	break;

case 0x72c7:
	// $72c7	ldx #$00
	regX = 0x0;
	// $72c9	lda $1c
	regA = mem[0x1c];
	// $72cb	cmp #$01
	temp = regA - 0x1;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $72cd	beq $72fd
	if (flagZ == 1) {
		cycles += 9;
		regPC = 0x72fd;
		break;
	};
	// $72cf	bcc $72e9
	if (flagC == 0) {
		cycles += 11;
		regPC = 0x72e9;
		break;
	};
	// $72d1	lda $18
	regA = mem[0x18];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $72d3	beq $72e9
	if (flagZ == 1) {
		cycles += 16;
		regPC = 0x72e9;
		break;
	};
	// $72d5	ldx #$20
	regX = 0x20;
	// $72d7	lda $021b
	regA = mem[0x21b];
	// $72da	ora $59
	regA = (regA | mem[0x59]) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $72dc	bne $72e9
	if (flagZ == 0) {
		cycles += 27;
		regPC = 0x72e9;
		break;
	};
	// $72de	lda $02fa
	regA = mem[0x2fa];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $72e1	bmi $72e9
	if (flagN == 1) {
		cycles += 33;
		regPC = 0x72e9;
		break;
	};
	// $72e3	lda $5c
	regA = mem[0x5c];
	// $72e5	and #$10
	regA = (regA & 0x10) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $72e7	beq $72f6
	if (flagZ == 1) {
		cycles += 40;
		regPC = 0x72f6;
		break;
	};
	cycles += 40;
	regPC = 0x72e9;

case 0x72e9:
	// $72e9	lda #$54
	regA = 0x54;
	// $72eb	ldy #$02
	regY = 0x2;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $72ed	sec
	flagC = 1;
	// $72ee	jsr $773f
	mem[0x0100 | regS] = 0x72;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xf1;
	regS = (regS - 1) & 0xff;
	regPC = 0x773f;
	cycles += 12;
	break;

case 0x72f1:
	// $72f1	lda #$00
	regA = 0x0;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $72f3	jsr $778b
	mem[0x0100 | regS] = 0x72;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xf6;
	regS = (regS - 1) & 0xff;
	regPC = 0x778b;
	cycles += 8;
	break;

case 0x72f6:
	// $72f6	lda #$cf
	regA = 0xcf;
	// $72f8	ldy $58
	regY = mem[0x58];
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $72fa	jmp $6f3e
	regPC = 0x6f3e;
	cycles += 8;
	break;

case 0x72fd:
	// $72fd	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 6;
	break;

case 0x72fe:
	// $72fe	sty $00
	mem[0x0] = regY;
	// $7300	stx $0d
	mem[0xd] = regX;
	// $7302	lda $05
	regA = mem[0x5];
	// $7304	lsr a
	temp = regA;
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	// $7305	ror $04
	temp = mem[0x4] | (flagC << 8);
	temp >>= 1;
	mem[0x4] = (temp & 0xff);
	// $7307	lsr a
	temp = regA;
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	// $7308	ror $04
	temp = mem[0x4] | (flagC << 8);
	temp >>= 1;
	mem[0x4] = (temp & 0xff);
	// $730a	lsr a
	temp = regA;
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	// $730b	ror $04
	temp = mem[0x4] | (flagC << 8);
	temp >>= 1;
	mem[0x4] = (temp & 0xff);
	// $730d	sta $05
	mem[0x5] = regA;
	// $730f	lda $07
	regA = mem[0x7];
	// $7311	clc
	flagC = 0;
	// $7312	adc #$04
	temp = 0x4;
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
	};
	// $7314	lsr a
	temp = regA;
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	// $7315	ror $06
	temp = mem[0x6] | (flagC << 8);
	temp >>= 1;
	mem[0x6] = (temp & 0xff);
	// $7317	lsr a
	temp = regA;
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	// $7318	ror $06
	temp = mem[0x6] | (flagC << 8);
	temp >>= 1;
	mem[0x6] = (temp & 0xff);
	// $731a	lsr a
	temp = regA;
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	// $731b	ror $06
	temp = mem[0x6] | (flagC << 8);
	flagC = temp & 1;
	temp >>= 1;
	mem[0x6] = (temp & 0xff);
	// $731d	sta $07
	mem[0x7] = regA;
	// $731f	ldx #$04
	regX = 0x4;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7321	jsr $7c1c
	mem[0x0100 | regS] = 0x73;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x24;
	regS = (regS - 1) & 0xff;
	regPC = 0x7c1c;
	cycles += 72;
	break;

case 0x7324:
	// $7324	lda #$70
	regA = 0x70;
	// $7326	sec
	flagC = 1;
	// $7327	sbc $00
	temp = mem[0x0];
	if (flagD == 0) {   // no decimal mode
		z = regA - temp - ((flagC == 0) ? 1 : 0);
		zl = z & 0xff;
		zh = z >> 8;
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		regA = zl;
	}
	else {   // decimal mode
		zl = ((regA & 0x0f) - (temp & 0x0f) - ((flagC == 0) ? 1 : 0)) & 0xff;
		if (zl > 15)
			zl -= 6;
		zh = (regA >> 4) - (temp >> 4) - (zl&0x10);
		if (zh > 15)
			zh -= 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
	};
	// $7329	cmp #$a0
	temp = regA - 0xa0;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $732b	bcc $733b
	if (flagC == 0) {
		cycles += 11;
		regPC = 0x733b;
		break;
	};
	cycles += 11;
	regPC = 0x732d;

case 0x732d:
	// $732d	pha
	mem[0x0100 | regS] = regA;
	regS = (regS - 1) & 0xff;
	// $732e	lda #$90
	regA = 0x90;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7330	jsr $7cde
	mem[0x0100 | regS] = 0x73;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x33;
	regS = (regS - 1) & 0xff;
	regPC = 0x7cde;
	cycles += 11;
	break;

case 0x7333:
	// $7333	pla
	regS = (regS + 1) & 0xff;
	regA = mem[0x0100 | regS];
	// $7334	sec
	flagC = 1;
	// $7335	sbc #$10
	temp = 0x10;
	if (flagD == 0) {   // no decimal mode
		z = regA - temp - ((flagC == 0) ? 1 : 0);
		zl = z & 0xff;
		zh = z >> 8;
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		regA = zl;
	}
	else {   // decimal mode
		zl = ((regA & 0x0f) - (temp & 0x0f) - ((flagC == 0) ? 1 : 0)) & 0xff;
		if (zl > 15)
			zl -= 6;
		zh = (regA >> 4) - (temp >> 4) - (zl&0x10);
		if (zh > 15)
			zh -= 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
	};
	// $7337	cmp #$a0
	temp = regA - 0xa0;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $7339	bcs $732d
	if (flagC == 1) {
		cycles += 12;
		regPC = 0x732d;
		break;
	};
	cycles += 12;
	regPC = 0x733b;

case 0x733b:
	// $733b	jsr $7cde
	mem[0x0100 | regS] = 0x73;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x3e;
	regS = (regS - 1) & 0xff;
	regPC = 0x7cde;
	cycles += 6;
	break;

case 0x733e:
	// $733e	ldx $0d
	regX = mem[0xd];
	// $7340	lda $0200,x
	regA = mem[0x200+regX];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7343	bpl $735b
	if (flagN == 0) {
		cycles += 9;
		regPC = 0x735b;
		break;
	};
	// $7345	cpx #$1b
	temp = regX - 0x1b;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $7347	beq $7355
	if (flagZ == 1) {
		cycles += 13;
		regPC = 0x7355;
		break;
	};
	// $7349	and #$0c
	regA = (regA & 0xc) & 0xff;
	// $734b	lsr a
	temp = regA;
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	// $734c	tay
	regY = regA;
	// $734d	lda $50f8,y
	regA = mem[0x50f8+regY];
	// $7350	ldx $50f9,y
	regX = mem[0x50f9+regY];
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7353	bne $7370
	if (flagZ == 0) {
		cycles += 29;
		regPC = 0x7370;
		break;
	};
	cycles += 29;
	regPC = 0x7355;

case 0x7355:
	// $7355	jsr $7465
	mem[0x0100 | regS] = 0x73;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x58;
	regS = (regS - 1) & 0xff;
	regPC = 0x7465;
	cycles += 6;
	break;

case 0x7358:
	// $7358	ldx $0d
	regX = mem[0xd];
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $735a	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 9;
	break;

case 0x735b:
	// $735b	cpx #$1b
	temp = regX - 0x1b;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $735d	beq $7376
	if (flagZ == 1) {
		cycles += 4;
		regPC = 0x7376;
		break;
	};
	// $735f	cpx #$1c
	temp = regX - 0x1c;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $7361	beq $737c
	if (flagZ == 1) {
		cycles += 8;
		regPC = 0x737c;
		break;
	};
	// $7363	bcs $7384
	if (flagC == 1) {
		cycles += 10;
		regPC = 0x7384;
		break;
	};
	// $7365	and #$18
	regA = (regA & 0x18) & 0xff;
	// $7367	lsr a
	temp = regA;
	temp >>= 1;
	regA = (temp & 0xff);
	// $7368	lsr a
	temp = regA;
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	// $7369	tay
	regY = regA;
	// $736a	lda $51de,y
	regA = mem[0x51de+regY];
	// $736d	ldx $51df,y
	regX = mem[0x51df+regY];
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	cycles += 26;
	regPC = 0x7370;

case 0x7370:
	// $7370	jsr $7d45
	mem[0x0100 | regS] = 0x73;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x73;
	regS = (regS - 1) & 0xff;
	regPC = 0x7d45;
	cycles += 6;
	break;

case 0x7373:
	// $7373	ldx $0d
	regX = mem[0xd];
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7375	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 9;
	break;

case 0x7376:
	// $7376	jsr $750b
	mem[0x0100 | regS] = 0x73;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x79;
	regS = (regS - 1) & 0xff;
	regPC = 0x750b;
	cycles += 6;
	break;

case 0x7379:
	// $7379	ldx $0d
	regX = mem[0xd];
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $737b	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 9;
	break;

case 0x737c:
	// $737c	lda $5250
	regA = mem[0x5250];
	// $737f	ldx $5251
	regX = mem[0x5251];
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7382	bne $7370
	if (flagZ == 0) {
		cycles += 10;
		regPC = 0x7370;
		break;
	};
	cycles += 10;
	regPC = 0x7384;

case 0x7384:
	// $7384	lda #$70
	regA = 0x70;
	// $7386	ldx #$f0
	regX = 0xf0;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7388	jsr $7ce0
	mem[0x0100 | regS] = 0x73;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x8b;
	regS = (regS - 1) & 0xff;
	regPC = 0x7ce0;
	cycles += 10;
	break;

case 0x738b:
	// $738b	ldx $0d
	regX = mem[0xd];
	// $738d	lda $5c
	regA = mem[0x5c];
	// $738f	and #$03
	regA = (regA & 0x3) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7391	bne $7396
	if (flagZ == 0) {
		cycles += 10;
		regPC = 0x7396;
		break;
	};
	// $7393	dec $0200,x
	temp = (mem[0x200+regX] - 1) & 0xff;
	mem[0x200+regX] = temp;
	flagZ = (temp == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	cycles += 17;
	regPC = 0x7396;

case 0x7396:
	// $7396	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 6;
	break;

case 0x7397:
	// $7397	sed
	flagD = 1;
	// $7398	adc $52,x
	temp = mem[(0x52+regX)&0xff];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		flagZ = (zl == 0 ? 1 : 0);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		flagN = (zl >> 7) & 1;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $739a	sta $52,x
	mem[(0x52+regX)&0xff] = regA;
	// $739c	bcc $73b0
	if (flagC == 0) {
		cycles += 12;
		regPC = 0x73b0;
		break;
	};
	// $739e	lda $53,x
	regA = mem[(0x53+regX)&0xff];
	// $73a0	adc #$00
	temp = 0x0;
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $73a2	sta $53,x
	mem[(0x53+regX)&0xff] = regA;
	// $73a4	and #$0f
	regA = (regA & 0xf) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $73a6	bne $73b0
	if (flagZ == 0) {
		cycles += 26;
		regPC = 0x73b0;
		break;
	};
	// $73a8	lda #$b0
	regA = 0xb0;
	// $73aa	sta $68
	mem[0x68] = regA;
	// $73ac	ldx $18
	regX = mem[0x18];
	// $73ae	inc $57,x
	temp = (mem[(0x57+regX)&0xff] + 1) & 0xff;
	mem[(0x57+regX)&0xff] = temp;
	flagZ = (temp == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	cycles += 40;
	regPC = 0x73b0;

case 0x73b0:
	// $73b0	cld
	flagD = 0;
	// $73b1	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 8;
	break;

case 0x73b2:
	// $73b2	lda $18
	regA = mem[0x18];
	// $73b4	asl a
	temp = (regA << 1);
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $73b5	asl a
	temp = (regA << 1);
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $73b6	sta $08
	mem[0x8] = regA;
	// $73b8	lda $6f
	regA = mem[0x6f];
	// $73ba	and #$fb
	regA = (regA & 0xfb) & 0xff;
	// $73bc	ora $08
	regA = (regA | mem[0x8]) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $73be	sta $6f
	mem[0x6f] = regA;
	// $73c0	sta $3200
	writeMem(0x3200, regA);
	// $73c3	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 31;
	break;

case 0x73c4:
	// $73c4	lda $1c
	regA = mem[0x1c];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $73c6	beq $73ca
	if (flagZ == 1) {
		cycles += 5;
		regPC = 0x73ca;
		break;
	};
	cycles += 5;
	regPC = 0x73c8;

case 0x73c8:
	// $73c8	clc
	flagC = 0;
	// $73c9	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 8;
	break;

case 0x73ca:
	// $73ca	lda $5d
	regA = mem[0x5d];
	// $73cc	and #$04
	regA = (regA & 0x4) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $73ce	bne $73c8
	if (flagZ == 0) {
		cycles += 7;
		regPC = 0x73c8;
		break;
	};
	// $73d0	lda $1d
	regA = mem[0x1d];
	// $73d2	ora $1e
	regA = (regA | mem[0x1e]) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $73d4	beq $73c8
	if (flagZ == 1) {
		cycles += 15;
		regPC = 0x73c8;
		break;
	};
	// $73d6	ldy #$00
	regY = 0x0;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $73d8	jsr $77f6
	mem[0x0100 | regS] = 0x73;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xdb;
	regS = (regS - 1) & 0xff;
	regPC = 0x77f6;
	cycles += 23;
	break;

case 0x73db:
	// $73db	ldx #$00
	regX = 0x0;
	// $73dd	stx $10
	mem[0x10] = regX;
	// $73df	lda #$01
	regA = 0x1;
	// $73e1	sta $00
	mem[0x0] = regA;
	// $73e3	lda #$a7
	regA = 0xa7;
	// $73e5	sta $0e
	mem[0xe] = regA;
	// $73e7	lda #$10
	regA = 0x10;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $73e9	sta $00
	mem[0x0] = regA;
	cycles += 20;
	regPC = 0x73eb;

case 0x73eb:
	// $73eb	lda $1d,x
	regA = mem[(0x1d+regX)&0xff];
	// $73ed	ora $1e,x
	regA = (regA | mem[(0x1e+regX)&0xff]) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $73ef	beq $7458
	if (flagZ == 1) {
		cycles += 10;
		regPC = 0x7458;
		break;
	};
	// $73f1	stx $0f
	mem[0xf] = regX;
	// $73f3	lda #$5f
	regA = 0x5f;
	// $73f5	ldx $0e
	regX = mem[0xe];
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $73f7	jsr $7c03
	mem[0x0100 | regS] = 0x73;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xfa;
	regS = (regS - 1) & 0xff;
	regPC = 0x7c03;
	cycles += 24;
	break;

case 0x73fa:
	// $73fa	lda #$40
	regA = 0x40;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $73fc	jsr $7cde
	mem[0x0100 | regS] = 0x73;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xff;
	regS = (regS - 1) & 0xff;
	regPC = 0x7cde;
	cycles += 8;
	break;

case 0x73ff:
	// $73ff	lda $0f
	regA = mem[0xf];
	// $7401	lsr a
	temp = regA;
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	// $7402	sed
	flagD = 1;
	// $7403	adc #$01
	temp = 0x1;
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
	};
	// $7405	cld
	flagD = 0;
	// $7406	sta $0d
	mem[0xd] = regA;
	// $7408	lda #$0d
	regA = 0xd;
	// $740a	sec
	flagC = 1;
	// $740b	ldy #$01
	regY = 0x1;
	// $740d	ldx #$00
	regX = 0x0;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $740f	jsr $773f
	mem[0x0100 | regS] = 0x74;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x12;
	regS = (regS - 1) & 0xff;
	regPC = 0x773f;
	cycles += 28;
	break;

		};

	};

	void exec7400 () {

		switch (regPC) {

case 0x7412:
	// $7412	lda #$40
	regA = 0x40;
	// $7414	tax
	regX = regA;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7415	jsr $7ce0
	mem[0x0100 | regS] = 0x74;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x18;
	regS = (regS - 1) & 0xff;
	regPC = 0x7ce0;
	cycles += 10;
	break;

case 0x7418:
	// $7418	ldy #$00
	regY = 0x0;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $741a	jsr $6f35
	mem[0x0100 | regS] = 0x74;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x1d;
	regS = (regS - 1) & 0xff;
	regPC = 0x6f35;
	cycles += 8;
	break;

case 0x741d:
	// $741d	lda $0f
	regA = mem[0xf];
	// $741f	clc
	flagC = 0;
	// $7420	adc #$1d
	temp = 0x1d;
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
	};
	// $7422	ldy #$02
	regY = 0x2;
	// $7424	sec
	flagC = 1;
	// $7425	ldx #$00
	regX = 0x0;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7427	jsr $773f
	mem[0x0100 | regS] = 0x74;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x2a;
	regS = (regS - 1) & 0xff;
	regPC = 0x773f;
	cycles += 19;
	break;

case 0x742a:
	// $742a	lda #$00
	regA = 0x0;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $742c	jsr $7bd1
	mem[0x0100 | regS] = 0x74;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x2f;
	regS = (regS - 1) & 0xff;
	regPC = 0x7bd1;
	cycles += 8;
	break;

case 0x742f:
	// $742f	ldy #$00
	regY = 0x0;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $7431	jsr $6f35
	mem[0x0100 | regS] = 0x74;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x34;
	regS = (regS - 1) & 0xff;
	regPC = 0x6f35;
	cycles += 8;
	break;

case 0x7434:
	// $7434	ldy $10
	regY = mem[0x10];
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $7436	jsr $6f1a
	mem[0x0100 | regS] = 0x74;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x39;
	regS = (regS - 1) & 0xff;
	regPC = 0x6f1a;
	cycles += 9;
	break;

case 0x7439:
	// $7439	inc $10
	temp = (mem[0x10] + 1) & 0xff;
	mem[0x10] = temp;
	// $743b	ldy $10
	regY = mem[0x10];
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $743d	jsr $6f1a
	mem[0x0100 | regS] = 0x74;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x40;
	regS = (regS - 1) & 0xff;
	regPC = 0x6f1a;
	cycles += 14;
	break;

case 0x7440:
	// $7440	inc $10
	temp = (mem[0x10] + 1) & 0xff;
	mem[0x10] = temp;
	// $7442	ldy $10
	regY = mem[0x10];
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $7444	jsr $6f1a
	mem[0x0100 | regS] = 0x74;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x47;
	regS = (regS - 1) & 0xff;
	regPC = 0x6f1a;
	cycles += 14;
	break;

case 0x7447:
	// $7447	inc $10
	temp = (mem[0x10] + 1) & 0xff;
	mem[0x10] = temp;
	// $7449	lda $0e
	regA = mem[0xe];
	// $744b	sec
	flagC = 1;
	// $744c	sbc #$08
	temp = 0x8;
	if (flagD == 0) {   // no decimal mode
		z = regA - temp - ((flagC == 0) ? 1 : 0);
		zl = z & 0xff;
		zh = z >> 8;
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		regA = zl;
	}
	else {   // decimal mode
		zl = ((regA & 0x0f) - (temp & 0x0f) - ((flagC == 0) ? 1 : 0)) & 0xff;
		if (zl > 15)
			zl -= 6;
		zh = (regA >> 4) - (temp >> 4) - (zl&0x10);
		if (zh > 15)
			zh -= 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
	};
	// $744e	sta $0e
	mem[0xe] = regA;
	// $7450	ldx $0f
	regX = mem[0xf];
	// $7452	inx
	regX++;
	regX&= 0xff;
	// $7453	inx
	regX++;
	regX&= 0xff;
	// $7454	cpx #$14
	temp = regX - 0x14;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $7456	bcc $73eb
	if (flagC == 0) {
		cycles += 26;
		regPC = 0x73eb;
		break;
	};
	cycles += 26;
	regPC = 0x7458;

case 0x7458:
	// $7458	sec
	flagC = 1;
	// $7459	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 8;
	break;

case 0x745a:
	// $745a	ldx #$1a
	regX = 0x1a;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	cycles += 2;
	regPC = 0x745c;

case 0x745c:
	// $745c	lda $0200,x
	regA = mem[0x200+regX];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $745f	beq $7464
	if (flagZ == 1) {
		cycles += 6;
		regPC = 0x7464;
		break;
	};
	// $7461	dex
	regX--;
	regX&= 0xff;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7462	bpl $745c
	if (flagN == 0) {
		cycles += 10;
		regPC = 0x745c;
		break;
	};
	cycles += 10;
	regPC = 0x7464;

case 0x7464:
	// $7464	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 6;
	break;

case 0x7465:
	// $7465	lda $021b
	regA = mem[0x21b];
	// $7468	cmp #$a2
	temp = regA - 0xa2;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $746a	bcs $748e
	if (flagC == 1) {
		cycles += 8;
		regPC = 0x748e;
		break;
	};
	// $746c	ldx #$0a
	regX = 0xa;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	cycles += 10;
	regPC = 0x746e;

case 0x746e:
	// $746e	lda $50ec,x
	regA = mem[0x50ec+regX];
	// $7471	lsr a
	temp = regA;
	temp >>= 1;
	regA = (temp & 0xff);
	// $7472	lsr a
	temp = regA;
	temp >>= 1;
	regA = (temp & 0xff);
	// $7473	lsr a
	temp = regA;
	temp >>= 1;
	regA = (temp & 0xff);
	// $7474	lsr a
	temp = regA;
	temp >>= 1;
	regA = (temp & 0xff);
	// $7475	clc
	flagC = 0;
	// $7476	adc #$f8
	temp = 0xf8;
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
	};
	// $7478	eor #$f8
	regA = (regA ^ 0xf8) & 0xff;
	// $747a	sta $7e,x
	mem[(0x7e+regX)&0xff] = regA;
	// $747c	lda $50ed,x
	regA = mem[0x50ed+regX];
	// $747f	lsr a
	temp = regA;
	temp >>= 1;
	regA = (temp & 0xff);
	// $7480	lsr a
	temp = regA;
	temp >>= 1;
	regA = (temp & 0xff);
	// $7481	lsr a
	temp = regA;
	temp >>= 1;
	regA = (temp & 0xff);
	// $7482	lsr a
	temp = regA;
	temp >>= 1;
	regA = (temp & 0xff);
	// $7483	clc
	flagC = 0;
	// $7484	adc #$f8
	temp = 0xf8;
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $7486	eor #$f8
	regA = (regA ^ 0xf8) & 0xff;
	// $7488	sta $8a,x
	mem[(0x8a+regX)&0xff] = regA;
	// $748a	dex
	regX--;
	regX&= 0xff;
	// $748b	dex
	regX--;
	regX&= 0xff;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $748c	bpl $746e
	if (flagN == 0) {
		cycles += 50;
		regPC = 0x746e;
		break;
	};
	cycles += 50;
	regPC = 0x748e;

case 0x748e:
	// $748e	lda $021b
	regA = mem[0x21b];
	// $7491	eor #$ff
	regA = (regA ^ 0xff) & 0xff;
	// $7493	and #$70
	regA = (regA & 0x70) & 0xff;
	// $7495	lsr a
	temp = regA;
	temp >>= 1;
	regA = (temp & 0xff);
	// $7496	lsr a
	temp = regA;
	temp >>= 1;
	regA = (temp & 0xff);
	// $7497	lsr a
	temp = regA;
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	// $7498	tax
	regX = regA;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	cycles += 16;
	regPC = 0x7499;

case 0x7499:
	// $7499	stx $09
	mem[0x9] = regX;
	// $749b	ldy #$00
	regY = 0x0;
	// $749d	lda $50ec,x
	regA = mem[0x50ec+regX];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $74a0	bpl $74a3
	if (flagN == 0) {
		cycles += 11;
		regPC = 0x74a3;
		break;
	};
	// $74a2	dey
	regY--;
	regY&= 0xff;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	cycles += 13;
	regPC = 0x74a3;

case 0x74a3:
	// $74a3	clc
	flagC = 0;
	// $74a4	adc $7d,x
	temp = mem[(0x7d+regX)&0xff];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $74a6	sta $7d,x
	mem[(0x7d+regX)&0xff] = regA;
	// $74a8	tya
	regA = regY;
	// $74a9	adc $7e,x
	temp = mem[(0x7e+regX)&0xff];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $74ab	sta $7e,x
	mem[(0x7e+regX)&0xff] = regA;
	// $74ad	sta $04
	mem[0x4] = regA;
	// $74af	sty $05
	mem[0x5] = regY;
	// $74b1	ldy #$00
	regY = 0x0;
	// $74b3	lda $50ed,x
	regA = mem[0x50ed+regX];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $74b6	bpl $74b9
	if (flagN == 0) {
		cycles += 34;
		regPC = 0x74b9;
		break;
	};
	// $74b8	dey
	regY--;
	regY&= 0xff;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	cycles += 36;
	regPC = 0x74b9;

case 0x74b9:
	// $74b9	clc
	flagC = 0;
	// $74ba	adc $89,x
	temp = mem[(0x89+regX)&0xff];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $74bc	sta $89,x
	mem[(0x89+regX)&0xff] = regA;
	// $74be	tya
	regA = regY;
	// $74bf	adc $8a,x
	temp = mem[(0x8a+regX)&0xff];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $74c1	sta $8a,x
	mem[(0x8a+regX)&0xff] = regA;
	// $74c3	sta $06
	mem[0x6] = regA;
	// $74c5	sty $07
	mem[0x7] = regY;
	// $74c7	lda $02
	regA = mem[0x2];
	// $74c9	sta $0b
	mem[0xb] = regA;
	// $74cb	lda $03
	regA = mem[0x3];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $74cd	sta $0c
	mem[0xc] = regA;
	// $74cf	jsr $7c49
	mem[0x0100 | regS] = 0x74;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xd2;
	regS = (regS - 1) & 0xff;
	regPC = 0x7c49;
	cycles += 44;
	break;

case 0x74d2:
	// $74d2	ldy $09
	regY = mem[0x9];
	// $74d4	lda $50e0,y
	regA = mem[0x50e0+regY];
	// $74d7	ldx $50e1,y
	regX = mem[0x50e1+regY];
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $74da	jsr $7d45
	mem[0x0100 | regS] = 0x74;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xdd;
	regS = (regS - 1) & 0xff;
	regPC = 0x7d45;
	cycles += 17;
	break;

case 0x74dd:
	// $74dd	ldy $09
	regY = mem[0x9];
	// $74df	lda $50e1,y
	regA = mem[0x50e1+regY];
	// $74e2	eor #$04
	regA = (regA ^ 0x4) & 0xff;
	// $74e4	tax
	regX = regA;
	// $74e5	lda $50e0,y
	regA = mem[0x50e0+regY];
	// $74e8	and #$0f
	regA = (regA & 0xf) & 0xff;
	// $74ea	eor #$04
	regA = (regA ^ 0x4) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $74ec	jsr $7d45
	mem[0x0100 | regS] = 0x74;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xef;
	regS = (regS - 1) & 0xff;
	regPC = 0x7d45;
	cycles += 25;
	break;

case 0x74ef:
	// $74ef	ldy #$ff
	regY = 0xff;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	cycles += 2;
	regPC = 0x74f1;

case 0x74f1:
	// $74f1	iny
	regY++;
	regY&= 0xff;
	// $74f2	lda ($0b),y
	regA = mem[(mem[0xb] | (mem[0xc]<<8))+regY];
	// $74f4	sta ($02),y
	mem[(mem[0x2] | (mem[0x3]<<8)) + regY] = regA;
	// $74f6	iny
	regY++;
	regY&= 0xff;
	// $74f7	lda ($0b),y
	regA = mem[(mem[0xb] | (mem[0xc]<<8))+regY];
	// $74f9	eor #$04
	regA = (regA ^ 0x4) & 0xff;
	// $74fb	sta ($02),y
	mem[(mem[0x2] | (mem[0x3]<<8)) + regY] = regA;
	// $74fd	cpy #$03
	temp = regY - 0x3;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $74ff	bcc $74f1
	if (flagC == 0) {
		cycles += 32;
		regPC = 0x74f1;
		break;
	};
	// $7501	jsr $7c39
	mem[0x0100 | regS] = 0x75;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x4;
	regS = (regS - 1) & 0xff;
	regPC = 0x7c39;
	cycles += 38;
	break;

case 0x7504:
	// $7504	ldx $09
	regX = mem[0x9];
	// $7506	dex
	regX--;
	regX&= 0xff;
	// $7507	dex
	regX--;
	regX&= 0xff;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7508	bpl $7499
	if (flagN == 0) {
		cycles += 9;
		regPC = 0x7499;
		break;
	};
	// $750a	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 15;
	break;

case 0x750b:
	// $750b	ldx #$00
	regX = 0x0;
	// $750d	stx $17
	mem[0x17] = regX;
	// $750f	ldy #$00
	regY = 0x0;
	// $7511	lda $61
	regA = mem[0x61];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7513	bpl $751b
	if (flagN == 0) {
		cycles += 12;
		regPC = 0x751b;
		break;
	};
	// $7515	ldy #$04
	regY = 0x4;
	// $7517	txa
	regA = regX;
	// $7518	sec
	flagC = 1;
	// $7519	sbc $61
	temp = mem[0x61];
	if (flagD == 0) {   // no decimal mode
		z = regA - temp - ((flagC == 0) ? 1 : 0);
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh == 0) ? 1 : 0;
		flagZ = (zl == 0 ? 1 : 0);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		flagN = (zl >> 7) & 1;
		regA = zl;
	}
	else {   // decimal mode
		zl = ((regA & 0x0f) - (temp & 0x0f) - ((flagC == 0) ? 1 : 0)) & 0xff;
		if (zl > 15)
			zl -= 6;
		zh = (regA >> 4) - (temp >> 4) - (zl&0x10);
		if (zh > 15)
			zh -= 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	cycles += 21;
	regPC = 0x751b;

case 0x751b:
	// $751b	sta $08
	mem[0x8] = regA;
	// $751d	bit $08
	temp =mem[0x8];
	flagN = (temp >> 7);
	flagV = ((temp >> 6) & 1);
	flagZ = ((temp & regA) != 0) ? 0 : 1;
	// $751f	bmi $7523
	if (flagN == 1) {
		cycles += 8;
		regPC = 0x7523;
		break;
	};
	// $7521	bvc $752a
	if (flagV == 0) {
		cycles += 10;
		regPC = 0x752a;
		break;
	};
	cycles += 10;
	regPC = 0x7523;

case 0x7523:
	// $7523	ldx #$04
	regX = 0x4;
	// $7525	lda #$80
	regA = 0x80;
	// $7527	sec
	flagC = 1;
	// $7528	sbc $08
	temp = mem[0x8];
	if (flagD == 0) {   // no decimal mode
		z = regA - temp - ((flagC == 0) ? 1 : 0);
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh == 0) ? 1 : 0;
		flagZ = (zl == 0 ? 1 : 0);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		flagN = (zl >> 7) & 1;
		regA = zl;
	}
	else {   // decimal mode
		zl = ((regA & 0x0f) - (temp & 0x0f) - ((flagC == 0) ? 1 : 0)) & 0xff;
		if (zl > 15)
			zl -= 6;
		zh = (regA >> 4) - (temp >> 4) - (zl&0x10);
		if (zh > 15)
			zh -= 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	cycles += 9;
	regPC = 0x752a;

case 0x752a:
	// $752a	stx $08
	mem[0x8] = regX;
	// $752c	sty $09
	mem[0x9] = regY;
	// $752e	lsr a
	temp = regA;
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	// $752f	and #$fe
	regA = (regA & 0xfe) & 0xff;
	// $7531	tay
	regY = regA;
	// $7532	lda $526e,y
	regA = mem[0x526e+regY];
	// $7535	ldx $526f,y
	regX = mem[0x526f+regY];
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7538	jsr $6ad3
	mem[0x0100 | regS] = 0x75;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x3b;
	regS = (regS - 1) & 0xff;
	regPC = 0x6ad3;
	cycles += 26;
	break;

case 0x753b:
	// $753b	lda $2405
	regA = readMem(0x2405);
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $753e	bpl $7554
	if (flagN == 0) {
		cycles += 6;
		regPC = 0x7554;
		break;
	};
	// $7540	lda $5c
	regA = mem[0x5c];
	// $7542	and #$04
	regA = (regA & 0x4) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7544	beq $7554
	if (flagZ == 1) {
		cycles += 13;
		regPC = 0x7554;
		break;
	};
	// $7546	iny
	regY++;
	regY&= 0xff;
	// $7547	iny
	regY++;
	regY&= 0xff;
	// $7548	sec
	flagC = 1;
	// $7549	ldx $0c
	regX = mem[0xc];
	// $754b	tya
	regA = regY;
	// $754c	adc $0b
	temp = mem[0xb];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		flagZ = (zl == 0 ? 1 : 0);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		flagN = (zl >> 7) & 1;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $754e	bcc $7551
	if (flagC == 0) {
		cycles += 29;
		regPC = 0x7551;
		break;
	};
	// $7550	inx
	regX++;
	regX&= 0xff;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	cycles += 31;
	regPC = 0x7551;

case 0x7551:
	// $7551	jsr $6ad3
	mem[0x0100 | regS] = 0x75;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x54;
	regS = (regS - 1) & 0xff;
	regPC = 0x6ad3;
	cycles += 6;
	break;

case 0x7554:
	// $7554	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 6;
	break;

case 0x7555:
	// $7555	lda $1c
	regA = mem[0x1c];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7557	bne $755a
	if (flagZ == 0) {
		cycles += 5;
		regPC = 0x755a;
		break;
	};
	// $7559	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 11;
	break;

case 0x755a:
	// $755a	ldx #$00
	regX = 0x0;
	// $755c	lda $021c
	regA = mem[0x21c];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $755f	bmi $756b
	if (flagN == 1) {
		cycles += 8;
		regPC = 0x756b;
		break;
	};
	// $7561	beq $756b
	if (flagZ == 1) {
		cycles += 10;
		regPC = 0x756b;
		break;
	};
	// $7563	ror a
	temp = regA | (flagC << 8);
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	// $7564	ror a
	temp = regA | (flagC << 8);
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	// $7565	ror a
	temp = regA | (flagC << 8);
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	// $7566	sta $3c02
	writeMem(0x3c02, regA);
	// $7569	ldx #$80
	regX = 0x80;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	cycles += 22;
	regPC = 0x756b;

case 0x756b:
	// $756b	stx $3c00
	writeMem(0x3c00, regX);
	// $756e	ldx #$01
	regX = 0x1;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7570	jsr $75cd
	mem[0x0100 | regS] = 0x75;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x73;
	regS = (regS - 1) & 0xff;
	regPC = 0x75cd;
	cycles += 12;
	break;

case 0x7573:
	// $7573	sta $3c01
	writeMem(0x3c01, regA);
	// $7576	dex
	regX--;
	regX&= 0xff;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7577	jsr $75cd
	mem[0x0100 | regS] = 0x75;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x7a;
	regS = (regS - 1) & 0xff;
	regPC = 0x75cd;
	cycles += 12;
	break;

case 0x757a:
	// $757a	sta $3c04
	writeMem(0x3c04, regA);
	// $757d	lda $021b
	regA = mem[0x21b];
	// $7580	cmp #$01
	temp = regA - 0x1;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $7582	beq $7588
	if (flagZ == 1) {
		cycles += 12;
		regPC = 0x7588;
		break;
	};
	// $7584	txa
	regA = regX;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7585	sta $3c03
	writeMem(0x3c03, regA);
	cycles += 18;
	regPC = 0x7588;

case 0x7588:
	// $7588	lda $02f6
	regA = mem[0x2f6];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $758b	beq $759e
	if (flagZ == 1) {
		cycles += 6;
		regPC = 0x759e;
		break;
	};
	// $758d	lda $021b
	regA = mem[0x21b];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7590	bmi $759e
	if (flagN == 1) {
		cycles += 12;
		regPC = 0x759e;
		break;
	};
	// $7592	ora $59
	regA = (regA | mem[0x59]) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7594	beq $759e
	if (flagZ == 1) {
		cycles += 17;
		regPC = 0x759e;
		break;
	};
	// $7596	lda $6d
	regA = mem[0x6d];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7598	beq $75ae
	if (flagZ == 1) {
		cycles += 22;
		regPC = 0x75ae;
		break;
	};
	// $759a	dec $6d
	temp = (mem[0x6d] - 1) & 0xff;
	mem[0x6d] = temp;
	flagZ = (temp == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $759c	bne $75bf
	if (flagZ == 0) {
		cycles += 29;
		regPC = 0x75bf;
		break;
	};
	cycles += 29;
	regPC = 0x759e;

case 0x759e:
	// $759e	lda $6c
	regA = mem[0x6c];
	// $75a0	and #$0f
	regA = (regA & 0xf) & 0xff;
	// $75a2	sta $6c
	mem[0x6c] = regA;
	// $75a4	sta $3a00
	writeMem(0x3a00, regA);
	// $75a7	lda $02fc
	regA = mem[0x2fc];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $75aa	sta $6e
	mem[0x6e] = regA;
	// $75ac	bpl $75bf
	if (flagN == 0) {
		cycles += 21;
		regPC = 0x75bf;
		break;
	};
	cycles += 21;
	regPC = 0x75ae;

case 0x75ae:
	// $75ae	dec $6e
	temp = (mem[0x6e] - 1) & 0xff;
	mem[0x6e] = temp;
	flagZ = (temp == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $75b0	bne $75bf
	if (flagZ == 0) {
		cycles += 7;
		regPC = 0x75bf;
		break;
	};
	// $75b2	lda #$04
	regA = 0x4;
	// $75b4	sta $6d
	mem[0x6d] = regA;
	// $75b6	lda $6c
	regA = mem[0x6c];
	// $75b8	eor #$14
	regA = (regA ^ 0x14) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $75ba	sta $6c
	mem[0x6c] = regA;
	// $75bc	sta $3a00
	writeMem(0x3a00, regA);
	cycles += 24;
	regPC = 0x75bf;

case 0x75bf:
	// $75bf	lda $69
	regA = mem[0x69];
	// $75c1	tax
	regX = regA;
	// $75c2	and #$3f
	regA = (regA & 0x3f) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $75c4	beq $75c7
	if (flagZ == 1) {
		cycles += 9;
		regPC = 0x75c7;
		break;
	};
	// $75c6	dex
	regX--;
	regX&= 0xff;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	cycles += 11;
	regPC = 0x75c7;

case 0x75c7:
	// $75c7	stx $69
	mem[0x69] = regX;
	// $75c9	stx $3600
	writeMem(0x3600, regX);
	// $75cc	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 13;
	break;

case 0x75cd:
	// $75cd	lda $6a,x
	regA = mem[(0x6a+regX)&0xff];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $75cf	bmi $75dd
	if (flagN == 1) {
		cycles += 6;
		regPC = 0x75dd;
		break;
	};
	// $75d1	lda $66,x
	regA = mem[(0x66+regX)&0xff];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $75d3	bpl $75e7
	if (flagN == 0) {
		cycles += 12;
		regPC = 0x75e7;
		break;
	};
	// $75d5	lda #$10
	regA = 0x10;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $75d7	sta $66,x
	mem[(0x66+regX)&0xff] = regA;
	cycles += 18;
	regPC = 0x75d9;

case 0x75d9:
	// $75d9	lda #$80
	regA = 0x80;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $75db	bmi $75e9
	if (flagN == 1) {
		cycles += 4;
		regPC = 0x75e9;
		break;
	};
	cycles += 4;
	regPC = 0x75dd;

case 0x75dd:
	// $75dd	lda $66,x
	regA = mem[(0x66+regX)&0xff];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $75df	beq $75e7
	if (flagZ == 1) {
		cycles += 6;
		regPC = 0x75e7;
		break;
	};
	// $75e1	bmi $75e7
	if (flagN == 1) {
		cycles += 8;
		regPC = 0x75e7;
		break;
	};
	// $75e3	dec $66,x
	temp = (mem[(0x66+regX)&0xff] - 1) & 0xff;
	mem[(0x66+regX)&0xff] = temp;
	flagZ = (temp == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $75e5	bne $75d9
	if (flagZ == 0) {
		cycles += 16;
		regPC = 0x75d9;
		break;
	};
	cycles += 16;
	regPC = 0x75e7;

case 0x75e7:
	// $75e7	lda #$00
	regA = 0x0;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 2;
	regPC = 0x75e9;

case 0x75e9:
	// $75e9	sta $6a,x
	mem[(0x6a+regX)&0xff] = regA;
	// $75eb	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 10;
	break;

case 0x75ec:
	// $75ec	stx $0d
	mem[0xd] = regX;
	// $75ee	lda #$50
	regA = 0x50;
	// $75f0	sta $02f9
	mem[0x2f9] = regA;
	// $75f3	lda $0200,y
	regA = mem[0x200+regY];
	// $75f6	and #$78
	regA = (regA & 0x78) & 0xff;
	// $75f8	sta $0e
	mem[0xe] = regA;
	// $75fa	lda $0200,y
	regA = mem[0x200+regY];
	// $75fd	and #$07
	regA = (regA & 0x7) & 0xff;
	// $75ff	lsr a
	temp = regA;
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	// $7600	tax
	regX = regA;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7601	beq $7605
	if (flagZ == 1) {
		cycles += 30;
		regPC = 0x7605;
		break;
	};
	// $7603	ora $0e
	regA = (regA | mem[0xe]) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 33;
	regPC = 0x7605;

case 0x7605:
	// $7605	sta $0200,y
	mem[0x200+regY] = regA;
	// $7608	lda $1c
	regA = mem[0x1c];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $760a	beq $761d
	if (flagZ == 1) {
		cycles += 10;
		regPC = 0x761d;
		break;
	};
	// $760c	lda $0d
	regA = mem[0xd];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $760e	beq $7614
	if (flagZ == 1) {
		cycles += 15;
		regPC = 0x7614;
		break;
	};
	// $7610	cmp #$04
	temp = regA - 0x4;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $7612	bcc $761d
	if (flagC == 0) {
		cycles += 19;
		regPC = 0x761d;
		break;
	};
	cycles += 19;
	regPC = 0x7614;

case 0x7614:
	// $7614	lda $7659,x
	regA = mem[0x7659+regX];
	// $7617	ldx $19
	regX = mem[0x19];
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7619	clc
	flagC = 0;
	// $761a	jsr $7397
	mem[0x0100 | regS] = 0x76;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x1d;
	regS = (regS - 1) & 0xff;
	regPC = 0x7397;
	cycles += 15;
	break;

case 0x761d:
	// $761d	ldx $0200,y
	regX = mem[0x200+regY];
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7620	beq $7656
	if (flagZ == 1) {
		cycles += 6;
		regPC = 0x7656;
		break;
	};
	// $7622	jsr $745a
	mem[0x0100 | regS] = 0x76;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x25;
	regS = (regS - 1) & 0xff;
	regPC = 0x745a;
	cycles += 12;
	break;

case 0x7625:
	// $7625	bmi $7656
	if (flagN == 1) {
		cycles += 2;
		regPC = 0x7656;
		break;
	};
	// $7627	inc $02f6
	temp = (mem[0x2f6] + 1) & 0xff;
	mem[0x2f6] = temp;
	flagZ = (temp == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $762a	jsr $6a9d
	mem[0x0100 | regS] = 0x76;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x2d;
	regS = (regS - 1) & 0xff;
	regPC = 0x6a9d;
	cycles += 14;
	break;

case 0x762d:
	// $762d	jsr $7203
	mem[0x0100 | regS] = 0x76;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x30;
	regS = (regS - 1) & 0xff;
	regPC = 0x7203;
	cycles += 6;
	break;

case 0x7630:
	// $7630	lda $0223,x
	regA = mem[0x223+regX];
	// $7633	and #$1f
	regA = (regA & 0x1f) & 0xff;
	// $7635	asl a
	temp = (regA << 1);
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $7636	eor $02af,x
	regA = (regA ^ mem[0x2af+regX]) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7639	sta $02af,x
	mem[0x2af+regX] = regA;
	// $763c	jsr $745c
	mem[0x0100 | regS] = 0x76;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x3f;
	regS = (regS - 1) & 0xff;
	regPC = 0x745c;
	cycles += 24;
	break;

case 0x763f:
	// $763f	bmi $7656
	if (flagN == 1) {
		cycles += 2;
		regPC = 0x7656;
		break;
	};
	// $7641	inc $02f6
	temp = (mem[0x2f6] + 1) & 0xff;
	mem[0x2f6] = temp;
	flagZ = (temp == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $7644	jsr $6a9d
	mem[0x0100 | regS] = 0x76;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x47;
	regS = (regS - 1) & 0xff;
	regPC = 0x6a9d;
	cycles += 14;
	break;

case 0x7647:
	// $7647	jsr $7203
	mem[0x0100 | regS] = 0x76;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x4a;
	regS = (regS - 1) & 0xff;
	regPC = 0x7203;
	cycles += 6;
	break;

case 0x764a:
	// $764a	lda $0246,x
	regA = mem[0x246+regX];
	// $764d	and #$1f
	regA = (regA & 0x1f) & 0xff;
	// $764f	asl a
	temp = (regA << 1);
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $7650	eor $02d2,x
	regA = (regA ^ mem[0x2d2+regX]) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7653	sta $02d2,x
	mem[0x2d2+regX] = regA;
	cycles += 18;
	regPC = 0x7656;

case 0x7656:
	// $7656	ldx $0d
	regX = mem[0xd];
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7658	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 9;
	break;

case 0x7659:
	// dc.b $10		; $7659
	// dc.b $05		; $765a
	// dc.b $02		; $765b

case 0x765c:
	// $765c	lda $1c
	regA = mem[0x1c];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $765e	bpl $7698
	if (flagN == 0) {
		cycles += 5;
		regPC = 0x7698;
		break;
	};
	// $7660	ldx #$02
	regX = 0x2;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7662	sta $5d
	mem[0x5d] = regA;
	// $7664	sta $32
	mem[0x32] = regA;
	// $7666	sta $33
	mem[0x33] = regA;
	cycles += 16;
	regPC = 0x7668;

case 0x7668:
	// $7668	ldy #$00
	regY = 0x0;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	cycles += 2;
	regPC = 0x766a;

case 0x766a:
	// $766a	lda $001d,y
	regA = mem[0x1d+regY];
	// $766d	cmp $52,x
	temp = regA - mem[(0x52+regX)&0xff];
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	// $766f	lda $001e,y
	regA = mem[0x1e+regY];
	// $7672	sbc $53,x
	temp = mem[(0x53+regX)&0xff];
	if (flagD == 0) {   // no decimal mode
		z = regA - temp - ((flagC == 0) ? 1 : 0);
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh == 0) ? 1 : 0;
		flagZ = (zl == 0 ? 1 : 0);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		flagN = (zl >> 7) & 1;
		regA = zl;
	}
	else {   // decimal mode
		zl = ((regA & 0x0f) - (temp & 0x0f) - ((flagC == 0) ? 1 : 0)) & 0xff;
		if (zl > 15)
			zl -= 6;
		zh = (regA >> 4) - (temp >> 4) - (zl&0x10);
		if (zh > 15)
			zh -= 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $7674	bcc $7699
	if (flagC == 0) {
		cycles += 18;
		regPC = 0x7699;
		break;
	};
	// $7676	iny
	regY++;
	regY&= 0xff;
	// $7677	iny
	regY++;
	regY&= 0xff;
	// $7678	cpy #$14
	temp = regY - 0x14;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $767a	bcc $766a
	if (flagC == 0) {
		cycles += 26;
		regPC = 0x766a;
		break;
	};
	cycles += 26;
	regPC = 0x767c;

case 0x767c:
	// $767c	dex
	regX--;
	regX&= 0xff;
	// $767d	dex
	regX--;
	regX&= 0xff;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $767e	bpl $7668
	if (flagN == 0) {
		cycles += 6;
		regPC = 0x7668;
		break;
	};
	// $7680	lda $33
	regA = mem[0x33];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7682	bmi $7692
	if (flagN == 1) {
		cycles += 11;
		regPC = 0x7692;
		break;
	};
	// $7684	cmp $32
	temp = regA - mem[0x32];
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $7686	bcc $7692
	if (flagC == 0) {
		cycles += 16;
		regPC = 0x7692;
		break;
	};
	// $7688	adc #$02
	temp = 0x2;
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
	};
	// $768a	cmp #$1e
	temp = regA - 0x1e;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $768c	bcc $7690
	if (flagC == 0) {
		cycles += 22;
		regPC = 0x7690;
		break;
	};
	// $768e	lda #$ff
	regA = 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 24;
	regPC = 0x7690;

case 0x7690:
	// $7690	sta $33
	mem[0x33] = regA;
	cycles += 3;
	regPC = 0x7692;

case 0x7692:
	// $7692	lda #$00
	regA = 0x0;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7694	sta $1c
	mem[0x1c] = regA;
	// $7696	sta $31
	mem[0x31] = regA;
	cycles += 8;
	regPC = 0x7698;

case 0x7698:
	// $7698	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 6;
	break;

case 0x7699:
	// $7699	stx $0b
	mem[0xb] = regX;
	// $769b	sty $0c
	mem[0xc] = regY;
	// $769d	txa
	regA = regX;
	// $769e	lsr a
	temp = regA;
	temp >>= 1;
	regA = (temp & 0xff);
	// $769f	tax
	regX = regA;
	// $76a0	tya
	regA = regY;
	// $76a1	lsr a
	temp = regA;
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	// $76a2	adc $0c
	temp = mem[0xc];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $76a4	sta $0d
	mem[0xd] = regA;
	// $76a6	sta $32,x
	mem[(0x32+regX)&0xff] = regA;
	// $76a8	ldx #$1b
	regX = 0x1b;
	// $76aa	ldy #$12
	regY = 0x12;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	cycles += 30;
	regPC = 0x76ac;

case 0x76ac:
	// $76ac	cpx $0d
	temp = regX - mem[0xd];
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $76ae	beq $76cf
	if (flagZ == 1) {
		cycles += 5;
		regPC = 0x76cf;
		break;
	};
	// $76b0	lda $31,x
	regA = mem[(0x31+regX)&0xff];
	// $76b2	sta $34,x
	mem[(0x34+regX)&0xff] = regA;
	// $76b4	lda $32,x
	regA = mem[(0x32+regX)&0xff];
	// $76b6	sta $35,x
	mem[(0x35+regX)&0xff] = regA;
	// $76b8	lda $33,x
	regA = mem[(0x33+regX)&0xff];
	// $76ba	sta $36,x
	mem[(0x36+regX)&0xff] = regA;
	// $76bc	lda $001b,y
	regA = mem[0x1b+regY];
	// $76bf	sta $001d,y
	mem[0x1d+regY] = regA;
	// $76c2	lda $001c,y
	regA = mem[0x1c+regY];
	// $76c5	sta $001e,y
	mem[0x1e+regY] = regA;
	// $76c8	dey
	regY--;
	regY&= 0xff;
	// $76c9	dey
	regY--;
	regY&= 0xff;
	// $76ca	dex
	regX--;
	regX&= 0xff;
	// $76cb	dex
	regX--;
	regX&= 0xff;
	// $76cc	dex
	regX--;
	regX&= 0xff;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $76cd	bne $76ac
	if (flagZ == 0) {
		cycles += 59;
		regPC = 0x76ac;
		break;
	};
	cycles += 59;
	regPC = 0x76cf;

case 0x76cf:
	// $76cf	lda #$0b
	regA = 0xb;
	// $76d1	sta $34,x
	mem[(0x34+regX)&0xff] = regA;
	// $76d3	lda #$00
	regA = 0x0;
	// $76d5	sta $35,x
	mem[(0x35+regX)&0xff] = regA;
	// $76d7	sta $36,x
	mem[(0x36+regX)&0xff] = regA;
	// $76d9	lda #$f0
	regA = 0xf0;
	// $76db	sta $5d
	mem[0x5d] = regA;
	// $76dd	ldx $0b
	regX = mem[0xb];
	// $76df	ldy $0c
	regY = mem[0xc];
	// $76e1	lda $53,x
	regA = mem[(0x53+regX)&0xff];
	// $76e3	sta $001e,y
	mem[0x1e+regY] = regA;
	// $76e6	lda $52,x
	regA = mem[(0x52+regX)&0xff];
	// $76e8	sta $001d,y
	mem[0x1d+regY] = regA;
	// $76eb	ldy #$00
	regY = 0x0;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $76ed	beq $767c
	if (flagZ == 1) {
		cycles += 49;
		regPC = 0x767c;
		break;
	};

case 0x76f0:
	// $76f0	tya
	regA = regY;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $76f1	bpl $76fc
	if (flagN == 0) {
		cycles += 60;
		regPC = 0x76fc;
		break;
	};
	// $76f3	jsr $7708
	mem[0x0100 | regS] = 0x76;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xf6;
	regS = (regS - 1) & 0xff;
	regPC = 0x7708;
	cycles += 66;
	break;

case 0x76f6:
	// $76f6	jsr $76fc
	mem[0x0100 | regS] = 0x76;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xf9;
	regS = (regS - 1) & 0xff;
	regPC = 0x76fc;
	cycles += 6;
	break;

case 0x76f9:
	// $76f9	jmp $7708
	regPC = 0x7708;
	cycles += 3;
	break;

case 0x76fc:
	// $76fc	tay
	regY = regA;
	// $76fd	txa
	regA = regX;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $76fe	bpl $770e
	if (flagN == 0) {
		cycles += 6;
		regPC = 0x770e;
		break;
	};
	// $7700	jsr $7708
	mem[0x0100 | regS] = 0x77;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x3;
	regS = (regS - 1) & 0xff;
	regPC = 0x7708;
	cycles += 12;
	break;

case 0x7703:
	// $7703	jsr $770e
	mem[0x0100 | regS] = 0x77;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x6;
	regS = (regS - 1) & 0xff;
	regPC = 0x770e;
	cycles += 6;
	break;

case 0x7706:
	// $7706	eor #$80
	regA = (regA ^ 0x80) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 2;
	regPC = 0x7708;

case 0x7708:
	// $7708	eor #$ff
	regA = (regA ^ 0xff) & 0xff;
	// $770a	clc
	flagC = 0;
	// $770b	adc #$01
	temp = 0x1;
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		flagZ = (zl == 0 ? 1 : 0);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		flagN = (zl >> 7) & 1;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $770d	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 12;
	break;

case 0x770e:
	// $770e	sta $0c
	mem[0xc] = regA;
	// $7710	tya
	regA = regY;
	// $7711	cmp $0c
	temp = regA - mem[0xc];
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $7713	beq $7725
	if (flagZ == 1) {
		cycles += 10;
		regPC = 0x7725;
		break;
	};
	// $7715	bcc $7728
	if (flagC == 0) {
		cycles += 12;
		regPC = 0x7728;
		break;
	};
	// $7717	ldy $0c
	regY = mem[0xc];
	// $7719	sta $0c
	mem[0xc] = regA;
	// $771b	tya
	regA = regY;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $771c	jsr $7728
	mem[0x0100 | regS] = 0x77;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x1f;
	regS = (regS - 1) & 0xff;
	regPC = 0x7728;
	cycles += 26;
	break;

case 0x771f:
	// $771f	sec
	flagC = 1;
	// $7720	sbc #$40
	temp = 0x40;
	if (flagD == 0) {   // no decimal mode
		z = regA - temp - ((flagC == 0) ? 1 : 0);
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh == 0) ? 1 : 0;
		flagZ = (zl == 0 ? 1 : 0);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		flagN = (zl >> 7) & 1;
		regA = zl;
	}
	else {   // decimal mode
		zl = ((regA & 0x0f) - (temp & 0x0f) - ((flagC == 0) ? 1 : 0)) & 0xff;
		if (zl > 15)
			zl -= 6;
		zh = (regA >> 4) - (temp >> 4) - (zl&0x10);
		if (zh > 15)
			zh -= 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $7722	jmp $7708
	regPC = 0x7708;
	cycles += 7;
	break;

case 0x7725:
	// $7725	lda #$20
	regA = 0x20;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7727	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 8;
	break;

case 0x7728:
	// $7728	jsr $776c
	mem[0x0100 | regS] = 0x77;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x2b;
	regS = (regS - 1) & 0xff;
	regPC = 0x776c;
	cycles += 6;
	break;

case 0x772b:
	// $772b	lda $772f,x
	regA = mem[0x772f+regX];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $772e	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 10;
	break;

case 0x772f:
	// dc.b $00		; $772f
	// dc.b $02		; $7730
	// dc.b $05		; $7731
	// dc.b $07		; $7732
	// dc.b $0a		; $7733
	// dc.b $0c		; $7734
	// dc.b $0f		; $7735
	// dc.b $11		; $7736
	// dc.b $13		; $7737
	// dc.b $15		; $7738
	// dc.b $17		; $7739
	// dc.b $19		; $773a
	// dc.b $1a		; $773b
	// dc.b $1c		; $773c
	// dc.b $1d		; $773d
	// dc.b $1f		; $773e

case 0x773f:
	// $773f	php
	mem[0x0100 | regS] = ((flagN<<7)|(flagV<<6)|(0x20)|(flagB<<4)|(flagD<<3)|(flagI<<2)|(flagZ<<1)|flagC);
	regS = (regS - 1) & 0xff;
	// $7740	stx $17
	mem[0x17] = regX;
	// $7742	dey
	regY--;
	regY&= 0xff;
	// $7743	sty $16
	mem[0x16] = regY;
	// $7745	clc
	flagC = 0;
	// $7746	adc $16
	temp = mem[0x16];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
	};
	// $7748	sta $15
	mem[0x15] = regA;
	// $774a	plp
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	flagV = (temp >> 6) & 1;
	flagB = (temp >> 4) & 1;
	flagD = (temp >> 3) & 1;
	flagI = (temp >> 2) & 1;
	flagC = temp & 1;
	// $774b	tax
	regX = regA;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	cycles += 25;
	regPC = 0x774c;

case 0x774c:
	// $774c	php
	mem[0x0100 | regS] = ((flagN<<7)|(flagV<<6)|(0x20)|(flagB<<4)|(flagD<<3)|(flagI<<2)|(flagZ<<1)|flagC);
	regS = (regS - 1) & 0xff;
	// $774d	lda $00,x
	regA = mem[(0x0+regX)&0xff];
	// $774f	lsr a
	temp = regA;
	temp >>= 1;
	regA = (temp & 0xff);
	// $7750	lsr a
	temp = regA;
	temp >>= 1;
	regA = (temp & 0xff);
	// $7751	lsr a
	temp = regA;
	temp >>= 1;
	regA = (temp & 0xff);
	// $7752	lsr a
	temp = regA;
	temp >>= 1;
	regA = (temp & 0xff);
	// $7753	plp
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	flagN = (temp >> 7) & 1;
	flagV = (temp >> 6) & 1;
	flagB = (temp >> 4) & 1;
	flagD = (temp >> 3) & 1;
	flagI = (temp >> 2) & 1;
	flagZ = (temp >> 1) & 1;
	flagC = temp & 1;
	// $7754	jsr $7785
	mem[0x0100 | regS] = 0x77;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x57;
	regS = (regS - 1) & 0xff;
	regPC = 0x7785;
	cycles += 25;
	break;

case 0x7757:
	// $7757	lda $16
	regA = mem[0x16];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7759	bne $775c
	if (flagZ == 0) {
		cycles += 5;
		regPC = 0x775c;
		break;
	};
	// $775b	clc
	flagC = 0;
	cycles += 7;
	regPC = 0x775c;

case 0x775c:
	// $775c	ldx $15
	regX = mem[0x15];
	// $775e	lda $00,x
	regA = mem[(0x0+regX)&0xff];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7760	jsr $7785
	mem[0x0100 | regS] = 0x77;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x63;
	regS = (regS - 1) & 0xff;
	regPC = 0x7785;
	cycles += 13;
	break;

case 0x7763:
	// $7763	dec $15
	temp = (mem[0x15] - 1) & 0xff;
	mem[0x15] = temp;
	// $7765	ldx $15
	regX = mem[0x15];
	// $7767	dec $16
	temp = (mem[0x16] - 1) & 0xff;
	mem[0x16] = temp;
	flagZ = (temp == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $7769	bpl $774c
	if (flagN == 0) {
		cycles += 15;
		regPC = 0x774c;
		break;
	};
	// $776b	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 21;
	break;

case 0x776c:
	// $776c	ldy #$00
	regY = 0x0;
	// $776e	sty $0b
	mem[0xb] = regY;
	// $7770	ldy #$04
	regY = 0x4;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	cycles += 7;
	regPC = 0x7772;

case 0x7772:
	// $7772	rol $0b
	temp = (mem[0xb] << 1)  | flagC;
	flagC = temp >> 8;
	mem[0xb] = (temp & 0xff);
	// $7774	rol a
	temp = (regA << 1)  | flagC;
	regA = (temp & 0xff);
	// $7775	cmp $0c
	temp = regA - mem[0xc];
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $7777	bcc $777b
	if (flagC == 0) {
		cycles += 12;
		regPC = 0x777b;
		break;
	};
	// $7779	sbc $0c
	temp = mem[0xc];
	if (flagD == 0) {   // no decimal mode
		z = regA - temp - ((flagC == 0) ? 1 : 0);
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh == 0) ? 1 : 0;
		flagZ = (zl == 0 ? 1 : 0);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		flagN = (zl >> 7) & 1;
		regA = zl;
	}
	else {   // decimal mode
		zl = ((regA & 0x0f) - (temp & 0x0f) - ((flagC == 0) ? 1 : 0)) & 0xff;
		if (zl > 15)
			zl -= 6;
		zh = (regA >> 4) - (temp >> 4) - (zl&0x10);
		if (zh > 15)
			zh -= 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	cycles += 15;
	regPC = 0x777b;

case 0x777b:
	// $777b	dey
	regY--;
	regY&= 0xff;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $777c	bne $7772
	if (flagZ == 0) {
		cycles += 4;
		regPC = 0x7772;
		break;
	};
	// $777e	lda $0b
	regA = mem[0xb];
	// $7780	rol a
	temp = (regA << 1)  | flagC;
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $7781	and #$0f
	regA = (regA & 0xf) & 0xff;
	// $7783	tax
	regX = regA;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7784	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 19;
	break;

case 0x7785:
	// $7785	bcc $778b
	if (flagC == 0) {
		cycles += 2;
		regPC = 0x778b;
		break;
	};
	// $7787	and #$0f
	regA = (regA & 0xf) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7789	beq $77b2
	if (flagZ == 1) {
		cycles += 6;
		regPC = 0x77b2;
		break;
	};
	cycles += 6;
	regPC = 0x778b;

case 0x778b:
	// $778b	ldx $17
	regX = mem[0x17];
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $778d	beq $77b2
	if (flagZ == 1) {
		cycles += 5;
		regPC = 0x77b2;
		break;
	};
	// $778f	and #$0f
	regA = (regA & 0xf) & 0xff;
	// $7791	clc
	flagC = 0;
	// $7792	adc #$01
	temp = 0x1;
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		flagZ = (zl == 0 ? 1 : 0);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		flagN = (zl >> 7) & 1;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $7794	php
	mem[0x0100 | regS] = ((flagN<<7)|(flagV<<6)|(0x20)|(flagB<<4)|(flagD<<3)|(flagI<<2)|(flagZ<<1)|flagC);
	regS = (regS - 1) & 0xff;
	// $7795	asl a
	temp = (regA << 1);
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $7796	tay
	regY = regA;
	// $7797	lda $56d4,y
	regA = mem[0x56d4+regY];
	// $779a	asl a
	temp = (regA << 1);
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $779b	sta $0b
	mem[0xb] = regA;
	// $779d	lda $56d5,y
	regA = mem[0x56d5+regY];
	// $77a0	rol a
	temp = (regA << 1)  | flagC;
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $77a1	and #$1f
	regA = (regA & 0x1f) & 0xff;
	// $77a3	ora #$40
	regA = (regA | 0x40) & 0xff;
	// $77a5	sta $0c
	mem[0xc] = regA;
	// $77a7	lda #$00
	regA = 0x0;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $77a9	sta $08
	mem[0x8] = regA;
	// $77ab	sta $09
	mem[0x9] = regA;
	// $77ad	jsr $6ad7
	mem[0x0100 | regS] = 0x77;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xb0;
	regS = (regS - 1) & 0xff;
	regPC = 0x6ad7;
	cycles += 54;
	break;

case 0x77b0:
	// $77b0	plp
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	flagN = (temp >> 7) & 1;
	flagV = (temp >> 6) & 1;
	flagB = (temp >> 4) & 1;
	flagD = (temp >> 3) & 1;
	flagI = (temp >> 2) & 1;
	flagZ = (temp >> 1) & 1;
	flagC = temp & 1;
	// $77b1	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 10;
	break;

case 0x77b2:
	// $77b2	jmp $7bcb
	regPC = 0x7bcb;
	cycles += 3;
	break;

case 0x77b5:
	// $77b5	asl $5f
	temp = (mem[0x5f] << 1);
	flagC = temp >> 8;
	mem[0x5f] = (temp & 0xff);
	// $77b7	rol $60
	temp = (mem[0x60] << 1)  | flagC;
	flagC = temp >> 8;
	mem[0x60] = (temp & 0xff);
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = ((temp & 0xff) >> 7) & 1;
	// $77b9	bpl $77bd
	if (flagN == 0) {
		cycles += 12;
		regPC = 0x77bd;
		break;
	};
	// $77bb	inc $5f
	temp = (mem[0x5f] + 1) & 0xff;
	mem[0x5f] = temp;
	flagZ = (temp == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	cycles += 17;
	regPC = 0x77bd;

case 0x77bd:
	// $77bd	lda $5f
	regA = mem[0x5f];
	// $77bf	bit $77d1
	temp =mem[0x77d1];
	flagN = (temp >> 7);
	flagV = ((temp >> 6) & 1);
	flagZ = ((temp & regA) != 0) ? 0 : 1;
	// $77c2	beq $77c8
	if (flagZ == 1) {
		cycles += 9;
		regPC = 0x77c8;
		break;
	};
	// $77c4	eor #$01
	regA = (regA ^ 0x1) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $77c6	sta $5f
	mem[0x5f] = regA;
	cycles += 14;
	regPC = 0x77c8;

case 0x77c8:
	// $77c8	ora $60
	regA = (regA | mem[0x60]) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $77ca	bne $77ce
	if (flagZ == 0) {
		cycles += 5;
		regPC = 0x77ce;
		break;
	};
	// $77cc	inc $5f
	temp = (mem[0x5f] + 1) & 0xff;
	mem[0x5f] = temp;
	flagZ = (temp == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	cycles += 10;
	regPC = 0x77ce;

case 0x77ce:
	// $77ce	lda $5f
	regA = mem[0x5f];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $77d0	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 9;
	break;

case 0x77d1:
	// dc.b $02		; $77d1

case 0x77d2:
	// $77d2	clc
	flagC = 0;
	// $77d3	adc #$40
	temp = 0x40;
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		flagZ = (zl == 0 ? 1 : 0);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		flagN = (zl >> 7) & 1;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	cycles += 4;
	regPC = 0x77d5;

case 0x77d5:
	// $77d5	bpl $77df
	if (flagN == 0) {
		cycles += 2;
		regPC = 0x77df;
		break;
	};
	// $77d7	and #$7f
	regA = (regA & 0x7f) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $77d9	jsr $77df
	mem[0x0100 | regS] = 0x77;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xdc;
	regS = (regS - 1) & 0xff;
	regPC = 0x77df;
	cycles += 10;
	break;

case 0x77dc:
	// $77dc	jmp $7708
	regPC = 0x7708;
	cycles += 3;
	break;

case 0x77df:
	// $77df	cmp #$41
	temp = regA - 0x41;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $77e1	bcc $77e7
	if (flagC == 0) {
		cycles += 4;
		regPC = 0x77e7;
		break;
	};
	// $77e3	eor #$7f
	regA = (regA ^ 0x7f) & 0xff;
	// $77e5	adc #$00
	temp = 0x0;
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		flagZ = (zl == 0 ? 1 : 0);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		flagN = (zl >> 7) & 1;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	cycles += 8;
	regPC = 0x77e7;

case 0x77e7:
	// $77e7	tax
	regX = regA;
	// $77e8	lda $57b9,x
	regA = mem[0x57b9+regX];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $77eb	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 12;
	break;

case 0x77ec:
	// dc.b $00		; $77ec
	// dc.b $00		; $77ed
	// dc.b $00		; $77ee
	// dc.b $00		; $77ef
	// dc.b $00		; $77f0
	// dc.b $00		; $77f1
	// dc.b $00		; $77f2
	// dc.b $00		; $77f3
	// dc.b $00		; $77f4
	// dc.b $00		; $77f5

case 0x77f6:
	// $77f6	lda $2803
	regA = readMem(0x2803);
	// $77f9	and #$03
	regA = (regA & 0x3) & 0xff;
	// $77fb	asl a
	temp = (regA << 1);
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $77fc	tax
	regX = regA;
	// $77fd	lda #$10
	regA = 0x10;
	// $77ff	sta $00
	mem[0x0] = regA;
	// $7801	lda $7888,x
	regA = mem[0x7888+regX];
	// $7804	sta $09
	mem[0x9] = regA;
	// $7806	lda $7887,x
	regA = mem[0x7887+regX];
	// $7809	sta $08
	mem[0x8] = regA;
	// $780b	adc ($08),y
	temp = mem[(mem[0x8] | (mem[0x9]<<8))+regY];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		flagZ = (zl == 0 ? 1 : 0);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		flagN = (zl >> 7) & 1;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $780d	sta $08
	mem[0x8] = regA;
	// $780f	bcc $7813
	if (flagC == 0) {
		cycles += 39;
		regPC = 0x7813;
		break;
	};
	// $7811	inc $09
	temp = (mem[0x9] + 1) & 0xff;
	mem[0x9] = temp;
	flagZ = (temp == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	cycles += 44;
	regPC = 0x7813;

		};

	};

	void exec7800 () {

		switch (regPC) {

case 0x7813:
	// $7813	tya
	regA = regY;
	// $7814	asl a
	temp = (regA << 1);
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $7815	tay
	regY = regA;
	// $7816	lda $7871,y
	regA = mem[0x7871+regY];
	// $7819	ldx $7872,y
	regX = mem[0x7872+regY];
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $781c	jsr $7c03
	mem[0x0100 | regS] = 0x78;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x1f;
	regS = (regS - 1) & 0xff;
	regPC = 0x7c03;
	cycles += 20;
	break;

case 0x781f:
	// $781f	lda #$70
	regA = 0x70;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7821	jsr $7cde
	mem[0x0100 | regS] = 0x78;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x24;
	regS = (regS - 1) & 0xff;
	regPC = 0x7cde;
	cycles += 8;
	break;

case 0x7824:
	// $7824	ldy #$00
	regY = 0x0;
	// $7826	ldx #$00
	regX = 0x0;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	cycles += 4;
	regPC = 0x7828;

case 0x7828:
	// $7828	lda ($08,x)
	regA = mem[mem[(0x8+regX)&0xff] | (mem[(0x9+regX)&0xff]<<8)];
	// $782a	sta $0b
	mem[0xb] = regA;
	// $782c	lsr a
	temp = regA;
	temp >>= 1;
	regA = (temp & 0xff);
	// $782d	lsr a
	temp = regA;
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $782e	jsr $784d
	mem[0x0100 | regS] = 0x78;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x31;
	regS = (regS - 1) & 0xff;
	regPC = 0x784d;
	cycles += 19;
	break;

case 0x7831:
	// $7831	lda ($08,x)
	regA = mem[mem[(0x8+regX)&0xff] | (mem[(0x9+regX)&0xff]<<8)];
	// $7833	rol a
	temp = (regA << 1)  | flagC;
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $7834	rol $0b
	temp = (mem[0xb] << 1)  | flagC;
	flagC = temp >> 8;
	mem[0xb] = (temp & 0xff);
	// $7836	rol a
	temp = (regA << 1)  | flagC;
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $7837	lda $0b
	regA = mem[0xb];
	// $7839	rol a
	temp = (regA << 1)  | flagC;
	regA = (temp & 0xff);
	// $783a	asl a
	temp = (regA << 1);
	flagC = temp >> 8;
	regA = (temp & 0xff);
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $783b	jsr $7853
	mem[0x0100 | regS] = 0x78;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x3e;
	regS = (regS - 1) & 0xff;
	regPC = 0x7853;
	cycles += 28;
	break;

case 0x783e:
	// $783e	lda ($08,x)
	regA = mem[mem[(0x8+regX)&0xff] | (mem[(0x9+regX)&0xff]<<8)];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7840	sta $0b
	mem[0xb] = regA;
	// $7842	jsr $784d
	mem[0x0100 | regS] = 0x78;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x45;
	regS = (regS - 1) & 0xff;
	regPC = 0x784d;
	cycles += 15;
	break;

case 0x7845:
	// $7845	lsr $0b
	temp = mem[0xb];
	flagC = temp & 1;
	temp >>= 1;
	mem[0xb] = (temp & 0xff);
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = ((temp & 0xff) >> 7) & 1;
	// $7847	bcc $7828
	if (flagC == 0) {
		cycles += 7;
		regPC = 0x7828;
		break;
	};
	cycles += 7;
	regPC = 0x7849;

case 0x7849:
	// $7849	dey
	regY--;
	regY&= 0xff;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $784a	jmp $7c39
	regPC = 0x7c39;
	cycles += 5;
	break;

case 0x784d:
	// $784d	inc $08
	temp = (mem[0x8] + 1) & 0xff;
	mem[0x8] = temp;
	flagZ = (temp == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $784f	bne $7853
	if (flagZ == 0) {
		cycles += 7;
		regPC = 0x7853;
		break;
	};
	// $7851	inc $09
	temp = (mem[0x9] + 1) & 0xff;
	mem[0x9] = temp;
	flagZ = (temp == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	cycles += 12;
	regPC = 0x7853;

case 0x7853:
	// $7853	and #$3e
	regA = (regA & 0x3e) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7855	bne $785b
	if (flagZ == 0) {
		cycles += 4;
		regPC = 0x785b;
		break;
	};
	// $7857	pla
	regS = (regS + 1) & 0xff;
	regA = mem[0x0100 | regS];
	// $7858	pla
	regS = (regS + 1) & 0xff;
	regA = mem[0x0100 | regS];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7859	bne $7849
	if (flagZ == 0) {
		cycles += 14;
		regPC = 0x7849;
		break;
	};
	cycles += 14;
	regPC = 0x785b;

case 0x785b:
	// $785b	cmp #$0a
	temp = regA - 0xa;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $785d	bcc $7861
	if (flagC == 0) {
		cycles += 4;
		regPC = 0x7861;
		break;
	};
	// $785f	adc #$0d
	temp = 0xd;
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		flagZ = (zl == 0 ? 1 : 0);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		flagN = (zl >> 7) & 1;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	cycles += 6;
	regPC = 0x7861;

case 0x7861:
	// $7861	tax
	regX = regA;
	// $7862	lda $56d2,x
	regA = mem[0x56d2+regX];
	// $7865	sta ($02),y
	mem[(mem[0x2] | (mem[0x3]<<8)) + regY] = regA;
	// $7867	iny
	regY++;
	regY&= 0xff;
	// $7868	lda $56d3,x
	regA = mem[0x56d3+regX];
	// $786b	sta ($02),y
	mem[(mem[0x2] | (mem[0x3]<<8)) + regY] = regA;
	// $786d	iny
	regY++;
	regY&= 0xff;
	// $786e	ldx #$00
	regX = 0x0;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7870	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 34;
	break;

case 0x7871:
	// dc.b $64		; $7871
	// dc.b $b6		; $7872
	// dc.b $64		; $7873
	// dc.b $b6		; $7874
	// dc.b $0c		; $7875
	// dc.b $aa		; $7876
	// dc.b $0c		; $7877
	// dc.b $a2		; $7878
	// dc.b $0c		; $7879
	// dc.b $9a		; $787a
	// dc.b $0c		; $787b
	// dc.b $92		; $787c
	// dc.b $64		; $787d
	// dc.b $c6		; $787e
	// dc.b $64		; $787f
	// dc.b $9d		; $7880
	// dc.b $50		; $7881
	// dc.b $39		; $7882
	// dc.b $50		; $7883
	// dc.b $39		; $7884
	// dc.b $50		; $7885
	// dc.b $39		; $7886
	// dc.b $1e		; $7887
	// dc.b $57		; $7888
	// dc.b $8f		; $7889
	// dc.b $78		; $788a
	// dc.b $46		; $788b
	// dc.b $79		; $788c
	// dc.b $f3		; $788d
	// dc.b $79		; $788e
	// dc.b $0b		; $788f
	// dc.b $15		; $7890
	// dc.b $1b		; $7891
	// dc.b $35		; $7892
	// dc.b $4d		; $7893
	// dc.b $65		; $7894
	// dc.b $7f		; $7895
	// dc.b $8d		; $7896
	// dc.b $93		; $7897
	// dc.b $9f		; $7898
	// dc.b $ab		; $7899
	// dc.b $64		; $789a
	// dc.b $d2		; $789b
	// dc.b $3b		; $789c
	// dc.b $2e		; $789d
	// dc.b $c2		; $789e
	// dc.b $6c		; $789f
	// dc.b $5a		; $78a0
	// dc.b $4c		; $78a1
	// dc.b $93		; $78a2
	// dc.b $6f		; $78a3
	// dc.b $bd		; $78a4
	// dc.b $1a		; $78a5
	// dc.b $4c		; $78a6
	// dc.b $12		; $78a7
	// dc.b $b0		; $78a8
	// dc.b $40		; $78a9
	// dc.b $6b		; $78aa
	// dc.b $2c		; $78ab
	// dc.b $0a		; $78ac
	// dc.b $6c		; $78ad
	// dc.b $5a		; $78ae
	// dc.b $4c		; $78af
	// dc.b $93		; $78b0
	// dc.b $6e		; $78b1
	// dc.b $0b		; $78b2
	// dc.b $6e		; $78b3
	// dc.b $c0		; $78b4
	// dc.b $52		; $78b5
	// dc.b $6c		; $78b6
	// dc.b $92		; $78b7
	// dc.b $b8		; $78b8
	// dc.b $50		; $78b9
	// dc.b $4d		; $78ba
	// dc.b $82		; $78bb
	// dc.b $f2		; $78bc
	// dc.b $58		; $78bd
	// dc.b $90		; $78be
	// dc.b $4c		; $78bf
	// dc.b $4d		; $78c0
	// dc.b $f0		; $78c1
	// dc.b $4c		; $78c2
	// dc.b $80		; $78c3
	// dc.b $33		; $78c4
	// dc.b $70		; $78c5
	// dc.b $c2		; $78c6
	// dc.b $42		; $78c7
	// dc.b $5a		; $78c8
	// dc.b $4c		; $78c9
	// dc.b $4c		; $78ca
	// dc.b $82		; $78cb
	// dc.b $bb		; $78cc
	// dc.b $52		; $78cd
	// dc.b $0b		; $78ce
	// dc.b $58		; $78cf
	// dc.b $b2		; $78d0
	// dc.b $42		; $78d1
	// dc.b $6c		; $78d2
	// dc.b $9a		; $78d3
	// dc.b $c3		; $78d4
	// dc.b $4a		; $78d5
	// dc.b $82		; $78d6
	// dc.b $64		; $78d7
	// dc.b $0a		; $78d8
	// dc.b $5a		; $78d9
	// dc.b $90		; $78da
	// dc.b $00		; $78db
	// dc.b $f6		; $78dc
	// dc.b $6c		; $78dd
	// dc.b $09		; $78de
	// dc.b $b2		; $78df
	// dc.b $3b		; $78e0
	// dc.b $2e		; $78e1
	// dc.b $c1		; $78e2
	// dc.b $4c		; $78e3
	// dc.b $4c		; $78e4
	// dc.b $b6		; $78e5
	// dc.b $2b		; $78e6
	// dc.b $20		; $78e7
	// dc.b $0d		; $78e8
	// dc.b $a6		; $78e9
	// dc.b $c1		; $78ea
	// dc.b $70		; $78eb
	// dc.b $48		; $78ec
	// dc.b $50		; $78ed
	// dc.b $b6		; $78ee
	// dc.b $52		; $78ef
	// dc.b $3b		; $78f0
	// dc.b $d2		; $78f1
	// dc.b $90		; $78f2
	// dc.b $00		; $78f3
	// dc.b $da		; $78f4
	// dc.b $64		; $78f5
	// dc.b $90		; $78f6
	// dc.b $4c		; $78f7
	// dc.b $c9		; $78f8
	// dc.b $d8		; $78f9
	// dc.b $be		; $78fa
	// dc.b $0a		; $78fb
	// dc.b $32		; $78fc
	// dc.b $42		; $78fd
	// dc.b $9b		; $78fe
	// dc.b $c2		; $78ff
	// dc.b $67		; $7900
	// dc.b $68		; $7901
	// dc.b $4d		; $7902
	// dc.b $ae		; $7903
	// dc.b $a1		; $7904
	// dc.b $4e		; $7905
	// dc.b $48		; $7906
	// dc.b $50		; $7907
	// dc.b $b6		; $7908
	// dc.b $52		; $7909
	// dc.b $3b		; $790a
	// dc.b $d2		; $790b
	// dc.b $90		; $790c
	// dc.b $00		; $790d
	// dc.b $be		; $790e
	// dc.b $0a		; $790f
	// dc.b $b6		; $7910
	// dc.b $1e		; $7911
	// dc.b $94		; $7912
	// dc.b $d2		; $7913
	// dc.b $a2		; $7914
	// dc.b $92		; $7915
	// dc.b $0a		; $7916
	// dc.b $2c		; $7917
	// dc.b $ca		; $7918
	// dc.b $4e		; $7919
	// dc.b $7a		; $791a
	// dc.b $65		; $791b
	// dc.b $bd		; $791c
	// dc.b $1a		; $791d
	// dc.b $4c		; $791e
	// dc.b $12		; $791f
	// dc.b $92		; $7920
	// dc.b $13		; $7921
	// dc.b $18		; $7922
	// dc.b $62		; $7923
	// dc.b $ca		; $7924
	// dc.b $64		; $7925
	// dc.b $f2		; $7926
	// dc.b $42		; $7927
	// dc.b $20		; $7928
	// dc.b $6e		; $7929
	// dc.b $a3		; $792a
	// dc.b $52		; $792b
	// dc.b $82		; $792c
	// dc.b $40		; $792d
	// dc.b $18		; $792e
	// dc.b $62		; $792f
	// dc.b $ca		; $7930
	// dc.b $64		; $7931
	// dc.b $f2		; $7932
	// dc.b $42		; $7933
	// dc.b $18		; $7934
	// dc.b $6e		; $7935
	// dc.b $a3		; $7936
	// dc.b $52		; $7937
	// dc.b $80		; $7938
	// dc.b $00		; $7939
	// dc.b $20		; $793a
	// dc.b $62		; $793b
	// dc.b $ca		; $793c
	// dc.b $64		; $793d
	// dc.b $f2		; $793e
	// dc.b $64		; $793f
	// dc.b $08		; $7940
	// dc.b $c2		; $7941
	// dc.b $bd		; $7942
	// dc.b $1a		; $7943
	// dc.b $4c		; $7944
	// dc.b $00		; $7945
	// dc.b $0b		; $7946
	// dc.b $15		; $7947
	// dc.b $19		; $7948
	// dc.b $31		; $7949
	// dc.b $41		; $794a
	// dc.b $57		; $794b
	// dc.b $73		; $794c
	// dc.b $7f		; $794d
	// dc.b $89		; $794e
	// dc.b $95		; $794f
	// dc.b $a1		; $7950
	// dc.b $8a		; $7951
	// dc.b $5a		; $7952
	// dc.b $84		; $7953
	// dc.b $12		; $7954
	// dc.b $cd		; $7955
	// dc.b $82		; $7956
	// dc.b $b9		; $7957
	// dc.b $e6		; $7958
	// dc.b $b2		; $7959
	// dc.b $40		; $795a
	// dc.b $74		; $795b
	// dc.b $f2		; $795c
	// dc.b $4d		; $795d
	// dc.b $83		; $795e
	// dc.b $d4		; $795f
	// dc.b $f0		; $7960
	// dc.b $b2		; $7961
	// dc.b $42		; $7962
	// dc.b $b9		; $7963
	// dc.b $e6		; $7964
	// dc.b $b2		; $7965
	// dc.b $42		; $7966
	// dc.b $4d		; $7967
	// dc.b $f0		; $7968
	// dc.b $0e		; $7969
	// dc.b $64		; $796a
	// dc.b $0a		; $796b
	// dc.b $12		; $796c
	// dc.b $b8		; $796d
	// dc.b $46		; $796e
	// dc.b $10		; $796f
	// dc.b $62		; $7970
	// dc.b $4b		; $7971
	// dc.b $60		; $7972
	// dc.b $82		; $7973
	// dc.b $72		; $7974
	// dc.b $b5		; $7975
	// dc.b $c0		; $7976
	// dc.b $be		; $7977
	// dc.b $a8		; $7978
	// dc.b $0a		; $7979
	// dc.b $64		; $797a
	// dc.b $c5		; $797b
	// dc.b $92		; $797c
	// dc.b $f0		; $797d
	// dc.b $74		; $797e
	// dc.b $9d		; $797f
	// dc.b $c2		; $7980
	// dc.b $6c		; $7981
	// dc.b $9a		; $7982
	// dc.b $c3		; $7983
	// dc.b $4a		; $7984
	// dc.b $82		; $7985
	// dc.b $6f		; $7986
	// dc.b $a4		; $7987
	// dc.b $f2		; $7988
	// dc.b $bd		; $7989
	// dc.b $d2		; $798a
	// dc.b $f0		; $798b
	// dc.b $6c		; $798c
	// dc.b $9e		; $798d
	// dc.b $0a		; $798e
	// dc.b $c2		; $798f
	// dc.b $42		; $7990
	// dc.b $a4		; $7991
	// dc.b $f2		; $7992
	// dc.b $b0		; $7993
	// dc.b $74		; $7994
	// dc.b $9d		; $7995
	// dc.b $c2		; $7996
	// dc.b $6c		; $7997
	// dc.b $9a		; $7998
	// dc.b $c3		; $7999
	// dc.b $4a		; $799a
	// dc.b $82		; $799b
	// dc.b $6f		; $799c
	// dc.b $a4		; $799d
	// dc.b $f2		; $799e
	// dc.b $bd		; $799f
	// dc.b $d2		; $79a0
	// dc.b $f0		; $79a1
	// dc.b $58		; $79a2
	// dc.b $ed		; $79a3
	// dc.b $12		; $79a4
	// dc.b $b5		; $79a5
	// dc.b $e8		; $79a6
	// dc.b $29		; $79a7
	// dc.b $d2		; $79a8
	// dc.b $0d		; $79a9
	// dc.b $72		; $79aa
	// dc.b $2c		; $79ab
	// dc.b $90		; $79ac
	// dc.b $0c		; $79ad
	// dc.b $12		; $79ae
	// dc.b $c6		; $79af
	// dc.b $2c		; $79b0
	// dc.b $48		; $79b1
	// dc.b $4e		; $79b2
	// dc.b $9d		; $79b3
	// dc.b $ac		; $79b4
	// dc.b $49		; $79b5
	// dc.b $f0		; $79b6
	// dc.b $48		; $79b7
	// dc.b $00		; $79b8
	// dc.b $2d		; $79b9
	// dc.b $28		; $79ba
	// dc.b $cf		; $79bb
	// dc.b $52		; $79bc
	// dc.b $b0		; $79bd
	// dc.b $6e		; $79be
	// dc.b $cd		; $79bf
	// dc.b $82		; $79c0
	// dc.b $be		; $79c1
	// dc.b $0a		; $79c2
	// dc.b $b6		; $79c3
	// dc.b $00		; $79c4
	// dc.b $53		; $79c5
	// dc.b $64		; $79c6
	// dc.b $0a		; $79c7
	// dc.b $12		; $79c8
	// dc.b $0d		; $79c9
	// dc.b $0a		; $79ca
	// dc.b $b6		; $79cb
	// dc.b $1a		; $79cc
	// dc.b $48		; $79cd
	// dc.b $00		; $79ce
	// dc.b $18		; $79cf
	// dc.b $68		; $79d0
	// dc.b $6a		; $79d1
	// dc.b $4e		; $79d2
	// dc.b $48		; $79d3
	// dc.b $48		; $79d4
	// dc.b $0b		; $79d5
	// dc.b $a6		; $79d6
	// dc.b $ca		; $79d7
	// dc.b $72		; $79d8
	// dc.b $b5		; $79d9
	// dc.b $c0		; $79da
	// dc.b $18		; $79db
	// dc.b $68		; $79dc
	// dc.b $6a		; $79dd
	// dc.b $4e		; $79de
	// dc.b $48		; $79df
	// dc.b $46		; $79e0
	// dc.b $0b		; $79e1
	// dc.b $a6		; $79e2
	// dc.b $ca		; $79e3
	// dc.b $72		; $79e4
	// dc.b $b0		; $79e5
	// dc.b $00		; $79e6
	// dc.b $20		; $79e7
	// dc.b $68		; $79e8
	// dc.b $6a		; $79e9
	// dc.b $4e		; $79ea
	// dc.b $4d		; $79eb
	// dc.b $c2		; $79ec
	// dc.b $18		; $79ed
	// dc.b $5c		; $79ee
	// dc.b $9e		; $79ef
	// dc.b $52		; $79f0
	// dc.b $cd		; $79f1
	// dc.b $80		; $79f2
	// dc.b $0b		; $79f3
	// dc.b $11		; $79f4
	// dc.b $17		; $79f5
	// dc.b $31		; $79f6
	// dc.b $45		; $79f7
	// dc.b $5f		; $79f8
	// dc.b $6b		; $79f9
	// dc.b $73		; $79fa
	// dc.b $7d		; $79fb
	// dc.b $89		; $79fc
	// dc.b $93		; $79fd
	// dc.b $b2		; $79fe
	// dc.b $4e		; $79ff
	// dc.b $9d		; $7a00
	// dc.b $90		; $7a01
	// dc.b $b8		; $7a02
	// dc.b $00		; $7a03
	// dc.b $76		; $7a04
	// dc.b $56		; $7a05
	// dc.b $2a		; $7a06
	// dc.b $26		; $7a07
	// dc.b $b0		; $7a08
	// dc.b $40		; $7a09
	// dc.b $be		; $7a0a
	// dc.b $42		; $7a0b
	// dc.b $a6		; $7a0c
	// dc.b $64		; $7a0d
	// dc.b $c1		; $7a0e
	// dc.b $5c		; $7a0f
	// dc.b $48		; $7a10
	// dc.b $52		; $7a11
	// dc.b $be		; $7a12
	// dc.b $0a		; $7a13
	// dc.b $0a		; $7a14
	// dc.b $64		; $7a15
	// dc.b $c5		; $7a16
	// dc.b $92		; $7a17
	// dc.b $0c		; $7a18
	// dc.b $26		; $7a19
	// dc.b $b8		; $7a1a
	// dc.b $50		; $7a1b
	// dc.b $6a		; $7a1c
	// dc.b $7c		; $7a1d
	// dc.b $0c		; $7a1e
	// dc.b $52		; $7a1f
	// dc.b $74		; $7a20
	// dc.b $ec		; $7a21
	// dc.b $4d		; $7a22
	// dc.b $c0		; $7a23
	// dc.b $a4		; $7a24
	// dc.b $ec		; $7a25
	// dc.b $0a		; $7a26
	// dc.b $8a		; $7a27
	// dc.b $d4		; $7a28
	// dc.b $ec		; $7a29
	// dc.b $0a		; $7a2a
	// dc.b $64		; $7a2b
	// dc.b $c5		; $7a2c
	// dc.b $92		; $7a2d
	// dc.b $0d		; $7a2e
	// dc.b $f2		; $7a2f
	// dc.b $b8		; $7a30
	// dc.b $5a		; $7a31
	// dc.b $93		; $7a32
	// dc.b $4e		; $7a33
	// dc.b $69		; $7a34
	// dc.b $60		; $7a35
	// dc.b $4d		; $7a36
	// dc.b $c0		; $7a37
	// dc.b $9d		; $7a38
	// dc.b $2c		; $7a39
	// dc.b $6c		; $7a3a
	// dc.b $4a		; $7a3b
	// dc.b $0d		; $7a3c
	// dc.b $a6		; $7a3d
	// dc.b $c1		; $7a3e
	// dc.b $70		; $7a3f
	// dc.b $48		; $7a40
	// dc.b $68		; $7a41
	// dc.b $2d		; $7a42
	// dc.b $8a		; $7a43
	// dc.b $0d		; $7a44
	// dc.b $d2		; $7a45
	// dc.b $82		; $7a46
	// dc.b $4e		; $7a47
	// dc.b $3b		; $7a48
	// dc.b $66		; $7a49
	// dc.b $91		; $7a4a
	// dc.b $6c		; $7a4b
	// dc.b $0c		; $7a4c
	// dc.b $0a		; $7a4d
	// dc.b $0c		; $7a4e
	// dc.b $12		; $7a4f
	// dc.b $c5		; $7a50
	// dc.b $8b		; $7a51
	// dc.b $9d		; $7a52
	// dc.b $2c		; $7a53
	// dc.b $6c		; $7a54
	// dc.b $4a		; $7a55
	// dc.b $0b		; $7a56
	// dc.b $3a		; $7a57
	// dc.b $a2		; $7a58
	// dc.b $6c		; $7a59
	// dc.b $bd		; $7a5a
	// dc.b $0a		; $7a5b
	// dc.b $3a		; $7a5c
	// dc.b $40		; $7a5d
	// dc.b $a6		; $7a5e
	// dc.b $60		; $7a5f
	// dc.b $b9		; $7a60
	// dc.b $6c		; $7a61
	// dc.b $0d		; $7a62
	// dc.b $f0		; $7a63
	// dc.b $2d		; $7a64
	// dc.b $b1		; $7a65
	// dc.b $76		; $7a66
	// dc.b $52		; $7a67
	// dc.b $5c		; $7a68
	// dc.b $c2		; $7a69
	// dc.b $c2		; $7a6a
	// dc.b $6c		; $7a6b
	// dc.b $8b		; $7a6c
	// dc.b $64		; $7a6d
	// dc.b $2a		; $7a6e
	// dc.b $27		; $7a6f
	// dc.b $18		; $7a70
	// dc.b $54		; $7a71
	// dc.b $69		; $7a72
	// dc.b $d8		; $7a73
	// dc.b $28		; $7a74
	// dc.b $48		; $7a75
	// dc.b $0b		; $7a76
	// dc.b $b2		; $7a77
	// dc.b $4a		; $7a78
	// dc.b $e6		; $7a79
	// dc.b $b8		; $7a7a
	// dc.b $00		; $7a7b
	// dc.b $18		; $7a7c
	// dc.b $54		; $7a7d
	// dc.b $69		; $7a7e
	// dc.b $d8		; $7a7f
	// dc.b $28		; $7a80
	// dc.b $46		; $7a81
	// dc.b $0b		; $7a82
	// dc.b $b2		; $7a83
	// dc.b $4a		; $7a84
	// dc.b $e7		; $7a85
	// dc.b $20		; $7a86
	// dc.b $54		; $7a87
	// dc.b $69		; $7a88
	// dc.b $d8		; $7a89
	// dc.b $2d		; $7a8a
	// dc.b $c2		; $7a8b
	// dc.b $18		; $7a8c
	// dc.b $5c		; $7a8d
	// dc.b $ca		; $7a8e
	// dc.b $56		; $7a8f
	// dc.b $98		; $7a90
	// dc.b $00		; $7a91
	// dc.b $52		; $7a92

case 0x7a93:
	// $7a93	ldx #$02
	regX = 0x2;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	cycles += 2;
	regPC = 0x7a95;

case 0x7a95:
	// $7a95	lda $2400,x
	regA = readMem(0x2400+regX);
	// $7a98	asl a
	temp = (regA << 1);
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $7a99	lda $7a,x
	regA = mem[(0x7a+regX)&0xff];
	// $7a9b	and #$1f
	regA = (regA & 0x1f) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7a9d	bcc $7ad6
	if (flagC == 0) {
		cycles += 14;
		regPC = 0x7ad6;
		break;
	};
	// $7a9f	beq $7ab1
	if (flagZ == 1) {
		cycles += 16;
		regPC = 0x7ab1;
		break;
	};
	// $7aa1	cmp #$1b
	temp = regA - 0x1b;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $7aa3	bcs $7aaf
	if (flagC == 1) {
		cycles += 20;
		regPC = 0x7aaf;
		break;
	};
	// $7aa5	tay
	regY = regA;
	// $7aa6	lda $5e
	regA = mem[0x5e];
	// $7aa8	and #$07
	regA = (regA & 0x7) & 0xff;
	// $7aaa	cmp #$07
	temp = regA - 0x7;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	// $7aac	tya
	regA = regY;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7aad	bcc $7ab1
	if (flagC == 0) {
		cycles += 33;
		regPC = 0x7ab1;
		break;
	};
	cycles += 33;
	regPC = 0x7aaf;

case 0x7aaf:
	// $7aaf	sbc #$01
	temp = 0x1;
	if (flagD == 0) {   // no decimal mode
		z = regA - temp - ((flagC == 0) ? 1 : 0);
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh == 0) ? 1 : 0;
		flagZ = (zl == 0 ? 1 : 0);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		flagN = (zl >> 7) & 1;
		regA = zl;
	}
	else {   // decimal mode
		zl = ((regA & 0x0f) - (temp & 0x0f) - ((flagC == 0) ? 1 : 0)) & 0xff;
		if (zl > 15)
			zl -= 6;
		zh = (regA >> 4) - (temp >> 4) - (zl&0x10);
		if (zh > 15)
			zh -= 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	cycles += 2;
	regPC = 0x7ab1;

case 0x7ab1:
	// $7ab1	sta $7a,x
	mem[(0x7a+regX)&0xff] = regA;
	// $7ab3	lda $2006
	regA = readMem(0x2006);
	// $7ab6	and #$80
	regA = (regA & 0x80) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7ab8	beq $7abe
	if (flagZ == 1) {
		cycles += 12;
		regPC = 0x7abe;
		break;
	};
	// $7aba	lda #$f0
	regA = 0xf0;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7abc	sta $72
	mem[0x72] = regA;
	cycles += 17;
	regPC = 0x7abe;

case 0x7abe:
	// $7abe	lda $72
	regA = mem[0x72];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7ac0	beq $7aca
	if (flagZ == 1) {
		cycles += 5;
		regPC = 0x7aca;
		break;
	};
	// $7ac2	dec $72
	temp = (mem[0x72] - 1) & 0xff;
	mem[0x72] = temp;
	// $7ac4	lda #$00
	regA = 0x0;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7ac6	sta $7a,x
	mem[(0x7a+regX)&0xff] = regA;
	// $7ac8	sta $77,x
	mem[(0x77+regX)&0xff] = regA;
	cycles += 20;
	regPC = 0x7aca;

case 0x7aca:
	// $7aca	clc
	flagC = 0;
	// $7acb	lda $77,x
	regA = mem[(0x77+regX)&0xff];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7acd	beq $7af2
	if (flagZ == 1) {
		cycles += 8;
		regPC = 0x7af2;
		break;
	};
	// $7acf	dec $77,x
	temp = (mem[(0x77+regX)&0xff] - 1) & 0xff;
	mem[(0x77+regX)&0xff] = temp;
	flagZ = (temp == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $7ad1	bne $7af2
	if (flagZ == 0) {
		cycles += 16;
		regPC = 0x7af2;
		break;
	};
	// $7ad3	sec
	flagC = 1;
	// $7ad4	bcs $7af2
	if (flagC == 1) {
		cycles += 20;
		regPC = 0x7af2;
		break;
	};
	cycles += 20;
	regPC = 0x7ad6;

case 0x7ad6:
	// $7ad6	cmp #$1b
	temp = regA - 0x1b;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $7ad8	bcs $7ae3
	if (flagC == 1) {
		cycles += 4;
		regPC = 0x7ae3;
		break;
	};
	// $7ada	lda $7a,x
	regA = mem[(0x7a+regX)&0xff];
	// $7adc	adc #$20
	temp = 0x20;
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		flagZ = (zl == 0 ? 1 : 0);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		flagN = (zl >> 7) & 1;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $7ade	bcc $7ab1
	if (flagC == 0) {
		cycles += 12;
		regPC = 0x7ab1;
		break;
	};
	// $7ae0	beq $7ae3
	if (flagZ == 1) {
		cycles += 14;
		regPC = 0x7ae3;
		break;
	};
	// $7ae2	clc
	flagC = 0;
	cycles += 16;
	regPC = 0x7ae3;

case 0x7ae3:
	// $7ae3	lda #$1f
	regA = 0x1f;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7ae5	bcs $7ab1
	if (flagC == 1) {
		cycles += 4;
		regPC = 0x7ab1;
		break;
	};
	// $7ae7	sta $7a,x
	mem[(0x7a+regX)&0xff] = regA;
	// $7ae9	lda $77,x
	regA = mem[(0x77+regX)&0xff];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7aeb	beq $7aee
	if (flagZ == 1) {
		cycles += 14;
		regPC = 0x7aee;
		break;
	};
	// $7aed	sec
	flagC = 1;
	cycles += 16;
	regPC = 0x7aee;

case 0x7aee:
	// $7aee	lda #$78
	regA = 0x78;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7af0	sta $77,x
	mem[(0x77+regX)&0xff] = regA;
	cycles += 6;
	regPC = 0x7af2;

case 0x7af2:
	// $7af2	bcc $7b17
	if (flagC == 0) {
		cycles += 2;
		regPC = 0x7b17;
		break;
	};
	// $7af4	lda #$00
	regA = 0x0;
	// $7af6	cpx #$01
	temp = regX - 0x1;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $7af8	bcc $7b10
	if (flagC == 0) {
		cycles += 8;
		regPC = 0x7b10;
		break;
	};
	// $7afa	beq $7b08
	if (flagZ == 1) {
		cycles += 10;
		regPC = 0x7b08;
		break;
	};
	// $7afc	lda $71
	regA = mem[0x71];
	// $7afe	and #$0c
	regA = (regA & 0xc) & 0xff;
	// $7b00	lsr a
	temp = regA;
	temp >>= 1;
	regA = (temp & 0xff);
	// $7b01	lsr a
	temp = regA;
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7b02	beq $7b10
	if (flagZ == 1) {
		cycles += 21;
		regPC = 0x7b10;
		break;
	};
	// $7b04	adc #$02
	temp = 0x2;
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		flagZ = (zl == 0 ? 1 : 0);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		flagN = (zl >> 7) & 1;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $7b06	bne $7b10
	if (flagZ == 0) {
		cycles += 25;
		regPC = 0x7b10;
		break;
	};
	cycles += 25;
	regPC = 0x7b08;

case 0x7b08:
	// $7b08	lda $71
	regA = mem[0x71];
	// $7b0a	and #$10
	regA = (regA & 0x10) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7b0c	beq $7b10
	if (flagZ == 1) {
		cycles += 7;
		regPC = 0x7b10;
		break;
	};
	// $7b0e	lda #$01
	regA = 0x1;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 9;
	regPC = 0x7b10;

case 0x7b10:
	// $7b10	sec
	flagC = 1;
	// $7b11	adc $73
	temp = mem[0x73];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $7b13	sta $73
	mem[0x73] = regA;
	// $7b15	inc $74,x
	temp = (mem[(0x74+regX)&0xff] + 1) & 0xff;
	mem[(0x74+regX)&0xff] = temp;
	flagZ = (temp == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	cycles += 14;
	regPC = 0x7b17;

case 0x7b17:
	// $7b17	dex
	regX--;
	regX&= 0xff;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7b18	bmi $7b1d
	if (flagN == 1) {
		cycles += 4;
		regPC = 0x7b1d;
		break;
	};
	// $7b1a	jmp $7a95
	regPC = 0x7a95;
	cycles += 7;
	break;

case 0x7b1d:
	// $7b1d	lda $71
	regA = mem[0x71];
	// $7b1f	and #$03
	regA = (regA & 0x3) & 0xff;
	// $7b21	tay
	regY = regA;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $7b22	beq $7b36
	if (flagZ == 1) {
		cycles += 9;
		regPC = 0x7b36;
		break;
	};
	// $7b24	lsr a
	temp = regA;
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	// $7b25	adc #$00
	temp = 0x0;
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
	};
	// $7b27	eor #$ff
	regA = (regA ^ 0xff) & 0xff;
	// $7b29	sec
	flagC = 1;
	// $7b2a	adc $73
	temp = mem[0x73];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		flagZ = (zl == 0 ? 1 : 0);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		flagN = (zl >> 7) & 1;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $7b2c	bcc $7b38
	if (flagC == 0) {
		cycles += 22;
		regPC = 0x7b38;
		break;
	};
	// $7b2e	cpy #$02
	temp = regY - 0x2;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $7b30	bcs $7b34
	if (flagC == 1) {
		cycles += 26;
		regPC = 0x7b34;
		break;
	};
	// $7b32	inc $70
	temp = (mem[0x70] + 1) & 0xff;
	mem[0x70] = temp;
	flagZ = (temp == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	cycles += 31;
	regPC = 0x7b34;

case 0x7b34:
	// $7b34	inc $70
	temp = (mem[0x70] + 1) & 0xff;
	mem[0x70] = temp;
	flagZ = (temp == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	cycles += 5;
	regPC = 0x7b36;

case 0x7b36:
	// $7b36	sta $73
	mem[0x73] = regA;
	cycles += 3;
	regPC = 0x7b38;

case 0x7b38:
	// $7b38	lda $5e
	regA = mem[0x5e];
	// $7b3a	lsr a
	temp = regA;
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7b3b	bcs $7b64
	if (flagC == 1) {
		cycles += 7;
		regPC = 0x7b64;
		break;
	};
	// $7b3d	ldy #$00
	regY = 0x0;
	// $7b3f	ldx #$02
	regX = 0x2;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	cycles += 11;
	regPC = 0x7b41;

case 0x7b41:
	// $7b41	lda $74,x
	regA = mem[(0x74+regX)&0xff];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7b43	beq $7b4e
	if (flagZ == 1) {
		cycles += 6;
		regPC = 0x7b4e;
		break;
	};
	// $7b45	cmp #$10
	temp = regA - 0x10;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $7b47	bcc $7b4e
	if (flagC == 0) {
		cycles += 10;
		regPC = 0x7b4e;
		break;
	};
	// $7b49	adc #$ef
	temp = 0xef;
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $7b4b	iny
	regY++;
	regY&= 0xff;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $7b4c	sta $74,x
	mem[(0x74+regX)&0xff] = regA;
	cycles += 18;
	regPC = 0x7b4e;

case 0x7b4e:
	// $7b4e	dex
	regX--;
	regX&= 0xff;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7b4f	bpl $7b41
	if (flagN == 0) {
		cycles += 4;
		regPC = 0x7b41;
		break;
	};
	// $7b51	tya
	regA = regY;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7b52	bne $7b64
	if (flagZ == 0) {
		cycles += 8;
		regPC = 0x7b64;
		break;
	};
	// $7b54	ldx #$02
	regX = 0x2;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	cycles += 10;
	regPC = 0x7b56;

case 0x7b56:
	// $7b56	lda $74,x
	regA = mem[(0x74+regX)&0xff];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7b58	beq $7b61
	if (flagZ == 1) {
		cycles += 6;
		regPC = 0x7b61;
		break;
	};
	// $7b5a	clc
	flagC = 0;
	// $7b5b	adc #$ef
	temp = 0xef;
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		flagZ = (zl == 0 ? 1 : 0);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		flagN = (zl >> 7) & 1;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $7b5d	sta $74,x
	mem[(0x74+regX)&0xff] = regA;
	// $7b5f	bmi $7b64
	if (flagN == 1) {
		cycles += 16;
		regPC = 0x7b64;
		break;
	};
	cycles += 16;
	regPC = 0x7b61;

case 0x7b61:
	// $7b61	dex
	regX--;
	regX&= 0xff;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7b62	bpl $7b56
	if (flagN == 0) {
		cycles += 4;
		regPC = 0x7b56;
		break;
	};
	cycles += 4;
	regPC = 0x7b64;

case 0x7b64:
	// $7b64	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 6;
	break;

case 0x7b65:
	// $7b65	pha
	mem[0x0100 | regS] = regA;
	regS = (regS - 1) & 0xff;
	// $7b66	tya
	regA = regY;
	// $7b67	pha
	mem[0x0100 | regS] = regA;
	regS = (regS - 1) & 0xff;
	// $7b68	txa
	regA = regX;
	// $7b69	pha
	mem[0x0100 | regS] = regA;
	regS = (regS - 1) & 0xff;
	// $7b6a	cld
	flagD = 0;
	// $7b6b	lda $01ff
	regA = mem[0x1ff];
	// $7b6e	ora $01d0
	regA = (regA | mem[0x1d0]) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 23;
	regPC = 0x7b71;

case 0x7b71:
	// $7b71	bne $7b71
	if (flagZ == 0) {
		cycles += 2;
		regPC = 0x7b71;
		break;
	};
	// $7b73	inc $5e
	temp = (mem[0x5e] + 1) & 0xff;
	mem[0x5e] = temp;
	// $7b75	lda $5e
	regA = mem[0x5e];
	// $7b77	and #$03
	regA = (regA & 0x3) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7b79	bne $7b83
	if (flagZ == 0) {
		cycles += 14;
		regPC = 0x7b83;
		break;
	};
	// $7b7b	inc $5b
	temp = (mem[0x5b] + 1) & 0xff;
	mem[0x5b] = temp;
	// $7b7d	lda $5b
	regA = mem[0x5b];
	// $7b7f	cmp #$04
	temp = regA - 0x4;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	cycles += 24;
	regPC = 0x7b81;

case 0x7b81:
	// $7b81	bcs $7b81
	if (flagC == 1) {
		cycles += 2;
		regPC = 0x7b81;
		break;
	};
	cycles += 2;
	regPC = 0x7b83;

case 0x7b83:
	// $7b83	jsr $7a93
	mem[0x0100 | regS] = 0x7b;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x86;
	regS = (regS - 1) & 0xff;
	regPC = 0x7a93;
	cycles += 6;
	break;

case 0x7b86:
	// $7b86	lda $6f
	regA = mem[0x6f];
	// $7b88	and #$c7
	regA = (regA & 0xc7) & 0xff;
	// $7b8a	bit $74
	temp =mem[0x74];
	flagN = (temp >> 7);
	flagV = ((temp >> 6) & 1);
	flagZ = ((temp & regA) != 0) ? 0 : 1;
	// $7b8c	bpl $7b90
	if (flagN == 0) {
		cycles += 10;
		regPC = 0x7b90;
		break;
	};
	// $7b8e	ora #$08
	regA = (regA | 0x8) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 12;
	regPC = 0x7b90;

case 0x7b90:
	// $7b90	bit $75
	temp =mem[0x75];
	flagN = (temp >> 7);
	flagV = ((temp >> 6) & 1);
	flagZ = ((temp & regA) != 0) ? 0 : 1;
	// $7b92	bpl $7b96
	if (flagN == 0) {
		cycles += 5;
		regPC = 0x7b96;
		break;
	};
	// $7b94	ora #$10
	regA = (regA | 0x10) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 7;
	regPC = 0x7b96;

case 0x7b96:
	// $7b96	bit $76
	temp =mem[0x76];
	flagN = (temp >> 7);
	flagV = ((temp >> 6) & 1);
	flagZ = ((temp & regA) != 0) ? 0 : 1;
	// $7b98	bpl $7b9c
	if (flagN == 0) {
		cycles += 5;
		regPC = 0x7b9c;
		break;
	};
	// $7b9a	ora #$20
	regA = (regA | 0x20) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 7;
	regPC = 0x7b9c;

case 0x7b9c:
	// $7b9c	sta $6f
	mem[0x6f] = regA;
	// $7b9e	sta $3200
	writeMem(0x3200, regA);
	// $7ba1	lda $72
	regA = mem[0x72];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7ba3	beq $7ba9
	if (flagZ == 1) {
		cycles += 12;
		regPC = 0x7ba9;
		break;
	};
	// $7ba5	lda #$80
	regA = 0x80;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7ba7	bne $7bb7
	if (flagZ == 0) {
		cycles += 16;
		regPC = 0x7bb7;
		break;
	};
	cycles += 16;
	regPC = 0x7ba9;

case 0x7ba9:
	// $7ba9	lda $68
	regA = mem[0x68];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7bab	beq $7bb7
	if (flagZ == 1) {
		cycles += 5;
		regPC = 0x7bb7;
		break;
	};
	// $7bad	lda $5c
	regA = mem[0x5c];
	// $7baf	ror a
	temp = regA | (flagC << 8);
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7bb0	bcc $7bb4
	if (flagC == 0) {
		cycles += 12;
		regPC = 0x7bb4;
		break;
	};
	// $7bb2	dec $68
	temp = (mem[0x68] - 1) & 0xff;
	mem[0x68] = temp;
	flagZ = (temp == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	cycles += 17;
	regPC = 0x7bb4;

case 0x7bb4:
	// $7bb4	ror a
	temp = regA | (flagC << 8);
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	// $7bb5	ror a
	temp = regA | (flagC << 8);
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	// $7bb6	ror a
	temp = regA | (flagC << 8);
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 6;
	regPC = 0x7bb7;

case 0x7bb7:
	// $7bb7	sta $3c05
	writeMem(0x3c05, regA);
	// $7bba	pla
	regS = (regS + 1) & 0xff;
	regA = mem[0x0100 | regS];
	// $7bbb	tax
	regX = regA;
	// $7bbc	pla
	regS = (regS + 1) & 0xff;
	regA = mem[0x0100 | regS];
	// $7bbd	tay
	regY = regA;
	// $7bbe	pla
	regS = (regS + 1) & 0xff;
	regA = mem[0x0100 | regS];
	// $7bbf	rti
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	flagN = (temp >> 7) & 1;
	flagV = (temp >> 6) & 1;
	flagB = (temp >> 4) & 1;
	flagD = (temp >> 3) & 1;
	flagI = (temp >> 2) & 1;
	flagZ = (temp >> 1) & 1;
	flagC = temp & 1;
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 26;
	break;

case 0x7bc0:
	// $7bc0	lda #$b0
	regA = 0xb0;
	// $7bc2	ldy #$00
	regY = 0x0;
	// $7bc4	sta ($02),y
	mem[(mem[0x2] | (mem[0x3]<<8)) + regY] = regA;
	// $7bc6	iny
	regY++;
	regY&= 0xff;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $7bc7	sta ($02),y
	mem[(mem[0x2] | (mem[0x3]<<8)) + regY] = regA;
	// $7bc9	bne $7c39
	if (flagZ == 0) {
		cycles += 20;
		regPC = 0x7c39;
		break;
	};
	cycles += 20;
	regPC = 0x7bcb;

case 0x7bcb:
	// $7bcb	bcc $7bd1
	if (flagC == 0) {
		cycles += 2;
		regPC = 0x7bd1;
		break;
	};
	// $7bcd	and #$0f
	regA = (regA & 0xf) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7bcf	beq $7bd6
	if (flagZ == 1) {
		cycles += 6;
		regPC = 0x7bd6;
		break;
	};
	cycles += 6;
	regPC = 0x7bd1;

case 0x7bd1:
	// $7bd1	and #$0f
	regA = (regA & 0xf) & 0xff;
	// $7bd3	clc
	flagC = 0;
	// $7bd4	adc #$01
	temp = 0x1;
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		flagZ = (zl == 0 ? 1 : 0);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		flagN = (zl >> 7) & 1;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	cycles += 6;
	regPC = 0x7bd6;

case 0x7bd6:
	// $7bd6	php
	mem[0x0100 | regS] = ((flagN<<7)|(flagV<<6)|(0x20)|(flagB<<4)|(flagD<<3)|(flagI<<2)|(flagZ<<1)|flagC);
	regS = (regS - 1) & 0xff;
	// $7bd7	asl a
	temp = (regA << 1);
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $7bd8	ldy #$00
	regY = 0x0;
	// $7bda	tax
	regX = regA;
	// $7bdb	lda $56d4,x
	regA = mem[0x56d4+regX];
	// $7bde	sta ($02),y
	mem[(mem[0x2] | (mem[0x3]<<8)) + regY] = regA;
	// $7be0	lda $56d5,x
	regA = mem[0x56d5+regX];
	// $7be3	iny
	regY++;
	regY&= 0xff;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $7be4	sta ($02),y
	mem[(mem[0x2] | (mem[0x3]<<8)) + regY] = regA;
	// $7be6	jsr $7c39
	mem[0x0100 | regS] = 0x7b;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xe9;
	regS = (regS - 1) & 0xff;
	regPC = 0x7c39;
	cycles += 37;
	break;

case 0x7be9:
	// $7be9	plp
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	flagN = (temp >> 7) & 1;
	flagV = (temp >> 6) & 1;
	flagB = (temp >> 4) & 1;
	flagD = (temp >> 3) & 1;
	flagI = (temp >> 2) & 1;
	flagZ = (temp >> 1) & 1;
	flagC = temp & 1;
	// $7bea	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 10;
	break;

case 0x7beb:
	// dc.b $4a		; $7beb
	// dc.b $29		; $7bec
	// dc.b $0f		; $7bed
	// dc.b $09		; $7bee
	// dc.b $e0		; $7bef

case 0x7bf0:
	// $7bf0	ldy #$01
	regY = 0x1;
	// $7bf2	sta ($02),y
	mem[(mem[0x2] | (mem[0x3]<<8)) + regY] = regA;
	// $7bf4	dey
	regY--;
	regY&= 0xff;
	// $7bf5	txa
	regA = regX;
	// $7bf6	ror a
	temp = regA | (flagC << 8);
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	// $7bf7	sta ($02),y
	mem[(mem[0x2] | (mem[0x3]<<8)) + regY] = regA;
	// $7bf9	iny
	regY++;
	regY&= 0xff;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $7bfa	bne $7c39
	if (flagZ == 0) {
		cycles += 24;
		regPC = 0x7c39;
		break;
	};
	cycles += 24;
	regPC = 0x7bfc;

case 0x7bfc:
	// $7bfc	lsr a
	temp = regA;
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	// $7bfd	and #$0f
	regA = (regA & 0xf) & 0xff;
	// $7bff	ora #$c0
	regA = (regA | 0xc0) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7c01	bne $7bf0
	if (flagZ == 0) {
		cycles += 8;
		regPC = 0x7bf0;
		break;
	};
	cycles += 8;
	regPC = 0x7c03;

		};

	};

	void exec7c00 () {

		switch (regPC) {

case 0x7c03:
	// $7c03	ldy #$00
	regY = 0x0;
	// $7c05	sty $05
	mem[0x5] = regY;
	// $7c07	sty $07
	mem[0x7] = regY;
	// $7c09	asl a
	temp = (regA << 1);
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $7c0a	rol $05
	temp = (mem[0x5] << 1)  | flagC;
	mem[0x5] = (temp & 0xff);
	// $7c0c	asl a
	temp = (regA << 1);
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $7c0d	rol $05
	temp = (mem[0x5] << 1)  | flagC;
	mem[0x5] = (temp & 0xff);
	// $7c0f	sta $04
	mem[0x4] = regA;
	// $7c11	txa
	regA = regX;
	// $7c12	asl a
	temp = (regA << 1);
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $7c13	rol $07
	temp = (mem[0x7] << 1)  | flagC;
	mem[0x7] = (temp & 0xff);
	// $7c15	asl a
	temp = (regA << 1);
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $7c16	rol $07
	temp = (mem[0x7] << 1)  | flagC;
	flagC = temp >> 8;
	mem[0x7] = (temp & 0xff);
	// $7c18	sta $06
	mem[0x6] = regA;
	// $7c1a	ldx #$04
	regX = 0x4;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	cycles += 46;
	regPC = 0x7c1c;

case 0x7c1c:
	// $7c1c	lda $02,x
	regA = mem[(0x2+regX)&0xff];
	// $7c1e	ldy #$00
	regY = 0x0;
	// $7c20	sta ($02),y
	mem[(mem[0x2] | (mem[0x3]<<8)) + regY] = regA;
	// $7c22	lda $03,x
	regA = mem[(0x3+regX)&0xff];
	// $7c24	and #$0f
	regA = (regA & 0xf) & 0xff;
	// $7c26	ora #$a0
	regA = (regA | 0xa0) & 0xff;
	// $7c28	iny
	regY++;
	regY&= 0xff;
	// $7c29	sta ($02),y
	mem[(mem[0x2] | (mem[0x3]<<8)) + regY] = regA;
	// $7c2b	lda $00,x
	regA = mem[(0x0+regX)&0xff];
	// $7c2d	iny
	regY++;
	regY&= 0xff;
	// $7c2e	sta ($02),y
	mem[(mem[0x2] | (mem[0x3]<<8)) + regY] = regA;
	// $7c30	lda $01,x
	regA = mem[(0x1+regX)&0xff];
	// $7c32	and #$0f
	regA = (regA & 0xf) & 0xff;
	// $7c34	ora $00
	regA = (regA | mem[0x0]) & 0xff;
	// $7c36	iny
	regY++;
	regY&= 0xff;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $7c37	sta ($02),y
	mem[(mem[0x2] | (mem[0x3]<<8)) + regY] = regA;
	cycles += 57;
	regPC = 0x7c39;

case 0x7c39:
	// $7c39	tya
	regA = regY;
	// $7c3a	sec
	flagC = 1;
	// $7c3b	adc $02
	temp = mem[0x2];
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		flagZ = (zl == 0 ? 1 : 0);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		flagN = (zl >> 7) & 1;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $7c3d	sta $02
	mem[0x2] = regA;
	// $7c3f	bcc $7c43
	if (flagC == 0) {
		cycles += 12;
		regPC = 0x7c43;
		break;
	};
	// $7c41	inc $03
	temp = (mem[0x3] + 1) & 0xff;
	mem[0x3] = temp;
	flagZ = (temp == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	cycles += 17;
	regPC = 0x7c43;

case 0x7c43:
	// $7c43	rts
	regS = (regS + 1) & 0xff;
	regPC = mem[0x0100 | regS];
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	regPC |= (temp << 8);
	cycles += 6;
	break;

case 0x7c44:
	// dc.b $a9		; $7c44
	// dc.b $d0		; $7c45
	// dc.b $4c		; $7c46
	// dc.b $c2		; $7c47
	// dc.b $7b		; $7c48

case 0x7c49:
	// $7c49	lda $05
	regA = mem[0x5];
	// $7c4b	cmp #$80
	temp = regA - 0x80;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $7c4d	bcc $7c60
	if (flagC == 0) {
		cycles += 7;
		regPC = 0x7c60;
		break;
	};
	// $7c4f	eor #$ff
	regA = (regA ^ 0xff) & 0xff;
	// $7c51	sta $05
	mem[0x5] = regA;
	// $7c53	lda $04
	regA = mem[0x4];
	// $7c55	eor #$ff
	regA = (regA ^ 0xff) & 0xff;
	// $7c57	adc #$00
	temp = 0x0;
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		flagZ = (zl == 0 ? 1 : 0);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		flagN = (zl >> 7) & 1;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $7c59	sta $04
	mem[0x4] = regA;
	// $7c5b	bcc $7c5f
	if (flagC == 0) {
		cycles += 24;
		regPC = 0x7c5f;
		break;
	};
	// $7c5d	inc $05
	temp = (mem[0x5] + 1) & 0xff;
	mem[0x5] = temp;
	flagZ = (temp == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	cycles += 29;
	regPC = 0x7c5f;

case 0x7c5f:
	// $7c5f	sec
	flagC = 1;
	cycles += 2;
	regPC = 0x7c60;

case 0x7c60:
	// $7c60	rol $08
	temp = (mem[0x8] << 1)  | flagC;
	mem[0x8] = (temp & 0xff);
	// $7c62	lda $07
	regA = mem[0x7];
	// $7c64	cmp #$80
	temp = regA - 0x80;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $7c66	bcc $7c79
	if (flagC == 0) {
		cycles += 12;
		regPC = 0x7c79;
		break;
	};
	// $7c68	eor #$ff
	regA = (regA ^ 0xff) & 0xff;
	// $7c6a	sta $07
	mem[0x7] = regA;
	// $7c6c	lda $06
	regA = mem[0x6];
	// $7c6e	eor #$ff
	regA = (regA ^ 0xff) & 0xff;
	// $7c70	adc #$00
	temp = 0x0;
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		flagZ = (zl == 0 ? 1 : 0);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		flagN = (zl >> 7) & 1;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $7c72	sta $06
	mem[0x6] = regA;
	// $7c74	bcc $7c78
	if (flagC == 0) {
		cycles += 29;
		regPC = 0x7c78;
		break;
	};
	// $7c76	inc $07
	temp = (mem[0x7] + 1) & 0xff;
	mem[0x7] = temp;
	flagZ = (temp == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	cycles += 34;
	regPC = 0x7c78;

case 0x7c78:
	// $7c78	sec
	flagC = 1;
	cycles += 2;
	regPC = 0x7c79;

case 0x7c79:
	// $7c79	rol $08
	temp = (mem[0x8] << 1)  | flagC;
	flagC = temp >> 8;
	mem[0x8] = (temp & 0xff);
	// $7c7b	lda $05
	regA = mem[0x5];
	// $7c7d	ora $07
	regA = (regA | mem[0x7]) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7c7f	beq $7c8b
	if (flagZ == 1) {
		cycles += 13;
		regPC = 0x7c8b;
		break;
	};
	// $7c81	ldx #$00
	regX = 0x0;
	// $7c83	cmp #$02
	temp = regA - 0x2;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $7c85	bcs $7cab
	if (flagC == 1) {
		cycles += 19;
		regPC = 0x7cab;
		break;
	};
	// $7c87	ldy #$01
	regY = 0x1;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $7c89	bne $7c9b
	if (flagZ == 0) {
		cycles += 23;
		regPC = 0x7c9b;
		break;
	};
	cycles += 23;
	regPC = 0x7c8b;

case 0x7c8b:
	// $7c8b	ldy #$02
	regY = 0x2;
	// $7c8d	ldx #$09
	regX = 0x9;
	// $7c8f	lda $04
	regA = mem[0x4];
	// $7c91	ora $06
	regA = (regA | mem[0x6]) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7c93	beq $7cab
	if (flagZ == 1) {
		cycles += 12;
		regPC = 0x7cab;
		break;
	};
	// $7c95	bmi $7c9b
	if (flagN == 1) {
		cycles += 14;
		regPC = 0x7c9b;
		break;
	};
	cycles += 14;
	regPC = 0x7c97;

case 0x7c97:
	// $7c97	iny
	regY++;
	regY&= 0xff;
	// $7c98	asl a
	temp = (regA << 1);
	flagC = temp >> 8;
	regA = (temp & 0xff);
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7c99	bpl $7c97
	if (flagN == 0) {
		cycles += 6;
		regPC = 0x7c97;
		break;
	};
	cycles += 6;
	regPC = 0x7c9b;

case 0x7c9b:
	// $7c9b	tya
	regA = regY;
	// $7c9c	tax
	regX = regA;
	// $7c9d	lda $05
	regA = mem[0x5];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 7;
	regPC = 0x7c9f;

case 0x7c9f:
	// $7c9f	asl $04
	temp = (mem[0x4] << 1);
	flagC = temp >> 8;
	mem[0x4] = (temp & 0xff);
	// $7ca1	rol a
	temp = (regA << 1)  | flagC;
	regA = (temp & 0xff);
	// $7ca2	asl $06
	temp = (mem[0x6] << 1);
	flagC = temp >> 8;
	mem[0x6] = (temp & 0xff);
	// $7ca4	rol $07
	temp = (mem[0x7] << 1)  | flagC;
	flagC = temp >> 8;
	mem[0x7] = (temp & 0xff);
	// $7ca6	dey
	regY--;
	regY&= 0xff;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $7ca7	bne $7c9f
	if (flagZ == 0) {
		cycles += 21;
		regPC = 0x7c9f;
		break;
	};
	// $7ca9	sta $05
	mem[0x5] = regA;
	cycles += 24;
	regPC = 0x7cab;

case 0x7cab:
	// $7cab	txa
	regA = regX;
	// $7cac	sec
	flagC = 1;
	// $7cad	sbc #$0a
	temp = 0xa;
	if (flagD == 0) {   // no decimal mode
		z = regA - temp - ((flagC == 0) ? 1 : 0);
		zl = z & 0xff;
		zh = z >> 8;
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		regA = zl;
	}
	else {   // decimal mode
		zl = ((regA & 0x0f) - (temp & 0x0f) - ((flagC == 0) ? 1 : 0)) & 0xff;
		if (zl > 15)
			zl -= 6;
		zh = (regA >> 4) - (temp >> 4) - (zl&0x10);
		if (zh > 15)
			zh -= 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
	};
	// $7caf	eor #$ff
	regA = (regA ^ 0xff) & 0xff;
	// $7cb1	asl a
	temp = (regA << 1);
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $7cb2	ror $08
	temp = mem[0x8] | (flagC << 8);
	flagC = temp & 1;
	temp >>= 1;
	mem[0x8] = (temp & 0xff);
	// $7cb4	rol a
	temp = (regA << 1)  | flagC;
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $7cb5	ror $08
	temp = mem[0x8] | (flagC << 8);
	flagC = temp & 1;
	temp >>= 1;
	mem[0x8] = (temp & 0xff);
	// $7cb7	rol a
	temp = (regA << 1)  | flagC;
	regA = (temp & 0xff);
	// $7cb8	asl a
	temp = (regA << 1);
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $7cb9	sta $08
	mem[0x8] = regA;
	// $7cbb	ldy #$00
	regY = 0x0;
	// $7cbd	lda $06
	regA = mem[0x6];
	// $7cbf	sta ($02),y
	mem[(mem[0x2] | (mem[0x3]<<8)) + regY] = regA;
	// $7cc1	lda $08
	regA = mem[0x8];
	// $7cc3	and #$f4
	regA = (regA & 0xf4) & 0xff;
	// $7cc5	ora $07
	regA = (regA | mem[0x7]) & 0xff;
	// $7cc7	iny
	regY++;
	regY&= 0xff;
	// $7cc8	sta ($02),y
	mem[(mem[0x2] | (mem[0x3]<<8)) + regY] = regA;
	// $7cca	lda $04
	regA = mem[0x4];
	// $7ccc	iny
	regY++;
	regY&= 0xff;
	// $7ccd	sta ($02),y
	mem[(mem[0x2] | (mem[0x3]<<8)) + regY] = regA;
	// $7ccf	lda $08
	regA = mem[0x8];
	// $7cd1	and #$02
	regA = (regA & 0x2) & 0xff;
	// $7cd3	asl a
	temp = (regA << 1);
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $7cd4	ora $01
	regA = (regA | mem[0x1]) & 0xff;
	// $7cd6	ora $05
	regA = (regA | mem[0x5]) & 0xff;
	// $7cd8	iny
	regY++;
	regY&= 0xff;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $7cd9	sta ($02),y
	mem[(mem[0x2] | (mem[0x3]<<8)) + regY] = regA;
	// $7cdb	jmp $7c39
	regPC = 0x7c39;
	cycles += 91;
	break;

case 0x7cde:
	// $7cde	ldx #$00
	regX = 0x0;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	cycles += 2;
	regPC = 0x7ce0;

case 0x7ce0:
	// $7ce0	ldy #$01
	regY = 0x1;
	// $7ce2	sta ($02),y
	mem[(mem[0x2] | (mem[0x3]<<8)) + regY] = regA;
	// $7ce4	dey
	regY--;
	regY&= 0xff;
	// $7ce5	tya
	regA = regY;
	// $7ce6	sta ($02),y
	mem[(mem[0x2] | (mem[0x3]<<8)) + regY] = regA;
	// $7ce8	iny
	regY++;
	regY&= 0xff;
	// $7ce9	iny
	regY++;
	regY&= 0xff;
	// $7cea	sta ($02),y
	mem[(mem[0x2] | (mem[0x3]<<8)) + regY] = regA;
	// $7cec	iny
	regY++;
	regY&= 0xff;
	// $7ced	txa
	regA = regX;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7cee	sta ($02),y
	mem[(mem[0x2] | (mem[0x3]<<8)) + regY] = regA;
	// $7cf0	jmp $7c39
	regPC = 0x7c39;
	cycles += 41;
	break;

case 0x7cf3:
	// $7cf3	ldx #$fe
	regX = 0xfe;
	// $7cf5	txs
	regS = regX;
	// $7cf6	cld
	flagD = 0;
	// $7cf7	lda #$00
	regA = 0x0;
	// $7cf9	tax
	regX = regA;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	cycles += 10;
	regPC = 0x7cfa;

case 0x7cfa:
	// $7cfa	dex
	regX--;
	regX&= 0xff;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7cfb	sta $0300,x
	mem[0x300+regX] = regA;
	// $7cfe	sta $0200,x
	mem[0x200+regX] = regA;
	// $7d01	sta $0100,x
	mem[0x100+regX] = regA;
	// $7d04	sta $00,x
	mem[(0x0+regX)&0xff] = regA;
	// $7d06	bne $7cfa
	if (flagZ == 0) {
		cycles += 23;
		regPC = 0x7cfa;
		break;
	};
	// $7d08	ldy $2007
	regY = readMem(0x2007);
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $7d0b	bmi $7d50
	if (flagN == 1) {
		cycles += 29;
		regPC = 0x7d50;
		break;
	};
	// $7d0d	inx
	regX++;
	regX&= 0xff;
	// $7d0e	stx $4000
	mem[0x4000] = regX;
	// $7d11	lda #$e2
	regA = 0xe2;
	// $7d13	sta $4001
	mem[0x4001] = regA;
	// $7d16	lda #$b0
	regA = 0xb0;
	// $7d18	sta $4003
	mem[0x4003] = regA;
	// $7d1b	sta $32
	mem[0x32] = regA;
	// $7d1d	sta $33
	mem[0x33] = regA;
	// $7d1f	lda #$03
	regA = 0x3;
	// $7d21	sta $6f
	mem[0x6f] = regA;
	// $7d23	sta $3200
	writeMem(0x3200, regA);
	// $7d26	and $2800
	regA = (regA & readMem(0x2800)) & 0xff;
	// $7d29	sta $71
	mem[0x71] = regA;
	// $7d2b	lda $2801
	regA = readMem(0x2801);
	// $7d2e	and #$03
	regA = (regA & 0x3) & 0xff;
	// $7d30	asl a
	temp = (regA << 1);
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $7d31	asl a
	temp = (regA << 1);
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $7d32	ora $71
	regA = (regA | mem[0x71]) & 0xff;
	// $7d34	sta $71
	mem[0x71] = regA;
	// $7d36	lda $2802
	regA = readMem(0x2802);
	// $7d39	and #$02
	regA = (regA & 0x2) & 0xff;
	// $7d3b	asl a
	temp = (regA << 1);
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $7d3c	asl a
	temp = (regA << 1);
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $7d3d	asl a
	temp = (regA << 1);
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $7d3e	ora $71
	regA = (regA | mem[0x71]) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7d40	sta $71
	mem[0x71] = regA;
	// $7d42	jmp $6803
	regPC = 0x6803;
	cycles += 106;
	break;

case 0x7d45:
	// $7d45	ldy #$00
	regY = 0x0;
	// $7d47	sta ($02),y
	mem[(mem[0x2] | (mem[0x3]<<8)) + regY] = regA;
	// $7d49	iny
	regY++;
	regY&= 0xff;
	// $7d4a	txa
	regA = regX;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7d4b	sta ($02),y
	mem[(mem[0x2] | (mem[0x3]<<8)) + regY] = regA;
	// $7d4d	jmp $7c39
	regPC = 0x7c39;
	cycles += 21;
	break;

case 0x7d50:
	// $7d50	sta $4000,x
	mem[0x4000+regX] = regA;
	// $7d53	sta $4100,x
	mem[0x4100+regX] = regA;
	// $7d56	sta $4200,x
	mem[0x4200+regX] = regA;
	// $7d59	sta $4300,x
	mem[0x4300+regX] = regA;
	// $7d5c	sta $4400,x
	mem[0x4400+regX] = regA;
	// $7d5f	sta $4500,x
	mem[0x4500+regX] = regA;
	// $7d62	sta $4600,x
	mem[0x4600+regX] = regA;
	// $7d65	sta $4700,x
	mem[0x4700+regX] = regA;
	// $7d68	inx
	regX++;
	regX&= 0xff;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7d69	bne $7d50
	if (flagZ == 0) {
		cycles += 44;
		regPC = 0x7d50;
		break;
	};
	// $7d6b	sta $3400
	writeMem(0x3400, regA);
	// $7d6e	ldx #$00
	regX = 0x0;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	cycles += 50;
	regPC = 0x7d70;

case 0x7d70:
	// $7d70	lda $00,x
	regA = mem[(0x0+regX)&0xff];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7d72	bne $7dbb
	if (flagZ == 0) {
		cycles += 6;
		regPC = 0x7dbb;
		break;
	};
	// $7d74	lda #$11
	regA = 0x11;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 8;
	regPC = 0x7d76;

case 0x7d76:
	// $7d76	sta $00,x
	mem[(0x0+regX)&0xff] = regA;
	// $7d78	tay
	regY = regA;
	// $7d79	eor $00,x
	regA = (regA ^ mem[(0x0+regX)&0xff]) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7d7b	bne $7dbb
	if (flagZ == 0) {
		cycles += 12;
		regPC = 0x7dbb;
		break;
	};
	// $7d7d	tya
	regA = regY;
	// $7d7e	asl a
	temp = (regA << 1);
	flagC = temp >> 8;
	regA = (temp & 0xff);
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7d7f	bcc $7d76
	if (flagC == 0) {
		cycles += 18;
		regPC = 0x7d76;
		break;
	};
	// $7d81	inx
	regX++;
	regX&= 0xff;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7d82	bne $7d70
	if (flagZ == 0) {
		cycles += 22;
		regPC = 0x7d70;
		break;
	};
	// $7d84	sta $3400
	writeMem(0x3400, regA);
	// $7d87	txa
	regA = regX;
	// $7d88	sta $00
	mem[0x0] = regA;
	// $7d8a	rol a
	temp = (regA << 1)  | flagC;
	flagC = temp >> 8;
	regA = (temp & 0xff);
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 33;
	regPC = 0x7d8b;

case 0x7d8b:
	// $7d8b	sta $01
	mem[0x1] = regA;
	// $7d8d	ldy #$00
	regY = 0x0;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	cycles += 5;
	regPC = 0x7d8f;

case 0x7d8f:
	// $7d8f	ldx #$11
	regX = 0x11;
	// $7d91	lda ($00),y
	regA = mem[(mem[0x0] | (mem[0x1]<<8))+regY];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7d93	bne $7dbf
	if (flagZ == 0) {
		cycles += 9;
		regPC = 0x7dbf;
		break;
	};
	cycles += 9;
	regPC = 0x7d95;

case 0x7d95:
	// $7d95	txa
	regA = regX;
	// $7d96	sta ($00),y
	mem[(mem[0x0] | (mem[0x1]<<8)) + regY] = regA;
	// $7d98	eor ($00),y
	regA = (regA ^ mem[(mem[0x0] | (mem[0x1]<<8))+regY]) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7d9a	bne $7dbf
	if (flagZ == 0) {
		cycles += 15;
		regPC = 0x7dbf;
		break;
	};
	// $7d9c	txa
	regA = regX;
	// $7d9d	asl a
	temp = (regA << 1);
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $7d9e	tax
	regX = regA;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7d9f	bcc $7d95
	if (flagC == 0) {
		cycles += 23;
		regPC = 0x7d95;
		break;
	};
	// $7da1	iny
	regY++;
	regY&= 0xff;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $7da2	bne $7d8f
	if (flagZ == 0) {
		cycles += 27;
		regPC = 0x7d8f;
		break;
	};
	// $7da4	sta $3400
	writeMem(0x3400, regA);
	// $7da7	inc $01
	temp = (mem[0x1] + 1) & 0xff;
	mem[0x1] = temp;
	// $7da9	ldx $01
	regX = mem[0x1];
	// $7dab	cpx #$04
	temp = regX - 0x4;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $7dad	bcc $7d8f
	if (flagC == 0) {
		cycles += 43;
		regPC = 0x7d8f;
		break;
	};
	// $7daf	lda #$40
	regA = 0x40;
	// $7db1	cpx #$40
	temp = regX - 0x40;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $7db3	bcc $7d8b
	if (flagC == 0) {
		cycles += 49;
		regPC = 0x7d8b;
		break;
	};
	// $7db5	cpx #$48
	temp = regX - 0x48;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $7db7	bcc $7d8f
	if (flagC == 0) {
		cycles += 53;
		regPC = 0x7d8f;
		break;
	};
	// $7db9	bcs $7e24
	if (flagC == 1) {
		cycles += 55;
		regPC = 0x7e24;
		break;
	};
	cycles += 55;
	regPC = 0x7dbb;

case 0x7dbb:
	// $7dbb	ldy #$00
	regY = 0x0;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $7dbd	beq $7dcd
	if (flagZ == 1) {
		cycles += 4;
		regPC = 0x7dcd;
		break;
	};
	cycles += 4;
	regPC = 0x7dbf;

case 0x7dbf:
	// $7dbf	ldy #$00
	regY = 0x0;
	// $7dc1	ldx $01
	regX = mem[0x1];
	// $7dc3	cpx #$04
	temp = regX - 0x4;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $7dc5	bcc $7dcd
	if (flagC == 0) {
		cycles += 9;
		regPC = 0x7dcd;
		break;
	};
	// $7dc7	iny
	regY++;
	regY&= 0xff;
	// $7dc8	cpx #$44
	temp = regX - 0x44;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $7dca	bcc $7dcd
	if (flagC == 0) {
		cycles += 15;
		regPC = 0x7dcd;
		break;
	};
	// $7dcc	iny
	regY++;
	regY&= 0xff;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	cycles += 17;
	regPC = 0x7dcd;

case 0x7dcd:
	// $7dcd	cmp #$10
	temp = regA - 0x10;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	// $7dcf	rol a
	temp = (regA << 1)  | flagC;
	regA = (temp & 0xff);
	// $7dd0	and #$1f
	regA = (regA & 0x1f) & 0xff;
	// $7dd2	cmp #$02
	temp = regA - 0x2;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	// $7dd4	rol a
	temp = (regA << 1)  | flagC;
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $7dd5	and #$03
	regA = (regA & 0x3) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 12;
	regPC = 0x7dd7;

case 0x7dd7:
	// $7dd7	dey
	regY--;
	regY&= 0xff;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $7dd8	bmi $7dde
	if (flagN == 1) {
		cycles += 4;
		regPC = 0x7dde;
		break;
	};
	// $7dda	asl a
	temp = (regA << 1);
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $7ddb	asl a
	temp = (regA << 1);
	flagC = temp >> 8;
	regA = (temp & 0xff);
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7ddc	bcc $7dd7
	if (flagC == 0) {
		cycles += 10;
		regPC = 0x7dd7;
		break;
	};
	cycles += 10;
	regPC = 0x7dde;

case 0x7dde:
	// $7dde	lsr a
	temp = regA;
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	// $7ddf	ldx #$14
	regX = 0x14;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7de1	bcc $7de5
	if (flagC == 0) {
		cycles += 6;
		regPC = 0x7de5;
		break;
	};
	// $7de3	ldx #$1d
	regX = 0x1d;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	cycles += 8;
	regPC = 0x7de5;

case 0x7de5:
	// $7de5	stx $3a00
	writeMem(0x3a00, regX);
	// $7de8	ldx #$00
	regX = 0x0;
	// $7dea	ldy #$08
	regY = 0x8;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	cycles += 8;
	regPC = 0x7dec;

case 0x7dec:
	// $7dec	bit $2001
	temp =readMem(0x2001);
	flagN = (temp >> 7);
	flagV = ((temp >> 6) & 1);
	flagZ = ((temp & regA) != 0) ? 0 : 1;
	// $7def	bpl $7dec
	if (flagN == 0) {
		cycles += 6;
		regPC = 0x7dec;
		break;
	};
	cycles += 6;
	regPC = 0x7df1;

case 0x7df1:
	// $7df1	bit $2001
	temp =readMem(0x2001);
	flagN = (temp >> 7);
	flagV = ((temp >> 6) & 1);
	flagZ = ((temp & regA) != 0) ? 0 : 1;
	// $7df4	bmi $7df1
	if (flagN == 1) {
		cycles += 6;
		regPC = 0x7df1;
		break;
	};
	// $7df6	dex
	regX--;
	regX&= 0xff;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7df7	sta $3400
	writeMem(0x3400, regA);
	// $7dfa	bne $7dec
	if (flagZ == 0) {
		cycles += 14;
		regPC = 0x7dec;
		break;
	};
	// $7dfc	dey
	regY--;
	regY&= 0xff;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $7dfd	bne $7dec
	if (flagZ == 0) {
		cycles += 18;
		regPC = 0x7dec;
		break;
	};
	// $7dff	stx $3a00
	writeMem(0x3a00, regX);
	// $7e02	ldy #$08
	regY = 0x8;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	cycles += 24;
	regPC = 0x7e04;

case 0x7e04:
	// $7e04	bit $2001
	temp =readMem(0x2001);
	flagN = (temp >> 7);
	flagV = ((temp >> 6) & 1);
	flagZ = ((temp & regA) != 0) ? 0 : 1;
	// $7e07	bpl $7e04
	if (flagN == 0) {
		cycles += 6;
		regPC = 0x7e04;
		break;
	};
	cycles += 6;
	regPC = 0x7e09;

case 0x7e09:
	// $7e09	bit $2001
	temp =readMem(0x2001);
	flagN = (temp >> 7);
	flagV = ((temp >> 6) & 1);
	flagZ = ((temp & regA) != 0) ? 0 : 1;
	// $7e0c	bmi $7e09
	if (flagN == 1) {
		cycles += 6;
		regPC = 0x7e09;
		break;
	};
	// $7e0e	dex
	regX--;
	regX&= 0xff;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7e0f	sta $3400
	writeMem(0x3400, regA);
	// $7e12	bne $7e04
	if (flagZ == 0) {
		cycles += 14;
		regPC = 0x7e04;
		break;
	};
	// $7e14	dey
	regY--;
	regY&= 0xff;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $7e15	bne $7e04
	if (flagZ == 0) {
		cycles += 18;
		regPC = 0x7e04;
		break;
	};
	// $7e17	tax
	regX = regA;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7e18	bne $7dde
	if (flagZ == 0) {
		cycles += 22;
		regPC = 0x7dde;
		break;
	};
	cycles += 22;
	regPC = 0x7e1a;

case 0x7e1a:
	// $7e1a	sta $3400
	writeMem(0x3400, regA);
	// $7e1d	lda $2007
	regA = readMem(0x2007);
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7e20	bmi $7e1a
	if (flagN == 1) {
		cycles += 10;
		regPC = 0x7e1a;
		break;
	};
	cycles += 10;
	regPC = 0x7e22;

case 0x7e22:
	// $7e22	bpl $7e22
	if (flagN == 0) {
		cycles += 2;
		regPC = 0x7e22;
		break;
	};
	cycles += 2;
	regPC = 0x7e24;

case 0x7e24:
	// $7e24	lda #$00
	regA = 0x0;
	// $7e26	tay
	regY = regA;
	// $7e27	tax
	regX = regA;
	// $7e28	sta $08
	mem[0x8] = regA;
	// $7e2a	lda #$50
	regA = 0x50;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 11;
	regPC = 0x7e2c;

case 0x7e2c:
	// $7e2c	sta $09
	mem[0x9] = regA;
	// $7e2e	lda #$04
	regA = 0x4;
	// $7e30	sta $0b
	mem[0xb] = regA;
	// $7e32	lda #$ff
	regA = 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 10;
	regPC = 0x7e34;

case 0x7e34:
	// $7e34	eor ($08),y
	regA = (regA ^ mem[(mem[0x8] | (mem[0x9]<<8))+regY]) & 0xff;
	// $7e36	iny
	regY++;
	regY&= 0xff;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $7e37	bne $7e34
	if (flagZ == 0) {
		cycles += 9;
		regPC = 0x7e34;
		break;
	};
	// $7e39	inc $09
	temp = (mem[0x9] + 1) & 0xff;
	mem[0x9] = temp;
	// $7e3b	dec $0b
	temp = (mem[0xb] - 1) & 0xff;
	mem[0xb] = temp;
	flagZ = (temp == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $7e3d	bne $7e34
	if (flagZ == 0) {
		cycles += 21;
		regPC = 0x7e34;
		break;
	};
	// $7e3f	sta $0d,x
	mem[(0xd+regX)&0xff] = regA;
	// $7e41	inx
	regX++;
	regX&= 0xff;
	// $7e42	sta $3400
	writeMem(0x3400, regA);
	// $7e45	lda $09
	regA = mem[0x9];
	// $7e47	cmp #$58
	temp = regA - 0x58;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $7e49	bcc $7e2c
	if (flagC == 0) {
		cycles += 38;
		regPC = 0x7e2c;
		break;
	};
	// $7e4b	bne $7e4f
	if (flagZ == 0) {
		cycles += 40;
		regPC = 0x7e4f;
		break;
	};
	// $7e4d	lda #$68
	regA = 0x68;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 42;
	regPC = 0x7e4f;

case 0x7e4f:
	// $7e4f	cmp #$80
	temp = regA - 0x80;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $7e51	bcc $7e2c
	if (flagC == 0) {
		cycles += 4;
		regPC = 0x7e2c;
		break;
	};
	// $7e53	sta $0300
	mem[0x300] = regA;
	// $7e56	ldx #$04
	regX = 0x4;
	// $7e58	stx $3200
	writeMem(0x3200, regX);
	// $7e5b	stx $15
	mem[0x15] = regX;
	// $7e5d	ldx #$00
	regX = 0x0;
	// $7e5f	cmp $0200
	temp = regA - mem[0x200];
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $7e62	beq $7e65
	if (flagZ == 1) {
		cycles += 25;
		regPC = 0x7e65;
		break;
	};
	// $7e64	inx
	regX++;
	regX&= 0xff;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	cycles += 27;
	regPC = 0x7e65;

case 0x7e65:
	// $7e65	lda $0300
	regA = mem[0x300];
	// $7e68	cmp #$88
	temp = regA - 0x88;
	flagC = ((temp >> 8) != 0) ? 0 : 1;
	flagZ = ((temp & 0xff) == 0) ? 1 : 0;
	flagN = (temp >> 7) & 1;
	// $7e6a	beq $7e6d
	if (flagZ == 1) {
		cycles += 8;
		regPC = 0x7e6d;
		break;
	};
	// $7e6c	inx
	regX++;
	regX&= 0xff;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	cycles += 10;
	regPC = 0x7e6d;

case 0x7e6d:
	// $7e6d	stx $16
	mem[0x16] = regX;
	// $7e6f	lda #$10
	regA = 0x10;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7e71	sta $00
	mem[0x0] = regA;
	cycles += 8;
	regPC = 0x7e73;

case 0x7e73:
	// $7e73	ldx #$24
	regX = 0x24;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	cycles += 2;
	regPC = 0x7e75;

case 0x7e75:
	// $7e75	lda $2001
	regA = readMem(0x2001);
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7e78	bpl $7e75
	if (flagN == 0) {
		cycles += 6;
		regPC = 0x7e75;
		break;
	};
	cycles += 6;
	regPC = 0x7e7a;

case 0x7e7a:
	// $7e7a	lda $2001
	regA = readMem(0x2001);
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7e7d	bmi $7e7a
	if (flagN == 1) {
		cycles += 6;
		regPC = 0x7e7a;
		break;
	};
	// $7e7f	dex
	regX--;
	regX&= 0xff;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7e80	bpl $7e75
	if (flagN == 0) {
		cycles += 10;
		regPC = 0x7e75;
		break;
	};
	cycles += 10;
	regPC = 0x7e82;

case 0x7e82:
	// $7e82	bit $2002
	temp =readMem(0x2002);
	flagN = (temp >> 7);
	flagV = ((temp >> 6) & 1);
	flagZ = ((temp & regA) != 0) ? 0 : 1;
	// $7e85	bmi $7e82
	if (flagN == 1) {
		cycles += 6;
		regPC = 0x7e82;
		break;
	};
	// $7e87	sta $3400
	writeMem(0x3400, regA);
	// $7e8a	lda #$00
	regA = 0x0;
	// $7e8c	sta $02
	mem[0x2] = regA;
	// $7e8e	lda #$40
	regA = 0x40;
	// $7e90	sta $03
	mem[0x3] = regA;
	// $7e92	lda $2005
	regA = readMem(0x2005);
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7e95	bpl $7ef2
	if (flagN == 0) {
		cycles += 26;
		regPC = 0x7ef2;
		break;
	};
	// $7e97	ldx $15
	regX = mem[0x15];
	// $7e99	lda $2003
	regA = readMem(0x2003);
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7e9c	bpl $7ea8
	if (flagN == 0) {
		cycles += 35;
		regPC = 0x7ea8;
		break;
	};
	// $7e9e	eor $0009
	regA = (regA ^ mem[0x9]) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7ea1	bpl $7ea8
	if (flagN == 0) {
		cycles += 41;
		regPC = 0x7ea8;
		break;
	};
	// $7ea3	dex
	regX--;
	regX&= 0xff;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7ea4	beq $7ea8
	if (flagZ == 1) {
		cycles += 45;
		regPC = 0x7ea8;
		break;
	};
	// $7ea6	stx $15
	mem[0x15] = regX;
	cycles += 48;
	regPC = 0x7ea8;

case 0x7ea8:
	// $7ea8	ldy $7ebb,x
	regY = mem[0x7ebb+regX];
	// $7eab	lda #$b0
	regA = 0xb0;
	// $7ead	sta ($02),y
	mem[(mem[0x2] | (mem[0x3]<<8)) + regY] = regA;
	// $7eaf	dey
	regY--;
	regY&= 0xff;
	// $7eb0	dey
	regY--;
	regY&= 0xff;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	cycles += 16;
	regPC = 0x7eb1;

case 0x7eb1:
	// $7eb1	lda $7ec0,y
	regA = mem[0x7ec0+regY];
	// $7eb4	sta ($02),y
	mem[(mem[0x2] | (mem[0x3]<<8)) + regY] = regA;
	// $7eb6	dey
	regY--;
	regY&= 0xff;
	flagZ = (regY == 0) ? 1 : 0;
	flagN = (regY >> 7) & 1;
	// $7eb7	bpl $7eb1
	if (flagN == 0) {
		cycles += 14;
		regPC = 0x7eb1;
		break;
	};
	// $7eb9	jmp $7f9d
	regPC = 0x7f9d;
	cycles += 17;
	break;

case 0x7ebc:
	// dc.b $33		; $7ebc
	// dc.b $1d		; $7ebd
	// dc.b $17		; $7ebe
	// dc.b $0d		; $7ebf
	// dc.b $80		; $7ec0
	// dc.b $a0		; $7ec1
	// dc.b $00		; $7ec2
	// dc.b $00		; $7ec3
	// dc.b $00		; $7ec4
	// dc.b $70		; $7ec5
	// dc.b $00		; $7ec6
	// dc.b $00		; $7ec7
	// dc.b $ff		; $7ec8
	// dc.b $92		; $7ec9
	// dc.b $ff		; $7eca
	// dc.b $73		; $7ecb
	// dc.b $d0		; $7ecc
	// dc.b $a1		; $7ecd
	// dc.b $30		; $7ece
	// dc.b $02		; $7ecf
	// dc.b $00		; $7ed0
	// dc.b $70		; $7ed1
	// dc.b $00		; $7ed2
	// dc.b $00		; $7ed3
	// dc.b $7f		; $7ed4
	// dc.b $fb		; $7ed5
	// dc.b $0d		; $7ed6
	// dc.b $e0		; $7ed7
	// dc.b $00		; $7ed8
	// dc.b $b0		; $7ed9
	// dc.b $7e		; $7eda
	// dc.b $fa		; $7edb
	// dc.b $11		; $7edc
	// dc.b $c0		; $7edd
	// dc.b $78		; $7ede
	// dc.b $fe		; $7edf
	// dc.b $00		; $7ee0
	// dc.b $b0		; $7ee1
	// dc.b $13		; $7ee2
	// dc.b $c0		; $7ee3
	// dc.b $00		; $7ee4
	// dc.b $d0		; $7ee5
	// dc.b $15		; $7ee6
	// dc.b $c0		; $7ee7
	// dc.b $00		; $7ee8
	// dc.b $d0		; $7ee9
	// dc.b $17		; $7eea
	// dc.b $c0		; $7eeb
	// dc.b $00		; $7eec
	// dc.b $d0		; $7eed
	// dc.b $7a		; $7eee
	// dc.b $f8		; $7eef
	// dc.b $00		; $7ef0
	// dc.b $d0		; $7ef1

case 0x7ef2:
	// $7ef2	lda #$50
	regA = 0x50;
	// $7ef4	ldx #$00
	regX = 0x0;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7ef6	jsr $7bfc
	mem[0x0100 | regS] = 0x7e;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0xf9;
	regS = (regS - 1) & 0xff;
	regPC = 0x7bfc;
	cycles += 10;
	break;

case 0x7ef9:
	// $7ef9	lda #$69
	regA = 0x69;
	// $7efb	ldx #$93
	regX = 0x93;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7efd	jsr $7c03
	mem[0x0100 | regS] = 0x7f;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x0;
	regS = (regS - 1) & 0xff;
	regPC = 0x7c03;
	cycles += 10;
	break;

case 0x7f00:
	// $7f00	lda #$30
	regA = 0x30;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7f02	jsr $7cde
	mem[0x0100 | regS] = 0x7f;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x5;
	regS = (regS - 1) & 0xff;
	regPC = 0x7cde;
	cycles += 8;
	break;

case 0x7f05:
	// $7f05	ldx #$03
	regX = 0x3;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	cycles += 2;
	regPC = 0x7f07;

case 0x7f07:
	// $7f07	lda $2800,x
	regA = readMem(0x2800+regX);
	// $7f0a	and #$01
	regA = (regA & 0x1) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7f0c	stx $0b
	mem[0xb] = regX;
	// $7f0e	jsr $7bd1
	mem[0x0100 | regS] = 0x7f;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x11;
	regS = (regS - 1) & 0xff;
	regPC = 0x7bd1;
	cycles += 15;
	break;

case 0x7f11:
	// $7f11	ldx $0b
	regX = mem[0xb];
	// $7f13	lda $2800,x
	regA = readMem(0x2800+regX);
	// $7f16	and #$02
	regA = (regA & 0x2) & 0xff;
	// $7f18	lsr a
	temp = regA;
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7f19	jsr $7bd1
	mem[0x0100 | regS] = 0x7f;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x1c;
	regS = (regS - 1) & 0xff;
	regPC = 0x7bd1;
	cycles += 17;
	break;

case 0x7f1c:
	// $7f1c	ldx $0b
	regX = mem[0xb];
	// $7f1e	dex
	regX--;
	regX&= 0xff;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7f1f	bpl $7f07
	if (flagN == 0) {
		cycles += 7;
		regPC = 0x7f07;
		break;
	};
	// $7f21	lda #$7a
	regA = 0x7a;
	// $7f23	ldx #$9d
	regX = 0x9d;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7f25	jsr $7c03
	mem[0x0100 | regS] = 0x7f;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x28;
	regS = (regS - 1) & 0xff;
	regPC = 0x7c03;
	cycles += 17;
	break;

case 0x7f28:
	// $7f28	lda #$10
	regA = 0x10;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7f2a	jsr $7cde
	mem[0x0100 | regS] = 0x7f;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x2d;
	regS = (regS - 1) & 0xff;
	regPC = 0x7cde;
	cycles += 8;
	break;

case 0x7f2d:
	// $7f2d	lda $2802
	regA = readMem(0x2802);
	// $7f30	and #$02
	regA = (regA & 0x2) & 0xff;
	// $7f32	lsr a
	temp = regA;
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	// $7f33	adc #$01
	temp = 0x1;
	if (flagD == 0) {   // no decimal mode
		z = regA + temp + flagC;
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh & 1);
		flagZ = (zl == 0 ? 1 : 0);
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		flagN = (zl >> 7) & 1;
		regA = zl;
	}
	else {   // decimal mode 
		zl = ((regA & 0x0f) + (temp & 0x0f) + flagC)&0xff;
		zh = ((regA >> 4) + (temp >> 4) + (zl>15 ? 1 : 0));
		if (zl > 9) {
			zl += 6;
			zh++;
		};
		if (zh > 9)
			zh += 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $7f35	jsr $7bd1
	mem[0x0100 | regS] = 0x7f;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x38;
	regS = (regS - 1) & 0xff;
	regPC = 0x7bd1;
	cycles += 16;
	break;

case 0x7f38:
	// $7f38	lda $2801
	regA = readMem(0x2801);
	// $7f3b	and #$03
	regA = (regA & 0x3) & 0xff;
	// $7f3d	tax
	regX = regA;
	// $7f3e	lda $7ff5,x
	regA = mem[0x7ff5+regX];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7f41	jsr $7bd1
	mem[0x0100 | regS] = 0x7f;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x44;
	regS = (regS - 1) & 0xff;
	regPC = 0x7bd1;
	cycles += 18;
	break;

case 0x7f44:
	// $7f44	lda $16
	regA = mem[0x16];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7f46	beq $7f4f
	if (flagZ == 1) {
		cycles += 5;
		regPC = 0x7f4f;
		break;
	};
	// $7f48	ldx #$88
	regX = 0x88;
	// $7f4a	lda #$50
	regA = 0x50;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7f4c	jsr $7bfc
	mem[0x0100 | regS] = 0x7f;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x4f;
	regS = (regS - 1) & 0xff;
	regPC = 0x7bfc;
	cycles += 15;
	break;

case 0x7f4f:
	// $7f4f	ldx #$96
	regX = 0x96;
	// $7f51	stx $000c
	mem[0xc] = regX;
	// $7f54	ldx #$07
	regX = 0x7;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	cycles += 8;
	regPC = 0x7f56;

case 0x7f56:
	// $7f56	lda $0d,x
	regA = mem[(0xd+regX)&0xff];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7f58	beq $7f91
	if (flagZ == 1) {
		cycles += 6;
		regPC = 0x7f91;
		break;
	};
	// $7f5a	pha
	mem[0x0100 | regS] = regA;
	regS = (regS - 1) & 0xff;
	// $7f5b	stx $000b
	mem[0xb] = regX;
	// $7f5e	ldx $000c
	regX = mem[0xc];
	// $7f61	txa
	regA = regX;
	// $7f62	sec
	flagC = 1;
	// $7f63	sbc #$08
	temp = 0x8;
	if (flagD == 0) {   // no decimal mode
		z = regA - temp - ((flagC == 0) ? 1 : 0);
		zl = z & 0xff;
		zh = z >> 8;
		flagC = (zh == 0) ? 1 : 0;
		if ((~(regA ^ temp) & (regA ^ zl) & 0x80) != 0)
			flagV = 1;
		else
			flagV = 0;
		regA = zl;
	}
	else {   // decimal mode
		zl = ((regA & 0x0f) - (temp & 0x0f) - ((flagC == 0) ? 1 : 0)) & 0xff;
		if (zl > 15)
			zl -= 6;
		zh = (regA >> 4) - (temp >> 4) - (zl&0x10);
		if (zh > 15)
			zh -= 6;
		regA = (zl & 0x0f) | ((zh << 4)&0xff);
		flagC = (zh > 15 ? 1 : 0);
	};
	// $7f65	sta $000c
	mem[0xc] = regA;
	// $7f68	lda #$20
	regA = 0x20;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7f6a	jsr $7c03
	mem[0x0100 | regS] = 0x7f;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x6d;
	regS = (regS - 1) & 0xff;
	regPC = 0x7c03;
	cycles += 35;
	break;

case 0x7f6d:
	// $7f6d	lda #$70
	regA = 0x70;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7f6f	jsr $7cde
	mem[0x0100 | regS] = 0x7f;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x72;
	regS = (regS - 1) & 0xff;
	regPC = 0x7cde;
	cycles += 8;
	break;

case 0x7f72:
	// $7f72	lda $000b
	regA = mem[0xb];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7f75	jsr $7bd1
	mem[0x0100 | regS] = 0x7f;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x78;
	regS = (regS - 1) & 0xff;
	regPC = 0x7bd1;
	cycles += 10;
	break;

case 0x7f78:
	// $7f78	lda $56d4
	regA = mem[0x56d4];
	// $7f7b	ldx $56d5
	regX = mem[0x56d5];
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7f7e	jsr $7d45
	mem[0x0100 | regS] = 0x7f;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x81;
	regS = (regS - 1) & 0xff;
	regPC = 0x7d45;
	cycles += 14;
	break;

case 0x7f81:
	// $7f81	pla
	regS = (regS + 1) & 0xff;
	regA = mem[0x0100 | regS];
	// $7f82	pha
	mem[0x0100 | regS] = regA;
	regS = (regS - 1) & 0xff;
	// $7f83	lsr a
	temp = regA;
	temp >>= 1;
	regA = (temp & 0xff);
	// $7f84	lsr a
	temp = regA;
	temp >>= 1;
	regA = (temp & 0xff);
	// $7f85	lsr a
	temp = regA;
	temp >>= 1;
	regA = (temp & 0xff);
	// $7f86	lsr a
	temp = regA;
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7f87	jsr $7bd1
	mem[0x0100 | regS] = 0x7f;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x8a;
	regS = (regS - 1) & 0xff;
	regPC = 0x7bd1;
	cycles += 21;
	break;

case 0x7f8a:
	// $7f8a	pla
	regS = (regS + 1) & 0xff;
	regA = mem[0x0100 | regS];
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7f8b	jsr $7bd1
	mem[0x0100 | regS] = 0x7f;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x8e;
	regS = (regS - 1) & 0xff;
	regPC = 0x7bd1;
	cycles += 10;
	break;

case 0x7f8e:
	// $7f8e	ldx $000b
	regX = mem[0xb];
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	cycles += 4;
	regPC = 0x7f91;

case 0x7f91:
	// $7f91	dex
	regX--;
	regX&= 0xff;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7f92	bpl $7f56
	if (flagN == 0) {
		cycles += 4;
		regPC = 0x7f56;
		break;
	};
	// $7f94	lda #$7f
	regA = 0x7f;
	// $7f96	tax
	regX = regA;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7f97	jsr $7c03
	mem[0x0100 | regS] = 0x7f;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x9a;
	regS = (regS - 1) & 0xff;
	regPC = 0x7c03;
	cycles += 14;
	break;

case 0x7f9a:
	// $7f9a	jsr $7bc0
	mem[0x0100 | regS] = 0x7f;
	regS = (regS - 1) & 0xff;
	mem[0x0100 | regS] = 0x9d;
	regS = (regS - 1) & 0xff;
	regPC = 0x7bc0;
	cycles += 6;
	break;

case 0x7f9d:
	// $7f9d	lda #$00
	regA = 0x0;
	// $7f9f	ldx #$04
	regX = 0x4;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	cycles += 4;
	regPC = 0x7fa1;

case 0x7fa1:
	// $7fa1	rol $2003,x
	temp = (readMem(0x2003+regX) << 1)  | flagC;
	flagC = temp >> 8;
	writeMem(0x2003+regX, (temp & 0xff));
	// $7fa4	ror a
	temp = regA | (flagC << 8);
	flagC = temp & 1;
	temp >>= 1;
	regA = (temp & 0xff);
	// $7fa5	dex
	regX--;
	regX&= 0xff;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7fa6	bpl $7fa1
	if (flagN == 0) {
		cycles += 13;
		regPC = 0x7fa1;
		break;
	};
	// $7fa8	tay
	regY = regA;
	// $7fa9	ldx #$07
	regX = 0x7;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	cycles += 17;
	regPC = 0x7fab;

case 0x7fab:
	// $7fab	rol $2400,x
	temp = (readMem(0x2400+regX) << 1)  | flagC;
	flagC = temp >> 8;
	writeMem(0x2400+regX, (temp & 0xff));
	// $7fae	rol a
	temp = (regA << 1)  | flagC;
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $7faf	dex
	regX--;
	regX&= 0xff;
	flagZ = (regX == 0) ? 1 : 0;
	flagN = (regX >> 7) & 1;
	// $7fb0	bpl $7fab
	if (flagN == 0) {
		cycles += 13;
		regPC = 0x7fab;
		break;
	};
	// $7fb2	tax
	regX = regA;
	// $7fb3	eor $08
	regA = (regA ^ mem[0x8]) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7fb5	stx $08
	mem[0x8] = regX;
	// $7fb7	php
	mem[0x0100 | regS] = ((flagN<<7)|(flagV<<6)|(0x20)|(flagB<<4)|(flagD<<3)|(flagI<<2)|(flagZ<<1)|flagC);
	regS = (regS - 1) & 0xff;
	// $7fb8	lda #$04
	regA = 0x4;
	// $7fba	sta $3200
	writeMem(0x3200, regA);
	// $7fbd	rol $2003
	temp = (readMem(0x2003) << 1)  | flagC;
	flagC = temp >> 8;
	writeMem(0x2003, (temp & 0xff));
	// $7fc0	rol a
	temp = (regA << 1)  | flagC;
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $7fc1	rol $2004
	temp = (readMem(0x2004) << 1)  | flagC;
	flagC = temp >> 8;
	writeMem(0x2004, (temp & 0xff));
	// $7fc4	rol a
	temp = (regA << 1)  | flagC;
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $7fc5	rol $2407
	temp = (readMem(0x2407) << 1)  | flagC;
	flagC = temp >> 8;
	writeMem(0x2407, (temp & 0xff));
	// $7fc8	rol a
	temp = (regA << 1)  | flagC;
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $7fc9	rol $2406
	temp = (readMem(0x2406) << 1)  | flagC;
	flagC = temp >> 8;
	writeMem(0x2406, (temp & 0xff));
	// $7fcc	rol a
	temp = (regA << 1)  | flagC;
	flagC = temp >> 8;
	regA = (temp & 0xff);
	// $7fcd	rol $2405
	temp = (readMem(0x2405) << 1)  | flagC;
	flagC = temp >> 8;
	writeMem(0x2405, (temp & 0xff));
	// $7fd0	rol a
	temp = (regA << 1)  | flagC;
	regA = (temp & 0xff);
	// $7fd1	tax
	regX = regA;
	// $7fd2	plp
	regS = (regS + 1) & 0xff;
	temp = mem[0x0100 | regS];
	flagN = (temp >> 7) & 1;
	flagV = (temp >> 6) & 1;
	flagB = (temp >> 4) & 1;
	flagD = (temp >> 3) & 1;
	flagI = (temp >> 2) & 1;
	flagZ = (temp >> 1) & 1;
	flagC = temp & 1;
	// $7fd3	bne $7fde
	if (flagZ == 0) {
		cycles += 78;
		regPC = 0x7fde;
		break;
	};
	// $7fd5	eor $0a
	regA = (regA ^ mem[0xa]) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7fd7	bne $7fde
	if (flagZ == 0) {
		cycles += 83;
		regPC = 0x7fde;
		break;
	};
	// $7fd9	tya
	regA = regY;
	// $7fda	eor $09
	regA = (regA ^ mem[0x9]) & 0xff;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	// $7fdc	beq $7fe0
	if (flagZ == 1) {
		cycles += 90;
		regPC = 0x7fe0;
		break;
	};
	cycles += 90;
	regPC = 0x7fde;

case 0x7fde:
	// $7fde	lda #$80
	regA = 0x80;
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 2;
	regPC = 0x7fe0;

case 0x7fe0:
	// $7fe0	sta $3c05
	writeMem(0x3c05, regA);
	// $7fe3	sta $3200
	writeMem(0x3200, regA);
	// $7fe6	sta $3000
	writeMem(0x3000, regA);
	// $7fe9	stx $0a
	mem[0xa] = regX;
	// $7feb	sty $09
	mem[0x9] = regY;
	// $7fed	lda $2007
	regA = readMem(0x2007);
	flagZ = (regA == 0) ? 1 : 0;
	flagN = (regA >> 7) & 1;
	cycles += 22;
	regPC = 0x7ff0;

case 0x7ff0:
	// $7ff0	bpl $7ff0
	if (flagN == 0) {
		cycles += 2;
		regPC = 0x7ff0;
		break;
	};
	// $7ff2	jmp $7e73
	regPC = 0x7e73;
	cycles += 5;
	break;

case 0x7ff5:
	// dc.b $01		; $7ff5
	// dc.b $04		; $7ff6
	// dc.b $05		; $7ff7
	// dc.b $06		; $7ff8
	// dc.b $4e		; $7ff9
	// dc.b $65		; $7ffa
	// dc.b $7b		; $7ffb
	// dc.b $f3		; $7ffc
	// dc.b $7c		; $7ffd
	// dc.b $f3		; $7ffe
	// dc.b $7c		; $7fff
	break;
		};

	};
// ********************** generated code end
};