package structures;
import helpers.CombatHeuristics;
import io.CombatControl;
import io.CombatState;

import java.awt.geom.Point2D;

import objects.Saucer;
import objects.Ship;
import objects.TargetObject;
import utils.Utils;


public class TargetConfiguration {
	
	public Ship ship;

	public TargetObject object;
	
	public TargetInformation info;
	
	public TargetInformation preCalc;
	
	public int angleIndex;	
	
	public int turnTime;
	
	public int preWaitTime;
	
	public int postWaitTime;
	
	public int shootTime;
	
	public int collisionTime;
	
	public boolean isValid;
	
	public boolean isDifficult;
	
	public double prioritization;	
	
	public TargetConfiguration()
	{
		object = null;	
		info = new TargetInformation();
		preCalc = new TargetInformation();
		angleIndex = 0;
		turnTime = 0;
		preWaitTime = 0;
		postWaitTime = 0;
		shootTime = 0;
		collisionTime = Integer.MAX_VALUE;
		isDifficult = false;
		isValid = true;
		prioritization = 1.0;
	}
	
	public void preCalculate(int time, CombatState state, CombatControl control, boolean calcAngleDif)
	{	
		double moveX = ship.getMovementX();
		double moveY = ship.getMovementY();
		if(!state.isMovementPhase())
		{
			if(control.isAccelerating())
			{
				moveX = Math.floor(ship.getSpeedX() + Utils.TABLES.astCos(ship.getAngleIndex()) / 128.0) / 8.0;
				moveY = Math.floor(ship.getSpeedY() + Utils.TABLES.astSin(ship.getAngleIndex()) / 128.0) / 8.0;
			} else {
				double vX = ship.getSpeedX();
				double vY = ship.getSpeedY();
				if(vX > 0)
				{
					vX = Math.max(0, vX - Math.floor(vX) / 128.0 - 1 / 256.0);
				} else if (vX < 0) {
					vX = Math.min(0, vX - Math.floor(vX) / 128.0);
				}
				if(vY > 0)
				{
					vY = Math.max(0, vY - Math.floor(vY) / 128.0 - 1 / 256.0);
				} else if(vY < 0)
				{
					vY = Math.min(0, vY - Math.floor(vY) / 128.0);
				}
				moveX = Math.floor(vX) / 8.0;
				moveY = Math.floor(vY) / 8.0;
			}
		}
		Point2D.Double shotOffset = Utils.TABLES.getShotOffset(angleIndex);
		int shipX = ship.getPosX() + (int)Math.round(moveX * time);
		int shipY = ship.getPosY() + (int)Math.round(moveY * time);
		int objectX = object.getPosX() + (int)Math.round(object.getMovementX() * time);
		int objectY = object.getPosY() + (int)Math.round(object.getMovementY() * time);
		preCalc.difX = Utils.getXDif(shipX + (int)shotOffset.x, objectX);
		preCalc.difY = Utils.getYDif(shipY + (int)shotOffset.y, objectY);
		preCalc.distance = Utils.TABLES.dist(preCalc.difX, preCalc.difY);
		preCalc.angle = Utils.TABLES.atan2(preCalc.difY, preCalc.difX);
		if(calcAngleDif)
		{
			int index = ship.getAngleIndex();
			if(control.isTurningLeft()) 
			{
				index = Utils.incByteAngle(index);
			} else if(control.isTurningRight()) 
			{
				index = Utils.decByteAngle(index);	
			} 
			preCalc.angleDif = Utils.getAngleDif(Utils.TABLES.getAngle(index), preCalc.angle);			
		}
		double realDist = Utils.TABLES.dist(Utils.getXDif(shipX, objectX), Utils.getYDif(shipY, objectY));
		if(realDist > ship.getRadius() + object.getRadius() + 2
			&& preCalc.distance < Utils.TABLES.getShotSpeed(angleIndex) * Utils.PHOTON_TORPEDO_MIN_ALIVE_TIME)
		{
			isValid = true;
		} else {
			isValid = false;
		}
	}
	
	public void calculate(Ship ship, TargetObject object)
	{
		this.ship = ship;
		this.object = object;
		turnTime = 0;
		preWaitTime = 0;
		postWaitTime = 0;
		shootTime = 0;
		collisionTime = Integer.MAX_VALUE;
		isDifficult = false;
		isValid = true;
		this.info.setInfo(ship, object);
	}
	
	public void reCalculate()
	{
		this.info.setInfo(ship, object);
	}
	
	public void setConfig(TargetConfiguration other)
	{
		ship = other.ship;
		object = other.object;
		turnTime = other.turnTime;
		preWaitTime = other.preWaitTime;
		postWaitTime = other.postWaitTime;
		shootTime = other.shootTime;
		collisionTime = other.collisionTime;
		isDifficult = other.isDifficult;
		isValid = other.isValid;
		prioritization = other.prioritization;
		info.setInfo(other.info);
	}
	
	public void passTime(int time)
	{
		if(preWaitTime > 0)
		{
			preWaitTime = Math.max(0, preWaitTime - time);
		}		
		collisionTime = Math.max(0, collisionTime - time);
	}
	
	public boolean isReadyToFire(CombatState state)
	{
		boolean saucerCamp = CombatHeuristics.isSaucerCampingReasonable(state, this);
		if(saucerCamp)
		{
			return object instanceof Saucer 
				&& Math.max(getWaitTime(), turnTime) < 2 
				&& object.getID() > -1;
		} else {
			return Math.max(getWaitTime(), turnTime) < 2 
				&& object.getID() > -1;
		}		
	}
	
	public boolean isBetterThan(double otherRating)
	{
		if(Math.max(getWaitTime(), turnTime) + 2 < collisionTime)
		{
			return getRating() < otherRating;
		} else {
			return false;
		}
	}
	
	public boolean isWorseThan(int otherRating)
	{
		if(Math.max(getWaitTime(), turnTime) + 2 < collisionTime)
		{
			return getRating() > otherRating;
		} else {
			return true;
		}
	}
	
	public double getRating()
	{
		return (Math.max(0, Math.max(preWaitTime, turnTime) - ship.getTimeToNextTorpedo()) + shootTime) * prioritization;
	}
	
	public int getWaitTime()
	{
		return preWaitTime + postWaitTime;
	}
	
	public int getFinalTimeTaken()
	{
		return Math.max(ship.getTimeToNextTorpedo(), Math.max(getWaitTime(), turnTime)) + shootTime;
	}
	
	public int getTimeTaken()
	{
		return Math.max(getWaitTime(), turnTime) + shootTime;
	}
}
