#ifndef OBJECTLINK_H
#define OBJECTLINK_H

#include <vector>
#include <set>

/// \brief Represents a weighted link between two objects.

/**
 *  This class takes two objects and a weight that
 *  describes the likeliness of the link. It is used
 *  when matching objects of consecutive frames, ruling
 *  out ambiguities by comparing and manipulating
 *  the likeliness values.
 */

template<class T>
class ObjectLink
{
public:
    T* prev;      ///<  Object of the previous frame.
    T* curr;      ///<  Object of the current frame.
    double match; ///<  Link-likeliness value.



    /// \brief Property for derived classes, to make them linkable with ObjectLink.

    /**
     *  This class equips a derived class with the properties
     *  required to be handled by the @see ObjectLink class.
     */
    class Linkable {
    public:
        std::set<ObjectLink<T>*> linkToNext;
        std::set<ObjectLink<T>*> linkToPrev;
        virtual ~Linkable() {
            while (linkToNext.size() > 0)
                delete (*(linkToNext.begin()));
            while (linkToPrev.size() > 0)
                delete (*(linkToPrev.begin()));
        }
    };



    ObjectLink(T* prev, T*curr, double match)
    : prev(prev), curr(curr), match(match)
    {
        prev->linkToNext.insert(this);
        curr->linkToPrev.insert(this);
    }

    ~ObjectLink()
    {
        prev->linkToNext.erase(this);
        curr->linkToPrev.erase(this);
    }



    /// \brief Remove obviously superflous links.

    /**
     *  This subroutine eliminates links between objects that are obviously
     *  unlikely. The rationale is as follows: Several possible linkes
     *  exist from previous to current frame, and from current to previous
     *  frame. However, if there is only one link in any direction, then obviously
     *  all non-matching links in the oposite direction are invalid.
     *
     *  This leads to an iterative cleanup, until no further links can be removed.
     */
    static void eliminateObvious(std::vector<T*> &prev, std::vector<T*> &curr)
    {
        int nlast = prev.size();
        int ncurr = curr.size();
        bool change = true;
        while (change) {
                change = false;

                // obvious cases (unique links backward)
                for (int icurr = 0;icurr<ncurr;icurr++)
                {
                    T* acurr = curr[icurr];
                    int count = acurr->linkToPrev.size();

                    if (count==1) {
                    // if unique link back to last, make sure this
                    // is also the only link from the past.
                        ObjectLink<T>* link = *(acurr->linkToPrev.begin());
                        T* alast = link->prev;
                        if (alast->linkToNext.size()>1) {
                            change = true;
                            alast->linkToNext.erase(link);
                            while (alast->linkToNext.size()>0)
                                delete (*(alast->linkToNext.begin()));
                            alast->linkToNext.insert(link);
                        }
                    }
                }

                // obvious cases (unique links forward)
                for (int ilast = 0;ilast<nlast;ilast++)
                {
                    T* alast = prev[ilast];
                    int count = alast->linkToNext.size();

                    if (count==1) {
                    // if unique link forward, make sure this
                    // is also the only link backwards.
                        ObjectLink<T>* link = *(alast->linkToNext.begin());
                        T* acurr = link->curr;
                        if (acurr->linkToPrev.size()>1) {
                            change = true;
                            acurr->linkToPrev.erase(link);
                            while (acurr->linkToPrev.size()>0)
                                delete (*(acurr->linkToPrev.begin()));
                            acurr->linkToPrev.insert(link);
                        }
                    }
                }


                // remove improbable cases (backward)
                for (int icurr = 0;icurr<ncurr;icurr++)
                {
                    T* acurr = curr[icurr];
                    int count = acurr->linkToPrev.size();

                    if (count>1) {

                        // determine maximum probability
                        typename std::set<ObjectLink<T>*>::iterator i = acurr->linkToPrev.begin();
                        double maxprob = (*i)->match;
                        i++;
                        for(;i!=acurr->linkToPrev.end();i++) {
                            double prob = (*i)->match;
                            maxprob = maxprob>prob?maxprob:prob;
                        }

                        // remove cases with a significantly lower probability
                        for(i=acurr->linkToPrev.begin();i!=acurr->linkToPrev.end();i++) {
                            double prob = (*i)->match;
                            if (maxprob-prob > 1) {
                                change = true;
                                delete (*i);
                                break; // better break, because iterator integrity is broken.
                            }
                        }
                    }
                }

                // remove improbable cases (forward)
                for (int ilast = 0;ilast<nlast;ilast++)
                {
                    T* alast = prev[ilast];
                    int count = alast->linkToNext.size();

                    if (count>1) {

                        // determine maximum probability
                        typename std::set<ObjectLink<T>*>::iterator i = alast->linkToNext.begin();
                        double maxprob = (*i)->match;
                        i++;
                        for(;i!=alast->linkToNext.end();i++) {
                            double prob = (*i)->match;
                            maxprob = maxprob>prob?maxprob:prob;
                        }

                        // remove cases with a significantly lower probability
                        for(i=alast->linkToNext.begin();i!=alast->linkToNext.end();i++) {
                            double prob = (*i)->match;
                            if (maxprob-prob > 1) {
                                change = true;
                                delete (*i);
                                break; // better break, because iterator integrity is broken.
                            }
                        }
                    }
                }

            }
    }


    static void balanceInfo(std::vector<T*> &prev, std::vector<T*> &curr)
    {
            int nlast = prev.size();
            int ncurr = curr.size();
            int count_none = 0;
            int count_mult = 0;
            for (int icurr = 0;icurr<ncurr;icurr++)
            {
                T* acurr = curr[icurr];
                int count = acurr->linkToPrev.size();

                if (count==0)
                    ++count_none;
                else if (count>1)
                    ++count_mult;
            }
            int count_kill = 0;
            for (int ilast = 0;ilast<nlast;ilast++)
            {
                T* alast = prev[ilast];
                int count = alast->linkToNext.size();

                if (count==0)
                    ++count_kill;
            }

            if (count_mult)
                printf("%s: New objects: %d   Dead objects: %d   Confused objects: %d\n",T().className().c_str(), count_none,count_kill,count_mult);
    }


    /// \brief In case of multiple links, remove all but the most likely ones.

    static void eliminateAll(std::vector<T*> &prev, std::vector<T*> &curr)
    {
        int nlast = prev.size();
        int ncurr = curr.size();

        // first, eliminate cases with a lower priority
        bool change = true;
        while (change) {
            change = false;

            for (int icurr = 0;icurr<ncurr;icurr++)
            {
                T* acurr = curr[icurr];
                int count = acurr->linkToPrev.size();

                if (count>1) {

                    // determine maximum probability
                    typename std::set<ObjectLink<T>*>::iterator i = acurr->linkToPrev.begin();
                    double maxprob = (*i)->match;
                    i++;
                    for(;i!=acurr->linkToPrev.end();i++) {
                        double prob = (*i)->match;
                        maxprob = maxprob>prob?maxprob:prob;
                    }

                    // remove a case with a lower probability
                    for(i=acurr->linkToPrev.begin();i!=acurr->linkToPrev.end();i++) {
                        double prob = (*i)->match;
                        if (maxprob-prob > 0) {
                            change = true;
                            delete (*i);
                            break;
                        }
                    }
                    if (change) break;
                }
            }
            if (change) eliminateObvious(prev,curr);
        }

        // finally, keep only the first occuring case
        // (in case of equal probabilities)
        change = true;
        while (change) {
            change = false;

            for (int icurr = 0;icurr<ncurr;icurr++)
            {
                T* acurr = curr[icurr];
                int count = acurr->linkToPrev.size();

                if (count>1) {

                    // determine maximum probability
                    typename std::set<ObjectLink<T>*>::iterator i = acurr->linkToPrev.begin();
                    i++;
                    delete(*i);
                    change = true;
                    break;
                }
            }
            if (change) eliminateObvious(prev,curr);
        }

    }

};


#endif
