#ifndef FRAMESTATUS_AIMFORTARGETS_H
#define FRAMESTATUS_AIMFORTARGETS_H

#include "cmath"
#include "FrameStatus.h"
#include "../MovingObjects/Point.h"

/// For each turn step (left or right) required to aim for
/// the target, the distance is incremented by this fraction.
const double TURN_PENALTY = 0.02;

/// Maximum distance value. Above that, distances are truncated.
const double OUTER_DIST = 300;

/// Maximum shoting distance.
const double MAX_SHOT_DIST = 440;

inline
FrameStatus::Aim
FrameStatus::aimForTargets()
{

    int nasteroids = asteroids.size();
    const double a = Ship::stepAngle[ship->step]/180.*M_PI;

    double min_dist = 1e6;

    /// In a preliminary round, check where the majority of
    /// nearby asteroids can be found.

    double prevx = 0;
    double prevy = 0;
    for (int i=0; i<nasteroids; ++i)
    {

        /// Asteroids that have already been shot at are
        /// only included, if they might break into
        /// smaller ones.
        if (asteroids[i]->sf==14) {
            int prevShots = 0;
            for (int j=0;j<4;j++)
                if (asteroids[i]->underfire[j]<MAX_SHOT_AGE) prevShots++;
            if (prevShots>=MAX_SHOT_CNT) continue;
        }

        double dx = asteroids[i]->x - ship->getx();
        double dy = asteroids[i]->y - ship->gety();
        normalize(dx,dy);
        double dist = sqrt(dx*dx+dy*dy);

        if (dist<min_dist) min_dist = dist;

        prevx += dx*(100/dist);
        prevy += dy*(100/dist);
    }
    double prevd = sqrt(prevx*prevx+prevy*prevy);
    prevx /= prevd;
    prevy /= prevy;
    double preva  = atan2(prevx,prevy);

    if (ufo)
    {
        double dx = ufo->x - ship->getx();
        double dy = ufo->y - ship->gety();
        normalize(dx,dy);
        double dist = sqrt(dx*dx+dy*dy);

        if (dist<min_dist) min_dist = dist;
    }

    freeRadius = min_dist;


    Aim result;

    min_dist = 1e6;

    // aim for asteroids
    for (int i=0; i<nasteroids; ++i)
    {

        /// If there are many asteroids, do not
        /// aim at a target that was already shot for.
        if (nasteroids>2) {
            int prevShots = 0;
            for (int j=0;j<4;j++)
                if (asteroids[i]->underfire[j]<MAX_SHOT_AGE) prevShots++;
            if (prevShots>=MAX_SHOT_CNT) continue;
        }

        /// One thing to consider is the distance to the
        /// target (at the time of impact)
        Point pos = asteroids[i]->getPos_shotBy(*ship,false);
        double dx = pos.x - ship->getx();
        double dy = pos.y - ship->gety();
        normalize(dx,dy);
        double dist = sqrt(dx*dx+dy*dy);

        //if (dist>MAX_SHOT_DIST) continue;

        /// If the distance is beyound a certain treshhold,
        /// the excess distance is scaled down. Thus, far
        /// targets are valued rather by the angle offset
        /// than by the sheer distance.
        if (dist>OUTER_DIST)
            dist = OUTER_DIST + (dist-OUTER_DIST)*0.1;

        /// The other thing to consider is the required
        /// turning of our ship,
        /// which costs extra time, and might bring other
        /// valuable targets out of range.
        double da = atan2(dx,dy)-a;
        normalize(da);
        double turns = da/SHIP_ROT;

        /// Now, we say that turning the ship increases distance
        /// by some percent per turn step
        dist *= pow(1.+TURN_PENALTY,fabs(turns))/(1.+TURN_PENALTY);


        /// Also, smaller targets are prefered (clean up the small
        /// pieces after cracking a big asteroid)
        if (asteroids[i]->sf== 0) dist *= 1.2;
        if (asteroids[i]->sf==15) dist *= 1.1;


        if (dist < min_dist)
        {
            min_dist = dist;
            result.dist = dist;
            result.dx = roundd(dx);
            result.dy = roundd(dy);
            result.target = asteroids[i];
        }
    }

    // aim for saucers
    if (ufo)
    {
        Point pos = ufo->getPos_shotBy(*ship);
        double dx = pos.x - ship->getx();
        double dy = pos.y - ship->gety();
        normalize(dx,dy);
        double dist = sqrt(dx*dx+dy*dy);
        dist = dist>OUTER_DIST?OUTER_DIST:dist;

        if (0.6*dist < min_dist)
        {
            min_dist = dist;
            result.dist = dist;
            result.dx = roundd(dx);
            result.dy = roundd(dy);
            result.target = ufo;
        }
    }



    if (result.target) {

        /// Final check: If the target is straight in front
        /// of our ship, and departing, we aim at a periodic
        /// equivalent.

        double tu = result.target->getu();
        double tv = result.target->getv();
        double align = (tu*ship->getu() + tv*ship->getv())
                     / (sqrt(tu*tu+tv*tv)*(ship->speed));

        if (result.dist > 180 && align>cos(3./180.*M_PI)) {

            printf("Decide to turn.\n");

            /// There are two relevant choices, left and
            /// right. We compute both positions and chose
            /// the appropriate one.

            double x1;
            double y2;

            if (tu>0) x1 = result.dx -1024.;
            else      x1 = result.dx +1024.;

            if (tv>0) y2 = result.dy -768.;
            else      y2 = result.dy +768.;

            double d1 = sqrt(x1*x1+result.dy*result.dy);
            double d2 = sqrt(y2*y2+result.dx*result.dx);

            if (d1<d2) result.dx = x1;
            else       result.dy = y2;
        }
    }


    return result;

}

#endif
