// ----------------------------------------------------------------------------
// 'The Sniper'  -  c't Asteroids bot  -  Thorsten Denhard, 2008
// ----------------------------------------------------------------------------

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <winsock2.h>

#include "networkengine.h"

NetworkEngine::NetworkEngine (SOCKET socket, ADDRESS serverIP) : 
  mSocket                       (socket),
  mServerIP                     (serverIP),
  mFrame                        (),
  mActionList                   (NULL),
  mActionListStart              (0),
  mActionListSize               (0),
  mPrevFrame                    (0),
  mPrevFullFrameNumber          (0),
  mFullFrameNumber              (0),
  mLastDeliveredFullFrameNumber (0),
  mFrameDelay                   (0),
  mFrameLoss                    (0),
  mPing                         (0),
  mLeftRightBalance             (0),
  mLastLeftRightBalance         (0),
  mActionsByPing                (),
  mReceiveCountsByPing          (),
  mPrioAction1                  (),
  mPrioAction2                  (),
  mLastAction                   (),
  mPrioActionFlag               (0)
{
  mActionList          = new KeysPacket[512];
  mActionsByPing       = new KeysPacket[256];
  mReceiveCountsByPing = new int[256];
  
  int i = 0;

  for (i=0; i<256; ++i)
  {
    mReceiveCountsByPing[i] = 0;
  }
}

NetworkEngine::~NetworkEngine ()
{
  delete[] mActionList;
  delete[] mActionsByPing;
  delete[] mReceiveCountsByPing;
}

void 
NetworkEngine::removeScheduledActions ()
{
  mActionListStart = 0;
  mActionListSize  = 0;
}

void 
NetworkEngine::scheduleAction (const KeysPacket& keys)
{
  int backIndex = (mActionListStart + mActionListSize) % 512;
  mActionList[backIndex] = keys;
  mActionListSize += 1;
}

void 
NetworkEngine::getLatestFramePacket (FramePacket& packetRet, int& frameDelayRet)
{
  packetRet                     = mFrame;
  frameDelayRet                 = mFrameDelay;
  mLastDeliveredFullFrameNumber = mFullFrameNumber;
}

void 
NetworkEngine::receivePacket ()
{
  sockaddr_in sender;
  int sender_size = sizeof sender;
  fd_set readfds, writefds, exceptfds;

  do
  {
    FD_ZERO(&readfds);
    FD_ZERO(&writefds);
    FD_ZERO(&exceptfds);
    FD_SET(mSocket, &readfds);
    FD_SET(mSocket, &exceptfds);

    select(int (mSocket) + 1, &readfds, &writefds, &exceptfds, NULL);
    
    {
      int bytes_received = recv (mSocket, (char *)&mFrame, sizeof mFrame, 0);

      if (bytes_received != sizeof mFrame)
      {
        int err = WSAGetLastError();
        printf ("! Error: recvfrom (%d)\n", err);
        exit(1);
      }

      unsigned long int oldNum = mPrevFullFrameNumber;
      
      if (mFrame.mFrameNum > mPrevFrame)
      {
        mPrevFullFrameNumber   += (mFrame.mFrameNum - mPrevFrame);
        mFullFrameNumber        = mPrevFullFrameNumber;
        mPrevFrame              = mFrame.mFrameNum;
      }
      else if (mPrevFrame - mFrame.mFrameNum > 128)
      {
        // Wrap-around
        mPrevFullFrameNumber   += (255 - (mPrevFrame - mFrame.mFrameNum) + 1);
        mFullFrameNumber        = mPrevFullFrameNumber;
        mPrevFrame              = mFrame.mFrameNum;    
      }
      else
      {
        // Too many packet losses to keep track. Start counting again.
        mPrevFrame                 = mFrame.mFrameNum;
        mPrevFullFrameNumber       = mPrevFrame;
        mFullFrameNumber           = mPrevFullFrameNumber;
      }
      
      mFrameLoss = mFullFrameNumber - oldNum - 1;
      
      int oPing = mFrame.mPing;
      int nPing = mPing;
      
      if (oPing == nPing)
      {
        mFrameDelay = 0;
      }
      else if (nPing > oPing)
      {
        mFrameDelay = nPing - oPing;
      }
      else if (oPing - nPing > 128)
      {
        // Wrap-around
        mFrameDelay = 255 - (oPing - nPing) + 1;
      }
      else
      {
        // Too many packet losses to keep track. 
        mFrameDelay = 0;
      }
      
      // Left-right balance must be updated at receive time, not while sending!
      // Only now do we know how many mame-frames have been influenced by the last command.
      
      if (mActionsByPing[oPing].isLeft ())
      {
        mLeftRightBalance += mFrameLoss;
      }      
      else if (mActionsByPing[oPing].isRight ())
      {
        mLeftRightBalance -= mFrameLoss;
      }      

      mLastLeftRightBalance = mLeftRightBalance;
      
      if (mActionsByPing[oPing].isLeft ())
      {
        mLeftRightBalance += 1;
      }      
      else if (mActionsByPing[oPing].isRight ())
      {
        mLeftRightBalance -= 1;
      }      

      // It's all bullshit :-( Take some desparate measures to cope with the 
      // case of "minor action loss", i.e. sporadic ping-mismatches of 1 with 
      // no frame loss.
      // This is the case I encountered in a LAN-setup; anything more serious is
      // definitely _not_ correctly handled...
      // Now heres the case: 
      //  - we sent action-17
      //  - we received backping-17
      //  - we sent action-18
      //  - we just received backping-17 again
      //  Most likely we will send (in the near future) action-19 early enough again, so 
      //  that effectively action-18 is not evaluated at all! Instead action-17 is doubly 
      //  evalutated...
      //  Now what to do?      
      //  
      //     L     R     F     H     N    [Second]
      //
      // L   -     RR    RF    RH    R
      //
      // R   LL    -     LF    LH    L
      //
      // F   L     R     -     H     N
      //
      // H   L     R     F     -     N
      //
      // N   L     R     F     H     -

      mReceiveCountsByPing[oPing] += 1;

      int surplus = mReceiveCountsByPing[oPing] - 1;
      
      if (surplus         >  0 &&
          mPrioActionFlag == 0)
      {
        if (mActionsByPing[oPing].isLeft ())
        {
          if (mLastAction.isLeft ())
          {
            AST_NOP;
          }
          else if (true != mLastAction.isRight      () &&
                   true != mLastAction.isFire       () &&
                   true != mLastAction.isHyperspace ())
          {
            // Only counteraction, last action was nop
            mPrioAction2    = KeysPacket::RIGHT ();        
            mPrioActionFlag = 1;
          }
          else
          {
            // Counteract and repeat
            mPrioAction1    = KeysPacket::RIGHT ();        
            mPrioAction2    = mLastAction;        
            mPrioActionFlag = 2;
          }
        }
        else if (mActionsByPing[oPing].isRight ())
        {
          if (mLastAction.isRight ())
          {
            AST_NOP;
          }
          else if (true != mLastAction.isLeft       () &&
                   true != mLastAction.isFire       () &&
                   true != mLastAction.isHyperspace ())
          {
            // Only counteraction, last action was nop
            mPrioAction2    = KeysPacket::LEFT ();        
            mPrioActionFlag = 1;
          }
          else
          {
            // Counteract and repeat
            mPrioAction1    = KeysPacket::LEFT ();        
            mPrioAction2    = mLastAction;        
            mPrioActionFlag = 2;
          }
        }
        else
        {
          bool same = false;

          if (mActionsByPing[oPing].isFire () && 
              mLastAction          .isFire ())
          {
            same = true;
          }
          else if (mActionsByPing[oPing].isHyperspace () && 
                   mLastAction          .isHyperspace ())
          {
            same = true;
          }
          else if (true != mLastAction          .isLeft       () &&
                   true != mLastAction          .isRight      () &&
                   true != mLastAction          .isFire       () &&
                   true != mLastAction          .isHyperspace () &&
                   true != mActionsByPing[oPing].isLeft       () &&
                   true != mActionsByPing[oPing].isRight      () &&
                   true != mActionsByPing[oPing].isFire       () &&
                   true != mActionsByPing[oPing].isHyperspace ())
          {
            same = true;
          }

          if (true != same)
          {
            // No counteraction, just repeat last action
            mPrioAction2    = mLastAction;        
            mPrioActionFlag = 1;
          }
        }
      }
    }

    FD_ZERO(&readfds);
    FD_ZERO(&writefds);
    FD_ZERO(&exceptfds);
    FD_SET(mSocket, &readfds);
    timeval zero;
    zero.tv_sec = zero.tv_usec = 0;
    select(int (mSocket) + 1, &readfds, &writefds, &exceptfds, &zero);
  } while(FD_ISSET(mSocket, &readfds));
}

void 
NetworkEngine::sendPacket ()
{
  sockaddr_in server;
  memset(&server, 0, sizeof server);
  server.sin_family = AF_INET;
  server.sin_port = htons(1979);
  server.sin_addr.s_addr = mServerIP;

  {
    mPing += 1;

    KeysPacket keys;
    bool fromList = false;

    if (mPrioActionFlag == 2)
    {
      keys = mPrioAction1;
      mPrioActionFlag = 1;
    }
    else if (mPrioActionFlag == 1)
    {
      keys = mPrioAction2;
      mPrioActionFlag = 0;
    }
    else if (mActionListSize > 0)
    {
      keys = mActionList[mActionListStart];
      fromList = true;
    }

    keys.mPing = mPing;
    
    bool ok = (sizeof (keys) == 
               sendto (mSocket, (char *)&keys, sizeof keys, 0, (sockaddr*)&server, sizeof server));

    if (! ok)
    {
      int err = WSAGetLastError();
      if (err != WSAEWOULDBLOCK)
      {
        printf ("! Error: sendto (%d)\n", err);
        exit(1);
      }
      else
      {
        printf ("! Suspicious: WSAEWOULDBLOCK\n");
      }
      
      mPing -= 1;
    }
    else
    {
      // Packet sent, update left-right-balance and
      // remove from scheduled list

      mActionsByPing      [mPing] = keys;
      mReceiveCountsByPing[mPing] = 0;

      if (true == fromList)
      {
        mActionListSize  -= 1;
        mActionListStart  = (mActionListStart + 1) % 512;
      }
      
      mLastAction = keys;
    }
  }
}

void 
NetworkEngine::forceLeftRightBalance (unsigned char value)
{
  mLeftRightBalance     = value;
  mLastLeftRightBalance = value;

  int i = 0;

  for (i=0; i<256; ++i)
  {
    mActionsByPing      [i] = KeysPacket ();
    mReceiveCountsByPing[i] = 0;
  }
}

unsigned char
NetworkEngine::getLeftRightBalance ()
{
  return mLeftRightBalance;
}

unsigned char
NetworkEngine::getLastLeftRightBalance ()
{
  return mLastLeftRightBalance;
}

