package de.curdreinert.asteroids.display;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.geom.AffineTransform;
import java.awt.image.BufferedImage;

import javax.swing.JComponent;

import de.curdreinert.asteroids.geometry.Position;
import de.curdreinert.asteroids.geometry.Vector;
import de.curdreinert.asteroids.intelligence.Analysis;
import de.curdreinert.asteroids.screenobject.ScreenObject;
import de.curdreinert.asteroids.screenobject.Ship;

public class Display extends JComponent {

	private static final int Y_BASE = 128;

	private static final int BORDER = 8;

	private static final int WIDTH = 1024;

	private static final int HEIGHT = 768;

	private static final Dimension DIMENSION = new Dimension(
			WIDTH + 2 * BORDER, HEIGHT + 2 * BORDER);

	private Analysis analysis;

	private Graphics2D graphics;

	private VectorPlotter vectorPlotter = new VectorPlotter();

	public Display() {
		super();
		setSize(DIMENSION);
		setPreferredSize(DIMENSION);
	}

	public void update(Graphics g) {
		// don't
	}

	public void display(Analysis analysis) {
		this.analysis = analysis;
		repaint();
	}

	public void paint(Graphics g) {
		Graphics2D graphics2D = (Graphics2D) g;

		BufferedImage image = new BufferedImage((int) DIMENSION.getWidth(),
				(int) DIMENSION.getHeight(), BufferedImage.TYPE_3BYTE_BGR);
		graphics = image.createGraphics();
		graphics.setTransform(new AffineTransform(1.0, 0.0, 0.0, -1.0, BORDER,
				HEIGHT + BORDER + Y_BASE));
		graphics.setBackground(Color.BLACK);
		graphics.clearRect(-BORDER, -BORDER, WIDTH + BORDER, HEIGHT + BORDER);
		graphics.setPaintMode();
		graphics.setStroke(new BasicStroke(2));
		if (analysis != null) {
			vectorPlotter.plot(graphics, analysis.getScreen().getData());
			drawAnalysis();
		}
		graphics2D.drawImage(image, 0, 0, null);

	}

	private void drawAnalysis() {
		Ship ship = analysis.getScreen().getShip();
		if (ship != null) {
			mark(ship, 20, Color.YELLOW);
			drawVector(ship, ship.getViewDirection(), Color.YELLOW);
			drawVector(ship, analysis.getTargetVector(), Color.BLUE);
		}
		mark(analysis.getTarget(), 18, Color.BLUE);
		mark(analysis.getTargetTrajectory(), 1, Color.BLUE);
		mark(analysis.getProspectedTargetPosition(), 5, Color.BLUE);
		mark(analysis.getDangerousShots().toArray(new ScreenObject[0]), 5,
				Color.RED);
		mark(analysis.getHarmlessShots().toArray(new ScreenObject[0]), 5,
				Color.GREEN);
		mark(analysis.getDangerousAsteroids().toArray(new ScreenObject[0]), 20,
				Color.RED);
		mark(analysis.getHarmlessAsteroids().toArray(new ScreenObject[0]), 20,
				Color.GREEN);
	}

	private void drawVector(ScreenObject from, Vector vector, Color color) {
		if (from != null) {
			drawVector(from.getPosition(), vector, color);
		}
	}

	private void drawVector(Position from, Vector vector, Color color) {
		if (from == null || vector == null) {
			return;
		}
		graphics.setColor(color);
		graphics.drawLine(from.getX(), from.getY(), from.getX()
				+ (int) vector.getX(), from.getY() + (int) vector.getY());
	}

	private void mark(ScreenObject[] objects, int radius, Color color) {
		if (objects == null) {
			return;
		}
		for (ScreenObject object : objects) {
			mark(object, radius, color);
		}
	}

	private void mark(ScreenObject object, int radius, Color color) {
		if (object == null) {
			return;
		}
		mark(object.getPosition(), radius, color);
		if (object.getMovement() != null) {
			drawVector(object.getPosition(), object.getMovement().multiply(10),
					color);
		}
	}

	private void mark(Position[] trajectory, int radius, Color color) {
		if (trajectory != null) {
			for (Position position : trajectory) {
				mark(position, radius, color);
			}
		}
	}

	private void mark(Position position, int radius, Color color) {
		if (position != null) {
			graphics.setColor(color);
			graphics.drawOval(position.getX() - radius, position.getY()
					- radius, 2 * radius, 2 * radius);
		}
	}
}
