#ifndef REPOSITORY_
#define REPOSITORY_

#include "Universe.h"
#include "Snapshot.h"
#include "Keys.h"
#include "Action.h"
#include "rdtsc.h"
#include "defines.h"

#include <wx/string.h>
#include <wx/thread.h>



//verwaltet den Datenaustausch und die Mutexe
class Repository
{
private:
	Action udpIDToActionMapping[256];
	Universe* universe;
	char keys;
	unsigned char udpID;
	char lastOutput;
	Snapshot* snapshot;
	
	#ifndef _BOT_SINGLETHREADED_
	wxMutex*     universeMutex;
	wxCondition* universeCondition;
	
	wxMutex*     keysMutex;
	wxCondition* keysCondition;
	
	wxMutex* 		snapshotMutex;
	wxCondition* 	snapshotCondition;
	
	wxMutex* 		actionMutex;
	wxCondition* 	actionCondition;
	#endif
	
	#ifdef _MEASURE_TIME_
	time_tsc setSnapshotTime;
	time_tsc popSnapshotTime;
	time_tsc setUniverseTime;
	time_tsc popUniverseTime;
	#endif
public:
	Repository();
	
	Universe* popUniverse();
	void      setUniverse(Universe* universe);
	
	char getLastOutput();
	char popKeys();
	void setKeys(char keys);
	
	Action getAction(unsigned char index);
	Action touchAction(unsigned char index);
	unsigned char setAction(Action action);
	
	Angle estimatedAngleOffset(unsigned char start);
	Angle clearActions(unsigned char start, unsigned char end);
	Angle missedAngleOffset(unsigned char start);
	
	void setSnapshot(Snapshot* snapshot);
	Snapshot* popSnapshot();
	
	void setShoot(bool);
	void setHyperspace(bool);
	void setAccelerate(bool);
	void setRight(bool);
	void setLeft(bool);
	
	unsigned char getUDPID()
	{
		return udpID;
	}
};
	

inline unsigned char Repository::setAction(Action action)
{
	#ifndef _BOT_SINGLETHREADED_
	wxMutexLocker lock(*actionMutex);
	#endif

	udpIDToActionMapping[udpID] = action;
	
	#ifndef _BOT_SINGLETHREADED_
	actionCondition->Signal();
	#endif
	
	return udpID++;	
}

inline Action Repository::getAction(unsigned char index)
{
	Action action;
	#ifndef _BOT_SINGLETHREADED_
	wxMutexLocker lock(*actionMutex);
	#endif
	
	action = udpIDToActionMapping[index];
	//udpIDToActionMapping[index].keys = Key::NONE;
	
	#ifdef _BOT_MULTITHREADED_
	actionCondition->Signal();
	#endif 
	
	return action;
}

inline Angle Repository::estimatedAngleOffset(unsigned char start)
{
	
	#ifndef _BOT_SINGLETHREADED_
	wxMutexLocker lock(*actionMutex);
	#endif
	Angle offset = 0;
	
	while(start != udpID)
	{
		if(udpIDToActionMapping[start].keys & Key::LEFT)
			offset += 3;
		
		if(udpIDToActionMapping[start].keys & Key::RIGHT)
			offset -= 3;
			
		start++;
	}
	
	#ifndef _BOT_SINGLETHREADED_
	actionCondition->Signal();
	#endif 
	
	return offset;
}

inline Angle Repository::clearActions(unsigned char start, unsigned char end)
{
	
	#ifndef _BOT_SINGLETHREADED_
	wxMutexLocker lock(*actionMutex);
	#endif
	Angle offset = 0;
	
	while(start != end)
	{		
		udpIDToActionMapping[start].keys = Key::NONE;		
		start++;
	}
	
	#ifndef _BOT_SINGLETHREADED_
	actionCondition->Signal();
	#endif 
	
	return offset;
}

inline Angle Repository::missedAngleOffset(unsigned char start)
{
	
	#ifndef _BOT_SINGLETHREADED_
	wxMutexLocker lock(*actionMutex);
	#endif
	Angle offset = 0;
	
	while(start != udpID)
	{
		if(udpIDToActionMapping[start].keys & Key::LEFT)
			offset += 3;
		
		if(udpIDToActionMapping[start].keys & Key::RIGHT)
			offset -= 3;
			
		udpIDToActionMapping[start].keys = Key::NONE;
			
		start++;
	}
	
	#ifndef _BOT_SINGLETHREADED_
	actionCondition->Signal();
	#endif 
	
	return offset;
}
	

inline Universe* Repository::popUniverse()
{
	#ifdef _MEASURE_TIME_	
	time_tsc start, end;
	start = getMicroSeconds();	
	//wxLogDebug("Repository::popUniverse start %lld",start);
	#endif
	
	#ifndef _BOT_SINGLETHREADED_
	wxMutexLocker lock(*universeMutex);
	while(!universe)
		universeCondition->Wait();
	#endif
	
	Universe* result = universe;
	universe = 0;
	
	#ifdef _MEASURE_TIME_
	end = getMicroSeconds();
	popUniverseTime = end;
	wxLogDebug("Time to pop Universe %lld",popUniverseTime-setUniverseTime);
	wxLogDebug("Repository::popUniverse end %lld ==> time %lld",end, end- start);
	#endif
	
	return result;
}

inline void Repository::setUniverse(Universe* universe)
{
	#ifdef _MEASURE_TIME_	
	time_tsc start, end;
	start = getMicroSeconds();
	//wxLogDebug("Repository::setUniverse start %lld",start);
	#endif
	
	#ifndef _BOT_SINGLETHREADED_
	wxMutexLocker lock(*universeMutex);
	#endif
	
	this->universe = universe;
	
	#ifndef _BOT_SINGLETHREADED_
	universeCondition->Signal();
	#endif
	
	#ifdef _MEASURE_TIME_
	end = getMicroSeconds();
	setUniverseTime = end;
	wxLogDebug("Repository::setUniverse end %lld ==> time %lld",end,end-start);
	#endif
}

inline char Repository::getLastOutput()
{
	return lastOutput;
}

inline char Repository::popKeys()
{
	#ifndef _BOT_SINGLETHREADED_
	wxMutexLocker lock(*keysMutex);
	while(keys == Key::EMPTY)
		keysCondition->Wait();
	#endif
	
	lastOutput = keys;
	keys = Key::EMPTY;
	return lastOutput;
}

inline void Repository::setKeys(char keys)
{
	#ifndef _BOT_SINGLETHREADED_
	wxMutexLocker lock(*keysMutex);
	#endif
	
	this->keys = keys;
	
	#ifndef _BOT_SINGLETHREADED_
	keysCondition->Signal();
	#endif
}

inline void Repository::setShoot(bool set)
{	
	#ifndef _BOT_SINGLETHREADED_
	wxMutexLocker lock(*keysMutex);
	#endif
	
	keys = set ? keys | Key::SHOOT : keys & ~Key::SHOOT;
	
	#ifndef _BOT_SINGLETHREADED_
	keysCondition->Signal();
	#endif
}

inline void Repository::setHyperspace(bool set)
{	
	#ifndef _BOT_SINGLETHREADED_
	wxMutexLocker lock(*keysMutex);
	#endif
	keys = set ? keys | Key::HYPERSPACE : keys & ~Key::HYPERSPACE;
	
	#ifndef _BOT_SINGLETHREADED_
	keysCondition->Signal();
	#endif
}

inline void Repository::setAccelerate(bool set)
{	
	#ifndef _BOT_SINGLETHREADED_
	wxMutexLocker lock(*keysMutex);
	#endif
	
	keys = set ? keys | Key::ACCELERATE : keys & ~Key::ACCELERATE;
	
	#ifndef _BOT_SINGLETHREADED_
	keysCondition->Signal();
	#endif
}

inline void Repository::setRight(bool set)
{	
	#ifndef _BOT_SINGLETHREADED_
	wxMutexLocker lock(*keysMutex);
	#endif
	
	keys = set ? keys | Key::RIGHT : keys & ~Key::RIGHT;
	
	#ifndef _BOT_SINGLETHREADED_
	keysCondition->Signal();
	#endif
}

inline void Repository::setLeft(bool set)
{	
	#ifndef _BOT_SINGLETHREADED_
	wxMutexLocker lock(*keysMutex);
	#endif
	
	keys = set ? keys | Key::LEFT : keys & ~Key::LEFT;
	
	#ifndef _BOT_SINGLETHREADED_
	keysCondition->Signal();
	#endif
}

inline void Repository::setSnapshot(Snapshot* snapshot)
{
	#ifdef _MEASURE_TIME_	
	time_tsc start, end;
	start = getMicroSeconds();
	//wxLogDebug("Repository::setSnapshot start %lld",start);
	#endif
	
	#ifndef _BOT_SINGLETHREADED_
	wxMutexLocker lock(*snapshotMutex);
	#endif
	
	this->snapshot = snapshot;
	
	#ifndef _BOT_SINGLETHREADED_
	snapshotCondition->Signal();
	#endif
	
	#ifdef _MEASURE_TIME_
	end = getMicroSeconds();
	setSnapshotTime=end;
	wxLogDebug("Repository::setSnapshot end %lld ==> time %lld",end,end-start);
	#endif
}

inline Snapshot* Repository::popSnapshot()
{
	#ifdef _MEASURE_TIME_	
	time_tsc start, end;
	start = getMicroSeconds();
	
	
	//wxLogDebug("Repository::poptSnapshot start %lld",start);
	#endif
	
	#ifndef _BOT_SINGLETHREADED_
	wxMutexLocker lock(*snapshotMutex);
	
	while(!snapshot)
		snapshotCondition->Wait();
	#endif
	
	Snapshot * result= snapshot;
	
	snapshot= 0;
	
	#ifdef _MEASURE_TIME_
	end = getMicroSeconds();
	popSnapshotTime = end;
	wxLogDebug("Time to pop Snapshot %lld",popSnapshotTime - setSnapshotTime);
	wxLogDebug("Repository::popSnapshot end %lld ==> time %lld",end,end-start);
	#endif
	
	return result;
}

#endif /*REPOSITORY_*/
