/////////////////////////////////////////////////////////////////////
// Code fuer den Kreativ-Wettbewerb zum 25. c't-Geburtstag von     //
// Josef Schtzenberger, basierend auf den Delphi Code von         //
// Joe Merten, ajm@jme.de, 29.06.2008                              //
// Getestet mit Delphi 7 unter Windows XP                          //
/////////////////////////////////////////////////////////////////////

Unit AstGame;

{$ifdef VER150}
  {$Define UseIndyUDP}     // Indy (D7)
 // {$Define UseUdpSocket}   // Borland Sockets (D7)
{$else}
  {$Define UseNMUDP}       // NetMasters UDP (D5)
{$endif}

interface

uses
  Classes,
{$ifdef VER150}
  Types,   // TPoint bei D7
           // Fuer D6 weiss ich's nicht
           // Bei D5 steckt TPoint in der Unit Windows
{$endif}

{$ifdef UseIndyUDP}
  IdBaseComponent, IdComponent,   IdUDPBase, IdUDPServer,
  IdSocketHandle,  IdStackConsts, IdUDPClient,
{$endif UseIndyUDP}

{$ifdef UseNMUDP}
  NMUDP, WinSock,    // wegen SetBroadcast (SetSockOpt)
{$endif UseNMUDP}

{$ifdef UseUdpSocket}
  Sockets,
  // WinSock,      // wegen TSockAddr
  // WsaError,
{$endif UseUdpSocket}

     Windows, // fuer GetTickCount
     ExtCtrls,SysUtils,Graphics;

const
    AstUdpPort         = 1979;

    AstKey_Hyperspace  = $01;
    AstKey_Fire        = $02;
    AstKey_Thrust      = $04;
    AstKey_Right       = $08;
    AstKey_Left        = $10;
    AstKey_StartButton = $20;

    AstShipShotDist    = 557;

type
  TAstKeys_=packed record
    aSignature: Array[0..5] of Char;
    nKeys: Byte;
    nPing: Byte;
    end;

type
  TAstVectorFrame_=packed record
    aVectorRam: Array[0..1023] of Byte;
    nFrameNr  : Byte;     // wird bei jedem Frame inkrementiert
    nPing     : Byte;     // Der Server schickt das letzte empfangene ping-Byte zurck
    end;

const
  AstObjectId_Unknown = 0;
  AstObjectId_Stone1  = 1;
  AstObjectId_Stone2  = 2;
  AstObjectId_Stone3  = 3;
  AstObjectId_Stone4  = 4;
  AstObjectId_Saucer  = 5;
  AstObjectId_Shot    = 6;
  AstObjectId_Ship    = 7;
  AstObjectId_Char    = 8;
  AstObjectId_String  = 9;
  AstObjectId_Explosion = 10;
  AstObjectId_Max     = 10;

  StoneSizeBig    = 40;
  StoneSizeMedium = 16;
  StoneSizeSmall  = 9;
type
  TAstObject=class
  public
    StartByte:Byte;
    Collision:integer;  //KollisionsKurs Frames bis zur Kollision
    AimFrame:integer;   //ZielerfassungsFrame
    FireFrame:integer;  //AbschussFrame
    ownshot : boolean;
    nshots  : integer;  //Anzahl Schsse
    shot    : integer;  //Schussziel
    frames  : integer;  //Lebensdauer
    id      : integer;
    bVisible: Boolean;
    x,y     : Integer;
    dx,dy   : Integer;  // Bewegungsrichtung
    nShipVX : Integer;  // Blickrichtung
    nShipVY : Integer;  // Blickrichtung
    nObjId  : Integer;
    nScale  : Integer;  // 0=gross, 15=mittel, 14=klein
    nRadiusX: Integer;
    nRadiusY: Integer;
    bMarked : Boolean;
    cChar   : Char;
    sText   : ShortString; // ShortString wegen Performance
    xa,ya   : array[0..7] of integer; //Geschwindigkeitsberechnung
  public
    constructor Create;
    destructor Destroy; override;
    procedure Dump;
    procedure Paint(aOut: TCanvas);
    function GetDist(p: TAstObject): TPoint;
    end;

  TAstObjArr=class
  private
    aArr: Array of TAstObject;
    bRef: Boolean;
  private
    function GetCount: Integer;
    function GetObj(n: Integer): TAstObject;
  public
    constructor Create(bRef_: Boolean);
    destructor Destroy; override;
    procedure Clear;
    procedure Add(pObj: TAstObject);

    property Count: Integer read GetCount;
    property Obj[n: Integer]: TAstObject read GetObj; default;
    end;

type
  TAstObjects=class
  public
    aObjs     : TAstObjArr;
    pShip     : TAstObject;
    pSaucer   : TAstObject;
    apShots   : TAstObjArr;
    apStones  : TAstObjArr;
    apExplos  : TAstObjArr;
    apChars   : TAstObjArr;
    nBigStones: Integer;
    aBmp      : TBitmap;
    nShipsLeft: Integer;
    nScore    : Integer;
    nHighscore: Integer;
    sHighName : ShortString;   // Zum eintragen der Initialien in die Highscore Tabelle
  private
    function GetCount: Integer;
    function Add(nObjId: Integer; x,y,nScale: Integer): TAstObject;
  public
    constructor Create;
    destructor Destroy; override;
    procedure Clear;
    procedure AddStone(nObjId: Integer; x,y,nScale,id: Integer);
    procedure AddEplosion(nObjId: Integer; x,y,nScale,id: Integer);
    procedure AddSaucer(x,y,nScale,id: Integer);
    procedure AddShot(x,y,nScale,id: Integer);
    procedure AddShip(x,y,vx,vy,id: Integer);
    procedure AddChar(c: Char; x,y,nScale: Integer);
    procedure AddString(s: ShortString; x,y,nScale: Integer);
    procedure Dump;
    procedure Paint(aOut: TCanvas);

    property Count: Integer read GetCount;

    property Objects  : TAstObjArr  read aObjs;
    property Ship     : TAstObject  read pShip;
    property Saucer   : TAstObject  read pSaucer;
    property Stones   : TAstObjArr  read apStones;
    property Explos   : TAstObjArr  read apExplos;
    property Shots    : TAstObjArr  read apShots;
    property BigStones: Integer     read nBigStones;
    property ShipsLeft: Integer     read nShipsLeft;
    property Score    : Integer     read nScore;
    property Highscore: Integer     read nHighscore;
    property HighName : ShortString read sHighName;
    end;

type
  TAstGameStatus = (
    ags_Unknown,
    ags_Standby,   // Wartet auf Start Button
    ags_Playing,   // Aktiv im Spiel
    ags_GameOver,  // Letztes Schiff verloren
    ags_Highscore  // Enter Highscore
  );

type
  TAstGameStatistics = record
    nWave             : Integer; // Level 1...
    nScoreWave        : Integer; // Punkte bisher in diesem Level
    nStonesWave       : Integer; // Anzahl der Steine zu Level begin
    nStonesAll        : Integer; // Summe der Steineanzahl aller Level
    nSaucerBigWave    : Integer; // Bisherige Anzahl grosser Ufos im aktuellen Level
    nSaucerBigAll     : Integer; // Bisherige Anzahl grosser Ufos seit ags_Playing
    nSaucerSmaWave    : Integer; // Bisherige Anzahl kleiner Ufos im aktuellen Level
    nSaucerSmaAll     : Integer; // Bisherige Anzahl kleiner Ufos seit ags_Playing
    nSaucerBigShotWave: Integer; // Anzahl abgeschossener grosser Ufos im aktuellen Level
    nSaucerBigShotAll : Integer; // Anzahl abgeschossener grosser Ufos seit ags_Playing
    nSaucerSmaShotWave: Integer; // Anzahl abgeschossener kleiner Ufos im aktuellen Level
    nSaucerSmaShotAll : Integer; // Anzahl abgeschossener kleiner Ufos seit ags_Playing
    end;

type
  TAstGame=class(TComponent)
  private
{$ifdef UseIndyUDP}
    aUdpSocket : TIdUDPServer;
{$endif UseIndyUDP}
{$ifdef UseNMUDP}
    aUdpSocket : TNMUDP;
{$endif UseNMUDP}
{$ifdef UseUdpSocket}
    aUdpSocket : TUdpSocket;
{$endif UseUdpSocket}

    bConnected : Boolean;     // Wird bei Frameempfang auf True gesetzt und bei Timeout auf False

    sRemoteHost: String;
    aTimer     : TTimer;
    nTimerInterval: Integer;

    pPaintBox  : TPaintBox;

    aTxBuf     : TAstKeys_;
    aRxBuf     : TAstVectorFrame_;
    nKeys      : Byte;
    nKeys0     : Byte;

    nRxFrameCount     : Integer;  // Anzahl bisher empfangener Frames
    nRxFrameNr0       : Integer;  // Nummer des zuletzt empfangenen Frames fuer Missing Frame Detection
    bRxFrameMissing   : Boolean;  // True, wenn letzter Frameempfang kein Folgeframe von dem davor empfangenen ist
    nRxFramesMissing  : Integer;  // Groesse der Luecke wenn bRxFrameMissing=True
    nRxFrameMissCount : Integer;  // Anzahl der Frameluecken
    nRxFrameMissCountA: Integer;  // Summe der verlorenen Frames

    nFrameTxTS    : Cardinal;  // Timestamp des zuletzt gesendeten Frames

    nFramePingData : Integer;  // Aktuelles TX Ping Byte, wird mit jedem TX erhoeht
    bFramePing     : Boolean;
    nFramePingTS   : Cardinal; // Zur Messung TX-RX der Antwortzeit
    nFramePingData0: Integer;  // Das Ping-Byte, welches zum Zeitpunkt nFramePingTS abgeschickt wurde

    nFrameRxTS : Cardinal; // Zur Messung Frameempfangsrate
    nFrameRxMS : Cardinal; // Zeit zwischen den beiden zuletzt empfangenen Frames
    nPingMS    : Cardinal; // Zuletzt gemessene Ping Zeit

    aObjs      : TAstObjects;
    aObjs0     : TAstObjects;

    eGameStatus : TAstGameStatus;
    eGameStatus0: TAstGameStatus;
    nGameStartT: Cardinal;
    nGameStopT : Cardinal;
    nGameTimeMS: Cardinal;
    aGameStats : TAstGameStatistics;

    nShipsLeft : Integer;
    nScore     : Integer;
    nHighscore : Integer;
    nScore0Objs: Integer;
    nScore100k : Integer;
    sHighName  : ShortString;   // Zum eintragen der Initialien in die Highscore Tabelle

    fOnRxFrame : TNotifyEvent;
    fOnRxFrame2: TNotifyEvent;
   private
{$ifdef UseIndyUDP}
    procedure OnIdUDPRXStatus(axSender: TObject; const axStatus: TIdStatus; const asStatusText: String);
    procedure OnIdUDPRXUDPRead(Sender: TObject; AData: TStream; ABinding: TIdSocketHandle);
{$endif UseIndyUDP}
{$ifdef UseNMUDP}
    procedure OnNMUDPStatus(Sender: TComponent; status: String);
    procedure OnNMUDPDataReceived(Sender: TComponent; NumberBytes: Integer;  FromIP: String; Port: Integer);
{$endif UseNMUDP}
{$ifdef UseUdpSocket}
    procedure OnUdpSocketReceive(Sender: TObject; Buf: PAnsiChar; var DataLen: Integer);
    procedure OnUdpSocketError(Sender: TObject; SocketError: Integer);
{$endif UseUdpSocket}
    procedure SendTxBuf;
    procedure RxBufReceived;
    procedure OnTimer(Sender: TObject);
    procedure ImpInterpretScreen(pRxBuf: TAstVectorFrame_);
    procedure SetRemoteHost(s: String);
    procedure SetTimerInterval(n: Integer);
    function  GetConnectedStr: String;
    function  GetGameStatusStr: String;
    function  GetGameTimeStr: String;
    function  GetShipAngle:integer;
  public
    WByte: Byte;
    WBytearr: array[0..1] of byte;
    LastShotByte: Byte;
    LastRotFrame: integer;
    LastFrames2HitTarget:integer;
    LastFireFrame:integer;
    constructor Create(aOwner: TComponent); override;
    destructor Destroy; override;
    procedure ResetRxStats;
    procedure ResetGameStats;

    procedure ResetKeys;
    procedure RotateLeft;
    procedure RotateRight;
    procedure Thrust;
    procedure Fire;
    procedure Hyperspace;
    procedure StartButton;

    property ShipAngle: integer read GetShipAngle;
    property Keys: Byte read nKeys write nKeys;
    property Objects: TAstObjects read aObjs;
    property Objects0: TAstObjects read aObjs0;
    property ConnectionStatus: Boolean read bConnected;
    property ConnectionStatusStr: String read GetConnectedStr;
    property GameStatus: TAstGameStatus read eGameStatus;
    property GameStatus0: TAstGameStatus read eGameStatus0;
    property GameStatusStr: String read GetGameStatusStr;
    property Ships    : Integer read nShipsLeft;
    property Score    : Integer read nScore;
    property Highscore: Integer read nHighscore;
    property PlayingTime: Cardinal read nGameTimeMS;
    property PlayingTimeStr: String read GetGameTimeStr;
    property GameStats: TAstGameStatistics read aGameStats;
    property HighName : ShortString read sHighName;
    property FrameRxMS: Cardinal read nFrameRxMS;
    property PingMS: Cardinal read nPingMS;
    property RxFrameCount     : Integer read nRxFrameCount;
    property RxFrameMissing   : Boolean read bRxFrameMissing   ;
    property RxFramesMissing  : Integer read nRxFramesMissing  ;
    property RxFrameMissCount : Integer read nRxFrameMissCount ;
    property RxFrameMissCountA: Integer read nRxFrameMissCountA;
  published
    property RemoteHost   : String read sRemoteHost write SetRemoteHost;
    property TimerInterval: Integer read nTimerInterval write SetTimerInterval;
    property PaintBox     : TPaintBox read pPaintBox write pPaintBox;
    property OnRxFrame    : TNotifyEvent read fOnRxFrame  write fOnRxFrame;
    property OnRxFrame2   : TNotifyEvent read fOnRxFrame2 write fOnRxFrame2;
    end;

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

// Routinen zur Zeitmessung
Function GetTimeMS: Cardinal;
Function GetTimeDiffMS(nStart,nStop: Cardinal): Cardinal;
Function GetTimeGoneMS(nStart: Cardinal): Cardinal;
Function GetTimeStrMMMSS(nMS: Cardinal): String;

Function IntToStrP(n: Integer): String;

implementation
 uses astmain,math,astmath,AstStat;
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// Routinen zur Zeitmessung

Function GetTimeMS: Cardinal;
  begin
  Result:=GetTickCount;
  end;

{$ifopt R+}{$Define RngChk}{$R-}{$endif}
{$ifopt Q+}{$Define OvlChk}{$Q-}{$endif}
Function GetTimeDiffMS(nStart,nStop: Cardinal): Cardinal;
  begin
  Dec(nStop,nStart);
  Result:=nStop;
  end;
{$ifdef OvlChk}{$Undef OvlChk}{$Q+}{$endif}
{$ifdef RngChk}{$Undef RngChk}{$R+}{$endif}

Function GetTimeGoneMS(nStart: Cardinal): Cardinal;
  begin
  Result:=GetTimeDiffMS(nStart,GetTimeMS);
  end;

Function GetTimeStrMMMSS(nMS: Cardinal): String;
  var nMin: Cardinal;
      nSec: Cardinal;
  begin
  nSec:=nMS div 1000;
  nMin:=nSec div 60;
  Dec(nSec,nMin*60);

  Result:=Format('%d:%.2d',[nMin,nSec]);
  end;

Function IntToStrP(n: Integer): String;
  var s  : String;
      i: integer;
  begin
  s:=IntToStr(n);
  i:=Length(s)-2;
  while i>1 do
    begin
    Insert('.',s,i);
    Dec(i,3);
    end;
  Result:=s;
  end;

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

constructor TAstObject.Create;
  begin
  inherited;
  frames:=0;
  shot:=0;
  end;

destructor TAstObject.Destroy;
  begin
  inherited;
  end;

procedure TAstObject.Dump;
  begin
  Writeln(IntToHex(nObjId,4),' x=',x,'  y=',y,' Scale=',nScale);
  end;

var nPaintBla: Integer=0;
function Sign2(const AValue: Integer): TValueSign;
begin
  if AValue < 0 then
    Result := NegativeValue
  else
    Result := PositiveValue;
end;
function calcKatheten(x2,y2,a:integer):Tpoint;
var x,y,r:double;
begin
  x:=x2;
  y:=y2;
  if a<>0 then
  r:=Sqrt(x*x+y*y)/a*2 else r:=1;
  result.X:=Round(x2/r);
  result.y:=Round(y2/r);
end;
procedure wrapLine(x1,y1,dx,dy,a:integer;aOut: TCanvas);
var x,y,s:double;p:TPoint;
begin
  if dx=0 then s:=9999*sign2(dy) else
  s:=dy/dx;
  p:=calcKatheten(dx,dy,a);
  if x1+p.x>512 then
  begin
    y:=y1+s*(512-x1);
    aOut.MoveTo(0,Round(y));
    aOut.LineTo(x1+p.x-512,y1+p.y);
  end;
  if x1+p.x<0 then
  begin
    y:=y1-s*x1;
    aOut.MoveTo(512,Round(y));
    aOut.LineTo(x1+p.x+512,y1+p.y);
  end;
  if s=0 then s:=0.0001*sign2(dx);
  if y1+p.y>384 then
  begin
    x:=x1+1/s*(384-y1);
    aOut.MoveTo(Round(x),0);
    aOut.LineTo(x1+p.x,y1+p.y-384);
  end;
  if y1+p.y<0 then
  begin
    x:=x1-1/s*y1;
    aOut.MoveTo(Round(x),384);
    aOut.LineTo(x1+p.x,y1+p.y+384);
  end;
end;
procedure TAstObject.Paint(aOut: TCanvas);
  var x1,y1: Integer;
      sx,sy: Integer;
//      p:Tpoint;
  begin
 // exit;
  aOut.Pen.Style:=psClear;
  aOut.Brush.Style:=bsSolid;
  aOut.Brush.Color:=clWhite;
//  y1:=895-y;  // Y-Koordinaten gehen von 128 bis 895
  y1:=y;
  x1:=x;
  x1:=x1 div 2;
  y1:=y1 div 2;
{  s:=1;
  case nScale of
    0 : s:=12; // gross
    15: s:= 8; // mittel
    14: s:= 4; // klein
    end;}
  sx:=nRadiusX div 2;
  sy:=nRadiusY div 2;
//  if nObjId>15 then s:=3;
  case nObjId of
    1: aOut.Brush.Color:=clNavy;
    2: aOut.Brush.Color:=clGreen;
    3: aOut.Brush.Color:=clMaroon;
    4: aOut.Brush.Color:=clAqua;
    5: aOut.Brush.Color:=clFuchsia;
    6: aOut.Brush.Color:=clYellow;
    7: aOut.Brush.Color:=clYellow;
    end;
  if nObjId<>AstObjectId_Ship then
    begin
    aOut.Pen.Style:=psSolid;
    aOut.Pen.Color:=aOut.Brush.Color;
    if not bMarked then aOut.Brush.Style:=bsClear;
    aOut.Ellipse(x1-sx,y1-sy,x1+sx,y1+sy);
    if nObjId<5 then
    begin
     aOut.Pen.Color := clWhite;
     aOut.MoveTo(x1,y1);
     aOut.LineTo(x1+dx div 5,y1+dy div 5);
    end;
    end
  else begin
    aOut.Pen.Style:=psSolid;
    aOut.Pen.Color:=clYellow;
    aOut.MoveTo(x1,y1);
    aOut.LineTo(x1+nShipVX,y1+nShipVY);
    wrapLine(x1,y1,nShipVX,nShipVY,612, aOut);
  end;
  if nObjId>AstObjectId_Max then
    begin
    aOut.Font.Color:=clWhite;
    aOut.TextOut(x1,y1,IntToHex(nObjId,4));
    end;
  if nObjId=AstObjectId_Char then
    begin
    aOut.Font.Color:=$000000FF; // Rot
    aOut.TextOut(x1,y1,cChar);
    end;
  if nObjId=AstObjectId_String then
    begin
    if nScale=0 then aOut.Font.Height:=16       // z.B. Highscoreanzeige oben in der Mitte
    else if nScale=1 then aOut.Font.Height:=22  // z.B. Highscoreeingabetext
    else if nScale=2 then aOut.Font.Height:=30  // z.B. Initialieneingabe
    else begin
      aOut.Font.Height:=10;
      end;
    aOut.Font.Color:=$00FF0000; // Blau
    aOut.Font.Color:=aOut.Font.Color;
    aOut.TextOut(x1,y1,sText);
    end;
  Inc(nPaintBla);
  end;

function TAstObject.GetDist(p: TAstObject): TPoint;
  var x1,y1: Integer;
  begin
  x1:=x-p.x;
  y1:=y-p.y;

  if x1>= 512 then Dec(x1,1024);
  if x1<=-512 then Inc(x1,1024);
  if y1>= 384 then Dec(y1,768);
  if y1<=-384 then Inc(y1,768);

  Result.X:=x1;
  Result.Y:=y1;
  end;

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

constructor TAstObjArr.Create(bRef_: Boolean);
  begin
  inherited Create;
  bRef:=bRef_
  end;

destructor TAstObjArr.Destroy;
  begin
  Clear;
  inherited;
  end;

procedure TAstObjArr.Clear;
  var n,i: Integer;
  begin
  if not bRef then
    begin
    n:=Length(aArr);
    for i:=0 to n-1 do
      begin
      aArr[i].Free;
      end;
    end;
  SetLength(aArr,0);
  end;

function TAstObjArr.GetCount: Integer;
  begin
  Result:=Length(aArr);
  end;

function TAstObjArr.GetObj(n: Integer): TAstObject;
  begin
  if n>high(aArr) then
  Result:=aArr[0]else
 Result:=aArr[n];
  end;

procedure TAstObjArr.Add(pObj: TAstObject);
  begin
  SetLength(aArr,Length(aArr)+1);
  aArr[Length(aArr)-1]:=pObj;
  end;

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

constructor TAstObjects.Create;
  begin
  inherited;
  apShots:=TAstObjArr.Create(True);
  apStones:=TAstObjArr.Create(True);
  apExplos:=TAstObjArr.Create(True); 
  apChars:=TAstObjArr.Create(True);
  aObjs:=TAstObjArr.Create(False);
  end;

destructor TAstObjects.Destroy;
  begin
  Clear;
  aBmp.Free;
  aBmp:=Nil;
  inherited;
  end;

procedure TAstObjects.Clear;
  begin
  pShip:=Nil;
  pSaucer:=Nil;
  apShots.Clear;
  apStones.Clear;
  apExplos.Clear;
  apChars.Clear;
  aObjs.Clear;
  nBigStones:=0;
  nShipsLeft:=0;
  nScore:=0;
  nHighscore:=0;
  sHighName:='';
  end;

function  TAstObjects.Add(nObjId: Integer; x,y,nScale: Integer): TAstObject;
  var aObj: TAstObject;
  begin
  aObj:=TAstObject.Create;
  aObj.nObjId:=nObjId;
  aObj.x:=x;
  aObj.y:=y;
  aObj.nScale:=nScale;
  aObjs.Add(aObj);
  Result:=aObj;
  end;

procedure TAstObjects.AddStone(nObjId: Integer; x,y,nScale,id: Integer);
  var pObj: TAstObject;
  begin
  pObj:=Add(nObjId,x,y,nScale);
  case nScale of
    0 : begin
        pObj.nRadiusX:=StoneSizeBig; // 40;
        Inc(nBigStones);
        end;
    15: pObj.nRadiusX:=StoneSizeMedium; // 20;
    14: pObj.nRadiusX:=StoneSizeSmall; //  8;
    end;
  pObj.nRadiusY:=pObj.nRadiusX;
  pObj.id:=id;
  apStones.Add(pObj);
  end;
procedure TAstObjects.AddEplosion(nObjId: Integer; x,y,nScale,id: Integer);
  var pObj: TAstObject;
  begin
  pObj:=Add(nObjId,x,y,nScale);
  case nScale of
    0 : begin
        pObj.nRadiusX:=40; // 40;
        end;
    15: pObj.nRadiusX:=16; // 20;
    14: pObj.nRadiusX:= 9; //  8;
    end;
  pObj.nRadiusY:=pObj.nRadiusX;
  pObj.id:=id;
  apExplos.Add(pObj);                    
  end;

procedure TAstObjects.AddSaucer(x,y,nScale,id: Integer);
  var pObj: TAstObject;
  begin
  pObj:=Add(AstObjectId_Saucer,x,y,nScale);
  pObj.id:=id;
  pSaucer:=pObj;
  case nScale of
    15: begin pObj.nRadiusX:=20; pObj.nRadiusY:=15; end;
    14: begin pObj.nRadiusX:=10; pObj.nRadiusY:= 6; end;
    end;
  end;

procedure TAstObjects.AddShot(x,y,nScale,id: Integer);
  var pObj: TAstObject;
  begin
  pObj:=Add(AstObjectId_Shot,x,y,nScale);
  pObj.id:=id;
  pObj.nRadiusX:=2;
  pObj.nRadiusY:=2;

  apShots.Add(pObj);
  end;

procedure TAstObjects.AddShip(x,y,vx,vy,id: Integer);
  var pObj: TAstObject;
  begin
  pObj:=Add(AstObjectId_Ship,x,y,0);
  pObj.nRadiusX:=13;
  pObj.nRadiusY:=13;
  pObj.nShipVX:=vx;
  pObj.nShipVY:=vy;
  pObj.id:=id;
  pShip:=pObj;
  end;

procedure TAstObjects.AddChar(c: Char; x,y,nScale: Integer);
  var pObj: TAstObject;
  begin
  pObj:=Add(AstObjectId_Char,x,y,nScale);
  pObj.cChar:=c;
  apChars.Add(pObj);
  end;

procedure TAstObjects.AddString(s: ShortString; x,y,nScale: Integer);
  var pObj: TAstObject;
  begin
  pObj:=Add(AstObjectId_String,x,y,nScale);
  pObj.sText:=s;
  apChars.Add(pObj);
  end;

function TAstObjects.GetCount: Integer;
  begin
  Result:=aObjs.Count;
  end;

procedure TAstObjects.Dump;
  var n,i: Integer;
  begin
  n:=Count;
  for i:=0 to n-1 do
    begin
    aObjs[i].Dump;
    end;
  end;

procedure TAstObjects.Paint(aOut: TCanvas);
  var n,i: Integer;
  begin
  nPaintBla:=0;
  if aBmp=Nil then
    begin
    aBmp:=TBitmap.Create;
    aBmp.Width:=1024 div 2;
    aBmp.Height:=768 div 2;
    aBmp.Canvas.Font.Name:='Courier New';
    aBmp.Canvas.Font.Style:=[fsBold];
    //aBmp.Canvas.Font.Name:='System';
    end;
   aBmp.Canvas.Pen.Style:=psClear;
  aBmp.Canvas.Brush.Style:=bsSolid;
  aBmp.Canvas.Brush.Color:=clBlack;
  aBmp.Canvas.Rectangle(0,0,aBmp.Width+1,aBmp.Height+1);
  n:=aObjs.Count;
  for i:=0 to n-1 do
    begin
    aObjs[i].Paint(aBmp.Canvas);
    end;
  aOut.Draw(0,0,aBmp);
 end;

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

constructor TAstGame.Create;

{$ifdef UseIndyUDP}
Procedure ImpMakeIndySocket;
var optval:integer;
begin
    aUdpSocket:=TIdUDPServer.Create(Self); // Client
    optval  := -1;
    aUdpSocket.Binding.SetSockOpt(Id_SOL_SOCKET,Id_SO_REUSEADDR ,PChar(@OptVal), SizeOf(optval)	);
    aUdpSocket.DefaultPort:=AstUdpPort;
    aUdpSocket.OnUDPRead:=OnIdUDPRXUDPRead;
    aUdpSocket.OnStatus:=OnIdUDPRXStatus;
    aUdpSocket.Active:=True;
end;
{$endif UseIndyUDP}

{$ifdef UseNMUDP}
  Procedure ImpMakeNMSocket;
    begin
    aUdpSocket:=TNMUDP.Create(Self); // Server

    aUdpSocket.LocalPort:=AstUdpPort+1; // Darf nicht 0 sein
    aUdpSocket.RemoteHost:=sRemoteHost;
    aUdpSocket.RemotePort:=AstUdpPort;
    aUdpSocket.ReportLevel:=Status_Basic;
    // Status_None = 0
    // Status_Informational = 1
    // Status_Basic = 2
    // Status_Routinen = 4
    // Status_Debug = 8
    // Status_Trace = 16

    aUdpSocket.OnStatus:=OnNMUDPStatus;
    aUdpSocket.OnDataReceived:=OnNMUDPDataReceived;

    // aUdpSocket.Active:=True; nicht erforlderlich bei NMUDP, gibts auch garnicht
    end;
{$endif UseNMUDP}

{$ifdef UseUdpSocket}
  Procedure ImpMakeUdpSocket;
    begin
    aUdpSocket:=TUdpSocket.Create(Self);
    aUdpSocket.OnReceive:=OnUdpSocketReceive;
    aUdpSocket.OnError:=OnUdpSocketError;
    aUdpSocket.RemoteHost:=sRemoteHost;
    aUdpSocket.RemotePort:=IntToStr(AstUdpPort);
    aUdpSocket.BlockMode:=bmNonBlocking;
    aUdpSocket.Active:=True;
    end;
{$endif UseUdpSocket}

  Procedure ImpMakeSocket;
    begin
{$ifdef UseIndyUDP}
    ImpMakeIndySocket;
{$endif UseIndyUDP}
{$ifdef UseNMUDP}
    ImpMakeNMSocket;
{$endif UseNMUDP}
{$ifdef UseUdpSocket}
    ImpMakeUdpSocket;
{$endif UseUdpSocket}
    end;


  procedure ImpMakeTimer;
    begin
    aTimer:=TTimer.Create(Self);
    aTimer.OnTimer:=OnTimer;
    aTimer.Interval:=nTimerInterval;
    aTimer.Enabled:=True;
    end;

  begin
  inherited;
  WByte:=0;
  FillChar(aRxBuf,SizeOf(aRxBuf),0);
  FillChar(aTxBuf,SizeOf(aTxBuf),0);
  aTxBuf.aSignature:='ctmame';

  aObjs:=TAstObjects.Create;
  aObjs0:=TAstObjects.Create;

  sRemoteHost:='127.0.0.1';
 //   sRemoteHost:='192.168.6.171';
   // Kurzes Timerintervall wird fuer UseUdpSocket benoetigt
  nTimerInterval:=5;

  ImpMakeSocket;
  ImpMakeTimer;


  end;

destructor TAstGame.Destroy;
  begin
  aTimer.Enabled:=False;

{$ifdef UseIndyUDP}
  // Indy macht beim Beenden zuweilen Aerger
  aUdpSocket.Active:=False;
  aUdpSocket.OnUDPRead:=Nil;
  aUdpSocket.OnStatus:=Nil;
{$endif UseIndyUDP}

{$ifdef UseNMUDP}
  // aUdpSocket.Active:=False; gibt's hier nicht
  aUdpSocket.OnStatus:=Nil;
  aUdpSocket.OnDataReceived:=Nil;
{$endif UseNMUDP}

{$ifdef UseUdpSocket}
  // Damit nichts mehr asynchron reinkommt
  aUdpSocket.Active:=False;
  aUdpSocket.OnReceive:=Nil;
  aUdpSocket.OnError:=Nil;
{$endif UseUdpSocket}

  aObjs.Free;
  aObjs0.Free;
  inherited;
  end;

procedure TAstGame.ResetRxStats;
  begin
  nRxFrameCount:=0;
  nRxFrameNr0:=0;
  bRxFrameMissing:=False;
  nRxFramesMissing:=0;
  nRxFrameMissCount:=0;
  nRxFrameMissCountA:=0;
  end;

procedure TAstGame.ResetGameStats;
  begin
  FillChar(aGameStats,SizeOf(aGameStats),0);
  end;

procedure TAstGame.SetRemoteHost(s: String);
  begin
  sRemoteHost:=s;
{$ifdef UseIndyUDP}
  // Keine besondere Behandlung notwendig
  // sRemoteHost wird bei SendBuffer() uebergeben
{$endif UseIndyUDP}

{$ifdef UseNMUDP}
    aUdpSocket.RemoteHost:=s;
{$endif UseNMUDP}

{$ifdef UseUdpSocket}
  aUdpSocket.RemoteHost:=s;
  aUdpSocket.Active:=True;
{$endif UseUdpSocket}
  ResetRxStats;
  end;

procedure TAstGame.SetTimerInterval(n: Integer);
  begin
  if n=nTimerInterval then Exit;
  nTimerInterval:=n;
  aTimer.Interval:=nTimerInterval;
  aTimer.Enabled:=True;
  end;

function TAstGame.GetConnectedStr: String;
  begin
  if bConnected then
    begin
    Result:='Connected';
    end
  else begin
    Result:='No Answer';
    StatGameOver(self);
    end;
  end;

function TAstGame.GetGameStatusStr: String;
  begin
  case eGameStatus of
    ags_Unknown  : Result:='Unknown';
    ags_Standby  : Result:='Standby';
    ags_Playing  : Result:='Playing';
    ags_GameOver : Result:='Game Over';
    ags_Highscore: Result:='Highscore';
    else Result:='Status='+IntToStr(Integer(eGameStatus));
    end;
  end;

function TAstGame.GetShipAngle: integer;
  begin
//  Result:=Round(3*PI*WByte/128*1000{*180/PI});
//  while result>round(PI*1000) do dec(Result,round(PI*2000));
    Result:=Round(WinkelByte[WByte]/180*PI);
  end;

function TAstGame.GetGameTimeStr: String;
  begin
  Result:=GetTimeStrMMMSS(nGameTimeMS);
  end;

procedure TAstGame.ResetKeys;
  begin
  nKeys:=0;
  end;

procedure TAstGame.RotateLeft;
  begin
  {$R-};{$Q-};
  if  WByte=0 then WByte:=255 else dec(WByte);
  {$Q+};{$R+};
 LastRotFrame:=nRxFrameCount;
  if  nKeys and AstKey_Right>0 then
  assert(0=9);
//  OutputDebugString(PChar('left----- '+inttostr(wbyte)));
  nKeys:=nKeys or AstKey_Left;
  nKeys:=nKeys and not AstKey_Right;
  end;

procedure TAstGame.RotateRight;
  begin
 {$R-}; {$Q-}
  if  WByte=255 then WByte:=0 else inc(WByte);
 {$Q+}; {$R+}
  LastRotFrame:=nRxFrameCount;
  if  nKeys and AstKey_Left>0 then
  assert(0=9);
//  OutputDebugString(PChar('right----- '+inttostr(wbyte)));
  nKeys:=nKeys or AstKey_Right;
  nKeys:=nKeys and not AstKey_Left;
  end;

procedure TAstGame.Thrust;
  begin
  nKeys:=nKeys or AstKey_Thrust;
  end;

procedure TAstGame.Fire;
  begin
  LastShotByte:=WByte;
//  LastWxy:=round(ArcTan2(Objects.pShip.nShipVY, Objects.pShip.nShipVX)*180/PI*1000);
  myOutputDebug(['shot----- ',wbyte,'frame=',RxFrameCount-285]);
  nKeys:=nKeys or AstKey_Fire;
  end;

procedure TAstGame.Hyperspace;
  begin
  nKeys:=nKeys or AstKey_Hyperspace;
  end;

procedure TAstGame.StartButton;
  begin
  nKeys:=nKeys or AstKey_StartButton;
  end;

procedure TAstGame.OnTimer(Sender: TObject);
  begin
  // Wenn der Letzte RX zu lange her ist, dann
  // sende ich zyklisch TX Frames raus
  if GetTimeGoneMS(nFrameTxTS)>=1000 then
    begin
    bConnected:=False;
    SendTxBuf;
    //Writeln('TX');
    end;

{$ifdef UseUdpSocket}
  // Erforderlich fuer Empfang bei Borland UDP Sockets
  aUdpSocket.ReceiveBuf(aRxBuf,SizeOf(aRxBuf));
{$endif UseUdpSocket}
  end;

procedure TAstGame.SendTxBuf;
{$ifdef UseNMUDP}
  var aStream: TMemoryStream;
{$endif UseNMUDP}
  begin

  Inc(nFramePingData);
  if nFramePingData>255 then nFramePingData:=0;
  aTxBuf.nPing:=nFramePingData;

  if bFramePing and (GetTimeGoneMS(nFramePingTS)>1000) then
    begin
    // Nach 1s noch keine Ping Antwort
    // Evtl. Frameverlust
    // Bei 16ms je Frame bin ich nach ca. 4s 1x die 256 durch
    bFramePing:=False;
    end;

  if not bFramePing then
    begin
    // Neuen Ping auf den Weg schicken
    nFramePingTS:=GetTimeMS;
    bFramePing:=True;
    nFramePingData0:=nFramePingData;
    end;

  nFrameTxTS:=GetTimeMS;

  try
{$ifdef UseIndyUDP}
    // 18.04.2008: Indy bringt hier beim Beenden oefters mal eine Exception
    // EIdSocketError "Socket-Fehler # 10038 Socket-Operation auf Nicht-Socket."
    aUdpSocket.SendBuffer(sRemoteHost,AstUdpPort,aTxBuf,SizeOf(aTxBuf));
{$endif UseIndyUDP}

{$ifdef UseNMUDP}
    aStream:=TMemoryStream.Create;
    try
      aStream.Write(aTxBuf,SizeOf(aTxBuf));
      aUdpSocket.SendStream(aStream);
    finally
      aStream.Free;
      end;
{$endif UseNMUDP}

{$ifdef UseUdpSocket}

    aUdpSocket.SendBuf(aTxBuf,SizeOf(aTxBuf));
 //   OutputDebugString(PChar('send------'+inttostr(aTxBuf.nKeys)+'ping------'+inttostr(aTxBuf.nPing)));

{$endif UseUdpSocket}
  except
    end;
  end;

procedure TAstGame.RxBufReceived;
  var nT    : Cardinal;
      nT1   : Cardinal;
      nExpNr: Integer;
  begin
  bConnected:=True;

  // Pingzeit bestimmen
  if bFramePing then
    begin
    if aRxBuf.nPing=nFramePingData0 then
      begin
      nPingMS:=GetTimeGoneMS(nFramePingTS);
      bFramePing:=False;
      //Writeln('P=',nPingMS);
      end;
    end;

  // RX Framerate bestimmen
  nT:=GetTimeMS;
  nT1:=GetTimeDiffMS(nFrameRxTS,nT);
  if nFrameRxTS<>0 then
    begin
    nFrameRxMS:=nT1;
    end;
  nFrameRxTS:=nT;

  if nRxFrameCount>0 then
    begin
    nExpNr:=nRxFrameNr0+1;
    if nExpNr>255 then nExpNr:=0;
    if nExpNr<0 then nExpNr:=255;
    if nExpNr<>aRxBuf.nFrameNr then
      begin
      // Frame Lost
      bRxFrameMissing:=True;
      nRxFramesMissing:=aRxBuf.nFrameNr-nExpNr;
      if nRxFramesMissing<0 then Inc(nRxFramesMissing,256);
      Inc(nRxFrameMissCount);
      Inc(nRxFrameMissCountA,nRxFramesMissing);
      end;
    end;
  Inc(nRxFrameCount);
  nRxFrameNr0:=aRxBuf.nFrameNr;

  ImpInterpretScreen(aRxBuf);
  WBytearr[1]:=WBytearr[0];
  WBytearr[0]:=WByte;
  if Assigned(fOnRxFrame) then
    begin
    fOnRxFrame(Self);
    end;

  aTxBuf.nKeys:=nKeys;
  nKeys0:=nKeys;
  SendTxBuf;

  if pPaintBox<>Nil then
    begin
    Objects.Paint(pPaintBox.Canvas);
    end;

  if Assigned(fOnRxFrame2) then
    begin
    fOnRxFrame2(Self);
    end;

{$ifdef UseUdpSocket}
  // Erforderlich fuer Empfang bei Borland UDP Sockets
  aUdpSocket.ReceiveBuf(aRxBuf,SizeOf(aRxBuf));
{$endif UseUdpSocket}
  end;

{$ifdef UseIndyUDP}
procedure TAstGame.OnIdUDPRXStatus(axSender: TObject; const axStatus: TIdStatus; const asStatusText: String);
  begin
  assert(axSender=axSender);
  // Todo
  end;

procedure TAstGame.OnIdUDPRXUDPRead(Sender: TObject; AData: TStream; ABinding: TIdSocketHandle);
  var nDataLen: Integer;
  begin
  if (aBinding.PeerIP=sRemoteHost) and (aBinding.PeerPort=AstUdpPort) then
    begin
    nDataLen:=AData.Size;

    //if (nDataLen>0) and (nDataLen<>1026) then
    //  begin
    //  Writeln(nDataLen);
    //  end;

    if nDataLen=1026 then
      begin
      AData.Read(aRxBuf,1026);
      RxBufReceived;
      end;
    end;
  end;
{$endif UseIndyUDP}

{$ifdef UseNMUDP}
procedure TAstGame.OnNMUDPStatus(Sender: TComponent; status: String);
  begin
  // Todo
  end;

procedure TAstGame.OnNMUDPDataReceived(Sender: TComponent; NumberBytes: Integer;  FromIP: String; Port: Integer);
  var aStream: TMemoryStream;
  begin
  if (FromIP=sRemoteHost) and (Port=AstUdpPort) then
    begin
    //if (NumberBytes>0) and (NumberBytes<>1026) then
    //  begin
    //  Writeln(NumberBytes);
    //  end;

    if NumberBytes=1026 then
      begin
      aStream:=TMemoryStream.Create;
      try
        aUdpSocket.ReadStream(aStream);
        if aStream.Size=1026 then
          begin
          aStream.Read(aRxBuf,SizeOf(aRxBuf));
          RxBufReceived;
          end;
      finally
        aStream.Free;
        end;
      end;
    end;
  end;
{$endif UseNMUDP}

{$ifdef UseUdpSocket}
procedure TAstGame.OnUdpSocketReceive(Sender: TObject; Buf: PAnsiChar; var DataLen: Integer);
  begin
  //if (DataLen>0) and (DataLen<>1026) then
  //  begin
  //  Writeln(DataLen);
  //  end;
  if DataLen=1026 then
    begin
    RxBufReceived;
    end;
  end;

procedure TAstGame.OnUdpSocketError(Sender: TObject; SocketError: Integer);
  begin

  if (SocketError<>10035) and
     (SocketError<>10054) then // Kein RemoteHost
   OutputDebugString(PChar('OnUdpSocketError'));
  if  (SocketError=10054) then // Kein RemoteHost
      StatGameOver(self);
 //  begin
  //  Writeln('UDP Error ',SocketError,': ',SysErrorMessage(SocketError));
  //  end;
  end;
{$endif UseUdpSocket}

procedure TAstGame.ImpInterpretScreen(pRxBuf: TAstVectorFrame_);
  var p       : PWord;
      nCmd    : Word;
      nOp     : Word;
      nArg    : Word;
      nCounter: Integer;
      vx,vy,vs: Integer;
      dx,dy   : Integer;
      nBrg    : Integer;
      nShipDetect: Integer;
      v1x,v1y : Integer;

      bWriteln: Boolean;
//      nRad: Integer;

      bStartText: Boolean;
      bEndeText : Boolean;
      bHighText : Boolean;

      sText: ShortString;
      bNewChar: Boolean;
      nTextX,nTextY,nTextSize: Integer;

      nScoreAdd: Integer;
      pObjsTmp : TAstObjects;

  Function ImpHandleChars: Boolean;
    var c: Char;
    begin
    c:=#0;
    case nArg of
      $0B2C: c:=' ';
      $0A78: c:='A';
      $0A80: c:='B';
      $0A8D: c:='C';
      $0A93: c:='D';
      $0A9B: c:='E';
      $0AA3: c:='F';
      $0AAA: c:='G';
      $0AB3: c:='H';
      $0ABA: c:='I';
      $0AC1: c:='J';
      $0AC7: c:='K';
      $0ACD: c:='L';
      $0AD2: c:='M';
      $0AD8: c:='N';
      $0ADD: c:='O';
      $0AE3: c:='P';
      $0AEA: c:='Q';
      $0AF3: c:='R';
      $0AFB: c:='S';
      $0B02: c:='T';
      $0B08: c:='U';
      $0B0e: c:='V';
      $0B13: c:='W';
      $0B1a: c:='X';
      $0B1f: c:='Y';
      $0B26: c:='Z';
      //$0 : c:='0'; fuer die Ziffer 0 wird offenbar der Buchstabe O verwendet
      $0B2E: c:='1';
      $0B32: c:='2';
      $0B3A: c:='3';
      $0B41: c:='4';
      $0B48: c:='5';
      $0B4F: c:='6';
      $0B56: c:='7';
      $0B5B: c:='8';
      $0B63: c:='9';
      end;

    if c<>#0 then
      begin
      if sText='' then
        begin
        nTextX:=vx;
        nTextY:=vy;
        nTextSize:=vs;
        end;
      Insert(c,sText,MaxInt);
      //aObjs.AddChar(c,vx,vy,vs);
      Inc(vx,20);  // Notwendig auch fuer AddString zur korrekten Darstellung der Highscores
      //if c='I' then Dec(vx,10); // nur fuer die Optik
      end;
    Result:=c<>#0;
    end;

  procedure ImpEvaluateText;
    var i: Integer;
        c0,c1,c2: Char;
    begin
    if sText='' then Exit;

    // Korrektur von '1OO' (Nullen als Buchstaben) auf '100' (Nullen als Ziffern)
    // Jeder Buchstabe 'O', der rechts oder links neben sich einen anderen Buchstaben
    // hat, den lasse ich als Buchstaben. Alle anderen wandle ich in die Ziffer '0' Null.
    c0:=#0;
    c1:=sText[1];
  //  c2:=#0;
    for i:=1 to Length(sText) do
      begin
      if i<Length(sText) then c2:=sText[i+1] else c2:=#0;
      if c1='O' then
        begin
        if (c0 in ['0'..'9',' ',#0]) and
           (c2 in ['0'..'9',' ',#0,'O']) then
          begin
          // Ist eine Ziffer Null
          c1:='0';
          sText[i]:=c1;
          end;
        end;
      c0:=c1;
      c1:=c2;
      end;

//    for i:=1 to Length(sText) do
//      begin
//      if sText[i]=' ' then sText[i]:='_';
//      end;

    aObjs.AddString(sText,nTextX,nTextY,nTextSize);

    if (nTextX=100) and (nTextY=19) and (nTextSize=1) then
      begin
      // Score-Anzeige Player 1
      Val(sText,aObjs.nScore,i);
      end;
    if (nTextX=480) and (nTextY=19) and (nTextSize=0) then
      begin
      // Highscore-Anzeige in der Mitte des Screens
      Val(sText,aObjs.nHighscore,i);
      end;

    if (nTextX=400) and (nTextY=667) and (nTextSize=2) then
      begin
      // Highscore-Initialien
      aObjs.sHighName:=sText;
      end;

//    Writeln('"',sText,'"');
//    bWriteln:=True;

    if sText='STARTKNOEPFE DRUECKEN' then bStartText:=True;
    if sText='SPIELENDE'             then bEndeText:=True;
    if sText='IHR ERGEBNIS IST EINES DER ZEHN BESTEN' then bHighText:=True;

    sText:='';
    end;

  begin
  // Swappen der beiden Buffer
  pObjsTmp:=aObjs;
  aObjs:=aObjs0;
  aObjs0:=pObjsTmp;

  aObjs.Clear;
  nCounter:=0;
  nShipDetect:=0;
  p:=@pRxBuf.aVectorRam[0];
  vx:=0;
  vy:=0;
  vs:=0;
  v1x:=0;
  v1y:=0;
  sText:='';
  bStartText:=False;
  bEndeText :=False;
  bHighText :=False;

  bWriteln:=False;

  if (p^<>$E001) and (p^<>$E201) then Exit;
  while True do
    begin
    bNewChar:=False;
    nCmd:=p^;
    nOp:=nCmd shr 12;
    nArg:=nCmd and $0FFF;
    case nOp of
      $0A: begin // LABS
        // Y-Koordinaten kommen 895..128 und ich rechne um nach 0..767
        vy:=895-nArg;
        Inc(p);
        vx:=p^ and $0FFF;
        vs:=p^ shr 12;
        end;
      $0B: begin // HALT
        break;
        end;
      $0C: begin // JSLR
        case nArg of
          $08F3: aObjs.AddStone(AstObjectId_Stone1,vx,vy,vs,nCounter){};
          $08FF: aObjs.AddStone(AstObjectId_Stone2,vx,vy,vs,nCounter){};
          $090D: aObjs.AddStone(AstObjectId_Stone3,vx,vy,vs,nCounter){};
          $091A: aObjs.AddStone(AstObjectId_Stone4,vx,vy,vs,nCounter){};
          $0929: aObjs.AddSaucer(vx,vy,vs,nCounter);
          $0880: aObjs.AddEplosion(AstObjectId_Explosion,vx,vy,vs,nCounter); // klmit Explosion
          $0896: aObjs.AddEplosion(AstObjectId_Explosion,vx,vy,vs,nCounter); // Explosion
          $08B5: aObjs.AddEplosion(AstObjectId_Explosion,vx,vy,vs,nCounter); // klmit Explosion
          $08D0: aObjs.AddEplosion(AstObjectId_Explosion,vx,vy,vs,nCounter); // kleine Explosion
          $0852: ; // "(C) 1979 ATARI INC", Positionierung ist implizit
          $0A6D: Inc(aObjs.nShipsLeft);
          else begin
            if ImpHandleChars then
              begin
              bNewChar:=True;
              end
            else
              begin
              with aObjs.Add(nArg,vx,vy,vs) do
                begin
                //nRadiusX:=nRad;
                //nRadiusY:=nRad;
                nObjId:=nArg;
                //Inc(nRad);
                Writeln(IntToHex(nArg,4));
                bWriteln:=True;
                end;
              end;
            end;
          end;
        end;
      $0D: begin // RTSL
        end;
      $0E: begin // JMPL
        end;
      $0F: begin // SVEC
        end;
      else begin
        // Raumschiff und Schuesse
        // nOp ist Skalierung;
        dy:=nArg and $03FF;
        //if (nArg and $0400) <>0 then dy:=-dy;
        if (nArg and $0400) =0 then dy:=-dy;
        Inc(p);
        dx:=p^ and $03FF;
        if (p^ and $0400) <>0 then dx:=-dx;
        nBrg:=p^ shr 12;
        if (dx=0) and (dy=0) and (nBrg=15) then
          begin
          aObjs.AddShot(vx,vy,vs,nCounter);
          end;
        if (nOp=6) and (nBrg=12) and (dy<>0) and (dy<>0) then
          begin
          if nShipDetect=0 then
            begin
            v1x:=dx;
            v1y:=dy;
            nShipDetect:=1;
            end
          else if nShipDetect=1 then
            begin
            dx:=v1x-dx;
            dy:=v1y-dy;
            aObjs.AddShip(vx,vy,dx,dy,nCounter);
            nShipDetect:=2;
            //Writeln(dx,',',dy);
            end;
          end
        else begin
          if nShipDetect=1 then nShipDetect:=0;
          end;
        end;
      end;

    if not bNewChar and (sText<>'') then
      begin
      ImpEvaluateText;
      end;

    Inc(p);

    Inc(nCounter);
    if nCounter>=1000 then
      begin
      // Scheint was nicht zu stimmen
      aObjs.Clear;
      break;
      end;
    end;

  //////////////////////////////////////////////////////////////////////////////
  // Weitere Auswertungen

  nShipsLeft:=aObjs.nShipsLeft;
  nHighscore:=aObjs.nHighscore;
  sHighName :=aObjs.sHighName;

  eGameStatus0:=eGameStatus;

  if bStartText then
    begin
    // Der Text "Startknoepfe druecken" wird angezeigt
    eGameStatus:=ags_Standby;
    end;

  if bEndeText then
    begin
    // Der Text "SPIELENDE" wird angezeigt
    eGameStatus:=ags_GameOver;
    end;

  if bHighText then
    begin
    // Der Text "IHR ERGEBNIS IST EINES DER ZEHN BESTEN" wird angezeigt
    eGameStatus:=ags_Highscore;
    end;

  if bWriteln then
    begin
    if Assigned(pPaintBox) then
      begin
      aObjs.Paint(pPaintBox.Canvas);
      end;
    Writeln('----');
    end;

  if nShipsLeft>0 then
    begin
    eGameStatus:=ags_Playing;
    end
  else if (aObjs.pShip=Nil) and (nShipsLeft=0) then
    begin
    if eGameStatus=ags_Playing then
      begin
      eGameStatus:=ags_GameOver;
      StatGameOver(self);
      end
    else begin
      if (eGameStatus<>ags_Highscore) and (eGameStatus<>ags_GameOver) then
        begin
        eGameStatus:=ags_Standby;
        end;
      end;
    end;

  //if eGameStatus<>eGameStatus0 then
  //  begin
  //  Writeln('Status ',Integer(eGameStatus0),' -> ',Integer(eGameStatus));
  //  end;

  if eGameStatus=ags_Playing then
    begin
    if eGameStatus0=ags_Playing then
      begin
      // Im laufenden Spiel, evtl ueberlauf des Punktezaehlers
      if aObjs.nScore<nScore0Objs then
        begin
        Inc(nScore100k,100000);
        end;
      end
    else begin
      // Gerade ein neues Spiel begonnen
      // Hunderttausender Ueberlaufzaehler fuer Score zuruecksetzen
      nScore100k:=0;
      // Und System Timestamp Speichern
      nGameStartT:=GetTimeMS;
      ResetGameStats;
      startsyncwbyte:=true;
      ShotLengthFinder.Reset;
    //  StatNewGame(self);
      end;
    nGameStopT:=GetTimeMS;
    nGameTimeMS:=GetTimeDiffMS(nGameStartT,nGameStopT);
    end
  else begin
    // Spiel ist vorbei
    // Score uebernehmen, falls offensichtlich abweichend
    if nScore mod 100000 <> aObjs.nScore then
      begin
      nScore100k:=0;
      end;
    end;

  // nScoreAdd zum pruefen was abgeschossen wurde
  nScoreAdd:=nScore;

  nScore:=aObjs.nScore+nScore100k;
  nScore0Objs:=aObjs.nScore;

  nScoreAdd:=nScore-nScoreAdd;

  // Game Statistics
  if aObjs.BigStones>aObjs0.BigStones then
    begin
    // Neues Level
    Inc(aGameStats.nWave);
    aGameStats.nStonesWave:=aObjs.BigStones;
    Inc(aGameStats.nStonesAll,aGameStats.nStonesWave);
    aGameStats.nScoreWave:=0;
    aGameStats.nSaucerSmaShotWave:=0;
    aGameStats.nSaucerBigShotWave:=0;
    aGameStats.nSaucerSmaWave:=0;
    aGameStats.nSaucerBigWave:=0;
   // StatNewLevel(self);
    end;

  Inc(aGameStats.nScoreWave,nScoreAdd);

  if nScoreAdd=1000 then
    begin
    // Kleines UFO erwischt
    Inc(aGameStats.nSaucerSmaShotWave);
    Inc(aGameStats.nSaucerSmaShotAll);
    end;
  if nScoreAdd=200 then
    begin
    // Grosses UFO erwischt
    Inc(aGameStats.nSaucerBigShotWave);
    Inc(aGameStats.nSaucerBigShotAll);
    end;

  if (aObjs.pSaucer<>Nil) and (aObjs0.pSaucer=Nil) then
    begin
    // Neues UFO auf dem Screen
    if aObjs.pSaucer.nScale=14 then
      begin
      // Kleines UFO
      Inc(aGameStats.nSaucerSmaWave);
      Inc(aGameStats.nSaucerSmaAll);
      end
    else begin
      // Grosses UFO
      Inc(aGameStats.nSaucerBigWave);
      Inc(aGameStats.nSaucerBigAll);
      end;
    end;

  end;

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

end.
