package doertbrexel.operations;

import doertbrexel.GameStatus;
import doertbrexel.ProcessorContext;

/**
 * @author G.Doert-Brexel
 * 
 * Diese Familie von Befehlen mit den Op-Codes 0 ... 9
 * zeichnet einen Vektor mit der Helligkeit Z (0 = unsichtbar, ...,
 * 15 = ganz hell) in Richtung (X, Y), wobei X und Y relativ zur aktuellen 
 * Strahlposition zu verstehen sind. Die Vorzeichenbits Xs und Ys geben
 * die Richtung an (0 = positiv = rechts bzw. oben).
 * X und Y werden durch einen Skalierungsfaktor dividiert, der sich als
 * Summe aus dem Op-Code und dem globalen Skalierungsfaktor GSF ergibt.
 * 
 * Y = Bits 0 ... 9 aus erstem Wort     Ys Bit 10 aus erstem Wort
 * X = Bits 0 ... 9 aus zweitem Wort    Xs Bit 10 aus zweitem Wort
 * 
 * Helligkeit Z = Bits 12 ... 15 aus zweitem Wort

 *
 */
public class VctrOperation extends TwoWordOperation
{
	private int	mDeltaX		= 0;
	private int	mDeltaY		= 0;
	private int	mBrightness	= 0;

	protected VctrOperation(int pAddressPointer, int pOpcode, int pFirstWord,
			int pSecondWord)
	{
		super(pAddressPointer, pOpcode, pFirstWord, pSecondWord);
		mDeltaY = mFirstWord & 0x000003FF;
		if ((mFirstWord & 0x00000400) == 0x00000400)
			mDeltaY = -mDeltaY;
		mDeltaX = mSecondWord & 0x000003FF;
		if ((mSecondWord & 0x00000400) == 0x00000400)
			mDeltaX = -mDeltaX;
		mBrightness = (mSecondWord & 0x0000F000) >>> 12;
	}

	public String toString()
	{
		return getAddress() + " VCTR - " + super.toString() + " -> DeltaX=" + mDeltaX
				+ " DeltaY=" + mDeltaY + " Brightness=" + mBrightness;
	}

	public int getScalingFactor(int pGlobalScalingFactor)
	{
		return mOpcode + pGlobalScalingFactor;
	}

	@Override
	public boolean execute(ProcessorContext processorContext, GameStatus pGameStatus)
	{
		if (mBrightness == 15 && mDeltaX == 0 && mDeltaY == 0)
		{
			pGameStatus.addShot(processorContext.absX, processorContext.absY);
		} else if (mBrightness == 12 && mDeltaX != 0 && mDeltaY != 0)
		{
			//			System.out.println(this + " globalScalingFactor=" + processorContext.globalScalingFactor);
			if (mOpcode == 6)
			{
				switch (processorContext.shipstep)
				{
					case 0 :
						processorContext.relX = mDeltaX;
						processorContext.relY = mDeltaY;
						++processorContext.shipstep;
						break;
					case 1 :
						pGameStatus.setSpaceShip(processorContext.absX,
								processorContext.absY, processorContext.relX - mDeltaX,
								processorContext.relY - mDeltaY);
						++processorContext.shipstep;
						break;
				}
			}
		} else
			processorContext.shipstep = 0;
		return false;
	}
}