/* If we are under DOS, install the first half of the file.
 * Otherwise, install the second half.
 **************************************************************/

#ifdef __TURBOC__

/* Read DOS */
/* by Jason Hunter
 * April, 1995
*/

/* This file's primary purpose is to give DOS the ability to read
 * a set of bytes from the Linux partition.  It has to look at the
 * partition table and find where the Linux partition is and what
 * the drive's parameters look like so it can convert a ulong
 * representative of a logical byte number in the partition to
 * a track #, sector #, head assignment, and offset in that sector.
 * It's in essence a DOS version of Linux's lseek off /dev/hda4.
*/

#include <stdio.h>
#include <bios.h>
#include <dos.h>
#include <stdlib.h>
#include <conio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>

#define DEBUG 0  /* true==1 */

/* globals */
int HEADS;
int SECTORS;
long start;
long num_sect;


void convert(unsigned long x, int *head, int *track, int *sector, int *offset)
{
  unsigned long logicsect;
  unsigned long absolsect;

  logicsect = x / 512;
  absolsect = logicsect+start;
  *track = (absolsect / (SECTORS*HEADS));  /* if we had to % we'd fall off */
  *head = (absolsect / SECTORS) % HEADS;
  *sector = (absolsect % SECTORS) +1;
  *offset = x % 512;

  if (DEBUG)
  {
    printf ("Beginning at sect=%ld...\n",start);
    printf ("byte=%4ld logic: %-4ld absol: %-4ld head: %-2d track: %-3d sect: %-4d  offset: %-d\n",
                x,logicsect,absolsect,*head,*track,*sector,*offset);
  }

  if (absolsect > start+num_sect)
  {
    fprintf(stderr,"You cannot access ABOVE the Linux partition.\n");
    exit(-1);
  }
}


void examine_drive(void)
{
  int rc;
  int i,j,k;
  char buf[512];
  int run[16];

  int boot, head, sect, cyl;
  int first_head, first_sect, first_cyl;
  int end_head, end_sect, end_cyl;
  int system;

  if (DEBUG)
    printf ("\n\n------------------\n");

  /* read the partition table from the first hard drive */
  rc = biosdisk (2, 128, 0, 0, 1, 1, buf);
  if (DEBUG)
    printf ("rc is %d\n",rc);

  /* For fun, tell what's the OEM for the table */
  if (DEBUG)
    printf ("\nOEM:\n%c%c%c%c\n",buf[2],buf[3],buf[4],buf[5]);

  /* scan through the four entries looking for Linux */
  j=1;

  for (k=0;k<4;k++)
  {
    for (i=0;i<16;i++)
    {
      run[i] = (int) buf[446+(k*16)+i];
      if (run[i] < 0)
        run[i] = run[i] - 0xff00;
      if (DEBUG)
      {
        printf("%x, ",run[i]);
        if (!(j % 16)) printf ("\n\n");
        j++;
      }
    }
    if (run[4] == 0x83) break;   /* we found the Linux partition */
  }

  /* If you never found Linux... */
  if (k==4)  /* error!! */
  {
    fprintf(stderr,"Could not find Linux partition\n");
    exit (-1);
  }

  boot=run[0];
  first_head=run[1];
  first_sect=run[2];
  first_cyl=run[3];

  first_cyl += (first_sect / 64)*256;  /* add top 2 bits of first_sect */
  first_sect = first_sect % 192;       /* and remove them from first_sect */

  system=run[4];

  end_head=run[5];
  end_sect=run[6];
  end_cyl=run[7];

  end_cyl += (end_sect / 64)*256;      /* add top 2 bits of end_sect */
  end_sect = end_sect % 192;           /* and remove them from end_sect */

  start = (long) run[11];
  start = start*256 + run[10];
  start = start*256 + run[9];
  start = start*256 + run[8];
  num_sect = (long) run[15];
  num_sect = num_sect*256 + run[14];
  num_sect = num_sect*256 + run[13];
  num_sect = num_sect*256 + run[12];

  HEADS = end_head+1;
  SECTORS = start/(first_cyl*HEADS);

  /* Brag about the results */
  if (DEBUG)
  {
    printf ("\n      THE DRIVE HAS:\n");
    printf ("    number of heads: %d\n",HEADS);
    printf ("  number of sectors: %d\n",SECTORS);
    printf ("\n   LINUX ITSELF HAS:\n");
    printf ("     boot indicator: %xh\n",boot);
    printf ("     beginning head: %d\n",first_head);
    printf ("   beginning sector: %d\n",first_sect);
    printf (" beginning cylinder: %d\n",first_cyl);
    printf ("   system indicator: %xh\n",system);
    printf ("        ending head: %d\n",end_head);
    printf ("      ending sector: %d\n",end_sect);
    printf ("    ending cylinder: %d\n",end_cyl);
    printf ("relative start sect: %ld\n",start);
    printf ("  number of sectors: %ld\n",num_sect);
  }

}


int readdisk( char *buf, unsigned long loc, size_t size )
{
  int head,cyl,sect,offset;
  int i,rc;
  char temp[20480];           /* when this gets big, it tromps data */

  convert (loc,&head,&cyl,&sect,&offset);

  rc = biosdisk (2, 128, head,cyl,sect,(size/512)+2,temp);

  if (DEBUG)
    printf ("rc is %d\n",rc);

  memcpy(buf,temp+offset,size);  /* no ending NULL */

  if (rc==0)
    return size;
  else
    return 0;
}




/* If we're under Linux (for testing) include the following */

#else


/* Read Linux */
/* Written by Dave Lutz */

/* This file is to be used for testing this software under Linux. */

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

int IN;				/* holds the open input dev */
typedef unsigned char byte;

int open_dev( char *name )
{
  return( IN = open( name, O_RDONLY ) ); /* open the file */
}

int readdisk( byte *buf, unsigned long loc, size_t size )
{
  lseek( IN, loc, SEEK_SET );
  return( read( IN, buf, size ) );
}

#endif
