/*
 * File:	wx_item.cc
 * Purpose:	Panel items implementation
 * Author:	Julian Smart
 * Created:	1993
 * Updated:	August 1994
 * RCS_ID:      $Id: wx_item.cc,v 1.1 1994/08/14 21:59:17 edz Exp $
 * Copyright:	(c) 1993, AIAI, University of Edinburgh
 */

static const char sccsid[] = "%W% %G%";

/* When implementing a new item, be sure to:
 *
 * - add the item to the parent panel
 * - set window_parent to the parent
 * - NULL any extra child window pointers not created for this item
 *   (e.g. label control that wasn't needed)
 * - delete any extra child windows in the destructor (e.g. label control)
 * - implement GetSize and SetSize
 * - to find panel position if coordinates are (-1, -1), use GetPosition
 * - call AdvanceCursor after creation, for panel layout mechanism.
 *
 */

#include "wx.h"
#pragma hdrstop
#include "wx_privt.h"
#include <iostream.h>
#include <stdio.h>
#include <stdlib.h>

 /*
  * If we have Fafa lib, include header.
  * else, force Windows Look
  */

#if FAFA_LIB
#include "fafa.h"
#endif

#if WINDOWS_LOOK
#define STATIC_CLASS     "STATIC"
#define STATIC_FLAGS     (SS_LEFT|WS_CHILD|WS_VISIBLE)
//#define STATIC_IS_FAFA   FALSE
#define CHECK_CLASS      "BUTTON"
#define CHECK_FLAGS      (BS_AUTOCHECKBOX|WS_TABSTOP|WS_CHILD)
#define CHECK_IS_FAFA   FALSE
#define RADIO_CLASS      "BUTTON"
#define RADIO_FLAGS      (BS_AUTORADIOBUTTON|WS_CHILD|WS_VISIBLE)
#define RADIO_SIZE       20
#define RADIO_IS_FAFA   FALSE
#define PURE_WINDOWS
#define GROUP_CLASS      "BUTTON"
#define GROUP_FLAGS      (BS_GROUPBOX|WS_CHILD|WS_VISIBLE)
//#define GROUP_IS_FAFA   FALSE
#else
#define STATIC_CLASS     "FafaStatic"
#define STATIC_FLAGS     (FS_Y4|FS_X4|FS_DEFAULT)
//#define STATIC_IS_FAFA   TRUE
#define CHECK_CLASS      "FafaCheck"
#define CHECK_FLAGS      (FC_REC_DWN|FC_DEFAULT|WS_VISIBLE)
#define CHECK_IS_FAFA   TRUE
#define RADIO_CLASS      "FafaCheck"
#define RADIO_FLAGS      (FC_CIR_DWN|FC_RADIO|WS_CHILD|WS_VISIBLE)
#define RADIO_SIZE       20
#define RADIO_IS_FAFA   TRUE
#define GROUP_CLASS      "FafaStatic"
#define GROUP_FLAGS      (FS_CADRE_DOWN|WS_CHILD|WS_VISIBLE)
//#define GROUP_IS_FAFA   TRUE
#endif
#define BITCHECK_FLAGS   (FB_BITMAP|FC_BUTTONDRAW|FC_DEFAULT|WS_VISIBLE)
#define BITRADIO_FLAGS   (FC_BUTTONDRAW|FB_BITMAP|FC_RADIO|WS_CHILD|WS_VISIBLE)

#define MEANING_CHARACTER '0'
#define DEFAULT_ITEM_WIDTH  200
#define DEFAULT_ITEM_HEIGHT 80
#define EDIT_CONTROL_FACTOR (15.0/10.0)
                                        // Scale font to get edit control height

/* Use this if you don't have precompiled headers
#include <windows.h>
#include "common.h"

#include "wx_item.h"
#include "wx_lbox.h"
#include "wx_rbox.h"
#include "wx_buttn.h"
#include "wx_choic.h"
#include "wx_check.h"
#include "wx_messg.h"
#include "wx_slidr.h"
#include "wx_menu.h"
#include "wx_txt.h"
#include "wx_mtxt.h"

#include "wx_event.h"
#include "wx_utils.h"
#include "wx_main.h"
#include "wx_frame.h"
*/

#if CTL3D
#include <ctl3d.h>
#endif

// Find maximum size of window/rectangle
void wxFindMaxSize(HWND hwnd, RECT *rect);
wxList wxScrollBarList(wxKEY_INTEGER);

// Subclassed EDIT control to intercept VK_RETURN messages

#if !defined(APIENTRY)	// NT defines APIENTRY, 3.x not
#define APIENTRY FAR PASCAL
#endif
 
#ifdef WIN32
#define _EXPORT /**/
#else
#define _EXPORT _export
typedef signed short int SHORT ;
#endif
// The function
extern LONG APIENTRY _EXPORT
  wxSubclassedControlProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam);
// The MakeProcInstance version of the function
FARPROC wxEditControlSubClassProc = 0;
wxList *wxControlHandleList = NULL;
wxItem *wxFindControlFromHandle(HWND hWnd);
void wxAddControlHandle(HWND hWnd, wxItem *item);

// Item members
wxItem::wxItem(void)
{
  isFafa = FALSE ;
  oldWndProc = 0;
}

wxItem::~wxItem(void)
{
  // item may be a menu, so check.
  wxObject *obj = (wxObject *)GetParent();
  if (!obj || !wxSubType(obj->__type, wxTYPE_PANEL)) return;

  // If we delete an item, we should initialize the parent panel,
  // because it could now be invalid.
  wxPanel *panel = (wxPanel *)GetParent();
  if (panel)
  {
    panel->last_created = NULL;
    panel->cursor_x = PANEL_LEFT_MARGIN;
    panel->cursor_y = PANEL_TOP_MARGIN;
    panel->max_height = 0;
    panel->max_line_height = 0;
    panel->max_width = 0;
    panel->hSpacing = PANEL_HSPACING;
    panel->vSpacing = PANEL_VSPACING;
    panel->initial_hspacing = panel->hSpacing ;
    panel->initial_vspacing = panel->vSpacing ;
    panel->current_hspacing = panel->hSpacing ;
    panel->current_vspacing = panel->vSpacing ;

    panel->new_line = FALSE;
    panel->label_position = wxHORIZONTAL;
    panel->has_child = FALSE ;
    panel->last_created = 0 ;
  }
}

void wxItem::GetSize(int *width, int *height)
{
  HWND wnd = (HWND)ms_handle;
  RECT rect;
  rect.left = -1; rect.right = -1; rect.top = -1; rect.bottom = -1;

  wxFindMaxSize(wnd, &rect);

  *width = rect.right - rect.left;
  *height = rect.bottom - rect.top;
}

void wxItem::GetPosition(int *x, int *y)
{
  HWND wnd = (HWND)ms_handle;
  wxWindow *parent = GetParent();
  RECT rect;
  rect.left = -1; rect.right = -1; rect.top = -1; rect.bottom = -1;

  wxFindMaxSize(wnd, &rect);

  // Since we now have the absolute screen coords,
  // if there's a parent we must subtract its top left corner
  POINT point;
  point.x = rect.left;
  point.y = rect.top;
  if (parent)
  {
    wxWnd *cparent = (wxWnd *)(parent->handle);
    ::ScreenToClient(cparent->handle, &point);
  }

  *x = point.x;
  *y = point.y;
}

void wxItem::SetSize(int x, int y, int width, int height)
{
  wxWindow::SetSize(x, y, width, height);
}

void wxItem::SetClientSize(int width, int height)
{
  SetSize(-1, -1, width, height);
}

void wxItem::SetLabel(char *label)
{
  SetWindowText((HWND)ms_handle, label);
}

char *wxItem::GetLabel(void)
{
  GetWindowText((HWND)ms_handle, wxBuffer, 1000);
  return wxBuffer;
}

void wxItem::SetFocus(void)
{
  wxWindow::SetFocus();
}

void wxItem::Show(Bool show)
{
  HWND wnd = (HWND)ms_handle;
  int cshow;
  if (show)
    cshow = SW_SHOW;
  else
    cshow = SW_HIDE;
  ShowWindow(wnd, cshow);
  if (show)
    BringWindowToTop(wnd);
}

float wxItem::GetCharHeight(void)
{
  TEXTMETRIC lpTextMetric;
  HWND wnd = (HWND)ms_handle;
  HDC dc = GetDC(wnd);

  GetTextMetrics(dc, &lpTextMetric);
  ReleaseDC(wnd, dc);

  return (float)lpTextMetric.tmHeight;
}

float wxItem::GetCharWidth(void)
{
  TEXTMETRIC lpTextMetric;
  HWND wnd = (HWND)ms_handle;
  HDC dc = GetDC(wnd);

  GetTextMetrics(dc, &lpTextMetric);
  ReleaseDC(wnd, dc);

  return (float)lpTextMetric.tmAveCharWidth;
}

void wxItem::GetTextExtent(const char *string, float *x, float *y,
                           float *descent, float *externalLeading,
                           wxFont *the_font)
{
  HWND wnd = (HWND)ms_handle;
  HDC dc = GetDC(wnd);
  HFONT fnt = 0;
  HFONT was = 0;

  if (the_font && (fnt=the_font->GetInternalFont(dc)))
    was = SelectObject(dc,fnt) ;

  SIZE sizeRect;
  TEXTMETRIC tm;
  GetTextExtentPoint(dc, string, strlen(string), &sizeRect);
  GetTextMetrics(dc, &tm);

  if (the_font && fnt && was)
    SelectObject(dc,was) ;
  ReleaseDC(wnd, dc);

  *x = (float)sizeRect.cx;
  *y = (float)sizeRect.cy;
  if (descent) *descent = (float)tm.tmDescent;
  if (externalLeading) *externalLeading = (float)tm.tmExternalLeading;

}

// Call this repeatedly for several wnds to find the overall size
// of the widget.
// Call it initially with -1 for all values in rect.
// Keep calling for other widgets, and rect will be modified
// to calculate largest bounding rectangle.
void wxFindMaxSize(HWND wnd, RECT *rect)
{
  int left = rect->left;
  int right = rect->right;
  int top = rect->top;
  int bottom = rect->bottom;

  GetWindowRect(wnd, rect);

  if (left < 0)
    return;

  if (left < rect->left)
    rect->left = left;

  if (right > rect->right)
    rect->right = right;

  if (top < rect->top)
    rect->top = top;

  if (bottom > rect->bottom)
    rect->bottom = bottom;

}

// General Windows item code
// wxWnd (wx_win.cc) catches commands from child items, finds the child
// and sends a Command message to the child.

// Buttons

BOOL wxButton::MSWCommand(UINT param, WORD id)
{
  if (param == BN_CLICKED)
  {
    wxCommandEvent event(wxEVENT_TYPE_BUTTON_COMMAND);
    event.eventObject = this;
    ProcessCommand(event);
    return TRUE;
  }
  else return FALSE;
}

BOOL wxCheckBox::MSWCommand(UINT param, WORD id)
{
  wxCommandEvent event(wxEVENT_TYPE_CHECKBOX_COMMAND);
  event.commandInt = GetValue();
  event.eventObject = this;
  ProcessCommand(event);
  return TRUE;
}

BOOL wxText::MSWCommand(UINT param, WORD id)
{
/*
  // Debugging
  wxDebugMsg("Edit control %d: ", (int)id);
  switch (param)
  {
    case EN_SETFOCUS:
      wxDebugMsg("EN_SETFOCUS\n");
      break;
    case EN_KILLFOCUS:
      wxDebugMsg("EN_KILLFOCUS\n");
      break;
    case EN_CHANGE:
      wxDebugMsg("EN_CHANGE\n");
      break;
    case EN_UPDATE:
      wxDebugMsg("EN_UPDATE\n");
      break;
    case EN_ERRSPACE:
      wxDebugMsg("EN_ERRSPACE\n");
      break;
    case EN_MAXTEXT:
      wxDebugMsg("EN_MAXTEXT\n");
      break;
    case EN_HSCROLL:
      wxDebugMsg("EN_HSCROLL\n");
      break;
    case EN_VSCROLL:
      wxDebugMsg("EN_VSCROLL\n");
      break;
    default:
      wxDebugMsg("Unknown EDIT notification\n");
      break;
  }
*/
  int eventTyp = 0;
  switch (param)
  {
    case EN_SETFOCUS:
      eventTyp = wxEVENT_TYPE_SET_FOCUS;
      break;
    case EN_KILLFOCUS:
      eventTyp = wxEVENT_TYPE_KILL_FOCUS;
      break;
    case EN_CHANGE:
      break;
    case EN_UPDATE:
      eventTyp = wxEVENT_TYPE_TEXT_COMMAND;
      break;
    case EN_ERRSPACE:
      break;
    case EN_MAXTEXT:
      break;
    case EN_HSCROLL:
       break;
    case EN_VSCROLL:
      break;
    default:
      break;
  }
  if (eventTyp != 0)
  {
    wxCommandEvent event(eventTyp);
    event.commandString = GetValue();
    event.eventObject = this;
    ProcessCommand(event);
    return TRUE;
  }
  else
    return FALSE;
}

BOOL wxListBox::MSWCommand(UINT param, WORD id)
{
  wxCommandEvent event(wxEVENT_TYPE_LISTBOX_COMMAND);
/*
  if (param == LBN_SELCANCEL)
  {
    event.extraLong = FALSE;
  }
*/
  if (param == LBN_SELCHANGE)
  {
/*
    if (multiple != wxMULTIPLE)
    {
      event.commandInt = GetSelection();
      event.clientData = GetClientData(event.commandInt);
      event.commandString = copystring(GetStringSelection());
    }
*/
    int *liste ;
    int count = GetSelections(&liste) ;
    if (count)
    {
      event.commandInt = liste[0] ;
      event.clientData = GetClientData(event.commandInt);
      event.commandString = copystring(GetString(event.commandInt));
    }
    else
    {
      event.commandInt = -1 ;
      event.commandString = copystring("") ;
    }

    event.eventObject = this;
    ProcessCommand(event);
    delete event.commandString ;
/*
    if (multiple != wxMULTIPLE)
      delete[] event.commandString;
*/
    return TRUE;
  }
  else if (param == LBN_DBLCLK)
  {
    wxPanel *parent = (wxPanel *)GetParent();
    if (parent)
      parent->OnDefaultAction(this);
    return TRUE;
  }
  return FALSE;
}

BOOL wxRadioBox::MSWCommand(UINT param, WORD id)
{
  if (param == BN_CLICKED)
  {
#ifdef WIN32
    for (int i = 0; i < no_items; i++)
      if (id == GetWindowLong(radioButtons[i], GWL_ID))
        selected = i;
#else
    for (int i = 0; i < no_items; i++)
      if (id == GetWindowWord(radioButtons[i], GWW_ID))
        selected = i;
#endif

    wxCommandEvent event(wxEVENT_TYPE_RADIOBOX_COMMAND);
    event.commandInt = selected;
    event.eventObject = this;
    ProcessCommand(event);
    return TRUE;
  }
  else return FALSE;
}

BOOL wxChoice::MSWCommand(UINT param, WORD id)
{
  if (param == CBN_SELCHANGE)
  {
    wxCommandEvent event(wxEVENT_TYPE_CHOICE_COMMAND);
    event.commandInt = GetSelection();
    event.eventObject = this;
    event.commandString = copystring(GetStringSelection());
    ProcessCommand(event);
    delete[] event.commandString;
    return TRUE;
  }
  else return FALSE;
}

wxButton::wxButton(void)
{
  wxWinType = wxTYPE_HWND;
  windows_id = 0;
  ms_handle = 0;
}

wxButton::wxButton(wxPanel *panel, wxFunction Function, char *label,
		   int x, int y, int width, int height,
                   long style, char *name):
  wxbButton(panel, Function, label, x, y, width, height, style, name)
{
  Create(panel, Function, label, x, y, width, height, style, name);
}

wxButton::wxButton(wxPanel *panel, wxFunction Function, wxBitmap *bitmap,
		   int x, int y, int width, int height,
                   long style, char *name):
  wxbButton(panel, Function, bitmap, x, y, width, height, style, name)
{
  Create(panel, Function, bitmap, x, y, width, height, style, name);
}

Bool wxButton::Create(wxPanel *panel, wxFunction Function, char *label,
		   int x, int y, int width, int height,
                   long style, char *name)
{
  if (panel) panel->AddChild(this);
  buttonFont = panel->buttonFont ;
  labelFont = panel->labelFont ;
  backColour = panel->backColour ;
  labelColour = panel->labelColour ;
  buttonColour = panel->buttonColour ;

  wxWinType = wxTYPE_HWND;
  windowStyle = style;

  wxWnd *cparent = NULL;
  if (panel)
    cparent = (wxWnd *)(panel->handle);

  panel->GetValidPosition(&x, &y);

  windows_id = (int)NewId();

/* Sorry, but Fafa seems to mess up default button setting,
 * so we're reverting to normal Windows buttons this time.
 * JACS 29/3/94
#if FAFA_LIB
  HWND wx_button =
      CreateWindowEx(0, "FafaButton", label, FB_TEXT | WS_TABSTOP | WS_CHILD,
                     0, 0, 0, 0, cparent->handle, (HMENU)windows_id,
                     wxhInstance, NULL);
#else
*/
  HWND wx_button =
    CreateWindowEx(0, "BUTTON", label, BS_PUSHBUTTON | WS_TABSTOP | WS_CHILD,
                    0, 0, 0, 0, cparent->handle, (HMENU)windows_id,
                    wxhInstance, NULL);

#if CTL3D
  Ctl3dSubclassCtl(wx_button);
#endif

  ms_handle = (HANDLE)wx_button;

  HDC the_dc = GetWindowDC((HWND)ms_handle) ;
  if (buttonFont && buttonFont->GetInternalFont(the_dc))
    SendMessage((HWND)ms_handle,WM_SETFONT,
                (WPARAM)buttonFont->GetInternalFont(the_dc),0L);
  ReleaseDC((HWND)ms_handle,the_dc) ;

  SetSize(x, y, width, height);
  ShowWindow(wx_button, SW_SHOW);

  Callback(Function);

  panel->AdvanceCursor(this);
  return TRUE;
}

Bool wxButton::Create(wxPanel *panel, wxFunction Function, wxBitmap *bitmap,
		   int x, int y, int width, int height,
                   long style, char *name)
{
  if (panel) panel->AddChild(this);
  buttonFont = panel->buttonFont ;
  labelFont = panel->labelFont ;
  backColour = panel->backColour ;
  labelColour = panel->labelColour ;
  buttonColour = panel->buttonColour ;
  wxWinType = wxTYPE_HWND;
  windowStyle = style;

  wxWnd *cparent = NULL;
  if (panel)
    cparent = (wxWnd *)(panel->handle);

  panel->GetValidPosition(&x, &y);

  windows_id = (int)NewId();

#if FAFA_LIB
  if (width<0)
    width = bitmap->GetWidth() ;
  if (height<0)
    height = bitmap->GetHeight() ;
  width += FB_MARGIN ;
  height+= FB_MARGIN ;
  HWND wx_button =
      CreateWindowEx(0, "FafaButton", "?", FB_BITMAP | WS_TABSTOP | WS_CHILD,
                     0, 0, 0, 0, cparent->handle, (HMENU)windows_id,
                     wxhInstance, NULL);
      SendMessage((HWND)wx_button,WM_CHANGEBITMAP,
                  (WPARAM)((bitmap->GetHeight()<<8)+bitmap->GetWidth()),
                  (LPARAM)bitmap->ms_bitmap);
#else
  HWND wx_button =
    CreateWindowEx(0, "BUTTON", "not implemented", BS_PUSHBUTTON | WS_TABSTOP | WS_CHILD,
                    0, 0, 0, 0, cparent->handle, (HMENU)windows_id,
                    wxhInstance, NULL);
#if CTL3D
  Ctl3dSubclassCtl(wx_button);
#endif
#endif

  ms_handle = (HANDLE)wx_button;

  HDC the_dc = GetWindowDC((HWND)ms_handle) ;
  if (buttonFont && buttonFont->GetInternalFont(the_dc))
    SendMessage((HWND)ms_handle,WM_SETFONT,
                (WPARAM)buttonFont->GetInternalFont(the_dc),0L);
  ReleaseDC((HWND)ms_handle,the_dc) ;

  SetSize(x, y, width, height);
  ShowWindow(wx_button, SW_SHOW);

  Callback(Function);

  panel->AdvanceCursor(this);
  return TRUE;
}

wxButton::~wxButton(void)
{
}

void wxButton::SetBackgroundColour(wxColour*col)
{
}

void wxButton::SetLabelColour(wxColour*col) 
{
}

void wxButton::SetButtonColour(wxColour*col)
{
}

void wxButton::SetLabel(char *label)
{
#if FAFA_LIB
    // This message will switch from FB_BITMAP style to FB_TEXT, if needed.
    SendMessage((HWND)ms_handle,WM_CHANGEBITMAP,
                (WPARAM)0,
                (LPARAM)NULL);
#endif
  SetWindowText((HWND)ms_handle, label);
}

void wxButton::SetLabel(wxBitmap *bitmap)
{
#if FAFA_LIB
    SendMessage((HWND)ms_handle,WM_CHANGEBITMAP,
                (WPARAM)((bitmap->GetHeight()<<8)+bitmap->GetWidth()),
                (LPARAM)bitmap->ms_bitmap);
#endif
}

char *wxButton::GetLabel(void)
{
  GetWindowText((HWND)ms_handle, wxBuffer, 300);
  return wxBuffer;
}

void wxButton::SetSize(int x, int y, int width, int height)
{
  int currentX, currentY;
  GetPosition(&currentX, &currentY);
  if (x == -1)
    x = currentX;
  if (y == -1)
    y = currentY;

  char buf[300];

  float current_width;

  int cx;
  int cy;
  float cyf;

  HWND button = (HWND)ms_handle;
  wxGetCharSize(button, &cx, &cy,buttonFont);

  GetWindowText(button, buf, 300);
  GetTextExtent(buf, &current_width, &cyf,NULL,NULL,buttonFont);
  if (width < 0)
   width = (int)(current_width + 3*cx) ;
 if (height<0)
   height = (int)(cyf*EDIT_CONTROL_FACTOR) ;
  MoveWindow(button, x, y, width, height, TRUE);

  OnSize(width, height);
}

void wxButton::SetFocus(void)
{
/*
  wxPanel *panel = (wxPanel *)GetParent();
  if (panel)
  {
    wxWnd *wnd = (wxWnd *)panel->handle;
    SendMessage(wnd->handle, DM_SETDEFID, windows_id, 0L);
  }
*/
  wxItem::SetFocus();
}

void wxButton::SetDefault(void)
{
  wxPanel *panel = (wxPanel *)GetParent();
  if (panel)
    panel->defaultItem = this;

  if (panel)
  {
    wxWnd *wnd = (wxWnd *)panel->handle;
    SendMessage(wnd->handle, DM_SETDEFID, windows_id, 0L);
  }
}

// Menus

// Construct a menu with optional title (then use append)
wxMenu::wxMenu(char *Title, wxFunction func):wxbMenu(Title, func)
{
  mustBeBreaked = FALSE ;
  no_items = 0;
  menu_bar = NULL;
  wxWinType = wxTYPE_HMENU;
  ms_handle = (HANDLE)CreatePopupMenu();
  save_ms_handle = NULL ;
  top_level_menu = this;
  if (title)
  {
    Append(-2,title) ;
    AppendSeparator() ;
  }

  Callback(func);
}

// The wxWindow destructor will take care of deleting the submenus.
wxMenu::~wxMenu(void)
{
  if (ms_handle)
    DestroyMenu((HMENU)ms_handle);
  ms_handle = NULL;

  // Windows seems really bad on Menu de-allocation...
  // After many try, here is what I do: RemoveMenu() will ensure
  // that popup are "disconnected" from their parent; then call
  // delete method on each child (which in turn do a recursive job),
  // and finally, DestroyMenu()
  //
  // With that, BoundCheckers is happy, and no complaints...
/*  
  int N = 0 ;
  if (ms_handle)
    N = GetMenuItemCount((HMENU)ms_handle);
  for (int i = N-1; i >= 0; i--)
    RemoveMenu((HMENU)ms_handle, i, MF_BYPOSITION);
*/
  wxNode *node = menuItems.First();
  while (node)
  {
    wxMenuItem *item = (wxMenuItem *)node->Data();
    item->menuBar = NULL;
/*
    // Delete child menus.
    // Beware: they must not be appended to children list!!!
    // (because order of delete is significant)
    if (item->subMenu)
      delete item->subMenu ;
    item->subMenu = NULL ;
*/
    wxNode *next = node->Next();
    delete item;
    delete node;
    node = next;
  }
/*
  if (ms_handle)
    DestroyMenu((HMENU)ms_handle);
  ms_handle = NULL;
*/
}

void wxMenu::Break(void)
{
  mustBeBreaked = TRUE ;
}

// Ordinary menu item
void wxMenu::Append(int Id, char *Label, char *helpString, Bool checkable)
{
  // 'checkable' parameter is useless for Windows.
  wxMenuItem *item = new wxMenuItem;
  item->checkable = checkable ;
  item->itemId = Id;
  item->itemName = copystring(Label);
  item->subMenu = NULL;
  if (helpString)
    item->helpString = copystring(helpString);

  menuItems.Append(item);

  int ms_flags = mustBeBreaked? MF_MENUBREAK : 0 ;
  mustBeBreaked = FALSE ;

  if (ms_handle)
    AppendMenu((HMENU)ms_handle, MF_STRING|ms_flags, Id, Label);
  else if (save_ms_handle) // For Dynamic Menu Append, Thx!
    AppendMenu((HMENU)save_ms_handle, MF_STRING|ms_flags, Id, Label);

  if (Id==-2)
  {
    int ms_flag = MF_DISABLED;
    if (ms_handle)
      EnableMenuItem((HMENU)ms_handle, no_items, MF_BYPOSITION | ms_flag);
    else if (save_ms_handle) // For Dynamic Menu Append, Thx!!
      EnableMenuItem((HMENU)save_ms_handle, no_items, MF_BYPOSITION | ms_flag);
  }

  no_items ++;
}

void wxMenu::AppendSeparator(void)
{
  int ms_flags = mustBeBreaked? MF_MENUBREAK : 0 ;
  mustBeBreaked = FALSE ;

  if (ms_handle)
    AppendMenu((HMENU)ms_handle, MF_SEPARATOR|ms_flags, NULL, NULL);
  else if (save_ms_handle) // For Dynamic Manu Append, Thx!
    AppendMenu((HMENU)save_ms_handle, MF_SEPARATOR|ms_flags, NULL, NULL);

  wxMenuItem *item = new wxMenuItem;
  item->checkable = FALSE ;
  item->itemId = -1;
  menuItems.Append(item);
  no_items ++;
}

// Pullright item
void wxMenu::Append(int Id, char *Label, wxMenu *SubMenu, char *helpString)
{
  SubMenu->top_level_menu = top_level_menu;
  SubMenu->window_parent = this;
  // To do a clean delete, don't append to children list (see ::~wxMenu)
  // Yes you do for my code says JACS..., I'm uncommenting it.
  children->Append(SubMenu);            // Store submenu for later deletion

  wxMenuItem *item = new wxMenuItem;
  item->checkable = FALSE ;
  item->itemId = Id;
  item->itemName = copystring(Label);
  if (helpString)
    item->helpString = copystring(helpString);
  item->subMenu = SubMenu;

  menuItems.Append(item);

  int ms_flags = mustBeBreaked? MF_MENUBREAK : 0 ;
  mustBeBreaked = FALSE ;

  HMENU menu = (HMENU)ms_handle;
  HMENU child = (HMENU)SubMenu->ms_handle;
  //
  // After looking Bounds Checker result, it seems that all
  // menus must be individually destroyed. So, don't reset ms_handle,
  // to  allow ~wxMenu to do the job.
  // NO, my code should work I THINK - JACS
  //
  SubMenu->save_ms_handle = (HANDLE)child;
  SubMenu->ms_handle = NULL;
  AppendMenu(menu, MF_POPUP | MF_STRING | ms_flags, (UINT)child, Label);

  no_items ++;
}

void wxMenu::Enable(int Id, Bool Flag)
{
  int ms_flag;
  if (Flag)
    ms_flag = MF_ENABLED;
  else
    ms_flag = MF_GRAYED;

  wxMenuItem *item = FindItemForId(Id) ;
  if (item==NULL)
    return;

  if (item->subMenu==NULL)
  {
    // Because you reset ms_handle, that cannot works in cascaded menus, no???
    if (ms_handle)
      EnableMenuItem((HMENU)ms_handle, Id, MF_BYCOMMAND | ms_flag);
    else if (save_ms_handle)
      EnableMenuItem((HMENU)save_ms_handle, Id, MF_BYCOMMAND | ms_flag);
  }
  else
  {
    wxMenu *father = item->subMenu->top_level_menu ;
    wxNode *node = father->menuItems.First() ;
    int i=0 ;
    while (node)
    {
      wxMenuItem *matched = (wxMenuItem*)node->Data() ;
      if (matched==item)
        break ;
      i++ ;
      node = node->Next() ;
    }
    EnableMenuItem((HMENU)father->save_ms_handle, i, MF_BYPOSITION | ms_flag);
  }
}

void wxMenu::Check(int Id, Bool Flag)
{
  wxMenuItem *item = FindItemForId(Id) ;
  if (!item->checkable)
    return ;
  int ms_flag;
  if (Flag)
    ms_flag = MF_CHECKED;
  else
    ms_flag = MF_UNCHECKED;
  if (ms_handle)
    CheckMenuItem((HMENU)ms_handle, Id, MF_BYCOMMAND | ms_flag);
  else if (save_ms_handle)
    CheckMenuItem((HMENU)save_ms_handle, Id, MF_BYCOMMAND | ms_flag);
}

Bool wxMenu::Checked(int Id)
{
  int Flag ;
  if (ms_handle)
    Flag=GetMenuState((HMENU)ms_handle, Id, MF_BYCOMMAND) ;
  else if (ms_handle)
    Flag=GetMenuState((HMENU)save_ms_handle, Id, MF_BYCOMMAND) ;
  if (Flag&MF_CHECKED)
    return TRUE ;
  else
    return FALSE ;
}

void wxMenu::SetTitle(char *label)
{
  if (title)
    delete[] title ;
  if (label)
    title = copystring(label) ;
  else
    title = copystring(" ") ;
  if (ms_handle)
    ModifyMenu((HMENU)ms_handle, 0,
             MF_BYPOSITION | MF_STRING | MF_DISABLED,
             -2,title);
  else if (save_ms_handle)
    ModifyMenu((HMENU)save_ms_handle, 0,
             MF_BYPOSITION | MF_STRING | MF_DISABLED,
             -2,title);
}

char *wxMenu::GetTitle()
{
   return(title) ;
}

void wxMenu::SetLabel(int Id,char *label)
{
  wxMenuItem *item = FindItemForId(Id) ;
  if (item==NULL)
    return;

  if (item->subMenu==NULL)
  {
    if (ms_handle)
    {
      UINT was_flag = GetMenuState((HMENU)ms_handle,Id,MF_BYCOMMAND) ;
      ModifyMenu((HMENU)ms_handle,Id,MF_BYCOMMAND|MF_STRING|was_flag,Id,label) ;
    }
    else if (save_ms_handle)
    {
      UINT was_flag = GetMenuState((HMENU)save_ms_handle,Id,MF_BYCOMMAND) ;
      ModifyMenu((HMENU)save_ms_handle,Id,MF_BYCOMMAND|MF_STRING|was_flag,Id,label) ;
    }
  }
  else
  {
    wxMenu *father = item->subMenu->top_level_menu ;
    wxNode *node = father->menuItems.First() ;
    int i = 0 ;
    while (node)
    {
      wxMenuItem *matched = (wxMenuItem*)node->Data() ;
      if (matched==item)
        break ;
      i++ ;
      node = node->Next() ;
    }
    // Here, we have the position.
    ModifyMenu((HMENU)father->save_ms_handle,i,
               MF_BYPOSITION|MF_STRING|MF_POPUP,
               (UINT)item->subMenu->save_ms_handle,label) ;
  }
}

char *wxMenu::GetLabel(int Id)
{
  static char tmp[128] ;
  int len ;
  if (ms_handle)
    len = GetMenuString((HMENU)ms_handle,Id,tmp,127,MF_BYCOMMAND) ;
  else if (save_ms_handle)
    len = GetMenuString((HMENU)save_ms_handle,Id,tmp,127,MF_BYCOMMAND) ;
  else
    len = 0 ;
  tmp[len] = '\0' ;
  return(tmp) ;
}

BOOL wxMenu::MSWCommand(UINT param, WORD id)
{
  wxCommandEvent event(wxEVENT_TYPE_MENU_COMMAND);
  event.eventObject = this;
  event.commandInt = id;
  ProcessCommand(event);
  return TRUE;
}

extern wxMenu *wxCurrentPopupMenu;
Bool wxWindow::PopupMenu(wxMenu *menu, float x, float y)
{
  HWND hWnd = GetHWND();
  HMENU hMenu = (HMENU)menu->ms_handle;
  POINT point;
  point.x = (int)x;
  point.y = (int)y;
  ::ClientToScreen(hWnd, &point);
  wxCurrentPopupMenu = menu;
  ::TrackPopupMenu(hMenu, 0, point.x, point.y, 0, hWnd, NULL);
  wxYield();
  wxCurrentPopupMenu = NULL;
  return TRUE;
}

// Menu Bar
wxMenuBar::wxMenuBar(void)
{
  wxWinType = wxTYPE_HMENU;

  n = 0;
  menus = NULL;
  titles = NULL;
  menu_bar_frame = NULL;
}

wxMenuBar::wxMenuBar(int N, wxMenu *Menus[], char *Titles[]):wxbMenuBar(N, Menus, Titles)
{
  wxWinType = wxTYPE_HMENU;
}

wxMenuBar::~wxMenuBar(void)
{
  HMENU menu = (HMENU)ms_handle;


  // In fact, don't want menu to be destroyed before MDI
  // shuffling has taken place. Let it be destroyed
  // automatically when the window is destroyed.

//  DestroyMenu(menu);
//  ms_handle = NULL;

  int i;
/*
  // See remarks in ::~wxMenu() method
  // BEWARE - this may interfere with MDI fixes, so
  // may need to remove
  int N = 0 ;

  if (menu_bar_frame && (menu_bar_frame->frame_type == wxSDI))
  {
    if (menu)
      N = GetMenuItemCount(menu) ;
    for (i = N-1; i >= 0; i--)
      RemoveMenu(menu, i, MF_BYPOSITION);
  }
*/
  for (i = 0; i < n; i++)
  {
    delete menus[i];
    delete[] titles[i];
  }
  delete[] menus;
  delete[] titles;

/* Don't destroy menu here, in case we're MDI and
   need to do some shuffling with VALID menu handles.
  if (menu)
    DestroyMenu(menu);
  ms_handle = NULL;
*/
}

void wxMenuBar::Append(wxMenu *menu, char *title)
{
  n ++;
  wxMenu **new_menus = new wxMenu *[n];
  char **new_titles = new char *[n];

  int i;
  for (i = 0; i < n - 1; i++)
    {
      new_menus[i] = menus[i];
      menus[i] = NULL;
      new_titles[i] = titles[i];
      titles[i] = NULL;
    }
  if (menus)
   {
     delete[] menus;
     delete[] titles;
   }
  menus = new_menus;
  titles = new_titles;

  menus[n-1] = menu;
  titles[n-1] = copystring(title);

  menu->menu_bar = this;
}

// Must only be used AFTER menu has been attached to frame,
// otherwise use individual menus to enable/disable items
void wxMenuBar::Enable(int Id, Bool Flag)
{
  int ms_flag;
  if (Flag)
    ms_flag = MF_ENABLED;
  else
    ms_flag = MF_GRAYED;

  wxMenu *itemMenu = NULL;
  wxMenuItem *item = FindItemForId(Id, &itemMenu) ;
  if (!item)
    return;

  if (itemMenu->ms_handle)
    EnableMenuItem((HMENU)itemMenu->ms_handle, Id, MF_BYCOMMAND | ms_flag);
  else if (itemMenu->save_ms_handle)
    EnableMenuItem((HMENU)itemMenu->save_ms_handle, Id, MF_BYCOMMAND | ms_flag);
  
}

void wxMenuBar::EnableTop(int pos,Bool flag)
{
  int ms_flag;
  if (flag)
    ms_flag = MF_ENABLED;
  else
    ms_flag = MF_GRAYED;

  EnableMenuItem((HMENU)ms_handle, pos, MF_BYPOSITION | ms_flag);
  wxWnd *cframe = (wxWnd*)menu_bar_frame->handle ;
  HWND hand = (HWND)cframe->handle ;
  DrawMenuBar(hand) ;
}

// Must only be used AFTER menu has been attached to frame,
// otherwise use individual menus
void wxMenuBar::Check(int Id, Bool Flag)
{
  wxMenu *itemMenu = NULL;
  wxMenuItem *item = FindItemForId(Id, &itemMenu) ;
  if (!item)
    return;
    
  if (!item->checkable)
    return ;
  int ms_flag;
  if (Flag)
    ms_flag = MF_CHECKED;
  else
    ms_flag = MF_UNCHECKED;

  if (itemMenu->ms_handle)
    CheckMenuItem((HMENU)itemMenu->ms_handle, Id, MF_BYCOMMAND | ms_flag);
  else if (itemMenu->save_ms_handle)
    CheckMenuItem((HMENU)itemMenu->save_ms_handle, Id, MF_BYCOMMAND | ms_flag);

//  CheckMenuItem((HMENU)ms_handle, Id, MF_BYCOMMAND | ms_flag);
}

Bool wxMenuBar::Checked(int Id)
{
  wxMenu *itemMenu = NULL;
  wxMenuItem *item = FindItemForId(Id, &itemMenu) ;
  if (!item)
    return FALSE;

  int Flag ;

  if (itemMenu->ms_handle)
    Flag=GetMenuState((HMENU)itemMenu->ms_handle, Id, MF_BYCOMMAND) ;
  else if (itemMenu->save_ms_handle)
    Flag=GetMenuState((HMENU)itemMenu->save_ms_handle, Id, MF_BYCOMMAND) ;
  
//  Flag=GetMenuState((HMENU)ms_handle, Id, MF_BYCOMMAND) ;

  if (Flag&MF_CHECKED)
    return TRUE ;
  else
    return FALSE ;
}

void wxMenuBar::SetLabel(int Id,char *label)
{
  wxMenu *itemMenu = NULL;
  wxMenuItem *item = FindItemForId(Id, &itemMenu) ;

  if (!item)
    return;

  if (itemMenu->ms_handle)
  {
    UINT was_flag = GetMenuState((HMENU)itemMenu->ms_handle,Id,MF_BYCOMMAND) ;
    ModifyMenu((HMENU)itemMenu->ms_handle,Id,MF_BYCOMMAND|MF_STRING|was_flag,Id,label) ;
  }
  else if (itemMenu->save_ms_handle)
  {
    UINT was_flag = GetMenuState((HMENU)itemMenu->save_ms_handle,Id,MF_BYCOMMAND) ;
    ModifyMenu((HMENU)itemMenu->save_ms_handle,Id,MF_BYCOMMAND|MF_STRING|was_flag,Id,label) ;
  }
}

char *wxMenuBar::GetLabel(int Id)
{
  wxMenu *itemMenu = NULL;
  wxMenuItem *item = FindItemForId(Id, &itemMenu) ;

  if (!item)
    return NULL;

  static char tmp[128] ;
  int len = 0;
  if (itemMenu->ms_handle)
  {
    len = GetMenuString((HMENU)itemMenu->ms_handle,Id,tmp,127,MF_BYCOMMAND) ;
  }
  else if (itemMenu->save_ms_handle)
  {
    len = GetMenuString((HMENU)itemMenu->save_ms_handle,Id,tmp,127,MF_BYCOMMAND) ;
  }

//  int len = GetMenuString((HMENU)ms_handle,Id,tmp,127,MF_BYCOMMAND) ;
  tmp[len] = '\0' ;
  char *p = copystring(tmp) ;
  return(p) ;
}

void wxMenuBar::SetLabelTop(int pos,char *label)
{
  UINT was_flag = GetMenuState((HMENU)ms_handle,pos,MF_BYPOSITION) ;
  if (was_flag&MF_POPUP)
  {
    was_flag &= 0xff ;
    HMENU popup = GetSubMenu((HMENU)ms_handle,pos) ;
    ModifyMenu((HMENU)ms_handle,pos,MF_BYPOSITION|MF_STRING|was_flag,(UINT)popup,label) ;
  }
  else
    ModifyMenu((HMENU)ms_handle,pos,MF_BYPOSITION|MF_STRING|was_flag,pos,label) ;
}

char *wxMenuBar::GetLabelTop(int pos)
{
  static char tmp[128] ;
  int len = GetMenuString((HMENU)ms_handle,pos,tmp,127,MF_BYPOSITION) ;
  tmp[len] = '\0' ;
  char *p = copystring(tmp) ;
  return(p) ;
}

void wxFrame::SetMenuBar(wxMenuBar *menu_bar)
{
  int i;
  HMENU menu = CreateMenu();
#if DEBUG
  wxDebugMsg("wxFrame::SetMenuBar: New menu bar HMENU is %d\n", menu);
#endif

  for (i = 0; i < menu_bar->n; i ++)
  {
    HMENU popup = (HMENU)menu_bar->menus[i]->ms_handle;
    //
    // After looking Bounds Checker result, it seems that all
    // menus must be individually destroyed. So, don't reset ms_handle,
    // to  allow ~wxMenu to do the job.
    //
    menu_bar->menus[i]->save_ms_handle = (HANDLE)popup;
    // Uncommenting for the moment... JACS
    menu_bar->menus[i]->ms_handle = NULL;
    AppendMenu(menu, MF_POPUP | MF_STRING, (UINT)popup, menu_bar->titles[i]);
  }

  menu_bar->ms_handle = (HANDLE)menu;
  if (wx_menu_bar)
    delete wx_menu_bar;

  wxWnd *cframe = (wxWnd *)handle;
  cframe->hMenu = menu;

  switch (frame_type)
  {
    case wxMDI_PARENT:
    {
      wxMDIFrame *mdi_frame = (wxMDIFrame *)cframe;
      HMENU subMenu = GetSubMenu(mdi_frame->window_menu, 0);
#if DEBUG
      wxDebugMsg("Window submenu is %d\n", subMenu);
#endif

      // Try to insert Window menu in front of Help, otherwise append it.
      int N = GetMenuItemCount(menu);
      Bool success = FALSE;
      for (i = 0; i < N; i++)
      {
        char buf[100];
        int chars = GetMenuString(menu, i, buf, 100, MF_BYPOSITION);
        if ((chars > 0) && (strcmp(buf, "&Help") == 0 ||
                            strcmp(buf, "Help") == 0))
        {
           success = TRUE;
           InsertMenu(menu, i, MF_BYPOSITION | MF_POPUP | MF_STRING,
                      (UINT)subMenu, "&Window");
           break;
        }
      }
      if (!success)
        AppendMenu(menu, MF_POPUP,
                         (UINT)subMenu,
                         "&Window");
      mdi_frame->parent_frame_active = TRUE;
#ifdef WIN32
      SendMessage(mdi_frame->client_hwnd, WM_MDISETMENU,
                  (WPARAM)menu,
                  (LPARAM)subMenu);
#else
      SendMessage(mdi_frame->client_hwnd, WM_MDISETMENU, 0,
                  MAKELPARAM(menu, subMenu));
#endif
      DrawMenuBar(mdi_frame->handle);
      break;
    }
    case wxMDI_CHILD:
    {
      wxMDIFrame *parent = (wxMDIFrame *)GetParent()->handle;
      parent->parent_frame_active = FALSE;
      HMENU subMenu = GetSubMenu(parent->window_menu, 0);
#if DEBUG
      wxDebugMsg("Window submenu is %d\n", subMenu);
#endif

      // Try to insert Window menu in front of Help, otherwise append it.
      int N = GetMenuItemCount(menu);
      Bool success = FALSE;
      for (i = 0; i < N; i++)
      {
        char buf[100];
        int chars = GetMenuString(menu, i, buf, 100, MF_BYPOSITION);
        if ((chars > 0) && (strcmp(buf, "&Help") == 0 ||
                            strcmp(buf, "Help") == 0))
        {
           success = TRUE;
           InsertMenu(menu, i, MF_BYPOSITION | MF_POPUP | MF_STRING,
                      (UINT)subMenu, "&Window");
           break;
        }
      }
      if (!success)
        AppendMenu(menu, MF_POPUP,
                         (UINT)subMenu,
                         "&Window");
#ifdef WIN32
      SendMessage(parent->client_hwnd, WM_MDISETMENU,
                  (WPARAM)menu,
                  (LPARAM)subMenu);
#else
      SendMessage(parent->client_hwnd, WM_MDISETMENU, 0,
                  MAKELPARAM(menu, subMenu));
#endif

      DrawMenuBar(parent->handle);
      break;
    }
    default:
    case wxSDI:
    {
      SetMenu(cframe->handle, menu);
      break;
    }
  }
  wx_menu_bar = menu_bar;
  menu_bar->menu_bar_frame = this;
}

// Single check box item
wxCheckBox::wxCheckBox(void)
{
  wxWinType = wxTYPE_HWND;
  windows_id = 0;
  ms_handle = 0;
  isFafa = CHECK_IS_FAFA ;
}

// Single check box item
wxCheckBox::wxCheckBox(wxPanel *panel, wxFunction func, char *Title,
                       int x, int y, int width, int height, long style, char *name):
  wxbCheckBox(panel, func, Title, x, y, width, height, style, name)
{
  Create(panel, func, Title, x, y, width, height, style, name);
}

wxCheckBox::wxCheckBox(wxPanel *panel, wxFunction func, wxBitmap *bitmap,
                       int x, int y, int width, int height, long style, char *name):
  wxbCheckBox(panel, func, bitmap, x, y, width, height, style, name)
{
  Create(panel, func, bitmap, x, y, width, height, style, name);
}

// Single check box item
Bool wxCheckBox::Create(wxPanel *panel, wxFunction func, char *Title,
                       int x, int y, int width, int height, long style, char *name)
{
  if (panel) panel->AddChild(this);
  buttonFont = panel->buttonFont ;
  labelFont = panel->labelFont ;
  backColour = panel->backColour ;
  labelColour = panel->labelColour ;
  buttonColour = panel->buttonColour ;
  wxWinType = wxTYPE_HWND;
  windowStyle = style;
  wxWnd *cparent = NULL;
  if (panel)
    cparent = (wxWnd *)panel->handle;

  panel->GetValidPosition(&x, &y);

  windows_id = (int)NewId();

  isFafa = CHECK_IS_FAFA ;
  checkWidth = -1 ;
  checkHeight = -1 ;
  HWND wx_button = CreateWindowEx(0, CHECK_CLASS, Title,
                    CHECK_FLAGS,
                    0, 0, 0, 0, cparent->handle, (HMENU)windows_id,
                    wxhInstance, NULL);
#if CTL3D
  Ctl3dSubclassCtl(wx_button);
#endif

  ms_handle = (HANDLE)wx_button;

  HDC the_dc = GetWindowDC((HWND)ms_handle) ;
  if (buttonFont && buttonFont->GetInternalFont(the_dc))
    SendMessage((HWND)ms_handle,WM_SETFONT,
                (WPARAM)buttonFont->GetInternalFont(the_dc),0L);
  ReleaseDC((HWND)ms_handle,the_dc) ;

  SetSize(x, y, width, height);

  ShowWindow(wx_button, SW_SHOW);
  panel->AdvanceCursor(this);
  Callback(func);
  return TRUE;
}

Bool wxCheckBox::Create(wxPanel *panel, wxFunction func, wxBitmap *bitmap,
                       int x, int y, int width, int height, long style, char *name)
{
  if (panel) panel->AddChild(this);
  buttonFont = panel->buttonFont ;
  labelFont = panel->labelFont ;
  backColour = panel->backColour ;
  labelColour = panel->labelColour ;
  buttonColour = panel->buttonColour ;
  wxWinType = wxTYPE_HWND;
  windowStyle = style;
  wxWnd *cparent = NULL;
  if (panel)
    cparent = (wxWnd *)panel->handle;

  panel->GetValidPosition(&x, &y);

  windows_id = (int)NewId();

#if FAFA_LIB && !CTL3D
  if (width<0)
    width = bitmap->GetWidth() ;
  if (height<0)
    height = bitmap->GetHeight() ;
  checkWidth = width ;
  checkHeight = height ;
  width += FB_MARGIN ;
  height += FB_MARGIN ;
  HWND wx_button = CreateWindowEx(0, "FafaCheck", "toggle",
                    BITCHECK_FLAGS,
                    0, 0, 0, 0, cparent->handle, (HMENU)windows_id,
                    wxhInstance, NULL);
      SendMessage((HWND)wx_button,WM_CHANGEBITMAP,
                  (WPARAM)((bitmap->GetHeight()<<8)+bitmap->GetWidth()),
                  (LPARAM)bitmap->ms_bitmap);
  isFafa = TRUE;
#else
  isFafa = CHECK_IS_FAFA;
  checkWidth = -1 ;
  checkHeight = -1 ;
  HWND wx_button = CreateWindowEx(0, CHECK_CLASS, "toggle",
                    CHECK_FLAGS,
                    0, 0, 0, 0, cparent->handle, (HMENU)windows_id,
                    wxhInstance, NULL);
#if CTL3D
  Ctl3dSubclassCtl(wx_button);
#endif
#endif

  ms_handle = (HANDLE)wx_button;

  HDC the_dc = GetWindowDC((HWND)ms_handle) ;
  if (buttonFont && buttonFont->GetInternalFont(the_dc))
    SendMessage((HWND)ms_handle,WM_SETFONT,
                (WPARAM)buttonFont->GetInternalFont(the_dc),0L);
  ReleaseDC((HWND)ms_handle,the_dc) ;

  SetSize(x, y, width, height);

  ShowWindow(wx_button, SW_SHOW);
  panel->AdvanceCursor(this);
  Callback(func);
  return TRUE;
}

wxCheckBox::~wxCheckBox(void)
{
}

void wxCheckBox::SetBackgroundColour(wxColour*col)
{
}

void wxCheckBox::SetLabelColour(wxColour*col) 
{
}

void wxCheckBox::SetButtonColour(wxColour*col)
{
}

void wxCheckBox::SetLabel(char *label)
{
#if FAFA_LIB && !CTL3D
    checkWidth = checkHeight = -1 ;
    // This message will switch from FB_BITMAP style to FB_TEXT, if needed.
    SendMessage((HWND)ms_handle,WM_CHANGEBITMAP,
                (WPARAM)0,
                (LPARAM)NULL);
#endif
  SetWindowText((HWND)ms_handle, label);
}

char *wxCheckBox::GetLabel()
{
  char buf[300];
  GetWindowText((HWND)ms_handle, buf, 300);
  return copystring(buf);
}

void wxCheckBox::SetLabel(wxBitmap *bitmap)
{
#if FAFA_LIB && !CTL3D
    checkWidth = bitmap->GetWidth() ;
    checkHeight = bitmap->GetHeight() ;
    SendMessage((HWND)ms_handle,WM_CHANGEBITMAP,
                (WPARAM)((bitmap->GetHeight()<<8)+bitmap->GetWidth()),
                (LPARAM)bitmap->ms_bitmap);
#endif
}

void wxCheckBox::SetSize(int x, int y, int width, int height)
{
  int currentX, currentY;
  GetPosition(&currentX, &currentY);
  if (x == -1)
    x = currentX;
  if (y == -1)
    y = currentY;

  char buf[300];

  float current_width;

  int cx;
  int cy;
  float cyf;

  HWND button = (HWND)ms_handle;
#if FAFA_LIB && !CTL3D
  if (checkWidth<0)
#endif
  {
    wxGetCharSize(button, &cx, &cy,buttonFont);

    GetWindowText(button, buf, 300);
    GetTextExtent(buf, &current_width, &cyf,NULL,NULL,buttonFont);
    if (width < 0)
      width = (int)(current_width + RADIO_SIZE) ;
    if (height<0)
      height = (int)(cyf) ;
  }
#if FAFA_LIB && !CTL3D
  else
  {
    if (width<0)
      width = checkWidth + FB_MARGIN ;
    if (height<0)
      height = checkHeight + FB_MARGIN ;
  }
#endif
  MoveWindow(button, x, y, width, height, TRUE);

  OnSize(width, height);
}


void wxCheckBox::SetValue(Bool val)
{
/*
// Following necessary for Win32s, because Win32s translate BM_SETCHECK
#if FAFA_LIB && !CTL3D
  SendMessage((HWND)ms_handle, FAFA_SETCHECK, val, 0);
#else
  SendMessage((HWND)ms_handle, BM_SETCHECK, val, 0);
#endif
*/
#if FAFA_LIB
  SendMessage((HWND)ms_handle, isFafa?FAFA_SETCHECK:BM_SETCHECK, val, 0);
#else
  SendMessage((HWND)ms_handle, BM_SETCHECK, val, 0);
#endif
}

Bool wxCheckBox::GetValue(void)
{
/*
// Following necessary for Win32s, because Win32s translate BM_SETCHECK
#if FAFA_LIB && !CTL3D
  return (Bool)(0x003 & SendMessage((HWND)ms_handle, FAFA_GETCHECK, 0, 0));
#else
  return (Bool)(0x003 & SendMessage((HWND)ms_handle, BM_GETCHECK, 0, 0));
#endif
*/
#if FAFA_LIB
  return (Bool)(0x003 & SendMessage((HWND)ms_handle,
                isFafa?FAFA_GETCHECK:BM_GETCHECK, 0, 0));
#else
  return (Bool)(0x003 & SendMessage((HWND)ms_handle, BM_GETCHECK, 0, 0));
#endif
}

wxChoice::wxChoice(void)
{
  no_strings = 0;
  wxWinType = wxTYPE_HWND;
  static_label = NULL;
  windows_id = 0;
  ms_handle = 0;
}

wxChoice::wxChoice(wxPanel *panel, wxFunction func, char *Title,
                   int x, int y, int width, int height, int N, char **Choices,
                   long style, char *name):
  wxbChoice(panel, func, Title, x, y, width, height, N, Choices, style, name)
{
  Create(panel, func, Title, x, y, width, height, N, Choices, style, name);
}

Bool wxChoice::Create(wxPanel *panel, wxFunction func, char *Title,
                   int x, int y, int width, int height, int N, char **Choices,
                   long style, char *name)
{
  if (panel) panel->AddChild(this);
  buttonFont = panel->buttonFont ;
  labelFont = panel->labelFont ;
  backColour = panel->backColour ;
  labelColour = panel->labelColour ;
  buttonColour = panel->buttonColour ;
  no_strings = N;

  wxWinType = wxTYPE_HWND;
  windowStyle = style;

  wxWnd *cparent = NULL;
  if (panel)
    cparent = (wxWnd *)(panel->handle);

  panel->GetValidPosition(&x, &y);

  char *the_label = NULL ;

  if (Title)
  {
    the_label = new char[strlen(Title)+1] ;
    if (style&wxFIXED_LENGTH)
      for (int i=0;i<(int)strlen(Title);i++)
        the_label[i]=MEANING_CHARACTER ;
    else
      strcpy(the_label,Title) ;
    the_label[strlen(Title)] = '\0' ;
  }

  if (Title)
  {
    static_label = CreateWindowEx(0, STATIC_CLASS, the_label,
                         STATIC_FLAGS,
                         0, 0, 0, 0, cparent->handle, (HMENU)NewId(),
                         wxhInstance, NULL);
#if CTL3D
    Ctl3dSubclassCtl(static_label);
#endif
    HDC the_dc = GetWindowDC(static_label) ;
    if (labelFont && labelFont->GetInternalFont(the_dc))
      SendMessage(static_label,WM_SETFONT,
                  (WPARAM)labelFont->GetInternalFont(the_dc),0L);
    ReleaseDC(static_label,the_dc) ;
  }
  else
    static_label = NULL;

  windows_id = (int)NewId();

  HWND wx_combo = CreateWindowEx(0, "COMBOBOX", NULL,
                   WS_CHILD | CBS_DROPDOWNLIST | WS_HSCROLL | WS_VSCROLL
                   | WS_BORDER | WS_TABSTOP | WS_VISIBLE,
                   0, 0, 0, 0, cparent->handle, (HMENU)windows_id,
                   wxhInstance, NULL);
#if CTL3D
  Ctl3dSubclassCtl(wx_combo);
#endif

  ms_handle = (HANDLE)wx_combo;

  HDC the_dc = GetWindowDC((HWND)ms_handle) ;
  if (panel->buttonFont && panel->buttonFont->GetInternalFont(the_dc))
    SendMessage((HWND)ms_handle,WM_SETFONT,
                (WPARAM)panel->buttonFont->GetInternalFont(the_dc),0L);
  ReleaseDC((HWND)ms_handle,the_dc) ;


  for (int i = 0; i < N; i++)
    SendMessage(wx_combo, CB_INSERTSTRING, i, (LONG)Choices[i]);
  SendMessage(wx_combo, CB_SETCURSEL, i, 0);

  SetSize(x, y, width, height);
  panel->AdvanceCursor(this);
  Callback(func);

  if (Title)
  {
    if (style&wxFIXED_LENGTH)
      SetLabel(Title) ;
    if (the_label)
      delete [] the_label ;
  }

  return TRUE;
}

wxChoice::~wxChoice(void)
{
  if (static_label)
    DestroyWindow(static_label);
  static_label = NULL;
}

void wxChoice::SetBackgroundColour(wxColour*col)
{
}

void wxChoice::SetLabelColour(wxColour*col) 
{
}

void wxChoice::SetButtonColour(wxColour*col)
{
}


void wxChoice::Append(char *Item)
{
  SendMessage((HWND)ms_handle, CB_ADDSTRING, 0, (LONG)Item);

  no_strings ++;
}

// Unfortunately, under XView it doesn't redisplay until user resizes
// window. Any suggestions folks?
void wxChoice::Clear(void)
{
  SendMessage((HWND)ms_handle, CB_RESETCONTENT, 0, 0);

  no_strings = 0;
}


int wxChoice::GetSelection(void)
{
  return (int)SendMessage((HWND)ms_handle, CB_GETCURSEL, 0, 0);
}

void wxChoice::SetSelection(int n)
{
  SendMessage((HWND)ms_handle, CB_SETCURSEL, n, 0);
}

int wxChoice::FindString(char *s)
{
 int pos = (int)SendMessage((HWND)ms_handle, CB_FINDSTRINGEXACT, -1, (LONG)s);
 if (pos == LB_ERR)
   return -1;
 else
   return pos;
}

char *wxChoice::GetString(int n)
{
  int len = (int)SendMessage((HWND)ms_handle, CB_GETLBTEXT, n, (long)wxBuffer);
  wxBuffer[len] = 0;
  return wxBuffer;
}

void wxChoice::SetSize(int x, int y, int width, int height)
{
// This flag is controlled by wx_setup.h
// ALS_CHOICE_SIZE is an experimental form of this method. Please, let this
// code here, when I've more time I'll attempt to merge both...
//(ALS_CHOICE_SIZE enables you to specify an height!=1, which is the
// displayed height)
#if !ALS_CHOICE_SIZE
  int currentX, currentY;
  GetPosition(&currentX, &currentY);
  if (x == -1)
    x = currentX;
  if (y == -1)
    y = currentY;

  char buf[300];

  int cx; // button font dimensions
  int cy;
  int clx; // label font dimensions
  int cly;
  wxGetCharSize((HWND)ms_handle, &cx, &cy, buttonFont);

  float label_width, label_height, label_x, label_y;
  float control_width, control_height, control_x, control_y;

  // Ignore height parameter because height doesn't
  // mean 'initially displayed' height, it refers to the
  // drop-down menu as well. The wxWindows interpretation
  // is different; also, getting the size returns the
  // _displayed_ size (NOT the drop down menu size)
  // so setting-getting-setting size would not work.
  height = -1;

  // Deal with default size (using -1 values)
  if (width <= 0)
  {
    // Find the longest string
    if (no_strings == 0)
      control_width = 100;
    else
    {
      float len, ht;
      float longest = 0.0;
      int i;
      for (i = 0; i < no_strings; i++)
      {
        char *s = GetString(i);
        GetTextExtent(s, &len, &ht, NULL, NULL,buttonFont);
        if ( len > longest) longest = len;
      }

      control_width = (float)(int)(longest + cx*5);
    }
  }

  // Choice drop-down list depends on number of items (limited to 10)
  if (height <= 0)
  {
    if (no_strings == 0)
      height = (int)(EDIT_CONTROL_FACTOR*cy*10.0);
    else height = (int)(EDIT_CONTROL_FACTOR*cy*(min(10, no_strings) + 1));
  }

  if (static_label)
  {
    // Find size of label
    wxGetCharSize((HWND)ms_handle, &clx, &cly,labelFont);
    GetWindowText(static_label, buf, 300);
    GetTextExtent(buf, &label_width, &label_height, NULL, NULL,labelFont);

    // Given size is total label + edit size, so find individual
    // control sizes on that basis.
    if (labelPosition == wxHORIZONTAL)
    {
      if (height<=0)
        height = (int)((max(cy,cly))*EDIT_CONTROL_FACTOR) ;

      label_x = (float)x;
      label_y = (float)y; // + (height - (int)label_height)/2; // Centre vertically
      label_width += (float)clx;

      control_x = label_x + label_width + clx;
      control_y = (float)y;
      if (width >= 0)
        control_width = width - (control_x - label_x);
      control_height = (float)height;
    }
    else // wxVERTICAL
    {
      label_x = (float)x;
      label_y = (float)y;

      control_x = (float)x;
      control_y = label_y + label_height + 3; // Allow for 3D border

      if (width >= 0)
        control_width = (float)width;

      if (height<=0)
        control_height = (float)(int)(cy*EDIT_CONTROL_FACTOR) ;
      else
        control_height = height - label_height - 3;
    }
    MoveWindow(static_label, (int)label_x, (int)label_y,
               (int)label_width, (int)label_height, TRUE);
  }
  else
  {
    control_x = (float)x;
    control_y = (float)y;
    if (width >= 0)
      control_width = (float)width;
    control_height = (float)height;
  }

  // Calculations may have made text size too small
  if (control_height <= 0)
    control_height = (float)(int)(cy*EDIT_CONTROL_FACTOR) ;

  if (control_width <= 0)
    control_width = 100;

  MoveWindow((HWND)ms_handle, (int)control_x, (int)control_y,
                              (int)control_width, (int)control_height, TRUE);

  OnSize(width, height);
#else
  int currentX, currentY;
  GetPosition(&currentX, &currentY);
  if (x == -1)
    x = currentX;
  if (y == -1)
    y = currentY;

  char buf[300];

  int y_offset = y;
  int x_offset = x;
  float current_width;

  int cx;
  int cy;
  float cyf;
  int label_width ;
  int label_height ;

  HWND wnd = (HWND)ms_handle;

  if (static_label)
  {
    wxGetCharSize(wnd, &cx, &cy,labelFont);
    GetWindowText(static_label, buf, 300);
    GetTextExtent(buf, &current_width, &cyf, NULL, NULL,labelFont);

    label_width = (int)(current_width+cx) ;
    label_height = (int)cyf ;
    MoveWindow(static_label,x_offset,y_offset,label_width,label_height,TRUE);
    if (labelPosition == wxVERTICAL)
      y_offset += cy;
    else
      x_offset += (int)(current_width + cx);
  }

  wxGetCharSize(wnd, &cx, &cy,buttonFont);
  if (width <= 0)
  {
    // Find the longest string
    if (no_strings == 0)
      width = 100;
    else
    {
      float len, ht;
      float longest = 0.0;
      int i;
      for (i = 0; i < no_strings; i++)
      {
        char *s = GetString(i);
        GetTextExtent(s, &len, &ht, NULL, NULL,buttonFont);
        if ( len > longest) longest = len;
      }

      width = (int)(longest + cx*5);
    }
  }
  // Choice drop-down list depends on number of items (limited to 10)
  // If height>0, this is the height of the combox box selection field!!
  // Be careful if you modify this part of code, I've spend a full day
  // to adjust all computations...
  // Major pb comes from height<=0, I've not found the REAL formula - Then
  // I let the original (empiric?) method, which gives us not so bad
  // results, after all.
  if (height <= 0)
  {
    if (no_strings == 0)
      height = cy*10;
    else height = (int)(cy*(min(10, no_strings) + 2)); // Beware: 2 is MULT
    MoveWindow(wnd, x_offset, y_offset, width, height, TRUE);
  }
  else
  {
    int hitem = height ;
    if (no_strings == 0)
      height = hitem*10+2 ;
    else
      height = hitem*min(10,no_strings) +2 ;              //Beware: 2 is ADDED
    MoveWindow(wnd, x_offset, y_offset, width, height, TRUE);
    SendMessage(wnd,CB_SETITEMHEIGHT,(WPARAM)-1,hitem) ;
  }

  if (static_label && labelPosition==wxHORIZONTAL)
  {
  // center label verticaly

  LRESULT choiceHeight=SendMessage(wnd,CB_GETITEMHEIGHT,(WPARAM)-1,(LPARAM)0) ;
    MoveWindow(static_label, (int)x,     (int)(y+(choiceHeight-cyf)/2),
                             label_width,label_height,
                             TRUE) ;
  }
  OnSize(width, height);
#endif
}

void wxChoice::GetSize(int *width, int *height)
{
  RECT rect;
  rect.left = -1; rect.right = -1; rect.top = -1; rect.bottom = -1;

  wxFindMaxSize((HWND)ms_handle, &rect);

  if (static_label)
  {
    wxFindMaxSize(static_label, &rect);
  }

  *width = rect.right - rect.left;
  *height = rect.bottom - rect.top;
}

void wxChoice::GetPosition(int *x, int *y)
{
  HWND wnd = (HWND)ms_handle;
  wxWindow *parent = GetParent();
  RECT rect;
  rect.left = -1; rect.right = -1; rect.top = -1; rect.bottom = -1;

  wxFindMaxSize(wnd, &rect);
  if (static_label)
    wxFindMaxSize(static_label, &rect);

  // Since we now have the absolute screen coords,
  // if there's a parent we must subtract its top left corner
  POINT point;
  point.x = rect.left;
  point.y = rect.top;
  if (parent)
  {
    wxWnd *cparent = (wxWnd *)(parent->handle);
    ::ScreenToClient(cparent->handle, &point);
  }

  *x = point.x;
  *y = point.y;
}

char *wxChoice::GetLabel(void)
{
  if (static_label)
  {
    GetWindowText(static_label, wxBuffer, 300);
    return wxBuffer;
  }
  else return NULL;
}

void wxChoice::SetLabel(char *label)
{
  if (static_label)
  {
    float w, h;
    RECT rect;

    wxWindow *parent = GetParent();
    GetWindowRect(static_label, &rect);

    // Since we now have the absolute screen coords,
    // if there's a parent we must subtract its top left corner
    POINT point;
    point.x = rect.left;
    point.y = rect.top;
    if (parent)
    {
      wxWnd *cparent = (wxWnd *)(parent->handle);
      ::ScreenToClient(cparent->handle, &point);
    }

    GetTextExtent(label, &w, &h, NULL, NULL,labelFont);
    MoveWindow(static_label, point.x, point.y, (int)(w + 10), (int)h,
               TRUE);
    SetWindowText(static_label, label);
  }
}


// Listbox item
wxListBox::wxListBox(void)
{
  wxWinType = wxTYPE_HWND;
  static_label = NULL;
  windows_id = 0;
  no_items = 0;
  ms_handle = 0;
}

wxListBox::wxListBox(wxPanel *panel, wxFunction func,
                       char *Title, Bool Multiple,
                       int x, int y, int width, int height,
                       int N, char **Choices, long style, char *name):
  wxbListBox(panel, func, Title, Multiple, x, y, width, height, N, Choices,
             style, name)
{
  Create(panel, func, Title, Multiple, x, y, width, height, N, Choices,
         style, name);
}

Bool wxListBox::Create(wxPanel *panel, wxFunction func,
                       char *Title, Bool Multiple,
                       int x, int y, int width, int height,
                       int N, char **Choices, long style, char *name)
{
  if (panel) panel->AddChild(this);
  buttonFont = panel->buttonFont ;
  labelFont = panel->labelFont ;
  backColour = panel->backColour ;
  labelColour = panel->labelColour ;
  buttonColour = panel->buttonColour ;
  multiple = Multiple & wxMULTIPLE_MASK;
  wxWinType = wxTYPE_HWND;
  windowStyle = style;
  wxWnd *cparent = NULL;
  if (panel)
    cparent = (wxWnd *)(panel->handle);

  panel->GetValidPosition(&x, &y);

  char *the_label = NULL ;

  if (Title)
  {
    the_label = new char[strlen(Title)+1] ;
    if (style&wxFIXED_LENGTH)
      for (int i=0;i<(int)strlen(Title);i++)
        the_label[i]=MEANING_CHARACTER ;
    else
      strcpy(the_label,Title) ;
    the_label[strlen(Title)] = '\0' ;
  }

  // If label exists, create a static control for it.
  if (Title)
  {
    static_label = CreateWindowEx(0, STATIC_CLASS, the_label,
                         STATIC_FLAGS,
                         0, 0, 0, 0, cparent->handle, (HMENU)NewId(),
                         wxhInstance, NULL);
#if CTL3D
    Ctl3dSubclassCtl(static_label);
#endif
    HDC the_dc = GetWindowDC(static_label) ;
    if (labelFont && labelFont->GetInternalFont(the_dc))
      SendMessage(static_label,WM_SETFONT,
                  (WPARAM)labelFont->GetInternalFont(the_dc),0L);
    ReleaseDC(static_label,the_dc) ;

  }
  else
    static_label = NULL;


  DWORD wstyle;
  if (multiple == wxMULTIPLE)
    wstyle = WS_VSCROLL | WS_BORDER | LBS_MULTIPLESEL | LBS_NOTIFY | WS_TABSTOP;
  else if (multiple == wxEXTENDED)
    wstyle = WS_VSCROLL | WS_BORDER | LBS_EXTENDEDSEL | LBS_NOTIFY | WS_TABSTOP ;
  else
    wstyle = WS_VSCROLL | WS_BORDER | LBS_NOTIFY | WS_TABSTOP;
  if ((Multiple&wxALWAYS_SB) || (style & wxALWAYS_SB))
    wstyle |= LBS_DISABLENOSCROLL ;
  if (style & wxHSCROLL)
    wstyle |= WS_HSCROLL;

  windows_id = (int)NewId();

  HWND wx_list = CreateWindowEx(0, "LISTBOX", NULL,
                         wstyle | WS_CHILD,
                         0, 0, 0, 0, cparent->handle, (HMENU)windows_id,
                         wxhInstance, NULL);
#if CTL3D
  Ctl3dSubclassCtl(wx_list);
#endif

  for (int i = 0; i < N; i++)
    SendMessage(wx_list, LB_ADDSTRING, 0, (LONG)Choices[i]);
  if (!Multiple)
    SendMessage(wx_list, LB_SETCURSEL, 0, 0);

  ShowWindow(wx_list, SW_SHOW);
  no_items = N;

  ms_handle = (HANDLE)wx_list;

  HDC the_dc = GetWindowDC((HWND)ms_handle) ;
  if (buttonFont && buttonFont->GetInternalFont(the_dc))
    SendMessage((HWND)ms_handle,WM_SETFONT,
                (WPARAM)buttonFont->GetInternalFont(the_dc),0L);
  ReleaseDC((HWND)ms_handle,the_dc) ;

  SetSize(x, y, width, height);
  panel->AdvanceCursor(this);
  Callback(func);

  if (Title)
  {
    if (style&wxFIXED_LENGTH)
      SetLabel(Title) ;
    if (the_label)
      delete [] the_label ;
  }

  return TRUE;
}

wxListBox::~wxListBox(void)
{
  if (selections)
    delete[] selections;
  if (static_label)
    DestroyWindow(static_label);
  static_label = NULL;
}

void wxListBox::SetBackgroundColour(wxColour*col)
{
}

void wxListBox::SetLabelColour(wxColour*col) 
{
}

void wxListBox::SetButtonColour(wxColour*col)
{
}

void wxListBox::SetFirstItem(int N)
{
  SendMessage((HWND)ms_handle,LB_SETTOPINDEX,(WPARAM)N,(LPARAM)0) ;
}

void wxListBox::SetFirstItem(char *s)
{
int N = FindString(s) ;

  if (N>=0)
    SetFirstItem(N) ;
}

void wxListBox::Delete(int N)
{
  SendMessage((HWND)ms_handle, LB_DELETESTRING, N, 0);
  no_items --;
  SetHorizontalExtent(NULL);
}

void wxListBox::Append(char *Item)
{
  SendMessage((HWND)ms_handle, LB_ADDSTRING, 0, (LONG)Item);
  no_items ++;
  SetHorizontalExtent(Item);
}

void wxListBox::Append(char *Item, char *Client_data)
{
  int index = (int)SendMessage((HWND)ms_handle, LB_ADDSTRING, 0, (LONG)Item);
  SendMessage((HWND)ms_handle, LB_SETITEMDATA, index, (LONG)Client_data);
  no_items ++;
  SetHorizontalExtent(Item);
}

void wxListBox::Set(int n, char *choices[])
{
  ShowWindow((HWND)ms_handle, SW_HIDE);
  for (int i = 0; i < n; i++)
    SendMessage((HWND)ms_handle, LB_ADDSTRING, 0, (LONG)choices[i]);
  no_items = n;
  SetHorizontalExtent(NULL);
  ShowWindow((HWND)ms_handle, SW_SHOW);
}

int wxListBox::FindString(char *s)
{
 int pos = (int)SendMessage((HWND)ms_handle, LB_FINDSTRINGEXACT, -1, (LONG)s);
 if (pos == LB_ERR)
   return -1;
 else
   return pos;
}

void wxListBox::Clear(void)
{
  SendMessage((HWND)ms_handle, LB_RESETCONTENT, 0, 0);

  no_items = 0;
  SendMessage((HWND)ms_handle, LB_SETHORIZONTALEXTENT, LOWORD(0), 0L);
}

void wxListBox::SetSelection(int N, Bool select)
{
  if (multiple != wxSINGLE)
    SendMessage((HWND)ms_handle, LB_SETSEL, select, N);
  else
  {
    if (!select) N = -N;
    SendMessage((HWND)ms_handle, LB_SETCURSEL, N, 0);
  }
}

Bool wxListBox::Selected(int N)
{
  return (Bool)SendMessage((HWND)ms_handle, LB_GETSEL, N, 0);
}

void wxListBox::Deselect(int N)
{
  if (multiple != wxSINGLE)
    SendMessage((HWND)ms_handle, LB_SETSEL, FALSE, N);
  else
    SendMessage((HWND)ms_handle, LB_SETCURSEL, -N, 0);
}

char *wxListBox::GetClientData(int N)
{
  return (char *)SendMessage((HWND)ms_handle, LB_GETITEMDATA, N, 0);
}

// Return number of selections and an array of selected integers
// Use selections field to store data, which will be cleaned up
// by destructor if necessary.
int wxListBox::GetSelections(int **list_selections)
{
  HWND listbox = (HWND)ms_handle;
  if (selections)
    { delete[] selections; selections = NULL; };
  if (multiple == wxSINGLE)
  {
    int sel = (int)SendMessage(listbox, LB_GETCURSEL, 0, 0);
    if (sel == LB_ERR)
      return 0;
    selections = new int[1];
    selections[0] = sel;
    *list_selections = selections;
    return 1;
  }
  else
  {
    int no_sel = (int)SendMessage(listbox, LB_GETSELCOUNT, 0, 0);
    if (no_sel == 0)
      return 0;
    selections = new int[no_sel];
    SendMessage(listbox, LB_GETSELITEMS, no_sel, (LONG)selections);
    *list_selections = selections;
    return no_sel;
  }
}

// Get single selection, for single choice list items
int wxListBox::GetSelection(void)
{
  HWND listbox = (HWND)ms_handle;
  if (selections)
    { delete[] selections; selections = NULL; };
  if (multiple == wxSINGLE)
  {
    int sel = (int)SendMessage(listbox, LB_GETCURSEL, 0, 0);
    if (sel == LB_ERR)
      return -1;
    else
    {
      return sel;
    }
  }
  else return -1;
}

// Find string for position
char *wxListBox::GetString(int N)
{
  int len = (int)SendMessage((HWND)ms_handle, LB_GETTEXT, N, (LONG)wxBuffer);
  wxBuffer[len] = 0;
  return wxBuffer;
}

void wxListBox::SetSize(int x, int y, int width, int height)
{
  int currentX, currentY;
  GetPosition(&currentX, &currentY);
  if (x == -1)
    x = currentX;
  if (y == -1)
    y = currentY;

  char buf[300];

  int cx; // button font dimensions
  int cy;
  int clx; // label font dimensions
  int cly;

  wxGetCharSize((HWND)ms_handle, &cx, &cy,buttonFont);

  float label_width, label_height, label_x, label_y;
  float control_width, control_height, control_x, control_y;

  // Deal with default size (using -1 values)
  if (width<=0)
    width = DEFAULT_ITEM_WIDTH;

  if (height<=0)
    height = DEFAULT_ITEM_HEIGHT;

  if (static_label)
  {
    // Find size of label
    wxGetCharSize((HWND)ms_handle, &clx, &cly,labelFont);
    GetWindowText(static_label, buf, 300);
    GetTextExtent(buf, &label_width, &label_height, NULL, NULL,labelFont);

    // Given size is total label + edit size, find individual
    // control sizes on that basis.
    if (labelPosition == wxHORIZONTAL)
    {
      label_x = (float)x;
      label_y = (float)y;
      label_width += (float)clx;

      control_x = label_x + label_width + clx;
      control_y = (float)y;
      control_width = width - (control_x - label_x);
      control_height = (float)height;
    }
    else // wxVERTICAL
    {
      label_x = (float)x;
      label_y = (float)y;

      control_x = (float)x;
      control_y = label_y + label_height + 3; // Allow for 3D border
      control_width = (float)width;
      control_height = height - label_height - 3;
    }

    MoveWindow(static_label, (int)label_x, (int)label_y,
               (int)label_width, (int)label_height, TRUE);
  }
  else
  {
    control_x = (float)x;
    control_y = (float)y;
    control_width = (float)width;
    control_height = (float)height;
  }

  // Calculations may have made size too small
  if (control_height <= 0)
    control_height = DEFAULT_ITEM_HEIGHT;

  if (control_width <= 0)
    control_width = DEFAULT_ITEM_WIDTH;

//  wxDebugMsg("About to set the listbox height to %d", (int)control_height);
  MoveWindow((HWND)ms_handle, (int)control_x, (int)control_y,
                              (int)control_width, (int)control_height, TRUE);

  OnSize(width, height);
}

void wxListBox::GetSize(int *width, int *height)
{
  RECT rect;
  rect.left = -1; rect.right = -1; rect.top = -1; rect.bottom = -1;

  wxFindMaxSize((HWND)ms_handle, &rect);

  if (static_label)
  {
    wxFindMaxSize(static_label, &rect);
  }

  *width = rect.right - rect.left;
  *height = rect.bottom - rect.top;
}

void wxListBox::GetPosition(int *x, int *y)
{
  wxWindow *parent = GetParent();
  RECT rect;
  rect.left = -1; rect.right = -1; rect.top = -1; rect.bottom = -1;

  wxFindMaxSize((HWND)ms_handle, &rect);
  if (static_label)
    wxFindMaxSize(static_label, &rect);

  // Since we now have the absolute screen coords,
  // if there's a parent we must subtract its top left corner
  POINT point;
  point.x = rect.left;
  point.y = rect.top;
  if (parent)
  {
    wxWnd *cparent = (wxWnd *)(parent->handle);
    ::ScreenToClient(cparent->handle, &point);
  }

  *x = point.x;
  *y = point.y;
}

char *wxListBox::GetLabel(void)
{
  if (static_label)
  {
    GetWindowText(static_label, wxBuffer, 300);
    return wxBuffer;
  }
  else return NULL;
}

void wxListBox::SetLabel(char *label)
{
  if (static_label)
  {
    float w, h;
    RECT rect;

    wxWindow *parent = GetParent();
    GetWindowRect(static_label, &rect);

    // Since we now have the absolute screen coords,
    // if there's a parent we must subtract its top left corner
    POINT point;
    point.x = rect.left;
    point.y = rect.top;
    if (parent)
    {
      wxWnd *cparent = (wxWnd *)(parent->handle);
      ::ScreenToClient(cparent->handle, &point);
    }

    GetTextExtent((LPSTR)label, &w, &h, NULL, NULL,labelFont);
    MoveWindow(static_label, point.x, point.y, (int)(w + 10), (int)h,
               TRUE);
    SetWindowText(static_label, label);
  }
}

// Windows-specific code to set the horizontal extent of
// the listbox, if necessary. If s is non-NULL, it's
// used to calculate the horizontal extent.
// Otherwise, all strings are used.
void wxListBox::SetHorizontalExtent(char *s)
{
  // Only necessary if we want a horizontal scrollbar
  if (!(windowStyle & wxHSCROLL))
    return;
  TEXTMETRIC lpTextMetric;

  HWND hWnd = GetHWND();

  if (s)
  {
    int existingExtent = (int)SendMessage(hWnd, LB_GETHORIZONTALEXTENT, 0, 0L);
    HDC dc = GetWindowDC(hWnd);
    GetTextMetrics(dc, &lpTextMetric);
    int extentX = (int)(::GetTextExtent(dc, (LPSTR)s, strlen(s)) + lpTextMetric.tmAveCharWidth);
    ReleaseDC(hWnd, dc);
    if (extentX > existingExtent)
      SendMessage(hWnd, LB_SETHORIZONTALEXTENT, LOWORD(extentX), 0L);
    return;
  }
  else
  {
    int largestExtent = 0;
    int existingExtent = (int)SendMessage(hWnd, LB_GETHORIZONTALEXTENT, 0, 0L);
    HDC dc = GetWindowDC(hWnd);
    GetTextMetrics(dc, &lpTextMetric);
    for (int i = 0; i < no_items; i++)
    {
      int len = (int)SendMessage(hWnd, LB_GETTEXT, i, (LONG)wxBuffer);
      wxBuffer[len] = 0;
      int extentX = (int)(::GetTextExtent(dc, (LPSTR)wxBuffer, len) + lpTextMetric.tmAveCharWidth);
      if (extentX > largestExtent)
        largestExtent = extentX;
    }
    ReleaseDC(hWnd, dc);
    SendMessage(hWnd, LB_SETHORIZONTALEXTENT, LOWORD(largestExtent), 0L);
  }
}

// Radio box item
wxRadioBox::wxRadioBox(void)
{
  wxWinType = wxTYPE_HWND;
  static_label = NULL;
  windows_id = 0;
  no_items = 0;
  ms_handle = 0;
  radioButtons = NULL;
  majorDim = 0 ;
  selected = -1;
  radioWidth = NULL ;
  radioHeight = NULL ;
  isFafa = RADIO_IS_FAFA ;
}

wxRadioBox::wxRadioBox(wxPanel *panel, wxFunction func,
                       char *Title,
                       int x, int y, int width, int height,
                       int N, char **Choices,
                       int MajorDim,long style, char *name):
  wxbRadioBox(panel, func, Title, x, y, width, height, N, Choices,
              MajorDim, style, name)
{
  Create(panel, func, Title, x, y, width, height, N, Choices, MajorDim, style, name);
}

/*
 * Causes problems for the Turbo C++ for Windows linker
 *
 */
#ifndef __BORLANDC__
wxRadioBox::wxRadioBox(wxPanel *panel, wxFunction func,
                       char *Title,
                       int x, int y, int width, int height,
                       int N, wxBitmap **Choices,
                       int MajorDim,long style, char *name):
  wxbRadioBox(panel, func, Title, x, y, width, height, N, Choices,
              MajorDim, style, name)
{
  Create(panel, func, Title, x, y, width, height, N, Choices, MajorDim, style, name);
}
#endif

Bool wxRadioBox::Create(wxPanel *panel, wxFunction func,
                       char *Title,
                       int x, int y, int width, int height,
                       int N, char **Choices,
                       int MajorDim, long style, char *name)
{
  if (panel) panel->AddChild(this);
  buttonFont = panel->buttonFont ;
  labelFont = panel->labelFont ;
  backColour = panel->backColour ;
  labelColour = panel->labelColour ;
  buttonColour = panel->buttonColour ;
  if (MajorDim==0)
    MajorDim = N ;
  majorDim = MajorDim ;
  windowStyle = style ;
  selected = -1;
  ms_handle = 0;
  isFafa = RADIO_IS_FAFA ;
  static_label = NULL;
  wxWinType = wxTYPE_HWND;
  wxWnd *cparent = NULL;
  if (panel)
    cparent = (wxWnd *)(panel->handle);

  panel->GetValidPosition(&x, &y);

  char *the_label = NULL ;

  if (Title)
  {
    the_label = new char[strlen(Title)+1] ;
    if (style&wxFIXED_LENGTH)
      for (int i=0;i<(int)strlen(Title);i++)
        the_label[i]=MEANING_CHARACTER ;
    else
      strcpy(the_label,Title) ;
    the_label[strlen(Title)] = '\0' ;
  }
  else the_label=copystring("") ;

  static_label = NULL;
  HWND the_handle ;

  ms_handle = CreateWindow(GROUP_CLASS,Title,
                           GROUP_FLAGS,
                           0,0,0,0,
                           cparent->handle,(HMENU)NewId(),wxhInstance,NULL) ;

  the_handle = cparent->handle ;

#if CTL3D
    Ctl3dSubclassCtl(ms_handle);
#endif
  HDC the_dc = GetWindowDC(ms_handle) ;
  if (labelFont && labelFont->GetInternalFont(the_dc))
    SendMessage(ms_handle,WM_SETFONT,
                (WPARAM)labelFont->GetInternalFont(the_dc),0L);
  ReleaseDC(ms_handle,the_dc) ;

  the_handle = cparent->handle;

  // Some radio boxes test consecutive id.
  (void)NewId() ;
  radioButtons = new HWND[N];
  radioWidth = new int[N] ;
  radioHeight = new int[N] ;
  for (int i = 0; i < N; i++)
  {
    radioWidth[i] = radioHeight[i] = -1 ;
    long groupStyle = 0;
    if (i == 0 && style==0)
      groupStyle = WS_GROUP;
    long newId = NewId();
    radioButtons[i] = CreateWindowEx(0, RADIO_CLASS, Choices[i], 
                          groupStyle | RADIO_FLAGS,0,0,0,0,
                          the_handle, (HMENU)newId, wxhInstance, NULL);
#if CTL3D
    Ctl3dSubclassCtl(radioButtons[i]);
#endif
    HDC the_dc = GetWindowDC((HWND)radioButtons[i]) ;
    if (buttonFont && buttonFont->GetInternalFont(the_dc))
      SendMessage((HWND)radioButtons[i],WM_SETFONT,
                  (WPARAM)buttonFont->GetInternalFont(the_dc),0L);
    ReleaseDC((HWND)radioButtons[i],the_dc) ;
    subControls.Append((wxObject *)newId);
  }
  (void)NewId() ;

  no_items = N;
  SetSelection(0);

  SetSize(x, y, width, height);
  panel->AdvanceCursor(this);
  Callback(func);

  if (Title)
  {
    if (style&wxFIXED_LENGTH)
      SetLabel(Title) ;
  }
  if (the_label)
    delete [] the_label ;

  return TRUE;
}

#ifndef __BORLANDC__
Bool wxRadioBox::Create(wxPanel *panel, wxFunction func,
                       char *Title,
                       int x, int y, int width, int height,
                       int N, wxBitmap **Choices,
                       int MajorDim, long style, char *name)
{
  if (panel) panel->AddChild(this);
  buttonFont = panel->buttonFont ;
  labelFont = panel->labelFont ;
  backColour = panel->backColour ;
  labelColour = panel->labelColour ;
  buttonColour = panel->buttonColour ;
  if (MajorDim==0)
    MajorDim = N ;
  majorDim = MajorDim ;
  windowStyle = style ;
  selected = -1;
  ms_handle = 0;
  wxWinType = wxTYPE_HWND;
  wxWnd *cparent = NULL;
  if (panel)
    cparent = (wxWnd *)(panel->handle);

  panel->GetValidPosition(&x, &y);

  char *the_label = NULL ;

  if (Title)
  {
    the_label = new char[strlen(Title)+1] ;
    if (style&wxFIXED_LENGTH)
      for (int i=0;i<(int)strlen(Title);i++)
        the_label[i]=MEANING_CHARACTER ;
    else
      strcpy(the_label,Title) ;
    the_label[strlen(Title)] = '\0' ;
  }
  else
    the_label = copystring("") ;

  static_label = NULL;
  HWND the_handle ;

  ms_handle = CreateWindow(GROUP_CLASS,the_label,
                           GROUP_FLAGS,
                           0,0,0,0,
                           cparent->handle,(HMENU)NewId(),wxhInstance,NULL) ;

  HDC the_dc = GetWindowDC(ms_handle) ;
  if (labelFont && labelFont->GetInternalFont(the_dc))
    SendMessage(ms_handle,WM_SETFONT,
                (WPARAM)labelFont->GetInternalFont(the_dc),0L);
  ReleaseDC(ms_handle,the_dc) ;
  the_handle = cparent->handle;

  (void)NewId() ;
  radioButtons = new HWND[N];
  radioWidth = new int[N] ;
  radioHeight = new int[N] ;
#if FAFA_LIB && !CTL3D
  isFafa = TRUE ; //always !!
#else
  isFafa = RADIO_IS_FAFA ;
#endif
  for (int i = 0; i < N; i++)
  {
    long groupStyle = 0;
    if (i == 0 && style==0)
      groupStyle = WS_GROUP;
    long newId = NewId();
#if FAFA_LIB && !CTL3D
    radioWidth[i]  = Choices[i]->GetWidth()  + FB_MARGIN ;
    radioHeight[i] = Choices[i]->GetHeight() + FB_MARGIN ;
#else
    radioWidth[i]  = Choices[i]->GetWidth();
    radioHeight[i] = Choices[i]->GetHeight();
#endif
    char tmp[32] ;
    sprintf(tmp,"Toggle%d",i) ;
#if FAFA_LIB && !CTL3D
    radioButtons[i] = CreateWindowEx(0, "FafaCheck", tmp,
                          groupStyle | BITRADIO_FLAGS,0,0,0,0,
                          the_handle, (HMENU)newId, wxhInstance, NULL);
    SendMessage((HWND)radioButtons[i],WM_CHANGEBITMAP,
                  (WPARAM)((Choices[i]->GetHeight()<<8)+Choices[i]->GetWidth()),
                  (LPARAM)Choices[i]->ms_bitmap);

#else
    radioButtons[i] = CreateWindowEx(0, RADIO_CLASS, tmp,
                          groupStyle | RADIO_FLAGS,0,0,0,0,
                          the_handle, (HMENU)newId, wxhInstance, NULL);
#if CTL3D
    Ctl3dSubclassCtl(radioButtons[i]);
#endif
#endif
    HDC the_dc = GetWindowDC((HWND)radioButtons[i]) ;
    if (buttonFont && buttonFont->GetInternalFont(the_dc))
      SendMessage((HWND)radioButtons[i],WM_SETFONT,
                  (WPARAM)buttonFont->GetInternalFont(the_dc),0L);
    ReleaseDC((HWND)radioButtons[i],the_dc) ;
    subControls.Append((wxObject *)newId);
  }
  (void)NewId() ;
  no_items = N;
  SetSelection(0);

  SetSize(x, y, width, height);
  panel->AdvanceCursor(this);
  Callback(func);

  if (Title)
  {
    if (style&wxFIXED_LENGTH)
      SetLabel(Title) ;
  }
  if (the_label)
    delete [] the_label ;

  return TRUE;
}
#endif

wxRadioBox::~wxRadioBox(void)
{
  if (static_label)
    DestroyWindow(static_label);
  static_label = NULL;
  if (radioButtons)
  {
    for (int i = 0; i < no_items; i++)
      DestroyWindow(radioButtons[i]);
    delete[] radioButtons;
  }
  if (radioWidth)
    delete[] radioWidth ;
  if (radioHeight)
    delete[] radioHeight ;
  if (ms_handle)
    DestroyWindow(ms_handle) ;
  ms_handle = NULL ;

}

void wxRadioBox::SetBackgroundColour(wxColour*col)
{
}

void wxRadioBox::SetLabelColour(wxColour*col) 
{
}

void wxRadioBox::SetButtonColour(wxColour*col)
{
}

char *wxRadioBox::GetLabel(int item)
{
  char buf[300];
  GetWindowText((HWND)radioButtons[item], buf, 300);
  return copystring(buf);
}

void wxRadioBox::SetLabel(int item,char *label)
{
#if FAFA_LIB && !CTL3D
    // This message will switch from FB_BITMAP style to FB_TEXT, if needed.
    SendMessage((HWND)radioButtons[item],WM_CHANGEBITMAP,
                (WPARAM)0,
                (LPARAM)NULL);
#endif
  radioWidth[item] = radioHeight[item] = -1 ;
  SetWindowText((HWND)radioButtons[item], label);
}

void wxRadioBox::SetLabel(int item,wxBitmap *bitmap)
{
#if FAFA_LIB && !CTL3D
    SendMessage((HWND)radioButtons[item],WM_CHANGEBITMAP,
                (WPARAM)((bitmap->GetHeight()<<8)+bitmap->GetWidth()),
                (LPARAM)bitmap->ms_bitmap);
    radioWidth[item] = bitmap->GetWidth() + FB_MARGIN ;
    radioHeight[item] = bitmap->GetHeight() + FB_MARGIN ;
#endif
}

int wxRadioBox::FindString(char *s)
{
 for (int i = 0; i < no_items; i++)
 {
  GetWindowText(radioButtons[i], wxBuffer, 1000);
  if (strcmp(wxBuffer, s) == 0)
    return i;
 }
 return -1;
}

void wxRadioBox::SetSelection(int N)
{
  if ((N < 0) || (N >= no_items))
    return;

// Following necessary for Win32s, because Win32s translate BM_SETCHECK
#if FAFA_LIB
  if (selected >= 0 && selected < no_items)
    SendMessage(radioButtons[selected], isFafa?FAFA_SETCHECK:BM_SETCHECK, 0, 0L);
    
  SendMessage(radioButtons[N], isFafa?FAFA_SETCHECK:BM_SETCHECK, 1, 0L);
#else
  if (selected >= 0 && selected < no_items)
    SendMessage(radioButtons[selected], BM_SETCHECK, 0, 0L);
  
  SendMessage(radioButtons[N], BM_SETCHECK, 1, 0L);
#endif
  selected = N;
}

// Get single selection, for single choice list items
int wxRadioBox::GetSelection(void)
{
  return selected;
}

// Find string for position
char *wxRadioBox::GetString(int N)
{
  GetWindowText(radioButtons[N], wxBuffer, 1000);
  return wxBuffer;
}

void wxRadioBox::SetSize(int x, int y, int width, int height)
{
  int currentX, currentY;
  GetPosition(&currentX, &currentY);
  if (x == -1)
    x = currentX;
  if (y == -1)
    y = currentY;

  char buf[400];

  int y_offset = y;
  int x_offset = x;
  float current_width;

  float cyf;
  HWND wnd = (HWND)ms_handle;

  int cx1,cy1 ;
  wxGetCharSize(wnd, &cx1, &cy1,buttonFont);
  // Attempt to have a look coherent with other platforms:
  // We compute the biggest toggle dim, then we align all
  // items according this value.
  int maxWidth =  -1;
  int maxHeight = -1 ;

  for (int i = 0 ; i < no_items; i++)
  {
    int eachWidth;
    int eachHeight ;
    if (radioWidth[i]<0)
    {
      // It's a labelled toggle
      GetWindowText(radioButtons[i], buf, 300);
      GetTextExtent(buf, &current_width, &cyf,NULL,NULL, buttonFont);
      eachWidth = (int)(current_width + RADIO_SIZE);
      eachHeight = (int)((3*cyf)/2);
    }
    else
    {
      eachWidth = radioWidth[i] ;
      eachHeight = radioHeight[i] ;
    }
    if (maxWidth<eachWidth) maxWidth = eachWidth ;
    if (maxHeight<eachHeight) maxHeight = eachHeight ;
  }

  if (ms_handle)
  {
    int totWidth ;
    int totHeight;

    int nbHor,nbVer;

    if (labelPosition==wxVERTICAL)
    {
      nbVer = majorDim ;
      nbHor = (no_items+majorDim-1)/majorDim ;
    }
    else
    {
      nbHor = majorDim ;
      nbVer = (no_items+majorDim-1)/majorDim ;
    }

    // this formula works, but I don't know why.
    // Please, be sure what you do if you modify it!!
    if (radioWidth[0]<0)
      totHeight = (nbVer * maxHeight) + cy1/2 ;
    else
      totHeight = nbVer * (maxHeight+cy1/2) ;
    totWidth  = nbHor * (maxWidth+cx1) ;

#if (!CTL3D)
    // Requires a bigger group box in plain Windows
    MoveWindow(ms_handle,x_offset,y_offset,totWidth+cx1,totHeight+(3*cy1)/2,TRUE) ;
#else
    MoveWindow(ms_handle,x_offset,y_offset,totWidth+cx1,totHeight+cy1,TRUE) ;
#endif
    x_offset += cx1;
    y_offset += cy1;
  }

#if (!CTL3D)
  y_offset += (int)(cy1/2); // Fudge factor since buttons overlapped label
                            // JACS 2/12/93. CTL3D draws group label quite high.
#endif
  int startX = x_offset ;
  int startY = y_offset ;

  for ( i = 0 ; i < no_items; i++)
  {
    // Bidimensional radio adjustment
    if (i&&((i%majorDim)==0)) // Why is this omitted for i = 0?
    {
      if (labelPosition==wxVERTICAL)
      {
        y_offset = startY;
        x_offset += maxWidth + cx1 ;
      }
      else
      {
        x_offset = startX ;
        y_offset += maxHeight ;
        if (radioWidth[0]>0)
          y_offset += cy1/2 ;
      }
    }
    int eachWidth ;
    int eachHeight ;
    if (radioWidth[i]<0)
    {
      // It's a labeled item
      GetWindowText(radioButtons[i], buf, 300);
      GetTextExtent(buf, &current_width, &cyf,NULL,NULL,buttonFont);

      // How do we find out radio button bitmap size!!
      // By adjusting them carefully, manually :-)
      eachWidth = (int)(current_width + RADIO_SIZE);
      eachHeight = (int)((3*cyf)/2);
    }
    else
    {
      eachWidth = radioWidth[i] ;
      eachHeight = radioHeight[i] ;
    }

    MoveWindow(radioButtons[i],x_offset,y_offset,eachWidth,eachHeight,TRUE);
    if (labelPosition == wxVERTICAL)
    {
      y_offset += maxHeight;
      if (radioWidth[0]>0)
        y_offset += cy1/2 ;
    }
    else
      x_offset += maxWidth + cx1;
  }
  OnSize(width, height);
}

void wxRadioBox::GetSize(int *width, int *height)
{
  RECT rect;
  rect.left = -1; rect.right = -1; rect.top = -1; rect.bottom = -1;

  if (static_label)
    wxFindMaxSize(static_label, &rect);

  if (ms_handle)
    wxFindMaxSize(ms_handle, &rect);

  for (int i = 0; i < no_items; i++)
    wxFindMaxSize(radioButtons[i], &rect);

  *width = rect.right - rect.left;
  *height = rect.bottom - rect.top;
}

void wxRadioBox::GetPosition(int *x, int *y)
{
  wxWindow *parent = GetParent();
  RECT rect;
  rect.left = -1; rect.right = -1; rect.top = -1; rect.bottom = -1;

  for (int i = 0; i < no_items; i++)
    wxFindMaxSize(radioButtons[i], &rect);

  if (static_label)
    wxFindMaxSize(static_label, &rect);

  if (ms_handle)
    wxFindMaxSize(ms_handle, &rect);

  // Since we now have the absolute screen coords,
  // if there's a parent we must subtract its top left corner
  POINT point;
  point.x = rect.left;
  point.y = rect.top;
  if (parent)
  {
    wxWnd *cparent = (wxWnd *)(parent->handle);
    ::ScreenToClient(cparent->handle, &point);
  }

  *x = point.x;
  *y = point.y;
}

char *wxRadioBox::GetLabel(void)
{
  if (static_label)
  {
    GetWindowText(static_label, wxBuffer, 300);
    return wxBuffer;
  }
  else return NULL;
}

void wxRadioBox::SetLabel(char *label)
{
  if (static_label)
  {
    float w, h;
    RECT rect;

    wxWindow *parent = GetParent();
    GetWindowRect(static_label, &rect);

    // Since we now have the absolute screen coords,
    // if there's a parent we must subtract its top left corner
    POINT point;
    point.x = rect.left;
    point.y = rect.top;
    if (parent)
    {
      wxWnd *cparent = (wxWnd *)(parent->handle);
      ::ScreenToClient(cparent->handle, &point);
    }

    GetTextExtent((LPSTR)label, &w, &h, NULL, NULL,labelFont);
    MoveWindow(static_label, point.x, point.y, (int)(w + 10), (int)h,
               TRUE);
    SetWindowText(static_label, label);
  }
}

void wxRadioBox::SetFocus(void)
{
  if (no_items > 0)
   ::SetFocus(radioButtons[0]);
}

void wxRadioBox::Show(Bool show)
{
  int cshow;
  if (show)
    cshow = SW_SHOW;
  else
    cshow = SW_HIDE;
  if (static_label)
    ShowWindow(static_label, cshow);
  for (int i = 0; i < no_items; i++)
    ShowWindow(radioButtons[i], cshow);
}

// Enable a specific button
void wxRadioBox::Enable(int item, Bool enable)
{
  if (item<0)
    wxWindow::Enable(enable) ;
  else if (item < no_items)
    ::EnableWindow(radioButtons[item], enable);
}

// Show a specific button
void wxRadioBox::Show(int item, Bool show)
{
  if (item<0)
    wxRadioBox::Show(show) ;
  else if (item < no_items)
  {
    int cshow;
    if (show)
      cshow = SW_SHOW;
    else
      cshow = SW_HIDE;
    ShowWindow(radioButtons[item], cshow);
  }
}

// Message
wxMessage::wxMessage(void)
{
  wxWinType = wxTYPE_HWND;
  ms_handle = 0;
}

wxMessage::wxMessage(wxPanel *panel, char *label, int x, int y, long style, char *name):
  wxbMessage(panel, label, x, y, style, name)
{
  Create(panel, label, x, y, style, name);
}
  
Bool wxMessage::Create(wxPanel *panel, char *label, int x, int y, long style, char *name)
{
  if (panel) panel->AddChild(this);
  buttonFont = panel->buttonFont ;
  labelFont = panel->labelFont ;
  backColour = panel->backColour ;
  labelColour = panel->labelColour ;
  buttonColour = panel->buttonColour ;
  wxWinType = wxTYPE_HWND;
  windowStyle = style;
  wxWnd *cparent = NULL;
  if (panel)
    cparent = (wxWnd *)(panel->handle);

  HWND static_item = CreateWindowEx(0, STATIC_CLASS, label,
                         STATIC_FLAGS,
                         0, 0, 0, 0, cparent->handle, (HMENU)NewId(),
                         wxhInstance, NULL);
#if CTL3D
  Ctl3dSubclassCtl(static_item);
#endif

  ms_handle = (HANDLE)static_item;

  HDC the_dc = GetWindowDC((HWND)ms_handle) ;
  if (labelFont && labelFont->GetInternalFont(the_dc))
    SendMessage((HWND)ms_handle,WM_SETFONT,
                (WPARAM)labelFont->GetInternalFont(the_dc),0L);
  ReleaseDC((HWND)ms_handle,the_dc) ;

  panel->GetValidPosition(&x, &y);

  SetSize(x, y, -1, -1);
  panel->AdvanceCursor(this);
  return TRUE;
}

wxMessage::~wxMessage(void)
{
}

void wxMessage::SetBackgroundColour(wxColour*col)
{
}

void wxMessage::SetLabelColour(wxColour*col) 
{
}

void wxMessage::SetButtonColour(wxColour*col)
{
}

void wxMessage::SetSize(int x, int y, int width, int height)
{
  int currentX, currentY;
  GetPosition(&currentX, &currentY);
  if (x == -1)
    x = currentX;
  if (y == -1)
    y = currentY;

  char buf[300];

  float current_width;

  int cx;
  int cy;
  float cyf;

  wxGetCharSize((HWND)ms_handle, &cx, &cy,labelFont);

  GetWindowText((HWND)ms_handle, buf, 300);
  GetTextExtent(buf, &current_width, &cyf, NULL, NULL,labelFont);
  if (width < 0)
    width = (int)(current_width + cx) ;
  if (height<0)
    height = (int)(cyf) ;
  MoveWindow((HWND)ms_handle, x, y, width, height, TRUE);

  OnSize(width, height);
}

void wxMessage::SetLabel(char *label)
{
  float w, h;
  RECT rect;

  wxWindow *parent = GetParent();
  GetWindowRect((HWND)ms_handle, &rect);

  // Since we now have the absolute screen coords,
  // if there's a parent we must subtract its top left corner
  POINT point;
  point.x = rect.left;
  point.y = rect.top;
  if (parent)
  {
    wxWnd *cparent = (wxWnd *)(parent->handle);
    ::ScreenToClient(cparent->handle, &point);
  }

  GetTextExtent(label, &w, &h, NULL, NULL, labelFont);
  MoveWindow((HWND)ms_handle, point.x, point.y, (int)(w + 10), (int)h,
             TRUE);
  SetWindowText((HWND)ms_handle, label);
}

// Text item
wxText::wxText(void)
{
  wxWinType = wxTYPE_HWND;
  static_label = NULL;
  windows_id = 0;
  ms_handle = 0;
  globalHandle = 0;
}

wxText::wxText(wxPanel *panel, wxFunction Function, char *label, char *value,
               int x, int y, int width, int height, long style, char *name):
  wxbText(panel, Function, label, value, x, y, width, height, style, name)
{
  Create(panel, Function, label, value, x, y, width, height, style, name);
}
  
Bool wxText::Create(wxPanel *panel, wxFunction Function, char *label, char *value,
               int x, int y, int width, int height, long style, char *name)
{
  if (panel) panel->AddChild(this);
  buttonFont = panel->buttonFont ;
  labelFont = panel->labelFont ;
  backColour = panel->backColour ;
  labelColour = panel->labelColour ;
  buttonColour = panel->buttonColour ;
  wxWinType = wxTYPE_HWND;
  windowStyle = style;
  wxWnd *cparent = NULL;
  if (panel)
    cparent = (wxWnd *)(panel->handle);

  panel->GetValidPosition(&x, &y);

  char *the_label = NULL ;

  if (label)
  {
    the_label = new char[strlen(label)+1] ;
    if (style&wxFIXED_LENGTH)
      for (int i=0;i<(int)strlen(label);i++)
        the_label[i]=MEANING_CHARACTER ;
    else
      strcpy(the_label,label) ;
    the_label[strlen(label)] = '\0' ;
  }

  // If label exists, create a static control for it.
  if (label)
  {
    static_label = CreateWindowEx(0, STATIC_CLASS, the_label,
                         STATIC_FLAGS,
                         0, 0, 0, 0, cparent->handle, (HMENU)NewId(),
                         wxhInstance, NULL);
#if CTL3D
    Ctl3dSubclassCtl(static_label);
#endif
    HDC the_dc = GetWindowDC(static_label) ;
    if (labelFont && labelFont->GetInternalFont(the_dc))
      SendMessage(static_label,WM_SETFONT,
                  (WPARAM)labelFont->GetInternalFont(the_dc),0L);
    ReleaseDC(static_label,the_dc) ;
  }
  else
    static_label = NULL;

  windows_id = (int)NewId();

  // Obscure method from the MS Developer's Network Disk for
  // using global memory instead of the local heap, which
  // runs out far too soon. Solves the problem with
  // failing to appear.
  globalHandle=GlobalAlloc(GMEM_MOVEABLE | GMEM_ZEROINIT,
                         256L);

  HWND edit = CreateWindowEx(0, "EDIT", NULL,
                        ES_AUTOHSCROLL | ES_LEFT | WS_BORDER |
                        WS_VISIBLE | WS_CHILD | WS_TABSTOP,
                        0, 0, 0, 0, cparent->handle, (HMENU)windows_id,
                        globalHandle, NULL);
#if CTL3D
  Ctl3dSubclassCtl(edit);
#endif
/* The following code subclasses the EDIT control (again! -- CTL3D may
 * already have done it) to intercept the ENTER key, which only
 * works if the style wxPROCESS_ENTER has been used.
 */
  ms_handle = (HANDLE)edit;
  
  wxAddControlHandle(edit, this);
  oldWndProc = (FARPROC) GetWindowLong(edit, GWL_WNDPROC);
  if (!wxEditControlSubClassProc)
    wxEditControlSubClassProc = MakeProcInstance((FARPROC) wxSubclassedControlProc, wxhInstance);
  SetWindowLong(edit, GWL_WNDPROC, (LONG) wxEditControlSubClassProc);

  HDC the_dc = GetWindowDC((HWND)ms_handle) ;
  if (buttonFont && buttonFont->GetInternalFont(the_dc))
    SendMessage((HWND)ms_handle,WM_SETFONT,
                (WPARAM)buttonFont->GetInternalFont(the_dc),0L);
  ReleaseDC((HWND)ms_handle,the_dc) ;

  if ((width < 0) && value && strlen(value))
  {
  char *the_value ;
    the_value = new char[strlen(value)+1] ;
    if (style&wxFIXED_LENGTH)
      for (int i=0;i<(int)strlen(value);i++)
        the_value[i]=MEANING_CHARACTER ;
    else
      strcpy(the_value,value) ;
    the_value[strlen(value)] = '\0' ;

    int cx,cy ;
    wxGetCharSize((HWND)ms_handle,&cx,&cy,buttonFont) ;
    float current_width,cyf ;
    GetTextExtent(the_value,&current_width,&cyf, NULL, NULL,buttonFont) ;
    width = (int)current_width + 2*cx ;
    if (label && strlen(label) && labelPosition==wxHORIZONTAL)
    {
      GetTextExtent(the_label,&current_width,&cyf, NULL, NULL,labelFont) ;
      width += (int)current_width + cx ;
    }
    delete [] the_value ;
  }

  SetSize(x, y, width, height);

  if (value)
    SetWindowText(edit, value);

  panel->AdvanceCursor(this);
  Callback(Function);

  if (label)
  {
    if (style&wxFIXED_LENGTH)
      SetLabel(label) ;
    if (the_label)
      delete [] the_label ;
  }

  return TRUE;
}

wxText::~wxText(void)
{
  if (static_label)
    DestroyWindow(static_label);
}

void wxText::SetBackgroundColour(wxColour*col)
{
}

void wxText::SetLabelColour(wxColour*col) 
{
}

void wxText::SetButtonColour(wxColour*col)
{
}

char *wxText::GetLabel(void)
{
  if (static_label)
  {
    GetWindowText(static_label, wxBuffer, 300);
    return wxBuffer;
  }
  else return NULL;
}

void wxText::SetLabel(char *label)
{
  if (static_label)
  {
    float w, h;
    RECT rect;

    wxWindow *parent = GetParent();
    GetWindowRect(static_label, &rect);

    // Since we now have the absolute screen coords,
    // if there's a parent we must subtract its top left corner
    POINT point;
    point.x = rect.left;
    point.y = rect.top;
    if (parent)
    {
      wxWnd *cparent = (wxWnd *)(parent->handle);
      ::ScreenToClient(cparent->handle, &point);
    }

    GetTextExtent(label, &w, &h, NULL, NULL,labelFont);
    MoveWindow(static_label, point.x, point.y, (int)(w + 10), (int)h,
               TRUE);
    SetWindowText(static_label, label);
  }
}

char *wxText::GetValue(void)
{
/* Causes problems with Borland C++
  *(LPINT)wxBuffer = 500;
  int n = (int)SendMessage((HWND)ms_handle, EM_GETLINE, 0, (LONG)wxBuffer);
  wxBuffer[n] = 0;
  return wxBuffer;
*/
  GetWindowText((HWND)ms_handle, wxBuffer, 500);
  return wxBuffer;
}

void wxText::SetValue(char *value)
{
  if (!value)
    value = "";
    
  // If newlines are denoted by just 10, must stick 13 in front.
  int singletons = 0;
  int len = strlen(value);
  int i;
  for (i = 0; i < len; i ++)
  {
    if ((i > 0) && (value[i] == 10) && (value[i-1] != 13))
      singletons ++;
  }
  if (singletons > 0)
  {
    char *tmp = new char[len + singletons + 1];
    int j = 0;
    for (i = 0; i < len; i ++)
    {
      if ((i > 0) && (value[i] == 10) && (value[i-1] != 13))
      {
        tmp[j] = 13;
        j ++;
      }
      tmp[j] = value[i];
      j ++;
    }
    tmp[j] = 0;
    SetWindowText((HWND)ms_handle, tmp);
    delete[] tmp;
  }
  else
    SetWindowText((HWND)ms_handle, value);
}

void wxText::GetSize(int *width, int *height)
{
  RECT rect;
  rect.left = -1; rect.right = -1; rect.top = -1; rect.bottom = -1;

  wxFindMaxSize((HWND)ms_handle, &rect);

  if (static_label)
  {
    wxFindMaxSize(static_label, &rect);
  }

  *width = rect.right - rect.left;
  *height = rect.bottom - rect.top;
}

void wxText::GetPosition(int *x, int *y)
{
  wxWindow *parent = GetParent();
  RECT rect;
  rect.left = -1; rect.right = -1; rect.top = -1; rect.bottom = -1;

  wxFindMaxSize((HWND)ms_handle, &rect);
  if (static_label)
    wxFindMaxSize(static_label, &rect);

  // Since we now have the absolute screen coords,
  // if there's a parent we must subtract its top left corner
  POINT point;
  point.x = rect.left;
  point.y = rect.top;
  if (parent)
  {
    wxWnd *cparent = (wxWnd *)(parent->handle);
    ::ScreenToClient(cparent->handle, &point);
  }

  *x = point.x;
  *y = point.y;
}

void wxText::SetSize(int x, int y, int width, int height)
{
  int currentX, currentY;
  GetPosition(&currentX, &currentY);
  if (x == -1)
    x = currentX;
  if (y == -1)
    y = currentY;

  char buf[300];
  int cx; // button font dimensions
  int cy;
  int clx; // label font dimensions
  int cly;

  wxGetCharSize((HWND)ms_handle, &cx, &cy,buttonFont);

  float label_width, label_height, label_x, label_y;
  float control_width, control_height, control_x, control_y;

  // Deal with default size (using -1 values)
  if (width<=0)
    width = DEFAULT_ITEM_WIDTH;

  if (static_label)
  {
    // Find size of label
    wxGetCharSize((HWND)ms_handle, &clx, &cly,labelFont);
    GetWindowText(static_label, buf, 300);
    GetTextExtent(buf, &label_width, &label_height, NULL, NULL,labelFont);

    // Given size is total label + edit size, so find individual
    // control sizes on that basis.
    if (labelPosition == wxHORIZONTAL)
    {
      if (height<=0)
        height = (int)((max(cly, cy))*EDIT_CONTROL_FACTOR) ;

      label_x = (float)x;
      label_y = (float)(y+(height - (int)label_height)/2); // Centre vertically
      label_width += (float)clx;

      control_x = label_x + label_width + cx;
      control_y = (float)y;
      control_width = width - (control_x - label_x);
      control_height = (float)height;
    }
    else // wxVERTICAL
    {
      label_x = (float)x;
      label_y = (float)y;

      control_x = (float)x;
      control_y = label_y + label_height + 3; // Allow for 3D border
      control_width = (float)width;

      if (height<=0)
        control_height = (float)(int)(cy*EDIT_CONTROL_FACTOR) ;
      else
        control_height = height - label_height - 3;
    }

    MoveWindow(static_label, (int)label_x, (int)label_y,
               (int)label_width, (int)label_height, TRUE);
  }
  else
  {
    control_x = (float)x;
    control_y = (float)y;
    control_width = (float)width;
    control_height = (float)height;
  }

  // Calculations may have made text size too small
  if (control_height <= 0)
    control_height = (float)(int)(cy*EDIT_CONTROL_FACTOR) ;

  if (control_width <= 0)
    control_width = DEFAULT_ITEM_WIDTH;

  MoveWindow((HWND)ms_handle, (int)control_x, (int)control_y,
                              (int)control_width, (int)control_height, TRUE);
  OnSize(width, height);
}

void wxText::SetFocus(void)
{
  wxItem::SetFocus();
}

// Clipboard operations
void wxText::Copy(void)
{
  HWND hWnd = GetHWND();
  SendMessage(hWnd, WM_COPY, 0, 0L);
}

void wxText::Cut(void)
{
  HWND hWnd = GetHWND();
  SendMessage(hWnd, WM_CUT, 0, 0L);
}

void wxText::Paste(void)
{
  HWND hWnd = GetHWND();
  SendMessage(hWnd, WM_PASTE, 0, 0L);
}

// Multi-line Text item
wxMultiText::wxMultiText(void)
{
}

wxMultiText::wxMultiText(wxPanel *panel, wxFunction Function, char *label, char *value,
               int x, int y, int width, int height, long style, char *name):
  wxbMultiText(panel, Function, label, value, x, y, width, height, style, name)
{
  Create(panel, Function, label, value, x, y, width, height, style, name);
}

Bool wxMultiText::Create(wxPanel *panel, wxFunction Function, char *label, char *value,
               int x, int y, int width, int height, long style, char *name)
{
  if (panel) panel->AddChild(this);
  buttonFont = panel->buttonFont ;
  labelFont = panel->labelFont ;
  backColour = panel->backColour ;
  labelColour = panel->labelColour ;
  buttonColour = panel->buttonColour ;
  if (height == -1) height = 50;

  window_parent = panel;
  labelPosition = panel->label_position;
  
  wxWinType = wxTYPE_HWND;
  windowStyle = style;
  wxWnd *cparent = NULL;
  if (panel)
    cparent = (wxWnd *)(panel->handle);

  panel->GetValidPosition(&x, &y);

  char *the_label = NULL ;

  if (label)
  {
    the_label = new char[strlen(label)+1] ;
    if (style&wxFIXED_LENGTH)
      for (int i=0;i<(int)strlen(label);i++)
        the_label[i]=MEANING_CHARACTER ;
    else
      strcpy(the_label,label) ;
    the_label[strlen(label)] = '\0' ;
  }

  // If label exists, create a static control for it.
  if (label)
  {
    static_label = CreateWindowEx(0, STATIC_CLASS, the_label,
                         STATIC_FLAGS,
                         0, 0, 0, 0, cparent->handle, (HMENU)NewId(),
                         wxhInstance, NULL);
#if CTL3D
    Ctl3dSubclassCtl(static_label);
#endif
    HDC the_dc = GetWindowDC(static_label) ;
    if (labelFont && labelFont->GetInternalFont(the_dc))
      SendMessage(static_label,WM_SETFONT,
                  (WPARAM)labelFont->GetInternalFont(the_dc),0L);
    ReleaseDC(static_label,the_dc) ;
  }
  else
    static_label = NULL;

  windows_id = (int)NewId();

  // Obscure method from the MS Developer's Network Disk for
  // using global memory instead of the local heap, which
  // runs out far too soon. Solves the problem with
  // failing to appear.
  globalHandle=GlobalAlloc(GMEM_MOVEABLE | GMEM_ZEROINIT,
                         256L);

  long msStyle = ES_MULTILINE | ES_LEFT | ES_WANTRETURN |
               WS_BORDER | WS_VISIBLE | WS_CHILD | WS_TABSTOP |
               WS_VSCROLL;

  if (windowStyle & wxREADONLY)
    msStyle |= ES_READONLY;

  if (windowStyle & wxHSCROLL)
    msStyle |= (WS_HSCROLL | ES_AUTOHSCROLL) ;

  HWND edit = CreateWindowEx(0, "EDIT", label,
               msStyle,
               0, 0, 0, 0, cparent->handle, (HMENU)windows_id,
               globalHandle, NULL);
#if CTL3D
  Ctl3dSubclassCtl(edit);
#endif

  ms_handle = (HANDLE)edit;

  HDC the_dc = GetWindowDC((HWND)ms_handle) ;
  if (buttonFont && buttonFont->GetInternalFont(the_dc))
    SendMessage((HWND)ms_handle,WM_SETFONT,
                (WPARAM)buttonFont->GetInternalFont(the_dc),0L);
  ReleaseDC((HWND)ms_handle,the_dc) ;

  SetSize(x, y, width, height);

  if (value)
    SetWindowText(edit, value);

  panel->AdvanceCursor(this);
  Callback(Function);

  if (label)
  {
    if (style&wxFIXED_LENGTH)
      SetLabel(label) ;
    if (the_label)
      delete [] the_label ;
  }

  return TRUE;

}

void wxMultiText::SetBackgroundColour(wxColour*col)
{
}

void wxMultiText::SetLabelColour(wxColour*col) 
{
}

void wxMultiText::SetButtonColour(wxColour*col)
{
}

char *wxMultiText::GetValue(void)
{
  // Fix by /Roffe for Borland 4.0.
  int TL = GetWindowTextLength((HWND)ms_handle);
  if (TL > 999)
    TL = 999;

  // Calling SendMessage this way is equivalent with a GetWindowText call.
  // Note that (some of) the parameters have to be casted appropriately...
  SendMessage((HWND)ms_handle, WM_GETTEXT, TL+1, (LPARAM)((LPSTR) wxBuffer));

  if(TL == 999)
    wxBuffer[999] = '\0';

  return wxBuffer;

/* Not sure if this is still needed.

// Julian! I can't see why this job below should have to be done -
// the edit control has the es_wantreturn attribute byte set,
// and will thus have the cr/lf characters inserted appropriately... /Roffe

  int buf_len = 1000;
  int no_chars = 0;
  int no_lines = (int)SendMessage((HWND)ms_handle, EM_GETLINECOUNT, 0, 0L);
  for (int i = 0; i < no_lines; i++)
  {
    *(LPINT)(wxBuffer+no_chars) = buf_len;
    int n = (int)SendMessage((HWND)ms_handle, EM_GETLINE, i,
                         (LONG)(wxBuffer+no_chars));
    no_chars += n;
    buf_len -= (n + 2);
    if (i < (no_lines - 1))
    {
      wxBuffer[no_chars] = 13;
      no_chars ++;
      wxBuffer[no_chars] = 10;
      no_chars ++;
    }
  }
  wxBuffer[no_chars] = 0;
  return wxBuffer;
*/
}

void wxMultiText::GetValue(char *buffer, int maxSize)
{
  // Fix by /Roffe for Borland 4.0.
  int TL = GetWindowTextLength((HWND)ms_handle);

  // We still want to keep within the limits for an edit control, won't we?
  if (maxSize > 32766)
    maxSize = 32766;

  if (TL > maxSize)
    TL = maxSize;

  // Here we use GetWindowText instead.
  GetWindowText((HWND)ms_handle, (LPSTR) buffer, TL);

  if(TL == maxSize)
    buffer[maxSize] = '\0';

/*
  buffer[0] = 0;
  int no_chars = 0;
  int no_lines = (int)SendMessage((HWND)ms_handle, EM_GETLINECOUNT, 0, 0L);
  for (int i = 0; i < no_lines; i++)
  {
    *(LPINT)(buffer+no_chars) = maxSize;
    int n = (int)SendMessage((HWND)ms_handle, EM_GETLINE, i,
                         (LONG)(buffer+no_chars));
    no_chars += n;
    maxSize -= (n + 2);
    if (i < (no_lines - 1))
    {
      buffer[no_chars] = 13;
      no_chars ++;
      buffer[no_chars] = 10;
      no_chars ++;
    }
  }
  buffer[no_chars] = 0;
*/
}

void wxMultiText::SetSize(int x, int y, int width, int height)
{
  int currentX, currentY;
  GetPosition(&currentX, &currentY);
  if (x == -1)
    x = currentX;
  if (y == -1)
    y = currentY;

  char buf[300];

  int cx;
  int cy;
  int clx;
  int cly;

  if (height <= 0)
    height = DEFAULT_ITEM_HEIGHT;

  wxGetCharSize((HWND)ms_handle, &cx, &cy,buttonFont);

  float label_width, label_height, label_x, label_y;
  float control_width, control_height, control_x, control_y;

  // Deal with default size (using -1 values)
  if (width<=0)
    width = DEFAULT_ITEM_WIDTH;

  if (static_label)
  {
    // Find size of label
    wxGetCharSize((HWND)ms_handle, &clx, &cly,labelFont);
    GetWindowText(static_label, buf, 300);
    GetTextExtent(buf, &label_width, &label_height, NULL, NULL,labelFont);

    // Given size is total label + edit size, so find individual
    // control sizes on that basis.
    if (labelPosition == wxHORIZONTAL)
    {
      label_x = (float)x;
      label_y = (float)y;
      label_width += (float)clx;

      control_x = label_x + label_width + cx;
      control_y = (float)y;
      control_width = width - (control_x - label_x);
      control_height = (float)height;
    }
    else // wxVERTICAL
    {
      label_x = (float)x;
      label_y = (float)y;

      control_x = (float)x;
      control_y = label_y + label_height + 3; // Allow for 3D border
      control_width = (float)width;

      control_height = height - label_height - 3;
    }

    MoveWindow(static_label, (int)label_x, (int)label_y,
               (int)label_width, (int)label_height, TRUE);
  }
  else
  {
    control_x = (float)x;
    control_y = (float)y;
    control_width = (float)width;
    control_height = (float)height;
  }

  // Calculations may have made text size too small
  if (control_height <= 0)
    control_height = DEFAULT_ITEM_HEIGHT;

  if (control_width <= 0)
    control_width = DEFAULT_ITEM_WIDTH;

//  wxDebugMsg("About to set the multitext height to %d", (int)control_height);

  MoveWindow((HWND)ms_handle, (int)control_x, (int)control_y,
                              (int)control_width, (int)control_height, TRUE);
  OnSize(width, height);
}


// Slider
wxSlider::wxSlider(void)
{
  wxWinType = wxTYPE_HWND;
  static_label = NULL;
  edit_value = NULL;
  static_min = NULL;
  windows_id = 0;
  page_size = 0;
  s_max = 0;
  s_min = 0;
  ms_handle = 0;
  static_max = 0;
}

wxSlider::wxSlider(wxPanel *panel, wxFunction func, char *label, int value,
           int min_value, int max_value, int width, int x, int y,
           long style, char *name):
  wxbSlider(panel, func, label, value, min_value, max_value, width, x, y,
            style, name)
{
  Create(panel, func, label, value, min_value, max_value, width, x, y,
         style, name);
}

Bool wxSlider::Create(wxPanel *panel, wxFunction func, char *label, int value,
           int min_value, int max_value, int width, int x, int y, long style, char *name)
{
  if (panel) panel->AddChild(this);
  buttonFont = panel->buttonFont ;
  labelFont = panel->labelFont ;
  backColour = panel->backColour ;
  labelColour = panel->labelColour ;
  buttonColour = panel->buttonColour ;
  wxWinType = wxTYPE_HWND;
  windowStyle = style;
  wxWnd *cparent = NULL;
  if (panel)
    cparent = (wxWnd *)(panel->handle);

  panel->GetValidPosition(&x, &y);

  int cx;
  int cy;
  wxGetCharSize(cparent->handle, &cx, &cy,buttonFont);

  char *the_label = NULL ;

  if (label)
  {
    the_label = new char[strlen(label)+1] ;
    if (style&wxFIXED_LENGTH)
      for (int i=0;i<(int)strlen(label);i++)
        the_label[i]=MEANING_CHARACTER ;
    else
      strcpy(the_label,label) ;
    the_label[strlen(label)] = '\0' ;
  }

  // If label exists, create a static control for it.
  if (label)
  {
    static_label = CreateWindowEx(0, STATIC_CLASS, the_label,
                         STATIC_FLAGS,
                         0, 0, 0, 0, cparent->handle, (HMENU)NewId(),
                         wxhInstance, NULL);
#if CTL3D
    Ctl3dSubclassCtl(static_label);
#endif
    HDC the_dc = GetWindowDC(static_label) ;
    if (labelFont && labelFont->GetInternalFont(the_dc))
      SendMessage(static_label,WM_SETFONT,
                  (WPARAM)labelFont->GetInternalFont(the_dc),0L);
    ReleaseDC(static_label,the_dc) ;
  }
  else
    static_label = NULL;

  edit_value = CreateWindowEx(0, "EDIT", NULL,
                         ES_AUTOHSCROLL | ES_LEFT | WS_VISIBLE | WS_CHILD |
                         WS_TABSTOP,
                         0, 0, 0, 0, cparent->handle, (HMENU)NewId(),
                         wxhInstance, NULL);
#if CTL3D
  Ctl3dSubclassCtl(edit_value);
#endif

  // Now create min static control
  sprintf(wxBuffer, "%d", min_value);
  static_min = CreateWindowEx(0, STATIC_CLASS, wxBuffer,
                         STATIC_FLAGS,
                         0, 0, 0, 0, cparent->handle, (HMENU)NewId(),
                         wxhInstance, NULL);
#if CTL3D
  Ctl3dSubclassCtl(static_min);
#endif

  // Now create slider
  windows_id = (int)NewId();
  HWND scroll_bar = CreateWindowEx(0, "SCROLLBAR", wxBuffer,
                         SBS_HORZ | WS_CHILD | WS_VISIBLE,
                         0, 0, 0, 0, cparent->handle, (HMENU)windows_id,
                         wxhInstance, NULL);
#if CTL3D
  Ctl3dSubclassCtl(scroll_bar);
#endif

  wxScrollBarList.Append((long)scroll_bar, this);

  page_size = (int)((max_value-min_value)/10);
  s_max = max_value;
  s_min = min_value;

  SetScrollRange(scroll_bar, SB_CTL, min_value, max_value, FALSE);
  SetScrollPos(scroll_bar, SB_CTL, value, FALSE);
  ShowWindow(scroll_bar, SW_SHOW);

  ms_handle = (HANDLE)scroll_bar;

  // Finally, create max value static item
  sprintf(wxBuffer, "%d", max_value);
  static_max = CreateWindowEx(0, STATIC_CLASS, wxBuffer,
                         STATIC_FLAGS,
                         0, 0, 0, 0, cparent->handle, (HMENU)NewId(),
                         wxhInstance, NULL);
#if CTL3D
  Ctl3dSubclassCtl(static_max);
#endif

  HDC the_dc = GetWindowDC((HWND)static_max) ;
  if (buttonFont && buttonFont->GetInternalFont(the_dc))
  {
    SendMessage((HWND)static_min,WM_SETFONT,
                (WPARAM)buttonFont->GetInternalFont(the_dc),0L);
    SendMessage((HWND)static_max,WM_SETFONT,
                (WPARAM)buttonFont->GetInternalFont(the_dc),0L);
    SendMessage((HWND)edit_value,WM_SETFONT,
                (WPARAM)buttonFont->GetInternalFont(the_dc),0L);
  }
  ReleaseDC((HWND)static_max,the_dc) ;

  SetSize(x, y, width, -1);
  SetValue(value);

  panel->AdvanceCursor(this);
  Callback(func);

  if (label)
  {
    if (style&wxFIXED_LENGTH)
      SetLabel(label) ;
    if (the_label)
      delete [] the_label ;
  }

  return TRUE;
}

// Called from wx_win.cc: wxWnd::OnHScroll
void wxSliderEvent(HWND bar, WORD wParam, WORD pos)
{
    wxNode *node = (wxNode *)wxScrollBarList.Find((long)bar);
    if (!node)
      return;

    wxSlider *slider = (wxSlider *)node->Data();
    int position = GetScrollPos(bar, SB_CTL);

    int nScrollInc;
    switch ( wParam )
    {
            case SB_LINEUP:
                    nScrollInc = -1;
                    break;

            case SB_LINEDOWN:
                    nScrollInc = 1;
                    break;

            case SB_PAGEUP:
                    nScrollInc = -slider->page_size;
                    break;

            case SB_PAGEDOWN:
                    nScrollInc = slider->page_size;;
                    break;

            case SB_THUMBTRACK:
                    nScrollInc = pos - position;
                    break;

            default:
                    nScrollInc = 0;
    }

    if (nScrollInc != 0)
    {
      int new_pos = position + nScrollInc;
      if (!(new_pos < slider->s_min || new_pos > slider->s_max))
      {
        slider->SetValue(new_pos);
        wxCommandEvent event(wxEVENT_TYPE_SLIDER_COMMAND);
        event.commandInt = new_pos;
        event.eventObject = slider;
        slider->ProcessCommand(event);
      }
    }
}

wxSlider::~wxSlider(void)
{
    if (static_min)
    {
      DestroyWindow(static_min);
    }
    if (static_max)
    {
      DestroyWindow(static_max);
    }
    if (edit_value)
    {
      DestroyWindow(edit_value);
    }
    if (static_label)
    {
      DestroyWindow(static_label);
    }
    wxScrollBarList.DeleteObject(this);
}

void wxSlider::SetBackgroundColour(wxColour*col)
{
}

void wxSlider::SetLabelColour(wxColour*col) 
{
}

void wxSlider::SetButtonColour(wxColour*col)
{
}

int wxSlider::GetValue(void)
{
  return GetScrollPos((HWND)ms_handle, SB_CTL);
}

char *wxSlider::GetLabel(void)
{
  if (static_label)
  {
    GetWindowText(static_label, wxBuffer, 300);
    return wxBuffer;
  }
  else return NULL;
}

void wxSlider::SetValue(int value)
{
  SetScrollPos((HWND)ms_handle, SB_CTL, value, TRUE);

  sprintf(wxBuffer, "%d", value);
  SetWindowText(edit_value, wxBuffer);
}

void wxSlider::SetLabel(char *label)
{
  if (static_label)
  {
    float w, h;
    RECT rect;

    wxWindow *parent = GetParent();
    GetWindowRect(static_label, &rect);

    // Since we now have the absolute screen coords,
    // if there's a parent we must subtract its top left corner
    POINT point;
    point.x = rect.left;
    point.y = rect.top;
    if (parent)
    {
      wxWnd *cparent = (wxWnd *)(parent->handle);
      ::ScreenToClient(cparent->handle, &point);
    }

    GetTextExtent(label, &w, &h, NULL, NULL,labelFont);
    MoveWindow(static_label, point.x, point.y, (int)(w + 10), (int)h,
               TRUE);
    SetWindowText(static_label, label);
  }
}

void wxSlider::GetSize(int *width, int *height)
{
  RECT rect;
  rect.left = -1; rect.right = -1; rect.top = -1; rect.bottom = -1;

  wxFindMaxSize((HWND)ms_handle, &rect);

  if (static_label)
  {
    wxFindMaxSize(static_label, &rect);
  }
  if (static_min)
  {
    wxFindMaxSize(static_min, &rect);
  }
  if (static_max)
  {
    wxFindMaxSize(static_max, &rect);
  }
  if (edit_value)
  {
    wxFindMaxSize(edit_value, &rect);
  }

  *width = rect.right - rect.left;
  *height = rect.bottom - rect.top;
}

void wxSlider::GetPosition(int *x, int *y)
{
  wxWindow *parent = GetParent();
  RECT rect;
  rect.left = -1; rect.right = -1; rect.top = -1; rect.bottom = -1;

  wxFindMaxSize((HWND)ms_handle, &rect);

  if (static_label)
  {
    wxFindMaxSize(static_label, &rect);
  }
  if (static_min)
  {
    wxFindMaxSize(static_min, &rect);
  }
  if (static_max)
  {
    wxFindMaxSize(static_max, &rect);
  }
  if (edit_value)
  {
    wxFindMaxSize(edit_value, &rect);
  }

  // Since we now have the absolute screen coords,
  // if there's a parent we must subtract its top left corner
  POINT point;
  point.x = rect.left;
  point.y = rect.top;
  if (parent)
  {
    wxWnd *cparent = (wxWnd *)(parent->handle);
    ::ScreenToClient(cparent->handle, &point);
  }

  *x = point.x;
  *y = point.y;
}

void wxSlider::SetSize(int x, int y, int width, int height)
{
  int currentX, currentY;
  GetPosition(&currentX, &currentY);
  if (x == -1)
    x = currentX;
  if (y == -1)
    y = currentY;

  char buf[300];

  int x_offset = x;
  int y_offset = y;
  float current_width;

  int cx;     // slider,min,max sizes
  int cy;
  float cyf;
  int cxs;    // label sizes
  int cys;
  int dy,dys ;  // Adjustement values (vertical) if font sizes differs.

  wxGetCharSize((HWND)ms_handle, &cx, &cy,buttonFont);

  if (static_label)
  {
    wxGetCharSize((HWND)ms_handle, &cxs, &cys,labelFont);
    GetWindowText(static_label, buf, 300);
    GetTextExtent(buf, &current_width, &cyf,NULL,NULL, labelFont);
    if (cys>cy)
    {
      dys = 0 ;
      dy = (cys-cy)/2 ; // center slider
    }
    else
    {
      dys = (cy-cys)/2; // center label
      dy = 0 ;
    }
    MoveWindow(static_label, x_offset, y+dys, (int)(current_width + 2*cxs), (int)cyf,
               TRUE);
    if (labelPosition==wxHORIZONTAL)
    {
      x_offset += (int)(current_width + 2*cxs + cxs);
      y_offset += dy ;
    }
    else
      y_offset += cys ;
  }

  float min_len;
  GetWindowText(static_min, buf, 300);
  GetTextExtent(buf, &min_len, &cyf,NULL,NULL, buttonFont);

  float max_len;
  GetWindowText(static_max, buf, 300);
  GetTextExtent(buf, &max_len, &cyf,NULL,NULL, buttonFont);

  if (edit_value)
  {
    int new_width = (int)(max(min_len, max_len));
    MoveWindow(edit_value, x_offset, y_offset, new_width, (int)cyf, TRUE);
    x_offset += new_width + cx;
  }

  MoveWindow(static_min, x_offset, y_offset, (int)min_len, cy, TRUE);
  x_offset += (int)(min_len + cx);

  int slider_length = (int)(width - x_offset - max_len - cx);

  // Slider must have a minimum/default length
  if (slider_length < 0)
    slider_length = 100;

  MoveWindow((HWND)ms_handle, x_offset, y_offset, slider_length, cy, TRUE);
  x_offset += slider_length + cx;

  MoveWindow(static_max, x_offset, y_offset, (int)max_len, cy, TRUE);
  OnSize(width, height);
}

extern wxList wxModelessWindows;

// Yeuch! Have to put this here since 'delete menu_bar' is not allowed before
// wxMenuBar has been defined!

wxFrame::~wxFrame(void)
{
  if (wx_menu_bar)
    delete wx_menu_bar;

  if (icon)
    delete icon;

  for (int i = 0; i < wxMAX_STATUS; i++)
    if (status_window[i])
    {
      status_window[i]->DestroyWindow();
      delete status_window[i];
    }

  if (this == wxTheApp->wx_frame)
  {
     wxTheApp->wx_frame = NULL;
     PostQuitMessage(0);
  }

  wxModelessWindows.DeleteObject(this);
}

HMENU wxFrame::GetWinMenu(void)
{
  if (handle)
    return ((wxWnd *)handle)->hMenu;
  else return 0;
}

/*
// Not currently used
void wxConvertDialogToPixels(wxWindow *control, int *x, int *y)
{
  if (control->window_parent && control->window_parent->is_dialog)
  {
    DWORD word = GetDialogBaseUnits();
    int xs = LOWORD(word);
    int ys = HIWORD(word);
    *x = (int)(*x * xs/4);
    *y = (int)(*y * ys/8);
  }
  else
  {
    *x = *x;
    *y = *y;
  }
}
*/

// Sub-classed edit control proc
LONG APIENTRY _EXPORT
  wxSubclassedControlProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
  wxItem *item = wxFindControlFromHandle(hWnd);

  if (!item)
  {
    wxDebugMsg("Panic! Cannot find wxItem for this HWND in wxSubclassedControlProc.\n");
    return FALSE;
  }

  switch (message)
  {
    case WM_GETDLGCODE:
    {
      if (item->GetWindowStyleFlag() & wxPROCESS_ENTER)
        return DLGC_WANTALLKEYS;
      break;
    }
    case WM_CHAR: // Always an ASCII character
    {
      if (wParam == VK_RETURN)
      {
        wxCommandEvent event(wxEVENT_TYPE_TEXT_ENTER_COMMAND);
        event.commandString = ((wxText *)item)->GetValue();
        event.eventObject = item;
        item->ProcessCommand(event);
      }
      break;
    }
    default:
    {
      return CallWindowProc(item->oldWndProc, hWnd, message, wParam, lParam);
    }
  }
  return CallWindowProc(item->oldWndProc, hWnd, message, wParam, lParam);
}

wxItem *wxFindControlFromHandle(HWND hWnd)
{
  if (!wxControlHandleList)
    return NULL;
  wxNode *node = wxControlHandleList->Find((long)hWnd);
  if (!node)
    return NULL;
  return (wxItem *)node->Data();
}

void wxAddControlHandle(HWND hWnd, wxItem *item)
{
  if (!wxControlHandleList)
    wxControlHandleList = new wxList(wxKEY_INTEGER);
  wxControlHandleList->Append((long)hWnd, item);
}

